/**
 * @license
 * Copyright (c) 2014, 2024, Oracle and/or its affiliates.
 * Licensed under The Universal Permissive License (UPL), Version 1.0
 * as shown at https://oss.oracle.com/licenses/upl/
 * @ignore
 */
import { Obj, Point, ArrayUtils, BaseComponentDefaults, CSSStyle, Agent, ColorUtils, JsonUtils, Container, Stroke, Line, SimpleMarker, Displayable, HtmlTooltipManager, ResourceUtils, SelectionEffectUtils, MouseEvent, KeyboardEvent, KeyboardHandler, Rectangle, TextUtils, BackgroundOutputText, OutputText, BackgroundMultilineText, MultilineText, TextObjPeer, EventManager, EventFactory, ToolkitUtils, Dimension, Rect, SimpleObjPeer, PathUtils, Path, Automation, BaseComponent, Math as Math$1, PatternFill, LinearGradientFill, SolidFill, Matrix, Polygon, Circle, CustomDatatipPeer, CustomAnimation, Animator, SvgDocumentUtils, IconButton, AnimFadeIn, AnimFadeOut, ParallelPlayable, Shape, Polyline, DataAnimationHandler, SequentialPlayable, AnimPopIn, PolygonUtils, LayoutUtils, ImageMarker, PixelMap, ClipPath, CategoryRolloverHandler, PanZoomHandler, MarqueeHandler, SimpleScrollbar, AriaUtils, SelectionHandler, BlackBoxAnimationHandler, Playable, Context, BaseComponentCache } from 'ojs/ojdvt-toolkit';
import { LinearScaleAxisValueFormatter, BaseAxisInfo, DataAxisInfoMixin } from 'ojs/ojdvt-axis';
import { Legend } from 'ojs/ojlegend-toolkit';
import { Overview } from 'ojs/ojdvt-overview';

const DvtChartDataItemUtils = {
  /**
   * Returns the default data item id based on series name and group name.
   * @param {string} series The series name.
   * @param {string} group The group name.
   * @return {string} The data item id.
   */
  createDataItemId: (series, group) => {
    return series + '; ' + group;
  },

  /**
   * Returns whether a and b are equal ids. The ids can be a string or and array of strings.
   * @param {object} a
   * @param {object} b
   * @param {dvt.Context} context The chart context
   * @return {boolean}
   */
  isEqualId: (a, b, context) => {
    if (a == null || b == null)
      // don't consider undefined ids as equal
      return false;
    return Obj.compareValues(context, a, b);
  }
};

/**
 * Creates an object representing the ID of a chart data item.
 * @constructor
 * @param {string} id The ID for the data item, if available.
 * @param {string} series The series ID for the chart data item.
 * @param {string|Array} group The group ID for the chart data item.
 * @param {dvt.Context} context The context for the chart.
 */
class DvtChartDataItem {
  constructor(id, series, group, context) {
    // Expose as named properties to simplify uptake.
    this.id = id;
    this.series = series;
    this.group = group;
    this.context = context;
  }

  /**
   * Determines if two DvtChartDataItem objects are equal.
   *
   * @param {DvtChartDataItem} dataItem The data item that will be used to test for equality.
   * @return {boolean} True if the two DvtChartDataItem objects are equal
   */
  equals(dataItem) {
    // Note that the id is not compared, because the series and group ids are considered the primary identifiers.
    // However, for nested items, we have to compare the id.
    if (dataItem instanceof DvtChartDataItem) {
      if (this.id != null || dataItem.id != null) {
        return DvtChartDataItemUtils.isEqualId(this.id, dataItem.id, this.context);
      }
      return (
        DvtChartDataItemUtils.isEqualId(this.series, dataItem.series, this.context) &&
        DvtChartDataItemUtils.isEqualId(this.group, dataItem.group, this.context)
      );
    }
    return false;
  }

  /**
   * @override
   */
  toString() {
    if (this.id != null && typeof this.id !== 'object') return this.id.toString();
    else return DvtChartDataItemUtils.createDataItemId(this.series, this.group);
  }

  /**
   * @override
   */
  valueOf() {
    return this.toString();
  }
}

/**
 * Utility functions for Chart.
 * @class
 */
const DvtChartTypeUtils = {
  /** @private @const */
  _SUPPORTED_TYPES: [
    'bar',
    'line',
    'area',
    'lineWithArea',
    'combo',
    'pie',
    'bubble',
    'scatter',
    'funnel',
    'pyramid',
    'stock',
    'boxPlot'
  ],

  /**
   * Returns true if the chart's type is valid.
   * @param {Chart} chart
   * @return {boolean}
   */
  isValidType: (chart) => {
    return DvtChartTypeUtils._SUPPORTED_TYPES.indexOf(chart.getType()) >= 0;
  },

  /**
   * Returns true if the chart is a spark.
   * @param {Chart} chart
   * @return {boolean}
   */
  isSpark: (chart) => {
    return chart.getOptions()['__spark'];
  },

  /**
   * Returns true if the chart is an overview background.
   * @param {Chart} chart
   * @return {boolean}
   */
  isOverview: (chart) => {
    return chart.getOptions()['_isOverview'];
  },

  /**
   * Returns true if the chart is a vertical type.
   * @param {Chart} chart
   * @return {boolean}
   */
  isVertical: (chart) => {
    return !DvtChartTypeUtils.isHorizontal(chart) && !DvtChartTypeUtils.isPolar(chart);
  },

  /**
   * Returns true if the chart is a horizontal type.
   * @param {Chart} chart
   * @return {boolean}
   */
  isHorizontal: (chart) => {
    return (
      chart.getOptions()['orientation'] == 'horizontal' &&
      !DvtChartTypeUtils.isPolar(chart) &&
      !DvtChartTypeUtils.isStock(chart) &&
      (DvtChartTypeUtils.isBLAC(chart) || DvtChartTypeUtils.isFunnel(chart))
    );
  },

  /**
   * Returns true if the chart is polar.
   * @param {Chart} chart
   * @return {boolean}
   */
  isPolar: (chart) => {
    return (
      chart.getOptions()['coordinateSystem'] == 'polar' &&
      !DvtChartTypeUtils.isStock(chart) &&
      !DvtChartTypeUtils.isBoxPlot(chart)
    );
  },

  /**
   * Returns true if the chart is a combo type.
   * @param {Chart} chart
   * @return {boolean}
   */
  isCombo: (chart) => {
    return chart.getType() == 'combo';
  },

  /**
   * Returns true if the chart is a bar graph.
   * @param {Chart} chart
   * @return {boolean}
   */
  isBar: (chart) => {
    return chart.getType() == 'bar';
  },

  /**
   * Returns true if the chart is a line graph.
   * @param {Chart} chart
   * @return {boolean}
   */
  isLine: (chart) => {
    return chart.getType() == 'line';
  },

  /**
   * Returns true if the chart is a line with area graph.
   * @param {Chart} chart
   * @return {boolean}
   */
  isLineWithArea: (chart) => {
    return chart.getType() == 'lineWithArea';
  },

  /**
   * Returns true if the chart is an area graph.
   * @param {Chart} chart
   * @return {boolean}
   */
  isArea: (chart) => {
    return chart.getType() == 'area';
  },

  /**
   * Returns true if the chart is a stock chart.
   * @param {Chart} chart
   * @return {boolean}
   */
  isStock: (chart) => {
    return chart.getType() == 'stock';
  },

  /**
   * Returns true if the chart is a box plot.
   * @param {Chart} chart
   * @return {boolean}
   */
  isBoxPlot: (chart) => {
    return chart.getType() == 'boxPlot';
  },

  /**
   * Returns true if the chart is a scatter graph.
   * @param {Chart} chart
   * @return {boolean}
   */
  isScatter: (chart) => {
    return chart.getType() == 'scatter';
  },

  /**
   * Returns true if the chart is a bubble graph.
   * @param {Chart} chart
   * @return {boolean}
   */
  isBubble: (chart) => {
    return chart.getType() == 'bubble';
  },

  /**
   * Returns true if the chart is a pie graph.
   * @param {Chart} chart
   * @return {boolean}
   */
  isPie: (chart) => {
    return chart.getType() == 'pie';
  },

  /**
   * Returns true if the chart is a funnel graph.
   * @param {Chart} chart
   * @return {boolean}
   */
  isFunnel: (chart) => {
    return chart.getType() == 'funnel';
  },

  /**
   * Returns true if the chart is a pyramid.
   * @param {Chart} chart
   * @return {boolean}
   */
  isPyramid: (chart) => {
    return chart.getType() == 'pyramid';
  },

  /**
   * Returns true if the chart supports dual-y.
   * @param {Chart} chart
   * @return {boolean}
   */
  isDualY: (chart) => {
    // Verify the chart type
    if (
      !DvtChartTypeUtils.hasAxes(chart) ||
      DvtChartTypeUtils.isScatterBubble(chart) ||
      DvtChartTypeUtils.isPolar(chart)
    )
      return false;

    // Dual-Y
    return true;
  },

  /**
   * Returns true if the chart is type bar, line, area, combo, stock, or boxPlot.
   * @param {Chart} chart
   * @return {boolean}
   */
  isBLAC: (chart) => {
    var type = chart.getType();
    return (
      type == 'bar' ||
      type == 'line' ||
      type == 'area' ||
      type == 'lineWithArea' ||
      type == 'combo' ||
      type == 'stock' ||
      type == 'boxPlot'
    );
  },

  /**
   * Returns true if the chart is type scatter or bubble.
   * @param {Chart} chart
   * @return {boolean}
   */
  isScatterBubble: (chart) => {
    var type = chart.getType();
    return type == 'scatter' || type == 'bubble';
  },

  /**
   * Returns true if the chart is type line, area, or lineWithArea.
   * @param {Chart} chart
   * @return {boolean}
   */
  isLineArea: (chart) => {
    var type = chart.getType();
    return type == 'line' || type == 'area' || type == 'lineWithArea';
  },

  /**
   * Returns whether zoom and scroll is supported for the chart type
   * @param {Chart} chart
   * @return {boolean}
   */
  isScrollSupported: (chart) => {
    return (
      !DvtChartTypeUtils.isPie(chart) &&
      !DvtChartTypeUtils.isFunnel(chart) &&
      !DvtChartTypeUtils.isPolar(chart) &&
      !DvtChartTypeUtils.isPyramid(chart)
    );
  },

  /**
   * Returns whether overview scrollbar is supported for the chart type
   * @param {Chart} chart
   * @return {boolean}
   */
  isOverviewSupported: (chart) => {
    return DvtChartTypeUtils.isBLAC(chart) && DvtChartTypeUtils.isVertical(chart);
  },

  /**
   * Returns true if the chart has axes.
   * @param {Chart} chart
   * @return {boolean}
   */
  hasAxes: (chart) => {
    return !(
      chart.getType() == 'pie' ||
      chart.getType() == 'funnel' ||
      chart.getType() == 'pyramid'
    );
  },

  /**
   * Returns whether the legend is rendered
   * (only type of the chart, legend.rendered and series.displayInLegend are considered)
   * @param {Chart} chart instance of the chart
   * @return {boolean} true, if the legend is rendered
   */
  isLegendRendered: (chart) => {
    var options = chart.getOptions();
    var hasLargeSeriesCount = chart.getOptionsCache().getFromCache('hasLargeSeriesCount');
    var legend = options['legend'];

    if (legend['rendered'] === 'off') {
      return false;
    }

    // Legend will not be rendered if rendered: auto is provided and
    // the series data count is large
    // The same check is being done in DvtChartLegendRenderer.render method.
    if (legend['rendered'] === 'auto' && hasLargeSeriesCount) {
      return false;
    }

    var hasSeriesDisplayedInLegend = chart
      .getOptionsCache()
      .getFromCache('hasSeriesDisplayedInLegend');
    var hasCustomLegendSections = legend['sections'] && legend['sections'].length !== 0;

    var isStock = DvtChartTypeUtils.isStock(chart);
    var isFunnel = DvtChartTypeUtils.isFunnel(chart);
    var isPyramid = DvtChartTypeUtils.isPyramid(chart);

    // For rendered: auto; Stock, funnel and pyramid charts do not render the legend unless
    // displayInLegend property is set in at least one series or
    // custom legend sections are provided.
    if (
      (isStock || isFunnel || isPyramid) &&
      !hasSeriesDisplayedInLegend &&
      !hasCustomLegendSections
    ) {
      return false;
    }
    return true;
  },

  /**
   * Returns true if the chart has a time axis.
   * @param {Chart} chart
   * @return {boolean}
   */
  hasTimeAxis: (chart) => {
    return (
      DvtChartTypeUtils.isBLAC(chart) && DvtChartTypeUtils.getTimeAxisType(chart) != 'disabled'
    );
  },

  /**
   * Returns true if the chart has a group axis.
   * @param {Chart} chart
   * @return {boolean}
   */
  hasGroupAxis: (chart) => {
    return (
      DvtChartTypeUtils.isBLAC(chart) && DvtChartTypeUtils.getTimeAxisType(chart) == 'disabled'
    );
  },

  /**
   * Returns time axis type of the chart.
   * @param {Chart} chart
   * @return {boolean}
   */
  getTimeAxisType: (chart) => {
    var timeAxisType = chart.getOptions()['timeAxisType'];
    if (
      timeAxisType &&
      timeAxisType != 'auto' &&
      DvtChartTypeUtils.isBLAC(chart) &&
      !DvtChartTypeUtils.isPolar(chart)
    )
      return timeAxisType;
    if (DvtChartTypeUtils.isStock(chart)) return 'skipGaps';
    return 'disabled';
  },

  /**
   * Returns if the chart contains mixed frequency data.
   * @param {Chart} chart The chart that will be rendered.
   * @return {boolean} True if chart has mixed data.
   */
  isMixedFrequency: (chart) => {
    return DvtChartTypeUtils.getTimeAxisType(chart) == 'mixedFrequency';
  }
};

const DvtChartCoordUtils = {
  /**
   * Converts polar coord to cartesian coord.
   * @param {number} r The radius.
   * @param {number} theta The angle.
   * @param {dvt.Rectangle} availSpace The availSpace, to compute the center.
   * @return {dvt.Point} The cartesian coord.
   */
  polarToCartesian: (r, theta, availSpace) => {
    var x = availSpace.x + availSpace.w / 2 + r * Math.sin(theta);
    var y = availSpace.y + availSpace.h / 2 - r * Math.cos(theta);
    return new Point(x, y);
  },

  /**
   * Converts the axis coordinate into the plot area coordinate.
   * @param {Chart} chart
   * @param {dvt.Point} coord The axis coordinate.
   * @param {dvt.Rectangle} availSpace
   * @return {dvt.Point} The plot area coordinate.
   */
  convertAxisCoord: (chart, coord, availSpace) => {
    if (DvtChartTypeUtils.isPolar(chart)) {
      var cartesian = DvtChartCoordUtils.polarToCartesian(coord.y, coord.x, availSpace);
      return new Point(cartesian.x, cartesian.y);
    } else if (DvtChartTypeUtils.isHorizontal(chart)) return new Point(coord.y, coord.x);
    else return new Point(coord.x, coord.y);
  }
};

const DvtChartFormatUtils$1 = {
  /**
   * Returns the valueFormat of the specified type.
   * @param {Chart} chart
   * @param {string} type The valueFormat type, e.g. series, group, or x.
   * @return {object} The valueFormat.
   */
  getValueFormat: (chart, type) => {
    var valueFormats = chart.getOptions()['valueFormats'];
    if (!valueFormats) return {};

    if (valueFormats[type]) return valueFormats[type];

    // For chart with time axis, if group valueFormat is not defined, fall back to x.
    if (type == 'group' && DvtChartTypeUtils.hasTimeAxis(chart))
      return DvtChartFormatUtils$1.getValueFormat(chart, 'x');

    // For BLAC charts, if y/y2 valueFormat is not defined, fall back to value.
    if (
      (type == 'y' || type == 'y2' || type == 'min' || type == 'max') &&
      DvtChartTypeUtils.isBLAC(chart)
    )
      return DvtChartFormatUtils$1.getValueFormat(chart, 'value');

    return {};
  },

  /**
   * Formats value with the converter from the valueFormat.
   * @param {Chart} chart
   * @param {object} valueFormat
   * @param {number} value The value to format.
   * @param {number} min (optional) Min value of the axis corresponding to the value.  This should be provided only if the
   *                     label should be formatted in the context of the axis extents.
   * @param {number} max (optional) Max value of the axis corresponding to the value.  This should be provided only if the
   *                     label should be formatted in the context of the axis extents.
   * @param {number} majorIncrement (optional) major increment of the axis corresponding to the value.  This should be
   *                                provided only if the label should be formatted in the context of the axis extents.
   * @return {string} The formatted value string.
   */
  formatVal: (chart, valueFormat, value, min, max, majorIncrement) => {
    var scaling = 'auto';
    var autoPrecision = 'on';
    var converter;
    // override from valueFormat
    if (valueFormat['scaling']) scaling = valueFormat['scaling'];
    if (valueFormat['autoPrecision']) autoPrecision = valueFormat['autoPrecision'];
    if (valueFormat['converter']) converter = valueFormat['converter'];

    // Retrieve the extent information
    min = min != null ? min : value;
    max = max != null ? max : value;
    majorIncrement = majorIncrement != null ? majorIncrement : 0;

    // Create the formatter
    var formatter = new LinearScaleAxisValueFormatter(
      min,
      max,
      majorIncrement,
      scaling,
      autoPrecision,
      chart.getOptions().translations
    );
    if (converter && converter['format']) return formatter.format(value, converter);
    else return formatter.format(value);
  },

  /**
   * Formats date with the converter from the valueFormat.
   * @param {object} valueFormat
   * @param {number} date The date to format.
   * @return {string} The formatted date string.
   */
  formatDateVal: (valueFormat, date) => {
    var converter = valueFormat['converter'];
    if (!converter) {
      return null;
    }
    if (converter['format']) {
      let _date = date;
      if (typeof date === 'number' && converter.resolvedOptions) {
        _date = new Date(date).toISOString();
      }
      return converter['format'](_date);
    }
    return null;
  }
};

/**
 * Data related utility functions for Chart.
 * @class
 */
const DvtChartDataUtils = {
  /** @private */
  _SERIES_TYPE_RAMP: ['bar', 'line', 'area'],
  /**
   * Returns true if the specified chart has data.
   * @param {Chart} chart
   * @return {boolean}
   */
  hasData: (chart) => {
    var options = chart.getOptions();

    // Check that there is a data object with at least one series
    if (!options || !options['series'] || options['series'].length < 1) return false;

    // Check that the minimum number of data points is present
    var minDataCount = 1;
    var seriesCount = DvtChartDataUtils.getSeriesCount(chart);
    for (var i = 0; i < seriesCount; i++) {
      var seriesItem = DvtChartDataUtils.getSeriesItem(chart, i);
      if (seriesItem && seriesItem['items'] && seriesItem['items'].length >= minDataCount)
        return true;
    }

    return false;
  },

  /**
   * Returns true if the specified chart doesn't have valid data.
   * @param {Chart} chart
   * @return {boolean}
   */
  hasInvalidData: (chart) => {
    return !DvtChartDataUtils.hasData(chart) || DvtChartDataUtils.hasInvalidTimeData(chart);
  },

  /**
   * Returns true if the specified chart has a timeAxis without valid numerical values.
   * @param {Chart} chart
   * @return {boolean}
   */
  hasInvalidTimeData: (chart) => {
    if (
      DvtChartTypeUtils.isFunnel(chart) ||
      DvtChartTypeUtils.isPie(chart) ||
      DvtChartTypeUtils.isPyramid(chart)
    )
      return false;

    var options = chart.getOptions();
    var groupCount = DvtChartDataUtils.getGroupCount(chart);

    // Check that there is a data object with at least one series
    if (!options || !options['series'] || options['series'].length < 1) return true;
    // Check that there is a data object with at least one group
    if (groupCount < 1) return true;

    var seriesIndex, groupIndex;
    var seriesCount = DvtChartDataUtils.getSeriesCount(chart);

    if (DvtChartTypeUtils.isMixedFrequency(chart)) {
      // Mixed frequency time axis uses x values to specify the dates
      for (seriesIndex = 0; seriesIndex < seriesCount; seriesIndex++) {
        for (groupIndex = 0; groupIndex < groupCount; groupIndex++) {
          var dataItem = DvtChartDataUtils.getDataItem(chart, seriesIndex, groupIndex);
          if (dataItem && (dataItem['x'] == null || isNaN(dataItem['x'])))
            //Invalid values will either be NaN or null depending on the browser
            return true;
        }
      }
    } else if (DvtChartTypeUtils.hasTimeAxis(chart)) {
      // Check that all values are numbers
      for (groupIndex = 0; groupIndex < groupCount; groupIndex++) {
        var groupItem = DvtChartDataUtils.getGroup(chart, groupIndex);
        if (groupItem == null || isNaN(groupItem)) return true;
      }
    }

    return false;
  },

  /**
   * Returns true if the specified chart series has non-null data.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @return {boolean}
   */
  hasSeriesData: (chart, seriesIndex) => {
    var seriesItem = DvtChartDataUtils.getSeriesItem(chart, seriesIndex);
    var dataItems = seriesItem['items'];
    if (dataItems) {
      for (var i = 0; i < dataItems.length; i++) {
        if (dataItems[i] != null) return true;
      }
    }

    // No data items or no non-null data items
    return false;
  },

  /**
   * Returns the number of series in the specified chart.
   * @param {Chart} chart
   * @return {number}
   */
  getSeriesCount: (chart) => {
    var seriesArray = chart.getOptions()['series'];
    return seriesArray ? seriesArray.length : 0;
  },

  /**
   * Returns the number of rendered y2 series in the specified chart. If type is specified returns the number of y2 series of that type.
   * @param {Chart} chart
   * @param {String} type (optional)
   * @param {Boolean} bIncludeHiddenSeries (optional) Whether or not to include hidden y2 series in the total, defaults to false.
   * @return {number}
   */
  getY2SeriesCount: (chart, type, bIncludeHiddenSeries) => {
    var y2SeriesCount = 0;
    var seriesCount = DvtChartDataUtils.getSeriesCount(chart);
    for (var seriesIndex = 0; seriesIndex < seriesCount; seriesIndex++) {
      if (type && DvtChartDataUtils.getSeriesType(chart, seriesIndex) != type) continue;
      if (!bIncludeHiddenSeries && !DvtChartDataUtils.isSeriesRendered(chart, seriesIndex))
        continue;
      if (DvtChartDataUtils.isAssignedToY2(chart, seriesIndex)) y2SeriesCount++;
    }
    return y2SeriesCount;
  },

  /**
   * Returns the id for the specified series.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @return {string} The id of the series.
   */
  getSeries: (chart, seriesIndex) => {
    var seriesItem = DvtChartDataUtils.getSeriesItem(chart, seriesIndex);
    if (seriesItem) {
      if (seriesItem['id']) return seriesItem['id'];
      else if (seriesItem['name'] || seriesItem['name'] === '') return seriesItem['name'];
      return String(seriesIndex);
    }
    return null;
  },

  /**
   * Returns the label for the specified series.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @return {string} The label for the series.
   */
  getSeriesLabel: (chart, seriesIndex) => {
    var seriesItem = DvtChartDataUtils.getSeriesItem(chart, seriesIndex);
    if (seriesItem && (seriesItem['name'] || seriesItem['name'] === '')) return seriesItem['name'];
    return null;
  },

  /**
   * Returns the index for the specified series.
   * @param {Chart} chart
   * @param {string} series The id of the series
   * @return {number} The index of the series.
   */
  getSeriesIdx: (chart, series) => {
    var numSeries = DvtChartDataUtils.getSeriesCount(chart);
    for (var seriesIndex = 0; seriesIndex < numSeries; seriesIndex++) {
      var seriesId = DvtChartDataUtils.getSeries(chart, seriesIndex);
      if (seriesId == series) return seriesIndex;
    }

    // No match found
    return -1;
  },

  /**
   * Returns the style index for the specified series.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @return {number} The index to use when looking for style information.
   */
  getSeriesStyleIdx: (chart, seriesIndex) => {
    if (chart.getOptionsCache().getFromCache('hasLargeSeriesCount')) return seriesIndex;

    var series = DvtChartDataUtils.getSeries(chart, seriesIndex);
    if (series == null) return seriesIndex;

    return chart.getSeriesStyleArray().indexOf(series);
  },

  /**
   * Returns the series item for the specified index.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @return {object} The series data item.
   */
  getSeriesItem: (chart, seriesIndex) => {
    if (isNaN(seriesIndex) || seriesIndex == null || seriesIndex < 0) return null;

    var options = chart.getOptions();
    if (options['series'] && options['series'].length > seriesIndex)
      return options['series'][seriesIndex];
    return null;
  },

  /**
   * Returns the data item for the specified index.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @return {object} The data item.
   */
  getDataItem: (chart, seriesIndex, groupIndex) => {
    if (isNaN(groupIndex) || groupIndex == null || groupIndex < 0) return null;

    var seriesItem = DvtChartDataUtils.getSeriesItem(chart, seriesIndex);
    if (seriesItem && seriesItem['items'] && seriesItem['items'].length > groupIndex)
      return seriesItem['items'][groupIndex];

    return null;
  },

  /**
   * Returns the raw data item for the specified index. Used for drill events
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @return {object} The data item.
   */
  getRawDataItem: (chart, seriesIndex, groupIndex) => {
    if (isNaN(groupIndex) || groupIndex == null || groupIndex < 0) {
      return null;
    }
    var dataItem = chart.getRawOptions().series[seriesIndex].items[groupIndex];
    if (typeof dataItem === 'number') {
      return dataItem;
    }

    // for missing data in group or series
    if (!dataItem) {
      return null;
    }

    var clonedData = Object.assign({}, dataItem);
    clonedData['itemData'] = dataItem['_itemData'];
    delete clonedData['_itemData'];
    return clonedData;
  },

  /**
   * Returns the id of the data item.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @return {string} The data item id.
   */
  getDataItemId: (chart, seriesIndex, groupIndex) => {
    var dataItem = DvtChartDataUtils.getDataItem(chart, seriesIndex, groupIndex);
    if (!dataItem) return null;

    if (dataItem['id'] != null) return dataItem['id'];

    // default id
    return DvtChartDataItemUtils.createDataItemId(
      DvtChartDataUtils.getSeries(chart, seriesIndex),
      DvtChartDataUtils.getGroup(chart, groupIndex)
    );
  },

  /**
   * Returns the specified nested data item.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @param {number} itemIndex
   * @return {number}
   */
  getNestedDataItem: (chart, seriesIndex, groupIndex, itemIndex) => {
    if (isNaN(itemIndex) || itemIndex == null || itemIndex < 0) return null;

    var dataItem = DvtChartDataUtils.getDataItem(chart, seriesIndex, groupIndex);
    if (dataItem && dataItem['items'] && dataItem['items'].length > itemIndex)
      return dataItem['items'][itemIndex];

    return null;
  },

  /**
   * Returns the number of nested data items for the specified data item.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @return {number}
   */
  getNestedDataItemCount: (chart, seriesIndex, groupIndex) => {
    var dataItem = DvtChartDataUtils.getDataItem(chart, seriesIndex, groupIndex);

    if (dataItem && dataItem['items']) return dataItem['items'].length;

    return 0;
  },

  /**
   * Returns the id of the nested data item.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @param {number} itemIndex
   * @return {string} The nested data item id.
   */
  getNestedDataItemId: (chart, seriesIndex, groupIndex, itemIndex) => {
    var nestedDataItem = DvtChartDataUtils.getNestedDataItem(
      chart,
      seriesIndex,
      groupIndex,
      itemIndex
    );
    if (!nestedDataItem) return null;

    if (nestedDataItem['id'] != null) return nestedDataItem['id'];

    // default id
    return (
      DvtChartDataUtils.getDataItemId(chart, seriesIndex, groupIndex) + '; ' + String(itemIndex)
    );
  },

  /**
   * Returns the index of the nested data item.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @param {string} id The id of the nested data item.
   * @return {number}
   */
  getNestedDataItemIdx: (chart, seriesIndex, groupIndex, id) => {
    var numItems = DvtChartDataUtils.getNestedDataItemCount(chart, seriesIndex, groupIndex);
    for (var itemIndex = 0; itemIndex < numItems; itemIndex++) {
      var itemId = DvtChartDataUtils.getNestedDataItemId(chart, seriesIndex, groupIndex, itemIndex);
      if (DvtChartDataItemUtils.isEqualId(itemId, id, chart.getCtx())) return itemIndex;
    }

    // No match found
    return -1;
  },

  /**
   * Returns the number of groups in the specified chart.
   * @param {Chart} chart
   * @return {number}
   */
  getGroupCount: (chart) => {
    return DvtChartDataUtils._getGroupsArray(chart).length;
  },

  /**
   * Returns the group id for the specified group index.
   * @param {Chart} chart
   * @param {Number} groupIndex The group index.
   * @return {string} The group id, null if the index is invalid.
   */
  getGroup: (chart, groupIndex) => {
    if (
      groupIndex != null &&
      groupIndex >= 0 &&
      groupIndex < DvtChartDataUtils.getGroupCount(chart)
    ) {
      var group = DvtChartDataUtils._getGroupsArray(chart)[groupIndex];
      if (group) {
        if (group['id'] != null) return group['id'];
        else if (group['name'] != null) return group['name'];
        return String(groupIndex);
      }
    }

    return null;
  },

  /**
   * Returns the index of the group with the specified id.
   * @param {Chart} chart
   * @param {string|Array} group The group whose index will be returned.
   * @return {number} The index of the group
   */
  getGroupIdx: (chart, group) => {
    var groups = DvtChartDataUtils.getGroups(chart);
    for (var i = 0; i < groups.length; i++) {
      if (
        group instanceof Array && groups[i] instanceof Array
          ? ArrayUtils.equals(group, groups[i])
          : group === groups[i]
      )
        return i;
    }
    return -1;
  },

  /**
   * Returns the group label for the specified group index.
   * @param {Chart} chart
   * @param {Number} groupIndex The group index.
   * @return {string} The group label, null if the index is invalid.
   */
  getGroupLabel: (chart, groupIndex) => {
    if (groupIndex >= 0 && groupIndex < DvtChartDataUtils.getGroupCount(chart)) {
      var group = DvtChartDataUtils._getGroupsArray(chart)[groupIndex];
      if (group) {
        if (group['name'] != null) return group['name'];
        else if (group['id'] != null || typeof group !== 'string')
          // Empty or null group name allowed if id is specified
          return '';
        return group;
      }
    }

    return null;
  },

  /**
   * Returns a list of the group ids in the chart's data.
   * @param {Chart} chart
   * @return {Array} An array of the group id's.
   */
  getGroups: (chart) => {
    var cacheKey = 'groups';
    var groups = chart.getOptionsCache().getFromCache(cacheKey);
    if (!groups) {
      var groupCount = DvtChartDataUtils.getGroupCount(chart);
      groups = [];
      for (var groupIndex = 0; groupIndex < groupCount; groupIndex++) {
        groups.push(DvtChartDataUtils.getGroup(chart, groupIndex));
      }
      chart.getOptionsCache().putToCache(cacheKey, groups);
    }
    return groups;
  },

  /**
   * Returns a structure containing the ids and names associated with each innermost group item.
   * @param {Chart} chart
   * @return {Array} An array of objects containing the ids and names associated with each innermost group item.
   * @private
   */
  _getGroupsArray: (chart) => {
    var options = chart.getOptions();
    var cacheKey = 'groupsArray';
    var groupsArray = chart.getOptionsCache().getFromCache(cacheKey);

    if (!groupsArray) {
      groupsArray = [];

      if (options['groups'])
        groupsArray = DvtChartDataUtils._getNestedGroups(options['groups'], groupsArray);

      for (var i = 0; i < groupsArray.length; i++) {
        if (groupsArray[i]['id'].length == 1) {
          groupsArray[i]['id'] = groupsArray[i]['id'][0];
          groupsArray[i]['name'] = groupsArray[i]['name'][0];
        }
      }

      chart.getOptionsCache().putToCache(cacheKey, groupsArray);
    }

    return groupsArray;
  },

  /**
   * Returns a structure containing the ids and names associated with each innermost group item.
   * @param {Array} groups An array of chart groups
   * @param {Array} groupsArray The array of objects associated with each group item
   * @return {Array} An array of objects containing the ids and names associated with each innermost group item.
   * @private
   */
  _getNestedGroups: (groups, groupsArray) => {
    if (!groups || groups.length == 0) return [];

    for (var i = 0; i < groups.length; i++) {
      var group = groups[i];
      var elementId = null;
      var elementName = null;
      if (group != null) {
        elementId = group.id != null ? group.id : group.name;
        elementId = elementId != null ? elementId : group;
        elementName = group['name'] ? group['name'] : group;
      }
      if (typeof elementId == 'object') elementId = null;
      if (typeof elementName == 'object') elementName = null;

      if (group && group['groups']) {
        var innerGroupArray = DvtChartDataUtils._getNestedGroups(group['groups'], []);
        if (innerGroupArray.length == 0) innerGroupArray.push({ id: [], name: [] });
        for (var j = 0; j < innerGroupArray.length; j++) {
          innerGroupArray[j]['id'].unshift(elementId);
          innerGroupArray[j]['name'].unshift(elementName);
        }
        groupsArray = groupsArray.concat(innerGroupArray);
      } else groupsArray.push({ id: [elementId], name: [elementName] });
    }
    return groupsArray;
  },

  /**
   * Returns the number of levels of hierarchical groups
   * @param {Chart} chart
   * @return {number} The number of label levels.
   */
  getNumLevels: (chart) => {
    var cacheKey = 'groupsNumLevels';
    var numLevels = chart.getOptionsCache().getFromCache(cacheKey);
    if (numLevels != null) return numLevels;

    // Compute the value
    numLevels = 0;
    var groupsArray = DvtChartDataUtils._getGroupsArray(chart);
    for (var i = 0; i < groupsArray.length; i++) {
      var group = groupsArray[i];
      if (group && group['id']) {
        var length = Array.isArray(group['id']) ? group['id'].length : 1;
        numLevels = Math.max(numLevels, length);
      }
    }

    // Put the computed value into the cache and return
    chart.getOptionsCache().putToCache(cacheKey, numLevels);
    return numLevels;
  },

  /**
   * Returns the value for the specified data item.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @param {number} itemIndex
   * @return {number} The value of the specified data item.
   */
  getVal: (chart, seriesIndex, groupIndex, itemIndex) => {
    // Use the cached value if it has been computed before
    var val;
    var cacheKey = 'value';
    var isNested = typeof itemIndex == 'number' && itemIndex >= 0;
    if (!isNested) {
      val = chart.getOptionsCache().getFromCachedMap2D(cacheKey, seriesIndex, groupIndex);
      if (val !== undefined)
        // anything that's defined, including null
        return val;
    }

    var dataItem = isNested
      ? DvtChartDataUtils.getNestedDataItem(chart, seriesIndex, groupIndex, itemIndex)
      : DvtChartDataUtils.getDataItem(chart, seriesIndex, groupIndex);

    val = null;
    if (dataItem != null) {
      if (typeof dataItem != 'object')
        // Number, just return
        val = dataItem;
      else if (DvtChartTypeUtils.isStock(chart) && dataItem['close'] != null)
        // Use the close value for stock
        val = dataItem['close'];
      else if (dataItem['value'] != null)
        // Object with value property
        val = dataItem['value'];
      else if (dataItem['y'] != null)
        // Object with y property
        val = dataItem['y'];
    }

    //  - timeaxis chart breaks when series value is NaN
    val = val || (val === 0 ? 0 : null);

    // Cache the value
    if (!isNested) chart.getOptionsCache().putToCachedMap2D(cacheKey, seriesIndex, groupIndex, val);

    return val;
  },

  /**
   * Returns the cumulative value for the specified data item, taking into account stacked values.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @param {boolean} bIncludeHiddenSeries True if hidden series should be included in the value calculation.
   * @return {number} The value of the specified data item.
   */
  getCumulativeVal: (chart, seriesIndex, groupIndex, bIncludeHiddenSeries) => {
    if (!DvtChartDataUtils.isStacked(chart)) {
      return DvtChartDataUtils.getVal(chart, seriesIndex, groupIndex);
    }

    // Use the cached value if it has been computed before
    var cacheKey = bIncludeHiddenSeries ? 'cumValueH' : 'cumValue';
    var cumVal = chart.getCache().getFromCachedMap2D(cacheKey, seriesIndex, groupIndex);
    if (cumVal !== undefined)
      // anything that's defined, including null
      return cumVal;

    // Match the series type and add up the values
    var seriesType = DvtChartDataUtils.getSeriesType(chart, seriesIndex);
    var bAssignedToY2 = DvtChartDataUtils.isAssignedToY2(chart, seriesIndex);
    var value = DvtChartDataUtils.getVal(chart, seriesIndex, groupIndex);
    var category = DvtChartDataUtils.getStackCategory(chart, seriesIndex);

    // Determine if to add to negative stack
    // If a bar chart use item value, otherwise use the whole series
    var isNegative =
      seriesType == 'bar' ? value < 0 : DvtChartDataUtils.isSeriesNegative(chart, seriesIndex);

    cumVal = 0;
    for (var i = seriesIndex; i >= 0; i--) {
      // Looping in reverse order to leverage previoulsy cached cumulative values
      // Skip series that are not rendered
      if (!bIncludeHiddenSeries && !DvtChartDataUtils.isDataItemRendered(chart, i, groupIndex))
        continue;

      // Skip series that don't match the type
      if (seriesType != DvtChartDataUtils.getSeriesType(chart, i)) continue;

      // Skip series who aren't assigned to the same y axis
      if (bAssignedToY2 != DvtChartDataUtils.isAssignedToY2(chart, i)) continue;

      // Add up all the values for items in the group in the same stack.  Null values are treated as 0.
      if (DvtChartDataUtils.getStackCategory(chart, i) == category) {
        var groupValue = DvtChartDataUtils.getVal(chart, i, groupIndex);

        // only add up positive values of items if current bar being processed is positive, and vice versa.
        var isCurrentNegative =
          seriesType == 'bar' ? groupValue < 0 : DvtChartDataUtils.isSeriesNegative(chart, i);

        if ((isNegative && isCurrentNegative) || (!isNegative && !isCurrentNegative)) {
          var prevCumVal = chart.getCache().getFromCachedMap2D(cacheKey, i, groupIndex);
          if (prevCumVal !== undefined) {
            cumVal = value + (prevCumVal || 0);
            break;
          }
          cumVal += groupValue == null || isNaN(groupValue) ? 0 : groupValue;
        }
      }
    }

    // Cache the value
    chart.getCache().putToCachedMap2D(cacheKey, seriesIndex, groupIndex, cumVal);
    return cumVal;
  },

  /**
   * Returns the low value for the specified data item.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @return {number} The value of the specified data item.
   */
  getLowVal: (chart, seriesIndex, groupIndex) => {
    var dataItem = DvtChartDataUtils.getDataItem(chart, seriesIndex, groupIndex);
    if (dataItem == null)
      // null or undefined, return null
      return null;
    if (DvtChartDataUtils.getSeriesType(chart, seriesIndex) == 'candlestick') {
      if (dataItem['low'] == null && dataItem['close'] != null) {
        if (dataItem['open'] != null) return Math.min(dataItem['close'], dataItem['open']);
        return dataItem['close'];
      }
      return dataItem['low'];
    }
    if (dataItem['low'] != null && dataItem['close'] == null) return dataItem['low'];
    return null;
  },

  /**
   * Returns the high value for the specified data item.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @return {number} The value of the specified data item.
   */
  getHighVal: (chart, seriesIndex, groupIndex) => {
    var dataItem = DvtChartDataUtils.getDataItem(chart, seriesIndex, groupIndex);
    if (dataItem == null)
      // null or undefined, return null
      return null;
    if (DvtChartDataUtils.getSeriesType(chart, seriesIndex) == 'candlestick') {
      if (dataItem['high'] == null) {
        if (dataItem['open'] != null) return Math.max(dataItem['close'], dataItem['open']);
        return dataItem['close'];
      }
      return dataItem['high'];
    }
    if (dataItem['high'] != null && dataItem['close'] == null) return dataItem['high'];
    return null;
  },

  /**
   * Returns the X value of a data point.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @return {number} The X value.
   */
  getXVal: (chart, seriesIndex, groupIndex) => {
    var dataItem = DvtChartDataUtils.getDataItem(chart, seriesIndex, groupIndex);
    return DvtChartDataUtils.getXValFromItem(chart, dataItem, groupIndex);
  },

  /**
   * Returns the X value given an item object and the group index. Used by series items and ref obj items.
   * @param {Chart} chart
   * @param {object} item The item object of a series or ref obj.
   * @param {number} groupIndex
   * @return {number} The X value.
   */
  getXValFromItem: (chart, item, groupIndex) => {
    if (item != null && item['x'] != null) return item['x'];
    if (DvtChartTypeUtils.hasGroupAxis(chart)) return groupIndex;
    if (DvtChartTypeUtils.hasTimeAxis(chart) && !DvtChartTypeUtils.isMixedFrequency(chart))
      return DvtChartDataUtils.getGroupLabel(chart, groupIndex);
    return null;
  },

  /**
   * Returns the target value for the specified data item in funnel charts.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @return {number} The target value of the specified data item.
   */
  getTargetVal: (chart, seriesIndex) => {
    var dataItem = DvtChartDataUtils.getDataItem(chart, seriesIndex, 0);
    if (dataItem == null || typeof dataItem != 'object') return null;
    return dataItem['targetValue'];
  },

  /**
   * Returns the z value for the specified data item.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @param {number} defaultVal The default value if the z is not specified,
   * @return {number} The z value of the specified data item.
   */
  getZVal: (chart, seriesIndex, groupIndex, defaultVal) => {
    var dataItem = DvtChartDataUtils.getDataItem(chart, seriesIndex, groupIndex);
    if (dataItem == null || typeof dataItem != 'object') return defaultVal;
    if (dataItem['z'] != null)
      // Object with value property
      return Math.max(0, dataItem['z']); // override any negative z-values as 0
    return defaultVal;
  },

  /**
   * Returns true if the stock value is rising, false otherwise. Returns true for null or equal values.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @return {boolean}
   */
  isStockValRising: (chart, seriesIndex, groupIndex) => {
    // Note: We return true for equality or null values, because stocks are generally shown as black for 0 or positive.
    var dataItem = DvtChartDataUtils.getDataItem(chart, seriesIndex, groupIndex);
    return dataItem ? dataItem['open'] <= dataItem['close'] : true;
  },

  /**
   * Returns the categories of the specified item.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @param {number=} groupIndex
   * @param {number} itemIndex
   * @return {string}
   */
  getCategories: (chart, seriesIndex, groupIndex, itemIndex) => {
    var nestedDataItem = DvtChartDataUtils.getNestedDataItem(
      chart,
      seriesIndex,
      groupIndex,
      itemIndex
    );
    if (nestedDataItem && nestedDataItem['categories']) return nestedDataItem['categories'];

    var dataItem = DvtChartDataUtils.getDataItem(chart, seriesIndex, groupIndex);
    if (dataItem && dataItem['categories']) return dataItem['categories'];

    var seriesItem = DvtChartDataUtils.getSeriesItem(chart, seriesIndex);
    if (seriesItem && seriesItem['categories']) return seriesItem['categories'];

    var series = DvtChartDataUtils.getSeries(chart, seriesIndex);
    if (series != null) return [series];

    return [];
  },

  /**
   * Returns true if the series is assigned to the y2 axis.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @return {boolean} True if the series is assigned to the y2 axis.
   */
  isAssignedToY2: (chart, seriesIndex) => {
    if (!chart.getOptionsCache().getFromCache('hasY2Assignment')) return false;
    var seriesItem = DvtChartDataUtils.getSeriesItem(chart, seriesIndex);
    return seriesItem && seriesItem['assignedToY2'] == 'on' && DvtChartTypeUtils.isDualY(chart);
  },

  /**
   * Returns the array of initially selected objects for a chart.  This information is based on the data object.
   * @param {Chart} chart The chart that will be rendered.
   * @return {array} The array of selected objects.
   */
  getInitialSelection: (chart) => {
    var selection = chart.getOptions()['selection'];
    var hasDataProvider = chart.getOptions()['data'] != null;
    if (!selection) selection = [];

    // Process the data item ids and fill in series and group information
    var peers = chart.getChartObjPeers();
    for (var i = 0; i < selection.length; i++) {
      var id = selection[i]['id'] != null && !hasDataProvider ? selection[i]['id'] : selection[i]; // check first if selection object has an id value
      if (selection[i]['id'] == null && !selection[i]['series'] && !selection[i]['group']) {
        selection[i] = { id: id };
      }

      // If id is defined, but series and group are not
      if (id != null && !(selection[i]['series'] && selection[i]['group'])) {
        for (var j = 0; j < peers.length; j++) {
          var peer = peers[j];
          if (DvtChartDataItemUtils.isEqualId(id, peer.getDataItemId(), chart.getCtx())) {
            selection[i]['series'] = peer.getSeries();
            selection[i]['group'] = peer.getGroup();
            break;
          }
        }
      }
    }

    return selection;
  },

  /**
   * Returns the current selection for the chart.  This selection is in the format of the data['selection'] API.
   * @param {Chart} chart The chart that will be rendered.
   * @return {array} The array of selected objects.
   */
  getCurrentSelection: (chart) => {
    var selection = [];
    var handler = chart.getSelectionHandler();
    if (handler) {
      var selectedIds = handler.getSelectedIds();
      for (var i = 0; i < selectedIds.length; i++) {
        var selectedId = selectedIds[i]; // selectedId is an instance of DvtChartDataItem
        selection.push({ series: selectedId.series, group: selectedId.group, id: selectedId.id });
      }
    }

    return selection;
  },

  /**
   * Returns whether the stock chart has a volume series
   * @param {Chart} chart
   * @return {boolean}
   */
  hasVolumeSeries: (chart) => {
    var hasVolume = chart.getOptionsCache().getFromCache('hasVolume');
    return hasVolume ? hasVolume : false;
  },

  /**
   * Returns whether the data point is currently selected.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @param {number} itemIndex
   * @return {boolean}
   */

  isDataSelected: (chart, seriesIndex, groupIndex, itemIndex) => {
    var isNested = !isNaN(itemIndex) && itemIndex != null && itemIndex >= 0;
    var id = isNested
      ? DvtChartDataUtils.getNestedDataItemId(chart, seriesIndex, groupIndex, itemIndex)
      : DvtChartDataUtils.getDataItemId(chart, seriesIndex, groupIndex);
    var series = DvtChartDataUtils.getSeries(chart, seriesIndex);
    var group = DvtChartDataUtils.getGroup(chart, groupIndex);

    // Check based on the selection attribute instead of asking the selectionHandler because
    // this method is called before the initial selection is set.
    var selection = chart.getOptions()['selection'];
    if (!selection) selection = [];

    for (var i = 0; i < selection.length; i++) {
      if (
        DvtChartDataItemUtils.isEqualId(id, selection[i], chart.getCtx()) ||
        DvtChartDataItemUtils.isEqualId(id, selection[i]['id'], chart.getCtx())
      )
        return true;
      if (
        selection[i]['id'] == null &&
        DvtChartDataItemUtils.isEqualId(series, selection[i]['series'], chart.getCtx()) &&
        DvtChartDataItemUtils.isEqualId(group, selection[i]['group'], chart.getCtx())
      )
        return true;
    }

    return false;
  },

  /**
   * Returns the default data label for the specified data point. It ignores the dataLabel function
   * @param {Chart} chart
   * @param {number} seriesIndex The series index.
   * @param {number} groupIndex The group index.
   * @param {number} itemIndex The nested item index.
   * @param {number} type (optional) Data label type: low, high, or value.
   * @param {boolean} isStackLabel true if label for stack cummulative, false otherwise
   * @return {string} The default data label, null if the index is invalid.
   */
  getDefaultDataLabel: (chart, seriesIndex, groupIndex, itemIndex, type, isStackLabel) => {
    var label;
    if (isStackLabel) label = DvtChartDataUtils.getCumulativeVal(chart, seriesIndex, groupIndex);
    else {
      var isNested = !isNaN(itemIndex) && itemIndex != null && itemIndex >= 0;
      var dataItem = isNested
        ? DvtChartDataUtils.getNestedDataItem(chart, seriesIndex, groupIndex, itemIndex)
        : DvtChartDataUtils.getDataItem(chart, seriesIndex, groupIndex);
      if (!dataItem) return null;

      label = dataItem['label'];
      // Range series data label support
      if (type == 'low') label = label instanceof Array ? label[0] : label;
      else if (type == 'high') label = label instanceof Array ? label[1] : null;
    }

    if (label != null) {
      // Numbers will be formatted, while all other labels will be treated as strings
      if (typeof label == 'number') {
        // Find the extents of the corresponding axis
        // Note: We assume y axis here because charts with numerical x axis would not pass a single value for the label.
        var min, max, majorIncrement;
        var bAssignedToY2 = DvtChartDataUtils.isAssignedToY2(chart, seriesIndex);
        var axis = bAssignedToY2 && chart.y2Axis ? chart.y2Axis : chart.yAxis;
        if (axis) {
          var axisInfo = axis.getInfo();
          min = axisInfo.getGlobalMin();
          max = axisInfo.getGlobalMax();
          majorIncrement = axisInfo.getMajorIncrement();
        }

        var valueFormat = DvtChartFormatUtils$1.getValueFormat(chart, 'label');
        return DvtChartFormatUtils$1.formatVal(chart, valueFormat, label, min, max, majorIncrement);
      }
      return label;
    }
    return null;
  },

  /**
   * Returns the stack category of the specified series. If the chart is unstacked, returns the series name.
   * @param {Chart} chart
   * @param {Number} seriesIndex The series index.
   * @return {String} The stack category for stacked; the series name for unstacked.
   */
  getStackCategory: (chart, seriesIndex) => {
    var cacheKey = 'stackCategory';
    var stackCategory = chart.getCache().getFromCachedMap(cacheKey, seriesIndex);
    if (typeof stackCategory != 'undefined') {
      return stackCategory;
    }

    if (DvtChartDataUtils.isStacked(chart))
      stackCategory = DvtChartDataUtils.getSeriesItem(chart, seriesIndex)['stackCategory'] || null;
    else stackCategory = DvtChartDataUtils.getSeries(chart, seriesIndex) || null; // each series is its own stack category

    chart.getCache().putToCachedMap(cacheKey, seriesIndex, stackCategory);
    return stackCategory;
  },

  /**
   * Returns the lists of the stack categories for a series type.
   * @param {Chart} chart
   * @param {String} type (optional) The series type.
   * @param {Boolean} bIncludeHiddenSeries (optional) Whether or not to include hidden series categories, defaults to false.
   * @return {Object} An object containing the arrays of stack categories for y and y2 axis respectively.
   */
  getStackCategories: (chart, type, bIncludeHiddenSeries) => {
    var yCategories = [],
      y2Categories = [];
    var yCategoriesHash = {},
      y2CategoriesHash = {}; // this is for performance to track categories processed
    var cacheKey = 'stackCategories';
    var categories = chart.getCache().getFromCachedMap2D(cacheKey, type, bIncludeHiddenSeries);
    if (categories) return categories;

    var seriesCount = DvtChartDataUtils.getSeriesCount(chart);
    for (var s = 0; s < seriesCount; s++) {
      // Intentionally split up below for readability
      if (!DvtChartDataUtils.isSeriesRendered(chart, s) && !bIncludeHiddenSeries)
        // skip unrendered series
        continue;
      else if (type) {
        // Treat candlestick and boxPlot as bar, as it inherits all the gap properties from bar.
        var seriesType = DvtChartDataUtils.getSeriesType(chart, s);
        if (seriesType == 'candlestick' || seriesType == 'boxPlot') seriesType = 'bar';

        if (type != seriesType) continue;
      }

      var category = DvtChartDataUtils.getStackCategory(chart, s);
      if (DvtChartDataUtils.isAssignedToY2(chart, s)) {
        if (!y2CategoriesHash[category]) {
          y2Categories.push(category);
          y2CategoriesHash[category] = true;
        }
      } else if (!yCategoriesHash[category]) {
        yCategories.push(category);
        yCategoriesHash[category] = true;
      }
    }

    categories = { y: yCategories, y2: y2Categories };
    chart.getCache().putToCachedMap2D(cacheKey, type, bIncludeHiddenSeries, categories);
    return categories;
  },

  /**
   * Computes z-value width of the specified stack category in a group.
   * @param {Chart} chart
   * @param {String} category The stack category. Use the series name for unstacked charts.
   * @param {Number} groupIndex
   * @param {Boolean} isY2 Whether the stack belongs to y2 axis.
   * @return {Number} The z-value width.
   */
  getBarCategoryZ: (chart, category, groupIndex, isY2) => {
    var width = 0;
    var seriesCount = DvtChartDataUtils.getSeriesCount(chart);
    for (var s = 0; s < seriesCount; s++) {
      var seriesType = DvtChartDataUtils.getSeriesType(chart, s);
      if (
        (seriesType != 'bar' && seriesType != 'candlestick' && seriesType != 'boxPlot') ||
        DvtChartDataUtils.getStackCategory(chart, s) != category ||
        !DvtChartDataUtils.isSeriesRendered(chart, s)
      )
        continue;

      // Compute the maximum z-value of the bars in the stack.
      var isSeriesY2 = DvtChartDataUtils.isAssignedToY2(chart, s);
      if ((isY2 && isSeriesY2) || (!isY2 && !isSeriesY2))
        width = Math.max(width, DvtChartDataUtils.getZVal(chart, s, groupIndex, 1));
    }

    return width;
  },

  /**
   * Returns the marker position for the specified index.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @param {number} itemIndex
   * @param {dvt.Rectangle} availSpace
   * @return {dvt.Point} The marker position.
   */
  getMarkerPos: (chart, seriesIndex, groupIndex, itemIndex, availSpace) => {
    var xAxis = chart.xAxis;
    var yAxis = DvtChartDataUtils.isAssignedToY2(chart, seriesIndex) ? chart.y2Axis : chart.yAxis;
    var isPolar = DvtChartTypeUtils.isPolar(chart);
    var bRange = DvtChartDataUtils.isRangeSeries(chart, seriesIndex);
    var isNested = !isNaN(itemIndex) && itemIndex != null && itemIndex >= 0;

    // Get the x-axis position
    var xValue = DvtChartDataUtils.getXVal(chart, seriesIndex, groupIndex);
    var yValue = isNested
      ? DvtChartDataUtils.getVal(chart, seriesIndex, groupIndex, itemIndex)
      : DvtChartDataUtils.getCumulativeVal(chart, seriesIndex, groupIndex);
    var xCoord, yCoord;

    // Get the position of the marker
    if (DvtChartTypeUtils.isBubble(chart)) {
      // The yValue for polar shouldn't go below the minimum because it will appear on the opposite side of the chart
      if (isPolar && yValue < yAxis.getInfo().getViewportMin()) return null;
      // Markers for most graph types must be within the plot area to be rendered.  Bubble markers
      // do not, as they are available clipped to the plot area bounds.
      if (isPolar) xCoord = xAxis.getCoordAt(xValue);
      // if we use unbounded here, the value will wrap around the angle.
      else xCoord = xAxis.getUnboundedCoordAt(xValue);
      yCoord = yAxis.getUnboundedCoordAt(yValue);
    } else if (bRange) {
      var lowCoord = yAxis.getCoordAt(DvtChartDataUtils.getLowVal(chart, seriesIndex, groupIndex));
      var highCoord = yAxis.getCoordAt(
        DvtChartDataUtils.getHighVal(chart, seriesIndex, groupIndex)
      );

      xCoord = xAxis.getCoordAt(DvtChartDataUtils.getXVal(chart, seriesIndex, groupIndex));
      yCoord = (lowCoord + highCoord) / 2;
    } else {
      xCoord = xAxis.getCoordAt(xValue);
      yCoord = yAxis.getCoordAt(yValue);
    }
    if (xCoord == null || yCoord == null) return null;

    return DvtChartCoordUtils.convertAxisCoord(chart, new Point(xCoord, yCoord), availSpace);
  },

  /**
   * Returns the marker position for the specified index. Optimized for large data
   * scatter and bubble charts. Differs from the normal getMarkerPosition in that
   * elements just outside the viewport are ignored.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @return {dvt.Point} The marker position.
   */
  getScatterBubbleMarkerPos: (chart, seriesIndex, groupIndex) => {
    var xAxis = chart.xAxis;
    var yAxis = chart.yAxis;

    // Get the x-axis position
    var xValue = DvtChartDataUtils.getXVal(chart, seriesIndex, groupIndex);
    var yValue = DvtChartDataUtils.getVal(chart, seriesIndex, groupIndex);

    // Get the position of the marker
    var xCoord = xAxis.getCoordAt(xValue);
    var yCoord = yAxis.getCoordAt(yValue);
    if (xCoord == null || yCoord == null) return null;
    return new Point(xCoord, yCoord);
  },

  /**
   * Returns true if all the values of the series are negative.
   * @param {Chart} chart
   * @param {Number} seriesIndex
   * @return {boolean}
   */
  isSeriesNegative: (chart, seriesIndex) => {
    if (!chart.getOptionsCache().getFromCache('hasNegativeValues')) return false;

    var isSeriesNegative = chart
      .getOptionsCache()
      .getFromCachedMap('isSeriesNegative', seriesIndex);
    if (isSeriesNegative != null) return isSeriesNegative;
    var groupCount = DvtChartDataUtils.getGroupCount(chart);
    isSeriesNegative = true;
    for (var i = 0; i < groupCount; i++) {
      // Use first non zero value to set series type(negative or positive)
      var value = DvtChartDataUtils.getVal(chart, seriesIndex, i);
      if (value > 0) {
        isSeriesNegative = false;
        break;
      }
    }
    chart.getOptionsCache().putToCachedMap('isSeriesNegative', seriesIndex, isSeriesNegative);
    return isSeriesNegative;
  },

  /**
   * Returns an array containing the hierarchical groups  associated with each innermost group item.
   * @param {Chart} chart
   * @return {Array} An array containing the hierarchical groups  associated with each innermost group item in chart.
   */
  getGroupsDataForContext: (chart) => {
    var cacheKey = 'groupsDataArray';
    var groupsDataArray = chart.getOptionsCache().getFromCache(cacheKey);

    if (!groupsDataArray) {
      var rawOptions = chart.getRawOptions();
      groupsDataArray = DvtChartDataUtils._getNestedGroupsData(rawOptions['groups']);
      chart.getOptionsCache().putToCache(cacheKey, groupsDataArray);
    }

    return groupsDataArray;
  },

  getSeriesDataForContext: (chart, seriesIndex) => {
    var rawOptions = chart.getRawOptions();
    var seriesData = rawOptions['series'][seriesIndex];
    var clonedSeriesData = Object.assign({}, seriesData);
    clonedSeriesData.items = []; // Object.assign doesn't clone array properties
    for (var i = 0; i < seriesData.items.length; i++) {
      var item;
      var seriesItem = seriesData.items[i];
      if (typeof seriesItem === 'object') {
        item = Object.assign({}, seriesData.items[i]);
      } else {
        item = seriesItem;
      }
      if (item && typeof item === 'object') {
        delete item['_itemData'];
      }
      clonedSeriesData.items.push(item);
    }
    return clonedSeriesData;
  },

  /**
   * Returns a structure containing the hierarchical groups  associated with each innermost group item.
   * @param {Array} groups An array of chart groups
   * @return {Array} An array of objects containing the hierarchical groups associated with each innermost group items in groups.
   * @private
   */
  _getNestedGroupsData: (groups) => {
    if (!groups) return [];
    var groupsDataArray = [];

    for (var i = 0; i < groups.length; i++) {
      var group = groups[i];

      if (group['groups']) {
        var innerGroupData = DvtChartDataUtils._getNestedGroupsData(group['groups']);
        for (var j = 0; j < innerGroupData.length; j++) {
          innerGroupData[j].unshift(group);
        }
        groupsDataArray = groupsDataArray.concat(innerGroupData);
      } else groupsDataArray.push([group]);
    }
    return groupsDataArray;
  },

  /**
   * Populates a cache map indicating if a data item is the outermost bar. Null items are ignored.
   * @param {Chart} chart
   * @private
   */
  _computeOutermostBarMap: (chart) => {
    if (!chart.getOptionsCache().getFromCache('outermostBar')) {
      var stackMap = {};
      var numSeries = DvtChartDataUtils.getSeriesCount(chart);
      var hasNegativeValues = chart.getOptionsCache().getFromCache('hasNegativeValues');

      // Iterate through the series and create an object maping stack category to a list of series.
      for (var seriesIndex = numSeries - 1; seriesIndex >= 0; seriesIndex--) {
        if (
          !DvtChartDataUtils.isSeriesRendered(chart, seriesIndex) ||
          DvtChartDataUtils.getSeriesType(chart, seriesIndex) != 'bar'
        )
          continue;
        var stackCategory = DvtChartDataUtils.getStackCategory(chart, seriesIndex) || '';
        var bAssignedToY2 = DvtChartDataUtils.isAssignedToY2(chart, seriesIndex);
        var stackKey = stackCategory + bAssignedToY2;
        if (stackMap[stackKey]) stackMap[stackKey].push(seriesIndex);
        else stackMap[stackKey] = [seriesIndex];
      }

      // For each stack category, iterate through each group and the series that are part of the stack.
      // If both the positive and negative outermost series are found then we can exit the series loop.
      var numGroups = DvtChartDataUtils.getGroupCount(chart);
      for (var key in stackMap) {
        var seriesList = stackMap[key];
        for (var groupIndex = 0; groupIndex < numGroups; groupIndex++) {
          var hasPositiveOutermost = false;
          var hasNegativeOutermost = false;
          for (var seriesListIndex = 0; seriesListIndex < seriesList.length; seriesListIndex++) {
            if (
              (!hasNegativeValues && hasPositiveOutermost) ||
              (hasPositiveOutermost && hasNegativeOutermost)
            )
              break;

            seriesIndex = seriesList[seriesListIndex];
            if (!DvtChartDataUtils.isDataItemRendered(chart, seriesIndex, groupIndex)) continue;

            var value = DvtChartDataUtils.getVal(chart, seriesIndex, groupIndex);
            if (value == null) {
              chart
                .getOptionsCache()
                .putToCachedMap2D('outermostBar', seriesIndex, groupIndex, false);
            } else if (!hasPositiveOutermost && value >= 0) {
              hasPositiveOutermost = true;
              chart
                .getOptionsCache()
                .putToCachedMap2D('outermostBar', seriesIndex, groupIndex, true);
            } else if (!hasNegativeOutermost && value < 0) {
              hasNegativeOutermost = true;
              chart
                .getOptionsCache()
                .putToCachedMap2D('outermostBar', seriesIndex, groupIndex, true);
            }
          }
        }
      }
    }
  },

  /**
   * Whether or not a bar is the outermost bar in its group or category.
   * @param {Chart} chart
   * @param {Number} seriesIndex The series index.
   * @param {Number} groupIndex The group index.
   * @return {boolean}  true if the bar is the outermost bar in its group or category, otherwise false.
   */
  isOutermostBar: (chart, seriesIndex, groupIndex) => {
    DvtChartDataUtils._computeOutermostBarMap(chart);
    return (
      chart.getOptionsCache().getFromCachedMap2D('outermostBar', seriesIndex, groupIndex) || false
    );
  },

  /**
   * Returns whether the data item is filtered.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @return {boolean}
   */
  isDataItemFiltered: (chart, seriesIndex, groupIndex) => {
    var dataItem = DvtChartDataUtils.getDataItem(chart, seriesIndex, groupIndex);
    if (dataItem && dataItem['_filtered']) return true;
    return false;
  },

  /**
   * Returns true if the chart series should be stacked.
   * @param {Chart} chart
   * @return {boolean}
   */
  isStacked: (chart) => {
    // To be stacked, the attribute must be set and the chart must be a supporting type.
    // If the series count is less than 2, assume unstacked because it's faster to render.
    var options = chart.getOptions();
    if (
      options['stack'] != 'on' ||
      DvtChartTypeUtils.isMixedFrequency(chart) ||
      DvtChartDataUtils.getSeriesCount(chart) < 2
    )
      return false;

    return DvtChartTypeUtils.isBLAC(chart);
  },

  /**
   * Returns true if the chart is split dual-Y.
   * @param {Chart} chart
   * @return {boolean}
   */
  isSplitDualY: (chart) => {
    if (
      DvtChartTypeUtils.isStock(chart) &&
      DvtChartDataUtils.hasVolumeSeries(chart) &&
      !DvtChartTypeUtils.isOverview(chart)
    )
      return true;

    return (
      chart.getOptions()['splitDualY'] == 'on' &&
      DvtChartDataUtils.hasY2Data(chart) &&
      !DvtChartDataUtils.hasY2DataOnly(chart)
    );
  },

  /**
   * Returns true if the chart has y2 data items only.
   * @param {Chart} chart
   * @return {boolean}
   */
  hasY2DataOnly: (chart) => {
    if (!DvtChartTypeUtils.isDualY(chart)) return false;

    // Verify that all the series are y2
    return (
      DvtChartDataUtils.getY2SeriesCount(chart, null, true) ==
      DvtChartDataUtils.getSeriesCount(chart)
    );
  },

  /**
   * Returns true if the chart has y2 data items.
   * @param {Chart} chart
   * @param {string} type Optional series type to look for.
   * @return {boolean}
   */
  hasY2Data: (chart, type) => {
    if (!DvtChartTypeUtils.isDualY(chart)) return false;

    // Verify the chart has at least one y2 series
    return DvtChartDataUtils.getY2SeriesCount(chart, null, true) > 0;
  },

  /**
   * Returns true if the chart has y2 data items.
   * @param {Chart} chart
   * @return {boolean}
   */
  hasY2BarData: (chart) => {
    return DvtChartDataUtils.hasY2Data(chart, 'bar');
  },

  /**
   * @param {Chart} chart
   * @return {boolean} true if one of the series type is bar.
   */
  hasBarSeries: (chart) => {
    return DvtChartDataUtils._hasSeriesType(chart, 'bar');
  },

  /**
   * @param {Chart} chart
   * @return {boolean} true if one of the series type is line.
   */
  hasLineSeries: (chart) => {
    return DvtChartDataUtils._hasSeriesType(chart, 'line');
  },

  /**
   * @param {Chart} chart
   * @return {boolean} true if one of the series type is area.
   */
  hasAreaSeries: (chart) => {
    return DvtChartDataUtils._hasSeriesType(chart, 'area');
  },

  /**
   * @param {Chart} chart
   * @return {boolean} true if one of the series type is lineWithArea.
   */
  hasLineWithAreaSeries: (chart) => {
    return DvtChartDataUtils._hasSeriesType(chart, 'lineWithArea');
  },

  /**
   * Returns true if one of the series type is stock.
   * @param {Chart} chart
   * @return {boolean}
   */
  hasCandlestickSeries: (chart) => {
    return DvtChartDataUtils._hasSeriesType(chart, 'candlestick');
  },

  /**
   * Returns true if one of the series type is boxPlot.
   * @param {Chart} chart
   * @return {boolean}
   */
  hasBoxPlotSeries: (chart) => {
    return DvtChartDataUtils._hasSeriesType(chart, 'boxPlot');
  },

  /**
   * @param {Chart} chart
   * @param {string} type The series type.
   * @return {boolean} True if one of the series is the specified series type. Only works for BLAC.
   * @private
   */
  _hasSeriesType: (chart, type) => {
    if (DvtChartTypeUtils.isBLAC(chart)) {
      var seriesCount = DvtChartDataUtils.getSeriesCount(chart);

      for (var seriesIndex = 0; seriesIndex < seriesCount; seriesIndex++) {
        // Ignore the series if it isn't rendered
        if (!DvtChartDataUtils.isSeriesRendered(chart, seriesIndex)) continue;
        else if (DvtChartDataUtils.getSeriesType(chart, seriesIndex) == type) return true;
      }
    }
    return false;
  },

  /**
   * Returns all reference objects for the current chart.
   * @param {Chart} chart
   * @return {array} The array of reference object definitions.
   */
  getRefObjs: (chart) => {
    var x = DvtChartDataUtils.getAxisRefObjs(chart, 'x');
    var y = DvtChartDataUtils.getAxisRefObjs(chart, 'y');
    var y2 = DvtChartDataUtils.getAxisRefObjs(chart, 'y2');
    return x.concat(y, y2);
  },

  /**
   * Returns all reference objects for the axis.
   * @param {Chart} chart
   * @param {string} axisType 'x', 'y', 'or 'y2'
   * @return {array} The array of reference object definitions.
   */
  getAxisRefObjs: (chart, axisType) => {
    var options = chart.getOptions();
    if (options && options[axisType + 'Axis'] && options[axisType + 'Axis']['referenceObjects'])
      return options[axisType + 'Axis']['referenceObjects'];
    return [];
  },

  /**
   * Returns the series type for the specified data item.  Returns "auto" for chart types
   * that do not support multiple series types.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @return {string} The series type.
   */
  getSeriesType: (chart, seriesIndex) => {
    var cacheKey = 'seriesType';
    var seriesType = chart.getOptionsCache().getFromCachedMap(cacheKey, seriesIndex);
    if (seriesType) {
      return seriesType;
    }

    if (!DvtChartTypeUtils.isBLAC(chart)) {
      chart.getOptionsCache().putToCachedMap(cacheKey, seriesIndex, 'auto');
      return 'auto';
    }

    var seriesItem = DvtChartDataUtils.getSeriesItem(chart, seriesIndex);
    seriesType = seriesItem ? seriesItem['type'] : null;

    // Error prevention for candlestick series in non stock type charts
    if (!DvtChartTypeUtils.isStock(chart) && seriesType == 'candlestick') seriesType = 'auto';

    if (!seriesType || seriesType == 'auto') {
      // Series type not specified, get default
      if (DvtChartTypeUtils.isBar(chart)) seriesType = 'bar';
      else if (DvtChartTypeUtils.isLine(chart)) seriesType = 'line';
      else if (DvtChartTypeUtils.isArea(chart)) seriesType = 'area';
      else if (DvtChartTypeUtils.isLineWithArea(chart)) seriesType = 'lineWithArea';
      else if (DvtChartTypeUtils.isStock(chart)) seriesType = 'candlestick';
      else if (DvtChartTypeUtils.isBoxPlot(chart)) seriesType = 'boxPlot';
      else if (DvtChartTypeUtils.isCombo(chart)) {
        var styleIndex = DvtChartDataUtils.getSeriesStyleIdx(chart, seriesIndex);
        var typeIndex = styleIndex % DvtChartDataUtils._SERIES_TYPE_RAMP.length;
        seriesType = DvtChartDataUtils._SERIES_TYPE_RAMP[typeIndex];
      }
    }

    chart.getOptionsCache().putToCachedMap(cacheKey, seriesIndex, seriesType);
    return seriesType;
  },

  /**
   * Returns the array containing the hidden categories for the chart.
   * @param {Chart} chart
   * @return {array}
   */
  getHiddenCategories: (chart) => {
    var options = chart.getOptions();
    if (!options['hiddenCategories']) options['hiddenCategories'] = [];

    return options['hiddenCategories'];
  },

  /**
   * Returns the array containing the highlighted categories for the chart.
   * @param {Chart} chart
   * @return {array}
   */
  getHighlightedCategories: (chart) => {
    var options = chart.getOptions();
    if (!options['highlightedCategories']) options['highlightedCategories'] = [];

    return options['highlightedCategories'];
  },

  /**
   * Returns true if the specified series should be rendered.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @return {boolean} True if the series should be rendered.
   */
  isSeriesRendered: (chart, seriesIndex) => {
    // Check if any category is hidden
    var hiddenCategories = DvtChartDataUtils.getHiddenCategories(chart);
    if (hiddenCategories.length > 0) {
      if (
        ArrayUtils.hasAnyItem(
          hiddenCategories,
          DvtChartDataUtils.getCategories(chart, seriesIndex)
        )
      ) {
        return false;
      }
    }

    return true;
  },

  /**
   * Returns true if the specified data item should be rendered.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @param {number} itemIndex
   * @return {boolean} True if the series should be rendered.
   */
  isDataItemRendered: (chart, seriesIndex, groupIndex, itemIndex) => {
    // Use the cached value if it has been computed before
    var ret;
    var cacheKey = 'isDataItemRendered';
    var isNested = !isNaN(itemIndex) && itemIndex != null && itemIndex >= 0;
    if (!isNested) {
      ret = chart.getOptionsCache().getFromCachedMap2D(cacheKey, seriesIndex, groupIndex);
      if (ret !== undefined)
        // anything that's defined, including null
        return ret;
    }

    ret = true;
    if (!DvtChartDataUtils.isSeriesRendered(chart, seriesIndex)) ret = false;
    else {
      // Check if any category is hidden
      var hiddenCategories = DvtChartDataUtils.getHiddenCategories(chart);
      if (hiddenCategories.length > 0) {
        if (
          DvtChartTypeUtils.isPie(chart) ||
          DvtChartTypeUtils.isFunnel(chart) ||
          DvtChartTypeUtils.isPyramid(chart)
        )
          groupIndex = 0;

        if (
          ArrayUtils.hasAnyItem(
            hiddenCategories,
            DvtChartDataUtils.getCategories(chart, seriesIndex, groupIndex)
          )
        ) {
          ret = false;
        }

        // nested item
        if (
          ArrayUtils.hasAnyItem(
            hiddenCategories,
            DvtChartDataUtils.getCategories(chart, seriesIndex, groupIndex, itemIndex)
          )
        ) {
          ret = false;
        }
      }
    }

    // Cache the value
    if (!isNested) chart.getOptionsCache().putToCachedMap2D(cacheKey, seriesIndex, groupIndex, ret);

    return ret;
  },

  /**
   * Returns whether the series is a range series.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @return {boolean}
   */
  isRangeSeries: (chart, seriesIndex) => {
    var optionsCache = chart.getOptionsCache();
    if (!optionsCache.getFromCache('hasLowHighSeries')) return false;

    // Use the cached value if it has been computed before
    var cacheKey = 'isRange';
    var isRange = optionsCache.getFromCachedMap(cacheKey, seriesIndex);
    if (isRange != null) return isRange;

    isRange = false;
    var seriesType = DvtChartDataUtils.getSeriesType(chart, seriesIndex);
    if (seriesType == 'bar' || seriesType == 'area') {
      for (var g = 0; g < DvtChartDataUtils.getGroupCount(chart); g++) {
        if (
          DvtChartDataUtils.getLowVal(chart, seriesIndex, g) != null ||
          DvtChartDataUtils.getHighVal(chart, seriesIndex, g) != null
        ) {
          isRange = true;
          break;
        }
      }
    }

    // Cache the value
    chart.getOptionsCache().putToCachedMap(cacheKey, seriesIndex, isRange);
    return isRange;
  },

  /**
   * Returns whether the series is drillable.
   * @param {Chart} chart
   * @param {Number} seriesIndex
   * @return {Boolean}
   */
  isSeriesDrillable: (chart, seriesIndex) => {
    var series = DvtChartDataUtils.getSeriesItem(chart, seriesIndex);
    var drilling = series != null ? series['drilling'] : 'inherit';
    if (drilling == 'on') return true;
    else if (drilling == 'off') return false;

    drilling = chart.getOptions()['drilling'];
    return drilling == 'on' || drilling == 'seriesOnly';
  },

  /**
   * Returns whether the data item is drillable.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @param {number} itemIndex
   * @return {Boolean}
   */
  isDataItemDrillable: (chart, seriesIndex, groupIndex, itemIndex) => {
    var dataItem = DvtChartDataUtils.getNestedDataItem(chart, seriesIndex, groupIndex, itemIndex);
    if (!dataItem) dataItem = DvtChartDataUtils.getDataItem(chart, seriesIndex, groupIndex);

    var drilling = dataItem != null ? dataItem['drilling'] : 'inherit';
    if (drilling == 'on') return true;
    else if (drilling == 'off') return false;

    drilling = chart.getOptions()['drilling'];
    return drilling == 'on';
  },

  /**
   * Returns weather multiseries drill is enabled in the chart
   * @param {Chart} chart The pie chart
   */
  isMultiSeriesDrillEnabled: (chart) => {
    return chart.getOptions().multiSeriesDrilling === 'on';
  }
};

/**
 * Default values and utility functions for component versioning.
 * @class
 * @constructor
 * @param {dvt.Context} context The rendering context.
 * @extends {dvt.BaseComponentDefaults}
 */
class DvtChartDefaults extends BaseComponentDefaults {
  constructor(context) {
    /**
     * Defaults for version 1.
     * @const
     */
    const SKIN_ALTA = {
      skin: CSSStyle.SKIN_ALTA,
      emptyText: null,
      type: 'bar',
      stack: 'off',
      stackLabel: 'off',
      orientation: 'vertical',
      polarGridShape: 'circle',
      selectionMode: 'none',
      hideAndShowBehavior: 'none',
      hoverBehavior: 'none',
      zoomAndScroll: 'off',
      zoomDirection: 'auto',
      initialZooming: 'none',
      dragMode: 'user',
      sorting: 'off',
      otherThreshold: 0,
      animationOnDataChange: 'none',
      animationOnDisplay: 'none',
      __sparkBarSpacing: 'subpixel',
      __spark: false,
      dataCursor: 'auto',
      dataCursorBehavior: 'auto',
      drilling: 'off',
      highlightMatch: 'all',
      series: [],
      groups: [],
      title: {
        style: new CSSStyle(
          BaseComponentDefaults.FONT_FAMILY_ALTA_BOLD_13 + 'color: #252525;'
        ),
        halign: 'start'
      },
      subtitle: {
        style: new CSSStyle(BaseComponentDefaults.FONT_FAMILY_ALTA_12 + 'color: #252525;')
      },
      footnote: {
        style: new CSSStyle(BaseComponentDefaults.FONT_FAMILY_ALTA_11 + 'color: #333333;'),
        halign: 'start'
      },
      titleSeparator: { upperColor: '#74779A', lowerColor: '#FFFFFF', rendered: 'off' },
      touchResponse: 'auto',
      _statusMessageStyle: new CSSStyle(
        BaseComponentDefaults.FONT_FAMILY_ALTA_13 + 'color: #252525;'
      ),
      _dropColor: '#D9F4FA',

      xAxis: {
        tickLabel: { rendered: 'on' },
        majorTick: { rendered: 'auto' },
        minorTick: { rendered: 'auto' },
        axisLine: { rendered: 'on' },
        scale: 'linear'
      },
      yAxis: {
        tickLabel: { rendered: 'on' },
        majorTick: { rendered: 'auto' },
        minorTick: { rendered: 'auto' },
        axisLine: { rendered: 'auto' },
        scale: 'linear'
      },
      y2Axis: {
        tickLabel: { rendered: 'on' },
        majorTick: { rendered: 'auto' },
        minorTick: { rendered: 'auto' },
        axisLine: { rendered: 'auto' },
        scale: 'linear',
        alignTickMarks: 'on'
      },
      pieCenter: { labelStyle: new CSSStyle('') },
      zAxis: {}, // this will be used for dataMin/Max calculations
      plotArea: { backgroundColor: null },

      legend: {
        position: 'auto',
        rendered: 'auto',
        layout: { gapRatio: 1.0 },
        seriesSection: {},
        referenceObjectSection: {},
        sections: []
      },

      overview: {
        rendered: 'off'
      },

      dnd: {
        drag: {
          items: {},
          series: {},
          groups: {}
        },
        drop: {
          plotArea: {},
          xAxis: {},
          yAxis: {},
          y2Axis: {},
          legend: {}
        }
      },

      styleDefaults: {
        colors: CSSStyle.COLORS_ALTA,
        borderColor: 'auto',
        borderWidth: 'auto',
        patterns: [
          'smallDiagonalRight',
          'smallChecker',
          'smallDiagonalLeft',
          'smallTriangle',
          'smallCrosshatch',
          'smallDiamond',
          'largeDiagonalRight',
          'largeChecker',
          'largeDiagonalLeft',
          'largeTriangle',
          'largeCrosshatch',
          'largeDiamond'
        ],
        shapes: ['square', 'circle', 'diamond', 'plus', 'triangleDown', 'triangleUp'],
        seriesEffect: 'color',
        threeDEffect: 'off',
        selectionEffect: 'highlight',
        animationDuration: 1000,
        animationIndicators: 'all',
        animationUpColor: '',
        animationDownColor: '',
        lineStyle: 'solid',
        lineType: 'auto',
        markerDisplayed: 'auto',
        markerColor: null,
        markerShape: 'auto',
        markerSize: 10,
        marqueeColor: '',
        marqueeBorderColor: '',
        pieFeelerColor: '#BAC5D6',
        pieInnerRadius: 0,
        selectedInnerColor: '#ffffff',
        selectedOuterColor: '#5a5a5a',
        sliceLabelType: 'percent',
        otherColor: '#4b4b4b',
        dataItemGaps: 'auto',
        dataLabelStyle: new CSSStyle(''),
        _dataLabelStyle: new CSSStyle(''),
        dataLabelPosition: 'auto',
        funnelBackgroundColor: '#EDEDED',
        x1Format: {},
        y1Format: {},
        y2Format: {},
        zFormat: {},
        _defaultSliceLabelColor: '#333333',
        _scrollbarHeight: 3,
        _scrollbarTrackColor: '#F0F0F0',
        _scrollbarHandleColor: '#9E9E9E',
        hoverBehaviorDelay: 200,
        dataCursor: { markerSize: 8, markerDisplayed: 'on', lineStyle: 'solid' },
        groupSeparators: { rendered: 'on', color: 'rgba(138,141,172,0.4)' },
        tooltipLabelStyle: new CSSStyle(''),
        tooltipValueStyle: new CSSStyle(''),
        stackLabelStyle: new CSSStyle(''),
        boxPlot: {
          whiskerSvgStyle: {},
          whiskerEndSvgStyle: { strokeWidth: 2 },
          whiskerEndLength: '9px',
          medianSvgStyle: { strokeWidth: 3 }
        }
      },

      layout: {
        gapWidthRatio: null,
        gapHeightRatio: null, // gap ratio is dynamic based on the component size
        // TODO, the following are internal and should be moved to a _layout object
        outerGapWidth: 10,
        outerGapHeight: 8,
        titleSubtitleGapWidth: 14,
        titleSubtitleGapHeight: 4,
        titleSeparatorGap: 6,
        titlePlotAreaGap: 16,
        footnoteGap: 10,
        verticalAxisGap: 6,
        legendGapWidth: 15,
        legendGapHeight: 10,
        tickLabelGapHeight: 8,
        tickLabelGapWidth: 9
      },

      _locale: 'en-us',
      _resources: {}
    };
    super({ alta: SKIN_ALTA }, context);
  }

  /**
   * Scales down gap widths based on the width of the component.
   * @param {Chart} chart The chart that is being rendered.
   * @param {Number} defaultWidth The default gap width.
   * @return {Number}
   */
  static getGapWidth(chart, defaultWidth) {
    return Math.ceil(defaultWidth * chart.getGapWidthRatio());
  }

  /**
   * Scales down gap heights based on the height of the component.
   * @param {Chart} chart The chart that is being rendered.
   * @param {Number} defaultHeight The default gap height.
   * @return {Number}
   */
  static getGapHeight(chart, defaultHeight) {
    return Math.ceil(defaultHeight * chart.getGapHeightRatio());
  }

  /**
   * @override
   */
  getNoCloneObject() {
    return {
      series: { items: { _itemData: true } },
      data: true,
      valueFormats: {
        close: { converter: true },
        high: { converter: true },
        label: { converter: true },
        low: { converter: true },
        open: { converter: true },
        q1: { converter: true },
        q2: { converter: true },
        q3: { converter: true },
        targetValue: { converter: true },
        value: { converter: true },
        volume: { converter: true },
        x: { converter: true },
        y: { converter: true },
        y2: { converter: true },
        z: { converter: true }
      },
      pieCenter: { converter: true },
      xAxis: {
        tickLabel: { converter: true }
      },
      yAxis: {
        tickLabel: { converter: true }
      },
      y2Axis: {
        tickLabel: { converter: true }
      }
    };
  }
}

/**
 * Utility functions for pie chart.
 * @class
 */
const DvtChartPieUtils = {
  OTHER_ID: '_dvtOther',

  /**
   * Generates the slice ID of a pie series
   * @param {Chart} chart The pie chart
   * @param {Number} seriesIndex The series index
   * @return {DvtChartDataItem} The slice ID
   */
  getSliceId: (chart, seriesIndex) => {
    var dataItem = DvtChartDataUtils.getDataItem(chart, seriesIndex, 0);
    var id = dataItem ? dataItem['id'] : null;
    var series = DvtChartDataUtils.getSeries(chart, seriesIndex);
    var group = DvtChartDataUtils.getGroup(chart, 0);
    return new DvtChartDataItem(id, series, group, chart.getCtx());
  },

  /**
   * Generates the slice ID of a pie "Other" series
   * @param {Chart} chart The pie chart
   * @return {DvtChartDataItem} The slice ID
   */
  getOtherSliceId: (chart) => {
    var group = DvtChartDataUtils.getGroup(chart, 0);
    return new DvtChartDataItem(null, DvtChartPieUtils.OTHER_ID, group, chart.getCtx());
  },

  /**
   * Returns an array of series indices that will be rendered on the pie chart.
   * The array is sorted if sorting is enabled, and does not include the series that will be grouped under "Other" slice.
   * The array includes hidden series and series with non-positive values.
   * @param {Chart} chart The pie chart
   * @return {Array} The array containing series indices
   */
  getRenderedSeriesIndices: (chart) => {
    return DvtChartPieUtils._getSeriesIndicesArrays(chart).rendered;
  },

  /**
   * Returns whether the pie has at least one series (visible or hidden) that is grouped under "Other".
   * @param {Chart} chart The pie chart
   * @return {Boolean}
   */
  hasOtherSeries: (chart) => {
    return DvtChartPieUtils._getSeriesIndicesArrays(chart).other.length > 0;
  },

  /**
   * Computes the total value of the "Other" slice. Only includes visible series with positive values.
   * @param {Chart} chart The pie chart
   * @return {Number} The total value
   */
  getOtherVal: (chart) => {
    var otherSeries = DvtChartPieUtils._getSeriesIndicesArrays(chart).other;
    var otherValue = 0;
    for (var i = 0; i < otherSeries.length; i++) {
      var seriesIndex = otherSeries[i];
      // Only add the values of visible series
      if (DvtChartDataUtils.isSeriesRendered(chart, seriesIndex)) {
        var value = DvtChartDataUtils.getVal(chart, seriesIndex, 0);
        if (value > 0) otherValue += value;
      }
    }
    return otherValue;
  },

  /**
   * Generates the slice IDs of the series that are grouped under "Other".
   * @param {Chart} chart The pie chart
   * @return {Array} The array containing slice IDs
   */
  getOtherSliceIds: (chart) => {
    var otherSeries = DvtChartPieUtils._getSeriesIndicesArrays(chart).other;
    var seriesIds = [];
    for (var i = 0; i < otherSeries.length; i++) {
      var seriesIndex = otherSeries[i];
      seriesIds.push(DvtChartPieUtils.getSliceId(chart, seriesIndex));
    }
    return seriesIds;
  },

  /**
   * Returns whether the "Other" slice is selected. It is selected if all the series in it are selected.
   * @param {Chart} chart The pie chart
   * @param {Array} selected An array containing the ID objects of the selected slices.
   * @return {Boolean} Whether the "Other" slice is selected
   */
  isOtherSliceSelected: (chart, selected) => {
    var otherIds = DvtChartPieUtils.getOtherSliceIds(chart);

    for (var j = 0; j < otherIds.length; j++) {
      var sliceId = otherIds[j];
      var sliceSelected = false;

      // Check if this slice is in the selected list
      for (var i = 0; i < selected.length; i++) {
        if (
          (selected[i]['id'] != null && sliceId.id === selected[i]['id']) ||
          (sliceId.series === selected[i]['series'] && sliceId.group === selected[i]['group'])
        ) {
          sliceSelected = true;
          break;
        }
      }

      if (!sliceSelected) return false;
    }

    return true;
  },

  /**
   * Divides the series indices into two arrays. The first array contains the series that are not grouped under "Other",
   * sorted by value if sorting is enabled. The second array contains the series that belongs to "Other". The arrays
   * include hidden series and series with non-positive values.
   * @param {Chart} chart The pie chart
   * @return {Object} An object in the form {rendered: firstArray, other: secondArray}. firstArray and secondArray are
   *     as described above.
   * @private
   */
  _getSeriesIndicesArrays: (chart) => {
    var renderedSeries = [];
    var otherSeries = [];

    var seriesCount = DvtChartDataUtils.getSeriesCount(chart);
    var options = chart.getOptions();
    var otherThreshold = options['otherThreshold'] * DvtChartPieUtils.getTotalVal(chart);

    for (var seriesIndex = 0; seriesIndex < seriesCount; seriesIndex++) {
      // Skip the series if its value is 0 or negative
      var value = DvtChartDataUtils.getVal(chart, seriesIndex, 0);

      // Do not use "Other" if the threshold is zero
      if (otherThreshold > 0 && value < otherThreshold) otherSeries.push(seriesIndex);
      else renderedSeries.push(seriesIndex);
    }

    // Sort the slices if enabled
    if (options['sorting'] == 'ascending') {
      renderedSeries.sort((a, b) => {
        return DvtChartDataUtils.getVal(chart, a, 0) - DvtChartDataUtils.getVal(chart, b, 0);
      });
    } else if (options['sorting'] == 'on' || options['sorting'] == 'descending') {
      renderedSeries.sort((a, b) => {
        return DvtChartDataUtils.getVal(chart, b, 0) - DvtChartDataUtils.getVal(chart, a, 0);
      });
    }

    return { rendered: renderedSeries, other: otherSeries };
  },

  /**
   * Computes the total value of a pie chart, including hidden series.
   * @param {Chart} chart The pie chart.
   */
  getTotalVal: (chart) => {
    var seriesCount = DvtChartDataUtils.getSeriesCount(chart);
    var totalValue = 0;

    for (var seriesIndex = 0; seriesIndex < seriesCount; seriesIndex++) {
      // Skip the series if its value is 0 or negative
      var value = DvtChartDataUtils.getVal(chart, seriesIndex, 0);
      if (value > 0) {
        totalValue += value;
      }
    }

    return totalValue;
  },

  /**
   * Returns the pie slice explode for the specified series.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @return {number} The pie slice explode from 0 to 100.
   */
  getSliceExplode: (chart, seriesIndex) => {
    // Series Override
    var seriesItem = DvtChartDataUtils.getSeriesItem(chart, seriesIndex);
    if (seriesItem && seriesItem['pieSliceExplode']) return seriesItem['pieSliceExplode'];
    else return 0;
  },
  /**
   * Returns the slice corresponding to the seriesIndex or null.
   * If seriesIndex is null or seriesIndex is part of "other", the "other" slice is returned
   * @param {Chart} chart
   * @param {Number} seriesIndex
   * @return {DvtChartPieSlice} slice
   */
  getSliceBySeriesIdx: (chart, seriesIndex) => {
    var slices = chart.pieChart.__getSlices();
    for (var i = 0; i < slices.length; i++) {
      if (slices[i].getSeriesIndex() == seriesIndex) return slices[i];
    }
    return null;
  }
};

/**
 * Style related utility functions for Chart.
 * @class
 */
const DvtChartStyleUtils = {
  /** @const */
  MARKER_DATA_LABEL_GAP: 4, // space separating the data label from the marker
  SERIES_PATTERN_BG_COLOR: '#FFFFFF',
  /**
   * Returns the series effect for the specified chart.
   * @param {Chart} chart
   * @return {string} The series effect.
   */
  getSeriesEffect: (chart) => {
    // Style Defaults
    var options = chart.getOptions();
    return options['styleDefaults']['seriesEffect'];
  },

  /**
   * Returns the color for the specified data item.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @return {string} The color string.
   */
  getColor: (chart, seriesIndex, groupIndex) => {
    // Data Override
    var dataItem = DvtChartDataUtils.getDataItem(chart, seriesIndex, groupIndex);
    if (dataItem && dataItem['color']) return dataItem['color'];

    // Series Override
    var seriesItem = DvtChartDataUtils.getSeriesItem(chart, seriesIndex);
    if (seriesItem && seriesItem['color']) return seriesItem['color'];

    // Stock Candlestick: Use rising/falling instead of series color or styleDefaults colors.
    var seriesType = DvtChartDataUtils.getSeriesType(chart, seriesIndex);
    if (seriesType == 'candlestick')
      return DvtChartStyleUtils.getStockItemColor(chart, seriesIndex, groupIndex);

    // Style Defaults
    var options = chart.getOptions();
    var defaultColors = options['styleDefaults']['colors'];
    var styleIndex = DvtChartDataUtils.getSeriesStyleIdx(chart, seriesIndex);
    var colorIndex = styleIndex % defaultColors.length;
    return defaultColors[colorIndex];
  },
  /**
   * Returns the stock item color for the specified data item for stock charts.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @return {string} The color string.
   */
  getStockItemColor: (chart, seriesIndex, groupIndex) => {
    var options = chart.getOptions();
    if (DvtChartDataUtils.isStockValRising(chart, seriesIndex, groupIndex))
      return options['styleDefaults']['stockRisingColor'];
    else return options['styleDefaults']['stockFallingColor'];
  },
  /**
   * Returns the color for the specified volume item for stock charts.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @return {string} The color string.
   */
  getStockVolumeColor: (chart, seriesIndex, groupIndex) => {
    // Data Override
    var dataItem = DvtChartDataUtils.getDataItem(chart, seriesIndex, groupIndex);
    if (dataItem && dataItem['color']) return dataItem['color'];

    // Volume Color
    var options = chart.getOptions();
    if (options['styleDefaults']['stockVolumeColor'])
      return options['styleDefaults']['stockVolumeColor'];

    // Rising/Falling Color
    return DvtChartStyleUtils.getStockItemColor(chart, seriesIndex, groupIndex);
  },
  /**
   * Returns the splitterPosition to be used as a number from 0 to 1.
   * @param {Chart} chart
   * @return {number}
   */
  getSplitterPos: (chart) => {
    var options = chart.getOptions();

    var splitterPosition = options['splitterPosition'];
    if (splitterPosition != null) return splitterPosition;
    else if (DvtChartTypeUtils.isStock(chart)) return 0.8;
    else return 0.5;
  },
  /**
   * Returns the pattern for the specified data item.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @param {number} itemIndex
   * @return {string} The pattern string.
   */
  getPattern: (chart, seriesIndex, groupIndex, itemIndex) => {
    // Nested Data Override
    var nestedDataItem = DvtChartDataUtils.getNestedDataItem(
      chart,
      seriesIndex,
      groupIndex,
      itemIndex
    );
    if (nestedDataItem && nestedDataItem['pattern'] && nestedDataItem['pattern'] != 'auto')
      return nestedDataItem['pattern'];

    // Data Override
    var dataItem = DvtChartDataUtils.getDataItem(chart, seriesIndex, groupIndex);
    if (dataItem && dataItem['pattern'] && dataItem['pattern'] != 'auto')
      return dataItem['pattern'];

    //get series type instead of chart type, in case its a combo chart
    var seriesType = DvtChartDataUtils.getSeriesType(chart, seriesIndex);
    //prevent line/area markers from using series/styleDefaults pattern
    if ((seriesType == 'line' || seriesType == 'area') && groupIndex != null) return null;

    // Series Override
    var seriesItem = DvtChartDataUtils.getSeriesItem(chart, seriesIndex);
    if (seriesItem && seriesItem['pattern'] && seriesItem['pattern'] != 'auto')
      return seriesItem['pattern'];

    // Style Defaults
    if (DvtChartStyleUtils.getSeriesEffect(chart) == 'pattern') {
      // For candlestick series, use pattern based on rising/falling value.
      if (
        DvtChartTypeUtils.isStock &&
        DvtChartDataUtils.getSeriesType(chart, seriesIndex) == 'candlestick'
      ) {
        var bRisingValue = DvtChartDataUtils.isStockValRising(chart, seriesIndex, groupIndex);
        var bRtl = Agent.isRightToLeft(chart.getCtx());
        if (bRisingValue) return bRtl ? 'smallDiagonalLeft' : 'smallDiagonalRight';
        // Falling Value
        else return bRtl ? 'smallDiagonalRight' : 'smallDiagonalLeft';
      } else {
        var options = chart.getOptions();
        var defaultPatterns = options['styleDefaults']['patterns'];
        var styleIndex = DvtChartDataUtils.getSeriesStyleIdx(chart, seriesIndex);
        var patternIndex = styleIndex % defaultPatterns.length;
        return defaultPatterns[patternIndex];
      }
    } else return null;
  },
  /**
   * Returns the border color for markers belonging to the specified data item.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @param {number} itemIndex
   * @return {string} The border color string.
   */
  getMarkerBorderColor: (chart, seriesIndex, groupIndex, itemIndex) => {
    // Return custom border color from API settings if found.
    var borderColor = DvtChartStyleUtils.getBorderColor(chart, seriesIndex, groupIndex, itemIndex);
    if (borderColor) return borderColor;

    return DvtChartStyleUtils.getDefaultMarkerBorderColor(
      chart,
      seriesIndex,
      groupIndex,
      itemIndex
    );
  },
  /**
   * Returns the default marker color for the marker
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @param {number} itemIndex
   */
  getDefaultMarkerBorderColor: (chart, seriesIndex, groupIndex, itemIndex) => {
    if (
      chart.getCtx().getThemeBehavior() === 'redwood' &&
      !DvtChartDataUtils.isRangeSeries(chart, seriesIndex)
    ) {
      return DvtChartStyleUtils.getMarkerColor(chart, seriesIndex, groupIndex, itemIndex);
    }
    // If data item gaps defined, use the background color to simulate gaps.
    if (
      DvtChartStyleUtils.getDataItemGaps(chart) > 0 &&
      DvtChartDataUtils.getSeriesType(chart, seriesIndex) != 'lineWithArea'
    ) {
      return DvtChartStyleUtils.getBackgroundColor(chart, true);
    }
    //  - In alta, automatically apply borders to bubbles in bubble charts using the 'color' seriesEffect for better readability
    if (
      DvtChartTypeUtils.isBubble(chart) &&
      DvtChartStyleUtils.getSeriesEffect(chart) != 'gradient'
    ) {
      var markerColor = DvtChartStyleUtils.getMarkerColor(chart, seriesIndex, groupIndex);
      if (markerColor) return ColorUtils.adjustHSL(markerColor, 0, 0.15, -0.25);
    }
    return null;
  },

  /**
   * Returns the border color for the specified data item.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @param {number} itemIndex
   * @return {string} The border color string.
   */
  getBorderColor: (chart, seriesIndex, groupIndex, itemIndex) => {
    // Nested Data Override
    var nestedDataItem = DvtChartDataUtils.getNestedDataItem(
      chart,
      seriesIndex,
      groupIndex,
      itemIndex
    );
    if (nestedDataItem && nestedDataItem['borderColor']) return nestedDataItem['borderColor'];

    // Data Override
    var dataItem = DvtChartDataUtils.getDataItem(chart, seriesIndex, groupIndex);
    if (dataItem && dataItem['borderColor']) return dataItem['borderColor'];

    // Series Override
    var seriesItem = DvtChartDataUtils.getSeriesItem(chart, seriesIndex);
    if (seriesItem && seriesItem['borderColor']) return seriesItem['borderColor'];

    // Style Defaults
    var options = chart.getOptions();
    var styleDefaults = options['styleDefaults'];
    return styleDefaults['borderColor'] != 'auto' ? styleDefaults['borderColor'] : null;
  },
  /**
   * Returns the border color for the specified data item if specified in the api.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @param {number} itemIndex
   * @return {number} The border width.
   */
  getUserBorderWidth: (chart, seriesIndex, groupIndex, itemIndex) => {
    // Nested Data Override
    var nestedDataItem = DvtChartDataUtils.getNestedDataItem(
      chart,
      seriesIndex,
      groupIndex,
      itemIndex
    );
    if (nestedDataItem && nestedDataItem['borderWidth'] != null)
      return nestedDataItem['borderWidth'];

    // Data Override
    var dataItem = DvtChartDataUtils.getDataItem(chart, seriesIndex, groupIndex);
    if (dataItem && dataItem['borderWidth'] != null) return dataItem['borderWidth'];

    // Series Override
    var seriesItem = DvtChartDataUtils.getSeriesItem(chart, seriesIndex);
    if (seriesItem && seriesItem['borderWidth'] != null) return seriesItem['borderWidth'];

    // Style Defaults
    var styleDefaults = chart.getOptions()['styleDefaults'];
    if (styleDefaults['borderWidth'] != 'auto') return styleDefaults['borderWidth'];

    return null;
  },
  /**
   * Returns the default border width for chart items.
   * @param {Chart} chart
   */
  getDefaultBorderWidth: (chart) => {
    // The borderWidth is reduced for scatter/bubble because it looks hairy otherwise
    return DvtChartTypeUtils.isScatterBubble(chart) || DvtChartTypeUtils.isLineArea(chart)
      ? 1.25
      : 1;
  },
  /**
   * Returns the default border width for chart markers.
   * @param {Chart} chart
   */
  getDefaultMarkerBorderWidth: (chart) => {
    // override border-width in css
    if (chart.getCtx().getThemeBehavior() === 'redwood') {
      return 0;
    }
    return DvtChartStyleUtils.getDefaultBorderWidth(chart);
  },
  /**
   * Returns the border color for the specified data item.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @param {number} itemIndex
   * @return {number} The border width.
   */
  getBorderWidth: (chart, seriesIndex, groupIndex, itemIndex) => {
    var customWidth = DvtChartStyleUtils.getUserBorderWidth(
      chart,
      seriesIndex,
      groupIndex,
      itemIndex
    );
    if (customWidth != null) {
      return customWidth;
    }
    return DvtChartStyleUtils.getDefaultBorderWidth(chart);
  },
  /**
   * Returns the marker color for the specified data item.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @param {number} itemIndex
   * @return {string} The marker color string.
   */
  getMarkerColor: (chart, seriesIndex, groupIndex, itemIndex) => {
    if (!DvtChartStyleUtils.isMarkerDisplayed(chart, seriesIndex, groupIndex, itemIndex))
      return DvtChartStyleUtils.getColor(chart, seriesIndex, groupIndex);

    // Nested Data Override
    var nestedDataItem = DvtChartDataUtils.getNestedDataItem(
      chart,
      seriesIndex,
      groupIndex,
      itemIndex
    );
    if (nestedDataItem && nestedDataItem['color']) return nestedDataItem['color'];

    // Data Override: Note that the data object defines a single 'color' attribute
    var dataItem = DvtChartDataUtils.getDataItem(chart, seriesIndex, groupIndex);
    if (dataItem && dataItem['color']) return dataItem['color'];

    // Series Override
    var seriesItem = DvtChartDataUtils.getSeriesItem(chart, seriesIndex);
    if (seriesItem && seriesItem['markerColor']) return seriesItem['markerColor'];

    // Style Defaults
    var options = chart.getOptions();
    var defaultMarkerColor = options['styleDefaults']['markerColor'];
    if (defaultMarkerColor)
      // Return the default if set
      return defaultMarkerColor;
    else {
      // Otherwise return the series color
      return DvtChartStyleUtils.getColor(chart, seriesIndex, groupIndex);
    }
  },

  /**
   * Returns the marker shape for the specified data item.  Returns the actual shape
   * if the marker shape is set to "auto".
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @param {number} itemIndex
   * @return {string} The marker shape.
   */
  getMarkerShape: (chart, seriesIndex, groupIndex, itemIndex) => {
    // Style Defaults
    var options = chart.getOptions();
    var shape = options['styleDefaults']['markerShape'];

    // Series Override
    var seriesItem = DvtChartDataUtils.getSeriesItem(chart, seriesIndex);
    if (seriesItem && seriesItem['markerShape']) shape = seriesItem['markerShape'];

    // Data Item Override
    var dataItem = DvtChartDataUtils.getDataItem(chart, seriesIndex, groupIndex);
    if (dataItem && dataItem['markerShape']) shape = dataItem['markerShape'];

    // Nested Item Override
    var nestedDataItem = DvtChartDataUtils.getNestedDataItem(
      chart,
      seriesIndex,
      groupIndex,
      itemIndex
    );
    if (nestedDataItem && nestedDataItem['markerShape']) shape = nestedDataItem['markerShape'];

    // Convert automatic shape to actual shape
    if (shape == 'auto') {
      if (
        DvtChartTypeUtils.isBubble(chart) ||
        DvtChartDataUtils.getSeriesType(chart, seriesIndex) == 'boxPlot' ||
        DvtChartDataUtils.isRangeSeries(chart, seriesIndex)
      )
        shape = 'circle';
      else {
        var styleIndex = DvtChartDataUtils.getSeriesStyleIdx(chart, seriesIndex);

        // Iterate through the shape ramp to find the right shape
        var shapeRamp = options['styleDefaults']['shapes'];
        var shapeIndex = styleIndex % shapeRamp.length;
        shape = shapeRamp[shapeIndex];
      }
    }

    return shape;
  },

  /**
   * Returns the marker size for the specified data item.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @param {number} itemIndex
   * @return {number} The marker size.
   */
  getMarkerSize: (chart, seriesIndex, groupIndex, itemIndex) => {
    var markerSize;
    if (DvtChartTypeUtils.isBubble(chart)) {
      markerSize = chart
        .getOptionsCache()
        .getFromCachedMap2D('bubbleSizeCache', seriesIndex, groupIndex);
      if (markerSize) return markerSize;
    }

    var nestedDataItem = DvtChartDataUtils.getNestedDataItem(
      chart,
      seriesIndex,
      groupIndex,
      itemIndex
    );
    if (nestedDataItem && nestedDataItem['markerSize'] != null)
      // Nested Data Override
      markerSize = Number(nestedDataItem['markerSize']);
    else {
      var dataItem = DvtChartDataUtils.getDataItem(chart, seriesIndex, groupIndex);
      if (dataItem && dataItem['markerSize'] != null)
        // Data Override
        markerSize = Number(dataItem['markerSize']);
      else {
        var seriesItem = DvtChartDataUtils.getSeriesItem(chart, seriesIndex);
        if (seriesItem && seriesItem['markerSize'] != null)
          // Series Override
          markerSize = Number(seriesItem['markerSize']);
        // Style Defaults
        else markerSize = Number(chart.getOptions()['styleDefaults']['markerSize']);
      }
    }

    // Scale down for chart overview
    if (DvtChartTypeUtils.isOverview(chart)) markerSize = Math.ceil(markerSize * 0.6);
    return markerSize;
  },

  /**
   * Returns the whether markers are displayed for the specified line or area series.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @param {number} itemIndex
   * @return {boolean} Whether markers should be displayed.
   */
  isMarkerDisplayed: (chart, seriesIndex, groupIndex, itemIndex) => {
    var displayed;
    var nestedDataItem = DvtChartDataUtils.getNestedDataItem(
      chart,
      seriesIndex,
      groupIndex,
      itemIndex
    );
    if (nestedDataItem && nestedDataItem['markerDisplayed'] != null)
      // nested data item override
      displayed = nestedDataItem['markerDisplayed'];
    else {
      var dataItem = DvtChartDataUtils.getDataItem(chart, seriesIndex, groupIndex);
      if (dataItem && dataItem['markerDisplayed'] != null)
        // data item override
        displayed = dataItem['markerDisplayed'];
      else {
        var seriesItem = DvtChartDataUtils.getSeriesItem(chart, seriesIndex);
        if (seriesItem && seriesItem['markerDisplayed'] != null)
          // series item override
          displayed = seriesItem['markerDisplayed'];
        // style defaults
        else displayed = chart.getOptions()['styleDefaults']['markerDisplayed'];
      }
    }

    if (displayed === 'on') return true;
    else if (displayed === 'off') return false;
    else if (DvtChartTypeUtils.isCombo(chart) && displayed === 'auto') return true;
    else
      return (
        DvtChartTypeUtils.isScatterBubble(chart) ||
        DvtChartDataUtils.getSeriesType(chart, seriesIndex) == 'boxPlot' ||
        DvtChartStyleUtils.getLineType(chart, seriesIndex) == 'none'
      );
  },

  /**
   * Returns the marker size for the specified data item.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @param {number} itemIndex
   * @param {string} sourceType
   * @return {string} The marker source for the type passed in.
   */
  getImageSource: (chart, seriesIndex, groupIndex, itemIndex, sourceType) => {
    // Nested Data Override
    var nestedDataItem = DvtChartDataUtils.getNestedDataItem(
      chart,
      seriesIndex,
      groupIndex,
      itemIndex
    );
    if (nestedDataItem && nestedDataItem[sourceType]) return nestedDataItem[sourceType];

    // Data Override
    var dataItem = DvtChartDataUtils.getDataItem(chart, seriesIndex, groupIndex);
    if (dataItem && dataItem[sourceType]) return dataItem[sourceType];

    // Series Override
    var seriesItem = DvtChartDataUtils.getSeriesItem(chart, seriesIndex);
    if (seriesItem && seriesItem[sourceType]) return seriesItem[sourceType];

    return undefined;
  },

  /**
   * Returns the line width for the specified series.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @return {number} The line width.
   */
  getLineWidth: (chart, seriesIndex) => {
    var seriesItem = DvtChartDataUtils.getSeriesItem(chart, seriesIndex);
    var options = chart.getOptions();
    var lineWidth;

    if (seriesItem && seriesItem['lineWidth'])
      // Series Override
      lineWidth = seriesItem['lineWidth'];
    else if (options['styleDefaults']['lineWidth'])
      // Style Defaults
      lineWidth = options['styleDefaults']['lineWidth'];
    else if (DvtChartDataUtils.getSeriesType(chart, seriesIndex) == 'lineWithArea') lineWidth = 2;
    else lineWidth = 3;

    // Scale down for chart overview
    if (DvtChartTypeUtils.isOverview(chart)) lineWidth = Math.ceil(lineWidth * 0.6);

    return lineWidth;
  },

  /**
   * Returns the line style for the specified series.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @return {string} The line style.
   */
  getLineStyle: (chart, seriesIndex) => {
    // Series Override
    var seriesItem = DvtChartDataUtils.getSeriesItem(chart, seriesIndex);
    if (seriesItem && seriesItem['lineStyle']) return seriesItem['lineStyle'];

    // Style Defaults
    var options = chart.getOptions();
    return options['styleDefaults']['lineStyle'];
  },

  /**
   * Returns the line type for the specified series.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @return {string} The line type.
   */
  getLineType: (chart, seriesIndex) => {
    var lineType;
    var seriesItem = DvtChartDataUtils.getSeriesItem(chart, seriesIndex);

    if (seriesItem && seriesItem['lineType'])
      // Series Override
      lineType = seriesItem['lineType'];
    // Style Defaults
    else lineType = chart.getOptions()['styleDefaults']['lineType'];

    if (lineType == 'auto')
      lineType = DvtChartTypeUtils.isScatterBubble(chart) ? 'none' : 'straight';

    // Centered segmented/stepped are not supported for polar and scatter/bubble
    if (DvtChartTypeUtils.isPolar(chart) || DvtChartTypeUtils.isScatterBubble(chart)) {
      if (lineType == 'centeredSegmented') lineType = 'segmented';
      if (lineType == 'centeredStepped') lineType = 'stepped';
    }

    return lineType;
  },

  /**
   * Returns the bar spacing behavior.  Only applies for spark charts.
   * @param {Chart} chart
   * @return {string} The bar spacing behavior
   */
  getBarSpacing: (chart) => {
    var options = chart.getOptions();
    return options['__sparkBarSpacing'];
  },

  /**
   * Returns the maxBarWidth (in pixels) of the bars.
   * @param {Chart} chart
   * @return {number}
   */
  getMaxBarWidth: (chart) => {
    var maxBarWidth = chart.getOptions()['styleDefaults']['maxBarWidth'];
    return maxBarWidth != null && !DvtChartTypeUtils.isPolar(chart) ? maxBarWidth : Infinity;
  },
  /**
   * Returns the bar width for the specified series and group.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @return {number} The bar width.
   */
  getBarWidth: (chart, seriesIndex, groupIndex) => {
    // If the chart doesn't have Z values, the stack widths are all the same, so override the seriesIndex and groupIndex so
    // the computation doesn't need to be repeated for every group and we can use the cached value instead.
    if (chart.getOptionsCache().getFromCache('hasConstantZValue')) {
      seriesIndex = 0;
      groupIndex = 0;
    }

    var cacheKey = 'barWidth';
    var barWidth = chart.getCache().getFromCachedMap2D(cacheKey, seriesIndex, groupIndex);
    if (barWidth != null) return barWidth;

    var ratio =
      DvtChartDataUtils.getZVal(chart, seriesIndex, groupIndex, 1) /
      chart.getOptions()['_averageGroupZ'];
    barWidth = Math.min(
      ratio * DvtChartStyleUtils.getGroupWidth(chart),
      DvtChartStyleUtils.getMaxBarWidth(chart)
    );

    chart.getCache().putToCachedMap2D(cacheKey, seriesIndex, groupIndex, barWidth);
    return barWidth;
  },
  /**
   * Returns the bar width for the specified stack category.
   * @param {Chart} chart
   * @param {string} category The stack category. Use the series name for unstacked charts.
   * @param {number} groupIndex
   * @param {boolean} isY2 Whether the stack is assigned to Y2.
   * @return {number} The stack width.
   */
  getBarStackWidth: (chart, category, groupIndex, isY2) => {
    // If the chart doesn't have Z values, the stack widths are all the same, so override the groupIndex so
    // the computation doesn't need to be repeated for every group and we can use the cached value instead.
    if (chart.getOptionsCache().getFromCache('hasConstantZValue')) groupIndex = 0;

    var cacheKey = isY2 ? 'y2BarStackWidth' : 'yBarStackWidth';
    var barStackWidth = chart.getCache().getFromCachedMap2D(cacheKey, category, groupIndex);
    if (barStackWidth != null) return barStackWidth;

    var ratio =
      DvtChartDataUtils.getBarCategoryZ(chart, category, groupIndex, isY2) /
      chart.getOptions()['_averageGroupZ'];
    barStackWidth = Math.min(
      ratio * DvtChartStyleUtils.getGroupWidth(chart),
      DvtChartStyleUtils.getMaxBarWidth(chart)
    );
    chart.getCache().putToCachedMap2D(cacheKey, category, groupIndex, barStackWidth);
    return barStackWidth;
  },
  /**
   * Computes the offsets of the bar stack categories relative to the group coordinate and stores it in a map.
   * @param {Chart} chart
   * @param {Number} groupIndex
   * @return {Object} An object containing two maps for y and y2 respectively. Each map contains the categories as the
   *    keys and the offsets as the values.
   */
  getBarCategoryOffsetMap: (chart, groupIndex) => {
    // If the chart doesn't have Z values, the stack widths are all the same, so override the groupIndex so
    // the computation doesn't need to be repeated for every group and we can use the cached value instead.
    if (chart.getOptionsCache().getFromCache('hasConstantZValue')) groupIndex = 0;

    var cacheKey = 'barCategoryOffsetMap';
    var yOffsetMaps = chart.getCache().getFromCachedMap(cacheKey, groupIndex);
    if (yOffsetMaps) return yOffsetMaps;

    var bStacked = DvtChartDataUtils.isStacked(chart);
    var categories = DvtChartDataUtils.getStackCategories(chart, 'bar');
    var isMixedFreq = DvtChartTypeUtils.isMixedFrequency(chart);
    var isSplitDualY = DvtChartDataUtils.isSplitDualY(chart);
    var yOffsetMap = {},
      y2OffsetMap = {};
    var yTotalWidth = 0,
      y2TotalWidth = 0;
    var stackWidth, i;

    // Populate offset maps
    if (bStacked) {
      // Use stack categories to get the width of each stack
      // Iterate through the y-axis stack categories and store the offsets relative the the start coord of the y stack
      for (i = 0; i < categories['y'].length; i++) {
        stackWidth = DvtChartStyleUtils.getBarStackWidth(
          chart,
          categories['y'][i],
          groupIndex,
          false
        );
        if (isMixedFreq) yOffsetMap[categories['y'][i]] = -0.5 * stackWidth;
        else {
          yOffsetMap[categories['y'][i]] = yTotalWidth;
          yTotalWidth += stackWidth;
        }
      }

      if (!isSplitDualY) y2TotalWidth = yTotalWidth;

      // Iterate through the y2-axis stack categories and store the offsets relative the the start coord of the y2 stack
      for (i = 0; i < categories['y2'].length; i++) {
        stackWidth = DvtChartStyleUtils.getBarStackWidth(
          chart,
          categories['y2'][i],
          groupIndex,
          true
        );
        if (isMixedFreq) y2OffsetMap[categories['y2'][i]] = -0.5 * stackWidth;
        else {
          y2OffsetMap[categories['y2'][i]] = y2TotalWidth;
          y2TotalWidth += stackWidth;
        }
      }

      if (!isSplitDualY) yTotalWidth = y2TotalWidth;
    } else {
      // The width of each bar series item is the width of the stack
      var seriesCount = DvtChartDataUtils.getSeriesCount(chart);
      for (var z = 0; z < seriesCount; z++) {
        var seriesType = DvtChartDataUtils.getSeriesType(chart, z);
        if (
          (seriesType != 'bar' && seriesType != 'candlestick' && seriesType != 'boxPlot') ||
          !DvtChartDataUtils.isSeriesRendered(chart, z)
        )
          continue;

        var isY2Series = DvtChartDataUtils.isAssignedToY2(chart, z);
        var category = DvtChartDataUtils.getStackCategory(chart, z);
        stackWidth = DvtChartStyleUtils.getBarWidth(chart, z, groupIndex);

        if (!isY2Series) {
          if (isMixedFreq) yOffsetMap[category] = -0.5 * stackWidth;
          else {
            yOffsetMap[category] = yTotalWidth;
            yTotalWidth += stackWidth;
          }
        } else {
          if (isMixedFreq) y2OffsetMap[category] = -0.5 * stackWidth;
          else {
            y2OffsetMap[category] = y2TotalWidth;
            y2TotalWidth += stackWidth;
          }
        }
      }
    }

    // Now shift each bar by half the total stack width
    for (var yCategory in yOffsetMap)
      yOffsetMap[yCategory] -=
        !isSplitDualY && !bStacked ? (yTotalWidth + y2TotalWidth) / 2 : yTotalWidth / 2;
    for (var y2Category in y2OffsetMap)
      y2OffsetMap[y2Category] -=
        !isSplitDualY && !bStacked
          ? (yTotalWidth + y2TotalWidth) / 2 - yTotalWidth
          : y2TotalWidth / 2;

    yOffsetMaps = { y: yOffsetMap, y2: y2OffsetMap };
    chart.getCache().putToCachedMap(cacheKey, groupIndex, yOffsetMaps);
    return yOffsetMaps;
  },
  /**
   * Returns the ratio of data item gaps to be used as a number from 0 to 1.
   * @param {Chart} chart
   * @return {number}
   */
  getDataItemGaps: (chart) => {
    var cacheKey = 'dataItemGaps';
    var ret = chart.getOptionsCache().getFromCache(cacheKey);
    if (ret != null) {
      return ret;
    }

    var options = chart.getOptions();

    if (options['styleDefaults']['sliceGaps'] != null) {
      // Backwards compatibility with sliceGaps, which is a number between 0 and 1
      ret = options['styleDefaults']['sliceGaps'];
    } else {
      // dataItemGaps: Currently a percentage string to be converted to ratio or "auto"
      var dataItemGaps = options['styleDefaults']['dataItemGaps'];
      if (dataItemGaps == 'auto') {
        // Auto is 50% for 2D charts, 0% for 3D charts
        dataItemGaps = options['styleDefaults']['threeDEffect'] == 'on' ? '0%' : '50%';
      }

      // Process the percentage value
      var percentIndex = dataItemGaps && dataItemGaps.indexOf ? dataItemGaps.indexOf('%') : -1;
      if (percentIndex >= 0) {
        dataItemGaps = dataItemGaps.substring(0, percentIndex);
        ret = dataItemGaps / 100;
      } else {
        // Not a valid string, return 0.
        ret = 0;
      }
    }

    chart.getOptionsCache().putToCache(cacheKey, ret);
    return ret;
  },
  /**
   * Returns true if the specified data item is selectable.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @return {boolean} True if the data item is selectable.
   */
  isSelectable: (chart, seriesIndex, groupIndex) => {
    // Series Override
    var seriesItem = DvtChartDataUtils.getSeriesItem(chart, seriesIndex);
    if (seriesItem && seriesItem['_selectable'] == 'off') return false;

    // Otherwise the requirements are that selection is enabled and the object corresponds to a data item.
    return (
      chart.isSelectionSupported() &&
      seriesIndex != null &&
      seriesIndex >= 0 &&
      groupIndex != null &&
      groupIndex >= 0
    );
  },
  /**
   * Returns the display animation for the specified chart.
   * @param {Chart} chart
   * @return {string}
   */
  getAnimOnDisplay: (chart) => {
    return chart.getOptions()['animationOnDisplay'];
  },

  /**
   * Returns the data change animation for the specified chart.
   * @param {Chart} chart
   * @return {string}
   */
  getAnimOnDataChange: (chart) => {
    return chart.getOptions()['animationOnDataChange'];
  },

  /**
   * Returns the animation duration in seconds for the specified chart.  This duration is
   * intended to be passed to the animatino handler, and is not in the same units
   * as the API.
   * @param {Chart} chart
   * @return {number} The animation duration in seconds.
   */
  getAnimDur: (chart) => {
    return (
      CSSStyle.getTimeMilliseconds(chart.getOptions()['styleDefaults']['animationDuration']) /
      1000
    );
  },

  /**
   * Returns the animation indicators property for the specified chart.
   * @param {Chart} chart
   * @return {string}  The animation indicators value.
   */
  getAnimIndicators: (chart) => {
    return chart.getOptions()['styleDefaults']['animationIndicators'];
  },

  /**
   * Returns the animation indicators up color.
   * @param {Chart} chart
   * @return {string}  The animation indicator up color.
   */
  getAnimUpColor: (chart) => {
    return chart.getOptions()['styleDefaults']['animationUpColor'];
  },

  /**
   * Returns the animation indicators down color.
   * @param {Chart} chart
   * @return {string}  The animation indicator down color.
   */
  getAnimDownColor: (chart) => {
    return chart.getOptions()['styleDefaults']['animationDownColor'];
  },
  /**
   * Returns the inner color of the selection feedback.
   * @param {Chart} chart
   * @return {string}
   */
  getSelectedInnerColor: (chart) => {
    return chart.getOptions()['styleDefaults']['selectedInnerColor'];
  },

  /**
   * Returns the outer color of the selection feedback.
   * @param {Chart} chart
   * @return {string}
   */
  getSelectedOuterColor: (chart) => {
    return chart.getOptions()['styleDefaults']['selectedOuterColor'];
  },
  /**
   * Returns whether the selected items are highlighted.
   * @param {Chart} chart
   * @return {boolean}
   */
  isSelectionHighlighted: (chart) => {
    var effect = chart.getOptions()['styleDefaults']['selectionEffect'];
    return effect == 'highlight' || effect == 'highlightAndExplode';
  },
  /**
   * Returns whether the selected items are exploded (only applies to pie).
   * @param {Chart} chart
   * @return {boolean}
   */
  isSelectionExploded: (chart) => {
    var effect = chart.getOptions()['styleDefaults']['selectionEffect'];
    return effect == 'explode' || effect == 'highlightAndExplode';
  },
  /**
   * Returns the data label style for the specified data point.
   * @param {Chart} chart
   * @param {number} seriesIndex The series index.
   * @param {number} groupIndex The group index.
   * @param {number} itemIndex The nested item index.
   * @param {Color} dataColor The color of the marker this is associated with.
   * @param {string} position The position returned by the getDataLabelPosition function, not the API values.
   * @param {string=} type Data label type: low, high, or value.
   * @return {string} The data label, null if the index is invalid.
   */
  getDataLabelStyle: (chart, seriesIndex, groupIndex, itemIndex, dataColor, position, type) => {
    var styleDefaults = chart.getOptions().styleDefaults;
    var labelStyleArray = [styleDefaults._dataLabelStyle, styleDefaults.dataLabelStyle];
    var contrastingColor;
    const seriesType = DvtChartDataUtils.getSeriesType(chart, seriesIndex);
    const supportsOutline = DvtChartStyleUtils.supportsLabelOutline(chart, seriesIndex);
    if (
      !supportsOutline &&
      dataColor &&
      (seriesType == 'bar' || DvtChartTypeUtils.isBubble(chart)) &&
      (position == 'center' ||
        position == 'inBottom' ||
        position == 'inTop' ||
        position == 'inRight' ||
        position == 'inLeft')
    ) {
      // issue identified by JET-56558, JET-59519, JET-59520 will be solved by JET-65212
      contrastingColor =
        DvtChartStyleUtils.getPattern(chart, seriesIndex, groupIndex, itemIndex) != null
          ? '#000000'
          : ColorUtils.getContrastingTextColor(dataColor);
      labelStyleArray.push(new CSSStyle('color: ' + contrastingColor + ';'));
    }
    labelStyleArray.push(
      DvtChartStyleUtils._parseLowHighArray(
        chart.getOptions()['styleDefaults']['dataLabelStyle'],
        type
      )
    );

    var dataItem = DvtChartDataUtils.getDataItem(chart, seriesIndex, groupIndex);
    if (dataItem)
      labelStyleArray.push(
        new CSSStyle(DvtChartStyleUtils._parseLowHighArray(dataItem['labelStyle'], type))
      );

    var nestedDataItem = DvtChartDataUtils.getNestedDataItem(
      chart,
      seriesIndex,
      groupIndex,
      itemIndex
    );
    if (nestedDataItem)
      labelStyleArray.push(
        new CSSStyle(DvtChartStyleUtils._parseLowHighArray(nestedDataItem['labelStyle'], type))
      );

    // In high contrast mode, force use of contrasting color and ignore custom color
    if (contrastingColor && Agent.isHighContrast())
      labelStyleArray.push(new CSSStyle('color: ' + contrastingColor + ';'));

    return CSSStyle.mergeStyles(labelStyleArray);
  },

  /**
   * Returns an object which indicates where the labels has collisions.
   * @param {Chart} chart
   * @param {string} centeredTextCoords Centered data label coordinates
   * @param {object} textDims Data label dimensions
   * @param {object} plotAreaDims Plot area dimensions
   * @param {object} isHoriz Whether or not the chart is horizontal.
   * @return {object} Indicates where the label has collisions: xAxis, yAxis, y2Axis, legend, top of chart
   */
  getDataLabelCollisions: (chart, centeredTextCoords, textDims, plotAreaDims, isHoriz) => {
    var textX =
      centeredTextCoords && centeredTextCoords.x
        ? centeredTextCoords.x - textDims.w / 2 || textDims.x
        : textDims.x;
    var textY =
      centeredTextCoords && centeredTextCoords.y
        ? centeredTextCoords.y - textDims.h / 2 || textDims.y
        : textDims.y;
    var buffer = DvtChartStyleUtils.MARKER_DATA_LABEL_GAP / 2;
    var options = chart.getOptions();
    var legendGap = isHoriz
      ? DvtChartDefaults.getGapHeight(chart, options['layout']['legendGapHeight'])
      : DvtChartDefaults.getGapWidth(chart, options['layout']['legendGapWidth']);
    return {
      xAxis: textY + textDims.h + buffer > plotAreaDims.y + plotAreaDims.h,
      yAxis: textX + buffer < plotAreaDims.x,
      y2Axis: textX + textDims.w + buffer > plotAreaDims.x + plotAreaDims.w,
      legend: textX + textDims.w + buffer > plotAreaDims.x + plotAreaDims.w + legendGap,
      top: textY < chart.getDimensions().y // No need for buffer because textY accounts for typography baseline
    };
  },
  /**
   * Returns an adjusted label position. It flips or adjusts the label position if there are collisions detected for that label.
   * @param {Chart} chart
   * @param {string} position Initial label position
   * @param {object} detectedCollisions Indicates where label has collision: xAxis, yAxis, y2Axis, legend, top of chart
   * @param {boolean} hasY2Axis Whether the chart has a y2 axis
   * @return {string} The adjusted data label position.
   */
  adjustDataLabelPos: (position, detectedCollisions, hasY2Axis) => {
    var adjustedPosition = position;

    // flip alg for collisions
    if ((position == 'left' || position == 'center') && detectedCollisions.yAxis) {
      adjustedPosition = 'right';
    } else if (
      (position == 'right' || position == 'center') &&
      ((hasY2Axis && detectedCollisions.y2Axis) || detectedCollisions.legend)
    ) {
      adjustedPosition = 'left';
    } else if ((position == 'top' || position == 'center') && detectedCollisions.top) {
      adjustedPosition = 'bottom';
    } else if ((position == 'bottom' || position == 'center') && detectedCollisions.xAxis) {
      adjustedPosition = 'top';
    }
    return adjustedPosition;
  },
  /**
   * Parses the data label attribute. If a single value is provided, it will apply to all labels. If an array of
   * two values is provided, the first and second value will apply to the low and high label respectively.
   * @param {object} value The attribute value.
   * @param {type} type Data label type: low, high, or value.
   * @return {object} The value corresponding to the type.
   * @private
   */
  _parseLowHighArray: (value, type) => {
    if (value instanceof Array) return type == 'high' ? value[1] : value[0];
    else return value;
  },

  /**
   * Returns whether the overview is rendered.
   * @param {Chart} chart
   * @return {boolean}
   */
  isOverviewRendered: (chart) => {
    var options = chart.getOptions();
    return DvtChartTypeUtils.isOverviewSupported(chart) && options['overview']['rendered'] != 'off';
  },

  /**
   * Returns the height of the overview scrollbar.
   * @param {Chart} chart
   * @return {number} The height.
   */
  getOverviewHeight: (chart) => {
    var options = chart.getOptions();
    var height = options['overview']['height'];
    if (height == null) height = DvtChartTypeUtils.hasTimeAxis(chart) ? 0.25 : 0.2; // use default ratio

    return DvtChartStyleUtils.getSizeInPixels(height, chart.getHeight());
  },

  /**
   * Computes the size of a subcomponent in pixels by parsing the user input.
   * @param {object} size The size input given by the user. It can be in percent, pixels, or number.
   * @param {number} totalSize The total size of the component in pixels.
   * @return {number} The size of the subcomponent in pixels.
   */
  getSizeInPixels: (size, totalSize) => {
    if (typeof size == 'string') {
      if (size.slice(-1) == '%') return (totalSize * Number(size.slice(0, -1))) / 100;
      else if (size.slice(-2) == 'px') return Number(size.slice(0, -2));
      else size = Number(size);
    }

    if (typeof size == 'number') {
      if (size <= 1)
        // assume to be ratio
        return totalSize * size;
      // assume to be absolute size in pixels
      else return size;
    } else return 0;
  },
  /**
   * Returns the plot area background color.
   * @param {Chart} chart
   * @param {Boolean} useDefault Whether it should fall back to the white default if the background color is not defined.
   * @return {String}
   */
  getBackgroundColor: (chart, useDefault) => {
    var options = chart.getOptions();
    if (options['plotArea']['backgroundColor']) return options['plotArea']['backgroundColor'];
    else return useDefault ? '#FFFFFF' : null;
  },
  /**
   * Returns the delay before mouse over event is triggerd.
   * This is used for highlighting in chart.
   * @param {Chart} chart
   * @return {number} The delay in ms.
   */
  getHoverBehaviorDelay: (chart) => {
    var delay = chart.getOptions()['styleDefaults']['hoverBehaviorDelay'];
    if (delay) {
      delay = CSSStyle.getTimeMilliseconds(delay);

      if (DvtChartTypeUtils.isScatterBubble(chart) || DvtChartTypeUtils.isLine(chart)) {
        return 0.75 * delay;
      } else {
        return 1.25 * delay;
      }
    } else {
      return 0;
    }
  },
  /**
   * Returns true if the marker stroke should be optimized by moving onto a container.
   * @param {Chart} chart
   * @return {boolean}
   */
  optimizeMarkerStroke: (chart) => {
    return DvtChartTypeUtils.isScatterBubble(chart) || DvtChartTypeUtils.isBoxPlot(chart);
  },
  /**
   * Returns the chart x-axis group width.
   * Use this method instead of calling chart.xAxis.getInfo().getGroupWidth() directly for better performance.
   * @param {Chart} chart
   * @return {number}
   */
  getGroupWidth: (chart) => {
    // Use the cached value if it has been computed before
    var cacheKey = 'groupWidth';
    var width = chart.getCache().getFromCache(cacheKey);
    if (width == null) {
      width = chart.xAxis.getInfo().getGroupWidth();
      chart.getCache().putToCache(cacheKey, width);
    }
    return width;
  },
  /**
   * Returns true if the chart supports stacking and stack label is enabled.
   * @param {Chart} chart
   * @return {boolean}
   */
  isStackLabelRendered: (chart) => {
    // To have stack labels, the attribute must be set and the chart must be a supporting type.
    var options = chart.getOptions();
    //check if stack is enabled first
    if (options['stack'] != 'on' || DvtChartTypeUtils.isMixedFrequency(chart)) {
      return false;
    }

    if (options['stackLabel'] === 'on') return DvtChartTypeUtils.isBLAC(chart);

    return false;
  },
  /**
   * Returns true if the marker fill should be optimized by moving onto a container.
   * @param {Chart} chart
   * @return {boolean}
   */
  optimizeMarkerFill: (chart) => {
    return DvtChartTypeUtils.isLineArea(chart);
  },
  /**
   * Returns the className for the specified data item.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @return {string} The class string.
   */
  getClassName: (chart, seriesIndex, groupIndex) => {
    var dataItem = DvtChartDataUtils.getDataItem(chart, seriesIndex, groupIndex);
    if (dataItem && (dataItem['className'] || dataItem['svgClassName']))
      return dataItem['className'] || dataItem['svgClassName'];

    var seriesItem = DvtChartDataUtils.getSeriesItem(chart, seriesIndex);
    if (seriesItem && (seriesItem['className'] || seriesItem['svgClassName']))
      return seriesItem['className'] || seriesItem['svgClassName'];
    else return null;
  },
  /**
   * Returns the areaClassName for the specified series.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @return {string} The class string.
   */
  getAreaClassName: (chart, seriesIndex) => {
    var seriesItem = DvtChartDataUtils.getSeriesItem(chart, seriesIndex);
    if (seriesItem && (seriesItem['areaClassName'] || seriesItem['areaSvgClassName']))
      return seriesItem['areaClassName'] || seriesItem['areaSvgClassName'];
    else if (seriesItem && (seriesItem['className'] || seriesItem['svgClassName']))
      return seriesItem['className'] || seriesItem['svgClassName'];
    return null;
  },
  /**
   * Returns the className for the specified data item marker.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @param {number} itemIndex
   * @return {string} The class string.
   */
  getMarkerClassName: (chart, seriesIndex, groupIndex, itemIndex) => {
    // Nested Data Override
    var nestedDataItem = DvtChartDataUtils.getNestedDataItem(
      chart,
      seriesIndex,
      groupIndex,
      itemIndex
    );
    if (nestedDataItem && (nestedDataItem['className'] || nestedDataItem['svgClassName']))
      return nestedDataItem['className'] || nestedDataItem['svgClassName'];

    // Data Override
    var dataItem = DvtChartDataUtils.getDataItem(chart, seriesIndex, groupIndex);
    if (dataItem && (dataItem['className'] || dataItem['svgClassName']))
      return dataItem['className'] || dataItem['svgClassName'];

    // Series Override
    var seriesItem = DvtChartDataUtils.getSeriesItem(chart, seriesIndex);
    if (seriesItem && (seriesItem['markerClassName'] || seriesItem['markerSvgClassName']))
      return seriesItem['markerClassName'] || seriesItem['markerSvgClassName'];

    if (
      DvtChartTypeUtils.isScatterBubble(chart) &&
      seriesItem &&
      (seriesItem['className'] || seriesItem['svgClassName'])
    )
      return seriesItem['className'] || seriesItem['svgClassName'];
    else return null;
  },
  /**
   * Returns the style for the specified data item.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @return {object} The object defining the style.
   */
  getStyle: (chart, seriesIndex, groupIndex) => {
    var dataItem = DvtChartDataUtils.getDataItem(chart, seriesIndex, groupIndex);
    if (dataItem && (dataItem['style'] || dataItem['svgStyle']))
      return dataItem['style'] || dataItem['svgStyle'];

    var seriesItem = DvtChartDataUtils.getSeriesItem(chart, seriesIndex);
    if (seriesItem && (seriesItem['style'] || seriesItem['svgStyle']))
      return seriesItem['style'] || seriesItem['svgStyle'];
    return null;
  },
  /**
   * Returns the areaStyle for the specified series.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @return {object} The object defining the style.
   */
  getAreaStyle: (chart, seriesIndex) => {
    var seriesItem = DvtChartDataUtils.getSeriesItem(chart, seriesIndex);
    if (seriesItem && (seriesItem['areaStyle'] || seriesItem['areaSvgStyle']))
      return seriesItem['areaStyle'] || seriesItem['areaSvgStyle'];
    else if (seriesItem && (seriesItem['style'] || seriesItem['svgStyle']))
      return seriesItem['style'] || seriesItem['svgStyle'];
    return null;
  },
  /**
   * Returns the style for the specified data item marker.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @param {number} itemIndex
   * @return {object} The object defining the style.
   */
  getMarkerStyle: (chart, seriesIndex, groupIndex, itemIndex) => {
    // Nested Data Override
    var nestedDataItem = DvtChartDataUtils.getNestedDataItem(
      chart,
      seriesIndex,
      groupIndex,
      itemIndex
    );
    if (nestedDataItem && (nestedDataItem['style'] || nestedDataItem['svgStyle']))
      return nestedDataItem['style'] || nestedDataItem['svgStyle'];

    // Data Override
    var dataItem = DvtChartDataUtils.getDataItem(chart, seriesIndex, groupIndex);
    if (dataItem && (dataItem['style'] || dataItem['svgStyle']))
      return dataItem['style'] || dataItem['svgStyle'];

    // Series Override
    var seriesItem = DvtChartDataUtils.getSeriesItem(chart, seriesIndex);
    if (seriesItem && (seriesItem['markerStyle'] || seriesItem['markerSvgStyle']))
      return seriesItem['markerStyle'] || seriesItem['markerSvgStyle'];

    if (
      DvtChartTypeUtils.isScatterBubble(chart) &&
      seriesItem &&
      (seriesItem['style'] || seriesItem['svgStyle'])
    )
      return seriesItem['style'] || seriesItem['svgStyle'];
    else return null;
  },
  /**
   * Returns an object containing the computed box plot options for the specified data item.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @return {object}
   */
  getBoxPlotStyleOptions: (chart, seriesIndex, groupIndex) => {
    // Data item override
    var dataItem = DvtChartDataUtils.getDataItem(chart, seriesIndex, groupIndex);
    var dataItemOptions = dataItem && dataItem['boxPlot'] ? dataItem['boxPlot'] : {};

    // Series override
    var seriesItem = DvtChartDataUtils.getSeriesItem(chart, seriesIndex);
    var seriesOptions = seriesItem && seriesItem['boxPlot'] ? seriesItem['boxPlot'] : {};

    // Style defaults
    var styleDefaults = chart.getOptions()['styleDefaults'];
    var styleDefaultsOptions = styleDefaults['boxPlot'];

    // Merge the three options
    var boxPlotOptions = JsonUtils.merge(
      dataItemOptions,
      JsonUtils.merge(seriesOptions, styleDefaultsOptions)
    );

    // Default color ramp support. Pass the data color privately for the hover color
    var defaultColor = DvtChartStyleUtils.getColor(chart, seriesIndex, groupIndex);
    boxPlotOptions['_color'] = defaultColor;
    var defaultBoxColor = ColorUtils.getBrighter(defaultColor, 0.8);
    if (!boxPlotOptions['q2Color']) boxPlotOptions['q2Color'] = defaultBoxColor;
    if (!boxPlotOptions['q3Color']) boxPlotOptions['q3Color'] = defaultBoxColor;

    // Default pattern ramp support (not public API, but needed for seriesEffect=pattern)
    if (DvtChartStyleUtils.getSeriesEffect(chart) == 'pattern') {
      var defaultPattern = DvtChartStyleUtils.getPattern(chart, seriesIndex, groupIndex);
      boxPlotOptions['_q2Pattern'] = defaultPattern;
      boxPlotOptions['_q3Pattern'] = defaultPattern;
    }

    // Border color and border width. Box plot has a border by default
    var defaultBorderColor = DvtChartStyleUtils.getBorderColor(chart, seriesIndex, groupIndex);
    boxPlotOptions['borderColor'] = defaultBorderColor ? defaultBorderColor : defaultColor;
    boxPlotOptions['borderWidth'] = DvtChartStyleUtils.getBorderWidth(
      chart,
      seriesIndex,
      groupIndex
    );

    // Default whisker and median line color
    var defaultLineColor = ColorUtils.getDarker(defaultColor, 0.1);
    DvtChartStyleUtils._setBoxPlotDefaultLineColor(boxPlotOptions, 'whisker', defaultLineColor);
    DvtChartStyleUtils._setBoxPlotDefaultLineColor(boxPlotOptions, 'whiskerEnd', defaultLineColor);
    DvtChartStyleUtils._setBoxPlotDefaultLineColor(boxPlotOptions, 'median', defaultLineColor);

    return boxPlotOptions;
  },
  /**
   * Sets the default line color for box plot shapes.
   * @param {object} boxPlotOptions The box plot style option object.
   * @param {string} prefix The option name prefix.
   * @param {string} defaultLineColor The default line color to set if the color is not user-specified.
   * @private
   */
  _setBoxPlotDefaultLineColor: (boxPlotOptions, prefix, defaultLineColor) => {
    // The deprecated *Style option name has to be merged with *SvgStyle to maintain backwards compatibility.
    // The reason is that the default options in DvtChartDefaults are only defined for *SvgStyle.
    var lineSvgStyle = JsonUtils.merge(
      boxPlotOptions[prefix + 'Style'],
      boxPlotOptions[prefix + 'SvgStyle']
    );
    boxPlotOptions[prefix + 'SvgStyle'] = lineSvgStyle;
    boxPlotOptions[prefix + 'Style'] = null; // nullify so *Style will not be applied

    // Set the default line color if the stroke is not set in the svgStyle, and if svgClassName is not set.
    if (
      lineSvgStyle &&
      !lineSvgStyle['stroke'] &&
      !boxPlotOptions[prefix + 'ClassName'] &&
      !boxPlotOptions[prefix + 'SvgClassName']
    )
      boxPlotOptions[prefix + 'SvgStyle']['stroke'] = defaultLineColor;
  },
  /**
   * Returns an object containing information about the data item used by tooltip and dataLabel callbacks.
   * @param {Chart} chart
   * @param {number} seriesIndex The series index.
   * @param {number} groupIndex The group index.
   * @param {number} itemIndex The nested item index.
   * @return {object} An object containing information about the data item.
   */
  getDataContext: (chart, seriesIndex, groupIndex, itemIndex) => {
    var seriesItem = DvtChartDataUtils.getSeriesItem(chart, seriesIndex);
    var dataItem = DvtChartDataUtils.getDataItem(chart, seriesIndex, groupIndex);
    var nestedDataItem = DvtChartDataUtils.getNestedDataItem(
      chart,
      seriesIndex,
      groupIndex,
      itemIndex
    );
    var rawOptions = chart.getRawOptions();
    var isOtherSlice = DvtChartTypeUtils.isPie(chart) && (seriesIndex == null || seriesIndex < 0);
    var chartOptions = chart.getOptions();

    var dataContext;
    if (isOtherSlice) {
      var otherStr = chartOptions.translations.labelOther;
      dataContext = {
        id: otherStr,
        series: otherStr,
        value: DvtChartPieUtils.getOtherVal(chart),
        color: chartOptions['styleDefaults']['otherColor']
      };
    } else if (nestedDataItem) {
      var rawData = rawOptions['series'][seriesIndex]['items'][groupIndex];
      if (rawData._noTemplate) {
        rawData = rawData._itemData;
      } else if (rawData && typeof rawData === 'object') {
        rawData = Object.assign({}, rawData);
        delete rawData['_itemData'];
      }
      dataContext = {
        id: DvtChartDataUtils.getNestedDataItemId(chart, seriesIndex, groupIndex, itemIndex),
        data: [rawData, rawData['items'][itemIndex]],
        value: DvtChartDataUtils.getVal(chart, seriesIndex, groupIndex, itemIndex),
        y: DvtChartDataUtils.getVal(chart, seriesIndex, groupIndex, itemIndex),
        color: DvtChartStyleUtils.getMarkerColor(chart, seriesIndex, groupIndex, itemIndex),
        itemData: chartOptions['series'][seriesIndex]['items'][groupIndex]['_itemData']
      };
    } else if (dataItem) {
      var rawDataObj = rawOptions['series'][seriesIndex]['items'][groupIndex];
      if (rawDataObj._noTemplate) {
        rawDataObj = rawDataObj._itemData;
      } else if (rawDataObj && typeof rawDataObj === 'object') {
        rawDataObj = Object.assign({}, rawDataObj);
        delete rawDataObj['_itemData'];
      }
      dataContext = {
        id: DvtChartDataUtils.getDataItemId(chart, seriesIndex, groupIndex),
        data: rawDataObj,
        value: DvtChartDataUtils.getVal(chart, seriesIndex, groupIndex),
        targetValue: DvtChartDataUtils.getTargetVal(chart, seriesIndex, groupIndex),
        x: DvtChartDataUtils.getXVal(chart, seriesIndex, groupIndex),
        y: DvtChartDataUtils.getVal(chart, seriesIndex, groupIndex),
        z: DvtChartDataUtils.getZVal(chart, seriesIndex, groupIndex),
        low: DvtChartDataUtils.getLowVal(chart, seriesIndex, groupIndex),
        high: DvtChartDataUtils.getHighVal(chart, seriesIndex, groupIndex),
        open: dataItem['open'],
        q1: dataItem['q1'],
        q2: dataItem['q2'],
        q3: dataItem['q3'],
        close: dataItem['close'],
        volume: dataItem['volume'],
        color: DvtChartStyleUtils.getColor(chart, seriesIndex, groupIndex),
        itemData: chartOptions['series'][seriesIndex]['items'][groupIndex]['_itemData']
      };
    } else if (seriesItem) {
      dataContext = {
        id: DvtChartDataUtils.getSeries(chart, seriesIndex),
        color: DvtChartStyleUtils.getColor(chart, seriesIndex)
      };
    }

    if (dataContext) {
      dataContext['component'] = chartOptions['_widgetConstructor'];

      if (isOtherSlice || nestedDataItem || dataItem) {
        dataContext['group'] = DvtChartDataUtils.getGroup(chart, groupIndex);
        dataContext['groupData'] = DvtChartDataUtils.getGroupsDataForContext(chart)[groupIndex];
      }

      if (!isOtherSlice && (nestedDataItem || dataItem || seriesItem)) {
        dataContext['series'] = DvtChartDataUtils.getSeries(chart, seriesIndex);
        dataContext['seriesData'] = DvtChartDataUtils.getSeriesDataForContext(chart, seriesIndex);
      }

      if (DvtChartTypeUtils.isPie(chart) && chart.pieChart)
        dataContext['totalValue'] = chart.pieChart.getTotalValue();

      var barDimensions = chart
        .getOptionsCache()
        .getFromCachedMap2D('barDims', seriesIndex, groupIndex);
      if (barDimensions) dataContext['dimensions'] = barDimensions;

      dataContext = chart.getCtx().fixRendererContext(dataContext);
    }

    return dataContext || {};
  },
  /**
   * Returns an object containing information about the data item used by tooltip and dataLabel callbacks.
   * @param {Chart} chart
   * @param {number} seriesIndex The series index.
   * @param {number} groupIndex The group index.
   * @param {number} itemIndex The nested item index.
   * @return {object} An object containing information about the data item.
   */
  getShortDescContext: (chart, seriesIndex, groupIndex, itemIndex) => {
    // Only data items have tooltips
    if (seriesIndex < 0 || groupIndex < 0) return null;

    var dataItem = DvtChartDataUtils.getDataItem(chart, seriesIndex, groupIndex);
    var rawOptions = chart.getRawOptions();
    var chartOptions = chart.getOptions();

    var rawData = rawOptions['series'][seriesIndex]['items'][groupIndex];
    if (rawData._noTemplate) {
      rawData = rawData._itemData;
    } else if (rawData && typeof rawData === 'object') {
      rawData = Object.assign({}, rawData);
      delete rawData['_itemData'];
    }
    return {
      id: DvtChartDataUtils.getDataItemId(chart, seriesIndex, groupIndex),
      data: rawData,
      value: DvtChartDataUtils.getVal(chart, seriesIndex, groupIndex),
      targetValue: DvtChartDataUtils.getTargetVal(chart, seriesIndex, groupIndex),
      label: DvtChartDataUtils.getDefaultDataLabel(chart, seriesIndex, groupIndex, itemIndex),
      totalValue:
        DvtChartTypeUtils.isPie(chart) && chart.pieChart ? chart.pieChart.getTotalValue() : null,
      x: DvtChartDataUtils.getXVal(chart, seriesIndex, groupIndex),
      y: DvtChartDataUtils.getVal(chart, seriesIndex, groupIndex),
      z: DvtChartDataUtils.getZVal(chart, seriesIndex, groupIndex),
      low: DvtChartDataUtils.getLowVal(chart, seriesIndex, groupIndex),
      high: DvtChartDataUtils.getHighVal(chart, seriesIndex, groupIndex),
      q1: dataItem['q1'],
      q2: dataItem['q2'],
      q3: dataItem['q3'],
      volume: dataItem['volume'],
      open: dataItem['open'],
      close: dataItem['close'],
      itemData: chartOptions['series'][seriesIndex]['items'][groupIndex]['_itemData'],
      group: DvtChartDataUtils.getGroup(chart, groupIndex),
      groupData: DvtChartDataUtils.getGroupsDataForContext(chart)[groupIndex],
      series: DvtChartDataUtils.getSeries(chart, seriesIndex),
      seriesData: DvtChartDataUtils.getSeriesDataForContext(chart, seriesIndex)
    };
  },
  /**
   * Returns the data label for the specified data point.
   * @param {Chart} chart
   * @param {number} seriesIndex The series index.
   * @param {number} groupIndex The group index.
   * @param {number} itemIndex The nested item index.
   * @param {number} type (optional) Data label type: low, high, or value.
   * @param {boolean} isStackLabel true if label for stack cummulative, false otherwise
   * @return {string} The data label, null if the index is invalid.
   */
  getDataLabel: (chart, seriesIndex, groupIndex, itemIndex, type, isStackLabel) => {
    var funcLabel;
    var defaultLabel = DvtChartDataUtils.getDefaultDataLabel(
      chart,
      seriesIndex,
      groupIndex,
      itemIndex,
      type,
      isStackLabel
    );

    // Use data Label function if there is one
    var dataLabelFunc = chart.getOptions()['dataLabel'];
    if (dataLabelFunc && !isStackLabel) {
      var dataContext = DvtChartStyleUtils.getDataContext(
        chart,
        seriesIndex,
        groupIndex,
        itemIndex
      );
      dataContext['label'] = defaultLabel;
      funcLabel = dataLabelFunc(dataContext);
      if (typeof funcLabel == 'number') {
        var valueFormat = DvtChartFormatUtils.getValueFormat(chart, 'label');
        funcLabel = DvtChartFormatUtils.formatVal(chart, valueFormat, funcLabel);
      }
    }
    if (chart.Options.stackLabelProvider && isStackLabel) {
      var stackLabelContext = DvtChartStyleUtils.getStackLabelContext(chart, groupIndex);
      funcLabel = chart.Options.stackLabelProvider(stackLabelContext);
    }

    return funcLabel ? funcLabel : defaultLabel;
  },

  /**
   * Returns an object containing information about the stack item used by stack label callbacks.
   * @param {Chart} chart
   * @return {object} An object containing information about the data item.
   */
  getStackLabelContext: (chart, groupIndex) => {
    var seriesCount = DvtChartDataUtils.getSeriesCount(chart);
    var chartOptions = chart.getOptions();
    var rawData = [];
    var itemData = [];
    var value = 0;
    var groups = DvtChartDataUtils.getGroup(chart, groupIndex);
    for (var seriesIndex = 0; seriesIndex < seriesCount; seriesIndex++) {
      var item = chartOptions['series'][seriesIndex]['items'][groupIndex];
      rawData.push(item);
      if (item) {
        itemData.push(item['_itemData']);
      }
      value += DvtChartDataUtils.getVal(chart, seriesIndex, groupIndex);
    }
    var groupData = DvtChartDataUtils.getGroupsDataForContext(chart)[groupIndex] || [];
    var stackLabelContext;
    //these all have to be arrays that get extended with each call
    stackLabelContext = {
      groups: groups,
      data: rawData,
      groupData: groupData,
      value: value,
      itemData: itemData
    };
    return stackLabelContext;
  },
  /**
   * @param {Chart} chart
   * @return {boolean} true if one of the series is centeredSegmented or centeredStepped line or area.
   */
  hasCenteredSeries: (chart) => {
    if (!DvtChartTypeUtils.isBLAC(chart)) return false;

    var seriesCount = DvtChartDataUtils.getSeriesCount(chart);
    for (var seriesIndex = 0; seriesIndex < seriesCount; seriesIndex++) {
      // Ignore the series if it isn't rendered
      if (!DvtChartDataUtils.isSeriesRendered(chart, seriesIndex)) continue;
      else if (DvtChartDataUtils.getSeriesType(chart, seriesIndex) != 'bar') {
        // line or area
        var lineType = DvtChartStyleUtils.getLineType(chart, seriesIndex);
        if (lineType == 'centeredSegmented' || lineType == 'centeredStepped') return true;
      }
    }
    return false;
  },
  /**
   * @param {Chart} chart
   * @return {boolean} true if one of the series is segmented or stepped line or area.
   */
  hasUncenteredSeries: (chart) => {
    if (!DvtChartTypeUtils.isBLAC(chart)) return false;

    var seriesCount = DvtChartDataUtils.getSeriesCount(chart);
    for (var seriesIndex = 0; seriesIndex < seriesCount; seriesIndex++) {
      // Ignore the series if it isn't rendered
      if (!DvtChartDataUtils.isSeriesRendered(chart, seriesIndex)) continue;
      else if (DvtChartDataUtils.getSeriesType(chart, seriesIndex) != 'bar') {
        // line or area
        var lineType = DvtChartStyleUtils.getLineType(chart, seriesIndex);
        if (lineType == 'segmented' || lineType == 'stepped') return true;
      }
    }
    return false;
  },

  /**
   * Returns whether data label contrast outline is supported for the chart type.
   * @param {Chart} chart
   * @return {boolean}
   */
  supportsLabelOutline: (chart, seriesIndex) => {
    var seriesType = DvtChartDataUtils.getSeriesType(chart, seriesIndex);
    if (
      seriesType === 'area' ||
      seriesType === 'line' ||
      seriesType === 'lineWithArea' ||
      DvtChartTypeUtils.isScatterBubble(chart)
    ) {
      return true;
    }
    return false;
  }
};

/**
 * Data cursor component.
 * @extends {dvt.Container}
 * @class DvtChartDataCursor  Creates a data cursor component.
 * @constructor
 * @param {dvt.Context} context The context object.
 * @param {object} options The data cursor options.
 * @param {boolean} bHoriz True if this is a data cursor for horizontal charts.
 */
class DvtChartDataCursor extends Container {
  /**
   * Initializes the data cursor.
   * @param {dvt.Context} context The context object.
   * @param {object} options The data cursor options.
   * @param {boolean} bHoriz True if this is a data cursor for horizontal charts.
   */
  constructor(context, options, bHoriz) {
    super(context);

    this._bHoriz = bHoriz;
    this._options = options;

    // Data cursor is never the target of mouse events
    this.setMouseEnabled(false);

    // Initially invisible until shown
    this.setVisible(false);

    //******************************************* Data Cursor Line ******************************************************/
    var lineWidth = options['lineWidth'];
    var lineColor = options['lineColor'];
    var stroke = new Stroke(
      lineColor,
      1,
      lineWidth,
      false,
      Stroke.getDefaultDashProps(options['lineStyle'], lineWidth)
    );

    this._cursorLine = new Line(this.getCtx(), 0, 0, 0, 0, 'dcLine');
    this._cursorLine.setStroke(stroke);
    this.addChild(this._cursorLine);

    //******************************************* Data Cursor Line Border ******************************************************/
    this._cursorOuterLine = new Line(this.getCtx(), 0, 0, 0, 0, 'dcLine');
    this._cursorOuterLine.setClassName('oj-chart-data-cursor-outer-line');
    this.addChild(this._cursorOuterLine);
    // default marker
    this.addMarker(SimpleMarker.CIRCLE, options.markerSize);
  }

  addMarker(markerShape, markerSize) {
    if (this._options.markerDisplayed !== 'off') {
      if (this._marker) {
        this.removeChild(this._marker);
      }
      this._marker = new Container(this._context);
      this._marker.setMouseEnabled(false);
      this.addChild(this._marker);

      var lineColor = this._options.lineColor;
      var lineWidth = this._options.lineWidth;
      var outerShape = new SimpleMarker(
        this._context,
        markerShape,
        0,
        0,
        markerSize + 4 * lineWidth,
        markerSize + 4 * lineWidth
      );
      outerShape.setSolidFill(lineColor);
      this._marker.addChild(outerShape);

      var middleShape = new SimpleMarker(
        this._context,
        markerShape,
        0,
        0,
        markerSize + 2 * lineWidth,
        markerSize + 2 * lineWidth
      );
      middleShape.setSolidFill('white');
      this._marker.addChild(middleShape);

      // Inner circle will be filled to correspond to the data item color
      this._markerInnerShape = new SimpleMarker(
        this._context,
        markerShape,
        0,
        0,
        markerSize,
        markerSize
      );
      this._marker.addChild(this._markerInnerShape);
    }
  }

  /**
   * Renders this data cursor.
   * @param {dvt.Rectangle} plotAreaBounds The bounds of the plot area.
   * @param {number} dataX The x coordinate of the actual data point, where the marker should be placed.
   * @param {number} dataY The y coordinate of the actual data point, where the marker should be placed.
   * @param {number} lineCoord The x coordinate of a vertical data cursor, or the y coordinate of a horizontal data cursor.
   * @param {string} text The text for the datatip.
   * @param {string} dataColor The primary color of the associated data item.
   */
  render(plotAreaBounds, dataX, dataY, lineCoord, text, dataColor) {
    var bHoriz = this.isHorizontal();
    var bRtl = Agent.isRightToLeft(this.getCtx());
    var tooltipBounds;

    // convert local coords to scaled stage coord for tooltip positioning
    // however datacursor and marker position are in local coords for svg rendering.

    var scaledPlotXY = this._parent.localToStage(new Point(plotAreaBounds.x, plotAreaBounds.y));
    var scaledPlotX2Y2 = this._parent.localToStage(
      new Point(plotAreaBounds.x + plotAreaBounds.w, plotAreaBounds.y + plotAreaBounds.h)
    );
    var scaledDataXY = this._parent.localToStage(new Point(dataX, dataY));
    var scaledCoord = this._parent.localToStage(
      new Point(bHoriz ? 0 : lineCoord, bHoriz ? lineCoord : 0)
    );

    var scaledLineCoord = bHoriz ? scaledCoord.y : scaledCoord.x;

    if (text != null && text != '') {
      // First render the datatip to retrieve its size.
      var stagePageCoords = this.getCtx().getStageAbsolutePosition();
      var tooltipManager = this.getCtx().getTooltipManager(DvtChartDataCursor.TOOLTIP_ID);
      tooltipManager.showDatatip(
        scaledDataXY.x + stagePageCoords.x,
        scaledDataXY.y + stagePageCoords.y,
        text,
        dataColor,
        false
      );
      tooltipBounds = tooltipManager.getTooltipBounds(); // tooltipBounds is in the page coordinate space

      // Then reposition to the right location
      var markerSizeOuter = this._options['markerSize'] + 4 * this._options['lineWidth'];
      var tooltipX, tooltipY; // tooltipX and tooltipY in the stage coordinate space
      if (bHoriz) {
        tooltipX = bRtl
          ? scaledPlotXY.x - 0.75 * tooltipBounds.w
          : scaledPlotXY.x + Math.abs(scaledPlotXY.x - scaledPlotX2Y2.x) - tooltipBounds.w / 4;
        tooltipY = scaledLineCoord - tooltipBounds.h / 2;

        // Add a buffer between the tooltip and data point. This may be rejected in positionTip due to viewport location.
        if (!bRtl && tooltipX - scaledDataXY.x < markerSizeOuter)
          tooltipX = scaledDataXY.x + markerSizeOuter;
        else if (bRtl && scaledDataXY.x - (tooltipX + tooltipBounds.w) < markerSizeOuter)
          tooltipX = scaledDataXY.x - markerSizeOuter - tooltipBounds.w;
      } else {
        tooltipX = scaledLineCoord - tooltipBounds.w / 2;
        tooltipY = scaledPlotXY.y - 0.75 * tooltipBounds.h;

        // Add a buffer between the tooltip and data point. This may be rejected in positionTip due to viewport location.
        if (dataY - (tooltipY + tooltipBounds.h) < markerSizeOuter)
          tooltipY = dataY - markerSizeOuter - tooltipBounds.h;
      }
      tooltipManager.positionTip(tooltipX + stagePageCoords.x, tooltipY + stagePageCoords.y);
      // Finally retrieve the rendered bounds to calculate the attachment point for the cursor line.
      tooltipBounds = tooltipManager.getTooltipBounds(); // tooltipBounds is in the page coordinate space
      tooltipBounds.x -= stagePageCoords.x;
      tooltipBounds.y -= stagePageCoords.y;

      // adjust the tooltipBounds according to the scale so that data cursor length is correct.
      var x1y1 = this._parent.stageToLocal(new Point(tooltipBounds.x, tooltipBounds.y));
      var x2y2 = this._parent.stageToLocal(
        new Point(tooltipBounds.x + tooltipBounds.w, tooltipBounds.y + tooltipBounds.h)
      );
      tooltipBounds.x = x1y1.x;
      tooltipBounds.y = x1y1.y;
      tooltipBounds.w = Math.abs(x2y2.x - x1y1.x);
      tooltipBounds.h = Math.abs(x2y2.y - x1y1.y);
    }

    // Position the cursor line. Use 1px fudge factor to ensure that the line connects to the tooltip.
    var cursorLineWidth = this._cursorLine.getStroke().getWidth();
    var cursorOuterLineCoord = lineCoord + Math.ceil(cursorLineWidth / 2);
    if (bHoriz) {
      this._cursorLine.setTranslateY(lineCoord);
      this._cursorOuterLine.setTranslateY(cursorOuterLineCoord);
      var x1Pos, x2Pos;
      if (bRtl) {
        x1Pos = tooltipBounds ? tooltipBounds.x + tooltipBounds.w - 1 : plotAreaBounds.x;
        x2Pos = plotAreaBounds.x + plotAreaBounds.w;
      } else {
        x1Pos = plotAreaBounds.x;
        const plotAreaEnd = plotAreaBounds.x + plotAreaBounds.w;
        x2Pos = Math.min(tooltipBounds ? tooltipBounds.x + 1 : plotAreaEnd, plotAreaEnd);
      }
      this._cursorLine.setX1(x1Pos);
      this._cursorLine.setX2(x2Pos);
      this._cursorOuterLine.setX1(x1Pos);
      this._cursorOuterLine.setX2(x2Pos);
    } else {
      // Vertical
      this._cursorLine.setTranslateX(lineCoord);
      this._cursorOuterLine.setTranslateX(cursorOuterLineCoord);

      // Position the cursor line
      var y1Pos = tooltipBounds ? tooltipBounds.y + tooltipBounds.h - 1 : plotAreaBounds.y;
      var y2Pos = plotAreaBounds.y + plotAreaBounds.h;
      this._cursorLine.setY1(y1Pos);
      this._cursorLine.setY2(y2Pos);
      this._cursorOuterLine.setY1(y1Pos);
      this._cursorOuterLine.setY2(y2Pos);
    }

    if (this._marker) {
      // Position the marker
      this._marker.setTranslate(dataX, dataY);

      // Set the marker color
      var markerColor = this._options['markerColor'];
      this._markerInnerShape.setSolidFill(markerColor ? markerColor : dataColor);

      // : Workaround firefox issue
      Agent.workaroundFirefoxRepaint(this._marker);
    }
  }

  /**
   * Returns true if this is a data cursor for a horizontal graph.
   * @return {boolean}
   */
  isHorizontal() {
    return this._bHoriz;
  }

  /**
   * Returns the behavior of the data cursor.
   * @return {string}
   */
  getBehavior() {
    return this._behavior ? this._behavior : 'auto';
  }

  /**
   * Specifies the behavior of the data cursor.
   * @param {string} behavior
   */
  setBehavior(behavior) {
    this._behavior = behavior;
  }
}

DvtChartDataCursor.TOOLTIP_ID = '_dvtDataCursor';

/**
 * Reference object related utility functions for Chart.
 * @class
 */
const DvtChartRefObjUtils = {
  /**
   * Returns the type of the reference object.
   * @param {object} refObj The reference object definition.
   * @return {string} The type of the reference object.
   */
  getType: (refObj) => {
    if (refObj['type'] == 'area') return 'area';
    // default to "line"
    else return 'line';
  },

  /**
   * Returns the location of the reference object.
   * @param {object} refObj The reference object definition.
   * @return {string} The location of the reference object.
   */
  getLocation: (refObj) => {
    if (refObj['location'] == 'front') return 'front';
    // default to "back"
    else return 'back';
  },

  /**
   * Returns the color of the reference object.
   * @param {Chart} chart
   * @param {object} refObj The reference object definition.
   * @return {string} The color.
   */
  getColor: (chart, refObj) => {
    if (refObj['color']) return refObj['color'];
    else
      return DvtChartRefObjUtils.getType(refObj) === 'line'
        ? chart.getOptions()['_defaultReferenceObjectLineColor']
        : chart.getOptions()['_defaultReferenceObjectAreaColor'];
  },

  /**
   * Returns the line width of the reference line.
   * @param {Chart} chart
   * @param {object} refObj The reference object definition.
   * @return {number} The line width.
   */
  getLineWidth: (chart, refObj) => {
    if (refObj['lineWidth']) return refObj['lineWidth'];
    else return parseFloat(chart.getOptions()['_defaultReferenceObjectLineWidth']);
  },
  /**
   * Returns the line type of the reference line.
   * @param {object} refObj The reference object definition.
   * @return {number} The line type.
   */
  getLineType: (refObj) => {
    if (refObj['lineType']) return refObj['lineType'];
    else return 'straight';
  },
  /**
   * Returns true if the specified reference object should be rendered.
   * @param {Chart} chart
   * @param {object} refObj
   * @return {boolean}
   */
  isObjRendered: (chart, refObj) => {
    var hiddenCategories = DvtChartDataUtils.getHiddenCategories(chart);
    if (hiddenCategories.length > 0) {
      var categories = DvtChartRefObjUtils.getRefObjCategories(refObj);
      if (categories && ArrayUtils.hasAnyItem(hiddenCategories, categories)) {
        return false;
      }
    }
    return refObj['visibility'] !== 'hidden';
  },

  /**
   * Returns the id of the reference object.
   * @param {object} refObj
   * @return {string}
   */
  getId: (refObj) => {
    return refObj['id'] != null ? refObj['id'] : refObj['text'];
  },
  /**
   * Returns the categories of the reference object.
   * @param {object} refObj
   * @return {string}
   */
  getRefObjCategories: (refObj) => {
    return refObj['categories'] ? refObj['categories'] : [DvtChartRefObjUtils.getId(refObj)];
  },
  /**
   * Returns reference object based on id.
   * @param {object} chart
   * @param {string} id The id of the ref obj.
   * @return {object}
   */
  getRefObj: (chart, id) => {
    var refObjs = DvtChartDataUtils.getRefObjs(chart);
    for (var i = 0; i < refObjs.length; i++) {
      if (DvtChartRefObjUtils.getId(refObjs[i]) == id) {
        return refObjs[i];
      }
    }
    return undefined;
  },

  /**
   * Returns the low value of the refObj item.
   * @param {object} item
   * @return {number}
   */
  getLowVal: (item) => {
    if (item == null) return null;
    return item['low'];
  },
  /**
   * Returns the high value of the refObj item.
   * @param {object} item
   * @return {number}
   */
  getHighVal: (item) => {
    if (item == null) return null;
    return item['high'];
  },
  /**
   * Retuns the x value of the refObj item at the given index
   * @param {Chart} chart
   * @param {object} items
   * @param {number} index
   * @return {number}
   */
  getXVal: (chart, items, index) => {
    return DvtChartDataUtils.getXValFromItem(chart, items[index], index);
  }
};

/**
 * Utility functions for Chart.
 * @class
 */
const DvtChartTooltipUtils = {
  /**
   * Returns the datatip color for the tooltip of a data item with the given series
   * and group indices.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @param {number} itemIndex
   * @return {string} The datatip color.
   */
  getDatatipColor: (chart, seriesIndex, groupIndex, itemIndex) => {
    if (DvtChartTypeUtils.isStock(chart)) return DvtChartStyleUtils.getColor(chart, 0, groupIndex);

    if (itemIndex != null && itemIndex >= 0)
      // nested item
      return DvtChartStyleUtils.getMarkerColor(chart, seriesIndex, groupIndex, itemIndex);

    return DvtChartStyleUtils.getColor(chart, seriesIndex, groupIndex);
  },

  /**
   * Returns the datatip string for a data item with the given series and group indices.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @param {number} itemIndex
   * @param {boolean} isTabular Whether the datatip is in a table format.
   * @return {string|Node|Array<Node>} The datatip string.
   */
  getDatatip: (chart, seriesIndex, groupIndex, itemIndex, isTabular) => {
    if (DvtChartTypeUtils.isSpark(chart) || DvtChartTypeUtils.isOverview(chart)) return null;

    // Only data items have tooltips
    if (seriesIndex < 0 || groupIndex < 0) return null;

    // Custom Tooltip via Function
    var customTooltip = chart.getOptions()['tooltip'];
    var tooltipFunc = customTooltip ? customTooltip['renderer'] : null;

    if (isTabular && tooltipFunc) {
      var tooltipManager = chart
        .getCtx()
        .getTooltipManager(
          DvtChartTooltipUtils.isDataCursorEnabled(chart) ? DvtChartDataCursor.TOOLTIP_ID : null
        );
      var dataContext = DvtChartStyleUtils.getDataContext(
        chart,
        seriesIndex,
        groupIndex,
        itemIndex
      );

      // Get customized labels
      if (DvtChartTypeUtils.isPie(chart)) {
        var slice = DvtChartPieUtils.getSliceBySeriesIdx(chart, seriesIndex);
        dataContext['label'] = slice.getSliceLabelString();
      } else {
        dataContext['label'] = DvtChartStyleUtils.getDataLabel(
          chart,
          seriesIndex,
          groupIndex,
          itemIndex
        );
      }

      return tooltipManager.getCustomTooltip(tooltipFunc, dataContext);
    }

    // Custom Tooltip Support for nested item
    var dataItem = DvtChartDataUtils.getNestedDataItem(chart, seriesIndex, groupIndex, itemIndex);
    // Custom Tooltip via Short Desc
    if (dataItem && dataItem.shortDesc != null) {
      return dataItem.shortDesc;
    }
    // check in data item
    var parentItem = DvtChartDataUtils.getDataItem(chart, seriesIndex, groupIndex);
    if (parentItem && parentItem.shortDesc != null) {
      return Displayable.resolveShortDesc(parentItem.shortDesc, () =>
        DvtChartStyleUtils.getShortDescContext(chart, seriesIndex, groupIndex, itemIndex)
      );
    }

    // Default Tooltip Support
    var datatipRows;
    if (DvtChartTypeUtils.isStock(chart))
      datatipRows = DvtChartTooltipUtils._getStockDatatip(chart, 0, groupIndex, isTabular);
    else {
      datatipRows = [];
      DvtChartTooltipUtils._addSeriesDatatip(
        datatipRows,
        chart,
        seriesIndex,
        groupIndex,
        isTabular
      );
      DvtChartTooltipUtils._addGroupDatatip(datatipRows, chart, seriesIndex, groupIndex, isTabular);
      DvtChartTooltipUtils._addValDatatip(
        datatipRows,
        chart,
        seriesIndex,
        groupIndex,
        itemIndex,
        isTabular
      );
    }

    return DvtChartTooltipUtils._processDatatip(datatipRows, chart, isTabular);
  },

  /**
   * Returns the datatip string for an "Other" slice.
   * @param {Chart} chart
   * @param {number} otherValue The value of the "Other" slice
   * @param {boolean} isTabular Whether the datatip is in a table format.
   * @return {string|Node|Array<Node>} The datatip string.
   */
  getOtherSliceDatatip: (chart, otherValue, isTabular) => {
    var otherStr = chart.getOptions().translations.labelOther;

    // Custom Tooltip via Function
    var customTooltip = chart.getOptions()['tooltip'];
    var tooltipFunc = customTooltip ? customTooltip['renderer'] : null;
    if (isTabular && tooltipFunc) {
      var slice = DvtChartPieUtils.getSliceBySeriesIdx(chart, null);
      var dataContext = DvtChartStyleUtils.getDataContext(chart, null, 0);
      dataContext['label'] = slice.getSliceLabelString();
      return chart.getCtx().getTooltipManager().getCustomTooltip(tooltipFunc, dataContext);
    }

    // Default Tooltip
    var datatipRows = [];
    DvtChartTooltipUtils._addDatatipRow(
      datatipRows,
      chart,
      'series',
      'labelSeries',
      otherStr,
      isTabular
    );
    DvtChartTooltipUtils._addGroupDatatip(datatipRows, chart, 0, 0, isTabular);
    DvtChartTooltipUtils._addDatatipRow(
      datatipRows,
      chart,
      'value',
      'labelValue',
      otherValue,
      isTabular
    );
    return DvtChartTooltipUtils._processDatatip(datatipRows, chart, isTabular);
  },

  /**
   * Final processing for the datatip.
   * @param {Array<string|Node>} datatipRows The current datatip.
   * @param {Chart} chart The owning chart instance.
   * @param {boolean} isTabular Whether the datatip is in a table format.
   * @return {string|Node} The processed datatip.
   * @private
   */
  _processDatatip: (datatipRows, chart, isTabular) => {
    // Don't render tooltip if empty
    if (datatipRows.length === 0) return null;

    // Add outer table tags
    if (isTabular)
      return HtmlTooltipManager.createElement('table', null, datatipRows, [
        'oj-dvt-datatip-table'
      ]);
    else return datatipRows.join('');
  },

  /**
   * Returns the tooltip for the reference object.
   * @param {Chart} chart
   * @param {object} refObj The reference object definition.
   * @param {object} axisType The type of axis could be 'yAxis', 'y2Axis' or 'xAxis'.
   * @param {object} index The index of the reference object in the axis.
   * @return {string} The tooltip for the reference object.
   */
  getRefObjTooltip: (chart, refObj, axisType, index) => {
    // Custom Tooltip via Function -- only if refObj['id'] is defined for backwards compat
    var customTooltip = chart.getOptions()['tooltip'];
    var tooltipFunc = customTooltip ? customTooltip['renderer'] : null;
    if (tooltipFunc && refObj['id'] != null) {
      var tooltipManager = chart
        .getCtx()
        .getTooltipManager(
          DvtChartTooltipUtils.isDataCursorEnabled(chart) ? DvtChartDataCursor.TOOLTIP_ID : null
        );
      var dataContext = {
        id: DvtChartRefObjUtils.getId(refObj),
        label: refObj['text'],
        data: chart.getRawOptions()[axisType]['referenceObjects'][index],
        value: refObj['value'],
        low: DvtChartRefObjUtils.getLowVal(refObj),
        high: DvtChartRefObjUtils.getHighVal(refObj),
        color: DvtChartRefObjUtils.getColor(chart, refObj)
      };
      return tooltipManager.getCustomTooltip(tooltipFunc, dataContext);
    }

    return refObj['shortDesc'];
  },
  /**
   * Returns the datatip for a data item with the given series and group indices.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @param {boolean} isTabular Whether the datatip is in a table format.
   * @return {Array<string|Node>} The datatip string.
   * @private
   */
  _getStockDatatip: (chart, seriesIndex, groupIndex, isTabular) => {
    var dataItem = DvtChartDataUtils.getDataItem(chart, seriesIndex, groupIndex);

    // Default Tooltip
    var datatipRows = [];
    DvtChartTooltipUtils._addGroupDatatip(datatipRows, chart, seriesIndex, groupIndex, isTabular);
    if (dataItem) {
      DvtChartTooltipUtils._addDatatipRow(
        datatipRows,
        chart,
        'open',
        'labelOpen',
        dataItem['open'],
        isTabular
      );
      DvtChartTooltipUtils._addDatatipRow(
        datatipRows,
        chart,
        'close',
        'labelClose',
        dataItem['close'],
        isTabular
      );
      DvtChartTooltipUtils._addDatatipRow(
        datatipRows,
        chart,
        'high',
        'labelHigh',
        dataItem['high'],
        isTabular
      );
      DvtChartTooltipUtils._addDatatipRow(
        datatipRows,
        chart,
        'low',
        'labelLow',
        dataItem['low'],
        isTabular
      );
      DvtChartTooltipUtils._addDatatipRow(
        datatipRows,
        chart,
        'volume',
        'labelVolume',
        dataItem['volume'],
        isTabular
      );
    }
    return datatipRows;
  },
  /**
   * Adds the series string to the datatip.
   * @param {Array<string|Node>} datatipRows The current datatip. This array will be mutated.
   * @param {Chart} chart The owning chart instance.
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @param {boolean} isTabular Whether the datatip is in a table format.
   * @private
   */
  _addSeriesDatatip: (datatipRows, chart, seriesIndex, groupIndex, isTabular) => {
    var seriesLabel = DvtChartDataUtils.getSeriesLabel(chart, seriesIndex);
    DvtChartTooltipUtils._addDatatipRow(
      datatipRows,
      chart,
      'series',
      'labelSeries',
      seriesLabel,
      isTabular
    );
  },
  /**
   * Adds the group string to the datatip.
   * @param {Array<string|Node>} datatipRows The current datatip. This array will be mutated.
   * @param {Chart} chart The owning chart instance.
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @param {boolean} isTabular Whether the datatip is in a table format.
   * @private
   */
  _addGroupDatatip: (datatipRows, chart, seriesIndex, groupIndex, isTabular) => {
    var groupLabel;
    if (DvtChartTypeUtils.hasTimeAxis(chart)) {
      var valueFormat = DvtChartFormatUtils$1.getValueFormat(chart, 'group');
      var value = DvtChartDataUtils.getXVal(chart, seriesIndex, groupIndex);
      groupLabel = DvtChartFormatUtils$1.formatDateVal(valueFormat, value);
      if (groupLabel == null) groupLabel = chart.xAxis.getInfo().formatLabel(value);
    } else groupLabel = DvtChartDataUtils.getGroupLabel(chart, groupIndex);

    var numLevels = DvtChartDataUtils.getNumLevels(chart);
    var defaultLabel = 'labelGroup';
    if (numLevels == 1 || !Array.isArray(groupLabel))
      DvtChartTooltipUtils._addDatatipRow(
        datatipRows,
        chart,
        'group',
        defaultLabel,
        groupLabel,
        isTabular
      );
    else {
      // hierarchical groups
      for (var levelIndex = numLevels - 1; levelIndex >= 0; levelIndex--) {
        DvtChartTooltipUtils._addDatatipRow(
          datatipRows,
          chart,
          'group',
          defaultLabel,
          groupLabel[levelIndex],
          isTabular,
          levelIndex
        );
        if (groupLabel[levelIndex]) defaultLabel = null;
      }
    }
  },

  /**
   * Adds the value string to the datatip.
   * @param {Array<string|Node>} datatipRows The current datatip. This array will be mutated.
   * @param {Chart} chart The owning chart instance.
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @param {number} itemIndex
   * @param {boolean} isTabular Whether the datatip is in a table format.
   * @private
   */
  _addValDatatip: (datatipRows, chart, seriesIndex, groupIndex, itemIndex, isTabular) => {
    var val = DvtChartDataUtils.getVal(chart, seriesIndex, groupIndex, itemIndex);
    var xVal = DvtChartDataUtils.getXVal(chart, seriesIndex, groupIndex);
    var zVal = DvtChartDataUtils.getZVal(chart, seriesIndex, groupIndex);
    var lowVal = DvtChartDataUtils.getLowVal(chart, seriesIndex, groupIndex);
    var highVal = DvtChartDataUtils.getHighVal(chart, seriesIndex, groupIndex);
    var isNested = itemIndex != null && itemIndex >= 0;

    if (DvtChartTypeUtils.isScatterBubble(chart)) {
      // Add the x and y values
      DvtChartTooltipUtils._addDatatipRow(datatipRows, chart, 'x', 'labelX', xVal, isTabular);
      DvtChartTooltipUtils._addDatatipRow(datatipRows, chart, 'y', 'labelY', val, isTabular);

      // Also add the z value for a bubble chart
      if (DvtChartTypeUtils.isBubble(chart))
        DvtChartTooltipUtils._addDatatipRow(datatipRows, chart, 'z', 'labelZ', zVal, isTabular);
    } else if (DvtChartTypeUtils.isPie(chart) || DvtChartTypeUtils.isPyramid(chart)) {
      DvtChartTooltipUtils._addDatatipRow(
        datatipRows,
        chart,
        'value',
        'labelValue',
        val,
        isTabular
      );
    } else if (DvtChartTypeUtils.isFunnel(chart)) {
      DvtChartTooltipUtils._addDatatipRow(
        datatipRows,
        chart,
        'value',
        'labelValue',
        val,
        isTabular
      );
      var target = DvtChartDataUtils.getTargetVal(chart, seriesIndex);
      if (target != null)
        DvtChartTooltipUtils._addDatatipRow(
          datatipRows,
          chart,
          'targetValue',
          'labelTargetValue',
          target,
          isTabular
        );
    } else if (DvtChartDataUtils.getSeriesType(chart, seriesIndex) == 'boxPlot' && !isNested) {
      var dataItem = DvtChartDataUtils.getDataItem(chart, seriesIndex, groupIndex);
      DvtChartTooltipUtils._addDatatipRow(
        datatipRows,
        chart,
        'high',
        'labelHigh',
        highVal,
        isTabular
      );
      DvtChartTooltipUtils._addDatatipRow(
        datatipRows,
        chart,
        'q3',
        'labelQ3',
        dataItem['q3'],
        isTabular
      );
      DvtChartTooltipUtils._addDatatipRow(
        datatipRows,
        chart,
        'q2',
        'labelQ2',
        dataItem['q2'],
        isTabular
      );
      DvtChartTooltipUtils._addDatatipRow(
        datatipRows,
        chart,
        'q1',
        'labelQ1',
        dataItem['q1'],
        isTabular
      );
      DvtChartTooltipUtils._addDatatipRow(datatipRows, chart, 'low', 'labelLow', lowVal, isTabular);

      if (zVal != null)
        DvtChartTooltipUtils._addDatatipRow(datatipRows, chart, 'z', 'labelZ', zVal, isTabular);
    } else if (DvtChartTypeUtils.isBLAC(chart)) {
      var type = DvtChartDataUtils.isAssignedToY2(chart, seriesIndex) ? 'y2' : 'y';

      // Ad min/max for range bar/area. Add z-value if defined for bar width.
      if ((lowVal != null || highVal != null) && !isNested) {
        DvtChartTooltipUtils._addDatatipRow(
          datatipRows,
          chart,
          'high',
          'labelHigh',
          highVal,
          isTabular
        );
        DvtChartTooltipUtils._addDatatipRow(
          datatipRows,
          chart,
          'low',
          'labelLow',
          lowVal,
          isTabular
        );
        if (zVal != null)
          DvtChartTooltipUtils._addDatatipRow(datatipRows, chart, 'z', 'labelZ', zVal, isTabular);
      } else if (zVal != null && !isNested) {
        DvtChartTooltipUtils._addDatatipRow(datatipRows, chart, type, 'labelY', val, isTabular);
        DvtChartTooltipUtils._addDatatipRow(datatipRows, chart, 'z', 'labelZ', zVal, isTabular);
      } else
        DvtChartTooltipUtils._addDatatipRow(datatipRows, chart, type, 'labelValue', val, isTabular);
    }
  },

  /**
   * Adds a row of item to the datatip.
   * @param {Array<string|Node>} datatipRows The current datatip. This array will be mutated.
   * @param {Chart} chart The owning chart instance.
   * @param {number} type The item type, e.g. series, group, x, etc.
   * @param {number} defaultLabel The bundle resource string for the default label.
   * @param {number} value The item value.
   * @param {boolean} isTabular Whether the datatip is in a table format.
   * @param {number} index (optional) The index of the tooltipLabel string to be used
   * @private
   */
  _addDatatipRow: (datatipRows, chart, type, defaultLabel, value, isTabular, index) => {
    if (value == null || value === '') return;

    var options = chart.getOptions()['styleDefaults'];
    var valueFormat = DvtChartFormatUtils$1.getValueFormat(chart, type);
    var tooltipDisplay = valueFormat['tooltipDisplay'];
    var translations = chart.getOptions().translations;

    if (!tooltipDisplay || tooltipDisplay == 'auto') {
      if (type == 'group' && (DvtChartTypeUtils.isPie(chart) || DvtChartTypeUtils.isFunnel(chart)))
        tooltipDisplay = 'off';
    }

    if (tooltipDisplay == 'off') return;

    // Create tooltip label
    var tooltipLabel;
    if (typeof valueFormat['tooltipLabel'] === 'string') tooltipLabel = valueFormat['tooltipLabel'];
    else if (Array.isArray(valueFormat['tooltipLabel']))
      tooltipLabel = valueFormat['tooltipLabel'][index ? index : 0];

    if (tooltipLabel == null) {
      if (defaultLabel == null)
        // non-innermost hierarchical group labels
        tooltipLabel = '';
      else {
        // : Use date instead of group for time axis chart tooltips. Only doing this for JET right now until
        // 1.1.1, after which we use the options.translations across fwks.
        if (defaultLabel == 'labelGroup' && DvtChartTypeUtils.hasTimeAxis(chart))
          tooltipLabel = translations.labelDate;
        else tooltipLabel = translations[defaultLabel];
      }
    }

    // Create tooltip value
    if (type != 'series' && type != 'group')
      value = DvtChartFormatUtils$1.formatVal(chart, valueFormat, value);

    if (isTabular) {
      var tds = [
        HtmlTooltipManager.createElement('td', options['tooltipLabelStyle'], tooltipLabel, [
          'oj-dvt-datatip-label'
        ]),
        HtmlTooltipManager.createElement('td', options['tooltipValueStyle'], value, [
          'oj-dvt-datatip-value'
        ])
      ];
      datatipRows.push(HtmlTooltipManager.createElement('tr', null, tds));
    } else {
      datatipRows.push(
        (datatipRows.length > 0 ? '<br>' : '') +
          ResourceUtils.format(translations.labelAndValue, [tooltipLabel, value])
      );
    }
  },

  /**
   * Returns whether or not the data cursor is enabled
   * @param {Chart} chart
   * @return {boolean}
   */
  isDataCursorEnabled: (chart) => {
    if (
      DvtChartTypeUtils.isPie(chart) ||
      DvtChartTypeUtils.isFunnel(chart) ||
      DvtChartTypeUtils.isPolar(chart) ||
      DvtChartTypeUtils.isPyramid(chart) ||
      DvtChartDataUtils.hasInvalidData(chart)
    ) {
      return false;
    }

    var options = chart.getOptions();
    if (options['dataCursor'] == 'on') return true;
    if (options['dataCursor'] == 'off') return false;

    // auto
    return Agent.isTouchDevice() && DvtChartTypeUtils.isLineArea(chart);
  },
  /**
   * Returns the data cursor behavior
   * @param {Chart} chart
   * @return {string}
   */
  getDataCursorBehavior: (chart) => {
    var dataCursorBehavior = chart.getOptions()['dataCursorBehavior'];

    if (dataCursorBehavior !== 'auto') {
      return dataCursorBehavior;
    }
    // auto
    return DvtChartTypeUtils.isLineArea(chart) ? 'smooth' : 'snap';
  }
};

/**
 * Logical object for chart data object displayables.
 * @param {Chart} chart The owning chart instance.
 * @param {array} displayables The array of associated DvtDisplayables.
 * @param {number} seriesIndex
 * @param {number} groupIndex
 * @param {number} itemIndex
 * @param {dvt.Point} dataPos The coordinate of the data point relative to the plot area
 * @class
 * @constructor
 * @implements {DvtCategoricalObject}
 * @implements {DvtLogicalObject}
 * @implements {DvtSelectable}
 * @implements {DvtTooltipSource}
 * @implements {DvtDraggable}
 */
class DvtChartObjPeer {
  constructor(chart, displayables, seriesIndex, groupIndex, itemIndex, dataPos) {
    /**
     * @param {Chart} chart The owning chart instance.
     * @param {array} displayables The array of associated DvtDisplayables.
     * @param {number} seriesIndex
     * @param {number} groupIndex
     * @param {number} itemIndex
     * @param {dvt.Point} dataPos The coordinate of the data point relative to the plot area
     */
    this._chart = chart;
    this._displayables = displayables;
    this._seriesIndex = seriesIndex != null && seriesIndex >= 0 ? seriesIndex : -1;
    this._groupIndex = groupIndex != null && groupIndex >= 0 ? groupIndex : -1;
    this._itemIndex = itemIndex != null && itemIndex >= 0 ? itemIndex : -1;
    this._dataPos = dataPos;
    this._isSelected = false;
    this._isShowingKeyboardFocusEffect = false;

    // . Need to evaluate these up front because the series and group are used for animation
    this._series = DvtChartDataUtils.getSeries(chart, seriesIndex);
    this._group = DvtChartDataUtils.getGroup(chart, groupIndex);

    // Create the array specifying all categories that this data item or series belongs to
    this._categories = DvtChartDataUtils.getCategories(chart, seriesIndex, groupIndex, itemIndex);

    if (this._itemIndex != -1) {
      this._dataItemId = DvtChartDataUtils.getNestedDataItemId(
        chart,
        seriesIndex,
        groupIndex,
        itemIndex
      );
      this._drillable = DvtChartDataUtils.isDataItemDrillable(
        chart,
        seriesIndex,
        groupIndex,
        itemIndex
      );
    } else if (this._groupIndex != -1) {
      var dataItem = DvtChartDataUtils.getDataItem(chart, seriesIndex, groupIndex);
      if (dataItem) {
        this._dataItemId = DvtChartDataUtils.getDataItemId(chart, seriesIndex, groupIndex);
        this._drillable = DvtChartDataUtils.isDataItemDrillable(chart, seriesIndex, groupIndex);
      }
    } else {
      var seriesItem = DvtChartDataUtils.getSeriesItem(chart, seriesIndex);
      if (seriesItem) {
        this._drillable = DvtChartDataUtils.isSeriesDrillable(chart, seriesIndex);
      }
    }

    // Apply the cursor for drilling if specified
    if (this._drillable) {
      for (var i = 0; i < this._displayables.length; i++) {
        this._displayables[i].setCursor(SelectionEffectUtils.getSelectingCursor());
      }
    }

    // Apply the aria properties
    for (var index = 0; index < displayables.length; index++) {
      var displayable = displayables[index];
      // lines are not interactive so we shouldn't add wai-aria attributes
      if (!(displayable.chartShapeType === 'lineArea')) displayable.setAriaRole('img');
      this._updateAriaLabel(displayable);
    }
  }

  /**
   * Creates a data item to identify the specified displayable and registers it with the chart.
   * @param {dvt.Displayable} displayable The displayable to associate.
   * @param {Chart} chart The owning chart instance.
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @param {number} itemIndex The index of nested data item.
   * @param {dvt.Point} dataPos The coordinate of the data point relative to the plot area.
   */
  static associate(displayable, chart, seriesIndex, groupIndex, itemIndex, dataPos) {
    // Associate is expensive and not needed during zoom & scroll.
    if (!displayable || chart.getOptions()['_duringZoomAndScroll']) return;

    // Create the logical object.
    var identObj = new DvtChartObjPeer(
      chart,
      [displayable],
      seriesIndex,
      groupIndex,
      itemIndex,
      dataPos
    );

    // Register with the chart
    chart.registerObject(identObj);

    // Finally associate using the event manager
    chart.getEventManager().associate(displayable, identObj);
  }

  /**
   * @override
   */
  getId() {
    if (this._seriesIndex >= 0 && this._groupIndex >= 0)
      return new DvtChartDataItem(
        this._dataItemId,
        this.getSeries(),
        this.getGroup(),
        this._chart.getCtx()
      );
    else if (this._seriesIndex >= 0) return this.getSeries();
    else return null;
  }

  /**
   * Return the peer's data item id.  This is an optional id that is provided to simplifying
   * row key support for ADF and AMX.
   * @return {string} the peer's row key.
   */
  getDataItemId() {
    return this._dataItemId;
  }

  /**
   * Return the peer's nested data item index.
   * @return {number}
   */
  getNestedDataItemIndex() {
    return this._itemIndex;
  }

  /**
   * Return the peer's series.
   * @return {string} the peer's series.
   */
  getSeries() {
    return this._series;
  }

  /**
   * Return the peer's series index.
   * @return {Number} the peer's series index.
   */
  getSeriesIndex() {
    return this._seriesIndex;
  }

  /**
   * Return the peer's group.
   * @return {string} the peer's group.
   */
  getGroup() {
    return this._group;
  }

  /**
   * Return the peer's group index.
   * @return {Number} the peer's group index.
   */
  getGroupIndex() {
    return this._groupIndex;
  }

  /**
   * Returns whether the chart object is drillable
   * @return {boolean}
   */
  isDrillable() {
    return this._drillable;
  }

  /**
   * Returns whether the chart object is double clickable.
   * @return {boolean}
   */
  isDoubleClickable() {
    // : IE double clicking workaround in dvt.EventManager.
    return this.isSelectable() && this.isDrillable();
  }

  /**
   * Convenience function to return the peer's chart.
   * @return {Chart} the associated chart object.
   */
  getChart() {
    return this._chart;
  }

  //---------------------------------------------------------------------//
  // Tooltip Support: DvtTooltipSource impl                              //
  //---------------------------------------------------------------------//

  /**
   * @override
   */
  getDatatip() {
    return DvtChartTooltipUtils.getDatatip(
      this._chart,
      this._seriesIndex,
      this._groupIndex,
      this._itemIndex,
      true
    );
  }

  /**
   * @override
   */
  getDatatipColor() {
    return DvtChartTooltipUtils.getDatatipColor(
      this._chart,
      this._seriesIndex,
      this._groupIndex,
      this._itemIndex
    );
  }

  //---------------------------------------------------------------------//
  // Selection Support: DvtSelectable impl                               //
  //---------------------------------------------------------------------//

  /**
   * @override
   */
  isSelectable() {
    return DvtChartStyleUtils.isSelectable(
      this.getChart(),
      this.getSeriesIndex(),
      this.getGroupIndex()
    );
  }

  /**
   * @override
   */
  isSelected() {
    return this._isSelected;
  }

  /**
   * @override
   */
  setSelected(bSelected) {
    this._isSelected = bSelected;
    for (var i = 0; i < this._displayables.length; i++) {
      if (this._displayables[i].setSelected) {
        this._displayables[i].setSelected(bSelected);
        this._updateAriaLabel(this._displayables[i]);
      }
    }
  }

  /**
   * @override
   */
  showHoverEffect() {
    for (var i = 0; i < this._displayables.length; i++) {
      if (this._displayables[i].showHoverEffect) this._displayables[i].showHoverEffect();
    }
  }

  /**
   * @override
   */
  hideHoverEffect() {
    for (var i = 0; i < this._displayables.length; i++) {
      if (this._displayables[i].hideHoverEffect) this._displayables[i].hideHoverEffect();
    }
  }

  //---------------------------------------------------------------------//
  // Rollover and Hide/Show Support: DvtLogicalObject impl               //
  //---------------------------------------------------------------------//
  /**
   * @override
   */
  getDisplayables() {
    return this._displayables;
  }

  /**
   * @override
   */
  getAriaLabel() {
    var states = [];
    var translations = this.getChart().getOptions().translations;
    if (this.isSelectable())
      states.push(translations[this.isSelected() ? 'stateSelected' : 'stateUnselected']);
    if (this.isDrillable()) states.push(translations.stateDrillable);

    var shortDesc = DvtChartTooltipUtils.getDatatip(
      this._chart,
      this._seriesIndex,
      this._groupIndex,
      this._itemIndex,
      false
    );
    if (shortDesc == null && this._groupIndex < 0 && states.length > 0)
      shortDesc = DvtChartDataUtils.getSeriesLabel(this._chart, this._seriesIndex);

    return Displayable.generateAriaLabel(shortDesc, states, () =>
      DvtChartStyleUtils.getShortDescContext(
        this._chart,
        this._seriesIndex,
        this._groupIndex,
        this._itemIndex,
        false
      )
    );
  }

  /**
   * Updates the aria-label as needed. On desktop, we can defer the aria creation, and the aria-label will be updated
   * when the activeElement is set.
   * @param {dvt.Displayable} displayable The displayable object.
   * @private
   */
  _updateAriaLabel(displayable) {
    if (!Agent.deferAriaCreation()) displayable.setAriaProperty('label', this.getAriaLabel());
  }

  //---------------------------------------------------------------------//
  // Rollover and Hide/Show Support: DvtCategoricalObject impl           //
  //---------------------------------------------------------------------//
  /**
   * @override
   */
  getCategories(category) {
    return this._categories;
  }

  /**
   * @return {dvt.Point} The coordinate of the data point relative to the plot area
   */
  getDataPosition() {
    return this._dataPos;
  }

  //---------------------------------------------------------------------//
  // Keyboard Support: DvtKeyboardNavigable impl                        //
  //---------------------------------------------------------------------//
  /**
   * @override
   */
  getNextNavigable(event) {
    var keyCode;
    var next;

    keyCode = event.keyCode;
    if (event.type == MouseEvent.CLICK) {
      return this;
    } else if (keyCode == KeyboardEvent.SPACE && event.ctrlKey) {
      // multi-select node with current focus; so we navigate to ourself and then let the selection handler take
      // care of the selection
      return this;
    }

    var chart = this._chart;
    var chartObjs = chart.getChartObjPeers();

    var navigables = [];
    for (var i = 0; i < chartObjs.length; i++) {
      if (chartObjs[i].isNavigable()) navigables.push(chartObjs[i]);
    }

    var isUpDown =
      event.keyCode == KeyboardEvent.UP_ARROW || event.keyCode == KeyboardEvent.DOWN_ARROW;
    var isHoriz = DvtChartTypeUtils.isHorizontal(chart);
    var isRTL = Agent.isRightToLeft(chart.getCtx());
    // Box plot up/down should go up and down the nested items (left/right for horiz)
    if (
      DvtChartTypeUtils.isScatterBubble(chart) ||
      (DvtChartTypeUtils.isBoxPlot(chart) && ((isUpDown && !isHoriz) || (!isUpDown && isHoriz)))
    ) {
      next = KeyboardHandler.getNextAdjacentNavigable(this, event, navigables);
    }
    // Polar bars should be treated the same way as line/area charts
    else if (
      DvtChartTypeUtils.isLineArea(chart) ||
      DvtChartDataUtils.isStacked(chart) ||
      DvtChartTypeUtils.isPolar(chart)
    ) {
      next = this._findNextNavigable(event);
    } else if (DvtChartTypeUtils.isFunnel(chart) && isUpDown) {
      if (isRTL)
        event.keyCode =
          event.keyCode == KeyboardEvent.UP_ARROW
            ? KeyboardEvent.RIGHT_ARROW
            : KeyboardEvent.LEFT_ARROW;
      else event.keyCode = event.keyCode - 1;
      next = KeyboardHandler.getNextNavigable(this, event, navigables);
    } else if (DvtChartTypeUtils.isPyramid(chart) && !isUpDown) {
      if (isRTL)
        event.keyCode =
          event.keyCode == KeyboardEvent.RIGHT_ARROW
            ? KeyboardEvent.DOWN_ARROW
            : KeyboardEvent.UP_ARROW;
      else
        event.keyCode =
          event.keyCode == KeyboardEvent.RIGHT_ARROW
            ? KeyboardEvent.UP_ARROW
            : KeyboardEvent.DOWN_ARROW;
      next = KeyboardHandler.getNextNavigable(this, event, navigables);
    } else {
      // : ignoreBounds for the case of range bars or bars with negative values that don't overlap with the adjacent series.
      next = KeyboardHandler.getNextNavigable(this, event, navigables, true);
    }
    return next;
  }

  /**
   * @override
   */
  getKeyboardBoundingBox(targetCoordinateSpace) {
    if (this._displayables[0]) return this._displayables[0].getDimensions(targetCoordinateSpace);
    else return new Rectangle(0, 0, 0, 0);
  }

  /**
   * @override
   */
  getTargetElem() {
    if (this._displayables[0]) return this._displayables[0].getElem();
    return null;
  }

  /**
   * @override
   */
  showKeyboardFocusEffect() {
    if (this.isNavigable()) {
      this._isShowingKeyboardFocusEffect = true;
      this.showHoverEffect();
    }
  }

  /**
   * @override
   */
  hideKeyboardFocusEffect() {
    if (this.isNavigable()) {
      this._isShowingKeyboardFocusEffect = false;
      this.hideHoverEffect();
    }
  }

  /**
   * @override
   */
  isShowingKeyboardFocusEffect() {
    return this._isShowingKeyboardFocusEffect;
  }

  /**
   * Returns true if the object is navigable
   * @return {boolean}
   */
  isNavigable() {
    return this.getGroupIndex() != -1 && this.getSeriesIndex() != -1;
  }

  /**
   * Returns the next navigable object in the direction of the arrow for line/area
   * @param {dvt.BaseEvent} event
   * @return {DvtChartObjPeer}
   * @private
   */
  _findNextNavigable(event) {
    var keyCode = event.keyCode;
    var chart = this._chart;
    var context = chart.getCtx();

    var seriesIndex = this.getSeriesIndex();
    var groupIndex = this.getGroupIndex();
    var groupCount = DvtChartDataUtils.getGroupCount(chart);
    var nextSeriesIndex;
    var nextGroupIndex;

    var isHoriz = DvtChartTypeUtils.isHorizontal(chart);
    var isPolar = DvtChartTypeUtils.isPolar(chart);
    var isRTL = Agent.isRightToLeft(context);
    var isUp = isHoriz
      ? isRTL
        ? keyCode == KeyboardEvent.LEFT_ARROW
        : keyCode == KeyboardEvent.RIGHT_ARROW
      : keyCode == KeyboardEvent.UP_ARROW;
    var isDown = isHoriz
      ? isRTL
        ? keyCode == KeyboardEvent.RIGHT_ARROW
        : keyCode == KeyboardEvent.LEFT_ARROW
      : keyCode == KeyboardEvent.DOWN_ARROW;
    var isLeft = isHoriz
      ? keyCode == KeyboardEvent.UP_ARROW
      : isRTL
      ? keyCode == KeyboardEvent.RIGHT_ARROW
      : keyCode == KeyboardEvent.LEFT_ARROW;
    var isRight = isHoriz
      ? keyCode == KeyboardEvent.DOWN_ARROW
      : isRTL
      ? keyCode == KeyboardEvent.LEFT_ARROW
      : keyCode == KeyboardEvent.RIGHT_ARROW;

    var isStacked = DvtChartDataUtils.isStacked(chart);
    var isBar = DvtChartTypeUtils.isBar(chart);
    var nextObj;
    if (isUp) {
      nextGroupIndex = groupIndex;
      nextSeriesIndex = this._findNextUpSeries(chart, seriesIndex, groupIndex);
    } else if (isDown) {
      nextGroupIndex = groupIndex;
      nextSeriesIndex = this._findNextDownSeries(chart, seriesIndex, groupIndex);
    } else if (isRight || isLeft) {
      nextSeriesIndex = seriesIndex;
      nextGroupIndex = groupIndex;
      do {
        nextGroupIndex = isRight ? nextGroupIndex + 1 : nextGroupIndex - 1;
        if (isPolar && nextGroupIndex >= groupCount) nextGroupIndex = 0;
        if (isPolar && nextGroupIndex < 0) nextGroupIndex = groupCount - 1;
        nextObj = chart.getObject(nextSeriesIndex, nextGroupIndex);
        // if stacked and next group does not have item of same series,
        // shift focus to the item at the bottom of the stack
        if (nextObj == null && isStacked && isBar) {
          var nextSeries = this._findFirstSeriesInGroup(chart, nextGroupIndex);
          nextObj = chart.getObject(nextSeries, nextGroupIndex);
        }
      } while (
        nextObj == null &&
        ((isRight && nextGroupIndex < groupCount) || (isLeft && nextGroupIndex > -1))
      );
    }

    nextObj = nextObj || chart.getObject(nextSeriesIndex, nextGroupIndex);
    return nextObj && nextObj.isNavigable() ? nextObj : this;
  }

  /**
   * Returns the index of the next up series
   * @param {Chart} chart
   * @param {number} seriesIndex Current series index.
   * @param {number} groupIndex Current group index.
   * @return {number} Next up series index.
   * @private
   */
  _findNextUpSeries(chart, seriesIndex, groupIndex) {
    var isStacked = DvtChartDataUtils.isStacked(chart);
    var seriesCount = DvtChartDataUtils.getSeriesCount(chart);
    var currentValue = DvtChartDataUtils.getCumulativeVal(chart, seriesIndex, groupIndex);
    var nextValue = null;
    var nextSeriesIndex = null;
    for (var i = 0; i < seriesCount; i++) {
      if (
        !DvtChartDataUtils.isSeriesRendered(chart, i) ||
        DvtChartDataUtils.getVal(chart, i, groupIndex) == null ||
        (isStacked && chart.getObject(i, groupIndex) == null)
      )
        continue;
      var itemValue = DvtChartDataUtils.getCumulativeVal(chart, i, groupIndex);
      if (itemValue > currentValue || (itemValue == currentValue && i > seriesIndex)) {
        if ((nextValue !== null && itemValue < nextValue) || nextValue == null) {
          nextValue = itemValue;
          nextSeriesIndex = i;
        }
      }
    }
    return nextSeriesIndex;
  }

  /**
   * Returns the index of the next down series.
   * @param {Chart} chart
   * @param {number} seriesIndex Current series index.
   * @param {number} groupIndex Current group index.
   * @return {number} Next down series index.
   * @private
   */
  _findNextDownSeries(chart, seriesIndex, groupIndex) {
    var isStacked = DvtChartDataUtils.isStacked(chart);
    var seriesCount = DvtChartDataUtils.getSeriesCount(chart);
    var currentValue = DvtChartDataUtils.getCumulativeVal(chart, seriesIndex, groupIndex);
    var nextValue = null;
    var nextSeriesIndex = null;
    for (var i = seriesCount - 1; i >= 0; i--) {
      if (
        !DvtChartDataUtils.isSeriesRendered(chart, i) ||
        DvtChartDataUtils.getVal(chart, i, groupIndex) == null ||
        (isStacked && chart.getObject(i, groupIndex) == null)
      )
        continue;
      var itemValue = DvtChartDataUtils.getCumulativeVal(chart, i, groupIndex);
      if (itemValue < currentValue || (itemValue == currentValue && i < seriesIndex)) {
        if ((nextValue !== null && itemValue > nextValue) || nextValue == null) {
          nextValue = itemValue;
          nextSeriesIndex = i;
        }
      }
    }
    return nextSeriesIndex;
  }

  /**
   * Returns the index of the first series in the given group. This will be the item at the
   * bottom of the stack when chart is stacked.
   * @param {Chart} chart
   * @param {number} groupIndex The group index of the group.
   * @returns
   */
  _findFirstSeriesInGroup(chart, groupIndex) {
    var seriesCount = DvtChartDataUtils.getSeriesCount(chart);
    for (var idx = 0; idx < seriesCount; idx++) {
      if (
        DvtChartDataUtils.isSeriesRendered(chart, idx) &&
        DvtChartDataUtils.getVal(chart, idx, groupIndex) != null &&
        chart.getObject(idx, groupIndex) != null
      )
        return idx;
    }
    return null;
  }

  //---------------------------------------------------------------------//
  // DnD Support: DvtDraggable impl                                      //
  //---------------------------------------------------------------------//

  /**
   * @override
   */
  isDragAvailable(clientIds) {
    return true;
  }

  /**
   * @override
   */
  getDragTransferable() {
    return [this.getId()];
  }

  /**
   * @override
   */
  getDragFeedback() {
    // If more than one object is selected, return the displayables of all selected objects
    if (
      this._chart.isSelectionSupported() &&
      this._chart.getSelectionHandler().getSelectedCount() > 1
    ) {
      var selection = this._chart.getSelectionHandler().getSelection();
      var displayables = [];
      for (var i = 0; i < selection.length; i++) {
        displayables = displayables.concat(selection[i].getDisplayables());
      }
      return displayables;
    }

    // Otherwise, return its own displayables
    return this._displayables;
  }
}

/**
 * Logical object for reference object displayables.
 * @param {Chart} chart
 * @param {array} displayables The array of associated DvtDisplayables.
 * @param {object} refObj reference object
 * @param {number} index The reference objects position in the reference object array
 * @param {string} axisType The axis the reference object is on
 * @class
 * @constructor
 * @implements {DvtCategoricalObject}
 * @implements {DvtLogicalObject}
 * @implements {DvtTooltipSource}
 */
class DvtChartRefObjPeer {
  constructor(chart, displayables, refObj, index, axisType) {
    this._chart = chart;
    this._displayables = displayables;
    this._refObj = refObj;
    this._categories = DvtChartRefObjUtils.getRefObjCategories(this._refObj);

    // used for automation
    this._index = index;
    this._axisType = axisType;

    // WAI-ARIA
    for (var i = 0; i < displayables.length; i++) {
      var displayable = displayables[i];
      displayable.setAriaRole('img');
      displayable.setAriaProperty('label', refObj['shortDesc']);
    }
  }

  /**
   * @override
   */
  getCategories() {
    return this._categories;
  }

  /**
   * @override
   */
  getDisplayables() {
    return this._displayables;
  }

  /**
   * Returns the position of the reference object in the referenceObjects array.
   * @return {number} The position of this reference object.
   */
  getIndex() {
    return this._index;
  }

  /**
   * Returns which axis this reference object belongs to.
   * @return {string} xAxis, yAxis, or y2Axis
   */
  getAxisType() {
    return this._axisType;
  }

  /**
   * @override
   */
  getDatatip() {
    return DvtChartTooltipUtils.getRefObjTooltip(
      this._chart,
      this._refObj,
      this._axisType,
      this._index
    );
  }

  /**
   * @override
   */
  getDatatipColor() {
    return DvtChartRefObjUtils.getColor(this._chart, this._refObj);
  }
}

/**
 * Default values and utility functions for component versioning.
 * @class
 * @constructor
 * @param {dvt.Context} context The rendering context.
 * @extends {dvt.BaseComponentDefaults}
 */

class DvtAxisDefaults extends BaseComponentDefaults {
  constructor(context) {
    /**
     * Defaults for version 1.
     */
    const SKIN_ALTA = {
      position: null,
      baselineScaling: 'zero',
      axisLine: { lineColor: '#9E9E9E', lineWidth: 1, rendered: 'on' },
      majorTick: {
        lineColor: 'rgba(196,206,215,0.4)',
        baselineColor: 'auto',
        lineWidth: 1,
        rendered: 'auto',
        lineStyle: 'solid'
      },
      minorTick: {
        lineColor: 'rgba(196,206,215,0.2)',
        lineWidth: 1,
        rendered: 'off',
        lineStyle: 'solid'
      },
      tickLabel: {
        scaling: 'auto',
        style: new CSSStyle(BaseComponentDefaults.FONT_FAMILY_ALTA_11 + 'color: #333333;'),
        rotation: 'auto',
        rendered: 'on'
      },
      titleStyle: new CSSStyle(
        BaseComponentDefaults.FONT_FAMILY_ALTA_12 + 'color: #737373;'
      ),

      // For group axis, an optional offset expressed as a factor of the group size.
      startGroupOffset: 0,
      endGroupOffset: 0,

      //* ********** Internal Attributes *************************************************//
      layout: {
        titleGap: 6,
        radialLabelGap: 5,
        insideLabelGapWidth: 4,
        insideLabelGapHeight: 2,
        hierarchicalLabelGapHeight: 8,
        hierarchicalLabelGapWidth: 15
      },
      _locale: 'en-us'
    };
    super({ alta: SKIN_ALTA }, context);
  }

  /**
   * @override
   */
  getNoCloneObject() {
    return {
      tickLabel: { converter: true }
    };
  }

  /**
   * Adjusts the gap size based on the component options.
   * @param {dvt.Context} context The axis component context.
   * @param {Object} options The axis options.
   * @param {Number} defaultSize The default gap size.
   * @return {Number}
   */
  static getGapSize(context, options, defaultSize) {
    // adjust based on tick label font size
    var scalingFactor = Math.min(
      TextUtils.getTextStringHeight(context, options.tickLabel.style) / 14,
      1
    );
    return Math.ceil(defaultSize * scalingFactor);
  }
}

/**
 * Calculated axis information and drawable creation.  This class should
 * not be instantiated directly.
 * @param {dvt.Context} context
 * @param {object} options The object containing specifications and data for this component.
 * @param {dvt.Rectangle} availSpace The available space.
 * @class
 * @constructor
 * @extends {dvt.Obj}
 */

class DvtAxisInfo extends BaseAxisInfo {
  constructor(context, options, availSpace) {
    super(context, options, availSpace);
    this._title = null;
  }
  /**
   * Returns an array containing the tick labels for this axis.
   * @param {dvt.Context} context
   * @param {Number} levelIdx The level index (optional). 0 indicates the first level, 1 the second, etc. If skipped, 0 (the first level) is assumed.
   * @return {Array} The Array of dvt.Text objects.
   */
  getLabels(context, levelIdx) {
    return null; // subclasses should override
  }

  /**
   * Returns the title for this axis.
   * @return {dvt.Text} The dvt.Text object, if it exists.
   */
  getTitle() {
    return this._title;
  }

  /**
   * Sets the title for this axis.
   * @param {dvt.Text} title The axis title.
   */
  setTitle(title) {
    this._title = title;
  }

  /**
   * Returns the coordinates of the major ticks.
   * @return {array} Array of coords.
   */
  getMajorTickCoords() {
    return []; // subclasses should override
  }

  /**
   * Returns the coordinates of the minor ticks.
   * @return {array} Array of coords.
   */
  getMinorTickCoords() {
    return []; // subclasses should override
  }

  /**
   * Returns the coordinates of the baseline (value = 0). Only applies to numerical axis.
   * @return {number} Baseline coord.
   */
  getBaselineCoord() {
    return null; // subclasses should override
  }

  /**
   * Returns the datatip for the label at the given index and level.
   * @param {number} index
   * @param {number} level
   * @return {string} The datatip.
   */
  getDatatip(index, level) {
    return null; // subclasses should override
  }

  /**
   * Returns an object with the label's background labelStyles applied
   * @param {dvt.OutputText} label The label.
   * @param {dvt.Context} context
   * @return {dvt.Rect} The object to be rendered behind the label.
   */
  getLabelBackground(label, context) {
    return null; // subclasses should override
  }

  /**
   * Returns whether the label at the given index is drillable
   * @param {number} index The label index.
   * @return {boolean} Whether the label is drillable.
   */
  isDrillable(index) {
    return null; // subclasses should override
  }

  /**
   * Returns if the labels of the horizontal axis are rotated by 90 degrees.
   * @return {boolean} Whether the labels are rotated.
   */
  isLabelRotated() {
    return false;
  }

  /**
   * Creates a dvt.Text instance for the specified text label.
   * @param {dvt.Context} context
   * @param {string} label The label string.
   * @param {number} coord The coordinate for the text.
   * @param {dvt.CSSStyle=} style Optional style for the text label.
   * @param {boolean=} bMultiline Optional boolean to create dvt.MultilineText
   * @return {dvt.OutputText|dvt.BackgroundOutputText|dvt.MultilineText}
   * @protected
   */
  CreateLabel(context, label, coord, style, bMultiline) {
    var text;

    if (this.Position == 'tangential') {
      var vTol = (16 / 180) * Math.PI; // the mid area (15 degrees) where labels will be middle aligned.
      var hTol = (1 / 180) * Math.PI; // the tolerance (1 degree) where labels will be center aligned.

      var offset = 0.5 * this.getTickLabelHeight();
      var dist = this._radius + offset;
      if (coord < hTol || coord > 2 * Math.PI - hTol) dist += offset; // avoild collision with radial label

      var xcoord = Math.round(dist * Math.sin(coord));
      var ycoord = Math.round(-dist * Math.cos(coord));
      text = style
        ? new BackgroundOutputText(context, label, xcoord, ycoord, style)
        : new OutputText(context, label, xcoord, ycoord);

      // Align the label according to the angular position
      if (coord < hTol || Math.abs(coord - Math.PI) < hTol || coord > 2 * Math.PI - hTol)
        text.alignCenter();
      else if (coord < Math.PI) text.alignLeft();
      else text.alignRight();

      if (Math.abs(coord - Math.PI / 2) < vTol || Math.abs(coord - (3 * Math.PI) / 2) < vTol)
        text.alignMiddle();
      else if (coord < Math.PI / 2 || coord > (3 * Math.PI) / 2) text.alignBottom();
      else text.alignTop();
    } else {
      if (bMultiline)
        text = style
          ? new BackgroundMultilineText(context, label, coord, coord, style)
          : new MultilineText(context, label, coord, coord);
      else {
        text = style
          ? new BackgroundOutputText(context, label, coord, coord, style)
          : new OutputText(context, label, coord, coord);
        text.alignMiddle();
      }
      text.alignCenter();
    }
    if (text instanceof OutputText || text instanceof MultilineText)
      // DvtBackgroundTexts already created with its CSSStyle
      text.setCSSStyle(this.Options['tickLabel']['style']);
    return text;
  }

  /**
   * Checks all the labels for the axis and returns whether they overlap.
   * @param {Array} labelDims An array of dvt.Rectangle objects that describe the x, y, height, width of the axis labels.
   * @param {number} skippedLabels The number of labels to skip. If skippedLabels is 1 then every other label will be skipped.
   * @param {Array=} maxWidths An array of max sizes for for each label.
   * @return {boolean} True if any labels overlap or are greater than their corresponding max width.
   * @protected
   */
  IsOverlapping(labelDims, skippedLabels, maxWidths) {
    // If there are no labels, return
    if (!labelDims || labelDims.length <= 0) return false;

    var isVert = this.Position == 'left' || this.Position == 'right' || this.Position == 'radial';
    var isRTL = Agent.isRightToLeft(this.getCtx());
    var gap = this.GetTickLabelGapSize();

    var pointA1, pointA2, pointB1, pointB2;
    for (var j = 0; j < labelDims.length; j += skippedLabels + 1) {
      if (labelDims[j] == null) continue;

      if (maxWidths != null && labelDims[j].w > maxWidths[j]) return true;

      if (pointA1 == null || pointA2 == null) {
        // Set the first points
        if (isVert) {
          pointA1 = labelDims[j].y;
          pointA2 = labelDims[j].y + labelDims[j].h;
        } else {
          pointA1 = labelDims[j].x;
          pointA2 = labelDims[j].x + labelDims[j].w;
        }
        continue;
      }

      if (isVert) {
        pointB1 = labelDims[j].y;
        pointB2 = labelDims[j].y + labelDims[j].h;

        // Broken apart for clarity, next label may be above or below
        if (pointB1 >= pointA1 && pointB1 - gap < pointA2)
          // next label below
          return true;
        else if (pointB1 < pointA1 && pointB2 + gap > pointA1)
          // next label above
          return true;
      } else {
        pointB1 = labelDims[j].x;
        pointB2 = labelDims[j].x + labelDims[j].w;

        // Broken apart for clarity, next label is on the right for non-BIDI, left for BIDI
        if (!isRTL && pointB1 - gap < pointA2) return true;
        else if (isRTL && pointB2 + gap > pointA1) return true;
      }

      // Otherwise start evaluating from label j
      pointA1 = pointB1;
      pointA2 = pointB2;
    }
    return false;
  }

  /**
   * Compares two label dimensions and returns whether they overlap.
   * @param {Object} labelDims1 An object that describes the x, y, height, width of the first label.
   * @param {Object} labelDims2 An object that describes the x, y, height, width of the second label.
   * @return {boolean} True if the label dimensions overlap.
   * @protected
   */
  IsOverlappingDims(labelDims1, labelDims2) {
    if (!labelDims1 || !labelDims2) return false;

    var pointA1 = labelDims1.y;
    var pointA2 = labelDims1.y + labelDims1.h;
    var pointA3 = labelDims1.x;
    var pointA4 = labelDims1.x + labelDims1.w;

    var pointB1 = labelDims2.y;
    var pointB2 = labelDims2.y + labelDims2.h;
    var pointB3 = labelDims2.x;
    var pointB4 = labelDims2.x + labelDims2.w;

    var widthOverlap =
      (pointA3 <= pointB3 && pointB3 <= pointA4) ||
      (pointA3 <= pointB4 && pointB4 <= pointA4) ||
      (pointB3 <= pointA3 && pointA3 <= pointB4) ||
      (pointB3 <= pointA4 && pointA4 <= pointB4);
    var heightOverlap =
      (pointB1 >= pointA1 && pointB1 < pointA2) || (pointB1 <= pointA1 && pointB2 >= pointA1);

    return widthOverlap && heightOverlap;
  }

  /**
   * Returns the tick label gap size.
   * @return {number}
   * @protected
   */
  GetTickLabelGapSize() {
    // Create gap based on tick label height
    // GroupAxis and TimeAxis have smaller gaps since these axes become less useable as more labels are dropped
    var labelHeight = this.getTickLabelHeight();
    var gapHoriz =
      this instanceof DvtAxisInfo._constructors.group ? labelHeight * 0.24 : labelHeight * 0.79;
    var gapVert =
      this instanceof DvtAxisInfo._constructors.group ? labelHeight * 0.08 : labelHeight * 0.28;

    var isVert = this.Position == 'left' || this.Position == 'right' || this.Position == 'radial';
    return isVert || this.isLabelRotated() ? gapVert : gapHoriz;
  }

  /**
   * Returns the tick label height in px.
   * @return {number}
   */
  getTickLabelHeight() {
    return TextUtils.getTextStringHeight(this.getCtx(), this.Options['tickLabel']['style']);
  }

  /**
   * Checks the labels for the axis and skips them as necessary.
   * @param {Array} labels An array of dvt.Text labels for the axis.
   * @param {Array} labelDims An array of dvt.Rectangle objects that describe the x, y, height, width of the axis labels.
   * @return {Array} The array of dvt.Text labels for the axis.
   * @protected
   */
  SkipLabels(labels, labelDims) {
    var skippedLabels = 0;
    var bOverlaps = this.IsOverlapping(labelDims, skippedLabels);
    while (bOverlaps) {
      skippedLabels++;
      bOverlaps = this.IsOverlapping(labelDims, skippedLabels);
    }

    if (skippedLabels > 0) {
      var renderedLabels = [];
      for (var j = 0; j < labels.length; j += skippedLabels + 1) {
        renderedLabels.push(labels[j]);
      }
      return renderedLabels;
    } else {
      return labels;
    }
  }

  /**
   * Checks the labels for the tangential axis and skips them as necessary.
   * @param {Array} labels An array of dvt.Text labels for the axis.
   * @param {Array} labelDims An array of dvt.Rectangle objects that describe the x, y, height, width of the axis labels.
   * @return {Array} The array of dvt.Text labels for the tangential axis.
   * @protected
   */
  SkipTangentialLabels(labels, labelDims) {
    var renderedLabels = [];
    var numLabels = labels.length;
    var firstLabelDims = null;

    if (numLabels > 1) {
      var prevLabelDims;
      // Include label if it does not overlap with previously included label
      for (var j = 0; j < numLabels; j++) {
        if (!labelDims[j]) continue;
        if (
          !prevLabelDims ||
          (prevLabelDims && !this.IsOverlappingDims(prevLabelDims, labelDims[j]))
        ) {
          if (!firstLabelDims) firstLabelDims = labelDims[j];
          renderedLabels.push(labels[j]);
          prevLabelDims = labelDims[j];
        }
      }

      // Remove last included label if it overlaps with the first included label
      if (this.IsOverlappingDims(prevLabelDims, firstLabelDims)) renderedLabels.pop();

      return renderedLabels;
    }
    return labels;
  }

  /**
   * Returns an array of dvt.Rectangle objects that describe the x, y, width, height of the axis labels.
   * If level is set, it assumes that the labels are center-middle aligned.
   * @param {Array} labels An array of dvt.Text labels for the axis.
   * @param {dvt.Container} container
   * @param {Number} level (optional) Used for group axis hierarchical labels
   * @return {Array} An array of dvt.Rectangle objects
   * @protected
   */
  GetLabelDims(labels, container, level) {
    var labelDims = [];
    var isRotated = this.isLabelRotated(level);

    // Get the text dimensions
    for (var i = 0; i < labels.length; i++) {
      var text = labels[i];
      if (text == null) {
        labelDims.push(null);
        continue;
      }

      var dims = text.getDimensions(container);
      if (level != null) {
        dims.x = (isRotated ? text.getTranslateX() : text.getX()) - dims.w / 2;
        dims.y = (isRotated ? text.getTranslateY() : text.getY()) - dims.h / 2;
      }

      if (dims.w && dims.h)
        // Empty group axis labels with 0 height and width are possible, they should count as null
        labelDims.push(dims);
      else labelDims.push(null);
    }

    return labelDims;
  }

  /**
   * Returns the number of major tick counts for the axis.
   * @return {number} The number of major tick counts.
   */
  getMajorTickCount() {
    return null; // subclasses that allow major gridlines should implement
  }

  /**
   * Returns the number of minor tick counts for the axis.
   * @return {number} The number of minor tick counts.
   */
  getMinorTickCount() {
    return null; // subclasses that allow minor gridlines should implement
  }

  /**
   * Returns the major increment for the axis.
   * @return {number} The major increment.
   */
  getMajorIncrement() {
    return null; // subclasses that allow major gridlines should implement
  }

  /**
   * Returns the minor increment for the axis.
   * @return {number} The minor increment.
   */
  getMinorIncrement() {
    return null; // subclasses that allow minor gridlines should implement
  }

  /**
   * Returns the global min value of the axis.
   * @return {number} The global min value.
   */
  getGlobalMin() {
    return this.GlobalMin;
  }

  /**
   * Returns the global max value of the axis.
   * @return {number} The global max value.
   */
  getGlobalMax() {
    return this.GlobalMax;
  }

  /**
   * Returns the viewport min value of the axis.
   * @return {number} The viewport min value.
   */
  getViewportMin() {
    return this.MinValue;
  }

  /**
   * Returns the viewport max value of the axis.
   * @return {number} The viewport max value.
   */
  getViewportMax() {
    return this.MaxValue;
  }

  /**
   * Returns the data min value of the axis.
   * @return {number} The data min value.
   */
  getDataMin() {
    return this.DataMin;
  }

  /**
   * Returns the data max value of the axis.
   * @return {number} The data max value.
   */
  getDataMax() {
    return this.DataMax;
  }

  /**
   * Returns the minimum extent of the axis, i.e. the (maxLinearValue - minLinearValue) during maximum zoom.
   * @return {number} The minimum extent.
   */
  getMinExtent() {
    return 0;
  }

  /**
   * Returns the start coord.
   * @return {number}
   */
  getStartCoord() {
    return this.StartCoord;
  }

  /**
   * Returns the end coord.
   * @return {number}
   */
  getEndCoord() {
    return this.EndCoord;
  }

  /**
   * Returns how much the axis labels overflow over the start coord.
   * @return {number}
   */
  getStartOverflow() {
    return this.StartOverflow;
  }

  /**
   * Returns how much the axis labels overflow over the end coord.
   * @return {number}
   */
  getEndOverflow() {
    return this.EndOverflow;
  }

  /**
   * Gets the width of a group (for rendering bar chart)
   * @return {Number} the width of a group
   */
  getGroupWidth() {
    return 0;
  }

  /**
   * Returns a string or an array of groups names/ids of the ancestors of a group label at the given index and level.
   * @param {Number} index The index of the group label within it's level of labels
   * @param {Number=} level The level of the group labels
   * @return {String|Array} The group name/id, or an array of group names/ids.
   * @override
   */
  getGroup(index, level) {
    // only applies to group axis
    return null;
  }

  /**
   * Converts linear value to actual value.
   * For example, for a log scale, the linear value is the log of the actual value.
   * @param {number} value The linear value.
   * @return {number} The actual value.
   */
  linearToActual(value) {
    return value;
  }

  /**
   * Converts actual value to linear value.
   * For example, for a log scale, the linear value is the log of the actual value.
   * @param {number} value The actual value.
   * @return {number} The linear value.
   */
  actualToLinear(value) {
    return value;
  }

  /**
   * Increases the scale unit used until the data for the axis can be fit within a given number of majorTicks
   * Used exclusively to align y and y2 log axes when the yAxis majorTickCount is less than what the y2Axis needs.
   * @param {number} scaleUnit The current scale unit of the axis.
   * @param {number} tickCount The number of major ticks the axis will use.
   * @return {number} The new scale unit that wil allow the axis data to render within the given tickCount number.
   */
  alignLogScaleToTickCount(scaleUnit, tickCount) {
    return scaleUnit;
  }

  /**
   * Returns the scale unit, if one has been previously stored. Will only apply to log axes.
   * @return {number} The scale unit
   */
  getLogScaleUnit() {
    return null;
  }

  /**
   * Used by DvtDataAxisInfo, DvtGroupAxisInfo, DvtTimeAxisInto to pass constructors without created a circular dependency
   *
   * @param {'data'|'group'|'time'} type
   * @param {function} ctor
   */
  static registerConstructor(type, ctor) {
    DvtAxisInfo._constructors[type] = ctor;
  }

  static getConstructor(type) {
    return DvtAxisInfo._constructors[type];
  }

  /**
   * Creates an appropriate instance of DvtAxisInfo with the specified parameters.
   * @param {dvt.Context} context
   * @param {object} options The object containing specifications and data for this component.
   * @param {dvt.Rectangle} availSpace The available space.
   * @return {DvtAxisInfo}
   */
  static newInstance(context, options, availSpace) {
    if (options['timeAxisType'] && options['timeAxisType'] != 'disabled')
      return new DvtAxisInfo._constructors.time(context, options, availSpace);
    else if (options['_isGroupAxis'])
      return new DvtAxisInfo._constructors.group(context, options, availSpace);
    return new DvtAxisInfo._constructors.data(context, options, availSpace);
  }
}

DvtAxisInfo._constructors = [];

/**
 * Simple logical object for tooltip support.
 * @param {DvtAxis} axis The axis.
 * @param {dvt.OutputText} label The owning text instance.
 * @param {string|Array} group A string or an array of groups names/ids of the label and the ancestors.
 * @param {object} drillable Whether the label is drillable.
 * @param {string} tooltip The tooltip of the label.
 * @param {string} datatip The datatip of the label.
 * @param {object=} params Optional object containing additional parameters for use by component.
 * @class DvtAxisObjPeer
 * @constructor
 * @implements {dvt.TextObjPeer}
 * @implements {DvtLogicalObject}
 * @implements {DvtDraggable}
 */

class DvtAxisObjPeer extends TextObjPeer {
  /**
   * @param {DvtAxis} axis The axis.
   * @param {dvt.OutputText} label The owning text instance.
   * @param {string|Array} group A string or an array of groups names/ids of the label and the ancestors.
   * @param {object} drillable Whether the label is drillable.
   * @param {string} tooltip The tooltip of the label.
   * @param {string} datatip The datatip of the label.
   * @param {object=} params Optional object containing additional parameters for use by component.
   */
  constructor(axis, label, group, drillable, tooltip, datatip, params) {
    super(axis, label, tooltip, datatip, null, params);
    this._axis = axis;
    this._label = label;
    this._group = group;
    this._drillable = drillable;

    // Apply the cursor for drilling if specified
    if (this._drillable) label.setCursor(SelectionEffectUtils.getSelectingCursor());

    axis.__registerObject(this);
  }

  /**
   * Returns the label for this object.
   * @return {dvt.OutputText}
   */
  getLabel() {
    return this._label;
  }

  /**
   * Returns the id for this object.
   * @return {object} The id for this label.
   */
  getId() {
    return this._group;
  }

  /**
   * Returns whether the label is drillable.
   * @return {boolean}
   */
  isDrillable() {
    return this._drillable;
  }

  /**
   * Returns the group.
   * @return {string|Array}
   */
  getGroup() {
    return this._group;
  }

  //---------------------------------------------------------------------//
  // Keyboard Support: DvtKeyboardNavigable impl                         //
  //---------------------------------------------------------------------//
  /**
   * @override
   */
  getNextNavigable(event) {
    // TODO: Figure out if this is necessary
    if (event.type == MouseEvent.CLICK) return this;

    var navigables = this._axis.__getKeyboardObjects();
    return KeyboardHandler.getNextNavigable(
      this,
      event,
      navigables,
      false,
      this._axis.getCtx().getStage()
    );
  }

  //---------------------------------------------------------------------//
  // WAI-ARIA Support: DvtLogicalObject impl               //
  //---------------------------------------------------------------------//
  /**
   * @override
   */
  getAriaLabel() {
    var states;
    if (this.isDrillable()) {
      states = [this._axis.getOptions().translations.stateDrillable];
    }
    if (this.getDatatip() != null) {
      return Displayable.generateAriaLabel(this.getDatatip(), states);
    } else if (states != null) {
      return Displayable.generateAriaLabel(this.getLabel().getTextString(), states);
    }
  }

  //---------------------------------------------------------------------//
  // DnD Support: DvtDraggable impl                                      //
  //---------------------------------------------------------------------//

  /**
   * @override
   */
  isDragAvailable(clientIds) {
    return true;
  }

  /**
   * @override
   */
  getDragTransferable(mouseX, mouseY) {
    return [this.getId()];
  }

  /**
   * @override
   */
  getDragFeedback(mouseX, mouseY) {
    return [this.getDisplayable()];
  }
}

/**
 * Event Manager for DvtAxis.
 * @param {DvtAxis} axis
 * @class
 * @extends {dvt.EventManager}
 * @constructor
 */

class DvtAxisEventManager extends EventManager {
  constructor(axis) {
    super(axis.getCtx(), axis.processEvent, axis, axis);
    this._axis = axis;
  }

  /**
   * @override
   */
  OnClick(event) {
    super.OnClick(event);

    var obj = this.GetLogicalObject(event.target);
    if (!obj) return;

    var action = this.processDrillEvent(obj);

    // If an action occurs, the event should not bubble.
    if (action) event.stopPropagation();
  }

  /**
   * @override
   */
  HandleTouchClickInternal(evt) {
    var obj = this.GetLogicalObject(evt.target);
    if (!obj) return;

    var touchEvent = evt.touchEvent;
    var action = this.processDrillEvent(obj);
    if (action && touchEvent) touchEvent.preventDefault();
  }

  /**
   * Processes a drill on the specified group label.  Returns true if a drill event is fired.
   * @param {DvtGroupAxisObjPeer} obj The group label that was clicked.
   * @return {boolean} True if an event was fired.
   */
  processDrillEvent(obj) {
    // Drill Support
    if (obj instanceof DvtAxisObjPeer && obj.isDrillable()) {
      this.FireEvent(
        EventFactory.newChartDrillEvent(obj.getId(), null, obj.getGroup(), 'group'),
        this._axis
      );
      return true;
    }

    return false;
  }

  /**
   * @override
   */
  isDndSupported() {
    return true;
  }

  /**
   * @override
   */
  GetDragSourceType() {
    var obj = this.DragSource.getDragObject();
    if (obj instanceof DvtAxisObjPeer && obj.getGroup() != null) return 'groups';
    return null;
  }

  /**
   * @override
   */
  GetDragDataContexts(bSanitize) {
    var obj = this.DragSource.getDragObject();
    if (obj instanceof DvtAxisObjPeer) {
      var dataContext = {
        id: obj.getId(),
        group: obj.getGroup(),
        label: obj.getLabel().getTextString()
      };
      if (bSanitize) ToolkitUtils.cleanDragDataContext(dataContext);

      return [dataContext];
    }
    return [];
  }

  /**
   * Returns the parameters for the DvtComponentUIEvent for an object with the specified arguments.
   * @param {string} type The type of object that was the target of the event.
   * @param {object=} id The id of the object, if one exists.
   * @param {number=} index The index of the axis label, in regards to its level, if one is specified.
   * @param {number=} level The level of the axis label, if one is specified.
   * @return {object} the parameters for the DvtComponentUIEvent
   */
  static getUIParams(type, id, index, level) {
    return { type, id, index, level };
  }
}

/**
 * Renderer for DvtAxis.
 */
const DvtAxisRenderer = {
  /**
   * Fraction of width or height used to calculate initial preferred size for title width / height for axes.
   * @private
   */
  _PREFERRED_TITLE_PROPORTION: 0.8,

  /**
   * The max amount of lines we allow in title wrapping.
   * @private
   */
  _MAX_TITLE_LINE_WRAP: 3,
  /**
   * Returns the preferred dimensions for this component given the maximum available space. This will never be called for
   * radial axis.
   * @param {DvtAxis} axis
   * @param {number} availWidth
   * @param {number} availHeight
   * @return {dvt.Dimension} The preferred dimensions for the object.
   */
  getPreferredSize: (axis, availWidth, availHeight, ignoreRenderedOption) => {
    // Calculate the axis extents and increments
    var axisInfo = DvtAxisRenderer._createAxisInfo(
      axis,
      new Rectangle(0, 0, availWidth, availHeight)
    );
    var context = axis.getCtx();
    var options = axis.getOptions();

    // The axis will always return the full length of the dimension along which values are placed, so there's only one
    // size that we need to keep track of.  For example, this is the height on horizontal axes.
    var bHoriz = options['position'] == 'top' || options['position'] == 'bottom';

    // No size if not rendered or either dimension is 0. Since dimensions can also be 0 for rendered==off, either checks are ignored for the ignoreRenderedOption flag
    if (
      (options['rendered'] == 'off' || availWidth <= 0 || availHeight <= 0) &&
      !ignoreRenderedOption
    )
      return bHoriz ? new Dimension(availWidth, 0) : new Dimension(0, availHeight);

    // Allocate space for the title
    var availableTitleWidth =
      (bHoriz ? availWidth : availHeight) * DvtAxisRenderer._PREFERRED_TITLE_PROPORTION;
    var availableTitleHeight =
      (bHoriz ? availHeight : availWidth) * DvtAxisRenderer._PREFERRED_TITLE_PROPORTION;
    var titleHeightInfo = DvtAxisRenderer.getTitleHeight(
      context,
      options,
      availableTitleWidth,
      availableTitleHeight
    );
    var titleHeight = titleHeightInfo.height;

    // cache preferred title line count and available space used
    axis.getOptionsCache().putToCache('prefTitleLineCount', titleHeightInfo.lineCount);
    axis.getOptionsCache().putToCache('prefAvailableTitleWidth', availableTitleWidth);

    var size = titleHeight != 0 ? titleHeight + DvtAxisRenderer._getTitleGap(axis) : 0;

    // Allocate space for the tick labels
    if (options['tickLabel']['rendered'] == 'on' && options['tickLabel']['position'] != 'inside') {
      if (bHoriz) {
        // Horizontal Axis
        var labelHeight = TextUtils.getTextStringHeight(context, options['tickLabel']['style']);
        if (axisInfo instanceof DvtAxisInfo.getConstructor('data')) size += labelHeight;
        else if (axisInfo instanceof DvtAxisInfo.getConstructor('time'))
          size += axisInfo.getLabels(context, 1) != null ? labelHeight * 2 : labelHeight;
        else if (axisInfo instanceof DvtAxisInfo.getConstructor('group'))
          size = DvtAxisRenderer._getGroupAxisPreferredSize(
            axis,
            axisInfo,
            size,
            availHeight,
            bHoriz
          );
      } else {
        // Vertical Axis
        if (axisInfo instanceof DvtAxisInfo.getConstructor('data'))
          size += TextUtils.getMaxTextStringWidth(
            context,
            axisInfo.getAllLabels(context, true),
            axisInfo.Options['tickLabel']['style']
          );
        else if (axisInfo instanceof DvtAxisInfo.getConstructor('time')) {
          var innerLabels = axisInfo.getLabels(context);
          var innerLabelWidth = TextUtils.getMaxTextDimensions(innerLabels).w;
          var outerLabels = axisInfo.getLabels(context, 1);
          var outerLabelWidth =
            outerLabels != null ? TextUtils.getMaxTextDimensions(outerLabels).w : 0;
          size += Math.max(innerLabelWidth, outerLabelWidth);
        } else if (axisInfo instanceof DvtAxisInfo.getConstructor('group'))
          size = DvtAxisRenderer._getGroupAxisPreferredSize(
            axis,
            axisInfo,
            size,
            availWidth,
            bHoriz
          );
      }
    }

    if (bHoriz) return new Dimension(availWidth, Math.min(size, availHeight));
    return new Dimension(Math.min(size, availWidth), availHeight);
  },

  /**
   * Renders the axis and updates the available space.
   * @param {DvtAxis} axis The axis being rendered.
   * @param {dvt.Rectangle} availSpace The available space.
   */
  render: (axis, availSpace) => {
    // Calculate the axis extents and increments
    var axisInfo = DvtAxisRenderer._createAxisInfo(axis, availSpace);
    var options = axis.getOptions();

    if (options['rendered'] == 'off') return;

    axis.__setBounds(availSpace.clone());

    DvtAxisRenderer._renderBackground(axis, availSpace);

    // Render the title
    DvtAxisRenderer._renderTitle(axis, axisInfo, availSpace);

    // Render the tick labels
    DvtAxisRenderer._renderLabels(axis, axisInfo, availSpace);
  },

  /**
   * Creates and returns the DvtAxisInfo for the specified axis.
   * @param {DvtAxis} axis The axis being rendered.
   * @param {dvt.Rectangle} availSpace The available space.
   * @return {DvtAxisInfo}
   * @private
   */
  _createAxisInfo: (axis, availSpace) => {
    var axisInfo = DvtAxisInfo.newInstance(axis.getCtx(), axis.getOptions(), availSpace);
    axis.__setInfo(axisInfo);
    return axisInfo;
  },

  /**
   * Returns the gap between the title and the tick labels.
   * @param {DvtAxis} axis
   * @return {number}
   * @private
   */
  _getTitleGap: (axis) => {
    var options = axis.getOptions();
    return DvtAxisDefaults.getGapSize(axis.getCtx(), options, options['layout']['titleGap']);
  },

  /**
   * Renders the axis invisble background. Needed for DnD drop effect.
   * @param {DvtAxis} axis The axis being rendered.
   * @param {dvt.Rectangle} availSpace The available space.
   * @private
   */
  _renderBackground: (axis, availSpace) => {
    var options = axis.getOptions();
    if (!options['dnd']) return;

    var dropOptions = options['dnd']['drop'];
    var isDropTarget =
      Object.keys(dropOptions['xAxis']).length > 0 ||
      Object.keys(dropOptions['yAxis']).length > 0 ||
      Object.keys(dropOptions['y2Axis']).length > 0;
    var dragOptions = options['dnd']['drag'];
    var isDraggable =
      Object.keys(dragOptions['groups']).length > 0 &&
      axis.getInfo() instanceof DvtAxisInfo.getConstructor('group');

    if (isDropTarget || isDraggable) {
      var position = options['position'];
      var isHoriz = position == 'top' || position == 'bottom';
      var yGap = isHoriz ? 4 : 10;
      var xGap = isHoriz ? 10 : 4;
      var background = new Rect(
        axis.getCtx(),
        availSpace.x - xGap,
        availSpace.y - yGap,
        availSpace.w + 2 * xGap,
        availSpace.h + 2 * yGap
      );

      if (isDraggable) background.setClassName('oj-draggable');

      background.setInvisibleFill();
      axis.getCache().putToCache('background', background);
      axis.addChild(background);
    }
  },

  /**
   * Renders the axis title and updates the available space.
   * @param {DvtAxis} axis The axis being rendered.
   * @param {DvtAxisInfo} axisInfo The axis model.
   * @param {dvt.Rectangle} availSpace The available space.
   * @private
   */
  _renderTitle: (axis, axisInfo, availSpace) => {
    // Note: DvtAxisRenderer.getPreferredSize must be updated for any layout changes to this function.
    var options = axis.getOptions();
    if (!options['title']) return;

    // Create the title object and add to axis
    var position = options['position'];

    if (position === 'radial' || position === 'tangential') return; // polar chart doesn't have axis titles

    var bHoriz = options['position'] === 'top' || options['position'] === 'bottom';
    var maxLabelWidth = bHoriz ? availSpace.w : availSpace.h;
    var maxLabelHeight = bHoriz ? availSpace.h : availSpace.w;
    var titleStyle = options['titleStyle'];
    var isMultiLine = DvtAxisRenderer.isWrapEnabled(titleStyle);
    // JET-48758 - Set maxlineCount to originally calculated value in getPrefferedSize if available width for title is less than it was in getPreferredSize.
    // Otherwise, the title might occupy more space due to wrapping which might lead to missing ticklabels.
    var maxLineCount;
    var prefAvailableTitleWidth = axis.getOptionsCache().getFromCache('prefAvailableTitleWidth');
    if (prefAvailableTitleWidth > maxLabelWidth) {
      maxLineCount = axis.getOptionsCache().getFromCache('prefTitleLineCount');
    }
    var title = DvtAxisRenderer._createText(
      axis.getEventManager(),
      axis,
      options['title'],
      titleStyle,
      0,
      0,
      maxLabelWidth,
      maxLabelHeight,
      DvtAxisEventManager.getUIParams('title'),
      isMultiLine,
      maxLineCount
    );

    if (title) {
      // Position the title based on text size and axis position
      var gap = DvtAxisRenderer._getTitleGap(axis);
      var overflow = (axisInfo.getStartOverflow() - axisInfo.getEndOverflow()) / 2;
      var isRTL = Agent.isRightToLeft(axis.getCtx());
      var titleHeight = title.getDimensions().h;
      title.alignCenter();

      // Position the label and update the space
      if (position === 'top') {
        title.setX(availSpace.x + overflow + availSpace.w / 2);
        title.setY(availSpace.y);
        availSpace.y += titleHeight + gap;
        availSpace.h -= titleHeight + gap;
      } else if (position === 'bottom') {
        title.setX(availSpace.x + overflow + availSpace.w / 2);
        title.setY(availSpace.y + availSpace.h - titleHeight);
        availSpace.h -= titleHeight + gap;
      } else if (position === 'left') {
        title.alignMiddle();
        title.setRotation(isRTL ? Math.PI / 2 : (3 * Math.PI) / 2);
        title.setTranslate(availSpace.x + titleHeight / 2, availSpace.y + availSpace.h / 2);
        availSpace.x += titleHeight + gap;
        availSpace.w -= titleHeight + gap;
      } else if (position === 'right') {
        title.alignMiddle();
        title.setRotation(isRTL ? Math.PI / 2 : (3 * Math.PI) / 2);
        title.setTranslate(
          availSpace.x + availSpace.w - titleHeight / 2,
          availSpace.y + availSpace.h / 2
        );
        availSpace.w -= titleHeight + gap;
      }

      axisInfo.setTitle(title);
    }
  },

  /**
   * Renders the tick labels and updates the available space.
   * @param {DvtAxis} axis The axis being rendered.
   * @param {DvtAxisInfo} axisInfo The axis model.
   * @param {dvt.Rectangle} availSpace The available space.
   * @private
   */
  _renderLabels: (axis, axisInfo, availSpace) => {
    // Note: DvtAxisRenderer.getPreferredSize must be updated for any layout changes to this function.
    var options = axis.getOptions();
    if (options['tickLabel']['rendered'] == 'on') {
      // Axis labels are positioned based on the position of the axis.  In layout
      // mode, the labels will be positioned as close to the title as possible to
      // calculate the actual space used.
      var position = options['position'];
      if (position === 'top' || position === 'bottom')
        DvtAxisRenderer._renderLabelsHoriz(axis, axisInfo, availSpace);
      else if (position === 'tangential')
        DvtAxisRenderer._renderLabelsTangent(axis, axisInfo, availSpace);
      else DvtAxisRenderer._renderLabelsVert(axis, axisInfo, availSpace);

      // Render the label separators (applicable only to group axis)
      DvtAxisRenderer._renderGroupSeparators(axis, axisInfo, availSpace);
    }
  },

  /**
   * Renders tick labels for a horizontal axis and updates the available space.
   * @param {DvtAxis} axis The axis being rendered.
   * @param {DvtAxisInfo} axisInfo The axis model.
   * @param {dvt.Rectangle} availSpace The available space.
   * @private
   */
  _renderLabelsHoriz: (axis, axisInfo, availSpace) => {
    // Note: DvtAxisRenderer.getPreferredSize must be updated for any layout changes to this function.
    // Position and add the axis labels.
    var context = axis.getCtx();
    var options = axis.getOptions();
    var position = options['position'];
    var isTickInside = options['tickLabel']['position'] == 'inside';
    var isRTL = Agent.isRightToLeft(context);
    var isGroupAxis = axisInfo instanceof DvtAxisInfo.getConstructor('group');
    var isHierarchical = isGroupAxis && axisInfo.getNumLevels() > 1;

    var levelIdx = isHierarchical ? 0 : null;
    var labels = axisInfo.getLabels(context, levelIdx);

    gap = isHierarchical
      ? DvtAxisDefaults.getGapSize(
          context,
          options,
          options['layout']['hierarchicalLabelGapHeight']
        )
      : 0;
    while (labels) {
      var height = 0;
      var maxLvlHeight = 0;

      for (var i = 0; i < labels.length; i++) {
        var label = labels[i];

        if (label == null) continue;

        var isMultiline =
          label instanceof MultilineText || label instanceof BackgroundMultilineText;

        if (axisInfo.isLabelRotated(levelIdx)) {
          // Truncate to fit. Multiline texts only need fitting here if wrap was disabled.
          var fitText = !isMultiline || (isMultiline && !label.isWrapEnabled());
          if (fitText && !TextUtils.fitText(label, availSpace.h, availSpace.w, axis)) continue;

          // position and add the axis labels
          if (!isRTL) label.alignRight();
          else label.alignLeft();

          if (isHierarchical) {
            height = label.getDimensions().w;
            label.setTranslateY(availSpace.h - height);
            maxLvlHeight = Math.max(maxLvlHeight, height);
          } else label.setTranslateY(availSpace.y);
        } else {
          // not rotated
          if (!isTickInside && label.getDimensions().h - 1 > availSpace.h)
            // -1 to prevent rounding error ()
            continue;

          if (isHierarchical && position === 'bottom') label.setY(availSpace.h);
          else if (position === 'bottom') label.setY(availSpace.y);
          else label.setY(availSpace.y + availSpace.h);

          if (
            !isHierarchical &&
            ((position === 'bottom' && !isTickInside) || (position === 'top' && isTickInside))
          )
            label.alignTop();
          else if (isHierarchical && position === 'top') label.alignTop();
          else label.alignBottom();

          if (isHierarchical) maxLvlHeight = Math.max(maxLvlHeight, label.getDimensions().h);
          else if (isTickInside) {
            var gap = DvtAxisDefaults.getGapSize(
              context,
              options,
              options['layout']['insideLabelGapWidth']
            );
            isRTL ? label.alignRight() : label.alignLeft();
            label.setX(label.getX() + gap * (isRTL ? -1 : 1));
          }
        }

        // group axis labels store the true index of a label in the hierarchy of levels
        // true index necessary for getting proper attributes from axisInfo
        var index = isGroupAxis ? axisInfo.getLabelIdx(label) : i;

        // support for categorical axis tooltip and datatip
        var datatip = axisInfo.getDatatip(index, levelIdx);
        var tooltip = label.getUntruncatedTextString();
        // drilling support
        var drillable = axisInfo.isDrillable(index, levelIdx);
        var group = axisInfo.getGroup(index, levelIdx);

        // Associate with logical object to support automation and tooltips
        var params = DvtAxisEventManager.getUIParams(
          'tickLabel',
          label.getTextString(),
          index,
          levelIdx
        );

        axis
          .getEventManager()
          .associate(
            label,
            new DvtAxisObjPeer(axis, label, group, drillable, tooltip, datatip, params)
          );

        if (!isHierarchical) maxLvlHeight = Math.max(maxLvlHeight, label.getDimensions().h);
        else axisInfo.setLastRenderedLevel(levelIdx);

        axis.addChild(label);
      }
      if (isHierarchical) {
        for (i = 0; i < labels.length; i++) {
          label = labels[i];
          if (label == null) continue;

          var isRotated = axisInfo.isLabelRotated(levelIdx);
          var isOuterLevel = levelIdx < axisInfo.getNumLevels() - 1;
          if (!isRotated && isOuterLevel) {
            // non-rotated outer multiline texts need height adjustment to center
            label.setY(availSpace.h - maxLvlHeight / 2);
            label.alignMiddle();
          } // all rotated texts need height adjustment
          else label.setTranslateY(availSpace.h - maxLvlHeight);
        }

        availSpace.y += maxLvlHeight + gap;
        availSpace.h -= maxLvlHeight + gap;
        levelIdx++;
        labels = axisInfo.getLabels(axis.getCtx(), levelIdx);
      } else {
        availSpace.y += maxLvlHeight;
        availSpace.h -= maxLvlHeight;
        labels = null;
      }
    }

    // Render the nested labels (level 2) for time axis.
    if (axisInfo instanceof DvtAxisInfo.getConstructor('time')) {
      labels = axisInfo.getLabels(axis.getCtx());
      var lv2Labels = axisInfo.getLabels(axis.getCtx(), 1);
      var offset = 0;

      if (lv2Labels != null) {
        for (i = 0; i < lv2Labels.length; i++) {
          label = lv2Labels[i];
          if (label == null) continue;
          if (label.getDimensions().h - 1 > availSpace.h)
            // -1 to prevent rounding error ()
            continue;

          // Associate with logical object to support automation and tooltips
          axis
            .getEventManager()
            .associate(
              label,
              new SimpleObjPeer(
                null,
                null,
                null,
                DvtAxisEventManager.getUIParams('tickLabel', label.getTextString())
              )
            );

          // align with level 1 label
          var overflow1 = 0;
          var overflow2 = 0;
          var maxOverflow = axisInfo.getOptions()['_maxOverflowCoord'];
          var minOverflow = axisInfo.getOptions()['_minOverflowCoord'];
          if (labels[i] != null) {
            offset = labels[i].getDimensions().w / 2;
            overflow1 = axisInfo._level1Overflow[i];
            overflow2 = axisInfo._level2Overflow[i];
          }

          // Code below skips attempt to align level2 label if it overflows and level1 label does not
          // This is because if the level2 label overflows it should not be moved inward, else we risk creating a label overlap
          if (overflow1 == 0 && overflow2 == 0) {
            // Check that shifting by offset will not cause overflow
            var x = label.getX();
            var newCoord;
            if (isRTL) {
              newCoord = x + offset <= maxOverflow ? x + offset : maxOverflow;
              label.setX(newCoord);
            } else {
              newCoord = x - offset >= minOverflow ? x - offset : minOverflow;
              label.setX(newCoord);
            }
          } else if (overflow1 < 0)
            // level1 label is at the left edge, push level2 label out to minOverflow coord
            label.setX(minOverflow);
          else if (overflow1 > 0)
            // level1 label is at the right edge, push level2 label out to maxOverflow coord
            label.setX(maxOverflow);

          label.alignTop();
          label.setY(availSpace.y);
          axis.addChild(label);
        }
      }
    }
  },

  /**
   * Renders tick labels for a vertical axis and updates the available space.
   * @param {DvtAxis} axis The axis being rendered.
   * @param {DvtAxisInfo} axisInfo The axis model.
   * @param {dvt.Rectangle} availSpace The available space.
   * @private
   */
  _renderLabelsVert: (axis, axisInfo, availSpace) => {
    // Note: DvtAxisRenderer.getPreferredSize must be updated for any layout changes to this function.
    var options = axis.getOptions();
    var position = options['position'];
    var context = axis.getCtx();
    var isRTL = Agent.isRightToLeft(context);
    var isNumerical = axisInfo instanceof DvtAxisInfo.getConstructor('data');
    var isTickInside = options['tickLabel']['position'] == 'inside';
    var labels;
    var gap;
    var maxLvlWidth;
    var isGroupAxis = axisInfo instanceof DvtAxisInfo.getConstructor('group');
    var isHierarchical = isGroupAxis && axisInfo.getNumLevels() > 1;

    // Hierarchical group axis labels
    var levelIdx = isHierarchical ? 0 : null;
    labels = axisInfo.getLabels(axis.getCtx(), levelIdx);

    var labelX = 0;
    if (!isHierarchical) {
      // Categorical and time labels are aligned left is position=right, aligned right if position=left.
      // Numerical labels are always aligned right.
      if (position === 'radial') {
        gap = DvtAxisDefaults.getGapSize(context, options, options['layout']['radialLabelGap']);
        labelX = availSpace.x + availSpace.w / 2;
        if (isRTL) labelX += gap + TextUtils.getMaxTextDimensions(labels).w;
        else labelX -= gap;
      } else if (position === 'left') {
        labelX = availSpace.x + availSpace.w;
        if (isNumerical && isTickInside) labelX += TextUtils.getMaxTextDimensions(labels).w;
      } else {
        // position == 'right'
        labelX = availSpace.x;
        if (isNumerical && !isTickInside) labelX += TextUtils.getMaxTextDimensions(labels).w;
      }
    } else {
      gap = DvtAxisDefaults.getGapSize(
        context,
        options,
        options['layout']['hierarchicalLabelGapWidth']
      );
      maxLvlWidth = TextUtils.getMaxTextDimensions(labels).w;
    }

    var formatLabelVert = (label, index) => {
      var isMultiline =
        label instanceof MultilineText || label instanceof BackgroundMultilineText;
      var fitText = !isMultiline || (isMultiline && !label.isWrapEnabled()); // Multiline texts only need fitting if wrap was disabled.

      if (isHierarchical && TextUtils.getMaxTextDimensions(labels).w - 1 > availSpace.w)
        // -1 to prevent rounding error ()
        return;
      else if (
        !isHierarchical &&
        !isTickInside &&
        fitText &&
        !TextUtils.fitText(label, availSpace.w, availSpace.h, axis)
      )
        return;

      // group axis labels store the true index of a label in the hierarchy of levels
      // true index necessary for getting proper attributes from axisInfo
      index = isGroupAxis ? axisInfo.getLabelIdx(label) : index;

      // support for categorical axis tooltip and datatip
      var datatip = axisInfo.getDatatip(index, levelIdx);
      var tooltip = label.getUntruncatedTextString();
      // drilling support
      var drillable = axisInfo.isDrillable(index, levelIdx);
      var group = axisInfo.getGroup(index, levelIdx);

      // Associate with logical object to support automation and tooltips
      var params = DvtAxisEventManager.getUIParams(
        'tickLabel',
        label.getTextString(),
        index,
        levelIdx
      );

      axis
        .getEventManager()
        .associate(
          label,
          new DvtAxisObjPeer(axis, label, group, drillable, tooltip, datatip, params)
        );

      if (!isHierarchical) {
        label.setX(labelX);
        if (!isNumerical && position == 'right') label.alignLeft();
        else label.alignRight();

        if (isTickInside) {
          label.alignBottom();
          label.setY(
            label.getY() -
              DvtAxisDefaults.getGapSize(
                context,
                options,
                options['layout']['insideLabelGapHeight']
              )
          );
        }

        if (position === 'radial') {
          var labelY = label.getY();
          label.setY(availSpace.y + availSpace.h / 2 - labelY);

          // draw bounding box to improve readability
          var bboxDims = label.getDimensions();
          var padding = bboxDims.h * 0.15;
          var cmd = PathUtils.roundedRectangle(
            bboxDims.x - padding,
            bboxDims.y,
            bboxDims.w + 2 * padding,
            bboxDims.h,
            2,
            2,
            2,
            2
          );
          var bbox = new Path(axis.getCtx(), cmd);
          var bgColor = label.getCSSStyle().getStyle(CSSStyle.BACKGROUND_COLOR);
          var plotAreaPosition =
            labelY + bboxDims.h / 2 > axisInfo.getEndCoord() &&
            axis.getOptions()['polarGridShape'] == 'circle'
              ? 'outside'
              : 'inside';
          if (bgColor) bbox.setSolidFill(bgColor);
          else bbox.setClassName('oj-chart-polar-axis-tick-label-' + plotAreaPosition);
          axis.addChild(bbox);
        }
      } else {
        label.alignRight();
        label.setX(isRTL ? availSpace.w : availSpace.x + maxLvlWidth);
        axisInfo.setLastRenderedLevel(levelIdx);
      }
      axis.addChild(label);
    };

    while (labels) {
      for (var i = 0; i < labels.length; i++) {
        var label = labels[i];
        if (label != null) formatLabelVert(label, i);
      }
      if (isHierarchical) {
        availSpace.x += maxLvlWidth + gap;
        availSpace.w -= maxLvlWidth + gap;
        levelIdx++;
        labels = axisInfo.getLabels(axis.getCtx(), levelIdx);
        maxLvlWidth = labels ? TextUtils.getMaxTextDimensions(labels).w : null;
      } else break;
    }

    if (axisInfo instanceof DvtAxisInfo.getConstructor('time')) {
      // Render the nested labels (level 2).
      var lv2Labels = axisInfo.getLabels(axis.getCtx(), 1);
      if (lv2Labels != null) {
        for (i = 0; i < lv2Labels.length; i++) {
          label = lv2Labels[i];
          if (label != null) formatLabelVert(label, i);
        }
      }
    }
  },

  /**
   * Renders tick labels for a tangential axis and updates the available space.
   * @param {DvtAxis} axis The axis being rendered.
   * @param {DvtAxisInfo} axisInfo The axis model.
   * @param {dvt.Rectangle} availSpace The available space.
   * @private
   */
  _renderLabelsTangent: (axis, axisInfo, availSpace) => {
    var labels = axisInfo.getLabels(axis.getCtx());
    for (var i = 0; i < labels.length; i++) {
      var label = labels[i];
      if (label == null) continue;
      var maxWidth = availSpace.w / 2 - Math.abs(label.getX());
      var maxHeight = availSpace.h / 2 - Math.abs(label.getY());
      if (TextUtils.fitText(label, maxWidth, maxHeight, axis)) {
        // truncation

        // group axis labels store the true index of a label in the hierarchy of levels
        // true index necessary for getting proper attributes from axisInfo
        var index =
          axisInfo instanceof DvtAxisInfo.getConstructor('group') ? axisInfo.getLabelIdx(label) : i;

        // support for categorical axis tooltip and datatip
        var datatip = axisInfo.getDatatip(index);
        var tooltip = label.getUntruncatedTextString();
        // drilling support
        var drillable = axisInfo.isDrillable(index);
        var group = axisInfo.getGroup(index);

        // Associate with logical object to support automation and tooltips
        var params = DvtAxisEventManager.getUIParams(
          'tickLabel',
          label.getTextString(),
          index,
          null
        );
        axis
          .getEventManager()
          .associate(
            label,
            new DvtAxisObjPeer(axis, label, group, drillable, tooltip, datatip, params)
          );

        label.setTranslateX(availSpace.x + availSpace.w / 2);
        label.setTranslateY(availSpace.y + availSpace.h / 2);
        axis.addChild(label);
      }
    }
  },

  /**
   * Creates and adds a dvt.Text object to a container. Will truncate and add tooltip as necessary.
   * @param {dvt.EventManager} eventManager
   * @param {dvt.Container} container The container to add the text object to.
   * @param {String} textString The text string of the text object.
   * @param {dvt.CSSStyle} cssStyle The css style to apply to the text object.
   * @param {number} x The x coordinate of the text object.
   * @param {number} y The y coordinate of the text object.
   * @param {number} width The width of available text space.
   * @param {number} height The height of the available text space.
   * @param {object} params Additional parameters that will be passed to the logical object.
   * @param {boolean=} bMultiLine True if text can use multiple lines
   * @param {number} maxLineCount Maximum number of lines allowed in the case of multi line texts.
   * @return {dvt.OutputText|dvt.MultilineText} The created text object. Can be null if no text object could be created in the given space.
   * @private
   */
  _createText: (
    eventManager,
    container,
    textString,
    cssStyle,
    x,
    y,
    width,
    height,
    params,
    bMultiLine,
    maxLineCount
  ) => {
    var text;
    if (bMultiLine) {
      text = new MultilineText(container.getCtx(), textString, x, y);
      text.setMaxLines(maxLineCount != null ? maxLineCount : DvtAxisRenderer._MAX_TITLE_LINE_WRAP);
      text.setCSSStyle(cssStyle);
      text.wrapText(width, height, 1);
    } else {
      text = new OutputText(container.getCtx(), textString, x, y);
      text.setCSSStyle(cssStyle);
    }

    if (TextUtils.fitText(text, width, height, container)) {
      // Associate with logical object to support automation and truncation
      eventManager.associate(
        text,
        new SimpleObjPeer(text.getUntruncatedTextString(), null, null, params)
      );
      return text;
    }

    return null;
  },

  /**
   * Renders the separators between group labels
   * @param {DvtAxis} axis The axis being rendered.
   * @param {DvtAxisInfo} axisInfo The axis model.
   * @param {dvt.Rectangle} availSpace The available space.
   * @private
   */
  _renderGroupSeparators: (axis, axisInfo, availSpace) => {
    if (
      axisInfo instanceof DvtAxisInfo.getConstructor('group') &&
      axisInfo.areSeparatorsRendered()
    ) {
      var numLevels = axisInfo.getNumLevels();
      var separatorStartLevel = axisInfo.getSeparatorStartLevel();

      // only draw separators when there is more than one level of labels, and at least one level to apply separators to
      if (numLevels <= 1 || separatorStartLevel <= 0) return;

      var options = axis.getOptions();
      var position = options['position'];
      var isHoriz = position == 'top' || position == 'bottom';
      var context = axis.getCtx();
      var isRTL = Agent.isRightToLeft(context);

      var color = axisInfo.getSeparatorColor();
      var lineStroke = new Stroke(color, 1, 1);
      var prevLevelSize = 0;
      var gap = isHoriz
        ? DvtAxisDefaults.getGapSize(
            context,
            options,
            options['layout']['hierarchicalLabelGapHeight']
          )
        : DvtAxisDefaults.getGapSize(
            context,
            options,
            options['layout']['hierarchicalLabelGapWidth']
          );
      var startOffset = options['startGroupOffset'];
      var endOffset = options['endGroupOffset'];

      var x1, y1, x2, y2, x3, x4;
      /*
       * orientation = 'vertical'                     if rotated:
       * (x1, y1)                        (x2, y1)     (x1, y1)                 (x2, y1)
       *    |                               |            |     rotated label      |
       *    ------------- label -------------            --------------------------
       * (x1, y2)    (x3, y2)(x4, y2)    (x2, y2)     (x1, y2)                 (x2, y2)
       *
       *
       * orientation = 'horizontal'
       * (x1, y1) _______ (x2, y1)
       *         |
       *         |
       *         | label
       *         |
       *         |
       * (x1, y2) _______ (x2, y2)
       */

      // process from the innermost level that was rendered
      for (var level = separatorStartLevel; level >= 0; level--) {
        var labels = axisInfo.getLabels(axis.getCtx(), level);
        var maxDims = TextUtils.getMaxTextDimensions(labels);
        var isRotated = axisInfo.isLabelRotated(level);
        var levelSize = isRotated || !isHoriz ? maxDims.w : maxDims.h;

        if (levelSize == 0) {
          // no labels to draw separators between
          prevLevelSize = levelSize;
          continue;
        }

        // variables to keep track of whether certain edge cases apply
        var prevLabelRendered = false; // previous label exists, does not have blank name, and is within the viewport
        var prevLabelEmpty = null; // previous label exists, but has a blank name (uneven heirarchy)

        // Start drawing separators from second innermost level rendered.
        if (level < separatorStartLevel) {
          for (var i = 0; i < labels.length; i++) {
            var label = labels[i];
            if (label == null) continue;

            var index = axisInfo.getLabelIdx(label);
            var isEmptyLabel = axisInfo.getLabelAt(index, level).length == 0; // label exists, but has a blank name (uneven heirarchy)

            if (isEmptyLabel) continue;

            // empty label at first or last position in the outermost level
            var eraseCornerEdge =
              isEmptyLabel && level == 0 && (index == 0 || index == labels.length - 1);

            var isFirstLabel = label && labels[index - 1] == null;
            var isLastLabel = label && labels[index + 1] == null;

            var start = axisInfo.getStartIdx(index, level);
            var end = axisInfo.getEndIdx(index, level);

            if (isHoriz) {
              // HORIZONTAL AXIS SEPARATORS

              // draw vertical lines, when necessary, around label
              if (label) {
                var yCoord;
                if (
                  label instanceof MultilineText ||
                  label instanceof BackgroundMultilineText
                )
                  yCoord = label.getYAlignCoord();
                else yCoord = label.getY();

                x1 = axisInfo.getCoordAt(start - startOffset);
                y1 = !isRotated
                  ? yCoord - levelSize / 2 - prevLevelSize * 0.5 - gap
                  : yCoord + prevLevelSize * 0.5;
                x2 = axisInfo.getCoordAt(end + endOffset);
                y2 = !isRotated ? yCoord : yCoord + levelSize + prevLevelSize + 2 * gap;

                if ((!isEmptyLabel || !eraseCornerEdge) && prevLabelRendered == false && x1 != null)
                  DvtAxisRenderer._addSeparatorLine(axis, lineStroke, x1, y2, x1, y1);

                if (x2 != null && !eraseCornerEdge)
                  DvtAxisRenderer._addSeparatorLine(axis, lineStroke, x2, y2, x2, y1);
              }

              // draw horizontal lines, when necessary, around non-empty labels
              if (!isEmptyLabel) {
                if (label)
                  var labelWidth = isRotated ? label.getDimensions().h : label.getDimensions().w;

                x1 =
                  isFirstLabel && prevLabelEmpty == false
                    ? axisInfo.getStartCoord()
                    : axisInfo.getBoundedCoordAt(start - startOffset);
                if (isFirstLabel) isFirstLabel = false;
                var nextLabel = axisInfo.getLabelAt(index + 1, level);
                x2 =
                  isLastLabel && nextLabel && nextLabel.length > 0
                    ? axisInfo.getEndCoord()
                    : axisInfo.getBoundedCoordAt(end + endOffset);

                x3 = label
                  ? isRTL
                    ? label.getX() + labelWidth * 0.5
                    : label.getX() - labelWidth * 0.5
                  : axisInfo.getBoundedCoordAt(end + endOffset);
                x4 = label
                  ? isRTL
                    ? label.getX() - labelWidth * 0.5
                    : label.getX() + labelWidth * 0.5
                  : axisInfo.getBoundedCoordAt(start - startOffset);

                if (label) {
                  if (isRotated)
                    // draw horizontal line beneath rotated label
                    DvtAxisRenderer._addSeparatorLine(axis, lineStroke, x1, y2, x2, y2);
                  else {
                    // draw horizontal lines on either size of rendered label
                    var spacing = isRTL
                      ? -label.getDimensions().h * 0.5
                      : label.getDimensions().h * 0.5; // small space between end of horizontal lines and label
                    var drawRightLine = isRTL ? x1 > x3 - spacing : x1 < x3 - spacing;
                    var drawLeftLine = isRTL ? x4 + spacing > x2 : x4 + spacing < x2;

                    if (drawRightLine)
                      DvtAxisRenderer._addSeparatorLine(axis, lineStroke, x1, y2, x3 - spacing, y2);

                    if (drawLeftLine)
                      DvtAxisRenderer._addSeparatorLine(axis, lineStroke, x4 + spacing, y2, x2, y2);
                  }
                }
              }
            } else {
              // VERTICAL AXIS SEPARATORS

              // draw horizontal lines, when necessary, around label
              if (label) {
                x1 = !isRTL ? label.getX() + gap * 0.5 : label.getX() - levelSize - gap * 0.5;
                y1 = axisInfo.getCoordAt(start - startOffset);
                x2 = !isRTL ? label.getX() - levelSize - gap * 0.5 : label.getX() + gap * 0.5;
                y2 = axisInfo.getCoordAt(end + endOffset);

                if (
                  ((!isEmptyLabel && prevLabelRendered == false) ||
                    (index == 0 && isEmptyLabel && level != 0)) &&
                  y1 != null
                )
                  DvtAxisRenderer._addSeparatorLine(axis, lineStroke, x1, y1, x2, y1);

                if (y2 != null && !eraseCornerEdge)
                  DvtAxisRenderer._addSeparatorLine(axis, lineStroke, x2, y2, x1, y2);
              }

              // draw vertical lines, when necessary, around non-empty labels
              if (!isEmptyLabel) {
                y1 =
                  isFirstLabel && prevLabelEmpty == false
                    ? 0
                    : axisInfo.getBoundedCoordAt(start - startOffset);
                if (isFirstLabel) isFirstLabel = false;
                nextLabel = axisInfo.getLabelAt(index + 1, level);
                y2 =
                  isLastLabel && nextLabel && nextLabel.length > 0
                    ? axisInfo.getEndCoord()
                    : axisInfo.getBoundedCoordAt(end + endOffset);

                if (label)
                  // draw vertical line around label
                  DvtAxisRenderer._addSeparatorLine(axis, lineStroke, x2, y1, x2, y2);
              }
            }
            // information about previous label
            prevLabelRendered = !isEmptyLabel && label != null;
            prevLabelEmpty = label != null || (label == null && isEmptyLabel); // TODO TAMIKA: IS THIS NECESSARY
          }
        }
        prevLevelSize = levelSize; // save height or width of previous level
      }
    }
    return;
  },

  /**
   * Renders separator line
   * @param {DvtAxis} axis The axis on which the separators are rendered.
   * @param {dvt.Stroke} lineStroke The stroke for the line.
   * @param {Number} x1 The first xCoordinate of the line.
   * @param {Number} y1 The first yCoordinate of the line.
   * @param {Number} x2 The second xCoordinate of the line.
   * @param {Number} y2 The second yCoordinate of the line.
   * @private
   */
  _addSeparatorLine: (axis, lineStroke, x1, y1, x2, y2) => {
    var line = new Line(axis.getCtx(), x1, y1, x2, y2);
    line.setStroke(lineStroke);
    line.setPixelHinting(true);
    axis.addChild(line);

    return;
  },

  /**
   * Gets the preferred size for a group axis, which may include hierarchical labels
   * @param {DvtAxis} axis The axis
   * @param {DvtGroupAxisInfo} axisInfo The group axis info
   * @param {Number} size The current preferred size of the axis
   * @param {Number} availSize The maximum availHeight or availWidth of the axis
   * @param {Boolean} bHoriz Whether or not the axis is vertical of horizontal
   * @return {Number}
   * @private
   */
  _getGroupAxisPreferredSize: (axis, axisInfo, size, availSize, bHoriz) => {
    var context = axis.getCtx();
    var options = axis.getOptions();
    var numLevels = axisInfo.getNumLevels();
    var gapName = bHoriz ? 'hierarchicalLabelGapHeight' : 'hierarchicalLabelGapWidth';
    var gap =
      numLevels > 1 ? DvtAxisDefaults.getGapSize(context, options, options['layout'][gapName]) : 0;
    for (var level = 0; level < numLevels; level++) {
      // allocate space outermost to innermost
      var labelSize; // corresponds to label height if bHoriz, label width if not

      if (axisInfo.isAutoRotate()) {
        // performance optimization
        var labelStrings = [];
        var labelStyles = [];
        // increase performance by only measuring a subset of labels to begin with, as majority will be skipped.
        var increment = axisInfo.getSkipIncrement();

        for (var i = 0; i < axisInfo.getGroupCount(); i += increment) {
          labelStrings.push(axisInfo.getLabelAt(i, 0));
          var style = axisInfo.getLabelStyleAt(i, 0);
          if (!style) style = axisInfo.Options['tickLabel']['style'];
          labelStyles.push(style);
        }
        labelSize = TextUtils.getMaxTextStringWidth(context, labelStrings, labelStyles);
      } else {
        var labels = axisInfo.getLabels(context, level);
        if (bHoriz) {
          var maxDims = TextUtils.getMaxTextDimensions(labels);
          labelSize = axisInfo.isLabelRotated(level) ? maxDims.w : maxDims.h;
        } else labelSize = TextUtils.getMaxTextDimensions(labels).w;
      }

      if (size + labelSize <= availSize) size += labelSize + gap;
      else {
        if (level == 0)
          // Outermost level labels were too big, assign all of availSize
          size = availSize;
        break;
      }
    }
    if (level != 0) size -= gap; // last hierarchical level rendered doesn't need gap

    return size;
  },

  /**
   * Get the height and line count of the axis title.
   * @param {dvt.Context} context The axis context
   * @param {Object} options The options for the axis
   * @param {Number} availWidth The maximum available width for the title
   * @param {Number} availHeight The maximum available height for the title
   * @return {Object} Contains the height and line count of the axis title
   */
  getTitleHeight: (context, options, availWidth, availHeight) => {
    var titleHeight = 0;
    var text;

    if (options['title']) {
      if (DvtAxisRenderer.isWrapEnabled(options['titleStyle'])) {
        text = new MultilineText(context, options['title'], 0, 0);
        text.setMaxLines(DvtAxisRenderer._MAX_TITLE_LINE_WRAP);
        text.setCSSStyle(options['titleStyle']);
        text.wrapText(availWidth, availHeight, 1);
        titleHeight =
          TextUtils.getTextStringHeight(context, options['titleStyle']) * text.getLineCount();
      } else titleHeight = TextUtils.getTextStringHeight(context, options['titleStyle']);
    }

    return {
      height: titleHeight,
      lineCount: text ? text.getLineCount() : 0
    };
  },

  /**
   * Returns true if the white space property is not nowrap.
   * @param {dvt.CSSStyle} cssStyle The css style to be evaluated
   * @return {boolean}
   */
  isWrapEnabled: (cssStyle) => {
    var whiteSpaceValue = cssStyle.getStyle(CSSStyle.WHITE_SPACE);
    // checking noWrap for backwards compatibility
    if (whiteSpaceValue == 'nowrap' || whiteSpaceValue == 'noWrap') return false;
    return true;
  }
};

/**
 * Calculated axis information and drawable creation for a group axis.
 * @param {dvt.Context} context
 * @param {object} options The object containing specifications and data for this component.
 * @param {dvt.Rectangle} availSpace The available space.
 * @class
 * @constructor
 * @extends {DvtAxisInfo}
 */
class DvtGroupAxisInfo extends DvtAxisInfo {
  constructor(context, options, availSpace) {
    super(context, options, availSpace);

    /**
     * The max amount of lines we allow in label wrapping.
     * Needed to prevent rotated labels from greedily wrapping along the length of the xAxis
     * @private
     */
    this._MAX_LINE_WRAP = 3;

    /**
     * The threshold for how small the group width can become before skipping label measurement checks
     * and defaulting to rotation in order to improve performance.
     * @private
     */
    this._ROTATE_THRESHOLD = 12;

    // Flip horizontal axes for BIDI
    var isRTL = Agent.isRightToLeft(context);
    var isHoriz = this.Position == 'top' || this.Position == 'bottom';
    if (isHoriz && isRTL) {
      var temp = this.StartCoord;
      this.StartCoord = this.EndCoord;
      this.EndCoord = temp;
    }

    this._levelsArray = [];
    this._groupCount = this._generateLevelsArray(options['groups'], 0, this._levelsArray, 0); // populates this._levelsArray and returns groupCount
    this._numLevels = this._levelsArray.length;
    this._areSeparatorsRendered = options['groupSeparators']['rendered'] != 'off';
    this._separatorColor = options['groupSeparators']['color'];
    this._lastRenderedLevel = null;
    this._drilling = options['drilling'];

    // Calculate the increment and add offsets if specified
    var endOffset = options['endGroupOffset'] > 0 ? Number(options['endGroupOffset']) : 0;
    var startOffset = options['startGroupOffset'] > 0 ? Number(options['startGroupOffset']) : 0;

    // Set the axis min/max
    this.DataMin = 0;
    this.DataMax = this._groupCount - 1;

    this.GlobalMin = options['min'] == null ? this.DataMin - startOffset : options['min'];
    this.GlobalMax = options['max'] == null ? this.DataMax + endOffset : options['max'];

    // Set min/max by start/endGroup
    var startIndex = this.getGroupIndex(options['viewportStartGroup']);
    var endIndex = this.getGroupIndex(options['viewportEndGroup']);
    if (startIndex != -1) this.MinValue = startIndex - startOffset;
    if (endIndex != -1) this.MaxValue = endIndex + endOffset;

    // Set min/max by viewport min/max
    if (options['viewportMin'] != null) this.MinValue = options['viewportMin'];
    if (options['viewportMax'] != null) this.MaxValue = options['viewportMax'];

    // If min/max is still undefined, fall back to global min/max
    if (this.MinValue == null) this.MinValue = this.GlobalMin;
    if (this.MaxValue == null) this.MaxValue = this.GlobalMax;

    if (this.GlobalMin > this.MinValue) this.GlobalMin = this.MinValue;
    if (this.GlobalMax < this.MaxValue) this.GlobalMax = this.MaxValue;

    this._groupWidthRatios = options['_groupWidthRatios'];
    this._processGroupWidthRatios();

    this._startBuffer = isRTL ? options['rightBuffer'] : options['leftBuffer'];
    this._endBuffer = isRTL ? options['leftBuffer'] : options['rightBuffer'];

    this._isLabelRotated = [];
    for (var i = 0; i < this._numLevels; i++) this._isLabelRotated.push(false);

    this._renderGridAtLabels = options['_renderGridAtLabels'];

    this._labels = null;

    // Initial height/width that will be available for the labels. Used for text wrapping.
    this._maxSpace = isHoriz ? availSpace.h : availSpace.w;
    if (options['rendered'] != 'off') {
      var titleHeight = DvtAxisRenderer.getTitleHeight(
        context,
        options,
        isHoriz ? availSpace.w : availSpace.h,
        isHoriz ? availSpace.h : availSpace.w
      ).height;
      this._maxSpace -=
        titleHeight != 0
          ? titleHeight +
            DvtAxisDefaults.getGapSize(context, options, options['layout']['titleGap'])
          : 0;
    }

    this._maxLineWrap = this._MAX_LINE_WRAP;
  }

  /**
   * Processes group width ratios to support bar chart with varying widths.
   * @private
   */
  _processGroupWidthRatios() {
    // Edge case: less than two groups
    if (!this._groupWidthRatios || this._groupWidthRatios.length < 2) {
      this._groupWidthRatios = null;
      return;
    }

    // Compute the sums of the group widths that are contained within the viewport
    var sum = 0;
    var groupMin, groupMax;
    for (var g = 0; g < this._groupCount; g++) {
      groupMin = g == 0 ? this.MinValue : Math.max(g - 0.5, this.MinValue);
      groupMax = g == this._groupCount - 1 ? this.MaxValue : Math.min(g + 0.5, this.MaxValue);
      if (groupMax > groupMin) sum += (groupMax - groupMin) * this._groupWidthRatios[g];
    }

    // Divide the total viewport length (in pixels) proportionally based on the group width ratios.
    var totalWidth = this.EndCoord - this.StartCoord;
    this._groupWidths = this._groupWidthRatios.map((ratio) => {
      return (ratio * totalWidth) / sum;
    });

    // Construct borderValues array which stores the the value location of the group boundaries.
    this._borderValues = [];
    for (var g = 0; g < this._groupWidthRatios.length - 1; g++) {
      this._borderValues.push(g + 0.5);
    }

    // Construct borderCoords array which stores the coord location of the group boundaries.
    this._borderCoords = [];
    var anchor = Math.min(Math.max(Math.round(this.MinValue), 0), this._borderValues.length - 1);
    this._borderCoords[anchor] =
      this.StartCoord + (this._borderValues[anchor] - this.MinValue) * this._groupWidths[anchor];
    for (
      var g = anchor + 1;
      g < this._borderValues.length;
      g++ // compute borderCoords after the anchor
    )
      this._borderCoords[g] = this._borderCoords[g - 1] + this._groupWidths[g];
    for (
      var g = anchor - 1;
      g >= 0;
      g-- // compute borderCoords before the anchor
    )
      this._borderCoords[g] = this._borderCoords[g + 1] - this._groupWidths[g + 1];
  }

  /**
   * Rotates the labels of the horizontal axis by 90 degrees and skips the labels if necessary.
   * @param {Array} labels An array of dvt.Text labels for the axis.
   * @param {dvt.Container} container
   * @param {number} overflow How much overflow the rotated labels will have.
   * @param {number} level The level the labels array corresponds to
   * @return {Array} The array of dvt.Text labels for the axis.
   * @private
   */
  _rotateLabels(labels, container, overflow, level) {
    var text;
    var x;
    var context = this.getCtx();
    var isRTL = Agent.isRightToLeft(context);
    var isHierarchical = this._numLevels > 1;

    if (level == null) level = this._numLevels - 1;

    this._isLabelRotated[level] = true;

    // TODO: For hierarchical labels, overflow change due to rotation is ignored.
    // Ideally we need to set the overflow at the end after all levels have been rotated.
    if (!isHierarchical) this._setOverflow(overflow, overflow, labels);

    for (var i = 0; i < labels.length; i++) {
      text = labels[i];
      if (text == null) continue;
      x = text.getX();

      // Wrap multiline text in new height/width dimensions
      var isMultiline =
        text instanceof MultilineText || text instanceof BackgroundMultilineText;
      if (isMultiline) {
        var groupSpan =
          this.getGroupWidth() * (this.getEndIdx(i, level) - this.getStartIdx(i, level) + 1);
        // Estimate if there is room for at least one wrap, and either attempt to wrap or disable wrap on the text
        // Note: This estimate may end up disabling wrap for text that may have just fit, but sufficiently excludes text that
        // will have definitely not fit.
        if (text.getLineHeight() * 2 < groupSpan && this._maxSpace > 0)
          text.wrapText(this._maxSpace, text.getLineHeight() * this._MAX_LINE_WRAP, 1);
        else text.setWrapEnabled(false);
      }

      text.setX(0);
      text.setY(0);
      if (isRTL) text.setRotation(Math.PI / 2);
      else text.setRotation((3 * Math.PI) / 2);
      text.setTranslateX(x);
    }

    var labelDims = this.GetLabelDims(labels, container, level); // the guess returns the exact heights

    // Wrapped labels
    if (DvtAxisRenderer.isWrapEnabled(this.Options['tickLabel']['style']) && this._maxSpace > 0) {
      var updateLabelDims = this._sanitizeWrappedText(
        context,
        labelDims,
        labels,
        true,
        isHierarchical
      );
      // Recalculate label dims for skipping
      if (updateLabelDims) labelDims = this.GetLabelDims(labels, container, level);
    }

    return this.SkipLabels(labels, labelDims);
  }

  /**
   * Checks if any label should be re-wrapped due to overlap and re-wraps text if needed to minimize overlap.
   * Updates remaining space available for wrapping hierarchical labels.
   * @param {dvt.Context} context
   * @param {Array} labelDims An array of dvt.Text dimensions for the axis.
   * @param {Array} labels An array of dvt.Text labels for the axis.
   * @param {Boolean} isRotated Whether or not labels are horizontal and rotated, or vertical.
   * @param {Boolean} isHierarchical Whether or not axis has hierarchical labels
   * @return {Boolean} Whether or not labels were re-wrapped
   * @private
   */
  _sanitizeWrappedText(context, labelDims, labels, isRotated, isHierarchical) {
    // Check any label should be wrapped
    var updateLabelDims = this._calculateMaxWrap(labelDims, labels, isRotated);

    var totalSpace = 0;
    // Re-wraps text if needed to minimize overlap, updates remaining space available for wrapping hierarchical labels.
    for (var i = 0; i < labels.length; i++) {
      var text = labels[i];
      if (!text) continue;

      var isMultiline =
        text instanceof MultilineText || text instanceof BackgroundMultilineText;

      // Re-wrap to minimize overlap
      if (updateLabelDims && isMultiline && text.isWrapEnabled())
        text.wrapText(this._maxSpace, text.getLineHeight() * this._maxLineWrap, 1);

      // Keep track of maximum height of this rotated level.
      if (isHierarchical) totalSpace = Math.max(totalSpace, text.getDimensions().w);

      // Make sure texts, which may or may not have been re-wrapped are aligned center
      text.alignMiddle();
    }

    // Update remaining space available for wrapping hierarchical labels.
    if (isHierarchical) {
      var gap = isRotated
        ? this.Options['layout']['hierarchicalLabelGapHeight']
        : this.Options['layout']['hierarchicalLabelGapWidth'];
      this._maxSpace -= totalSpace + DvtAxisDefaults.getGapSize(context, this.Options, gap);
    }

    return updateLabelDims;
  }

  /**
   * Updates the maximum lines labels will be allowed to have to minimize overlap
   * @param {Array} labelDims An arry of the current label dimensions
   * @param {Array} labels An array of dvt.Text labels for the axis.
   * @param {Boolean} isRotated Whether or not the axis is horizontal
   * @return {Boolean} Whether or not labels will need to be re-wrapped
   * @private
   */
  _calculateMaxWrap(labelDims, labels, isRotated) {
    var updateLabelDims = false;

    // Estimate and update label dims of text if they were to be wrapped with current maxLineWrap
    // Decrease maxLineWrap until 1 if it is estimated text will still overlap
    while (this.IsOverlapping(labelDims, 0) && this._maxLineWrap > 1) {
      updateLabelDims = true;
      for (var i = 0; i < labels.length; i++) {
        var text = labels[i];
        if (text instanceof MultilineText || text instanceof BackgroundMultilineText) {
          if (text.getLineCount() == this._maxLineWrap) {
            var lineHeight = text.getLineHeight();
            if (isRotated) labelDims[i].w -= lineHeight;
            else {
              labelDims[i].y += lineHeight * 0.5;
              labelDims[i].h -= lineHeight;
            }
          }
        }
      }
      this._maxLineWrap--;
    }

    return updateLabelDims;
  }

  /**
   * @override
   */
  isLabelRotated(level) {
    if (level == null) level = this._numLevels - 1;
    return this._isLabelRotated[level];
  }

  /**
   * Sets the start/end overflow of the axis.
   * @param {number} startOverflow How much the first label overflows beyond the start coord.
   * @param {number} endOverflow How much the last label overflows beyonod the end coord.
   * @param {array} labels An array of dvt.Text labels for a specific level. The x coordinated of the labels will be recalculated.
   * @private
   */
  _setOverflow(startOverflow, endOverflow, labels) {
    // TODO: hierarchical labels -- when more than one level is rotated, _setOverflow is incorrect
    //       due to text.setX(coord) (should be setting the translateX instead).

    startOverflow = Math.max(startOverflow - this._startBuffer, 0);
    endOverflow = Math.max(endOverflow - this._endBuffer, 0);

    // Revert the start/endCoord to the original positions before applying the new overflow values
    var isRTL = Agent.isRightToLeft(this.getCtx());
    this.StartCoord += (startOverflow - this.StartOverflow) * (isRTL ? -1 : 1);
    this.EndCoord -= (endOverflow - this.EndOverflow) * (isRTL ? -1 : 1);

    // Reprocess since startCoord and endCoord have changed
    this._processGroupWidthRatios();

    // Recalculate coords for all levels.
    for (var j = 0; j < this._numLevels; j++) {
      labels = this._labels[j];
      // Adjust the label coords
      for (var i = 0; i < labels.length; i++) {
        var text = labels[i];
        if (text) {
          var coord = this._getLabelCoord(j, this.getLabelIdx(text));
          if (this._numLevels > 1) {
            text.setTranslateX(coord);
          } else {
            text.setX(coord);
          }
        }
      }
    }

    this.StartOverflow = startOverflow;
    this.EndOverflow = endOverflow;
  }

  /**
   * @override
   */
  getLabels(context, level) {
    if (level == null)
      // Default to returning inner most labels
      level = this._numLevels - 1;

    if (!this._labels) this._generateLabels(context);

    return this._labels[level];
  }

  /**
   * Gets the coordinate of a group label based on it's position in the hierarchy
   * @param {number} level
   * @param {number} index
   * @private
   * @return {number} The label coord
   */
  _getLabelCoord(level, index) {
    var startValue = this.getStartIdx(index, level);
    var endValue = this.getEndIdx(index, level);
    if (startValue == null || endValue == null) return null;

    if (startValue < this.MinValue && endValue > this.MinValue) startValue = this.MinValue;
    if (endValue > this.MaxValue && startValue < this.MaxValue) endValue = this.MaxValue;
    var center = endValue ? startValue + (endValue - startValue) / 2 : startValue;
    return this.getCoordAt(center);
  }

  /**
   * Generates the labels
   * @param {dvt.Context} context
   * @private
   */
  _generateLabels(context) {
    var labels = [];
    this._labels = [];
    var container = context.getStage();
    var isHoriz = this.Position == 'top' || this.Position == 'bottom';
    var isRTL = Agent.isRightToLeft(context);
    var isHierarchical = this._numLevels > 1;
    var groupWidth = this.getGroupWidth();
    var availSize = this._maxSpace;
    var gapName = isHoriz ? 'hierarchicalLabelGapHeight' : 'hierarchicalLabelGapWidth';
    var gap = isHierarchical
      ? DvtAxisDefaults.getGapSize(context, this.Options, this.Options['layout'][gapName])
      : 0;
    var rotationEnabled = this.Options['tickLabel']['rotation'] == 'auto' && isHoriz;
    var groupSpansMap = {};

    // autoRotate used to enhance performance
    var autoRotate = this.isAutoRotate();

    // Attempt text wrapping if:
    // 1. white-space != 'nowrap'
    // 2. vertical or horizontal axis
    // 3. groupWidth > textHeight -> wrapping is only necessary when more than one text line can tentatively fit in the groupWidth
    var tickLabelStyle = this.Options['tickLabel']['style'];
    var wrapping =
      DvtAxisRenderer.isWrapEnabled(this.Options['tickLabel']['style']) &&
      this.Position != 'tangential' &&
      groupWidth > TextUtils.getTextStringHeight(context, tickLabelStyle);

    // Iterate and create the labels
    var label, firstLabel, lastLabel;
    var cssStyle;
    var text;

    for (var level = 0; level < this._numLevels; level++) {
      var levels = this._levelsArray[level];
      // if autoRotate, increase performance by only generating a subset of labels to begin with, as majority will be skipped.
      var increment = autoRotate ? this.getSkipIncrement() : 1;
      groupSpansMap[level] = [];

      for (var i = 0; i < levels.length; i += increment) {
        if (levels[i]) {
          label = this.getLabelAt(i, level);
          // No text object created when group name is null or ''
          if (label === '' || (!label && label != 0)) {
            labels.push(null);
            continue;
          }

          var coord = this._getLabelCoord(level, i);
          if (coord != null) {
            // get categorical axis label style, if it exists
            cssStyle = this.getLabelStyleAt(i, level);
            var bMultiline =
              !autoRotate && wrapping && typeof label != 'number' && label.indexOf(' ') >= 0;
            text = this.CreateLabel(context, label, coord, cssStyle, bMultiline);

            // JET-53254: Make non multiline labels accessibile to assisstive technology
            if (this.isDrillable(i, level) && !bMultiline) {
              text.setAriaProperty('hidden', null);
            }

            var groupSpan =
              groupWidth * (this.getEndIdx(i, level) - this.getStartIdx(i, level) + 1);
            groupSpansMap[level].push(groupSpan);
            var bWrappedLabel =
              bMultiline &&
              this._isTextWrapNeeded(
                context,
                label,
                cssStyle,
                rotationEnabled,
                isHoriz ? groupSpan : availSize
              );
            // wrap text in the width available for each group
            if (bWrappedLabel && availSize > 0) {
              if (isHoriz) text.wrapText(groupSpan, availSize, 1, true);
              else text.wrapText(availSize, text.getLineHeight() * this._maxLineWrap, 1, false);
            } else if (bMultiline && (!isHoriz || availSize < 0))
              // Multiline texts on vertical axis will not attempt further wrapping
              text.setWrapEnabled(false);

            text._index = i; // group axis labels should reference label._index for its index
            labels.push(text);

            // Store first and last label
            if (!firstLabel && level == 0) firstLabel = text;
            if (level == 0) lastLabel = text;
          } else labels.push(null);
        } else labels.push(null);
      }

      // Adjust availSize for generating wrapped hierarchical levels
      if (wrapping && isHierarchical) {
        var totalSpace = 0;
        for (var j = 0; j < labels.length; j++) {
          if (!labels[j]) continue;

          var dims = labels[j].getDimensions();
          totalSpace = Math.max(totalSpace, isHoriz ? dims.h : dims.w);
        }
        availSize -= totalSpace + gap;
      }

      this._labels.push(labels);
      labels = [];
    }

    labels = this._labels[this._numLevels - 1];
    var labelDims = [];

    if (!firstLabel) return;

    if (this.Position == 'tangential') {
      labelDims = this.GetLabelDims(labels, container); // actual dims
      this._labels[0] = this.SkipTangentialLabels(labels, labelDims);
      return;
    }

    var firstLabelDim = firstLabel.getDimensions();
    var resetOverflow = false;

    if (isHoriz) {
      var startOverflow, endOverflow;
      if (this.Options['_startOverflow'] != null && this.Options['_endOverflow'] != null) {
        // Use the preset value if available (during z&s animation)
        startOverflow = this.Options['_startOverflow'];
        endOverflow = this.Options['_endOverflow'];
      } else {
        // wrapping combined with rotation eliminate the potential for overflow
        // Set the overflow depending on how much the first and the last label go over the bounds
        var lastLabelDim = lastLabel.getDimensions();
        startOverflow = isRTL
          ? firstLabelDim.w + firstLabelDim.x - this.StartCoord
          : this.StartCoord - firstLabelDim.x;
        endOverflow = isRTL
          ? this.EndCoord - lastLabelDim.x
          : lastLabelDim.w + lastLabelDim.x - this.EndCoord;
      }

      resetOverflow = startOverflow > this._startBuffer || endOverflow > this._endBuffer;
    }

    for (level = 0; level < this._numLevels; level++) {
      labels = this._labels[level];

      if (autoRotate)
        this._labels[level] = this._rotateLabels(labels, container, firstLabelDim.h / 2, level);
      else {
        labelDims = this.GetLabelDims(labels, container, level); // maximum estimate

        var labelsOverlapping = this.IsOverlapping(labelDims, 0, groupSpansMap[level]);
        if (!labelsOverlapping) this._labels[level] = labels; // all labels can fit

        // Rotate and skip the labels if necessary
        if (isHoriz) {
          // horizontal axis
          if (rotationEnabled) {
            if (labelsOverlapping) {
              this._labels[level] = this._rotateLabels(
                labels,
                container,
                firstLabelDim.h / 2,
                level
              );
            } else {
              resetOverflow ? this._setOverflow(startOverflow, endOverflow, labels) : null;
              this._labels[level] = labels; // all labels can fit
              if (isHierarchical) {
                // Adjust maxHeight for wrapping rotated hierarchical levels
                var totalHeight = 0;
                for (j = 0; j < labelDims.length; j++) {
                  if (labelDims[j]) {
                    totalHeight = Math.max(totalHeight, labelDims[j].h);
                  }
                }
                this._maxSpace -= totalHeight + gap;
              }
            }
          } else {
            // no rotation
            resetOverflow ? this._setOverflow(startOverflow, endOverflow, labels) : null;
            labelDims = this.GetLabelDims(labels, container); // get actual dims for skipping
            this._labels[level] = this.SkipLabels(labels, labelDims);
          }
        } else {
          // vertical axis
          // Wrapped labels
          if (wrapping) {
            var updateLabelDims = this._sanitizeWrappedText(
              context,
              labelDims,
              labels,
              false,
              isHierarchical
            );

            // Recalculate label dims for skipping
            if (updateLabelDims) labelDims = this.GetLabelDims(labels, container, level);
          }

          this._labels[level] = this.SkipLabels(labels, labelDims);
        }
      }
    }
  }

  /**
   * @override
   */
  getMajorTickCoords() {
    var coords = [],
      coord;

    // when drawing lines between labels, polar charts need gridline drawn after last label, cartesian charts do not.
    var maxIndex = this.Position == 'tangential' ? this.getGroupCount() : this.getGroupCount() - 1;

    for (var i = 0; i < this._levelsArray[0].length; i++) {
      if (this._levelsArray[0][i]) {
        var start = this.getStartIdx(i, 0);
        var end = this.getEndIdx(i, 0);
        /* If placing gridlines at labels, use the coordinates at the labels
         * Else if placing gridlines in between labels, use the value halfway between two consecutive coordinates */
        if (this._renderGridAtLabels) coord = this.getCoordAt(start + (end - start) * 0.5);
        // start == end for non-hierarchical labels
        else coord = end + 0.5 < maxIndex ? this.getCoordAt(end + 0.5) : null;

        if (coord != null) coords.push(coord);
      }
    }
    return coords;
  }

  /**
   * @override
   */
  getMinorTickCoords() {
    var coords = [],
      coord;
    if (!this._levelsArray[1])
      // minor ticks only rendered if two levels exist
      return coords;

    for (var i = 0; i < this._levelsArray[1].length; i++) {
      if (this._levelsArray[1][i]) {
        var start = this.getStartIdx(i, 1);
        var end = this.getEndIdx(i, 1);
        /* If placing gridlines at labels, use the coordinates at the labels
         * Else if placing gridlines in between labels, use the value halfway between two consecutive coordinates */
        if (this._renderGridAtLabels) coord = this.getCoordAt(start + (end - start) * 0.5);
        else coord = end + 0.5 < this.getGroupCount() - 1 ? this.getCoordAt(end + 0.5) : null;

        if (coord != null) coords.push(coord);
      }
    }
    return coords;
  }

  /**
   * @override
   */
  getUnboundedValAt(coord) {
    if (coord == null) return null;

    if (this._groupWidthRatios) {
      // Find the anchor, i.e. the group boundary closest to the coord.
      var anchor = this._borderCoords.length;
      for (var g = 0; g < this._borderCoords.length; g++) {
        if (coord <= this._borderCoords[g]) {
          anchor = g;
          break;
        }
      }
      // Compute the value based on the group width at the anchor.
      if (anchor == 0)
        return this._borderValues[0] - (this._borderCoords[0] - coord) / this._groupWidths[0];
      else
        return (
          this._borderValues[anchor - 1] +
          (coord - this._borderCoords[anchor - 1]) / this._groupWidths[anchor]
        );
    } else {
      // Even group widths
      var incr = (this.EndCoord - this.StartCoord) / (this.MaxValue - this.MinValue);
      return this.MinValue + (coord - this.StartCoord) / incr;
    }
  }

  /**
   * @override
   */
  getUnboundedCoordAt(value) {
    if (value == null) return null;

    if (this._groupWidthRatios) {
      // Find the anchor, i.e. the group boundary closest to the value.
      var anchor = this._borderValues.length;
      for (var g = 0; g < this._borderValues.length; g++) {
        if (value <= this._borderValues[g]) {
          anchor = g;
          break;
        }
      }
      // Compute the coord based on the group width at the anchor.
      if (anchor == 0)
        return this._borderCoords[0] - this._groupWidths[0] * (this._borderValues[0] - value);
      else
        return (
          this._borderCoords[anchor - 1] +
          this._groupWidths[anchor] * (value - this._borderValues[anchor - 1])
        );
    } else {
      // Even group widths
      var incr = (this.EndCoord - this.StartCoord) / (this.MaxValue - this.MinValue);
      return this.StartCoord + (value - this.MinValue) * incr;
    }
  }

  /**
   * Returns the group label for the specified group.
   * @param {number} index The index of the group label within it's level.
   * @param {number} level (optional) The level of the group label.
   * @return {string} The group label.
   */
  getLabelAt(index, level) {
    if (level == null) level = this._numLevels - 1;

    index = Math.round(index);
    if (index < 0) return null;

    var label =
      this._levelsArray[level] && this._levelsArray[level][index]
        ? this._levelsArray[level][index]['item']
        : null;

    if (label) {
      if (label['name']) label = label['name'];
      else if (label['id'] != null)
        // Empty or null group name allowed if id is specified
        label = '';
    }
    return label;
  }

  /**
   * Returns the group name or id for the specified group.
   * @param {number} index The index of the group within it's level.
   * @param {number} level (optional) The level of the group label.
   * @return {string} The group name or id.
   */
  getGroupAt(index, level) {
    if (level == null) level = this._numLevels - 1;

    index = Math.round(index);
    if (index < 0) return null;

    var label =
      this._levelsArray[level] && this._levelsArray[level][index]
        ? this._levelsArray[level][index]['item']
        : null;

    if (label) {
      if (label['id']) return label['id'];
      else if (label['name'] || label['name'] === '') return label['name'];
    }

    return label;
  }

  /**
   * Returns the style for the group label at the specified index and level.
   * @param {number} index The group index.
   * @param {number} level (optional) The level of the group label.
   * @return {dvt.CSSStyle}
   */
  getLabelStyleAt(index, level) {
    var labelStyle = this._getGroupAttribute(index, level, 'labelStyle');

    if (labelStyle) {
      var cssStyle = new CSSStyle(labelStyle);
      if (!cssStyle.getStyle('font-size'))
        // dvt.BackgroundOutputText needs font-size for adjusting select browser mis-alignment cases
        cssStyle.setStyle('font-size', this.Options['tickLabel']['style'].getStyle('font-size'));
      return cssStyle;
    }
    return null;
  }

  /**
   * @override
   */
  getDatatip(index, level) {
    // categorical label datatip is the given shortDesc if it exists
    return this._getGroupAttribute(index, level, 'shortDesc');
  }

  /**
   * @override
   */
  isDrillable(index, level) {
    var drilling = this._getGroupAttribute(index, level, 'drilling');

    if (drilling == 'on') return true;
    else if (drilling == 'off') return false;
    else return this._drilling == 'on' || this._drilling == 'groupsOnly';
  }

  /**
   * Returns a string or an array of groups names/ids of the ancestors of a group label at the given index and level.
   * @param {Number} index The index of the group label within it's level of labels
   * @param {Number=} level The level of the group labels
   * @return {String|Array} The group name/id, or an array of group names/ids.
   * @override
   */
  getGroup(index, level) {
    if (index < 0 || index > this.getGroupCount() - 1) return null;

    if (this._numLevels == 1)
      // skip the expensive computation below
      return this.getGroupAt(index);

    var groupLabels = [];
    if (level == null) level = this._numLevels - 1;
    var startIndex = this.getStartIdx(index, level);
    for (var levelIndex = 0; levelIndex <= level; levelIndex++) {
      var levelArray = this._levelsArray[levelIndex];
      for (var i = 0; i < levelArray.length; i++) {
        if (
          this.getStartIdx(i, levelIndex) <= startIndex &&
          this.getEndIdx(i, levelIndex) >= startIndex
        ) {
          groupLabels.push(this.getGroupAt(i, levelIndex));
          continue;
        }
      }
    }
    if (groupLabels.length > 0)
      return groupLabels.length === 1 && this._numLevels === 1 ? groupLabels[0] : groupLabels;
    return null;
  }

  /**
   * @override
   */
  getLabelBackground(label, context, level) {
    if (level == null) level = this._numLevels - 1;
    var style = label.getCSSStyle();
    if (style) {
      var bgColor = style.getStyle(CSSStyle.BACKGROUND_COLOR);
      var borderColor = style.getStyle(CSSStyle.BORDER_COLOR);
      var borderWidth = style.getStyle(CSSStyle.BORDER_WIDTH);
      var borderRadius = style.getStyle(CSSStyle.BORDER_RADIUS);

      // Create element for label background if group labelStyle has the background-related attributes that we support
      if (bgColor != null || borderColor != null || borderWidth != null || borderRadius != null) {
        var bboxDims = label.getDimensions();
        var padding = bboxDims.h * 0.15;

        // Chrome & IE handle 'vAlign = bottom' in a way that label and the background are misaligned, this corrects the dvt.Rect
        if (
          (Agent.engine === 'blink' ||
            Agent.browser === 'ie' ||
            Agent.browser === 'edge') &&
          label.getVertAlignment() === OutputText.V_ALIGN_BOTTOM
        )
          bboxDims.y += bboxDims.h / 2;

        var bbox = new Rect(
          context,
          bboxDims.x - padding,
          bboxDims.y,
          bboxDims.w + 2 * padding,
          bboxDims.h
        );

        var bgStyle = new CSSStyle();
        if (bgColor != null) bgStyle.setStyle(CSSStyle.BACKGROUND_COLOR, bgColor);
        else bbox.setInvisibleFill();
        bgStyle.setStyle(CSSStyle.BORDER_COLOR, borderColor);
        bgStyle.setStyle(CSSStyle.BORDER_WIDTH, borderWidth);
        bgStyle.setStyle(CSSStyle.BORDER_RADIUS, borderRadius);
        bbox.setCSSStyle(bgStyle);

        if (this._isLabelRotated[level]) bbox.setMatrix(label.getMatrix());
        bbox.setMouseEnabled(false);
        return bbox;
      }
      return null;
    } else return null;
  }

  /**
   * Returns the index for the specified group.
   * @param {string} group The group.
   * @return {number} The group index. -1 if the group doesn't exist.
   */
  getGroupIndex(group) {
    if (group == null) return -1;

    var index = -1;
    for (var i = 0; i < this._groupCount; i++) {
      var curGroup = this.getGroup(i);
      var matches =
        group instanceof Array && curGroup instanceof Array
          ? ArrayUtils.equals(group, curGroup)
          : group == curGroup;
      if (matches) {
        index = i;
        break;
      }
    }
    return index;
  }

  /**
   * @override
   */
  getMinExtent() {
    return 1;
  }

  /**
   * @override
   */
  getGroupWidth() {
    // returns the average group width
    return Math.abs(this.EndCoord - this.StartCoord) / Math.abs(this.MaxValue - this.MinValue);
  }

  /**
   * Returns the number of groups in the specified chart.
   * @return {number}
   */
  getGroupCount() {
    return this._groupCount;
  }

  /**
   * Returns the number of label levels
   * @return {number}
   */
  getNumLevels() {
    return this._numLevels;
  }

  /**
   * Conducts a DFS on a hierarchical group object to update the levelsArray
   * @param {object} groupsArray An array of chart groups
   * @param {number} level The level in the hierarchy
   * @param {object} levelsArray A structure of hierarchical group labels by level
   * @param {number} groupIndex The index of the current group
   * @return {groupIndex} A running count of the number of leaf groups
   * @private
   */
  _generateLevelsArray(groupsArray, level, levelsArray, groupIndex) {
    for (var i = 0; i < groupsArray.length; i++) {
      // Add new array if at first group in a new level
      if (!levelsArray[level]) levelsArray[level] = [];

      // Store object for group
      levelsArray[level].push({
        item: groupsArray[i],
        start: groupIndex,
        end: groupIndex,
        position: i
      });

      if (groupsArray[i] && groupsArray[i]['groups']) {
        var lastIndex = levelsArray[level].length - 1;
        // Find the index of the last innermost group nested within this group item
        var currentLeafIndex = this._generateLevelsArray(
          groupsArray[i]['groups'],
          level + 1,
          levelsArray,
          levelsArray[level][lastIndex]['start']
        );
        if (groupIndex != currentLeafIndex) {
          levelsArray[level][lastIndex]['end'] = currentLeafIndex - 1; // start and end index used for centering group labels
          groupIndex = currentLeafIndex;
        } else groupIndex++;
      } else groupIndex++;
    }
    return groupIndex;
  }

  /**
   * Returns the value for the given attribute for the group item specified by index and level
   * @param {number} index
   * @param {number} level
   * @param {string} attribute The desired atribute
   * @return {string} The value of the desires attribute
   * @private
   */
  _getGroupAttribute(index, level, attribute) {
    if (level == null) level = this._numLevels - 1;
    var groupItem =
      this._levelsArray[level] && this._levelsArray[level][index]
        ? this._levelsArray[level][index]['item']
        : null;
    return groupItem ? groupItem[attribute] : null;
  }

  /**
   * Returns whether or not to render group separators
   * @return {boolean}
   */
  areSeparatorsRendered() {
    return this._areSeparatorsRendered;
  }

  /**
   * Returns the color of the group separators
   * @return {boolean}
   */
  getSeparatorColor() {
    return this._separatorColor;
  }

  /**
   * Returns the start index for the group item specified by index and level
   * @param {number} index
   * @param {number} level
   * @return {number} The start index
   */
  getStartIdx(index, level) {
    if (level == null) level = this._numLevels - 1;
    var startIndex =
      this._levelsArray[level] && this._levelsArray[level][index]
        ? this._levelsArray[level][index]['start']
        : null;
    return startIndex;
  }

  /**
   * Returns the end index for the group item specified by index and level
   * @param {number} index
   * @param {number} level
   * @return {number} The end index
   */
  getEndIdx(index, level) {
    if (level == null) level = this._numLevels - 1;
    var endIndex =
      this._levelsArray[level] && this._levelsArray[level][index]
        ? this._levelsArray[level][index]['end']
        : null;
    return endIndex;
  }

  /**
   * Returns the position for the group item specified by index and level, in reference to it's parent
   * @param {number} index
   * @param {number} level
   * @return {number} The position of the group item in it's parent's array of children
   */
  getPosition(index, level) {
    if (level == null) level = this._numLevels - 1;
    var endIndex =
      this._levelsArray[level] && this._levelsArray[level][index]
        ? this._levelsArray[level][index]['position']
        : null;
    return endIndex;
  }

  /**
   * Returns the group index range and groupData array for the groupPath
   * @param {Array} groupPath
   */
  getItemsRange(groupPath) {
    if (!(groupPath instanceof Array)) {
      groupPath = [groupPath];
    }
    var startIndex = 0;
    var endIndex = this._groupCount;
    var groupData = [];

    for (var i = 0; i < groupPath.length; i++) {
      var searchLevel = this._levelsArray[i];
      var groupId = groupPath[i];
      for (var j = 0; j < searchLevel.length; j++) {
        var itemId = searchLevel[j].item.id ? searchLevel[j].item.id : searchLevel[j].item.name;
        if (itemId === groupId) {
          var start = searchLevel[j].start;
          var end = searchLevel[j].end;

          if (startIndex <= start && endIndex >= end) {
            startIndex = start;
            endIndex = end;
            groupData.push(searchLevel[j].item);
          }
        }
      }
    }
    return { startIndex: startIndex, endIndex: endIndex, groupData: groupData };
  }

  /**
   * Returns whether or not the axis is shifted
   * @return {boolean}
   */
  isRenderGridAtLabels() {
    return this._renderGridAtLabels;
  }

  /**
   * Store the index of the innermost level that was able to be rendered
   * @param {number} level The innermost level rendered
   */
  setLastRenderedLevel(level) {
    this._lastRenderedLevel = level;
  }

  /**
   * Returns the index of the innermost level that was able to be rendered
   * @return {number} The innermost level rendered
   */
  getLastRenderedLevel() {
    return this._lastRenderedLevel;
  }

  /**
   * Returns the index of the level there we will begin to draw hierarchical group separators
   * @return {number} The group separators start level
   */
  getSeparatorStartLevel() {
    var startLevel = this._lastRenderedLevel;

    // The start level of the separators may itself have skipped labels, but all levels after must not skip labels
    // We reset the startLevel when we find a level that skips labels
    for (var i = this._lastRenderedLevel - 1; i >= 0; i--) {
      if (this._labels[i].length != this._levelsArray[i].length) startLevel = i;
    }

    return startLevel;
  }

  /**
   * Returns the true index of the given group label
   * @param {dvt.OutputText} label The group label
   * @return {Number} The index of the group label in regards to it's position in it's level of labels
   */
  getLabelIdx(label) {
    return label._index >= 0 ? label._index : null;
  }

  /**
   * Returns the maximum lines allowed for wrapped labels.
   * @return {number}
   */
  getMaxLineWrap() {
    return this._maxLineWrap;
  }

  /**
   * Returns whether or not we should attempt to wrap a horizontal multiline text object
   * @param {dvt.Context} context
   * @param {String} label The label string of the text object
   * @param {dvt.CSSStyle} style The cssstyle of the text object
   * @param {Boolean} rotationEnabled Whether or not the text object is on an axis that enables label rotation
   * @param {Number} maxWidth The maximum width that will be given to the text object to wrap horizontally
   * @return {boolean}
   * @private
   */
  _isTextWrapNeeded(context, label, style, rotationEnabled, maxWidth) {
    var textWidth = TextUtils.getTextStringWidth(context, label, style);

    // Only attempt to wrap text horizontally if:
    // 1. The textWidth is longer that the maxWidth.
    // 2. The maximum possible width of each potential wrapped line is less than the maxWidth,
    //    or rotation is not enabled.
    // Note: This estimate may still attempt to wrap text that may not fully fit and eventually be rotated
    if (textWidth >= maxWidth && (textWidth / this._maxLineWrap < maxWidth || !rotationEnabled))
      return true;

    return false;
  }

  /**
   * Returns whether or not labels will be rotated automatically.
   * @return {boolean}
   */
  isAutoRotate() {
    var isHoriz = this.Position == 'top' || this.Position == 'bottom';
    var isHierarchical = this._numLevels > 1;
    var groupWidth = this.getGroupWidth();
    var rotationEnabled = this.Options['tickLabel']['rotation'] == 'auto' && isHoriz;
    return !isHierarchical && rotationEnabled && groupWidth < this._ROTATE_THRESHOLD;
  }

  /**
   * Returns the number by which labels can be safely skipped to improve performance.
   * @return {number}
   */
  getSkipIncrement() {
    // increase performance by only measuring a subset of labels to begin with, as majority will be skipped.
    var increment = 1;
    increment = this._ROTATE_THRESHOLD / (2 * this.getGroupWidth());
    if (this.Options['_duringZoomAndScroll']) {
      increment *= 4; // during animation, the labels can be more sparse to increase performance
    }
    return Math.max(1, Math.floor(increment));
  }
}

DvtAxisInfo.registerConstructor('group', DvtGroupAxisInfo);

/**
 *  Provides automation services for a DVT component.
 *  @class DvtAxisAutomation
 *  @param {DvtAxis} dvtComponent
 *  @implements {dvt.Automation}
 *  @constructor
 */

class DvtAxisAutomation extends Automation {
  constructor(dvtComponent) {
    super(dvtComponent);
    this._options = this._comp.getOptions();
    this._axisInfo = this._comp.getInfo();
  }

  /**
   * Valid subIds inlcude:
   * <ul>
   * <li>item[groupIndex0]...[groupIndexN]</li>
   * <li>title</li>
   * </ul>
   * @override
   */
  GetSubIdForDomElement(displayable) {
    var logicalObj = this._comp.getEventManager().GetLogicalObject(displayable);
    if (logicalObj && logicalObj instanceof SimpleObjPeer) {
      if (logicalObj.getParams().type === 'title')
        // return chart axis title subId
        return 'title';
      else if (this._options.groups) {
        // return group axis label subId
        var level = logicalObj.getParams().level || 0;
        var labelIndex = this._axisInfo.getStartIdx(logicalObj.getParams().index, level);
        var indexList = '';
        // Loop from outermost level to desired level
        for (var levelIdx = 0; levelIdx <= level; levelIdx++) {
          var labels = this._axisInfo.getLabels(this._comp.getCtx(), levelIdx);
          // Find label at each level that belongs in hierarchy for the specified label, and append position to subId index list
          for (var i = 0; i < labels.length; i++) {
            var index = this._axisInfo.getLabelIdx(labels[i]); // true group axis label index
            if (
              this._axisInfo.getStartIdx(index, levelIdx) <= labelIndex &&
              this._axisInfo.getEndIdx(index, levelIdx) >= labelIndex
            ) {
              indexList += '[' + this._axisInfo.getPosition(index, levelIdx) + ']';
            }
          }
        }
        // Return subId
        if (indexList.length > 0) return 'item' + indexList;
      }
    }
    return null;
  }

  /**
   * Valid subIds inlcude:
   * <ul>
   * <li>item[groupIndex0]...[groupIndexN]</li>
   * <li>title</li>
   * </ul>
   * @override
   */
  getDomElementForSubId(subId) {
    if (subId === 'title') {
      // process chart axis title subId
      var title = this._axisInfo.getTitle();
      if (title) return title.getElem();
    } else if (this._axisInfo instanceof DvtGroupAxisInfo) {
      // process group axis label subId
      var numIndices = subId.split('[').length - 1;
      var labelLevel = numIndices - 1;
      var labelIndex = 0;
      var startIndex = 0;
      // Loop from outermost level to specified level
      for (var levelIdx = 0; levelIdx <= labelLevel; levelIdx++) {
        var openParen = subId.indexOf('[');
        var closeParen = subId.indexOf(']');
        var groupIndex = Number(subId.substring(openParen + 1, closeParen));
        subId = subId.substring(closeParen + 1);
        var labels = this._axisInfo.getLabels(this._comp.getCtx(), levelIdx);
        var index; // true group axis label index
        for (var j = 0; j < labels.length; j++) {
          index = this._axisInfo.getLabelIdx(labels[j]);
          if (this._axisInfo.getStartIdx(index, levelIdx) === startIndex) {
            labelIndex = index;
            break;
          }
        }
        for (var i = labelIndex; i < labels.length; i++) {
          index = this._axisInfo.getLabelIdx(labels[i]);
          if (this._axisInfo.getPosition(index, levelIdx) === groupIndex) {
            if (subId.length === 0) return labels[i].getElem();
            else startIndex = this._axisInfo.getStartIdx(index, levelIdx);
            break;
          }
        }
      }
    }
    return null;
  }

  /**
   * Returns the logical object belonging to the axis label corresponding to the groupId.
   * @param {Array<string>} groupId An array of group ids associated with the axisLabel
   * @return {object} obj Logical object used corresponding to axis label belonging to groupId
   */
  findPeerFromGroupId(groupId) {
    let objs = this._comp._navigablePeers;
    for (let i = 0; i < objs.length; i++) {
      let obj = objs[i];

      if (
        (typeof obj._group === 'string' && groupId.length === 1 && groupId[0] === obj._group) ||
        (Array.isArray(obj._group) && ArrayUtils.equals(obj._group, groupId))
      ) {
        return obj;
      }
    }
    return null;
  }

  /**
   * Dispatches synthetic drill event from axis label. Used by webdriver.
   * @param {Array<string> } groupId An array of group ids associated with the axisLabel
   */

  dispatchDrillEvent(groupId) {
    let obj = this.findPeerFromGroupId(groupId);
    this._comp.getEventManager().processDrillEvent(obj);
  }
}

/**
 *  @param {dvt.EventManager} manager The owning dvt.EventManager
 *  @param {DvtAxis} axis
 *  @class DvtAxisKeyboardHandler
 *  @extends {dvt.KeyboardHandler}
 *  @constructor
 */
class DvtAxisKeyboardHandler extends KeyboardHandler {
  constructor(manager, axis) {
    super(manager);
    this._axis = axis;
  }

  /**
   * @override
   */
  processKeyDown(event) {
    var keyCode = event.keyCode;
    var currentNavigable = this._eventManager.getFocus();
    var nextNavigable = null;

    if (keyCode === KeyboardEvent.TAB) {
      if (currentNavigable) {
        EventManager.consumeEvent(event);
        nextNavigable = currentNavigable;
      }

      // navigate to the default
      var navigables = this._axis.__getKeyboardObjects();
      if (navigables.length > 0) {
        EventManager.consumeEvent(event);
        nextNavigable = this.getDefaultNavigable(navigables);
      }
    } else if (keyCode === KeyboardEvent.ENTER) {
      if (currentNavigable) {
        this._eventManager.processDrillEvent(currentNavigable);
        EventManager.consumeEvent(event);
      }
    } else nextNavigable = super.processKeyDown(event);

    return nextNavigable;
  }
}

/**
 * @class
 * @constructor
 * @extends {dvt.BaseComponent}
 */
class DvtAxis extends BaseComponent {
  constructor(context, callback, callbackObj) {
    super(context, callback, callbackObj);

    // Create the defaults object
    this.Defaults = new DvtAxisDefaults(context);

    // Create the event handler and add event listeners
    this.EventManager = new DvtAxisEventManager(this);
    this.EventManager.addListeners(this);

    // Set up keyboard handler if the axis is interactive
    this.EventManager.setKeyboardHandler(new DvtAxisKeyboardHandler(this.EventManager, this));

    this._bounds = null;
  }

  /**
   * @override
   * @protected
   */
  SetOptions(options) {
    if (options) {
      var isRendered = options['rendered'] !== 'off';

      // Combine the user options with the defaults and store. If the axis isn't rendered, no need to apply defaults.
      if (isRendered) {
        this.Options = this.Defaults.calcOptions(options);
      } else {
        // Convert user option object styles to CSSStyle
        options.tickLabel.style = new CSSStyle(options.tickLabel.style);
        options.titleStyle = new CSSStyle(options.titleStyle);
        this.Options = options;
      }
    } else if (!this.Options)
      // Create a default options object if none has been specified
      this.Options = this.GetDefaults();

    // cache isRtl value
    this.Options.isRTL = Agent.isRightToLeft(this.getCtx());
  }

  /**
   * Returns the preferred dimensions for this component given the maximum available space.
   * @param {object} options The object containing specifications and data for this component.
   * @param {Number} maxWidth The maximum width available.
   * @param {Number} maxHeight The maximum height available.
   * @param {boolean} ignoreRenderedOption rendered= "off" will be ignored if this is true
   * @return {dvt.Dimension} The preferred dimensions for the object.
   */
  getPreferredSize(options, maxWidth, maxHeight, ignoreRenderedOption) {
    // Update the options object.
    this.SetOptions(options);

    // Ask the axis to render its context in the max space and find the space used
    return DvtAxisRenderer.getPreferredSize(this, maxWidth, maxHeight, ignoreRenderedOption);
  }

  /**
   * Renders the component at the specified size.
   * @param {object} options The object containing specifications and data for this component.
   * @param {number} width The width of the component.
   * @param {number} height The height of the component.
   * @param {number=} x x position of the component.
   * @param {number=} y y position of the component.
   */
  render(options, width, height, x, y) {
    this.getCache().clearCache();

    // Update the options object.
    this.SetOptions(options);
    this._navigablePeers = [];

    this.Width = width;
    this.Height = height;

    // Clear any contents rendered previously
    this.removeChildren();

    // Set default values to undefined properties.
    if (!x) {
      x = 0;
    }

    if (!y) {
      y = 0;
    }

    // Render the axis
    var availSpace = new Rectangle(x, y, width, height);
    DvtAxisRenderer.render(this, availSpace);
  }

  /**
   * Registers the object peer with the axis.  The peer must be registered to participate
   * in interactivity.
   * @param {DvtAxisObjPeer} peer
   */
  __registerObject(peer) {
    // peer is navigable if associated with axis item using datatip or drilling is enabled
    if (peer.getDatatip() != null || peer.isDrillable()) this._navigablePeers.push(peer);
  }

  /**
   * Returns the keyboard navigables within the axis.
   * @return {array}
   */
  __getKeyboardObjects() {
    return this._navigablePeers;
  }

  /**
   * Returns whether or not the axis has navigable peers
   * @return {boolean}
   */
  isNavigable() {
    return this._navigablePeers.length > 0;
  }

  /**
   * Returns the keyboard-focused object of the axis
   * @return {DvtKeyboardNavigable} The focused object.
   */
  getKeyboardFocus() {
    if (this.EventManager != null) return this.EventManager.getFocus();
    return null;
  }

  /**
   * Sets the navigable as the keyboard-focused object of the axis. It matches the id in case it has been rerendered.
   * @param {DvtKeyboardNavigable} navigable The focused object.
   * @param {boolean} isShowingFocusEffect Whether the keyboard focus effect should be used.
   */
  setKeyboardFocus(navigable, isShowingFocusEffect) {
    if (this.EventManager == null) return;

    var peers = this.__getKeyboardObjects();
    var id = navigable.getId();
    var matchFound = false;
    for (var i = 0; i < peers.length; i++) {
      var otherId = peers[i].getId();
      if (
        (id instanceof Array && otherId instanceof Array && ArrayUtils.equals(id, otherId)) ||
        id === otherId
      ) {
        this.EventManager.setFocusObj(peers[i]);
        matchFound = true;
        if (isShowingFocusEffect) peers[i].showKeyboardFocusEffect();
        break;
      }
    }
    if (!matchFound)
      this.EventManager.setFocusObj(
        this.EventManager.getKeyboardHandler().getDefaultNavigable(peers)
      );

    // Update the accessibility attributes
    var focus = this.getKeyboardFocus();
    if (focus) {
      var displayable = focus.getDisplayable();
      displayable.setAriaProperty('label', focus.getAriaLabel());
      this.getCtx().setActiveElement(displayable);
    }
  }

  /**
   * Processes the specified event.
   * @param {object} event
   * @param {object} source The component that is the source of the event, if available.
   */
  processEvent(event, source) {
    // Dispatch the event to the callback if it originated from within this component.
    if (this === source) {
      this.dispatchEvent(event);
    }
  }

  /**
   * Returns the axisInfo for the axis
   * @return {DvtAxisInfo} the axisInfo
   */
  getInfo() {
    return this.Info;
  }

  /**
   * Sets the object containing calculated axis information and support
   * for creating drawables.
   * @param {DvtAxisInfo} axisInfo
   */
  __setInfo(axisInfo) {
    this.Info = axisInfo;
  }

  /**
   * Returns the axis width
   * @return {number}
   */
  getWidth() {
    return this.Width;
  }

  /**
   * Returns the axis height
   * @return {number}
   */
  getHeight() {
    return this.Height;
  }

  /**
   * Stores the bounds for this axis
   * @param {dvt.Rectangle} bounds
   */
  __setBounds(bounds) {
    this._bounds = bounds;
  }

  /**
   * Returns the bounds for this axis
   * @return {dvt.Rectangle} the object containing the bounds for this axis
   */
  __getBounds() {
    return this._bounds;
  }

  /**
   * Returns the automation object for this axis
   * @return {dvt.Automation} The automation object
   */
  getAutomation() {
    return new DvtAxisAutomation(this);
  }

  /**
   * Returns a copy of the default options for the specified skin.
   * @param {string} skin The skin whose defaults are being returned.
   * @return {object} The object containing defaults for this component.
   */
  static getDefaults(skin) {
    return new DvtAxisDefaults().getDefaults(skin);
  }
}

const DvtChartBehaviorUtils = {
  /**
   * Returns the hide and show behavior for the specified chart.
   * @param {Chart} chart
   * @return {string}
   */
  getHideAndShowBehavior: (chart) => {
    return chart.getOptions()['hideAndShowBehavior'];
  },

  /**
   * Returns the hover behavior for the specified chart.
   * @param {Chart} chart
   * @return {string}
   */
  getHoverBehavior: (chart) => {
    return chart.getOptions()['hoverBehavior'];
  },

  /**
   * Returns whether scroll is enabled.
   * @param {Chart} chart
   * @return {boolean}
   */
  isScrollable: (chart) => {
    if (!DvtChartTypeUtils.isScrollSupported(chart)) return false;
    return chart.getOptions()['zoomAndScroll'] != 'off';
  },

  /**
   * Returns whether zoom is enabled.
   * @param {Chart} chart
   * @return {boolean}
   */
  isZoomable: (chart) => {
    if (!DvtChartTypeUtils.isScrollSupported(chart)) return false;
    var zs = chart.getOptions()['zoomAndScroll'];
    return zs == 'live' || zs == 'delayed';
  },

  /**
   * Returns the zoom direction of the chart.
   * @param {Chart} chart
   * @return {string}
   */
  getZoomDir: (chart) => {
    if (DvtChartTypeUtils.isScatterBubble(chart)) return chart.getOptions()['zoomDirection'];
    else return 'auto';
  },

  /**
   * Returns whether zoom/scroll is live.
   * @param {Chart} chart
   * @return {boolean}
   */
  isLiveScroll: (chart) => {
    if (!DvtChartTypeUtils.isScrollSupported(chart)) return false;
    var zs = chart.getOptions()['zoomAndScroll'];
    return zs == 'live' || zs == 'liveScrollOnly';
  },

  /**
   * Returns whether zoom/scroll is delayed.
   * @param {Chart} chart
   * @return {boolean}
   */
  isDelayedScroll: (chart) => {
    if (!DvtChartTypeUtils.isScrollSupported(chart)) return false;
    var zs = chart.getOptions()['zoomAndScroll'];
    return zs == 'delayed' || zs == 'delayedScrollOnly';
  },

  /**
   * Returns whether items in plotarea are draggable.
   * @param {Chart} chart
   * @return {boolean}
   */
  isPlotAreaDraggable: (chart) => {
    var options = chart.getOptions();
    var chartDrag = options['dnd'] ? options['dnd']['drag']['items'] : {}; // for draggable effect
    return Object.keys(chartDrag).length > 0;
  },

  /**
   * Returns whether plot area is a drop target.
   * @param {Chart} chart
   * @return {boolean}
   */
  isPlotAreaDropTarget: (chart) => {
    var options = chart.getOptions();
    var chartDrop = options['dnd'] ? options['dnd']['drop']['plotArea'] : {}; // for drop effect
    return Object.keys(chartDrop).length > 0;
  },

  /**
   * Returns whether horizontal scrollbar is supported for the chart type
   * @param {Chart} chart
   * @return {boolean}
   */
  isHorizScrollbarSupported: (chart) => {
    var direction = DvtChartBehaviorUtils.getZoomDir(chart);
    if (DvtChartTypeUtils.isPolar(chart)) return false;
    return (
      (DvtChartTypeUtils.isBLAC(chart) && DvtChartTypeUtils.isVertical(chart)) ||
      (DvtChartTypeUtils.isScatterBubble(chart) && direction != 'y')
    );
  },

  /**
   * Returns whether vertical scrollbar is supported for the chart type
   * @param {Chart} chart
   * @return {boolean}
   */
  isVertScrollbarSupported: (chart) => {
    var direction = DvtChartBehaviorUtils.getZoomDir(chart);
    if (DvtChartTypeUtils.isPolar(chart)) return false;
    return (
      (DvtChartTypeUtils.isBLAC(chart) && DvtChartTypeUtils.isHorizontal(chart)) ||
      (DvtChartTypeUtils.isScatterBubble(chart) && direction != 'x')
    );
  }
};

/**
 * Text related utility functions.
 * @class
 */
const DvtChartTextUtils = {
  /**
   * Creates and adds a dvt.Text object to a container. Will truncate and add tooltip as necessary.
   * @param {dvt.EventManager} eventManager
   * @param {dvt.Container} container The container to add the text object to.
   * @param {String} textString The text string of the text object.
   * @param {dvt.CSSStyle} cssStyle The css style to apply to the text object.
   * @param {number} x The x coordinate of the text object.
   * @param {number} y The y coordinate of the text object.
   * @param {number} width The width of available text space.
   * @param {number} height The height of the available text space.
   * @return {dvt.Text} The created text object. Can be null if no text object could be created in the given space.
   */
  createText: (eventManager, container, textString, cssStyle, x, y, width, height) => {
    var text = new OutputText(container.getCtx(), textString, x, y);
    text.setCSSStyle(cssStyle);

    if (TextUtils.fitText(text, width, height, container)) {
      // Associate with logical object to support truncation
      eventManager.associate(text, new SimpleObjPeer(text.getUntruncatedTextString()));
      return text;
    }
    return null;
  },
  /**
   * Returns whether the chart has title, subtitle, or footnote.
   * @param {Chart} chart
   * @return {boolean} True if the chart has title, subtitle, or footnote.
   */
  areTitlesRendered: (chart) => {
    var options = chart.getOptions();
    return options['title']['text'] || options['subtitle']['text'] || options['footnote']['text'];
  },

  /**
   * Renders the empty text for the component.
   * @param {Chart} chart The chart being rendered.
   * @param {dvt.Container} container The container to render into.
   * @param {dvt.Rectangle} availSpace The available space.
   */
  renderEmptyText: (chart, container, availSpace) => {
    // Get the empty text string
    var options = chart.getOptions();
    if (DvtChartDataUtils.hasInvalidTimeData(chart) && DvtChartDataUtils.hasData(chart))
      var emptyTextStr = options.translations.labelInvalidData;
    else {
      emptyTextStr = options['emptyText'];
      if (!emptyTextStr) {
        emptyTextStr = options.translations.labelNoData;
      }
    }

    chart.renderEmptyText(
      container,
      emptyTextStr,
      new Rectangle(availSpace.x, availSpace.y, availSpace.w, availSpace.h),
      chart.getEventManager(),
      options['_statusMessageStyle']
    );
  }
};

/**
 * Axis related utility functions for Chart.
 * @class
 */
const DvtChartAxisUtils = {
  /**
   * Returns the position of the x axis relative to the chart.
   * @param {Chart} chart
   * @return {string} The axis position
   */
  getXAxisPos: (chart) => {
    if (DvtChartTypeUtils.isPolar(chart)) return 'tangential';
    if (DvtChartTypeUtils.isHorizontal(chart))
      return Agent.isRightToLeft(chart.getCtx()) ? 'right' : 'left';
    return 'bottom';
  },

  /**
   * Returns the baselineScaling of the specified axis.
   * @param {DvtChartImpl} chart
   * @param {string} type The axis type: x, y, or y2
   * @return {string} The axis position
   */
  getBaselineScaling: (chart, type) => {
    var axis = type + 'Axis';
    var baselineScaling = chart.getOptions()[axis]['baselineScaling'];
    if (baselineScaling && (baselineScaling == 'zero' || baselineScaling == 'min'))
      return baselineScaling;
    else if (DvtChartTypeUtils.isStock(chart)) return 'min';
    return 'zero';
  },

  /**
   * Returns the position of the y axis relative to the chart.
   * @param {Chart} chart
   * @return {string} The axis position
   */
  getYAxisPos: (chart) => {
    var position = chart.getOptions()['yAxis']['position'];

    if (DvtChartTypeUtils.isPolar(chart)) return 'radial';
    else if (DvtChartTypeUtils.isHorizontal(chart)) {
      if (position && (position == 'top' || position == 'bottom')) return position;
      return 'bottom';
    }

    if (DvtChartTypeUtils.isStock(chart)) position = position ? position : 'end';
    if (!Agent.isRightToLeft(chart.getCtx()))
      return position && position == 'end' ? 'right' : 'left';
    return position && position == 'end' ? 'left' : 'right';
  },

  /**
   * Returns the position of the y2 axis relative to the chart.
   * @param {Chart} chart
   * @return {string} The axis position
   */
  getY2AxisPos: (chart) => {
    var position = chart.getOptions()['y2Axis']['position'];

    if (DvtChartTypeUtils.isHorizontal(chart)) {
      if (position && (position == 'top' || position == 'bottom')) return position;
      return 'top';
    }

    if (!Agent.isRightToLeft(chart.getCtx()))
      return position && position == 'start' ? 'left' : 'right';
    return position && position == 'start' ? 'right' : 'left';
  },

  /**
   * Returns the offset before and after the groups for the specified chart.
   * @param {Chart} chart
   * @return {number} The offset factor.
   */
  getAxisOffset: (chart) => {
    // Use the cached value if it has been computed before
    var cacheKey = 'axisOffset';
    var axisOffset = chart.getCache().getFromCache(cacheKey);
    if (axisOffset != null) return axisOffset;

    var groupSeparators = chart.getOptions()['styleDefaults']['groupSeparators'];
    if (
      DvtChartTypeUtils.hasGroupAxis(chart) &&
      DvtChartDataUtils.getNumLevels(chart) > 1 &&
      groupSeparators['rendered'] == 'on'
    ) {
      // Use 0.5 offset for hierarchical group axis charts with groupSeparators, to ensure even spacing of the separators at start and end.
      axisOffset = 0.5;
    } else if (
      DvtChartDataUtils.hasBarSeries(chart) ||
      DvtChartStyleUtils.hasCenteredSeries(chart) ||
      DvtChartDataUtils.hasCandlestickSeries(chart) ||
      DvtChartDataUtils.hasBoxPlotSeries(chart) ||
      (DvtChartTypeUtils.isBLAC(chart) && DvtChartDataUtils.getGroupCount(chart) == 1)
    ) {
      // Use the offset for any chart with bars or centered lines/areas, or for single point line/area chart
      axisOffset = 0.5;
    } else if (
      !DvtChartTypeUtils.isSpark(chart) &&
      !DvtChartBehaviorUtils.isScrollable(chart) &&
      !DvtChartTypeUtils.isOverview(chart)
    ) {
      // Also add offset for line/area charts
      var maxOffset = DvtChartTypeUtils.isHorizontal(chart) ? 0.2 : 0.5;
      axisOffset = maxOffset - maxOffset / Math.sqrt(DvtChartDataUtils.getGroupCount(chart));
    } else {
      // Otherwise no offset
      axisOffset = 0;
    }

    chart.getCache().putToCache(cacheKey, axisOffset);
    return axisOffset;
  },

  /**
   * Returns whether the grid lines should be shifted by 1/2, so that the grid lines are drawn between labels, instead of
   * at labels. True if all the series are either bars or centeredSegmented/centeredStepped lines/areas.
   * @param {Chart} chart
   * @return {boolean}
   */
  isGridShifted: (chart) => {
    if (!DvtChartTypeUtils.isBLAC(chart)) return false;

    // Hierarchical charts will render grid lines between labels by default
    if (DvtChartDataUtils.getNumLevels(chart) > 1) return true;

    var seriesCount = DvtChartDataUtils.getSeriesCount(chart);
    for (var i = 0; i < seriesCount; i++) {
      // Ignore the series if it isn't rendered
      if (!DvtChartDataUtils.isSeriesRendered(chart, i)) continue;
      var seriesType = DvtChartDataUtils.getSeriesType(chart, i);
      var lineType = DvtChartStyleUtils.getLineType(chart, i);
      if (seriesType != 'bar' && lineType != 'centeredSegmented' && lineType != 'centeredStepped')
        return false;
    }

    return true;
  },

  /**
   * Returns whether the polar chart gridlines are polygonal.
   * @param {DvtChartImp} chart
   * @return {boolean}
   */
  isGridPolygonal: (chart) => {
    if (!DvtChartTypeUtils.isBLAC(chart) || DvtChartDataUtils.hasBarSeries(chart)) return false;
    return chart.getOptions()['polarGridShape'] == 'polygon';
  },

  /**
   * Returns whether an axis is rendered (only tick labels and axis title are considered parts of the axis).
   * @param {Chart} chart
   * @param {string} type The axis type: x, y, or y2.
   * @return {boolean} True if the axis is rendered.
   */
  isAxisRendered: (chart, type) => {
    // Check if the current chart type supports having axes
    if (!DvtChartTypeUtils.hasAxes(chart)) {
      return false;
    }

    var options = chart.getOptions();
    var ovContent = options.overview.content && options.overview.content[type + 'Axis'];
    var ignoreRenderedOption = ovContent && ovContent.rendered == 'on';
    // For y/y2, evaluate if there's any series assigned to them
    if (type == 'y' && DvtChartDataUtils.hasY2DataOnly(chart) && !ignoreRenderedOption)
      return false;
    if (type == 'y2' && !DvtChartDataUtils.hasY2Data(chart) && !ignoreRenderedOption) return false;

    // Check the chart options
    var axisOptions = options[type + 'Axis'];
    if (axisOptions['rendered'] == 'off' && !ignoreRenderedOption) return false;
    if (
      axisOptions['tickLabel']['rendered'] == 'off' &&
      !axisOptions['title'] &&
      !ignoreRenderedOption
    )
      return false;

    return true;
  },

  /**
   * Returns true if the axis line for the specified axis is to be rendered.
   * @param {Chart} chart
   * @param {string} type The axis type: x, y, or y2.
   * @return {boolean}
   */
  isAxisLineRendered: (chart, type) => {
    var axisOptions = chart.getOptions()[type + 'Axis'];
    if (axisOptions['rendered'] == 'off' || axisOptions['axisLine']['rendered'] == 'off')
      return false;
    else if (
      axisOptions['axisLine']['rendered'] == 'auto' &&
      type != 'x' &&
      DvtChartTypeUtils.isBLAC(chart) &&
      !DvtChartTypeUtils.isPolar(chart)
    )
      return false; // yAxis lines not rendered for blac cartesian with axisLine.rendered="auto"
    return true;
  },

  /**
   * Returns true if the major tick for the specified axis is to be rendered.
   * @param {Chart} chart
   * @param {string} type The axis type: x, y, or y2.
   * @return {boolean}
   */
  isMajorTickRendered: (chart, type) => {
    var axisOptions = chart.getOptions()[type + 'Axis'];
    if (axisOptions['rendered'] == 'off' || axisOptions['majorTick']['rendered'] == 'off')
      return false;
    else if (
      axisOptions['majorTick']['rendered'] == 'auto' &&
      type == 'x' &&
      DvtChartTypeUtils.isBLAC(chart) &&
      !DvtChartTypeUtils.isPolar(chart)
    )
      return false; // xAxis ticks not rendered for blac cartesian with axisLine.rendered="auto"
    return true;
  },

  /**
   * Returns true if the minor tick for the specified axis is to be rendered.
   * @param {Chart} chart
   * @param {string} type The axis type: x, y, or y2.
   * @return {boolean}
   */
  isMinorTickRendered: (chart, type) => {
    var axisOptions = chart.getOptions()[type + 'Axis'];
    if (axisOptions['rendered'] == 'off' || axisOptions['minorTick']['rendered'] == 'off')
      return false;
    else if (axisOptions['minorTick']['rendered'] == 'on') return true;
    return DvtChartAxisUtils.isLog(chart, type);
  },

  /**
   * Returns true if the axis scale is logarithmic.
   * @param {Chart} chart
   * @param {string} type The axis type: x, y, or y2.
   * @return {boolean}
   */
  isLog: (chart, type) => {
    var axisOptions = chart.getOptions()[type + 'Axis'];
    return axisOptions['scale'] == 'log';
  },

  /**
   * Returns the height of the axis tick label
   * @param {Chart} chart
   * @param {string} type The axis type: x, y, or y2
   * @return {number} Height in px
   */
  getTickLabelHeight: (chart, type) => {
    var options = chart.getOptions();
    var axisOptions = options[type + 'Axis'];

    // Manually construct the tick label style
    var tickLabelStyle = axisOptions['tickLabel']['style'];
    if (!(tickLabelStyle instanceof CSSStyle))
      tickLabelStyle = new CSSStyle(tickLabelStyle);
    tickLabelStyle.mergeUnder(DvtAxis.getDefaults(options['skin'])['tickLabel']['style']); // merge with the default

    return TextUtils.getTextStringHeight(chart.getCtx(), tickLabelStyle);
  },

  /**
   * Returns the tick label gap size for the axis.
   * @param {Chart} chart
   * @param {string} type The type of the axis: x, y, or y2.
   * @return {number} Gap size.
   */
  getTickLabelGapSize: (chart, type) => {
    if (DvtChartAxisUtils.isTickLabelInside(chart, type)) return 0;

    var options = chart.getOptions();
    var isHoriz = DvtChartTypeUtils.isHorizontal(chart);

    var scalingFactor = DvtChartAxisUtils.getGapScalingFactor(chart, type);
    var gapWidth = Math.ceil(options['layout']['tickLabelGapWidth'] * scalingFactor);
    var gapHeight = Math.ceil(options['layout']['tickLabelGapHeight'] * scalingFactor);

    if (type == 'x') return isHoriz ? gapWidth : gapHeight;
    return isHoriz ? gapHeight : gapWidth;
  },

  /**
   * Returns the scaling factor for a gap based on the axis tick label font size.
   * @param {Chart} chart
   * @param {string} type The type of the axis: x, y, or y2.
   * @return {number} Scaling factor.
   */
  getGapScalingFactor: (chart, type) => {
    if (DvtChartAxisUtils.isAxisRendered(chart, type))
      return DvtChartAxisUtils.getTickLabelHeight(chart, type) / 14; // 14px is the default label height, assuming 11px font size
    return 0;
  },

  /**
   * Returns the position of the axis tick label is inside the plot area.
   * @param {Chart} chart
   * @param {string} type The type of the axis: x, y, or y2.
   * @return {boolean}
   */
  isTickLabelInside: (chart, type) => {
    if (
      DvtChartTypeUtils.isPolar(chart) ||
      DvtChartTypeUtils.isScatterBubble(chart) ||
      (DvtChartTypeUtils.isBLAC(chart) && type == 'x')
    )
      return false;

    return chart.getOptions()[type + 'Axis']['tickLabel']['position'] == 'inside';
  },

  /**
   * Returns the viewport min/max of the x-axis. If viewportMin/Max is not defined, it assumes viewportStart/EndGroup to
   * be the viewport min/max.
   * @param {Chart} chart
   * @param {boolean} useGlobal Whether the method returns the global min/max if the viewport min/max is defined.
   * @return {object} An object containing min and max.
   */
  getXAxisViewportMinMax: (chart, useGlobal) => {
    var cacheKey = useGlobal ? 'xAxisViewportMinMaxUG' : 'xAxisViewportMinMax';
    var minMax = chart.getCache().getFromCache(cacheKey);
    if (minMax) return minMax;

    var options = chart.getOptions()['xAxis'];
    var isGroupAxis = DvtChartTypeUtils.hasGroupAxis(chart);
    var groupOffset = DvtChartAxisUtils.getAxisOffset(chart);

    if (useGlobal) var globalMinMax = DvtChartAxisUtils.getXAxisGlobalMinMax(chart);

    var min = null;
    if (options['viewportMin'] != null) min = options['viewportMin'];
    else if (options['viewportStartGroup'] != null)
      min = isGroupAxis
        ? DvtChartDataUtils.getGroupIdx(chart, options['viewportStartGroup']) - groupOffset
        : options['viewportStartGroup'];
    else if (useGlobal) {
      min = globalMinMax['min'];
    }

    var max = null;
    if (options['viewportMax'] != null) max = options['viewportMax'];
    else if (options['viewportEndGroup'] != null)
      max = isGroupAxis
        ? DvtChartDataUtils.getGroupIdx(chart, options['viewportEndGroup']) + groupOffset
        : options['viewportEndGroup'];
    else if (useGlobal) {
      max = globalMinMax['max'];
    }

    // Cache the value
    minMax = { min: min, max: max };
    chart.getCache().putToCache(cacheKey, minMax);

    return minMax;
  },

  /**
   * Returns the global min/max of the x-axis.
   * @param {Chart} chart
   * @return {object} An object containing min and max.
   */
  getXAxisGlobalMinMax: (chart) => {
    var options = chart.getOptions()['xAxis'];
    var isGroupAxis = DvtChartTypeUtils.hasGroupAxis(chart);
    var groupOffset = DvtChartAxisUtils.getAxisOffset(chart);

    if (!isGroupAxis) var minMax = DvtChartAxisUtils.getMinMaxVal(chart, 'x');

    var min = null;
    if (options['min'] != null) min = options['min'];
    else if (isGroupAxis) min = 0 - groupOffset;
    else min = minMax['min'];

    var max = null;
    if (options['max'] != null) max = options['max'];
    else if (isGroupAxis) max = DvtChartDataUtils.getGroupCount(chart) - 1 + groupOffset;
    else max = minMax['max'];

    return { min: min, max: max };
  },

  /**
   * Applies the chart initial zooming by updating the viewportMin/Max in the options object.
   * @param {Chart} chart
   * @param {dvt.Rectangle} availSpace The available axis space, to determine the amount of initial zooming.
   */
  applyInitialZooming: (chart, availSpace) => {
    var options = chart.getOptions();
    var axisOptions = options['xAxis'];
    var initialZooming = options['initialZooming'];
    if (
      !DvtChartTypeUtils.isBLAC(chart) ||
      options['zoomAndScroll'] == 'off' ||
      initialZooming == 'none'
    )
      return;

    // If the chart has been initially zoomed before, but is rerendered with the same options (possibly resized), the
    // initial zooming level has the be recomputed.
    if (options['_initialZoomed']) {
      if (initialZooming == 'last') axisOptions['viewportMin'] = null;
      // initialZooming = first
      else axisOptions['viewportMax'] = null;
    }

    var viewportMinMax = DvtChartAxisUtils.getXAxisViewportMinMax(chart, false);
    var viewportMin = viewportMinMax['min'];
    var viewportMax = viewportMinMax['max'];
    if (
      (initialZooming == 'last' && viewportMin != null) ||
      (initialZooming == 'first' && viewportMax != null)
    )
      return;

    var axisWidth = DvtChartTypeUtils.isHorizontal(chart) ? availSpace.h : availSpace.w; // estimated
    var maxNumGroups =
      Math.floor(axisWidth / (2 * DvtChartAxisUtils.getTickLabelHeight(chart, 'x'))) +
      DvtChartAxisUtils.getAxisOffset(chart);
    var numGroups = DvtChartDataUtils.getGroupCount(chart) - 1; // -1 because we count the number of group gaps
    if (numGroups <= maxNumGroups) return;

    var globalMin, globalMax;
    if (DvtChartTypeUtils.hasGroupAxis(chart)) {
      globalMin = 0;
      globalMax = numGroups; // numGroups is already subtracted by 1!
    } else {
      var globalMinMax = DvtChartAxisUtils.getMinMaxVal(chart, 'x');
      globalMin = globalMinMax['min'];
      globalMax = globalMinMax['max'];
    }
    var maxViewportSize = (maxNumGroups / numGroups) * (globalMax - globalMin);

    if (options['initialZooming'] == 'last') {
      if (viewportMax == null) viewportMax = globalMax;
      axisOptions['viewportMin'] = Math.max(viewportMax - maxViewportSize, globalMin);
    } else {
      // initialZooming = first
      if (viewportMin == null) viewportMin = globalMin;
      axisOptions['viewportMax'] = Math.min(viewportMin + maxViewportSize, globalMax);
    }

    // Add flag to indicate that the viewportMin/Max is the result of initial zooming.
    options['_initialZoomed'] = true;
  },

  /**
   * Returns true if the y axis needs to be adjust to account for data labels.
   * @param {Chart} chart
   * @return {boolean}
   */
  isYAdjustmentNeeded: (chart) => {
    var dataLabelPosition = chart.getOptions()['styleDefaults']['dataLabelPosition'];
    var hasDataOutsideBarEdge = chart.getOptionsCache().getFromCache('hasDataOutsideBarEdge');
    var hasOutsideBarEdge = dataLabelPosition == 'outsideBarEdge' || hasDataOutsideBarEdge;
    var isStackLabelRendered = DvtChartStyleUtils.isStackLabelRendered(chart);
    if (DvtChartDataUtils.hasBarSeries(chart) && (hasOutsideBarEdge || isStackLabelRendered)) {
      return true;
    }
    return false;
  },

  /**
   * Whether the axis contains the specified point.
   * @param {DvtAxis} axis
   * @param {dvt.Point} relPos Point coords relative to the stage.
   * @return {boolean}
   */
  axisContainsPoint: (axis, relPos) => {
    if (!axis) return false;

    // Increase the hit area
    var position = axis.getOptions()['position'];
    var isHoriz = position == 'top' || position == 'bottom';
    var yGap = isHoriz ? 4 : 10;
    var xGap = isHoriz ? 10 : 4;

    var bounds = axis.__getBounds().clone();
    bounds.x -= xGap;
    bounds.y -= yGap;
    bounds.w += 2 * xGap;
    bounds.h += 2 * yGap;

    var axisPos = axis.stageToLocal(relPos);
    return bounds.containsPoint(axisPos.x, axisPos.y);
  },

  /**
   * @param {Chart} chart
   * @return {boolean} true if the chart is a standalone plot area.
   */
  isStandalonePlotArea: (chart) => {
    if (DvtChartTextUtils.areTitlesRendered(chart)) return false;
    if (DvtChartTypeUtils.isLegendRendered(chart)) return false;
    if (DvtChartAxisUtils.isAxisRendered(chart, 'x')) return false;
    if (DvtChartAxisUtils.isAxisRendered(chart, 'y')) return false;
    if (DvtChartAxisUtils.isAxisRendered(chart, 'y2')) return false;
    return true;
  },

  /**
   * @param {Chart} chart
   * @return {boolean} true if the chart is a standalone x-axis.
   */
  isStandaloneXAxis: (chart) => {
    var options = chart.getOptions();
    if (DvtChartTextUtils.areTitlesRendered(chart)) return false;
    if (options['legend']['rendered'] != 'off') return false;
    if (options['plotArea']['rendered'] != 'off') return false;
    if (DvtChartAxisUtils.isAxisRendered(chart, 'y')) return false;
    if (DvtChartAxisUtils.isAxisRendered(chart, 'y2')) return false;
    return true;
  },

  /**
   * @param {Chart} chart
   * @return {boolean} true if the chart is a standalone y-axis.
   */
  isStandaloneYAxis: (chart) => {
    var options = chart.getOptions();
    if (DvtChartTextUtils.areTitlesRendered(chart)) return false;
    if (options['legend']['rendered'] != 'off') return false;
    if (DvtChartAxisUtils.isAxisRendered(chart, 'x')) return false;
    if (options['plotArea']['rendered'] != 'off') return false;
    if (DvtChartAxisUtils.isAxisRendered(chart, 'y2')) return false;
    return true;
  },

  /**
   * @param {Chart} chart
   * @return {boolean} true if the chart is a standalone y2-axis.
   */
  isStandaloneY2Axis: (chart) => {
    var options = chart.getOptions();
    if (DvtChartTextUtils.areTitlesRendered(chart)) return false;
    if (options['legend']['rendered'] != 'off') return false;
    if (DvtChartAxisUtils.isAxisRendered(chart, 'x')) return false;
    if (DvtChartAxisUtils.isAxisRendered(chart, 'y')) return false;
    if (options['plotArea']['rendered'] != 'off') return false;
    return true;
  },

  /**
   * Returns the min and max groupIndex that are entirely within the chart viewport for the specified seriesIndex.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @return {object} An object containing min and max properties.
   */
  getViewportMinMaxGroupIdx: (chart, seriesIndex) => {
    var minMaxValue = DvtChartAxisUtils.getXAxisViewportMinMax(chart, true);
    var hasGroupAxis = DvtChartTypeUtils.hasGroupAxis(chart);
    var groupCount = DvtChartDataUtils.getGroupCount(chart);

    var minValue = minMaxValue['min'];
    var minIndex = 0;
    if (minValue != null) {
      if (hasGroupAxis) minIndex = Math.ceil(minValue);
      else {
        // TODO: faster with binary search
        for (var g = 0; g < groupCount; g++) {
          var xValue = DvtChartDataUtils.getXVal(chart, seriesIndex, g);
          if (xValue >= minValue) {
            minIndex = g;
            break;
          }
        }
      }
    }

    var maxValue = minMaxValue['max'];
    var maxIndex = groupCount - 1;
    if (hasGroupAxis) maxIndex = Math.floor(maxValue);
    else {
      // TODO: faster with binary search
      for (var groupIdx = groupCount - 1; groupIdx >= minIndex; groupIdx--) {
        var xVal = DvtChartDataUtils.getXVal(chart, seriesIndex, groupIdx);
        if (xVal <= maxValue) {
          maxIndex = groupIdx;
          break;
        }
      }
    }

    return { min: minIndex, max: maxIndex };
  },

  /**
   * Returns the number of groups within the viewport.
   * @param {Chart} chart
   * @return {number}
   */
  getViewportGroupCount: (chart) => {
    var viewportMinMax = DvtChartAxisUtils.getXAxisViewportMinMax(chart, true);
    var globalMinMax = DvtChartAxisUtils.getXAxisGlobalMinMax(chart);
    var ratio =
      (viewportMinMax['max'] - viewportMinMax['min']) / (globalMinMax['max'] - globalMinMax['min']);
    return isNaN(ratio) ? 1 : ratio * DvtChartDataUtils.getGroupCount(chart);
  },

  /**
   * Compute the Y corresponding to the X along the line from (x1,y1) to (x2,y2).
   * @param {boolean} isLog
   * @param {number} x1
   * @param {number} y1
   * @param {number} x2
   * @param {number} y2
   * @param {number} x
   * @return {number} The y value.
   * @private
   */
  _computeYAlongLine: (isLog, x1, y1, x2, y2, x) => {
    if (isLog) {
      y1 = Math$1.log10(y1);
      y2 = Math$1.log10(y2);
    }
    var y = y1 + ((y2 - y1) * (x - x1)) / (x2 - x1);
    return isLog ? Math.pow(10, y) : y;
  },

  /**
   * Returns the min and max values from the data.
   * @param {Chart} chart
   * @param {string} type The type of value to find: "x", "y", "y2", "z".
   * @param {boolean=} isDataOnly Use data points only in min/max computation. Excludes bubble radii and viewport edge values. Defaults to false.
   * @return {object} An object containing the minValue and the maxValue.
   */
  getMinMaxVal: (chart, type, isDataOnly) => {
    // Use the cached value if it has been computed before
    var cacheKey = type + (isDataOnly ? 'MinMaxDO' : 'MinMax');
    var minMax = chart.getCache().getFromCache(cacheKey);
    if (minMax) return minMax;

    // Use user specified values if set to improve performance
    var axisOptions = chart.getOptions()[type + 'Axis'];
    if (axisOptions['dataMax'] != null && axisOptions['dataMin'] != null && isDataOnly) {
      // Cache the value
      minMax = { min: axisOptions['dataMin'], max: axisOptions['dataMax'] };
      chart.getCache().putToCache(cacheKey, minMax);
      return minMax;
    }

    var hasTimeAxis = DvtChartTypeUtils.hasTimeAxis(chart);
    if (axisOptions['max'] != null && axisOptions['min'] != null && !isDataOnly && !hasTimeAxis) {
      // Cache the value
      minMax = { min: axisOptions['min'], max: axisOptions['max'] };
      chart.getCache().putToCache(cacheKey, minMax);
      return minMax;
    }

    // TODO support for null or NaN values

    var isLog = type != 'z' && DvtChartAxisUtils.isLog(chart, type);

    // Y2 values pull from the y data value
    var isY2Value = type == 'y2';
    if (isY2Value) type = 'y';

    // Y values may be listed directly as numbers
    var isYValue = type == 'y';

    // limitToViewport is computing the min/max based on only the values that are within the viewport.
    // Only implemented for BLAC chart "y" and "y2" axis.
    var limitToViewport = !isDataOnly && isYValue && DvtChartTypeUtils.isBLAC(chart);

    // Include hidden series if hideAndShowBehavior occurs without rescale or for time axis
    var bIncludeHiddenSeries =
      DvtChartBehaviorUtils.getHideAndShowBehavior(chart) == 'withoutRescale' ||
      (type == 'x' && DvtChartTypeUtils.hasTimeAxis(chart));

    var maxValue = -Infinity;
    var minValue = Infinity;
    var seriesCount = DvtChartDataUtils.getSeriesCount(chart);

    for (var seriesIndex = 0; seriesIndex < seriesCount; seriesIndex++) {
      var seriesItem = DvtChartDataUtils.getSeriesItem(chart, seriesIndex);
      var seriesType = DvtChartDataUtils.getSeriesType(chart, seriesIndex);
      var isRange =
        isYValue &&
        (DvtChartDataUtils.isRangeSeries(chart, seriesIndex) ||
          seriesType == 'candlestick' ||
          seriesType == 'boxPlot');

      // Skip the series if it is hidden
      if (!bIncludeHiddenSeries && !DvtChartDataUtils.isSeriesRendered(chart, seriesIndex))
        continue;

      // Skip the series if it's not assigned to the right y axis
      var isY2Series = DvtChartDataUtils.isAssignedToY2(chart, seriesIndex);
      if (isYValue && isY2Value != isY2Series) continue;

      // Loop through the data
      var seriesData = seriesItem['items'];
      if (!seriesData) continue;

      var minGroupIndex = 0;
      var maxGroupIndex = seriesData.length - 1;
      if (limitToViewport) {
        var minMaxGroupIndex = DvtChartAxisUtils.getViewportMinMaxGroupIdx(chart, seriesIndex);
        minGroupIndex = minMaxGroupIndex['min'];
        maxGroupIndex = minMaxGroupIndex['max'];
      }

      for (var groupIndex = minGroupIndex; groupIndex <= maxGroupIndex; groupIndex++) {
        // Skip the data item if it is hidden
        if (
          !bIncludeHiddenSeries &&
          !DvtChartDataUtils.isDataItemRendered(chart, seriesIndex, groupIndex)
        )
          continue;

        var data = seriesData[groupIndex];

        var value = null;
        if (isYValue) {
          if (!isRange)
            value = DvtChartDataUtils.getCumulativeVal(
              chart,
              seriesIndex,
              groupIndex,
              bIncludeHiddenSeries
            );
        } else if (type == 'x' && hasTimeAxis && !DvtChartTypeUtils.isMixedFrequency(chart)) {
          // Take time value from the groups array and transfer it to the data item
          value = DvtChartDataUtils.getGroupLabel(chart, groupIndex);
          if (data != null) data['x'] = value;
        } else if (data != null) value = data[type];

        if (type == 'z' && value <= 0)
          // exclude 0 for bubble radius min/max
          continue;

        if (!isRange && value != null && typeof value == 'number' && !(isLog && value <= 0)) {
          var radius = 0;
          if (DvtChartTypeUtils.isBubble(chart) && !isDataOnly && type != 'z') {
            var markerSize = DvtChartStyleUtils.getMarkerSize(chart, seriesIndex, groupIndex);
            radius = DvtChartAxisUtils.getBubbleAxisRadius(chart, type, markerSize);
          }

          maxValue = Math.max(maxValue, isLog ? value * Math.pow(10, radius) : value + radius);
          minValue = Math.min(minValue, isLog ? value / Math.pow(10, radius) : value - radius);
        }

        if (isRange) {
          var high = DvtChartDataUtils.getHighVal(chart, seriesIndex, groupIndex);
          var low = DvtChartDataUtils.getLowVal(chart, seriesIndex, groupIndex);

          if (!(isLog && (high <= 0 || low <= 0))) {
            maxValue = Math.max(maxValue, high, low);
            minValue = Math.min(minValue, high, low);
          }
        }

        // Include the Y values at the X-axis min/max edges of the viewport to make the Y-axis rescale smoothly
        // TODO: Implement for range series
        if (limitToViewport && !isRange) {
          // Computation only applies to first and last group in the viewport. Get the adjacent group just outside
          // the viewport to get the viewport edge value.
          var adjacentIndex = null;
          var edgeX = null;
          if (minGroupIndex > 0 && groupIndex == minGroupIndex) {
            adjacentIndex = groupIndex - 1;
            edgeX = DvtChartAxisUtils.getXAxisViewportMinMax(chart, true)['min'];
          } else if (maxGroupIndex < seriesData.length - 1 && groupIndex == maxGroupIndex) {
            adjacentIndex = groupIndex + 1;
            edgeX = DvtChartAxisUtils.getXAxisViewportMinMax(chart, true)['max'];
          }

          if (adjacentIndex != null) {
            var x = DvtChartDataUtils.getXVal(chart, seriesIndex, groupIndex);
            var adjacentX = DvtChartDataUtils.getXVal(chart, seriesIndex, adjacentIndex);
            var adjacentValue = DvtChartDataUtils.getCumulativeVal(
              chart,
              seriesIndex,
              adjacentIndex
            );
            var edgeValue = DvtChartAxisUtils._computeYAlongLine(
              isLog,
              x,
              value || 0,
              adjacentX,
              adjacentValue || 0,
              edgeX
            );
            maxValue = Math.max(maxValue, edgeValue);
            minValue = Math.min(minValue, edgeValue);
          }
        }

        // Only include nested item when type is not x.
        if (type != 'x') {
          var nestedDataCount = DvtChartDataUtils.getNestedDataItemCount(
            chart,
            seriesIndex,
            groupIndex
          );
          if (nestedDataCount > 0) {
            for (var itemIndex = 0; itemIndex < nestedDataCount; itemIndex++) {
              var item = DvtChartDataUtils.getNestedDataItem(
                chart,
                seriesIndex,
                groupIndex,
                itemIndex
              );
              var isItemNumeric = typeof item === 'number';
              maxValue = isItemNumeric ? Math.max(maxValue, item) : Math.max(maxValue, item.value);
              minValue = isItemNumeric ? Math.min(minValue, item) : Math.min(minValue, item.value);
            }
          }
        }
      }
    }

    //Loop through reference objects and include their min/max values in the calculation
    var refObjects = null;
    if (type == 'x') refObjects = DvtChartDataUtils.getAxisRefObjs(chart, 'x');
    else if (isY2Value)
      // check isY2Value first, because isYValue will also be true
      refObjects = DvtChartDataUtils.getAxisRefObjs(chart, 'y2');
    else if (isYValue) refObjects = DvtChartDataUtils.getAxisRefObjs(chart, 'y');

    if (refObjects != null) {
      for (var i = 0; i < refObjects.length; i++) {
        var refObj = refObjects[i];
        var items = refObj['items']; //reference objects with varied min/max or values have the 'items' object populated
        var hidden =
          DvtChartBehaviorUtils.getHideAndShowBehavior(chart) == 'withRescale' &&
          !DvtChartRefObjUtils.isObjRendered(chart, refObj);
        // If refObj has varied values , loop through and evaluate all values in 'items'
        // Else just evaluate min/max/value on refObj
        if (hidden) continue;

        if (items && !hidden) {
          var minItemIndex = 0;
          var maxItemIndex = items.length - 1;
          if (limitToViewport) {
            var minMaxItemIndex = DvtChartAxisUtils.getViewportMinMaxIdx(chart, items);
            minItemIndex = minMaxItemIndex['min'];
            maxItemIndex = minMaxItemIndex['max'];
          }

          for (var j = minItemIndex; j <= maxItemIndex; j++) {
            if (items[j] == null) continue;

            var refObjLow = DvtChartRefObjUtils.getLowVal(items[j]);
            var refObjHigh = DvtChartRefObjUtils.getHighVal(items[j]);
            var val = isNaN(items[j]) ? items[j]['value'] : items[j];

            if (refObjLow != null && isFinite(refObjLow)) {
              minValue = Math.min(minValue, refObjLow);
              maxValue = Math.max(maxValue, refObjLow);
            }
            if (refObjHigh != null && isFinite(refObjHigh)) {
              minValue = Math.min(minValue, refObjHigh);
              maxValue = Math.max(maxValue, refObjHigh);
            }
            if (val != null && isFinite(val)) {
              minValue = Math.min(minValue, val);
              maxValue = Math.max(maxValue, val);
            }
          }
        } else {
          var refLow = DvtChartRefObjUtils.getLowVal(refObj);
          var refHigh = DvtChartRefObjUtils.getHighVal(refObj);
          var refVal = refObj['value'];
          if (refLow != null && isFinite(refLow)) {
            minValue = Math.min(minValue, refLow);
            maxValue = Math.max(maxValue, refLow);
          }
          if (refHigh != null && isFinite(refHigh)) {
            minValue = Math.min(minValue, refHigh);
            maxValue = Math.max(maxValue, refHigh);
          }
          if (refVal != null && isFinite(refVal)) {
            minValue = Math.min(minValue, refVal);
            maxValue = Math.max(maxValue, refVal);
          }
        }
      }
    }

    // Cache the value
    minMax = { min: minValue, max: maxValue };
    chart.getCache().putToCache(cacheKey, minMax);

    return minMax;
  },

  /**
   * Estimates how much axis range (in values, not pixels) the bubble radius would span.
   * @param {Chart} chart
   * @param {string} axisType 'x' or 'y'
   * @param {number} markerSize The size of the bubble marker
   * @return {number}
   */
  getBubbleAxisRadius: (chart, axisType, markerSize) => {
    if (!markerSize) return 0;
    var cacheKey = axisType == 'x' ? '_xAxisBubbleRatio' : '_yAxisBubbleRatio';
    var axisBubbleRatio = chart.getCache().getFromCache(cacheKey);
    return (markerSize / 2) * axisBubbleRatio;
  },

  /**
   *
   * Returns the min and max ref obj item index that are entirely within the chart viewport.
   * @param {Chart} chart
   * @param {object} items The array of reference object items
   * @return {object} An object containing min and max properties.
   */
  getViewportMinMaxIdx: (chart, items) => {
    var minMaxValue = DvtChartAxisUtils.getXAxisViewportMinMax(chart, true);
    var hasGroupAxis = DvtChartTypeUtils.hasGroupAxis(chart);

    var minValue = minMaxValue['min'];
    var minIndex = 0;
    if (minValue != null) {
      if (hasGroupAxis) minIndex = Math.ceil(minValue);
      else {
        // TODO: faster with binary search
        for (var i = 0; i < items.length; i++) {
          var xValue = DvtChartRefObjUtils.getXVal(chart, items, i);
          if (xValue >= minValue) {
            minIndex = i;
            break;
          }
        }
      }
    }

    var maxValue = minMaxValue['max'];
    var maxIndex = items.length - 1;
    if (hasGroupAxis) maxIndex = Math.floor(maxValue);
    else {
      // TODO: faster with binary search
      for (var z = items.length - 1; z >= minIndex; z--) {
        var xVal = DvtChartDataUtils.getXVal(chart, items, z);
        if (xVal <= maxValue) {
          maxIndex = z;
          break;
        }
      }
    }

    return { min: minIndex, max: maxIndex };
  }
};

const DvtChartGroupUtils = {
  /**
   * Computes the ratios of the axis group widths (for bars with varying widths).
   * @param {Chart} chart
   * @return {Array} The array of the axis group width ratios.
   */
  getGroupWidthRatios: (chart) => {
    if (
      !DvtChartDataUtils.hasBarSeries(chart) &&
      !DvtChartDataUtils.hasCandlestickSeries(chart) &&
      !DvtChartDataUtils.hasBoxPlotSeries(chart)
    )
      return null;

    var options = chart.getOptions();
    var barGapRatio = DvtChartGroupUtils.getBarGapRatio(chart);

    if (barGapRatio >= 1) {
      options['_averageGroupZ'] = Infinity; // so that all bars have zero width
      return null;
    }

    options['_averageGroupZ'] = 0; // reset the value

    // Compute the total z-values of the bars occupying each group
    var numGroups = DvtChartDataUtils.getGroupCount(chart);
    var isSplitDualY = DvtChartDataUtils.isSplitDualY(chart);
    var categories = DvtChartDataUtils.getStackCategories(chart, 'bar');
    var numYCategories = categories['y'].length;
    var numY2Categories = categories['y2'].length;
    var groupWidths, yWidth, y2Width, i;

    var barWidthSum, gapWidthSum;
    var hasConstantZValue = chart.getOptionsCache().getFromCache('hasConstantZValue');

    if (hasConstantZValue) {
      var constantZValue = chart.getOptionsCache().getFromCache('constantZValue');
      var barWidth =
        constantZValue *
        (isSplitDualY
          ? Math.max(numYCategories, numY2Categories)
          : numYCategories + numY2Categories);
      barWidthSum = barWidth * numGroups;

      // The gap size is the same for all groups, regardless of the bar width.
      gapWidthSum = (barWidthSum * barGapRatio) / (1 - barGapRatio);

      // no need for group widths if z is constant
      groupWidths = null;
    } else {
      var barWidths = [];

      for (var g = 0; g < numGroups; g++) {
        yWidth = 0;
        for (i = 0; i < numYCategories; i++) {
          yWidth += DvtChartDataUtils.getBarCategoryZ(chart, categories['y'][i], g, false);
        }
        y2Width = 0;
        for (i = 0; i < numY2Categories; i++) {
          y2Width += DvtChartDataUtils.getBarCategoryZ(chart, categories['y2'][i], g, true);
        }
        barWidths.push(isSplitDualY ? Math.max(yWidth, y2Width) : yWidth + y2Width);
      }

      barWidthSum = barWidths.reduce((prev, cur) => {
        return prev + cur;
      });

      // The gap size is the same for all groups, regardless of the bar width.
      gapWidthSum = (barWidthSum * barGapRatio) / (1 - barGapRatio);
      groupWidths = barWidths.map((_barWidth) => {
        // divide the gaps evenly
        return _barWidth + gapWidthSum / numGroups;
      });
    }

    // Store the average z-value. This is useful because when we call groupAxisInfo.getGroupWidth(), it returns the average
    // group width. Thus, we can convert z-value to pixels using (zValue / averageGroupZ * groupAxisInfo.getGroupWidth()).
    options['_averageGroupZ'] = (barWidthSum + gapWidthSum) / numGroups;

    return groupWidths;
  },

  /**
   * Returns the bar gap ratio.
   * @param {Chart} chart
   * @return {Number} The bar gap ratio, between 0 and 1
   */
  getBarGapRatio: (chart) => {
    var cacheKey = 'barGapRatio';
    var barGapRatio = chart.getCache().getFromCache(cacheKey);
    if (barGapRatio) return barGapRatio;

    barGapRatio = chart.getOptions()['styleDefaults']['barGapRatio'];
    if (typeof barGapRatio == 'string' && barGapRatio.slice(-1) == '%')
      // parse percent input
      barGapRatio = Number(barGapRatio.slice(0, -1)) / 100;

    if (barGapRatio != null && !isNaN(barGapRatio)) return Number(barGapRatio);

    var categories = DvtChartDataUtils.getStackCategories(chart, 'bar');
    var numYStacks = categories['y'].length;
    var numY2Stacks = categories['y2'].length;
    var numStacks = DvtChartDataUtils.isSplitDualY(chart)
      ? Math.max(numYStacks, numY2Stacks)
      : numYStacks + numY2Stacks;

    // Fall back to the default
    if (DvtChartTypeUtils.isPolar(chart)) barGapRatio = numStacks == 1 ? 0 : 0.25;
    else
      barGapRatio =
        numStacks == 1 ? 0.37 + 0.26 / DvtChartAxisUtils.getViewportGroupCount(chart) : 0.25;

    chart.getCache().putToCache(cacheKey, barGapRatio);
    return barGapRatio;
  },

  /**
   * Returns the bar information needed to draw it for the specified index.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @param {dvt.Rectangle} availSpace
   * @return {Object} The coordinates associated with this bar for rendering purposes.
   */
  getBarInfo: (chart, seriesIndex, groupIndex, availSpace) => {
    var bHoriz = DvtChartTypeUtils.isHorizontal(chart);
    var bStacked = DvtChartDataUtils.isStacked(chart);
    var isRTL = Agent.isRightToLeft(chart.getCtx());
    var xAxis = chart.xAxis;
    var bRange = DvtChartDataUtils.isRangeSeries(chart, seriesIndex);
    var offsetMap = DvtChartStyleUtils.getBarCategoryOffsetMap(chart, groupIndex);

    // Get the x-axis position
    var xValue = DvtChartDataUtils.getXVal(chart, seriesIndex, groupIndex);
    var xCoord = xAxis.getUnboundedCoordAt(xValue);

    // Find the corresponding y axis
    var bY2Series = DvtChartDataUtils.isAssignedToY2(chart, seriesIndex);
    var yAxis = bY2Series ? chart.y2Axis : chart.yAxis;
    var axisCoord = yAxis.getBaselineCoord();

    // Get the y-axis position
    var yCoord, baseCoord;
    if (bRange) {
      var lowValue = DvtChartDataUtils.getLowVal(chart, seriesIndex, groupIndex);
      var highValue = DvtChartDataUtils.getHighVal(chart, seriesIndex, groupIndex);
      if (lowValue == null || isNaN(lowValue) || highValue == null || isNaN(highValue))
        // Don't render bars whose value is null
        return null;

      yCoord = yAxis.getBoundedCoordAt(lowValue);
      baseCoord = yAxis.getBoundedCoordAt(highValue);

      // Don't render bars whose start and end points are both out of bounds
      if (yCoord == baseCoord && yAxis.getCoordAt(lowValue) == null) return null;
    } else {
      var yValue = DvtChartDataUtils.getVal(chart, seriesIndex, groupIndex);
      var totalYValue = DvtChartDataUtils.getCumulativeVal(chart, seriesIndex, groupIndex);
      if (yValue == null || isNaN(yValue))
        // Don't render bars whose value is null
        return null;

      yCoord = yAxis.getBoundedCoordAt(totalYValue);
      baseCoord = bStacked ? yAxis.getBoundedCoordAt(totalYValue - yValue) : axisCoord;

      // Don't render bars whose start and end points are both out of bounds
      if (yCoord == baseCoord && yAxis.getCoordAt(totalYValue) == null) return null;
    }

    // Calculate the width for the bar.
    var category = DvtChartDataUtils.getStackCategory(chart, seriesIndex);
    var barWidth = DvtChartStyleUtils.getBarWidth(chart, seriesIndex, groupIndex);
    var stackWidth = bStacked
      ? DvtChartStyleUtils.getBarStackWidth(chart, category, groupIndex, bY2Series)
      : barWidth;

    // : Mac FF pixel snaps greedily, so there must be 2 or more pixels of gaps.
    if (DvtChartStyleUtils.getBarSpacing(chart) == 'pixel' && Agent.browser === 'firefox') {
      var groupWidth = barWidth / (1 - DvtChartGroupUtils.getBarGapRatio(chart));
      if (barWidth > 1 && groupWidth - barWidth < 2) {
        barWidth--;
        stackWidth = barWidth;
      }
    }

    // Calculate the actual coords for the bar
    var itemOffset = offsetMap[bY2Series ? 'y2' : 'y'][category] + 0.5 * (stackWidth - barWidth);
    var x1 = isRTL && !bHoriz ? xCoord - itemOffset - barWidth : xCoord + itemOffset;
    var x2 = x1 + barWidth;

    // Store the center of the data point relative to the plot area (for marquee selection)
    var dataPosX = (x1 + x2) / 2;
    var dataPosY = bRange ? (yCoord + baseCoord) / 2 : yCoord;
    var dataPos = DvtChartCoordUtils.convertAxisCoord(
      chart,
      new Point(dataPosX, dataPosY),
      availSpace
    );
    return {
      x1: x1,
      x2: x2,
      axisCoord: axisCoord,
      baseCoord: baseCoord,
      yCoord: yCoord,
      dataPos: dataPos,
      barWidth: barWidth
    };
  },

  /**
   * Returns the data label position for the specified data point.
   * @param {Chart} chart
   * @param {number} seriesIndex The series index.
   * @param {number} groupIndex The group index.
   * @param {number} itemIndex The nested item index.
   * @param {string=} type Data label type: low, high, or value.
   * @param {boolean} isStackLabel true if label for stack cummulative, false otherwise
   * @return {string} The data label position. Uses an internal list different from the API values.
   * Possible values are: center, inLeft, inRight, inTop, inBottom, left, right, top, bottom, none
   */
  getDataLabelPos: (chart, seriesIndex, groupIndex, itemIndex, type, isStackLabel) => {
    var nestedData = DvtChartDataUtils.getNestedDataItem(chart, seriesIndex, groupIndex, itemIndex);
    var data = DvtChartDataUtils.getDataItem(chart, seriesIndex, groupIndex);
    var position;

    if (isStackLabel) position = 'outsideBarEdge';
    else {
      position =
        nestedData && nestedData['labelPosition']
          ? nestedData['labelPosition']
          : data['labelPosition'];

      if (!position) position = chart.getOptions()['styleDefaults']['dataLabelPosition'];
      position = DvtChartStyleUtils._parseLowHighArray(position, type);

      if (position == 'none') return 'none';
    }

    var bRTL = Agent.isRightToLeft(chart.getCtx());
    var bHorizontal = DvtChartTypeUtils.isHorizontal(chart);
    var bPolar = DvtChartTypeUtils.isPolar(chart);

    if (DvtChartTypeUtils.isFunnel(chart) || DvtChartTypeUtils.isPyramid(chart)) {
      return 'center';
    }
    // Bar series
    else if (DvtChartDataUtils.getSeriesType(chart, seriesIndex) == 'bar') {
      // Only center is supported for polar bar
      if (position == 'center' || bPolar) return 'center';

      // Only insideBarEdge, outsideBarEdge, and center are supported for cartesian bar.
      // outsideBarEdge is not supported for stacked because it'll be covered by the bar above.
      var isStacked = DvtChartDataUtils.isStacked(chart);
      if (position != 'insideBarEdge') {
        if (isStacked && !isStackLabel) return 'center';
        else if (position != 'outsideBarEdge') position = 'insideBarEdge';
      }
      if (position == 'insideBarEdge' && !isStacked) {
        var styleDefaultsDataLabel = chart.getOptions()['styleDefaults']['dataLabelStyle'];
        var style = data['labelStyle']
          ? CSSStyle.mergeStyles([styleDefaultsDataLabel, new CSSStyle(data['labelStyle'])])
          : styleDefaultsDataLabel;
        var textDim;

        if (bHorizontal) {
          var text = DvtChartStyleUtils.getDataLabel(
            chart,
            seriesIndex,
            groupIndex,
            itemIndex,
            type
          );
          textDim = TextUtils.getTextStringWidth(chart.getCtx(), text, style);
        } else textDim = TextUtils.getTextStringHeight(chart.getCtx(), style);

        var barInfo = DvtChartGroupUtils.getBarInfo(chart, seriesIndex, groupIndex);
        var barHeight = barInfo ? Math.abs(barInfo.baseCoord - barInfo.yCoord) : 0;

        if (barHeight <= textDim) position = 'outsideBarEdge';
      }

      // Determine if the label is positioned in the low or high position
      var bNegative;
      if (type == 'low') bNegative = data['low'] <= data['high'];
      else if (type == 'high') bNegative = data['high'] < data['low'];
      else bNegative = DvtChartDataUtils.getVal(chart, seriesIndex, groupIndex, itemIndex) < 0;

      if (position == 'outsideBarEdge') {
        if (bHorizontal) return (!bNegative && !bRTL) || (bNegative && bRTL) ? 'right' : 'left';
        else return bNegative ? 'bottom' : 'top';
      } else {
        // insideBarEdge
        if (bHorizontal) return (!bNegative && !bRTL) || (bNegative && bRTL) ? 'inRight' : 'inLeft';
        else return bNegative ? 'inBottom' : 'inTop';
      }
    }

    // Scatter, Bubble, Line, or Area series
    else {
      if (position == 'center') return 'center';
      if (position == 'belowMarker') return 'bottom';
      if (position == 'aboveMarker') return 'top';

      if (position != 'afterMarker' && position != 'beforeMarker') {
        if (DvtChartTypeUtils.isBubble(chart)) return 'center';
        else if (type == 'low' && !bPolar) {
          if (!bHorizontal) return 'bottom';
          else position = 'beforeMarker';
        } else if (type == 'high' && !bPolar) {
          if (!bHorizontal) return 'top';
          else position = 'afterMarker';
        } else position = 'afterMarker';
      }

      if ((!bRTL && position == 'afterMarker') || (bRTL && position == 'beforeMarker'))
        return 'right';
      else return 'left';
    }
  }
};

/**
 * Series effect utility functions for Chart.
 * @class
 */
const DvtChartSeriesEffectUtils = {
  /**
   * Returns the fill for a bar with the given series and group.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @param {number} barWidth The width of the bar
   * @return {dvt.Fill}
   */
  getBarFill: (chart, seriesIndex, groupIndex, barWidth) => {
    var color = DvtChartStyleUtils.getColor(chart, seriesIndex, groupIndex);
    var pattern = DvtChartStyleUtils.getPattern(chart, seriesIndex, groupIndex);
    return DvtChartSeriesEffectUtils.getRectangleFill(chart, color, pattern, barWidth);
  },

  /**
   * Returns the fill for a rectangular shape
   * @param {Chart} chart
   * @param {string} color
   * @param {string} pattern
   * @param {number} width The width of the rectangle
   * @return {dvt.Fill}
   */
  getRectangleFill: (chart, color, pattern, width) => {
    var seriesEffect = DvtChartStyleUtils.getSeriesEffect(chart);

    if (pattern) return new PatternFill(pattern, color);
    else if (seriesEffect == 'gradient' && width > 3) {
      // to improve performance, don't use gradient if rect is too thin
      var colors;
      var stops;
      var angle = DvtChartTypeUtils.isHorizontal(chart) ? 270 : 0;
      colors = [
        ColorUtils.adjustHSL(color, 0, -0.09, 0.04),
        ColorUtils.adjustHSL(color, 0, -0.04, -0.05)
      ];
      stops = [0, 1.0];

      return new LinearGradientFill(angle, colors, null, stops);
    } // seriesEffect="color"
    else return new SolidFill(color);
  },

  /**
   * Returns the fill for an area with the given series and group.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @return {dvt.Fill}
   */
  getAreaFill: (chart, seriesIndex) => {
    var isLineWithArea = DvtChartDataUtils.getSeriesType(chart, seriesIndex) == 'lineWithArea';

    // Get the color
    var color;
    var seriesItem = DvtChartDataUtils.getSeriesItem(chart, seriesIndex);
    if (seriesItem && seriesItem['areaColor']) color = seriesItem['areaColor'];
    else {
      color = DvtChartStyleUtils.getColor(chart, seriesIndex);
      if (isLineWithArea) color = ColorUtils.setAlpha(color, 0.2);
    }

    // All series effects are based off of the color
    var pattern = DvtChartStyleUtils.getPattern(chart, seriesIndex);
    var seriesEffect = DvtChartStyleUtils.getSeriesEffect(chart);

    if (pattern) return new PatternFill(pattern, color);
    else if (seriesEffect == 'gradient') {
      var colors, stops;
      var angle = DvtChartTypeUtils.isHorizontal(chart) ? 180 : 270;
      if (isLineWithArea) {
        var alpha = ColorUtils.getAlpha(color);
        colors = [
          ColorUtils.setAlpha(color, Math.min(alpha + 0.2, 1)),
          ColorUtils.setAlpha(color, Math.max(alpha - 0.15, 0))
        ];
        stops = [0, 1.0];
      } else {
        colors = [
          ColorUtils.adjustHSL(color, 0, -0.09, 0.04),
          ColorUtils.adjustHSL(color, 0, -0.04, -0.05)
        ];
        stops = [0, 1.0];
      }
      return new LinearGradientFill(angle, colors, null, stops);
    } // seriesEffect="color"
    else return new SolidFill(color);
  },

  /**
   * Returns the fill for a marker with the given series and group.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @param {number} itemIndex
   * @return {dvt.Fill}
   */
  getMarkerFill: (chart, seriesIndex, groupIndex, itemIndex) => {
    // All series effects are based off of the color
    var color = DvtChartStyleUtils.getMarkerColor(chart, seriesIndex, groupIndex, itemIndex);
    var pattern = DvtChartStyleUtils.getPattern(chart, seriesIndex, groupIndex, itemIndex);

    if (pattern) return new PatternFill(pattern, color);

    // Only bubble markers use series effect(gradient)
    if (DvtChartTypeUtils.isBubble(chart)) {
      var seriesEffect = DvtChartStyleUtils.getSeriesEffect(chart);

      if (seriesEffect == 'gradient') {
        var colors = [
          ColorUtils.adjustHSL(color, 0, -0.09, 0.04),
          ColorUtils.adjustHSL(color, 0, -0.04, -0.05)
        ];
        var stops = [0, 1.0];
        return new LinearGradientFill(270, colors, null, stops);
      }
    }

    // seriesEffect="color" or line/scatter marker
    return new SolidFill(color);
  },

  /**
   * Returns the fill for a funnel or pyramid slice with the given series and group.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @param {string} color  The color that is associated with the funnel or pyramid slice, to apply the effect onto it.
   * @param {dvt.Rectangle} dimensions  The dimensions of this funnel or pyramid slice, to pass as limits for the gradient effect.
   * @param {boolean} bBackground
   * @return {dvt.Fill}
   */
  getFunnelPyramidSliceFill: (chart, seriesIndex, color, dimensions, bBackground) => {
    var pattern = DvtChartStyleUtils.getPattern(chart, seriesIndex, 0);
    var seriesEffect = DvtChartStyleUtils.getSeriesEffect(chart);

    if (pattern && !bBackground) {
      // Need to rotate the pattern if vertical to counteract the chart rotation.
      var matrix;
      if (chart.getOptions()['orientation'] == 'vertical' || DvtChartTypeUtils.isPyramid(chart)) {
        if (Agent.isRightToLeft(chart.getCtx())) matrix = new Matrix(0, -1, 1, 0);
        else matrix = new Matrix(0, 1, -1, 0);
      }
      return new PatternFill(pattern, color, null, matrix);
    } else if (seriesEffect == 'gradient') {
      var angle = DvtChartTypeUtils.isPyramid(chart) ? 180 : 90;
      var colors, stops;
      if (chart.getOptions()['styleDefaults']['threeDEffect'] == 'on') {
        colors = [
          ColorUtils.adjustHSL(color, 0, 0, -0.1),
          ColorUtils.adjustHSL(color, 0, 0, 0.12),
          color
        ];
        stops = [0, 0.65, 1.0];
      } else {
        colors = [
          ColorUtils.adjustHSL(color, 0, -0.09, 0.04),
          ColorUtils.adjustHSL(color, 0, -0.04, -0.05)
        ];
        stops = [0, 1.0];
      }
      return new LinearGradientFill(angle, colors, null, stops, [
        dimensions.x,
        dimensions.y,
        dimensions.w,
        dimensions.h
      ]);
    } // seriesEffect="color"
    else return new SolidFill(color);
  }
};

/**
 *  Creates a funnel shape.
 *  @extends {dvt.Path}
 *  @class DvtChartFunnelSlice  Creates a funnel slice object.
 *  @constructor
 *  @param {Chart} chart  The chart being rendered.
 *  @param {number} seriesIndex  The index of this slice.
 *  @param {number} numDrawnSeries  The number of series already drawn. Should be total number of series - seriesIndex - 1 if none are skipped.
 *  @param {number} funnelWidth The available width for the whole funnel.
 *  @param {number} funnelHeight The available height for the whole funnel.
 *  @param {number} startPercent The cumulative value of all the slices that come before. The start/leftmost value of the slice.
 *  @param {number} valuePercent The percent value for the slice. Dictates the width.
 *  @param {number} fillPercent The actual/target value, how much of the slice is filled.
 *  @param {number} gap The gap distance between slices.
 */
class DvtChartFunnelSlice extends Path {
  /**
   *  Object initializer.
   *  @param {Chart} chart  The chart being rendered.
   *  @param {number} seriesIndex  The index of this slice.
   *  @param {number} numDrawnSeries  The number of series already drawn. Should be total number of series - seriesIndex - 1 if none are skipped.
   *  @param {number} funnelWidth The available width for the whole funnel.
   *  @param {number} funnelHeight The available height for the whole funnel.
   *  @param {number} startPercent The cumulative value of all the slices that come before. The start/leftmost value of the slice.
   *  @param {number} valuePercent The percent value for the slice. Dictates the width.
   *  @param {number} fillPercent The actual/target value, how much of the slice is filled.
   *  @param {number} gap The gap distance between slices.
   *  @protected
   */
  constructor(
    chart,
    seriesIndex,
    numDrawnSeries,
    funnelWidth,
    funnelHeight,
    startPercent,
    valuePercent,
    fillPercent,
    gap
  ) {
    super(chart.getCtx());

    /** The ratio of rx/ry in the 3D funnel opening
     * @private */
    this._FUNNEL_3D_WIDTH_RATIO = 0.08;
    /** Angle for creating the funnel sides
     * @private */
    this._FUNNEL_ANGLE_2D = 36;
    /** Ratio between the smallest and largest slices
     * @private */
    this._FUNNEL_RATIO = 1 / 3;
    /** Color for funnel slice border. Could be overridden in styleDefaults.
     * @private */
    this._BORDER_COLOR = '#FFFFFF';
    /** Minimum number of characters to use when truncating.
     * @private */
    this._MIN_CHARS_DATA_LABEL = 3;
    /** Length of the first line.
     * @private */
    this._LINE_FRACTION = 2 / 3;
    /** Fraction into which the funnel area is divided by the first line.
     * @private */
    this._AREA_FRACTION = 0.41;
    /** Fraction into which the funnel height is divided by the first line.
     * @private */
    this._HEIGHT_FRACTION = 0.28;
    /** Length of the second line.
     * @private */
    this._LINE_FRACTION_2 = 0.4;
    /** Fraction into which the funnel area is divided by the second line.
     * @private */
    this._AREA_FRACTION_2 = 0.8;
    /** Fraction into which the funnel height is divided by the second line.
     * @private */
    this._HEIGHT_FRACTION_2 = 0.7;

    this._chart = chart;
    var styleDefaults = chart.getOptions()['styleDefaults'];
    this._seriesIndex = seriesIndex;
    this._numDrawnSeries = numDrawnSeries;
    this._funnelWidth = funnelWidth;
    this._funnelHeight = funnelHeight;
    this._startPercent = startPercent;
    this._valuePercent = valuePercent;
    this._fillPercent = fillPercent;
    this._3dRatio = styleDefaults['threeDEffect'] == 'on' ? 1 : 0;
    this._gap = gap;
    var cmds = this._getPath();
    this._dataColor = DvtChartStyleUtils.getColor(this._chart, this._seriesIndex, 0);
    // Read the color from backgroundColor for backwards compatibility
    this._backgroundColor = styleDefaults['backgroundColor']
      ? styleDefaults['backgroundColor']
      : styleDefaults['funnelBackgroundColor'];
    this.setCmds(cmds['slice']);

    if (cmds['bar']) {
      this._bar = new Path(this.getCtx(), cmds['bar']);
      this.addChild(this._bar);
      this._bar.setMouseEnabled(false); // want the mouse interaction to only be with the slice.
    }

    this._setColorProps(cmds['sliceBounds']);
    this._label = this._getSliceLabel(cmds['sliceBounds'], cmds['barBounds']);

    if (this._label != null) {
      this._label.setMouseEnabled(false);
      this.addChild(this._label);
    }
  }

  /**
   * Creates the path commands that represent this slice
   * @return {object} The commands for drawing this slice. An object containing the sliceCommands, barCommands, sliceBounds and barBounds
   * @private
   */
  _getPath() {
    var isBiDi = Agent.isRightToLeft(this.getCtx());
    var gapCount = DvtChartDataUtils.getSeriesCount(this._chart);
    var offset = (this._numDrawnSeries + 1) * this._gap;
    var angle = Math$1.degreesToRads(this._FUNNEL_ANGLE_2D - 2 * this._3dRatio);

    var totalWidth = this._funnelWidth - gapCount * this._gap;
    var rx = totalWidth / Math.sin(Math$1.degreesToRads(this._FUNNEL_ANGLE_2D));
    var ry = this._funnelHeight / Math.sin(angle);
    var ratio =
      ((this._3dRatio * this._funnelWidth) / this._funnelHeight) * this._FUNNEL_3D_WIDTH_RATIO;
    if (ratio < 0.00001) ratio = 0;

    // Dividing the funnel into three trapezoids to come up with a better approximation for the dimensions. We draw two lines,
    // at .28 and .7 of the height, and they split the area to the ratio .41: .39: .2.
    var b1 = this._funnelHeight;
    var b2 = this._funnelHeight * this._FUNNEL_RATIO;
    var p1, p2; // The percent at which we are calculating the width
    var b11, b12, b21, b22; // The first and second base of the trapezoid into which the percent we are looking at falls.
    var f1, f2; // The fraction of the area included in the trapezoid we are considering.
    var t1, t2; // Total width of the trapezoid we are considering.
    var h1, h2; // Horizontal distance from the slice to the center of the ellipse for drawing the funnel arcs.

    // calculating first edge
    if (this._startPercent < this._AREA_FRACTION) {
      p1 = this._startPercent;
      b11 = b1;
      b21 = this._funnelHeight * this._LINE_FRACTION;
      f1 = this._AREA_FRACTION;
      t1 = totalWidth * this._HEIGHT_FRACTION;
      h1 = totalWidth * (1 - this._HEIGHT_FRACTION);
    } else if (this._startPercent < this._AREA_FRACTION_2) {
      p1 = this._startPercent - this._AREA_FRACTION;
      b11 = this._funnelHeight * this._LINE_FRACTION;
      b21 = this._funnelHeight * this._LINE_FRACTION_2;
      f1 = this._AREA_FRACTION_2 - this._AREA_FRACTION;
      t1 = totalWidth * (this._HEIGHT_FRACTION_2 - this._HEIGHT_FRACTION);
      h1 = totalWidth * (1 - this._HEIGHT_FRACTION_2);
    } else {
      p1 = this._startPercent - this._AREA_FRACTION_2;
      b11 = this._funnelHeight * this._LINE_FRACTION_2;
      b21 = b2;
      f1 = 1 - this._AREA_FRACTION_2;
      t1 = totalWidth * (1 - this._HEIGHT_FRACTION_2);
      h1 = 0;
    }

    // Calculating second edge
    if (this._startPercent + this._valuePercent < this._AREA_FRACTION) {
      b12 = b1;
      b22 = this._funnelHeight * this._LINE_FRACTION;
      p2 = this._startPercent + this._valuePercent;
      f2 = this._AREA_FRACTION;
      t2 = totalWidth * this._HEIGHT_FRACTION;
      h2 = totalWidth * (1 - this._HEIGHT_FRACTION);
    } else if (this._startPercent + this._valuePercent < this._AREA_FRACTION_2) {
      b12 = this._funnelHeight * this._LINE_FRACTION;
      b22 = this._funnelHeight * this._LINE_FRACTION_2;
      p2 = this._startPercent + this._valuePercent - this._AREA_FRACTION;
      f2 = this._AREA_FRACTION_2 - this._AREA_FRACTION;
      t2 = totalWidth * (this._HEIGHT_FRACTION_2 - this._HEIGHT_FRACTION);
      h2 = totalWidth * (1 - this._HEIGHT_FRACTION_2);
    } else {
      b12 = this._funnelHeight * this._LINE_FRACTION_2;
      b22 = b2;
      p2 = this._startPercent + this._valuePercent - this._AREA_FRACTION_2;
      f2 = 1 - this._AREA_FRACTION_2;
      t2 = totalWidth * (1 - this._HEIGHT_FRACTION_2);
      h2 = 0;
    }

    var w1 = Math.sqrt(((f1 - p1) / f1) * b11 * b11 + (p1 / f1) * b21 * b21);
    var w2 = Math.sqrt(((f2 - p2) / f2) * b12 * b12 + (p2 / f2) * b22 * b22);

    var startAngle = 0.98 * Math.asin((((w1 - b21) * t1) / (b11 - b21) + h1) / rx);
    var endAngle = 0.98 * Math.asin((((w2 - b22) * t2) / (b12 - b22) + h2) / rx);

    var c1 = ((1 + this._FUNNEL_RATIO) / 2) * this._funnelHeight + ry;
    var c2 = ((1 - this._FUNNEL_RATIO) / 2) * this._funnelHeight - ry;
    var ar, arcDir1, arcDir2;

    if (isBiDi) {
      ar = [
        rx * Math.sin(startAngle) + offset,
        c1 - ry * Math.cos(startAngle),
        rx * Math.sin(endAngle) + offset,
        c1 - ry * Math.cos(endAngle),
        rx * Math.sin(endAngle) + offset,
        c2 + ry * Math.cos(endAngle),
        rx * Math.sin(startAngle) + offset,
        c2 + ry * Math.cos(startAngle)
      ];
      arcDir1 = 0;
      arcDir2 = 1;
    } else {
      ar = [
        this._funnelWidth - offset - rx * Math.sin(startAngle),
        c1 - ry * Math.cos(startAngle),
        this._funnelWidth - offset - rx * Math.sin(endAngle),
        c1 - ry * Math.cos(endAngle),
        this._funnelWidth - offset - rx * Math.sin(endAngle),
        c2 + ry * Math.cos(endAngle),
        this._funnelWidth - offset - rx * Math.sin(startAngle),
        c2 + ry * Math.cos(startAngle)
      ];
      arcDir1 = 1;
      arcDir2 = 0;
    }

    var pathCommands = PathUtils.moveTo(ar[0], ar[1]);
    var barCommands = null;
    pathCommands += PathUtils.arcTo(
      (ratio * (ar[1] - ar[7])) / 2,
      (ar[1] - ar[7]) / 2,
      Math.PI,
      arcDir2,
      ar[6],
      ar[7]
    );
    pathCommands += PathUtils.arcTo(
      (ratio * (ar[1] - ar[7])) / 2,
      (ar[1] - ar[7]) / 2,
      Math.PI,
      arcDir2,
      ar[0],
      ar[1]
    );
    pathCommands += PathUtils.arcTo(rx, ry, angle, arcDir1, ar[2], ar[3]);
    pathCommands += PathUtils.arcTo(
      (ratio * (ar[3] - ar[5])) / 2,
      (ar[3] - ar[5]) / 2,
      Math.PI,
      arcDir2,
      ar[4],
      ar[5]
    );
    pathCommands += PathUtils.arcTo(rx, ry, angle, arcDir1, ar[6], ar[7]);
    var sliceBounds = new Rectangle(
      Math.min(ar[0], ar[2]),
      ar[5],
      Math.abs(ar[0] - ar[2]),
      Math.abs(ar[3] - ar[5])
    );

    if (this._fillPercent != null) {
      // creating the bar commands for 3D slices if applicable
      var percent = Math.max(Math.min(this._fillPercent, 1), 0);
      var alpha = isBiDi ? -percent * Math.PI : percent * Math.PI;
      barCommands = PathUtils.moveTo(ar[0], ar[1]);
      barCommands += PathUtils.arcTo(rx, ry, angle, arcDir1, ar[2], ar[3]);
      barCommands += PathUtils.arcTo(
        (ratio * (ar[3] - ar[5])) / 2,
        (ar[3] - ar[5]) / 2,
        alpha,
        arcDir2,
        ar[2] + ((ratio * (ar[3] - ar[5])) / 2) * Math.sin(alpha),
        (ar[5] + ar[3]) / 2 + ((ar[3] - ar[5]) / 2) * Math.cos(alpha)
      );
      // Edge cases require different bar shapes so they don't spill out of the slice.
      if (this._fillPercent > 0.95)
        barCommands += PathUtils.arcTo(
          rx,
          ry,
          angle,
          arcDir1,
          ar[6],
          ar[1] + percent * (ar[7] - ar[1])
        );
      else if (this._fillPercent < 0.05)
        barCommands += PathUtils.arcTo(
          rx,
          ry,
          angle,
          arcDir2,
          ar[6],
          ar[1] + percent * (ar[7] - ar[1])
        );
      else
        barCommands += PathUtils.lineTo(
          ar[6] + ((ratio * (ar[1] - ar[7])) / 2) * Math.sin(alpha),
          (ar[7] + ar[1]) / 2 + ((ar[1] - ar[7]) / 2) * Math.cos(alpha)
        );
      barCommands += PathUtils.arcTo(
        (ratio * (ar[1] - ar[7])) / 2,
        (ar[1] - ar[7]) / 2,
        alpha,
        arcDir1,
        ar[0],
        ar[1]
      );
      barCommands += PathUtils.closePath();
      var barBounds = new Rectangle(
        Math.min(ar[0], ar[2]),
        ar[5] + Math.abs(ar[3] - ar[5]) * (1 - percent),
        Math.abs(ar[0] - ar[2]),
        Math.abs(ar[3] - ar[5]) * percent
      );
    }
    return {
      slice: pathCommands,
      bar: barCommands,
      sliceBounds: sliceBounds,
      barBounds: barCommands ? barBounds : sliceBounds
    };
  }

  /**
   * Creates a single slice label dvt.Text object associated with this slice.
   * @param {dvt.Rectangle} sliceBounds The space occupied by the slice this is associated with.
   * @param {dvt.Rectangle} barBounds The space occupied by the colored bar this is associated with. Could affect the color.
   * @return {dvt.OutputText} slice label for this slice
   * @private
   */
  _getSliceLabel(sliceBounds, barBounds) {
    // Get and create the label string
    var labelString = DvtChartStyleUtils.getDataLabel(this._chart, this._seriesIndex, 0);
    if (!labelString)
      // if no data label set on the data item, set it from the series
      labelString = DvtChartDataUtils.getSeriesLabel(this._chart, this._seriesIndex);

    // Return if no label or label position none
    if (
      !labelString ||
      DvtChartGroupUtils.getDataLabelPos(this._chart, this._seriesIndex, 0) == 'none'
    )
      return;

    var label = new MultilineText(this.getCtx(), labelString, 0, 0);

    // Have to move the style setting first because was using wrong font size to come up with truncated text
    var isPatternBg = DvtChartStyleUtils.getPattern(this._chart, this._seriesIndex, 0) != null;
    var styleDefaults = this._chart.getOptions().styleDefaults;
    var labelStyleArray = [
      styleDefaults._dataLabelStyle,
      styleDefaults.dataLabelStyle,
      new CSSStyle(
        DvtChartDataUtils.getDataItem(this._chart, this._seriesIndex, 0)['labelStyle']
      )
    ];
    var style = CSSStyle.mergeStyles(labelStyleArray);
    label.setCSSStyle(style);

    // Truncating text and dropping if doesn't fit.
    if (
      !TextUtils.fitText(
        label,
        sliceBounds.h - this._3dRatio * (0.8 - this._valuePercent) * 50,
        sliceBounds.w,
        this,
        this._MIN_CHARS_DATA_LABEL
      )
    )
      return;

    var textDim = label.getDimensions();
    var pos = this._getLabelPos(sliceBounds);
    // Checking if the text starts within the bounding box of the colored bar.
    if (isPatternBg) {
      var padding = textDim.h * 0.15;
      var displacement = Agent.isRightToLeft(this.getCtx()) ? 0.5 : -0.5;
      var cmd = PathUtils.roundedRectangle(
        textDim.x - padding,
        textDim.y,
        textDim.w + 2 * padding,
        textDim.h,
        2,
        2,
        2,
        2
      );
      var bbox = new Path(this.getCtx(), cmd);
      bbox.setSolidFill(DvtChartStyleUtils.SERIES_PATTERN_BG_COLOR, 0.9);
      pos = pos.translate(displacement * textDim.h, -displacement * textDim.w);
      bbox.setMatrix(pos);
      this.addChild(bbox);
    }
    var labelColor = isPatternBg
      ? ColorUtils.getContrastingTextColor(DvtChartStyleUtils.SERIES_PATTERN_BG_COLOR)
      : barBounds.containsPoint(sliceBounds.x, sliceBounds.y + (sliceBounds.h - textDim.w) / 2)
      ? ColorUtils.getContrastingTextColor(this._dataColor)
      : ColorUtils.getContrastingTextColor(this._backgroundColor);
    var labelColorStyle = new CSSStyle({ color: labelColor });
    // Don't want to override the color if it was set above, unless in high contrast mode.
    labelStyleArray.splice(1, 0, labelColorStyle);
    if (Agent.isHighContrast()) {
      labelStyleArray.push(labelColorStyle);
    }
    style = CSSStyle.mergeStyles(labelStyleArray);
    label.setCSSStyle(style);

    label.setMatrix(this._getLabelPos(sliceBounds));
    label.alignCenter();
    label.alignMiddle();
    return label;
  }

  /**
   * Calculates the position of the text within this slice. Comes up with the translation/rotation matrix.
   * @param {dvt.Rectangle} sliceBounds The space occupied by the slice.
   * @return {dvt.Matrix} The matrix representing the transformation for placing the text.
   * @private
   */
  _getLabelPos(sliceBounds) {
    var displacement =
      this._3dRatio *
      ((((sliceBounds.h * this._funnelWidth) / this._funnelHeight) * this._FUNNEL_3D_WIDTH_RATIO) /
        2); // to make up for the 3D funnel opening
    // Rotate the text
    var rotationMatrix = new Matrix();
    if (Agent.isRightToLeft(this.getCtx())) {
      rotationMatrix = rotationMatrix.rotate(Math.PI / 2);
      rotationMatrix = rotationMatrix.translate(
        sliceBounds.x + sliceBounds.w / 2 - displacement,
        sliceBounds.y + sliceBounds.h / 2
      );
    } else {
      rotationMatrix = rotationMatrix.rotate((3 * Math.PI) / 2);
      rotationMatrix = rotationMatrix.translate(
        sliceBounds.x + sliceBounds.w / 2 + displacement,
        sliceBounds.y + sliceBounds.h / 2
      );
    }
    return rotationMatrix;
  }

  /**
   * Passing on the colors for the funnel slice object. Sets the slice fill and border color, as well as the selection and hover colors by reading them from the chart.
   * @param {dvt.Rectangle} sliceBounds The space occupied by the slice. This is used for calculating the gradient effect bounds.
   * @private
   */
  _setColorProps(sliceBounds) {
    var sliceFill = DvtChartSeriesEffectUtils.getFunnelPyramidSliceFill(
      this._chart,
      this._seriesIndex,
      this._dataColor,
      sliceBounds
    );
    var sliceBorder = DvtChartStyleUtils.getBorderColor(this._chart, this._seriesIndex, 0);

    var userBorderWidth = DvtChartStyleUtils.getUserBorderWidth(this._chart, this._seriesIndex, 0);
    var defaultBorderWidth = DvtChartStyleUtils.getDefaultBorderWidth(
      this._chart,
      this._seriesIndex,
      0
    );
    var borderIsFrom3D = false;
    var isRedwood = this.getCtx().getThemeBehavior() === 'redwood';
    if (sliceBorder == null && this._3dRatio > 0) {
      borderIsFrom3D = true;
      sliceBorder = this._BORDER_COLOR;
    }
    var borderWidth = userBorderWidth !== null ? userBorderWidth : defaultBorderWidth;
    if (
      sliceBorder &&
      !(isRedwood && this._chart.isSelectionSupported() && !this._bar && borderIsFrom3D)
    ) {
      this.setSolidStroke(sliceBorder, null, borderWidth);
    }
    var hoverColor = SelectionEffectUtils.getHoverBorderColor(this._dataColor);
    var backgroundFill = DvtChartSeriesEffectUtils.getFunnelPyramidSliceFill(
      this._chart,
      this._seriesIndex,
      this._backgroundColor,
      sliceBounds,
      true
    );
    if (this._bar) {
      this.setFill(backgroundFill);
      this._bar.setFill(sliceFill);
      this._bar.setStyle(DvtChartStyleUtils.getStyle(this._chart, this._seriesIndex, 0));
      this._bar.setClassName(DvtChartStyleUtils.getClassName(this._chart, this._seriesIndex, 0));
    } else {
      this.setFill(sliceFill);
      this.setStyle(DvtChartStyleUtils.getStyle(this._chart, this._seriesIndex, 0));
      this.setClassName(DvtChartStyleUtils.getClassName(this._chart, this._seriesIndex, 0));
    }
    // Save the original border stroke
    this.OriginalStroke = this.getStroke();
    var shapeForSelection = this._bar != null ? this._bar : this;

    if (this._chart.isSelectionSupported()) {
      this.setCursor(SelectionEffectUtils.getSelectingCursor());
      if (isRedwood) {
        if (this._bar != null) {
          shapeForSelection.setSolidStroke(this._dataColor, null, 0);
          shapeForSelection.setFeedbackClassName('oj-dvt-selectable');
        } else {
          var strokeColor = sliceBorder != null && !borderIsFrom3D ? sliceBorder : this._dataColor;
          var strokeWidth = userBorderWidth !== null ? userBorderWidth : 0;
          shapeForSelection.setSolidStroke(strokeColor, null, strokeWidth);
          shapeForSelection.setFeedbackClassName('oj-dvt-selectable');
          if (sliceBorder) {
            if (borderIsFrom3D) {
              shapeForSelection.addClassName('oj-dvt-default-border-color');
            }
            if (userBorderWidth == null) {
              shapeForSelection.addClassName('oj-dvt-default-border-width');
            }
          }
        }
      } else {
        var innerColor = DvtChartStyleUtils.getSelectedInnerColor(this._chart);
        var outerColor = DvtChartStyleUtils.getSelectedOuterColor(this._chart)
          ? DvtChartStyleUtils.getSelectedOuterColor(this._chart)
          : this._dataColor;
        // Set the selection strokes
        shapeForSelection.setHoverStroke(new Stroke(hoverColor, 1, 2));
        shapeForSelection.setSelectedStroke(
          new Stroke(innerColor, 1, 1.5),
          new Stroke(outerColor, 1, 4.5)
        );
        shapeForSelection.setSelectedHoverStroke(
          new Stroke(innerColor, 1, 1.5),
          new Stroke(hoverColor, 1, 4.5)
        );
      }
    }
  }

  /**
   * Gets the percent values associated with the slice for animation
   * @return {array} the start, value, fill percents, and alpha for this slice.
   */
  getAnimParams() {
    return [
      this._startPercent,
      this._valuePercent,
      this._fillPercent,
      this.getAlpha(),
      this._3dRatio
    ];
  }

  /**
   * Sets the percent values associated with the slice for animation
   * @param {array} ar The new start, value, and fill percents for this slice
   */
  setAnimParams(ar) {
    this._startPercent = ar[0];
    this._valuePercent = ar[1];
    this._fillPercent = this._fillPercent != null ? ar[2] : null;
    this.setAlpha(ar[3]);
    this._3dRatio = ar[4];
    var cmds = this._getPath();

    this.setCmds(cmds['slice']);
    if (cmds['bar'] && this._bar) this._bar.setCmds(cmds['bar']);
    if (this._label) this._label.setMatrix(this._getLabelPos(cmds['sliceBounds']));
  }

  /**
   * @override
   */
  setSelected(selected) {
    if (this._bar != null) {
      if (this.IsSelected == selected) return;
      this.IsSelected = selected;
      this._bar.setSelected(selected);
    } else super.setSelected(selected);

    var dims = this.getDimensions();
    var shapeForSelection = this._bar != null ? this._bar : this;
    var displacement = 3;
    // To make the selection effect more apparent - make the bars slightly smaller
    var w = dims.w;
    if (selected) {
      shapeForSelection.setScaleX((w - displacement) / w);
      shapeForSelection.setTranslateX(Math.ceil(displacement / 2) + (displacement / w) * dims.x);
    } else {
      shapeForSelection.setScaleX(1);
      shapeForSelection.setTranslateX(0);
    }
  }

  /**
   * @override
   */
  showHoverEffect() {
    if (this._bar != null) {
      this._bar.showHoverEffect();
    } else super.showHoverEffect();
  }

  /**
   * @override
   */
  hideHoverEffect() {
    if (this._bar != null) {
      this._bar.hideHoverEffect();
    } else super.hideHoverEffect();
  }

  /**
   * @override
   */
  copyShape() {
    return new DvtChartFunnelSlice(
      this._chart,
      this._seriesIndex,
      this._numDrawnSeries,
      this._funnelWidth,
      this._funnelHeight,
      this._startPercent,
      this._valuePercent,
      this._fillPercent,
      this._gap
    );
  }
}

/**
 * Utility functions for Chart eventing and interactivity.
 * @class
 */
const DvtChartEventUtils = {
  /**
   * Updates the visibility of the specified category.  Returns true if the visibility is successfully
   * updated.  This function will return false if the visibility change is rejected, such as when
   * hiding the last visible series with hideAndShowBehavior withRescale.
   * @param {Chart} chart
   * @param {string} category
   * @param {string} visibility The new visibility of the category.
   * @return {boolean} True if the visibility was successfully updated.
   */
  setVisibility: (chart, category, visibility) => {
    //  - HIDE & SHOW FOR REFERENCE OBJECTS
    var refObj = DvtChartRefObjUtils.getRefObj(chart, category);
    if (refObj != null) {
      refObj['visibility'] = visibility;
    }
    // Update the categories list
    var hiddenCategories = DvtChartDataUtils.getHiddenCategories(chart);
    var index = hiddenCategories.indexOf(category);
    if (visibility == 'hidden' && index < 0) hiddenCategories.push(category);
    else if (visibility == 'visible' && index >= 0) hiddenCategories.splice(index, 1);

    // Update the legend
    var options = chart.getOptions();
    if (options && options['legend'] && options['legend']['sections']) {
      // Iterate through any sections defined
      for (var i = 0; i < options['legend']['sections'].length; i++) {
        var dataSection = options['legend']['sections'][i];
        if (dataSection && dataSection['items']) {
          // Find the matching item and apply visibility
          for (var j = 0; j < dataSection['items'].length; j++) {
            if (dataSection['items'][j]['id'] == category)
              dataSection['items'][j]['categoryVisibility'] = visibility;
          }
        }
      }

      return true;
    }

    return false;
  },

  /**
   * Processes an array of DvtChartDataItems representing the current selection, making them ready to be
   * dispatched to the callback. This is primarily used for expanding the "Other" data item into its contained
   * ids.
   * @param {Chart} chart
   * @param {array} selection The array of unprocessed ids.
   * @return {array} The array of processed ids, ready to be dispatched.
   */
  processIds: (chart, selection) => {
    var ret = [];
    for (var i = 0; i < selection.length; i++) {
      var item = selection[i];
      if (item.series == DvtChartPieUtils.OTHER_ID) {
        // Get the slice ids of the slices that are grouped into "other" slice
        var otherItems = DvtChartPieUtils.getOtherSliceIds(chart);
        ret = ret.concat(otherItems);
      } else ret.push(item);
    }
    return ret;
  },

  /**
   * Enlarge marquee rectangular bounds by one pixel in all directions to cover the points at the edges.
   * @param {object} event The marquee event.
   */
  adjustBounds: (event) => {
    if (event.x != null) event.x -= 1;
    if (event.w != null) event.w += 2;
    if (event.y != null) event.y -= 1;
    if (event.h != null) event.h += 2;
  },

  /**
   * Gets the chart data items that are inside a marquee.
   * @param {Chart} chart The chart.
   * @param {object} event The marquee event, which contains the rectangular bounds (relative to stage).
   * @return {array} Array of peer objects that are inside the rectangle.
   */
  getBoundedObjs: (chart, event) => {
    // Include the filtered data items so that they are included if they are inside the marquee rectangle
    var peers = chart.getChartObjPeers().concat(DvtChartEventUtils.getFilteredChartObjPeers(chart));
    var boundedPeers = [];

    for (var i = 0; i < peers.length; i++) {
      var peer = peers[i];
      var dataPos = peer.getDataPosition();

      if (dataPos) {
        // since marquee coords are converted from (possibly scaled) stage to local (unscaled svg) coords,
        // we want to get the unscaled stage coord here.
        dataPos = chart
          .getPlotArea()
          .ConvertCoordSpaceRect(
            { x: dataPos.x, y: dataPos.y, h: 0, w: 0 },
            chart.getCtx().getStage()
          );
        var withinX = event.x == null || (dataPos.x >= event.x && dataPos.x <= event.x + event.w);
        var withinY = event.y == null || (dataPos.y >= event.y && dataPos.y <= event.y + event.h);
        if (withinX && withinY) boundedPeers.push(peer);
      }
    }

    return boundedPeers;
  },

  /**
   * Returns the peers of the chart data items that are filtered.
   * @param {Chart} chart
   * @return {Array}
   */
  getFilteredChartObjPeers: (chart) => {
    if (!chart.getCache().getFromCache('dataFiltered')) return [];

    var cacheKey = 'filteredChartObjPeers';
    var filteredPeers = chart.getCache().getFromCache(cacheKey);

    if (!filteredPeers) {
      filteredPeers = [];
      for (var s = 0; s < DvtChartDataUtils.getSeriesCount(chart); s++) {
        for (var g = 0; g < DvtChartDataUtils.getGroupCount(chart); g++) {
          if (!DvtChartDataUtils.isDataItemFiltered(chart, s, g)) continue;

          var dataPos;
          if (DvtChartDataUtils.getSeriesType(chart, s) == 'bar')
            dataPos = DvtChartGroupUtils.getBarInfo(chart, s, g).dataPos;
          else dataPos = DvtChartDataUtils.getMarkerPos(chart, s, g);

          filteredPeers.push(new DvtChartObjPeer(chart, [], s, g, null, dataPos));
        }
      }
      chart.getCache().putToCache(cacheKey, filteredPeers);
    }

    return filteredPeers;
  },

  /**
   * Gets the chart axis bounds corresponding to the bounding rectangle.
   * @param {Chart} chart The chart.
   * @param {object} event The marquee event, which contains the rectangular bounds (relative to stage).
   * @param {boolean} limitExtent Whether the result should be limited to the axis min/max extent.
   * @return {object} An object containing xMin, xMax, yMin, yMax, startGroup, endGroup corresponding to the bounds.
   */
  getAxisBounds: (chart, event, limitExtent) => {
    // Get the bounds in the axis coordinates
    var plotArea = chart.getPlotArea();
    var minPt = plotArea.stageToLocal(new Point(event.x, event.y));
    var maxPt = plotArea.stageToLocal(new Point(event.x + event.w, event.y + event.h));
    // Reset null values because they would be treated as zeros by stageToLocal()
    if (event.x == null) {
      minPt.x = null;
      maxPt.x = null;
    }
    if (event.y == null) {
      minPt.y = null;
      maxPt.y = null;
    }
    var coords = DvtChartEventUtils._convertToAxisCoord(chart, minPt.x, maxPt.x, minPt.y, maxPt.y);

    // Compute the axis bounds. Skip if the event values are null due to dragging on the axis.
    var xMinMax = {},
      yMinMax = {},
      y2MinMax = {},
      startEndGroup = {};
    if (chart.xAxis) {
      xMinMax = DvtChartEventUtils._getAxisMinMax(
        chart.xAxis,
        coords.xMin,
        coords.xMax,
        limitExtent
      );
      startEndGroup = DvtChartEventUtils.getAxisStartEndGroup(
        chart.xAxis,
        xMinMax.min,
        xMinMax.max
      );
    }
    if (chart.yAxis)
      yMinMax = DvtChartEventUtils._getAxisMinMax(
        chart.yAxis,
        coords.yMin,
        coords.yMax,
        limitExtent
      );
    if (chart.y2Axis)
      y2MinMax = DvtChartEventUtils._getAxisMinMax(
        chart.y2Axis,
        coords.yMin,
        coords.yMax,
        limitExtent
      );

    return {
      xMin: xMinMax.min,
      xMax: xMinMax.max,
      unchanged: xMinMax.unchanged,
      yMin: yMinMax.min,
      yMax: yMinMax.max,
      y2Min: y2MinMax.min,
      y2Max: y2MinMax.max,
      startGroup: startEndGroup.startGroup,
      endGroup: startEndGroup.endGroup
    };
  },

  /**
   * Gets the axis min/max values corresponding to the bounding coords.
   * @param {DvtChartAxis} axis
   * @param {number} minCoord The coord of the minimum value of the axis.
   * @param {number} maxCoord The coord of the maximum value of the axis.
   * @param {boolean} limitExtent Whether the result should be limited to the axis min/max extent.
   * @return {object} An object containing the axis min/max value.
   * @private
   */
  _getAxisMinMax: (axis, minCoord, maxCoord, limitExtent) => {
    if (minCoord == null || maxCoord == null) return { min: null, max: null };

    var min = axis.getUnboundedLinearValAt(minCoord);
    var max = axis.getUnboundedLinearValAt(maxCoord);

    if (limitExtent) {
      // Limit to min extent
      var minExtent = axis.getInfo().getMinExtent();
      if (max - min < minExtent) {
        var center = (max + min) / 2;
        max = center + minExtent / 2;
        min = center - minExtent / 2;
      }
      return DvtChartEventUtils._limitToGlobal(axis, min, max);
    }
    return DvtChartEventUtils.getActualMinMax(axis, min, max);
  },

  /**
   * Gets the chart axis bounds corresponding to the deltas in coords. The results are bounded by axis global min/max and
   * minimum axis extent.
   * @param {Chart} chart The chart.
   * @param {number} xMinDelta The delta coord of the left end of the horizontal axis.
   * @param {number} xMaxDelta The delta coord of the right end of the horizontal axis.
   * @param {number} yMinDelta The delta coord of the top end of the vertical axis.
   * @param {number} yMaxDelta The delta coord of the bottom end of the vertical axis.
   * @return {object} An object containing xMin, xMax, yMin, yMax, startGroup, endGroup corresponding to the bounds.
   */
  getAxisBoundsByDelta: (chart, xMinDelta, xMaxDelta, yMinDelta, yMaxDelta) => {
    // Convert the deltas to the axis coordinates
    var deltas = DvtChartEventUtils._convertToAxisCoord(
      chart,
      xMinDelta,
      xMaxDelta,
      yMinDelta,
      yMaxDelta
    );
    var zoomDirection = DvtChartBehaviorUtils.getZoomDir(chart);

    // Compute the axis bounds.
    var xMinMax = {},
      yMinMax = {},
      y2MinMax = {},
      startEndGroup = {};
    if (chart.xAxis && zoomDirection != 'y') {
      xMinMax = DvtChartEventUtils._getAxisMinMaxByDelta(chart.xAxis, deltas.xMin, deltas.xMax);
      startEndGroup = DvtChartEventUtils.getAxisStartEndGroup(
        chart.xAxis,
        xMinMax.min,
        xMinMax.max
      );
    }
    if (chart.yAxis && zoomDirection != 'x')
      yMinMax = DvtChartEventUtils._getAxisMinMaxByDelta(chart.yAxis, deltas.yMin, deltas.yMax);
    if (chart.y2Axis)
      y2MinMax = DvtChartEventUtils._getAxisMinMaxByDelta(chart.y2Axis, deltas.yMin, deltas.yMax);

    return {
      xMin: xMinMax.min,
      xMax: xMinMax.max,
      unchanged: xMinMax.unchanged,
      yMin: yMinMax.min,
      yMax: yMinMax.max,
      y2Min: y2MinMax.min,
      y2Max: y2MinMax.max,
      startGroup: startEndGroup.startGroup,
      endGroup: startEndGroup.endGroup
    };
  },

  /**
   * Gets the axis min/max values corresponding to the delta coords. The results are bounded by axis global min/max and
   * minimum axis extent.
   * @param {DvtChartAxis} axis
   * @param {number} minDelta The delta coord of the minimum value of the axis.
   * @param {number} maxDelta The delta coord of the maximum value of the axis.
   * @return {object} An object containing the axis min/max value.
   * @private
   */
  _getAxisMinMaxByDelta: (axis, minDelta, maxDelta) => {
    var min = axis.getLinearViewportMin();
    var max = axis.getLinearViewportMax();

    // Don't do the computation if the min/max won't change. This is to prevent rounding errors.
    if (maxDelta == minDelta && axis.isFullViewport())
      return DvtChartEventUtils.getActualMinMax(axis, min, max);

    var minDeltaVal = axis.getUnboundedLinearValAt(minDelta) - axis.getUnboundedLinearValAt(0);
    var maxDeltaVal = axis.getUnboundedLinearValAt(maxDelta) - axis.getUnboundedLinearValAt(0);

    // Make sure that the min/max is not less than the minimum axis extent
    var weight = 1;
    var newExtent = max + maxDeltaVal - (min + minDeltaVal);
    var minExtent = axis.getInfo().getMinExtent();
    if (minDelta != maxDelta && newExtent < minExtent)
      weight = (max - min - minExtent) / (minDeltaVal - maxDeltaVal);

    min += minDeltaVal * weight;
    max += maxDeltaVal * weight;

    return DvtChartEventUtils._limitToGlobal(axis, min, max);
  },

  /**
   * Convert from real coord to axis coord.
   * @param {Chart} chart
   * @param {number} xMin The minimum x in real coord.
   * @param {number} xMax The maximum x in real coord.
   * @param {number} yMin The minimum y in real coord.
   * @param {number} yMax The maximum y in real coord.
   * @return {object} An object containing the axis xMin/Max and yMin/Max.
   * @private
   */
  _convertToAxisCoord: (chart, xMin, xMax, yMin, yMax) => {
    var axisCoord = {};
    var isRTL = Agent.isRightToLeft(chart.getCtx());
    if (DvtChartTypeUtils.isHorizontal(chart)) {
      axisCoord.xMin = yMin;
      axisCoord.xMax = yMax;
      axisCoord.yMin = isRTL ? xMax : xMin;
      axisCoord.yMax = isRTL ? xMin : xMax;
    } else {
      axisCoord.xMin = isRTL ? xMax : xMin;
      axisCoord.xMax = isRTL ? xMin : xMax;
      axisCoord.yMin = yMax;
      axisCoord.yMax = yMin;
    }
    return axisCoord;
  },

  /**
   * Limits the min/max values to the global extents of the axis.
   * @param {DvtChartAxis} axis
   * @param {number} min Linearized min value of the axis.
   * @param {number} max Linearzied max value of the axis.
   * @return {object} An object containing the actual axis min/max value after limiting.
   * @private
   */
  _limitToGlobal: (axis, min, max) => {
    var globalMin = axis.getLinearGlobalMin();
    var globalMax = axis.getLinearGlobalMax();

    // Limit to global min/max
    if (max - min >= globalMax - globalMin) {
      min = globalMin;
      max = globalMax;
    } else if (min < globalMin) {
      max += globalMin - min;
      min = globalMin;
    } else if (max > globalMax) {
      min -= max - globalMax;
      max = globalMax;
    }

    return DvtChartEventUtils.getActualMinMax(axis, min, max);
  },

  /**
   * Returns an object containing the actual min/max based on the linearized min/max.
   * @param {DvtChartAxis} axis
   * @param {number} min Linearized min value of the axis.
   * @param {number} max Linearzied max value of the axis.
   * @return {object} An object containing the actual axis min/max value.
   */
  getActualMinMax: (axis, min, max) => {
    var actualMinMax = { min: axis.linearToActual(min), max: axis.linearToActual(max) };

    // For group axis, skip if the bounds don't change so we don't render the sparse axis labels unnecessarily ()
    if (axis.isGroupAxis()) {
      var currentMin = axis.getLinearViewportMin();
      var currentMax = axis.getLinearViewportMax();

      // Instead of strict equality, we need to add some tolerance to handle rounding errors
      var tolerance = 0.0001;
      if (Math.abs(min - currentMin) < tolerance && Math.abs(max - currentMax) < tolerance)
        actualMinMax.unchanged = true;
    }

    return actualMinMax;
  },

  /**
   * Returns the start/endGroup of the axis.
   * @param {DvtChartAxis} axis
   * @param {number} min The minimum value of the axis.
   * @param {number} max The maximum value of the axis.
   * @return {object} An object containing the axis start/endGroup.
   */
  getAxisStartEndGroup: (axis, min, max) => {
    if (axis.isGroupAxis() && min != null && max != null) {
      var startIdx = Math.ceil(min);
      var endIdx = Math.floor(max);
      if (endIdx >= startIdx) {
        var startGroup = axis.getInfo().getGroup(startIdx);
        var endGroup = axis.getInfo().getGroup(endIdx);
        return { startGroup: startGroup, endGroup: endGroup };
      }
    }
    return { startGroup: null, endGroup: null };
  },

  /**
   * Sets initial selection for the graph.
   * @param {Chart} chart The chart being rendered.
   * @param {array} selection The array of initially selected objects.
   */
  setInitialSelection: (chart, selection) => {
    var handler = chart.getSelectionHandler();
    if (!handler) return;

    if (!selection || selection.length == 0) {
      handler.clearSelection();
      return;
    }

    // Construct a keySet of selection ids
    var selectionIds = [];
    for (var i = 0; i < selection.length; i++) {
      if (selection[i]['id'] != null) selectionIds.push(selection[i]['id']);
      else if (selection[i]['series'] != null && selection[i]['group'] != null)
        selectionIds.push(
          DvtChartDataItemUtils.createDataItemId(selection[i]['series'], selection[i]['group'])
        );
    }
    var ctx = chart.getCtx();
    var selectionSet = new ctx.KeySetImpl(selectionIds);

    // Now go through the peers and add the peers that can be found inside the selectionSet to the selectedIds array.
    var peers = chart.getChartObjPeers();
    var selectedIds = [];
    for (var j = 0; j < peers.length; j++) {
      var peer = peers[j];
      if (!peer.isSelectable()) continue;

      // We have to check both id and series+group combination because the user can specify selection using either
      var peerDataId = peer.getDataItemId();
      var peerDataItemId = DvtChartDataItemUtils.createDataItemId(
        peer.getSeries(),
        peer.getGroup()
      );
      if (selectionSet.has(peerDataId) || selectionSet.has(peerDataItemId)) {
        selectedIds.push(peer.getId());
      }
    }

    handler.processInitialSelections(selectedIds, peers);
  },

  /**
   * Returns the keyboard navigable objects for the chart.
   * @param {Chart} chart
   * @return {array}
   */
  getKeyboardNavigables: (chart) => {
    var navigables = [];
    // only process pie chart with valid/non empty data
    if (DvtChartTypeUtils.isPie(chart) && chart.pieChart) {
      var slices = chart.pieChart.__getSlices();
      for (var i = 0; i < slices.length; i++) {
        // exclude hidden slices that may be included during delete animation
        if (DvtChartDataUtils.isSeriesRendered(chart, slices[i].getSeriesIndex())) {
          navigables.push(slices[i]);
        }
      }
    } else {
      var peers = chart.getChartObjPeers();
      for (var z = 0; z < peers.length; z++) {
        if (peers[z].isNavigable()) {
          navigables.push(peers[z]);
        }
      }
    }
    return navigables;
  },

  /**
   * Add data, itemData, seriesData, and groupData to the event payload.
   * @param {Chart} chart
   * @param {object} eventPayload The event payload to decorate. Currently contains series, group, and id.
   */
  addEventData: (chart, eventPayload) => {
    var seriesIndex = DvtChartDataUtils.getSeriesIdx(chart, eventPayload['series']);
    var groupIndex = DvtChartDataUtils.getGroupIdx(chart, eventPayload['group']);
    var itemIndex = DvtChartDataUtils.getNestedDataItemIdx(
      chart,
      seriesIndex,
      groupIndex,
      eventPayload['id']
    );
    var dataContext = DvtChartStyleUtils.getDataContext(chart, seriesIndex, groupIndex, itemIndex);

    if (dataContext) {
      eventPayload['data'] = dataContext['data'];
      eventPayload['itemData'] = dataContext['itemData']; // data provider row data
      eventPayload['seriesData'] = dataContext['seriesData'];
      if (dataContext['groupData']) eventPayload['groupData'] = dataContext['groupData'];
      // getDataContext() doesn't support group objects
      else if (groupIndex != null)
        eventPayload['groupData'] = DvtChartDataUtils.getGroupsDataForContext(chart)[groupIndex];
    }
  },

  /**
   * Adds group drill details in the eventPayload
   * @param {Chart} chart
   * @param {object} eventPayload The event payload to decorate. Currently contains series, group, and id.
   */
  addGroupDrillEventData: (chart, eventPayload) => {
    var newEvent = {};
    Object.assign(newEvent, eventPayload);

    var axis = chart.xAxis.getInfo();
    var range = axis.getItemsRange(eventPayload.group);
    delete newEvent.series;

    var startIndex = range.startIndex;
    var endIndex = range.endIndex;
    var itemsList = [];
    for (var i = startIndex; i < endIndex + 1; i++) {
      var seriesCount = DvtChartDataUtils.getSeriesCount(chart);
      for (var j = 0; j < seriesCount; j++) {
        itemsList.push(DvtChartDataUtils.getRawDataItem(chart, j, i));
      }
    }

    newEvent.items = itemsList;
    newEvent.groupData = range.groupData;

    return newEvent;
  },

  /**
   * Adds multiseries drill details in the eventPayload
   * @param {Chart} chart
   * @param {object} eventPayload The event payload to decorate. Currently contains series, group, and id.
   */
  addMultiSeriesDrillEventData: (chart, eventPayload) => {
    var newEvent = {};
    Object.assign(newEvent, eventPayload);

    var seriesItems = DvtChartPieUtils._getSeriesIndicesArrays(chart).other;
    delete newEvent.group;

    var seriesData = [];
    var items = [];
    var series = [];
    for (var i = 0; i < seriesItems.length; i++) {
      var seriesContext = DvtChartDataUtils.getSeriesDataForContext(chart, seriesItems[i]);
      var item = DvtChartDataUtils.getRawDataItem(chart, seriesItems[i], 0);

      seriesData.push(seriesContext);
      items.push(item);
      series.push(DvtChartDataUtils.getSeries(chart, seriesItems[i]));
    }
    newEvent.seriesData = seriesData;
    newEvent.items = items;
    newEvent.series = series;

    return newEvent;
  },

  /**
   * Adds series drill details in the eventPayload
   * @param {Chart} chart
   * @param {object} eventPayload The event payload to decorate. Currently contains series, group, and id.
   */
  addSeriesDrillEventData: (chart, eventPayload) => {
    var newEvent = {};
    Object.assign(newEvent, eventPayload);

    var seriesIndex = DvtChartDataUtils.getSeriesIdx(chart, eventPayload.id);
    var numGroups = DvtChartDataUtils.getGroupCount(chart);
    newEvent.series = eventPayload.id;
    delete newEvent.group;

    newEvent.seriesData = DvtChartDataUtils.getSeriesDataForContext(chart, seriesIndex);

    var items = [];
    for (var i = 0; i < numGroups; i++) {
      var item = DvtChartDataUtils.getRawDataItem(chart, seriesIndex, i);
      items.push(item);
    }
    newEvent.items = items;
    return newEvent;
  },

  /**
   * Adds item drill details in the eventPayload
   * @param {Chart} chart
   * @param {object} eventPayload The event payload to decorate. Currently contains series, group, and id.
   */
  addItemDrillEventData: (chart, eventPayload) => {
    var newEvent = {};
    Object.assign(newEvent, eventPayload);

    var seriesIndex = DvtChartDataUtils.getSeriesIdx(chart, eventPayload.series);
    var groupIndex = DvtChartDataUtils.getGroupIdx(chart, eventPayload.group);
    var itemIndex = DvtChartDataUtils.getNestedDataItemIdx(
      chart,
      seriesIndex,
      groupIndex,
      eventPayload.id
    );

    newEvent.seriesData = DvtChartDataUtils.getSeriesDataForContext(chart, seriesIndex);
    newEvent.groupData = DvtChartDataUtils.getGroupsDataForContext(chart)[groupIndex];
    newEvent.data = DvtChartDataUtils.getRawDataItem(chart, seriesIndex, groupIndex, itemIndex);
    newEvent.itemData = DvtChartDataUtils.getDataItem(chart, seriesIndex, groupIndex, itemIndex)[
      '_itemData'
    ];

    return newEvent;
  },

  /**
   * Create a background rect that allows DND theming
   * @param {Chart} chart The chart being rendered.
   * @param {dvt.Container} container The container to render into.
   * @param {dvt.Rectangle} availSpace The available space.
   * @param {boolean} bHorizontal Is chart horizontal
   */
  addPlotAreaDnDBackground: (chart, container, availSpace, bHorizontal) => {
    var isPlotAreaDraggable = DvtChartBehaviorUtils.isPlotAreaDraggable(chart);
    var isPlotAreaDropTarget = DvtChartBehaviorUtils.isPlotAreaDropTarget(chart);

    if (isPlotAreaDropTarget || isPlotAreaDraggable) {
      var background = new Rect(
        chart.getCtx(),
        0,
        0,
        bHorizontal ? availSpace.w : availSpace.h,
        bHorizontal ? availSpace.h : availSpace.w
      );
      background.setInvisibleFill();
      container.addChild(background);
      chart.getCache().putToCache('plotAreaBackground', background);

      if (isPlotAreaDraggable) background.setClassName('oj-draggable');
    }
  }
};

/**
 * Renderer for funnel chart.
 * @class
 */
const DvtChartFunnelRenderer = {
  /** @private @const */
  _DEFAULT_3D_GAP_RATIO: 1 / 36,
  /** @private @const */
  _DEFAULT_2D_GAP_RATIO: 1 / 70,
  /** @private @const */
  _MAX_WIDTH_FOR_GAPS: 0.25,
  /** @private @const */
  _GROUP_IDX: 0,
  /** @private @const */
  _SLICE_VALUE_THRESHOLD: 0.0001,

  /**
   * Renders the funnel into the available space.
   * @param {Chart} chart The chart being rendered.
   * @param {dvt.Container} container The container to render to.
   * @param {dvt.Rectangle} availSpace The available space.
   */
  render: (chart, container, availSpace) => {
    // Creating a container for the funnel so that it can be rotated if vertical, also for animation.
    var funnelContainer = new Container(chart.getCtx());
    funnelContainer.setTranslate(availSpace.x, availSpace.y);
    container.addChild(funnelContainer);
    chart.setPlotArea(funnelContainer);
    var isHorizontal = DvtChartTypeUtils.isHorizontal(chart);

    var bbox;

    if (isHorizontal) bbox = new Rectangle(0, 0, availSpace.w, availSpace.h);
    else {
      //rotate the container and the bounding rect
      var rotationMatrix = new Matrix();
      var dirFactor = Agent.isRightToLeft(chart.getCtx()) ? -1 : 1;
      rotationMatrix = rotationMatrix.translate(-availSpace.h / 2, -availSpace.w / 2);
      rotationMatrix = rotationMatrix.rotate((dirFactor * Math.PI) / 2);
      rotationMatrix = rotationMatrix.translate(
        availSpace.x + availSpace.w / 2,
        availSpace.y + availSpace.h / 2
      );
      bbox = new Rectangle(0, 0, availSpace.h, availSpace.w);
      funnelContainer.setMatrix(rotationMatrix);
    }

    DvtChartEventUtils.addPlotAreaDnDBackground(chart, funnelContainer, availSpace, isHorizontal);

    if (!DvtChartFunnelRenderer._renderFunnelSlices(chart, funnelContainer, bbox))
      DvtChartTextUtils.renderEmptyText(chart, container, availSpace);

    // Initial Selection
    var selected = DvtChartDataUtils.getInitialSelection(chart);
    DvtChartEventUtils.setInitialSelection(chart, selected);

    // Initial Highlighting
    chart.highlight(DvtChartDataUtils.getHighlightedCategories(chart));
  },

  /**
   * Renders all funnel slices for the given chart.
   * @param {Chart} chart
   * @param {dvt.Container} container The container to render to.
   * @param {dvt.Rectangle} availSpace
   * @return {boolean} true if funnel slices have been rendered, false otherwise
   * @private
   */
  _renderFunnelSlices: (chart, container, availSpace) => {
    var options = chart.getOptions();
    var seriesCount = DvtChartDataUtils.getSeriesCount(chart);

    // Compute the gap size
    var gapRatio = DvtChartStyleUtils.getDataItemGaps(chart);
    var defaultGapSize =
      (options['styleDefaults']['threeDEffect'] == 'on'
        ? DvtChartFunnelRenderer._DEFAULT_3D_GAP_RATIO
        : DvtChartFunnelRenderer._DEFAULT_2D_GAP_RATIO) * availSpace.w;
    var maxGapSize = Math.min(
      (DvtChartFunnelRenderer._MAX_WIDTH_FOR_GAPS * availSpace.w) / (seriesCount - 1),
      defaultGapSize
    );
    var gapSize = gapRatio * maxGapSize;

    var totalValue = 0; // the total value represented by the funnel
    var numDrawnSeries = 0; // to keep track of how many series are drawn, so we don't add too many gaps if there are zero values
    var cumulativeValue = 0; // keeping track of the total up to this series

    // Iterate through the data to calculate the total value
    for (var seriesIndex = 0; seriesIndex < seriesCount; seriesIndex++) {
      // Skip the series if it shouldn't be rendered
      if (!DvtChartDataUtils.isDataItemRendered(chart, seriesIndex)) continue;

      // Do not render if the value is not positive
      var value = DvtChartDataUtils.getTargetVal(chart, seriesIndex);
      if (value == null)
        value = DvtChartDataUtils.getVal(chart, seriesIndex, DvtChartFunnelRenderer._GROUP_IDX);
      if (value <= 0) continue;
      totalValue += value;
    }

    if (totalValue == 0) return false;

    // Iterate through the data
    for (var seriesIdx = seriesCount - 1; seriesIdx >= 0; seriesIdx--) {
      // Skip the series if it shouldn't be rendered
      if (!DvtChartDataUtils.isDataItemRendered(chart, seriesIdx)) continue;

      // Do not render if the value is not positive
      var val = DvtChartDataUtils.getVal(chart, seriesIdx, DvtChartFunnelRenderer._GROUP_IDX);
      var targetValue = DvtChartDataUtils.getTargetVal(chart, seriesIdx);
      if ((val <= 0 && targetValue == null) || (targetValue != null && targetValue <= 0)) continue;

      //  - rendering issues for funnel/pie/sunburst charts
      var sliceValue = targetValue != null ? targetValue : val;
      if (sliceValue < DvtChartFunnelRenderer._SLICE_VALUE_THRESHOLD * totalValue) continue;

      var slice;

      if (targetValue != null) {
        cumulativeValue += targetValue / totalValue;
        slice = new DvtChartFunnelSlice(
          chart,
          seriesIdx,
          numDrawnSeries,
          availSpace.w,
          availSpace.h,
          1 - cumulativeValue,
          targetValue / totalValue,
          val / targetValue,
          gapSize
        );
      } else {
        cumulativeValue += val / totalValue;
        slice = new DvtChartFunnelSlice(
          chart,
          seriesIdx,
          numDrawnSeries,
          availSpace.w,
          availSpace.h,
          1 - cumulativeValue,
          val / totalValue,
          null,
          gapSize
        );
      }

      numDrawnSeries++; // keeping track of how many series have been drawn to create the gap.
      container.addChild(slice);
      DvtChartObjPeer.associate(slice, chart, seriesIdx, DvtChartFunnelRenderer._GROUP_IDX);
    }
    return true;
  }
};

/**
 *  Creates a pyramid shape.
 *  @extends {dvt.Path}
 *  @class DvtChartPyramidSlice  Creates a pyramid slice object.
 *  @constructor
 *  @param {Chart} chart  The chart being rendered.
 *  @param {number} seriesIndex  The index of this slice.
 *  @param {number} numDrawnSeries  The number of series already drawn. Should be total number of series - seriesIndex - 1 if none are skipped.
 *  @param {number} pyramidWidth The available width for the whole pyramid.
 *  @param {number} pyramidHeight The available height for the whole pyramid.
 *  @param {number} startPercent The cumulative value of all the slices that come before. The start/leftmost value of the slice.
 *  @param {number} valuePercent The percent value for the slice. Dictates the width.
 *  @param {number} gap The gap distance between slices.
 */
class DvtChartPyramidSlice extends Path {
  /**
   *  Object initializer.
   *  @param {Chart} chart  The chart being rendered.
   *  @param {number} seriesIndex  The index of this slice.
   *  @param {number} numDrawnSeries  The number of series already drawn. Should be total number of series - seriesIndex - 1 if none are skipped.
   *  @param {number} pyramidWidth The available width for the whole pyramid.
   *  @param {number} pyramidHeight The available height for the whole pyramid.
   *  @param {number} startPercent The cumulative value of all the slices that come before. The start/leftmost value of the slice.
   *  @param {number} valuePercent The percent value for the slice. Dictates the width.
   *  @param {number} gap The gap distance between slices.
   *  @protected
   */
  constructor(
    chart,
    seriesIndex,
    numDrawnSeries,
    pyramidWidth,
    pyramidHeight,
    startPercent,
    valuePercent,
    gap
  ) {
    super(chart.getCtx());

    /** Minimum number of characters to use when truncating.
     * @private
     * */
    this._MIN_CHARS_DATA_LABEL = 3;

    /** Horizontal padding for the slice label to prevent the text from bleeding
     * @private
     * */
    this._SLICE_LABEL_HORIZONTAL_PADDING = 4;

    /** Factor by which the total width is used for rendering the 3D side
     * @private */
    this._3D_WIDTH_FACTOR = 0.2;

    this._chart = chart;
    var styleDefaults = chart.getOptions()['styleDefaults'];
    this._seriesIndex = seriesIndex;
    this._numDrawnSeries = numDrawnSeries;
    this._pyramidWidth = pyramidWidth;
    this._pyramidHeight = pyramidHeight;
    this._startPercent = startPercent;
    this._valuePercent = valuePercent;
    this._3dRatio = styleDefaults['threeDEffect'] == 'on' ? 1 : 0;
    this._gap = gap;
    var cmds = this._getPath();
    this._dataColor = DvtChartStyleUtils.getColor(this._chart, this._seriesIndex, 0);

    if (this._3dRatio > 0) {
      this.setCmds(cmds['threeDPathTop']);
      this._threeDPathSide = new Path(this.getCtx(), cmds['threeDPathSide']);
      this._mainFace = new Path(this.getCtx(), cmds['slice']);
      this.addChild(this._threeDPathSide);
      this.addChild(this._mainFace);
    } else this.setCmds(cmds['slice']);

    this._setColorProps(cmds['sliceBounds']);
    this._label = this._getSliceLabel(cmds['sliceBounds']);

    if (this._label != null) {
      this._label.setMouseEnabled(false);
      this.addChild(this._label);
    }
  }

  /**
   * Creates the path commands that represent this slice
   * @return {object} The commands for drawing this slice. An object containing the sliceCommands and sliceBounds
   * @private
   */
  _getPath() {
    var isBiDi = Agent.isRightToLeft(this.getCtx());
    var seriesCount = DvtChartDataUtils.getSeriesCount(this._chart);
    var offset = (seriesCount - this._numDrawnSeries - 1) * this._gap;
    var center3DOffset = this._3dRatio * this._3D_WIDTH_FACTOR * 0.98 * this._pyramidWidth;
    var chartOptions = this._chart.getOptions();

    var heightToBottomEdge = Math.sqrt(
      Math.pow(this._pyramidHeight - (seriesCount - 1) * this._gap, 2) * (1 - this._startPercent)
    ); // height to bottom edge
    var heightToTopEdge =
      this._startPercent + this._valuePercent >= 1
        ? 0
        : Math.sqrt(
            Math.pow(this._pyramidHeight - (seriesCount - 1) * this._gap, 2) *
              (1 - this._startPercent - this._valuePercent)
          ); //height to top edge

    var topEdgeWidth =
      (1 - this._3dRatio * this._3D_WIDTH_FACTOR) *
      0.98 *
      this._pyramidWidth *
      (heightToTopEdge / this._pyramidHeight);
    var gapAdjustedHeightToBottom =
      this._numDrawnSeries == 0 ? heightToBottomEdge : heightToBottomEdge - this._gap;
    var bottomEdgeWidth =
      (1 - this._3dRatio * this._3D_WIDTH_FACTOR) *
      0.98 *
      this._pyramidWidth *
      (gapAdjustedHeightToBottom / this._pyramidHeight);

    var topLeftCoords, topRightCoords, bottomLeftCoords, bottomRightCoords; // naming is based on left to right case

    if (isBiDi) {
      topRightCoords = [
        this._pyramidWidth - 0.5 * this._pyramidWidth + center3DOffset / 2 - topEdgeWidth / 2,
        offset + heightToTopEdge
      ];
      bottomRightCoords = [
        this._pyramidWidth - 0.5 * this._pyramidWidth + center3DOffset / 2 - bottomEdgeWidth / 2,
        offset + heightToBottomEdge
      ];
      bottomLeftCoords = [
        this._pyramidWidth - 0.5 * this._pyramidWidth + center3DOffset / 2 + bottomEdgeWidth / 2,
        offset + heightToBottomEdge
      ];
      topLeftCoords = [
        this._pyramidWidth - 0.5 * this._pyramidWidth + center3DOffset / 2 + topEdgeWidth / 2,
        offset + heightToTopEdge
      ];
    } else {
      topLeftCoords = [
        0.5 * this._pyramidWidth - center3DOffset / 2 - topEdgeWidth / 2,
        offset + heightToTopEdge
      ];
      bottomLeftCoords = [
        0.5 * this._pyramidWidth - center3DOffset / 2 - bottomEdgeWidth / 2,
        offset + heightToBottomEdge
      ];
      bottomRightCoords = [
        0.5 * this._pyramidWidth - center3DOffset / 2 + bottomEdgeWidth / 2,
        offset + heightToBottomEdge
      ];
      topRightCoords = [
        0.5 * this._pyramidWidth - center3DOffset / 2 + topEdgeWidth / 2,
        offset + heightToTopEdge
      ];
    }

    var threeDPathTop, threeDPathSide;
    if (chartOptions['styleDefaults']['threeDEffect'] == 'on') {
      // 3D angle is 45 degrees
      // Our pyramids are not correctly drawn to perspective.  We need to fudge the position of the invisble vertex to avoid it
      // becoming visible for wide pyramids.  Proposed hack is to allow the angle of the invisible edge to be no larger than
      // 90% of the bottom angles of the front face
      var baseAngle = Math.atan((2 * heightToBottomEdge) / bottomEdgeWidth);
      var invisibleAngle = Math.min(baseAngle * 0.9, Math.PI / 4);
      var bottomEdge3DOffset =
        (this._3dRatio * this._3D_WIDTH_FACTOR * bottomEdgeWidth * Math.sqrt(2)) / 2;
      var topEdge3DOffset =
        (this._3dRatio * this._3D_WIDTH_FACTOR * topEdgeWidth * Math.sqrt(2)) / 2;
      var topEdge3DHiddenXOffset = topEdge3DOffset / Math.tan(invisibleAngle);

      // 3D Top
      threeDPathTop = PathUtils.moveTo(topRightCoords[0], topRightCoords[1]);
      threeDPathTop += PathUtils.lineTo(
        topRightCoords[0] + (isBiDi ? -1 : 1) * Math.max(topEdge3DOffset, 1),
        topRightCoords[1] - Math.max(topEdge3DOffset, 0.5)
      );
      threeDPathTop += PathUtils.lineTo(
        topLeftCoords[0] + (isBiDi ? -1 : 1) * Math.max(topEdge3DHiddenXOffset, 1),
        topLeftCoords[1] - Math.max(topEdge3DOffset, 0.5)
      );
      threeDPathTop += PathUtils.lineTo(topLeftCoords[0], topLeftCoords[1]);
      threeDPathTop += PathUtils.closePath();

      // 3D Side
      threeDPathSide = PathUtils.moveTo(topRightCoords[0], topRightCoords[1]);
      threeDPathSide += PathUtils.lineTo(
        topRightCoords[0] + (isBiDi ? -1 : 1) * Math.max(topEdge3DOffset, 1),
        topRightCoords[1] - Math.max(topEdge3DOffset, 0.5)
      );
      threeDPathSide += PathUtils.lineTo(
        bottomRightCoords[0] + (isBiDi ? -1 : 1) * bottomEdge3DOffset,
        bottomRightCoords[1] - bottomEdge3DOffset
      );
      threeDPathSide += PathUtils.lineTo(bottomRightCoords[0], bottomRightCoords[1]);
      threeDPathSide += PathUtils.closePath();
    }

    //2D Main face path commands
    var pathCommands = PathUtils.moveTo(topRightCoords[0], topRightCoords[1]);
    pathCommands += PathUtils.lineTo(bottomRightCoords[0], bottomRightCoords[1]);
    pathCommands += PathUtils.lineTo(bottomLeftCoords[0], bottomLeftCoords[1]);
    pathCommands += PathUtils.lineTo(topLeftCoords[0], topLeftCoords[1]);
    pathCommands += PathUtils.closePath();

    var topBottomEdgeRatio = topEdgeWidth / bottomEdgeWidth;
    // alternative calculations for top slice
    var sliceHeight =
      0.5 * Math.abs(topLeftCoords[1] - bottomLeftCoords[1]) * (1 + topBottomEdgeRatio);
    var sliceWidth =
      0.5 * Math.abs(bottomRightCoords[0] - bottomLeftCoords[0]) * (1 + topBottomEdgeRatio) -
      this._SLICE_LABEL_HORIZONTAL_PADDING;
    var sliceY = bottomLeftCoords[1] - sliceHeight;
    var sliceX = bottomLeftCoords[0] + (isBiDi ? -1 : 1) * ((bottomEdgeWidth - sliceWidth) / 2);
    var sliceBounds = new Rectangle(sliceX, sliceY, sliceWidth, sliceHeight);

    return { slice: pathCommands, sliceBounds, threeDPathTop, threeDPathSide };
  }

  /**
   * Creates a single slice label dvt.Text object associated with this slice.
   * @param {dvt.Rectangle} sliceBounds The space occupied by the slice this is associated with.
   * @return {dvt.OutputText} slice label for this slice
   * @private
   */
  _getSliceLabel(sliceBounds) {
    // Get and create the label string
    var labelString = DvtChartStyleUtils.getDataLabel(this._chart, this._seriesIndex, 0);
    if (!labelString)
      // if no data label set on the data item, set it from the series
      labelString = DvtChartDataUtils.getSeriesLabel(this._chart, this._seriesIndex);

    // Return if no label or label position none
    if (
      !labelString ||
      DvtChartGroupUtils.getDataLabelPos(this._chart, this._seriesIndex, 0) == 'none'
    )
      return;

    var label = new MultilineText(this.getCtx(), labelString, 0, 0);

    // Have to move the style setting first because was using wrong font size to come up with truncated text
    var isPatternBg = DvtChartStyleUtils.getPattern(this._chart, this._seriesIndex, 0) != null;
    var styleDefaults = this._chart.getOptions().styleDefaults;
    var labelStyleArray = [
      styleDefaults._dataLabelStyle,
      styleDefaults.dataLabelStyle,
      new CSSStyle(
        DvtChartDataUtils.getDataItem(this._chart, this._seriesIndex, 0)['labelStyle']
      )
    ];
    var style = CSSStyle.mergeStyles(labelStyleArray);
    label.setCSSStyle(style);

    // Truncating text and dropping if doesn't fit.
    if (
      !TextUtils.fitText(label, sliceBounds.w, sliceBounds.h, this, this._MIN_CHARS_DATA_LABEL)
    )
      return;

    var textDim = label.getDimensions();
    var pos = this._getLabelPos(sliceBounds);
    // Checking if the text starts within the bounding box.
    if (isPatternBg) {
      var padding = textDim.h * 0.15;
      var cmd = PathUtils.roundedRectangle(
        textDim.x - padding,
        textDim.y,
        textDim.w + 2 * padding,
        textDim.h,
        2,
        2,
        2,
        2
      );
      var bbox = new Path(this.getCtx(), cmd);
      bbox.setSolidFill(DvtChartStyleUtils.SERIES_PATTERN_BG_COLOR, 0.9);
      pos = pos.translate(-0.5 * textDim.w, -0.5 * textDim.h);
      bbox.setMatrix(pos);
      this.addChild(bbox);
    }
    var labelColor = isPatternBg
      ? ColorUtils.getContrastingTextColor(DvtChartStyleUtils.SERIES_PATTERN_BG_COLOR)
      : sliceBounds.containsPoint(sliceBounds.x + (sliceBounds.w - textDim.w) / 2, sliceBounds.y)
      ? ColorUtils.getContrastingTextColor(this._dataColor)
      : ColorUtils.getContrastingTextColor(null);
    // Don't want to override the color if it was set above, unless in high contrast mode.
    var labelColorStyle = new CSSStyle({ color: labelColor });
    labelStyleArray.splice(1, 0, labelColorStyle);
    if (Agent.isHighContrast()) {
      labelStyleArray.push(labelColorStyle);
    }
    style = CSSStyle.mergeStyles(labelStyleArray);
    label.setCSSStyle(style);
    label.setMatrix(this._getLabelPos(sliceBounds));
    label.alignCenter();
    label.alignMiddle();
    return label;
  }

  /**
   * Calculates the position of the text within this slice. Comes up with the translation matrix.
   * @param {dvt.Rectangle} sliceBounds The space occupied by the slice.
   * @return {dvt.Matrix} The matrix representing the transformation for placing the text.
   * @private
   */
  _getLabelPos(sliceBounds) {
    // Rotate the text
    var matrix = new Matrix();
    if (Agent.isRightToLeft(this.getCtx())) {
      matrix = matrix.translate(
        sliceBounds.x - sliceBounds.w / 2,
        sliceBounds.y + sliceBounds.h / 2
      );
    } else {
      matrix = matrix.translate(
        sliceBounds.x + sliceBounds.w / 2,
        sliceBounds.y + sliceBounds.h / 2
      );
    }
    return matrix;
  }

  /**
   * Passing on the colors for the pyramid slice object. Sets the slice fill and border color, as well as the selection and hover colors by reading them from the chart.
   * @param {dvt.Rectangle} sliceBounds The space occupied by the slice. This is used for calculating the gradient effect bounds.
   * @private
   */
  _setColorProps(sliceBounds) {
    var sliceFill = DvtChartSeriesEffectUtils.getFunnelPyramidSliceFill(
      this._chart,
      this._seriesIndex,
      this._dataColor,
      sliceBounds
    );
    var sliceBorder = DvtChartStyleUtils.getBorderColor(this._chart, this._seriesIndex, 0);
    var userBorderWidth = DvtChartStyleUtils.getUserBorderWidth(this._chart, this._seriesIndex, 0);
    var defaultBorderWidth = DvtChartStyleUtils.getDefaultBorderWidth(
      this._chart,
      this._seriesIndex,
      0
    );
    var borderWidth = userBorderWidth !== null ? userBorderWidth : defaultBorderWidth;

    var borderIsFrom3D = false;
    var isRedwood = this.getCtx().getThemeBehavior() === 'redwood';
    if (sliceBorder == null && this._3dRatio > 0) {
      borderIsFrom3D = true;
      sliceBorder = '#FFFFFF';
    }

    if (sliceBorder && !(isRedwood && this._chart.isSelectionSupported() && borderIsFrom3D)) {
      this.setSolidStroke(sliceBorder, null, borderWidth);
      if (this._3dRatio > 0) {
        this._threeDPathSide.setSolidStroke(sliceBorder, null, borderWidth);
        this._mainFace.setSolidStroke(sliceBorder, null, borderWidth);
      }
    }

    var hoverColor = SelectionEffectUtils.getHoverBorderColor(this._dataColor);

    if (this._3dRatio > 0) {
      var isSeriesEffectColor = !sliceFill.getPattern && !sliceFill.getAlphas;
      this._mainFace.setFill(sliceFill);
      this._mainFace.setStyle(DvtChartStyleUtils.getStyle(this._chart, this._seriesIndex, 0));
      this._mainFace.setClassName(
        DvtChartStyleUtils.getClassName(this._chart, this._seriesIndex, 0)
      );
      this._threeDPathSide.setFill(
        isSeriesEffectColor
          ? new SolidFill(ColorUtils.getDarker(sliceFill.getColor(), 0.3))
          : sliceFill
      );
      this._threeDPathSide.setStyle(DvtChartStyleUtils.getStyle(this._chart, this._seriesIndex, 0));
      this._threeDPathSide.setClassName(
        DvtChartStyleUtils.getClassName(this._chart, this._seriesIndex, 0)
      );
      this.setFill(
        isSeriesEffectColor
          ? new SolidFill(ColorUtils.getDarker(sliceFill.getColor(), 0.3))
          : sliceFill
      );
      this.setStyle(DvtChartStyleUtils.getStyle(this._chart, this._seriesIndex, 0));
      this.setClassName(DvtChartStyleUtils.getClassName(this._chart, this._seriesIndex, 0));
    } else {
      this.setFill(sliceFill);
      this.setStyle(DvtChartStyleUtils.getStyle(this._chart, this._seriesIndex, 0));
      this.setClassName(DvtChartStyleUtils.getClassName(this._chart, this._seriesIndex, 0));
    }

    // Save the original border stroke
    this.OriginalStroke = this.getStroke();

    if (this._chart.isSelectionSupported()) {
      this.setCursor(SelectionEffectUtils.getSelectingCursor());

      if (this._3dRatio > 0) {
        this._mainFace.setCursor(SelectionEffectUtils.getSelectingCursor());
        this._threeDPathSide.setCursor(SelectionEffectUtils.getSelectingCursor());
        if (isRedwood) {
          this._mainFace.setFeedbackClassName('oj-dvt-selectable');
          this._threeDPathSide.setFeedbackClassName('oj-dvt-selectable');
          var strokeColor = sliceBorder != null && !borderIsFrom3D ? sliceBorder : this._dataColor;
          var strokeWidth = userBorderWidth !== null ? userBorderWidth : 0;
          this._mainFace.setSolidStroke(strokeColor, null, strokeWidth);
          this._threeDPathSide.setSolidStroke(strokeColor, null, strokeWidth);
          if (sliceBorder) {
            if (borderIsFrom3D) {
              this._mainFace.addClassName('oj-dvt-default-border-color');
              this._threeDPathSide.addClassName('oj-dvt-default-border-color');
            }
            if (userBorderWidth == null) {
              this._mainFace.addClassName('oj-dvt-default-border-width');
              this._threeDPathSide.addClassName('oj-dvt-default-border-width');
            }
          }
        } else {
          var innerColor = DvtChartStyleUtils.getSelectedInnerColor(this._chart);
          var outerColor = DvtChartStyleUtils.getSelectedOuterColor(this._chart)
            ? DvtChartStyleUtils.getSelectedOuterColor(this._chart)
            : this._dataColor;
          this._mainFace.setHoverStroke(new Stroke(hoverColor, 1, 2));
          this._threeDPathSide.setHoverStroke(new Stroke(hoverColor, 1, 2));
          this._mainFace.setSelectedStroke(
            new Stroke(innerColor, 1, 1.5),
            new Stroke(outerColor, 1, 4.5)
          );
          this._mainFace.setSelectedHoverStroke(
            new Stroke(innerColor, 1, 1.5),
            new Stroke(hoverColor, 1, 4.5)
          );
        }
      } else {
        this.setCursor(SelectionEffectUtils.getSelectingCursor());
        if (isRedwood) {
          this.setFeedbackClassName('oj-dvt-selectable');
          var strokeColor = sliceBorder != null && !borderIsFrom3D ? sliceBorder : this._dataColor;
          var strokeWidth = userBorderWidth !== null ? userBorderWidth : 0;
          this.setSolidStroke(strokeColor, null, strokeWidth);
          if (sliceBorder) {
            if (borderIsFrom3D) {
              this.addClassName('oj-dvt-default-border-color');
            }
            if (userBorderWidth == null) {
              this.addClassName('oj-dvt-default-border-width');
            }
          }
        } else {
          var innerColor = DvtChartStyleUtils.getSelectedInnerColor(this._chart);
          var outerColor = DvtChartStyleUtils.getSelectedOuterColor(this._chart)
            ? DvtChartStyleUtils.getSelectedOuterColor(this._chart)
            : this._dataColor;
          this.setHoverStroke(new Stroke(hoverColor, 1, 2));
          this.setSelectedStroke(
            new Stroke(innerColor, 1, 1.5),
            new Stroke(outerColor, 1, 4.5)
          );
          this.setSelectedHoverStroke(
            new Stroke(innerColor, 1, 1.5),
            new Stroke(hoverColor, 1, 4.5)
          );
        }
      }
    }
  }

  /**
   * Gets the percent values associated with the slice for animation
   * @return {array} the start, value and alpha for this slice.
   */
  getAnimParams() {
    return [this._startPercent, this._valuePercent, this.getAlpha(), this._3dRatio];
  }

  /**
   * Sets the percent values associated with the slice for animation
   * @param {array} ar The new start and value for this slice
   */
  setAnimParams(ar) {
    this._startPercent = ar[0];
    this._valuePercent = ar[1];
    this.setAlpha(ar[2]);
    this._3dRatio = ar[3];
    var cmds = this._getPath();

    if (this._threeDPathSide && this._mainFace) {
      this.setCmds(cmds['threeDPathTop']);
      this._threeDPathSide.setCmds(cmds['threeDPathSide']);
      this._mainFace.setCmds(cmds['slice']);
    } else this.setCmds(cmds['slice']);

    if (this._label) this._label.setMatrix(this._getLabelPos(cmds['sliceBounds']));
  }

  /**
   * @override
   */
  setSelected(selected) {
    if (this._3dRatio > 0) {
      this._mainFace.setSelected(selected);
      if (this.getCtx().getThemeBehavior() === 'redwood') {
        this._threeDPathSide.setSelected(selected);
      }
      if (selected) {
        var dims = this.getDimensions();
        var displacementX = 3;
        var displacementY = 5;
        // To make the selection effect more apparent - make the bars slightly smaller
        var w = dims.w;
        var h = dims.h;
        var scaleX = (w - displacementX) / w;
        var scaleY = (h - displacementY) / h;
        this._mainFace.setScaleX(scaleX);
        this._mainFace.setScaleY(scaleY);
        this._mainFace.setTranslateX(Math.ceil(displacementX / 2) + (displacementX / w) * dims.x);
        this._mainFace.setTranslateY(Math.ceil(displacementY / 2) + (displacementY / h) * dims.y);
      } else {
        this._mainFace.setScaleX(1);
        this._mainFace.setScaleY(1);
        this._mainFace.setTranslateX(0);
        this._mainFace.setTranslateY(0);
      }
    } else super.setSelected(selected);
  }

  /**
   * @override
   */
  showHoverEffect() {
    if (this.getCtx().getThemeBehavior() !== 'redwood' || this._3dRatio === 0) {
      super.showHoverEffect();
    }
    if (this._3dRatio > 0) {
      this._threeDPathSide.showHoverEffect();
      this._mainFace.showHoverEffect();
    }
  }

  /**
   * @override
   */
  hideHoverEffect() {
    if (this.getCtx().getThemeBehavior() !== 'redwood' || this._3dRatio === 0) {
      super.hideHoverEffect();
    }
    if (this._3dRatio > 0) {
      this._threeDPathSide.hideHoverEffect();
      this._mainFace.hideHoverEffect();
    }
  }

  /**
   * @override
   */
  copyShape() {
    return new DvtChartPyramidSlice(
      this._chart,
      this._seriesIndex,
      this._numDrawnSeries,
      this._pyramidWidth,
      this._pyramidHeight,
      this._startPercent,
      this._valuePercent,
      this._gap
    );
  }

  /**
   * Gets the fill for the main face shape.
   * @return {dvt.SoldFill}
   */
  getPrimaryFill() {
    return this._mainFace ? this._mainFace.getFill() : this.getFill();
  }
}

/**
 * Renderer for pyramid chart.
 * @class
 */
const DvtChartPyramidRenderer = {
  /** @private @const */
  _DEFAULT_GAP_RATIO: 1 / 70,
  /** @private @const */
  _MAX_HEIGHT_FOR_GAPS: 0.25,
  /** @private @const */
  _GROUP_IDX: 0,

  /**
   * Renders the pyramid into the available space.
   * @param {Chart} chart The chart being rendered.
   * @param {dvt.Container} container The container to render to.
   * @param {dvt.Rectangle} availSpace The available space.
   */
  render: (chart, container, availSpace) => {
    // Creating a container for the pyramid so that it can be rotated if vertical, also for animation.
    var pyramidContainer = new Container(chart.getCtx());
    pyramidContainer.setTranslate(availSpace.x, availSpace.y);
    container.addChild(pyramidContainer);
    chart.setPlotArea(pyramidContainer);

    var bbox = new Rectangle(0, 0, availSpace.w, availSpace.h);

    DvtChartEventUtils.addPlotAreaDnDBackground(chart, pyramidContainer, availSpace, true);

    if (!DvtChartPyramidRenderer._renderPyramidSlices(chart, pyramidContainer, bbox))
      DvtChartTextUtils.renderEmptyText(chart, container, availSpace);

    // Initial Selection
    var selected = DvtChartDataUtils.getInitialSelection(chart);
    DvtChartEventUtils.setInitialSelection(chart, selected);

    // Initial Highlighting
    chart.highlight(DvtChartDataUtils.getHighlightedCategories(chart));
  },

  /**
   * Renders all pyramid slices for the given chart.
   * @param {Chart} chart
   * @param {dvt.Container} container The container to render to.
   * @param {dvt.Rectangle} availSpace
   * @return {boolean} true if pyramid slices have been rendered, false otherwise
   * @private
   */
  _renderPyramidSlices: (chart, container, availSpace) => {
    var seriesCount = DvtChartDataUtils.getSeriesCount(chart);

    // Compute the gap size
    var gapRatio = DvtChartStyleUtils.getDataItemGaps(chart);
    var defaultGapSize = DvtChartPyramidRenderer._DEFAULT_GAP_RATIO * availSpace.h;
    var maxGapSize = Math.min(
      (DvtChartPyramidRenderer._MAX_HEIGHT_FOR_GAPS * availSpace.h) / (seriesCount - 1),
      defaultGapSize
    );
    var gapSize = gapRatio * maxGapSize;

    var totalValue = 0; // the total value represented by the pyramid
    var numDrawnSeries = 0; // to keep track of how many series are drawn, so we don't add too many gaps if there are zero values
    var cumulativeValue = 0; // keeping track of the total up to this series

    // Iterate through the data to calculate the total value
    for (var seriesIndex = 0; seriesIndex < seriesCount; seriesIndex++) {
      // Skip the series if it shouldn't be rendered
      if (!DvtChartDataUtils.isDataItemRendered(chart, seriesIndex)) continue;

      // Do not render if the value is not positive
      var value = DvtChartDataUtils.getVal(chart, seriesIndex, DvtChartPyramidRenderer._GROUP_IDX);
      if (value <= 0) continue;
      totalValue += value;
    }

    if (totalValue == 0) return false;

    // Iterate through the data
    for (var seriesIdx = 0; seriesIdx < seriesCount; seriesIdx++) {
      // Skip the series if it shouldn't be rendered
      if (!DvtChartDataUtils.isDataItemRendered(chart, seriesIdx)) continue;

      // Do not render if the value is not positive
      var val = DvtChartDataUtils.getVal(chart, seriesIdx, DvtChartPyramidRenderer._GROUP_IDX);
      if (val <= 0) continue;

      var slice = new DvtChartPyramidSlice(
        chart,
        seriesIdx,
        numDrawnSeries,
        availSpace.w,
        availSpace.h,
        cumulativeValue,
        val / totalValue,
        gapSize
      );
      cumulativeValue += val / totalValue;

      numDrawnSeries++; // keeping track of how many series have been drawn to create the gap.
      container.addChild(slice);
      DvtChartObjPeer.associate(slice, chart, seriesIdx, DvtChartPyramidRenderer._GROUP_IDX);
    }
    return true;
  }
};

/**
 *  Creates a selectable shape using SVG path commands.
 *  @extends {dvt.Path}
 *  @class DvtChartSelectableWedge  Creates an arbitrary shape using SVG path commands.
 *  @constructor
 *  @param {dvt.Context} context
 *  @param {number} cx  The center x position.
 *  @param {number} cy  The center y position.
 *  @param {number} rx  The horizontal radius of the ellipse.
 *  @param {number} ry  The vertical radius of the ellipse.
 *  @param {number} sa  The starting angle in degrees (following the normal anti-clockwise is positive convention).
 *  @param {number} ae  The angle extent in degrees (following the normal anti-clockwise is positive convention).
 *  @param {number} gap The data item gap.
 *  @param {number} ir The inner radius.
 *  @param {String} id  Optional ID for the shape (see {@link  dvt.Displayable#setId}).
 */
class DvtChartSelectableWedge extends Path {
  /**
   *  Object initializer.
   *  @param {dvt.Context} context
   *  @param {Object} cmds  Optional string of SVG path commands (see comment for
   *                        {@link dvt.Path#setCmds}), or an array containing
   *                        consecutive command and coordinate entries (see comment
   *                        for {@link dvt.Path#setCommands}).
   * @param {String} id  Optional ID for the shape (see {@link  dvt.Displayable#setId}).
   *  @protected
   */
  /**
   *  Sets the path commands based on the wedge parameters
   *  @param {number} cx  The center x position.
   *  @param {number} cy  The center y position.
   *  @param {number} rx  The horizontal radius.
   *  @param {number} ry  The vertical radius.
   *  @param {number} sa  The starting angle in degrees (following the normal anti-clockwise is positive convention).
   *  @param {number} ae  The angle extent in degrees (following the normal anti-clockwise is positive convention).
   *  @param {number} gap The gap between wedges.
   *  @param {number} ir The inner radius of the wedge.
   */
  setWedgeParams(cx, cy, rx, ry, sa, ae, gap, ir) {
    this._cx = cx;
    this._cy = cy;
    this._rx = rx;
    this._ry = ry;
    this._sa = sa;
    this._ae = ae;
    this._gap = gap;
    this._ir = ir;
    var cmds = this._makeWedgePath(0);
    this.setCmds(cmds);
  }

  /**
   * Returns the path string for a wedge based on the set params.
   *  @param {number} inset  The number of pixels to inset the path.
   *  @return {String} the path commands for creating the wedge
   * @private
   */
  _makeWedgePath(inset) {
    var rx = Math.max(this._rx - inset, 0);
    var ry = Math.max(this._ry - inset, 0);
    var gap = this._ae == 360 || rx < inset ? 0 : this._gap + 2 * inset;
    var ir = this._ir ? this._ir + inset : 0;

    var angleExtentRads =
      this._ae == 360 ? Math$1.degreesToRads(359.99) : Math$1.degreesToRads(this._ae);

    var startAngleRads = Math$1.degreesToRads(this._sa);
    var dataItemGaps = gap / 2;

    var gapAngle = dataItemGaps < rx ? Math.asin(dataItemGaps / rx) : 0;
    var centerLineAngle = -angleExtentRads / 2 - startAngleRads;

    // distanceToStartPoint should be correlated with the dataItemGaps in that dimension- needed for 3D pies because rx != ry.
    var distanceToStartPointX = Math.min(
      dataItemGaps * 5,
      angleExtentRads > 0 ? Math.abs(dataItemGaps / Math.sin(angleExtentRads / 2)) : 0
    );
    var distanceToStartPointY = rx == 0 ? distanceToStartPointX : (distanceToStartPointX * ry) / rx;

    var startPointX = this._cx + Math.cos(centerLineAngle) * distanceToStartPointX;
    var startPointY = this._cy + Math.sin(centerLineAngle) * distanceToStartPointY;

    var arcPointX = this._cx + Math.cos(-gapAngle - startAngleRads) * rx;
    var arcPointY = this._cy + Math.sin(-gapAngle - startAngleRads) * ry;

    var arcPoint2X = this._cx + Math.cos(-startAngleRads - angleExtentRads + gapAngle) * rx;
    var arcPoint2Y = this._cy + Math.sin(-startAngleRads - angleExtentRads + gapAngle) * ry;

    var outerAngle = Math$1.calculateAngleBetweenTwoVectors(
      arcPoint2X - this._cx,
      arcPoint2Y - this._cy,
      arcPointX - this._cx,
      arcPointY - this._cy
    );
    outerAngle = Math.min(outerAngle, angleExtentRads);
    var pathCommands;
    if (ir > 0) {
      var innerGapAngle = dataItemGaps < ir ? Math.asin(dataItemGaps / ir) : 0;
      var innerPointX = this._cx + Math.cos(-innerGapAngle - startAngleRads) * ir;
      var innerPointY = this._cy + Math.sin(-innerGapAngle - startAngleRads) * ir;

      var innerPoint2X =
        this._cx + Math.cos(-startAngleRads - angleExtentRads + innerGapAngle) * ir;
      var innerPoint2Y =
        this._cy + Math.sin(-startAngleRads - angleExtentRads + innerGapAngle) * ir;

      var innerAngle = Math$1.calculateAngleBetweenTwoVectors(
        innerPoint2X - this._cx,
        innerPoint2Y - this._cy,
        innerPointX - this._cx,
        innerPointY - this._cy
      );
      innerAngle = Math.min(innerAngle, outerAngle, angleExtentRads);

      if (this._ae == 360) {
        pathCommands = PathUtils.moveTo(arcPoint2X, arcPoint2Y);
        pathCommands += PathUtils.arcTo(rx, ry, angleExtentRads, 1, arcPointX, arcPointY);
        pathCommands += PathUtils.lineTo(arcPoint2X, arcPoint2Y);
        pathCommands += PathUtils.moveTo(innerPointX, innerPointY);
        pathCommands += PathUtils.arcTo(ir, ir, angleExtentRads, 0, innerPoint2X, innerPoint2Y);
      } else {
        pathCommands = PathUtils.moveTo(innerPoint2X, innerPoint2Y);
        pathCommands += PathUtils.lineTo(arcPoint2X, arcPoint2Y);
        pathCommands += PathUtils.arcTo(rx, ry, outerAngle, 1, arcPointX, arcPointY);
        pathCommands += PathUtils.lineTo(innerPointX, innerPointY);
        pathCommands += PathUtils.arcTo(ir, ir, innerAngle, 0, innerPoint2X, innerPoint2Y);
      }
    } else {
      if (this._ae == 360) {
        pathCommands = PathUtils.moveTo(arcPoint2X, arcPoint2Y);
        pathCommands += PathUtils.arcTo(rx, ry, angleExtentRads, 1, arcPointX, arcPointY);
      } else {
        pathCommands = PathUtils.moveTo(startPointX, startPointY);
        pathCommands += PathUtils.lineTo(arcPoint2X, arcPoint2Y);
        pathCommands += PathUtils.arcTo(rx, ry, outerAngle, 1, arcPointX, arcPointY);
      }
    }

    pathCommands += PathUtils.closePath();
    return pathCommands;
  }

  /**
   * Helper function that creates and adds the shapes used for displaying hover and selection effects. Should only be
   * called on hover or select operations, since it assumes that the fill, stroke, and shape size are already determined.
   * @private
   */
  _initializeSelectionEffects() {
    // Calculate the geometry of the shapes used for the selection effects
    var isRedwood = this.getCtx().getThemeBehavior() === 'redwood';
    var outerBorderWidth =
      this.isSelected() || isRedwood
        ? DvtChartSelectableWedge._OUTER_BORDER_WIDTH
        : DvtChartSelectableWedge._OUTER_BORDER_WIDTH_HOVER;
    var outerChildCmds = this._makeWedgePath(outerBorderWidth);
    var innerChildCmds = this._makeWedgePath(
      outerBorderWidth + DvtChartSelectableWedge._INNER_BORDER_WIDTH
    );

    // Just update the geometries if already initialized
    if (this.OuterChild) {
      this.OuterChild.setCmds(outerChildCmds);
      this.InnerChild.setCmds(innerChildCmds);
      return;
    }

    this.OuterChild = new Path(this.getCtx(), outerChildCmds);
    this.OuterChild.setInvisibleFill();
    this.OuterChild.setMouseEnabled(true);
    this.addChild(this.OuterChild);

    this.InnerChild = new Path(this.getCtx(), innerChildCmds);
    this.InnerChild.setInvisibleFill();
    this.InnerChild.setMouseEnabled(true);
    this.addChild(this.InnerChild);
  }

  /**
   * Helper function to apply border colors for hover and selection.
   * @param {string=} outerBorderColor
   * @param {string=} innerBorderColor
   * @private
   */
  _showNestedBorders(outerBorderColor, innerBorderColor) {
    // Ensure that selection and hover shapes are created
    this._initializeSelectionEffects();
    // Modify the shapes based on which borders should be shown
    if (innerBorderColor) {
      this.setSolidFill(outerBorderColor);
      this.setStroke(null);
      this.setClassName().setStyle();

      this.OuterChild.setSolidFill(innerBorderColor);
      this.OuterChild.setClassName().setStyle();

      this.InnerChild.setFill(this._fill);
      this.InnerChild.setClassName(this._shapeClassName).setStyle(this._shapeStyle);
    } else if (outerBorderColor) {
      this.setSolidFill(outerBorderColor);
      this.setStroke(null);
      this.setClassName().setStyle();

      this.OuterChild.setFill(this._fill);
      this.OuterChild.setClassName(this._shapeClassName).setStyle(this._shapeStyle);

      this.InnerChild.setInvisibleFill();
      this.InnerChild.setClassName().setStyle();
    } else {
      this.setFill(this._fill);
      this.setStroke(this._shapeStroke);
      this.setClassName(this._shapeClassName).setStyle(this._shapeStyle);

      this.OuterChild.setInvisibleFill();
      this.OuterChild.setClassName().setStyle();

      this.InnerChild.setInvisibleFill();
      this.InnerChild.setClassName().setStyle();
    }
  }

  /**
   * Specifies the colors needed to generate the selection effect.
   * @param {dvt.Fill} fill
   * @param {dvt.Stroke} stroke
   * @param {string} dataColor The color of the data.
   * @param {string} innerColor The color of the inner selection border.
   * @param {string} outerColor The color of the outer selection border.
   * @param {string} className The className of the shape.
   * @param {object} style The style of the shape.
   */
  setStyleProperties(fill, stroke, dataColor, innerColor, outerColor, className, style) {
    this._fill = fill;
    // Save original stroke style to get reapplied in _showNestedBorders. Cannot use this._stroke, as it gets overwritten during select and hover
    this._shapeStroke = stroke;
    var isRedwood = this.getCtx().getThemeBehavior() === 'redwood';
    this._hoverColor = isRedwood
      ? dataColor
      : SelectionEffectUtils.getHoverBorderColor(dataColor);
    this._innerColor = innerColor;
    this._outerColor = outerColor;
    this._shapeClassName = className;
    this._shapeStyle = style;
    this.setStyle(style).setClassName(className);

    // Apply the fill and stroke
    this.setFill(fill);
    if (stroke) this.setStroke(stroke);
  }

  /**
   * @override
   */
  showHoverEffect() {
    this.IsShowingHoverEffect = true;
    var isRedwood = this.getCtx().getThemeBehavior() === 'redwood';
    var outerColor = isRedwood && this.isSelected() ? this._outerColor : this._hoverColor;
    this._showNestedBorders(outerColor, this._innerColor);
  }

  /**
   * @override
   */
  hideHoverEffect() {
    this.IsShowingHoverEffect = false;
    if (this.isSelected()) this._showNestedBorders(this._outerColor, this._innerColor);
    else this._showNestedBorders();
  }

  /**
   * @override
   */
  setSelected(selected) {
    if (this.IsSelected == selected) return;

    this.IsSelected = selected;
    var isRedwood = this.getCtx().getThemeBehavior() === 'redwood';
    if (this.isHoverEffectShown()) {
      var outerColor = isRedwood && this.isSelected() ? this._outerColor : this._hoverColor;
      this._showNestedBorders(outerColor, this._innerColor);
    } else if (this.isSelected()) this._showNestedBorders(this._outerColor, this._innerColor);
    else this._showNestedBorders();
  }

  /**
   * @override
   */
  UpdateSelectionEffect() {
    // noop: Selection effects fully managed by this class
  }
}

/** @private @const */
DvtChartSelectableWedge._OUTER_BORDER_WIDTH = 2;

/** @private @const */
DvtChartSelectableWedge._OUTER_BORDER_WIDTH_HOVER = 1.25;

/** @private @const */
DvtChartSelectableWedge._INNER_BORDER_WIDTH = 1;

/*---------------------------------------------------------------------*/
/* Class DvtChartPieLabelInfo       Slice label information               */
/*---------------------------------------------------------------------*/

/** A property bag used to pass around information used for label placement
 *
 * @constructor
 */
class DvtChartPieLabelInfo {
  constructor() {
    this._sliceLabel = null; // instance of dvt.OutputText or dvt.MultilineText
    this._slice = null; // DvtSlice we will associate _sliceLabel with, if we can fit the label
    this._angle = -1;

    // this._position is the normalized midpoint angle, where 0 degrees is at 12 o'clock
    //    and angular measures are degrees away from 12 o'clock (so 90 degrees
    //    can be either at 3 o'clock or 9 o'clock on the unit circle)
    this._position = -1;
    this._width = -1;
    this._height = -1;
    this._x = -1;
    this._y = -1;

    this._initialNumLines = -1;

    this._hasFeeler = false;

    this._maxY = -1;
    this._minY = -1;
  }

  /**
   * @return {number} Angle of the text in this slice label
   */
  getAngle() {
    return this._angle;
  }

  /**
   * @param {number} angle Sets the angle of the text in this slice label
   */
  setAngle(angle) {
    this._angle = angle;
  }

  /**
   * @return {number} The height of this slice label
   */
  getHeight() {
    return this._height;
  }

  /**
   * @param {number} height The height of this slice label
   */
  setHeight(height) {
    this._height = height;
  }

  /**
   * @return {number}
   */
  getInitialNumLines() {
    return this._initialNumLines;
  }

  /**
   * @param {number} numLines
   */
  setInitialNumLines(numLines) {
    this._initialNumLines = numLines;
  }

  /**
   * @return {number} The maximum Y position of this slice label
   */
  getMaxY() {
    return this._maxY;
  }

  /**
   * @param {number} maxY The maximum Y position of this slice label
   */
  setMaxY(maxY) {
    this._maxY = maxY;
  }

  /**
   * @return {number} The minimum Y position of this slice label
   */
  getMinY() {
    return this._minY;
  }

  /**
   * @param {number} minY The minimum Y position of this slice label
   */
  setMinY(minY) {
    this._minY = minY;
  }

  /**
   * bound the value of y within minY and maxY
   * assumes that maxY > minY
   * @param {number} y value
   * @return {number} bounded y value
   */
  boundY(y) {
    if (this._minY <= this._maxY) {
      y = Math.max(y, this._minY);
      y = Math.min(y, this._maxY);
    }
    return y;
  }

  /**
   * @return {boolean}
   */
  hasFeeler() {
    return this._hasFeeler;
  }

  /**
   * @param {boolean} hasFeeler
   */
  setHasFeeler(hasFeeler) {
    this._hasFeeler = hasFeeler;
  }

  /**
   * Returns the normalized midpoint angle, where 0 degrees is at 12 o'clock
   * and angular measures are degrees away from 12 o'clock (so 90 degrees
   * can be either at 3 o'clock or 9 o'clock on the unit circle)
   *
   * @return {number}
   */
  getPosition() {
    return this._position;
  }

  /**
   * Sets the normalized midpoint angle, where 0 degrees is at 12 o'clock
   * and angular measures are degrees away from 12 o'clock (so 90 degrees
   * can be either at 3 o'clock or 9 o'clock on the unit circle)
   *
   * @param {number} position
   */
  setPosition(position) {
    this._position = position;
  }

  /**
   * The slice that we want to associate the label with
   *
   * @return {DvtChartPieSlice}
   */
  getSlice() {
    return this._slice;
  }

  /**
   * @param {DvtChartPieSlice} slice
   */
  setSlice(slice) {
    this._slice = slice;
  }

  /**
   * The displayable associated with this SliceLabelInfo
   *
   * @return {dvt.OutputText|dvt.MultilineText}
   */
  getSliceLabel() {
    return this._sliceLabel;
  }

  /**
   * Sets the displayable this label info will layout
   *
   * @param {dvt.OutputText|dvt.MultilineText} label
   */
  setSliceLabel(label) {
    this._sliceLabel = label;
  }

  /**
   * @return {number} The width of this label
   */
  getWidth() {
    return this._width;
  }

  /**
   * @param {number} width
   */
  setWidth(width) {
    this._width = width;
  }

  /**
   * @return {number} The x-coordinate of the reference point for this label
   */
  getX() {
    return this._x;
  }

  /**
   * @param {number} x
   */
  setX(x) {
    this._x = x;
  }

  /**
   * @return {number} The y-coordinate of hte reference point for this label
   */
  getY() {
    return this._y;
  }

  /**
   * @param {number} y
   */
  setY(y) {
    this._y = y;
  }
}

/**
 * @class DvtChartPieRenderUtils
 */
const DvtChartPieRenderUtils = {
  // surface types
  SURFACE_CRUST: 0,
  SURFACE_LEFT: 1,
  SURFACE_RIGHT: 2,

  /**
   * Returns a <code>Point</code> object representing a point at a given
   * angle at a distance specified by the rx and ry radius from the center cx, cy.
   *
   * Function reflects input angle over the y-axis.  It then scales the
   * cosine and sine of this angle by rx and ry, and then translates
   * the cosine and sine values by cx and cy.  The reflected, scaled, and
   * translated angle's cosine and sine values are then returned
   *
   * @param {number} angle A <code>Number</code> representing the desired angle in degrees.
   * @param {number} cx    A <code>Number</code> indicating the center horizontal position.
   * @param {number} cy    A <code>Number</code> indicating the center vertical position.
   * @param {number} rx    A <code>Number</code> indicating the horizontal radius.
   * @param {number} ry    A <code>Number</code> indicating the vertical radius.
   *
   * @return {object} A point object with the calculated x and y fields.
   */

  // original code taken from com.oracle.dvt.shape.draw.utils.RenderUtils
  // function originally called rotatePoint -- but that was a serious misnomer
  reflectAngleOverYAxis: (angle, cx, cy, rx, ry) => {
    var radian = Math$1.degreesToRads(360 - angle);
    var cosine = Math.cos(radian);
    var sine = Math.sin(radian);

    return { x: cx + cosine * rx, y: cy + sine * ry };
  },

  /**
   * Returns an array of colors (with no alphas) for use in creating a gradient, based on a base color and where the gradient
   * will be applied
   *
   * @param {String} baseColor
   * @param {String} style Either '2D', '3D', 'CRUST',
   *                          'SIDE', or 'BORDER'
   * @return {Array}
   */
  getGradientColors: (baseColor, style) => {
    if (style == '2D' || style == '3D')
      return [
        ColorUtils.adjustHSL(baseColor, 0, -0.04, -0.05),
        ColorUtils.adjustHSL(baseColor, 0, -0.09, 0.04)
      ];
    else if (style == 'CRUST')
      return [
        ColorUtils.adjustHSL(baseColor, 0, -0.04, -0.05),
        ColorUtils.adjustHSL(baseColor, 0, 0, -0.14)
      ];
    else if (style == 'SIDE')
      return [
        ColorUtils.adjustHSL(baseColor, 0, -0.1, 0.06),
        ColorUtils.adjustHSL(baseColor, 0, -0.04, -0.05)
      ];
    return undefined;
  },

  /**
   * Returns an array of alphas for use in creating a gradient, based on an initial alpha value and where the gradient
   * will be applied
   *
   * @param {number} baseAlpha
   * @param {String} style Either '2D', '3D', 'CRUST',
   *                          'SIDE', or 'BORDER'
   *
   * @return {Array}
   */
  getGradientAlphas: (baseAlpha, style) => {
    var alpha = baseAlpha == null || isNaN(baseAlpha) || baseAlpha == 0 ? 1.0 : baseAlpha;
    if (style == '2D') return [alpha, alpha, alpha];
    else if (style == 'BORDER')
      return [alpha / (0xff / 0xa0), alpha / (0xff / 0x30), alpha / (0xff / 0x60)];
    else if (style == '3D') return [alpha, alpha, alpha, alpha, alpha];
    else if (style == 'CRUST') return [alpha, alpha, alpha, alpha];
    else if (style == 'SIDE') return [alpha, alpha];

    return undefined;
  },

  /*
   * Static methods for generating the physical shapes that make up the different pieces of a DvtChartPieSlice
   */

  /**
   * @this {DvtChartPieSlice}
   * Returns an array of dvt.Shape objects representing the top of a pie slice
   *
   * @param {DvtChartPieSlice} slice The slice to generate the top for
   * @param {dvt.Fill} fill The fill for the top
   * @return {Array} An array of dvt.Shape objects representing the top of this pie slice
   */
  createTopSurface: (slice, fill) => {
    var pieChart = slice.getPieChart();
    var context = pieChart.getCtx();
    var pieCenter = slice.getCenter();

    var innerRadius = slice.getInnerRadius();
    var sliceGaps =
      pieChart.is3D() ||
      slice.getSliceGaps() >
        Math.sin(Math$1.degreesToRads(slice.getAngleExtent())) * slice._radiusX + 1
        ? null
        : slice.getSliceGaps();
    var wedge = new DvtChartSelectableWedge(context);
    wedge.setWedgeParams(
      pieCenter.x,
      pieCenter.y,
      slice._radiusX,
      slice._radiusY,
      slice.getAngleStart(),
      slice.getAngleExtent(),
      sliceGaps,
      innerRadius
    );

    var innerColor = DvtChartStyleUtils.getSelectedInnerColor(pieChart.chart);
    var outerColor = DvtChartStyleUtils.getSelectedOuterColor(pieChart.chart);
    var stroke = new Stroke(slice.getStrokeColor(), 1, slice.getBorderWidth());

    var seriesIndex = slice.getSeriesIndex();
    var className = DvtChartStyleUtils.getClassName(pieChart.chart, seriesIndex, 0);
    var style = DvtChartStyleUtils.getStyle(pieChart.chart, seriesIndex, 0);
    wedge.setStyleProperties(
      fill,
      stroke,
      slice.getFillColor(),
      innerColor,
      outerColor,
      className,
      style
    );

    var shapes = [wedge];

    // Associate the shapes with the slice for use during event handling
    DvtChartPieRenderUtils.associate(slice, shapes);

    return shapes;
  },

  /**
   * Associates the specified displayables with the specified slice.
   * @param {DvtChartPieSlice} slice The owning slice.
   * @param {array} displayables The displayables to associate.
   */
  associate: (slice, displayables) => {
    if (!displayables) return;

    for (var i = 0; i < displayables.length; i++)
      slice.getPieChart().chart.getEventManager().associate(displayables[i], slice);
  },
  /**
   * Generates any lateral (non-top) pie surface
   *
   * @param {DvtChartPieSlice} slice
   * @param {number} pathType One of DvtChartPieRenderUtils.SURFACE_CRUST,
   *                          DvtChartPieRenderUtils.SURFACE_LEFT, or DvtChartPieRenderUtils.SURFACE_RIGHT
   * @param {dvt.Fill} fill The fill for the lateral surface
   *
   * @return {Array} An array of dvt.Shape objects representing this lateral surface
   */
  // replaces PieSlice._draw
  createLateralSurface: (slice, pathType, fill) => {
    // handle the case where we are animating a slice insert
    // initially, this slice will have 0 extent. in this case
    // don't generate any surface
    if (slice.getAngleExtent() == 0) {
      return [];
    }

    var talpha = ColorUtils.getAlpha(slice.getFillColor());
    var shapes = [];

    if (talpha > 0) {
      if (
        pathType == DvtChartPieRenderUtils.SURFACE_LEFT ||
        pathType == DvtChartPieRenderUtils.SURFACE_RIGHT
      ) {
        shapes.push(DvtChartPieRenderUtils._generateLateralShape(slice, pathType, null, fill));
      } else if (pathType == DvtChartPieRenderUtils.SURFACE_CRUST) {
        var pathCommands = DvtChartPieRenderUtils._createCrustPathCommands(slice);

        var len = pathCommands.length;
        for (var i = 0; i < len; i++) {
          shapes.push(
            DvtChartPieRenderUtils._generateLateralShape(slice, pathType, pathCommands[i], fill)
          );
        }
      }
    }

    // Associate the shapes with the slice for use during event handling
    DvtChartPieRenderUtils.associate(slice, shapes);

    return shapes;
  },

  /**
   * Create the gradient fill used for lateral surfaces.
   * @param {DvtChartPieSlice} slice
   * @param {String} objType One of 'CRUST' or 'SIDE'
   * @return {dvt.LinearGradientFill}
   */
  generateLateralGradientFill: (slice, objType) => {
    var angle = 270;
    var arColors = DvtChartPieRenderUtils.getGradientColors(
      ColorUtils.getRGB(slice.getFillColor()),
      objType
    );
    var arAlphas = DvtChartPieRenderUtils.getGradientAlphas(
      ColorUtils.getAlpha(slice.getFillColor()),
      objType
    );
    var arRatios = [0, 1.0];
    var arBounds = null;
    return new LinearGradientFill(angle, arColors, arAlphas, arRatios, arBounds);
  },

  /**
   * Private method that generates an array of dvt.Shape objects for different lateral pie surfaces
   *
   * @param {DvtChartPieSlice} slice
   * @param {number} pathType One of DvtChartPieRenderUtils.SURFACE_CRUST,
   *                          DvtChartPieRenderUtils.SURFACE_LEFT, or DvtChartPieRenderUtils.SURFACE_RIGHT
   * @param {String} pathCommand  A string of SVG commands in SVG "d" attribute format. Used when pathType is
   *                              DvtChartPieRenderUtils.SURFACE_CRUST. Can be set to null otherwise
   * @param {dvt.Fill} fill The fill to apply to the shapes
   *
   * @return {dvt.Shape} A right or left pie surface, or a piece of a crust, as described in pathCommands
   *
   * @private
   */
  _generateLateralShape: (slice, pathType, pathCommand, fill) => {
    var pie = slice.getPieChart();
    var context = pie.getCtx();
    // left side points and right side points
    if (
      pathType == DvtChartPieRenderUtils.SURFACE_LEFT ||
      pathType == DvtChartPieRenderUtils.SURFACE_RIGHT
    ) {
      var angle = slice.getAngleStart();
      var arc = slice.getAngleExtent();
      var xCenter = slice.getCenter().x;
      var yCenter = slice.getCenter().y;
      var xRadius = slice._radiusX;
      var yRadius = slice._radiusY;
      var depth = slice.getDepth();

      var pt =
        pathType == DvtChartPieRenderUtils.SURFACE_LEFT
          ? DvtChartPieRenderUtils.reflectAngleOverYAxis(
              angle + arc,
              xCenter,
              yCenter,
              xRadius,
              yRadius
            )
          : DvtChartPieRenderUtils.reflectAngleOverYAxis(angle, xCenter, yCenter, xRadius, yRadius);
      var pointArray = DvtChartPieRenderUtils._generateInnerPoints(
        xCenter,
        yCenter,
        pt.x,
        pt.y,
        depth
      );

      var points = [];
      for (var i = 0; i < pointArray.length; i++) {
        points.push(pointArray[i].x, pointArray[i].y);
      }

      var polygon = new Polygon(context, points);

      polygon.setFill(fill);
      if (slice.getStrokeColor()) polygon.setSolidStroke(slice.getStrokeColor());

      return polygon;
    } // draw piece of pie crust
    else {
      if (pathCommand) {
        var path = new Path(context, null);

        path.setCmds(pathCommand);
        path.setTranslate(slice.__getExplodeOffsetX(), slice.__getExplodeOffsetY());

        path.setFill(fill);
        if (slice.getStrokeColor()) {
          path.setSolidStroke(slice.getStrokeColor());
        }

        return path;
      }
    }

    return null;
  },

  /**
   * Returns an array of path commands describing how to draw a pie crust
   *
   * @param {DvtChartPieSlice} slice
   *
   * @return {Array} An array of strings of SVG commands in SVG "d" attribute format.
   *                 e.g., [ [command1 x1, y1, ..., commandi xn, yn, ...], [commandj xs, ys, ...] ]
   *
   * @private
   */
  _createCrustPathCommands: (slice) => {
    var angle = slice.getAngleStart();
    var arc = slice.getAngleExtent();
    var angleEnd = angle + arc;
    var xCenter = slice.getCenter().x;
    var yCenter = slice.getCenter().y;
    var xRadius = slice._radiusX;
    var yRadius = slice._radiusY;
    var depth = slice.getDepth();

    // If slice crosses 0 degrees (right horizontal x-axis), we need to break crust into 2 pieces joined at the crossing
    // point so that the right side of the slice appears to be a solid 3D wall. If slice crosses 180 degrees (left
    // horizontal x-axis), we need to break crust into 2 pieces joined at the crossing point so that the left side of the
    // slice appears to be a solid 3D wall.
    var arOuterPath = [];
    if (angle < 180.0 && angleEnd > 360.0) {
      arOuterPath.push(
        DvtChartPieRenderUtils._makeOuterPath(
          xCenter,
          yCenter,
          xRadius,
          yRadius,
          depth,
          angle,
          180.0 - angle
        )
      ); // left
      arOuterPath.push(
        DvtChartPieRenderUtils._makeOuterPath(
          xCenter,
          yCenter,
          xRadius,
          yRadius,
          depth,
          360.0,
          angleEnd - 360.0
        )
      ); // right
      arOuterPath.push(
        DvtChartPieRenderUtils._makeOuterPath(
          xCenter,
          yCenter,
          xRadius,
          yRadius,
          depth,
          180.0,
          180.0
        )
      ); // center
    } else if (angleEnd > 360.0) {
      arOuterPath.push(
        DvtChartPieRenderUtils._makeOuterPath(
          xCenter,
          yCenter,
          xRadius,
          yRadius,
          depth,
          angle,
          360.0 - angle
        )
      );
      arOuterPath.push(
        DvtChartPieRenderUtils._makeOuterPath(
          xCenter,
          yCenter,
          xRadius,
          yRadius,
          depth,
          360.0,
          angleEnd - 360.0
        )
      );
    } else if (angle < 180.0 && angleEnd > 180.0) {
      arOuterPath.push(
        DvtChartPieRenderUtils._makeOuterPath(
          xCenter,
          yCenter,
          xRadius,
          yRadius,
          depth,
          angle,
          180.0 - angle
        )
      );
      arOuterPath.push(
        DvtChartPieRenderUtils._makeOuterPath(
          xCenter,
          yCenter,
          xRadius,
          yRadius,
          depth,
          180.0,
          angleEnd - 180.0
        )
      );
    } else
      arOuterPath.push(
        DvtChartPieRenderUtils._makeOuterPath(xCenter, yCenter, xRadius, yRadius, depth, angle, arc)
      );

    return arOuterPath;
  },

  /**
   * Returns the path string for a segment of the outer crust of a pie slice.
   * @param {number} cx The x coordinate of the center of the pie.
   * @param {number} cy The y coordinate of the center of the pie.
   * @param {number} rx The radius of the pie.
   * @param {number} ry The radius of the pie.
   * @param {number} depth The depth of the pie.
   * @param {number} startAngle The start angle in degrees.
   * @param {number} angleExtent The angular extent in degrees.  Always less than 180 degrees (half the pie).
   * @return {String} An SVG string that represents part of the crust.
   * @private
   */
  _makeOuterPath: (cx, cy, rx, ry, depth, startAngle, angleExtent) => {
    var angleExtentRads = Math$1.degreesToRads(angleExtent);
    var endAngleRads = -(Math$1.degreesToRads(startAngle) + angleExtentRads);

    // Calculate the start and end points on the top curve
    var startPointTop = DvtChartPieRenderUtils.reflectAngleOverYAxis(startAngle, cx, cy, rx, ry);
    var endPointTopX = cx + Math.cos(endAngleRads) * rx;
    var endPointTopY = cy + Math.sin(endAngleRads) * ry;

    // Top Curve
    var pathCommands = PathUtils.moveTo(startPointTop.x, startPointTop.y);
    pathCommands += PathUtils.arcTo(rx, ry, angleExtentRads, 0, endPointTopX, endPointTopY);

    // Line to Bottom Curve
    pathCommands += PathUtils.lineTo(endPointTopX, endPointTopY + depth);

    // Bottom Curve
    pathCommands += PathUtils.arcTo(
      rx,
      ry,
      angleExtentRads,
      1,
      startPointTop.x,
      startPointTop.y + depth
    );

    // Line to Top Curve
    pathCommands += PathUtils.lineTo(startPointTop.x, startPointTop.y);

    return pathCommands;
  },
  /**
   * Private function to generate the points for the left or right pie surface
   *
   * @param {number} cx The x-coordinate of the center of the pie slice
   * @param {number} cy The y-coordinate of the center of the pie slice
   * @param {number} xpos The x-coordinate of the top, outside (left or right) edge of the pie slice
   * @param {number} ypos The y-coordinate of the top, outside (left or right) edge of the pie slice
   * @param {number} tilt Pie tilt
   *
   * @return {Array} An array of points that are the coordinates for the left or right surface of a pie slice
   *
   * @private
   */
  _generateInnerPoints: (cx, cy, xpos, ypos, tilt) => {
    var pointArray = [];
    pointArray.push({ x: cx, y: cy });
    pointArray.push({ x: xpos, y: ypos });
    pointArray.push({ x: xpos, y: ypos + tilt });
    pointArray.push({ x: cx, y: cy + tilt });
    return pointArray;
  }
};

/*---------------------------------------------------------------------*/
/*   DvtChartPieLabelUtils                                                  */
/*---------------------------------------------------------------------*/

/**
 * @class DvtChartPieLabelUtils
 */
const DvtChartPieLabelUtils = {
  /** @private */
  _MAX_LINES_PER_LABEL: 3,
  /** @private */
  _COLLISION_MARGIN: 1,
  /** @private */
  _LEFT_SIDE_LABELS: 1,
  /** @private */
  _RIGHT_SIDE_LABELS: 2,
  /** @private */
  _OUTSIDE_LABEL_DISTANCE: 1.04, // distance from the slice, as a ratio of the radius

  //constants for column layout
  /** @private */
  _FEELER_RAD_MINSIZE: 0.1, // ratio to the pie diameter
  /** @private */
  _FEELER_HORIZ_MINSIZE: 0.1, // ratio to the pie diameter
  /** @private */
  _LABEL_TO_FEELER_OFFSET: 0.5, // ratio to the label height
  /** @private */
  _LABEL_TO_FEELER_DISTANCE: 3, // in pixels
  /** @private */
  _NO_COLLISION: 0,
  /** @private */
  _HALF_COLLISION: 1,
  /** @private */
  _ALL_COLLISION: 2,

  /**
   * Public entry point called by DvtChartPie to layout the pie chart's labels and feelers.
   * @param {DvtChartPie} pie the pie chart
   */
  layoutLabelsAndFeelers: (pie) => {
    var labelPosition = pie.getLabelPos();
    DvtChartPieLabelUtils._layoutInsideLabels(pie, labelPosition == 'auto');
    DvtChartPieLabelUtils._layoutOutsideLabelsAndFeelers(pie);
  },

  /**
   * Lays out labels that appear within the pie slices.
   * @param {DvtChartPie} pie the pie chart
   * @param {boolean} isHybrid Whether the labeling is inside/outside hybrid
   * @private
   */
  _layoutInsideLabels: (pie, isHybrid) => {
    var slices = pie.__getSlices();

    for (var i = 0; i < slices.length; i++) {
      var slice = slices[i];
      // Only doing layout for inside labels, so skip any labels that have a position of none or outside
      var labelPosition = pie.getSeriesLabelPos(slice.getSeriesIndex());
      if (
        labelPosition == 'none' ||
        labelPosition == 'outsideSlice' ||
        DvtChartPieLabelUtils._skipSliceLabel(pie, slice)
      )
        continue;

      var midAngle = slice.getAngleStart() + slice.getAngleExtent() / 2;
      var ir = slice.getInnerRadius();
      var center = slice.getCenter();
      var posX = 0;
      var posY = 0;
      var sliceLabel = DvtChartPieLabelUtils._createLabel(slice, true);

      if (slices.length == 1) {
        // Center the label
        posX = center.x;
        posY = center.y;
      } else {
        var offset = Math.max(0.45, 0.65 - (0.45 * ir) / Math.max(slice.getRadiusY(), 0.001));
        var midPt = DvtChartPieRenderUtils.reflectAngleOverYAxis(
          midAngle,
          center.x,
          center.y,
          ir + (slice.getRadiusX() - ir) * offset,
          ir + (slice.getRadiusY() - ir) * offset
        );

        posX = midPt.x;
        posY = midPt.y;
      }

      sliceLabel.setX(posX);
      sliceLabel.setY(posY);
      sliceLabel.alignMiddle();
      sliceLabel.alignCenter();

      // Find the estimated dimensions of the label
      var sliceLabelDims = sliceLabel.getDimensions();

      // Find the largest rectangle that will fit.  The height is accurate, so we only need to check the width.
      var x1 = posX;
      var x2 = posX;
      var y1 = posY - sliceLabelDims.h / 2;
      var y2 = posY + sliceLabelDims.h / 2;

      // Calculate the left-most x1 that will fit
      while (slice.contains(x1, y1) && slice.contains(x1, y2)) {
        x1--;
      }

      // Calculate the right-most x2 that will fit
      while (slice.contains(x2, y1) && slice.contains(x2, y2)) {
        x2++;
      }

      // Add a 3-pixel buffer on each side (accounts for the potential extra pixel in the while loop on failed check)
      x1 = Math.ceil(x1 + 3);
      x2 = Math.floor(x2 - 3);

      // Adjust the anchor point to the midpoint of available space if truncation would occur centered at current anchor
      var usableSpace = 2 * Math.min(posX - x1, x2 - posX);
      if (usableSpace < sliceLabelDims.w) {
        sliceLabel.setX((x1 + x2) / 2);
        usableSpace = x2 - x1;
      }

      // Don't want to use the automatic hybrid layout if slice label position is specifically set as center
      if (isHybrid && labelPosition != 'center') {
        var textWidth = sliceLabel.getDimensions().w;
        if (textWidth < usableSpace) slice.setSliceLabel(sliceLabel);
        else slice.setSliceLabel(null); // use outside label
      } else {
        // Truncate the text.  It will be added to the chart later, so remove if it is added.
        var stage = pie.getCtx().getStage();
        var minChars = !DvtChartPieLabelUtils._isTextLabel(pie, slice)
          ? sliceLabel.getTextString().length
          : null;
        if (TextUtils.fitText(sliceLabel, usableSpace, sliceLabelDims.h, stage, minChars)) {
          stage.removeChild(sliceLabel);
          slice.setSliceLabel(sliceLabel);
        }
      }

      // If hybrid labeling, the slice can animate from having a outsideLabel + feeler to being inside. In this case, we
      // need to clear the outside feeler so that it doesn't stay around.
      if (slice.getSliceLabel() != null) slice.setNoOutsideFeeler();
    }
  },

  /**
   * Lays out labels (and feelers if necessary) that appear outside the pie slices
   * @param {DvtChartPie} pie The pie chart
   * @private
   */
  _layoutOutsideLabelsAndFeelers: (pie) => {
    var leftLabels = [];
    var rightLabels = [];

    // -----------------------------------------------------------
    // Build arrays of Left side and Right side Labels
    //
    // When computing the positioning of the labels, we consider
    // angles to be measured from the 12 o'clock position,
    // i.e., 12 o'clock is 0 degrees.
    // Angular measurements then range from 0 to 180.
    // A value of 90 degrees can then be either at the
    // 3 o'clock position or at the 9 o'clock position
    // -----------------------------------------------------------
    var alabels = DvtChartPieLabelUtils._generateInitialLayout(pie);

    leftLabels = alabels[0];
    rightLabels = alabels[1];

    // -----------------------------------------------------------
    // Evaluate initial label layout from generateInitialLayout
    // -----------------------------------------------------------
    var leftColl = DvtChartPieLabelUtils._refineInitialLayout(
      pie,
      leftLabels,
      DvtChartPieLabelUtils._LEFT_SIDE_LABELS
    );
    var rightColl = DvtChartPieLabelUtils._refineInitialLayout(
      pie,
      rightLabels,
      DvtChartPieLabelUtils._RIGHT_SIDE_LABELS
    );

    if (
      leftColl == DvtChartPieLabelUtils._HALF_COLLISION &&
      rightColl != DvtChartPieLabelUtils._NO_COLLISION
    )
      DvtChartPieLabelUtils._columnLabels(pie, leftLabels, true, true, true);
    if (
      leftColl != DvtChartPieLabelUtils._NO_COLLISION &&
      rightColl == DvtChartPieLabelUtils._HALF_COLLISION
    )
      DvtChartPieLabelUtils._columnLabels(pie, rightLabels, false, true, true);

    DvtChartPieLabelUtils._setLabelsAndFeelers(
      pie,
      leftLabels,
      DvtChartPieLabelUtils._LEFT_SIDE_LABELS
    );
    DvtChartPieLabelUtils._setLabelsAndFeelers(
      pie,
      rightLabels,
      DvtChartPieLabelUtils._RIGHT_SIDE_LABELS
    );
  },

  /**
   * Create a label for the given pie slice. Label positioning is done elsewhere
   * @param {DvtChartPieSlice} slice
   * @param {boolean} isInside True if the label is inside the slice.
   * @return {dvt.OutputText|dvt.MultilineText}
   * @private
   */
  _createLabel: (slice, isInside) => {
    var isHighContrast = Agent.isHighContrast();
    var pieChart = slice.getPieChart();

    var context = pieChart.getCtx();
    var isOtherSlice = slice.getId().series === DvtChartPieUtils.OTHER_ID;
    var hasPattern =
      DvtChartStyleUtils.getPattern(pieChart.chart, slice.getSeriesIndex(), 0) != null ||
      (isOtherSlice && DvtChartStyleUtils.getSeriesEffect(pieChart) == 'pattern');
    var needsPatternBg = isInside && hasPattern;
    var sliceLabel;
    if (needsPatternBg) {
      sliceLabel = new BackgroundOutputText(context);
    } else {
      sliceLabel = isInside ? new OutputText(context) : new MultilineText(context);
    }

    // Apply the label color- read all applicable styles and merge them.
    var contrastColor = needsPatternBg
      ? ColorUtils.getContrastingTextColor(DvtChartStyleUtils.SERIES_PATTERN_BG_COLOR)
      : isInside
      ? ColorUtils.getContrastingTextColor(slice.getFillColor())
      : ColorUtils.getContrastingTextColor(ColorUtils.getColorFromName('black'));
    var contrastColorStyle = new CSSStyle({ color: contrastColor });
    var styleDefaults = pieChart.getOptions().styleDefaults;
    var labelStyleArray = [styleDefaults._dataLabelStyle];
    if (isInside) {
      labelStyleArray.push(contrastColorStyle);
    }
    labelStyleArray.push(styleDefaults.dataLabelStyle);
    var dataItem = DvtChartDataUtils.getDataItem(pieChart.chart, slice.getSeriesIndex(), 0);
    if (dataItem) {
      labelStyleArray.push(new CSSStyle(dataItem.labelStyle));
    }

    if (isHighContrast) {
      labelStyleArray.push(contrastColorStyle);
    }
    if (needsPatternBg) {
      var backgroundColorStyle = new CSSStyle({
        'background-color': 'rgba(255, 255, 255, 0.9)',
        'border-radius': '2px'
      });
      labelStyleArray.push(backgroundColorStyle);
    }
    var style = CSSStyle.mergeStyles(labelStyleArray);
    sliceLabel.setCSSStyle(style);

    var labelStr = DvtChartPieLabelUtils.generateSliceLabelString(
      slice,
      styleDefaults['sliceLabelType']
    );
    sliceLabel.setTextString(labelStr);
    slice.setSliceLabelString(labelStr);

    return sliceLabel;
  },
  /**
   * Create the center content for pie chart.
   * @param {DvtChartPie} pieChart the pie chart
   */
  createPieCenter: (pieChart) => {
    var options = pieChart.getOptions();
    var context = pieChart.getCtx();
    var pieCenter = DvtChartPieLabelUtils.getPieCenterOptions(pieChart, options);
    var centerLabel = pieCenter['label'];
    var centerRenderer = pieCenter['renderer'];
    var dataLabelPosition = pieChart.getLabelPos();
    var customTooltip = options['tooltip'];
    var tooltipFunc = customTooltip ? customTooltip['renderer'] : null;
    var centerCoord = pieChart.getCenter();
    var innerRadius = pieChart.getInnerRadius();

    if (!centerLabel && !centerRenderer) return;

    var radiusX = pieChart.getRadiusX();
    var defaultInnerRadius = dataLabelPosition == 'outsideSlice' ? 0.9 * radiusX : 0.5 * radiusX;
    innerRadius = innerRadius > 0 ? innerRadius : defaultInnerRadius;
    var innerSquareDimension = innerRadius * Math.sqrt(2);
    if (centerLabel) {
      var centerText = new MultilineText(context);
      var centerStyle = pieCenter['labelStyle'];
      centerText.setCSSStyle(centerStyle);

      // Scaling and Converter option handler
      if (typeof centerLabel === 'number') {
        centerLabel = DvtChartFormatUtils$1.formatVal(
          pieChart,
          pieCenter,
          centerLabel,
          centerLabel,
          centerLabel,
          0
        );
      }

      centerText.setTextString(centerLabel);
      if (TextUtils.fitText(centerText, innerSquareDimension, innerSquareDimension, pieChart)) {
        var textDim = centerText.getDimensions();
        centerText.setY(centerCoord.y - textDim.h / 2);
        centerText.setX(centerCoord.x);
        centerText.alignCenter();

        // Associate with logical object to support automation and truncation
        if (!tooltipFunc)
          pieChart.chart.getEventManager().associate(
            centerText,
            new SimpleObjPeer(centerText.getTextString(), null, null, {
              type: 'pieCenterLabel'
            })
          );
        pieChart.addChild(centerText);
        pieChart.setCenterLabel(centerText);
      }
    }

    // If there is a tooltip callback function, overlay a circular object over the center area.
    if (tooltipFunc) {
      var centerOverlay = new Circle(context, centerCoord.x, centerCoord.y, innerRadius);
      centerOverlay.setInvisibleFill();
      pieChart.addChild(centerOverlay);
      var tooltipManager = pieChart.getCtx().getTooltipManager();
      pieChart.chart.getEventManager().associate(
        centerOverlay,
        new CustomDatatipPeer(tooltipManager, tooltipFunc, '#4b4b4b', {
          component: options['_widgetConstructor'],
          label: centerLabel
        })
      );
    }

    if (centerRenderer) {
      var dataContext = {
        outerBounds: {
          x: centerCoord.x - innerRadius,
          y: centerCoord.y - innerRadius,
          width: 2 * innerRadius,
          height: 2 * innerRadius
        },
        innerBounds: {
          x: centerCoord.x - innerSquareDimension / 2,
          y: centerCoord.y - innerSquareDimension / 2,
          width: innerSquareDimension,
          height: innerSquareDimension
        },
        label: centerLabel,
        totalValue: pieChart.getTotalValue(),
        component: options['_widgetConstructor']
      };
      dataContext = context.fixRendererContext(dataContext);
      var parentDiv = context.getContainer();

      // Remove existing overlay if there is one
      var existingOverlay = pieChart.chart.pieCenterDiv;
      if (existingOverlay) parentDiv.removeChild(existingOverlay);

      var customContent = centerRenderer(dataContext);
      if (!customContent) return;
      var newOverlay = context.createOverlayDiv();
      if (Array.isArray(customContent)) {
        customContent.forEach((node) => {
          newOverlay.appendChild(node); // @HTMLUpdateOK
        });
      } else if (typeof customContent === 'string') {
        newOverlay.textContent = customContent;
        newOverlay.style.position = 'absolute';
        newOverlay.style.left = `${centerCoord.x}px`;
        newOverlay.style.top = `${centerCoord.y}px`;
        newOverlay.style.transform = 'translate(-50%, -50%)';
        newOverlay.style.width = `${2 * innerRadius}px`;
        newOverlay.style.height = `${2 * innerRadius}px`;
        newOverlay.style.display = 'flex';
        newOverlay.style.textAlign = 'center';
        newOverlay.style.alignItems = 'center';
      } else {
        newOverlay.appendChild(customContent); // @HTMLUpdateOK
      }
      pieChart.chart.pieCenterDiv = newOverlay;
      parentDiv.appendChild(newOverlay); // @HTMLUpdateOK

      // Invoke the overlay attached callback if one is available.
      var callback = context.getOverlayAttachedCallback();
      if (callback) callback(newOverlay);
    }
  },
  /**
   * Returns the untruncated label text for a given pie slice
   * @param {DvtChartPieSlice} slice
   * @param {string} labelType The label type.
   * @return {string} The full, untruncated label string, or null if the slice's pie chart is configured to not display labels
   */
  generateSliceLabelString: (slice, labelType) => {
    var functionLabel;
    var defaultLabel = DvtChartPieLabelUtils.getDefaultSliceLabelString(slice, labelType);

    // Use data Label function if there is one
    var dataLabelFunc = slice.getPieChart().getOptions()['dataLabel'];

    if (dataLabelFunc) {
      var dataContext = DvtChartStyleUtils.getDataContext(slice._chart, slice.getSeriesIndex(), 0);
      dataContext['label'] = defaultLabel;
      functionLabel = dataLabelFunc(dataContext);
      if (typeof functionLabel == 'number') {
        var valueFormat = DvtChartFormatUtils$1.getValueFormat(slice.getPieChart().chart, 'label');
        functionLabel = DvtChartFormatUtils$1.formatVal(
          slice.getPieChart(),
          valueFormat,
          functionLabel
        );
      }
    }

    return functionLabel ? functionLabel : defaultLabel;
  },
  /**
   * Returns the default label text for a given pie slice and ignores the dataLabel function
   * @param {DvtChartPieSlice} slice
   * @param {string} labelType The label type.
   * @return {string} The full, default label string, or null if the slice's pie chart is configured to not display labels
   */
  getDefaultSliceLabelString: (slice, labelType) => {
    var pieChart = slice.getPieChart();

    // If customLabel exists it will be the slice label. If a converter is set, apply converter to customLabel
    var customLabel = slice.getCustomLabel();
    var valueFormat = DvtChartFormatUtils$1.getValueFormat(pieChart.chart, 'label');
    if (customLabel != null) {
      if (typeof customLabel == 'number')
        return DvtChartFormatUtils$1.formatVal(pieChart, valueFormat, customLabel);
      return customLabel;
    }

    if (labelType == 'percent') return DvtChartPieLabelUtils.generateSlicePercentageString(slice);
    else if (labelType == 'number')
      return DvtChartFormatUtils$1.formatVal(pieChart, valueFormat, slice.getVal());
    else if (labelType == 'text') return slice.getSeriesLabel();
    else if (labelType == 'textAndPercent')
      return (
        slice.getSeriesLabel() + ', ' + DvtChartPieLabelUtils.generateSlicePercentageString(slice)
      );
    return null;
  },
  /**
   * Returns the default percentage string for a given pie slice
   * @param {DvtChartPieSlice} slice
   * @return {string} The default percentage string for the slice
   */
  generateSlicePercentageString: (slice) => {
    var pieChart = slice.getPieChart();

    var totalValue = pieChart.getTotalValue();
    var percentage = totalValue == 0 ? 0 : slice.getVal() / totalValue;
    var onesDigits = percentage < 1 ? 1 : 0;
    var tenthsDigits = percentage < 0.1 ? 2 : onesDigits;
    var decDigits = percentage < 0.01 ? 3 : tenthsDigits;
    // For pies smaller than a certain size, drop to fewer significant digits so that the labels can display.
    if (pieChart.getRadiusX() * 2 < 150) decDigits = Math.max(decDigits - 1, 0);

    var formats = pieChart.getOptions().valueFormats;
    var customConverter;
    if (formats && formats.label && formats.label.converter) {
      customConverter = formats.label.converter;
    }
    var percentConverter = pieChart.getCtx().getNumberConverter({
      style: 'percent',
      maximumFractionDigits: decDigits,
      minimumFractionDigits: decDigits
    });
    var spercent = '';

    if (customConverter && customConverter['format']) {
      spercent = customConverter['format'](percentage);
    } else if (percentConverter && percentConverter['format']) {
      // Apply the percent converter if present
      spercent = percentConverter['format'](percentage);
    } else {
      percentage *= 100;
      // If the percentage is 100%, make sure to display it without any fractions ("100%" not "100.0%")
      spercent =
        DvtChartFormatUtils$1.formatVal(
          pieChart,
          {},
          percentage,
          null,
          null,
          percentage == 100 ? 1 : Math.pow(10, -1 * decDigits)
        ) + '%';
    }

    return spercent;
  },
  /**
   * Called after initial naive layout is generated to resolve label collisions
   *
   * @param {DvtChartPie} pie
   * @param {Array} labelInfoArray An array of DvtChartPieLabelInfo objects
   * @param {number} side Either DvtChartPieLabelUtils._LEFT_SIDE_LABELS or DvtChartPieLabelUtils._RIGHT_SIDE_LABELS
   * @return {number} DvtChartPieLabelUtils._ALL_COLLISION, DvtChartPieLabelUtils._HALF_COLLISION, or DvtChartPieLabelUtils._NO_COLLISION
   * @private
   */
  _refineInitialLayout: (pie, labelInfoArray, side) => {
    if (labelInfoArray && labelInfoArray.length > 0) {
      var lastY = pie.__getFrame().y; //think again!!
      var collisionTop = false;
      var collisionCentral = false;
      var collisionBottom = false;
      var labelBottom = 0;

      var labelInfo;

      var bottomQuarter = false;
      var prevBottomQuarter;
      var collide = false;
      var isLeftSideLabels = side == DvtChartPieLabelUtils._LEFT_SIDE_LABELS;

      for (var i = 0; i < labelInfoArray.length; i++) {
        labelInfo = labelInfoArray[i];

        prevBottomQuarter = bottomQuarter;
        if (labelInfo.getPosition() > 90) bottomQuarter = true;

        labelBottom = labelInfo.getY() + labelInfo.getHeight();

        collide = lastY - labelInfo.getY() > DvtChartPieLabelUtils._COLLISION_MARGIN;

        if (collide) {
          if (!bottomQuarter) {
            collisionTop = true;
          } else if (!prevBottomQuarter) {
            collisionCentral = true;
          } else {
            collisionBottom = true;
          }
        }

        if (labelBottom > lastY) {
          lastY = labelBottom;
        }
      }

      if ((collisionTop && collisionBottom) || collisionCentral) {
        DvtChartPieLabelUtils._columnLabels(pie, labelInfoArray, isLeftSideLabels, true, true);
        return DvtChartPieLabelUtils._ALL_COLLISION;
      } else if (collisionTop) {
        DvtChartPieLabelUtils._columnLabels(pie, labelInfoArray, isLeftSideLabels, true, false); //top only
        return DvtChartPieLabelUtils._HALF_COLLISION;
      } else if (collisionBottom) {
        DvtChartPieLabelUtils._columnLabels(pie, labelInfoArray, isLeftSideLabels, false, true); //bottom only
        return DvtChartPieLabelUtils._HALF_COLLISION;
      }
      return DvtChartPieLabelUtils._NO_COLLISION;
    }
    return undefined;
  },

  // ported over from PieChart.as, renderLabelsAndFeelers
  /**
   *
   * Sets the location of the label objects as specified in the DvtChartPieLabelInfo objects in alabels.
   * When this method returns, the DvtChartPie labels corresponding to the DvtChartPieLabelInfo objects in alabels
   * will have their layout positions set, and will be ready to render
   *
   * @param {DvtChartPie} pie
   * @param {Array} alabels An array of DvtChartPieLabelInfo objects.
   * @param {number} side Either DvtChartPieLabelUtils._LEFT_SIDE_LABELS or DvtChartPieLabelUtils._RIGHT_SIDE_LABELS
   * @private
   */
  _setLabelsAndFeelers: (pie, alabels, side) => {
    if (alabels == null || alabels.length <= 0) return;

    var i;
    var slice; // instance of DvtChartPieSlice
    var sliceLabel; // instance of dvt.OutputText or dvt.MultilineText
    var labelInfo; // instance of DvtChartPieLabelInfo
    var isLeftSide = side == DvtChartPieLabelUtils._LEFT_SIDE_LABELS;
    var frame = pie.__getFrame();

    var excessWidth = Infinity;
    var excessLength;

    // Determine how much the horizontal feelers can be shortened
    for (i = 0; i < alabels.length; i++) {
      labelInfo = alabels[i];
      slice = labelInfo.getSlice();

      if (labelInfo.hasFeeler()) {
        excessLength = DvtChartPieLabelUtils._calculateFeeler(labelInfo, slice, isLeftSide);

        // For numeric labels, the minLabelWidth is equal to the labelWidth because anything shorter will be skipped.
        // For text labels, the minLabelWidth is one character + ellipses ("S...") because if it is shorter than that then
        // fitText() will completely remove the label. We estimate that width as twice the font height.
        var fontHeight = TextUtils.getTextStringHeight(
          pie.getCtx(),
          labelInfo.getSliceLabel().getCSSStyle()
        );
        var labelWidth = labelInfo.getWidth();
        var minLabelWidth = DvtChartPieLabelUtils._isTextLabel(pie, slice)
          ? Math.min(2 * fontHeight, labelWidth)
          : labelWidth;
        var maxLabelWidth = DvtChartPieLabelUtils.getMaxLabelWidth(
          pie,
          labelInfo.getX(),
          isLeftSide
        );

        // Remove feelers for labels that will not be rendered and ignore for excess width calculation
        if (maxLabelWidth + excessLength < minLabelWidth || labelInfo.getWidth() == 0) {
          labelInfo.setSliceLabel(null);
          slice.setNoOutsideFeeler();
          continue;
        }
        excessWidth = Math.min(excessWidth, excessLength);
      } else slice.setNoOutsideFeeler();
    }

    for (i = 0; i < alabels.length; i++) {
      labelInfo = alabels[i];
      slice = labelInfo.getSlice();
      sliceLabel = labelInfo.getSliceLabel();
      if (!sliceLabel) continue;

      if (labelInfo.hasFeeler()) {
        // shorten the horizontal feelers
        if (isLeftSide) {
          labelInfo.setX(labelInfo.getX() + excessWidth);
        } else {
          labelInfo.setX(labelInfo.getX() - excessWidth);
        }
        // setup the feeler line (let it clip if needed)
        DvtChartPieLabelUtils._calculateFeeler(labelInfo, slice, isLeftSide);
      }

      sliceLabel.setY(labelInfo.getY());
      sliceLabel.setX(labelInfo.getX());

      // perform 'logical' clipping ourselves
      if (
        labelInfo.getY() < frame.y ||
        labelInfo.getY() + labelInfo.getHeight() > frame.y + frame.h
      ) {
        slice.setSliceLabel(null);
        slice.setNoOutsideFeeler(); //  - don't show feelers if the label is 'clipped' (invisible)
      } else {
        DvtChartPieLabelUtils._truncateSliceLabel(pie, slice, labelInfo, isLeftSide);
        // If slice label has zero dimensions, don't add it to the slice, and disable feelers
        if (labelInfo.getWidth() == 0 || labelInfo.getHeight() == 0) {
          slice.setSliceLabel(null);
          slice.setNoOutsideFeeler();
        } else slice.setSliceLabel(sliceLabel);
      }
    }
  },

  // replaces PieChart.drawFeeler
  /**
   *
   * Sets the feeler line that extends from the pie to the targetPt on the given slice. This method computes where
   * on the pie the feeler line should start, and then the start point and targetPt are set on the input slice.
   *
   * @param {DvtChartPieLabelInfo} labelInfo A DvtChartPieLabelInfo object
   * @param {DvtChartPieSlice} slice A DvtChartPieSlice object
   * @param {boolean} isLeft Boolean indicating if these labels are on the left side of the pie
   * @return {number} The excess length of the horizontal feeler, i.e. the length of the horizontal feeler minus the minimum length
   * @private
   */
  _calculateFeeler: (labelInfo, slice, isLeft) => {
    var targetX = labelInfo.getX();
    var targetY =
      labelInfo.getY() + labelInfo.getHeight() * DvtChartPieLabelUtils._LABEL_TO_FEELER_OFFSET;
    var minHorizLength = DvtChartPieLabelUtils._FEELER_HORIZ_MINSIZE * slice.getRadiusX();

    var midX;
    if (isLeft) {
      targetX += DvtChartPieLabelUtils._LABEL_TO_FEELER_DISTANCE;
      midX = targetX + minHorizLength;
    } else {
      targetX -= DvtChartPieLabelUtils._LABEL_TO_FEELER_DISTANCE;
      midX = targetX - minHorizLength;
    }

    var midPt = {
      x: midX,
      y: targetY
    };
    var endPt = {
      x: targetX,
      y: targetY
    };
    var ma = labelInfo.getAngle();
    var tilt = DvtChartPieLabelUtils._adjustForDepth(ma, slice.getDepth());

    var startPt = DvtChartPieRenderUtils.reflectAngleOverYAxis(
      ma,
      slice.getCenter().x,
      slice.getCenter().y + tilt,
      slice.getRadiusX(),
      slice.getRadiusY()
    );

    // make set the first section of the feeler radial if possible
    var pa = Math$1.degreesToRads(labelInfo.getPosition());
    var radFeelerAngle = Math.abs(Math.atan2(midPt.x - startPt.x, startPt.y - midPt.y));
    var horizOffset = (startPt.y - midPt.y) * Math.tan(pa); // * pieChart.getRadiusX() / pieChart.getRadiusY();
    if (
      (pa > Math.PI / 2 && radFeelerAngle > Math.PI / 2 && radFeelerAngle < pa) ||
      (pa < Math.PI / 2 && radFeelerAngle < Math.PI / 2 && radFeelerAngle > pa)
    ) {
      if (isLeft) {
        midPt.x = startPt.x - horizOffset;
      } else {
        midPt.x = startPt.x + horizOffset;
      }
    }

    //store outside feeler points on slice
    //and let slice draw the feeler so that we can
    //easily redraw it when selecting
    slice.setOutsideFeelerPoints(startPt, midPt, endPt);
    return Math.abs(endPt.x - midPt.x) - minHorizLength;
  },

  /**

  * Generates the offset of a label feeler to account for the depth of a 3d pie.
  *
  * @param {number} ma The angle on the unit circle from which the leaderline should originate from
  * @param {number} pieDepth The pie chart's depth
  *
  * @return {number} The offset for the feeler line
  * @private
  */
  _adjustForDepth: (ma, pieDepth) => {
    var depth = 0;
    var leftMidHi = 189;
    var rightMidHi = 351;

    if (ma > leftMidHi && ma < rightMidHi) {
      depth = pieDepth;
    }

    return depth;
  },

  /**
   *  Finds the label corresponding to the most horizontal slice
   *
   *  @param {Array} alabels An array of DvtChartPieLabelInfo objects
   *  @return {number} the index of the most horizontal slice
   *  @private
   */
  _getMiddleLabel: (alabels) => {
    var bestAngle = 91;
    var bestIndex = -1;
    for (var i = 0; i < alabels.length; i++) {
      var pa = alabels[i].getPosition();
      if (Math.abs(pa - 90) < bestAngle) {
        bestAngle = Math.abs(pa - 90);
        bestIndex = i;
      }
    }
    return bestIndex;
  },

  /**
   * Sets the label at its optimal position, assuming all other labels do not exist.
   * @param {DvtChartPie} pie
   * @param {DvtChartPieLabelInfo} labelInfo
   * @param {number} vertX The x-position where the labels are aligned.
   * @param {boolean} isLeft Whether the label is on the left side of the pie.
   * @private
   */
  _setOptimalLabelPos: (pie, labelInfo, vertX, isLeft) => {
    //set optimal X
    labelInfo.setX(vertX);

    //set optimal Y
    //  var a = pie.getRadiusX() * (1 + DvtChartPieLabelUtils._FEELER_RAD_MINSIZE);
    var b = pie.getRadiusY() * (1 + DvtChartPieLabelUtils._FEELER_RAD_MINSIZE);
    var angleInRad = Math$1.degreesToRads(labelInfo.getPosition());
    var heightFromCenter = b * Math.cos(angleInRad);
    var tilt = DvtChartPieLabelUtils._adjustForDepth(labelInfo.getAngle(), pie.getDepth());
    var optimalY =
      pie.getCenter().y -
      heightFromCenter -
      labelInfo.getHeight() * DvtChartPieLabelUtils._LABEL_TO_FEELER_OFFSET +
      tilt;
    labelInfo.setY(labelInfo.boundY(optimalY));
  },
  /**
   * Calculates the feeler angle for this label based on the projected x and y positions of the label outside of the slice
   * @param {DvtChartPieLabelInfo} labelInfo
   * @param {number} x The x position
   * @param {number} y The y position
   * @return {number} The angle for the feeler line
   * @private
   */
  _getRadFeelerAngle: (labelInfo, x, y) => {
    var slice = labelInfo.getSlice();
    var center = slice.getCenter();
    var ma = labelInfo.getAngle();
    var tilt = DvtChartPieLabelUtils._adjustForDepth(ma, slice.getDepth());
    var startPt = DvtChartPieRenderUtils.reflectAngleOverYAxis(
      ma,
      center.x,
      center.y + tilt,
      slice.getRadiusX(),
      slice.getRadiusY()
    );
    return Math.atan2(Math.abs(x - startPt.x), startPt.y - y);
  },
  /**
   *  Adjusts the label locations by positioning the labels vertically in a column
   *  @param {DvtChartPie} pie
   *  @param {Array} alabels An array of DvtChartPieLabelInfo objects
   *  @param {boolean} isLeft Boolean indicating if these labels are on the left side of the pie
   *  @param {boolean} isTop Boolean indicating if these labels are on the top of the pie
   *  @param {boolean} isBottom Boolean indicating if these labels are at the bottom of the pie
   *  @private
   */
  _columnLabels: (pie, alabels, isLeft, isTop, isBottom) => {
    var frame = pie.__getFrame();
    var minY = frame.y;
    var maxY = frame.y + frame.h;
    var i;
    var labelInfo;
    var pa = 0;
    var radFeelerAngle;

    //determine the position where the column will be aligned
    var vertX = pie.getCenter().x;
    var feelerX;
    var minFeelerDist =
      pie.getRadiusX() *
      (1 + DvtChartPieLabelUtils._FEELER_RAD_MINSIZE + DvtChartPieLabelUtils._FEELER_HORIZ_MINSIZE);

    if (isLeft) {
      vertX -= minFeelerDist;
      feelerX = vertX + pie.getRadiusX() * DvtChartPieLabelUtils._FEELER_HORIZ_MINSIZE;
    } else {
      vertX += minFeelerDist;
      feelerX = vertX - pie.getRadiusX() * DvtChartPieLabelUtils._FEELER_HORIZ_MINSIZE;
    }

    //set the minimum heights that ensures as many labels as possible are displayed
    for (i = 0; i < alabels.length; i++) {
      labelInfo = alabels[i];
      pa = Math$1.degreesToRads(labelInfo.getPosition());
      radFeelerAngle = DvtChartPieLabelUtils._getRadFeelerAngle(labelInfo, feelerX, minY);

      // Remove labels that are more than a certain angle away from the slice.
      if (
        radFeelerAngle - pa > 0.45 * Math.PI ||
        DvtChartPieLabelUtils._skipSliceLabel(pie, labelInfo.getSlice())
      ) {
        alabels.splice(i, 1);
        i--;
      } else {
        alabels[i].setMinY(minY);
        minY += alabels[i].getHeight();
      }
    }

    //set the maximum heights that ensures as many labels as possible are displayed
    for (i = alabels.length - 1; i >= 0; i--) {
      labelInfo = alabels[i];
      pa = Math$1.degreesToRads(labelInfo.getPosition());
      radFeelerAngle = DvtChartPieLabelUtils._getRadFeelerAngle(labelInfo, feelerX, maxY);

      // Remove labels that are more than a certain angle away from the slice.
      if (
        pa - radFeelerAngle > 0.45 * Math.PI ||
        DvtChartPieLabelUtils._skipSliceLabel(pie, labelInfo.getSlice())
      ) {
        alabels.splice(i, 1);
      } else {
        maxY -= alabels[i].getHeight();
        alabels[i].setMaxY(maxY);
      }
    }

    if (alabels.length == 0) return;

    var startIndex = DvtChartPieLabelUtils._getMiddleLabel(alabels);
    var startLabel = alabels[startIndex];

    //if the column is only partial but there are too many labels, then set the whole side as column
    if (isTop && !isBottom) {
      if (startLabel.getMinY() + startLabel.getHeight() > pie.getCenter().y) {
        isBottom = true;
      }
    }
    if (isBottom && !isTop) {
      if (startLabel.getMaxY() < pie.getCenter().y) {
        isTop = true;
      }
    }

    var labelPostion = startLabel.getPosition();
    if ((isBottom && isTop) || (labelPostion > 90 && isBottom) || (labelPostion <= 90 && isTop)) {
      DvtChartPieLabelUtils._setOptimalLabelPos(pie, startLabel, vertX, isLeft);
      startLabel.setHasFeeler(true);
    }

    var highestY = startLabel.getY();
    var lowestY = startLabel.getY() + startLabel.getHeight();

    var optimalY;
    var labelHeight;

    if (isTop) {
      //labels above the start label
      for (i = startIndex - 1; i >= 0; i--) {
        labelInfo = alabels[i];
        labelHeight = labelInfo.getHeight();
        DvtChartPieLabelUtils._setOptimalLabelPos(pie, labelInfo, vertX, isLeft);
        labelInfo.setHasFeeler(true);

        //avoid collision with the label below
        optimalY = labelInfo.getY();
        if (optimalY + labelHeight < highestY) {
          highestY = optimalY;
        } else {
          highestY -= labelHeight;
        }
        labelInfo.setY(highestY);
      }
    }

    if (isBottom) {
      //labels below the start label
      for (i = startIndex + 1; i < alabels.length; i++) {
        labelInfo = alabels[i];
        labelHeight = labelInfo.getHeight();
        DvtChartPieLabelUtils._setOptimalLabelPos(pie, labelInfo, vertX, isLeft);
        labelInfo.setHasFeeler(true);

        //avoid collision with the label above
        optimalY = labelInfo.getY();
        if (optimalY > lowestY) {
          lowestY = optimalY + labelHeight;
        } else {
          lowestY += labelHeight;
        }
        labelInfo.setY(lowestY - labelHeight);
      }
    }
  },
  /**
   *
   * Truncates the label for the last time after the final X position is calculated
   *
   * @param {DvtChartPie} pie
   * @param {DvtChartPieSlice} slice
   * @param {DvtChartPieLabelInfo} labelInfo
   * @param {boolean} isLeft Boolean indicating whether or not this slice is on the left side of the pie
   *
   * @return {boolean} True if the height is modified after truncation, false otherwise
   * @private
   */
  _truncateSliceLabel: (pie, slice, labelInfo, isLeft) => {
    var sliceLabel = labelInfo.getSliceLabel();
    var style = sliceLabel.getCSSStyle();
    var maxLabelWidth = 0;
    var tmDimPt;

    // before setting the label displayable, make sure it is added to the DOM
    // necessary because the displayable will try to wrap, and to do that,
    // it needs to get the elements dimensions, which it can only do if it's
    // added to the DOM
    var numChildren = pie.getNumChildren();
    var removeTextArea = false;
    if (!pie.contains(sliceLabel)) {
      pie.addChild(sliceLabel);
      removeTextArea = true;
    }

    sliceLabel.setCSSStyle(style);
    var labelStr = slice.getSliceLabelString();
    sliceLabel.setTextString(labelStr);

    if (removeTextArea) {
      pie.removeChildAt(numChildren);
    }

    maxLabelWidth = DvtChartPieLabelUtils.getMaxLabelWidth(pie, labelInfo.getX(), isLeft);

    // truncates with larger space
    tmDimPt = DvtChartPieLabelUtils._getTextDim(
      pie,
      slice,
      sliceLabel,
      maxLabelWidth,
      labelInfo.getInitialNumLines()
    );

    // Update labelinfo
    labelInfo.setWidth(tmDimPt.x);

    if (labelInfo.getHeight() != tmDimPt.y) {
      labelInfo.setHeight(tmDimPt.y); // new height
      return true;
    }

    return false;
  },

  /**
   * Create initial layout, placing each label in its ideal location. Locations will be subsequently updated
   * to account for collisions
   * @param {DvtChartPie} pie
   * @return {Array}  An array with two elements. The first element is an array of DvtChartPieLabelInfo objects for the
   *                  labels on the left side of the pie.  The second element is an array of DvtChartPieLabelInfo objects
   *                  for the labels on the right side of the pie.
   * @private
   */
  _generateInitialLayout: (pie) => {
    var arArrays = [];
    var leftLabels = [];
    var rightLabels = [];

    var slices = pie.__getSlices();
    var frame = pie.__getFrame();

    for (var i = 0; i < slices.length; i++) {
      var slice = slices[i];
      // Only doing layout for outside labels, so skip any labels that have a position of none or inside, or was already positioned inside in auto layout
      var labelPosition = pie.getSeriesLabelPos(slice.getSeriesIndex());
      if (
        slice.getSliceLabel() != null ||
        labelPosition == 'none' ||
        labelPosition == 'center' ||
        DvtChartPieLabelUtils._skipSliceLabel(pie, slice)
      )
        continue;

      var s_label = DvtChartPieLabelUtils._createLabel(slice, false);

      var ma = slice.getAngleStart() + slice.getAngleExtent() / 2;
      if (ma > 360) ma -= 360;
      if (ma < 0) ma += 360;

      var labelPt = DvtChartPieRenderUtils.reflectAngleOverYAxis(
        ma,
        pie.getCenter().x,
        pie.getCenter().y,
        pie.getRadiusX() * DvtChartPieLabelUtils._OUTSIDE_LABEL_DISTANCE,
        pie.getRadiusY() * DvtChartPieLabelUtils._OUTSIDE_LABEL_DISTANCE
      );

      var isLeftSide = ma >= 90 && ma < 270;
      var maxLabelWidth = DvtChartPieLabelUtils.getMaxLabelWidth(pie, labelPt.x, isLeftSide);

      var tmDimPt = DvtChartPieLabelUtils._getTextDim(
        pie,
        slice,
        s_label,
        maxLabelWidth,
        DvtChartPieLabelUtils._MAX_LINES_PER_LABEL
      ); // set up for word wrap
      var midArea = 15;

      if (ma < 180 - midArea && ma > midArea) {
        //upper half
        labelPt.y -= tmDimPt.y * 1;
      } else if (ma < midArea || ma > 360 - midArea) {
        //right side, near horizontal
        labelPt.y -= tmDimPt.y * 0.5;
        labelPt.x += tmDimPt.y * 0.2;
      } else if (ma > 180 - midArea && ma < 180 + midArea) {
        //left side, near horizontal
        labelPt.y -= tmDimPt.y * 0.5;
        labelPt.x -= tmDimPt.y * 0.2;
      }

      var tilt = DvtChartPieLabelUtils._adjustForDepth(ma, pie.getDepth());
      labelPt.y += tilt;

      if (slices.length == 1)
        // only 1 label
        labelPt.x -= tmDimPt.x / 2; //position the label at the center

      if (labelPt.y < frame.y || labelPt.y + tmDimPt.y > frame.y + frame.h)
        // label will not fit with appropriate spacing
        continue;

      var pa;
      if (ma >= 90.0 && ma < 270.0) {
        // left side
        // right align
        s_label.alignRight();
        //        s_label.alignTop();  // alignTop impl buggy - too much interline space in FF
        // normalize from 0 to 180
        pa = ma - 90.0;
        DvtChartPieLabelUtils._createLabelInfo(
          slice,
          s_label,
          ma,
          pa,
          tmDimPt,
          labelPt,
          leftLabels
        );
      } else {
        // right side
        // normalize from 0 to 180
        pa = ma <= 90.0 ? Math.abs(90 - ma) : 180 - (ma - 270);
        DvtChartPieLabelUtils._createLabelInfo(
          slice,
          s_label,
          ma,
          pa,
          tmDimPt,
          labelPt,
          rightLabels
        );
      }
    }

    arArrays[0] = leftLabels;
    arArrays[1] = rightLabels;

    return arArrays;
  },
  /**
   * Create the DvtChartPieLabelInfo property bag object for a given slice and inserts it into labelInfoArray,
   * it its properly sorted position (where top-most labels are at the start of the array)
   *
   * @param {DvtChartPieSlice} slice
   * @param {dvt.OutputText|dvt.MultilineText} sliceLabel  The physical label we will associate with thie DvtChartPieLabelInfo. This
   label will be the one eventually associated with the input slice, if this
  label gets rendered
  * @param {number} ma The angle for the feeler line, with 0 degrees being the standard
  *                    0 degrees in the trigonometric sense (3 o'clock position)
  * @param {number} pa The normalized midpoint angle, where 0 degrees is at 12 o'clock
  *                    and angular measures are degrees away from 12 o'clock (so 90 degrees
  *                    can be either at 3 o'clock or 9 o'clock on the unit circle. Used to order slice
  *                    labels from top to bottom
  * @param {object} tmDimPt Object representing the width and height of the slice label
  * @param {object} labelPt The outside endpoint of the feeler line
  * @param {Array} labelInfoArray Where we store the newly created DvtChartPieLabelInfo
  * @private
  */
  _createLabelInfo: (slice, sliceLabel, ma, pa, tmDimPt, labelPt, labelInfoArray) => {
    // method carefully refactored from the end of PieChart.prepareLabels
    var insertPos = -1;
    var labelInfo;
    var s_label = sliceLabel;

    // insertion "sort"
    for (var j = 0; j < labelInfoArray.length; j++) {
      labelInfo = labelInfoArray[j];
      if (labelInfo.getPosition() > pa) {
        insertPos = j;
        break;
      }
    }

    if (insertPos == -1) insertPos = labelInfoArray.length;

    labelInfo = new DvtChartPieLabelInfo();

    labelInfo.setPosition(pa);
    labelInfo.setAngle(ma);
    labelInfo.setSliceLabel(s_label);
    labelInfo.setSlice(slice);
    labelInfo.setWidth(tmDimPt.x);
    labelInfo.setHeight(tmDimPt.y);
    labelInfo.setX(labelPt.x);
    labelInfo.setY(labelPt.y);
    labelInfo.setInitialNumLines(s_label.getLineCount());

    labelInfoArray.splice(insertPos, 0, labelInfo);
  },
  /**
   *
   * Wraps and truncates the text in the pieLabel, and returns a pt describing the new dimensions
   * @param {DvtChartPie} pieChart
   * @param {DvtChartPieSlice} slice
   * @param {dvt.MultilineText} sliceLabel the text instance to wrap and truncate
   * @param {Number} maxWidth the maxWidth of a line
   * @param {Number} maxLines the maximum number of lines to wrap, after which the rest of the text is truncated
   * @return {object} a point describing the new dimensions
   * @private
   */
  _getTextDim: (pieChart, slice, sliceLabel, maxWidth, maxLines) => {
    // Truncate and wrap the text to fit in the available space
    sliceLabel.setMaxLines(maxLines);
    var minChars = !DvtChartPieLabelUtils._isTextLabel(pieChart, slice)
      ? sliceLabel.getTextString().length
      : null;
    if (TextUtils.fitText(sliceLabel, maxWidth, Infinity, pieChart, minChars)) {
      // Add the label to the DOM to get dimensions
      pieChart.addChild(sliceLabel);
      var dimensions = sliceLabel.getDimensions();
      pieChart.removeChild(sliceLabel);
      return { x: dimensions.w, y: dimensions.h };
    }
    // It doesn't fit. return dimensions of 0x0
    return { x: 0, y: 0 };
  },
  /**
   * Checks whether the label for the given slice is text or numeric. Always returns true if a dataLabel function is specified, because the dataLabel could be returning labels that are all strings or all numbers or or a mixture of both.
   * @param {DvtChartPie} pie
   * @param {DvtChartPieSlice} slice
   * @return {boolean} whether the label associated with the slice is text. Returns true if a   dataLabel function is specified.
   * @private
   */
  _isTextLabel: (pie, slice) => {
    var customLabel = slice.getCustomLabel();
    var hasDataLabelFunc = pie.getOptions()['dataLabel'] != null;
    return (
      pie.getOptions()['styleDefaults']['sliceLabelType'].indexOf('text') != -1 ||
      (customLabel != null && typeof customLabel != 'number') ||
      hasDataLabelFunc
    );
  },
  /**
   *  Returns the maximum label length
   *  @param {DvtChartPie} pie
   *  @param {number} labelX the x point of the label
   *  @param {boolean} isLeftSide Indicates if the label is on left side of the pie chart
   *  @return {number} the maximum label length
   */
  getMaxLabelWidth: (pie, labelX, isLeftSide) => {
    var frame = pie.__getFrame();
    return isLeftSide ? labelX - frame.x : frame.x + frame.w - labelX;
  },
  /**
   *  Returns the pie center option
   * @param {DvtChartPie} pieChart the pie chart
   *  @param {object} options The options object
   *  @return {object} pieCenter object
   */
  getPieCenterOptions: (pieChart, options) => {
    var pieCenter = JsonUtils.clone(
      options['pieCenter'],
      null,
      pieChart.chart.Defaults.getNoCloneObject().pieCenter
    );
    var deprecatedPieCenter = options['pieCenterLabel'];
    if (deprecatedPieCenter) {
      var style = deprecatedPieCenter['style'];
      var text = deprecatedPieCenter['text'];
      if (text) pieCenter['label'] = text;
      if (style) pieCenter['labelStyle'] = new CSSStyle(style);
    }
    return pieCenter;
  },
  /**
   *  Returns true if slice label should be skipped. Currently we do this when the slice label is small on a pie with a large number of slices.
   *  @param {DvtChartPie} pie
   *  @param {DvtPieSlice} slice
   *  @return {boolean} true if slice label should be skipped, false otherwise
   *  @private
   */
  _skipSliceLabel: (pie, slice) => {
    return slice.getAngleExtent() < 3 && DvtChartDataUtils.getSeriesCount(pie.chart) > 120;
  }
};

/*---------------------------------------------------------------------*/
/*   DvtChartPieSlice                                                       */
/*---------------------------------------------------------------------*/

/*
 * Call chain:
 *
 * DvtChartPie creates each logical DvtChartPieSlice object.  The physical surface objects are
 * then created in DvtChartPie.render, by calling DvtChartPieSlice.preRender()
 *
 * In DvtChartPieSlice.preRender() we
 *
 * 1. setup the gradient used for this DvtChartPieSlice
 * 2. create the physical objects representing each surface
 *
 * The labels are then created and laid out by DvtSliceLabelLayout.layoutLabelsAndFeelers.
 *
 * After the label layout is complete, DvtChartPie then calls
 *
 * 1. render() to render the pie slice itself
 * 2. renderLabelAndFeeler() to render the pie label and feeler (if necessary)
 *
 */

/**
 * Creates an instance of DvtChartPieSlice
 *
 * @param {DvtChartPie} pieChart The pie chart that owns the pie slice.
 * @param {number=} seriesIndex The series index of this slice. If not provided, the slice is an "Other" slice.
 * @class DvtChartPieSlice
 * @constructor
 * @implements {DvtLogicalObject}
 * @implements {DvtCategoricalObject}
 * @implements {DvtTooltipSource}
 * @implements {DvtDraggable}
 */
class DvtChartPieSlice {
  /**
   * Object initializer
   * @param {DvtChartPie} pieChart The pie chart that owns the pie slice.
   * @param {number=} seriesIndex The series index of this slice. If not provided, the slice is an "Other" slice.
   * @protected
   */
  constructor(pieChart, seriesIndex) {
    this._pieChart = pieChart;
    this._chart = pieChart.chart;

    this._angleStart = 0;
    this._angleExtent = 0;

    this._topSurface = null; // an array of DvtShapes representing the top of the slice
    this._leftSurface = null; // an array of DvtShapes representing the left side of the slice
    // ("left" as seen from the tip of the slice)
    this._rightSurface = null; // an array of DvtShapes representing the right side of the slice
    // ("right" as seen from the tip of the slice)
    this._crustSurface = null; // an array of DvtShapes representing the crust of the slice

    this._explodeOffsetX = 0;
    this._explodeOffsetY = 0;

    this._sliceLabel = null;
    this._sliceLabelString = null;

    this._hasFeeler = false;
    this._feelerRad = null; // the section of the feeler closest to the pie
    this._feelerHoriz = null; // the section of the feeler closest to the label
    this._outsideFeelerStart = null; // a point class with x and y fields. This represents the point on the pie
    // from which the feeler originates in the unexploded state
    this._outsideFeelerMid = null; // a point class with x and y fields. This represents the point on the pie
    // from which the feeler bends
    this._outsideFeelerEnd = null; // a point class with x and y fields. This represents the point not on the pie
    // at which the feeler ends
    this._selected = false;
    this._selecting = false;

    this._centerX = this._pieChart.getCenter().x;
    this._centerY = this._pieChart.getCenter().y;

    this._radiusX = this._pieChart.getRadiusX();
    this._radiusY = this._pieChart.getRadiusY();

    this._depth = this._pieChart.getDepth();

    // Set rendering constants
    var options = this._chart.getOptions();

    if (seriesIndex != null) {
      // not "Other" slice
      var dataItem = DvtChartDataUtils.getDataItem(this._chart, seriesIndex, 0);
      this._value = DvtChartDataUtils.getVal(this._chart, seriesIndex, 0);
      this._explode = DvtChartPieUtils.getSliceExplode(this._chart, seriesIndex);
      this._fillColor = DvtChartStyleUtils.getColor(this._chart, seriesIndex, 0);
      this._fillPattern = DvtChartStyleUtils.getPattern(this._chart, seriesIndex, 0);
      this._strokeColor = DvtChartStyleUtils.getBorderColor(this._chart, seriesIndex);
      this._borderWidth = DvtChartStyleUtils.getBorderWidth(this._chart, seriesIndex);
      this._customLabel = dataItem ? dataItem['label'] : null;
      this._seriesLabel = DvtChartDataUtils.getSeries(this._chart, seriesIndex);
      this._drillable = DvtChartDataUtils.isDataItemDrillable(this._chart, seriesIndex, 0);
      this._id = DvtChartPieUtils.getSliceId(this._chart, seriesIndex);
      this._seriesIndex = seriesIndex;
      this._categories = DvtChartDataUtils.getCategories(this._chart, seriesIndex, 0);
    } else {
      // "Other" slice
      this._value = DvtChartPieUtils.getOtherVal(this._chart);
      this._explode = 0;
      this._fillColor = options['styleDefaults']['otherColor'];
      this._fillPattern = null;
      this._strokeColor = options['styleDefaults']['borderColor'];
      this._borderWidth = options['styleDefaults']['borderWidth'];
      this._customLabel = null;
      this._seriesLabel = options.translations.labelOther;
      this._drillable = DvtChartDataUtils.isMultiSeriesDrillEnabled(this._chart);
      this._id = DvtChartPieUtils.getOtherSliceId(this._chart);
      this._otherId = true;
    }
  }

  /**
   * Returns the owning DvtChartPie object.
   * @return {DvtChartPie}
   */
  getPieChart() {
    return this._pieChart;
  }

  /**
   * Render the pie slice only; rendering of label and feeler
   * occurs in DvtChartPieSlice.renderLabelAndFeelers
   * @param {boolean=} duringDisplayAnim Whether the render is during the display animation.
   */
  render(duringDisplayAnim) {
    var sortedSurfaces = DvtChartPieSlice._sortPieSurfaces(
      this._topSurface,
      this._leftSurface,
      this._rightSurface,
      this._crustSurface,
      this._angleStart,
      this._angleExtent
    );
    var len = sortedSurfaces.length;
    for (var i = 0; i < len; i++) {
      var shapeArray = sortedSurfaces[i];
      // shapeArray is an array of DvtShapes representing the given surface.
      // Iterate through this array and add each shape to the pieChart
      var shapeCount = shapeArray.length;
      for (var j = 0; j < shapeCount; j++) {
        var shapesContainer = this._pieChart.__getShapesContainer();
        shapesContainer.addChild(shapeArray[j]);
        if (shapeArray[j].render)
          // render is only defined on certain shape subclasses
          shapeArray[j].render();
      }
    }

    // Render label and feeler
    // assume that all layout and text truncation has already been done
    // so in theory, we just need to call addChild with the feeler and label
    if (this._sliceLabel) {
      this._pieChart.addChild(this._sliceLabel);

      // Associate the shapes with the slice for use during event handling
      DvtChartPieRenderUtils.associate(this, [this._sliceLabel]);

      if (duringDisplayAnim) {
        // Reuse the existing feeler instead of creating a new one for fade in animation
        this._pieChart.addChild(this._feelerRad);
        this._pieChart.addChild(this._feelerHoriz);
      } else this._renderOutsideFeeler();
    }

    // Perform initial explosion
    this._explodeSlice();

    // Apply the correct cursor if drilling or selection is enabled
    if (this._drillable || this._pieChart.chart.isSelectionSupported()) {
      var sliceDisplayables = this.getDisplayables();
      for (var i = 0; i < sliceDisplayables.length; i++) {
        sliceDisplayables[i].setCursor(SelectionEffectUtils.getSelectingCursor());
      }
    }

    // WAI-ARIA
    var displayable = this.getTopDisplayable();

    if (displayable) {
      displayable.setAriaRole('img');
      this._updateAriaLabel();
    }
  }

  /**
   * Create a feeler from pt1 to pt2.
   * @param {dvt.Point} pt1
   * @param {dvt.Point} pt2
   * @return {dvt.Line} feeler
   * @private
   */
  _feelerFromPts(pt1, pt2) {
    var feeler = new Line(this._pieChart.getCtx(), pt1.x, pt1.y, pt2.x, pt2.y);
    var color = this._pieChart.getOptions()['styleDefaults']['pieFeelerColor'];
    var stroke = new Stroke(color);
    feeler.setStroke(stroke);
    this._pieChart.addChild(feeler);
    return feeler;
  }

  /**
   * Render a feeler outside the slice.
   * @private
   */
  _renderOutsideFeeler() {
    if (!this._hasFeeler) return;

    var feelerRad = this._feelerFromPts(this._outsideFeelerStart, this._outsideFeelerMid);
    var feelerHoriz = this._feelerFromPts(this._outsideFeelerMid, this._outsideFeelerEnd);

    // Store a reference to it so that we can remove
    this._feelerRad = feelerRad;
    this._feelerHoriz = feelerHoriz;
  }

  /**
   * Creates the gradients and physical shapes for the pie surfaces. Z-Ordering of the shapes
   * and layout and creation of the pie label is done elsewhere.
   */
  preRender() {
    var fillType = this._bFillerSlice ? 'color' : DvtChartStyleUtils.getSeriesEffect(this._chart);
    var color = this.getFillColor();
    var pattern = this.getFillPattern();

    // Create the fills
    var topFill;
    if (fillType == 'pattern' || pattern != null) {
      topFill = new PatternFill(pattern, color);
      fillType = 'pattern';
    } else if (fillType == 'gradient') {
      var grAngle = 270;
      var style = !this._pieChart.is3D() ? '2D' : '3D';

      var arColors = DvtChartPieRenderUtils.getGradientColors(ColorUtils.getRGB(color), style);
      var arAlphas = DvtChartPieRenderUtils.getGradientAlphas(
        ColorUtils.getAlpha(color),
        style
      );
      var arRatios = [0, 1.0];
      var arBounds = [
        Math.floor(this._centerX - this._radiusX),
        Math.floor(this._centerY - this._radiusY),
        Math.ceil(2 * this._radiusX),
        Math.ceil(2 * this._radiusY)
      ];

      topFill = new LinearGradientFill(grAngle, arColors, arAlphas, arRatios, arBounds);
    } else topFill = new SolidFill(color);

    // Create the Top Surface
    this._topSurface = DvtChartPieRenderUtils.createTopSurface(this, topFill);

    // 3D Effect Support
    if (this._depth > 0 || this._radiusX != this._radiusY) {
      var useGradientFill = fillType == 'gradient';
      var lateralFill = new SolidFill(ColorUtils.getDarker(color, 0.4));
      var sideFill = useGradientFill
        ? DvtChartPieRenderUtils.generateLateralGradientFill(this, 'SIDE')
        : lateralFill;
      var crustFill = useGradientFill
        ? DvtChartPieRenderUtils.generateLateralGradientFill(this, 'CRUST')
        : lateralFill;

      // Create the side surfaces for the slice, which will be sorted later
      this._leftSurface = DvtChartPieRenderUtils.createLateralSurface(
        this,
        DvtChartPieRenderUtils.SURFACE_LEFT,
        sideFill
      );
      this._rightSurface = DvtChartPieRenderUtils.createLateralSurface(
        this,
        DvtChartPieRenderUtils.SURFACE_RIGHT,
        sideFill
      );
      this._crustSurface = DvtChartPieRenderUtils.createLateralSurface(
        this,
        DvtChartPieRenderUtils.SURFACE_CRUST,
        crustFill
      );
    }

    // Clear slice label and feelers from previous render
    this.setSliceLabel(null);
    this.setNoOutsideFeeler();
  }

  // This logic is ported over from PieChart.sortSliversBySlice, and pushed
  // from the PieChart to the PieSlice
  /**
   * Sorts this DvtChartPieSlice's surfaces by z-order
   *
   * @param {Array} topSurface An array of DvtShapes representing the top of this DvtChartPieSlice
   * @param {Array} leftSurface An array of DvtShapes representing the left side of this DvtChartPieSlice (as seen from
                    the tip of the slice)
   * @param {Array} rightSurface An array of DvtShapes representing the right side of this DvtChartPieSlice (as seen from
                    the tip of the slice)
   * @param {Array} crustSurface An array of DvtShapes representing the crust of this DvtChartPieSlice
   * @param {number} angleStart The starting position of this pie slice
   * @param {number} angleExtent The angular size of this pie slice
   *
   * @return {Array} A sorted array of arrays (two-dimensional array) of dvt.Shape objects for this slice to render.
   * @private
   */
  static _sortPieSurfaces(
    topSurface,
    leftSurface,
    rightSurface,
    crustSurface,
    angleStart,
    angleExtent
  ) {
    var sortedSurfaces = [];

    if (leftSurface && rightSurface && crustSurface) {
      // ported from PieChart.sortSliversBySlice
      // NOTE that instead of relying on the order in which the surfaces were
      // originally created, we just get them from associative array by name
      // the last slice to render,
      // or if a slice starts at 270 degrees/6 o'clock (Fix for )
      if (angleStart <= 270 && angleStart + angleExtent > 270) {
        //we're in the bottom-most slice, so add surfaces in back-to-front z-order:
        //left edge, right edge, crust
        sortedSurfaces.push(leftSurface);
        sortedSurfaces.push(rightSurface);
        sortedSurfaces.push(crustSurface);
      }

      // right-side of the pie
      else if (angleStart > 270 || angleStart + angleExtent <= 90) {
        //we're in the right side of the pie, so add surfaces in back-to-front z-order:
        //left edge, crust, right edge
        sortedSurfaces.push(leftSurface);
        sortedSurfaces.push(crustSurface);
        sortedSurfaces.push(rightSurface);
      } else {
        //we're in the left side of the pie, so add surfaces in back-to-front z-order:
        //right edge, crust, left edge
        sortedSurfaces.push(rightSurface);
        sortedSurfaces.push(crustSurface);
        sortedSurfaces.push(leftSurface);
      }
    }

    // top is rendered last
    sortedSurfaces.push(topSurface);

    return sortedSurfaces;
  }

  /**
   * Returns true if (x-y1) and (x-y2) have different signs.
   * @param {number} x
   * @param {number} y1
   * @param {number} y2
   * @return {boolean}
   */
  static oppositeDir(x, y1, y2) {
    var positive1 = x - y1 > 0;
    var positive2 = x - y2 > 0;
    return positive1 != positive2;
  }

  /**
   * Explodes the DvtChartPieSlice and feeler line.
   * @private
   */
  _explodeSlice() {
    if (this._explode != 0) {
      var arc = this._angleExtent;
      var angle = this._angleStart;
      var fAngle = angle + arc / 2;
      var radian = (360 - fAngle) * Math$1.RADS_PER_DEGREE;
      var tilt = this._pieChart.is3D() ? DvtChartPieSlice.THREED_TILT : 1;

      var explodeOffset = this._explode * this._pieChart.__calcMaxExplodeDistance();
      this._explodeOffsetX = Math.cos(radian) * explodeOffset;
      this._explodeOffsetY = Math.sin(radian) * tilt * explodeOffset;

      // To work around , in the 2D pie case, we need to poke the
      // DOM element that contains the shadow filter that is applied to the pie slices.
      // However, due to , just poking the DOM also causes jitter in the
      // slice animation.  To get rid of the jitter, we round the amount of the translation we
      // apply to the pie slice and we also shorten the duration of the animation to visually smooth
      // out the result of the rounding.
      // 
      if (Agent.browser === 'safari' || Agent.engine === 'blink') {
        this._explodeOffsetX = Math.round(this._explodeOffsetX);
        this._explodeOffsetY = Math.round(this._explodeOffsetY);
      }
    } else {
      this._explodeOffsetX = 0;
      this._explodeOffsetY = 0;
    }

    // now update each surface
    if (this._topSurface) {
      var offsets =
        this._pieChart.is3D() && this._topSurface[0].getSelectionOffset
          ? this._topSurface[0].getSelectionOffset()
          : [];
      DvtChartPieSlice._translateShapes(
        this._topSurface,
        offsets[0] ? offsets[0] + this._explodeOffsetX : this._explodeOffsetX,
        offsets[1] ? offsets[1] + this._explodeOffsetY : this._explodeOffsetY
      );
    }

    if (this._rightSurface) {
      DvtChartPieSlice._translateShapes(
        this._rightSurface,
        this._explodeOffsetX,
        this._explodeOffsetY
      );
    }

    if (this._leftSurface) {
      DvtChartPieSlice._translateShapes(
        this._leftSurface,
        this._explodeOffsetX,
        this._explodeOffsetY
      );
    }

    if (this._crustSurface) {
      DvtChartPieSlice._translateShapes(
        this._crustSurface,
        this._explodeOffsetX,
        this._explodeOffsetY
      );
    }

    // update the feeler line
    if (this._hasFeeler) {
      // get current starting x and y, and then update the feeler line only
      var oldStartX = this._outsideFeelerStart.x;
      var oldStartY = this._outsideFeelerStart.y;

      var newStartX = oldStartX + this._explodeOffsetX;
      var newStartY = oldStartY + this._explodeOffsetY;

      this._feelerRad.setX1(newStartX);
      this._feelerRad.setY1(newStartY);

      var oldMidX = this._outsideFeelerMid.x;
      var oldMidY = this._outsideFeelerMid.y;

      //The midpoint of the feeler has to be updated if the new radial feeler is pointing towards an opposite direction;
      //otherwise, the feeler will go through the slice.
      //The easy solution is to set the x/y of the midPt to be the same as startPt.
      if (DvtChartPieSlice.oppositeDir(oldMidX, oldStartX, newStartX)) {
        this._feelerRad.setX2(newStartX);
        this._feelerHoriz.setX1(newStartX);
      } else {
        this._feelerRad.setX2(oldMidX);
        this._feelerHoriz.setX1(oldMidX);
      }

      if (DvtChartPieSlice.oppositeDir(oldMidY, oldStartY, newStartY)) {
        this._feelerRad.setY2(newStartY);
        this._feelerHoriz.setY1(newStartY);
      } else {
        this._feelerRad.setY2(oldMidY);
        this._feelerHoriz.setY1(oldMidY);
      }
    }

    //update the label position
    if (this._sliceLabel && !this._hasFeeler) {
      this._sliceLabel.setTranslate(this._explodeOffsetX, this._explodeOffsetY);
    }
  }

  /**
   * Translates each element in an array of shapes by the same delta x and delta y
   *
   * @param {Array} shapes An array of dvt.Shape objects to translate
   * @param {number} tx
   * @param {number} ty
   *
   * @private
   */
  static _translateShapes(shapes, tx, ty) {
    if (!shapes) return;

    var len = shapes.length;

    for (var i = 0; i < len; i++) {
      var shape = shapes[i];
      shape.setTranslate(tx, ty);
    }
  }

  /** Getters and setters **/

  /**
   * Returns the x radius of the slice.
   * @return {number}
   */
  getRadiusX() {
    return this._radiusX;
  }

  /**
   * Returns the y radius of the slice.
   * @return {number}
   */
  getRadiusY() {
    return this._radiusY;
  }

  /**
   * Returns the center of the pie. This is used to animate between pies with different center points.
   * @return {dvt.Point}
   */
  getCenter() {
    return new Point(this._centerX, this._centerY);
  }

  /**
   * Returns the depth of the pie. This is used to animate between pies with and without 3d effect.
   * @return {number}
   */
  getDepth() {
    return this._depth;
  }

  /**
   * @return {number} The size of this pie slice's angle
   */
  getAngleExtent() {
    return this._angleExtent;
  }

  /**
   * @param {number} extent The size of this pie slice's angle
   */
  setAngleExtent(extent) {
    this._angleExtent = extent;
  }

  /**
   * @return {number} The starting angle location of this pie slice
   */
  getAngleStart() {
    return this._angleStart;
  }

  /**
   * @param {number} start The starting angle location of this pie slice
   */
  setAngleStart(start) {
    this._angleStart = start;
  }

  /**
   * @return {number} The x-offset for this pie slice. Zero if the slice is not exploded.
   */
  __getExplodeOffsetX() {
    return this._explodeOffsetX;
  }

  /**
   * @return {number} The y-offset for this pie slice. Zero if the slice is not exploded.
   */
  __getExplodeOffsetY() {
    return this._explodeOffsetY;
  }

  /**
   * Set the points for a feeler outside the slice.
   *
   * @param {object} startPt The starting point of the feeler, located on the pie slice. Point has an x and y field
   * @param {object} midPt The mid point of the feeler, located between the slice and the label. Point has an x and y field
   * @param {object} endPt The ending point of the feeler, located on the pie label. Point has an x and y field
   */
  setOutsideFeelerPoints(startPt, midPt, endPt) {
    this._outsideFeelerStart = startPt;
    this._outsideFeelerMid = midPt;
    this._outsideFeelerEnd = endPt;
    this._hasFeeler = true;
  }

  /**
   * Set the slice without feeler.
   */
  setNoOutsideFeeler() {
    this._outsideFeelerStart = null;
    this._outsideFeelerMid = null;
    this._outsideFeelerEnd = null;
    this._hasFeeler = false;
  }

  /**
   * Returns an array containing the label and feeler objects of the slice.
   * @return {array}
   */
  getLabelAndFeeler() {
    var ar = [];
    if (this._sliceLabel) ar.push(this._sliceLabel);
    if (this._feelerRad) ar.push(this._feelerRad);
    if (this._feelerHoriz) ar.push(this._feelerHoriz);
    return ar;
  }

  /**
   * @return {dvt.OutputText|dvt.MultilineText} The label for this slice
   */
  getSliceLabel() {
    return this._sliceLabel;
  }

  /**
   * @param {dvt.OutputText|dvt.MultilineText} sliceLabel
   */
  setSliceLabel(sliceLabel) {
    this._sliceLabel = sliceLabel;
  }

  /**
   * @return {String} Untruncated slice label if slice label is truncated.
   */
  getSliceLabelString() {
    return this._sliceLabelString;
  }

  /**
   * @param {String} labelStr Untruncated slice label if slice label is truncated.
   */
  setSliceLabelString(labelStr) {
    this._sliceLabelString = labelStr;
  }

  /**
   * @return {Array} The top surface displayables of this Pie Slice
   */
  getTopSurface() {
    return this._topSurface;
  }

  /**
   * Returns the numeric data value associated with this slice
   * @return {number}
   */
  getVal() {
    return this._value;
  }

  /**
   * @return {String} The series id
   */
  getId() {
    return this._id;
  }

  /**
   * @return {Number} The series index
   */
  getSeriesIndex() {
    return this._seriesIndex;
  }

  /**
   * Returns true if the specified displayable can be selected or hovered.
   * @param {dvt.Displayable} shape
   * @return {boolean}
   * @private
   */
  static _shapeIsSelectable(shape) {
    return shape instanceof DvtChartSelectableWedge;
  }

  /**
   * Returns true if this slice contains the given coordinates.
   * @param {number} x
   * @param {number} y
   * @return {boolean}
   */
  contains(x, y) {
    var ir = this._pieChart.getInnerRadius();
    var c = this._pieChart.getCenter();
    var cos = (x - c.x) / this._radiusX;
    var sin = (y - c.y) / this._radiusY;

    // Compute the angle
    var angle = -Math.atan2(sin, cos) * (180 / Math.PI); // in degrees
    // First adjust angle to be greater than the start angle.
    while (angle < this._angleStart) angle += 360;
    // Then adjust to be within 360 degrees of it
    while (angle - this._angleStart >= 360) angle -= 360;

    var distance = Math.pow(cos, 2) + Math.pow(sin, 2);
    var containsRadius = Math.sqrt(distance) > ir / this._radiusX && distance <= 1;
    var containsAngle = angle <= this._angleStart + this._angleExtent;
    return containsRadius && containsAngle;
  }

  //---------------------------------------------------------------------//
  // Animation Support                                                   //
  //---------------------------------------------------------------------//

  /**
   * @override
   */
  GetAnimParams() {
    var r = ColorUtils.getRed(this._fillColor);
    var g = ColorUtils.getGreen(this._fillColor);
    var b = ColorUtils.getBlue(this._fillColor);
    var a = ColorUtils.getAlpha(this._fillColor);
    return [
      this._value,
      this._radiusX,
      this._radiusY,
      this._explode,
      this._centerX,
      this._centerY,
      this._depth,
      r,
      g,
      b,
      a
    ];
  }

  /**
   * @override
   */
  SetAnimParams(params) {
    this._value = params[0];
    this._radiusX = params[1];
    this._radiusY = params[2];
    this._explode = params[3];
    this._centerX = params[4];
    this._centerY = params[5];
    this._depth = params[6];

    // Update the color.  Round them since color parts must be ints
    var r = Math.round(params[7]);
    var g = Math.round(params[8]);
    var b = Math.round(params[9]);
    var a = Math.round(params[10]);
    this._fillColor = ColorUtils.makeRGBA(r, g, b, a);
  }

  /**
   * Returns the animation params of a deleted slice.
   * @return {Array} animation params.
   */
  getDeletedAnimParams() {
    var params = this.GetAnimParams();
    params[0] = 0; // value
    params[1] = this.getInnerRadius(); // radiusX
    params[2] = this.getInnerRadius(); // radiusY
    params[3] = 0; // explode
    return params;
  }

  /**
   * Creates the update animation for this object.
   * @param {dvt.DataAnimationHandler} handler The animation handler, which can be used to chain animations.
   * @param {DvtChartPieSlice} oldSlice The old pie state to animate from.
   */
  animateUpdate(handler, oldSlice) {
    var startState = oldSlice.GetAnimParams();
    var endState = this.GetAnimParams();

    if (!ArrayUtils.equals(startState, endState)) {
      // Create the animation
      var anim = new CustomAnimation(
        this._pieChart.getCtx(),
        this,
        this.getPieChart().getAnimDuration()
      );
      anim
        .getAnimator()
        .addProp(
          Animator.TYPE_NUMBER_ARRAY,
          this,
          this.GetAnimParams,
          this.SetAnimParams,
          endState
        );
      handler.add(anim, 0);

      // Initialize to the start state
      this.SetAnimParams(startState);
    }
  }

  /**
   * Creates the insert animation for this object.
   * @param {dvt.DataAnimationHandler} handler The animation handler, which can be used to chain animations.
   */
  animateInsert(handler) {
    // Create the animation
    var anim = new CustomAnimation(
      this._pieChart.getCtx(),
      this,
      this.getPieChart().getAnimDuration()
    );
    anim
      .getAnimator()
      .addProp(
        Animator.TYPE_NUMBER_ARRAY,
        this,
        this.GetAnimParams,
        this.SetAnimParams,
        this.GetAnimParams()
      );
    handler.add(anim, 0);

    // Initialize to the start state
    this.SetAnimParams(this.getDeletedAnimParams());
  }

  /**
   * Creates the delete animation for this object.
   * @param {dvt.DataAnimationHandler} handler The animation handler, which can be used to chain animations.
   * @param {DvtChartPie} container The new container where the pie slice should be moved for animation.
   */
  animateDelete(handler, container) {
    var newSlices = container.__getSlices();
    var oldSlices = this.getPieChart().__getSlices();

    // Add the deleted slice to the new pie in the right spot
    var oldIndex = oldSlices.indexOf(this);
    var prevIndex = oldIndex - 1;
    if (prevIndex >= 0) {
      var prevId = oldSlices[prevIndex].getId();
      // Find the location of the previous slice
      for (var i = 0; i < newSlices.length; i++) {
        if (newSlices[i].getId().equals(prevId)) {
          newSlices.splice(i + 1, 0, this);
          break;
        }
      }
    } else newSlices.splice(0, 0, this);

    this._pieChart = container; // reparent this slice to the new pie

    // Create the animation to delete the slice
    var anim = new CustomAnimation(
      container.getCtx(),
      this,
      this.getPieChart().getAnimDuration()
    );
    anim
      .getAnimator()
      .addProp(
        Animator.TYPE_NUMBER_ARRAY,
        this,
        this.GetAnimParams,
        this.SetAnimParams,
        this.getDeletedAnimParams()
      );

    // Set the onEnd listener so that the slice can be deleted
    anim.setOnEnd(this._removeDeletedSlice, this);

    // Finally add the animation
    handler.add(anim, 0);
  }

  /**
   * Removes a deleted slice from the owning pie chart.  A re-render must be performed for the
   * results to be visible.
   * @private
   */
  _removeDeletedSlice() {
    var slices = this.getPieChart().__getSlices();
    var index = slices.indexOf(this);

    if (index >= 0) slices.splice(index, 1);
  }

  //---------------------------------------------------------------------//
  // DvtLogicalObject impl                                               //
  //---------------------------------------------------------------------//
  /**
   * @override
   */
  getDisplayables() {
    var ret = new Array();

    if (this._topSurface) ret = ret.concat(this._topSurface);

    if (this._leftSurface) ret = ret.concat(this._leftSurface);

    if (this._rightSurface) ret = ret.concat(this._rightSurface);

    if (this._crustSurface) ret = ret.concat(this._crustSurface);

    if (this._sliceLabel) ret.push(this._sliceLabel);

    if (this._feelerRad) ret.push(this._feelerRad);

    if (this._feelerHoriz) ret.push(this._feelerHoriz);

    return ret;
  }

  /**
   * @override
   */
  getAriaLabel() {
    var shortDesc;
    var translations = this._pieChart.getOptions().translations;
    if (this._seriesIndex == null)
      // other slice
      shortDesc = DvtChartTooltipUtils.getOtherSliceDatatip(this._chart, this._value, false);
    else
      shortDesc = DvtChartTooltipUtils.getDatatip(this._chart, this._seriesIndex, 0, null, false);

    // Include percentage in the shortDesc
    var percentageLabel = translations.labelPercentage;
    var percentage = DvtChartPieLabelUtils.generateSlicePercentageString(this);
    shortDesc +=
      '; ' + ResourceUtils.format(translations.labelAndValue, [percentageLabel, percentage]);

    var states = [];
    if (this.isSelectable())
      states.push(translations[this.isSelected() ? 'stateSelected' : 'stateUnselected']);

    if (DvtChartDataUtils.isDataItemDrillable(this._chart, this._seriesIndex, this._groupIndex))
      states.push(translations.stateDrillable);

    return Displayable.generateAriaLabel(shortDesc, states);
  }

  /**
   * Updates the aria-label as needed. On desktop, we can defer the aria creation, and the aria-label will be updated
   * when the activeElement is set.
   * @private
   */
  _updateAriaLabel() {
    var displayable = this.getTopDisplayable();
    if (displayable && !Agent.deferAriaCreation())
      displayable.setAriaProperty('label', this.getAriaLabel());
  }

  /**
   * Returns the displayable of the top surface.
   * @return {dvt.Shape}
   */
  getTopDisplayable() {
    if (this._topSurface && this._topSurface.length > 0) return this._topSurface[0];
    return null;
  }

  /**
   * @override
   */
  isSelectable() {
    return this._chart.isSelectionSupported();
  }

  /**
   * @override
   */
  isSelected() {
    return this._selected;
  }

  /**
   * @override
   */
  setSelected(bSelected, isInitial) {
    this._selected = bSelected;
    if (!this.getTopSurface()) {
      // Skip slices not rendered for performance optimization
      return;
    } else if (this._selected) {
      this._pieChart.bringToFrontOfSelection(this);
    } else if (!this._selecting) {
      this._pieChart.pushToBackOfSelection(this);
    }

    // Selection effect: Highlight
    if (DvtChartStyleUtils.isSelectionHighlighted(this._chart)) {
      var shapeArr = this.getDisplayables();
      for (var i = 0; i < shapeArr.length; i++) {
        if (DvtChartPieSlice._shapeIsSelectable(shapeArr[i])) {
          shapeArr[i].setSelected(bSelected);
        }
      }
    }

    // Selection effect: Explode
    if (DvtChartStyleUtils.isSelectionExploded(this._chart)) {
      var explode = bSelected ? 1 : 0;
      if (!isInitial && DvtChartStyleUtils.getAnimOnDataChange(this._chart) != 'none') {
        // animate the explosion
        var anim = new CustomAnimation(
          this._pieChart.getCtx(),
          this,
          this._pieChart.getAnimDuration() / 2
        );
        anim
          .getAnimator()
          .addProp(Animator.TYPE_NUMBER, this, this.getExplode, this.setExplode, explode);
        anim.play();
      } else this.setExplode(explode);
    }

    this._updateAriaLabel();
  }

  /**
   * @override
   */
  showHoverEffect() {
    this._selecting = true;
    this._pieChart.bringToFrontOfSelection(this);
    var shapeArr = this.getDisplayables();
    for (var i = 0; i < shapeArr.length; i++) {
      if (DvtChartPieSlice._shapeIsSelectable(shapeArr[i])) {
        shapeArr[i].showHoverEffect();
      }
    }
  }

  /**
   * @override
   */
  hideHoverEffect() {
    this._selecting = false;
    if (!this._selected) {
      this._pieChart.pushToBackOfSelection(this);
    }
    var shapeArr = this.getDisplayables();
    for (var i = 0; i < shapeArr.length; i++) {
      if (DvtChartPieSlice._shapeIsSelectable(shapeArr[i])) {
        shapeArr[i].hideHoverEffect();
      }
    }
  }

  //---------------------------------------------------------------------//
  // Tooltip Support: DvtTooltipSource impl                              //
  //---------------------------------------------------------------------//
  /**
   * @override
   */
  getDatatip(target, x, y) {
    if (target == this._sliceLabel) {
      if (this._sliceLabel && this._sliceLabel.isTruncated()) return this.getSliceLabelString();
    }
    return this.getTooltip();
  }

  /**
   * @override
   */
  getDatatipColor() {
    return this.getFillColor();
  }

  //---------------------------------------------------------------------//
  // Rollover and Hide/Show Support: DvtCategoricalObject impl           //
  //---------------------------------------------------------------------//
  /**
   * @override
   */
  getCategories() {
    if (this._categories && this._categories.length > 0) return this._categories;
    return [this.getId().series];
  }

  //---------------------------------------------------------------------//
  // Keyboard Support: DvtKeyboardNavigables impl                        //
  //---------------------------------------------------------------------//
  /**
   * @override
   */
  getNextNavigable(event) {
    var keyCode = event.keyCode;
    if (event.type == MouseEvent.CLICK) {
      return this;
    } else if (keyCode == KeyboardEvent.SPACE && event.ctrlKey) {
      // multi-select node with current focus; so we navigate to ourself and then let the selection handler take
      // care of the selection
      return this;
    }

    var rtl = Agent.isRightToLeft(this._chart.getCtx());
    var slices = this._pieChart.__getSlices();
    var idx = slices.indexOf(this);
    var next = null;

    if (
      keyCode == KeyboardEvent.RIGHT_ARROW ||
      (keyCode == KeyboardEvent.DOWN_ARROW && !rtl) ||
      (keyCode == KeyboardEvent.UP_ARROW && rtl)
    ) {
      if (idx < slices.length - 1) next = slices[idx + 1];
      else next = slices[0];
    } else if (
      keyCode == KeyboardEvent.LEFT_ARROW ||
      (keyCode == KeyboardEvent.DOWN_ARROW && rtl) ||
      (keyCode == KeyboardEvent.UP_ARROW && !rtl)
    ) {
      if (idx == 0) next = slices[slices.length - 1];
      else next = slices[idx - 1];
    }
    return next;
  }

  /**
   * @override
   */
  getKeyboardBoundingBox(targetCoordinateSpace) {
    var displayables = this.getDisplayables();
    if (displayables[0]) return displayables[0].getDimensions(targetCoordinateSpace);
    else return new Rectangle(0, 0, 0, 0);
  }

  /**
   * @override
   */
  getTargetElem() {
    var displayables = this.getDisplayables();
    if (displayables[0]) return displayables[0].getElem();
    return null;
  }

  /**
   * @override
   */
  showKeyboardFocusEffect() {
    this._isShowingKeyboardFocusEffect = true;
    this.showHoverEffect();
  }

  /**
   * @override
   */
  hideKeyboardFocusEffect() {
    this._isShowingKeyboardFocusEffect = false;
    this.hideHoverEffect();
  }

  /**
   * @override
   */
  isShowingKeyboardFocusEffect() {
    return this._isShowingKeyboardFocusEffect;
  }

  //---------------------------------------------------------------------//
  // DnD Support: DvtDraggable impl                                      //
  //---------------------------------------------------------------------//

  /**
   * @override
   */
  isDragAvailable() {
    return true;
  }

  /**
   * @override
   */
  getDragTransferable() {
    return [this.getId()];
  }

  /**
   * @override
   */
  getDragFeedback() {
    // If more than one object is selected, return the displayables of all selected objects
    if (
      this._chart.isSelectionSupported() &&
      this._chart.getSelectionHandler().getSelectedCount() > 1
    ) {
      var selection = this._chart.getSelectionHandler().getSelection();
      var displayables = [];
      for (var i = 0; i < selection.length; i++) {
        displayables = displayables.concat(selection[i].getDisplayables());
      }
      return displayables;
    }

    // Otherwise, return its own displayables
    return this.getDisplayables();
  }

  /**
   * Returns the current explode value for this pie slice
   * @return {number}
   */
  getExplode() {
    return this._explode;
  }

  /**
   * Sets the current explode value for this pie slice
   * @param {number} explode
   */
  setExplode(explode) {
    this._explode = explode;
    this._explodeSlice();
  }

  /**
   * Returns the user-defined label for this pie slice.
   * @return {string}
   */
  getCustomLabel() {
    return this._customLabel;
  }

  /**
   * Returns the default series label for this pie slice.
   * @return {string}
   */
  getSeriesLabel() {
    return this._seriesLabel;
  }

  /**
   * Returns the color of this pie slice, represented as a String
   * @return {String}
   */
  getFillColor() {
    return this._fillColor;
  }

  /**
   * Returns the name of the fill pattern for this pie slice
   * @return {string}
   */
  getFillPattern() {
    return this._fillPattern;
  }

  /**
   * Returns the color of this pie slice border
   * @return {String}
   */
  getStrokeColor() {
    return this._strokeColor;
  }

  /**
   * Returns the color of this pie slice border
   * @return {String}
   */
  getBorderWidth() {
    return this._borderWidth;
  }

  /**
   * Returns the slice gaps
   * @return {Number}
   */
  getSliceGaps() {
    // Slice gap is only supported if the pie is not tilted (depth = 0)
    if (this._depth == 0) return 3 * DvtChartStyleUtils.getDataItemGaps(this._chart);
    else return 0;
  }

  /**
   * Returns the inner radius
   * @return {Number}
   */
  getInnerRadius() {
    return this._pieChart.getInnerRadius();
  }

  /**
   * Returns the tooltip string associated with this slice
   * @return {String}
   */
  getTooltip() {
    if (this._seriesIndex == null)
      // other slice
      return DvtChartTooltipUtils.getOtherSliceDatatip(this._chart, this._value, true);

    return DvtChartTooltipUtils.getDatatip(this._chart, this._seriesIndex, 0, null, true);
  }

  /**
   * Returns whether the pie slice is drillable.
   * @return {boolean}
   */
  isDrillable() {
    return this._drillable;
  }

  /**
   * Creates a filler slice (for fan effect in display animation).
   * @param {DvtChartPie} pieChart
   * @param {number} value The value of the filler slice.
   * @return {DvtChartPieSlice} filler slice.
   */
  static createFillerSlice(pieChart, value) {
    var slice = new DvtChartPieSlice(pieChart);
    slice._value = value;
    slice._bFillerSlice = true;
    slice._centerX = pieChart.getCenter().x;
    slice._centerY = pieChart.getCenter().y;
    slice._fillColor = 'rgba(255,255,255,0)';
    slice._strokeColor = 'rgba(255,255,255,0)';
    slice._id = new DvtChartDataItem(null, null, null, null);
    return slice;
  }

  /**
   * Returns the seriesIndex of the slice
   * @return {Number}
   */
  getSeriesIndex() {
    return this._seriesIndex;
  }
}
/** @const */
DvtChartPieSlice.THREED_TILT = 0.59;

/**
 *  Provides automation services for a DVT component.
 *  @class DvtChartAutomation
 *  @param {Chart} dvtComponent
 *  @implements {dvt.Automation}
 *  @constructor
 */
class DvtChartAutomation extends Automation {
  constructor(dvtComponent) {
    super(dvtComponent);
    this._options = this._comp.getOptions();
    this._legend = this._comp.legend;
    this._xAxis = this._comp.xAxis;
    this._yAxis = this._comp.yAxis;
    this._y2Axis = this._comp.y2Axis;

    this._legendAutomation = this._legend ? this._legend.getAutomation() : null;
    this._xAxisAutomation = this._xAxis ? this._xAxis.getAutomation() : null;
    this._yAxisAutomation = this._yAxis ? this._yAxis.getAutomation() : null;
    this._y2AxisAutomation = this._y2Axis ? this._y2Axis.getAutomation() : null;
  }
  /**
   * Valid subIds inlcude:
   * <ul>
   * <li>dataItem[seriesIndex][itemIndex]</li>
   * <li>series[seriesIndex] / legend:section[sectionIndex]:item[itemIndex]</li>
   * <li>group[groupIndex0]...[groupIndexN]</li>
   * <li>axis["axisType"]:title</li>
   * <li>axis["axisType"]:referenceObject[index]</li>
   * </ul>
   * @override
   */
  GetSubIdForDomElement(displayable) {
    var axisSubId = null;
    if (displayable.isDescendantOf(this._xAxis)) {
      axisSubId = this._xAxisAutomation.GetSubIdForDomElement(displayable);
      return axisSubId ? this._convertAxisSubIdToChartSubId(axisSubId, 'xAxis') : null;
    } else if (displayable.isDescendantOf(this._yAxis)) {
      axisSubId = this._yAxisAutomation.GetSubIdForDomElement(displayable);
      return axisSubId ? this._convertAxisSubIdToChartSubId(axisSubId, 'yAxis') : null;
    } else if (displayable.isDescendantOf(this._y2Axis)) {
      axisSubId = this._y2AxisAutomation.GetSubIdForDomElement(displayable);
      return axisSubId ? this._convertAxisSubIdToChartSubId(axisSubId, 'y2Axis') : null;
    } else if (displayable.isDescendantOf(this._legend)) {
      var legendSubId = this._legendAutomation.GetSubIdForDomElement(displayable);
      return legendSubId ? this._convertLegendSubIdToChartSubId(legendSubId) : null;
    } else {
      var logicalObj = this._comp.getEventManager().GetLogicalObject(displayable);
      if (!logicalObj) return null;

      if (logicalObj instanceof SimpleObjPeer) {
        var type = logicalObj.getParams()['type'];
        if (type == 'pieCenterLabel') return 'pieCenterLabel';
        else if (type == 'plotArea') return 'plotArea';
      }

      if (logicalObj instanceof DvtChartPieSlice)
        // pie chart data items do not use ChartObjPeer and return only dataItem[seriesIndex]
        return 'dataItem[' + logicalObj.getSeriesIndex() + ']';

      if (logicalObj instanceof DvtChartObjPeer) {
        // Chart data items
        var seriesIndex = logicalObj.getSeriesIndex();
        var itemIndex = logicalObj.getGroupIndex(); // corresponds to data items position in its series array

        if (
          seriesIndex != null &&
          itemIndex >= 0 &&
          (this._options['type'] != 'funnel' || this._options['type'] != 'pyramid')
        )
          return 'dataItem[' + seriesIndex + '][' + itemIndex + ']';
        else if (
          seriesIndex != null &&
          itemIndex == DvtChartFunnelRenderer._GROUP_IDX &&
          (this._options['type'] == 'funnel' || this._options['type'] == 'pyramid')
        )
          return 'dataItem[' + seriesIndex + ']';
        // funnel or pyramid chart only returns dataItem[seriesIndex]
        else if (seriesIndex != null && (itemIndex == null || itemIndex < 0))
          // displayable represents a seriesItem e.g. line, area
          return 'series[' + seriesIndex + ']';
      } else if (logicalObj instanceof DvtChartRefObjPeer) {
        // reference objects
        var axisType = logicalObj.getAxisType();
        var refObjIndex = logicalObj.getIndex();
        return axisType && refObjIndex >= 0
          ? axisType + ':referenceObject[' + refObjIndex + ']'
          : null;
      }
    }
    return null;
  }

  /**
   * Takes the subId for a legend item and converts it to a valid subId for chart legends
   * @param {String} subId for legend
   * @return {String} series[seriesIndex] / legend:section[sectionIndex]:item[itemIndex]
   * @private
   */
  _convertLegendSubIdToChartSubId(subId) {
    // Get the legend item that corresponds to the legend subId
    var legendOptions = this._legend.getOptions();
    var legendItem = this._legendAutomation.getLegendItem(legendOptions, subId);
    if (legendItem) {
      // Get index of series item that has same name as legend items's text
      for (var s = 0; s < this._options['series'].length; s++) {
        var series = this._options['series'][s];
        if (series['name'] == legendItem['text']) return 'series[' + s + ']';
      }
      // legend item is not associated with a series
      return 'legend:' + subId;
    }
    return null;
  }

  /**
   * Takes the subId for an axis item and converts it to a valid subId for chart axes
   * @param {String} subId for returned by the axis
   * @param {String=} axisType The axisType
   * @return {String} group[groupIndex0]...[groupIndexN] or axis["axisType"]:title
   * @private
   */
  _convertAxisSubIdToChartSubId(subId, axisType) {
    if (subId == 'title' && axisType) return axisType + ':' + subId;
    else {
      // Take item[groupIndex0]...[groupIndexN] string and return group[groupIndex0]...[groupIndexN]
      var indexList = subId.substring(subId.indexOf('['));
      if (indexList) return 'group' + indexList;
    }

    return null;
  }

  /**
   * Valid subIds inlcude:
   * <ul>
   * <li>dataItem[seriesIndex][itemIndex]</li>
   * <li>series[seriesIndex] / legend:section[sectionIndex]:item[itemIndex]</li>
   * <li>group[groupIndex0]...[groupIndexN]</li>
   * <li>axis["axisType"]:title</li>
   * <li>axis["axisType"]:referenceObject[index]</li>
   * </ul>
   * @override
   */
  getDomElementForSubId(subId) {
    // First check for subIds that don't have to be parsed
    if (subId == Automation.TOOLTIP_SUBID)
      // TOOLTIP
      return this.GetTooltipElement(
        this._comp,
        DvtChartTooltipUtils.isDataCursorEnabled(this._comp) ? DvtChartDataCursor.TOOLTIP_ID : null
      );
    else if (subId == 'pieCenterLabel')
      // PIE CENTER LABEL
      return this._comp.pieChart.getCenterLabel().getElem();
    else if (subId == 'plotArea')
      // PLOT AREA
      return this._comp.getPlotArea().getElem();

    // CHART ELEMENTS
    var openParen1 = subId.indexOf('[');
    var closeParen1 = subId.indexOf(']');
    var openParen2, closeParen2, logicalObj;
    var colon = subId.indexOf(':');
    if ((openParen1 > 0 && closeParen1 > 0) || colon > 0) {
      var objType = colon < 0 ? subId.substring(0, openParen1) : subId.substring(0, colon);
      // GROUP AXIS LABELS
      if (objType == 'group') {
        return this._xAxisAutomation.getDomElementForSubId(subId);
      }
      // LEGEND ITEMS
      if (objType == 'series') {
        subId = this._convertToLegendSubId(subId);
        return this._legendAutomation.getDomElementForSubId(subId);
      } else if (subId.substring(0, colon) == 'legend') {
        subId = subId.substring(colon + 1);
        return this._legendAutomation.getDomElementForSubId(subId);
      }
      var seriesIndex = subId.substring(openParen1 + 1, closeParen1);
      // AXIS TITLE & REFERENCE OBJECTS
      if (objType == 'xAxis' || objType == 'yAxis' || objType == 'y2Axis') {
        var axisObjectType = subId.substring(colon + 1);
        if (axisObjectType == 'title') {
          // subId for axis title
          if (objType == 'xAxis')
            return this._xAxisAutomation.getDomElementForSubId(axisObjectType);
          else if (objType == 'yAxis')
            return this._yAxisAutomation.getDomElementForSubId(axisObjectType);
          else if (objType == 'y2Axis')
            return this._y2AxisAutomation.getDomElementForSubId(axisObjectType);
        } else {
          // subId for axis reference objects
          openParen2 = axisObjectType.indexOf('[');
          closeParen2 = axisObjectType.indexOf(']');
          if (axisObjectType.substring(0, openParen2) == 'referenceObject') {
            var index = axisObjectType.substring(openParen2 + 1, closeParen2);
            logicalObj = this._getRefObjPeer(index);
            if (logicalObj) return logicalObj.getDisplayables()[0].getElem();
          }
        }
      }

      // CHART DATA ITEMS
      if (this._options['type'] == 'pie') {
        var pieSlice = this._comp.pieChart.getSliceDisplayable(seriesIndex);
        if (pieSlice) return pieSlice.getElem();
      }
      // If funnel/pyramid chart set the default itemIndex, else parse it from the given subId
      var itemIndex;
      if (this._options['type'] == 'funnel') {
        itemIndex = DvtChartFunnelRenderer._GROUP_IDX;
      } else if (this._options['type'] == 'pyramid') {
        itemIndex = DvtChartPyramidRenderer._GROUP_IDX;
      } else {
        subId = subId.substring(closeParen1 + 1);
        openParen2 = subId.indexOf('[');
        closeParen2 = subId.indexOf(']');
        if (openParen2 >= 0 && closeParen2 >= 0) {
          itemIndex = subId.substring(openParen2 + 1, closeParen2);
        }
      }
      // Get the logical object and return the dom element of its associated displayable
      logicalObj = this._getChartObjPeer(seriesIndex, itemIndex);
      if (logicalObj) return logicalObj.getDisplayables()[0].getElem();
    }
    return null;
  }

  /**
   * Returns the DvtChartObjPeer for the given seriesIndex and itemIndex
   * @param {String} seriesIndex The seriesIndex for dataItem types
   * @param {String} itemIndex The itemIndex for dataItem types
   * @return {DvtChartObjPeer} The DvtChartObjPeer matching the parameters or null if none exists
   * @private
   */
  _getChartObjPeer(seriesIndex, itemIndex) {
    var peers = this._comp.getChartObjPeers();
    for (var i = 0; i < peers.length; i++) {
      var series = peers[i].getSeriesIndex();
      var item = peers[i].getGroupIndex(); // correspinds to the data item's position in its series array
      if (series == seriesIndex && item == itemIndex) return peers[i];
    }
    return null;
  }

  /**
   * Returns the DvtChartObjPeer for the given series and groupid
   * @param {String} seriesId The seriesId for dataItem types
   * @param {Array} groupId The groupIds for dataItem types
   * @param {number} itemIndex The index for dataItem corresponding to index in boxplot sub items.
   * @return {DvtChartObjPeer} The DvtChartObjPeer matching the parameters or null if none exists
   * @private
   */

  _getChartObjPeerFromId(seriesId, groupId, itemIndex) {
    var peers = this._comp.getChartObjPeers();
    for (var i = 0; i < peers.length; i++) {
      var series = peers[i].getSeries();
      var group = peers[i].getGroup();
      if (
        series === seriesId &&
        ((typeof group === 'string' && groupId.length === 1 && groupId[0] === group) ||
          (Array.isArray(group) && ArrayUtils.equals(group, groupId)))
      ) {
        if (itemIndex == null || itemIndex === peers[i]._itemIndex) {
          return peers[i];
        }
      }
    }
    return null;
  }

  /**
   * Returns the DvtChartRefObjPeer for the given index
   * @param {String} index The index of the object in the referenceObjects array
   * @return {DvtChartObjPeer} The DvtChartRefObjPeer matching the index or null if none exists
   * @private
   */
  _getRefObjPeer(index) {
    var peers = this._comp.getRefObjPeers();
    for (var i = 0; i < peers.length; i++) {
      if (index == peers[i].getIndex()) return peers[i];
    }
    return null;
  }

  /**
   * Takes the subId for a chart series and converts it to a valid subId for legend item
   * @param {String} subId series[seriesIndex]
   * @return {String} section[sectionIndex0]:item[itemIndex]
   * @private
   */
  _convertToLegendSubId(subId) {
    var openParen = subId.indexOf('[');
    var closeParen = subId.indexOf(']');
    var seriesIndex = subId.substring(openParen + 1, closeParen);

    var legendOptions = this._legend.getOptions();
    var series = this._options['series'][seriesIndex];

    var indices = this._legendAutomation.getIndicesFromSeries(series, legendOptions);
    return 'section' + indices;
  }

  /**
   * Returns an object containing data for a chart data item. Used for verification.
   * Valid verification values inlcude:
   * <ul>
   * <li>borderColor</li>
   * <li>color</li>
   * <li>label</li>
   * <li>targetValue</li>
   * <li>tooltip</li>
   * <li>value</li>
   * <li>open</li>
   * <li>close</li>
   * <li>high</li>
   * <li>low</li>
   * <li>volume</li>
   * <li>x</li>
   * <li>y</li>
   * <li>z</li>
   * <li>group</li>
   * <li>series</li>
   * <li>selected</li>
   * </ul>
   * @param {String} seriesIndex The seriesIndex for dataItem and series types, the itemIndex for group types
   * @param {String} itemIndex The itemIndex for dataItem types
   * @return {Object} An object containing data for the dataItem
   */
  getDataItem(seriesIndex, itemIndex) {
    if (
      this._options['type'] == 'pie' ||
      this._options['type'] == 'funnel' ||
      this._options['type'] == 'pyramid'
    )
      itemIndex = 0; //Not sure if neccessary but getDataItem will be null if itemIndex is null

    var dataItem = DvtChartDataUtils.getDataItem(this._comp, seriesIndex, itemIndex);

    if (dataItem) {
      return {
        borderColor: DvtChartStyleUtils.getBorderColor(this._comp, seriesIndex, itemIndex),
        color: DvtChartStyleUtils.getColor(this._comp, seriesIndex, itemIndex),
        label: DvtChartStyleUtils.getDataLabel(this._comp, seriesIndex, itemIndex),
        targetValue: DvtChartDataUtils.getTargetVal(this._comp, seriesIndex, itemIndex),
        tooltip: DvtChartTooltipUtils.getDatatip(this._comp, seriesIndex, itemIndex, null, false),
        value: DvtChartDataUtils.getVal(this._comp, seriesIndex, itemIndex),
        open: dataItem['open'],
        close: dataItem['close'],
        high: DvtChartDataUtils.getHighVal(this._comp, seriesIndex, itemIndex),
        low: DvtChartDataUtils.getLowVal(this._comp, seriesIndex, itemIndex),
        volume: dataItem['volume'],
        x: DvtChartDataUtils.getXVal(this._comp, seriesIndex, itemIndex),
        y: dataItem['y'],
        z: dataItem['z'],
        min: dataItem['min'],
        max: dataItem['max'],
        group: DvtChartDataUtils.getGroup(this._comp, itemIndex),
        series: DvtChartDataUtils.getSeries(this._comp, seriesIndex),
        selected: DvtChartDataUtils.isDataSelected(this._comp, seriesIndex, itemIndex)
      };
    }
    return null;
  }

  /**
   * Returns the group corresponding to the given index. Used for verification.
   * @param {String} itemIndex The index of the desired group
   * @return {String} The group corresponding to the given index
   */
  getGroup(itemIndex) {
    return DvtChartDataUtils.getGroup(this._comp, itemIndex);
  }

  /**
   * Returns the name of the series corresponding to the given index. Used for verification.
   * @param {String} seriesIndex The index of the desired series
   * @return {String} the name of the series corresponding to the given index
   */
  getSeries(seriesIndex) {
    return this._options['series'][seriesIndex]['name'];
  }

  /**
   * Returns the number of groups in the chart data. Used for verification.
   * @return {Number} The number of groups
   */
  getGroupCount() {
    return DvtChartDataUtils.getGroupCount(this._comp);
  }

  /**
   * Returns the number of series in the chart data. Used for verification.
   * @return {Number} The number of series
   */
  getSeriesCount() {
    return this._options['series'].length;
  }

  /**
   * Returns the chart title. Used for verification.
   * @return {String} The chart title
   */
  getTitle() {
    return this._options['title']['text'];
  }

  /**
   * Returns an object that represents the legend data. Used for verification.
   * Valid verification values inlcude:
   * <ul>
   * <li>bounds</li>
   * <li>title</li>
   * </ul>
   * @return {Object} An object that represents the legend data
   */
  getLegend() {
    var legendSpace = this._legend.__getBounds();
    var point = this._legend.localToStage(new Point(legendSpace.x, legendSpace.y));
    var legendBounds = {
      x: point.x,
      y: point.y,
      width: legendSpace.w,
      height: legendSpace.h
    };

    return {
      bounds: legendBounds,
      title: this._legend.getOptions()['title']
    };
  }

  /**
   * Returns an object that represents the plot area data. Used for verification.
   * Valid verification values inlcude:
   * <ul>
   * <li>bounds</li>
   * </ul>
   * @return {Object} An object that represents the plot area data
   */
  getPlotArea() {
    var plotAreaSpace = this._comp.__getPlotAreaSpace();

    var plotAreaBounds = {
      x: plotAreaSpace.x,
      y: plotAreaSpace.y,
      width: plotAreaSpace.w,
      height: plotAreaSpace.h
    };

    return { bounds: plotAreaBounds };
  }

  /**
   * Returns an object that represents the xAxis data. Used for verification.
   * Valid verification values inlcude:
   * <ul>
   * <li>bounds</li>
   * <li>title</li>
   * </ul>
   * @return {Object} An object that represents the xAxis data
   */
  getXAxis() {
    return this._getAxis('x');
  }

  /**
   * Returns an object that represents the yAxis data. Used for verification.
   * Valid verification values inlcude:
   * <ul>
   * <li>bounds</li>
   * <li>title</li>
   * </ul>
   * @return {Object} An object that represents the yAxis data
   */
  getYAxis() {
    return this._getAxis('y');
  }

  /**
   * Returns an object that represents the y2Axis data. Used for verification.
   * Valid verification values inlcude:
   * <ul>
   * <li>bounds</li>
   * <li>title</li>
   * </ul>
   * @return {Object} An object that represents the y2Axis data
   */
  getY2Axis() {
    return this._getAxis('y2');
  }

  /**
   * Returns an object that represents the axis data.
   * @param {string} type The axis type: x, y, or y2
   * @return {object} An object that represents the axis data
   * @private
   */
  _getAxis(type) {
    var axis = type == 'x' ? this._xAxis : type == 'y' ? this._yAxis : this._y2Axis;
    if (axis) {
      var axisSpace = axis.__getBounds();
      var stageCoord = axis.localToStage(new Point(axisSpace.x, axisSpace.y));
      var axisBounds = {
        x: stageCoord.x,
        y: stageCoord.y,
        width: axisSpace.w,
        height: axisSpace.h
      };

      var chart = this._comp;
      var getPreferredSize = (width, height) => {
        var axisOptions = axis.getOptions();
        var position = axisOptions['position'];
        var tickLabelGap = DvtChartAxisUtils.getTickLabelGapSize(chart, type);
        var outerGap =
          DvtChartAxisUtils.isStandaloneXAxis(chart) ||
          DvtChartAxisUtils.isStandaloneYAxis(chart) ||
          DvtChartAxisUtils.isStandaloneY2Axis(chart)
            ? 2
            : 0;

        // the preferred size computed by the axis excludes tick label gap, so we have to subtract the gap
        // before passing to the axis, and add it again later
        var prefSize;
        if (position == 'top' || position == 'bottom') {
          prefSize = axis.getPreferredSize(axisOptions, width, height - tickLabelGap - outerGap);
          prefSize.h = Math.ceil(prefSize.h + tickLabelGap + outerGap);
        } else {
          prefSize = axis.getPreferredSize(axisOptions, width - tickLabelGap - outerGap, height);
          prefSize.w = Math.ceil(prefSize.w + tickLabelGap + outerGap);
        }
        return { width: prefSize.w, height: prefSize.h };
      };

      return {
        bounds: axisBounds,
        title: this._options[type + 'Axis']['title'],
        getPreferredSize: getPreferredSize
      };
    }

    return null;
  }

  /**
   * @override
   */
  IsTooltipElement(domElement) {
    var id = domElement.getAttribute('id');
    if (
      id &&
      (id.indexOf(DvtChartDataCursor.TOOLTIP_ID) == 0 ||
        id.indexOf(HtmlTooltipManager._TOOLTIP_DIV_ID) == 0)
    )
      return true;
    return false;
  }

  /**
   * Method to fire synthetic item drill event. Used by chart webdriver.
   * @param {string} seriesId The series id of the data item.
   * @param {Array<string>} groupId The group id of the data item.
   * @param {number} itemIndex The index of data item belonging to boxplot sub items.
   */

  dispatchItemDrill(seriesId, groupId, itemIndex) {
    var obj = this._getChartObjPeerFromId(seriesId, groupId, itemIndex);
    if (obj) {
      this._comp.getEventManager().processDrillEvent(obj);
    }
  }

  /**
   * Method to fire synthetic multiseries drill event. Used by chart webdriver.
   */
  dispatchMultiSeriesDrill() {
    var pie = this._comp.pieChart;
    if (pie && pie.otherSlice) {
      this._comp.getEventManager().processDrillEvent(pie.otherSlice);
    }
  }

  /**
   * Fires synthetic group drill event from chart. Used by chart webdriver.
   * @param {Array<string>} groupId The group id of the axis label from which group drill event is fired.
   */
  dispatchGroupDrill(groupId) {
    if (this._xAxisAutomation && groupId) {
      this._xAxisAutomation.dispatchDrillEvent(groupId);
    }
  }

  /**
   * Fires synthetic series drill event from chart. Used by chart webdriver.
   * @param {string} seriesId The id of chart series from which series drill event is fired.
   */

  dispatchSeriesDrill(seriesId) {
    if (this._legendAutomation && seriesId) {
      this._legendAutomation.dispatchDrillEvent(seriesId);
    }
  }
}

/**
 * Event Manager for Chart.
 * @param {Chart} chart
 * @class
 * @extends {dvt.EventManager}
 * @constructor
 */
class DvtChartEventManager extends EventManager {
  constructor(chart) {
    super(chart.getCtx(), chart.processEvent, chart, chart);
    this._chart = chart;

    this._dragMode = null;
    this._dragButtonsVisible = Agent.isTouchDevice();

    /**
     * The pan button
     * @type {dvt.IconButton}
     */
    this.panButton = null;
    /**
     * The marquee zoom button
     * @type {dvt.IconButton}
     */
    this.zoomButton = null;
    /**
     * The marquee select button
     * @type {dvt.IconButton}
     */
    this.selectButton = null;

    // Event handlers
    this._dataCursorHandler = null;
    this._panZoomHandler = null;
    this._marqueeZoomHandler = null;
    this._marqueeSelectHandler = null;
  }

  /**
   * @override
   */
  addListeners(displayable) {
    SvgDocumentUtils.addDragListeners(
      this._chart,
      this._onDragStart,
      this._onDragMove,
      this._onDragEnd,
      this
    );
    super.addListeners(displayable);

    if (!Agent.isTouchDevice()) {
      displayable.addEvtListener(MouseEvent.MOUSEWHEEL, this.OnMouseWheel, false, this);
    }
  }

  /**
   * @override
   */
  RemoveListeners(displayable) {
    super.RemoveListeners(displayable);
    if (!Agent.isTouchDevice()) {
      displayable.removeEvtListener(MouseEvent.MOUSEWHEEL, this.OnMouseWheel, false, this);
    }
  }

  /**
   * Returns the logical object corresponding to the specified dvt.Displayable.  All high level event handlers,
   * such as the selection handlers, are designed to react to the logical objects.
   * @param {dvt.Displayable} target The displayable.
   * @return {object} The logical object corresponding to the target.
   */
  getLogicalObject(target) {
    return this.GetLogicalObject(target, true);
  }

  /**
   * Returns an event handler for the current drag mode.
   * @param {dvt.Point} relPos (optional) The current cursor position relative to the stage. If provided, the relPos will
   *    be considered in choosing the drag handler.
   * @return {dvt.MarqueeHandler or dvt.PanZoomHandler} Drag handler.
   * @private
   */
  _getDragHandler(relPos) {
    if (
      relPos &&
      this._chart.getOptions()['dragMode'] == 'user' &&
      DvtChartTypeUtils.isBLAC(this._chart) &&
      (this._dragMode == DvtChartEventManager.DRAG_MODE_PAN ||
        this._dragMode == DvtChartEventManager.DRAG_MODE_ZOOM)
    ) {
      // For BLAC chart on desktop, the pan and zoom modes are combined.
      // If the drag starts inside the plot area, it's a pan. If the drag starts inside the axis, it's a marquee zoom.
      if (this._panZoomHandler && this._panZoomHandler.isWithinBounds(relPos))
        this._dragMode = DvtChartEventManager.DRAG_MODE_PAN;
      else this._dragMode = DvtChartEventManager.DRAG_MODE_ZOOM;
    }

    if (this._dragMode == DvtChartEventManager.DRAG_MODE_PAN) return this._panZoomHandler;
    if (this._dragMode == DvtChartEventManager.DRAG_MODE_ZOOM) return this._marqueeZoomHandler;
    if (this._dragMode == DvtChartEventManager.DRAG_MODE_SELECT) return this._marqueeSelectHandler;
    return null;
  }

  /**
   * Drag start callback.
   * @param {dvt.BaseEvent} event
   * @return {boolean} Whether drag is initiated.
   * @private
   */
  _onDragStart(event) {
    if (EventManager.isTouchEvent(event)) return this._onTouchDragStart(event);
    else return this._onMouseDragStart(event);
  }

  /**
   * Drag move callback.
   * @param {dvt.BaseEvent} event
   * @return {boolean}
   * @private
   */
  _onDragMove(event) {
    if (EventManager.isTouchEvent(event)) return this._onTouchDragMove(event);
    else return this._onMouseDragMove(event);
  }

  /**
   * Drag end callback.
   * @param {dvt.BaseEvent} event
   * @return {boolean}
   * @private
   */
  _onDragEnd(event) {
    if (EventManager.isTouchEvent(event)) return this._onTouchDragEnd(event);
    else return this._onMouseDragEnd(event);
  }

  /**
   * Mouse drag start callback.
   * @param {dvt.BaseEvent} event
   * @return {boolean} Whether drag is initiated.
   * @private
   */
  _onMouseDragStart(event) {
    var relPos = this._context.pageToStageCoords(event.pageX, event.pageY);
    var dragHandler = this._getDragHandler(relPos);
    var chartEvent;

    // Do not initiate drag if the target is selectable. Drag only on left click.
    var obj = this.GetLogicalObject(event.target);
    var selectable = obj && obj.isSelectable && obj.isSelectable();
    if (!selectable && event.button == 0 && dragHandler) {
      chartEvent = dragHandler.processDragStart(relPos, event.ctrlKey);
      if (chartEvent) this._callback.call(this._callbackObj, chartEvent);

      this._chart.setCursor(dragHandler.getCursor(relPos));
      this.setDragButtonsVisible(false); // hide drag buttons on drag

      // Ensure the chart is currently focused so that it can accept cancel events
      if (this._chart != this.getCtx().getCurrentKeyboardFocus())
        this.getCtx().setCurrentKeyboardFocus(this._chart);
    }

    if (chartEvent) {
      if (this._dataCursorHandler) this._dataCursorHandler.processEnd();
      return true;
    }
    return false;
  }

  /**
   * Mouse drag move callback.
   * @param {dvt.BaseEvent} event
   * @private
   */
  _onMouseDragMove(event) {
    var relPos = this._context.pageToStageCoords(event.pageX, event.pageY);
    var dragHandler = this._getDragHandler(); // don't pass the relPos so that the drag mode stays
    var chartEvent;

    if (dragHandler) {
      chartEvent = dragHandler.processDragMove(relPos, event.ctrlKey);
      if (chartEvent) {
        this._callback.call(this._callbackObj, chartEvent);
        this.setDragButtonsVisible(false); // hide drag buttons on drag
      }
    }

    if (chartEvent) event.stopPropagation(); // prevent data cursor from appearing
  }

  /**
   * Mouse drag end callback.
   * @param {dvt.BaseEvent} event
   * @private
   */
  _onMouseDragEnd(event) {
    var relPos = this._context.pageToStageCoords(event.pageX, event.pageY);
    var dragHandler = this._getDragHandler(); // don't pass the relPos so that the drag mode stays
    var chartEvent;

    if (dragHandler) {
      chartEvent = dragHandler.processDragEnd(relPos, event.ctrlKey);
      if (chartEvent) {
        this._callback.call(this._callbackObj, chartEvent);
        this.autoToggleZoomButton();
      }

      this._chart.setCursor(dragHandler.getCursor(relPos));

      // Show the drag buttons
      var axisSpace = this._chart.__getAxisSpace();
      if (axisSpace) this.setDragButtonsVisible(axisSpace.containsPoint(relPos.x, relPos.y));
    }
  }

  /**
   * @override
   */
  OnMouseMove(event) {
    super.OnMouseMove(event);

    var relPos = this._context.pageToStageCoords(event.pageX, event.pageY);
    if (this._dataCursorHandler) {
      if (this.GetLogicalObjectAndDisplayable(event.target).displayable instanceof IconButton)
        // don't show DC over buttons
        this._dataCursorHandler.processEnd();
      else {
        relPos = this._component.stageToLocal(relPos);
        this._dataCursorHandler.processMove(relPos);
      }
    }

    // Update the cursor
    var dragHandler = this._getDragHandler(relPos);
    if (dragHandler) this._chart.setCursor(dragHandler.getCursor(relPos));
    else this._chart.setCursor('default');
  }

  /**
   * @override
   */
  OnMouseOut(event) {
    super.OnMouseOut(event);
    var relPos = this._context.pageToStageCoords(event.pageX, event.pageY);

    // Hide the drag buttons
    var axisSpace = this._chart.__getAxisSpace();
    if (axisSpace) this.setDragButtonsVisible(axisSpace.containsPoint(relPos.x, relPos.y));

    if (this._dataCursorHandler) {
      relPos = this._component.stageToLocal(relPos);
      this._dataCursorHandler.processMove(relPos);
    }

    var obj = this.GetLogicalObject(event.target);
    if (!obj) return;
  }

  /**
   * @override
   */
  OnMouseWheel(event) {
    if (!DvtChartBehaviorUtils.isZoomable(this._chart)) return;

    var delta = event.wheelDelta != null ? event.wheelDelta : 0;
    var relPos = this._context.pageToStageCoords(event.pageX, event.pageY);

    if (this._panZoomHandler) {
      var panZoomEvent = this._panZoomHandler.processMouseWheel(relPos, delta);
      if (panZoomEvent) {
        event.preventDefault();
        event.stopPropagation();
        this._callback.call(this._callbackObj, panZoomEvent);

        // Update the data cursor since the viewport has changed
        if (this._dataCursorHandler) {
          relPos = this._component.stageToLocal(relPos);
          this._dataCursorHandler.processMove(relPos);
        }
      }
    }
  }

  /**
   * @override
   */
  ShowFocusEffect(event, navigable) {
    if (this._dataCursorHandler) {
      var pos = navigable.getDataPosition();
      if (pos) {
        var plotAreaBounds = this._chart.__getPlotAreaSpace();
        this._dataCursorHandler.processMove(
          new Point(pos.x + plotAreaBounds.x, pos.y + plotAreaBounds.y)
        );
      }
    }
    super.ShowFocusEffect(event, navigable);
  }

  /**
   * @override
   */
  OnBlur(event) {
    if (this._dataCursorHandler) this._dataCursorHandler.processEnd();
    super.OnBlur(event);
  }

  /**
   * @override
   */
  OnClickInternal(event) {
    var obj = this.GetLogicalObject(event.target);
    var pos = this._context.pageToStageCoords(event.pageX, event.pageY);
    if (this.SeriesFocusHandler) this.SeriesFocusHandler.processSeriesFocus(pos, obj);

    if (!obj) return;

    // Only drill if not selectable. If selectable, drill with double click.
    if (!(obj.isSelectable && obj.isSelectable())) this.processDrillEvent(obj);
  }

  /**
   * @override
   */
  OnDblClickInternal(event) {
    var obj = this.GetLogicalObject(event.target);
    if (!obj) return;

    // Only double click to drill if selectable. Otherwise, drill with single click.
    if (obj.isSelectable && obj.isSelectable()) this.processDrillEvent(obj);
  }

  /**
   * @override
   */
  HandleTouchHoverStartInternal(event) {
    if (this._dataCursorHandler && !this.isTouchResponseTouchStart()) {
      var relPos = this._context.pageToStageCoords(event.touch.pageX, event.touch.pageY);
      this._dataCursorHandler.processMove(relPos);
      return false;
    }

    var dlo = this.GetLogicalObject(event.target);
    this.TouchManager.setTooltipEnabled(event.touch.identifier, this.getTooltipsEnabled(dlo));
    return false;
  }

  /**
   * @override
   */
  HandleTouchHoverMoveInternal(event) {
    if (this._dataCursorHandler && !this.isTouchResponseTouchStart()) {
      var relPos = this._context.pageToStageCoords(event.touch.pageX, event.touch.pageY);
      this._dataCursorHandler.processMove(relPos);
      return false;
    }

    var dlo = this.GetLogicalObject(event.target);
    this.TouchManager.setTooltipEnabled(event.touch.identifier, this.getTooltipsEnabled(dlo));
    return false;
  }

  /**
   * @override
   */
  HandleTouchHoverEndInternal(event) {
    this.endDrag();

    var obj = this.GetLogicalObject(event.target);
    if (!obj) return;

    // Only drill if not selectable. If selectable, drill using double click.
    if (!(obj.isSelectable && obj.isSelectable())) this.processDrillEvent(obj);
  }

  /**
   * @override
   */
  HandleTouchClickInternal(event) {
    var obj = this.GetLogicalObject(event.target);
    if (!obj) return;

    // Only drill if not selectable. If selectable, drill using double click.
    if (!(obj.isSelectable && obj.isSelectable())) this.processDrillEvent(obj);
  }

  /**
   * @override
   */
  HandleTouchDblClickInternal(event) {
    var obj = this.GetLogicalObject(event.target);
    if (!obj) return;

    // Only double click to drill if selectable. Otherwise, drill with single click.
    if (obj.isSelectable && obj.isSelectable()) {
      event.preventDefault();
      event.stopPropagation();
      this.processDrillEvent(obj);
    }
  }

  /**
   * Processes an drill on the specified chart item.
   * @param {DvtChartObjPeer} obj The chart item that was clicked.
   */
  processDrillEvent(obj) {
    if (obj && obj.isDrillable && obj.isDrillable()) {
      var id = obj.getId();
      if (obj instanceof DvtChartObjPeer) {
        // when clicked on line and area instead of marker, chart fires seriesDrill
        if (obj.getGroupIndex() === -1) {
          this.FireEvent(EventFactory.newChartDrillEvent(id, obj.getSeries(), null, 'series'));
        } else {
          this.FireEvent(
            EventFactory.newChartDrillEvent(
              id.id != null ? id.id : id,
              obj.getSeries(),
              obj.getGroup(),
              'item'
            )
          );
        }
      } else if (obj instanceof DvtChartPieSlice) {
        // chart does not use DvtChartObjPeer for pieslice
        var subType = id.series === '_dvtOther' ? 'multiSeries' : 'item';
        this.FireEvent(EventFactory.newChartDrillEvent(id.id, id.series, id.group, subType));
      }
    }
  }

  /**
   * @override
   */
  ProcessRolloverEvent(event, obj, bOver) {
    // Don't continue if not enabled
    var options = this._chart.getOptions();
    if (DvtChartBehaviorUtils.getHoverBehavior(this._chart) != 'dim') return;

    // Compute the new highlighted categories and update the options
    var categories = obj.getCategories ? obj.getCategories() : [];
    options['highlightedCategories'] = bOver ? categories.slice() : null;

    // Fire the event to the rollover handler, who will fire to the component callback.
    var rolloverEvent = EventFactory.newCategoryHighlightEvent(
      options['highlightedCategories'],
      bOver
    );
    var hoverBehaviorDelay = DvtChartStyleUtils.getHoverBehaviorDelay(this._chart);

    // Find all the objects that may need to be highlighted
    var objs = this._chart.getObjects();
    if (this._chart.pieChart) objs = objs.concat(this._chart.pieChart.__getSlices());

    this.RolloverHandler.processEvent(
      rolloverEvent,
      objs,
      hoverBehaviorDelay,
      options['highlightMatch'] == 'any'
    );
  }

  /**
   * Touch drag start callback.
   * @param {dvt.BaseEvent} event
   * @return {boolean} Whether drag is initiated.
   * @private
   */
  _onTouchDragStart(event) {
    var touches = event.touches;
    var chartEvent, dataCursorOn;

    if (touches.length == 1) {
      var relPos = this._context.pageToStageCoords(touches[0].pageX, touches[0].pageY);
      var dragHandler = this._getDragHandler();
      if (dragHandler) chartEvent = dragHandler.processDragStart(relPos, true);
      else if (this._dataCursorHandler && this.isTouchResponseTouchStart()) {
        this._dataCursorHandler.processMove(relPos);
        dataCursorOn = true;
      }
    } else if (
      touches.length == 2 &&
      this._panZoomHandler &&
      DvtChartBehaviorUtils.isZoomable(this._chart)
    ) {
      this.endDrag(); // clean 1-finger events before starting pinch zoom
      var relPos1 = this._context.pageToStageCoords(touches[0].pageX, touches[0].pageY);
      var relPos2 = this._context.pageToStageCoords(touches[1].pageX, touches[1].pageY);
      chartEvent = this._panZoomHandler.processPinchStart(relPos1, relPos2);
    }

    if (chartEvent) {
      this._callback.call(this._callbackObj, chartEvent);
      this.getCtx().getTooltipManager().hideTooltip();
    }

    if (chartEvent || dataCursorOn) {
      event.preventDefault();
      event.stopPropagation();
      this.setDragButtonsVisible(false); // hide drag buttons on drag
      return true;
    }

    return false;
  }

  /**
   * Touch drag move callback.
   * @param {dvt.BaseEvent} event
   * @private
   */
  _onTouchDragMove(event) {
    var touches = event.touches;
    var chartEvent, dataCursorOn;

    if (touches.length == 1) {
      var relPos = this._context.pageToStageCoords(touches[0].pageX, touches[0].pageY);
      var dragHandler = this._getDragHandler();
      if (dragHandler) chartEvent = dragHandler.processDragMove(relPos, true);
      else if (this._dataCursorHandler && this.isTouchResponseTouchStart()) {
        this._dataCursorHandler.processMove(relPos);
        dataCursorOn = true;
      }
    } else if (
      touches.length == 2 &&
      this._panZoomHandler &&
      DvtChartBehaviorUtils.isZoomable(this._chart)
    ) {
      var relPos1 = this._context.pageToStageCoords(touches[0].pageX, touches[0].pageY);
      var relPos2 = this._context.pageToStageCoords(touches[1].pageX, touches[1].pageY);
      chartEvent = this._panZoomHandler.processPinchMove(relPos1, relPos2);
    }

    if (chartEvent || dataCursorOn) {
      event.preventDefault();
    }

    if (chartEvent) {
      this._callback.call(this._callbackObj, chartEvent);
      this.getCtx().getTooltipManager().hideTooltip();
    }
  }

  /**
   * Touch drag end callback.
   * @param {dvt.BaseEvent} event
   * @private
   */
  _onTouchDragEnd(event) {
    // End 1-finger event
    var chartEvent1 = this.endDrag();

    // End 2-finger event
    var chartEvent2;
    if (this._panZoomHandler && DvtChartBehaviorUtils.isZoomable(this._chart)) {
      chartEvent2 = this._panZoomHandler.processPinchEnd();
      if (chartEvent2) this._callback.call(this._callbackObj, chartEvent2);
    }

    if (chartEvent1 || chartEvent2) {
      event.preventDefault();
      this.getCtx().getTooltipManager().hideTooltip();

      var touchManager = this.getTouchManager();
      var identifier =
        event['changedTouches'].length == 1 ? event['changedTouches'][0].identifier : null;
      var touchInfo = identifier != null ? touchManager.getTouchInfo(identifier) : null;

      // : Reset the touch manager if we will be processing a touchMove, because we do not use the touch manager to handle the dragging
      // but the dragging updates the touch manager state
      if (!touchInfo || touchInfo['touchMoved']) touchManager.reset();
    }

    this.setDragButtonsVisible(true);
  }

  /**
   * @override
   */
  endDrag() {
    var dragHandler = this._getDragHandler();
    var chartEvent;

    if (dragHandler) chartEvent = dragHandler.processDragEnd(null, true);

    if (this._dataCursorHandler) this._dataCursorHandler.processEnd();

    if (chartEvent) this._callback.call(this._callbackObj, chartEvent);

    return chartEvent;
  }

  /**
   * Zooms by the specified amount.
   * @param {number} dz A number specifying the zoom ratio. dz = 1 means no zoom.
   */
  zoomBy(dz) {
    if (this._panZoomHandler && DvtChartBehaviorUtils.isZoomable(this._chart)) {
      var chartEvent = this._panZoomHandler.zoomBy(dz);
      if (chartEvent) this._callback.call(this._callbackObj, chartEvent);
    }
  }

  /**
   * Pans by the specified amount.
   * @param {number} dx A number from specifying the pan ratio in the x direction, e.g. dx = 0.5 means pan end by 50%..
   * @param {number} dy A number from specifying the pan ratio in the y direction, e.g. dy = 0.5 means pan down by 50%.
   */
  panBy(dx, dy) {
    if (this._panZoomHandler && DvtChartBehaviorUtils.isScrollable(this._chart)) {
      var chartEvent = this._panZoomHandler.panBy(dx, dy);
      if (chartEvent) this._callback.call(this._callbackObj, chartEvent);
    }
  }

  /**
   * Helper function to hide tooltips and data cursor, generally in preparation for render or removal of the chart. This
   * is not done in hideTooltip to avoid interactions with the superclass, which would cause problems with the data cursor.
   */
  hideHoverFeedback() {
    // Hide tooltip and data cursor
    this.hideTooltip();

    // Hide the data cursor. This is necessary to hide the data cursor line when the user mouses over the tooltip div in
    // IE9, which does not support pointer-events.
    if (this._dataCursorHandler) this._dataCursorHandler.processEnd();
  }

  /**
   * @override
   */
  hideTooltip() {
    // Don't hide the tooltip if data cursor is shown on a touch device
    if (!this._dataCursorHandler || !this._dataCursorHandler.isDataCursorShown())
      super.hideTooltip();
  }

  /**
   * @override
   */
  getTooltipsEnabled(logicalObj) {
    // Don't allow tooltips to conflict with the data cursor
    if (
      this._dataCursorHandler &&
      (logicalObj instanceof DvtChartObjPeer ||
        logicalObj instanceof DvtChartRefObjPeer ||
        this._dataCursorHandler.isDataCursorShown())
    )
      return false;
    else return super.getTooltipsEnabled();
  }

  /**
   * Gets the data cursor handler.
   * @return {DvtChartDataCursorHandler} The data cursor handler.
   */
  getDataCursorHandler() {
    return this._dataCursorHandler;
  }

  /**
   * Sets the data cursor handler.
   * @param {DvtChartDataCursorHandler} handler The data cursor handler.
   */
  setDataCursorHandler(handler) {
    this._dataCursorHandler = handler;
  }

  /**
   * Sets the pan zoom handler.
   * @param {dvt.PanZoomHandler} handler The pan zoom handler.
   */
  setPanZoomHandler(handler) {
    this._panZoomHandler = handler;
  }

  /**
   * Sets the marquee zoom handler.
   * @param {dvt.MarqueeHandler} handler The marquee zoom handler.
   */
  setMarqueeZoomHandler(handler) {
    this._marqueeZoomHandler = handler;
  }

  /**
   * Sets the marquee select handler.
   * @param {dvt.MarqueeHandler} handler The marquee select handler.
   */
  setMarqueeSelectHandler(handler) {
    this._marqueeSelectHandler = handler;
  }

  /**
   * @override
   */
  getMarqueeGlassPane() {
    if (this._dragMode == DvtChartEventManager.DRAG_MODE_ZOOM)
      return this._marqueeZoomHandler.getGlassPane();
    else if (this._dragMode == DvtChartEventManager.DRAG_MODE_SELECT)
      return this._marqueeSelectHandler.getGlassPane();

    return null;
  }

  /**
   * Cancels marquee zoom/select.
   * @param {dvt.BaseEvent} event The event
   */
  cancelMarquee(event) {
    if (this._dragMode == DvtChartEventManager.DRAG_MODE_ZOOM) {
      if (this._marqueeZoomHandler.cancelMarquee()) event.preventDefault();
    } else if (this._dragMode == DvtChartEventManager.DRAG_MODE_SELECT) {
      // If marquee is in progress, re-render from the options obj, which has the old selection
      if (this._marqueeSelectHandler && this._marqueeSelectHandler.cancelMarquee())
        this._chart.render();
    }
  }

  /**
   * Gets the current drag mode.
   * @return {string} The drag mode.
   */
  getDragMode() {
    return this._dragMode;
  }

  /**
   * Sets the drag mode. If set to null, the drag mode will become the default one.
   * @param {string} dragMode The drag mode, or null.
   */
  setDragMode(dragMode) {
    if (dragMode == null) this._dragMode = this._getDefaultDragMode();
    else this._dragMode = dragMode;

    // If the chart is fully zoomed out, the pan mode should fall back to the zoom mode on desktop
    if (
      this._chart.xAxis.isFullViewport() &&
      (!this._chart.yAxis || this._chart.yAxis.isFullViewport())
    )
      this.autoToggleZoomButton();
  }

  /**
   * Returns the default drag mode for the chart.
   * @return {string} The default drag mode.
   * @private
   */
  _getDefaultDragMode() {
    if (Agent.isTouchDevice()) return DvtChartEventManager.DRAG_MODE_OFF;
    else if (DvtChartBehaviorUtils.isScrollable(this._chart))
      return DvtChartEventManager.DRAG_MODE_PAN;
    else if (this._chart.getOptions()['selectionMode'] == 'multiple')
      return DvtChartEventManager.DRAG_MODE_SELECT;
    else return null;
  }

  /**
   * Handles the zoom button click event.
   * @param {object} event
   */
  onZoomButtonClick(event) {
    if (this.zoomButton.isToggled()) {
      if (this.selectButton) this.selectButton.setToggled(false);
      this.setDragMode(DvtChartEventManager.DRAG_MODE_ZOOM);
    } else this.setDragMode(null);
  }

  /**
   * Handles the pan button click event.
   * @param {object} event
   */
  onPanButtonClick(event) {
    if (this.panButton.isToggled()) {
      if (this.selectButton) this.selectButton.setToggled(false);
      this.setDragMode(DvtChartEventManager.DRAG_MODE_PAN);
    } else this.setDragMode(null);
  }

  /**
   * Handles the select button click event.
   * @param {object} event
   */
  onSelectButtonClick(event) {
    if (this.selectButton.isToggled()) {
      if (this.zoomButton) this.zoomButton.setToggled(false);
      if (this.panButton) this.panButton.setToggled(false);
      this.setDragMode(DvtChartEventManager.DRAG_MODE_SELECT);
    } else this.setDragMode(null);
  }

  /**
   * Sets the visibility of the drag buttons.
   * @param {boolean} visible The visibility.
   */
  setDragButtonsVisible(visible) {
    if (visible && !this._dragButtonsVisible) {
      this._chart.showDragButtons();
      this._dragButtonsVisible = true;
    } else if (!visible && this._dragButtonsVisible) {
      this._chart.hideDragButtons();
      this._dragButtonsVisible = false;
    }
  }

  /**
   * Returns whether the drag buttons are visible.
   * @return {boolean}
   */
  areDragButtonsVisible() {
    return this._dragButtonsVisible;
  }

  /**
   * Toggles the marquee zoom button automatically:
   * - Marquee select button is unaffected.
   * - If the chart is fully zoomed out, turn on the marquee zoom mode; otherwise, turn it off.
   * Doesn't apply to touch devices.
   */
  autoToggleZoomButton() {
    if (Agent.isTouchDevice() || !this.zoomButton) return;

    if (this._chart.xAxis.isFullViewport() && this._chart.yAxis.isFullViewport()) {
      if (this._dragMode == DvtChartEventManager.DRAG_MODE_PAN) {
        this.zoomButton.setToggled(true);
        this.onZoomButtonClick(null);
      }
    } else {
      if (this._dragMode == DvtChartEventManager.DRAG_MODE_ZOOM) {
        this.zoomButton.setToggled(false);
        this.onZoomButtonClick(null);
      }
    }
  }

  /**
   * @override
   */
  GetTouchResponse() {
    if (this._dragMode && this._dragMode != DvtChartEventManager.DRAG_MODE_OFF) {
      return EventManager.TOUCH_RESPONSE_TOUCH_HOLD;
    } else return this._chart.getOptions()['touchResponse'];
  }

  // Drag & Drop Support
  /**
   * @override
   */
  isDndSupported() {
    return true;
  }

  /**
   * @override
   */
  GetDragSourceType() {
    var obj = this.DragSource.getDragObject();
    if (
      (obj instanceof DvtChartObjPeer && obj.getSeriesIndex() >= 0 && obj.getGroupIndex() >= 0) ||
      obj instanceof DvtChartPieSlice
    )
      return 'items';
    return null;
  }

  /**
   * @override
   */
  GetDragDataContexts(bSanitize) {
    // If more than one object is selected, return the contexts of all selected objects
    if (
      this._chart.isSelectionSupported() &&
      this._chart.getSelectionHandler().getSelectedCount() > 1
    ) {
      var selection = this._chart.getSelectionHandler().getSelection();
      var contexts = [];
      for (var i = 0; i < selection.length; i++) {
        var context = DvtChartStyleUtils.getDataContext(
          this._chart,
          selection[i].getSeriesIndex(),
          selection[i].getGroupIndex(),
          selection[i].getNestedDataItemIndex()
        );
        if (bSanitize) ToolkitUtils.cleanDragDataContext(context);
        contexts.push(context);
      }
      return contexts;
    }

    // Otherwise, return the context of the current drag object
    var obj = this.DragSource.getDragObject();
    var dataContext = null;
    if (obj instanceof DvtChartObjPeer)
      dataContext = DvtChartStyleUtils.getDataContext(
        this._chart,
        obj.getSeriesIndex(),
        obj.getGroupIndex(),
        obj.getNestedDataItemIndex()
      );
    if (obj instanceof DvtChartPieSlice)
      dataContext = DvtChartStyleUtils.getDataContext(this._chart, obj.getSeriesIndex(), 0);
    if (dataContext && bSanitize) ToolkitUtils.cleanDragDataContext(dataContext);
    return dataContext ? [dataContext] : null;
  }

  /**
   * @override
   */
  GetDropOffset(event) {
    var obj = this.DragSource.getDragObject();

    if (obj instanceof DvtChartObjPeer) {
      var dataPos = obj.getDataPosition();
      if (dataPos) {
        dataPos = this._chart.getPlotArea().localToStage(dataPos);
        var relPos = this._context.pageToStageCoords(event.pageX, event.pageY);
        return new Point(dataPos.x - relPos.x, dataPos.y - relPos.y);
      }
    }

    return null;
  }

  /**
   * @override
   */
  GetDropTargetType(event) {
    var relPos = this._context.pageToStageCoords(event.pageX, event.pageY);
    var dropOptions = this._chart.getOptions()['dnd']['drop'];

    var paBounds = this._chart.__getPlotAreaSpace();
    if (
      Object.keys(dropOptions['plotArea']).length > 0 &&
      paBounds.containsPoint(relPos.x, relPos.y)
    )
      return 'plotArea';

    if (
      Object.keys(dropOptions['xAxis']).length > 0 &&
      DvtChartAxisUtils.isAxisRendered(this._chart, 'x') &&
      DvtChartAxisUtils.axisContainsPoint(this._chart.xAxis, relPos)
    )
      return 'xAxis';

    if (
      Object.keys(dropOptions['yAxis']).length > 0 &&
      DvtChartAxisUtils.isAxisRendered(this._chart, 'y') &&
      DvtChartAxisUtils.axisContainsPoint(this._chart.yAxis, relPos)
    )
      return 'yAxis';

    if (
      Object.keys(dropOptions['y2Axis']).length > 0 &&
      DvtChartAxisUtils.isAxisRendered(this._chart, 'y2') &&
      DvtChartAxisUtils.axisContainsPoint(this._chart.y2Axis, relPos)
    )
      return 'y2Axis';

    return null;
  }

  /**
   * @override
   */
  GetDropEventPayload(event) {
    // Apply the drop offset if the drag source is a DVT component
    // NOTE: The drop offset is stored in dataTransfer, so it's only accessible from "drop" event. It can't be
    //       accessed from "dragEnter", "dragOver", and "dragLeave".
    var dataTransfer = event.getNativeEvent().dataTransfer;
    var offsetX = Number(dataTransfer.getData(EventManager.DROP_OFFSET_X_DATA_TYPE)) || 0;
    var offsetY = Number(dataTransfer.getData(EventManager.DROP_OFFSET_Y_DATA_TYPE)) || 0;

    var relPos = this._context.pageToStageCoords(event.pageX, event.pageY);
    return this._chart.getValsAt(relPos.x + offsetX, relPos.y + offsetY);
  }

  /**
   * @override
   */
  ShowDropEffect(event) {
    var dropObject = this._getDropObject(event);

    if (dropObject) {
      dropObject.setClassName('oj-active-drop');
      dropObject.setSolidFill(this._chart.getOptions()['_dropColor']);
    }
  }

  /**
   * @override
   */
  ClearDropEffect() {
    // Clear the plot area
    var plotArea = this._chart.getCache().getFromCache('plotAreaBackground');
    if (plotArea) {
      var plotAreaColor = DvtChartStyleUtils.getBackgroundColor(this._chart);
      if (plotAreaColor) plotArea.setSolidFill(plotAreaColor);
      else plotArea.setInvisibleFill();
      ToolkitUtils.removeClassName(plotArea.getElem(), 'oj-invalid-drop');
      ToolkitUtils.removeClassName(plotArea.getElem(), 'oj-active-drop');
    }

    // Clear the axes
    var clearAxisDropEffect = (axis) => {
      if (axis) {
        var background = axis.getCache().getFromCache('background');
        if (background) {
          background.setInvisibleFill();
          ToolkitUtils.removeClassName(background.getElem(), 'oj-invalid-drop');
          ToolkitUtils.removeClassName(background.getElem(), 'oj-active-drop');
        }
      }
    };
    clearAxisDropEffect(this._chart.xAxis);
    clearAxisDropEffect(this._chart.yAxis);
    clearAxisDropEffect(this._chart.y2Axis);
  }

  /**
   * @override
   */
  ShowRejectedDropEffect(event) {
    var dropObject = this._getDropObject(event);

    if (dropObject) dropObject.setClassName('oj-invalid-drop');
  }

  /**
   * Returns the background object that accepts DnD drops
   * @param {object} event
   * @return {dvt.Rect}
   * @private
   */
  _getDropObject(event) {
    var dropTargetType = this.GetDropTargetType(event);
    var dropObject;

    if (dropTargetType == 'plotArea') {
      dropObject = this._chart.getCache().getFromCache('plotAreaBackground');
    } else if (dropTargetType == 'xAxis') {
      dropObject = this._chart.xAxis.getCache().getFromCache('background');
    } else if (dropTargetType == 'yAxis') {
      dropObject = this._chart.yAxis.getCache().getFromCache('background');
    } else if (dropTargetType == 'y2Axis') {
      dropObject = this._chart.y2Axis.getCache().getFromCache('background');
    }

    return dropObject;
  }
}

/** @const */
DvtChartEventManager.DRAG_MODE_PAN = 'pan';
/** @const */
DvtChartEventManager.DRAG_MODE_ZOOM = 'zoom';
/** @const */
DvtChartEventManager.DRAG_MODE_SELECT = 'select';
/** @const */
DvtChartEventManager.DRAG_MODE_OFF = 'off';

/*---------------------------------------------------------------------------------*/
/*  DvtChartKeyboardHandler     Keyboard handler for Chart                         */
/*---------------------------------------------------------------------------------*/
/**
 *  @param {dvt.EventManager} manager The owning dvt.EventManager
 *  @param {Chart} chart
 *  @class DvtChartKeyboardHandler
 *  @extends {dvt.KeyboardHandler}
 *  @constructor
 */
class DvtChartKeyboardHandler extends KeyboardHandler {
  constructor(manager, chart) {
    super(manager);
    this._chart = chart;
  }

  /**
   * @override
   */
  isSelectionEvent(event) {
    return this.isNavigationEvent(event) && !event.ctrlKey;
  }

  /**
   * @override
   */
  isMultiSelectEvent(event) {
    return event.keyCode == KeyboardEvent.SPACE && event.ctrlKey;
  }

  /**
   * @override
   */
  processKeyDown(event) {
    var keyCode = event.keyCode;
    if (keyCode == KeyboardEvent.TAB) {
      var currentNavigable = this._eventManager.getFocus();
      if (currentNavigable) {
        EventManager.consumeEvent(event);
        return currentNavigable;
      }

      // navigate to the default
      var navigables = DvtChartEventUtils.getKeyboardNavigables(this._chart);
      if (navigables.length > 0) {
        EventManager.consumeEvent(event);
        return this.getDefaultNavigable(navigables);
      }
    } else if (keyCode == KeyboardEvent.ENTER) {
      var currentNavigable = this._eventManager.getFocus();
      if (currentNavigable) {
        this._eventManager.processDrillEvent(currentNavigable);
        EventManager.consumeEvent(event);
        return currentNavigable;
      }
    } else if (keyCode == KeyboardEvent.ESCAPE) {
      this._eventManager.cancelMarquee(event);
    } else if (keyCode == KeyboardEvent.PAGE_UP) {
      if (
        (event.ctrlKey || event.shiftKey || DvtChartTypeUtils.isBLAC(this._chart)) &&
        DvtChartTypeUtils.isVertical(this._chart)
      ) {
        // pan left
        this._eventManager.panBy(-0.25, 0);
      } else {
        // pan up. Also used for horizontal bar charts
        this._eventManager.panBy(0, -0.25);
      }
      EventManager.consumeEvent(event);
    } else if (keyCode == KeyboardEvent.PAGE_DOWN) {
      if (
        (event.ctrlKey || event.shiftKey || DvtChartTypeUtils.isBLAC(this._chart)) &&
        DvtChartTypeUtils.isVertical(this._chart)
      ) {
        // pan right
        this._eventManager.panBy(0.25, 0);
      } else {
        // pan down. Also used for horizontal bar charts
        this._eventManager.panBy(0, 0.25);
      }
      EventManager.consumeEvent(event);
    } else if (KeyboardEvent.isEquals(event) || KeyboardEvent.isPlus(event)) {
      // zoom in
      this._eventManager.zoomBy(1.5);
    } else if (KeyboardEvent.isMinus(event) || KeyboardEvent.isUnderscore(event)) {
      // zoom out
      this._eventManager.zoomBy(1 / 1.5);
    }

    return super.processKeyDown(event);
  }

  /**
   * @override
   */
  getDefaultNavigable(navigableItems) {
    if (!navigableItems || navigableItems.length <= 0) return null;

    var isPie = DvtChartTypeUtils.isPie(this._chart);
    var defaultNavigable, defaultSeries, defaultGroup;
    var navigable;

    // Pick the first group in the first series
    for (var i = 0; i < navigableItems.length; i++) {
      navigable = navigableItems[i];

      if (!defaultNavigable || navigable.getSeriesIndex() < defaultSeries) {
        defaultNavigable = navigable;
        defaultSeries = navigable.getSeriesIndex();
        if (!isPie) defaultGroup = navigable.getGroupIndex();
        continue;
      }

      if (!isPie && navigable.getGroupIndex() < defaultGroup) {
        defaultNavigable = navigable;
        defaultSeries = navigable.getSeriesIndex();
        defaultGroup = navigable.getGroupIndex();
      }
    }

    return defaultNavigable;
  }
}

/**
 *  Data change handler for a chart object peer.
 *  @extends {dvt.Obj}
 *  @class DvtChartDataChange  Data change Handler for a chart object peer.
 *  @constructor
 *  @param {DvtChartObjPeer} peer  The chart object peer to be animated on datachange.
 *  @param {Number} duration  the duration of the animation in seconds.
 */
class DvtChartDataChangeHandler {
  constructor(peer, duration, animId) {
    this._peer = peer;
    this._updateDuration = duration * 0.75;
    this._insertDuration = duration * 0.5;
    this._deleteDuration = duration * 0.5;
    this._shape = peer.getDisplayables()[0];
    this._animId = (peer.getDataItemId() || peer.getSeries() + '/' + peer.getGroup()) + animId;
  }

  /**
   * Creates an update animation from the old node to this node.
   * @param {dvt.DataAnimationHandler} handler The animation handler, which can
   *                                  be used to chain animations. Animations
   *                                  created should be added via
   *                                  dvt.DataAnimationHandler.add()
   * @param {DvtChartDataChange} oldNode The old node state to animate from.
   */
  animateUpdate(handler, oldNode) {
    var oldShape = oldNode._shape;

    // Use update animation defined by the shape if available.
    if (this._shape && this._shape.getUpdateAnim)
      handler.add(this._shape.getUpdateAnim(this._updateDuration, oldShape), 1);
  }

  /**
   * Creates an insert animation for this node.
   * @param {dvt.DataAnimationHandler} handler The animation handler, which can
   *                                  be used to chain animations. Animations
   *                                  created should be added via
   *                                  dvt.DataAnimationHandler.add()
   */
  animateInsert(handler) {
    // Use insert animation defined by the shape if available.
    if (this._shape && this._shape.getInsertAnim)
      handler.add(this._shape.getInsertAnim(this._insertDuration), 2);
    else {
      // Fade In
      var nodePlayable = new AnimFadeIn(
        this._shape.getCtx(),
        this._shape,
        this._insertDuration
      );
      handler.add(nodePlayable, 0);
    }
  }

  /**
   * Creates a delete animation for this node.
   * @param {dvt.DataAnimationHandler} handler The animation handler, which can
   *                                  be used to chain animations. Animations
   *                                  created should be added via
   *                                  dvt.DataAnimationHandler.add()
   * @param {dvt.Container} delContainer   The container to which deleted objects can
   *                                      be moved for animation.
   */
  animateDelete(handler, delContainer) {
    // Move from the old chart to the delete container on top of the new chart.
    delContainer.addChild(this._shape);

    // Use the delete animation defined by the shape if available.
    if (this._shape && this._shape.getDeleteAnim)
      handler.add(this._shape.getDeleteAnim(this._deleteDuration), 0);
    else {
      // Fade Out
      var nodePlayable = new AnimFadeOut(
        this._shape.getCtx(),
        this._shape,
        this._deleteDuration
      );
      handler.add(nodePlayable, 0);
    }
  }

  /**
   * @return {String} A unique id for object comparison during data change animation of a chart.
   */
  getId() {
    return this._animId;
  }

  /**
   * Sets the id for the data change object. Generally the id is created automatically, so this method should only be
   * used if the default id needs to be overridden.
   * @param {String} id A unique id for object comparison during data change animation of a chart.
   */
  setId(id) {
    this._animId = id;
  }

  /**
   *   Saves the psuedo old chart object.
   *   @param {Object} chart  a synopsis object created by Chart before
   *   the chart object is updated and rendered with new data.
   */
  setOldChart(chart) {
    this._oldChart = chart;
  }
}

/*---------------------------------------------------------------------*/
/*  DvtChartDataChangeUtils()                                                       */
/*---------------------------------------------------------------------*/

// Utilities for Chart data change animations

/**
 * @constructor
 */
const DvtChartDataChangeUtils = {
  DIR_UP: 0, // pointer directions
  DIR_DOWN: 1,
  DIR_NOCHANGE: 2,

  /**
   * Creates an update value direction pointer and positions it.
   * @param {Object} oldChart old chart.
   * @param {number} oldSIdx old series index.
   * @param {number} oldGIdx old group index.
   * @param {Chart} newChart new chart.
   * @param {number} newSIdx new series index.
   * @param {number} newGIdx new group index.
   * @return {dvt.Path} indicator.
   */
  makeIndicator: (oldChart, oldSIdx, oldGIdx, newChart, newSIdx, newGIdx) => {
    if (DvtChartTypeUtils.isPolar(newChart)) return null;

    var uiDirection = DvtChartDataChangeUtils.getDirection(
      oldChart,
      oldSIdx,
      oldGIdx,
      newChart,
      newSIdx,
      newGIdx
    );
    if (uiDirection == DvtChartDataChangeUtils.DIR_NOCHANGE) return null;

    var bDown = uiDirection === DvtChartDataChangeUtils.DIR_DOWN;
    var fc = bDown
      ? DvtChartStyleUtils.getAnimDownColor(newChart)
      : DvtChartStyleUtils.getAnimUpColor(newChart);

    //  Create a path object that draws the indicator (it will be positioned in _setAnimParams).
    var indicator = DvtChartDataChangeUtils._drawIndicator(
      newChart.getCtx(),
      bDown,
      DvtChartTypeUtils.isHorizontal(newChart),
      fc
    );
    newChart.getPlotArea().addChild(indicator);
    return indicator;
  },

  /**
   * Returns the direction of data change for use with animation indicators
   * @param {Object} oldChart old chart.
   * @param {number} oldSIdx old series index.
   * @param {number} oldGIdx old group index.
   * @param {Chart} newChart new chart.
   * @param {number} newSIdx new series index.
   * @param {number} newGIdx new group index.
   * @return {number} direction.
   */
  getDirection: (oldChart, oldSIdx, oldGIdx, newChart, newSIdx, newGIdx) => {
    var oldValue = DvtChartDataUtils.getVal(oldChart, oldSIdx, oldGIdx);
    var newValue = DvtChartDataUtils.getVal(newChart, newSIdx, newGIdx);

    if (newValue == null || oldValue == null || newValue == oldValue)
      return DvtChartDataChangeUtils.DIR_NOCHANGE;

    return newValue > oldValue ? DvtChartDataChangeUtils.DIR_UP : DvtChartDataChangeUtils.DIR_DOWN;
  },

  /**
   * Creates and returns a dvt.Path centered at (0,0) for the animation indicator.
   * @param {dvt.Context} context
   * @param {boolean} bDown True if the indicator represents a decrease in value.
   * @param {boolean} bHoriz True if the y axis is horizontal.
   * @param {string} fc The fill color of the indicator
   * @return {dvt.Path}
   * @private
   */
  _drawIndicator: (context, bDown, bHoriz, fc) => {
    var ptrCmds;
    if (bHoriz) {
      var bLeft = Agent.isRightToLeft(context) ? !bDown : bDown;
      ptrCmds = bLeft ? 'M3.5,-5L3.5,5L-3.5,0L3.5,-5' : 'M-3.5,-5L-3.5,5L3.5,0L-3.5,-5';
    } // Vertical
    else ptrCmds = bDown ? 'M-5,-3.5L5,-3.5L0,3.5L-5,-3.5Z' : 'M-5,3.5L5,3.5L0,-3.5L-5,3.5Z';

    var cssClass = bDown ? 'oj-chart-animation-down' : 'oj-chart-animation-up';
    var ret = new Path(context, ptrCmds);
    ret.setClassName(cssClass);
    ret.setStyle({ fill: fc });
    return ret;
  }
};

/**
 *  Data change Handler for 2D Bar Riser (implements DvtChartDataChangeHandler).
 *  @extends {DvtChartDataChangeHandler}
 *  @class DvtChartDataChangeBar  Data change Handler for 2D Bar Riser.
 *  @constructor
 *  @param {DvtChartObjPeer} peer The chart object peer for the shape to be animated.
 *  @param {number} duration  the animation duration is seconds.
 */
class DvtChartDataChangeBar extends DvtChartDataChangeHandler {
  constructor(peer, duration, animId) {
    super(peer, duration, animId);
    this._indicator = null;
  }

  /**
   * @override
   */
  animateInsert(handler) {
    var playable = this._shape.getInsertAnim(this._insertDuration);
    handler.add(playable, 2);
  }

  /**
   * @override
   */
  animateDelete(handler, delContainer) {
    // Move from the old chart to the new chart
    delContainer.addChild(this._shape);

    // Create the delete animation
    var playable = this._shape.getDeleteAnim(this._deleteDuration);
    handler.add(playable, 0);
  }

  /**
   * @override
   */
  animateUpdate(handler, oldDC) {
    var oldChart = this._oldChart;
    var newChart = this._peer.getChart();

    // Get the start and end state for the animation. Get flipped coordinates if orientation has changed.
    var bFlip =
      DvtChartTypeUtils.isHorizontal(oldChart) != DvtChartTypeUtils.isHorizontal(newChart);
    var startState = oldDC._getAnimParams(bFlip);
    var endState = this._getAnimParams();

    // Get the start and end state for the fill animation. If either shape is selected, skip the fill animation so
    // that it doesn't animate the black fill of the selection outer shape
    var startFill = oldDC._shape.getPrimaryFill();
    var endFill = this._shape.getPrimaryFill();
    var bSkipFillAnimation =
      oldDC._shape.isSelected() || this._shape.isSelected() || startFill.equals(endFill);

    if (ArrayUtils.equals(startState, endState) && startFill.equals(endFill)) return;

    var newSIdx = this._peer.getSeriesIndex();
    var oldSIdx = oldDC._peer.getSeriesIndex();
    var newGIdx = this._peer.getGroupIndex();
    var oldGIdx = oldDC._peer.getGroupIndex();

    //  Create an animate indicator if requested
    if (DvtChartStyleUtils.getAnimIndicators(newChart) !== 'none')
      this._indicator = DvtChartDataChangeUtils.makeIndicator(
        oldChart,
        oldSIdx,
        oldGIdx,
        newChart,
        newSIdx,
        newGIdx
      );

    // Initialize start state
    this._setAnimParams(startState);
    if (!bSkipFillAnimation) this._shape.setFill(startFill);

    // Create the animator for this bar update
    var nodePlayable = new CustomAnimation(this._shape.getCtx(), this, this._updateDuration);
    nodePlayable
      .getAnimator()
      .addProp(
        Animator.TYPE_NUMBER_ARRAY,
        this,
        this._getAnimParams,
        this._setAnimParams,
        endState
      );

    if (!bSkipFillAnimation)
      nodePlayable
        .getAnimator()
        .addProp(
          Animator.TYPE_FILL,
          this._shape,
          this._shape.getFill,
          this._shape.setFill,
          endFill
        );

    if (this._indicator) {
      nodePlayable.setOnEnd(this._onEndAnim, this);
      this._indicator.setAlpha(0);
    }

    handler.add(nodePlayable, 1); // create the playable
  }

  /**
   * Returns the geometry of the bar.
   * @param {boolean} bFlip True if the result should be flipped for horizontal/vertical orientation change.
   * @return {Array}
   * @private
   */
  _getAnimParams(bFlip) {
    return this._shape.getAnimParams(bFlip);
  }

  /**
   * Updates the geometry of the bar.
   * @param {Array} ar  an array containing the polygon points.
   * @private
   */
  _setAnimParams(ar) {
    this._shape.setAnimParams(ar, this._indicator);
  }

  /**
   * Callback to remove the indicator object at the end of the animation.
   * @private
   */
  _onEndAnim() {
    if (this._indicator) {
      this._indicator.getParent().removeChild(this._indicator);
      this._indicator = null;
    }
  }
}

/**
 *  Data change Handler for DvtChartFunnelSlice (implements DvtChartDataChangeHandler).
 *  @extends {DvtChartDataChangeHandler}
 *  @class DvtChartDataChangeFunnelSlice  Data change Handler for Funnel Slices.
 *  @constructor
 *  @param {DvtChartObjPeer} peer  The chart object peer for the shape to be animated.
 *  @param {Number} duration  the animation duration is seconds.
 */
class DvtChartDataChangeFunnelSlice extends DvtChartDataChangeHandler {
  /**
   * @override
   */
  animateUpdate(handler, oldDC) {
    var obj = this._shape;

    var startState = oldDC._shape.getAnimParams();
    var endState = obj.getAnimParams();

    var startFill = oldDC._shape.getFill();
    var endFill = this._shape.getFill();

    if (ArrayUtils.equals(startState, endState) && startFill.equals(endFill))
      // if no change,
      return; // nothing to animate.

    // Initialize start state
    obj.setAnimParams(startState);

    // Create the animator for this slice update
    var nodePlayable = new CustomAnimation(obj.getCtx(), this, this._updateDuration);
    nodePlayable
      .getAnimator()
      .addProp(Animator.TYPE_NUMBER_ARRAY, obj, obj.getAnimParams, obj.setAnimParams, endState);

    // TODO  this only works for slices without target values. Slices with target values are drawn within
    // DvtChartFunnelSlice, so wait until the animation code is collapsed to do that work.
    if (!startFill.equals(endFill)) {
      this._shape.setFill(startFill);
      nodePlayable
        .getAnimator()
        .addProp(Animator.TYPE_FILL, obj, obj.getFill, obj.setFill, endFill);
    }

    // TODO  this line of code makes no sense, since we never draw an indicator for funnel
    if (this._indicator) {
      nodePlayable.setOnEnd(this._onEndAnim, this);
    }

    handler.add(nodePlayable, 1); // create the playable
  }

  /**
   * @override
   */
  animateInsert(handler) {
    var obj = this._shape;

    var endState = obj.getAnimParams();
    var startState = endState.slice(0);
    startState[0] += startState[1] / 2;
    startState[1] = 0;
    startState[3] = 0; // start alpha

    obj.setAnimParams(startState); // set the start state
    var nodePlayable = new CustomAnimation(obj.getCtx(), this, this._insertDuration);
    nodePlayable
      .getAnimator()
      .addProp(Animator.TYPE_NUMBER_ARRAY, obj, obj.getAnimParams, obj.setAnimParams, endState);

    handler.add(nodePlayable, 2); // create the playable
  }

  /**
   * @override
   */
  animateDelete(handler, delContainer) {
    var obj = this._shape;

    delContainer.setClipPath(null); // remove clipping
    delContainer.addChild(obj); // move from existing container to the delete container on top of the new chart.

    var startState = obj.getAnimParams();
    var endState = startState.slice(0);

    endState[0] += startState[1] / 2;
    endState[1] = 0;
    endState[3] = 0; // end alpha

    var nodePlayable = new CustomAnimation(obj.getCtx(), this, this._deleteDuration);
    nodePlayable
      .getAnimator()
      .addProp(Animator.TYPE_NUMBER_ARRAY, obj, obj.getAnimParams, obj.setAnimParams, endState);

    handler.add(nodePlayable, 0); // create the playable
  }
}

/**
 *  Data change Handler for Line or Area (implements DvtChartDataChange).
 *  @extends {dvt.Obj}
 *  @class DvtChartDataChangeLineArea  Data change Handler for Line and Area.
 *  @constructor
 *  @param {DvtChartObjPeer} peer  The chart object peer for the shape to be animated.
 *  @param {number} duration  the animation duration is seconds.
 */
class DvtChartDataChangeLineArea extends DvtChartDataChangeHandler {
  constructor(peer, duration) {
    super(peer, duration, '');
    this._context = this._shape.getCtx();
    this._chart = this._peer.getChart();
    this._animId += '/' + (this._shape.isArea() ? 'area' : 'line');
  }
  /**
   * Creates the update animation for this Line or Area. Insert/delete of
   * groups within an existing series is treated as a special case of animateUpdate.
   * @param {dvt.DataAnimationHandler} handler The animation handler, which can be
   *                                          used to chain animations.
   * @param {DvtChartDataChangeLineArea} oldDC   The old node DC Handler to animate from.
   * @override
   */
  animateUpdate(handler, oldDC) {
    this._baseCoords = this._shape.getBaseCoords();
    this._coords = this._shape.getCoords();
    var isArea = this._shape.isArea();

    var oldChart = this._oldChart;
    var newChart = this._chart;
    var newSIdx = this._peer.getSeriesIndex();
    var oldSIdx = oldDC._peer.getSeriesIndex();
    var newGIdcs = this._shape.getCommonGroupIndices(oldDC._shape);
    var oldGIdcs = oldDC._shape.getCommonGroupIndices(this._shape);

    // Construct animation for the area base.
    var baseAnim;
    if (isArea) {
      var baseStartState = oldDC._getBaseAnimParams(this._shape);
      var baseEndState = this._getBaseAnimParams(oldDC._shape);
      DvtChartDataChangeLineArea._matchGroupIndices(baseStartState, baseEndState);
      if (!ArrayUtils.equals(baseStartState, baseEndState)) {
        this._setBaseAnimParams(baseStartState); // initialize the start state
        baseAnim = new CustomAnimation(this._context, this, this._updateDuration);
        baseAnim
          .getAnimator()
          .addProp(
            Animator.TYPE_NUMBER_ARRAY,
            this,
            this._getBaseAnimParams,
            this._setBaseAnimParams,
            baseEndState
          );
      }
    }

    // Construct animation for the line or the area top.
    var topAnim;
    var startState = oldDC._getAnimParams(this._shape);
    var endState = this._getAnimParams(oldDC._shape);
    DvtChartDataChangeLineArea._matchGroupIndices(startState, endState);
    if (!ArrayUtils.equals(startState, endState)) {
      this._setAnimParams(startState); // initialize the start state
      topAnim = new CustomAnimation(this._context, this, this._updateDuration);
      topAnim
        .getAnimator()
        .addProp(
          Animator.TYPE_NUMBER_ARRAY,
          this,
          this._getAnimParams,
          this._setAnimParams,
          endState
        );
    }

    // Create animate indicators if requested. If seriesType is lineWithArea, add indicators only to the line.
    var seriesType = DvtChartDataUtils.getSeriesType(this._peer.getChart(), newSIdx);
    if (
      DvtChartStyleUtils.getAnimIndicators(newChart) !== 'none' &&
      !(isArea && seriesType == 'lineWithArea')
    ) {
      var direction, indicator;
      for (var i = 0; i < newGIdcs.length; i++) {
        direction = DvtChartDataChangeUtils.getDirection(
          oldChart,
          oldSIdx,
          oldGIdcs[i],
          newChart,
          newSIdx,
          newGIdcs[i]
        );
        indicator = DvtChartDataChangeUtils.makeIndicator(
          oldChart,
          oldSIdx,
          oldGIdcs[i],
          newChart,
          newSIdx,
          newGIdcs[i]
        );
        if (indicator) this._shape.addIndicator(newGIdcs[i], direction, indicator);
      }
    }

    // Combine the top and base animation.
    if (baseAnim || topAnim) {
      var nodePlayable = new ParallelPlayable(this._context, baseAnim, topAnim);
      nodePlayable.setOnEnd(this._onAnimEnd, this);
      handler.add(nodePlayable, 1);
    }
  }

  /**
   * Creates the insert animation for this Line or Area.
   * @param {dvt.DataAnimationHandler} handler The animation handler, which can be used to chain animations.
   * @override
   */
  animateInsert(handler) {
    this._shape.setAlpha(0); // set alpha=0 so that the inserted object is hidden until the insert animation starts

    var nodePlayable = new AnimFadeIn(this._context, this._shape, this._insertDuration);
    handler.add(nodePlayable, 2);
  }

  /**
   * Creates the delete animation for this Line or Area
   * @param {dvt.DataAnimationHandler} handler The animation handler, which can be used to
   *                                          chain animations.
   * @param {dvt.Container} delContainer   The container to which the deleted objects should
   *                                      be moved for animation.
   * @override
   */
  animateDelete(handler, delContainer) {
    var nodePlayable;
    var seriesType = DvtChartDataUtils.getSeriesType(this._oldChart, this._peer.getSeriesIndex()); // get from old chart

    if (seriesType == 'area') {
      // For area chart, we need to add and fade out all of the areas (not just the deleted ones) to make sure that the
      // areas are in the correct z-order. Furthermore, the delContainer should be the areaContainer of the new chart
      // so that the deleted areas appear below the gridlines and other data items.
      var areaContainer = this._chart.__getAreaContainer(); // new chart's areaContainer
      this._deletedAreas = this._shape.getParent().getParent(); // the parent is the clipGroup, and the grandparent is the old chart's areaContainer
      if (areaContainer) areaContainer.addChild(this._deletedAreas);
      else return;
      nodePlayable = new AnimFadeOut(this._context, this._deletedAreas, this._deleteDuration);
      nodePlayable.setOnEnd(this._removeDeletedAreas, this);
      handler.add(nodePlayable, 0);
    } else {
      // Move from the old chart to the delete container on top of the new chart.
      delContainer.addChild(this._shape);
      nodePlayable = new AnimFadeOut(this._context, this._shape, this._deleteDuration);
      handler.add(nodePlayable, 0);
    }
  }

  /**
   * Removes the deleted areas at the end of delete animation.
   * @private
   */
  _removeDeletedAreas() {
    var areaContainer = this._chart.__getAreaContainer();
    if (areaContainer) areaContainer.removeChild(this._deletedAreas);
  }

  /**
   * Returns the animation params for the line or the area top.
   * @param {DvtChartLineArea} otherShape
   * @return {array} params
   * @private
   */
  _getAnimParams(otherShape) {
    return this._shape.getAnimParams(otherShape);
  }

  /**
   * Updates the animation params for the line or the area top.
   * @param {array} params
   * @private
   */
  _setAnimParams(params) {
    this._shape.setAnimParams(params);
  }

  /**
   * Returns the animation params for the area base.
   * @param {DvtChartLineArea} otherShape
   * @return {array} params
   * @private
   */
  _getBaseAnimParams(otherShape) {
    return this._shape.getBaseAnimParams(otherShape);
  }

  /**
   * Updates the animation params for the area base.
   * @param {array} params
   * @private
   */
  _setBaseAnimParams(params) {
    this._shape.setBaseAnimParams(params);
  }

  /**
   * Clean up at the end of update animation.
   * @private
   */
  _onAnimEnd() {
    this._shape.removeIndicators();
    this._shape.setCoords(this._coords, this._baseCoords);
  }

  /**
   * Sets the group indices of the startParams to be the group indices of the endParams.
   * Required because the group indices of the startParams is taken from the oldChart.
   * @param {array} startParams
   * @param {array} endParams
   * @private
   */
  static _matchGroupIndices(startParams, endParams) {
    // group index is the 4th, 8th, 12th... param
    for (var i = 3; i < startParams.length; i += 4) {
      startParams[i] = endParams[i];
    }
  }
}

/**
 *  Data change Handler for markers.
 *  @extends {DvtChartDataChangeHandler}
 *  @class DvtChartDataChangeMarker  Data change Handler for markers.
 *  @constructor
 *  @param {DvtChartObjPeer} peer  The chart object peer for the shape to be animated.
 *  @param {Number} duration  The animation duration is seconds.
 */
class DvtChartDataChangeMarker extends DvtChartDataChangeHandler {
  /**
   * @override
   */
  animateUpdate(handler, oldDC) {
    var startRect = oldDC._shape.getCenterDimensions();
    var endRect = this._shape.getCenterDimensions();

    // Return if no change in the geometry
    if (endRect.equals(startRect)) return;

    // Initialize the start state
    this._shape.setCenterDimensions(startRect);

    // Create the animator for this node
    var nodePlayable = new CustomAnimation(this._shape.getCtx(), this, this._updateDuration);
    nodePlayable
      .getAnimator()
      .addProp(
        Animator.TYPE_RECTANGLE,
        this._shape,
        this._shape.getCenterDimensions,
        this._shape.setCenterDimensions,
        endRect
      );

    // If animation indicators required, and the value changed, add visual effect to marker.
    var chart = this._peer.getChart();
    if (
      this.isValueChange(oldDC) &&
      DvtChartStyleUtils.getAnimIndicators(chart) != 'none' &&
      DvtChartTypeUtils.isScatterBubble(chart)
    ) {
      // Use the old shape for the update color overlay
      var overlay = oldDC._shape;
      overlay.setAlpha(0.9);
      overlay.setClassName('oj-chart-animation-marker');
      overlay.setCenterDimensions(startRect);
      this._peer.getChart().getPlotArea().addChild(overlay);

      //  Move and fade the overlay
      nodePlayable
        .getAnimator()
        .addProp(
          Animator.TYPE_RECTANGLE,
          overlay,
          overlay.getCenterDimensions,
          overlay.setCenterDimensions,
          endRect
        );
      nodePlayable
        .getAnimator()
        .addProp(Animator.TYPE_NUMBER, overlay, overlay.getAlpha, overlay.setAlpha, 0);

      // Set end listener to remove the overlay
      this._overlay = overlay;
      nodePlayable.setOnEnd(this._onEndAnim, this);
    }

    handler.add(nodePlayable, 1);
  }

  /**
   * @override
   */
  animateInsert(handler) {
    this._shape.setAlpha(0);
    var nodePlayable = new AnimFadeIn(this._shape.getCtx(), this._shape, this._insertDuration);

    handler.add(nodePlayable, 2);
  }

  /**
   * @override
   */
  animateDelete(handler, delContainer) {
    delContainer.addChild(this._shape); // Move from the old chart to the delete
    // container on top of the new chart.

    var nodePlayable = new AnimFadeOut(this._shape.getCtx(), this._shape, this._deleteDuration);

    handler.add(nodePlayable, 0);
  }

  /**
   * Check if there is data change.
   * @param {DvtChartDataChangeMarker} oldDC    The old node state to animate from.
   * @return {boolean}  true if node data has changed.
   */
  isValueChange(oldDC) {
    var bRet = false;

    if (oldDC) {
      var oldSIdx = oldDC._peer.getSeriesIndex();
      var oldGIdx = oldDC._peer.getGroupIndex();
      var newSIdx = this._peer.getSeriesIndex();
      var newGIdx = this._peer.getGroupIndex();
      var oldData = oldDC._oldChart.getOptions();
      var newData = this._peer.getChart().getOptions();

      var oldX = oldData['series'][oldSIdx]['items'][oldGIdx]['x'];
      var oldY = oldData['series'][oldSIdx]['items'][oldGIdx]['y'];
      var oldZ = oldData['series'][oldSIdx]['items'][oldGIdx]['z'];
      var newX = newData['series'][newSIdx]['items'][newGIdx]['x'];
      var newY = newData['series'][newSIdx]['items'][newGIdx]['y'];
      var newZ = newData['series'][newSIdx]['items'][newGIdx]['z'];

      bRet = newX !== oldX || newY !== oldY || newZ !== oldZ;
    }

    return bRet;
  }

  /**
   * Remove update animation overlay
   * @private
   */
  _onEndAnim() {
    if (this._overlay) {
      this._peer.getChart().getPlotArea().removeChild(this._overlay);
      this._overlay = null;
    }
  }
}

/**
 *  Data change Handler for DvtChartPyramidSlice (implements DvtChartDataChange).
 *  @extends {DvtChartDataChangeHandler}
 *  @class DvtChartDataChangePyramidSlice  Data change Handler for Pyramid Slices.
 *  @constructor
 *  @param {DvtChartObjPeer} peer  The chart object peer for the shape to be animated.
 *  @param {Number} duration  the animation duration is seconds.
 */
class DvtChartDataChangePyramidSlice extends DvtChartDataChangeHandler {
  /**
   * @override
   */
  animateUpdate(handler, oldDC) {
    var obj = this._shape;

    var startState = oldDC._shape.getAnimParams();
    var endState = obj.getAnimParams();

    var startFill = oldDC._shape.getPrimaryFill();
    var endFill = this._shape.getPrimaryFill();

    if (ArrayUtils.equals(startState, endState) && startFill.equals(endFill))
      // if no change,
      return; // nothing to animate.

    // Initialize start state
    obj.setAnimParams(startState);

    // Create the animator for this slice update
    var nodePlayable = new CustomAnimation(obj.getCtx(), this, this._updateDuration);
    nodePlayable
      .getAnimator()
      .addProp(Animator.TYPE_NUMBER_ARRAY, obj, obj.getAnimParams, obj.setAnimParams, endState);

    if (!startFill.equals(endFill)) {
      this._shape.setFill(startFill);
      nodePlayable
        .getAnimator()
        .addProp(Animator.TYPE_FILL, obj, obj.getFill, obj.setFill, endFill);
    }

    handler.add(nodePlayable, 1); // create the playable
  }

  /**
   * @override
   */
  animateInsert(handler) {
    var obj = this._shape;

    var endState = obj.getAnimParams();
    var startState = endState.slice(0);
    startState[1] = 0;
    startState[2] = 0; // start alpha

    obj.setAnimParams(startState); // set the start state
    var nodePlayable = new CustomAnimation(obj.getCtx(), this, this._insertDuration);
    nodePlayable
      .getAnimator()
      .addProp(Animator.TYPE_NUMBER_ARRAY, obj, obj.getAnimParams, obj.setAnimParams, endState);

    handler.add(nodePlayable, 2); // create the playable
  }

  /**
   * @override
   */
  animateDelete(handler, delContainer) {
    var obj = this._shape;

    delContainer.setClipPath(null); // remove clipping
    delContainer.addChild(obj); // move from existing container to the delete container on top of the new chart.

    var startState = obj.getAnimParams();
    var endState = startState.slice(0);

    endState[1] = 0;
    endState[2] = 0; // end alpha

    var nodePlayable = new CustomAnimation(obj.getCtx(), this, this._deleteDuration);
    nodePlayable
      .getAnimator()
      .addProp(Animator.TYPE_NUMBER_ARRAY, obj, obj.getAnimParams, obj.setAnimParams, endState);
    handler.add(nodePlayable, 0); // create the playable
  }
}

/**
 *  Data change handler for range markers (implements DvtChartDataChange).
 *  @extends {DvtChartDataChangeHandler}
 *  @class DvtChartDataChangeRangeMarker  Data change Handler for range markers.
 *  @constructor
 *  @param {DvtChartObjPeer} peer The chart object peer for the shape to be animated.
 *  @param {number} duration  the animation duration is seconds.
 */
class DvtChartDataChangeRangeMarker extends DvtChartDataChangeHandler {
  /**
   * @override
   */
  animateInsert(handler) {
    this._shape.setAlpha(0);
    var nodePlayable = new AnimFadeIn(this._shape.getCtx(), this._shape, this._insertDuration);

    handler.add(nodePlayable, 2);
  }

  /**
   * @override
   */
  animateDelete(handler, delContainer) {
    delContainer.addChild(this._shape); // Move from the old chart to the delete
    // container on top of the new chart.

    var nodePlayable = new AnimFadeOut(this._shape.getCtx(), this._shape, this._deleteDuration);

    handler.add(nodePlayable, 0);
  }

  /**
   * @override
   */
  animateUpdate(handler, oldDC) {
    var start = oldDC._shape.getAnimParams();
    var end = this._shape.getAnimParams();

    // Initialize the start state
    this._shape.setAnimParams(start);

    // Create the animator for this node
    var nodePlayable = new CustomAnimation(this._shape.getCtx(), this, this._updateDuration);
    nodePlayable
      .getAnimator()
      .addProp(
        Animator.TYPE_NUMBER_ARRAY,
        this._shape,
        this._shape.getAnimParams,
        this._shape.setAnimParams,
        end
      );

    handler.add(nodePlayable, 1);
  }
}

/**
 * A selectable polygon displayable.
 * @class DvtChartSelectableRectangularPolygon
 * @extends {dvt.Polygon}
 * @constructor
 * @param {dvt.Context} context
 * @param {array} arPoints The array of coordinates for this polygon, in the form [x1,y1,x2,y2...].
 * @param {string} id The optional id for the corresponding DOM element.
 */
class DvtChartSelectableRectangularPolygon extends Polygon {
  constructor(context, arPoints, id, doNotRender) {
    let coords, _x1, _x2, _y1, _y2;
    if (arPoints) {
      _x1 = Math.min(arPoints[0], arPoints[4]);
      _x2 = Math.max(arPoints[0], arPoints[4]);
      _y1 = Math.min(arPoints[1], arPoints[5]);
      _y2 = Math.max(arPoints[1], arPoints[5]);
      coords = [_x1, _y1, _x2, _y1, _x2, _y2, _x1, _y2];
    }

    super(context, coords, id, doNotRender);
    this._x1 = _x1;
    this._x2 = _x2;
    this._y1 = _y1;
    this._y2 = _y2;
  }

  /**
   * Specifies the colors needed to generate the selection effect.
   * @param {dvt.Fill} fill
   * @param {dvt.Stroke} stroke
   * @param {string} dataColor The color of the data.
   * @param {string} innerColor The color of the inner selection border.
   * @param {string} outerColor The color of the outer selection border.
   * @param {string} className The className of the shape.
   * @param {object} style The style of the shape.
   */
  setStyleProperties(fill, stroke, dataColor, innerColor, outerColor, className, style) {
    this._fill = fill;
    // Save original stroke style to get reapplied in _showNestedBorders. Cannot use this._stroke, as it gets overwritten during select and hover
    this._borderStroke = stroke;
    this._hoverColor = SelectionEffectUtils.getHoverBorderColor(dataColor);
    this._innerColor = innerColor;
    this._outerColor = outerColor;
    this._shapeClassName = className;
    this._shapeStyle = style;
    this.setStyle(style).setClassName(className);

    // Apply the fill and stroke
    this.setFill(fill);
    if (stroke) this.setStroke(stroke);
  }

  /**
   * To allow the updating of the size of the child shapes during animation
   * @param {array} ar The array of points.
   */
  setAnimParams(ar) {
    this._x1 = Math.min(ar[0], ar[4]);
    this._x2 = Math.max(ar[0], ar[4]);
    this._y1 = Math.min(ar[1], ar[5]);
    this._y2 = Math.max(ar[1], ar[5]);
    this.setPoints(ar);
    this._initializeSelectionEffects();
  }

  /**
   * @override
   */
  showHoverEffect() {
    if (this.IsShowingHoverEffect) return;

    this.IsShowingHoverEffect = true;
    if (this.getCtx().getThemeBehavior() === 'redwood') {
      if (!this.isSelected()) {
        var hoverColor = this.getFill().getColor();
        this._showNestedBorders(hoverColor, this._innerColor);
      }
    } else {
      this._showNestedBorders(this._hoverColor, this._innerColor);
    }
  }

  /**
   * @override
   */
  hideHoverEffect() {
    if (!this.IsShowingHoverEffect) return;

    this.IsShowingHoverEffect = false;
    if (this.getCtx().getThemeBehavior() === 'redwood') {
      if (!this.isSelected()) {
        this._showNestedBorders();
      }
    } else {
      if (this.isSelected()) this._showNestedBorders(this._outerColor, this._innerColor);
      else this._showNestedBorders();
    }
  }

  /**
   * @override
   */
  setSelected(selected) {
    if (this.IsSelected == selected) return;

    this.IsSelected = selected;
    if (this.getCtx().getThemeBehavior() === 'redwood') {
      var hoverColor = this.getPrimaryFill().getColor();
      if (this.isSelected()) {
        this._showNestedBorders(this._outerColor, this._innerColor);
      } else {
        if (this.isHoverEffectShown()) this._showNestedBorders(hoverColor, this._innerColor);
        else this._showNestedBorders();
      }
    } else {
      if (this.isHoverEffectShown()) this._showNestedBorders(this._hoverColor, this._innerColor);
      else if (this.isSelected()) this._showNestedBorders(this._outerColor, this._innerColor);
      else this._showNestedBorders();
    }
  }

  /**
   * @override
   */
  UpdateSelectionEffect() {
    // noop: Selection effects fully managed by this class
  }

  /**
   * Returns the primary dvt.Fill for this bar. Used for animation, since getFill may return the fill of the selection
   * shapes.
   * @return {dvt.Fill}
   */
  getPrimaryFill() {
    return this._fill;
  }

  /**
   * Helper function that creates and adds the shapes used for displaying hover and selection effects. Should only be
   * called on hover or select operations, since it assumes that the fill, stroke, and shape size are already determined.
   * @private
   */
  _initializeSelectionEffects() {
    // Calculate the geometry of the shapes used for the selection effects
    var outerBorderWidth = this.isSelected()
      ? DvtChartSelectableRectangularPolygon.OUTER_BORDER_WIDTH
      : DvtChartSelectableRectangularPolygon.OUTER_BORDER_WIDTH_HOVER;
    var outerChildPoints = this._createPointsArray(outerBorderWidth);
    var innerChildPoints = this._createPointsArray(
      outerBorderWidth + DvtChartSelectableRectangularPolygon.INNER_BORDER_WIDTH
    );

    // Just update the geometries if already initialized
    if (this.OuterChild) {
      this.OuterChild.setPoints(outerChildPoints);
      this.InnerChild.setPoints(innerChildPoints);
      return;
    }

    this.OuterChild = new Polygon(this.getCtx(), outerChildPoints);
    this.OuterChild.setInvisibleFill();
    this.OuterChild.setMouseEnabled(true);
    this.addChild(this.OuterChild);

    this.InnerChild = new Polygon(this.getCtx(), innerChildPoints);
    this.InnerChild.setInvisibleFill();
    this.InnerChild.setMouseEnabled(true);
    this.addChild(this.InnerChild);
  }

  /**
   * Helper function to apply border colors for hover and selection.
   * @param {string=} outerBorderColor
   * @param {string=} innerBorderColor
   * @private
   */
  _showNestedBorders(outerBorderColor, innerBorderColor) {
    // Ensure that selection and hover shapes are created
    this._initializeSelectionEffects();

    // Modify the shapes based on which borders should be shown
    if (innerBorderColor) {
      this.setSolidFill(outerBorderColor);
      this.setStroke(null);
      this.setClassName().setStyle();

      this.OuterChild.setSolidFill(innerBorderColor);
      this.OuterChild.setClassName().setStyle();

      this.InnerChild.setFill(this._fill);
      this.InnerChild.setClassName(this._shapeClassName).setStyle(this._shapeStyle);
    } else if (outerBorderColor) {
      this.setSolidFill(outerBorderColor);
      this.setStroke(null);
      this.setClassName().setStyle();

      this.OuterChild.setFill(this._fill);
      this.OuterChild.setClassName(this._shapeClassName).setStyle(this._shapeStyle);

      this.InnerChild.setInvisibleFill();
      this.InnerChild.setClassName().setStyle();
    } else {
      this.setFill(this._fill);
      this.setStroke(this._borderStroke);
      this.setClassName(this._shapeClassName).setStyle(this._shapeStyle);

      this.OuterChild.setInvisibleFill();
      this.OuterChild.setClassName().setStyle();

      this.InnerChild.setInvisibleFill();
      this.InnerChild.setClassName().setStyle();
    }
  }

  /**
   * Returns the points array for the polygon used to render the polygon, with an inset to show nested border effects.
   * @param {number} inset The number of pixels to inset the polygon.  Defaults to 0.
   * @return {array} The list of points for the polygon
   * @private
   */
  _createPointsArray(inset) {
    var x1 = this._x1 + inset;
    var x2 = this._x2 - inset;
    var y1 = this._y1 + inset;
    var y2 = this._y2 - inset;
    return [x1, y1, x2, y1, x2, y2, x1, y2];
  }
}

/** @const */
DvtChartSelectableRectangularPolygon.OUTER_BORDER_WIDTH = 2;

/** @const */
DvtChartSelectableRectangularPolygon.OUTER_BORDER_WIDTH_HOVER = 1.25;

/** @const */
DvtChartSelectableRectangularPolygon.INNER_BORDER_WIDTH = 1;

/**
 *  A selectable bar for charting.
 *  @class DvtChartBar
 *  @extends {dvt.Polygon}
 *  @constructor
 *  @param {Chart} chart
 *  @param {number} axisCoord The location of the axis line.
 *  @param {number} baselineCoord The location from which the bar grows.
 *  @param {number} endCoord The location where the bar length ends.
 *  @param {number} x1 The left coord of a vertical bar, or the top of a horizontal bar.
 *  @param {number} x2 The right coord of a vertical bar, or the bottom of a horizontal bar.
 *  @param {boolean} doNotRender Flag to indicate that this shape is only used as a property bag, so the DOM elem shouldn't be rendered.
 */
class DvtChartBar extends DvtChartSelectableRectangularPolygon {
  constructor(chart, axisCoord, baselineCoord, endCoord, x1, x2, doNotRender) {
    super(chart.getCtx(), null, null, doNotRender);
    /** @private @const */
    this._MAX_GAP_SIZE = 2;

    /** @private @const */
    this._MIN_BAR_LENGTH_FOR_GAPS = 5;

    /** @private @const */
    this._INDICATOR_OFFSET = 8;

    /** @private @const */
    this._MIN_BAR_WIDTH_FOR_GAPS_PIXEL_HINTING = 15;

    this._bHoriz = DvtChartTypeUtils.isHorizontal(chart);
    this._bStacked = DvtChartDataUtils.isStacked(chart);
    this._barGapRatio = DvtChartGroupUtils.getBarGapRatio(chart);
    this._dataItemGaps = DvtChartStyleUtils.getDataItemGaps(chart);
    this._axisCoord = axisCoord;
    this._doNotRender = !!doNotRender;

    // Calculate the points array and apply to the polygon
    this._setBarCoords(baselineCoord, endCoord, x1, x2, true);
  }

  /**
   * @override
   */
  setSelected(selected) {
    if (this.IsSelected == selected) return;

    this.IsSelected = selected;
    var hoverColor;
    if (this.getCtx().getThemeBehavior() === 'redwood') {
      var hoverColor = this.getPrimaryFill().getColor();
    } else {
      var hoverColor = this._hoverColor;
    }

    if (this.isSelected()) {
      // Remove the gaps from the sides of the bar per UX spec
      this._tempX1 = this._x1;
      this._tempX2 = this._x2;
      this._tempBaselineCoord = this._baselineCoord;
      this._x1 = this._origX1;
      this._x2 = this._origX2;
      this._baselineCoord = this._origBaselineCoord;
      this.setPoints(this._createPointsArray());
      if (this.getCtx().getThemeBehavior() === 'redwood') {
        this._showNestedBorders(this._outerColor, this._innerColor);
      } else {
        this._showNestedBorders(
          this.isHoverEffectShown() ? hoverColor : this._outerColor,
          this._innerColor
        );
      }
    } else {
      // Restore the gaps from the sides of the bar per UX spec
      this._x1 = this._tempX1;
      this._x2 = this._tempX2;
      this._baselineCoord = this._tempBaselineCoord;
      this.setPoints(this._createPointsArray());
      if (this.getCtx().getThemeBehavior() === 'redwood') {
        if (this.isHoverEffectShown()) this._showNestedBorders(hoverColor, this._innerColor);
        else this._showNestedBorders(null);
      } else this._showNestedBorders(this.isHoverEffectShown() ? hoverColor : null);
    }
  }

  /**
   * Returns the layout parameters for the current animation frame.
   * @param {boolean=} bFlip True if the result should be flipped for horizontal/vertical orientation change.
   * @return {array} The array of layout parameters.
   */
  getAnimParams(bFlip) {
    if (bFlip) {
      if (this._bHoriz)
        // flipping to vertical
        return [this._x2, this._x1, this._baselineCoord, this._endCoord];
      // flipping to horizontal
      else return [this._x1, this._x2, this._endCoord, this._baselineCoord];
    } else return [this._baselineCoord, this._endCoord, this._x1, this._x2];
  }

  /**
   * Sets the layout parameters for the current animation frame.
   * @param {array} params The array of layout parameters.
   * @param {dvt.Displayable=} indicator The animation indicator, whose geometry is centered at (0,0).
   */
  setAnimParams(params, indicator) {
    // Set bar coords but don't adjust for gaps, since they've already been factored in.
    this._setBarCoords(params[0], params[1], params[2], params[3], false);

    // Update animation indicator if present.
    if (indicator) {
      var indicatorPosition = this.getIndicatorPos();
      indicator.setTranslate(indicatorPosition.x, indicatorPosition.y);
      indicator.setAlpha(1);

      // Reparent to keep indicator on top
      indicator.getParent().addChild(indicator);
    }
  }

  /**
   * Returns a dvt.Playable containing the animation of the bar to its initial data value.
   * @param {number} duration The duration of the animation in seconds.
   * @return {dvt.Playable} A playable for the initial bar animation.
   */
  getDisplayAnim(duration) {
    // Current state is the end state
    var endState = this.getAnimParams();

    // Initialize the start state. To grow the bar, just set the end coord to the axis coord.
    this.setAnimParams([this._axisCoord, this._axisCoord, this._x1, this._x2]);

    // Create and return the playable
    var nodePlayable = new CustomAnimation(this.getCtx(), this, duration);
    nodePlayable
      .getAnimator()
      .addProp(
        Animator.TYPE_NUMBER_ARRAY,
        this,
        this.getAnimParams,
        this.setAnimParams,
        endState
      );
    return nodePlayable;
  }

  /**
   * Returns a dvt.Playable containing the animation to delete the bar.
   * @param {number} duration The duration of the animation in seconds.
   * @return {dvt.Playable}
   */
  getDeleteAnim(duration) {
    // End state is for the bar length to shrink to 0
    var endState = [this._baselineCoord, this._baselineCoord, this._x1, this._x2];

    // Create and return the playable
    var nodePlayable = new CustomAnimation(this.getCtx(), this, duration);
    nodePlayable
      .getAnimator()
      .addProp(
        Animator.TYPE_NUMBER_ARRAY,
        this,
        this.getAnimParams,
        this.setAnimParams,
        endState
      );
    nodePlayable
      .getAnimator()
      .addProp(Animator.TYPE_NUMBER, this, this.getAlpha, this.setAlpha, 0);
    return nodePlayable;
  }

  /**
   * Returns a dvt.Playable containing the insert animation of the bar.
   * @param {number} duration The duration of the animation in seconds.
   * @return {dvt.Playable}
   */
  getInsertAnim(duration) {
    // Initialize the alpha to fade in the bar
    this.setAlpha(0);

    // Create the playable
    var nodePlayable = this.getDisplayAnim(duration);
    nodePlayable
      .getAnimator()
      .addProp(Animator.TYPE_NUMBER, this, this.getAlpha, this.setAlpha, 1);
    return nodePlayable;
  }

  /**
   * Returns the position where the value change indicator should be displayed.
   * @return {dvt.Point}
   */
  getIndicatorPos() {
    var widthCoord = (this._x1 + this._x2) / 2;
    var x, y;
    if (this._bStacked) {
      // Center the indicator within the stacked bar
      var midLength = (this._endCoord + this._baselineCoord) / 2;
      x = this._bHoriz ? midLength : widthCoord;
      y = this._bHoriz ? widthCoord : midLength;
    } else {
      var lengthCoord =
        this._endCoord >= this._baselineCoord
          ? this._endCoord + this._INDICATOR_OFFSET
          : this._endCoord - this._INDICATOR_OFFSET;
      x = this._bHoriz ? lengthCoord : widthCoord;
      y = this._bHoriz ? widthCoord : lengthCoord;
    }

    return new Point(x, y);
  }

  /**
   * Stores the point coords defining the bar.
   * @param {number} baselineCoord The location from which the bar grows.
   * @param {number} endCoord The location where the bar length ends.
   * @param {number} x1 The left coord of a vertical bar, or the top of a horizontal bar.
   * @param {number} x2 The right coord of a vertical bar, or the bottom of a horizontal bar.
   * @param {boolean} bAdjustForGaps True if the specified coordinate should be adjusted to produce gaps.
   * @private
   */
  _setBarCoords(baselineCoord, endCoord, x1, x2, bAdjustForGaps) {
    // Store the geometry values
    this._baselineCoord = baselineCoord;
    this._endCoord = endCoord;
    this._x1 = x1;
    this._x2 = x2;

    // Bar width has to be at least 1px to prevent disappearing bars
    var barWidth = this._x2 - this._x1;
    if (barWidth < 1) {
      this._x1 = Math.floor(this._x1);
      this._x2 = this._x1 + 1;
      barWidth = 1;
    }

    // Store the values before the gaps are applied
    this._origX1 = this._x1;
    this._origX2 = this._x2;
    this._origBaselineCoord = this._baselineCoord;
    this._origSize = this._x2 - this._x1;

    // If data item gaps enabled, add gaps between bars.
    if (this._dataItemGaps > 0 && bAdjustForGaps && !this.isSelected()) {
      // Note: The gap sizes were found via experimentation and we may need to tweak them for browser updates. Firefox
      // vertical pixel hinting behavior requires double gaps.
      var gapSize = Math.ceil(this._MAX_GAP_SIZE * this._dataItemGaps);
      var barLength = Math.abs(this._baselineCoord - this._endCoord);
      var bStartsAtBaseline = this._axisCoord == this._baselineCoord;

      // Gaps between bars in stack
      if (barLength >= this._MIN_BAR_LENGTH_FOR_GAPS && this._bStacked && !bStartsAtBaseline)
        this._baselineCoord += this._endCoord > this._baselineCoord ? gapSize : -gapSize;

      // Gaps between bars in cluster
      if (barWidth >= DvtChartBar._MIN_BAR_WIDTH_FOR_GAPS) {
        if (
          Agent.getDevicePixelRatio() == 1 &&
          this._barGapRatio > 0 &&
          barWidth > this._MIN_BAR_WIDTH_FOR_GAPS_PIXEL_HINTING
        ) {
          // If devicePixelRatio is 1, we need to be extremely precise since anti-aliasing must be disabled for correct
          // gaps. We can only do this for barGapRatio > 0, as otherwise positioning is more important than crispness.

          // Don't do this for FF though, since it does pixel hinting incorrectly.
          if (Agent.browser !== 'firefox' && !this._doNotRender) this.setPixelHinting(true);

          // Round the coords for crisp looking bars
          this._x1 = Math.round(this._x1);
          this._x2 = Math.round(this._x2);

          // Update to use the rounded coords
          this._origX1 = this._x1;
          this._origX2 = this._x2;

          // Apply the gap
          this._x2 -= gapSize;
        } else {
          // Browser zoom or retina display.  Allow anti-aliasing to do the work.
          this._x1 += gapSize / 2;
          this._x2 -= gapSize / 2;
        }

        this._origSize -= gapSize;
      }
    }

    if (this._doNotRender) return;

    // Calculate the points array for the outer shape
    var points = this._createPointsArray();
    this.setPoints(points);

    // If the inner shapes are already defined, update them as well
    if (this.OuterChild)
      this.OuterChild.setPoints(
        this._createPointsArray(DvtChartSelectableRectangularPolygon.OUTER_BORDER_WIDTH)
      );

    if (this.InnerChild)
      this.InnerChild.setPoints(
        this._createPointsArray(
          DvtChartSelectableRectangularPolygon.OUTER_BORDER_WIDTH +
            DvtChartSelectableRectangularPolygon.INNER_BORDER_WIDTH
        )
      );
  }

  /**
   * Returns the points array for the polygon used to render the bar. An optional inset can be provided to show nested
   * border effects.
   * @param {number=} inset The number of pixels to inset the polygon.  Defaults to 0.
   * @return {array}
   * @private
   */
  _createPointsArray(inset) {
    var baselineCoord = this._baselineCoord;
    var endCoord = this._endCoord;
    var x1 = this._x1;
    var x2 = this._x2;

    // Check the inset if specified
    if (inset > 0) {
      // Ensure there's enough space for the inset
      if (Math.abs(x1 - x2) < 2 * inset || Math.abs(baselineCoord - endCoord) < 2 * inset)
        return [];

      // x1 is always less than x2
      x1 += inset;
      x2 -= inset;

      // Relationship between baseline and endCoord aren't so static
      if (endCoord < baselineCoord) {
        baselineCoord -= inset;
        endCoord += inset;
      } else {
        baselineCoord += inset;
        endCoord -= inset;
      }
    }

    if (this._bHoriz) return [endCoord, x1, endCoord, x2, baselineCoord, x2, baselineCoord, x1];
    else return [x1, endCoord, x2, endCoord, x2, baselineCoord, x1, baselineCoord];
  }

  /**
   * Returns the bounding box of the shape
   * @return {dvt.Rectangle} bbox
   */
  getBoundingBox() {
    var x = Math.min(this._x2, this._x1);
    var y = Math.min(this._endCoord, this._baselineCoord);
    var w = Math.abs(this._x2 - this._x1);
    var h = Math.abs(this._endCoord - this._baselineCoord);

    if (this._bHoriz) return new Rectangle(y, x, h, w);
    else return new Rectangle(x, y, w, h);
  }

  /**
   * Returns the non rounded width(horizontal) or height(vertical) of the bar
   * @return {number}
   */
  getOriginalBarSize() {
    return this._origSize;
  }

  /**
   * Returns the bounds of the displayable relative to the target coordinate space.  If the target
   * coordinate space is not specified, returns the bounds relative to this displayable.  This function does not take
   * into account any child displayables.
   * @param {dvt.Displayable} targetCoordinateSpace The displayable defining the target coordinate space.
   * @return {dvt.Rectangle} The bounds of the displayable relative to the target coordinate space.
   */
  getDimensionsSelf(targetCoordinateSpace) {
    // Note: In the near future, we will not support children for shapes, so this routine will be refactored into the
    //       existing getDimensions calls.  For now, components must be aware of the presence of children to use this.
    return this.ConvertCoordSpaceRect(this.getBoundingBox(), targetCoordinateSpace);
  }
}

/** @private @const */
DvtChartBar._MIN_BAR_WIDTH_FOR_GAPS = 5;

/**
 * Displayable for box and whisker shape (box plot).
 * @extends {dvt.Container}
 * @param {Chart} chart
 * @param {number} xCoord
 * @param {number} boxWidth
 * @param {number} lowCoord
 * @param {number} q1Coord
 * @param {number} q2Coord
 * @param {number} q3Coord
 * @param {number} highCoord
 * @param {object} styleOptions
 * @class
 * @constructor
 */
class DvtChartBoxAndWhisker extends Shape {
  constructor(
    chart,
    xCoord,
    boxWidth,
    lowCoord,
    q1Coord,
    q2Coord,
    q3Coord,
    highCoord,
    styleOptions
  ) {
    super(chart.getCtx());
    this._chart = chart;
    this._bHoriz = DvtChartTypeUtils.isHorizontal(chart);

    this._styleOptions = styleOptions;
    this._innerColor = DvtChartStyleUtils.getSelectedInnerColor(this._chart);
    this._outerColor = DvtChartStyleUtils.getSelectedOuterColor(this._chart);
    this._hoverColor = SelectionEffectUtils.getHoverBorderColor(this._styleOptions['_color']);

    var x1 = xCoord - boxWidth / 2;
    var x2 = xCoord + boxWidth / 2;
    if (
      DvtChartStyleUtils.getDataItemGaps(chart) > 0 &&
      boxWidth > DvtChartBar._MIN_BAR_WIDTH_FOR_GAPS
    )
      x2--;

    this._render(x1, x2, lowCoord, q1Coord, q2Coord, q3Coord, highCoord);
  }

  /**
   * Renders the box and whisker shapes.
   * @param {number} x1
   * @param {number} x2
   * @param {number} low
   * @param {number} q1
   * @param {number} q2
   * @param {number} q3
   * @param {number} high
   * @private
   */
  _render(x1, x2, low, q1, q2, q3, high) {
    this._cleanUp();
    var context = this.getCtx();

    // Round the coords to produce crisp edges
    this._x1 = Math.round(x1);
    this._x2 = Math.round(x2);
    this._low = Math.round(low);
    this._q1 = Math.round(q1);
    this._q2 = Math.round(q2);
    this._q3 = Math.round(q3);
    this._high = Math.round(high);

    // Ensure that the whisker ends are wider than the box and that the length is an odd integer
    var boxWidth = x2 - x1;
    var whiskerEndLength = DvtChartStyleUtils.getSizeInPixels(
      this._styleOptions['whiskerEndLength'],
      boxWidth
    );
    whiskerEndLength = Math.min(boxWidth, whiskerEndLength);
    whiskerEndLength = Math.floor((whiskerEndLength - 1) / 2) * 2 + 1;

    // Create the whiskers
    var whiskerX = Math.floor((x1 + x2) / 2) + 0.5;
    var whiskerX1 = whiskerX - Math.floor(whiskerEndLength / 2) - 0.5;
    var whiskerX2 = whiskerX + Math.floor(whiskerEndLength / 2) + 0.5;

    this._drawLine(whiskerX, this._low, whiskerX, this._high, 'whisker');
    this._drawLine(whiskerX1, this._low, whiskerX2, this._low, 'whiskerEnd');
    this._drawLine(whiskerX1, this._high, whiskerX2, this._high, 'whiskerEnd');

    // Create the box shape on top of whiskers
    this._q2Box = new Polygon(context, this._createQ2PointsArray(0));
    var q2Fill = DvtChartSeriesEffectUtils.getRectangleFill(
      this._chart,
      this._styleOptions['q2Color'],
      this._styleOptions['_q2Pattern'],
      boxWidth
    );
    this._q2Box.setFill(q2Fill);
    this._applyCustomStyle(this._q2Box, 'q2');
    this.addChild(this._q2Box);

    this._q3Box = new Polygon(context, this._createQ3PointsArray(0));
    var q3Fill = DvtChartSeriesEffectUtils.getRectangleFill(
      this._chart,
      this._styleOptions['q3Color'],
      this._styleOptions['_q3Pattern'],
      boxWidth
    );
    this._q3Box.setFill(q3Fill);
    this._applyCustomStyle(this._q3Box, 'q3');
    this.addChild(this._q3Box);

    // Create the median line
    this._drawMedianLine(0);

    // Create box border
    this._borderColor = this._styleOptions['borderColor'];
    if (this._borderColor) {
      this._borderWidth = this._styleOptions['borderWidth'];
      this._drawBorders(this._borderColor, this._borderWidth);
    }

    // Reapply selection effect (during animation)
    if (this.IsSelected) {
      this.IsSelected = false; // force selection effect to be reapplied
      this.setSelected(true);
    }
  }

  /**
   * @override
   */
  showHoverEffect() {
    if (this.IsShowingHoverEffect) return;

    this.IsShowingHoverEffect = true;

    if (this.isSelected()) {
      if (this.getCtx().getThemeBehavior() !== 'redwood')
        this._drawBorders(
          this._hoverColor,
          DvtChartSelectableRectangularPolygon.OUTER_BORDER_WIDTH,
          this._innerColor,
          DvtChartSelectableRectangularPolygon.INNER_BORDER_WIDTH
        );
    } else {
      this._drawBorders(
        this._hoverColor,
        DvtChartSelectableRectangularPolygon.OUTER_BORDER_WIDTH_HOVER,
        this._innerColor,
        DvtChartSelectableRectangularPolygon.INNER_BORDER_WIDTH
      );
    }
  }

  /**
   * @override
   */
  hideHoverEffect() {
    if (!this.IsShowingHoverEffect) return;

    this.IsShowingHoverEffect = false;

    if (this.isSelected())
      this._drawBorders(
        this._outerColor,
        DvtChartSelectableRectangularPolygon.OUTER_BORDER_WIDTH,
        this._innerColor,
        DvtChartSelectableRectangularPolygon.INNER_BORDER_WIDTH
      );
    else this._drawBorders(this._borderColor, this._borderWidth); // set original border
  }

  /**
   * @override
   */
  setSelected(selected) {
    if (this.IsSelected == selected) return;

    this.IsSelected = selected;

    if (this.isHoverEffectShown()) {
      if (selected) {
        if (this.getCtx().getThemeBehavior() === 'redwood') {
          this._drawBorders(
            this._outerColor,
            DvtChartSelectableRectangularPolygon.OUTER_BORDER_WIDTH,
            this._innerColor,
            DvtChartSelectableRectangularPolygon.INNER_BORDER_WIDTH
          );
        } else
          this._drawBorders(
            this._hoverColor,
            DvtChartSelectableRectangularPolygon.OUTER_BORDER_WIDTH,
            this._innerColor,
            DvtChartSelectableRectangularPolygon.INNER_BORDER_WIDTH
          );
      } else {
        this._drawBorders(
          this._hoverColor,
          DvtChartSelectableRectangularPolygon.OUTER_BORDER_WIDTH_HOVER,
          this._innerColor,
          DvtChartSelectableRectangularPolygon.INNER_BORDER_WIDTH
        );
      }
    } else {
      if (selected)
        this._drawBorders(
          this._outerColor,
          DvtChartSelectableRectangularPolygon.OUTER_BORDER_WIDTH,
          this._innerColor,
          DvtChartSelectableRectangularPolygon.INNER_BORDER_WIDTH
        );
      else this._drawBorders(this._hoverColor, this._borderWidth); // set original border
    }
  }

  /**
   * @override
   */
  UpdateSelectionEffect() {
    // noop: Selection effects fully managed by this class
  }

  /**
   * Returns a dvt.Playable containing the display animation for the shape.
   * @param {number} duration The duration of the animation in seconds.
   * @return {dvt.Playable} A playable for the display animation.
   */
  getDisplayAnim(duration) {
    return this.getInsertAnim(duration);
  }

  /**
   * Returns a dvt.Playable containing the animation to delete this displayable.
   * @param {number} duration The duration of the animation in seconds.
   * @return {dvt.Playable}
   */
  getDeleteAnim(duration) {
    // Animation: Shink into the median & fade out.
    var nodePlayable = new CustomAnimation(this.getCtx(), this, duration);

    var endState = [this._x1, this._x2, this._q2, this._q2, this._q2, this._q2, this._q2];
    nodePlayable
      .getAnimator()
      .addProp(
        Animator.TYPE_NUMBER_ARRAY,
        this,
        this._getAnimParams,
        this._setAnimParams,
        endState
      );

    nodePlayable
      .getAnimator()
      .addProp(Animator.TYPE_NUMBER, this, this.getAlpha, this.setAlpha, 0);

    return nodePlayable;
  }

  /**
   * Returns a dvt.Playable containing the insert animation for this displayable.
   * @param {number} duration The duration of the animation in seconds.
   * @return {dvt.Playable}
   */
  getInsertAnim(duration) {
    // Animation: Grow from the median.
    var nodePlayable = new CustomAnimation(this.getCtx(), this, duration);

    var endState = this._getAnimParams();
    this._setAnimParams([this._x1, this._x2, this._q2, this._q2, this._q2, this._q2, this._q2]);
    nodePlayable
      .getAnimator()
      .addProp(
        Animator.TYPE_NUMBER_ARRAY,
        this,
        this._getAnimParams,
        this._setAnimParams,
        endState
      );

    this.setAlpha(0);
    nodePlayable
      .getAnimator()
      .addProp(Animator.TYPE_NUMBER, this, this.getAlpha, this.setAlpha, 1);

    return nodePlayable;
  }

  /**
   * Returns a dvt.Playable containing the update animation for this displayable.
   * @param {number} duration The duration of the animation in seconds.
   * @param {DvtChartBoxAndWhisker} oldShape The old shape to animate from.
   * @return {dvt.Playable}
   */
  getUpdateAnim(duration, oldShape) {
    // Animation: Transition shape and color.
    var nodePlayable = new CustomAnimation(this.getCtx(), this, duration);

    var startQ2Fill = oldShape._getQ2Fill();
    var endQ2Fill = this._getQ2Fill();
    if (!startQ2Fill.equals(endQ2Fill)) {
      this._setQ2Fill(startQ2Fill);
      nodePlayable
        .getAnimator()
        .addProp(Animator.TYPE_FILL, this, this._getQ2Fill, this._setQ2Fill, endQ2Fill);
    }

    var startQ3Fill = oldShape._getQ3Fill();
    var endQ3Fill = this._getQ3Fill();
    if (!startQ3Fill.equals(endQ3Fill)) {
      this._setQ3Fill(startQ3Fill);
      nodePlayable
        .getAnimator()
        .addProp(Animator.TYPE_FILL, this, this._getQ3Fill, this._setQ3Fill, endQ3Fill);
    }

    var endState = this._getAnimParams();
    this._setAnimParams(oldShape._getAnimParams());
    nodePlayable
      .getAnimator()
      .addProp(
        Animator.TYPE_NUMBER_ARRAY,
        this,
        this._getAnimParams,
        this._setAnimParams,
        endState
      );

    return nodePlayable;
  }

  /**
   * Get animation params.
   * @return {array} The array of params.
   * @private
   */
  _getAnimParams() {
    return [this._x1, this._x2, this._low, this._q1, this._q2, this._q3, this._high];
  }

  /**
   * Set animation params.
   * @param {array} ar The array of params.
   * @private
   */
  _setAnimParams(ar) {
    this._render(ar[0], ar[1], ar[2], ar[3], ar[4], ar[5], ar[6]);
  }

  /**
   * Get the Q2 box fill for animation.
   * @return {dvt.Fill}
   * @private
   */
  _getQ2Fill() {
    // Return from the _styleOptions bc it's the source of truth.
    // Can't rely on the shape fill because the shape is recreated at each _render.
    return new SolidFill(this._styleOptions['q2Color']);
  }

  /**
   * Set the Q2 box fill for animation.
   * @param {dvt.Fill} fill
   * @private
   */
  _setQ2Fill(fill) {
    // Set the _styleOptions bc it's the source of truth. The color will be updated on _render.
    this._styleOptions['q2Color'] = fill.getColor();
  }

  /**
   * Get the Q3 box fill for animation.
   * @return {dvt.Fill}
   * @private
   */
  _getQ3Fill() {
    // Return from the _styleOptions bc it's the source of truth.
    // Can't rely on the shape fill because the shape is recreated at each _render.
    return new SolidFill(this._styleOptions['q3Color']);
  }

  /**
   * Set the Q3 box fill for animation.
   * @param {dvt.Fill} fill
   * @private
   */
  _setQ3Fill(fill) {
    // Set the _styleOptions bc it's the source of truth. The color will be updated on _render.
    this._styleOptions['q3Color'] = fill.getColor();
  }

  /**
   * Draws a line with the specified coords.
   * @param {number} x1
   * @param {number} y1
   * @param {number} x2
   * @param {number} y2
   * @param {string} prefix The _styleOptions prefix
   * @return {dvt.Line} The line shape.
   * @private
   */
  _drawLine(x1, y1, x2, y2, prefix) {
    var line = this._bHoriz
      ? new Line(this.getCtx(), y1, x1, y2, x2)
      : new Line(this.getCtx(), x1, y1, x2, y2);
    this._applyCustomStyle(line, prefix);
    line.setPixelHinting(true);
    this.addChild(line);
    return line;
  }

  /**
   * Draws the median line.
   * @param {number} inset The number of pixels to inset the median line.
   * @private
   */
  _drawMedianLine(inset) {
    if (!this._medianLine) {
      var line = new Line(this.getCtx(), 0, 0, 0, 0);
      this._applyCustomStyle(line, 'median');
      line.setPixelHinting(true);
      this.addChild(line);
      this._medianLine = line;
    }

    var x1 = this._x1 + inset;
    var x2 = this._x2 - inset;
    var q2 = this._q2;

    if (this._bHoriz) {
      this._medianLine.setX1(q2);
      this._medianLine.setX2(q2);
      this._medianLine.setY1(x1);
      this._medianLine.setY2(x2);
    } else {
      this._medianLine.setY1(q2);
      this._medianLine.setY2(q2);
      this._medianLine.setX1(x1);
      this._medianLine.setX2(x2);
    }
  }

  /**
   * Helper function to apply up to two border colors for the shape.
   * @param {string=} outerBorderColor
   * @param {number=} outerBorderWidth
   * @param {string=} innerBorderColor
   * @param {number=} innerBorderWidth
   * @private
   */
  _drawBorders(outerBorderColor, outerBorderWidth, innerBorderColor, innerBorderWidth) {
    // Initialize the border shapes if this is called the first time
    if (!this._outerBorderShape) {
      var childIndex = this.getChildIndex(this._q2Box);

      this._innerBorderShape = new Polygon(this.getCtx());
      this._innerBorderShape.setInvisibleFill();
      this.addChildAt(this._innerBorderShape, childIndex);

      this._outerBorderShape = new Polygon(this.getCtx());
      this._outerBorderShape.setInvisibleFill();
      this.addChildAt(this._outerBorderShape, childIndex);
    }

    var boxInset = 0;
    if (outerBorderWidth) {
      this._outerBorderShape.setPoints(this._createBoxPointsArray(0));
      this._outerBorderShape.setSolidFill(outerBorderColor);
      boxInset += outerBorderWidth;

      if (innerBorderWidth) {
        this._innerBorderShape.setPoints(this._createBoxPointsArray(outerBorderWidth));
        this._innerBorderShape.setSolidFill(innerBorderColor);
        boxInset += innerBorderWidth;
      } else this._innerBorderShape.setInvisibleFill();
    } else {
      this._outerBorderShape.setInvisibleFill();
      this._innerBorderShape.setInvisibleFill();
    }

    this._q2Box.setPoints(this._createQ2PointsArray(boxInset));
    this._q3Box.setPoints(this._createQ3PointsArray(boxInset));
    this._drawMedianLine(boxInset);
  }

  /**
   * Returns the points array for the polygon used to render the shape covering the entire box.
   * @param {number} inset The number of pixels to inset the polygon.
   * @return {Array}
   * @private
   */
  _createBoxPointsArray(inset) {
    var x1 = this._x1 + inset;
    var x2 = this._x2 - inset;

    var q1, q3;
    if (this._q1 < this._q3) {
      q1 = this._q1 + inset;
      q3 = this._q3 - inset;
    } else {
      q1 = this._q1 - inset;
      q3 = this._q3 + inset;
    }

    return this._createPointsArray(x1, x2, q1, q3);
  }

  /**
   * Returns the points array for the polygon used to render the shape covering the q2-q3 segment of the box.
   * @param {number} inset The number of pixels to inset the polygon.
   * @return {Array}
   * @private
   */
  _createQ3PointsArray(inset) {
    var x1 = this._x1 + inset;
    var x2 = this._x2 - inset;
    var q2 = this._q2; // don't apply inset for q2

    var q3;
    if (this._q2 < this._q3) q3 = this._q3 - inset;
    else q3 = this._q3 + inset;

    return this._createPointsArray(x1, x2, q2, q3);
  }

  /**
   * Returns the points array for the polygon used to render the shape covering the q1-q2 segment of the box.
   * @param {number} inset The number of pixels to inset the polygon.
   * @return {Array}
   * @private
   */
  _createQ2PointsArray(inset) {
    var x1 = this._x1 + inset;
    var x2 = this._x2 - inset;
    var q2 = this._q2; // don't apply inset for q2

    var q1;
    if (this._q2 < this._q1) q1 = this._q1 - inset;
    else q1 = this._q1 + inset;

    return this._createPointsArray(x1, x2, q1, q2);
  }

  /**
   * Returns the points array for a polygon with the specified coords.
   * @param {number} x1
   * @param {number} x2
   * @param {number} y1
   * @param {number} y2
   * @return {Array}
   * @private
   */
  _createPointsArray(x1, x2, y1, y2) {
    if (this._bHoriz) return [y1, x1, y1, x2, y2, x2, y2, x1];
    else return [x1, y1, x2, y1, x2, y2, x1, y2];
  }

  /**
   * Apply the custom style in the _styleOptions to the shape.
   * @param {dvt.Shape} shape The shape to apply the style to.
   * @param {string} prefix The style prefix, e.g. 'q2', 'whisker', etc.
   * @private
   */
  _applyCustomStyle(shape, prefix) {
    shape.setStyle(
      this._styleOptions[prefix + 'Style'] || this._styleOptions[prefix + 'SvgStyle'],
      true
    );
    shape.setClassName(
      this._styleOptions[prefix + 'ClassName'] || this._styleOptions[prefix + 'SvgClassName'],
      true
    );
  }

  /**
   * Clean up previous render.
   * @private
   */
  _cleanUp() {
    this.removeChildren();

    // Remove references to the old shapes to ensure that they're rerendered
    this._q2Box = null;
    this._q3Box = null;
    this._medianLine = null;
    this._outerBorderShape = null;
    this._innerBorderShape = null;
  }
}

/**
 * Displayable for stock bars.
 * @extends {dvt.Container}
 * @param {dvt.Context} context
 * @param {number} xCoord
 * @param {number} barWidth
 * @param {number} openCoord
 * @param {number} closeCoord
 * @param {number=} lowCoord
 * @param {number=} highCoord
 * @class
 * @constructor
 */
class DvtChartCandlestick extends Container {
  constructor(context, xCoord, barWidth, openCoord, closeCoord, lowCoord, highCoord) {
    super(context);

    /**
     * The minimum width fraction of the range bar with respect to the change bar width.
     * @const
     * @private
     */
    this._BAR_WIDTH = 0.3;

    // Calculate the bar width. For width >= 2, use even integer widths to ensure symmetry with range bar.
    barWidth = Math.max(Math.round(barWidth / 2) * 2, 1);
    var rangeWidth = Math.min(Math.ceil((this._BAR_WIDTH * barWidth) / 2) * 2, barWidth);

    // Calculate the x coords of the bar. Round the xCoord to ensure pixel location.
    var x1 = Math.round(xCoord) - barWidth / 2;
    var x2 = x1 + barWidth;

    // Create the range shape if coords provided
    if (lowCoord != null && highCoord != null) {
      var rangeX1 = Math.round(xCoord) - rangeWidth / 2;
      var rangeX2 = rangeX1 + rangeWidth;
      this._rangeShape = new DvtChartSelectableRectangularPolygon(context, [
        rangeX1,
        lowCoord,
        rangeX2,
        lowCoord,
        rangeX2,
        highCoord,
        rangeX1,
        highCoord
      ]);
      this.addChild(this._rangeShape);
    }

    // Create the change shape on top of range
    this._changeShape = new DvtChartSelectableRectangularPolygon(context, [
      x1,
      openCoord,
      x2,
      openCoord,
      x2,
      closeCoord,
      x1,
      closeCoord
    ]);
    this.addChild(this._changeShape);

    // Never anti-alias. Coords are carefully chosen to be perfectly aligned.
    this.setPixelHinting(true);
  }

  /**
   * Specifies the fill and stroke for the change shape.
   * @param {dvt.Fill} fill
   * @param {dvt.Stroke} stroke
   * @param {string} dataColor The primary color of this data item.
   * @param {string} innerColor The inner color of the selection effect.
   * @param {string} outerColor The outer color of the selection effect.
   */
  setChangeStyle(fill, stroke, dataColor, innerColor, outerColor) {
    this._changeShape.setStyleProperties(fill, stroke, dataColor, innerColor, outerColor);
  }

  /**
   * Specifies the fill and stroke for the range shape.
   * @param {dvt.Fill} fill
   * @param {dvt.Stroke} stroke
   * @param {string} rangeColor The primary color of the range bar.
   * @param {string} outerColor The outer color of the selection effect.
   */
  setRangeStyle(fill, stroke, rangeColor, outerColor) {
    if (!this._rangeShape) return;

    this._rangeShape.setStyleProperties(fill, stroke, rangeColor, null, outerColor);
  }

  /**
   * Set selected obj for stock
   * @param {boolean} selected Indicates if the container is selested. We never want to display the range as selected
   */
  setSelected(selected) {
    this._changeShape.setSelected(selected);
    if (this._rangeShape) this._rangeShape.setSelected(selected);
  }

  /**
   * Only show the hover effect for the change and range shape, ignore volume
   */
  showHoverEffect() {
    this._changeShape.showHoverEffect();
    if (this._rangeShape) this._rangeShape.showHoverEffect();
  }

  /**
   * Hide the hover effect for change and range shape, ignore volume
   */
  hideHoverEffect() {
    this._changeShape.hideHoverEffect();
    if (this._rangeShape) this._rangeShape.hideHoverEffect();
  }

  /**
   * Returns a dvt.ParallelPlayable containing the display animations for the stock bars
   * @param {number} duration The duration of the animation in seconds.
   * @return {dvt.ParallelPlayable} A playable for the initial bar animation.
   */
  getDisplayAnim(duration) {
    // Animation: Grow from the center of each bar.
    var nodePlayable = new CustomAnimation(this.getCtx(), this, duration);

    // Change Shape
    var endStateChange = this._changeShape.getPoints();
    this._changeShape.setPoints(DvtChartCandlestick._getInitialPoints(endStateChange));
    nodePlayable
      .getAnimator()
      .addProp(
        Animator.TYPE_NUMBER_ARRAY,
        this._changeShape,
        this._changeShape.getPoints,
        this._changeShape.setAnimParams,
        endStateChange
      );

    // Range Shape
    if (this._rangeShape) {
      var endStateRange = this._rangeShape.getPoints();
      this._rangeShape.setPoints(DvtChartCandlestick._getInitialPoints(endStateRange));
      nodePlayable
        .getAnimator()
        .addProp(
          Animator.TYPE_NUMBER_ARRAY,
          this._rangeShape,
          this._rangeShape.getPoints,
          this._rangeShape.setAnimParams,
          endStateRange
        );
    }

    return nodePlayable;
  }

  /**
   * Returns a dvt.Playable containing the animation to delete this displayable.
   * @param {number} duration The duration of the animation in seconds.
   * @return {dvt.Playable}
   */
  getDeleteAnim(duration) {
    // Animation: Shrink to the center of each bar.
    var nodePlayable = new CustomAnimation(this.getCtx(), this, duration);

    // Change Shape
    var endStateChange = DvtChartCandlestick._getInitialPoints(this._changeShape.getPoints());
    nodePlayable
      .getAnimator()
      .addProp(
        Animator.TYPE_NUMBER_ARRAY,
        this._changeShape,
        this._changeShape.getPoints,
        this._changeShape.setAnimParams,
        endStateChange
      );

    // Range Shape
    if (this._rangeShape) {
      var endStateRange = DvtChartCandlestick._getInitialPoints(this._rangeShape.getPoints());
      nodePlayable
        .getAnimator()
        .addProp(
          Animator.TYPE_NUMBER_ARRAY,
          this._rangeShape,
          this._rangeShape.getPoints,
          this._rangeShape.setAnimParams,
          endStateRange
        );
    }

    // Alpha Fade
    nodePlayable
      .getAnimator()
      .addProp(Animator.TYPE_NUMBER, this, this.getAlpha, this.setAlpha, 0);

    return nodePlayable;
  }

  /**
   * Returns a dvt.Playable containing the insert animation for this displayable.
   * @param {number} duration The duration of the animation in seconds.
   * @return {dvt.Playable}
   */
  getInsertAnim(duration) {
    // Initialize the alpha to fade in the bar
    this.setAlpha(0);

    // Create the playable
    var nodePlayable = this.getDisplayAnim(duration);
    nodePlayable
      .getAnimator()
      .addProp(Animator.TYPE_NUMBER, this, this.getAlpha, this.setAlpha, 1);
    return nodePlayable;
  }

  /**
   * Returns a dvt.Playable containing the update animation for this displayable.
   * @param {number} duration The duration of the animation in seconds.
   * @param {DvtChartCandlestick} oldShape The old shape to animate from.
   * @return {dvt.Playable}
   */
  getUpdateAnim(duration, oldShape) {
    // Animation: Transition from old points to new points arrays.
    var nodePlayable = new CustomAnimation(this.getCtx(), this, duration);

    // Change Shape: Points
    var endStateChange = this._changeShape.getPoints();
    this._changeShape.setPoints(oldShape._changeShape.getPoints());
    nodePlayable
      .getAnimator()
      .addProp(
        Animator.TYPE_NUMBER_ARRAY,
        this._changeShape,
        this._changeShape.getPoints,
        this._changeShape.setAnimParams,
        endStateChange
      );

    // Change Shape: Fill. Need the get the primary color, since it might be overwritten during selection
    var startFill = oldShape._changeShape.getPrimaryFill();
    var endFill = this._changeShape.getPrimaryFill();
    var bSkipFillAnimation =
      oldShape._changeShape.isSelected() ||
      this._changeShape.isSelected() ||
      startFill.equals(endFill);
    if (!bSkipFillAnimation) {
      this._changeShape.setFill(startFill);
      nodePlayable
        .getAnimator()
        .addProp(
          Animator.TYPE_FILL,
          this._changeShape,
          this._changeShape.getFill,
          this._changeShape.setFill,
          endFill
        );
    }
    // Range Shape: Points
    if (this._rangeShape && oldShape._rangeShape) {
      var endStateRange = this._rangeShape.getPoints();
      this._rangeShape.setPoints(oldShape._rangeShape.getPoints());
      nodePlayable
        .getAnimator()
        .addProp(
          Animator.TYPE_NUMBER_ARRAY,
          this._rangeShape,
          this._rangeShape.getPoints,
          this._rangeShape.setAnimParams,
          endStateRange
        );
    }
    return nodePlayable;
  }

  /**
   * @override
   */
  UpdateSelectionEffect() {
    // noop: Selection effects fully managed by this class
  }

  /**
   * Returns the array of points for initial animation of the specified points array.
   * @param {array} points
   * @return {array}
   * @private
   */
  static _getInitialPoints(points) {
    var x1 = points[0];
    var x2 = points[2];
    var y1 = points[1];
    var y2 = points[5];
    var yMid = (y1 + y2) / 2;
    return [x1, yMid, x2, yMid, x2, yMid, x1, yMid];
  }
}

/**
 * Property bag of line/area coordinate for DvtChartPolygonSet.
 * The coordinates are represented by x, y1, and y2. Usually, y1 == y2. If y1 != y2, it means that there's a jump in
 * the y value at that x position (from y1 to y2) due to a null in the data.
 * @class DvtChartCoord
 * @extends {DvtObject}
 * @constructor
 * @param {number} x The x coordinate.
 * @param {number} y1 The first y coordinate.
 * @param {number} y2 The second y coordinate.
 * @param {number} groupIndex The group index of the coordinate.
 * @param {string} group The group name of the coordinate.
 * @param {boolean} filtered Whether the coordinate is filtered for performance.
 */
class DvtChartCoord {
  constructor(x, y1, y2, groupIndex, group, filtered) {
    this.x = x;
    this.y1 = y1;
    this.y2 = y2;
    this.groupIndex = groupIndex;
    this.group = group;
    this.filtered = filtered;
  }

  /**
   * Returns whether the step from y1 to y2 is upward. "Upward" is defined as moving further from the baseline.
   * @param {number} baseline The coordinate of the baseline.
   * @return {boolean}
   */
  isUpstep(baseline) {
    return Math.abs(this.y2 - baseline) > Math.abs(this.y1 - baseline);
  }

  /**
   * Returns a clone of itself.
   * @return {DvtChartCoord}
   */
  clone() {
    return new DvtChartCoord(this.x, this.y1, this.y2, this.groupIndex, this.group, this.filtered);
  }
}

/**
 * A collection of line/area shapes for a chart series.
 * Usually there's only one shape for each series, but there can be multiple if there are null values in the data.
 * @class DvtChartLineArea
 * @extends {dvt.Container}
 * @constructor
 * @param {Chart} chart The chart.
 * @param {boolean} bArea Whether this is an Area (it is a Line otherwise).
 * @param {dvt.Rectangle} availSpace The available space.
 * @param {number} baseline The axis baseline coordinate.
 * @param {object} style The style of the shape.
 * @param {string} className The className of the shape.
 * @param {dvt.Fill} fill The fill of the shapes.
 * @param {dvt.Stroke} stroke The stroke of the shapes.
 * @param {string} type The type of the line or the area top: straight, curved, stepped, centeredStepped, segmented, or centeredSegmented.
 * @param {array} arCoord Array of DvtChartCoord representing the coordinates of the line or the area top.
 * @param {string} baseType The type of the area base: straight, curved, stepped, centeredStepped, segmented, or centeredSegmented.
 * @param {array} arBaseCoord Array of DvtChartCoord representing the coordinates of the area base.
 */
class DvtChartLineArea extends Container {
  constructor(
    chart,
    bArea,
    availSpace,
    baseline,
    style,
    className,
    fill,
    stroke,
    type,
    arCoord,
    baseType,
    arBaseCoord
  ) {
    super(chart.getCtx());

    /** @private **/
    this._INDICATOR_OFFSET = 8;

    // Calculate the points array and apply to the polygon
    this._chart = chart;
    this._bArea = bArea;
    this._availSpace = availSpace;
    this._baseline = baseline;
    this._style = style;
    this._className = className;
    this._fill = fill;
    this._stroke = stroke;
    this._type = type;
    this._baseType = baseType ? baseType : type;
    this._indicatorMap = {};

    this.setCoords(arCoord, arBaseCoord);
    this.chartShapeType = 'lineArea';
  }

  /**
   * Sets the coordinates of the shapes.
   * @param {array} arCoord Array of DvtChartCoord representing the coordinates of the line or the area top.
   * @param {array} arBaseCoord Array of DvtChartCoord representing the coordinates of the area base.
   */
  setCoords(arCoord, arBaseCoord) {
    this._arCoord = arCoord;
    if (arBaseCoord) this._arBaseCoord = arBaseCoord;

    this.removeChildren(); // clean up

    if (this._bArea) this._renderAreas();
    else this._renderLines();

    this._positionIndicators();
  }

  /**
   * Returns the coordinates of the line or the area top.
   * @return {array} Array of DvtChartCoord representing the coordinates of the line or the area top.
   */
  getCoords() {
    return this._arCoord;
  }

  /**
   * Returns the coordinates of the area base.
   * @return {array} Array of DvtChartCoord representing the coordinates of the area base.
   */
  getBaseCoords() {
    return this._arBaseCoord;
  }

  /**
   * Returns the axis baseline coordinate.
   * @return {number}
   */
  getBaseline() {
    return this._baseline;
  }

  /**
   * Returns whether this is an area (otherwise, it's a line).
   * @return {boolean}
   */
  isArea() {
    return this._bArea;
  }

  /**
   * Creates arrays of dvt.Point for drawing polygons or polylines based on the DvtChartCoord array.
   * An array of dvt.Point arrays is returned, with each array being a set of contiguous points.
   * @param {array} coords The array of DvtChartCoords representing axis coordinates.
   * @param {string} type The line type: straight, curved, stepped, centeredStepped, segmented, or centeredSegmented.
   * @return {array} The arrays of contiguous points for drawing polygons or polylines.
   * @private
   */
  _getPointArrays(coords, type) {
    var pointsArrays = [];
    var points = [];
    pointsArrays.push(points);
    coords = DvtChartLineArea._convertToPointCoords(coords);

    var isPolar = DvtChartTypeUtils.isPolar(this._chart);
    var isCentered = type == 'centeredStepped' || type == 'centeredSegmented';
    var isParallel = isCentered || type == 'stepped' || type == 'segmented';
    var groupWidth = DvtChartStyleUtils.getGroupWidth(this._chart);
    var dir =
      Agent.isRightToLeft(this.getCtx()) && DvtChartTypeUtils.isVertical(this._chart) ? -1 : 1;

    var lastCoord;
    if (isPolar)
      // initiate lastCoord to be the final point
      lastCoord = coords[coords.length - 1];

    var coord, xCoord, isY2, finalXCoord;
    var inBump = false; // flag to indicate whether we're in a bump (a section surrounded by y1 != y2 cliffs)
    for (var i = 0; i < coords.length; i++) {
      if (coords[i] == null) {
        if (!DvtChartTypeUtils.isMixedFrequency(this._chart)) {
          // Draw the last step
          if (isParallel && !isPolar && lastCoord && !isY2) {
            finalXCoord = isCentered
              ? lastCoord.x + 0.5 * groupWidth * dir
              : lastCoord.x + groupWidth * dir;
            this._pushCoord(points, finalXCoord, lastCoord.y);
          }
        }
        // Start a new list of points, except in ADF and MAF mixed freq where we want to connect points across nulls.
        points = [];
        pointsArrays.push(points);
        lastCoord = null;
        continue;
      }

      coord = coords[i];
      isY2 = coords[i]._isY2;
      xCoord = isCentered ? coord.x - (groupWidth / 2) * dir : coord.x;

      if (isY2) {
        if (inBump && isParallel) xCoord += groupWidth * dir; // draw the segment at the end of the bump
        inBump = !inBump;
      }

      if (type == 'curved' && isY2) points.push(null, null); // flag to indicate that the curve should be broken

      if (lastCoord && isParallel)
        // draw the step
        this._pushCoord(points, xCoord, lastCoord.y);

      if (!this._bArea && (type == 'segmented' || type == 'centeredSegmented')) {
        // Start a new list of points to break the segments
        points = [];
        pointsArrays.push(points);
      }

      this._pushCoord(points, xCoord, coord.y);
      lastCoord = coord;
    }

    // Draw the last step
    if (isParallel && !isPolar && lastCoord && !isY2) {
      finalXCoord = isCentered
        ? lastCoord.x + 0.5 * groupWidth * dir
        : lastCoord.x + groupWidth * dir;
      this._pushCoord(points, finalXCoord, lastCoord.y);
    }

    // Connect the last points with the first ones for polar
    if (isPolar && pointsArrays.length > 1) {
      var lastPoints = pointsArrays.pop();
      pointsArrays[0] = lastPoints.concat(pointsArrays[0]);
    }

    return pointsArrays;
  }

  /**
   * Converts the axis coordinate to the stage coordinate and pushes the points to the pointArray.
   * @param {array} pointArray The point array.
   * @param {number} x The x-axis coordinate.
   * @param {number} y The y-axis coordinate.
   * @private
   */
  _pushCoord(pointArray, x, y) {
    var coord = DvtChartCoordUtils.convertAxisCoord(
      this._chart,
      new Point(x, y),
      this._availSpace
    );

    // Round to 1 decimal to keep the DOM small, but prevent undesidered gaps due to rounding errors
    pointArray.push(Math.round(coord.x * 10) / 10, Math.round(coord.y * 10) / 10);
  }

  /**
   * Returns whether the points form a complete ring/donut shape. Only applicable to polar charts.
   * @return {boolean}
   * @private
   */
  _isRing() {
    if (
      !DvtChartTypeUtils.isPolar(this._chart) ||
      !DvtChartTypeUtils.hasGroupAxis(this._chart) ||
      this._arCoord.length < DvtChartDataUtils.getGroupCount(this._chart)
    )
      return false;

    // Check if there is any null that breaks the ring/donut.
    for (var i = 0; i < this._arCoord.length; i++) {
      if (this._arCoord[i].x == null) return false;
    }
    return true;
  }

  /**
   * Returns the spline type of the line/area based on the chart type.
   * @return {string} Spline type.
   * @private
   */
  _getSplineType() {
    if (DvtChartTypeUtils.isScatterBubble(this._chart)) return PathUtils.SPLINE_TYPE_CARDINAL;
    else if (DvtChartTypeUtils.isPolar(this._chart))
      return this._isRing()
        ? PathUtils.SPLINE_TYPE_CARDINAL_CLOSED
        : PathUtils.SPLINE_TYPE_CARDINAL;
    else if (DvtChartTypeUtils.isHorizontal(this._chart))
      return PathUtils.SPLINE_TYPE_MONOTONE_HORIZONTAL;
    else return PathUtils.SPLINE_TYPE_MONOTONE_VERTICAL;
  }

  /**
   * Renders lines.
   * @private
   */
  _renderLines() {
    var pointArrays = this._getPointArrays(this._arCoord, this._type);
    var line;
    for (var i = 0; i < pointArrays.length; i++) {
      var points = pointArrays[i];
      if (points && points.length > 1) {
        if (this._type == 'curved') {
          var cmd = DvtChartLineArea._getCurvedPathCommands(points, false, this._getSplineType());
          line = new Path(this.getCtx(), cmd);
          line.setFill(null);
        } else {
          // not curved
          if (this._isRing()) {
            // create a closed loop
            line = new Polygon(this.getCtx(), points);
            line.setFill(null);
          } else line = new Polyline(this.getCtx(), points);
        }
        line.setStroke(this._stroke);
        line.setClassName(this._className).setStyle(this._style);
        this.addChild(line);
      }
    }
  }

  /**
   * Renders areas.
   * @private
   */
  _renderAreas() {
    // If both the area has both top and bottom coords, remove the edge points that go to the baseline at the two ends.
    // These edge points are invisible, but may show up if border is turned on, and also during animation.
    var arCoord = this._arCoord;
    var arBaseCoord = this._arBaseCoord;
    if (!DvtChartTypeUtils.isPolar(this._chart) && arCoord.length > 0 && arBaseCoord.length > 0) {
      // Don't update the stored arrays (this._arCoord and this._arBaseCoord) as the edge points may be needed later for animation.
      arCoord = arCoord.slice(0);
      arBaseCoord = arBaseCoord.slice(0);

      if (arCoord[0].x != null && arBaseCoord[0].x != null) {
        DvtChartLineArea._removeAreaEdge(arCoord, 0, this._baseline);
        DvtChartLineArea._removeAreaEdge(arBaseCoord, 0, this._baseline);
        arBaseCoord[0].x = arCoord[0].x;
      }
      if (arCoord[arCoord.length - 1].x != null && arBaseCoord[arBaseCoord.length - 1].x != null) {
        DvtChartLineArea._removeAreaEdge(arCoord, arCoord.length - 1, this._baseline);
        DvtChartLineArea._removeAreaEdge(arBaseCoord, arBaseCoord.length - 1, this._baseline);
        arBaseCoord[arBaseCoord.length - 1].x = arCoord[arCoord.length - 1].x;
      }
    }

    var highArrays = this._getPointArrays(arCoord, this._type);
    var lowArrays = this._getPointArrays(arBaseCoord, this._baseType);

    if (highArrays.length != lowArrays.length) return;

    var area;
    for (var i = 0; i < highArrays.length; i++) {
      var highArray = highArrays[i];
      var lowArray = lowArrays[i];

      if (highArray.length < 2) continue;

      var highCurved = this._type == 'curved';
      var lowCurved = this._baseType == 'curved';

      // For polar with group axis, form an polygonal donut if possible
      if (this._isRing()) {
        if (!highCurved) highArray.push(highArray[0], highArray[1]);
        if (lowArray.length >= 2 && !lowCurved) lowArray.push(lowArray[0], lowArray[1]);
      }

      // Reverse the lowArray
      var revLowArray = [];
      for (var j = 0; j < lowArray.length; j += 2)
        revLowArray.unshift(lowArray[j], lowArray[j + 1]);

      // If either the top or the base is a curve, we have to draw a path. Otherwise, we can use polygon.
      if (highCurved || lowCurved) {
        var splineType = this._getSplineType();
        var cmd = highCurved
          ? DvtChartLineArea._getCurvedPathCommands(highArray, false, splineType)
          : PathUtils.polyline(highArray, false);
        cmd += lowCurved
          ? DvtChartLineArea._getCurvedPathCommands(revLowArray, true, splineType)
          : PathUtils.polyline(revLowArray, true);
        cmd += PathUtils.closePath();
        area = new Path(this.getCtx(), cmd);
      } else {
        // not curved
        // Add the reversed low points to the high points to form a range
        var points = revLowArray.concat(highArray);
        area = new Polygon(this.getCtx(), points);
      }

      area.setFill(this._fill);
      area.setClassName(this._className).setStyle(this._style);

      if (this._stroke) area.setStroke(this._stroke);
      this.addChild(area);
    }
  }

  /**
   * Positions the animation indicators.
   * @private
   */
  _positionIndicators() {
    var indicatorObj, indicator, pos, y, coord;
    for (var i = 0; i < this._arCoord.length; i++) {
      coord = this._arCoord[i];
      indicatorObj = this._indicatorMap[coord.groupIndex];

      if (indicatorObj && indicatorObj.indicator) {
        // If the coord has unequal y1 and y2, pick the one farthest from the baseline.
        y =
          (coord.isUpstep(this._baseline) ? coord.y2 : coord.y1) +
          this._INDICATOR_OFFSET *
            (indicatorObj.direction == DvtChartDataChangeUtils.DIR_UP ? -1 : 1);
        pos = DvtChartCoordUtils.convertAxisCoord(
          this._chart,
          new Point(coord.x, y),
          this._availSpace
        );

        indicator = indicatorObj.indicator;
        indicator.setTranslate(pos.x, pos.y);
        indicator.setAlpha(1); // show it because it's hidden when added
        indicator.getParent().addChild(indicator); // reparent to keep at top
      }
    }
  }

  /**
   * Returns the animation params for the line or the area top.
   * @param {DvtChartLineArea} other The shape it is animating from/to. If provided, the animation params are guaranteed
   *     to contain all the groups that the other shape has, in the correct order.
   * @return {array} The animation params in the form of [x y1 y2 groupIndex x y1 y2 groupIndex ...]
   */
  getAnimParams(other) {
    return DvtChartLineArea._coordsToAnimParams(
      this._arCoord,
      other ? other._arCoord : null,
      this._baseline
    );
  }

  /**
   * Updates the animation params for the line or the area top.
   * @param {array} params The animation params in the form of [x y1 y2 groupIndex x y1 y2 groupIndex ...]
   */
  setAnimParams(params) {
    var coords = DvtChartLineArea._animParamsToCoords(params);
    this.setCoords(coords);
  }

  /**
   * Returns the animation params for the area base.
   * @param {DvtChartLineArea} other The shape it is animating from/to. If provided, the animation params are guaranteed
   *     to contain all the groups that the other shape has, in the correct order.
   * @return {array} The animation params in the form of [x y1 y2 groupIndex x y1 y2 groupIndex ...]
   */
  getBaseAnimParams(other) {
    return DvtChartLineArea._coordsToAnimParams(
      this._arBaseCoord,
      other ? other._arBaseCoord : null,
      this._baseline
    );
  }

  /**
   * Updates the animation params for the area base.
   * @param {array} params The animation params in the form of [x y1 y2 groupIndex x y1 y2 groupIndex ...]
   */
  setBaseAnimParams(params) {
    this._arBaseCoord = DvtChartLineArea._animParamsToCoords(params);
  }

  /**
   * Returns a list of group indices that are common between this and other (used for generating animation indicators).
   * @param {DvtChartLineArea} other The shape it is animating from/to.
   * @return {array} The array of common group indices.
   */
  getCommonGroupIndices(other) {
    var indices = [];
    for (var i = 0; i < this._arCoord.length; i++) {
      if (this._arCoord[i].filtered || this._arCoord[i].x == null) continue;

      for (var j = 0; j < other._arCoord.length; j++) {
        if (other._arCoord[j].filtered || other._arCoord[j].x == null) continue;

        if (this._arCoord[i].group == other._arCoord[j].group) {
          indices.push(this._arCoord[i].groupIndex);
          break;
        }
      }
    }
    return indices;
  }

  /**
   * Adds an animation indicator.
   * @param {number} groupIndex The group index corresponding to the indicator.
   * @param {number} direction The direction of the indicator.
   * @param {dvt.Shape} indicator The indicator shape.
   */
  addIndicator(groupIndex, direction, indicator) {
    indicator.setAlpha(0); // hide it until animation starts
    this._indicatorMap[groupIndex] = { direction: direction, indicator: indicator };
  }

  /**
   * Removes all animation indicators.
   */
  removeIndicators() {
    for (var groupIndex in this._indicatorMap) {
      var indicator = this._indicatorMap[groupIndex].indicator;
      if (indicator) indicator.getParent().removeChild(indicator);
    }

    this._indicatorMap = {};
  }

  /**
   * Converts DvtChartCoord array into dvt.Point array. Excludes filtered points.
   * @param {array} coords DvtChartCoord array.
   * @return {array} dvt.Point array.
   * @private
   */
  static _convertToPointCoords(coords) {
    var pointCoords = [];
    for (var i = 0; i < coords.length; i++) {
      if (coords[i].filtered) continue;

      if (coords[i].x == null) pointCoords.push(null);
      else {
        // if y1 == y2, then add just one point. Otherwise, add two points.
        pointCoords.push(new Point(coords[i].x, coords[i].y1));
        if (coords[i].y1 != coords[i].y2) {
          var p2 = new Point(coords[i].x, coords[i].y2);
          p2._isY2 = true; // flag to indicate the point comes from y2
          pointCoords.push(p2);
        }
      }
    }
    return pointCoords;
  }

  /**
   * Converts array of DvtChartCoord into animation params.
   * @param {array} coords Array of DvtChartCoord.
   * @param {array} otherCoords The array of DvtChartCoord it is animating from/to. If provided, the animation params are
   *     guaranteed to contain all the groups that the other shape has, in the correct order.
   * @param {number} baseline The axis baseline coordinate.
   * @return {array} The animation params in the form of [x y1 y2 groupIndex x y1 y2 groupIndex ...].
   * @private
   */
  static _coordsToAnimParams(coords, otherCoords, baseline) {
    if (otherCoords && otherCoords.length > 0) {
      // Construct coords list that contains all the groups that this shape and other shape have.
      if (coords && coords.length > 0) {
        coords = coords.slice(0);
        var otherGroups = DvtChartLineArea._coordsToGroups(otherCoords);
        var groups = DvtChartLineArea._coordsToGroups(coords);
        var idx = coords.length;

        // Iterate otherGroups backwards. For each group, check if the current shape has it. If not, insert a dummy coord
        // into the array, which has an identical value to the coord before it (or after it if the insertion is at the start).
        var group, groupIdx, dummyCoord;
        for (var g = otherGroups.length - 1; g >= 0; g--) {
          group = otherGroups[g];
          groupIdx = groups.indexOf(group);
          if (groupIdx == -1) {
            // Group not found -- insert dummy coord
            if (idx == 0) {
              dummyCoord = coords[0].clone(); // copy coord after it
              coords[0] = coords[0].clone();
              DvtChartLineArea._removeCoordJump(dummyCoord, coords[0], baseline);
            } else {
              dummyCoord = coords[idx - 1].clone(); // copy coord before it
              coords[idx - 1] = coords[idx - 1].clone();
              DvtChartLineArea._removeCoordJump(coords[idx - 1], dummyCoord, baseline);
            }
            dummyCoord.groupIndex = -1;
            coords.splice(idx, 0, dummyCoord);
          } // Group found -- use idx to keep track of the last found group so we know where to add the dummy coord
          else idx = groupIdx;
        }
      } else {
        // this coords is empty, so return the baseline coords
        coords = [];
        for (var g = 0; g < otherCoords.length; g++) {
          coords.push(new DvtChartCoord(otherCoords[g].x, baseline, baseline));
        }
      }
    }

    // Construct the animation params
    var params = [];
    for (var i = 0; i < coords.length; i++) {
      if (coords[i].filtered) continue;

      if (coords[i].x == null) {
        params.push(Infinity); // placeholder for nulls
        params.push(Infinity);
        params.push(Infinity);
      } else {
        params.push(coords[i].x);
        params.push(coords[i].y1);
        params.push(coords[i].y2);
      }
      params.push(coords[i].groupIndex);
    }
    return params;
  }

  /**
   * Converts animation params into array of DvtChartCoord.
   * @param {array} params The animation params in the form of [x y1 y2 groupIndex x y1 y2 groupIndex ...].
   * @return {array} Array of DvtChartCoord.
   * @private
   */
  static _animParamsToCoords(params) {
    var coords = [];
    for (var i = 0; i < params.length; i += 4) {
      if (params[i] == Infinity || isNaN(params[i]))
        coords.push(new DvtChartCoord(null, null, null, params[i + 3]));
      else coords.push(new DvtChartCoord(params[i], params[i + 1], params[i + 2], params[i + 3]));
    }
    return coords;
  }

  /**
   * Converts array of DvtChartCoord into array of group names.
   * @param {array} coords Array of DvtChartCoord.
   * @return {array} Array of group names.
   * @private
   */
  static _coordsToGroups(coords) {
    var groups = [];
    for (var i = 0; i < coords.length; i++) {
      if (!coords[i].filtered) groups.push(coords[i].group);
    }
    return groups;
  }

  /**
   * Removes the jump (due to null values) between startCoord and endCoord, which are duplicates of each other:
   * - If the jump is upward, it eliminates the jump in the endCoord.
   * - If the jump is downward, it eliminates the jump in the startCoord.
   * @param {DvtChartCoord} startCoord The coord on the left (or right in R2L).
   * @param {DvtChartCoord} endCoord The coord on the right (or left in R2L).
   * @param {number} baseline The axis baseline coordinate.
   * @private
   */
  static _removeCoordJump(startCoord, endCoord, baseline) {
    if (startCoord.isUpstep(baseline)) endCoord.y1 = endCoord.y2;
    else startCoord.y2 = startCoord.y1;
  }

  /**
   * Returns the path commands for a curve that goes through the points.
   * @param {array} points The points array.
   * @param {boolean} connectWithLine Whether the first point is reached using lineTo. Otherwise, moveTo is used.
   * @param {string} splineType The spline type.
   * @return {string} Path commands.
   * @private
   */
  static _getCurvedPathCommands(points, connectWithLine, splineType) {
    // First we need to split the points into multiple arrays. The separator between arrays are the nulls, which
    // indicate that the two segments should be connected by a straight line instead of a curve.
    var arP = [];
    var p = [];
    arP.push(p);
    for (var i = 0; i < points.length; i += 2) {
      if (points[i] == null) {
        p = [];
        arP.push(p);
      } else p.push(points[i], points[i + 1]);
    }

    // Connect the last segment with the first one for polar
    if (splineType == PathUtils.SPLINE_TYPE_CARDINAL_CLOSED && arP.length > 1) {
      var lastPoints = arP.pop();
      arP[0] = lastPoints.concat(arP[0]);
      splineType = PathUtils.SPLINE_TYPE_CARDINAL; // multiple segments, so not a closed curve
    }

    var cmd = '';
    for (var i = 0; i < arP.length; i++) {
      p = arP[i];
      cmd += PathUtils.curveThroughPoints(p, connectWithLine, splineType);
      connectWithLine = true; // after the first segment, the rest are connected by straight lines
    }

    return cmd;
  }

  /**
   * Removes the edge point of an area. Used to create range area shape without the extra lines at the edges.
   * @param {array} arCoord The coord array.
   * @param {number} index The index of the edge to be removed.
   * @param {number} baseline The baseline coord.
   * @private
   */
  static _removeAreaEdge(arCoord, index, baseline) {
    var coord = arCoord[index].clone();
    if (coord.isUpstep(baseline)) coord.y1 = coord.y2;
    else coord.y2 = coord.y1;
    arCoord[index] = coord;
  }
}

/**
 *  A marker object for selectable invisible markers.
 *  @param {dvt.Context} context
 *  @param {number} type The marker type.
 *  @param {number} cx The x position of the center of the marker.
 *  @param {number} cy The y position of the center of the marker.
 *  @param {number} size The size of the marker.
 *  @param {boolean} bOptimizeStroke True if the stroke of the markers has been applied on a container.
 *  @extends {dvt.SimpleMarker}
 *  @class DvtChartLineMarker
 *  @constructor
 */
class DvtChartLineMarker extends SimpleMarker {
  constructor(context, type, cx, cy, size, bOptimizeStroke) {
    super(context, type, cx, cy, size, size, null, null, true);

    // Set the stroke if the container may have defined a different one.
    if (bOptimizeStroke) this.setStroke(DvtChartLineMarker.DEFAULT_STROKE);
  }

  /**
   * @override
   */
  setDataColor(dataColor) {
    this._dataColor = dataColor;
    this._hoverStroke = new Stroke(dataColor, 1, 1.5);
  }

  /**
   * @override
   */
  getDataColor() {
    return this._dataColor;
  }

  /**
   * @override
   */
  showHoverEffect() {
    this.IsShowingHoverEffect = true;
    this.setStroke(this._hoverStroke);
  }

  /**
   * @override
   */
  hideHoverEffect() {
    this.IsShowingHoverEffect = false;
    this.setStroke(
      this.isSelected() ? DvtChartLineMarker.SELECTED_STROKE : DvtChartLineMarker.DEFAULT_STROKE
    );
  }

  /**
   * @override
   */
  setSelected(selected) {
    if (this.IsSelected == selected) return;

    this.IsSelected = selected;
    if (this.isSelected()) {
      this.setFill(DvtChartLineMarker.SELECTED_FILL);
      this.setStroke(
        this.isHoverEffectShown() ? this._hoverStroke : DvtChartLineMarker.SELECTED_STROKE
      );
    } else {
      this.setInvisibleFill();
      this.setStroke(
        this.isHoverEffectShown() ? this._hoverStroke : DvtChartLineMarker.DEFAULT_STROKE
      );
    }
  }

  /**
   * @override
   */
  UpdateSelectionEffect() {
    // noop: Selection effects fully managed by this class
  }
}

/** @const */
DvtChartLineMarker.DEFAULT_STROKE = new Stroke('none');

/** @const */
DvtChartLineMarker.SELECTED_FILL = new SolidFill('#FFFFFF');

/** @const */
DvtChartLineMarker.SELECTED_STROKE = new Stroke('#5A5A5A', 1, 1.5);

/**
 *  A selectable polar bar for charting.
 *  @class DvtChartPolarBar
 *  @extends {dvt.Path}
 *  @constructor
 *  @param {Chart} chart
 *  @param {number} axisCoord The location of the axis line.
 *  @param {number} baselineCoord The location from which the bar grows.
 *  @param {number} endCoord The location where the bar length ends.
 *  @param {number} x1 The left coord of a vertical bar, or the top of a horizontal bar.
 *  @param {number} x2 The right coord of a vertical bar, or the bottom of a horizontal bar.
 *  @param {number} availSpace The plotAreaSpace, to convert the polar coordinates.
 */
class DvtChartPolarBar extends DvtChartSelectableWedge {
  constructor(chart, axisCoord, baselineCoord, endCoord, x1, x2, availSpace) {
    // Initialize the path
    super(chart.getCtx());

    /**
     * Min bar length for gaps to be added, in pixels.
     * @const
     * @private
     */
    this._MIN_BAR_LENGTH_FOR_GAPS = 4;

    /**
     * Gap between bars in a stack in pixels.
     * @const
     * @private
     */
    this._MAX_DATA_ITEM_GAP = 3;

    this._axisCoord = axisCoord;
    this._availSpace = availSpace.clone();
    this._bbox = null;
    this._dataItemGaps = DvtChartStyleUtils.getDataItemGaps(chart) * this._MAX_DATA_ITEM_GAP;

    // Calculate the path commands and apply to the path
    this._setBarCoords(baselineCoord, endCoord, x1, x2);
  }

  /**
   * Returns the layout parameters for the current animation frame.
   * @return {array} The array of layout parameters.
   */
  getAnimParams() {
    return [this._baselineCoord, this._endCoord, this._x1, this._x2];
  }

  /**
   * Sets the layout parameters for the current animation frame.
   * @param {array} params The array of layout parameters.
   * @param {dvt.Displayable=} indicator The animation indicator, whose geometry is centered at (0,0).
   */
  setAnimParams(params) {
    // Set bar coords but don't adjust for gaps, since they've already been factored in.
    this._setBarCoords(params[0], params[1], params[2], params[3]);
  }

  /**
   * Returns the primary dvt.Fill for this bar. Used for animation, since getFill may return the fill of the selection
   * shapes.
   * @return {dvt.Fill}
   */
  getPrimaryFill() {
    // Note: getFill is currently correct, but will change once we stop using filters.
    return this.getFill();
  }

  /**
   * Returns a dvt.Playable containing the animation of the bar to its initial data value.
   * @param {number} duration The duration of the animation in seconds.
   * @return {dvt.Playable} A playable for the initial bar animation.
   */
  getDisplayAnim(duration) {
    // Current state is the end state
    var endState = this.getAnimParams();

    // Initialize the start state. To grow the bar, just set the end coord to the baseline coord.
    this.setAnimParams([this._axisCoord, this._axisCoord, 0, 0]);

    // Create and return the playable
    var nodePlayable = new CustomAnimation(this.getCtx(), this, duration);
    nodePlayable
      .getAnimator()
      .addProp(
        Animator.TYPE_NUMBER_ARRAY,
        this,
        this.getAnimParams,
        this.setAnimParams,
        endState
      );
    return nodePlayable;
  }

  /**
   * Returns a dvt.Playable containing the animation to delete the bar.
   * @param {number} duration The duration of the animation in seconds.
   * @return {dvt.Playable}
   */
  getDeleteAnim(duration) {
    // End state is for the bar length to shrink to 0
    var endState = [this._baselineCoord, this._baselineCoord, this._x1, this._x2];

    // Create and return the playable
    var nodePlayable = new CustomAnimation(this.getCtx(), this, duration);
    nodePlayable
      .getAnimator()
      .addProp(
        Animator.TYPE_NUMBER_ARRAY,
        this,
        this.getAnimParams,
        this.setAnimParams,
        endState
      );
    nodePlayable
      .getAnimator()
      .addProp(Animator.TYPE_NUMBER, this, this.getAlpha, this.setAlpha, 0);
    return nodePlayable;
  }

  /**
   * Returns a dvt.Playable containing the insert animation of the bar.
   * @param {number} duration The duration of the animation in seconds.
   * @return {dvt.Playable}
   */
  getInsertAnim(duration) {
    // Initialize the alpha to fade in the bar
    this.setAlpha(0);

    // Current state is the end state
    var endState = this.getAnimParams();

    // Initialize the start state. To grow the bar, just set the end coord to the baseline coord.
    this.setAnimParams([this._baselineCoord, this._baselineCoord, this._x1, this._x2]);

    // Create and return the playable
    var nodePlayable = new CustomAnimation(this.getCtx(), this, duration);
    nodePlayable
      .getAnimator()
      .addProp(
        Animator.TYPE_NUMBER_ARRAY,
        this,
        this.getAnimParams,
        this.setAnimParams,
        endState
      );
    nodePlayable
      .getAnimator()
      .addProp(Animator.TYPE_NUMBER, this, this.getAlpha, this.setAlpha, 1);
    return nodePlayable;
  }

  /**
   * Stores the point coords defining the bar.
   * @param {number} baselineCoord The location from which the bar grows.
   * @param {number} endCoord The location where the bar length ends.
   * @param {number} x1 The left coord of a vertical bar, or the top of a horizontal bar.
   * @param {number} x2 The right coord of a vertical bar, or the bottom of a horizontal bar.
   * @private
   */
  _setBarCoords(baselineCoord, endCoord, x1, x2) {
    var cx = this._availSpace.x + this._availSpace.w / 2;
    var cy = this._availSpace.y + this._availSpace.h / 2;
    var r = Math.max(endCoord, baselineCoord);
    var ir =
      Math.abs(endCoord - baselineCoord) >= this._MIN_BAR_LENGTH_FOR_GAPS &&
      this._axisCoord != baselineCoord
        ? Math.min(endCoord, baselineCoord) + this._dataItemGaps
        : Math.min(endCoord, baselineCoord);
    var sa = 360 - Math$1.radsToDegrees(Math.max(x1, x2)) + 90;
    var ae = Math$1.radsToDegrees(Math.abs(x2 - x1));
    this.setWedgeParams(cx, cy, r, r, sa, ae, this._dataItemGaps, ir);

    // Calculate the coords to compute the bounding box
    var inner1 = DvtChartCoordUtils.polarToCartesian(baselineCoord, x1, this._availSpace);
    var inner2 = DvtChartCoordUtils.polarToCartesian(baselineCoord, x2, this._availSpace);
    var outer1 = DvtChartCoordUtils.polarToCartesian(endCoord, x1, this._availSpace);
    var outer2 = DvtChartCoordUtils.polarToCartesian(endCoord, x2, this._availSpace);
    var minX = Math.min(inner1.x, inner2.x, outer1.x, outer2.x);
    var maxX = Math.max(inner1.x, inner2.x, outer1.x, outer2.x);
    var minY = Math.min(inner1.y, inner2.y, outer1.y, outer2.y);
    var maxY = Math.max(inner1.y, inner2.y, outer1.y, outer2.y);
    this._bbox = new Rectangle(minX, minY, maxX - minX, maxY - minY);

    // Store the geometry values, needed for animation
    this._baselineCoord = baselineCoord;
    this._endCoord = endCoord;
    this._x1 = x1;
    this._x2 = x2;
  }

  /**
   * Returns the bounding box of the shape
   * @return {dvt.Rectangle} bbox
   */
  getBoundingBox() {
    return this._bbox;
  }

  /**
   * Returns the non rounded width of the bar
   * @return {number}
   */
  getOriginalBarSize() {
    return this._bbox.w;
  }
}

/**
 * A marker for range area chart.
 * @class DvtChartRangeMarker
 * @extends {dvt.Path}
 * @constructor
 * @param {dvt.Context} context
 * @param {number} x1 The x coord of the first point.
 * @param {number} y1 The y coord of the first point.
 * @param {number} x2 The x coord of the second point.
 * @param {number} y2 The y coord of the second point.
 * @param {number} markerSize The diameter of the marker.
 * @param {boolean} isInvisible Whether the marker is invisible unless hovered/selected.
 */
class DvtChartRangeMarker extends Path {
  constructor(context, x1, y1, x2, y2, markerSize, isInvisible) {
    super(context);
    this._markerSize = markerSize;
    this._isInvisible = isInvisible;

    this._drawPath(x1, y1, x2, y2);
  }

  /**
   * Draws the marker shape.
   * @param {number} x1 The x coord of the first point.
   * @param {number} y1 The y coord of the first point.
   * @param {number} x2 The x coord of the second point.
   * @param {number} y2 The y coord of the second point.
   * @private
   */
  _drawPath(x1, y1, x2, y2) {
    // Construct the path
    var angle = Math.atan2(y2 - y1, x2 - x1);
    var r = this._markerSize / 2;
    var lineAngle = Math.PI / 8;

    var cmds =
      PathUtils.moveTo(
        x1 + r * Math.cos(angle + lineAngle),
        y1 + r * Math.sin(angle + lineAngle)
      ) +
      PathUtils.arcTo(
        r,
        r,
        2 * (Math.PI - lineAngle),
        1,
        x1 + r * Math.cos(angle - lineAngle),
        y1 + r * Math.sin(angle - lineAngle)
      ) +
      PathUtils.lineTo(
        x2 - r * Math.cos(angle + lineAngle),
        y2 - r * Math.sin(angle + lineAngle)
      ) +
      PathUtils.arcTo(
        r,
        r,
        2 * (Math.PI - lineAngle),
        1,
        x2 - r * Math.cos(angle - lineAngle),
        y2 - r * Math.sin(angle - lineAngle)
      ) +
      PathUtils.closePath();

    this.setCmds(cmds);

    // Save the coords
    this._x1 = x1;
    this._y1 = y1;
    this._x2 = x2;
    this._y2 = y2;
  }

  /**
   * Specifies the styles needed to generate the selection effect.
   * @param {dvt.Fill} fill
   * @param {dvt.Stroke} stroke
   * @param {string} dataColor The color of the data.
   * @param {string} innerColor The color of the inner selection border.
   * @param {string} outerColor The color of the outer selection border.
   */
  setStyleProperties(fill, stroke, dataColor, innerColor, outerColor) {
    this._dataColor = dataColor;
    var hoverColor = SelectionEffectUtils.getHoverBorderColor(dataColor);

    if (this._isInvisible) {
      this.setInvisibleFill();
      this._hoverStroke = new Stroke(hoverColor, 1, 1.5);
    } else {
      this.setFill(fill);
      this.setStroke(stroke);
      this.setHoverStroke(new Stroke(innerColor, 1, 1), new Stroke(hoverColor, 1, 3.5));
      this.setSelectedStroke(
        new Stroke(innerColor, 1, 1.5),
        new Stroke(outerColor, 1, 4.5)
      );
      this.setSelectedHoverStroke(
        new Stroke(innerColor, 1, 1.5),
        new Stroke(hoverColor, 1, 4.5)
      );
    }
  }

  /**
   * @override
   */
  getDataColor() {
    return this._dataColor;
  }

  /**
   * @override
   */
  showHoverEffect() {
    if (this._isInvisible) {
      this.IsShowingHoverEffect = true;
      this.setStroke(this._hoverStroke);
    } else super.showHoverEffect();
  }

  /**
   * @override
   */
  hideHoverEffect() {
    if (this._isInvisible) {
      this.IsShowingHoverEffect = false;
      this.setStroke(
        this.isSelected() ? DvtChartLineMarker.SELECTED_STROKE : DvtChartLineMarker.DEFAULT_STROKE
      );
    } else super.hideHoverEffect();
  }

  /**
   * @override
   */
  setSelected(selected) {
    if (this._isInvisible) {
      if (this.IsSelected == selected) return;

      this.IsSelected = selected;
      if (this.isSelected()) {
        this.setFill(DvtChartLineMarker.SELECTED_FILL);
        this.setStroke(
          this.isHoverEffectShown() ? this._hoverStroke : DvtChartLineMarker.SELECTED_STROKE
        );
      } else {
        this.setInvisibleFill();
        this.setStroke(
          this.isHoverEffectShown() ? this._hoverStroke : DvtChartLineMarker.DEFAULT_STROKE
        );
      }
    } else super.setSelected(selected);
  }

  /**
   * @override
   */
  UpdateSelectionEffect() {
    if (!this._isInvisible) super.UpdateSelectionEffect();
  }

  /**
   * Returns the animation params for the marker.
   * @return {array} params
   */
  getAnimParams() {
    return [this._x1, this._y1, this._x2, this._y2];
  }

  /**
   * Updates the animation params for the marker.
   * @param {array} params
   */
  setAnimParams(params) {
    this._drawPath(params[0], params[1], params[2], params[3]);
  }

  /**
   * Returns whether the marker is invisible.
   * @return {boolean}
   */
  isInvisible() {
    return this._isInvisible;
  }

  /**
   * Returns the bounding box of the shape
   * @return {dvt.Rectangle} bbox
   */
  getBoundingBox() {
    return this.getBoundingBox1().getUnion(this.getBoundingBox2());
  }

  /**
   * Returns the bounding box of marker #1 (x1, y1)
   * @return {dvt.Rectangle} bbox
   */
  getBoundingBox1() {
    return new Rectangle(
      this._x1 - this._markerSize / 2,
      this._y1 - this._markerSize / 2,
      this._markerSize,
      this._markerSize
    );
  }

  /**
   * Returns the bounding box of marker #2 (x2, y2)
   * @return {dvt.Rectangle} bbox
   */
  getBoundingBox2() {
    return new Rectangle(
      this._x2 - this._markerSize / 2,
      this._y2 - this._markerSize / 2,
      this._markerSize,
      this._markerSize
    );
  }
}

/**
 *   Animation on Datachange functionality.
 *   @class
 */

const DvtChartAnimOnDC = {
  /**
   * Creates a dvt.Playable that performs animation between a chart's data states.
   * @param {Chart} oldChart
   * @param {Chart} newChart
   * @param {string} type The animation type.
   * @param {number} duration The duration of the animation in seconds.
   * @param {dvt.Container} delContainer The container for adding the deleted objects.
   * @return {dvt.Playable}
   */
  createAnim: (oldChart, newChart, type, duration, delContainer) => {
    if (!DvtChartAnimOnDC._canAnimate(oldChart, newChart)) {
      return null;
    }

    var ctx = newChart.getCtx();

    // Build arrays of old and new data change handlers.
    var arOldList = [];
    var arNewList = [];
    if (DvtChartTypeUtils.isPie(newChart)) {
      arOldList.push(oldChart.pieChart);
      arNewList.push(newChart.pieChart);
    } else {
      DvtChartAnimOnDC._buildAnimLists(ctx, arOldList, oldChart, arNewList, newChart, duration);
    }

    //  Walk the datachange handler arrays, and create animators for risers
    //  that need to be updated/deleted/inserted.
    var playable;
    var handler = new DataAnimationHandler(ctx, delContainer);
    handler.constructAnimation(arOldList, arNewList);
    if (handler.getNumPlayables() > 0) playable = handler.getAnimation(true);

    // Animating the fade-in of data labels if they exist for this chart.
    var newLabels = newChart.getDataLabels();
    if (playable && newLabels.length > 0) {
      for (var i = 0; i < newLabels.length; i++) newLabels[i].setAlpha(0);
      playable = new SequentialPlayable(
        ctx,
        playable,
        new AnimFadeIn(ctx, newLabels, duration / 4)
      );
    }

    return playable;
  },

  /**
   * Builds two (supplied) arrays of data change handlers (such as {@link DvtChartDataChange3DBar}
   * for the old and new charts. Also creates this._Y1Animation list of gridline
   * playables if axis scale change.
   * @param {dvt.Context} ctx
   * @param {array} arOldList The array to fill in the old peers.
   * @param {Chart} oldChart
   * @param {array} arNewList The array to fill in the new peers.
   * @param {Chart} newChart
   * @param {number} duration Animation duration.
   * @private
   */
  _buildAnimLists: (ctx, arOldList, oldChart, arNewList, newChart, duration) => {
    //  Create a list of DC handlers in arOldPeers and arNewPeers for the old and new peers.
    var i, j;
    var ar = oldChart.getChartObjPeers();
    var aOut = arOldList; // start on old peers first
    var peer, obj, dch;
    var isDataFiltered = newChart.getCache().getFromCache('dataFiltered');

    for (i = 0; i < 2; i++) {
      // loop over old peers and new peers
      var barCount = {}; // keeps track of how many bars have been handled for each series
      var lineAreaCount = {}; // keeps track of areas and lines for areas
      for (j = 0; j < ar.length; j++) {
        peer = ar[j];

        obj = peer.getDisplayables()[0];
        dch = null;

        if (obj instanceof DvtChartFunnelSlice) {
          dch = new DvtChartDataChangeFunnelSlice(peer, duration, '/funnel');
        } else if (obj instanceof DvtChartPyramidSlice) {
          dch = new DvtChartDataChangePyramidSlice(peer, duration, '/pyramid');
        } else if (obj instanceof DvtChartBar || obj instanceof DvtChartPolarBar) {
          dch = new DvtChartDataChangeBar(peer, duration, '/bar');

          // When the bars are filtered, the groups that are rendered would vary depending on data, so we workaround by
          // animating the rendered bars based on the ordering, regardless of which group they belong to. Otherwise,
          // we'd see undesired insert/delete animations every time.
          // Note that this approach is only correct if the number of groups stays the same before and after the animation.
          if (isDataFiltered) {
            var series = peer.getSeries();
            barCount[series] = barCount[series] ? barCount[series] + 1 : 1;
            dch.setId(series + '/' + barCount[series] + '/bar');
          }
        } else if (obj instanceof DvtChartLineArea) {
          dch = new DvtChartDataChangeLineArea(peer, duration);
          var lineAreaId = dch.getId();
          lineAreaCount[lineAreaId] = lineAreaCount[lineAreaId] ? lineAreaCount[lineAreaId] + 1 : 1;
          dch.setId(lineAreaId + '/' + lineAreaCount[lineAreaId]);
        } else if (obj instanceof SimpleMarker) {
          // DvtChartLineMarker is invisible unless selected.
          if (obj instanceof DvtChartLineMarker && !obj.isSelected()) continue;

          dch = new DvtChartDataChangeMarker(peer, duration, '/marker');
        } else if (obj instanceof DvtChartRangeMarker) {
          if (obj.isInvisible() && !obj.isSelected()) continue;

          dch = new DvtChartDataChangeRangeMarker(peer, duration, '/rangeMarker');
        } else if (obj instanceof DvtChartCandlestick) {
          dch = new DvtChartDataChangeHandler(peer, duration);
        } else if (obj instanceof DvtChartBoxAndWhisker) {
          dch = new DvtChartDataChangeHandler(peer, duration, '/boxAndWhisker');
        }

        if (dch) {
          aOut.push(dch);
          dch.setOldChart(oldChart);
        }
      }

      // repeat on the new chart's peer
      aOut = arNewList;
      ar = newChart.getChartObjPeers();
    }
  },

  /**
   * Checks if animation between the two charts is possible.
   * @param {Chart} oldChart
   * @param {Chart} newChart
   * @return {boolean} true if animation can be performed, else false.
   * @private
   */
  _canAnimate: (oldChart, newChart) => {
    //  Test for conditions for which we will not animate.
    if (DvtChartTypeUtils.isPie(oldChart) && DvtChartTypeUtils.isPie(newChart))
      return oldChart && newChart;
    else if (DvtChartTypeUtils.isPolar(oldChart) != DvtChartTypeUtils.isPolar(newChart))
      return false;
    else if (DvtChartTypeUtils.isBLAC(oldChart) && DvtChartTypeUtils.isBLAC(newChart)) return true;
    else if (
      DvtChartTypeUtils.isScatterBubble(oldChart) &&
      DvtChartTypeUtils.isScatterBubble(newChart)
    )
      return true;
    else if (oldChart.getType() == newChart.getType()) return true;
    else return false;
  }
};

/**
 *   Animation on Display funtionality.
 *   @class
 */
const DvtChartAnimOnDisplay = {
  /**
   *  Creates a dvt.Playable that performs initial animation for a chart.
   *  @param {Chart} chart
   *  @param {string} type The animation type.
   *  @param {number} duration The duration of the animation in seconds.
   *  @return {dvt.Playable} The animation of the chart objects that are subject to animation.
   */
  createAnim: (chart, type, duration) => {
    var arPlayables = [];

    if (DvtChartTypeUtils.isBLAC(chart)) {
      DvtChartAnimOnDisplay._animBarLineArea(chart, duration, arPlayables);
    } else if (
      DvtChartTypeUtils.isScatterBubble(chart) ||
      DvtChartTypeUtils.isFunnel(chart) ||
      DvtChartTypeUtils.isPyramid(chart)
    ) {
      DvtChartAnimOnDisplay._animBubbleScatterFunnelPyramid(chart, duration, arPlayables);
    } else if (DvtChartTypeUtils.isPie(chart) && chart.pieChart) {
      // Delegate to the pie to create the animation.
      return chart.pieChart.getDisplayAnim();
    }

    return arPlayables.length > 0 ? new ParallelPlayable(chart.getCtx(), arPlayables) : null;
  },

  /**
   *  Adds a list of playables that animates the chart on initial display, for
   *  the bar and line/area components (including visible markers) to the
   *  supplied array.
   *  @param {Chart} chart
   *  @param {number} duration The duration of the animation in seconds.
   *  @param {Array} arPlayables The array to which the playables should be added.
   *  @private
   */
  _animBarLineArea: (chart, duration, arPlayables) => {
    var objs = chart.getChartObjPeers();
    var objCount = objs ? objs.length : 0;

    if (objCount) {
      var obj, peer;
      var nodePlayable;

      for (var i = 0; i < objCount; i++) {
        peer = objs[i];

        obj = peer.getDisplayables()[0];
        var seriesType = DvtChartDataUtils.getSeriesType(peer.getChart(), peer.getSeriesIndex());

        nodePlayable = null;
        if (
          obj instanceof DvtChartBar ||
          obj instanceof DvtChartPolarBar ||
          obj instanceof DvtChartCandlestick ||
          obj instanceof DvtChartBoxAndWhisker
        ) {
          nodePlayable = obj.getDisplayAnim(duration);
        } else if (obj instanceof DvtChartLineArea) {
          if (seriesType == 'line')
            nodePlayable = DvtChartAnimOnDisplay._getLinePlayable(chart, obj, duration);
          else nodePlayable = DvtChartAnimOnDisplay._getAreaPlayable(chart, obj, duration);
        } else if (obj instanceof SimpleMarker || obj instanceof DvtChartRangeMarker) {
          // DvtChartLineMarker is invisible unless selected.
          if (obj instanceof DvtChartLineMarker && !obj.isSelected()) continue;

          // Fade-in the marker near the end of the line/area animation
          nodePlayable = new AnimFadeIn(chart.getCtx(), obj, duration - 0.8, 0.8);
        }

        if (nodePlayable) {
          arPlayables.push(nodePlayable);
        }
      } // end for
    } // end if objs
  },

  /**
   *  Adds a list of playables that animates the chart on initial display, for
   *  the bubble, scatter, funnel and pyramid components to the supplied array.
   *  @param {Chart} chart
   *  @param {number} duration The duration of the animation in seconds.
   *  @param {Array} arPlayables The array to which the playables should be added.
   *  @private
   */
  _animBubbleScatterFunnelPyramid: (chart, duration, arPlayables) => {
    var objs = chart.getObjects();
    var objCount = objs ? objs.length : 0;

    if (objCount) {
      var obj, peer;
      var nodePlayable;

      for (var i = 0; i < objCount; i++) {
        peer = objs[i];
        obj = peer.getDisplayables()[0];

        if (obj instanceof SimpleMarker)
          nodePlayable = new AnimPopIn(chart.getCtx(), obj, true, duration);
        else if (obj instanceof DvtChartFunnelSlice || obj instanceof DvtChartPyramidSlice) {
          nodePlayable = DvtChartAnimOnDisplay._getFunnelPyramidPlayable(chart, obj, duration);
        }

        if (nodePlayable) arPlayables.push(nodePlayable);
      }
    }
  },

  /**
   *   Returns a dvt.Playable representing the animation of an area polygon
   *   to its initial data value.
   *   @param {Chart} chart
   *   @param {DvtChartLineArea} shape  the area shape to be animated.
   *   @param {number} duration The duration of the animation in seconds.
   *   @return {dvt.Playable} a playable representing the animation of the area to its initial data value.
   *   @private
   */
  _getAreaPlayable: (chart, shape, duration) => {
    var context = chart.getCtx();
    var baselineCoord = shape.getBaseline();

    // Create animation for the area base
    var baseAnim;
    if (shape.isArea()) {
      var baseCoords = shape.getBaseCoords();
      var baseParams = shape.getBaseAnimParams();
      var baseEndState = baseParams.slice(0); // copy, we will update the original
      for (var j = 0; j < baseParams.length; j++) {
        if (j % 4 == 1 || j % 4 == 2)
          // y1 or y2
          baseParams[j] = baselineCoord;
      }
      shape.setBaseAnimParams(baseParams); // set initial position
      baseAnim = new CustomAnimation(context, shape, duration);
      baseAnim
        .getAnimator()
        .addProp(
          Animator.TYPE_NUMBER_ARRAY,
          shape,
          shape.getBaseAnimParams,
          shape.setBaseAnimParams,
          baseEndState
        );
    }

    // Create animation for the area top
    var coords = shape.getCoords();
    var params = shape.getAnimParams();
    var endState = params.slice(0); // copy, we will update the original
    for (var j = 0; j < params.length; j++) {
      if (j % 4 == 1 || j % 4 == 2)
        // y1 or y2
        params[j] = baselineCoord;
    }
    shape.setAnimParams(params); // set initial position
    var topAnim = new CustomAnimation(context, shape, duration);
    topAnim
      .getAnimator()
      .addProp(
        Animator.TYPE_NUMBER_ARRAY,
        shape,
        shape.getAnimParams,
        shape.setAnimParams,
        endState
      );

    // Combine the top and base animation
    var nodePlayable = new ParallelPlayable(chart.getCtx(), baseAnim, topAnim);
    nodePlayable.setOnEnd(() => {
      shape.setCoords(coords, baseCoords);
    });

    return nodePlayable;
  },

  /**
   *   Returns a dvt.Playable representing the animation of a funnel or pyramid slice to
   *   its initial data value and location.
   *   @param {Chart} chart
   *   @param {DvtChartFunnelSlice|DvtChartPyramidSlice} slice  The funnel or pyramid slice to be animated.
   *   @param {number} duration The duration of the animation in seconds.
   *   @return {dvt.Playable} a playable representing the animation of the slice polygon to its initial data value.
   *   @private
   */
  _getFunnelPyramidPlayable: (chart, slice, duration) => {
    var context = chart.getCtx();
    var arPoints = slice.getAnimParams();
    var endState1 = arPoints.slice(0);
    var endState2 = arPoints.slice(0); // copy, we will update the original
    arPoints[0] = 0;
    if (DvtChartTypeUtils.isFunnel(chart)) {
      arPoints[2] = 0;
      endState1[2] = 0;
    }

    slice.setAnimParams(arPoints); // set initial position
    var nodePlayable1 = new CustomAnimation(context, slice, duration / 2);
    nodePlayable1
      .getAnimator()
      .addProp(
        Animator.TYPE_NUMBER_ARRAY,
        slice,
        slice.getAnimParams,
        slice.setAnimParams,
        endState1
      );
    var nodePlayable2 = new CustomAnimation(context, slice, duration / 2);
    nodePlayable2
      .getAnimator()
      .addProp(
        Animator.TYPE_NUMBER_ARRAY,
        slice,
        slice.getAnimParams,
        slice.setAnimParams,
        endState2
      );

    return new SequentialPlayable(context, [nodePlayable1, nodePlayable2]);
  },

  /**
   *   Returns a dvt.Playable representing the animation of the line to
   *   its initial data value.
   *   @param {Chart} chart
   *   @param {DvtChartLineArea} line  the line shape to be animated.
   *   @param {number} duration The duration of the animation in seconds.
   *   @return {dvt.Playable} a playable representing the animation of the line to its initial data value.
   *   @private
   */
  _getLinePlayable: (chart, line, duration) => {
    var coords = line.getCoords();
    var params = line.getAnimParams();
    var endState = params.slice(0); // copy, we will update the original
    DvtChartAnimOnDisplay._getMeanPoints(params); // update params to initial coords
    line.setAnimParams(params); // set initial position

    var nodePlayable = new CustomAnimation(chart.getCtx(), line, duration);
    nodePlayable
      .getAnimator()
      .addProp(
        Animator.TYPE_NUMBER_ARRAY,
        line,
        line.getAnimParams,
        line.setAnimParams,
        endState
      );
    nodePlayable.setOnEnd(() => {
      line.setCoords(coords);
    });
    return nodePlayable;
  },

  /**
   * Updates the supplied array of line coordinates to reflect the mean x or y position of the line data.
   * @param {array} params  The line animation parameters; should be positive values for plot area coordinate points
   * @private
   */
  _getMeanPoints: (params) => {
    var mean = 0;
    var min = Number.MAX_VALUE;
    var max = -1 * Number.MAX_VALUE;
    var len = params.length;
    var i;

    for (i = 0; i < len; i++) {
      // find largest and smallest y-values
      var v = params[i];
      if (i % 4 == 0 || i % 4 == 3 || v == Infinity)
        // x value or groupIndex
        continue;
      if (v < min) min = v;
      if (v > max) max = v;
      mean += v;
    }

    // if more than 2 data points, discard smallest and largest values to get a generally more representative mean.
    if (len > 8) {
      mean -= 2 * min;
      mean -= 2 * max;
      mean /= len / 2 - 4;
    } else mean /= len / 2;

    for (i = 0; i < len; i++) {
      if (i % 4 == 1 || i % 4 == 2)
        // y1 or y2
        params[i] = mean;
    }
  }
};

/**
 * Calculated axis information and drawable creation for a data axis.
 * @param {dvt.Context} context
 * @param {object} options The object containing specifications and data for this component.
 * @param {dvt.Rectangle} availSpace The available space.
 * @class
 * @constructor
 * @extends {DvtAxisInfo}
 */
class DvtDataAxisInfo extends DataAxisInfoMixin(DvtAxisInfo) {
  /**
   * Returns the value correspoding to the first tick label (or gridline) of the axis.
   * @return {number} The value of the min label.
   */
  getMinLabel() {
    if (
      this.ZeroBaseline ||
      (this.Options['_continuousExtent'] == 'on' && this.Options['min'] == null)
    ) {
      // the tickLabels and gridlines should be at integer intervals from zero
      return Math.ceil(this.LinearMinValue / this.MajorIncrement) * this.MajorIncrement;
    }
    // the tickLabels and gridlines should be at integer intervals from the globalMin
    return (
      Math.ceil((this.LinearMinValue - this.LinearGlobalMin) / this.MajorIncrement) *
        this.MajorIncrement +
      this.LinearGlobalMin
    );
  }

  /**
   * @override
   */
  getLabels(context, levelIdx) {
    if (levelIdx && levelIdx > 0)
      // data axis has only one level
      return null;

    var labels = this.getAllLabels(context);
    var labelDims = [];
    var container = context.getStage();

    if (this.Position != 'tangential') {
      labelDims = this.GetLabelDims(labels, container);
      labels = this.SkipLabels(labels, labelDims);
    }

    return labels;
  }

  /**
   * Returns an array of either strings or dvt.OutputTexts containing all the tick labels for this axis.
   *
   * @param {dvt.Context} context
   * @param {Boolean} asString If true, function will return array of label strings and dvt.OutputText objects otherwise
   * @return {Array} An array of label strings or dvt.OutputText objects
   */
  getAllLabels(context, asString) {
    var labels = [];

    // when scaling is set then init formatter
    if (this.Options['tickLabel'] && this.Options['tickLabel']['scaling']) {
      var autoPrecision = this.Options['tickLabel']['autoPrecision']
        ? this.Options['tickLabel']['autoPrecision']
        : 'on';
      this._axisValueFormatter = new LinearScaleAxisValueFormatter(
        this.LinearMinValue,
        this.LinearMaxValue,
        this.MajorIncrement,
        this.Options['tickLabel']['scaling'],
        autoPrecision,
        this.Options.translations
      );
    }

    // Iterate on an integer to reduce rounding error.  We use <= since the first
    // tick is not counted in the tick count.
    for (var i = 0; i <= this.MajorTickCount; i++) {
      var value = i * this.MajorIncrement + this.getMinLabel();
      if (value - this.LinearMaxValue > this.MAJOR_TICK_INCREMENT_BUFFER)
        // Use buffer to address js arithmetic inaccurracy
        break;

      var coord = this.GetUnboundedCoordAt(value);
      if (this.Options['_skipHighestTick']) {
        if (value == this.LinearMaxValue) continue;
        if (
          this.Position != 'tangential' &&
          Math.abs(coord - this.MaxCoord) < this.getTickLabelHeight()
        )
          continue;
      }

      var label;
      if (this.IsLog) {
        // for log scale, format each label individually as the scaling don't need to match across labels
        value = this.linearToActual(value);
        this._axisValueFormatter = new LinearScaleAxisValueFormatter(
          value,
          value,
          value,
          this.Options['tickLabel']['scaling'],
          autoPrecision,
          this.Options.translations
        );
        label = this._formatValue(value);
      } else label = this._formatValue(value);

      if (asString) labels.push(label);
      else labels.push(this.CreateLabel(context, label, coord));
    }

    return labels;
  }

  /**
   * @override
   */
  getMajorTickCoords() {
    var coords = [];

    // Iterate on an integer to reduce rounding error.  We use <= since the first
    // tick is not counted in the tick count.
    for (var i = 0; i <= this.MajorTickCount; i++) {
      var value = i * this.MajorIncrement + this.getMinLabel();
      if (value - this.LinearMaxValue > this.MAJOR_TICK_INCREMENT_BUFFER)
        // Use buffer to address js arithmetic inaccurracy
        break;
      if (this.Options['_skipHighestTick'] && value == this.LinearMaxValue) continue;

      var coord = this.GetUnboundedCoordAt(value);
      coords.push(coord);
    }

    return coords;
  }

  /**
   * @override
   */
  getMinorTickCoords() {
    var coords = [];

    // Iterate on an integer to reduce rounding error.  We use <= since the first
    // tick is not counted in the tick count.
    // Start from i=-1 so that minorTicks that should get rendered before the first majorTick are evaluated
    for (var i = -1; i <= this.MajorTickCount; i++) {
      var value = i * this.MajorIncrement + this.getMinLabel();
      if (this.IsLog && this.MajorIncrement == 1 && this.MinorIncrement == 1) {
        // draw linear ticks from 2 to 9
        for (var j = 2; j <= 9; j++) {
          var linearValue = value + Math$1.log10(j);
          if (linearValue > this.LinearMaxValue) break;
          if (linearValue < this.LinearMinValue) continue;

          coord = this.GetUnboundedCoordAt(linearValue);
          coords.push(coord);
        }
      } else {
        for (var j = 1; j < this.MinorTickCount; j++) {
          var minorValue = value + j * this.MinorIncrement;
          if (minorValue > this.LinearMaxValue) break;
          if (minorValue < this.LinearMinValue) continue;

          var coord = this.GetUnboundedCoordAt(minorValue);
          coords.push(coord);
        }
      }
    }
    return coords;
  }

  /**
   * @param {number} value
   * @return {string} Formatted value.
   * @private
   */
  _formatValue(value) {
    if (this.Converter && this.Converter['format']) {
      if (this._axisValueFormatter) return this._axisValueFormatter.format(value, this.Converter);
      return this.Converter['format'](value);
    } else if (this._axisValueFormatter) return this._axisValueFormatter.format(value);

    // set the # of decimals of the value to the # of decimals of the major increment
    var t = Math$1.log10(this.MajorIncrement);
    var decimals = Math.max(Math.ceil(-t), 0);
    return value.toFixed(decimals);
  }

  /**
   * @override
   */
  getMajorTickCount() {
    return this.MajorTickCount;
  }

  /**
   * @override
   */
  getMinorTickCount() {
    return this.MinorTickCount;
  }

  /**
   * @override
   */
  getMajorIncrement() {
    return this.linearToActual(this.MajorIncrement);
  }

  /**
   * @override
   */
  getMinorIncrement() {
    return this.linearToActual(this.MinorIncrement);
  }

  /**
   * @override
   */
  getMinExtent() {
    return (this.LinearGlobalMax - this.LinearGlobalMin) / DvtDataAxisInfo.MAX_ZOOM;
  }

  /**
   * @override
   */
  getStartOverflow() {
    if (
      (this.Position == 'top' || this.Position == 'bottom') &&
      Agent.isRightToLeft(this.getCtx())
    )
      return this.EndOverflow;
    return this.StartOverflow;
  }

  /**
   * @override
   */
  getEndOverflow() {
    if (
      (this.Position == 'top' || this.Position == 'bottom') &&
      Agent.isRightToLeft(this.getCtx())
    )
      return this.StartOverflow;
    return this.EndOverflow;
  }

  /**
   * @override
   */
  alignLogScaleToTickCount(scaleUnit, tickCount) {
    // only applies to log axis
    if (this.IsLog) {
      var currentMajorTickCount = this.getMajorTickCount();

      // increase the scale unit until we can fit the data into the given number of tick counts
      while (tickCount < currentMajorTickCount) {
        // reset values before recalculation
        this.MajorIncrement = null;
        this.MajorTickCount = null;
        this.MinorIncrement = null;
        this.MinorTickCount = null;

        scaleUnit++;
        this.CalcMajorMinorIncr(scaleUnit);
        currentMajorTickCount = this.getMajorTickCount();
      }
    }

    return scaleUnit;
  }

  /**
   * @override
   */
  getLogScaleUnit() {
    return this.LogScaleUnit;
  }
}

DvtAxisInfo.registerConstructor('data', DvtDataAxisInfo);

/** @private @const */
DvtDataAxisInfo.MAX_ZOOM = 64;

/**
 * Calculated axis information and drawable creation for a time axis.
 * @param {dvt.Context} context
 * @param {object} options The object containing specifications and data for this component.
 * @param {dvt.Rectangle} availSpace The available space.
 * @class
 * @constructor
 * @extends {DvtAxisInfo}
 */

class DvtTimeAxisInfo extends DvtAxisInfo {
  constructor(context, options, availSpace) {
    super(context, options, availSpace);

    // Figure out the coords for the min/max values
    if (this.Position == 'top' || this.Position == 'bottom') {
      // Provide at least the minimum buffer at each side to accommodate labels
      if (!options['_isOverview'] && options['tickLabel']['rendered'] == 'on') {
        this.StartOverflow = Math.max(BaseAxisInfo.MIN_AXIS_BUFFER - options['leftBuffer'], 0);
        this.EndOverflow = Math.max(BaseAxisInfo.MIN_AXIS_BUFFER - options['rightBuffer'], 0);
      }

      // Axis is horizontal, so flip for BIDI if needed
      if (Agent.isRightToLeft(context)) {
        this._startCoord = this.EndCoord - this.EndOverflow;
        this._endCoord = this.StartCoord + this.StartOverflow;
      } else {
        this._startCoord = this.StartCoord + this.StartOverflow;
        this._endCoord = this.EndCoord - this.EndOverflow;
      }
    } else {
      // Vertical axis should go from top to bottom
      this._startCoord = this.StartCoord;
      this._endCoord = this.EndCoord;
    }

    var converter = options['tickLabel'] != null ? options['tickLabel']['converter'] : null;
    this._label1Converter = converter && converter[0] ? converter[0] : converter;
    this._label2Converter = converter && converter[1] ? converter[1] : null;
    this._dateToIsoWithTimeZoneConverter =
      context.getLocaleHelpers()['dateToIsoWithTimeZoneConverter'];

    this._groups = options['groups'];

    var timeAxisType = options['timeAxisType'];
    this._skipGaps = timeAxisType == 'skipGaps';
    this._mixedFrequency = timeAxisType == 'mixedFrequency';

    this.DataMin = options['dataMin'];
    this.DataMax = options['dataMax'];

    if (this._groups.length > 1)
      this._averageInterval = (this.DataMax - this.DataMin) / (this._groups.length - 1);
    else if (this.DataMax - this.DataMin > 0) this._averageInterval = this.DataMax - this.DataMin;
    else this._averageInterval = 6 * DvtTimeAxisInfo.TIME_MINUTE; // to get the time axis to show YMDHM information
    this._step = options['step'];

    // Calculate the increment and add offsets if specified
    var endOffset =
      options['endGroupOffset'] > 0 ? options['endGroupOffset'] * this._averageInterval : 0;
    var startOffset =
      options['startGroupOffset'] > 0 ? options['startGroupOffset'] * this._averageInterval : 0;

    this.GlobalMin = options['min'] != null ? options['min'] : this.DataMin - startOffset;
    this.GlobalMax = options['max'] != null ? options['max'] : this.DataMax + endOffset;

    // Set min/max by start/endGroup
    if (options['viewportStartGroup'] != null)
      this.MinValue = options['viewportStartGroup'] - startOffset;
    if (options['viewportEndGroup'] != null)
      this.MaxValue = options['viewportEndGroup'] + endOffset;

    // Set min/max by viewport min/max
    if (options['viewportMin'] != null) this.MinValue = options['viewportMin'];
    if (options['viewportMax'] != null) this.MaxValue = options['viewportMax'];

    // If min/max is still undefined, fall back to global min/max
    if (this.MinValue == null) this.MinValue = this.GlobalMin;
    if (this.MaxValue == null) this.MaxValue = this.GlobalMax;

    if (this.GlobalMin > this.MinValue) this.GlobalMin = this.MinValue;
    if (this.GlobalMax < this.MaxValue) this.GlobalMax = this.MaxValue;

    this._timeRange = this.MaxValue - this.MinValue;

    this._level1Labels = null;
    this._level2Labels = null;
    // Coordinates of labels need to be stored for gridline rendering
    this._level1Coords = null;
    this._level2Coords = null;
    this._isOneLevel = true;

    // Overflow of labels need to be stored for attempting to align level1 & level2 labels
    this._level1Overflow = [];
    this._level2Overflow = [];

    this._locale = options['_locale'].toLowerCase();

    this._monthResources = context.LocaleData.getMonthNames('abbreviated');
  }

  /**
   * Formats the label given an axis value (used for generating tooltips).
   * @param {Number} axisValue The axis value (in milliseconds)
   * @return {String} A formatted axis label
   */
  formatLabel(axisValue) {
    var date = new Date(axisValue);
    var twoLabels = this._formatAxisLabel(date, null, true);
    if (twoLabels[1] != null) {
      if (
        DvtTimeAxisInfo._getDMYOrder(this._locale) == 'YMD' ||
        (this._timeRange < DvtTimeAxisInfo.TIME_MONTH_MIN && this._step < DvtTimeAxisInfo.TIME_DAY)
      )
        // time showing HH:MM:SS or YMD order
        return twoLabels[1] + ' ' + twoLabels[0];
      else return twoLabels[0] + ' ' + twoLabels[1];
    } else return twoLabels[0];
  }

  /**
   * Calculates the level of granularity in the data
   * @return {Number} The level of granularity (Eg. Second, Minute, Hour, Day, Month, Year)
   *                  represented in milliseconds.
   * @private
   */
  _calculateGranularity() {
    if (
      this._step >= DvtTimeAxisInfo.TIME_YEAR_MIN ||
      this._timeRange >= 6 * DvtTimeAxisInfo.TIME_YEAR_MIN
    )
      return DvtTimeAxisInfo.TIME_YEAR_MIN;

    if (
      this._step >= DvtTimeAxisInfo.TIME_MONTH_MIN ||
      this._timeRange >= 6 * DvtTimeAxisInfo.TIME_MONTH_MIN
    )
      return DvtTimeAxisInfo.TIME_MONTH_MIN;

    if (this._step >= DvtTimeAxisInfo.TIME_DAY || this._timeRange >= 6 * DvtTimeAxisInfo.TIME_DAY)
      return DvtTimeAxisInfo.TIME_DAY;

    if (this._step >= DvtTimeAxisInfo.TIME_HOUR || this._timeRange >= 6 * DvtTimeAxisInfo.TIME_HOUR)
      return DvtTimeAxisInfo.TIME_HOUR;

    if (
      this._step >= DvtTimeAxisInfo.TIME_MINUTE ||
      this._timeRange >= 6 * DvtTimeAxisInfo.TIME_MINUTE
    )
      return DvtTimeAxisInfo.TIME_MINUTE;

    return DvtTimeAxisInfo.TIME_SECOND;
  }

  /**
   * Formats the given date with the given converter
   * @param {Date} date The current date
   * @param {Date} prevDate The date of the previous set of labels
   * @param {Object} converter The converter
   * @return {String} An axis label
   * @private
   */
  _formatAxisLabelWithConverter(date, prevDate, converter) {
    if (converter) {
      var label = null;
      var prevLabel = null;

      if (converter['format']) {
        label = converter['format'](
          this._dateToIsoWithTimeZoneConverter && date
            ? this._dateToIsoWithTimeZoneConverter(date)
            : date
        );
        prevLabel = converter['format'](
          this._dateToIsoWithTimeZoneConverter && prevDate
            ? this._dateToIsoWithTimeZoneConverter(prevDate)
            : prevDate
        );
      }
      if (prevLabel !== label || label == null) {
        return label;
      }

      if (!converter.resolvedOptions) {
        return null;
      }
      // JET-36468: do not skip label when it is same as prevLabel checking for granularity
      // for eg. if June and July both convert to 'J' and granularity is TIME_MONTH_MIN, we want to keep both 'J's
      // for eg. if Saturday and Sunday both convert to 'S' and granularity is TIME_MONTH_DAY, we want to keep both 'S's
      var granularity = this._calculateGranularity();
      if (
        granularity === DvtTimeAxisInfo.TIME_MONTH_MIN &&
        prevDate.getMonth() !== date.getMonth() &&
        converter.resolvedOptions()['month'] != undefined
      ) {
        return label;
      } else if (
        granularity === DvtTimeAxisInfo.TIME_DAY &&
        prevDate.getDate() !== date.getDate() &&
        converter.resolvedOptions()['day'] != undefined
      ) {
        return label;
      }

      return null;
    }
  }

  /**
   * Formats the level 1 and level 2 axis labels
   * @param {Date} date The current date
   * @param {Date} prevDate The date of the previous set of labels
   * @param {boolean} bOneLabel Whether we want to show only one label. Used for tooltip to get correct order for MDY
   * @return {Array} An array [level1Label, level2Label]
   * @private
   */
  _formatAxisLabel(date, prevDate, bOneLabel) {
    var label1 = null; // level 1 label
    var label2 = null; // level 2 label
    var isVert = this.Position == 'left' || this.Position == 'right';
    var granularity = this._calculateGranularity();

    // If dateTimeFormatter is used, use it
    if (this._label1Converter || this._label2Converter) {
      if (this._label1Converter)
        label1 = this._formatAxisLabelWithConverter(date, prevDate, this._label1Converter);
      if (this._label2Converter)
        label2 = this._formatAxisLabelWithConverter(date, prevDate, this._label2Converter);

      return [label1, label2];
    }

    if (granularity === DvtTimeAxisInfo.TIME_YEAR_MIN) {
      label1 = this._formatDate(date, false, false, true); // Year
    } else if (granularity === DvtTimeAxisInfo.TIME_MONTH_MIN) {
      if (prevDate == null || prevDate.getMonth() != date.getMonth())
        label1 = this._formatDate(date, false, true, false); // Month

      if (prevDate == null || prevDate.getYear() != date.getYear())
        label2 = this._formatDate(date, false, false, true); // Year
    } else if (granularity === DvtTimeAxisInfo.TIME_DAY) {
      if (bOneLabel) {
        label1 = this._formatDate(date, true, true, true); // Day, Month, Year
      } else {
        if (prevDate == null || prevDate.getDate() != date.getDate())
          label1 = this._formatDate(date, true, false, false); // Day

        if (prevDate == null || prevDate.getYear() != date.getYear())
          label2 = this._formatDate(date, false, true, true); // Year, Month
        else if (prevDate.getMonth() != date.getMonth())
          label2 = this._formatDate(date, false, true, false); // Month
      }
    } else {
      if (granularity === DvtTimeAxisInfo.TIME_HOUR) {
        if (prevDate == null || prevDate.getHours() != date.getHours())
          label1 = this._formatTime(date, false, false); // HH AM/PM or HH:MM
      } else if (granularity === DvtTimeAxisInfo.TIME_MINUTE) {
        if (prevDate == null || prevDate.getMinutes() != date.getMinutes())
          label1 = this._formatTime(date, true, false); // HH:MM
      } else {
        if (prevDate == null || prevDate.getSeconds() != date.getSeconds())
          label1 = this._formatTime(date, true, true); // HH:MM:SS
      }

      if (isVert) {
        if (prevDate == null || prevDate.getDate() != date.getDate())
          label2 = this._formatDate(date, true, true, false); // Month, Day
      } else {
        if (prevDate == null || prevDate.getYear() != date.getYear())
          label2 = this._formatDate(date, true, true, true); // Year, Month, Day
        else if (prevDate.getMonth() != date.getMonth())
          label2 = this._formatDate(date, true, true, false); // Month, Day
        else if (prevDate.getDate() != date.getDate())
          label2 = this._formatDate(date, true, false, false); // Day
      }
    }

    return [label1, label2];
  }

  /**
   * Returns the date as a DMY string
   * @param {Date} date The date
   * @param {boolean} showDay Whether the day is shown
   * @param {boolean} showMonth Whether the month is shown
   * @param {boolean} showYear Whether the year is shown
   * @return {string} The formatted string
   * @private
   */
  _formatDate(date, showDay, showMonth, showYear) {
    // . Manually add 543 years to the Gregorian year if using a Thai locale.
    // Should use date.toLocaleDateString once it's available on Safari
    var yearStr = date.getFullYear();

    var monthStr;
    if (this._monthResources && this._monthResources.length >= 12)
      monthStr = this._monthResources[date.getMonth()];
    else monthStr = date.toString().split(' ')[1]; // date.toString() returns "Day Mon Date HH:MM:SS TZD YYYY"
    var dayStr = date.getDate();

    // Add the day and year trailing characters if needed
    // These will be "" if not needed
    yearStr += DvtTimeAxisInfo._getYearTrailingCharacters(this._locale);
    dayStr += DvtTimeAxisInfo._getDayTrailingCharacters(this._locale);

    // Process the DMY Order
    var dmyOrder = DvtTimeAxisInfo._getDMYOrder(this._locale);

    var dateStr = '';

    for (var i = 0; i < dmyOrder.length; i++) {
      if (showDay && dmyOrder[i] == 'D') {
        dateStr += dayStr + ' ';
      } else if (showMonth && dmyOrder[i] == 'M') {
        dateStr += monthStr + ' ';
      } else if (showYear && dmyOrder[i] == 'Y') {
        dateStr += yearStr + ' ';
      }
    }

    return dateStr.length > 0 ? dateStr.slice(0, dateStr.length - 1) : dateStr;
  }

  /**
   * Returns the date as an HH:MM:SS string
   * @param {Date} date The date
   * @param {boolean} showMinute Whether the minute is shown
   * @param {boolean} showSecond Whether the second is shown
   * @return {string} The formatted string
   * @private
   */
  _formatTime(date, showMinute, showSecond) {
    var hours = date.getHours();
    var mins = date.getMinutes();
    var secs = date.getSeconds();

    var am = DvtTimeAxisInfo._getAMString(this._locale);
    var pm = DvtTimeAxisInfo._getPMString(this._locale);
    var ampmBefore = DvtTimeAxisInfo._getAMPMBefore(this._locale);

    var b12HFormat = am != '' && pm != '';
    var ampm;
    var timeLabel = '';

    if (Agent.isRightToLeft(this.getCtx())) timeLabel = '\u200F';

    if (b12HFormat) {
      if (hours < 12) {
        ampm = am;
        if (hours == 0) hours = 12;
      } else {
        ampm = pm;
        if (hours > 12) hours -= 12;
      }
      timeLabel += hours;

      if (showMinute || mins != 0) timeLabel += ':' + this._doubleDigit(mins);
    } else timeLabel += this._doubleDigit(hours) + ':' + this._doubleDigit(mins);

    if (showSecond) {
      timeLabel += ':' + this._doubleDigit(secs);
    }

    if (b12HFormat) {
      if (ampmBefore) return ampm + ' ' + timeLabel;
      else return timeLabel + ' ' + ampm;
    } else {
      return timeLabel;
    }
  }

  /**
   * Creates a double-digit number string for the HH:MM:SS format
   * @param {Number} num A number less than 100
   * @return {String} A double-digit number string
   * @private
   */
  _doubleDigit(num) {
    if (num < 10) {
      return '0' + num;
    }
    return '' + num;
  }

  /**
   * Returns the time label interval for mixed frequency data.
   * Makes sure that the interval is a regular time unit.
   * @return {number} The interval.
   * @private
   */
  _getMixedFrequencyStep() {
    if (this._timeRange >= 6 * DvtTimeAxisInfo.TIME_YEAR_MIN) return DvtTimeAxisInfo.TIME_YEAR_MIN;
    if (this._timeRange >= 6 * DvtTimeAxisInfo.TIME_MONTH_MIN)
      return DvtTimeAxisInfo.TIME_MONTH_MIN;
    if (this._timeRange >= 6 * DvtTimeAxisInfo.TIME_DAY) return DvtTimeAxisInfo.TIME_DAY;
    if (this._timeRange >= DvtTimeAxisInfo.TIME_DAY) return 3 * DvtTimeAxisInfo.TIME_HOUR;
    if (this._timeRange >= 6 * DvtTimeAxisInfo.TIME_HOUR) return DvtTimeAxisInfo.TIME_HOUR;
    if (this._timeRange >= DvtTimeAxisInfo.TIME_HOUR) return 15 * DvtTimeAxisInfo.TIME_MINUTE;
    if (this._timeRange >= 30 * DvtTimeAxisInfo.TIME_MINUTE) return 5 * DvtTimeAxisInfo.TIME_MINUTE;
    if (this._timeRange >= 6 * DvtTimeAxisInfo.TIME_MINUTE) return DvtTimeAxisInfo.TIME_MINUTE;
    if (this._timeRange >= DvtTimeAxisInfo.TIME_MINUTE) return 15 * DvtTimeAxisInfo.TIME_SECOND;
    if (this._timeRange >= 30 * DvtTimeAxisInfo.TIME_SECOND) return 5 * DvtTimeAxisInfo.TIME_SECOND;
    return DvtTimeAxisInfo.TIME_SECOND;
  }

  /**
   * Generates the level 1 and level 2 tick labels
   * @param {dvt.Context} context
   * @private
   */
  _generateLabels(context) {
    var labels1 = [];
    var labels2 = [];
    var labelInfos1 = [];
    var coords1 = [];
    var coords2 = [];
    var prevDate = null;
    var c1 = 0; // number of level 1 labels
    var c2 = 0; // number of level 2 labels
    var container = context.getStage(context);
    var isRTL = Agent.isRightToLeft(context);
    var isVert = this.Position == 'left' || this.Position == 'right';
    var scrollable = this.Options['zoomAndScroll'] != 'off';
    var first = true;

    //  : On Chrome, creating a gap value to be used for spacing level1 labels and level2 labels
    var levelsGap = 0;
    if (isVert && Agent.engine === 'blink') {
      levelsGap = this.getTickLabelHeight() * 0.16;
    }

    // Find the time positions where labels are located
    var times = [];
    if (this._step != null) {
      times = DvtTimeAxisInfo._getLabelPos(this.MinValue, this.MaxValue, this._step);
    } else if (this._mixedFrequency) {
      this._step = this._getMixedFrequencyStep();
      times = DvtTimeAxisInfo._getLabelPos(this.MinValue, this.MaxValue, this._step);
    } else {
      for (var i = 0; i < this._groups.length; i++) {
        if (this._groups[i] >= this.MinValue && this._groups[i] <= this.MaxValue)
          times.push(this._groups[i]);
      }
      this._step = this._averageInterval;

      if (!this._skipGaps) {
        // BUG JET-31414 (30393656) MISSING '28' ON X AXIS FOR COORDINATE TO VALUE CONVERSION DEMO
        // Since the axis labels are obtained from the groups, it may miss a few values.
        // Find and treat the missing values before proceeding.
        times = DvtTimeAxisInfo._treatMissingValues(times, this._calculateGranularity());

        // Since the contents of the times array might have been updated, the step value should
        // be updated.
        // The step value is approximated to the estimated average of the intervals in the updated
        // times array.
        // It is safe to do so even when there were no missing values as it should not have any side-effects
        // on rendering the axis.
        // If there are less than 2 values in the times array, proceed with the existing step value.
        if (times.length > 1) {
          this._step = (times[times.length - 1] - times[0]) / (times.length - 1);
        }
        // Check the width of the first level1 label. If we expect that we'll have more group labels than we can fit in the
        // available space, then render the time labels at a regular interval (using mixed freq algorithm).
        var labelWidth;
        if (isVert)
          labelWidth = TextUtils.getTextStringHeight(
            context,
            this.Options['tickLabel']['style']
          );
        else {
          var firstLabelString = this._formatAxisLabel(new Date(times[0] || this.MinValue))[0];
          labelWidth = TextUtils.getTextStringWidth(
            context,
            firstLabelString,
            this.Options['tickLabel']['style']
          );
        }
        var totalWidth = (labelWidth + this.GetTickLabelGapSize()) * (times.length - 1);
        var availWidth = Math.abs(this._endCoord - this._startCoord);
        if (totalWidth > availWidth) {
          this._step = this._getMixedFrequencyStep();
          times = DvtTimeAxisInfo._getLabelPos(this.MinValue, this.MaxValue, this._step);
        }
      }
    }
    if (times.length === 0) times = [this.MinValue]; // render at least one label
    // Create and format the labels
    for (var i = 0; i < times.length; i++) {
      var time = times[i];
      var coord = this.getCoordAt(time);
      if (coord == null) continue;

      var date = new Date(time);
      var twoLabels = this._formatAxisLabel(date, prevDate);

      var label1 = twoLabels[0];
      var label2 = twoLabels[1];
      //level 1 label
      if (label1 != null) {
        // If level 2 exists put a levelsGap space between labels. levelsGap is only non-zero on Chrome.
        labelInfos1.push({ text: label1, coord: label2 != null ? coord + levelsGap : coord });
        coords1.push(coord);
        c1++;
      } else {
        labelInfos1.push(null);
        coords1.push(null);
      }
      // Defer label1 creation for now for performance optimization.
      // Only the labels we expect not to skip will be created in skipLabelsUniform().
      labels1.push(null);

      // Make sure that the position of first level2 label is constant if the chart is scrollable to prevent jumping around
      if (scrollable && first) coord = this.MinValue ? this.getCoordAt(this.MinValue) : coord;
      first = false;

      //level 2 label
      if (label2 != null) {
        var text = this.CreateLabel(context, label2, coord - levelsGap);
        coords2.push(coord);
        if (!isVert)
          //set alignment now in order to determine if the labels will overlap
          isRTL ? text.alignRight() : text.alignLeft();
        labels2.push(text);
        this._isOneLevel = false;
        c2++;
      } else {
        labels2.push(null);
        coords2.push(null);
      }

      prevDate = date;
    }

    // skip level 1 labels every uniform interval
    c1 = this._skipLabelsUniform(labelInfos1, labels1, container, false, isRTL);

    if (!scrollable && c2 > 1 && c1 < 1.5 * c2) {
      // too few level 1 labels
      labels1 = labels2;
      labels2 = null;
      // center align the new level1 labels
      for (var j = 0; j < labels1.length; j++) {
        if (labels1[j] != null) labels1[j].alignCenter();
      }
      c1 = this._skipLabelsGreedy(labels1, this.GetLabelDims(labels1, container), false, isRTL);
    } else {
      // skip level 2 labels greedily
      c2 = this._skipLabelsGreedy(labels2, this.GetLabelDims(labels2, container), true, isRTL);
      if (c2 == 0) labels2 = null; // null it so DvtAxisRenderer.getPreferredSize won't allocate space for it
    }

    if (isVert && labels2 != null) this._skipVertLabels(labels1, labels2, container);

    this._level1Labels = labels1;
    this._level2Labels = labels2;

    // Store coordinates of labels for gridline rendering
    this._level1Coords = coords1;
    this._level2Coords = coords2;
  }

  /**
   * Returns how much a label overflows outside of rendering bounds.
   * @param {Number} coord The current coordinate of a label
   * @param {Number} labelLength The length of a label
   * @param {boolean} isStartAligned Whether or not the labels are text-anchored start, assumes center alignment if false.
   * @param {boolean} isRTL Whether or not the context is right to left.
   * @return {Number} The label overflow
   * @private
   */
  _getLabelOverflow(coord, labelLength, isStartAligned, isRTL) {
    var minOverflow = coord - (isStartAligned ? (isRTL ? labelLength : 0) : labelLength * 0.5);
    if (minOverflow < this.Options['_minOverflowCoord'])
      // Negative overflow : Label overflows the beginning of the axis
      return Math.floor(minOverflow - this.Options['_minOverflowCoord']);

    var maxOverflow = coord + (isStartAligned ? (isRTL ? 0 : labelLength) : labelLength * 0.5);
    if (maxOverflow > this.Options['_maxOverflowCoord'])
      // Negative overflow : Label overflows the beginning of the axis
      return Math.ceil(maxOverflow - this.Options['_maxOverflowCoord']);

    return 0; // No overflow
  }

  /**
   * Skip labels greedily. Delete all labels that overlap with the last rendered label.
   * @param {Array} labels An array of dvt.Text labels for the axis. This array will be modified by the method.
   * @param {Array} labelDims An array of dvt.Rectangle objects that describe the x, y, height, width of the axis labels.
   * @param {boolean} isStartAligned Whether or not the labels are text-anchored start, assumes center alignment if false.
   * @param {boolean} isRTL Whether or not the context is right to left.
   * @return {Number} The number of remaining labels after skipping.
   * @private
   */
  _skipLabelsGreedy(labels, labelDims, isStartAligned, isRTL) {
    // If there are no labels, return
    if (!labelDims || labelDims.length <= 0) return false;

    var isVert = this.Position == 'left' || this.Position == 'right';
    var labelHeight = this.getTickLabelHeight();
    var gap = isVert ? labelHeight * 0.08 : labelHeight * 0.24;

    var count = 0; // the number of non-null labels
    var pointA1, pointA2, pointB1, pointB2;

    // Check for potential overflow
    var label;
    var availWidth = Math.abs(this._endCoord - this._startCoord); // The available width for the axis
    for (var j = 0; j < labelDims.length; j++) {
      this._level2Overflow.push(0);
      if (labels[j] != null) {
        label = labels[j];
        var labelLength = label.getDimensions().w;

        if (labelDims[j].w > availWidth) labels[j] = null;
        else {
          var overflow = this._getLabelOverflow(label.getX(), labelLength, isStartAligned, isRTL);
          this._level2Overflow[j] = overflow;
          if (overflow != 0) {
            label.setX(label.getX() - overflow); // move label
            labelDims[j].x -= overflow; // adjust recorded dims so skipping takes into account new label position
          }
        }
      }
    }

    for (j = 0; j < labelDims.length; j++) {
      if (labelDims[j] == null) continue;

      if (isVert) {
        pointB1 = labelDims[j].y;
        pointB2 = labelDims[j].y + labelDims[j].h;
      } else {
        pointB1 = labelDims[j].x;
        pointB2 = labelDims[j].x + labelDims[j].w;
      }

      if (
        pointA1 != null &&
        pointA2 != null &&
        DvtTimeAxisInfo._isOverlapping(pointA1, pointA2, pointB1, pointB2, gap)
      )
        labels[j] = null;

      if (labels[j] != null) {
        // start evaluating from label j
        pointA1 = pointB1;
        pointA2 = pointB2;
        count++;
      }
    }

    return count;
  }

  /**
   * Skip labels uniformly (every regular interval).
   * @param {array} labelInfos An array of object containing text (the label text string) and coord (the label coordinate).
   * @param {array} labels An array of dvt.OutputText labels for the axis (initially empty). This array will be populated by the method.
   * @param {dvt.Container} container The label container.
   * @param {boolean} isRTL Whether or not the context is right to left.
   * @return {number} The number of remaining labels after skipping.
   * @private
   */
  _skipLabelsUniform(labelInfos, labels, container, isRTL) {
    var rLabelInfos = []; // contains rendered labels only
    var rLabelDims = [];

    // The available width for the axis
    var availWidth = Math.abs(this._endCoord - this._startCoord);

    for (var j = 0; j < labelInfos.length; j++) {
      if (labelInfos[j] != null) {
        rLabelInfos.push(labelInfos[j]);
        rLabelDims.push(null);
        this._level1Overflow.push(0);
      }
    }

    // Method that returns the label size. If the label object doesn't exist yet, it will create it and measure the
    // dimensions. Otherwise, it simply returns the stored dimensions.
    var isVert = this.Position == 'left' || this.Position == 'right';
    var _this = this;
    var getDim = (i) => {
      if (rLabelDims[i] == null) {
        rLabelInfos[i].label = _this.CreateLabel(
          container.getCtx(),
          rLabelInfos[i].text,
          rLabelInfos[i].coord
        );
        rLabelDims[i] = rLabelInfos[i].label.getDimensions(container);

        if (rLabelDims[i].w > availWidth) {
          rLabelInfos[i].label = null;
          rLabelDims[i].w = 0;
          rLabelDims[i].h = 0;
        } else {
          var overflow = _this._getLabelOverflow(
            rLabelInfos[i].coord,
            rLabelDims[i].w,
            false,
            isRTL
          );
          if (overflow != 0) {
            rLabelInfos[i].coord -= overflow;
            rLabelDims[i].x -= overflow; // adjust recorded dims so skipping takes into account new label position
            rLabelInfos[i].label.setX(rLabelInfos[i].label.getX() - overflow);
            _this._level1Overflow[i] = overflow;
          }
        }
      }
      return isVert ? rLabelDims[i].h : rLabelDims[i].w;
    };

    // Estimate the minimum amount of skipping by dividing the total label width (estimated) by the
    // available axis width.
    var totalWidth = (getDim(0) + this.GetTickLabelGapSize()) * (rLabelInfos.length - 1);
    var skip = availWidth > 0 ? Math.ceil(totalWidth / availWidth) - 1 : 0;

    // Iterate to find the minimum amount of skipping
    var bOverlaps = true;
    while (bOverlaps) {
      for (var j = 0; j < rLabelInfos.length; j++) {
        if (j % (skip + 1) == 0) {
          getDim(j); // create the label and obtain the dim
          rLabelInfos[j].skipped = false;
        } else rLabelInfos[j].skipped = true;
      }
      bOverlaps = this.IsOverlapping(rLabelDims, skip);
      skip++;
    }

    // Populate the labels array with non-skipped labels
    var count = 0; // # of rendered labels
    for (var j = 0; j < labelInfos.length; j++) {
      if (labelInfos[j] != null && !labelInfos[j].skipped) {
        labels[j] = labelInfos[j].label;
        count++;
      }
    }
    return count;
  }

  /**
   * Format the alignments of the vertical axis labels and skip them accordingly so that level1 and level2 don't overlap.
   * @param {Array} labels1 An array of level 1 dvt.Text labels for the axis. This array will be modified by the method.
   * @param {Array} labels2 An array of level 2 dvt.Text labels for the axis. This array will be modified by the method.
   * @param {dvt.Container} container
   * @private
   */
  _skipVertLabels(labels1, labels2, container) {
    var gap = this.getTickLabelHeight() * 0.08;

    // returns if two rectangles (dimsA and dimsB) overlap vertically
    var isOverlapping = (dimsA, dimsB) => {
      return DvtTimeAxisInfo._isOverlapping(
        dimsA.y,
        dimsA.y + dimsA.h,
        dimsB.y,
        dimsB.y + dimsB.h,
        gap
      );
    };

    var lastDims = null;
    var overlapping = false;

    // attempt to render both level 1 and level 2 and see if they fit on the axis
    for (var i = 0; i < labels1.length; i++) {
      if (labels1[i] && labels2[i]) {
        labels1[i].alignTop();
        labels2[i].alignBottom();
        if (lastDims && isOverlapping(lastDims, labels2[i].getDimensions())) {
          overlapping = true;
          break;
        }
        lastDims = labels1[i].getDimensions();
      } else if (labels1[i] || labels2[i]) {
        var label = labels1[i] ? labels1[i] : labels2[i];
        if (lastDims && isOverlapping(lastDims, label.getDimensions())) {
          overlapping = true;
          break;
        }
        lastDims = label.getDimensions();
      }
    }

    if (!overlapping) return; // if both levels fit, we're done
    var lastLv1Idx = null;
    var lastLv1Dims = null;
    var lastLv2Dims = null;
    var dims;

    // if they don't fit:
    // - for points that have level 2 labels, don't generate the level 1 (one level nesting)
    // - skip all level 1 labels that overlaps with level 2 labels
    for (i = 0; i < labels1.length; i++) {
      if (labels2[i]) {
        // if level 2 exists
        labels1[i] = null; // delete level 1
        labels2[i].alignMiddle();
        dims = labels2[i].getDimensions();
        if (lastLv1Dims && isOverlapping(lastLv1Dims, dims)) {
          labels1[lastLv1Idx] = null;
        }
        lastLv2Dims = dims;
      } else if (labels1[i]) {
        // if level 1 exists but not level 2
        dims = labels1[i].getDimensions();
        if (lastLv2Dims && isOverlapping(lastLv2Dims, dims)) {
          labels1[i] = null;
        } else {
          lastLv1Dims = dims;
          lastLv1Idx = i;
        }
      }
    }
  }

  /**
   * @override
   */
  getLabels(context, levelIdx) {
    if (levelIdx && levelIdx > 1)
      // time axis has no more than two levels
      return null;

    if (!this._level1Labels) this._generateLabels(context);

    if (levelIdx == 1) {
      return this._level2Labels;
    }

    return this._level1Labels;
  }

  /**
   * @override
   */
  getMajorTickCoords() {
    var coords = [];
    if (this._isOneLevel) {
      // only one level, level1 is majorTick
      for (var i = 0; i < this._level1Coords.length; i++) {
        if (this._level1Coords[i] != null && this._level1Labels[i] != null)
          coords.push(this._level1Coords[i]);
      }
    } else {
      // level1 is minorTick, level2 is majorTick
      // don't draw majorTick for the first level2 label bc it's not the beginning of period
      for (var i = 1; i < this._level2Coords.length; i++) {
        if (this._level2Coords[i] != null) coords.push(this._level2Coords[i]); // render gridline even if label is skipped
      }
    }

    return coords;
  }

  /**
   * @override
   */
  getMinorTickCoords() {
    if (this._isOneLevel)
      // minorTick only applies on timeAxis if there is more than one level
      return [];

    var coords = [];
    for (var i = 0; i < this._level1Coords.length; i++) {
      if (this._level1Coords[i] != null && this._level1Labels[i] != null)
        coords.push(this._level1Coords[i]);
    }

    return coords;
  }

  /**
   * @override
   */
  getUnboundedValAt(coord) {
    if (coord == null) return null;

    var ratio = (coord - this._startCoord) / (this._endCoord - this._startCoord);

    if (this._skipGaps) {
      var minVal = this._timeToIndex(this.MinValue);
      var maxVal = this._timeToIndex(this.MaxValue);
      return this._indexToTime(minVal + ratio * (maxVal - minVal));
    } else return this.MinValue + ratio * (this.MaxValue - this.MinValue);
  }

  /**
   * @override
   */
  getUnboundedCoordAt(value) {
    if (value == null) return null;

    var ratio;
    if (this._skipGaps) {
      var minVal = this._timeToIndex(this.MinValue);
      var maxVal = this._timeToIndex(this.MaxValue);
      var val = this._timeToIndex(value);
      ratio = (val - minVal) / (maxVal - minVal);
    } else ratio = (value - this.MinValue) / (this.MaxValue - this.MinValue);

    return this._startCoord + ratio * (this._endCoord - this._startCoord);
  }

  /**
   * @override
   */
  linearToActual(value) {
    if (value == null) return null;
    return this._skipGaps ? this._indexToTime(value) : value;
  }

  /**
   * @override
   */
  actualToLinear(value) {
    if (value == null) return null;
    return this._skipGaps ? this._timeToIndex(value) : value;
  }

  /**
   * Converts time to group index for regular time axis.
   * @param {number} time
   * @return {number} index
   * @private
   */
  _timeToIndex(time) {
    var endIndex = this._groups.length;
    for (var i = 0; i < this._groups.length; i++) {
      if (time <= this._groups[i]) {
        endIndex = i;
        break;
      }
    }
    var startIndex = endIndex - 1;

    var startTime =
      this._groups[startIndex] !== undefined
        ? this._groups[startIndex]
        : this._groups[0] - this._averageInterval;
    var endTime =
      this._groups[endIndex] !== undefined
        ? this._groups[endIndex]
        : this._groups[this._groups.length - 1] + this._averageInterval;

    return startIndex + (time - startTime) / (endTime - startTime);
  }

  /**
   * Converts group index to time for regular time axis.
   * @param {number} index
   * @return {number} time
   * @private
   */
  _indexToTime(index) {
    var endIndex = Math.min(Math.max(Math.ceil(index), 0), this._groups.length);
    var startIndex = endIndex - 1;

    var startTime =
      this._groups[startIndex] !== undefined
        ? this._groups[startIndex]
        : this._groups[0] - this._averageInterval;
    var endTime =
      this._groups[endIndex] !== undefined
        ? this._groups[endIndex]
        : this._groups[this._groups.length - 1] + this._averageInterval;

    return startTime + (index - startIndex) * (endTime - startTime);
  }

  /**
   * @override
   */
  getGroupWidth() {
    if (this._skipGaps)
      return Math.abs(
        this.getUnboundedCoordAt(this._indexToTime(1)) -
          this.getUnboundedCoordAt(this._indexToTime(0))
      );
    else
      return Math.abs(
        this.getUnboundedCoordAt(this.MinValue + this._averageInterval) -
          this.getUnboundedCoordAt(this.MinValue)
      );
  }

  /**
   * @override
   */
  getMinExtent() {
    return this._skipGaps
      ? 1
      : this._mixedFrequency
      ? Math.min((this.getGlobalMax() - this.getGlobalMin()) / 8, this._averageInterval)
      : this._averageInterval;
  }

  /**
   * @override
   */
  getStartOverflow() {
    if (
      (this.Position == 'top' || this.Position == 'bottom') &&
      Agent.isRightToLeft(this.getCtx())
    )
      return this.EndOverflow;
    else return this.StartOverflow;
  }

  /**
   * @override
   */
  getEndOverflow() {
    if (
      (this.Position == 'top' || this.Position == 'bottom') &&
      Agent.isRightToLeft(this.getCtx())
    )
      return this.StartOverflow;
    else return this.EndOverflow;
  }

  /**
   * Returns the am string for this locale if applicable.
   * @param {String} locale the locale for the axis.
   * @return {String} the string representing "am"
   * @private
   */
  static _getAMString(locale) {
    var language = locale.substring(0, 2);
    if (locale == 'en-au' || locale == 'en-ie' || locale == 'en-ph') return 'am';
    else if (locale == 'en-gb') return '';
    switch (language) {
      case 'en':
        return 'AM';
      case 'ar':
        return '\u0635';
      case 'el':
        return '\u03c0\u03bc';
      case 'ko':
        return '\uc624\uc804';
      case 'zh':
        return '\u4e0a\u5348';
      default:
        return '';
    }
  }

  /**
   * Returns the pm string for this locale if applicable.
   * @param {String} locale the locale for the axis.
   * @return {String} the string representing "pm"
   * @private
   */
  static _getPMString(locale) {
    var language = locale.substring(0, 2);
    if (locale == 'en-au' || locale == 'en-ie' || locale == 'en-ph') return 'pm';
    else if (locale == 'en-gb') return '';
    switch (language) {
      case 'en':
        return 'PM';
      case 'ar':
        return '\u0645';
      case 'el':
        return '\u03bc\u03bc';
      case 'ko':
        return '\uc624\ud6c4';
      case 'zh':
        return '\u4e0b\u5348';
      default:
        return '';
    }
  }

  /**
   * Returns whether the AM/PM string should be displayed before or after the time string based on locale.
   * @param {String} locale the locale for the axis
   * @return {boolean} whether the AM/PM string before the time.
   * @private
   */
  static _getAMPMBefore(locale) {
    var language = locale.substring(0, 2);
    if (language == 'ko' || language == 'zh') return true;
    else return false;
  }

  /**
   * Returns the DMY order based on the locale
   * @param {String} locale the locale for the axis
   * @return {String} the order of date, month and year
   * @private
   */
  static _getDMYOrder(locale) {
    var language = locale.substring(0, 2);
    if (locale == 'en-us' || locale == 'en-ph') return 'MDY';
    else if (
      language == 'fa' ||
      language == 'hu' ||
      language == 'ja' ||
      language == 'ko' ||
      language == 'lt' ||
      language == 'mn' ||
      language == 'zh'
    )
      return 'YMD';
    else return 'DMY';
  }

  /**
   * Returns the trailing characters for the year
   * @param {String} locale the locale for the axis
   * @return {String} the year trailing character by locale
   * @private
   */
  static _getYearTrailingCharacters(locale) {
    if (locale.indexOf('ja') == 0 || locale.indexOf('zh') == 0) return '\u5e74';
    else if (locale.indexOf('ko') == 0) return '\ub144';
    else return '';
  }

  /**
   * Returns the trailing characters for the day
   * @param {String} locale the locale for the axis
   * @return {String} the day trailing character by locale
   * @private
   */
  static _getDayTrailingCharacters(locale) {
    if (locale.indexOf('ja') == 0 || locale.indexOf('zh') == 0) return '\u65e5';
    else if (locale.indexOf('ko') == 0) return '\uc77c';
    else return '';
  }

  /**
   * Returns the positions of time axis labels, given the start, end, and step
   * @param {number} start The start time of the axis.
   * @param {number} end The end time of the axis.
   * @param {number} step The increment between labels.
   * @return {array} A list of label positions.
   * @private
   */
  static _getLabelPos(start, end, step) {
    // The time positions has to be at even intervals from the beginning of a year (January 1, 12:00:00 AM), otherwise
    // we may have labels such as [2013, 2014, 2015, ...] that are drawn at [June 8 2013, June 8 2014, June 8 2015, ...],
    // which is data misrepresentation.
    var anchor = new Date(start);
    var initialTimezoneOffset = anchor.getTimezoneOffset();
    anchor.setMonth(0, 1); // January 1
    anchor.setHours(0, 0, 0, 0); // 00:00:00
    var time = anchor.getTime();

    var times = [];
    if (step >= DvtTimeAxisInfo.TIME_YEAR_MIN && step <= DvtTimeAxisInfo.TIME_YEAR_MAX) {
      // Assume that the step is one year, which can mean different # of days depending on the year
      while (time < start) time = DvtTimeAxisInfo._addOneYear(time);
      while (time <= end) {
        times.push(time);
        time = DvtTimeAxisInfo._addOneYear(time);
      }
    } else if (step >= DvtTimeAxisInfo.TIME_MONTH_MIN && step <= DvtTimeAxisInfo.TIME_MONTH_MAX) {
      // Assume that the step is one month, which can mean different # of days depending on the month
      while (time < start) time = DvtTimeAxisInfo._addOneMonth(time);
      while (time <= end) {
        times.push(time);
        time = DvtTimeAxisInfo._addOneMonth(time);
      }
    } else {
      // . Correction is needed due to daylight savings.
      // Only apply daylight correction when step is less than a month. Daylight savings does not impact any step higher than month.
      // JET-52348 - Ideally we should be using Date api to add and substract date offsets to calculate labels since
      // Date will automatically handle daylight savings. This approach should solve issues for steps greater than month but we could encounter
      // offsets in axis labels when dataset values are in different daylight savings and chart step is low (eg day or hour);
      var timezoneCorrection = (initialTimezoneOffset - anchor.getTimezoneOffset()) * 60 * 1000;
      var correction = step < DvtTimeAxisInfo.TIME_MONTH_MIN ? timezoneCorrection : 0;
      time += Math.ceil((start - time - correction) / step) * step + correction;
      while (time <= end) {
        times.push(time);
        time += step;
      }
    }
    return times;
  }

  /**
   * Adds the time by one year, e.g. 2014 January 15 -> 2015 January 15 -> ...
   * @param {number} time The current time
   * @return {number} Next year
   * @private
   */
  static _addOneYear(time) {
    var date = new Date(time);
    date.setFullYear(date.getFullYear() + 1);
    return date.getTime();
  }

  /**
   * Adds the time by one month, e.g. January 15 -> February 15 -> March 15 -> ...
   * @param {number} time The current time
   * @return {number} Next month
   * @private
   */
  static _addOneMonth(time) {
    var date = new Date(time);
    date.setMonth(date.getMonth() + 1);
    return date.getTime();
  }

  /**
   * Calulates the GCD of all the intervals in the given array
   * @param {Array} intervals The array of intervals in the time axis values
   * @return {Number} The GCD of the intervals
   */
  static _getGCDInterval(intervals) {
    var gcd = (interval1, interval2) => {
      if (interval1 === 0) return interval2;
      return gcd(interval2 % interval1, interval1);
    };

    var result = intervals[0];
    for (var i = 1; i < intervals.length; i++) {
      result = gcd(result, intervals[i]);
      if (result === 1) return 1;
    }

    return result;
  }

  /**
   * Calculates interval between two times based on the current granularity level
   * @param {Date} previousValue
   * @param {Date} currentValue
   * @param {Number} granularity The granularity level
   * @return {Number} The interval in the current granular level
   *                  (Eg. if the interval is 1 month and graularity level is month, then returns 1)
   * @private
   * @static
   */
  static _calculateGranularInterval(previousDate, currentDate, granularity) {
    // Treat times for the day light savings.
    var prevTimezoneOffset = previousDate.getTimezoneOffset();
    var currentTimezoneOffset = currentDate.getTimezoneOffset();
    var currentTime = currentDate.getTime();
    var currentTimezoneOffsetCorrection = (prevTimezoneOffset - currentTimezoneOffset) * 1000 * 60;
    if (currentTimezoneOffsetCorrection > 0) {
      currentDate.setTime(currentTime + currentTimezoneOffsetCorrection);
    }

    var granularInterval;
    if (granularity === DvtTimeAxisInfo.TIME_YEAR_MIN)
      granularInterval = currentDate.getFullYear() - previousDate.getFullYear();
    else if (granularity === DvtTimeAxisInfo.TIME_MONTH_MIN)
      granularInterval =
        12 * (currentDate.getFullYear() - previousDate.getFullYear()) +
        (currentDate.getMonth() - previousDate.getMonth());
    else
      granularInterval = Math.round((currentDate.getTime() - previousDate.getTime()) / granularity);

    if (currentTimezoneOffsetCorrection > 0) {
      currentDate.setTime(currentTime); // reset to original value
    }

    return granularInterval;
  }

  /**
   * Checks if any of the times array has any missing value and fills in the gap.
   * @param {Array} times An array of times in milliseconds representing the axis values.
   * @param {Number} granularity The granularity level
   * @return {Array} The treated times array
   * @private
   * @static
   */
  static _treatMissingValues(times, granularity) {
    var initialInterval;
    var intervals = new Set();
    var hasMissingValues = false;
    var ret = times; // If there are no missing values, the original array will be returned.
    var intervalsCache = [];

    var timeLength = times.length;
    var i;
    var previousDate = new Date(times[0]);
    for (i = 1; i < timeLength; i++) {
      // Calculate the current interval.
      var currentDate = new Date(times[i]);
      var currentInterval = DvtTimeAxisInfo._calculateGranularInterval(
        previousDate,
        currentDate,
        granularity
      );
      previousDate = currentDate;

      // Cache the result so as to not calculate again while filling the missing values
      // Since the cache is only used for filling in the missing values, and the operation
      // is performed in the same order as this one, having the cache key in the format
      // '1546318800000-1514782800000' should be fine.
      intervalsCache.push(currentInterval);

      // Proceed only if the current interval is greater than the current
      // granularity level.
      if (currentInterval <= 0) {
        continue;
      }

      // Add the current interval to the set.
      // The GCD of all the intervals will be used to fill in the missing the
      // values.
      intervals.add(currentInterval);

      // During the initial iteration, we will not have any information on intervals,
      // so, store the current interval and continue.
      if (!initialInterval) {
        initialInterval = currentInterval;
        continue;
      }

      // Check if the current interval is regular.
      if (currentInterval !== initialInterval) {
        hasMissingValues = true;
      }
    }

    // If missing values are present, treat them.
    if (hasMissingValues) {
      // Get the GCD of intervals and it will be the minimum interval in the new set of times
      var minimumInterval = DvtTimeAxisInfo._getGCDInterval([...intervals]);
      ret = [];
      ret.push(times[0]);
      for (i = 1; i < timeLength; i++) {
        var currentValue = times[i];
        var previousValue = times[i - 1];

        // Retrieve the interval from cache.
        // As we are looping through the same array the cache will have the interval value
        // and will never be undefined.
        var currentInterval = intervalsCache[i - 1];

        // If no values are missing in this interval,
        // add current value to the return array and continue.
        // Note: Values are considered missing only when the interval is greater than the current granularity level (minimumInterval).
        // Example: Jan 1, Jan 15, Feb 1, Mar 1, May 1, Jun 1
        // Granularity = Month; Minimum Interval = 1 (1 month)
        // Mar 1, May 1 => has one missing value: Apr
        // Jan 1, Jan 15 => has no missing value
        if (currentInterval <= minimumInterval) {
          ret.push(currentValue);
          continue;
        }

        // Calculate the interval at which the values are to be filled
        var ratioOfCurrentIntervalToMinimumInterval = currentInterval / minimumInterval; // Should be a round number as minimumInterval is a divisor of currentInterval
        var numMissingValues = ratioOfCurrentIntervalToMinimumInterval - 1;
        var fillIntervalInMilliseconds = Math.round(
          (currentValue - previousValue) / ratioOfCurrentIntervalToMinimumInterval
        );

        // Fill in the missing values
        var j;
        for (j = 1; j <= numMissingValues; j++) {
          previousValue += fillIntervalInMilliseconds;
          ret.push(previousValue);
        }
        ret.push(currentValue); // Finally, add the current value
        previousValue = currentValue; // Update the previous value
      }
    }
    return ret;
  }

  /**
   * Determines if rectangle A (bounded by pointA1 and pointA2) and rectangle B (bounded by pointB1 and B2) overlap.
   * All the points should lie in one dimension.
   * @param {Number} pointA1
   * @param {Number} pointA2
   * @param {Number} pointB1
   * @param {Number} pointB2
   * @param {Number} gap The minimum gap between the two rectangles
   * @return {Boolean} whether rectangle A and B overlap
   * @private
   */
  static _isOverlapping(pointA1, pointA2, pointB1, pointB2, gap) {
    if (pointB1 >= pointA1 && pointB1 - gap < pointA2) return true;
    else if (pointB1 < pointA1 && pointB2 + gap > pointA1) return true;
    return false;
  }
}

// ------------------------
// Constants
//
/** @const */
DvtTimeAxisInfo.TIME_SECOND = 1000;
/** @const */
DvtTimeAxisInfo.TIME_MINUTE = 60 * DvtTimeAxisInfo.TIME_SECOND;
/** @const */
DvtTimeAxisInfo.TIME_HOUR = 60 * DvtTimeAxisInfo.TIME_MINUTE;
/** @const */
DvtTimeAxisInfo.TIME_DAY = 24 * DvtTimeAxisInfo.TIME_HOUR;
/** @const */
DvtTimeAxisInfo.TIME_MONTH_MIN = 28 * DvtTimeAxisInfo.TIME_DAY;
/** @const */
DvtTimeAxisInfo.TIME_MONTH_MAX = 31 * DvtTimeAxisInfo.TIME_DAY;
/** @const */
DvtTimeAxisInfo.TIME_YEAR_MIN = 365 * DvtTimeAxisInfo.TIME_DAY;
/** @const */
DvtTimeAxisInfo.TIME_YEAR_MAX = 366 * DvtTimeAxisInfo.TIME_DAY;

DvtAxisInfo.registerConstructor('time', DvtTimeAxisInfo);

/**
 * Renderer for the reference objects of a Chart.
 * @class
 */
const DvtChartRefObjRenderer = {
  /**
   * Renders the background reference objects.
   * @param {Chart} chart The chart being rendered.
   * @param {dvt.Container} container The container to render to.
   * @param {dvt.Rectangle} plotAreaBounds The bounds of the plot area.
   */
  renderBackgroundObjects: (chart, container, plotAreaBounds) => {
    DvtChartRefObjRenderer._renderObjects(chart, container, plotAreaBounds, 'back');
  },

  /**
   * Renders the foreground reference objects.
   * @param {Chart} chart The chart being rendered.
   * @param {dvt.Container} container The container to render to.
   * @param {dvt.Rectangle} plotAreaBounds The bounds of the plot area.
   */
  renderForegroundObjects: (chart, container, plotAreaBounds) => {
    DvtChartRefObjRenderer._renderObjects(chart, container, plotAreaBounds, 'front');
  },

  /**
   * Renders the reference objects for the given location.
   * @param {Chart} chart The chart being rendered.
   * @param {dvt.Container} container The container to render to.
   * @param {dvt.Rectangle} plotAreaBounds The bounds of the plot area.
   * @param {string} location The location of the reference objects.
   * @private
   */
  _renderObjects: (chart, container, plotAreaBounds, location) => {
    DvtChartRefObjRenderer._renderObjectsForAxis(
      chart,
      container,
      plotAreaBounds,
      location,
      chart.xAxis,
      DvtChartDataUtils.getAxisRefObjs(chart, 'x')
    );
    DvtChartRefObjRenderer._renderObjectsForAxis(
      chart,
      container,
      plotAreaBounds,
      location,
      chart.yAxis,
      DvtChartDataUtils.getAxisRefObjs(chart, 'y')
    );
    DvtChartRefObjRenderer._renderObjectsForAxis(
      chart,
      container,
      plotAreaBounds,
      location,
      chart.y2Axis,
      DvtChartDataUtils.getAxisRefObjs(chart, 'y2')
    );
  },

  /**
   * Renders the reference objects for the given location.
   * @param {Chart} chart The chart being rendered.
   * @param {dvt.Container} container The container to render to.
   * @param {dvt.Rectangle} plotAreaBounds The bounds of the plot area.
   * @param {string} location The location of the reference objects.
   * @param {DvtAxis} axis The axis corresponding to the reference objects.
   * @param {array} objects The array of reference objects.
   * @private
   */
  _renderObjectsForAxis: (chart, container, plotAreaBounds, location, axis, objects) => {
    if (!objects || !axis) return;

    // Loop through and render each reference object
    for (var i = 0; i < objects.length; i++) {
      var refObj = objects[i];

      if (!DvtChartRefObjUtils.isObjRendered(chart, refObj)) {
        continue;
      }

      if (!refObj) continue;

      if (DvtChartRefObjUtils.getLocation(refObj) != location) continue;

      var shape, outerShape;
      var type = DvtChartRefObjUtils.getType(refObj);
      if (type == 'area')
        shape = DvtChartRefObjRenderer._createReferenceArea(refObj, chart, plotAreaBounds, axis);
      else if (type == 'line') {
        shape = DvtChartRefObjRenderer._createReferenceLine(refObj, chart, plotAreaBounds, axis);
        outerShape = DvtChartRefObjRenderer._createReferenceLine(
          refObj,
          chart,
          plotAreaBounds,
          axis,
          true
        );
      }

      if (shape == null) continue;

      //  - HIDE & SHOW FOR REFERENCE OBJECTS
      // Associate for interactivity and tooltip support
      var yAxisType = axis == chart.yAxis ? 'yAxis' : 'y2Axis';
      var axisType = axis == chart.xAxis ? 'xAxis' : yAxisType;
      var refObjShapes = outerShape ? [shape, outerShape] : [shape];
      var refObjPeer = new DvtChartRefObjPeer(chart, refObjShapes, refObj, i, axisType);
      chart.registerObject(refObjPeer);
      chart.getEventManager().associate(shape, refObjPeer);

      // Add the shape to the container
      container.addChild(shape);

      if (outerShape) container.addChild(outerShape);
    }
  },

  /**
   * Creates a reference area.
   * @param {object} refObj The options object for the reference area.
   * @param {Chart} chart The chart being rendered.
   * @param {dvt.Rectangle} plotAreaBounds The bounds of the plot area.
   * @param {DvtAxis} axis The axis corresponding to the reference area.
   * @return {dvt.Shape} The reference area.
   * @private
   */
  _createReferenceArea: (refObj, chart, plotAreaBounds, axis) => {
    var context = chart.getCtx();
    var position = axis.getPosition();
    var bHoriz = position == 'top' || position == 'bottom';
    var bRadial = position == 'radial';
    var color = DvtChartRefObjUtils.getColor(chart, refObj);
    var lineType = DvtChartRefObjUtils.getLineType(refObj);
    var style = refObj['style'] || refObj['svgStyle'];
    var className = refObj['className'] || refObj['svgClassName'];
    var shape;

    if (refObj['items'] != null && (axis == chart.yAxis || axis == chart.y2Axis)) {
      // REF AREA WITH MULTIPLE VALUES
      var items = refObj['items'];
      var highCoords = [];
      var lowCoords = [];

      // Match the number of items with the group count
      if (chart.xAxis.isGroupAxis()) {
        while (items.length < DvtChartDataUtils.getGroupCount(chart)) {
          items.push(null);
        }
      }

      // Build arrays of high/low axis coords
      for (var pointIndex = 0; pointIndex < items.length; pointIndex++) {
        var dataItem = items[pointIndex];
        var lVal = DvtChartRefObjUtils.getLowVal(dataItem);
        var hVal = DvtChartRefObjUtils.getHighVal(dataItem);
        if (lVal == null || hVal == null) {
          highCoords.push(new DvtChartCoord());
          lowCoords.push(new DvtChartCoord());
          continue;
        }

        // x is always the xCoord, and min and max along the yAxis
        var lCoord = axis.getUnboundedCoordAt(lVal);
        var hCoord = axis.getUnboundedCoordAt(hVal);
        var xCoord = chart.xAxis.getUnboundedCoordAt(
          DvtChartRefObjUtils.getXVal(chart, items, pointIndex)
        );

        highCoords.push(new DvtChartCoord(xCoord, hCoord, hCoord));
        lowCoords.push(new DvtChartCoord(xCoord, lCoord, lCoord));
      }

      // Create the area shapes
      shape = new DvtChartLineArea(
        chart,
        true,
        plotAreaBounds,
        null,
        style,
        className,
        new SolidFill(color),
        null,
        lineType,
        highCoords,
        lineType,
        lowCoords
      );
    } else {
      // REF AREA WITH SINGLE VALUE
      var lowVal = DvtChartRefObjUtils.getLowVal(refObj);
      var highVal = DvtChartRefObjUtils.getHighVal(refObj);

      // Populate the default value if either low or high is missing or infinite
      if (lowVal == null || lowVal == -Infinity) lowVal = axis.getInfo().getGlobalMin();
      if (highVal == null || highVal == Infinity) highVal = axis.getInfo().getGlobalMax();

      // Find the coordinates
      var lowCoord = DvtChartRefObjRenderer._getAxisCoord(chart, axis, lowVal);
      var highCoord = DvtChartRefObjRenderer._getAxisCoord(chart, axis, highVal);

      if (DvtChartTypeUtils.isPolar(chart)) {
        var cmds;
        var cx = plotAreaBounds.x + plotAreaBounds.w / 2;
        var cy = plotAreaBounds.y + plotAreaBounds.h / 2;
        if (bRadial) {
          if (DvtChartAxisUtils.isGridPolygonal(chart)) {
            // draw polygonal donut
            var nSides = DvtChartDataUtils.getGroupCount(chart);
            var outerPoints = PolygonUtils.getRegularPolygonPoints(
              cx,
              cy,
              nSides,
              highCoord,
              0,
              1
            );
            var innerPoints = PolygonUtils.getRegularPolygonPoints(
              cx,
              cy,
              nSides,
              lowCoord,
              0,
              0
            );
            cmds =
              PathUtils.polyline(outerPoints) +
              PathUtils.polyline(innerPoints) +
              PathUtils.closePath();
          } else {
            // draw circular donut
            // To work around a chrome/safari bug, we draw two segments around each of the outer and inner arcs
            cmds =
              PathUtils.moveTo(cx, cy - highCoord) +
              PathUtils.arcTo(highCoord, highCoord, Math.PI, 1, cx, cy + highCoord) +
              PathUtils.arcTo(highCoord, highCoord, Math.PI, 1, cx, cy - highCoord) +
              PathUtils.moveTo(cx, cy - lowCoord) +
              PathUtils.arcTo(lowCoord, lowCoord, Math.PI, 0, cx, cy + lowCoord) +
              PathUtils.arcTo(lowCoord, lowCoord, Math.PI, 0, cx, cy - lowCoord) +
              PathUtils.closePath();
          }
        } else {
          // for tangential axis, draw circular segment. If polygonal, the shape will be clipped by the container.
          var radius = chart.getRadius();
          var pLow = DvtChartCoordUtils.polarToCartesian(radius, lowCoord, plotAreaBounds);
          var pHigh = DvtChartCoordUtils.polarToCartesian(radius, highCoord, plotAreaBounds);
          cmds =
            PathUtils.moveTo(cx, cy) +
            PathUtils.lineTo(pLow.x, pLow.y) +
            PathUtils.arcTo(
              radius,
              radius,
              highCoord - lowCoord,
              Agent.isRightToLeft(context) ? 0 : 1,
              pHigh.x,
              pHigh.y
            ) +
            PathUtils.lineTo(pHigh.x, pHigh.y) +
            PathUtils.closePath();
        }
        shape = new Path(context, cmds);
      } else {
        // draw rectangle
        var points;
        if (bHoriz)
          points = [
            lowCoord,
            0,
            highCoord,
            0,
            highCoord,
            plotAreaBounds.h,
            lowCoord,
            plotAreaBounds.h
          ];
        else
          points = [
            0,
            lowCoord,
            0,
            highCoord,
            plotAreaBounds.w,
            highCoord,
            plotAreaBounds.w,
            lowCoord
          ];
        shape = new Polygon(context, points);
      }
      shape.setSolidFill(color);
      shape.setStyle(style).setClassName(className);
    }

    return shape;
  },

  /**
   * Creates a reference line.
   * @param {object} refObj The options object for the reference line.
   * @param {Chart} chart The chart being rendered.
   * @param {dvt.Rectangle} plotAreaBounds The bounds of the plot area.
   * @param {DvtAxis} axis The axis corresponding to the reference line.
   * @param {boolean} isOuterLine True if this is the border line for the reference line.
   * @return {dvt.Shape} The reference line.
   * @private
   */
  _createReferenceLine: (refObj, chart, plotAreaBounds, axis, isOuterLine) => {
    var position = axis.getPosition();
    var bHoriz = position == 'top' || position == 'bottom';
    var bRadial = position == 'radial';
    var bTangential = position == 'tangential';

    // Set style attributes
    var lineWidth = DvtChartRefObjUtils.getLineWidth(chart, refObj);
    var lineType = DvtChartRefObjUtils.getLineType(refObj);
    var color = DvtChartRefObjUtils.getColor(chart, refObj);
    var style = refObj['style'] || refObj['svgStyle'];
    var className = isOuterLine
      ? 'oj-chart-reference-object-inner-line'
      : refObj['className'] || refObj['svgClassName'];
    var stroke = new Stroke(
      color,
      1,
      lineWidth,
      false,
      Stroke.getDefaultDashProps(refObj['lineStyle'], lineWidth)
    );

    var context = chart.getCtx();
    var shape;

    if (refObj['items'] != null && (axis == chart.yAxis || axis == chart.y2Axis)) {
      // REF LINE WITH MULTIPLE VALUES
      var items = refObj['items'];
      // Match the number of items with the group count
      if (chart.xAxis.isGroupAxis()) {
        while (items.length < DvtChartDataUtils.getGroupCount(chart)) {
          items.push(null);
        }
      }

      // Build array of axis coords
      var coords = [];
      for (var pointIndex = 0; pointIndex < items.length; pointIndex++) {
        var dataItem = items[pointIndex];

        // Extract the value for the dataItem
        var value = null;
        if (dataItem != null) {
          if (typeof dataItem != 'object') value = dataItem;
          else if (dataItem['value'] != null) value = dataItem['value'];
        }

        if (value == null) {
          coords.push(new DvtChartCoord());
          continue;
        }

        // y is always along the xAxis, and value for yAxis
        var yCoord = axis.getUnboundedCoordAt(value) - (isOuterLine ? lineWidth / 2 : 0);
        var xCoord = chart.xAxis.getUnboundedCoordAt(
          DvtChartRefObjUtils.getXVal(chart, items, pointIndex)
        );
        coords.push(new DvtChartCoord(xCoord, yCoord, yCoord));
      }

      // Create line shapes
      shape = new DvtChartLineArea(
        chart,
        false,
        plotAreaBounds,
        null,
        style,
        className,
        null,
        stroke,
        lineType,
        coords
      );
    } else if (refObj['value'] != null) {
      // REF LINE WITH SINGLE VALUE
      var lineCoord = DvtChartRefObjRenderer._getAxisCoord(chart, axis, refObj['value']);

      // Don't continue if the line is outside of the axis
      if (lineCoord == null || lineCoord == Infinity || lineCoord == -Infinity) return null;

      if (isOuterLine) lineCoord -= Math.ceil(lineWidth / 2);

      var cx = plotAreaBounds.x + plotAreaBounds.w / 2;
      var cy = plotAreaBounds.y + plotAreaBounds.h / 2;
      if (bRadial) {
        if (DvtChartAxisUtils.isGridPolygonal(chart)) {
          var points = PolygonUtils.getRegularPolygonPoints(
            cx,
            cy,
            DvtChartDataUtils.getGroupCount(chart),
            lineCoord,
            0
          );
          shape = new Polygon(context, points);
        } else shape = new Circle(context, cx, cy, lineCoord);
        shape.setFill(null);
      } else if (bTangential) {
        var cartesian = DvtChartCoordUtils.polarToCartesian(
          chart.getRadius(),
          lineCoord,
          plotAreaBounds
        );
        shape = new Line(context, cx, cy, cartesian.x, cartesian.y);
      } else {
        if (bHoriz) shape = new Line(context, lineCoord, 0, lineCoord, plotAreaBounds.h);
        else shape = new Line(context, 0, lineCoord, plotAreaBounds.w, lineCoord);
        shape.setPixelHinting(true);
      }
      shape.setStroke(stroke);
      shape.setStyle(style).setClassName(className);
    } // no line created
    else return null;

    return shape;
  },

  /**
   * Returns the coordinate of the specified value on the axis.  If the coordinate cannot be located, then returns null.
   * @param {Chart} chart
   * @param {DvtAxis} axis The axis corresponding to the reference object.
   * @param {object} value The value whose coordinate will be returned.
   * @return {number}
   * @private
   */
  _getAxisCoord: (chart, axis, value) => {
    if (axis.isGroupAxis()) {
      // For group axis, find the index of the group and pass it to the axis
      var index = DvtChartDataUtils.getGroupIdx(chart, value);
      if (index >= 0) return axis.getUnboundedCoordAt(index);
    }

    // If value is number, treat is as the group index for group axis
    if (!isNaN(value)) return axis.getUnboundedCoordAt(value);

    return null;
  }
};

/**
 * Bubble chart utility functions for Chart.
 * @class
 */
const DvtChartMarkerUtils = {
  /** @private */
  _MIN_BUBBLE_SIZE: 6,

  /** @private */
  _MAX_BUBBLE_SIZE_RATIO: 0.5,

  /**
   * Calculates the bubble sizes for the chart.
   * @param {Chart} chart
   * @param {dvt.Rectangle} availSpace
   */
  calcBubbleSizes: (chart, availSpace) => {
    // Calculate the min and max z values
    var minMax = DvtChartAxisUtils.getMinMaxVal(chart, 'z');
    var minZ = minMax['min'];
    var maxZ = minMax['max'];

    // Calculate the max allowed bubble sizes
    var minSize = DvtChartMarkerUtils._MIN_BUBBLE_SIZE;
    var maxSize = DvtChartMarkerUtils._MAX_BUBBLE_SIZE_RATIO * Math.min(availSpace.w, availSpace.h);

    // Loop through the data and update the sizes
    var seriesCount = DvtChartDataUtils.getSeriesCount(chart);
    var bIncludeHiddenSeries =
      DvtChartBehaviorUtils.getHideAndShowBehavior(chart) == 'withoutRescale';

    var optionsCache = chart.getOptionsCache();

    for (var seriesIndex = 0; seriesIndex < seriesCount; seriesIndex++) {
      if (!bIncludeHiddenSeries && !DvtChartDataUtils.isSeriesRendered(chart, seriesIndex))
        continue;

      var seriesItem = DvtChartDataUtils.getSeriesItem(chart, seriesIndex);
      var numGroups = seriesItem['items'] ? seriesItem['items'].length : 0;
      for (var j = 0; j < numGroups; j++) {
        var dataItem = seriesItem['items'][j];

        // If a z value exists, calculate and set the marker size
        if (dataItem) {
          var markerSize = LayoutUtils.getBubbleSize(
            dataItem['z'],
            minZ,
            maxZ,
            minSize,
            maxSize
          );
          optionsCache.putToCachedMap2D('bubbleSizeCache', seriesIndex, j, markerSize);
        }
      }
    }

    // The rest of the code is to determine how much the axis needs to be extended to cover the bubble
    // radii. _x/yAxisBubbleRatio will be used in DvtChartAxisUtils.getBubbleAxisRadius().
    // NOTE: The computed values here are approximations, so they can be grossly inaccurate in some
    //       circumstances. We can't get the exact values at this point since the axes are not rendered yet, but
    //       we need to approximate now in order to layout the axes correctly.
    var axisWidth, axisHeight;
    if (DvtChartTypeUtils.isPolar(chart)) {
      axisWidth = Infinity; // polar x-axis is circular
      axisHeight = chart.getRadius();
    } else {
      // At this point, we still don't know the actual dimensions of the axes since they're not rendered yet, so we
      // approximate based on the tick label height
      axisWidth = availSpace.w - 2.4 * DvtChartAxisUtils.getTickLabelHeight(chart, 'y');
      axisHeight = availSpace.h - 1.6 * DvtChartAxisUtils.getTickLabelHeight(chart, 'x');
    }

    // Subtract the axis width and height based on the largest bubble size. This is to approximate the extra axis
    // space that is allocated to accommodate the bubbles near the plot area borders.
    axisWidth -= 0.5 * maxSize;
    axisHeight -= 0.5 * maxSize;

    // Store the computed ratios in the options cache
    var cache = chart.getCache();
    var xAxisValueRange = DvtChartMarkerUtils._getAxisValueRange(chart, 'x');
    cache.putToCache('_xAxisBubbleRatio', xAxisValueRange / axisWidth);

    var yAxisValueRange = DvtChartMarkerUtils._getAxisValueRange(chart, 'y');
    cache.putToCache('_yAxisBubbleRatio', yAxisValueRange / axisHeight);
  },

  /**
   * Returns the axis value range, disregarding the bubble radii (which we don't know yet at this point)
   * @param {Chart} chart
   * @param {string} type The axis type, 'x' or 'y'.
   * @return {number} Axis value range.
   * @private
   */
  _getAxisValueRange: (chart, type) => {
    var axisOptions = chart.getOptions()[type + 'Axis'];
    var isLog = DvtChartAxisUtils.isLog(chart, type);
    var zeroBaseline = !isLog && DvtChartAxisUtils.getBaselineScaling(chart, type) == 'zero';
    var minMax = DvtChartAxisUtils.getMinMaxVal(chart, type, true);

    var min = axisOptions['min'];
    if (min == null) min = zeroBaseline ? Math.min(0, minMax['min']) : minMax['min'];

    var max = axisOptions['max'];
    if (max == null) max = zeroBaseline ? Math.max(0, minMax['max']) : minMax['max'];

    if (isLog && max > 0 && min > 0) return max == min ? 6 : Math$1.log10(max / min);
    return max == min ? 60 : max - min;
  },

  /**
   * Sorts the markers in order of descending marker size.
   * @param {array} markers The array of dvt.SimpleMarker objects or objects with size properties.
   */
  sortMarkers: (markers) => {
    markers.sort(DvtChartMarkerUtils._compareSize);
  },

  /**
   * Sorts the markers in order of descending marker size.
   * @param {array} markers The array of markerInfo objects with size properties.
   */
  sortMarkerInfos: (markers) => {
    markers.sort(DvtChartMarkerUtils._compareInfoSize);
  },

  /**
   * Compare function to sort markers in order of descending size.
   * @param {dvt.SimpleMarker | object} a
   * @param {dvt.SimpleMarker | object} b
   * @return {number} Comparison.
   * @private
   */
  _compareSize: (a, b) => {
    return b.getSize() - a.getSize();
  },

  /**
   * Compare function to sort markerInfo objects in order of descending size.
   * @param {object} a
   * @param {object} b
   * @return {number} Comparison.
   * @private
   */
  _compareInfoSize: (a, b) => {
    // We want to sort the markers from biggest to smallest
    return b.size - a.size;
  },

  /**
   * Returns true if the specified marker would be fully obscured.
   * @param {dvt.PixelMap} pixelMap
   * @param {number} markerX
   * @param {number} markerY
   * @param {number} markerSize
   * @return {boolean}
   */
  checkPixelMap: (pixelMap, markerX, markerY, markerSize) => {
    // Note: This function is conservative about the pixels occupied.
    var halfSize = markerSize / 2;
    var x1 = Math.max(Math.floor(markerX - halfSize), 0);
    var y1 = Math.max(Math.floor(markerY - halfSize), 0);
    var x2 = Math.max(Math.ceil(markerX + halfSize), 0);
    var y2 = Math.max(Math.ceil(markerY + halfSize), 0);
    return pixelMap.isObscured(x1, y1, x2, y2);
  },

  /**
   * Updates the pixel map for the specified marker.
   * @param {dvt.Map2D} pixelMap
   * @param {number} markerX
   * @param {number} markerY
   * @param {number} markerSize
   * @param {number} alpha
   */
  updatePixelMap: (pixelMap, markerX, markerY, markerSize, alpha) => {
    // Note: This function uses several approximations. Only 80% of the marker size is counted as occupied, partly to
    // account for marker shape. The coords are rounded, since the browser will likely anti-alias in the direction of
    // rounding.
    var halfSize = markerSize * 0.4;
    var x1 = Math.max(Math.round(markerX - halfSize), 0);
    var x2 = Math.max(Math.round(markerX + halfSize), 0);
    var y1 = Math.max(Math.round(markerY - halfSize), 0);
    var y2 = Math.max(Math.round(markerY + halfSize), 0);
    pixelMap.obscure(x1, y1, x2, y2, alpha);
  }
};

/**
 * Renderer for the plot area of a Chart.
 * @class
 */
const DvtChartPlotAreaRenderer = {
  /** @private @const */
  _MIN_TOUCH_MARKER_SIZE: 16, // minimum marker size for touch device,

  /** @private @const */
  _MIN_CHARS_DATA_LABEL: 3, // minimum number of chars to be displayed of a data label when truncatin,

  /**
   * The minimum number of data points after which data filtering will be enabled for scatter and bubble charts.
   * @const
   */
  FILTER_THRESHOLD_SCATTER_BUBBLE: 10000,

  /**
   * Renders the plot area into the available space.
   * @param {Chart} chart The chart being rendered.
   * @param {dvt.Container} container The container to render to.
   * @param {dvt.Rectangle} availSpace The available space.
   */
  render: (chart, container, availSpace) => {
    if (chart.getOptions()['plotArea']['rendered'] == 'off') {
      DvtChartPlotAreaRenderer._renderAxisLines(chart, container, availSpace);
    } else if (availSpace.w > 0 && availSpace.h > 0) {
      // TODO: change to formal location for displayed data
      chart._currentMarkers = [];
      chart._currentAreas = [];
      DvtChartPlotAreaRenderer._renderBackgroundObjs(chart, container, availSpace);
      DvtChartPlotAreaRenderer._renderTicks(chart, container, availSpace);
      DvtChartPlotAreaRenderer._renderForegroundObjs(chart, container, availSpace);
    }
  },

  /**
   * Renders objects in the background of the plot area.
   * @param {Chart} chart The chart being rendered.
   * @param {dvt.Container} container The container to render to.
   * @param {dvt.Rectangle} availSpace The available space.
   * @private
   */
  _renderBackgroundObjs: (chart, container, availSpace) => {
    // Chart background
    var options = chart.getOptions();
    var background = DvtChartPlotAreaRenderer._getBackgroundShape(chart, availSpace);

    var backgroundColor = DvtChartStyleUtils.getBackgroundColor(chart);
    if (backgroundColor) background.setSolidFill(backgroundColor);
    else background.setInvisibleFill(); // Always render a background plot area rectangle and save for interactivity

    container.addChild(background);
    chart.getCache().putToCache('plotAreaBackground', background);

    var isPlotAreaDraggable = DvtChartBehaviorUtils.isPlotAreaDraggable(chart);
    if (isPlotAreaDraggable) background.setClassName('oj-draggable');

    // Reference Objects
    if (
      options['xAxis']['referenceObjects'] ||
      options['yAxis']['referenceObjects'] ||
      options['y2Axis']['referenceObjects']
    ) {
      var clipGroup = DvtChartPlotAreaRenderer.createClippedGroup(chart, container, availSpace);
      DvtChartRefObjRenderer.renderBackgroundObjects(chart, clipGroup, availSpace);
    }

    // Draw area series behind the gridlines (because they would obscure the grids)
    if (DvtChartTypeUtils.isBLAC(chart)) {
      // Create area container for all BLAC types to allow delete animation from a chart with area to a chart without area
      var areaContainer = new Container(chart.getCtx());
      container.addChild(areaContainer);
      chart.__setAreaContainer(areaContainer);
      const comboSeriesOrder = options['comboSeriesOrder'];

      if (
        DvtChartDataUtils.hasAreaSeries(chart) &&
        (comboSeriesOrder === 'seriesType' || !DvtChartTypeUtils.isCombo(chart))
      )
        DvtChartPlotAreaRenderer._renderAreas(chart, areaContainer, availSpace, false);
    }
  },

  /**
   * Helper function to create the background shape.
   * @param {Chart} chart The chart being rendered.
   * @param {dvt.Rectangle} availSpace The available space.
   * @return {dvt.Shape} The background shape
   * @private
   */
  _getBackgroundShape: (chart, availSpace) => {
    var background;
    var context = chart.getCtx();
    if (DvtChartTypeUtils.isPolar(chart)) {
      var cx = availSpace.x + availSpace.w / 2;
      var cy = availSpace.y + availSpace.h / 2;
      if (DvtChartAxisUtils.isGridPolygonal(chart)) {
        var points = PolygonUtils.getRegularPolygonPoints(
          cx,
          cy,
          DvtChartDataUtils.getGroupCount(chart),
          chart.getRadius(),
          0
        );
        background = new Polygon(context, points);
      } else background = new Circle(context, cx, cy, chart.getRadius());
    } else
      background = new Rect(context, availSpace.x, availSpace.y, availSpace.w, availSpace.h);
    return background;
  },

  /**
   * Renders the major and minor ticks for the chart.
   * @param {Chart} chart The chart being rendered.
   * @param {dvt.Container} container The container to render to.
   * @param {dvt.Rectangle} availSpace The available space.
   * @private
   */
  _renderTicks: (chart, container, availSpace) => {
    // Minor Ticks
    if (chart.xAxis && DvtChartAxisUtils.isMinorTickRendered(chart, 'x'))
      DvtChartPlotAreaRenderer._renderMinorTicks(chart, container, chart.xAxis, availSpace);

    if (chart.yAxis && DvtChartAxisUtils.isMinorTickRendered(chart, 'y'))
      DvtChartPlotAreaRenderer._renderMinorTicks(chart, container, chart.yAxis, availSpace);

    if (chart.y2Axis && DvtChartAxisUtils.isMinorTickRendered(chart, 'y2'))
      DvtChartPlotAreaRenderer._renderMinorTicks(chart, container, chart.y2Axis, availSpace);

    // Major Ticks
    if (chart.xAxis && DvtChartAxisUtils.isMajorTickRendered(chart, 'x'))
      DvtChartPlotAreaRenderer._renderMajorTicks(chart, container, chart.xAxis, availSpace);

    if (chart.yAxis && DvtChartAxisUtils.isMajorTickRendered(chart, 'y'))
      DvtChartPlotAreaRenderer._renderMajorTicks(chart, container, chart.yAxis, availSpace);

    if (chart.y2Axis && DvtChartAxisUtils.isMajorTickRendered(chart, 'y2'))
      DvtChartPlotAreaRenderer._renderMajorTicks(chart, container, chart.y2Axis, availSpace);
  },

  /**
   * Renders the axis lines for the chart.
   * @param {Chart} chart The chart being rendered.
   * @param {dvt.Container} container The container to render to.
   * @param {dvt.Rectangle} availSpace The available space.
   * @private
   */
  _renderAxisLines: (chart, container, availSpace) => {
    if (chart.xAxis && chart.yAxis && DvtChartAxisUtils.isAxisLineRendered(chart, 'x'))
      DvtChartPlotAreaRenderer._renderAxisLine(
        chart,
        container,
        chart.xAxis,
        chart.yAxis,
        availSpace
      );

    // Render x-axis line based on y2 if there's no y1 or if split dual-Y
    if (chart.xAxis && chart.y2Axis && DvtChartAxisUtils.isAxisLineRendered(chart, 'x')) {
      if (!chart.yAxis || DvtChartDataUtils.isSplitDualY(chart))
        DvtChartPlotAreaRenderer._renderAxisLine(
          chart,
          container,
          chart.xAxis,
          chart.y2Axis,
          availSpace
        );
    }

    if (chart.yAxis && chart.xAxis && DvtChartAxisUtils.isAxisLineRendered(chart, 'y'))
      DvtChartPlotAreaRenderer._renderAxisLine(
        chart,
        container,
        chart.yAxis,
        chart.xAxis,
        availSpace
      );

    if (chart.y2Axis && chart.xAxis && DvtChartAxisUtils.isAxisLineRendered(chart, 'y2'))
      DvtChartPlotAreaRenderer._renderAxisLine(
        chart,
        container,
        chart.y2Axis,
        chart.xAxis,
        availSpace
      );
  },

  /**
   * Renders the major ticks for the axis.
   * @param {Chart} chart The chart being rendered.
   * @param {dvt.Container} container The container to render to.
   * @param {DvtChartAxis} axis The axis owning the major ticks.
   * @param {dvt.Rectangle} availSpace The available space.
   * @private
   */
  _renderMajorTicks: (chart, container, axis, availSpace) => {
    DvtChartPlotAreaRenderer._renderGridlines(
      chart,
      container,
      axis.getOptions()['majorTick'],
      axis.getPosition(),
      axis.getMajorTickCoords(),
      axis.getBaselineCoord(),
      availSpace
    );
  },

  /**
   * Renders the minor ticks for the axis.
   * @param {Chart} chart The chart being rendered.
   * @param {dvt.Container} container The container to render to.
   * @param {DvtChartAxis} axis The axis owning the minor ticks.
   * @param {dvt.Rectangle} availSpace The available space.
   * @private
   */
  _renderMinorTicks: (chart, container, axis, availSpace) => {
    DvtChartPlotAreaRenderer._renderGridlines(
      chart,
      container,
      axis.getOptions()['minorTick'],
      axis.getPosition(),
      axis.getMinorTickCoords(),
      null,
      availSpace
    );
  },

  /**
   * Renders the axis line for the axis.
   * Axis lines are drawn by the opposite axis. For example, x-axis line is drawn based on the y-axis coord (and will be
   * parallel to the y-axis gridlines), and vice versa.
   * @param {Chart} chart The chart being rendered.
   * @param {dvt.Container} container The container to render to.
   * @param {DvtChartAxis} oAxis The axis owning the axis line.
   * @param {DvtChartAxis} dAxis The axis drawing the axis line (i.e. the axis orthogonal to oAxis).
   * @param {dvt.Rectangle} availSpace The available space.
   * @private
   */
  _renderAxisLine: (chart, container, oAxis, dAxis, availSpace) => {
    var options = oAxis.getOptions();
    var position = options['position'];
    var coord =
      position == 'bottom' || position == 'right' || position == 'tangential'
        ? dAxis.getMaxCoord()
        : dAxis.getMinCoord();
    DvtChartPlotAreaRenderer._renderGridlines(
      chart,
      container,
      options['axisLine'],
      dAxis.getPosition(),
      [coord],
      null,
      availSpace
    );
  },

  /**
   * Renders the specified set of gridlines (ticks or axis lines).
   * @param {Chart} chart The chart being rendered.
   * @param {dvt.Container} container The container to render to.
   * @param {Object} options The options object of the gridline.
   * @param {String} position The axis position.
   * @param {Array} coords The array gridline coords.
   * @param {Number} baselineCoord The baseline coord (to use baseline style).
   * @param {dvt.Rectangle} availSpace The available space.
   * @private
   */
  _renderGridlines: (chart, container, options, position, coords, baselineCoord, availSpace) => {
    // Construct the default line stroke
    var lineColor = options['lineColor'];
    var type = options['lineStyle'];
    var lineStroke = new Stroke(
      lineColor,
      1,
      options['lineWidth'],
      false,
      Stroke.getDefaultDashProps(type, options['lineWidth'])
    );

    // Construct the baseline stroke
    var baselineColor = lineStroke.getColor();
    if (options['baselineColor'] != 'inherit') {
      if (options['baselineColor'] == 'auto')
        baselineColor = ColorUtils.getDarker(lineColor, 0.6);
      // derive the baselineColor from lineColor
      else baselineColor = options['baselineColor'];
    }
    var baselineWidth =
      options['baselineWidth'] != null ? options['baselineWidth'] : options['lineWidth'];
    var baselineType = options['baselineStyle'] ? options['baselineStyle'] : options['lineStyle'];
    var baselineStroke = new Stroke(
      baselineColor,
      1,
      baselineWidth,
      false,
      Stroke.getDefaultDashProps(baselineType, baselineWidth)
    );

    // Render a single path for horizontal and vertical gridlines otherwise render individual lines/circles
    var pathCmd = '';
    for (var i = 0; i < coords.length; i++) {
      var isBaseline = baselineCoord != null && coords[i] == baselineCoord;
      if (position == 'radial' || position == 'tangential' || isBaseline) {
        DvtChartPlotAreaRenderer._renderGridline(
          chart,
          container,
          position,
          coords[i],
          isBaseline ? baselineStroke : lineStroke,
          availSpace
        );
      } else if (position == 'top' || position == 'bottom')
        pathCmd +=
          PathUtils.moveTo(coords[i], availSpace.y) +
          PathUtils.verticalLineTo(availSpace.y + availSpace.h);
      else
        pathCmd +=
          PathUtils.moveTo(availSpace.x, coords[i]) +
          PathUtils.horizontalLineTo(availSpace.x + availSpace.w);
    }
    if (pathCmd != '') {
      var path = new Path(chart.getCtx(), pathCmd);
      if (!Agent.isTouchDevice() || Agent.getDevicePixelRatio() > 1)
        path.setPixelHinting(true);
      path.setStroke(lineStroke);
      path.setMouseEnabled(false);
      container.addChild(path);
    }
  },

  /**
   * Renders the specified gridline (tick or axis line).
   * @param {Chart} chart The chart being rendered.
   * @param {dvt.Container} container The container to render to.
   * @param {String} position The axis position.
   * @param {Number} coord The gridline coord.
   * @param {dvt.Stroke} stroke The gridline stroke.
   * @param {dvt.Rectangle} availSpace The available space.
   * @private
   */
  _renderGridline: (chart, container, position, coord, stroke, availSpace) => {
    var line;
    var context = container.getCtx();
    var usePixelHinting = !Agent.isTouchDevice() || Agent.getDevicePixelRatio() > 1;

    if (position == 'radial') {
      if (DvtChartAxisUtils.isGridPolygonal(chart)) {
        var points = PolygonUtils.getRegularPolygonPoints(
          0,
          0,
          DvtChartDataUtils.getGroupCount(chart),
          coord,
          0
        );
        line = new Polygon(context, points);
      } else line = new Circle(context, 0, 0, coord);
      line.setInvisibleFill();
      line.setTranslate(availSpace.x + availSpace.w / 2, availSpace.y + availSpace.h / 2);
    } else if (position == 'tangential') {
      line = new Line(
        context,
        0,
        0,
        chart.getRadius() * Math.sin(coord),
        -chart.getRadius() * Math.cos(coord)
      );
      var mod = coord % (Math.PI / 2); // use pixel hinting at 0, 90, 180, and 270 degrees
      if ((mod < 0.001 || mod > Math.PI / 2 - 0.001) && usePixelHinting) line.setPixelHinting(true);

      line.setTranslate(availSpace.x + availSpace.w / 2, availSpace.y + availSpace.h / 2);
    } else {
      if (position == 'top' || position == 'bottom')
        line = new Line(context, coord, availSpace.y, coord, availSpace.y + availSpace.h);
      else line = new Line(context, availSpace.x, coord, availSpace.x + availSpace.w, coord);
      if (usePixelHinting) line.setPixelHinting(true);
    }
    line.setStroke(stroke);
    line.setMouseEnabled(false);
    container.addChild(line);
  },

  /**
   * Renders objects in the foreground of the plot area.
   * @param {Chart} chart The chart being rendered.
   * @param {dvt.Container} container The container to render to.
   * @param {dvt.Rectangle} availSpace The available space.
   * @private
   */
  _renderForegroundObjs: (chart, container, availSpace) => {
    // : Axis lines are generally rendered in the back of the foreground, but we render them after the
    // bars if only bar series are present.  We can't apply this fix with lines/areas, since the markers must appear over
    // the axis lines.
    var options = chart.getOptions();

    const hasLayering = options['comboSeriesOrder'] === 'data' && DvtChartTypeUtils.isCombo(chart);
    var clipGroup = hasLayering
      ? null
      : DvtChartPlotAreaRenderer.createClippedGroup(chart, container, availSpace);

    // Render axis lines after the clipGroup
    DvtChartPlotAreaRenderer._renderAxisLines(chart, container, availSpace);

    // plotArea border. We add it here to be above the clipGroup, but below the other foreground items
    var plotAreaBorderColor = options['plotArea']['borderColor'];
    var plotAreaBorderWidth = options['plotArea']['borderWidth'];
    if (plotAreaBorderColor && plotAreaBorderWidth != 0) {
      var plotAreaBorder = DvtChartPlotAreaRenderer._getBackgroundShape(chart, availSpace);
      plotAreaBorder.setInvisibleFill();
      plotAreaBorder.setSolidStroke(plotAreaBorderColor, null, plotAreaBorderWidth);
      plotAreaBorder.setMouseEnabled(false);
      container.addChild(plotAreaBorder);
    }

    // Data Objects for BLAC
    if (DvtChartTypeUtils.isBLAC(chart)) {
      const seriesCount = options['series'].length;
      if (hasLayering) {
        for (let index = seriesCount - 1; index >= 0; index--) {
          const seriesType = DvtChartDataUtils.getSeriesType(chart, index);
          if (seriesType !== 'area' && seriesType !== 'lineWithArea')
            clipGroup = DvtChartPlotAreaRenderer.createClippedGroup(chart, container, availSpace);

          if (seriesType === 'area')
            DvtChartPlotAreaRenderer._renderAreas(chart, container, availSpace, false, index);
          else if (seriesType === 'lineWithArea')
            DvtChartPlotAreaRenderer._renderAreas(chart, container, availSpace, true, index);
          else if (seriesType === 'bar')
            DvtChartPlotAreaRenderer._renderBars(chart, clipGroup, availSpace, index);
          else if (seriesType === 'stock')
            DvtChartPlotAreaRenderer._renderStock(chart, clipGroup, index);
          else if (seriesType === 'boxPlot')
            DvtChartPlotAreaRenderer._renderBoxPlot(chart, clipGroup, availSpace, index);
          else if (seriesType === 'line')
            DvtChartPlotAreaRenderer._renderLines(chart, container, clipGroup, availSpace, index);
        }
      } else {
        // Areas were drawn in the background. Draw lineWithAreas, bars, and lines
        if (DvtChartDataUtils.hasLineWithAreaSeries(chart))
          DvtChartPlotAreaRenderer._renderAreas(chart, container, availSpace, true);

        if (DvtChartDataUtils.hasBarSeries(chart))
          DvtChartPlotAreaRenderer._renderBars(chart, clipGroup, availSpace);

        if (DvtChartDataUtils.hasCandlestickSeries(chart))
          DvtChartPlotAreaRenderer._renderStock(chart, clipGroup);

        if (DvtChartDataUtils.hasBoxPlotSeries(chart))
          DvtChartPlotAreaRenderer._renderBoxPlot(chart, clipGroup, availSpace);

        if (DvtChartDataUtils.hasLineSeries(chart))
          DvtChartPlotAreaRenderer._renderLines(chart, container, clipGroup, availSpace);
      }
    } else if (DvtChartTypeUtils.isScatterBubble(chart)) {
      DvtChartPlotAreaRenderer._renderScatterBubble(chart, container, clipGroup, true, availSpace);
    }

    // Reference Objects
    if (
      options['xAxis']['referenceObjects'] ||
      options['yAxis']['referenceObjects'] ||
      options['y2Axis']['referenceObjects']
    ) {
      clipGroup = DvtChartPlotAreaRenderer.createClippedGroup(chart, container, availSpace);
      DvtChartRefObjRenderer.renderForegroundObjects(chart, clipGroup, availSpace);
    }

    // Initial Selection
    var selected = DvtChartDataUtils.getInitialSelection(chart);
    DvtChartEventUtils.setInitialSelection(chart, selected);

    // Initial Highlighting
    chart.highlight(DvtChartDataUtils.getHighlightedCategories(chart));
  },

  /**
   * Renders a single data label associated with a data item.
   * @param {Chart} chart The chart being rendered.
   * @param {dvt.Container} container The container to render to.
   * @param {dvt.Rectangle} dataItemBounds The space occupied by the data item this is associated with.
   * @param {number} seriesIndex The series of this data label.
   * @param {number} groupIndex The group of this data label.
   * @param {number} itemIndex The nested item index of this data label.
   * @param {Color} dataColor The color of the data item this label is associated with.
   * @param {number} type (optional) Data label type: low, high, or value.
   * @param {boolean} isStackLabel true if label for stack cummulative, false otherwise
   * @param {number=} originalBarSize The non rounded width(vertical) or height(horizontal) of the bar
   * @private
   */
  _renderDataLabel: (
    chart,
    container,
    dataItemBounds,
    seriesIndex,
    groupIndex,
    itemIndex,
    dataColor,
    type,
    isStackLabel,
    originalBarSize
  ) => {
    if (DvtChartTypeUtils.isOverview(chart))
      // no data label in overview
      return;

    var isBar = DvtChartDataUtils.getSeriesType(chart, seriesIndex) == 'bar';
    const supportsOutline = DvtChartStyleUtils.supportsLabelOutline(chart, seriesIndex);
    var bHoriz = DvtChartTypeUtils.isHorizontal(chart);
    var barDataItemDims = {
      width: bHoriz ? dataItemBounds.w : originalBarSize,
      height: bHoriz ? originalBarSize : dataItemBounds.h
    };
    var chartOptions = chart.getOptions();
    var styleDefaults = chartOptions['styleDefaults'];
    var hasY2Axis = chart.y2Axis != null;
    var hasLegend = chartOptions.legend.rendered === 'on';

    if (isBar)
      chart.getOptionsCache().putToCachedMap2D('barDims', seriesIndex, groupIndex, barDataItemDims);

    var labelString = DvtChartStyleUtils.getDataLabel(
      chart,
      seriesIndex,
      groupIndex,
      itemIndex,
      type,
      isStackLabel
    );
    if (labelString == null) return;

    var position = DvtChartGroupUtils.getDataLabelPos(
      chart,
      seriesIndex,
      groupIndex,
      itemIndex,
      type,
      isStackLabel
    );
    if (position == 'none') return;

    var label = new OutputText(chart.getCtx(), labelString, 0, 0);
    label.setMouseEnabled(false);

    var style = isStackLabel
      ? styleDefaults['stackLabelStyle']
      : DvtChartStyleUtils.getDataLabelStyle(
          chart,
          seriesIndex,
          groupIndex,
          itemIndex,
          dataColor,
          position,
          type
        );
    label.setCSSStyle(style);

    if (supportsOutline) {
      label.addClassName('oj-chart-data-label-contrast');
    }

    label.setY(dataItemBounds.y + dataItemBounds.h / 2);
    label.setX(dataItemBounds.x + dataItemBounds.w / 2);
    label.alignCenter();
    label.alignMiddle();
    var textDim = label.getDimensions();
    var plotAreaDims = chart.getCache().getFromCache('plotAreaDims');
    if (!plotAreaDims) {
      plotAreaDims = chart.getPlotArea().getDimensions();
      chart.getCache().putToCache('plotAreaDims', plotAreaDims);
    }

    // Get the label coordinates for labels position outside the data item
    var getCenteredOutsideLabelCoords = (pos, itemBounds, labelDims, detectedCollisions) => {
      var coord = { x: null, y: null };

      // Check if the data label's position needs to be shift
      var nudgeUp =
        detectedCollisions && detectedCollisions.xAxis && (pos == 'left' || pos == 'right');
      var nudgeDown =
        detectedCollisions && detectedCollisions.top && (pos == 'left' || pos == 'right');
      var nudgeLeft =
        detectedCollisions &&
        (((hasY2Axis || !hasLegend) && detectedCollisions.y2Axis) || detectedCollisions.legend) &&
        (pos == 'top' || pos == 'bottom');
      var nudgeRight =
        detectedCollisions && detectedCollisions.yAxis && (pos == 'top' || pos == 'bottom');

      if (pos == 'left') {
        coord.x = itemBounds.x - labelDims.w / 2 - DvtChartStyleUtils.MARKER_DATA_LABEL_GAP;
      } else if (pos == 'right') {
        coord.x =
          itemBounds.x + itemBounds.w + labelDims.w / 2 + DvtChartStyleUtils.MARKER_DATA_LABEL_GAP;
      } else if (pos == 'top') {
        coord.y = itemBounds.y - labelDims.h / 2; // No need for buffer because itemBounds.y accounts for typography baseline
      } else if (pos == 'bottom') {
        coord.y =
          itemBounds.y +
          itemBounds.h +
          labelDims.h / 2 +
          DvtChartStyleUtils.MARKER_DATA_LABEL_GAP / 2;
      }

      var edgeLabelGap = 2 * DvtChartStyleUtils.MARKER_DATA_LABEL_GAP; // Doubling the gap around the edges because the single gap looks too small
      if (nudgeLeft) {
        coord.x = plotAreaDims.x + plotAreaDims.w - labelDims.w / 2 - edgeLabelGap;
      } else if (nudgeRight) {
        coord.x = labelDims.w / 2 + edgeLabelGap;
      } else if (nudgeUp) {
        coord.y = plotAreaDims.y + plotAreaDims.h - labelDims.h / 2 - edgeLabelGap;
      } else if (nudgeDown) {
        coord.y = labelDims.h / 2 + edgeLabelGap;
      }
      return coord;
    };

    var collision = styleDefaults['dataLabelCollision'];
    var outsideCenteredLabelCoords = getCenteredOutsideLabelCoords(
      position,
      dataItemBounds,
      textDim
    );
    if (collision === 'fitInBounds') {
      // Reset label position and coordinates if labels have collisions
      var detectedCollisions = DvtChartStyleUtils.getDataLabelCollisions(
        chart,
        outsideCenteredLabelCoords,
        textDim,
        plotAreaDims,
        bHoriz
      );
      var adjustedPosition = DvtChartStyleUtils.adjustDataLabelPos(
        position,
        detectedCollisions,
        hasY2Axis
      );

      if (adjustedPosition != position && DvtChartTypeUtils.isBubble(chart)) {
        // Adjust bubble label style if position changed
        style = isStackLabel
          ? styleDefaults['stackLabelStyle']
          : DvtChartStyleUtils.getDataLabelStyle(
              chart,
              seriesIndex,
              groupIndex,
              itemIndex,
              dataColor,
              adjustedPosition,
              type
            );
        label.setCSSStyle(style);
      }

      position = adjustedPosition;
      outsideCenteredLabelCoords = getCenteredOutsideLabelCoords(
        position,
        dataItemBounds,
        textDim,
        detectedCollisions
      );
    }

    if (outsideCenteredLabelCoords.x != null || outsideCenteredLabelCoords.y != null) {
      if (outsideCenteredLabelCoords.x != null) {
        label.setX(outsideCenteredLabelCoords.x);
      }
      if (outsideCenteredLabelCoords.y != null) {
        label.setY(outsideCenteredLabelCoords.y);
      }
    } else {
      // inside label
      if (isBar) {
        if (textDim.w > barDataItemDims.width || textDim.h > barDataItemDims.height) return; //dropping text if doesn't fit.

        if (position == 'inLeft') {
          label.setX(dataItemBounds.x + textDim.w / 2.0 + DvtChartStyleUtils.MARKER_DATA_LABEL_GAP);
        } else if (position == 'inRight') {
          label.setX(
            dataItemBounds.x +
              dataItemBounds.w -
              textDim.w / 2.0 -
              DvtChartStyleUtils.MARKER_DATA_LABEL_GAP
          );
        } else if (position == 'inTop') {
          label.setY(dataItemBounds.y + textDim.h / 2.0 + DvtChartStyleUtils.MARKER_DATA_LABEL_GAP);
        } else if (position == 'inBottom') {
          label.setY(
            dataItemBounds.y +
              dataItemBounds.h -
              textDim.h / 2.0 -
              DvtChartStyleUtils.MARKER_DATA_LABEL_GAP / 2
          );
        }
      } else if (DvtChartTypeUtils.isBubble(chart)) {
        dataItemBounds.x += DvtChartStyleUtils.MARKER_DATA_LABEL_GAP;
        dataItemBounds.y += DvtChartStyleUtils.MARKER_DATA_LABEL_GAP;
        dataItemBounds.h -= DvtChartStyleUtils.MARKER_DATA_LABEL_GAP * 2;
        dataItemBounds.w -= DvtChartStyleUtils.MARKER_DATA_LABEL_GAP * 2;

        var size = TextUtils.getOptimalFontSize(
          label.getCtx(),
          label.getTextString(),
          label.getCSSStyle(),
          dataItemBounds
        );
        label.setFontSize(size);
        if (
          !TextUtils.fitText(
            label,
            dataItemBounds.w,
            dataItemBounds.h,
            container,
            DvtChartPlotAreaRenderer._MIN_CHARS_DATA_LABEL
          )
        )
          return; //dropping text if doesn't fit.
      }

      const isPatternBg = DvtChartStyleUtils.getPattern(chart, seriesIndex, groupIndex) != null;
      if (isPatternBg && !supportsOutline) {
        DvtChartPlotAreaRenderer._addLabelContrastBackground(chart, container, label);
      }
    }

    // Truncate or drop labels if they go outside the plot area. 
    if (DvtChartTypeUtils.isBar(chart)) {
      var labelDim = label.getDimensions();
      var isLabelOutsidePlotArea = false;
      var maxWidth;
      if (position == 'left' && labelDim.x < 0) {
        isLabelOutsidePlotArea = true;
        maxWidth = dataItemBounds.x - DvtChartStyleUtils.MARKER_DATA_LABEL_GAP;
      } else if (
        position == 'right' &&
        labelDim.x + labelDim.w > plotAreaDims.w + DvtChartStyleUtils.MARKER_DATA_LABEL_GAP
      ) {
        isLabelOutsidePlotArea = true;
        maxWidth = plotAreaDims.w - dataItemBounds.w - DvtChartStyleUtils.MARKER_DATA_LABEL_GAP;
      } else if (position == 'top' && labelDim.y < 0) return;
      else if (
        position == 'bottom' &&
        labelDim.y + labelDim.h >
          plotAreaDims.y + plotAreaDims.h + DvtChartStyleUtils.MARKER_DATA_LABEL_GAP
      )
        return;

      if (isLabelOutsidePlotArea) {
        if (!isNaN(labelString)) return;
        else if (TextUtils.fitText(label, maxWidth, dataItemBounds.h, container, 1)) {
          // Recalculate x co-ordinate if truncated
          textDim = label.getDimensions();
          if (position == 'left') {
            label.setX(dataItemBounds.x - textDim.w / 2 - DvtChartStyleUtils.MARKER_DATA_LABEL_GAP);
          } else if (position == 'right') {
            label.setX(
              dataItemBounds.x +
                dataItemBounds.w +
                textDim.w / 2 +
                DvtChartStyleUtils.MARKER_DATA_LABEL_GAP
            );
          }
        } else return;
      }
    }

    // Reset the stroke so that the container properties aren't inherited
    if (DvtChartStyleUtils.optimizeMarkerStroke(chart)) label.setSolidStroke('none');

    container.addChild(label);
    chart.addDataLabel(label);
  },

  /**
   * Helper function. Calculates and passes the marker bounds to the data label rendering code.
   * @param {Chart} chart The chart being rendered.
   * @param {dvt.Container} container The container to render to.
   * @param {dvt.SimpleMarker} marker
   * @private
   */
  _renderDataLabelForMarker: (chart, container, marker) => {
    var logicalObject = chart.getEventManager().getLogicalObject(marker);
    if (!logicalObject) return;

    var seriesIndex = logicalObject.getSeriesIndex();
    var groupIndex = logicalObject.getGroupIndex();
    var itemIndex = logicalObject.getNestedDataItemIndex();

    if (marker instanceof SimpleMarker) {
      var markerBounds = new Rectangle(
        marker.getCx() - marker.getWidth() / 2,
        marker.getCy() - marker.getHeight() / 2,
        marker.getWidth(),
        marker.getHeight()
      );
      DvtChartPlotAreaRenderer._renderDataLabel(
        chart,
        container,
        markerBounds,
        seriesIndex,
        groupIndex,
        itemIndex,
        marker.getDataColor()
      );
    } else if (marker instanceof DvtChartRangeMarker) {
      DvtChartPlotAreaRenderer._renderDataLabel(
        chart,
        container,
        marker.getBoundingBox1(),
        seriesIndex,
        groupIndex,
        itemIndex,
        marker.getDataColor(),
        'low'
      );
      DvtChartPlotAreaRenderer._renderDataLabel(
        chart,
        container,
        marker.getBoundingBox2(),
        seriesIndex,
        groupIndex,
        itemIndex,
        marker.getDataColor(),
        'high'
      );
    }
  },

  /**
   * Renders the data markers for scatter and bubble chart.
   * @param {Chart} chart The chart being rendered.
   * @param {dvt.Container} container The container to render to.
   * @param {dvt.Container} clipGroup The group for clipping the lines and bubbles.
   * @param {boolean} bSortBySize True if markers should be sorted by size to reduce overlaps.
   * @param {dvt.Rectangle} availSpace The available space.
   * @private
   */
  _renderScatterBubble: (chart, container, clipGroup, bSortBySize, availSpace) => {
    // Performance optimization for huge data sets.
    var markerInfos = DvtChartPlotAreaRenderer._filterScatterBubble(chart, bSortBySize);

    // Loop through the series and draw line connectors.
    var seriesIndex;
    var seriesCount = DvtChartDataUtils.getSeriesCount(chart);
    for (seriesIndex = 0; seriesIndex < seriesCount; seriesIndex++) {
      // Draw the line connector if the series has one
      if (DvtChartStyleUtils.getLineType(chart, seriesIndex) != 'none')
        DvtChartPlotAreaRenderer._renderLinesForSeries(chart, clipGroup, seriesIndex, availSpace);
    }

    // Calculate the default stroke to save DOM calls.
    var borderColor = DvtChartStyleUtils.getMarkerBorderColor(chart);
    var borderWidth = DvtChartStyleUtils.getBorderWidth(chart);
    var defaultStroke = new Stroke(borderColor, 1, borderWidth);

    // Create the markers
    var markers = [];
    if (markerInfos) {
      // The stroke is optimized onto the parent container when possible.
      var bOptimizeStroke = DvtChartStyleUtils.optimizeMarkerStroke(chart);
      var defaultBorderColor = bOptimizeStroke ? defaultStroke.getColor() : null;
      var defaultBorderWidth = bOptimizeStroke ? defaultStroke.getWidth() : null;

      // Markers were filtered and markerInfo contains only the visible markers.
      for (var markerIndex = 0; markerIndex < markerInfos.length; markerIndex++) {
        var markerInfo = markerInfos[markerIndex];
        var marker = DvtChartPlotAreaRenderer._createMarker(
          chart,
          markerInfo,
          defaultBorderColor,
          defaultBorderWidth
        );
        markers.push(marker);
      }
    } else {
      for (seriesIndex = 0; seriesIndex < seriesCount; seriesIndex++) {
        var seriesMarkers = DvtChartPlotAreaRenderer._getMarkersForSeries(
          chart,
          seriesIndex,
          availSpace,
          defaultStroke
        );
        markers = markers.concat(seriesMarkers);
      }

      // Sort the markers from smallest to largest. Filtered markers were already sorted.
      if (bSortBySize) DvtChartMarkerUtils.sortMarkers(markers);
    }

    // Finally add the markers to their container.
    if (DvtChartTypeUtils.isBubble(chart))
      // For bubble, all the lines and markers are added to the clipGroups
      DvtChartPlotAreaRenderer._addMarkersToContainer(chart, clipGroup, markers, defaultStroke);
    // For scatter, don't want to have the markers in the clipGroup so they don't get clipped
    else DvtChartPlotAreaRenderer._addMarkersToContainer(chart, container, markers, defaultStroke);
  },

  /**
   * Renders the data markers for the specified line/area series.
   * @param {Chart} chart The chart being rendered.
   * @param {dvt.Container} container The container to render to.
   * @param {number} seriesIndex The series to render.
   * @param {dvt.Rectangle} availSpace The available space.
   * @private
   */
  _renderMarkersForSeries: (chart, container, seriesIndex, availSpace) => {
    // Calculate the default stroke to save DOM calls.
    var borderColor = DvtChartStyleUtils.getMarkerBorderColor(chart, seriesIndex);
    var borderWidth = DvtChartStyleUtils.getBorderWidth(chart, seriesIndex);
    var defaultStroke = new Stroke(borderColor, 1, borderWidth);

    var markers;
    if (DvtChartDataUtils.isRangeSeries(chart, seriesIndex))
      markers = DvtChartPlotAreaRenderer._getRangeMarkersForSeries(chart, seriesIndex, availSpace);
    else
      markers = DvtChartPlotAreaRenderer._getMarkersForSeries(
        chart,
        seriesIndex,
        availSpace,
        defaultStroke
      );

    DvtChartPlotAreaRenderer._addMarkersToContainer(chart, container, markers, defaultStroke);
  },

  _addLabelContrastBackground: (chart, container, label) => {
    var textDim = label.getDimensions();
    var padding = textDim.h * 0.15;
    var cmd = PathUtils.roundedRectangle(
      textDim.x - padding,
      textDim.y,
      textDim.w + 2 * padding,
      textDim.h,
      2,
      2,
      2,
      2
    );
    var bbox = new Path(chart.getCtx(), cmd);
    bbox.setSolidFill('#FFFFFF', 0.9);
    container.addChild(bbox);
  },

  /**
   * Adds the specified data markers to the container.
   * @param {Chart} chart The chart being rendered.
   * @param {dvt.Container} container The container to render to.
   * @param {array} markers The array of data markers.
   * @param {dvt.Stroke} defaultStroke The default stroke of the markers for optimization.
   * @private
   */
  _addMarkersToContainer: (chart, container, markers, defaultStroke) => {
    // Performance optimization for scatter and bubble: Create a container for the markers and data labels.
    var markerContainer = container;
    var bOptimizeStroke = DvtChartStyleUtils.optimizeMarkerStroke(chart);
    var bOptimizeFill = DvtChartStyleUtils.optimizeMarkerFill(chart);

    if (bOptimizeStroke || bOptimizeFill) {
      markerContainer = new Container(chart.getCtx());
      if (bOptimizeStroke) markerContainer.setStroke(defaultStroke);
      else markerContainer.setInvisibleFill();
      container.addChild(markerContainer);
    }

    // Add the markers to the container
    for (var i = 0; i < markers.length; i++) {
      markerContainer.addChild(markers[i]);

      // Data Label Support
      DvtChartPlotAreaRenderer._renderDataLabelForMarker(chart, markerContainer, markers[i]);
    }

    // TODO: change to formal location for displayed data
    chart._currentMarkers.push(markers);
  },

  /**
   * Returns an object with rendering information for a single marker. Returns null
   * if the marker should be skipped. The returned object must remain consistent
   * with that returned by _getScatterBubbleMarkerInfo.
   * @param {Chart} chart The chart being rendered.
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @param {number} itemIndex
   * @param {dvt.Rectangle} availSpace The available space.
   * @return {object} An object with rendering information. Fields not documented as it is intended for use within this
   *                 class only.
   * @private
   */
  _getMarkerInfo: (chart, seriesIndex, groupIndex, itemIndex, availSpace) => {
    var options = chart.getOptions();

    // Skip for null or undefined values
    var dataValue = DvtChartDataUtils.getVal(chart, seriesIndex, groupIndex, itemIndex);
    if (dataValue == null || isNaN(dataValue)) return null;

    // Filter markers to optimize rendering
    if (DvtChartDataUtils.isDataItemFiltered(chart, seriesIndex, groupIndex)) return null;

    // Determine whether a visible marker is to be displayed
    var numGroups = DvtChartDataUtils.getGroupCount(chart);
    var bMarkerDisplayed = DvtChartStyleUtils.isMarkerDisplayed(
      chart,
      seriesIndex,
      groupIndex,
      itemIndex
    );
    if (!bMarkerDisplayed && (itemIndex == null || itemIndex < 0)) {
      // If both previous and next values are null, then always display a marker
      // In computing the prev and next indices for polar chart, allow the index to wrap around
      var lastIndex = numGroups - 1;
      var isPolar = DvtChartTypeUtils.isPolar(chart);
      var prevIndex = isPolar && lastIndex > 0 && groupIndex == 0 ? lastIndex : groupIndex - 1;
      var nextIndex = isPolar && lastIndex > 0 && groupIndex == lastIndex ? 0 : groupIndex + 1;

      var prevValue = DvtChartDataUtils.getVal(chart, seriesIndex, prevIndex);
      var nextValue = DvtChartDataUtils.getVal(chart, seriesIndex, nextIndex);
      if ((prevValue == null || isNaN(prevValue)) && (nextValue == null || isNaN(nextValue)))
        bMarkerDisplayed = true;
    }

    // Skip hidden markers for overview, animation, and spark.
    if (!bMarkerDisplayed) {
      if (
        DvtChartTypeUtils.isSpark(chart) ||
        ((options['_duringZoomAndScroll'] || DvtChartTypeUtils.isOverview(chart)) &&
          !DvtChartDataUtils.isDataSelected(chart, seriesIndex, groupIndex, itemIndex))
      )
        return null;
    }

    // Skip if not visible. This check is relatively slow so we do it after the above checks.
    if (!DvtChartDataUtils.isDataItemRendered(chart, seriesIndex, groupIndex, itemIndex))
      return false;

    var bInViewport = true;
    var pos = DvtChartDataUtils.getMarkerPos(chart, seriesIndex, groupIndex, itemIndex, availSpace);
    var markerSize = DvtChartStyleUtils.getMarkerSize(chart, seriesIndex, groupIndex, itemIndex);
    if (availSpace && pos && markerSize)
      bInViewport = availSpace.intersects(
        new Rectangle(pos.x - markerSize / 2, pos.y - markerSize / 2, markerSize, markerSize)
      );

    if (!pos || !bInViewport) return null;

    return {
      seriesIndex: seriesIndex,
      groupIndex: groupIndex,
      itemIndex: itemIndex,
      x: pos.x,
      y: pos.y,
      size: markerSize,
      markerDisplayed: bMarkerDisplayed
    };
  },

  /**
   * Optimized version of _getMarkerInfo for large data bubble and scatter charts.
   * The returned object's fields must remain consistent with that returned by
   * _getMarkerInfo.
   * @param {Chart} chart The chart being rendered.
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @return {object} An object with rendering information. Fields not documented as it is intended for use within this
   *                 class only.
   * @private
   */
  _getScatterBubbleMarkerInfo: (chart, seriesIndex, groupIndex) => {
    // Skip if not visible. This check is relatively slow so we do it after the above checks.
    if (!DvtChartDataUtils.isDataItemRendered(chart, seriesIndex, groupIndex)) return false;

    var pos = DvtChartDataUtils.getScatterBubbleMarkerPos(chart, seriesIndex, groupIndex);
    if (!pos) return null;

    var markerSize = DvtChartStyleUtils.getMarkerSize(chart, seriesIndex, groupIndex);
    if (!markerSize) return null;

    return {
      seriesIndex: seriesIndex,
      groupIndex: groupIndex,
      x: pos.x,
      y: pos.y,
      size: markerSize
    };
  },

  /**
   * Creates and returns the array of dvt.SimpleMarker objects for the specified series.
   * @param {Chart} chart The chart being rendered.
   * @param {number} seriesIndex
   * @param {dvt.Rectangle} availSpace The available space.
   * @param {dvt.Stroke} defaultStroke The default stroke that will be applied to the container of the markers.
   * @return {array} The array of dvt.SimpleMarker objects for the specified series.
   * @private
   */
  _getMarkersForSeries: (chart, seriesIndex, availSpace, defaultStroke) => {
    // Skip the series if it shouldn't be rendered
    if (!DvtChartDataUtils.isSeriesRendered(chart, seriesIndex)) return [];

    // The stroke is optimized onto the parent container when possible.
    var bOptimizeStroke = DvtChartStyleUtils.optimizeMarkerStroke(chart);
    var defaultBorderColor = bOptimizeStroke ? defaultStroke.getColor() : null;
    var defaultBorderWidth = bOptimizeStroke ? defaultStroke.getWidth() : null;

    // Keep track of the markers so that they can be sorted and added
    var markers = [];

    // Loop through the groups in the series
    var numGroups = DvtChartDataUtils.getGroupCount(chart);
    for (var groupIndex = 0; groupIndex < numGroups; groupIndex++) {
      // Calculate the rendering info for the marker. If null is returned, skip.
      var markerInfo = DvtChartPlotAreaRenderer._getMarkerInfo(
        chart,
        seriesIndex,
        groupIndex,
        null,
        availSpace
      );
      if (!markerInfo) continue;

      var marker = DvtChartPlotAreaRenderer._createMarker(
        chart,
        markerInfo,
        defaultBorderColor,
        defaultBorderWidth
      );
      if (marker != null) markers.push(marker);
    }

    return markers;
  },

  /**
   * Creates the marker for the specified data item.
   * @param  {DvtChart} chart
   * @param  {object} markerInfo
   * @param  {string} defaultBorderColor
   * @param  {string} defaultBorderWidth
   * @return {dvt.SimpleMarker}
   * @private
   */
  _createMarker: (chart, markerInfo, defaultBorderColor, defaultBorderWidth) => {
    var isTouchDevice = Agent.isTouchDevice();
    var context = chart.getCtx();

    var bOptimizeStroke = DvtChartStyleUtils.optimizeMarkerStroke(chart);
    var bOptimizeInvisibleFill = DvtChartStyleUtils.optimizeMarkerFill(chart);

    // Create the marker
    var marker;
    var seriesIndex = markerInfo.seriesIndex;
    var groupIndex = markerInfo.groupIndex;
    var itemIndex = markerInfo.itemIndex;
    var dataColor = DvtChartStyleUtils.getMarkerColor(chart, seriesIndex, groupIndex, itemIndex);
    var markerShape = DvtChartStyleUtils.getMarkerShape(chart, seriesIndex, groupIndex, itemIndex);
    var markerDisplayed = markerInfo.markerDisplayed;
    if (markerDisplayed == null)
      markerDisplayed = DvtChartStyleUtils.isMarkerDisplayed(
        chart,
        seriesIndex,
        groupIndex,
        itemIndex
      );

    if (markerDisplayed) {
      // Support for visible markers
      var source = DvtChartStyleUtils.getImageSource(
        chart,
        seriesIndex,
        groupIndex,
        itemIndex,
        'source'
      );
      if (source) {
        var sourceSelected = DvtChartStyleUtils.getImageSource(
          chart,
          seriesIndex,
          groupIndex,
          itemIndex,
          'sourceSelected'
        );
        var sourceHover = DvtChartStyleUtils.getImageSource(
          chart,
          seriesIndex,
          groupIndex,
          itemIndex,
          'sourceHover'
        );
        var sourceHoverSelected = DvtChartStyleUtils.getImageSource(
          chart,
          seriesIndex,
          groupIndex,
          itemIndex,
          'sourceHoverSelected'
        );
        marker = new ImageMarker(
          context,
          markerInfo.x,
          markerInfo.y,
          markerInfo.size,
          markerInfo.size,
          null,
          source,
          sourceSelected,
          sourceHover,
          sourceHoverSelected
        );
        if (DvtChartStyleUtils.isSelectable(chart, seriesIndex, groupIndex)) {
          marker.setCursor(SelectionEffectUtils.getSelectingCursor());
        }
      } else {
        marker = new SimpleMarker(
          context,
          markerShape,
          markerInfo.x,
          markerInfo.y,
          markerInfo.size,
          markerInfo.size,
          null,
          null,
          true
        );

        // Apply the marker style
        marker.setFill(
          DvtChartSeriesEffectUtils.getMarkerFill(chart, seriesIndex, groupIndex, itemIndex)
        );

        var userBorderColor = DvtChartStyleUtils.getBorderColor(
          chart,
          seriesIndex,
          groupIndex,
          itemIndex
        );
        var markerDefaultBorderColor = DvtChartStyleUtils.getDefaultMarkerBorderColor(
          chart,
          seriesIndex,
          groupIndex,
          itemIndex
        );
        var userBorderWidth = DvtChartStyleUtils.getUserBorderWidth(
          chart,
          seriesIndex,
          groupIndex,
          itemIndex
        );
        var markerDefaultBorderWidth = DvtChartStyleUtils.getDefaultMarkerBorderWidth(
          chart,
          seriesIndex,
          groupIndex,
          itemIndex
        );

        var borderColor = userBorderColor || markerDefaultBorderColor;
        var borderWidth = userBorderWidth !== null ? userBorderWidth : markerDefaultBorderWidth;

        if (borderColor !== defaultBorderColor || borderWidth !== defaultBorderWidth) {
          marker.setSolidStroke(borderColor, null, borderWidth);
        }

        // Set the data color, used for data label generation
        marker.setDataColor(dataColor, true);

        if (chart.getCtx().getThemeBehavior() === 'redwood') {
          if (DvtChartTypeUtils.isBubble(chart)) {
            marker.addClassName('oj-dvt-bubble-marker');
          }
          if (!userBorderColor && !DvtChartTypeUtils.isBubble(chart)) {
            marker.addClassName('oj-dvt-default-border-color');
          }
          if (userBorderWidth == null) {
            marker.addClassName('oj-dvt-default-border-width');
          }
          if (DvtChartStyleUtils.isSelectable(chart, seriesIndex, groupIndex)) {
            marker.setCursor(SelectionEffectUtils.getSelectingCursor());
            if (DvtChartTypeUtils.isBubble(chart)) {
              marker.setFeedbackClassName('oj-dvt-selectable');
            } else {
              marker.setFeedbackClassName('oj-dvt-selectable-marker');
            }
          }
        } else {
          if (DvtChartStyleUtils.isSelectable(chart, seriesIndex, groupIndex)) {
            marker.setCursor(SelectionEffectUtils.getSelectingCursor());
          }
          // Apply the selection effects, which are also used for keyboard focus.
          var hoverColor = SelectionEffectUtils.getHoverBorderColor(dataColor);
          var innerColor = DvtChartStyleUtils.getSelectedInnerColor(chart);
          var outerColor = DvtChartStyleUtils.getSelectedOuterColor(chart);
          marker.setHoverStroke(
            new Stroke(innerColor, 1, 1),
            new Stroke(hoverColor, 1, 3.5)
          );
          marker.setSelectedStroke(
            new Stroke(innerColor, 1, 1.5),
            new Stroke(outerColor, 1, 4.5)
          );
          marker.setSelectedHoverStroke(
            new Stroke(innerColor, 1, 1.5),
            new Stroke(hoverColor, 1, 4.5)
          );
        }
        marker.addClassName(
          DvtChartStyleUtils.getMarkerClassName(chart, seriesIndex, groupIndex, itemIndex)
        );
        marker.setStyle(
          DvtChartStyleUtils.getMarkerStyle(chart, seriesIndex, groupIndex, itemIndex)
        );
      }

      // Make sure that the marker hit area is large enough for touch devices ()
      // Also make sure there is only 1 invisible marker ()
      if (isTouchDevice && markerInfo.size < DvtChartPlotAreaRenderer._MIN_TOUCH_MARKER_SIZE)
        DvtChartPlotAreaRenderer._addMarkerHitArea(
          marker,
          markerInfo.x,
          markerInfo.y,
          bOptimizeStroke
        );
    } else {
      // Support for invisible markers for tooltips/interactivity
      if (DvtChartStyleUtils.isSelectable(chart, seriesIndex, groupIndex)) {
        marker = new DvtChartLineMarker(
          context,
          markerShape,
          markerInfo.x,
          markerInfo.y,
          markerInfo.size,
          bOptimizeStroke
        );
        marker.setCursor(SelectionEffectUtils.getSelectingCursor());
        if (isTouchDevice)
          DvtChartPlotAreaRenderer._addMarkerHitArea(
            marker,
            markerInfo.x,
            markerInfo.y,
            bOptimizeStroke
          );
      } else {
        // Selection not supported
        if (isTouchDevice) markerInfo.size = DvtChartPlotAreaRenderer._MIN_TOUCH_MARKER_SIZE;

        marker = new DvtChartLineMarker(
          context,
          SimpleMarker.SQUARE,
          markerInfo.x,
          markerInfo.y,
          markerInfo.size,
          bOptimizeStroke
        );
      }

      if (!bOptimizeInvisibleFill) marker.setInvisibleFill();

      marker.setDataColor(dataColor);
    }

    // Associate the marker for interactivity
    DvtChartObjPeer.associate(marker, chart, seriesIndex, groupIndex, itemIndex, markerInfo);

    return marker;
  },

  /**
   * Adds hit area to marker.
   * @param {dvt.SimpleMarker} marker
   * @param {number} x The marker x position.
   * @param {number} y The marker y position.
   * @param {boolean} bOptimizeStroke Whether the marker stroke is defined in the marker container.
   * @private
   */
  _addMarkerHitArea: (marker, x, y, bOptimizeStroke) => {
    var hitArea = new Rect(
      marker.getCtx(),
      x - DvtChartPlotAreaRenderer._MIN_TOUCH_MARKER_SIZE / 2,
      y - DvtChartPlotAreaRenderer._MIN_TOUCH_MARKER_SIZE / 2,
      DvtChartPlotAreaRenderer._MIN_TOUCH_MARKER_SIZE,
      DvtChartPlotAreaRenderer._MIN_TOUCH_MARKER_SIZE
    );

    // If stroke is optimized by defining in the container, it needs to be removed from the hit area.
    if (bOptimizeStroke) hitArea.setSolidStroke('none');

    hitArea.setInvisibleFill();
    marker.addChild(hitArea);
  },

  /**
   * Creates and returns the array of DvtChartRangeMarker objects for the specified series.
   * @param {Chart} chart The chart being rendered.
   * @param {number} seriesIndex
   * @param {dvt.Rectangle} availSpace The available space.
   * @return {array} The array of DvtChartRangeMarker objects for the specified series.
   * @private
   */
  _getRangeMarkersForSeries: (chart, seriesIndex, availSpace) => {
    // Skip the series if it shouldn't be rendered
    if (!DvtChartDataUtils.isSeriesRendered(chart, seriesIndex)) return [];

    var isTouchDevice = Agent.isTouchDevice();
    var context = chart.getCtx();
    var xAxis = chart.xAxis;
    var yAxis = DvtChartDataUtils.isAssignedToY2(chart, seriesIndex) ? chart.y2Axis : chart.yAxis;
    var options = chart.getOptions();
    var numGroups = DvtChartDataUtils.getGroupCount(chart);

    // Keep track of the markers so that they can be sorted and added
    var markers = [];

    // Loop through the groups in the series
    for (var groupIndex = 0; groupIndex < numGroups; groupIndex++) {
      // Filter markers to optimize rendering
      if (DvtChartDataUtils.isDataItemFiltered(chart, seriesIndex, groupIndex)) continue;

      // Skip if not visible
      if (!DvtChartDataUtils.isDataItemRendered(chart, seriesIndex, groupIndex)) continue;

      // Get the axis values
      var xCoord = xAxis.getCoordAt(DvtChartDataUtils.getXVal(chart, seriesIndex, groupIndex));
      var lowCoord = yAxis.getUnboundedCoordAt(
        DvtChartDataUtils.getLowVal(chart, seriesIndex, groupIndex)
      );
      var highCoord = yAxis.getUnboundedCoordAt(
        DvtChartDataUtils.getHighVal(chart, seriesIndex, groupIndex)
      );

      if (xCoord == null || lowCoord == null || highCoord == null) continue;

      var bMarkerDisplayed = DvtChartStyleUtils.isMarkerDisplayed(chart, seriesIndex, groupIndex);
      if (!bMarkerDisplayed) {
        // If both previous and next values are null, then always display a marker
        // In computing the prev and next indices for polar chart, allow the index to wrap around
        var lastIndex = numGroups - 1;
        var isPolar = DvtChartTypeUtils.isPolar(chart);
        var prevIndex = isPolar && lastIndex > 0 && groupIndex == 0 ? lastIndex : groupIndex - 1;
        var nextIndex = isPolar && lastIndex > 0 && groupIndex == lastIndex ? 0 : groupIndex + 1;

        var prevLowValue = DvtChartDataUtils.getLowVal(chart, seriesIndex, prevIndex);
        var prevHighValue = DvtChartDataUtils.getHighVal(chart, seriesIndex, prevIndex);
        var nextLowValue = DvtChartDataUtils.getLowVal(chart, seriesIndex, nextIndex);
        var nextHighValue = DvtChartDataUtils.getHighVal(chart, seriesIndex, prevIndex);

        if (
          prevLowValue == null &&
          prevHighValue == null &&
          nextLowValue == null &&
          nextHighValue == null
        )
          bMarkerDisplayed = true;
      }

      // If the chart is inside overview or in the middle of animation, don't render invisible markers unless the marker is selected.
      if (
        (options['_duringZoomAndScroll'] ||
          DvtChartTypeUtils.isOverview(chart) ||
          DvtChartTypeUtils.isSpark(chart)) &&
        !bMarkerDisplayed &&
        !DvtChartDataUtils.isDataSelected(chart, seriesIndex, groupIndex)
      )
        continue;

      // Store the center of the data point relative to the plot area (for marquee selection)
      var pLow = DvtChartCoordUtils.convertAxisCoord(
        chart,
        new Point(xCoord, lowCoord),
        availSpace
      );
      var pHigh = DvtChartCoordUtils.convertAxisCoord(
        chart,
        new Point(xCoord, highCoord),
        availSpace
      );
      var dataPos = new Point((pLow.x + pHigh.x) / 2, (pLow.y + pHigh.y) / 2);

      // Create the marker. Even if the marker is invisible, we need it for keyboard and voiceover support.
      var markerSize = DvtChartStyleUtils.getMarkerSize(chart, seriesIndex, groupIndex);
      var marker = new DvtChartRangeMarker(
        context,
        pLow.x,
        pLow.y,
        pHigh.x,
        pHigh.y,
        markerSize,
        !bMarkerDisplayed
      );
      var fill = DvtChartSeriesEffectUtils.getMarkerFill(chart, seriesIndex, groupIndex);
      var borderColor = DvtChartStyleUtils.getMarkerBorderColor(chart, seriesIndex, groupIndex);
      var borderWidth = DvtChartStyleUtils.getBorderWidth(chart, seriesIndex, groupIndex);
      var stroke = new Stroke(borderColor, 1, borderWidth);
      var dataColor = DvtChartStyleUtils.getMarkerColor(chart, seriesIndex, groupIndex);
      var innerColor = DvtChartStyleUtils.getSelectedInnerColor(chart);
      var outerColor = DvtChartStyleUtils.getSelectedOuterColor(chart);
      marker.setStyleProperties(fill, stroke, dataColor, innerColor, outerColor);

      if (DvtChartStyleUtils.isSelectable(chart, seriesIndex, groupIndex))
        marker.setCursor(SelectionEffectUtils.getSelectingCursor());

      // Create hit area
      var hitArea = new Line(context, pLow.x, pLow.y, pHigh.x, pHigh.y);
      hitArea.setSolidStroke(
        'rgba(0,0,0,0)',
        null,
        isTouchDevice
          ? Math.max(markerSize, DvtChartPlotAreaRenderer._MIN_TOUCH_MARKER_SIZE)
          : markerSize
      );
      marker.addChild(hitArea);

      // Add it to the markers array for sorting and addition to the display list later
      markers.push(marker);

      // Associate the marker for interactivity
      DvtChartObjPeer.associate(marker, chart, seriesIndex, groupIndex, null, dataPos);
    }

    return markers;
  },

  /**
   * Renders all bar series for the given chart.
   * @param {Chart} chart
   * @param {dvt.Container} container The container to render to.
   * @param {dvt.Rectangle} availSpace
   * @private
   */
  _renderBars: (chart, container, availSpace, barSeriesIndex) => {
    var bHoriz = DvtChartTypeUtils.isHorizontal(chart);
    var bPolar = DvtChartTypeUtils.isPolar(chart);
    var bStock = DvtChartTypeUtils.isStock(chart);
    var bPixelSpacing = DvtChartStyleUtils.getBarSpacing(chart) == 'pixel';
    var isR2L = Agent.isRightToLeft(chart.getCtx());
    var hasStackLabel = DvtChartStyleUtils.isStackLabelRendered(chart);
    var isStacked = DvtChartDataUtils.isStacked(chart);

    // fix for chart stack label for data with series count < 2; 
    if (DvtChartDataUtils.getSeriesCount(chart) < 2 && hasStackLabel) {
      isStacked = true;
    }
    var innerColor = DvtChartStyleUtils.getSelectedInnerColor(chart);
    var outerColor = DvtChartStyleUtils.getSelectedOuterColor(chart);
    var duringZoomAndScroll = chart.getOptions()['_duringZoomAndScroll'];
    var groupWidth = DvtChartStyleUtils.getGroupWidth(chart);
    var prevStackYCoords = {};

    // Iterate through the data
    for (
      var seriesIndex = 0;
      seriesIndex < DvtChartDataUtils.getSeriesCount(chart);
      seriesIndex++
    ) {
      if (
        DvtChartDataUtils.getSeriesType(chart, seriesIndex) != 'bar' ||
        (barSeriesIndex !== undefined && barSeriesIndex !== seriesIndex)
      )
        continue;

      DvtChartPlotAreaRenderer._filterPointsForSeries(chart, seriesIndex);

      var pathCmd = ''; // path command for optimized rendering during zoom & scroll
      var itemStyleSet = chart.getOptionsCache().getFromCachedMap('itemStyleSet', seriesIndex);
      var stackCategory = DvtChartDataUtils.getStackCategory(chart, seriesIndex) || '';
      var bAssignedToY2 = DvtChartDataUtils.isAssignedToY2(chart, seriesIndex);

      var minMaxGroupIndex = DvtChartAxisUtils.getViewportMinMaxGroupIdx(chart, seriesIndex);
      for (
        var groupIndex = minMaxGroupIndex['min'] - 1;
        groupIndex <= minMaxGroupIndex['max'] + 1;
        groupIndex++
      ) {
        if (
          DvtChartDataUtils.isDataItemFiltered(chart, seriesIndex, groupIndex) ||
          !DvtChartDataUtils.isDataItemRendered(chart, seriesIndex, groupIndex)
        )
          continue;

        if (DvtChartDataUtils.getZVal(chart, seriesIndex, groupIndex) == 0) continue;

        var barInfo = DvtChartGroupUtils.getBarInfo(chart, seriesIndex, groupIndex, availSpace);
        if (barInfo == null) continue;

        // Get the y-axis position
        var yCoord = barInfo.yCoord;
        var baseCoord = barInfo.baseCoord;
        var axisCoord = barInfo.axisCoord;
        var x1 = barInfo.x1;
        var x2 = barInfo.x2;
        var barWidth = barInfo.barWidth;

        // For stack bars, don't render bar if it's height is less than .5px and
        // the gap between the previously rendered series is less than .5px
        if (isStacked) {
          var bNegative = DvtChartDataUtils.getVal(chart, seriesIndex, groupIndex) < 0;
          var stackKey = '' + groupIndex + stackCategory + bAssignedToY2 + bNegative;
          if (
            Math.abs(baseCoord - yCoord) < 0.5 &&
            Math.abs(yCoord - prevStackYCoords[stackKey]) < 0.5 &&
            !DvtChartDataUtils.isOutermostBar(chart, seriesIndex, groupIndex)
          )
            continue;
          else {
            if (prevStackYCoords[stackKey]) baseCoord = prevStackYCoords[stackKey];
            prevStackYCoords[stackKey] = yCoord;
          }
        }

        // Support for 0 value bars. If the bar is smaller than a pixel:
        // - draw as 1px bar if it's range series
        // - draw as an invisible 3px bar otherwise
        var bInvisible = false;
        if (Math.abs(yCoord - baseCoord) < 1) {
          if (DvtChartDataUtils.isRangeSeries(chart, seriesIndex)) yCoord--;
          else if (!isStacked || DvtChartDataUtils.isOutermostBar(chart, seriesIndex, groupIndex)) {
            bInvisible = true;
            if (yCoord > baseCoord || (bHoriz && !isR2L && yCoord == baseCoord))
              // if horizontal, R2L must be considered to draw bar on positive side of baseline
              yCoord = baseCoord + 3;
            else yCoord = baseCoord - 3;
          }
        }

        // Create and apply the style
        var shape;
        if (bPolar)
          shape = new DvtChartPolarBar(chart, axisCoord, baseCoord, yCoord, x1, x2, availSpace);
        else {
          // Optimize rendering during zoom & scroll by drawing all the bars belonging to one series as a single path.
          // If the items in the series have custom styles, we can't use this optimization.
          var bOptimize = duringZoomAndScroll && groupWidth < 5 && !itemStyleSet;
          shape = new DvtChartBar(chart, axisCoord, baseCoord, yCoord, x1, x2, bOptimize);
          if (bOptimize) {
            var bbox = shape.getBoundingBox();
            pathCmd +=
              PathUtils.moveTo(bbox.x, bbox.y) +
              PathUtils.horizontalLineTo(bbox.x + bbox.w) +
              PathUtils.verticalLineTo(bbox.y + bbox.h) +
              PathUtils.horizontalLineTo(bbox.x) +
              PathUtils.closePath();
            continue;
          }
        }
        container.addChild(shape);

        if (DvtChartStyleUtils.isSelectable(chart, seriesIndex, groupIndex))
          shape.setCursor(SelectionEffectUtils.getSelectingCursor());

        var fill,
          stroke = null;
        if (bInvisible)
          // Apply an invisible fill for small bars
          fill = SolidFill.invisibleFill();
        else {
          // Apply the specified style
          fill = DvtChartSeriesEffectUtils.getBarFill(chart, seriesIndex, groupIndex, barWidth);
          var borderColor = DvtChartStyleUtils.getBorderColor(chart, seriesIndex, groupIndex);
          var borderWidth = DvtChartStyleUtils.getBorderWidth(chart, seriesIndex, groupIndex);
          if (borderColor) stroke = new Stroke(borderColor, 1, borderWidth);
        }

        // Apply the fill, stroke, and selection colors
        var dataColor = DvtChartStyleUtils.getColor(chart, seriesIndex, groupIndex);
        var className = DvtChartStyleUtils.getClassName(chart, seriesIndex, groupIndex);
        var style = DvtChartStyleUtils.getStyle(chart, seriesIndex, groupIndex);
        shape.setStyleProperties(fill, stroke, dataColor, innerColor, outerColor, className, style);

        // Use pixel hinting for pixel bar spacing
        if (bPixelSpacing) shape.setPixelHinting(true);

        // Associate for interactivity
        DvtChartObjPeer.associate(shape, chart, seriesIndex, groupIndex, null, barInfo.dataPos);

        // Rendering data labels for this bar
        if (DvtChartDataUtils.isRangeSeries(chart, seriesIndex)) {
          DvtChartPlotAreaRenderer._renderDataLabel(
            chart,
            container,
            shape.getBoundingBox(),
            seriesIndex,
            groupIndex,
            null,
            dataColor,
            'low',
            false,
            shape.getOriginalBarSize()
          );
          DvtChartPlotAreaRenderer._renderDataLabel(
            chart,
            container,
            shape.getBoundingBox(),
            seriesIndex,
            groupIndex,
            null,
            dataColor,
            'high',
            false,
            shape.getOriginalBarSize()
          );
        } else
          DvtChartPlotAreaRenderer._renderDataLabel(
            chart,
            container,
            shape.getBoundingBox(),
            seriesIndex,
            groupIndex,
            null,
            dataColor,
            null,
            false,
            shape.getOriginalBarSize()
          );

        var markers = [];
        markers.push(shape);

        if (!(bStock && seriesIndex != 0)) {
          // TODO: change to formal location for displayed data
          chart._currentMarkers.push(markers);
        }

        // Render stack cumulative labels
        if (hasStackLabel && DvtChartDataUtils.isOutermostBar(chart, seriesIndex, groupIndex)) {
          DvtChartPlotAreaRenderer._renderDataLabel(
            chart,
            container,
            shape.getBoundingBox(),
            seriesIndex,
            groupIndex,
            null,
            null,
            null,
            true,
            shape.getOriginalBarSize()
          );
        }
      }

      if (pathCmd) {
        // If optimizing for zoom & scroll, create the path and set the series style properties
        var path = new Path(chart.getCtx(), pathCmd);

        var seriesColor = DvtChartStyleUtils.getColor(chart, seriesIndex);
        path.setSolidFill(seriesColor);

        var seriesBorderColor = DvtChartStyleUtils.getBorderColor(chart, seriesIndex);
        if (seriesBorderColor) {
          var seriesBorderWidth = DvtChartStyleUtils.getBorderWidth(chart, seriesIndex);
          path.setSolidStroke(seriesBorderColor, null, seriesBorderWidth);
        }

        var seriesClassName = DvtChartStyleUtils.getClassName(chart, seriesIndex);
        var seriesStyle = DvtChartStyleUtils.getStyle(chart, seriesIndex);
        path.setClassName(seriesClassName).setStyle(seriesStyle);

        if (bPixelSpacing) path.setPixelHinting(true);

        container.addChild(path);
      }
    }
  },

  /**
   * Renders all stock values and ranges for the given chart.
   * @param {Chart} chart
   * @param {dvt.Container} container The container to render to.
   * @private
   */
  _renderStock: (chart, container) => {
    var options = chart.getOptions();
    var xAxis = chart.xAxis;
    var yAxis = chart.yAxis;

    // Only a single series is supported right now
    if (DvtChartDataUtils.getSeriesType(chart, 0) != 'candlestick') return;

    // Iterate through the data
    var minMaxGroupIndex = DvtChartAxisUtils.getViewportMinMaxGroupIdx(chart, 0);
    for (
      var groupIndex = minMaxGroupIndex['min'] - 1;
      groupIndex <= minMaxGroupIndex['max'] + 1;
      groupIndex++
    ) {
      if (!DvtChartDataUtils.isDataItemRendered(chart, 0, groupIndex)) continue;

      var dataItem = DvtChartDataUtils.getDataItem(chart, 0, groupIndex);
      if (!dataItem) continue;

      var openValue = dataItem['open'];
      var closeValue = dataItem['close'];
      var lowValue = dataItem['low'];
      var highValue = dataItem['high'];

      var renderRange = lowValue != null && highValue != null;
      // Don't render bars whose value is null
      if (openValue == null || closeValue == null) continue;

      // Get the position on the x axis and the bar width.
      var xValue = DvtChartDataUtils.getXVal(chart, 0, groupIndex);
      var xCoord = xAxis.getUnboundedCoordAt(xValue);
      var barWidth = DvtChartStyleUtils.getBarWidth(chart, 0, groupIndex);

      // Get the position on the y axis
      var openCoord = yAxis.getBoundedCoordAt(openValue);
      var closeCoord = yAxis.getBoundedCoordAt(closeValue);
      var lowCoord,
        highCoord = null;
      if (renderRange) {
        lowCoord = yAxis.getBoundedCoordAt(lowValue);
        highCoord = yAxis.getBoundedCoordAt(highValue);
      }

      // Create the candlestick
      var candlestick = new DvtChartCandlestick(
        chart.getCtx(),
        xCoord,
        barWidth,
        openCoord,
        closeCoord,
        lowCoord,
        highCoord
      );
      container.addChild(candlestick);

      if (DvtChartStyleUtils.isSelectable(chart, 0, groupIndex))
        candlestick.setCursor(SelectionEffectUtils.getSelectingCursor());

      // Find the fill and stroke to be applied.
      var fill = DvtChartSeriesEffectUtils.getBarFill(chart, 0, groupIndex, barWidth);
      var stroke = null;
      var borderColor = DvtChartStyleUtils.getBorderColor(chart, 0, groupIndex);
      var borderWidth = DvtChartStyleUtils.getBorderWidth(chart, 0, groupIndex);
      if (borderColor) stroke = new Stroke(borderColor, 1, borderWidth);
      else if (fill instanceof PatternFill)
        // Patterns aren't usable here without the stroke.
        stroke = new Stroke(fill.getColor(), 1, borderWidth);

      // Find the hover and selected styles and pass to the candlestick.
      var dataColor = DvtChartStyleUtils.getColor(chart, 0, groupIndex);
      var innerColor = DvtChartStyleUtils.getSelectedInnerColor(chart);
      var outerColor = DvtChartStyleUtils.getSelectedOuterColor(chart);
      var rangeColor = options['styleDefaults']['stockRangeColor'];
      candlestick.setChangeStyle(fill, stroke, dataColor, innerColor, outerColor);
      candlestick.setRangeStyle(new SolidFill(rangeColor), stroke, rangeColor, outerColor);

      // Associate for interactivity
      var dataPos = new Point(xCoord, (openCoord + closeCoord) / 2);
      DvtChartObjPeer.associate(candlestick, chart, 0, groupIndex, null, dataPos);

      // TODO: Illegal private accesses for data cursor.
      // chart._currentMarkers will be removed when data cursor code is cleaned up
      var markers = [];
      markers.push(candlestick._changeShape);
      chart._currentMarkers.push(markers);
    }
  },

  /**
   * Renders box plot series for the given chart.
   * @param {Chart} chart
   * @param {dvt.Container} container The container to render to.
   * @param {dvt.Rectangle} availSpace
   * @private
   */
  _renderBoxPlot: (chart, container, availSpace, boxPlotSeriesIndex) => {
    var xAxis = chart.xAxis;

    for (
      var seriesIndex = 0;
      seriesIndex < DvtChartDataUtils.getSeriesCount(chart);
      seriesIndex++
    ) {
      if (boxPlotSeriesIndex !== undefined && boxPlotSeriesIndex !== seriesIndex) continue;
      var minMaxGroupIndex = DvtChartAxisUtils.getViewportMinMaxGroupIdx(chart, seriesIndex);

      for (
        var groupIndex = minMaxGroupIndex['min'] - 1;
        groupIndex <= minMaxGroupIndex['max'] + 1;
        groupIndex++
      ) {
        if (
          !DvtChartDataUtils.isDataItemRendered(chart, seriesIndex, groupIndex) ||
          DvtChartDataUtils.getSeriesType(chart, seriesIndex) != 'boxPlot'
        )
          continue;

        var dataItem = DvtChartDataUtils.getDataItem(chart, seriesIndex, groupIndex);
        if (!dataItem) continue;

        var lowValue = dataItem['low'];
        var q1Value = dataItem['q1'];
        var q2Value = dataItem['q2'];
        var q3Value = dataItem['q3'];
        var highValue = dataItem['high'];

        // Don't render anything if the required values aren't specified
        if (
          lowValue == null ||
          q1Value == null ||
          q2Value == null ||
          q3Value == null ||
          highValue == null
        )
          continue;

        // Get the position on the x axis and the box width

        var boxWidth = DvtChartStyleUtils.getBarWidth(chart, seriesIndex, groupIndex);
        var offsetMap = DvtChartStyleUtils.getBarCategoryOffsetMap(chart, groupIndex);
        var bAssignedToY2 = DvtChartDataUtils.isAssignedToY2(chart, seriesIndex);
        var category = DvtChartDataUtils.getStackCategory(chart, seriesIndex);
        var boxOffset = offsetMap[bAssignedToY2 ? 'y2' : 'y'][category] + boxWidth / 2;
        if (Agent.isRightToLeft(chart.getCtx()) && DvtChartTypeUtils.isVertical(chart))
          boxOffset *= -1;

        var xValue = DvtChartDataUtils.getXVal(chart, seriesIndex, groupIndex);
        var xCoord = xAxis.getUnboundedCoordAt(xValue) + boxOffset;

        // Get the position on the y axis
        var yAxis = bAssignedToY2 ? chart.y2Axis : chart.yAxis;
        var lowCoord = yAxis.getBoundedCoordAt(lowValue);
        var q1Coord = yAxis.getBoundedCoordAt(q1Value);
        var q2Coord = yAxis.getBoundedCoordAt(q2Value);
        var q3Coord = yAxis.getBoundedCoordAt(q3Value);
        var highCoord = yAxis.getBoundedCoordAt(highValue);

        // Create the candlestick
        var styleOptions = DvtChartStyleUtils.getBoxPlotStyleOptions(
          chart,
          seriesIndex,
          groupIndex
        );
        var boxAndWhisker = new DvtChartBoxAndWhisker(
          chart,
          xCoord,
          boxWidth,
          lowCoord,
          q1Coord,
          q2Coord,
          q3Coord,
          highCoord,
          styleOptions
        );
        container.addChild(boxAndWhisker);

        if (DvtChartStyleUtils.isSelectable(chart, seriesIndex, groupIndex))
          boxAndWhisker.setCursor(SelectionEffectUtils.getSelectingCursor());

        // Associate for interactivity
        var dataPos = DvtChartCoordUtils.convertAxisCoord(
          chart,
          new Point(xCoord, q2Coord),
          availSpace
        );
        DvtChartObjPeer.associate(boxAndWhisker, chart, seriesIndex, groupIndex, null, dataPos);

        DvtChartPlotAreaRenderer._renderBoxPlotMarkers(
          chart,
          container,
          seriesIndex,
          groupIndex,
          availSpace,
          xCoord
        );

        // TODO: Illegal private accesses for data cursor.
        // chart._currentMarkers will be removed when data cursor code is cleaned up
        chart._currentMarkers.push([boxAndWhisker._medianLine]);
      }
    }
  },

  /**
   * Renders box plot markers for the given data item.
   * @param {Chart} chart
   * @param {dvt.Container} container The container to render to.
   * @param {number} seriesIndex
   * @param {number} groupIndex
   * @param {dvt.Rectangle} availSpace
   * @param {number} xCoord The x coord of the markers.
   * @private
   */
  _renderBoxPlotMarkers: (chart, container, seriesIndex, groupIndex, availSpace, xCoord) => {
    // Calculate the default stroke to save DOM calls.
    var defaultBorderColor = DvtChartStyleUtils.getMarkerBorderColor(
      chart,
      seriesIndex,
      groupIndex
    );
    var defaultBorderWidth = DvtChartStyleUtils.getBorderWidth(chart, seriesIndex, groupIndex);
    var defaultStroke = new Stroke(defaultBorderColor, 1, defaultBorderWidth);

    // Keep track of the markers so that they can be sorted and added
    var markers = [];

    // Loop through the groups in the series
    var numItems = DvtChartDataUtils.getNestedDataItemCount(chart, seriesIndex, groupIndex);
    for (var itemIndex = 0; itemIndex < numItems; itemIndex++) {
      // Calculate the rendering info for the marker. If null is returned, skip.
      var markerInfo = DvtChartPlotAreaRenderer._getMarkerInfo(
        chart,
        seriesIndex,
        groupIndex,
        itemIndex,
        availSpace
      );
      if (!markerInfo) continue;

      // Box plot markers should also use the box offset, so pass it here instead of recalculating
      if (DvtChartTypeUtils.isHorizontal(chart)) markerInfo.y = xCoord;
      else markerInfo.x = xCoord;

      var marker = DvtChartPlotAreaRenderer._createMarker(
        chart,
        markerInfo,
        defaultBorderColor,
        defaultBorderWidth
      );
      if (marker != null) markers.push(marker);
    }

    DvtChartPlotAreaRenderer._addMarkersToContainer(chart, container, markers, defaultStroke);
  },

  /**
   * Renders all line series for the given chart.
   * @param {Chart} chart
   * @param {dvt.Container} container The container to render to.
   * @param {dvt.Container} clipGroup The group for clipping the line and the area.
   * @param {dvt.Rectangle} availSpace
   * @private
   */
  _renderLines: (chart, container, clipGroup, availSpace, lineSeriesIndex) => {
    // Find all series that are lines
    var lineSeries = [],
      seriesIndex;
    var seriesCount = DvtChartDataUtils.getSeriesCount(chart);
    for (seriesIndex = 0; seriesIndex < seriesCount; seriesIndex++) {
      // Skip the series if it shouldn't be rendered or if the series type is not line.
      if (
        !DvtChartDataUtils.isSeriesRendered(chart, seriesIndex) ||
        DvtChartDataUtils.getSeriesType(chart, seriesIndex) != 'line' ||
        (lineSeriesIndex !== undefined && lineSeriesIndex !== seriesIndex)
      )
        continue;
      else lineSeries.push(seriesIndex);
    }

    // Render the lines
    for (var lineIndex = 0; lineIndex < lineSeries.length; lineIndex++) {
      seriesIndex = lineSeries[lineIndex];

      if (DvtChartStyleUtils.getLineType(chart, seriesIndex) == 'none') continue;

      // Filter points to reduce render time
      DvtChartPlotAreaRenderer._filterPointsForSeries(chart, seriesIndex);

      DvtChartPlotAreaRenderer._renderLinesForSeries(chart, clipGroup, seriesIndex, availSpace);
    }

    // Render the markers
    for (lineIndex = 0; lineIndex < lineSeries.length; lineIndex++)
      DvtChartPlotAreaRenderer._renderMarkersForSeries(
        chart,
        container,
        lineSeries[lineIndex],
        availSpace
      );
  },

  /**
   * Renders all area series for the given chart.
   * @param {Chart} chart
   * @param {dvt.Container} container The container to render to.
   * @param {dvt.Rectangle} availSpace
   * @param {boolean} isLineWithArea True if rendering lineWithArea.
   * @private
   */
  _renderAreas: (chart, container, availSpace, isLineWithArea, areaSeriesIndex) => {
    // Find all series that are areas
    var seriesCount = DvtChartDataUtils.getSeriesCount(chart);
    var seriesType = isLineWithArea ? 'lineWithArea' : 'area';
    var yAreaSeries = [],
      y2AreaSeries = [];

    for (var seriesIndex = 0; seriesIndex < seriesCount; seriesIndex++) {
      // Skip the series if it shouldn't be rendered or if the series type is not area.
      if (
        !DvtChartDataUtils.isSeriesRendered(chart, seriesIndex) ||
        DvtChartDataUtils.getSeriesType(chart, seriesIndex) != seriesType ||
        (areaSeriesIndex !== undefined && areaSeriesIndex !== seriesIndex)
      )
        continue;

      if (DvtChartDataUtils.isAssignedToY2(chart, seriesIndex)) y2AreaSeries.push(seriesIndex);
      else yAreaSeries.push(seriesIndex);
    }

    if (yAreaSeries.length > 0)
      DvtChartPlotAreaRenderer._renderAreasForAxis(
        chart,
        container,
        yAreaSeries,
        chart.yAxis.getBaselineCoord(),
        availSpace,
        isLineWithArea
      );

    if (y2AreaSeries.length > 0)
      DvtChartPlotAreaRenderer._renderAreasForAxis(
        chart,
        container,
        y2AreaSeries,
        chart.y2Axis.getBaselineCoord(),
        availSpace,
        isLineWithArea
      );
  },

  /**
   * Renders all area series for the given y or y2 axis.
   * @param {Chart} chart
   * @param {dvt.Container} container The container to render to.
   * @param {array} areaSeries The series indices for the area series that belongs to the axis.
   * @param {number} baselineCoord The baseline coord of the axis.
   * @param {dvt.Rectangle} availSpace
   * @param {boolean} isLineWithArea True if rendering lineWithArea.
   * @private
   */
  _renderAreasForAxis: (
    chart,
    container,
    areaSeries,
    baselineCoord,
    availSpace,
    isLineWithArea
  ) => {
    var bStacked = DvtChartDataUtils.isStacked(chart);

    // Create group to clip the areas
    var clippedGroup = DvtChartPlotAreaRenderer.createClippedGroup(chart, container, availSpace);

    // For stacked areas, the bottom shape of the area has to follow the shape formed by the areas below it.
    // The prevCoords array stores the shape that has formed so far. For each group, it stores x, y1, and y2. When there's
    // a transition to/from null values, the shape at x jumps from y1 to y2. Otherwise, y1 == y2.
    var prevCoordsMap = {};
    var prevTypeMap = {};
    var prevCoordsMapNegative = {};
    var prevTypeMapNegative = {};

    // Construct baseline coords array to build the first area on the stack
    var baselineCoords = [];
    var groupCount = DvtChartDataUtils.getGroupCount(chart);
    for (var i = 0; i < groupCount; i++) {
      baselineCoords.push(
        new DvtChartCoord(
          null,
          baselineCoord,
          baselineCoord,
          i,
          DvtChartDataUtils.getGroup(chart, i),
          true
        )
      );
    }

    // Loop through the series
    for (var areaIndex = 0; areaIndex < areaSeries.length; areaIndex++) {
      var seriesIndex = areaSeries[areaIndex];
      var category = DvtChartDataUtils.getStackCategory(chart, seriesIndex);
      var isSeriesNegative = DvtChartDataUtils.isSeriesNegative(chart, seriesIndex);
      // Use previous coords of the last area on the current series' axis with the same stackCategory
      var prevCoords = prevCoordsMap[category];
      var prevType = prevTypeMap[category];
      var prevCoordsNegative = prevCoordsMapNegative[category];
      var prevTypeNegative = prevTypeMapNegative[category];

      if (DvtChartStyleUtils.getLineType(chart, seriesIndex) == 'none') {
        // render markers only
        DvtChartPlotAreaRenderer._renderMarkersForSeries(chart, container, seriesIndex, availSpace);
        continue;
      }

      var fill = DvtChartSeriesEffectUtils.getAreaFill(chart, seriesIndex);
      var borderColor = DvtChartStyleUtils.getBorderColor(chart, seriesIndex);
      var borderWidth = DvtChartStyleUtils.getBorderWidth(chart, seriesIndex);
      var className = DvtChartStyleUtils.getAreaClassName(chart, seriesIndex);
      var style = DvtChartStyleUtils.getAreaStyle(chart, seriesIndex);

      var stroke = borderColor ? new Stroke(borderColor, 1, borderWidth) : null;
      var type = DvtChartStyleUtils.getLineType(chart, seriesIndex);

      // Filter points to reduce render time
      DvtChartPlotAreaRenderer._filterPointsForSeries(chart, seriesIndex);

      var coords, baseCoords, baseType;
      var isRange = DvtChartDataUtils.isRangeSeries(chart, seriesIndex);
      if (isRange) {
        coords = DvtChartPlotAreaRenderer._getCoordsForSeries(
          chart,
          seriesIndex,
          availSpace,
          'high'
        );
        baseCoords = DvtChartPlotAreaRenderer._getCoordsForSeries(
          chart,
          seriesIndex,
          availSpace,
          'low'
        );
        baseType = type;
      } else {
        if (isSeriesNegative) {
          coords = DvtChartPlotAreaRenderer._getAreaCoordsForSeries(
            chart,
            seriesIndex,
            availSpace,
            prevCoordsNegative ? prevCoordsNegative : baselineCoords
          );
          baseCoords = prevCoordsNegative ? prevCoordsNegative : [];
          baseType = prevTypeNegative;
        } else {
          coords = DvtChartPlotAreaRenderer._getAreaCoordsForSeries(
            chart,
            seriesIndex,
            availSpace,
            prevCoords ? prevCoords : baselineCoords
          );
          baseCoords = prevCoords ? prevCoords : [];
          baseType = prevType;
        }
      }

      var area = new DvtChartLineArea(
        chart,
        true,
        availSpace,
        baselineCoord,
        style,
        className,
        fill,
        stroke,
        type,
        coords,
        baseType,
        baseCoords
      );
      clippedGroup.addChild(area);
      chart._currentAreas.push(area); // TODO: change to formal API for storage
      DvtChartObjPeer.associate(area, chart, seriesIndex); // Associate for interactivity

      // Store for the base of the next area in the stack
      if (isSeriesNegative) {
        prevCoordsMapNegative[category] = coords;
        prevTypeMapNegative[category] = type;
      } else {
        prevCoordsMap[category] = coords;
        prevTypeMap[category] = type;
      }

      // If not stacked, draw with each series so that lines and markers don't bleed through
      if (!bStacked) {
        // Draw line as data item gap only if the area doesn't have border. Otherwise the gap will be drawn on top of the border
        if (isLineWithArea || (DvtChartStyleUtils.getDataItemGaps(chart) > 0 && !borderColor))
          DvtChartPlotAreaRenderer._renderLinesForSeries(
            chart,
            clippedGroup,
            seriesIndex,
            availSpace,
            !isLineWithArea
          );

        // For lineWithArea, always draw the markers last since the areas are semi-transparent
        if (!isLineWithArea)
          DvtChartPlotAreaRenderer._renderMarkersForSeries(
            chart,
            container,
            seriesIndex,
            availSpace
          );

        // : New group generated for the next area so that, in unstacked charts, the clipGroup for each area is
        // added after the markers of previous series
        if (areaIndex + 1 < areaSeries.length)
          clippedGroup = DvtChartPlotAreaRenderer.createClippedGroup(chart, container, availSpace);
      }
    }

    // If stacked, draw lines and markers at the end so that the stacked areas don't overlap them
    for (areaIndex = 0; areaIndex < areaSeries.length; areaIndex++) {
      seriesIndex = areaSeries[areaIndex];
      if (DvtChartStyleUtils.getLineType(chart, seriesIndex) == 'none') continue; // markers are already rendered in previous loop

      // Draw line as data item gap only if the area doesn't have border. Otherwise the gap will be drawn on top of the border
      var hasBorder =
        DvtChartStyleUtils.getBorderColor(chart, seriesIndex) ||
        DvtChartStyleUtils.getBorderColor(chart, seriesIndex + 1);
      if (
        bStacked &&
        (isLineWithArea || (DvtChartStyleUtils.getDataItemGaps(chart) > 0 && !hasBorder))
      )
        DvtChartPlotAreaRenderer._renderLinesForSeries(
          chart,
          clippedGroup,
          seriesIndex,
          availSpace,
          !isLineWithArea
        );

      // Also draw markers last for unstacked lineWithArea so people can interact with them through the semi-transparent areas
      if (bStacked || isLineWithArea)
        DvtChartPlotAreaRenderer._renderMarkersForSeries(chart, container, seriesIndex, availSpace);
    }
  },

  /**
   * Returns the coordinates for the specified area series.
   * @param {Chart} chart The chart being rendered.
   * @param {number} seriesIndex The series being rendered.
   * @param {dvt.Rectangle} availSpace The available space.
   * @param {array} prevCoords The array of DvtChartCood from the previous series.
   * @return {array} The arrays of DvtChartCoord for the current series.
   * @private
   */
  _getAreaCoordsForSeries: (chart, seriesIndex, availSpace, prevCoords) => {
    var rawCoords = DvtChartPlotAreaRenderer._getCoordsForSeries(chart, seriesIndex, availSpace);
    var coords = [];
    for (var i = 0; i < prevCoords.length; i++) coords.push(prevCoords[i].clone());

    // Construct the coords based on the prevCoords.
    // At each point, if it's not null:
    // - if the previous point is null, update only the y2 so that the shape jumps up from the old y1 to the new y2;
    // - if the next point is null, update only the y1 so that the shape jumps down from the new y1 to the old y2;
    // - else, y1 == y2 and there's no jump at that point.
    var lastIndex = rawCoords.length - 1;
    var bPolar = DvtChartTypeUtils.isPolar(chart);
    for (var z = 0; z < rawCoords.length; z++) {
      if (rawCoords[z].x != null) {
        var coord = coords[rawCoords[z].groupIndex];
        var prevIndex = bPolar && z == 0 ? lastIndex : z - 1; // prev/nextIndex in polar is circular
        var nextIndex = bPolar && z == lastIndex ? 0 : z + 1;

        if (prevIndex >= 0 && rawCoords[prevIndex].x != null) coord.y1 = rawCoords[z].y1;
        if (nextIndex <= lastIndex && rawCoords[nextIndex].x != null) coord.y2 = rawCoords[z].y2;

        coord.x = rawCoords[z].x;

        // If y1!=y2, there's a jump so the point can't be filtered
        coord.filtered = coord.y1 == coord.y2 ? rawCoords[z].filtered : false;
      }
    }

    return coords;
  },

  /**
   * Create lines for a series (for line and lineWithArea) and add them to the container.
   * @param {Chart} chart
   * @param {dvt.Container} container The line container.
   * @param {Number} seriesIndex
   * @param {dvt.Rectangle} availSpace
   * @param {Boolean} isDataItemGap Whether this line is for drawing area gap.
   * @private
   */
  _renderLinesForSeries: (chart, container, seriesIndex, availSpace, isDataItemGap) => {
    // Get the style info
    var stroke;
    if (isDataItemGap) {
      var gapSize = DvtChartStyleUtils.getDataItemGaps(chart) * 2.5;
      stroke = new Stroke(DvtChartStyleUtils.getBackgroundColor(chart, true), 1, gapSize);
    } else {
      var color = DvtChartStyleUtils.getColor(chart, seriesIndex);
      var lineWidth = DvtChartStyleUtils.getLineWidth(chart, seriesIndex);
      var lineStyle = DvtChartStyleUtils.getLineStyle(chart, seriesIndex);
      stroke = new Stroke(
        color,
        1,
        lineWidth,
        false,
        Stroke.getDefaultDashProps(lineStyle, lineWidth)
      );
    }

    // Create the lines
    var baseline = DvtChartDataUtils.isAssignedToY2(chart, seriesIndex)
      ? chart.y2Axis.getBaselineCoord()
      : chart.yAxis.getBaselineCoord();
    var lineType = DvtChartStyleUtils.getLineType(chart, seriesIndex);
    var className = DvtChartStyleUtils.getClassName(chart, seriesIndex);
    var style = DvtChartStyleUtils.getStyle(chart, seriesIndex);

    var renderLine = (type) => {
      var coords = DvtChartPlotAreaRenderer._getCoordsForSeries(
        chart,
        seriesIndex,
        availSpace,
        type
      );
      var line = new DvtChartLineArea(
        chart,
        false,
        availSpace,
        baseline,
        style,
        className,
        null,
        stroke,
        lineType,
        coords
      );
      container.addChild(line);
      DvtChartObjPeer.associate(line, chart, seriesIndex); // Associate for interactivity
    };

    var isRange = DvtChartDataUtils.isRangeSeries(chart, seriesIndex);
    if (isRange) {
      renderLine('high');
      renderLine('low');
    } else renderLine('value');
  },

  /**
   * Filters the bubble and scatter markers for performance.
   * @param {Chart} chart The chart being rendered.
   * @param {boolean} bSortBySize True if markers should be sorted by size to reduce overlaps.
   * @return {array} The array of visible markerInfo objects, or null if filtering was not performed.
   * @private
   */
  _filterScatterBubble: (chart, bSortBySize) => {
    // Note: This function works by determining and marking the data to be filtered. When render of the data is called
    // afterwards, the filtering flag causes the item to be skipped.
    var seriesCount = DvtChartDataUtils.getSeriesCount(chart);
    var groupCount = DvtChartDataUtils.getGroupCount(chart);

    // Filter only if there are a sufficient number of data points. The filtering cost has overhead based on the size of
    // the availSpace, and is not generally worthwhile until several thousand markers can be skipped.
    if (seriesCount * groupCount < DvtChartPlotAreaRenderer.FILTER_THRESHOLD_SCATTER_BUBBLE)
      return null;

    // Gather the marker information objects for all data items.
    var markerInfo;
    var markerInfos = [];
    for (var seriesIndex = 0; seriesIndex < seriesCount; seriesIndex++) {
      // Skip the series if it shouldn't be rendered
      if (!DvtChartDataUtils.isSeriesRendered(chart, seriesIndex)) continue;

      for (var groupIndex = 0; groupIndex < groupCount; groupIndex++) {
        markerInfo = DvtChartPlotAreaRenderer._getScatterBubbleMarkerInfo(
          chart,
          seriesIndex,
          groupIndex
        );
        if (markerInfo != null) markerInfos.push(markerInfo);
      }
    }

    // Check the data count again now that obscured series accounted for.
    if (markerInfos.length < DvtChartPlotAreaRenderer.FILTER_THRESHOLD_SCATTER_BUBBLE) return null;

    if (bSortBySize) DvtChartMarkerUtils.sortMarkerInfos(markerInfos);

    // Create the data structure to track if pixels are occupied.
    var pixelMap = new PixelMap(25, new PixelMap(5, new PixelMap()));

    // Loop backwards to determine if objects are visible. Create an array of visible markers.
    var ret = [];
    var markerCount = markerInfos.length;
    for (var markerIndex = markerCount - 1; markerIndex >= 0; markerIndex--) {
      markerInfo = markerInfos[markerIndex];

      // Check if the pixels are occupied and adjust the filtered flag.
      var bObscured = DvtChartMarkerUtils.checkPixelMap(
        pixelMap,
        markerInfo.x,
        markerInfo.y,
        markerInfo.size
      );
      if (!bObscured) {
        var dataColor = DvtChartStyleUtils.getColor(
          chart,
          markerInfo.seriesIndex,
          markerInfo.groupIndex
        );
        var markerDisplayed = DvtChartStyleUtils.isMarkerDisplayed(chart, seriesIndex, groupIndex);
        var alpha = markerDisplayed ? ColorUtils.getAlpha(dataColor) : 0;
        if (alpha > 0) {
          // Update the pixel map for this marker
          DvtChartMarkerUtils.updatePixelMap(
            pixelMap,
            markerInfo.x,
            markerInfo.y,
            markerInfo.size,
            alpha
          );
          ret.push(markerInfo);
        }
      }
    }

    // Set the flag in the cache so that marquee selection will be based on the data rather than rendered objects.
    chart.getCache().putToCache('dataFiltered', true);

    // Markers were added top to bottom. Reverse for rendering order and return.
    ret.reverse();
    return ret;
  },

  /**
   * Filters the data points for line/area so that no more than one point is drawn per pixel.
   * @param {Chart} chart
   * @param {number} seriesIndex
   * @private
   */
  _filterPointsForSeries: (chart, seriesIndex) => {
    // TODO: implement filtering for range series
    if (DvtChartTypeUtils.isPolar(chart) || DvtChartDataUtils.isRangeSeries(chart, seriesIndex))
      return;

    var plotAreaDims = chart.__getPlotAreaSpace();
    var maxNumPts = DvtChartTypeUtils.isHorizontal(chart) ? plotAreaDims.h : plotAreaDims.w; // one point per pixel
    var seriesItems = DvtChartDataUtils.getSeriesItem(chart, seriesIndex)['items'];
    var isBar = DvtChartDataUtils.getSeriesType(chart, seriesIndex) == 'bar';
    var axisInfo = chart.xAxis.getInfo();
    var zoomFactor =
      (axisInfo.getDataMax() - axisInfo.getDataMin()) /
      (axisInfo.getViewportMax() - axisInfo.getViewportMin());
    var dataItem;
    // For line/area, pick two points (max and min) from each set, so each set can cover two pixels.
    // For bar, only pick the max for each set, so the set size should only cover 1/2 pixel to prevent sparse plot area.
    var numPixelsPerSet = isBar ? 0.5 : 2;
    var setSize = zoomFactor
      ? Math.round((numPixelsPerSet * (seriesItems.length / zoomFactor)) / maxNumPts)
      : 1;

    var minSetSize = isBar ? 2 : 3;
    if (setSize < minSetSize) {
      // Nothing should be filtered. Clear _filtered flags from previous rendering.
      for (var i = 0; i < seriesItems.length; i++) {
        dataItem = seriesItems[i];
        if (dataItem) dataItem['_filtered'] = false;
      }
      return;
    }

    var maxIndex, maxValue, minIndex, minValue, dataValue;
    var filtered = false;
    for (var y = 0; y < seriesItems.length; y += setSize) {
      maxIndex = -1;
      maxValue = -Infinity;
      minIndex = -1;
      minValue = Infinity;

      // Find the extreme points (min/max) of the set
      for (var j = y; j < Math.min(y + setSize, seriesItems.length); j++) {
        dataValue = DvtChartDataUtils.getCumulativeVal(chart, seriesIndex, j);
        dataItem = seriesItems[j];
        if (dataValue == null || dataItem == null) continue;
        if ((!isBar || dataValue > 0) && dataValue > maxValue) {
          // for bar, unfilter the maxValue only if the dataValue is positive
          maxIndex = j;
          maxValue = dataValue;
        }
        if ((!isBar || dataValue < 0) && dataValue < minValue) {
          // for bar, unfilter the minValue only if the dataValue is negative
          minIndex = j;
          minValue = dataValue;
        }
        dataItem['_filtered'] = true; // Filter every point in the meanwhile
        filtered = true;
      }

      // Unfilter the extreme points of the set
      for (var w = y; w < Math.min(y + setSize, seriesItems.length); w++) {
        dataItem = seriesItems[w];
        if (dataItem == null) continue;
        if (w == maxIndex || w == minIndex) dataItem['_filtered'] = false;
      }
    }

    chart.getCache().putToCache('dataFiltered', filtered);
  },

  /**
   * Creates and returns the coordinates for the specified series.
   * @param {Chart} chart The chart being rendered.
   * @param {number} seriesIndex The series being rendered.
   * @param {dvt.Rectangle} availSpace The available space.
   * @param {string} type (optional) The value type: 'value' (default), 'low', or 'high'.
   * @return {array} The array of DvtChartCoord.
   * @private
   */
  _getCoordsForSeries: (chart, seriesIndex, availSpace, type) => {
    var xAxis = chart.xAxis;
    var yAxis = chart.yAxis;
    if (DvtChartDataUtils.isAssignedToY2(chart, seriesIndex)) yAxis = chart.y2Axis;

    // Loop through the groups
    var coords = [];
    var minMaxGroupIndex = DvtChartAxisUtils.getViewportMinMaxGroupIdx(chart, seriesIndex);
    for (
      var groupIndex = minMaxGroupIndex['min'] - 1;
      groupIndex <= minMaxGroupIndex['max'] + 1;
      groupIndex++
    ) {
      var group = DvtChartDataUtils.getGroup(chart, groupIndex);
      if (group == null) continue;

      // Get the axis values
      var xValue = DvtChartDataUtils.getXVal(chart, seriesIndex, groupIndex);

      var yValue = null;
      if (type == 'low') yValue = DvtChartDataUtils.getLowVal(chart, seriesIndex, groupIndex);
      else if (type == 'high')
        yValue = DvtChartDataUtils.getHighVal(chart, seriesIndex, groupIndex);
      else if (DvtChartDataUtils.getVal(chart, seriesIndex, groupIndex) != null)
        yValue = DvtChartDataUtils.getCumulativeVal(chart, seriesIndex, groupIndex);

      // A null or undefined value begins another line or area and skips this data item
      if (
        yValue == null ||
        isNaN(yValue) ||
        !DvtChartDataUtils.isDataItemRendered(chart, seriesIndex, groupIndex)
      ) {
        // Skip this value since it's invalid
        coords.push(new DvtChartCoord(null, null, null, groupIndex, group, false));
        continue;
      }

      if (DvtChartTypeUtils.isPolar(chart))
        // The yValue for polar shouldn't go below the minimum because it will appear on the opposite side of the chart
        yValue = Math.max(yValue, yAxis.getInfo().getViewportMin());

      // Get the position on the axis
      var xCoord = xAxis.getUnboundedCoordAt(xValue);
      var yCoord = yAxis.getUnboundedCoordAt(yValue);

      if (xCoord == null || yCoord == null) {
        // Skip this value since it's invalid
        coords.push(new DvtChartCoord(null, null, null, groupIndex, group, false));
        continue;
      }

      var coord = new DvtChartCoord(
        xCoord,
        yCoord,
        yCoord,
        groupIndex,
        group,
        DvtChartDataUtils.isDataItemFiltered(chart, seriesIndex, groupIndex)
      );
      coords.push(coord);
    }

    return coords;
  },

  /**
   * Creates a container for plot area foreground objects with clipping.
   * @param {Chart} chart The chart being rendered.
   * @param {dvt.Container} container The container to render to.
   * @param {dvt.Rectangle} availSpace The available space.
   * @return {dvt.Container} The clipped container for plot area foreground objects.
   */
  createClippedGroup: (chart, container, availSpace) => {
    var clipGroup = new Container(container.getCtx());
    container.addChild(clipGroup);
    var clip = new ClipPath(chart.getId());

    //  - line gets cut off at plot area max
    //  - line charts vertex clipped if dataMax too close to globalMax
    // For a chart with a line/linewitharea series, increase clip group by several pixels if lines risk getting clipped
    var buffer = DvtChartPlotAreaRenderer._extendClipGroup(chart);

    // Add clip path depending on plot area shape
    if (DvtChartTypeUtils.isPolar(chart)) {
      var cx = availSpace.x + availSpace.w / 2;
      var cy = availSpace.y + availSpace.h / 2;
      if (DvtChartAxisUtils.isGridPolygonal(chart)) {
        var points = PolygonUtils.getRegularPolygonPoints(
          cx,
          cy,
          DvtChartDataUtils.getGroupCount(chart),
          chart.getRadius(),
          0
        );
        clip.addPolygon(points);
      } else clip.addCircle(cx, cy, chart.getRadius());
    } else if (DvtChartTypeUtils.isHorizontal(chart))
      clip.addRect(availSpace.x - buffer, availSpace.y, availSpace.w + 2 * buffer, availSpace.h);
    else clip.addRect(availSpace.x, availSpace.y - buffer, availSpace.w, availSpace.h + 2 * buffer);

    clipGroup.setClipPath(clip);
    return clipGroup;
  },

  /**
   * Checks or not the plot area should extend it's clip group, and returns how many pixels should be added.
   * Used for preventing clipping of line series at edge cases.
   * @param {Chart} chart
   * @return {number}
   * @private
   */
  _extendClipGroup: (chart) => {
    var hasLineSeries =
      DvtChartDataUtils.hasLineSeries(chart) || DvtChartDataUtils.hasLineWithAreaSeries(chart);
    if (hasLineSeries && !DvtChartTypeUtils.isSpark(chart)) {
      // Spark chart is already extended
      var lineWidth = DvtChartStyleUtils.getLineWidth(chart);
      var hasEdgeData = (axis) => {
        var axisInfo = axis.getInfo();
        var globalMaxCoord = axisInfo.getCoordAt(axisInfo.getGlobalMax());
        var dataMaxCoord = axisInfo.getCoordAt(axisInfo.getDataMax());
        var globalMinCoord = axisInfo.getCoordAt(axisInfo.getGlobalMin());
        var dataMinCoord = axisInfo.getCoordAt(axisInfo.getDataMin());

        if (
          globalMaxCoord != null &&
          dataMaxCoord != null &&
          dataMaxCoord - globalMaxCoord <= lineWidth / 2
        )
          return true;
        if (
          globalMinCoord != null &&
          dataMinCoord != null &&
          globalMinCoord - dataMinCoord <= lineWidth / 2
        )
          return true;

        return false;
      };

      if ((chart.yAxis && hasEdgeData(chart.yAxis)) || (chart.y2Axis && hasEdgeData(chart.y2Axis)))
        return Math.ceil(lineWidth / 2);
    }

    return 0;
  }
};

/*---------------------------------------------------------------------*/
/*  DvtChartDataCursorHandler                 Data Cursor Event Handler                  */
/*---------------------------------------------------------------------*/
/**
 *  @class  DvtChartDataCursorHandler
 *  @extends {dvt.Obj}
 *  @constructor
 */
class DvtChartDataCursorHandler {
  constructor(chart, dataCursor) {
    this._context = chart.getCtx();
    this._dataCursorShown = false;
    this._dataCursor = dataCursor;
    this._chart = chart;
  }

  // Show/hide the data cursor based on the global page coordinates of the action
  // Returns whether or not data cursor is shown
  processMove(pos, bSuppressEvent) {
    var plotRect = this._chart.__getPlotAreaSpace();
    if (
      plotRect &&
      plotRect.containsPoint(pos.x, pos.y) &&
      !this._chart.getOptions()['_duringZoomAndScroll']
    ) {
      // Show the data cursor only if the current point is within the plot area
      this._showDataCursor(plotRect, pos.x, pos.y, bSuppressEvent);
      return true;
    } else {
      this._removeDataCursor(bSuppressEvent);
    }
    return false;
  }

  processEnd(bSuppressEvent) {
    this._removeDataCursor(bSuppressEvent);
  }

  processOut(pos, bSuppressEvent) {
    var plotRect = this._chart.__getPlotAreaSpace();
    if (plotRect && !plotRect.containsPoint(pos.x, pos.y)) {
      this._removeDataCursor(bSuppressEvent);
    }
  }

  /**
   * Displays the data cursor.
   * @param {dvt.Rectangle} plotRect The bounds of the plot area
   * @param {number} x
   * @param {number} y
   * @param {object} targetObj
   * @private
   */
  _showDataCursor(plotRect, x, y, bSuppressEvent) {
    if (this._context.isOffscreen(true)) {
      this._removeDataCursor(bSuppressEvent);
      return;
    }

    var dataCursor = this._dataCursor;

    // Find the closest data point
    var closestMatch = this._getClosestMatch(x, y);
    if (closestMatch == null) {
      this._removeDataCursor(bSuppressEvent);
      return;
    }

    // Find the center of the data item
    var centerPoint = closestMatch.matchRegion.getCenter();

    var dcX = x;
    var dcY = y;
    // Adjust for snap behavior
    if (dataCursor.getBehavior() === 'snap') {
      if (dataCursor.isHorizontal())
        dcY = Math.min(Math.max(centerPoint.y, plotRect.y), plotRect.y + plotRect.h);
      else dcX = Math.min(Math.max(centerPoint.x, plotRect.x), plotRect.x + plotRect.w);
    }

    // If "dataCursor" attr is "auto", don't show the data cursor if tooltip text is null. Otherwise, always show the cursor.
    var logicalObject = closestMatch.logicalObject;
    var tooltipText = DvtChartTooltipUtils.getDatatip(
      this._chart,
      logicalObject.getSeriesIndex(),
      logicalObject.getGroupIndex(),
      logicalObject.getNestedDataItemIndex(),
      true
    );
    if (tooltipText == null) {
      dataCursor.setVisible(false);
      return;
    } else dataCursor.setVisible(true);

    var itemColor = DvtChartTooltipUtils.getDatatipColor(
      this._chart,
      logicalObject.getSeriesIndex(),
      logicalObject.getGroupIndex(),
      logicalObject.getNestedDataItemIndex()
    );
    var lineCoord = dataCursor.isHorizontal() ? dcY : dcX;
    var shape = logicalObject.getDisplayables()[0];
    // update the default maker shape and size
    if (shape instanceof SimpleMarker) {
      dataCursor.addMarker(shape.getType(), shape.getSize());
    }
    dataCursor.render(plotRect, centerPoint.x, centerPoint.y, lineCoord, tooltipText, itemColor);

    this._dataCursorShown = true;

    // fire optionChange event
    if (!bSuppressEvent) {
      var values = this._chart.getValsAt(x, y);
      this._chart.changeOption('dataCursorPosition', values);
    }
  }

  // Remove the data cursor
  _removeDataCursor(bSuppressEvent) {
    if (this._dataCursor.getVisible()) this._dataCursor.setVisible(false);

    this._context.getTooltipManager(DvtChartDataCursor.TOOLTIP_ID).hideTooltip();

    this._dataCursorShown = false;

    // fire optionChange event
    if (!bSuppressEvent) this._chart.changeOption('dataCursorPosition', null);
  }

  isDataCursorShown() {
    return this._dataCursorShown;
  }

  static _getClosestMatchSecondDirection(matchesInBounds, horizontal, x, y) {
    var closestMatch = null;
    var minDiff = Infinity;
    for (var i = matchesInBounds.length - 1; 0 <= i; i--) {
      var match = matchesInBounds[i];
      var lowerBound = horizontal ? match.matchRegion.x : match.matchRegion.y;
      var higherBound = horizontal
        ? match.matchRegion.x + match.matchRegion.w
        : match.matchRegion.y + match.matchRegion.h;
      var value = horizontal ? x : y;
      var midPoint = (lowerBound + higherBound) / 2;
      var diffValue = Math.round(Math.abs(midPoint - value));
      if (diffValue < minDiff) {
        minDiff = diffValue;
        closestMatch = match;
      }
    }
    return closestMatch;
  }

  static _getClosestMatchesFirstDirection(matches, horizontal, x, y, isHighlightMatched) {
    var minDiff = Infinity;
    var closestFirstDirectionMatches = new Array();
    // Get closest matches
    for (var i = 0; i < matches.length; i++) {
      var matchObj = matches[i];
      if (isHighlightMatched(matchObj.logicalObject)) {
        var matchRegion = matchObj.matchRegion;
        var lowerBound = horizontal ? matchRegion.y : matchRegion.x;
        var higherBound = horizontal
          ? matchRegion.y + matchRegion.h
          : matchRegion.x + matchRegion.w;
        var value = horizontal ? y : x;

        var midPoint = (lowerBound + higherBound) / 2;
        var diffValue = Math.round(Math.abs(midPoint - value));
        if (diffValue <= minDiff) {
          if (diffValue < minDiff) {
            closestFirstDirectionMatches = new Array();
          }
          closestFirstDirectionMatches.push(matchObj);
          minDiff = diffValue;
        }
      }
    }
    return closestFirstDirectionMatches;
  }

  // TODO JSDOC: This class needs to be rewritten to not access private properties and get rid of these implicit object returns.
  _findMatches() {
    var stage = this._context.getStage();
    var eventManager = this._chart.getEventManager();
    var matches = [];

    if (!this._chart._currentMarkers) return null;

    for (var i = 0; i < this._chart._currentMarkers.length; i++) {
      var markers = this._chart._currentMarkers[i];
      var numMarkers = markers.length;

      for (var idx = 0; idx < numMarkers; idx++) {
        var item = markers[idx];
        var logicalObject = eventManager.GetLogicalObject(item);

        // Find the bounding box of the item.  We use getDimensionsSelf, an optimized version of getDimensions, where
        // possible.  It's safe to use either API since chart data objects do not have children.
        var dims = item.getDimensionsSelf
          ? item.getDimensionsSelf(stage)
          : item.getDimensions(stage);

        var match = { matchRegion: dims, logicalObject: logicalObject };
        matches.push(match);
      }
    }
    return matches;
  }

  _getClosestMatch(x, y) {
    var horizontal = DvtChartTypeUtils.isHorizontal(this._chart);
    var useAllInGroup =
      DvtChartTypeUtils.isLineArea(this._chart) && !DvtChartTypeUtils.isMixedFrequency(this._chart);

    var matches = this._findMatches();

    var highlightedCategories = DvtChartDataUtils.getHighlightedCategories(this._chart);
    var isHighlightMatchAll = this._chart.getOptions()['highlightMatch'] == 'all';
    var matchFound =
      highlightedCategories.length > 0
        ? isHighlightMatchAll
          ? ArrayUtils.hasAllItems
          : ArrayUtils.hasAnyItem
        : null;
    var isHighlightMatched = (obj) => {
      return matchFound ? matchFound(obj.getCategories(), highlightedCategories) : true;
    };

    var matchesInBounds = DvtChartDataCursorHandler._getClosestMatchesFirstDirection(
      matches,
      horizontal,
      x,
      y,
      isHighlightMatched
    );

    // Non-numerical x axis
    if (!DvtChartTypeUtils.isScatterBubble(this._chart)) {
      var closestLowerBound = Infinity;
      var closestHigherBound = -Infinity;
      var closestGroup = null;

      for (var i = 0; i < matchesInBounds.length; i++) {
        var closestFirstDirectionMatch = matchesInBounds[i];
        closestLowerBound = Math.min(
          closestLowerBound,
          horizontal
            ? closestFirstDirectionMatch.matchRegion.y
            : closestFirstDirectionMatch.matchRegion.x
        );
        closestHigherBound = Math.max(
          closestHigherBound,
          horizontal
            ? closestFirstDirectionMatch.matchRegion.y + closestFirstDirectionMatch.matchRegion.h
            : closestFirstDirectionMatch.matchRegion.x + closestFirstDirectionMatch.matchRegion.w
        );
        closestGroup = closestFirstDirectionMatch.logicalObject.getGroupIndex();
      }

      for (var i = 0; i < matches.length; i++) {
        var match = matches[i];
        if (isHighlightMatched(match.logicalObject)) {
          if (useAllInGroup) {
            if (match.logicalObject.getGroupIndex() == closestGroup) {
              matchesInBounds.push(match);
            }
          } else {
            var lowerBound = horizontal ? match.matchRegion.y : match.matchRegion.x;
            var higherBound = horizontal
              ? match.matchRegion.y + match.matchRegion.h
              : match.matchRegion.x + match.matchRegion.w;
            var midPoint = (lowerBound + higherBound) / 2;
            if (closestHigherBound >= midPoint && closestLowerBound <= midPoint) {
              matchesInBounds.push(match);
            }
          }
        }
      }
    }
    return DvtChartDataCursorHandler._getClosestMatchSecondDirection(
      matchesInBounds,
      horizontal,
      x,
      y
    );
  }
}

/**
 * Axis component for use by Chart.  This class exposes additional functions needed for
 * rendering grid lines and data items.
 * @param {dvt.Context} context The rendering context.
 * @param {string} callback The function that should be called to dispatch component events.
 * @param {object} callbackObj The optional object instance on which the callback function is defined.
 * @class
 * @constructor
 * @extends {DvtAxis}
 */

class DvtChartAxis extends DvtAxis {
  /**
   * Converts the axis coord to plot area coord.
   * @param {number} coord Axis coord.
   * @return {number} Plot area coord.
   */
  axisToPlotArea(coord) {
    if (this.getOptions()['position'] == 'tangential') return coord;

    if (coord == null) return null;

    var ret = coord - this.getLeftOverflow();

    // Round to 1 decimal to keep the DOM small, but prevent undesidered gaps due to rounding errors
    return Math.round(ret * 10) / 10;
  }

  /**
   * Converts the plot area coord to axis coord.
   * @param {number} coord Plot area coord.
   * @param {boolean=} bRoundResult Whether the resulting coordinate is rounded to the nearest pixel.  Defaults to true.
   * @return {number} Axis coord.
   */
  plotAreaToAxis(coord, bRoundResult) {
    if (this.getOptions()['position'] == 'tangential') return coord;

    if (coord == null) return null;

    var ret = coord + this.getLeftOverflow();
    return bRoundResult === false ? ret : Math.round(ret);
  }

  /**
   * Converts linear value to actual value.
   * For example, for a log scale, the linear value is the log of the actual value.
   * @param {number} value The linear value.
   * @return {number} The actual value.
   */
  linearToActual(value) {
    return this.Info.linearToActual(value);
  }

  /**
   * Converts actual value to linear value.
   * For example, for a log scale, the linear value is the log of the actual value.
   * @param {number} value The actual value.
   * @return {number} The linear value.
   */
  actualToLinear(value) {
    return this.Info.actualToLinear(value);
  }

  /**
   * Returns the value for the specified coordinate along the axis.  Returns null
   * if the coordinate is not within the axis.
   * @param {number} coord The coordinate along the axis.
   * @return {object} The value at that coordinate.
   */
  getValAt(coord) {
    return this.Info.getValAt(this.plotAreaToAxis(coord));
  }

  /**
   * Returns the coordinate for the specified value.
   * @param {object} value The value to locate.
   * @return {number} The coordinate for the value.
   */
  getCoordAt(value) {
    return this.axisToPlotArea(this.Info.getCoordAt(value));
  }

  /**
   * Returns the coordinate for the specified value.  If a value is not within the axis,
   * returns the coordinate of the closest value within the axis.
   * @param {object} value The value to locate.
   * @return {number} The coordinate for the value.
   */
  getBoundedCoordAt(value) {
    return this.axisToPlotArea(this.Info.getBoundedCoordAt(value));
  }

  /**
   * Returns the value for the specified coordinate along the axis.
   * @param {number} coord The coordinate along the axis.
   * @return {object} The value at that coordinate.
   */
  getUnboundedValAt(coord) {
    return this.Info.getUnboundedValAt(this.plotAreaToAxis(coord));
  }

  /**
   * Returns the coordinate for the specified value.
   * @param {object} value The value to locate.
   * @return {number} The coordinate for the value.
   */
  getUnboundedCoordAt(value) {
    return this.axisToPlotArea(this.Info.getUnboundedCoordAt(value));
  }

  /**
   * Returns the baseline coordinate for the axis, if applicable.
   * @return {number} The baseline coordinate for the axis.
   */
  getBaselineCoord() {
    return this.axisToPlotArea(this.Info.getBaselineCoord());
  }

  /**
   * Returns the position of the axis relative to the chart.
   * @return {string} The position of the axis.
   */
  getPosition() {
    return this.getOptions()['position'];
  }

  /**
   * Returns true if this is a group axis.
   * @return {boolean}
   */
  isGroupAxis() {
    return this.Info instanceof DvtGroupAxisInfo;
  }

  /**
   * Returns the coordinates of the major ticks.
   * @return {array} Array of coords.
   */
  getMajorTickCoords() {
    var coords = this.Info ? this.Info.getMajorTickCoords() : [];
    for (var i = 0; i < coords.length; i++) coords[i] = this.axisToPlotArea(coords[i]);
    return coords;
  }

  /**
   * Returns the coordinates of the minor ticks.
   * @return {array} Array of coords.
   */
  getMinorTickCoords() {
    var coords = this.Info ? this.Info.getMinorTickCoords() : [];
    for (var i = 0; i < coords.length; i++) coords[i] = this.axisToPlotArea(coords[i]);
    return coords;
  }

  /**
   * Returns the coordinates of the baseline (value = 0). Only applies to numerical axis.
   * @return {number} Baseline coord.
   */
  getBaselineCoord() {
    return this.axisToPlotArea(this.Info.getBaselineCoord());
  }

  /**
   * Returns the linearized global min value of the axis.
   * @return {number} The global min value.
   */
  getLinearGlobalMin() {
    return this.actualToLinear(this.Info.getGlobalMin());
  }

  /**
   * Returns the linearized global max value of the axis.
   * @return {number} The global max value.
   */
  getLinearGlobalMax() {
    return this.actualToLinear(this.Info.getGlobalMax());
  }

  /**
   * Returns the linearized viewport min value of the axis.
   * @return {number} The viewport min value.
   */
  getLinearViewportMin() {
    return this.actualToLinear(this.Info.getViewportMin());
  }

  /**
   * Returns the linearized viewport max value of the axis.
   * @return {number} The viewport max value.
   */
  getLinearViewportMax() {
    return this.actualToLinear(this.Info.getViewportMax());
  }

  /**
   * Returns the linearized value for the specified coordinate along the axis.
   * @param {number} coord The coordinate along the axis.
   * @return {object} The linearized value at that coordinate.
   */
  getUnboundedLinearValAt(coord) {
    return this.Info.actualToLinear(this.getUnboundedValAt(coord));
  }

  /**
   * Returns whether the viewport is showing the full extent of the chart.
   * @return {boolean}
   */
  isFullViewport() {
    return (
      this.Info.getViewportMin() == this.Info.getGlobalMin() &&
      this.Info.getViewportMax() == this.Info.getGlobalMax()
    );
  }

  /**
   * Returns how much the axis labels overflow to the left.
   * @return {number}
   */
  getLeftOverflow() {
    return Agent.isRightToLeft(this.getCtx())
      ? this.Info.getEndOverflow()
      : this.Info.getStartOverflow();
  }

  /**
   * Returns how much the axis labels overflow to the right.
   * @return {number}
   */
  getRightOverflow() {
    return Agent.isRightToLeft(this.getCtx())
      ? this.Info.getStartOverflow()
      : this.Info.getEndOverflow();
  }

  /**
   * Returns the length of the axis.
   * @return {number} The axis length.
   */
  getLength() {
    return Math.abs(this.Info.getStartCoord() - this.Info.getEndCoord());
  }

  /**
   * Returns the minimum coordinate of the axis.
   * @return {number}
   */
  getMinCoord() {
    return this.axisToPlotArea(Math.min(this.Info.getStartCoord(), this.Info.getEndCoord()));
  }

  /**
   * Returns the maximum coordinate of the axis.
   * @return {number}
   */
  getMaxCoord() {
    return this.axisToPlotArea(Math.max(this.Info.getStartCoord(), this.Info.getEndCoord()));
  }
}

/**
 * Performs layout and positioning for the chart axes.
 * @class
 */
const DvtChartAxisRenderer = {
  /** @private */
  _DEFAULT_AXIS_MAX_SIZE: 0.33,

  /**
   * @this {DvtChartAxisRenderer}
   * Renders axes and updates the available space.
   * @param {Chart} chart The chart being rendered.
   * @param {dvt.Container} container The container to render into.
   * @param {dvt.Rectangle} availSpace The available space.
   */
  render: (chart, container, availSpace) => {
    if (!DvtChartTypeUtils.hasAxes(chart)) return;

    DvtChartAxisUtils.applyInitialZooming(chart, availSpace);

    // Approximate bubble sizes are needed at this point to compute the axis ranges
    if (DvtChartTypeUtils.isBubble(chart)) DvtChartMarkerUtils.calcBubbleSizes(chart, availSpace);

    // Render axes based on coordinate system
    if (DvtChartTypeUtils.isPolar(chart))
      DvtChartAxisRenderer._renderPolar(chart, container, availSpace);
    else DvtChartAxisRenderer._renderCartesian(chart, container, availSpace);
  },

  /**
   * Renders axes in Cartesian coordinate system.
   * @param {Chart} chart The chart being rendered.
   * @param {dvt.Container} container The container to render into.
   * @param {dvt.Rectangle} availSpace The available space.
   * @private
   */
  _renderCartesian: (chart, container, availSpace) => {
    var options = chart.getOptions();
    var isHoriz = DvtChartTypeUtils.isHorizontal(chart);
    var isSplitDualY = DvtChartDataUtils.isSplitDualY(chart);
    var isOverview = DvtChartTypeUtils.isOverview(chart);
    var totalAvailSpace = availSpace.clone();
    var yPosition = DvtChartAxisUtils.getYAxisPos(chart);
    var y2Position = DvtChartAxisUtils.getY2AxisPos(chart);
    var ignoreYRendering =
      !isOverview &&
      options.overview.content &&
      options.overview.content.yAxis &&
      options.overview.content.yAxis.rendered == 'on'; // main chart's y-axis.rendered = "off" should be ignored(for sizing purposes) if the overview has a y-axis
    var ignoreY2Rendering =
      !isOverview &&
      options.overview.content &&
      options.overview.content.y2Axis &&
      options.overview.content.y2Axis.rendered == 'on'; // main chart's y2-axis.rendered = "off" should be ignored(for sizing purposes) if the overview has a y2-axis

    DvtChartAxisRenderer._addAxisGaps(chart, availSpace);

    // Set which axis loses its last label when both plot areas are rendered to avoid overlapping labels
    if (isSplitDualY && yPosition == y2Position) {
      options['yAxis']['_skipHighestTick'] = isHoriz;
      options['y2Axis']['_skipHighestTick'] = !isHoriz;
    }

    // Layout Algorithm
    // 1. Get preferred size of y axes and allocate space.
    // 2. Get preferred size of x axis and allocate space.
    // 3. Update y axes with reduced size (due to x axis)

    // Get preferred sizing for the y axes
    var yInfo = DvtChartAxisRenderer._createYAxis(
      chart,
      container,
      availSpace,
      totalAvailSpace,
      ignoreYRendering
    );
    var y2Info = DvtChartAxisRenderer._createY2Axis(
      chart,
      container,
      availSpace,
      totalAvailSpace,
      ignoreY2Rendering
    );

    // Align the y and y2 axis tick marks if needed
    var bAligned =
      !isSplitDualY &&
      options['y2Axis']['alignTickMarks'] == 'on' &&
      options['y2Axis']['step'] == null;
    if (bAligned && yInfo && y2Info) {
      // Alignment won't happen below if yAxis.getPreferredSize() is not called in _createYAxis(), so we must call
      // _alignYAxes() again later after rendering yAxis.
      DvtChartAxisRenderer._alignYAxes(chart, yInfo, y2Info);

      //  - y2 tick label is missing sometimes
      // recalculate preferred dimensions to account for new set of labels, which may be wider than previous dimensions
      if (!isHoriz)
        y2Info.dim = DvtChartAxisRenderer._getPreferredSize(
          chart,
          y2Info.axis,
          chart.y2Axis,
          y2Info.options,
          'y2',
          availSpace,
          totalAvailSpace,
          ignoreY2Rendering
        );
    }

    var yGap = DvtChartAxisUtils.getTickLabelGapSize(chart, 'y');
    var y2Gap = DvtChartAxisUtils.getTickLabelGapSize(chart, 'y2');

    // Position the axes to reserve space
    if (isSplitDualY && yPosition == y2Position) {
      var maxSize; // align the two y axes
      if (isHoriz) {
        maxSize = Math.max(yInfo.dim.h + yGap, y2Info.dim.h + y2Gap);
        yInfo.dim.h = maxSize - yGap;
        y2Info.dim.h = maxSize - y2Gap;
      } else {
        maxSize = Math.max(yInfo.dim.w + yGap, y2Info.dim.w + y2Gap);
        yInfo.dim.w = maxSize - yGap;
        y2Info.dim.w = maxSize - y2Gap;
      }
      DvtChartAxisRenderer._positionAxis(availSpace.clone(), yInfo, yGap, ignoreYRendering); // clone so availSpace not subtracted twice
      DvtChartAxisRenderer._positionAxis(availSpace, y2Info, y2Gap, ignoreY2Rendering);
    } else {
      DvtChartAxisRenderer._positionAxis(availSpace, yInfo, yGap, ignoreYRendering);
      DvtChartAxisRenderer._positionAxis(availSpace, y2Info, y2Gap, ignoreY2Rendering);
    }

    // Spark Bar Spacing Support
    var numGroups = DvtChartDataUtils.getGroupCount(chart);
    if (DvtChartStyleUtils.getBarSpacing(chart) == 'pixel' && DvtChartTypeUtils.isBar(chart)) {
      // Adjust the width so that it's an even multiple of the number of groups
      if (availSpace.w > numGroups) {
        var newWidth = Math.floor(availSpace.w / numGroups) * numGroups;
        availSpace.x += (availSpace.w - newWidth) / 2;
        availSpace.w = newWidth;
      }
    }

    // Get preferred sizing for the x axes, render, and position.
    var xInfo = DvtChartAxisRenderer._createXAxis(chart, container, availSpace, totalAvailSpace);
    xInfo.axis.render(xInfo.options, xInfo.dim.w, xInfo.dim.h);
    DvtChartAxisRenderer._positionAxis(
      availSpace,
      xInfo,
      DvtChartAxisUtils.getTickLabelGapSize(chart, 'x')
    );

    var splitterPosition = DvtChartStyleUtils.getSplitterPos(chart);
    var isR2L = Agent.isRightToLeft(chart.getCtx());
    var yAvailSpace = DvtChartAxisRenderer._getSplitAvailSpace(
      availSpace,
      splitterPosition,
      isHoriz,
      isHoriz && isR2L
    );
    var y2AvailSpace = DvtChartAxisRenderer._getSplitAvailSpace(
      availSpace,
      1 - splitterPosition,
      isHoriz,
      !isHoriz || !isR2L
    );

    // Render and position the y axes
    if (isHoriz) {
      if (yInfo) {
        yInfo.axis.setTranslateX(availSpace.x);
        if (isSplitDualY)
          yInfo.axis.render(yInfo.options, yAvailSpace.w, yInfo.dim.h, yAvailSpace.x, 0);
        else yInfo.axis.render(yInfo.options, availSpace.w, yInfo.dim.h);
      }

      if (bAligned && yInfo && y2Info)
        // align again after rendering yAxis
        DvtChartAxisRenderer._alignYAxes(chart, yInfo, y2Info);

      if (y2Info) {
        y2Info.axis.setTranslateX(availSpace.x);
        if (isSplitDualY)
          y2Info.axis.render(y2Info.options, y2AvailSpace.w, y2Info.dim.h, y2AvailSpace.x, 0);
        else y2Info.axis.render(y2Info.options, availSpace.w, y2Info.dim.h);
      }

      DvtChartAxisRenderer._setOverflow(availSpace, yInfo, xInfo);
    } else {
      if (yInfo) {
        if (isSplitDualY)
          yInfo.axis.render(yInfo.options, yInfo.dim.w, yAvailSpace.h, 0, yAvailSpace.y);
        else yInfo.axis.render(yInfo.options, yInfo.dim.w, availSpace.h);
      }

      if (bAligned && yInfo && y2Info)
        // align again after rendering yAxis
        DvtCha