const fsx = require('fs-extra');
const glob = require('glob');
const path = require('path');
const os = require('os');
const PatternCompiler = require('./PatternCompiler');
const ts = require('typescript');
const pkg = require('./package.json');

/**
 * Traverse the message bundle and compile all strings.
 * @param {PatternCompiler} compiler The instance of the PatternCompiler
 * @param {Object} bundle The message bundle
 * @return {Object} A message bundle whose strings are wrapped by functions
 * generated from the PatternCompiler.
 * @private
 */
function traverse(compiler, bundle) {
  const metadata = {};
  // sort so that metadata '@' keys are processed first
  const output = Object.keys(bundle).sort().reduce((accum, key) => {
    if (key.match(/^@/)) {
      metadata[key.substring(1)] = getMetadataParamTypes(bundle[key]);
    } else {
      const value = bundle[key];
      if (typeof value !== 'string') {
        throw Error(`"${key}" value must be a string`);
      }
      return {
        ...accum,
        [key]: compiler.compile(value, metadata[key])
      };
    }
    return accum;
  }, {});
  return output;
}

function getMetadataParamTypes(metadata) {
  return Object.keys(metadata.placeholders || {}).reduce((accum, key) => (
    {
      ...accum,
      [key]: metadata.placeholders?.[key].type
    }
  ), {});
}

/**
 * Get the bundle as an object from the given filepath
 * @param {string} filepath
 * @return {object} The bundle from the file
 * @private
 */
function getBundle(filepath) {
  let bundle;
  if (fsx.existsSync(filepath)) {
    const bundleContents = fsx.readFileSync(filepath);
    bundle = JSON.parse(bundleContents);
  }
  return bundle;
}

/**
 * Process the given bundle for the given locale and write the compiled contents
 * to the given file path
 * @param {string} rootBundleFile The path to the root bundle JSON file
 * @param {string} bundleId The id of the bundle
 * @param {object} bundle The message bundle
 * @param {string} locale The locale of the bundle
 * @param {string} targetFile The path to the target file where the contents will
 * be written.
 * @param {string} exportType The type of export to produce, either 'default' or 'named'
 * @param {boolean?} withBundleType Whether to export the bundle type definition
 * @private
 */
function convertBundle(
  rootBundleFile,
  bundleId,
  bundle,
  locale,
  targetFile,
  exportType,
  withBundleType
) {
  const compiler = new PatternCompiler(locale);
  let processed;
  try {
    processed = traverse(compiler, bundle);
  } catch (ex) {
    throw Error(`While processing ${locale}/${bundleId}:\n${ex.message}`);
  }
  // apply hooks and type imports
  const convertor = customHooks.convertor;
  const typeImport = customHooks.typeImport
    ? Object.keys(customHooks.typeImport).map((k) => customHooks.typeImport[k])[0]
    : '';
  const otherImports = customHooks.otherImports ? [customHooks.otherImports] : [];
  const valueType = typeImport ? Object.keys(customHooks.typeImport)[0] : '';
  const translations = Object.keys(processed).map((messageKey) => {
    const entry = processed[messageKey];
    const translation = entry.formatter || '""';
    // Param types for TS
    const params = Object.keys(entry.paramTypes).map(
      (paramKey) => `${paramKey}:${entry.paramTypes[paramKey]}`
    );
    const paramString = params.length ? `p: {${params.join(',')}}` : '';
    const paramVar = params.length ? 'p' : undefined;
    const output = convertor
      ? `convert({bundleId:"${bundleId}",id:"${messageKey}",params:${paramVar},translation:${translation}})`
      : translation;
    return [messageKey, `(${paramString})${valueType && ':' + valueType} => ${output}`];
  });
  const targetDir = path.dirname(targetFile);
  fsx.ensureDirSync(targetDir);
  fsx.writeFileSync(
    targetFile,
    generateContent(
      rootBundleFile,
      otherImports.concat(typeImport),
      translations,
      withBundleType,
      convertor,
      exportType
    )
  );
}

function generateContent(
  rootBundleFile,
  imports,
  translations,
  withBundleType,
  convertor,
  exportType
) {
  return `// This file is auto-generated by ${pkg.name} from ${rootBundleFile}
// and should not be edited by hand. Update the JSON file and rerun the build to regenerate this content.
  ${imports.join('\n')}

${
  (convertor &&
    `
type ParamsType = {
  bundleId: string,
  id: string,
  params: { [key:string]: any } | undefined,
  translation: string
};

const convert = ${convertor}`) ||
  ''
}
const bundle = {
${translations.map((t) => `  "${t[0]}": ${t[1]}`).join(',\n')}
};
${
  (exportType === 'default' && `export default bundle;`) ||
  `${translations.map((t) => `export const ${t[0]} = bundle.${t[0]}`).join(';\n')}`
}
${withBundleType ? 'export type BundleType = typeof bundle;' : ''}
`;
}

/**
 * Test if a given directory name (base name only) is an NLS directory.
 * @param {string} name The directory name
 * @return True if NLS directory, false otherwise
 */
function isNlsDir(name) {
  return (name.match(/^[a-z]{2,3}$/i) || name.match(/^[a-z]{2,3}-.+/)) && !name.match(/\.\w+$/);
}

function transpile(files, module) {
  const moduleMap = {
    amd: ts.ModuleKind.AMD,
    'legacy-amd': ts.ModuleKind.AMD,
    esm: ts.ModuleKind.ES2020
  };
  const program = ts.createProgram(files, {
    module: moduleMap[module],
    target: 'ES2021',
    strict: true
  });
  program.emit();
}

function convertToLegacyAmd(files) {
  // Replace default export with top-level keys
  const amdDefaultExport = 'exports.default = bundle;';
  files.forEach((file) => {
    const contents = fsx.readFileSync(file).toString();
    if (contents.indexOf(amdDefaultExport) === -1) {
      throw Error(`No default export found in ${file}`);
    }
    fsx.writeFileSync(file, contents.replace(amdDefaultExport, 'Object.assign(exports, bundle);'));
  });
}

let customHooks = {};

/**
 * Compile the message bundle in ICU format. This function starts with the "root"
 * message bundle and attempts to discover bundles for all available locales by
 * traversing the root bundle's directory and looking for locale directory names.
 * Any file found under locale directories whose names match the root bundle will
 * also be compiled.
 * When creating the output file in the given targetDir, the locale directory
 * structure from the source will be recreated.
 * @param {object} props The properties with which to build the bundle
 * @param {string} props.rootDir The path to the root message bundle
 * @param {string} props.bundleName The name of the bundle file
 * @param {string} props.locale The locale of the root message bundle
 * @param {string} props.outDir The output directory where the built bundle will be written
 * @param {('amd'|'cjs'|'esm'|'legacy-amd')=} props.module The type of module to produce -- 'amd', 'cjs', 'esm', or 'legacy-amd'.
 * If not supplied, the Typscript will not be transpiled.
 * @param {('default'|'named')=} props.exportType The type of export to produce -- 'default' or 'named'
 * @param {boolean=} props.override Indicates the bundle is an override, and only the root
 * locale and those explicitly stated in [--supportedLocales] will be built.
 * @param {string[]=} props.additionalLocales An array of additional locales to build
 * @param {string=} props.hooks A path to the custom hooks file
 */
function build({
  rootDir,
  bundleName,
  locale,
  outDir,
  module,
  exportType = 'default',
  override,
  additionalLocales = [],
  hooks
}) {
  const jsonRegEx = /\.json$/;
  if (!jsonRegEx.test(bundleName)) {
    throw Error(`${bundleName} must be a JSON file`);
  }
  const isLegacyAmd = module === 'legacy-amd';
  const targetBundleName = bundleName.replace(jsonRegEx, '.ts');
  // Merge in supported locales to build. If physical dirs don't exist, they'll
  // inherit the root translations
  const supportedLocales = [
    ...new Set(
      // for override bundles, only look in root and additionalLocales folders
      (override ? [] : fsx.readdirSync(rootDir).filter(isNlsDir)).concat(additionalLocales).sort()
    )
  ];

  customHooks = hooks ? require(hooks) : {};

  // Traverse root bundle
  const rootBundleFile = path.relative(process.cwd(), path.join(rootDir, bundleName));
  const rootBundle = getBundle(rootBundleFile);
  let rootBundleKeys;
  if (rootBundle) {
    convertBundle(
      rootBundleFile,
      bundleName,
      rootBundle,
      locale,
      path.join(outDir, targetBundleName),
      isLegacyAmd ? 'default' : exportType,
      true
    );
    rootBundleKeys = new Set(Object.keys(rootBundle));
  }

  supportedLocales.forEach((locale) => {
    // Combine all levels into single bundle, starting with rootBundle
    const combinedBundle = Object.assign({}, rootBundle);
    const localeParts = locale.split('-');
    for (let i = 0, len = localeParts.length; i < len; i++) {
      // Build segment from region (0) to index
      let segment = localeParts.slice(0, i + 1).join('-');
      const perBundlePath = path.join(rootDir, segment, bundleName);
      const perBundle = getBundle(perBundlePath);
      Object.assign(combinedBundle, perBundle);
      // combinedBundle should not have keys not in rootBundle
      if (rootBundleKeys) {
        const extraKeys = Object.keys(combinedBundle).filter((k) => !rootBundleKeys.has(k));
        if (extraKeys.length) {
          throw Error(`${perBundlePath} has keys not found in ${rootBundleFile}: ${extraKeys}`);
        }
      }
    }

    convertBundle(
      rootBundleFile,
      bundleName,
      combinedBundle,
      locale,
      path.join(outDir, locale, targetBundleName),
      isLegacyAmd ? 'default' : exportType
    );
  });

  !override && fsx.writeFileSync(
    path.join(outDir, 'supportedLocales.ts'),
    `export default ${JSON.stringify(supportedLocales)};\n`
  );

  if (module) {
    const transpileFiles = glob.sync(`${path.resolve(outDir)}/**/*.ts`);
    transpile(transpileFiles, module);

    if (module === 'legacy-amd') {
      convertToLegacyAmd(glob.sync(`${outDir}/**/${bundleName.replace(jsonRegEx, '')}.js`));
      transpileFiles.forEach(fsx.removeSync);
    }
  }
}
module.exports = {
  build,
  isNlsDir
};
