/*!
 Copyright (c) 2021, 2024, Oracle and/or its affiliates.
 */
/**
 * @namespace apex.pwa
 * @since 21.2
 * @desc
 * <p>The apex.pwa namespace contains Oracle APEX functions related to Progressive Web App features.</p>
 * <p>These functions are useful only when an APEX application has enabled Progressive Web App.</p>
 */

// apex.pwa is already generated by the server, with {} being a fallback
apex.pwa = apex.pwa || {};

( function ( pwa, lang, $, debug, theme, message, actions, util, server, env, item ) {

    "use strict";

    const body$ = $("body"),
        // Constants for IDs and classes
        PWA_DIALOG = "a-pwaDialog",
        PWA_DIALOG_SHOW_INSTRUCTIONS = "a-pwaDialog--showInstructions",
        PWA_DIALOG_SHOW_INSTRUCTIONS_ACTION = "a-pwa-show-instructions",
        PWA_DIALOG_HIDE_INSTRUCTIONS_ACTION = "a-pwa-hide-instructions",
        PWA_INSTALL = "a-pwaInstall",
        PWA_INSTALL_AVAILABLE = "a-pwaInstall--available",
        PWA_INSTALL_ACTION = "a-pwa-install",
        PWA_PUSH_AVAILABLE = "a-pwaPush--available",
        // Constants for selectors
        SEL_PUSH_STATE = ".a-pwaPush--state",
        SEL_PUSH_SUBSCRIPTION_REGION = ".a-pwaPush--subscriptionRegion",
        SEL_PUSH_SUBSCRIPTION_NOT_SUPPORTED = ".a-pwaPush--subscriptionRegion--not-supported",
        // Constants for display modes
        DISPLAY_FULLSCREEN = "fullscreen",
        DISPLAY_STANDALONE = "standalone",
        DISPLAY_MINIMALUI = "minimal-ui",
        DISPLAY_BROWSER = "browser";

    // Initialize installPrompt for use later to show browser install prompt.
    let installPrompt;

    // Variable used for emitting installability log to the debug API when relevant.
    let installabilityLog;

    // Variable used to check is the current browser has support for service workers
    // which is required for most PWA features.
    const hasServiceWorker = "serviceWorker" in navigator;

    // Variable used to check if current URL is http,
    // given that https is required for most PWA features.
    // Exceptions are localhost.
    const isHttp = window.location.protocol === "http:"     &&
                   window.location.hostname !== "localhost" &&
                   window.location.hostname !== "127.0.0.1";

    /**
     * Determines if push notifications are supported
     * by the combination of current browser, device, application and user.
     * Parameter showDebug is used to show debug messages in the console.
     * @ignore
     */
    const isPushUsable = ( showDebug = true ) => {

        // Variables used to check elligibility of push notifications.
        const hasPushManager = "PushManager" in window;
        const hasNotification = "Notification" in window;
        const hasPushPublicKey = typeof pwa.publicKey === "string";
        const isPublicUser = env.APP_USER === "nobody";

        // If serviceWorker exists in the navigator object and
        // PushManager exists in the window object and
        // a public key was emitted from the server, then push is usable.
        if ( hasServiceWorker && hasPushManager && hasNotification && hasPushPublicKey && !isPublicUser ) {
            return true;
        }

        // Otherwise, push is not usable. Find out why.

        // showDebug is used to display more information in the console to developers, but we avoid it on page load
        // to avoid spamming the console, so it's only used when the user invokes the push notifications functions.
        if ( showDebug ) {
            // If current URL is http, then push is not usable.
            // Exceptions are localhost to avoid spamming the console with a false positive.
            if ( isHttp ) {
                debug.warn( "Push notifications are not supported on HTTP pages." );
            }

            // If a public key was not emitted from the server, then push is not usable.
            if ( !hasPushPublicKey ) {
                debug.warn( "Push notifications are not supported because the public key is not set for this application." );
            }

            // If the current user is nobody then it's the public user (not authenticated), then push is not usable.
            if ( isPublicUser ) {
                debug.warn( "Push notifications are not supported because the current user is not authenticated." );
            }

            // If serviceWorker does not exist in the navigator object or
            // PushManager does not exist in the window object, then push is not usable.
            if ( !hasServiceWorker || !hasPushManager || !hasNotification ) {
                debug.warn( "Push notifications are not supported by this browser." );
            }
        }

        return false;
    };

    /**
     * Initialize the UI for PWA Install components
     * This function runs on page load and upon other events
     * Performs hide and show for PWA install buttons
     * @ignore
     */
    const refreshInstallUI = async () => {
        if ( await pwa.isInstallable() ) {
            body$.addClass( PWA_INSTALL_AVAILABLE );
            actions.show( PWA_INSTALL_ACTION );
        } else {
            body$.removeClass( PWA_INSTALL_AVAILABLE );
            actions.hide( PWA_INSTALL_ACTION );

            // Give reasonable time (2 seconds) after page load to allow for the browser
            // events to detect PWA installability criteria.
            // If the PWA can't be installed emit the reason using debug.info
            setTimeout(() => {
                if ( installabilityLog ) {
                    debug.info( installabilityLog );
                }
            }, 2000);
        }
    };

    /**
     * Initialize the UI for PWA Push components
     * This function runs on page load and upon other events
     * @ignore
     */
    const refreshPushUI = async () => {

        // Variable used to check if push is supported by the current browser
        // false parameter is used to not show debug messages in the console.
        const pushUsable = isPushUsable( false );

        // if push is supported by the current browser
        // add the push available class to the body element, if not remove it
        if ( pushUsable ) {
            body$.addClass( PWA_PUSH_AVAILABLE );
        } else {
            body$.removeClass( PWA_PUSH_AVAILABLE );
        }

        // SEL_PUSH_STATE is a selector for the push subscription state text
        // Possible values are "Enabled", "Disabled" or "Not Supported"
        // This only gets computed when the SEL_PUSH_STATE exists on the page.
        const pushState = $( SEL_PUSH_STATE );
        if ( pushState.length ) {

            // Variable used to store the message code for the push state
            // Assume the default behavior is not supported
            let messageCode = "NOT_SUPPORTED";

            // if push is supported by the current browser
            // then the push state is either enabled or disabled
            if ( pushUsable ) {
                const hasPushSubscription = await pwa.hasPushSubscription();
                messageCode = hasPushSubscription ? "ON" : "OFF";
            }

            // Set the text of the push state object to the appropriate text message
            pushState.text( lang.getMessage( `APEX.FEATURE.CONFIG.${ messageCode }` ) );
        }
    };

    /**
     * Registers the service worker on the current page
     * Adds various PWA event listeners and actions
     * Triggers PWA components UI refresh
     * @ignore
     */
    pwa.init = ( { serviceWorkerPath } ) => {
        if ( hasServiceWorker ) {
            window.addEventListener( "load", async () => {
                try {
                    await navigator.serviceWorker.register( serviceWorkerPath );
                    debug.info( "Service worker registered." );
                } catch ( err ) {
                    debug.warn( "Service worker failed to register.", err );
                }
            } );

            window.addEventListener( "beforeinstallprompt", ( event ) => {
                // Prevent the mini-infobar from appearing on mobile
                event.preventDefault();

                // Store the event so it can be triggered later
                installPrompt = event;
                refreshInstallUI();
            } );

            window.addEventListener( "appinstalled", () => {
                // Clear the installPrompt
                installPrompt = null;

                // Hide install buttons
                body$.removeClass( PWA_INSTALL_AVAILABLE );
                actions.hide( PWA_INSTALL_ACTION );
            } );

            $( () => {
                // Event listener when clicking on the "Install App" action
                actions.add( {
                    name: PWA_INSTALL_ACTION,
                    label: lang.getMessage( "PWA.INSTALL" ),
                    icon: "fa fa-cloud-download",
                    action: pwa.openInstallDialog
                } );

                // Event listener when invoking the "Show instructions" action
                actions.add( {
                    name: PWA_DIALOG_SHOW_INSTRUCTIONS_ACTION,
                    label: null,
                    icon: null,
                    action: () => {
                        $( "." + PWA_DIALOG ).addClass( PWA_DIALOG_SHOW_INSTRUCTIONS );
                    }
                } );

                // Event listener when invoking the "Instructions back" action
                actions.add( {
                    name: PWA_DIALOG_HIDE_INSTRUCTIONS_ACTION,
                    label: null,
                    icon: null,
                    action: () => {
                        $( "." + PWA_DIALOG ).removeClass( PWA_DIALOG_SHOW_INSTRUCTIONS );
                    }
                } );

                // Event listener when clicking on the "Install App" button class.
                // This button is generally located in the navigation bar list
                // Usage of .not function below is to ensure we don't interfere
                // with the apex.actions handler
                $( "." + PWA_INSTALL )
                    .not( `:has([href="#action$${ PWA_INSTALL_ACTION }"])` )
                    .not( `[href="#action$${ PWA_INSTALL_ACTION }"]` )
                    .on( "click", pwa.openInstallDialog );
            } );

        } else {
            if ( isHttp ) {
                debug.warn( "Service workers are not supported on HTTP pages." );
            } else {
                debug.warn( "Service workers are not supported." );
            }
        }

        $( () => {
            refreshInstallUI();
            refreshPushUI();
        } );
    };

    /**
     * The APEX Push Notification feature page comes with components that should
     * be initialized from the client side due to its reliance on the browser
     * push manager. This function initializes the UI for the APEX push
     * notification feature page, which includes hidding/showing the relevant
     * region based on feature eligibility and setting the checkbox value based
     * on the current push subscription state.
     * @ignore
     */
    pwa.initPushSubscriptionPage = async () => {
        const hasPushSubscription = await pwa.hasPushSubscription();
        const pageItem = item(`P${ env.APP_PAGE_ID }_ENABLE_PUSH`);

        if ( isPushUsable( false ) ) {
            // Feature page generated by APEX contains a PXXX_ENABLE_PUSH checkbox item
            // Which is set from the client side based on the eligibility of hasPushSubscription
            pageItem.setValue( hasPushSubscription ? "Y" : "N" );
        } else {
            // if push is not usable, then hide the region that contains the checkbox and show
            // the region that contains the message that push is not supported
            $( SEL_PUSH_SUBSCRIPTION_REGION ).hide();
            $( SEL_PUSH_SUBSCRIPTION_NOT_SUPPORTED ).show();
            pageItem.setValue( "N" );
        }
    };

    /**
     * <p>Get the current display mode for the PWA.</p>
     * <p>Possible values are: fullscreen, standalone, minimal-ui, browser.</p>
     * <p>The display mode is set by the developer in the application definition.</p>
     * <p>This function is used to determine if the application is currently accessed through the PWA application
     * (eg. in fullscreen) or through the browser normally.</p>
     *
     * @function getDisplayMode
     * @memberof apex.pwa
     * @return {string} Current display mode for the PWA
     *
     * @example <caption>Returns the PWA current display mode.
     * Possible values are: fullscreen, standalone, minimal-ui, browser.</caption>
     *
     * const displayMode = apex.pwa.getDisplayMode();
     */
    pwa.getDisplayMode = () => {
        if ( theme.mq( `(display-mode: ${DISPLAY_FULLSCREEN})` ) ) {
            return DISPLAY_FULLSCREEN;
        } else if ( theme.mq( `(display-mode: ${DISPLAY_STANDALONE})` ) ) {
            return DISPLAY_STANDALONE;
        } else if ( theme.mq( `(display-mode: ${DISPLAY_MINIMALUI})` ) ) {
            return DISPLAY_MINIMALUI;
        } else {
            return DISPLAY_BROWSER;
        }
    };

    /**
     * <p>Determines if the current session is eligible for installation of the PWA.</p>
     * <p>This function will detect:</p>
     * <ul>
     * <li>the user's browser install prompt is available.</li>
     * <li>the PWA is already installed on the user's device.</li>
     * <li>the user session is currently in PWA mode.</li>
     * <li>the user is on iOS/iPadOS on Safari.</li>
     * </ul>
     * <p>Given the user's current device and browser, this function will determine if installing the
     * PWA is possible.</p>
     *
     * @function isInstallable
     * @memberof apex.pwa
     * @return {Promise} A Promise that resolves a boolean, based on installation eligibility
     *
     * @example <caption>Returns if the APEX application is installable as a PWA.</caption>
     *
     * const isInstallable = await apex.pwa.isInstallable();
     */
    pwa.isInstallable = async () => {

        let isInstallable = false;

        // Reset installability log
        installabilityLog = null;

        // Bug 33551886
        // BeforeInstallPromptEvent is a non-standard event supported in a few select browsers
        // https://caniuse.com/?search=BeforeInstallPromptEvent
        // If this event exists, then we should only show the install button if the installPrompt
        // has been populated before. Otherwise the install button should be hidden.
        // This is the main feature detection criteria
        if ( "BeforeInstallPromptEvent" in window ) {
            if ( installPrompt ) {
                isInstallable = true;
            } else {
                installabilityLog = 'PWA is not installable. Browser installation criteria are not met or PWA may already be installed.';
                isInstallable = false;
            }
        }

        // Validate if app is already installed
        // This check must be done on top level navigation only (must not be an iframe)
        if ( "getInstalledRelatedApps" in navigator && window === window.parent ) {
            const relatedApps = await navigator.getInstalledRelatedApps();

            if ( relatedApps && relatedApps.length ) {
                installabilityLog = 'PWA is not installable due to being already installed on current device.';
                isInstallable = false;
            }
        }

        // navigator.standalone returns a boolean indicating if the browser is running in standalone mode. This property
        // is available only on Apple iOS Safari only, which is useful to determine if the current device and browser is
        // Safari iOS. Reference: https://web.dev/learn/pwa/enhancements/#detecting-a-pwa-on-apple-mobile-platforms
        if ( "standalone" in navigator ) {
            // navigator.standalone is true if app is opened from Home Screen, so it's already installed
            // navigator.standalone is false if app is opened from Safari, so it's installable
            if ( navigator.standalone ) {
                installabilityLog = 'PWA is not installable due to current page already being in PWA mode.';
                isInstallable = false;
            } else {
                isInstallable = true;
            }
        }

        // If current page is iframe, then it's not installable
        if ( window !== window.parent ) {
            installabilityLog = 'PWA is not installable due to current page being an iframe.';
            isInstallable = false;
        }

        return isInstallable;
    };

    /**
     * @deprecated
     * @function getInstallText
     * @memberof apex.pwa
     * @return {Promise} A Promise that resolves the PWA dialog installation text
     */
    pwa.getInstallText = async () => {

        const isInstallable = await pwa.isInstallable();

        // Load all PWA messages first
        // We will load unnecessary messages but it will be useful
        // to avoid pinging the server many times
        await lang.loadMessages( ["APEX.PWA.%"] );

        // APEX.PWA.INSTRUCTIONS.IOS
        if ( "standalone" in navigator && isInstallable ) {
            return `<ol class="a-pwaDialog-steps">
  <li class="a-pwaDialog-step a-pwaDialog-step--1">
    <svg class="a-pwaDialog-stepImage" role="presentation" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 64 64"><g><path d="M50.21 19.87H40.8a1.641 1.641 0 0 0 0 3.28h9.41c.75 0 1.36.61 1.36 1.36v31.24c0 .75-.61 1.36-1.36 1.36H13.79c-.75 0-1.36-.61-1.36-1.36V24.5c0-.75.61-1.36 1.36-1.36h9.41a1.641 1.641 0 0 0 0-3.28h-9.41c-2.56 0-4.64 2.08-4.64 4.64v31.24c0 2.56 2.08 4.64 4.64 4.64h36.42c2.56 0 4.64-2.08 4.64-4.64V24.5a4.648 4.648 0 0 0-4.64-4.63z"/><path d="M22.8 15.2c.42 0 .84-.16 1.16-.48l6.4-6.41v31.63a1.641 1.641 0 0 0 3.28 0V8.32l6.4 6.41c.32.32.74.48 1.16.48a1.635 1.635 0 0 0 1.16-2.79l-9.19-9.19c-.33-.33-.75-.49-1.17-.49-.42 0-.84.16-1.17.48l-9.19 9.19a1.63 1.63 0 0 0 0 2.31c.33.32.74.48 1.16.48z"/></g></svg>
    <p class="a-pwaDialog-stepText">${lang.getMessage( "APEX.PWA.INSTRUCTIONS.IOS.STEP1" )}</p>
  </li>
  <li class="a-pwaDialog-step a-pwaDialog-step--2">
    <svg class="a-pwaDialog-stepImage" role="presentation" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 64 64"><g><path d="M46.41 30.55H33.46V17.02a1.451 1.451 0 0 0-2.9 0v13.53H17.61a1.451 1.451 0 0 0 0 2.9h12.95v13.53a1.451 1.451 0 0 0 2.9 0V33.45h12.95a1.451 1.451 0 0 0 0-2.9z" stroke-miterlimit="10"/><path d="M54.04 60.67H9.12c-3.65 0-6.62-2.97-6.62-6.62V9.14c0-3.65 2.97-6.62 6.62-6.62h44.92c3.65 0 6.62 2.97 6.62 6.62v44.92c-.01 3.64-2.98 6.61-6.62 6.61zM9.12 5.52c-1.99 0-3.62 1.62-3.62 3.62v44.92c0 1.99 1.62 3.62 3.62 3.62h44.92c1.99 0 3.62-1.62 3.62-3.62V9.14c0-1.99-1.62-3.62-3.62-3.62H9.12z"/></g></svg>
    <p class="a-pwaDialog-stepText">${lang.getMessage( "APEX.PWA.INSTRUCTIONS.IOS.STEP2" )}</p>
  </li>
  <li class="a-pwaDialog-step a-pwaDialog-step--3">
    <svg class="a-pwaDialog-stepImage" role="presentation" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 64 64"><g><path d="M32 60.41c-15.9 0-28.83-12.93-28.83-28.83S16.1 2.74 32 2.74s28.83 12.93 28.83 28.83S47.9 60.41 32 60.41zm0-53.97c-13.86 0-25.14 11.28-25.14 25.14S18.14 56.72 32 56.72s25.14-11.28 25.14-25.14S45.86 6.44 32 6.44z"/><path d="M42.5 20.8L26.72 36.57l-5.19-5.18a2.62 2.62 0 0 0-1.8-.71 2.655 2.655 0 0 0-1.94 4.46l7.06 7.06c.48.48 1.14.78 1.87.78.73 0 1.39-.3 1.87-.78l17.65-17.65c.44-.47.71-1.1.71-1.8a2.655 2.655 0 0 0-4.45-1.95z"/></g></svg>
    <p class="a-pwaDialog-stepText">${lang.getMessage( "APEX.PWA.INSTRUCTIONS.IOS.STEP3" )}</p>
  </li>
</ol>`;
        }

         // APEX.PWA.INSTRUCTIONS (Fallback message when not installable)
        if ( !isInstallable ) {
            return lang.getMessage( "APEX.PWA.INSTRUCTIONS" );
        }
    };

    /**
     * <p>For browsers with automatic PWA installation, this function triggers the installation process.</p>
     * <p>For browsers without automatic PWA installation, this function opens a dialog with the instruction
     * text.</p>
     * <p>This function is automatically invoked when clicking on any DOM element with the following class:
     * <code class="prettyprint">.a-pwaInstall</code>.
     * <p>This function is also invoked on <code class="prettyprint">apex.actions</code> with action name
     * <code class="prettyprint">a-pwa-install</code>.
     * <p>For example when creating a new APEX application with PWA enabled, a navigation bar entry is added with the
     * <code class="prettyprint">.a-pwaInstall</code> class and
     * <code class="prettyprint">href="#action$a-pwa-install"</code>.
     * Developers can add custom buttons to their application and use the
     * <code class="prettyprint">.a-pwaInstall</code> class or
     * <code class="prettyprint">href="#action$a-pwa-install"</code>
     * to trigger the PWA installation process.</p>
     * <p>Alternatively, developers can run this function to trigger the PWA installation process
     * programatically for a custom experience.</p>
     *
     * @function openInstallDialog
     * @memberof apex.pwa
     *
     * @example <caption>Opens the installation dialog for installing the PWA.</caption>
     *
     * apex.pwa.openInstallDialog();
     */
    pwa.openInstallDialog = async () => {
        if ( installPrompt ) {
            // Show the browser native install prompt
            installPrompt.prompt();
        } else {
            // Otherwise show instructions
            const instructions = await pwa.getInstallText();

            const _open = ( { manifest, scn } ) => {
                let dialogHeader = "";
                let dialogClass = PWA_DIALOG;
                let dialogActions = [];
                let screenshots = [];

                if ( manifest ) {
                    dialogHeader =
`<img class="a-pwaDialog-icon" src="${ util.escapeHTMLAttr( manifest.icons[1].src ) }?v=${util.escapeHTMLAttr(scn)}" alt="${ util.escapeHTMLAttr( manifest.name ) }" />
<div class="a-pwaDialog-about">
<div class="a-pwaDialog-name">${ util.escapeHTML( manifest.name ) }</div>
<div class="a-pwaDialog-description">${ util.escapeHTML( manifest.description || lang.getMessage( 'APEX.PWA.DIALOG.INTRO' ) ) }</div>
</div>`;

                    // If there are screenshots in the manifest, build a carousel of screenshots
                    if ( Array.isArray( manifest.screenshots ) && manifest.screenshots.length ) {
                        screenshots = manifest.screenshots.map(( screenshot ) => `<img class="a-pwaDialog-screenshot" src="${ util.escapeHTMLAttr( screenshot.src ) }?v=${ util.escapeHTMLAttr( scn ) }" alt="${ util.escapeHTMLAttr( screenshot.label ) }" />` );
                        dialogActions.push( `<button data-action="${ PWA_DIALOG_SHOW_INSTRUCTIONS_ACTION }" class="a-Button a-Button--full a-Button--large a-Button--hot">${ util.escapeHTML( lang.getMessage( 'APEX.PWA.DIALOG.SHOW.INSTRUCTIONS' ) ) }</button>` );
                        dialogActions.push( `<button data-action="${ PWA_DIALOG_HIDE_INSTRUCTIONS_ACTION }" class="a-Button a-Button--full a-Button--large">${ util.escapeHTML( lang.getMessage( 'APEX.PWA.DIALOG.HIDE.INSTRUCTIONS' ) ) }</button>` );
                    } else {
                        dialogClass += ' ' + PWA_DIALOG_SHOW_INSTRUCTIONS; // No screenshots, so show instructions directly
                    }
                } else {
                    dialogHeader = `<div class="a-pwaDialog-description">${ util.escapeHTML( lang.getMessage( 'APEX.PWA.DIALOG.INTRO' ) ) }</div>`;
                    dialogClass += ' ' + PWA_DIALOG_SHOW_INSTRUCTIONS;  // No manifest, so show instructions directly
                }

                // Build the content of the PWA install dialog
                const dialogContent =
`<div class="a-pwaDialog-header">${ dialogHeader }</div>
<div class="a-pwaDialog-instructions">${ instructions }</div>
<div class="a-pwaDialog-screenshots">${ screenshots.join("") }</div>
<div class="a-pwaDialog-actions">${ dialogActions.join("") }</div>`;

                // Open the dialog
                message.showDialog( dialogContent, {
                    id: PWA_DIALOG,
                    title: lang.getMessage( "APEX.PWA.DIALOG.TITLE" ),
                    unsafe: false,
                    width: "auto",
                    okButton: false,
                    dialogClass,
                    open: () => {
                        // Accessibility helper to close the dialog when clicking
                        // outside of the dialog, since this is going to be called
                        // mostly on mobile devices
                        $( ".ui-widget-overlay" ).on( "click", () => {
                            $( "#" + PWA_DIALOG ).dialog( "close" );
                        } );
                    },
                } );
            };

            // Fetch the manifest to reuse data from the manifest such as
            // the name of the app, description, icon, screenshots, and more.
            // The manifest data is already translated.
            try {
                const manifestEl = document.querySelector( "link[rel=manifest]" );
                const response = await fetch( manifestEl.href );
                const manifest = await response.json();
                // Get the SCN from the manifest URL to reuse in the images for this dialog
                const scn = manifestEl.href.split("?v=")[1];
                _open( { manifest, scn } );
            } catch ( err ) {
                debug.warn( `Fetching manifest error: ${ err.message }` );
                _open( {} );
            }
        }
    };

    /**
     * This function unregisters service workers and
     * deletes all APEX core and app caches
     * @ignore
     */
    pwa.cleanup = ( { serviceWorkerPath } ) => {
        if ( hasServiceWorker ) {
            navigator.serviceWorker.getRegistrations().then( ( registrations ) => {
                for ( let registration of registrations ) {
                    const registrationUrl = new URL( registration.active.scriptURL ).pathname;
                    if ( serviceWorkerPath.includes( registrationUrl ) ) {
                        registration.unregister();
                    }
                }
            } ).catch( debug.warn );
        }

        if ( "caches" in window ) {
            const CORE_CACHE_PREFIX = "APEX-CORE-";
            const APP_CACHE_PREFIX = "APEX-APP-";
            window.caches.keys().then( ( cacheNames ) =>
                Promise.all(
                    cacheNames.map( ( cacheName ) => {
                        // cleanup core cache if APEX version is a mismatch
                        if ( cacheName.startsWith( CORE_CACHE_PREFIX ) &&
                             cacheName.split( CORE_CACHE_PREFIX )[1] !== env.APEX_VERSION ) {
                            return window.caches.delete( cacheName );
                        }

                        // cleanup app cache for current app
                        if ( cacheName.startsWith( APP_CACHE_PREFIX ) &&
                             cacheName.split( "-" )[2] === env.APP_ID ) {
                            return window.caches.delete( cacheName );
                        }
                    } )
                )
            ).catch( debug.warn );
        }
    };

    /**
     * <p>This function returns the current user push subscription object.</p>
     *
     * @function getPushSubscription
     * @memberof apex.pwa
     * @return {Promise} A Promise that resolves to a PushSubscription object containing details of an existing subscription
     *
     * @example <caption>Returns a PushSubscription object containing details of an existing subscription.</caption>
     *
     * const subscription = await apex.pwa.getPushSubscription();
     */
    pwa.getPushSubscription = async () => {
        if ( !isPushUsable() ) { return; }

        const serviceWorkerRegistration = await navigator.serviceWorker.ready;

        if ( serviceWorkerRegistration ) {
            const subscription = await serviceWorkerRegistration.pushManager.getSubscription();

            if ( subscription ) {
                return subscription;
            }
        }
    };

    /**
     * <p>This function is used to verify if the current user is subscribed to push notifications.</p>
     *
     * @function hasPushSubscription
     * @memberof apex.pwa
     * @return {Promise} A Promise that resolves a boolean, based on the if the user is subscribed to push notifications.
     *
     * @example <caption>Returns if the current user is subscribed to push notifications.</caption>
     *
     * const hasPushSubscription = await apex.pwa.hasPushSubscription();
     */
    pwa.hasPushSubscription = async () => {

        // If push is not usable, early exit and return false
        if ( !isPushUsable() ) {
            return false;
        }

        // Returns a SHA-256 hash of the subscription object, which is used to compare with the list of push subscriptions
        // stored in the database and available as an array through pwa.pushSubscriptions
        const getSubscriptionHash = async ( subscription ) => {
            const msgUint8 = new TextEncoder().encode( subscription );                            // encode as (utf-8) Uint8Array
            const hashBuffer = await crypto.subtle.digest( 'SHA-256', msgUint8 );                 // hash the message
            const hashArray = Array.from( new Uint8Array( hashBuffer ) );                         // convert buffer to byte array
            const hashHex = hashArray.map( ( b ) => b.toString(16).padStart( 2, '0' ) ).join(''); // convert bytes to hex string
            return hashHex;
        };

        const subscription = await pwa.getPushSubscription();

        if ( subscription && pwa.pushSubscriptions && pwa.pushSubscriptions.length ) {
            // Get the hash of the subscription object
            const subscriptionHash = await getSubscriptionHash( JSON.stringify( subscription.toJSON() ) );

            // Check if the hash of the subscription object is in the list of push subscriptions output from the server
            return pwa.pushSubscriptions.some( pushSubscription => pushSubscription.toLowerCase() === subscriptionHash.toLowerCase() );
        }

        return false;
    };

    /**
     * <p>This function is used to subscribe the current user to push notifications.</p>
     *
     * @function subscribePushNotifications
     * @memberof apex.pwa
     *
     * @example <caption>Subscribe the current user to push notifications.</caption>
     *
     * await apex.pwa.subscribePushNotifications();
     */
    pwa.subscribePushNotifications = async () => {
        if ( !isPushUsable() ) { return; }

        const spinner = util.showSpinner();

        const serviceWorkerRegistration = await navigator.serviceWorker.ready;

        let subscription = await pwa.getPushSubscription();

        if ( !subscription ) {
            // Prompt notification subscription
            try {
                subscription = await serviceWorkerRegistration.pushManager.subscribe({
                    userVisibleOnly: true,
                    applicationServerKey: pwa.publicKey
                });
            } catch ( error ) {

                debug.error( `Push subscription error: ${ error.message }` );

                // Message code containing the error message for the end user
                const messageCode = "APEX.PWA.PUSH.SUBSCRIPTION_FAILED";

                // Load the error message for the end user
                await lang.loadMessages( [ messageCode ] );

                // Show the error message to the end user
                message.clearErrors();
                message.showErrors( [ {
                    type: "error",
                    location: "page",
                    message: lang.getMessage( messageCode ),
                    unsafe: false
                } ] );
            }
        }

        if ( subscription ) {
            // Save the subscription
            await server.widget( "pwa", {
                x01: "subscribe_push_notifications",
                x02: JSON.stringify( subscription.toJSON() )
            } );
        }

        spinner.remove();
    };

    /**
     * <p>This function is used to unsubscribe the current user to push notifications.</p>
     *
     * @function unsubscribePushNotifications
     * @memberof apex.pwa
     *
     * @example <caption>Unsubscribe the current user to push notifications.</caption>
     *
     * await apex.pwa.unsubscribePushNotifications();
     */
    pwa.unsubscribePushNotifications = async () => {
        if ( !isPushUsable() ) { return; }

        const spinner = util.showSpinner();
        const subscription = await pwa.getPushSubscription();

        if ( subscription ) {
            await server.widget( "pwa", {
                x01: "unsubscribe_push_notifications",
                x02: JSON.stringify( subscription.toJSON() )
            });

            await subscription.unsubscribe();
        }

        spinner.remove();
    };

} )( apex.pwa, apex.lang, apex.jQuery, apex.debug, apex.theme, apex.message, apex.actions, apex.util, apex.server, apex.env, apex.item );
