/*!
 * Copyright (c) 1999, 2023, Oracle and/or its affiliates.
 */

/*
 * This file holds the main APEX namespace and settings for jQuery.
 */

/**
 * <p>The apex namespace is the top level Oracle APEX namespace and contains a number of sub namespaces,
 * and a few common functions and properties.</p>
 *
 * <p>The apex namespace also contains information on APEX specific events.</p>
 *
 * @namespace apex
 */
window.apex = window.apex || {};

/**
 * This namespace property holds all the {@link item} interfaces that have been created on this page.
 * See {@link apex.fn:item|apex.item} for details.
 * @memberOf apex
 * @since 21.2
 */
apex.items = {};

/**
 * This namespace property holds all the {@link region} interfaces that have been created on this page.
 * See {@link apex.fn:region|apex.region} for details.
 * @memberOf apex
 * @since 21.2
 */
apex.regions = {};

/**
 * This object holds various environment values related to the APEX instance, current application and page.
 * @member {object} env
 * @memberof apex
 * @property {string} APP_USER           The current username
 * @property {string} APP_ID             The application ID
 * @property {string} APP_PAGE_ID        The page ID
 * @property {string} APP_SESSION        The session ID
 * @property {string} APP_FILES          The relative path of the application static files
 * @property {string} WORKSPACE_FILES    The relative path of the workspace static files
 * @property {string} APEX_FILES         The relative path of the files distributed with Oracle APEX
 * @property {string} APEX_VERSION       The full version of the Oracle APEX instance
 * @property {string} APEX_BASE_VERSION  The base version of the Oracle APEX instance
 *
 * @example <caption>Redirect to page 2 in the current application.</caption>
 * apex.navigation.redirect( "f?p=" + apex.env.APP_ID + ":2:" + apex.env.APP_SESSION );
 *
 * @since 21.2
 */

// apex.env is already generated by the server, with {} being a fallback
apex.env = apex.env || {};

// completing apex.env with the values already available on the page
apex.env.APEX_FILES = window.apex_img_dir || "";

/**
 * <p>This namespace property holds the jQuery function that APEX uses. Ideally there is just one copy
 * of jQuery on a page but it is possible to have multiple copies and even different versions of jQuery on a page.
 * This is sometimes necessary when using third party plugins that only work with an older version of jQuery.
 * Use this property in place of global variables $ or jQuery to ensure you are using the same jQuery library that
 * APEX is using.</p>
 *
 * @type function
 * @example <caption>The following function creates a local variable $ as a convenient way to reference jQuery
 * while ensuring that it is using the same jQuery that APEX uses.</caption>
 * function myFunction() {
 *     var $ = apex.jQuery;
 *     // use $ to access jQuery functionality
 * }
 */
apex.jQuery = jQuery;

(function( $ ) {
    "use strict";

/**
 * <p>This namespace property stores the current page context. The current page context is set to the HTML document
 * (same as apex.jQuery(document)).</p>
 *
 * @type jQuery
 * @example <caption> This selects all elements with a CSS class of my_class, in the context of the current page.</caption>
 * apex.jQuery( ".my_class", apex.gPageContext$ );
 */
apex.gPageContext$ = $( document );

apex.gParentPageContext$ = apex.gPageContext$;

/*
 * Set up an error handler that can inform the dev toolbar that an error happened.
 * When dev toolbar is not present this does nothing.
 * todo consider in the future to use this for actual persistent logging of the errors
 * todo the intention behind calling window.onerror in server and dynamic_actions_core changes due to this
 */
function notifyErrors() {
    if ( apex._dtNotifyErrors ) {
        apex._dtNotifyErrors();
    } else {
        apex._pageHasErrors = true;
    }
}
const originalOnError = window.onerror; // If someone else has set up a handler chain to it. Hope any that come after are as kind.
window.onerror = function( messageOrEvent, source, lineno, colno, error ) {
    notifyErrors();
    if ( originalOnError ) {
        originalOnError( messageOrEvent, source, lineno, colno, error );
    }
};

// these jQuery errors are logged but don't flow through onerror
const originalJQueryDeferredExceptionHook = $.Deferred.exceptionHook;
if ( originalJQueryDeferredExceptionHook ) {
    $.Deferred.exceptionHook = function( error, stack ) {
        originalJQueryDeferredExceptionHook( error, stack );
        // check for error and not an ajax abort to avoid indicating an error when there is no error logged to the console
        if ( error && error.statusText !== "abort" ) {
            notifyErrors();
        }
    };
}

    /**
     * <p>This event is triggered on the window a couple hundred milliseconds after the window stops resizing.
     * Listen for this event to adjust or resize page content after the window is done resizing. In some cases this is
     * a better alternative to the window resize event, which is triggered many times as the window is being resized,
     * because it is triggered just once after the window stops resizing.</p>
     *
     * @event apexwindowresized
     * @memberof apex
     * @property {Event} Event <code class="prettyprint">jQuery</code> event object
     *
     * @example <caption>This example responds to the apexwindowresized event and updates page content based on
     * the new height and width.</caption>
     * apex.jQuery( window ).on( "apexwindowresized", function( event ) {
     *     var window$ = apex.jQuery( this ),
     *         height = window$.height(),
     *         width = window$.width();
     *     // update page content based on new window height and width
     * });

     */

    let resizeTimerId,
        lastStoredHeight = 0,
        lastStoredWidth = 0;

    // A simple debouncer for page resize events.
    $( window ).resize(function() {
        let window$ = $( window );
        // Certain plugins (Flotchart) and browsers (IE 8 and below) sometimes spam window resize events when the window
        // is not actually resizing. This guard prevents such spam events from triggering
        // an apexwindowresized event, by checking to see if the window height and width has changed since the
        // event was last fired.
        if ( window$.height() === lastStoredHeight && window$.width() === lastStoredWidth) {
            return;
        }
        lastStoredHeight = window$.height();
        lastStoredWidth = window$.width();
        if ( resizeTimerId ) {
            clearTimeout( resizeTimerId );
        }
        resizeTimerId = setTimeout( function() {
            $( window ).trigger( "apexwindowresized" );
            resizeTimerId = null;
        }, 200);
    });

    let gUserHasTouched = false; // set to true if the user has touched the browser/device since the session started

    $( function() {
        let lTestColour;

        /* The following code detects high contrast mode. It works because in high contrast mode the reported
           background color will not be the same as it was explicitly set.*/

        // Add an element with explicit background color set
        $( "body" ).append( "<p class='hcmtest'></p>" );

        // Get the background color
        lTestColour = $( ".hcmtest" ).css( "background-color" ).toLowerCase();

        // Remove the testing DIV, we don't need it anymore
        $( ".hcmtest" ).remove();

        // Different browsers return the color in different ways
        if ( lTestColour !== "#878787" && lTestColour !== "rgb(135, 135, 135)" ) {

            // add utility class to body tag
            $( "body").addClass( "u-HCM" );

        }

        /*
         * Code to determine if the user is touching their device. This is saved in session storage
         */
        const sessionStore = apex.storage.getScopedSessionStorage( { prefix: "ORA_WWV_apex.core", useAppId: false} );
        if ( sessionStore.getItem( "userHasTouched" ) === "y" ) {
            gUserHasTouched = true;
        } else {
            // if user hasn't touched yet perhaps they will
            $( document.body ).one( "touchstart", function () {
                gUserHasTouched = true;
                // save in session
                sessionStore.setItem( "userHasTouched", "y" );
            } );
        }

        // Various overrides for jQuery UI based dialogs
        if ( $.ui && $.ui.dialog ) {
            $.widget( "ui.dialog", $.ui.dialog, {
                _create: function() {
                    this._super();

                    // Add aria-label to the close button. See #33670212
                    this.uiDialogTitlebarClose.attr( "aria-label", this.options.closeText );

                    // Add aria-describedby to the close button. See #33996908
                    this.uiDialogTitlebarClose.attr( "aria-describedby", $( "span.ui-dialog-title", this.uiDialogTitlebar ).attr( "id" ) );

                    // Bug 35378244 - MODAL DIALOGS HAVE ARIA-DESCRIBEDBY POINTING TO THE DIALOG CONTENT
                    // Remove aria-describedby from the dialog because it breaks accessible descriptions
                    // applied to elements inside of the dialog
                    this.uiDialog.removeAttr( "aria-describedby" );

                    // To simplify dialog styling, we:
                    //  - remove the "Close" text node of the X button (it already has the title and aria-label attributes which is enough)
                    //  - remove the span.ui-button-icon-space element
                    // See #34259635
                    this.uiDialogTitlebarClose.contents().filter( ( i, elem ) => ( elem.nodeType === Node.TEXT_NODE ) ).remove();
                    this.uiDialogTitlebarClose.find( "span.ui-button-icon-space" ).remove();

                    // Dialog titles should be h1, not span
                    // The title can still be updated dynamically, as the internal jqueryui reference
                    //  is based on the ui-dialog-title class.
                    // See #35378244
                    const currentTitle$ = this.uiDialogTitlebar.find( ".ui-dialog-title" );
                    const newTitle$ = $( "<h1>" ).attr( {
                        "id": currentTitle$.attr( "id" ),
                        "class": currentTitle$.attr( "class" ),
                        "aria-roledescription": apex.lang.getMessage( "APEX.DIALOG.TITLE" )
                    } ).html( currentTitle$.html() );
                    currentTitle$.replaceWith( newTitle$ );
                },
                /*
                 * See bug 34333105
                 * In Firefox, non-chained modal dialogs have trouble receiving focus when being opened via the keyboard.
                 * jQuery appears to erroneously trigger a focusin event on the whole document, right before triggering the correct one.
                 * Through a series of events, focus is denied and returned to the previous dialog, with no way to set it back.
                 * This has also been observed and mentioned here: github.com/jquery/jquery-ui/pull/1108
                 * The simplest solution is to account for this event and not deny the dialog interaction because of it.
                 */
                _allowInteraction: function( event ) {
                    return event.target === document || this._super( event );
                }
            } );

            /*
             * Variants of the hide()/show() functions that apply the is-closing class
             * which allows for a CSS class-based closing animation.
             * Dialogs can opt in by using this effect as such:
             *
             * .dialog({
             *     show: {
             *         effect: "apexDialogShow"
             *     }
             *     hide: {
             *         effect: "apexDialogHide"
             *     }
             * })
             *
             * Do not call these functions directly.
             * They will be invoked by the jQuery internals as needed.
             */
            $.fn.apexDialogHide = function( duration, easing, callback ) {
                const elem$ = this;
                const dur = apex.util.cssDurationToMilliseconds( elem$.css( "--js-dialog-close-timing" ) || "0ms" );
                const cls = "is-closing";

                elem$.addClass( cls );
                setTimeout( () => {
                    // only fully hide if the dialog was not re-opened during the closing animation
                    if( elem$.hasClass( cls ) ) {
                        elem$.removeClass( cls ).hide();
                        callback?.();
                    }
                }, dur );
            };

            $.fn.apexDialogShow = function( duration, easing, callback ) {
                // remove closing class just in case a close is ongoing
                this.removeClass( "is-closing" ).show();
                callback?.();
            };
        }

        /*
         * Logic to prevent user interaction of APEX readonly fields. Any element with class 'js-accessible-readonly'
         * will participate in this logic.
         */
        $( "body" ).on( "beforeinput", ".js-accessible-readonly", function( e ) {
            e.preventDefault();
        });

        // Adding the js-ready class after page rendering allows some components to be displayed smoothly
        queueMicrotask( () => {
            $( "body" ).addClass( "js-ready" );
        } );

    } );

    /**
     * <p>Determine if the user is or has been interacting with this web app using touch since the browser session
     * began. Note: it is possible for the user to touch for the first time after this function is called.</p>
     *
     * <p>It is rare to need know this information since the app should be designed to work for both touch and non-touch environments.</p>
     *
     * @returns {boolean} true if the user has been using touch to interact with the web app and false otherwise.
     */
    apex.userHasTouched = function() {
        return gUserHasTouched;
    };

 })( apex.jQuery );
