set define '^' verify off
prompt ...wwv_flow_string.sql
create or replace package wwv_flow_string authid definer as
--------------------------------------------------------------------------------
--
-- Copyright (c) Oracle Corporation 1999 - 2024. All Rights Reserved.
--
-- NAME
--   wwv_flow_string.sql - apex_string
--
-- DESCRIPTION
--   Utilities for varchar2, clob, apex_t_varchar2, apex_t_clob and
--   apex_t_number types.
--   Unless otherwise noted, the APIs expect arrays to be continuous (i.e.
--   without gaps that were generated by coll.delete(n) operations).
--
--   The major features of this package are:
--
--   * push:            Append values to a table.
--   * split%:          Split strings at separator patterns.
--   * string_to_table: Version of split for apex_application_global.vc_arr2,
--                      separator defaults to ':'.
--   * next_chunk:      Read chunks of strings from a clob, with increasing
--                      offset.
--   * join%:           Concatenate table values with separators in between.
--   * table_to_string: Version of join for apex_application_global.vc_arr2,
--                      separator defaults to ':'.
--   * grep:            Return values of the input table or string that match a
--                      regular expression.
--   * shuffle:         Randomly re-order the values of the input table.
--   * plist_*:         Retrieve, set and delete values of a property list
--                      (plist), which is a key/value store based on
--                      apex_t_varchar2. Entries at odd table positions (1, 3,
--                      ...) are keys, while entries at even positions (2,
--                      4, ...) are values. Native index by varchar2 PL/SQL
--                      tables are more efficient, but this structure is
--                      sometimes more convenient to use, e.g. when passing
--                      multiple keys/values as parameters.
--   * format:          Return a formatted string, with substitutions applied.
--   * get_initials:    Get N letter initials from the first N words.
--   * get_searchable_phrases: Return only the phrases in the given input
--                      strings, which are relevant for searching.
--
-- RUNTIME DEPLOYMENT: YES
-- PUBLIC:             YES
--
-- MODIFIED   (MM/DD/YYYY)
--   cneumuel  08/09/2016 - Created
--
--------------------------------------------------------------------------------

--==============================================================================
-- Append value to apex_t_varchar2 table.
--
-- PARAMETERS
-- * p_table:  The table.
-- * p_value:  The value to be added.
--
-- EXAMPLE
--   Append 2 values, then print the table.
--
--   declare
--       l_table apex_t_varchar2;
--   begin
--       apex_string.push(l_table, 'a');
--       apex_string.push(l_table, 'b');
--       sys.dbms_output.put_line(apex_string.join(l_table, ':'));
--   end;
--   -> a:b
--==============================================================================
procedure push (
    p_table in out nocopy wwv_flow_t_varchar2,
    p_value in            varchar2 );

--==============================================================================
-- Append value to apex_t_clob table.
--
-- PARAMETERS
-- * p_table:  The table.
-- * p_value:  The value to be added.
--
-- EXAMPLE
--   Append 2 values, then print the table.
--
--   declare
--       l_table apex_t_clob;
--   begin
--       apex_string.push(l_table, 'a');
--       apex_string.push(l_table, 'b');
--       sys.dbms_output.put_line(apex_string.join_clobs(l_table, ':'));
--   end;
--   -> a:b
--==============================================================================
procedure push (
    p_table in out nocopy wwv_flow_t_clob,
    p_value in            clob );

--==============================================================================
-- Append value to apex_t_number table.
--
-- PARAMETERS
-- * p_table:  The table.
-- * p_value:  The value to be added.
--
-- EXAMPLE
--   Append 2 values, then print the table.
--
--   declare
--       l_table apex_t_number;
--   begin
--       apex_string.push(l_table, 1);
--       apex_string.push(l_table, 2);
--       sys.dbms_output.put_line(apex_string.join(l_table, ':'));
--   end;
--   -> 1:2
--==============================================================================
procedure push (
    p_table in out nocopy wwv_flow_t_number,
    p_value in number );

--==============================================================================
-- Append collection values to apex_t_varchar2 table.
--
-- PARAMETERS
-- * p_table:  The table.
-- * p_values: Values that should be added to p_table.
--
-- EXAMPLE
--   Append single value and multiple values, then print the table.
--
--   declare
--       l_table apex_t_varchar2;
--   begin
--       apex_string.push(l_table, 'a');
--       apex_string.push(l_table, apex_t_varchar2('1','2','3'));
--       sys.dbms_output.put_line(apex_string.join(l_table, ':'));
--   end;
--   -> a:1:2:3
--==============================================================================
procedure push (
    p_table  in out nocopy wwv_flow_t_varchar2,
    p_values in            wwv_flow_t_varchar2 );

--==============================================================================
-- Append number collection values to apex_t_varchar2 table.
--
-- PARAMETERS
-- * p_table:       The table.
-- * p_values:      Values that should be added to p_table.
-- * p_format_mask: Format mask to use when converting numbers to strings.
--
-- EXAMPLE
--   Append single value and multiple values, then print the table.
--
--   declare
--       l_table apex_t_varchar2;
--   begin
--       apex_string.push(l_table, 'a');
--       apex_string.push(l_table, apex_t_number(1,2,3), 'FM990D00');
--       sys.dbms_output.put_line(apex_string.join(l_table, ':'));
--   end;
--   -> a:1.00:2.00:3.00
--==============================================================================
procedure push (
    p_table       in out nocopy wwv_flow_t_varchar2,
    p_values      in            wwv_flow_t_number,
    p_format_mask in            varchar2             default null );

--==============================================================================
-- Append collection values to apex_t_clob table.
--
-- PARAMETERS
-- * p_table:  The table.
-- * p_values: Values that should be added to p_table.
--
-- EXAMPLE
--   Append single value and multiple values, then print the table.
--
--   declare
--       l_table apex_t_clob;
--   begin
--       apex_string.push(l_table, 'a');
--       apex_string.push(l_table, apex_t_clob('1','2','3'));
--       sys.dbms_output.put_line(apex_string.join_clobs(l_table, ':'));
--   end;
--   -> a:1:2:3
--==============================================================================
procedure push (
    p_table  in out nocopy wwv_flow_t_clob,
    p_values in            wwv_flow_t_clob );

--==============================================================================
-- Append values of a PL/SQL table to apex_t_varchar2 table.
--
-- PARAMETERS
-- * p_table:  The table.
-- * p_values: Values that should be added to p_table.
--
-- EXAMPLE
--   Append, then print the table.
--
--   declare
--       l_table  apex_t_varchar2;
--       l_values apex_application_global.vc_arr2;
--   begin
--       l_values(1) := 'a';
--       l_values(2) := 'b';
--       apex_string.push(l_table, l_values);
--       sys.dbms_output.put_line(apex_string.join(l_table, ':'));
--   end;
--   -> a:b
--==============================================================================
procedure push (
    p_table  in out nocopy wwv_flow_t_varchar2,
    p_values in            wwv_flow_global.vc_arr2 );

--==============================================================================
-- Split input string at separator.
--
-- PARAMETERS
-- * p_str:    The input string.
-- * p_sep:    The separator. If null, split after each character. If a single
--             character, split at this character. If more than 1 character,
--             split at regular expression (max 512 characters).
--             The default is to split at line feed.
-- * p_limit:  Maximum number of splits, ignored if null. If smaller than the
--             total possible number of splits, the last table element contains
--             the rest.
--
-- EXAMPLES
--   apex_string.split(1||chr(10)||2||chr(10)||3)
--   -> apex_t_varchar2('1','2','3')
--
--   apex_string.split('1:2:3',':')
--   -> apex_t_varchar2('1','2','3')
--
--   apex_string.split('123',null)
--   -> apex_t_varchar2('1','2','3')
--
--   apex_string.split('1:2:3:4',':',2)
--   -> apex_t_varchar2('1','2:3:4')
--
--   apex_string.split('key1=val1, key2=val2','\s*[=,]\s*')
--   -> apex_t_varchar2('key1','val1','key2','val2')
--==============================================================================
function split (
    p_str   in varchar2,
    p_sep   in varchar2    default wwv_flow.LF,
    p_limit in pls_integer default null )
    return wwv_flow_t_varchar2;

--==============================================================================
-- Split input clob at separator. Returns a table of varchar2.
--
-- PARAMETERS
-- * p_str:    The input clob.
-- * p_sep:    The separator. If null, split after each character. If a single
--             character, split at this character. If more than 1 character,
--             split at regular expression (max 512 characters).
--             The default is to split at line feed.
--
-- EXAMPLE
--   apex_string.split('1:2:3',':')
--   -> apex_t_varchar2('1','2','3')
--==============================================================================
function split (
    p_str   in clob,
    p_sep   in varchar2    default wwv_flow.LF )
    return wwv_flow_t_varchar2;

--==============================================================================
-- Split input clob at separator. Returns a table of clobs.
--
-- PARAMETERS
-- * p_str:    The input clob.
-- * p_sep:    The separator. If null, split after each character. If a single
--             character, split at this character. If more than 1 character,
--             split at regular expression (max 512 characters).
--             The default is to split at line feed.
-- * p_limit:  Maximum number of splits, ignored if null. If smaller than the
--             total possible number of splits, the last table element contains
--             the rest.
--
-- EXAMPLE
--   apex_string.split('1:2:3',':')
--   -> apex_t_clob('1','2','3')
--==============================================================================
function split_clobs (
    p_str   in clob,
    p_sep   in varchar2    default wwv_flow.LF,
    p_limit in pls_integer default null )
    return wwv_flow_t_clob;

--==============================================================================
-- Split input at separator, values must all be numbers.
--
-- PARAMETERS
-- * p_str:    The input varchar2.
-- * p_sep:    The separator. If null, split after each character. If a single
--             character, split at this character. If more than 1 character,
--             split at regular expression (max 512 characters).
--             The default is to split at line feed.
--
-- EXAMPLE
--   apex_string.split_numbers('1:2:3',':')
--   -> apex_t_number(1,2,3)
--==============================================================================
function split_numbers (
    p_str   in varchar2,
    p_sep   in varchar2 default wwv_flow.LF )
    return wwv_flow_t_number;

--==============================================================================
-- Split input at separator, returning a vc_arr2.
--
-- PARAMETERS
-- * p_str:    The input varchar2.
-- * p_sep:    The separator (no regexp or split at char). Defaults to ':'.
--
-- EXAMPLE
--   declare
--       l_result apex_application_global.vc_arr2;
--   begin
--       l_result := apex_string.string_to_table('1:2:3',':');
--       sys.dbms_output.put_line(apex_string.table_to_string(l_result,'-'));
--   end;
--   -> 1-2-3
--==============================================================================
function string_to_table (
    p_str   in varchar2,
    p_sep   in varchar2    default ':' )
    return wwv_flow_global.vc_arr2;

--==============================================================================
-- Read a fixed-length string from a clob. This is just a small wrapper around
-- DBMS_LOB.READ, but it prevents common errors when incrementing the offset and
-- picking the maximum chunk size.
--
-- PARAMETERS
--   * p_str:    The input clob.
--   * p_chunk:  The chunk value (in/out).
--   * p_offset: The position in p_str, where the next chunk should be read from
--               (in/out).
--   * p_amount: The amount of characters that should be read (default 8191).
--
-- RETURNS
--   True if another chunk could be read. False if reading past the end of
--   p_str.
--
-- EXAMPLE
--   Print chunks of 20 bytes of the input clob.
--
--     declare
--         l_input  clob := 'The quick brown fox jumps over the lazy dog';
--         l_offset integer;
--         l_chunk  varchar2(20 char);
--     begin
--         while apex_string.next_chunk (
--                   p_str    => l_input,
--                   p_chunk  => l_chunk,
--                   p_offset => l_offset,
--                   p_amount => 20 )
--         loop
--             sys.dbms_output.put_line(l_chunk);
--         end loop;
--     end;
--
--   Output:
--     The quick brown fox
--     jumps over the lazy
--     dog
--
--==============================================================================
function next_chunk (
    p_str    in            clob,
    p_chunk  out           nocopy varchar2,
    p_offset in out nocopy integer,
    p_amount in            integer default 8191 )
    return boolean;

--==============================================================================
-- Return the values of the apex_t_varchar2 input table p_table as a
-- concatenated varchar2, separated by p_sep.
--
-- PARAMETERS
-- * p_table:  The input table.
-- * p_sep:    The separator, default is line feed.
--
-- EXAMPLE
--   Concatenate numbers, separated by ':'.
--
--   apex_string.join(apex_t_varchar2('a','b','c'),':')
--   -> a:b:c
--==============================================================================
function join (
    p_table in wwv_flow_t_varchar2,
    p_sep   in varchar2             default wwv_flow.LF)
    return varchar2;

--==============================================================================
-- Return the values of the apex_t_number input table p_table as a concatenated
-- varchar2, separated by p_sep.
--
-- PARAMETERS
-- * p_table:  The input table.
-- * p_sep:    The separator, default is line feed.
--
-- EXAMPLE
--   Concatenate numbers, separated by ':'.
--
--   apex_string.join(apex_t_number(1,2,3),':')
--   -> 1:2:3
--==============================================================================
function join (
    p_table in wwv_flow_t_number,
    p_sep   in varchar2 default wwv_flow.LF )
    return varchar2;

--==============================================================================
-- Return the values of the apex_t_varchar2 input table p_table as a
-- concatenated clob, separated by p_sep.
--
-- PARAMETERS
-- * p_table:  The input table.
-- * p_sep:    The separator, default is line feed.
-- * p_dur:    The duration of the clob, default sys.dbms_lob.call.
--
-- EXAMPLE
--   Concatenate numbers, separated by ':'
--
--   apex_string.join_clob(apex_t_varchar2('1','2','3'),':')
--   -> 1:2:3
--==============================================================================
function join_clob (
    p_table in wwv_flow_t_varchar2,
    p_sep   in varchar2             default wwv_flow.LF,
    p_dur   in pls_integer          default sys.dbms_lob.call )
    return clob;

--==============================================================================
-- Return the values of the apex_application_global.vc_arr2 input table p_table as a
-- concatenated clob, separated by p_sep.
--
-- PARAMETERS
-- * p_table:  The input table.
-- * p_sep:    The separator, default is line feed.
-- * p_dur:    The duration of the clob, default sys.dbms_lob.call.
--
-- EXAMPLE
--   Concatenate numbers, separated by ':'
--
--   declare
--       l_table apex_application_global.vc_arr2;
--   begin
--       l_table(1) := '1';
--       l_table(2) := '2';
--       l_table(3) := '3';
--
--       sys.dbms_output.put_line(apex_string.table_to_clob(l_table, ':'));
--   end;
--   -> 1:2:3
--==============================================================================
function table_to_clob (
    p_table in wwv_flow_global.vc_arr2,
    p_sep   in varchar2             default wwv_flow.LF,
    p_dur   in pls_integer          default sys.dbms_lob.call )
    return clob;

--==============================================================================
-- Return the values of the apex_t_clob input table p_table as a
-- concatenated clob, separated by p_sep.
--
-- PARAMETERS
-- * p_table:  The input table.
-- * p_sep:    The separator, default is line feed.
-- * p_dur:    The duration of the clob, default sys.dbms_lob.call.
--
-- EXAMPLE
--   Concatenate numbers, separated by ':'
--
--   apex_string.join_clob(apex_t_clob('1','2','3'),':')
--   -> 1:2:3
--==============================================================================
function join_clobs (
    p_table in wwv_flow_t_clob,
    p_sep   in varchar2             default wwv_flow.LF,
    p_dur   in pls_integer          default sys.dbms_lob.call )
    return clob;

--==============================================================================
-- Return the values of the apex_application_global.vc_arr2 input table p_table
-- as a concatenated varchar2, separated by p_sep.
--
-- PARAMETERS
-- * p_table:  The input table, assumes no holes and index starts at 1.
-- * p_sep:    The separator, default is ':'.
--
-- EXAMPLE
--   Concatenate numbers, separated by ':'.
--
--   declare
--       l_table apex_application_global.vc_arr2;
--   begin
--       l_table(1) := 'a';
--       l_table(2) := 'b';
--       l_table(3) := 'c';
--       sys.dbms_output.put_line(apex_string.table_to_string(l_table));
--   end;
--   -> a:b:c
--==============================================================================
function table_to_string (
    p_table in wwv_flow_global.vc_arr2,
    p_sep   in varchar2             default ':' )
    return varchar2;

--==============================================================================
-- Return values of the input table that match a regular expression.
--
-- PARAMETERS
-- * p_table:         The input table.
-- * p_pattern:       The regular expression.
-- * p_modifier:      The regular expression modifier.
-- * p_subexpression: The subexpression which should be returned. If null,
--                    return the complete table value. If 0 (the default),
--                    return the matched expression. If > 0, return the
--                    subexpression value. You can also pass a comma separated
--                    list of numbers, to get multiple subexpressions in the
--                    result.
-- * p_limit:         Limitation for the number of elements in the return table.
--                    If null (the default), there is no limit.
--
-- EXAMPLE
--   Collect and print basenames of sql files in input collection
--
--   declare
--       l_sqlfiles apex_t_varchar2;
--   begin
--       l_sqlfiles := apex_string.grep (
--                         p_table         => apex_t_varchar2('a.html','b.sql', 'C.SQL'),
--                         p_pattern       => '(\w+)\.sql',
--                         p_modifier      => 'i',
--                         p_subexpression => '1' );
--       sys.dbms_output.put_line(apex_string.join(l_sqlfiles, ':'));
--   end;
--   -> b:C
--==============================================================================
function grep (
    p_table         in wwv_flow_t_varchar2,
    p_pattern       in varchar2,
    p_modifier      in varchar2             default null,
    p_subexpression in varchar2             default '0',
    p_limit         in pls_integer          default null )
    return wwv_flow_t_varchar2;

--==============================================================================
-- Return values of the input varchar2 that match a regular expression.
--
-- PARAMETERS
-- * p_str:           The input varchar2.
-- * p_pattern:       The regular expression.
-- * p_modifier:      The regular expression modifier.
-- * p_subexpression: The subexpression which should be returned. If null,
--                    return the line where a match is found. If 0 (the
--                    default), return the matched expression. If > 0, return
--                    subexpression value. You can also pass a comma separated
--                    list of numbers, to get multiple subexpressions in the
--                    result.
-- * p_limit:         Limitation for the number of elements in the return table.
--                    If null (the default), there is no limit.
--
-- EXAMPLE
--   Collect and print key=value definitions
--
--   declare
--       l_plist apex_t_varchar2;
--   begin
--       l_plist := apex_string.grep (
--                      p_str           => 'define k1=v1'||chr(10)||
--                                         'define k2 = v2',
--                      p_pattern       => 'define\s+(\w+)\s*=\s*([^'||chr(10)||']*)',
--                      p_modifier      => 'i',
--                      p_subexpression => '1,2' );
--       sys.dbms_output.put_line(apex_string.join(l_plist, ':'));
--   end;
--   -> k1:v1:k2:v2
--==============================================================================
function grep (
    p_str           in varchar2,
    p_pattern       in varchar2,
    p_modifier      in varchar2             default null,
    p_subexpression in varchar2             default '0',
    p_limit         in pls_integer          default null )
    return wwv_flow_t_varchar2;

--==============================================================================
-- Return values of the input clob that match a regular expression.
--
-- PARAMETERS
-- * p_str:           The input clob.
-- * p_pattern:       The regular expression.
-- * p_modifier:      The regular expression modifier.
-- * p_subexpression: The subexpression which should be returned. If null,
--                    return the line where a match is found. If 0 (the
--                    default), return the matched expression. If > 0, return
--                    subexpression value. You can also pass a comma separated
--                    list of numbers, to get multiple subexpressions in the
--                    result.
-- * p_limit:         Limitation for the number of elements in the return table.
--                    If null (the default), there is no limit.
--
-- EXAMPLE
--   Collect and print key=value definitions
--
--   declare
--       l_plist apex_t_varchar2;
--   begin
--       l_plist := apex_string.grep (
--                      p_str           => to_clob('define k1=v1'||chr(10)||
--                                                 'define k2 = v2'),
--                      p_pattern       => 'define\s+(\w+)\s*=\s*([^'||chr(10)||']*)',
--                      p_modifier      => 'i',
--                      p_subexpression => '1,2' );
--       sys.dbms_output.put_line(apex_string.join(l_plist, ':'));
--   end;
--   -> k1:v1:k2:v2
--==============================================================================
function grep (
    p_str           in clob,
    p_pattern       in varchar2,
    p_modifier      in varchar2             default null,
    p_subexpression in varchar2             default '0',
    p_limit         in pls_integer          default null )
    return wwv_flow_t_varchar2;

--==============================================================================
-- Randomly re-order the values of the input table.
--
-- PARAMETERS
-- * p_table:         The input table, which will be modified by the procedure.
--
-- EXAMPLE
--   Shuffle and print l_table.
--
--   declare
--       l_table apex_t_varchar2 := apex_string.split('1234567890',null);
--   begin
--       apex_string.shuffle(l_table);
--       sys.dbms_output.put_line(apex_string.join(l_table,':'));
--   end;
--   -> a permutation of 1:2:3:4:5:6:7:8:9:0
--==============================================================================
procedure shuffle (
    p_table         in out nocopy wwv_flow_t_varchar2 );

--==============================================================================
-- Return the input table values, re-ordered.
--
-- PARAMETERS
-- * p_table:         The input table.
--
-- EXAMPLE
--   Shuffle and print l_table.
--
--   declare
--       l_table apex_t_varchar2 := apex_string.split('1234567890',null);
--   begin
--       sys.dbms_output.put_line(apex_string.join(apex_string.shuffle(l_table),':'));
--   end;
--   -> a permutation of 1:2:3:4:5:6:7:8:9:0
--==============================================================================
function shuffle (
    p_table         in wwv_flow_t_varchar2 )
    return wwv_flow_t_varchar2;

--==============================================================================
-- Return whether a key exists in the property list.
--
-- PARAMETERS
-- * p_table:         The input table.
-- * p_key:           The input key.
--
-- RAISES
-- * NO_DATA_FOUND:   Given key does not exist in table.
--
-- EXAMPLE
--   Print whether properties key1, key2 and key3 exist.
--
--   declare
--       l_plist apex_t_varchar2 := apex_t_varchar2('key1','foo','key2','bar');
--   begin
--       for i in 1 .. 3 loop
--           sys.dbms_output.put_line(
--               'key'||i||': '||
--               case apex_string.plist_exists(l_plist,'key'||i)
--               when true then 'TRUE'
--               else 'FALSE'
--               end);
--       end loop;
--   end;
--   -> key1:TRUE
--      key2:TRUE
--      key3:FALSE
--==============================================================================
function plist_exists (
    p_table in wwv_flow_t_varchar2,
    p_key   in varchar2 )
    return boolean;

--==============================================================================
-- Get property list value for key.
--
-- PARAMETERS
-- * p_table:         The input table.
-- * p_key:           The input key.
--
-- RAISES
-- * NO_DATA_FOUND:   Given key does not exist in table.
--
-- EXAMPLE
--   Get value of property "key2".
--
--   declare
--       l_plist apex_t_varchar2 := apex_t_varchar2('key1','foo','key2','bar');
--   begin
--       sys.dbms_output.put_line(apex_string.plist_get(l_plist,'key2'));
--   end;
--   -> bar
--==============================================================================
function plist_get (
    p_table in wwv_flow_t_varchar2,
    p_key   in varchar2 )
    return varchar2;

--==============================================================================
-- Get the first property list key that maps to a given value. Return null if
-- the value can not be found.
--
-- PARAMETERS
-- * p_table:         The input table.
-- * p_value:         The input value.
--
-- EXAMPLE
--   Get key of property "bar".
--
--   declare
--       l_plist apex_t_varchar2 := apex_t_varchar2('key1','foo','key2','bar');
--   begin
--       sys.dbms_output.put_line(apex_string.plist_get_key(l_plist,'bar'));
--   end;
--   -> key2
--==============================================================================
function plist_get_key (
    p_table in wwv_flow_t_varchar2,
    p_value in varchar2 )
    return varchar2;

--==============================================================================
-- Insert or update property list value for key.
--
-- PARAMETERS
-- * p_table:         The input table.
-- * p_key:           The input key.
-- * p_value:         The input value.
--
-- EXAMPLE
--   Set value of property "key2".
--
--   declare
--       l_plist apex_t_varchar2 := apex_t_varchar2('key1','foo');
--   begin
--       apex_string.plist_put(l_plist,'key2','bar');
--       sys.dbms_output.put_line(apex_string.plist_get(l_plist,'key2'));
--   end;
--   -> bar
--==============================================================================
procedure plist_put (
    p_table in out nocopy wwv_flow_t_varchar2,
    p_key   in varchar2,
    p_value in varchar2 );

--==============================================================================
-- Append key/value to the property list, without looking for duplicates.
--
-- PARAMETERS
-- * p_table:         The input table.
-- * p_key:           The input key.
-- * p_value:         The input value.
--
-- EXAMPLE
--   Append key2/bar.
--
--   declare
--       l_plist apex_t_varchar2 := apex_t_varchar2('key1','foo');
--   begin
--       apex_string.plist_push(l_plist,'key2','bar');
--       sys.dbms_output.put_line(apex_string.plist_get(l_plist,'key2'));
--   end;
--   -> bar
--==============================================================================
procedure plist_push (
    p_table in out nocopy wwv_flow_t_varchar2,
    p_key   in varchar2,
    p_value in varchar2 );

--==============================================================================
-- Remove the property list key from the table.
--
-- PARAMETERS
-- * p_table:         The input table.
-- * p_key:           The input key.
--
-- RAISES
-- * NO_DATA_FOUND:   Given key does not exist in table.
--
-- EXAMPLE
--   Remove value of property "key2".
--
--   declare
--       l_plist apex_t_varchar2 := apex_t_varchar2('key1','foo','key2','bar');
--   begin
--       apex_string.plist_delete(l_plist,'key2');
--       sys.dbms_output.put_line(apex_string.join(l_plist,':'));
--   end;
--   -> key1:foo
--==============================================================================
procedure plist_delete (
    p_table in out nocopy wwv_flow_t_varchar2,
    p_key   in varchar2 );

--==============================================================================
-- Return a formatted string, with substitutions applied.
--
-- Returns p_message after replacing each <n>th occurrence of %s with p<n> and
-- each occurrence of %<n> with p<n>. If p_max_length is not null,
-- substr(p<n>,1,p_arg_max_length) is used instead of p<n>. Use %% in p_message
-- to emit a single % character. Use %n to emit a newline.
--
-- PARAMETERS
-- * p_message:       Message string with substitution placeholders.
-- * p0-p19:          Substitution parameters.
-- * p_max_length:    If not null (default is 1000), cap each p<n> at
--                    p_max_length characters.
-- * p_prefix:        If set, remove leading white space and the given prefix
--                    from each line. This parameter can be used to simplify
--                    the formatting of indented multi-line text.
--
-- EXAMPLE
--   apex_string.format('%s+%s=%s', 1, 2, 'three')
--   -> 1+2=three
--
--   apex_string.format('%1+%2=%0', 'three', 1, 2)
--   -> 1+2=three
--
--   apex_string.format (
--       q'!begin
--         !    if not valid then
--         !        apex_debug.info('validation failed');
--         !    end if;
--         !end;!',
--       p_prefix => '!' )
--   -> begin
--          if not valid then
--              apex_debug.info('validation failed');
--          end if;
--      end;
--
--==============================================================================
function format (
    p_message    in varchar2,
    p0           in varchar2    default null,
    p1           in varchar2    default null,
    p2           in varchar2    default null,
    p3           in varchar2    default null,
    p4           in varchar2    default null,
    p5           in varchar2    default null,
    p6           in varchar2    default null,
    p7           in varchar2    default null,
    p8           in varchar2    default null,
    p9           in varchar2    default null,
    p10          in varchar2    default null,
    p11          in varchar2    default null,
    p12          in varchar2    default null,
    p13          in varchar2    default null,
    p14          in varchar2    default null,
    p15          in varchar2    default null,
    p16          in varchar2    default null,
    p17          in varchar2    default null,
    p18          in varchar2    default null,
    p19          in varchar2    default null,
    p_max_length in pls_integer default 1000,
    p_prefix     in varchar2    default null )
    return varchar2;

--==============================================================================
-- Get N letter initials from the first N words.
-- Words are separated by spaces or other special characters such as commas (,),
-- dashes (-) and quotes (').
-- If the input only has one word, returns the first N characters of that word.
--
-- PARAMETERS
-- * p_str:     The input string.
-- * p_cnt:     The N letter initials to get from the first N words. The default is 2.
--              Allowed values are 1 to 255.
--
-- EXAMPLE
--   Get the first 2 initials from "John Doe".
--
--   begin
--     sys.dbms_output.put_line(get_initials('John Doe'));
--   end;
--   -> JD
--
--   Get the first 3 initials from "Andres Homero Lozano Garza"
--
--   begin
--     sys.dbms_output.put_line(get_initials(p_str => 'Andres Homero Lozano Garza', p_cnt => 3));
--   end;
--   -> AHL
--
--   Get the first 3 initials from "JBHiFi"
--
--   begin
--     sys.dbms_output.put_line(get_initials(p_str => 'JBHiFi', p_cnt => 3));
--   end;
--   -> JBH
--==============================================================================
function get_initials (
    p_str  in varchar2,
    p_cnt  in pls_integer default 2 )
    return varchar2;

--==============================================================================
-- Return distinct phrases of 1-3 consecutive lower case words in the input
-- strings. Stopwords in the given language are ignored and split phrases.
--
-- This is a PL/SQL only implementation of a very small subset of what Oracle
-- Text provides. Consider using Oracle Text instead, if the features and
-- performance of this function are not sufficient.
--
-- PARAMETERS
-- * p_strings:         The input strings.
-- * p_max_words:       Maximum number of words in a phrse. The default is 3.
-- * p_language:        The language identifier for stopwords, defaults to "en".
--                      Supported values are "cn", "de", "en", "es", "fr", "it",
--                      "ja", "ko", "pt-br".
--
-- EXAMPLE
--   Print keywords in the given input string.
--
--     begin
--         sys.dbms_output.put_line (
--             apex_string.join (
--                 apex_string.get_searchable_phrases (
--                     p_strings => apex_t_varchar2 (
--                                      'Oracle APEX 19.1 is great.',
--                                      'Low code as it should be!' )),
--                 ':' ));
--
--     end;
--     -> oracle:oracle apex:oracle apex 19.1:apex:apex 19.1:19.1:great:low:
--        low code:code
--==============================================================================
function get_searchable_phrases (
    p_strings   in wwv_flow_t_varchar2,
    p_max_words in pls_integer default 3,
    p_language  in varchar2    default 'en' )
    return wwv_flow_t_varchar2;

--==============================================================================
-- Returns the first position in the list where p_value is stored. If it's not
-- found null will be returned.
--
-- Parameters:
-- * p_table:           The table.
-- * p_value:           Value that is being searched for.
--
-- Returns:
-- Index of the searched value in the table.
--
-- Example:
--   Print the index of the given input string in the table.
--
--     begin
--         sys.dbms_output.put_line (
--             apex_string.index_of (
--                 p_table => apex_t_varchar2 (
--                                'Dog',
--                                'Cat',
--                                'Capybara' ),
--                 p_value => 'Capybara' ) );
--     end;
--     -> 3
--==============================================================================
function index_of (
    p_table in wwv_flow_t_varchar2,
    p_value in varchar2 )
    return number;

--
--==============================================================================
-- Returns the first position in the list where p_value is stored. If it's not
-- found null will be returned.
--
-- Parameters:
-- * p_table:           The table.
-- * p_value:           Value that is being searched for.
--
-- Returns:
-- Index of the searched value in the table.
--
-- Example:
--   Print the index of the given input string in the table.
--
--     declare
--         l_list     apex_application_global.vc_arr2;
--     begin
--         l_list(1) := 'Dog';
--         l_list(2) := 'Capybara';
--         l_list(3) := 'Cat';
--         sys.dbms_output.put_line (
--             apex_string.index_of (
--                 p_table => l_list,
--                 p_value => 'Capybara' ) );
--     end;
--     -> 3
--==============================================================================
function index_of (
    p_table in wwv_flow_global.vc_arr2,
    p_value in varchar2 )
    return number;
--
--
--==============================================================================
-- Converts a wwv_flow_t_varchar2 record to a sys.json_object_t object type
-- Elements with odd numbers will be the attributes names.
-- Elements with even numbers will be the attributes values.
--
-- Parameters:
-- * p_plist:     The table.
--
-- Returns:
--   JSON_OBJECT_T with keys and values of the given p_plist.
--
-- Example:
-- Create JSON object: {"key1":"foo","key2":"bar"}
--
--  declare
--      l_attributes sys.json_object_t;
--  begin
--      l_attributes := wwv_flow_string.plist_to_json(wwv_flow_t_varchar2(
--                          'key1', 'foo' ,
--                          'key2', 'bar' ));
--     dbms_output.put_line(l_attributes.stringify);
--  end;
--
--
--==============================================================================
function plist_to_json (
    p_plist in wwv_flow_t_varchar2 )
    return sys.json_object_t;
--==============================================================================
-- Converts a wwv_flow_t_varchar2 record to a sys.json_object_t object type
-- and stringifies it.
-- Elements with odd numbers will be the attributes names.
-- Elements with even numbers will be the attributes values.
--
-- Parameters:
-- * p_plist:     The table.
--
-- Returns:
--   CLOB containing a JSON object with keys and values of the given p_plist.
--
-- Example:
-- Create JSON object: {"key1":"foo","key2":"bar"}
--
-- declare
--     l_attributes apex_application_page_regions.attributes%type;
-- begin
--     l_attributes := wwv_flow_string.plist_to_json_clob(wwv_flow_t_varchar2(
--                         'key1', 'foo' ,
--                         'key2', 'bar' ));
--     dbms_output.put_line(l_attributes);
-- end;
--==============================================================================
function plist_to_json_clob (
    p_plist in wwv_flow_t_varchar2 )
    return clob;
--
end wwv_flow_string;
/
show err
