set define '^' verify off
prompt ...wwv_flow_data_profile_dev.sql
create or replace package wwv_flow_data_profile_dev as
--------------------------------------------------------------------------------
--
--  Copyright (c) 2017, 2024, Oracle and/or its affiliates.
--
--    NAME
--      wwv_flow_data_profile_dev
--
--    DESCRIPTION
--      Data Profile processing logic specific to the APEX builder
--
--
--    MODIFIED   (MM/DD/YYYY)
--    cczarski    05/30/2017 - Created
--------------------------------------------------------------------------------

c_discover_hierarchy_levels constant number := 10;
c_max_initial_visible_cols  constant number := 150;

--==========================================================================
-- Converts the data type constant from WWV_FLOW_EXEC_API to the constants
-- used by data profiles.
--==========================================================================
function data_type_exec_to_profile(
    p_data_type   in wwv_flow_exec_api.t_data_type )
    return varchar2;

--==============================================================================
-- Copy a data profile. If the name for the new profile (p_name_to) is already
-- taken, a unique name will be generated by appending a "(n)" suffix, with "n"
-- being an numeric, increasing value.
--
-- Parameters:
-- * p_from_application_id:    Application ID to copy from
-- * p_name:                   Name of the profile to copy
-- * p_to_application_id:      Application ID to copy to
-- * p_to_name:                Name for the new Data Profile
-- * p_subscribe:              Whether the new profile subscribes to the source
--==============================================================================
function copy_data_profile (
    p_from_application_id   in number,
    p_name                  in varchar2,
    p_to_application_id     in number,
    p_to_name               in varchar2 default null,
    p_subscribe             in boolean  default false )
    return number;

--==============================================================================
-- Refresh a data profile from a master
--
-- Parameters:
-- * p_from_data_profile_id:   Profile to refresh from
-- * p_to_data_profile_id:     Profile to refresh to
-- * p_from_application_id:    Application ID to refresh from
-- * p_to_application_id:      Application ID to refresh to
--==============================================================================
procedure refresh_data_profile (
    p_from_data_profile_id  in number,
    p_to_data_profile_id    in number,
    p_from_application_id   in number,
    p_to_application_id     in number );

--==============================================================================
-- Samples a response and generates a data profile as wwv_flow_data_profile.t_data_profile
-- record structure. When P_AUTO_DETECT is set to false, ROW_SELECTOR and SINGLE_ROW attributes
-- the the P_DATA_PROFILE record structure should be set. When P_AUTO_DETECT is true, the procedure
-- will auto-detect them from the XML data.
--
-- Parameters:
-- * p_web_service_response:   CLOB with the JSON service response to sample
-- * p_data_profile:           data profile record structure to populate
-- * p_auto_detect:            whether to auto-detect the row selector (true) or not (false)
--==============================================================================
procedure sample_data_profile(
    p_web_service_response  in out nocopy clob,
    p_data_profile          in out nocopy wwv_flow_data_profile.t_data_profile,
    p_auto_detect           in            boolean default false );

--==============================================================================
procedure sample_data_profile(
    p_web_service_response  in            blob,
    p_data_profile          in out nocopy wwv_flow_data_profile.t_data_profile,
    p_auto_detect           in            boolean default false );

--==============================================================================
-- writes data profile information to a collection
--
-- Parameters:
-- * p_data_profile:           data profile record
-- * p_collection:             name of the collection to write info to
-- * p_include_arrays:         whether to include Array Column information or not.
--==============================================================================
procedure write_profile_to_collection(
    p_data_profile   in wwv_flow_data_profile.t_data_profile,
    p_collection     in varchar2    default 'WEB_SOURCE_INFO',
    p_include_arrays in boolean     default true );

--==============================================================================
-- Resequences the columns of a data profile
--
-- Parameters:
-- * p_data_profile_id:        ID of the data profile to resequence the columns
-- * p_interval:               New sequence interval
--==============================================================================
procedure resequence_columns(
    p_data_profile_id       in number,
    p_interval              in number default 10 );

--==============================================================================
-- Checks whether a SQL expression is valid within the data profile
--
-- Parameters:
-- * p_data_profile_id:        ID of the data profile to check the SQL expression for
-- * p_array_column_id:        ID of the array column to check the SQL expression for
-- * p_sql_expression:         SQL Expression to check
--==============================================================================
function check_sql_expression(
    p_data_profile_id in number,
    p_array_column_id in number    default null,
    p_sql_expression  in varchar2 )
    return varchar2;

--==============================================================================
-- Returns the resulting data type for a SQL expression
--
-- Parameters:
-- * p_data_profile_id:        ID of the data profile
-- * p_array_column_id:        ID of the array column to check the SQL expression for
-- * p_sql_expression:         SQL Expression to check
--==============================================================================
function get_sql_expr_datatype(
    p_data_profile_id       in number,
    p_array_column_id       in number    default null,
    p_sql_expression        in varchar2 )
    return varchar2;

--==============================================================================
-- generates a data profile from the JSON Schema being passed in.
--==============================================================================
function json_schema_discovery(
    p_json_schema       in clob )
    return wwv_flow_data_profile.t_data_profile;

--==========================================================================
-- Stores columns of a data profile from a collection.
--==========================================================================
procedure store_data_profile_columns(
    p_application_id      in number   default v('FB_FLOW_ID'),
    p_data_profile_id     in number,
    p_collection_name     in varchar2,
    p_replace             in boolean );

--==========================================================================
-- Stores a data profile from a collection.
--==========================================================================
procedure store_data_profile(
    p_application_id      in number              default v('FB_FLOW_ID'),
    p_data_profile_id     in number              default null,
    p_name                in varchar2,
    p_collection_name     in varchar2 );

end wwv_flow_data_profile_dev;
/
sho err

set define '^'
