set define '^' verify off
prompt ...htmldb_util
create or replace package htmldb_util authid definer as
--------------------------------------------------------------------------------
--
-- Copyright (c) 2003, 2024, Oracle and/or its affiliates.
--
-- Filename: core/htmldb_util.sql
--
-- Alias: APEX_UTIL
--
-- The `APEX_UTIL` package provides utilities you can use when programming in the
-- Oracle APEX environment. You can use the `APEX_UTIL` package to get and set session
-- state, to get files, to check authorizations for users, to reset different states
-- for users, to get and purge cache information, and to get and set preferences for
-- users.
--
-- Modified:
--   mhichwa   10/07/2003 - Created
--
--------------------------------------------------------------------------------

--##############################################################################
-- Constants:
--
-- The following constants are used by the APEX_UTIL package.
--##############################################################################
c_must_not_be_public_user constant varchar2(30) := 'MUST_NOT_BE_PUBLIC_USER'; -- Needs Review: Mark internal or add to public documentation?

c_install_script        constant varchar2(20) := 'INSTALL'; -- Needs Review: Mark internal or add to public documentation?
c_upgrade_script        constant varchar2(20) := 'UPGRADE'; -- Needs Review: Mark internal or add to public documentation?
c_deinstall_script      constant varchar2(20) := 'DEINSTALL'; -- Needs Review: Mark internal or add to public documentation?

c_output_as_dbms_output constant varchar2(20) := 'DBMS_OUTPUT'; -- Needs Review: Mark internal or add to public documentation?
c_output_as_file        constant varchar2(20) := 'FILE'; -- Needs Review: Mark internal or add to public documentation?

--##############################################################################
-- Internal:
-- This is an internal only API
-- Deprecated:
--! UNDOCUMENTED - DEPRECATED - DO NOT USE
--##############################################################################
function get_language_selector_list
    return varchar2;

--##############################################################################
--#
--# FEEDBACK
--#
--##############################################################################

--##############################################################################
-- The `SUBMIT_FEEDBACK` procedure enables you to write a procedure
-- to submit feedback, rather than using the feedback page generated by
-- Create Page Wizard.
--
-- Parameters:
-- * p_comment:             The comment to be submitted.
-- * p_type:                The type of feedback (1 is General Comment, 2 is Enhancement Request, 3 is Bug).
-- * p_application_id:      The ID of the application related to the feedback.
-- * p_page_id:             The ID of the page related to the feedback.
-- * p_email:               Email of the user providing the feedback.
-- * p_screen_width:        The width of the screen at the time the feedback was provided.
-- * p_screen_height:       The height of the screen at the time the feedback was provided.
-- * p_attribute_01 ... 08: A custom attribute for collecting feedback.
-- * p_label_01 ... 08:     The label for the corresponding custom attribute.
-- * p_rating:              User experience (3 is Good, 2 is Neutral, 1 is Bad).
-- * p_attachment_name:     Bind variable reference to the feedback form's "File Browse" page item.
--
--
-- Example:
--
-- The following example submits a bad user experience because of a bug on page
-- 22 within application 283.
--
--      begin
--          apex_util.submit_feedback (
--             p_comment        => 'This page does not render properly for me',
--             p_type           => 3,
--             p_rating         => 1,
--             p_application_id => 283,
--             p_page_id        => 22,
--             p_email          => 'user@xyz.corp',
--             p_attribute_01   => 'Charting',
--             p_label_01       => 'Component' );
--      end;
--
--
-- See Also:
-- SUBMIT_FEEDBACK_FOLLOWUP
--##############################################################################
procedure submit_feedback (
    p_comment          in varchar2 default null,
    p_type             in number   default 1,
    p_application_id   in varchar2 default null,
    p_page_id          in varchar2 default null,
    p_email            in varchar2 default null,
    p_screen_width     in varchar2 default null,
    p_screen_height    in varchar2 default null,
    p_attribute_01     in varchar2 default null,
    p_attribute_02     in varchar2 default null,
    p_attribute_03     in varchar2 default null,
    p_attribute_04     in varchar2 default null,
    p_attribute_05     in varchar2 default null,
    p_attribute_06     in varchar2 default null,
    p_attribute_07     in varchar2 default null,
    p_attribute_08     in varchar2 default null,
    p_label_01         in varchar2 default null,
    p_label_02         in varchar2 default null,
    p_label_03         in varchar2 default null,
    p_label_04         in varchar2 default null,
    p_label_05         in varchar2 default null,
    p_label_06         in varchar2 default null,
    p_label_07         in varchar2 default null,
    p_label_08         in varchar2 default null,
    p_rating           in number   default null,
    p_attachment_name  in varchar2 default null );

--==============================================================================
-- The `SUBMIT_FEEDBACK_FOLLOWUP` procedure enables you to submit a follow-up to
-- a feedback.
--
-- Parameters:
-- * p_feedback_id: The ID of feedback that this is a follow-up to.
-- * p_follow_up:   The text of the follow-up.
-- * p_email:       The email of the user providing the follow-up.
--
-- Example:
--
-- The following example submits a follow-up to a previously filed feedback.
--
--     begin
--        apex_util.submit_feedback_followup (
--            p_feedback_id    => 12345,
--            p_follow_up      => 'I tried this on another instance and it does not work there either',
--            p_email          => 'user@xyz.corp' );
--     end;
--
-- See Also:
-- SUBMIT_FEEDBACK
-- GET_FEEDBACK_FOLLOW_UP
--==============================================================================
procedure submit_feedback_followup (
    p_feedback_id      in number,
    p_follow_up        in varchar2 default null,
    p_email            in varchar2 default null );

--==============================================================================
-- Use the `GET_FEEDBACK_FOLLOW_UP` function to retrieve any remaining follow-up
-- associated with a specific feedback.
--
-- Parameters:
-- * p_feedback_id: The unique identifier of the feedback item.
-- * p_row:         Identifies which follow-up to retrieve and is ordered by
--                  `created_on_desc`.
-- * p_template:    The template to use to return the follow-up. Given the `<br/>`
--                  in the default template, the function can be used in a loop
--                  to return all the follow-up to a feedback.
-- Example:
--
-- The following example displays all the remaining follow-up for feedback with
-- the ID of 123.
--
--      declare
--         l_feedback_count  number;
--      begin
--         select count(*)
--           into l_feedback_count
--           from apex_team_feedback_followup
--          where feedback_id = 123;
--
--         for i in 1..l_feedback_count loop
--            htp.p(apex_util.get_feedback_follow_up (
--                     p_feedback_id => 123,
--                     p_row         => i,
--                     p_template    => '<br />#FOLLOW_UP# was created on #CREATED_ON# by #CREATED_BY#') );
--         end loop;
--      end;
--
-- See Also:
-- SUBMIT_FEEDBACK_FOLLOWUP
--==============================================================================
function get_feedback_follow_up (
    p_feedback_id    in number,
    p_row            in number default 1,
    p_template       in varchar2 default '<br />#CREATED_ON# (#CREATED_BY#) #FOLLOW_UP#')
    return varchar2;

--==============================================================================
-- The `FEEDBACK_ENABLED` function returns a boolean value to check if the application
-- feature 'Allow Feedback' is enabled.
--
-- Returns:
-- True, if 'Allow Feedback' is enabled, false otherwise.
--
-- Example:
--
--      begin
--          return apex_util.feedback_enabled;
--      end;
--==============================================================================
function feedback_enabled
      return boolean;

--==============================================================================
-- Use the `DELETE_FEEDBACK` procedure to delete feedback with the specified ID.
--
-- Parameters:
-- * p_feedback_id: The ID of the feedback.
--
-- Example:
-- The following example deletes a feedback with the specified ID.
--
--     begin
--        apex_util.delete_feedback (
--            p_feedback_id    => 12345 );
--     end;
--
-- See Also:
-- DELETE_FEEDBACK_ATTACHMENT
--==============================================================================
procedure delete_feedback (
    p_feedback_id      in number );

--==============================================================================
-- Use the `DELETE_FEEDBACK_ATTACHMENT` procedure to delete the attachment of a
-- feedback with the specified ID.
--
-- Parameters:
-- * p_feedback_id: The ID of the feedback.
--
-- Example:
-- The following example deletes the attachment of a feedback with the specified ID.
--
--     begin
--        apex_util.delete_feedback_attachment (
--            p_feedback_id    => 12345 );
--     end;
--
-- See Also:
-- DELETE_FEEDBACK
--==============================================================================
procedure delete_feedback_attachment (
    p_feedback_id      in number );

--==============================================================================
-- The `REPLY_TO_FEEDBACK` procedure enables you to submit a reply to
-- a feedback.
--
-- Parameters:
-- * p_feedback_id:       The ID of feedback.
-- * p_type:              The type of feedback (1 is General Comment, 2 is Enhancement Request, 3 is Bug).
-- * p_status:            The status of the feedback (0 is No Action, 1 is Acknowledged, 3 is Open and 4 is Closed).
-- * p_tags:              Used to categorize feedback, only to be displayed to developers (not in the current feature implementation).
-- * p_developer_comment: Comments not displayed to the users but could be displayed to developers (not in the current feature implementation).
-- * p_public_response:   Text that will be displayed to the users.
-- * p_followup:          The text of the follow-up.  When the 'Feedback' feature is added to an application, it will only use `p_public_response`.  It will not support passing data to `p_follow_up`.
--
-- Example:
-- In this example, the app admin replies to a feedback submitted by a user reporting a bug.
--
-- begin
--     apex_util.reply_to_feedback (
--              p_feedback_id       => 12345,
--              p_type              => 3,
--              p_status            => 1,
--              p_developer_comment => 'Created JIRA BUG-123!',
--              p_public_response   => 'Thank you for letting us know about this! We will fix it in the upcoming release!',
--              p_follow_up         => 'We really appreciate your support of our application.' );
-- end;
--
-- Example:
-- In this example, the developer leaves a comment about creating a ticket for the bug reported by the user,
-- and closes the feedback workflow.
--
-- begin
--     reply_to_feedback (
--     p_feedback_id       => 12345,
--     p_type              => 3,
--     p_status            => 4,
--     p_developer_comment => 'Filed ticket BUG-123!',
--     p_follow_up         => 'We really appreciate your support of our application.'  );
-- end;
--
--==============================================================================
procedure reply_to_feedback (
    p_feedback_id        in number,
    p_type               in number   default null,
    p_status             in number   default null,
    p_tags               in varchar2 default null,
    p_developer_comment  in varchar2 default null,
    p_public_response    in varchar2 default null,
    p_followup           in varchar2 default null );

--==============================================================================
--##############################################################################
--#
--# ACCESSIBILITY
--#
--##############################################################################
--
--==============================================================================
-- The function `IS_SCREEN_READER_SESSION` returns a boolean `TRUE` if the session
-- is in screen reader mode, and returns a boolean `FALSE` otherwise.
--
-- Example:
--
--     begin
--         if apex_util.is_screen_reader_session then
--             htp.p('Screen Reader Mode');
--         end if;
--     end;
--==============================================================================
function is_screen_reader_session
    return boolean;

--==============================================================================
-- The function `IS_SCREEN_READER_SESSION_YN` returns a 'Y' if the session
-- is in screen reader mode, and returns 'N' otherwise.
--
-- Example:
--      begin
--          if apex_util.is_screen_reader_session_yn = 'Y' then
--              htp.p('Screen Reader Mode');
--          end if;
--      end;
--
--==============================================================================
function is_screen_reader_session_yn
    return varchar2;

--==============================================================================
-- The `SET_SESSION_SCREEN_READER_ON` procedure puts the current session into
-- screen reader mode.
--
-- Example:
-- In this example, the current session is put into screen reader mode.
--
--      begin
--         apex_util.set_session_screen_reader_on;
--      end;
--
-- See Also:
-- SET_SESSION_SCREEN_READER_OFF
--==============================================================================
procedure set_session_screen_reader_on;

--==============================================================================
-- The `SET_SESSION_SCREEN_READER_OFF`switches off screen reader mode for the
-- current session.
--
-- Example:
-- In this example, the current session is put into standard mode.
--
--      begin
--         apex_util.set_session_screen_reader_off;
--      end;
--
-- See Also:
-- SET_SESSION_SCREEN_READER_ON
--==============================================================================
procedure set_session_screen_reader_off;

--==============================================================================
-- This procedure displays a link to the current page to turn on or off, toggle,
-- the mode. For example, if you are in standard mode, this procedure displays a
-- link that when clicked switches the screen reader mode on.
--
-- Parameters:
-- * p_on_message:  Optional text used for the link to switch to screen reader mode,
--                  when you are in standard mode. If this parameter is not passed,
--                  the default 'Set Screen Reader Mode On' text is displayed.
-- * p_off_message: Optional text used for the link to switch to standard mode,
--                  when you are in screen reader mode. If this parameter is not
--                  passed, the default 'Set Screen Reader Mode Off' text is displayed.
-- Example:
-- When running in standard mode, this procedure displays a link 'Set Screen Reader
-- Mode On', that when clicked refreshes the current page and switches on screen
-- reader mode. When running in screen reader mode, a link with the text 'Set Screen Reader Mode Off'
-- is displayed, that when clicked refreshes the current page and switches back to standard mode.
--
--      begin
--          apex_util.show_screen_reader_mode_toggle;
--      end;
--
-- See Also:
-- GET_SCREEN_READER_MODE_TOGGLE
--==============================================================================
procedure show_screen_reader_mode_toggle (
    p_on_message  in varchar2 default null,
    p_off_message in varchar2 default null );

--==============================================================================
-- This function returns a link to the current page to turn on or off, toggle,
-- the mode. For example, if you are in standard mode, this function returns a
-- link that when clicked switches the screen reader mode on.
--
-- Parameters:
-- * p_on_message:  Optional text used for the link to switch to screen reader mode,
--                  when you are in standard mode. If this parameter is not passed,
--                  the default 'Set Screen Reader Mode On' text is returned in the
--                  link.
-- * p_off_message: Optional text used for the link to switch to standard mode,
--                  when you are in screen reader mode. If this parameter is not
--                  passed, the default 'Set Screen Reader Mode Off' text is returned
--                  in the link.
--
-- Example:
-- When running in standard mode, this procedure returns a link with the text
-- 'Set Screen Reader Mode On', that when clicked refreshes the current page and
-- switches on screen reader mode. When running in screen reader mode, a link
-- with the text 'Set Screen Reader Mode Off' is returned, that when clicked refreshes
-- the current page and switches back to standard mode.
--
--      begin
--          htp.p(apex_util.get_screen_reader_mode_toggle);
--      end;
--
-- See Also:
-- SHOW_SCREEN_READER_MODE_TOGGLE
--==============================================================================
function get_screen_reader_mode_toggle (
    p_on_message  in varchar2 default null,
    p_off_message in varchar2 default null)
    return varchar2;

--==============================================================================
-- This function returns a boolean TRUE if the session is in high contrast mode
-- and returns a boolean FALSE if not in high contrast mode.
--
-- Example:
-- In this example, if the current session is running in high contrast mode,
-- a high contrast specific CSS file 'my_app_hc.css' is added to the HTML output
-- of the page.
--
--      begin
--          if apex_util.is_high_contrast_session then
--              apex_css.add_file (
--                  p_name => 'my_app_hc');
--          end if;
--      end;
--
-- See Also:
-- IS_HIGH_CONTRAST_SESSION_YN
--==============================================================================
function is_high_contrast_session
    return boolean;

--==============================================================================
-- This function returns Y if the session is in high contrast mode and N if
-- not in high contrast mode.
--
-- Example:
-- In this example, if the current session is running in high contrast mode,
-- a high contrast specific CSS file 'my_app_hc.css' is added to the HTML output
-- of the page.
--
--     begin
--          if apex_util.is_high_contrast_session_yn = 'Y' then
--              apex_css.add_file (
--                  p_name => 'my_app_hc');
--          end if;
--     end;
--
-- See Also:
-- IS_HIGH_CONTRAST_SESSION
--==============================================================================
function is_high_contrast_session_yn
    return varchar2;
--==============================================================================
-- This procedure switches on the high contrast mode for the current session.
--
-- Example:
-- This procedure switches on high contrast mode for the current session.
--
--      begin
--          apex_util.set_session_high_contrast_on;
--      end;
--
-- See Also:
-- SET_SESSION_HIGH_CONTRAST_OFF
--==============================================================================
procedure set_session_high_contrast_on;

--==============================================================================
-- This procedure switches off the high contrast mode for the current session.
--
-- Example:
-- In this example, high contrast mode is switched off for the current session.
--
--      begin
--          apex_util.set_session_high_contrast_off;
--      end;
--
-- See Also:
-- SET_SESSION_HIGH_CONTRAST_ON
--==============================================================================
procedure set_session_high_contrast_off;

--==============================================================================
-- This procedure displays a link to the current page to turn on or off, toggle,
-- the mode. For example, if you are in standard mode, this procedure displays a
-- link that when clicked switches the high contrast mode on.
--
-- Parameters:
-- * p_on_message:  Optional text used for the link to switch to high contrast mode,
--                  when you are in standard mode. If this parameter is not passed,
--                  the default 'Set High Contrast Mode On' text is displayed.
-- * p_off_message: Optional text used for the link to switch to standard mode,
--                  when you are in high contrast mode. If this parameter is not
--                  passed, the default 'Set High Contrast Mode Off' text is displayed.
-- Note:
-- There are also two translatable system messages that can be overridden at application
-- level to change the default link text that is returned for this toggle. They include:
--
-- + `APEX.SET_HIGH_CONTRAST_MODE_OFF` - Default text = 'Set High Contrast Mode Off'.
-- + `APEX.SET_HIGH_CONTRAST_MODE_ON`  - Default text = 'Set High Contrast Mode On'.
--
-- Example:
-- When running in standard mode, this procedure displays a link, 'Set High Contrast Mode On',
-- that when clicked refreshes the current page and switches on high contrast mode.
-- When running in high contrast mode, a link, 'Set High Contrast Mode Off', is displayed,
-- that refreshes the current page and switches back to standard mode when clicked.
--
--      begin
--          apex_util.show_high_contrast_mode_toggle;
--      end;
--
-- See Also:
-- GET_HIGH_CONTRAST_MODE_TOGGLE
--==============================================================================
procedure show_high_contrast_mode_toggle (
    p_on_message  in varchar2 default null,
    p_off_message in varchar2 default null);

--==============================================================================
-- This function returns a link to the current page to turn on or off, toggle,
-- the mode. For example, if you are in standard mode, this function returns a
-- link that when clicked switches the high contrast mode on.
--
-- Parameters:
-- * p_on_message:  Optional text used for the link to switch to high contrast mode,
--                  when you are in standard mode. If this parameter is not passed,
--                  the default 'Set High Contrast Mode On' text is returned in the link.
-- * p_off_message: Optional text used for the link to switch to standard mode,
--                  when you are in high contrast mode. If this parameter is not
--                  passed, the default 'Set High Contrast Mode Off' text is returned in the link.
-- Note:
-- There are also two translatable system messages that can be overridden at application
-- level to change the default link text that is returned for this toggle. They include:
--
-- + `APEX.SET_HIGH_CONTRAST_MODE_OFF` - Default text = 'Set High Contrast Mode Off'.
-- + `APEX.SET_HIGH_CONTRAST_MODE_ON`  - Default text = 'Set High Contrast Mode On'.
--
-- Example:
-- When running in standard mode, this function returns a link with the text
-- 'Set High Contrast Mode On'. When the link is clicked the current page is refreshed
-- and high contrast mode is switched on. When running in high contrast mode,
-- a link 'Set High Contrast Mode Off' is returned. When the link is clicked the
-- current page is refreshed and switched back to standard mode.
--
--      begin
--          htp.p(apex_util.get_high_contrast_mode_toggle);
--      end;
--
-- See Also:
-- SHOW_HIGH_CONTRAST_MODE_TOGGLE
--==============================================================================
function get_high_contrast_mode_toggle (
    p_on_message  in varchar2 default null,
    p_off_message in varchar2 default null)
    return varchar2;

--##############################################################################
--#
--# SESSION STATE
--#
--##############################################################################

--==============================================================================
-- This function returns the value in session state for a given item. You can use
-- this function in your Oracle APEX applications wherever you can use PL/SQL or SQL.
-- You can also use the shorthand function `V` in place of `APEX_UTIL.GET_SESSION_STATE`.
--
-- Tip:
-- For enhanced query performance, use the FAST DUAL functionality in the following SQL
-- code syntax:
-- `(select apex_util.get_session_state('P1_ITEM') from dual)`
--
-- Parameters:
-- * p_item:     The case insensitive name of the item for which you
--               wish to have the session state fetched.
--
-- Example:
-- The following example returns the value stored in session state for the item
-- `my_item`.
--
--      declare
--          l_item_value varchar2(255);
--      begin
--          l_item_value := apex_util.get_session_state('my_item');
--      end;
--
-- See Also:
-- GET_NUMERIC_SESSION_STATE
-- SET_SESSION_STATE
--==============================================================================
function get_session_state (
    p_item  in varchar2)
    return varchar2;

--==============================================================================
-- This function returns the numeric value in session state for a given
-- numeric item. You can use this function in Oracle APEX applications wherever
-- you can use PL/SQL or SQL. You can also use the shorthand function `NV` in place
-- of `APEX_UTIL.GET_NUMERIC_SESSION_STATE`.
--
-- Tip:
-- For enhanced query performance, use the FAST DUAL functionality in the following SQL
-- code syntax:
-- `(select apex_util.get_numeric_session_state('P1_ITEM') from dual)`
--
-- Parameters:
-- * p_item:     Case insensitive name of the numeric item for which you
--               wish to have the session state fetched.
--
-- Example:
-- The following example shows how to use the function to return the numeric value
-- stored in session state for the item `my_item`.
--
--      declare
--          l_item_value number;
--      begin
--          l_item_value := apex_util.get_numeric_session_state('my_item');
--      end;
--
-- See Also:
-- GET_SESSION_STATE
-- SET_SESSION_STATE
--==============================================================================
function get_numeric_session_state (
    p_item in varchar2)
    return number;

--==============================================================================
-- This function fetches session state for the current or specified application in
-- the current or specified session.
--
-- Parameters:
-- * p_item:     The name of an application-level item (not a page item) whose
--               current value is to be fetched.
-- * p_app:      The ID of the application that owns the item. (Leave `NULL`
--               for the current application).
-- * p_session:  The session ID from which to obtain the value. (Leave `NULL`
--               for the current session).
--
-- Example:
-- The following example shows how to use the `FETCH_APP_ITEM` function to obtain
-- the value of the application item 'F300_NAME' in application 300. As no value
-- is passed for `p_session`, this defaults to the current session state value.
--
--      declare
--          val varchar2(30);
--      begin
--          val := apex_util.fetch_app_item( p_item => 'F300_NAME',
--                                           p_app => 300);
--      end;
--==============================================================================
function fetch_app_item(
    p_item         in varchar2,
    p_app          in number default null,
    p_session      in number default null)
    return varchar2;

--==============================================================================
-- This function obtains the primary key of a file in the Oracle APEX file repository.
--! This function returns the numeric primary key of the named file in
--! wwv_flow_file_objects$.
--
-- Parameters:
-- * p_name: The `NAME` in `APEX_APPLICATION_FILES` of the file to be downloaded.
--           `APEX_APPLICATION_FILES` is a view on all files uploaded to your workspace.
-- Example:
-- The following example retrieves the database ID of the file with a filename of `F125.sql`.
--
--      declare
--          l_name varchar2(255);
--          l_file_id number;
--      begin
--          select name
--            into l_name
--            from apex_application_files
--            where filename = 'F125.sql';
--
--          l_file_id := apex_util.get_file_id (p_name => l_name);
--     end;
--
-- See Also:
-- GET_FILE
--==============================================================================
function get_file_id(
    p_name in varchar2)
    return number;

--==============================================================================
-- This procedure downloads files from the Oracle APEX file repository.
--
-- Note:
-- If you invoke this procedure during page processing, ensure that no page branch
-- is invoked under the same condition to avoid interference with the file retrieval.
-- This means that branches with any of the following conditions should NOT be set to fire:
-- + Branches with a 'When Button Pressed' attribute equal to the button that invokes the procedure.
-- + Branches with conditional logic defined that would succeed during page processing when the procedure is being invoked.
-- + As unconditional.
--
-- Parameters:
-- * p_file_id:      The ID in `APEX_APPLICATION_FILES` of the file to be downloaded.
--                   `APEX_APPLICATION_FILES` is a view on all files uploaded to your workspace.
--! The parameter value of p_mime_type will always be ignored.  The MIME type associated with the uploaded file
--! will always be used.  In the rare event that this stored MIME type is null, the specified
--! MIME type will be application/octet-stream.
-- * p_mime_type:    The mime type of the file to be downloaded. This parameter is desupported!
-- * p_inline:       Valid values include `YES` and `NO`.
--                   + `YES` to display the file inline in a browser.
--                   + `NO` to download the file as attachment.
--
-- Example:
-- The following example returns the file identified by the ID `8675309`. It is displayed inline in the browser.
--
--      begin
--        apex_util.get_file(
--            p_file_id   => '8675309',
--            p_inline    => 'YES');
--      end;
--
-- Example:
-- The following example demonstrates how to use the view `APEX_APPLICATION_FILES`:
--
--      declare
--          l_file_id number;
--      begin
--          select id
--            into l_file_id
--            from apex_application_files
--           where filename = 'myxml';
--
--          apex_util.get_file(
--               p_file_id   => l_file_id,
--               p_inline    => 'YES');
--      end;
--
-- See Also:
-- GET_FILE_ID
--==============================================================================
procedure get_file (
    p_file_id   in varchar2,
    p_mime_type in varchar2 default null, -- Internal:
    p_inline    in varchar2 default 'NO');

--==============================================================================
-- This procedure counts clicks from an application built in App Builder to an
-- external site. You can also use the shorthand version, procedure `z`, in place
-- of `APEX_UTIL.COUNT_CLICK`.
--
-- Parameters:
-- * p_url:             The URL to redirect to.
-- * p_cat:             A category to classify the click.
-- * p_id:              (Optional) Secondary ID to associate with the click.
-- * p_user:            (Optional) The application user ID.
-- * p_company:         This parameter is deprecated but needs to be
--                       maintained for backward compatibility.
-- * p_workspace:       (Optional) The workspace associated with the application.
-- * p_referrer_policy: The referrer-policy HTTP response header.
--
-- Example:
-- The following example demonstrates how to use the `COUNT_CLICK` procedure to
-- log how many users click on the `http://example.com` link specified. Once this
-- information is logged, you can view it by using the `APEX_WORKSPACE_CLICKS` view
-- and in the reports on this view available to workspace and site administrators.
--
--     declare
--         l_url          varchar2(255);
--         l_cat          varchar2(30);
--         l_workspace_id varchar2(30);
--     begin
--         l_url := 'http://example.com';
--         l_cat := 'yahoo';
--         l_workspace_id := to_char(apex_util.find_security_group_id('my_workspace'));
--
--         htp.p('<a href=apex_util.count_click?p_url=' || l_url || '&p_cat=' || l_cat || '&p_workspace=' || l_workspace_id || '>Click</a>');
--     end;
--
-- See Also:
-- FIND_SECURITY_GROUP_ID
--==============================================================================
procedure count_click (
    p_url             in varchar2,
    p_cat             in varchar2,
    p_id              in varchar2 default null,
    p_user            in varchar2 default null,
    -- Deprecated:
    p_company         in varchar2 default null,
    p_workspace       in varchar2 default null,
    p_referrer_policy in varchar2 default null );

--==============================================================================
-- Programmatically set session state for a current session.
-- This procedure must be called from Oracle APEX since it requires
-- the Oracle APEX environment.
--
-- Parameters:
-- * p_name:   Name of the application-level or page-level item for which you are setting sessions state.
-- * p_value:  Value of session state to set.
-- * p_commit: If TRUE (the default), commit after modifying session state. If
--             FALSE or if the existing value in session state equals `p_value`,
--             no commit is issued.
--             This parameter is ignored when the application's 'Session State
--             Changes' attribute is set to 'End Of Request'.
--
-- Example:
--   The following example shows how to use the `SET_SESSION_STATE procedure` to
--   set the value of the item `my_item` to `myvalue` in the current session.
--
--      begin
--          apex_util.set_session_state('my_item','myvalue');
--      end;
--
-- See Also:
-- GET_NUMERIC_SESSION_STATE
-- GET_SESSION_STATE
--==============================================================================
procedure set_session_state (
    p_name    in varchar2 default null,
    p_value   in varchar2 default null,
    p_commit  in boolean  default true );

--==============================================================================
-- To increase performance, Oracle APEX caches the results of authorization schemes
-- after they have been evaluated. You can use this procedure to undo caching, requiring
-- each authorization scheme be revalidated when it is next encountered during page
-- show or accept processing. You can use this procedure if you want users to have
-- the ability to change their responsibilities (their authorization profile) within
-- your application.
--
-- Deprecated:
-- You should no longer use this feature since it may be removed in future releases.
-- Use `APEX_AUTHORIZATION.RESET_CACHE` instead!
--
-- Example:
-- The following example shows how to use the `RESET_AUTHORIZATIONS` procedure to
-- clear the authorization scheme cache.
--
--      begin
--         apex_util.reset_authorizations;
--      end;
--
-- See Also:
-- APEX_AUTHORIZATION.RESET_CACHE
--==============================================================================
procedure reset_authorizations;

--==============================================================================
-- Given the name of an authorization scheme, this function determines if the
-- current user passes the security check.
--
-- You can also check for the predefined authorization 'MUST_NOT_BE_PUBLIC_USER'.
-- Use the constant `c_must_not_be_public_user` to reference it.
--
-- Deprecated:
-- You should no longer use this feature since it may be removed in future releases.
-- Use `APEX_AUTHORIZATION.IS_AUTHORIZED` instead!
--
-- Parameters:
-- * p_security_scheme: The name of the authorization scheme that determines if
--                      the user passes the security check.
--
-- Example:
-- The following example shows how to use the `PUBLIC_CHECK_AUTHORIZATION` function
-- to check if the current user passes the check defined in the 'my_auth_scheme'
-- authorization scheme.
--
--      declare
--          l_check_security  boolean;
--      begin
--          l_check_security := apex_util.public_check_authorization('my_auth_scheme');
--      end;
--
-- See Also:
-- APEX_AUTHORIZATION.IS_AUTHORIZED
--==============================================================================
function public_check_authorization (
    p_security_scheme in varchar2)
    return boolean;

--##############################################################################
--#
--# STRING / TABLE CONVERSION
--#
--##############################################################################

--==============================================================================
-- Given a string, this function returns a PL/SQL array of type `APEX_APPLICATION_GLOBAL.VC_ARR2`.
-- This array is a `VARCHAR2(32767)` table.
--
-- Deprecated:
-- You should no longer use this feature since it may be removed in future releases.
-- Use `APEX_STRING.STRING_TO_TABLE` instead!
--
-- Parameters:
-- * p_string:    String to be converted into a PL/SQL table of type `APEX_APPLICATION_GLOBAL.VC_ARR2`.
-- * p_separator: String separator. The default is a colon.
--
-- Example:
-- The following example demonstrates how the function is passed the string `One:Two:Three`
-- in the `p_string` parameter and returns a PL/SQL array of type `APEX_APPLICATION_GLOBAL.VC_ARR2`
-- containing three elements: the element at position 1 contains the value One,
-- position 2 contains the value Two, and position 3 contains the value Three.
-- This is then output using the HTP.P function call.
--
--      declare
--          l_vc_arr2    apex_application_global.vc_arr2;
--      begin
--          l_vc_arr2 := apex_util.string_to_table('One:Two:Three');
--          for z in 1..l_vc_arr2.count loop
--              htp.p(l_vc_arr2(z));
--          end loop;
--      end;
--
-- See Also:
-- APEX_STRING.STRING_TO_TABLE
-- TABLE_TO_STRING
-- APEX_STRING.SPLIT Function Signature 1
-- APEX_STRING.SPLIT Function Signature 2
-- APEX_STRING.SPLIT_NUMBERS
--==============================================================================
function string_to_table (
    p_string    in varchar2,
    p_separator in varchar2 default ':')
    return wwv_flow_global.vc_arr2;

--==============================================================================
-- Deprecated:
-- Please use `apex_string.join` and `apex_string.join_clob` instead.
-- ___
--
-- Given a a PL/SQL table of type `APEX_APPLICATION_GLOBAL.VC_ARR2`, this function
-- returns a delimited string separated by the supplied separator, or by the default
-- separator, a colon (:).
--
-- Parameters:
-- * p_table:  A PL/SQL table that is to be converted into a delimited string.
-- * p_string: The string separator. Default separator is a colon (:).
--
-- Example:
-- The following example returns a comma delimited string of contact names that
-- are associated with the provided `cust_id`.
--
--      create or replace function get_contacts (
--          p_cust_id  in  number )
--          return varchar2
--      is
--          l_vc_arr2   apex_application_global.vc_arr2;
--          l_contacts  varchar2(32000);
--      begin
--          select contact_name
--              bulk collect
--              into l_vc_arr2
--              from contacts
--          where cust_id = p_cust_id
--              order by contact_name;
--
--          l_contacts :=  apex_util.table_to_string (
--                             p_table  => l_vc_arr2,
--                             p_string => ', ');
--         return l_contacts;
--      end get_contacts;
--
-- See Also:
-- APEX_STRING.JOIN
-- APEX_STRING.JOIN_CLOB
-- STRING_TO_TABLE
--==============================================================================
function table_to_string (
    p_table     in wwv_flow_global.vc_arr2,
    p_string    in varchar2 default ':')
    return varchar2;

--==============================================================================
-- This function computes a hash value for all given values. Use this function to
-- implement lost update detection for data records.
--
-- Parameters:
-- * p_values:    The input values.
-- * p_salted:    If true (the default), salt hash with internal session information.
--
-- Example:
-- This example updates the `SAL` and `COMM` columns of a given record in the `EMP` table,
-- but throws an error if the column data has changed in the meantime.
--
--     declare
--         l_hash varchar2(4000);
--     begin
--         select apex_util.get_hash(apex_t_varchar2 (
--                    empno, sal, comm ))
--           into l_hash
--           from emp
--          where empno = :P1_EMPNO;
--
--         if :P1_HASH <> l_hash then
--             raise_application_error(-20001, 'Somebody already updated SAL/COMM');
--         end if;
--
--         update emp
--            set sal = :P1_SAL,
--               comm = :P1_COMM
--        where empno = :P1_EMPNO;
--
--     exception when no_data_found then
--         raise_application_error(-20001, 'Employee not found');
--     end;
--==============================================================================
function get_hash (
    p_values in wwv_flow_t_varchar2,
    p_salted in boolean default true )
    return varchar2;

--==============================================================================
-- Deprecated: 24.1. Use `UTL_URL.ESCAPE` instead.
-- ___
-- This function encodes (into HEX) the following special characters in a given string:
--
-- | Special Characters | After Encoding |
-- |--------------------|----------------|
-- | %                  | %25            |
-- | +                  | %2B            |
-- | space              | +              |
-- | .                  | %2E            |
-- | *                  | %2A            |
-- | ?                  | %3F            |
-- | \                  | %5C            |
-- | /                  | %2F            |
-- | >                  | %3E            |
-- | <                  | %3C            |
-- | }                  | %7B            |
-- | {                  | %7D            |
-- | ~                  | %7E            |
-- | [                  | %5B            |
-- | ]                  | %5D            |
-- | ;                  | %3B            |
-- | ?                  | %3F            |
-- | @                  | %40            |
-- | &                  | %26            |
-- | #                  | %23            |
-- | &#124;             | %7C            |
-- | ^                  | %5E            |
-- | :                  | %3A            |
-- | =                  | %3D            |
-- | $                  | %24            |
--
-- Parameters:
-- * p_url: The url to be encoded.
--
-- Example:
-- The following example shows how to use the `URL_ENCODE` function.
--
--      declare
--          l_url  varchar2(255);
--      begin
--          l_url := APEX_UTIL.URL_ENCODE('http://www.example.com?id=1&cat=foo');
--      end;
--
-- `l_url` would be returned as:
--
--      `http%3A%2F%2Fwww%2Eexamplel%2Ecom%3Fid%3D1%26cat%3Dfoo`
--==============================================================================
function url_encode (
    p_url       in varchar2)
    return varchar2;

--==============================================================================
-- Needs Review: Mark internal or add to public documentation?
-- Return the virtual GUID used by APEX to generate component primary keys
--==============================================================================
function get_apex_id
    return number;

--==============================================================================
-- Needs Review: Mark internal or add to public documentation?
-- Return the virtual GUID used by APEX to generate component primary keys
-- compressed into alpha numeric format
--==============================================================================
function get_apex_char_id
    return varchar2;

--==============================================================================
-- Needs Review: Mark internal or add to public documentation?
-- Return a minimum of 4 character alphanumeric compressed value for an integer
--
-- PARAMETERS
--   p_number: any integer
--==============================================================================
function compress_int (
    p_number in integer)
    return   varchar2;

--==============================================================================
-- Needs Review: Mark internal or add to public documentation?
-- Return the number used in compress_int
--
-- PARAMETERS
--   p_str: alphanumeric character of compressed value
--==============================================================================
function uncompress_int (p_str in varchar2)
    return number;

--==============================================================================
-- Needs Review: Mark internal or add to public documentation?
-- Used to get return a uuid formatted guid
--==============================================================================
function get_uuid
    return varchar2;

--==============================================================================
-- This procedure removes session state for a given page for the current session.
-- If `p_page_id` is not specified, then the current page will be cleared.
--
-- Parameters:
-- * p_page_id:   The ID of the page in the current application for which session
--                state is to be cleared for current session.
--
-- Example:
-- The following example demonstrates how to use the `clear_page_cache` procedure
-- to clear the current session state for the page with an ID of 10.
--
--      begin
--          apex_util.clear_page_cache(10);
--      end;
--==============================================================================
procedure clear_page_cache (
    p_page_id in number default null);

--==============================================================================
-- This procedure removes session state for a given application for the current session.
--
-- Parameters:
-- * p_app_id: The ID of the application for which session state is to be cleared
--             for the current session.
--
-- Example:
-- The following example demonstrates how to use the `clear_app_cache` procedure to
-- clear the current session state for the application with an ID of 100.
--
--     begin
--         apex_util.clear_app_cache('100');
--     end;
--
--==============================================================================
procedure clear_app_cache (
    p_app_id    in varchar2 default null);

--==============================================================================
-- This procedure removes session state and application system preferences for the
-- current user's session. Run this procedure if you reuse session IDs and want to
-- run `APEX` applications without the benefit of existing session state.
--
-- Example:
--     begin
--      apex_util.clear_user_cache;
--     end;
--==============================================================================
procedure clear_user_cache;

--==============================================================================
-- Needs Review: Mark internal or add to public documentation?
-- Pause for number of seconds identified by p_seconds
--  (capped at 120 seconds)
--==============================================================================
procedure pause (
    p_seconds   in number);

--##############################################################################
--#
--# GET_SINCE AND VARIANTS
--#
--##############################################################################


--==============================================================================
-- Needs Review: Mark internal or add to public documentation?
-- This function is used to get a string representation of time in days,
-- for example: 5 days, 2 weeks, 8 seconds.
--
-- Parameters:
-- * p_days: numeric number of days
--==============================================================================
function get_timeframe (
    p_days in number default null)
    return varchar2;

--==============================================================================
-- Overload: date_bool
-- The function `get_since` is used to get a string representation of the time passed
-- since the input date, i.e. it returns the relative date in words
-- (for example, 2 days from now, 30 minutes ago).
-- It also accepts a second optional `p_short parameter` to return "in 2d" or "30m" and so forth.
-- This function is equivalent to using the format masks `SINCE` and `SINCE_SHORT` available
-- within Oracle APEX and is useful within SQL queries or PL/SQL routines.
--
-- Parameters:
-- * p_date:  The date that is used to compute the time from `p_date` until `sysdate`.
-- * p_short: Boolean parameter to get the short since format. e.g. "10h" instead of "10 hours ago".
--
-- Example:
-- This example returns the `LAST_UPDATE` column with the default formatting:
--
--     select application_id,
--            application_name,
--            apex_util.get_since( last_updated_on ) last_update
--       from apex_applications
--       order by application_id;
--==============================================================================
function get_since(
    p_date  in date,
    p_short in boolean default false )
    return varchar2;

--==============================================================================
-- Overload: timestamp_bool
-- The function `get_since` is used to get a string representation of the time passed
-- since the input date, i.e. it returns the relative date in words
-- (for example, 2 days from now, 30 minutes ago).
-- It also accepts a second optional `p_short parameter` to return "in 2d" or "30m" and so forth.
-- This function is equivalent to using the format masks `SINCE` and `SINCE_SHORT` available
-- within Oracle APEX and is useful within SQL queries or PL/SQL routines.
--
-- Parameters:
-- * p_value: The timestamp that is used to compute the time from `p_value` until `sysdate`.
-- * p_short: Boolean parameter to get the short since format. e.g. "10h" instead of "10 hours ago".
--
-- Example:
-- This example returns the `LAST_UPDATE` column with the default formatting:
--
--     select application_id,
--            application_name,
--            apex_util.get_since( last_updated_on ) last_update
--       from apex_applications
--       order by application_id;
--==============================================================================
function get_since(
    p_value in timestamp,
    p_short in boolean default false )
    return varchar2;

--==============================================================================
-- Overload: tswtz
-- The function `get_since` is used to get a string representation of the time passed
-- since the input date, i.e. it returns the relative date in words
-- (for example, 2 days from now, 30 minutes ago).
-- It also accepts a second optional `p_short parameter` to return "in 2d" or "30m" and so forth.
-- This function is equivalent to using the format masks `SINCE` and `SINCE_SHORT` available
-- within Oracle APEX and is useful within SQL queries or PL/SQL routines.
--
-- Parameters:
-- * p_value: The timestamp that is used to compute the time from `p_value` until `sysdate`.
-- * p_short: Boolean parameter to get the short since format. e.g. "10h" instead of "10 hours ago".
--
-- Example:
-- This example returns the `LAST_UPDATE` column with the default formatting:
--
--     select application_id,
--            application_name,
--            apex_util.get_since( last_updated_on ) last_update
--       from apex_applications
--       order by application_id;
--==============================================================================
function get_since(
    p_value in timestamp with time zone,
    p_short in boolean default false )
    return varchar2;

--==============================================================================
-- Overload: tswltz_bool
-- The function `get_since` is used to get a string representation of the time passed
-- since the input date, i.e. it returns the relative date in words
-- (for example, 2 days from now, 30 minutes ago).
-- It also accepts a second optional `p_short parameter` to return "in 2d" or "30m" and so forth.
-- This function is equivalent to using the format masks `SINCE` and `SINCE_SHORT` available
-- within Oracle APEX and is useful within SQL queries or PL/SQL routines.
--
-- Parameters:
-- * p_value: The timestamp that is used to compute the time from `p_value` until `sysdate`.
-- * p_short: Boolean parameter to get the short since format. e.g. "10h" instead of "10 hours ago".
--
-- Example:
-- This example returns the `LAST_UPDATE` column with the default formatting:
--
--     select application_id,
--            application_name,
--            apex_util.get_since( last_updated_on ) last_update
--       from apex_applications
--       order by application_id;
--==============================================================================
function get_since(
    p_value in timestamp with local time zone,
    p_short in boolean default false )
    return varchar2;

--==============================================================================
-- Overload: date_var
-- The function `get_since` is used to get a string representation of the time passed
-- since the input date, i.e. it returns the relative date in words
-- (for example, 2 days from now, 30 minutes ago).
-- It also accepts a second optional `p_short parameter` to return "in 2d" or "30m" and so forth.
--
-- Parameters:
-- * p_date:  The date that is used to compute the time from `p_date` until `sysdate`.
-- * p_short: 'Y' or 'N' parameter to get the short since format. e.g. "10h" instead of "10 hours ago".
--
-- Example:
-- This example returns the `LAST_UPDATE` column with the short formatting:
--
--     select application_id,
--            application_name,
--            apex_util.get_since( last_updated_on, p_short => 'Y' ) last_update
--       from apex_applications
--       order by application_id;
--==============================================================================
function get_since(
    p_date  in date,
    p_short in varchar2)
    return varchar2;

--==============================================================================
-- Overload: timestamp_var
-- The function `get_since` is used to get a string representation of the time passed
-- since the input date, i.e. it returns the relative date in words
-- (for example, 2 days from now, 30 minutes ago).
-- It also accepts a second optional `p_short parameter` to return "in 2d" or "30m" and so forth.
--
-- Parameters:
-- * p_value: The timestamp that is used to compute the time from `p_date` until `sysdate`.
-- * p_short: 'Y' or 'N' parameter to get the short since format. e.g. "10h" instead of "10 hours ago".
--
-- Example:
-- This example returns the `LAST_UPDATE` column with the short formatting:
--
--     select application_id,
--            application_name,
--            apex_util.get_since( last_updated_on, p_short => 'Y' ) last_update
--       from apex_applications
--       order by application_id;
--==============================================================================
function get_since(
    p_value in timestamp,
    p_short in varchar2 )
    return varchar2;

--==============================================================================
-- Overload: tswtz_var
-- The function `get_since` is used to get a string representation of the time passed
-- since the input date, i.e. it returns the relative date in words
-- (for example, 2 days from now, 30 minutes ago).
-- It also accepts a second optional `p_short parameter` to return "in 2d" or "30m" and so forth.
--
-- Parameters:
-- * p_value: The timestamp that is used to compute the time from `p_date` until `sysdate`.
-- * p_short: 'Y' or 'N' parameter to get the short since format. e.g. "10h" instead of "10 hours ago".
--
-- Example:
-- This example returns the `LAST_UPDATE` column with the short formatting:
--
--     select application_id,
--            application_name,
--            apex_util.get_since( last_updated_on, p_short => 'Y' ) last_update
--       from apex_applications
--       order by application_id;
--==============================================================================
function get_since(
    p_value in timestamp with time zone,
    p_short in varchar2 )
    return varchar2;

--==============================================================================
-- Overload: tswltz_var
-- The function `get_since` is used to get a string representation of the time passed
-- since the input date, i.e. it returns the relative date in words
-- (for example, 2 days from now, 30 minutes ago).
-- It also accepts a second optional `p_short parameter` to return "in 2d" or "30m" and so forth.
--
-- Parameters:
-- * p_value: The timestamp that is used to compute the time from `p_date` until `sysdate`.
-- * p_short: 'Y' or 'N' parameter to get the short since format. e.g. "10h" instead of "10 hours ago".
--
-- Example:
-- This example returns the `LAST_UPDATE` column with the short formatting:
--
--     select application_id,
--            application_name,
--            apex_util.get_since( last_updated_on, p_short => 'Y' ) last_update
--       from apex_applications
--       order by application_id;
--==============================================================================
function get_since(
    p_value in timestamp with local time zone,
    p_short in varchar2 )
    return varchar2;

--##############################################################################
--#
--# USER PREFERENCES
--#
--##############################################################################

--==============================================================================
-- This procedure sets a preference that persists beyond the user's current session.
--
-- Parameters:
-- * p_preference:   Name of the preference (case-sensitive).
-- * p_value:        Value of the preference.
-- * p_user:         User for whom the preference is being set.
--
-- Example:
-- The following example shows how to use the `SET_PREFERENCE` procedure to set a
-- preference called 'default_view' to the value 'WEEKLY' that persists beyond session
-- for the currently authenticated user.
--
--     begin
--      apex_util.set_preference(
--          p_preference => 'default_view',
--          p_value      => 'WEEKLY',
--          p_user       => :APP_USER);
--     end;
--
-- See Also:
-- GET_PREFERENCE
-- REMOVE_PREFERENCE
--==============================================================================
procedure set_preference (
    p_preference   in varchar2 default null,
    p_value        in varchar2 default null,
    p_user         in varchar2 default null);

--==============================================================================
-- This function retrieves the value of a previously saved preference for a given user.
--
-- Parameters:
-- * p_preference: The name of the preference to retrieve the value of.
-- * p_user:       User for whom the preference is being retrieved.
--
-- Example:
-- The following example shows how to use the `GET_PREFERENCE` function to return
-- the value for the currently authenticated user's preference named 'default_view'.
--
--     declare
--     l_default_view    varchar2(255);
--     begin
--         l_default_view := apex_util.get_preference(
--                              p_preference => 'default_view',
--                              p_user       => :APP_USER);
--     end;
--
-- See Also:
-- SET_PREFERENCE
-- REMOVE_PREFERENCE
--==============================================================================
function get_preference (
    p_preference   in varchar2 default null,
    p_user         in varchar2 default v('USER'))
    return varchar2;

--==============================================================================
-- This procedure removes the preference for the supplied user.
--
-- Parameters:
-- * p_preference:   Name of the preference to remove.
-- * p_user:         User for whom the preference is defined.
--
-- Example:
-- The following example shows how to use the `REMOVE_PREFERENCE` procedure to
-- remove the preference 'default_view' for the currently authenticated user.
--
--     begin
--      apex_util.remove_preference(
--          p_preference => 'default_view',
--          p_user       => :APP_USER);
--     end;
--
-- See Also:
-- GET_PREFERENCE
-- SET_PREFERENCE
--==============================================================================
procedure remove_preference(
    p_preference   in varchar2 default null,
    p_user         in varchar2 default v('USER'));

--==============================================================================
-- This procedure removes the user's column heading sorting preference value.
--
-- Parameters:
-- * p_user:  Identifies the user for whom sorting preferences are removed.
--
-- Example:
-- The following example shows how to use the `REMOVE_SORT_PREFERENCES` procedure
-- to remove the currently authenticated user's column heading sorting preferences.
--
--     begin
--      apex_util.remove_sort_preferences (:APP_USER);
--     end;
--==============================================================================
procedure remove_sort_preferences (
    p_user         in varchar2 default v('USER'));

--##############################################################################
--#
--# USERS AND GROUPS
--#
--##############################################################################

--==============================================================================
-- Needs Review: Mark internal or add to public documentation?
-- Given a group name return a boolean true or false if the current
-- application user is part of that group.
--
-- EXAMPLE
--   if apex_util.current_user_in_group('MY_GROUP') then ...
--==============================================================================
function current_user_in_group (
   p_group_name          in varchar2 )
   return boolean;

--==============================================================================
-- Needs Review: Mark internal or add to public documentation?
-- Given a group numeric identifier return a boolean true or false if the current
-- application user is part of that group.
--
-- EXAMPLE
--   if apex_util.current_user_in_group(1234567) then ...
--
--   The above example assumes that 1234567 is a valid group ID.
--==============================================================================
function current_user_in_group (
   p_group_id            in number)
   return boolean;

--==============================================================================
-- This function returns a Boolean result based on the validity of the password
-- for a named user account in the current workspace.
-- This function returns `TRUE` if the password matches and it returns `FALSE`
-- if the password does not match.
--
-- Parameters:
-- * p_username: User name in account.
-- * p_password: Password to be compared with password stored in the account.
--
-- Returns:
-- * true:  The user credentials are valid.
-- * false: The user credentials are invalid.
-- * null:  Credentials checking was delayed because of too many wrong
--          combinations.
--
-- Example:
-- The following example shows how to use the `IS_LOGIN_PASSWORD_VALID` function
-- to check if the user 'FRANK' has the password 'tiger'. `TRUE` is returned if
-- this is a valid password for 'FRANK', `FALSE` is returned if it's not.
--
--     declare
--         val boolean;
--     begin
--         val := apex_util.is_login_password_valid (
--             p_username=>'FRANK',
--             p_password=>'tiger');
--     end;
--==============================================================================
function is_login_password_valid (
   p_username in varchar2 default null,
   p_password in varchar2 default null )
   return boolean;

--==============================================================================
-- This function returns a Boolean result based on whether the named user account is unique in the
-- workspace.
--
-- Parameters:
-- * p_username: Identifies the user name to be tested.
--
-- Example:
-- The following example shows how to use the `IS_USERNAME_UNIQUE` function. If the user 'FRANK'
-- already exists in the current workspace, `FALSE` is returned, otherwise `TRUE` is returned.
--
--     declare
--         val boolean;
--     begin
--         val := apex_util.is_username_unique(
--             p_username=>'FRANK');
--     end;
--==============================================================================
function is_username_unique (
   p_username              in varchar2)
   return boolean;

--==============================================================================
-- This function returns the numeric ID of a named user group in the workspace.
--
-- Parameters:
-- * p_group_name: Identifies the name of a user group in the workspace.
--
-- Example:
-- The following example shows how to use the `GET_GROUP_ID` function to return
-- the ID for the user group named 'Managers'.
--
--      declare
--          l_id number;
--      begin
--          l_id := apex_util.get_group_id(p_group_name => 'Managers');
--      end;
--
-- See Also:
-- GET_GROUP_NAME Function
--==============================================================================
function get_group_id (
   p_group_name          in varchar2 )
   return number;

--==============================================================================
-- Needs Review: Mark internal or add to public documentation?
-- Overload: varchar2
-- This function returns the name of a user group identified by an ID.
--
-- Parameters:
-- * p_group_id: Identifies the ID of a user group in the workspace.
--
-- Example:
-- The following example shows how to use the `GET_GROUP_NAME` function to return
-- the name of the user group with the ID '8922003'.
--
--     declare
--         l_name varchar2(255);
--     begin
--         l_name := apex_util.get_group_name (p_group_id => '8922003');
--     end;
--
-- See Also:
-- GET_GROUP_ID
--==============================================================================
function get_group_name (
   p_group_id            in varchar2 )
   return varchar2;

--==============================================================================
-- Overload: number
-- This function returns the name of a user group identified by a numeric ID.
--
-- Parameters:
-- * p_group_id: Identifies the numeric ID of a user group in the workspace.
--
-- Example:
-- The following example shows how to use the `GET_GROUP_NAME` function to return
-- the name of the user group with the ID 8922003.
--
--      declare
--      l_name varchar2(255);
--      begin
--          l_name := apex_util.get_group_name (p_group_id => 8922003);
--      end;
--
-- See Also:
-- GET_GROUP_ID
--==============================================================================
function get_group_name (
   p_group_id            in number )
   return varchar2;

--==============================================================================
-- This function returns the email address associated with the named user.
--
-- Note:
-- This function only works with the standard user management system. If you use
-- it in applications with other user repositories, this API will not return useful
-- information.
--
-- Parameters:
-- * p_username: The user name in the account.
--
-- Example:
-- The following example shows how to use the `GET_EMAIL` function to return the
-- email address of the user 'FRANK'.
--
--      declare
--          l_email varchar2(240);
--      begin
--          l_email := apex_util.get_email(p_username => 'FRANK');
--      end;
--
-- See Also:
-- SET_EMAIL
--==============================================================================
function get_email (
   p_username            in varchar2)
   return varchar2;

--==============================================================================
-- This function returns the `FIRST_NAME` field stored in the named user account record.
--
-- Note:
-- This function only works with the standard user management system. If you use
-- it in applications with other user repositories, this API will not return useful
-- information.
--
-- Parameters:
-- * p_username: The user name in the account.
--
-- Example:
-- The following example shows how to use the `GET_FIRST_NAME` function to return the
-- `FIRST_NAME` of the user 'FRANK'.
--
--      declare
--          l_fname varchar2(240);
--      begin
--          l_fname := apex_util.get_first_name (p_username => 'FRANK');
--      end;
--
-- See Also:
-- SET_FIRST_NAME
--==============================================================================
function get_first_name (
   p_username            in varchar2)
   return varchar2;

--==============================================================================
-- This function returns the `LAST_NAME` field stored in the named user account record.
--
-- Note:
-- This function only works with the standard user management system. If you use
-- it in applications with other user repositories, this API will not return useful
-- information.
--
-- Parameters:
-- * p_username: The user name in the account.
--
-- Example:
-- The following example shows how to use the `GET_LAST_NAME` function to return the
-- `LAST_NAME` of the user 'FRANK'.
--
--      declare
--          l_lname varchar2(240);
--      begin
--          l_lname := apex_util.get_last_name (p_username => 'FRANK');
--      end;
--
-- See Also:
-- SET_LAST_NAME
--==============================================================================
function get_last_name (
   p_username            in varchar2)
   return varchar2;

--==============================================================================
-- This function returns the numeric identifier for the current user.
--
-- Note:
-- This function only works with the standard user management system. If you use
-- it in applications with other user repositories, this API will not return useful
-- information.
--
-- Example:
-- The following example shows how to use the `GET_CURRENT_USER_ID` function to
-- return the ID of the current APEX user.
--
--     declare
--         val number;
--     begin
--         val := apex_util.get_current_user_id;
--     end;
--
-- See Also:
-- GET_USER_ID
--==============================================================================
function get_current_user_id
   return number;

--==============================================================================
-- This function returns the numeric ID of a named user in the workspace.
--
-- Note:
-- This function only works with the standard user management system. If you use
-- it in applications with other user repositories, this API will not return useful
-- information.
--
-- Parameters:
-- * p_username: Identifies the name of a user in the workspace.
--
-- Example:
-- The following example shows how to use the `GET_USER_ID` function to return the
-- ID for the user named 'FRANK'.
--
--      declare
--          l_id number;
--      begin
--          l_id := apex_util.get_user_id (p_username => 'FRANK');
--      end;
--
-- See Also:
-- GET_CURRENT_USER_ID
--==============================================================================
function get_user_id (
   p_username            in varchar2)
   return number;

--==============================================================================
-- This function returns the user name of a user account identified by a numeric ID.
--
-- Note:
-- This function only works with the standard user management system. If you use
-- it in applications with other user repositories, this API will not return useful
-- information.
--
-- Parameters:
-- * p_userid: 	Identifies the numeric ID of a user account in the workspace.
--
-- Example:
-- The following example uses `GET_USERNAME` to return the user name for the user
-- with an ID of 228922003.
--
--      declare
--          val varchar2(100);
--      begin
--          val := apex_util.get_username(p_userid => 228922003);
--      end;
--
-- See Also:
-- SET_USERNAME
--==============================================================================
function get_username (
   p_userid              in number)
   return varchar2;

--==============================================================================
-- This function returns the value of one of the attribute values (1 through 10)
-- of a named user in the Oracle APEX accounts table. These are only accessible
-- by using the APIs.
--
-- Parameters:
-- * p_username:          User name in the account.
-- * p_attribute_number:  Number of attributes in the user record (1 through 10).
--
-- Example:
--
-- The following example returns the value for the 1st attribute for the user
-- named 'FRANK'.
--
--     declare
--         val varchar2(4000);
--     begin
--         val := apex_util.get_attribute (
--             p_username         => 'FRANK',
--             p_attribute_number => 1 ):
--     end;
--
-- See Also:
-- SET_ATTRIBUTE
--==============================================================================
function get_attribute (
   p_username             in varchar2,
   p_attribute_number     in number)
   return varchar2;

--==============================================================================
-- This function returns a comma then a space-separated list of group names to
-- which the named user is a member.
--
-- Parameters:
-- * p_username: Identifies the user name in the account.
--
-- Example:
-- The following example shows how to use the `GET_GROUPS_USER_BELONGS_TO` to return
-- the list of groups to which the user 'FRANK' is a member.
--
--      declare
--          val varchar2(32765);
--      begin
--          val := apex_util.get_groups_user_belongs_to(p_username => 'FRANK');
--      end;
--
-- See Also:
-- EDIT_USER
--==============================================================================
function get_groups_user_belongs_to (
   p_username            in varchar2)
   return varchar2;

--==============================================================================
-- This function returns the default schema name associated with the current user.
--
-- Example:
-- The following example shows how to use the `GET_DEFAULT_SCHEMA` function. It
-- returns the default schema name associated with the current user into a local
-- variable.
--
--      declare
--          val varchar2(30);
--      begin
--          val := apex_util.get_default_schema;
--      end;
--==============================================================================
function get_default_schema
   return varchar2;

--==============================================================================
-- This procedure updates a user account with a new email address. To execute this
-- procedure, the current user must have administrative privileges in the workspace.
--
-- Note:
-- This procedure only works with the standard user management system. If you use
-- it in applications with other user repositories, this API will not be of use.
--
-- Parameters:
-- * p_userid: The numeric ID of the user account.
-- * p_email:  The email address to be saved in the user account.
--
-- Example:
-- The following example shows how to use the `SET_EMAIL` procedure to set the
-- value of `EMAIL` to 'frank.scott@example.com' for the user 'FRANK'.
--
--     begin
--         apex_util.set_email(
--             p_userid  => apex_util.get_user_id('FRANK'),
--             p_email   => 'frank.scott@example.com');
--     end;
--==============================================================================
procedure set_email (
   p_userid              in number,
   p_email               in varchar2);

--==============================================================================
-- This procedure updates a user account with a new `FIRST_NAME` value.
-- To execute this procedure, the current user must have administrative privileges
-- in the workspace.
--
-- Note:
-- This procedure only works with the standard user management system. If you use
-- it in applications with other user repositories, this API will not be of use.
--
-- Parameters:
-- * p_userid:      The numeric ID of the user account.
-- * p_first_name:  The `FIRST_NAME` value to be saved in the user account.
--
-- Example:
-- The following example shows how to use the `SET_FIRST_NAME` procedure to set
-- the value of `FIRST_NAME` to 'Frank' for the user 'FRANK'.
--
--      begin
--          apex_util.set_first_name(
--              p_userid       => apex_util.get_user_id('FRANK'),
--              p_first_name   => 'Frank');
--      end;
--
-- See Also:
-- GET_FIRST_NAME
-- GET_USER_ID
--==============================================================================
procedure set_first_name (
   p_userid              in number,
   p_first_name          in varchar2);

--==============================================================================
-- This procedure updates a user account with a new `LAST_NAME` value.
-- To execute this procedure, the current user must have administrative privileges
-- in the workspace.
--
-- Note:
-- This procedure only works with the standard user management system. If you use
-- it in applications with other user repositories, this API will not be of use.
--
-- Parameters:
-- * p_userid:    The numeric ID of the user account.
-- * p_last_name: The `LAST_NAME` value to be saved in the user account.
--
-- Example:
-- The following example shows how to use the `SET_LAST_NAME` procedure to set
-- the value of `LAST_NAME` to 'Smith' for the user 'FRANK'.
--
--      begin
--          apex_util.set_last_name(
--              p_userid     => apex_util.get_user_id('FRANK'),
--              p_last_name  => 'Smith');
--      end;
--
-- See Also:
-- GET_LAST_NAME
-- GET_USER_ID
--==============================================================================
procedure set_last_name (
   p_userid              in number,
   p_last_name           in varchar2);

--==============================================================================
-- This procedure updates a user account with a new `USER_NAME` value. To execute
-- this procedure, the current user must have administrative privileges in the workspace.
--
-- Parameters:
-- * p_userid:   The numeric ID of the user account.
-- * p_username: The `USER_NAME` value to be saved in the user account.
--
-- Example:
-- The following example shows how to use the `SET_USERNAME` procedure to set the value
-- of `USER_NAME` to 'USER-XRAY' for the user 'FRANK'.
--
--     begin
--         apex_util.set_username(
--             p_userid     => apex_util.get_user_id('FRANK'),
--             p_username   => 'USER-XRAY');
--     end;
--==============================================================================
procedure set_username (
   p_userid              in number,
   p_username            in varchar2);

--==============================================================================
-- This procedure sets the value of one of the attribute values (1 through 10)
-- of a user in the Oracle APEX accounts table.
--
-- Parameters:
-- * p_userid:           The numeric ID of the user account.
-- * p_attribute_number: Attribute number in the user record (1 through 10).
-- * p_attribute_value:  Value of the attribute located by `p_attribute_number`
--                       to be set in the user record.
--
-- Example:
-- The following example sets the number 1 attribute for user 'FRANK' with the
-- value 'foo'.
--
--      begin
--          apex_util.set_attribute (
--              p_userid           => apex_util.get_user_id(p_username => 'FRANK'),
--              p_attribute_number => 1,
--              p_attribute_value  => 'foo');
--      end;
-- See Also:
-- GET_USER_ID
-- SET_ATTRIBUTE
--==============================================================================
procedure set_attribute (
   p_userid               in number,
   p_attribute_number     in number,
   p_attribute_value      in varchar2);

--==============================================================================
-- This procedure produces an export file of the current workspace definition, workspace users, and
-- workspace groups when called from a page. To execute this procedure, the current user must have
-- administrative privilege in the workspace.
--
-- Parameters:
-- * p_export_format: Indicates how rows in the export file are formatted:
--                    + Specify 'UNIX' to have the resulting file contain rows delimited by line feeds.
--                    + Specify 'DOS' to have the resulting file contain rows delimited by carriage
--                      returns and line feeds.
--
-- Example:
-- The following example calls this procedure from a page to produce an export file containing the
-- current workspace definition, list of workspace users, and list of workspace groups. The file is
-- formatted with rows delimited by line feeds.
--
--     begin
--         apex_util.export_users;
--     end;
--==============================================================================
procedure export_users (
    p_export_format in varchar2 default 'UNIX');

--==============================================================================
-- This procedure creates a user group when you are using Oracle APEX authentication.
--
-- Note:
-- To execute this procedure within the context of an APEX application, the current
-- user must be an APEX workspace administrator and the application must permit
-- modification of the workspace repository.
--
-- Parameters:
-- * p_id                : The primary key of the user group.
-- * p_group_name        : The name of the group.
-- * p_security_group_id : Identifies which workspace this user group belongs to.
--                         Defaults to the current workspace.
-- * p_group_desc        : Text description used as a note to the developer.
--
-- Example:
-- The following example demonstrates how to use the `CREATE_USER_GROUP` procedure
-- to create a new group called 'Managers' with a description of 'text'.
-- Pass `NULL` for the `p_id` parameter to enable the database trigger to assign
-- the new primary key value.
--
--      begin
--          apex_util.create_user_group (
--              p_id                => null,
--              p_group_name        => 'Managers',
--              p_security_group_id => null,
--              p_group_desc        => 'text');
--      end;
-- See Also:
-- DELETE_USER_GROUP@number
-- DELETE_USER_GROUP@varchar2
--==============================================================================
procedure create_user_group (
    p_id                      in number    default null,
    p_group_name              in varchar2,
    p_security_group_id       in number    default null,
    p_group_desc              in varchar2  default null );

--==============================================================================
-- Overload: number
-- This procedure deletes a user group by providing the primary key of the group
-- when you are using Oracle APEX authentication. To execute this procedure,
-- the current user must have administrative privileges in the workspace.
--
-- Parameters:
-- * p_group_id: The primary key of the group.
--
-- Example:
-- The following example removes the user group called 'Managers' by providing
-- the user group's primary key.
--
--      declare
--          val number;
--      begin
--          val := apex_util.get_group_id (p_group_name => 'Managers');
--          apex_util.delete_user_group (p_group_id => val);
--      end;
--
-- See Also:
-- CREATE_USER_GROUP
-- DELETE_USER_GROUP@varchar2
--==============================================================================
procedure delete_user_group (
    p_group_id                 in number);

--==============================================================================
-- Overload: varchar2
-- This procedure deletes a user group by providing the name of the group
-- when you are using Oracle APEX authentication. To execute this procedure,
-- the current user must have administrative privileges in the workspace.
--
-- Parameters:
-- * p_group_name: The name of the group.
--
-- Example:
-- The following example removes the user group called 'Managers' by providing
-- the name of the user group.
--
--      begin
--          apex_util.delete_user_group (
--              p_group_name => 'Managers');
--      end;
--
-- See Also:
-- CREATE_USER_GROUP
-- DELETE_USER_GROUP@number
--==============================================================================
procedure delete_user_group (
    p_group_name              in varchar2);

--==============================================================================
-- This procedure modifies the group grants for a given group.
--
-- Parameters:
-- * p_group_name          : The target group name.
-- * p_granted_group_names : The names of groups to grant to `p_group_name`
--
-- Example:
-- This example creates three groups (`ACCTS_PAY`, `ACCTS_REC`, `MANAGER`) and then
-- grants `ACCTS_PAY` and `ACCTS_REC` to `MANAGER`.
--
--      begin
--        -- create three groups
--        apex_util.create_user_group (
--            p_group_name => 'ACCTS_PAY' );
--        apex_util.create_user_group (
--            p_group_name => 'ACCTS_REC' );
--        apex_util.create_user_group (
--            p_group_name => 'MANAGER' );
--
--        -- grant ACCTS_PAY and ACCTS_REC to MANAGER
--        apex_util.set_group_group_grants (
--            p_group_name          => 'MANAGER',
--            p_granted_group_names => apex_t_varchar2('ACCTS_PAY', 'ACCTS_REC') );
--      end;
--==============================================================================
procedure set_group_group_grants (
    p_group_name              in varchar2,
    p_granted_group_names     in wwv_flow_t_varchar2 );

--==============================================================================
-- This procedure modifies the group grants for a given user.
--
-- Parameters:
-- * p_user_name:            The target user name.
-- * p_granted_group_names : The names of groups that should to grant to p_user_name.
--
-- Example:
-- This example creates a user group ('MANAGER') and a user ('Example User') and
-- then grants 'MANAGER' to 'Example User'.
--
--     declare
--     l_random_password varchar2(255) := 'oracleapex';
--     begin
--     -- create group and user
--     apex_util.create_user_group (
--         p_group_name => 'MANAGER' );
--     apex_util.create_user (
--         p_user_name    => 'Example User',
--         p_web_password => l_random_password );
--     -- grant MANAGER to Example User
--     apex_util.set_group_user_grants (
--         p_user_name           => 'Example User',
--         p_granted_group_names => apex_t_varchar2('MANAGER') );
--     end;
--==============================================================================
procedure set_group_user_grants (
    p_user_name               in varchar2,
    p_granted_group_names     in wwv_flow_t_varchar2 );

--==============================================================================
-- This procedure creates a new account record in the Oracle APEX user accounts table.
--
-- Use this procedure to programmatically create user accounts for applications
-- that utilize the APEX Accounts authentication scheme. To execute this procedure
-- within the context of an APEX application, the current user must be an APEX workspace
-- administrator and the application must permit modification of the workspace repository.
--
-- When creating workspace developer or workspace administrator users, you must also
-- ensure that the user can authenticate to the development environment authentication scheme.
-- The `CREATE_USER` procedure only creates the APEX repository user. For example, if using
-- DB accounts authentication, you must also run `CREATE USER nnn IDENTIFIED BY yyy`.
--
-- Note: Currently, the following parameters are named inconsistently between the
-- `CREATE_USER`, `EDIT_USER`, and `FETCH_USER` APIs, although they all relate to
-- the `DEVELOPER_ROLE` field stored in the named user account record:
-- + `CREATE_USER` uses `p_developer_privs`.
-- + `EDIT_USER` uses `p_developer_roles`.
-- + `FETCH_USER` uses `p_developer_role`.
--
-- Parameters:
-- * p_user_id:                         Numeric primary key of user account.
-- * p_user_name:                       Alphanumeric name used for login.
-- * p_first_name:                      Informational only.
-- * p_last_name:                       Informational only.
-- * p_description:                     Informational only.
-- * p_email_address:                   The new user's email address.
-- * p_web_password:                    Password for the new user.
-- * p_web_password_format:             If the value you are passing for the `p_web_password` parameter
--                                      is in clear text format then use `CLEAR_TEXT`, otherwise
--                                      use `HEX_ENCODED_DIGEST_V2`.
--!                                     The range of values also includes HEX_ENCODED_DIGEST, DIGEST.
-- * p_group_ids:                       Colon separated list of numeric group IDs.
-- * p_developer_privs:                 Colon separated list of developer privileges.
--                                      - If `p_developer_privs` is not null, the user is given access to Team Development.
--                                      - If `p_developer_privs` contains 'ADMIN', the user is given App Builder and SQL
--                                        Workshop access.
--                                      - If `p_developer_privs` does not contain 'ADMIN' but contains 'EDIT', the user is
--                                        given App Builder access.
--                                      - If `p_developer_privs` does not contain 'ADMIN' but contains 'SQL', the user is
--                                        given SQL Workshop access.
--                                      - The following are acceptable values for this parameter:
--                                           - `NULL`: To create an end user (a user who can only authenticate to developed
--                                           applications).
--                                           - `CREATE:DATA_LOADER:EDIT:HELP:MONITOR:SQL`: To create a user with developer
--                                           privileges with access to App Builder and SQL Workshop.
--                                           - `ADMIN:CREATE:DATA_LOADER:EDIT:HELP:MONITOR:SQL`: To create a user with full
--                                           workspace administrator and developer privileges with access to App Builder,
--                                           SQL Workshop and Team Development.
--!                                     Further developer privs include: BROWSE:DB_MONITOR:USER_MANAGER
-- * p_default_schema:                  A database schema assigned to the user's workspace, used by default for browsing.
-- * p_default_date_format:             Oracle Date format for user, used for now only in SQL Workshop.
-- * p_allow_access_to_schemas:         Colon-separated list of schemas assigned to the user's workspace to which the user
--                                      is restricted (leave `NULL` for all).
-- * p_account_expiry:                  The date the password was last updated, which defaults to today's date on creation.
-- * p_account_locked:                  `Y` or `N` indicating if account is locked or unlocked.
-- * p_failed_access_attempts:          Number of consecutive login failures that have occurred, defaults to 0 on creation.
-- * p_change_password_on_first_use:    `Y` or `N` to indicate whether password must be changed on first use, defaults to
--                                      `Y` on creation.
-- * p_first_password_use_occurred:     `Y` or `N` to indicate whether login has occurred since password change, defaults
--                                      to `N` on creation.
-- * p_attribute_01 ... p_attribute_10: These attributes allow you to store arbitrary information about a given user.
--!                                     They are for use by flow developers who want to extend user information.
-- * p_allow_app_building_yn:           `Y` or `N` to indicate whether access to App Builder is enabled.
-- * p_allow_sql_workshop_yn:           `Y` or `N` to indicate whether access to SQL Workshop is enabled.
-- * p_allow_websheet_dev_yn:           `Y` or `N` to indicate whether access to Websheet development is enabled. Obsolete
--                                      from version 22.1.
-- * p_allow_team_development_yn:       `Y` or `N` to indicate whether access to Team Development is enabled.
--
-- Example:
-- This example creates ten new end users from sqlplus logged in as a schema assigned to the target workspace. End users
-- can only authenticate
-- to developed applications.
--
--     begin
--        for i in 1..10 loop
--           apex_util.create_user(
--              p_user_name     => 'USER_'||i,
--              p_email_address => 'user_'||i||'@example.com',
--              p_web_password  => 'user_'||i) ;
--        end loop;
--        commit;
--     end;
--
-- Example:
-- The following example creates an end user called `NEWUSER1` with a password of 'secret99'. End users can only authenticate
-- to developed applications.
--
--     begin
--         apex_util.create_user(
--                p_user_name    => 'NEWUSER1',
--                p_web_password => 'secret99');
--     end;
--
-- Example:
-- The following example creates a Workspace Administrator called `NEWUSER2`, who:
--
-- * has full workspace administration and developer privilege (`p_developer_privs` parameter set to
--  'ADMIN:CREATE:DATA_LOADER:EDIT:HELP:MONITOR:SQL')
-- * has access to 2 schemas, both their browsing default 'MY_SCHEMA' (`p_default_schema` parameter set to 'MY_SCHEMA')
-- and also 'MY_SCHEMA2' (`p_allow_access_to_schemas` parameter set to 'MY_SCHEMA2')
-- * does not have to change their password when they first login (`p_change_password_on_first_use` parameter set to `N`)
-- * and has their phone number stored in the first additional attribute (`p_attribute_01` parameter set to '123 456 7890').
--  ~~~
--  begin
--       apex_util.create_user(
--           p_user_name                     => 'NEWUSER2',
--           p_first_name                    => 'FRANK',
--           p_last_name                     => 'SMITH',
--           p_description                   => 'Description...',
--           p_email_address                 => 'frank@example.com',
--           p_web_password                  => 'password',
--           p_developer_privs               => 'ADMIN:CREATE:DATA_LOADER:EDIT:HELP:MONITOR:SQL',
--           p_default_schema                => 'MY_SCHEMA',
--           p_allow_access_to_schemas       => 'MY_SCHEMA2',
--           p_change_password_on_first_use  => 'N',
--           p_attribute_01                  => '123 456 7890');
--  end;
-- ~~~
--
-- See Also:
-- EDIT_USER
-- FETCH_USER@emp_info
-- FETCH_USER@short
-- FETCH_USER@account_info
-- REMOVE_USER@number
-- REMOVE_USER@varchar2
--==============================================================================
procedure create_user (
    p_user_id                      in number   default null,
    p_user_name                    in varchar2,
    p_first_name                   in varchar2 default null,
    p_last_name                    in varchar2 default null,
    p_description                  in varchar2 default null,
    p_email_address                in varchar2 default null,
    p_web_password                 in varchar2,
    p_web_password_format          in varchar2 default 'CLEAR_TEXT',
    p_group_ids                    in varchar2 default null,
    p_developer_privs              in varchar2 default null,
    p_default_schema               in varchar2 default null,
    p_default_date_format          in varchar2 default null,
    p_allow_access_to_schemas      in varchar2 default null,
    p_account_expiry               in date     default trunc(sysdate),
    p_account_locked               in varchar2 default 'N',
    p_failed_access_attempts       in number   default 0,
    p_change_password_on_first_use in varchar2 default 'Y',
    p_first_password_use_occurred  in varchar2 default 'N',
    p_attribute_01                 in varchar2 default null,
    p_attribute_02                 in varchar2 default null,
    p_attribute_03                 in varchar2 default null,
    p_attribute_04                 in varchar2 default null,
    p_attribute_05                 in varchar2 default null,
    p_attribute_06                 in varchar2 default null,
    p_attribute_07                 in varchar2 default null,
    p_attribute_08                 in varchar2 default null,
    p_attribute_09                 in varchar2 default null,
    p_attribute_10                 in varchar2 default null,
    p_allow_app_building_yn        in varchar2 default null,
    p_allow_sql_workshop_yn        in varchar2 default null,
    p_allow_websheet_dev_yn        in varchar2 default null, -- Ignore:
    p_allow_team_development_yn    in varchar2 default null );

--==============================================================================
-- Needs Review: Mark internal or add to public documentation?
-- This procedure allows for programmatic and bulk creation of users from a text
-- file.  The file must be in one of two formats.
--
-- FORMAT 1
--    username,email,PRIV1:PRIV2,password
--
-- FORMAT 2
--    username,email,PRIV1:PRIV2
--
-- PRIV1:PRIV2 is a colon delimited list of developer privileges of the
-- user.  Valid privileges are detailed in create_fnd_user spec.
--
-- Each line must end with a line feed (chr(10)).
--
-- PARAMETERS
--    p_id          The id of the file in wwv_flow_file_objects$
--    p_mode        Either CREATE or display depending on whether you
--                  actually want to create the users or just display them
--                  via htp.p
--    p_format      Can be either 1, 2 or 3.  Refers to the format of the file
--                  and the create user process.  1: passwords are contained in
--                  the file in clear text.  2: passwords have been supplied via
--                  the p_password parameter.  3: passwords should be randomly
--                  generated and then e-mailed to the user.
--    p_password    Only relevant in format 2.  Every user created is given the
--                  password supplied in this parameter.
--    p_app         The name of the application.  This name appears in the subject
--                  and body of the e-mail message sent to users when format is 3.
--    p_appurl      Optional URL can be supplied when file format is 3.  The URL will
--                  appear at the end of the mail message.
--    p_start       This parameter holds the start time of execution
--    p_end         Parameter holds the time the procedure completes
--    p_loaded      The number of users that were provisioned by the execution of the
--                  procedure.
--==============================================================================
procedure create_user_from_file (
    p_id in number,
    p_mode      in varchar2 default 'CREATE',
    p_format    in varchar2 default '1',
    p_password  in varchar2 default 'oracle',
    p_app       in varchar2 default null,
    p_appurl    in varchar2 default null,
    p_start    out varchar2,
    p_end      out varchar2,
    p_loaded   out varchar2 );

--==============================================================================
-- Overload: number
-- This procedure removes the user account identified by the primary key. To
-- execute this procedure, the current user must have administrative privilege in
-- the workspace.
--
-- Parameters:
-- * p_user_id: The numeric primary key of the user account record.
--
-- Example:
-- The following examples show how to use the `REMOVE_USER` procedure to remove
-- a user account by the primary key using the `p_user_id` parameter.
--
--     begin
--         apex_util.remove_user(p_user_id=> 99997);
--     end;
--
-- See Also:
-- REMOVE_USER@varchar2
--==============================================================================
procedure remove_user (
    p_user_id         in number);

--==============================================================================
-- Overload: varchar2
-- This procedure removes the user account identified by the user name. To
-- execute this procedure, the current user must have administrative privilege in
-- the workspace.
--
-- Parameters:
-- * p_user_name: The user name of the user account.
--
-- Example:
-- The following examples show how to use the `REMOVE_USER` procedure to remove
-- a user account by user name using the `p_user_name` parameter.
--
--     begin
--         wwv_flow_security.g_security_group_id := 20;
--     end;
--     /
--     begin
--        for i in 1..10 loop
--           apex_util.remove_user(
--              p_user_name => 'USER_'||i);
--        end loop;
--        commit;
--     end;
--
-- See Also:
-- REMOVE_USER@number
--==============================================================================
procedure remove_user (
    p_user_name       in varchar2 );

--==============================================================================
-- This procedure changes the password of the currently authenticated user, assuming
-- Oracle APEX user accounts are in use.
--
-- Parameters:
-- * p_new_password: The new password value in clear text.
--
-- Example:
-- The following example demonstrates how to use the `CHANGE_CURRENT_USER_PW`
-- procedure to change the password for the user who is currently authenticated,
-- assuming APEX accounts are in use.
--
--     begin
--         apex_util.change_current_user_pw ('secret99');
--     end;
--
-- See Also:
-- RESET_PW
-- RESET_PASSWORD
--==============================================================================
procedure change_current_user_pw (
    p_new_password    in varchar2 );

--==============================================================================
-- This procedure resets the password for a named user and emails it in a message
-- to the email address located for the named account in the current workspace.
-- To execute this procedure, the current user must have administrative privilege
-- in the workspace.
--
-- Parameters:
-- * p_user: The user name of the user account.
-- * p_msg:  Message text to be mailed to a user.
--
-- Example:
-- The following example shows how to use the `RESET_PW` procedure to reset the
-- password for the user 'FRANK'.
--
--     begin
--         apex_util.reset_pw(
--             p_user => 'FRANK',
--             p_msg  => 'Contact help desk at 555-1212 with questions');
--     end;
--
-- See Also:
-- CHANGE_CURRENT_USER_PW
-- RESET_PASSWORD
--==============================================================================
procedure reset_pw (
    p_user                  in varchar2,
    p_msg                   in varchar2 );

--==============================================================================
-- This procedure changes the password of `p_user_name` in the current workspace
-- to `p_new_password`. If `p_change_password_on_first_use` is `TRUE`, then the
-- user has to change the password on the next login.
--
-- This procedure can be used to implement a custom 'change my password' page in
-- an application.
--
-- Parameters:
-- * p_user_name:                    The user whose password should be changed.
--                                   The default is the currently logged in APEX
--                                   user.
-- * p_old_password:                 The current password of the user. The call succeeds
--                                   if the given value matches the current password or
--                                   it is `NULL` and the owner of the calling PL/SQL code
--                                   has APEX_ADMINISTRATOR_ROLE. If the value is not
--                                   the user's password, an error occurs.
-- * p_new_password:                 The new password.
-- * p_change_password_on_first_use: If `TRUE` (default), the user has to
--                                   change the password on the next login.
--
-- Raises:
-- * INVALID_CREDENTIALS:                        This error occurs if `p_old_password`
--                                               does not match the current password of
--                                               the user named in `p_user_name`.
-- * APEX.AUTHENTICATION.LOGIN_THROTTLE.COUNTER: Indicates that authentication was prevented by
--                                               login throttle.
-- * internal error:                             1. Occurs if `p_old_password` is `NULL` and the
--                                                  caller does not have `APEX_ADMINISTRATOR_ROLE`.
--                                               2. Indicates that the caller is not a valid workspace schema.
--
-- Example:
-- This example demonstrates how to change the password of the currently logged-in user to a new password.
--
--     begin
--       apex_util.reset_password (
--           p_old_password => :P111_OLD_PASSWORD,
--           p_new_password => :P111_NEW_PASSWORD );
--     end;
--
-- See Also:
-- CHANGE_CURRENT_USER_PW
-- RESET_PW
--==============================================================================
procedure reset_password (
    p_user_name                    in varchar2 default wwv_flow_security.g_user,
    p_old_password                 in varchar2 default null,
    p_new_password                 in varchar2,
    p_change_password_on_first_use in boolean default true );

--==============================================================================
-- Ignore: Obsolete
--==============================================================================
function user_in_group (
    p_group_name in varchar2)
    return boolean;

--==============================================================================
-- This function returns the `DEVELOPER_ROLE` field stored in the named user account record.
--
-- Parameters:
-- * p_username: Identifies a user name in a user account.
--
-- Example:
-- The following example shows how to use the `GET_USER_ROLES` function to return a
-- colon separated list of roles stored in the `DEVELOPER_ROLE` field for the user 'FRANK'.
--
--     declare
--        val varchar2(4000);
--     begin
--         val := apex_util.get_user_roles(p_username=>'FRANK');
--     end;
--
-- See Also:
-- CREATE_USER
-- EDIT_USER
-- FETCH_USER@emp_info
-- FETCH_USER@short
-- FETCH_USE@account_info
--==============================================================================
function get_user_roles (
    p_username              in varchar2)
    return varchar2;

--==============================================================================
-- This function returns the numeric security group ID of the named workspace.
--
-- Parameters:
-- * p_workspace: The name of the workspace.
--
-- Example:
-- The following example demonstrates how to use the `FIND_SECURITY_GROUP_ID` function
-- to return the security group ID for the workspace called 'DEMOS'.
--
--     declare
--        val number;
--     begin
--        val := apex_util.find_security_group_id(p_workspace=>'DEMOS');
--     end;
--
-- See Also:
-- FIND_WORKSPACE
-- SET_SECURITY_GROUP_ID
--==============================================================================
function find_security_group_id (
     p_workspace  in varchar2 default null)
     return number;

--==============================================================================
-- This function returns the workspace name associated with a security group ID.
--
-- Parameters:
-- * p_security_group_id: The security group ID of a workspace.
--
-- Example:
-- The following example demonstrates how to use the `FIND_WORKSPACE` function to
-- return the workspace name for the workspace with a security group ID of 20.
--
--     declare
--        val varchar2;
--     begin
--        val := apex_util.find_workspace(p_security_group_id=>'20');
--     end;
--
-- See Also:
-- FIND_SECURITY_GROUP_ID
--==============================================================================
function find_workspace (
     p_security_group_id  in varchar2 default null)
     return varchar2;

--==============================================================================
-- Overload: emp_info
-- This procedure fetches a user account record. To execute this procedure, the current user must
-- have administrative privileges in the workspace. Three overloaded versions of this procedure exist,
-- each with a distinct set of allowed parameters or signatures.
--! It fetches user information from
--! wwv_flow_fnd_user, wwv_flow_fnd_group_users and wwv_flow_developers table.
--
-- Note: Currently, the following parameters are named inconsistently between the
-- `CREATE_USER`, `EDIT_USER`, and `FETCH_USER` APIs, although they all relate to
-- the `DEVELOPER_ROLE` field stored in the named user account record:
-- + `CREATE_USER` uses `p_developer_privs`.
-- + `EDIT_USER` uses `p_developer_roles`.
-- + `FETCH_USER` uses `p_developer_role`.
--
-- Parameters:
-- * p_user_id:                 Numeric primary key of the user account.
-- * p_workspace:               The name of the workspace.
-- * p_user_name:               Alphanumeric name used for login. See also `GET_USERNAME`.
-- * p_first_name:              Informational. See also `GET_FIRST_NAME`.
-- * p_last_name:               Informational. See also `GET_LAST_NAME`.
-- * p_web_password:            Obfuscated account password.
-- * p_email_address:           Email address. See also `GET_EMAIL`.
-- * p_start_date:              Unused.
-- * p_end_date:                Unused.
-- * p_employee_id:             Unused.
-- * p_allow_access_to_schemas: A list of schemas assigned to the user's workspace to which the user is restricted.
-- * p_person_type:             Unused.
-- * p_default_schema:          A database schema assigned to the user's workspace, used by default for browsing.
-- * p_groups:                  A list of groups of which the user is a member. See also `GET_GROUPS_USER_BELONGS_TO`
--                              and `CURRENT_USER_IN_GROUP`.
-- * p_developer_role:          Colon separated list of developer privileges. The following are acceptable
--                              values for this parameter:
--                                 + `NULL`: Indicates an end user (a user who can only authenticate to developed
--                                 applications).
--                                 + `CREATE:DATA_LOADER:EDIT:HELP:MONITOR:SQL`: Indicates a user with developer
--                                 privileges.
--                                 + `ADMIN:CREATE:DATA_LOADER:EDIT:HELP:MONITOR:SQL`: Indicates a user with full
--                                 workspace administrator and developer privileges.
--                                 See also `GET_USER_ROLES`.
-- * p_description:             Informational.
--
-- Example:
-- The following example shows how to use the `FETCH_USER` procedure.
-- This procedure is passed the ID of the currently authenticated user for the only IN parameter `p_user_id`.
-- The code then stores all the other OUT parameter values in local variables.
--
--     declare
--         l_workspace                 varchar2(255);
--         l_user_name                 varchar2(100);
--         l_first_name                varchar2(255);
--         l_last_name                 varchar2(255);
--         l_web_password              varchar2(255);
--         l_email_address             varchar2(240);
--         l_start_date                date;
--         l_end_date                  date;
--         l_employee_id               number(15,0);
--         l_allow_access_to_schemas   varchar2(4000);
--         l_person_type               varchar2(1);
--         l_default_schema            varchar2(30);
--         l_groups                    varchar2(1000);
--         l_developer_role            varchar2(60);
--         l_description               varchar2(240);
--     begin
--         apex_util.fetch_users(
--             p_user_id                   => apex_util.get_current_user_id,
--             p_workspace                 => l_workspace,
--             p_user_name                 => l_user_name,
--             p_first_name                => l_first_name,
--             p_last_name                 => l_last_name,
--             p_web_password              => l_web_password,
--             p_email_address             => l_email_address,
--             p_start_date                => l_start_date,
--             p_end_date                  => l_end_date,
--             p_employee_id               => l_employee_id,
--             p_allow_access_to_schemas   => l_allow_access_to_schemas,
--             p_person_type               => l_person_type,
--             p_default_schema            => l_default_schema,
--             p_groups                    => l_groups,
--             p_developer_role            => l_developer_role,
--             p_description               => l_description);
--     end;
--
-- See Also:
-- CREATE_USER
-- EDIT_USER
-- FETCH_USER@short
-- FETCH_USER@account_info
--==============================================================================
procedure fetch_user (
     p_user_id                 in number,
     p_workspace               out varchar2,
     p_user_name               out varchar2,
     p_first_name              out varchar2,
     p_last_name               out varchar2,
     p_web_password            out varchar2,
     p_email_address           out varchar2,
     p_start_date              out varchar2,
     p_end_date                out varchar2,
     p_employee_id             out varchar2,
     p_allow_access_to_schemas out varchar2,
     p_person_type             out varchar2,
     p_default_schema          out varchar2,
     p_groups                  out varchar2,
     p_developer_role          out varchar2,
     p_description             out varchar2 );

--==============================================================================
-- Overload: short
-- This procedure fetches a user account record. To execute this procedure, the current user must
-- have administrative privileges in the workspace. Three overloaded versions of this procedure exist,
-- each with a distinct set of allowed parameters or signatures.
--! It fetches user information from
--! wwv_flow_fnd_user, wwv_flow_fnd_group_users and wwv_flow_developers table.
--
-- Note: Currently, the following parameters are named inconsistently between the
-- `CREATE_USER`, `EDIT_USER`, and `FETCH_USER` APIs, although they all relate to
-- the `DEVELOPER_ROLE` field stored in the named user account record:
-- + `CREATE_USER` uses `p_developer_privs`.
-- + `EDIT_USER` uses `p_developer_roles`.
-- + `FETCH_USER` uses `p_developer_role`.
--
-- Parameters:
-- * p_user_id:         Numeric primary key of the user account.
-- * p_user_name:       Alphanumeric name used for login. See also `GET_USERNAME`.
-- * p_first_name:      Informational. See also `GET_FIRST_NAME`.
-- * p_last_name:       Informational. See also `GET_LAST_NAME`.
-- * p_email_address:   Email address. See also `GET_EMAIL`.
-- * p_groups:          A list of groups of which the user is a member.
--                      See also `GET_GROUPS_USER_BELONGS_TO` and `CURRENT_USER_IN_GROUP`.
-- * p_developer_role:  Colon separated list of developer privileges. The following are acceptable
--                      values for this parameter:
--                         + `NULL`: Indicates an end user (a user who can only authenticate to developed
--                         applications).
--                         + `CREATE:DATA_LOADER:EDIT:HELP:MONITOR:SQL`: Indicates a user with developer
--                         privileges.
--                         + `ADMIN:CREATE:DATA_LOADER:EDIT:HELP:MONITOR:SQL`: Indicates a user with full
--                         workspace administrator and developer privileges.
--                         See also `GET_USER_ROLES`.
-- * p_description:     Informational.
--
-- Example:
-- The following example shows how to use the `FETCH_USER` procedure.
-- This procedure is passed the ID of the currently authenticated user for the only IN parameter `p_user_id`.
-- The code then stores all the other OUT parameter values in local variables.
--
--     declare
--         l_user_name                 varchar2(100);
--         l_first_name                varchar2(255);
--         l_last_name                 varchar2(255);
--         l_email_address             varchar2(240);
--         l_groups                    varchar2(1000);
--         l_developer_role            varchar2(60);
--         l_description               varchar2(240);
--     begin
--         apex_util.fetch_users(
--             p_user_id                   => apex_util.get_current_user_id,
--             p_user_name                 => l_user_name,
--             p_first_name                => l_first_name,
--             p_last_name                 => l_last_name,
--             p_email_address             => l_email_address,
--             p_groups                    => l_groups,
--             p_developer_role            => l_developer_role,
--             p_description               => l_description);
--     end;
--
-- See Also:
-- CREATE_USER
-- EDIT_USER
-- FETCH_USER@emp_info
-- FETCH_USER@account_info
--==============================================================================
procedure fetch_user (
     p_user_id         in number,
     p_user_name       out varchar2,
     p_first_name      out varchar2,
     p_last_name       out varchar2,
     p_email_address   out varchar2,
     p_groups          out varchar2,
     p_developer_role  out varchar2,
     p_description     out varchar2 );

--==============================================================================
-- Overload: account_info
-- This procedure fetches a user account record. To execute this procedure, the current user must
-- have administrative privileges in the workspace. Three overloaded versions of this procedure exist,
-- each with a distinct set of allowed parameters or signatures.
--! It fetches user information from
--! wwv_flow_fnd_user, wwv_flow_fnd_group_users and wwv_flow_developers table.
--
-- Note: Currently, the following parameters are named inconsistently between the
-- `CREATE_USER`, `EDIT_USER`, and `FETCH_USER` APIs, although they all relate to
-- the `DEVELOPER_ROLE` field stored in the named user account record:
-- + `CREATE_USER` uses `p_developer_privs`.
-- + `EDIT_USER` uses `p_developer_roles`.
-- + `FETCH_USER` uses `p_developer_role`.
--
-- Parameters:
-- * p_user_id:                      Numeric primary key of the user account.
-- * p_workspace:                    The name of the workspace.
-- * p_user_name:                    Alphanumeric name used for login. See also `GET_USERNAME`.
-- * p_first_name:                   Informational. See also `GET_FIRST_NAME`.
-- * p_last_name:                    Informational. See also `GET_LAST_NAME`.
-- * p_web_password:                 Obfuscated account password.
-- * p_email_address:                Email address. See also `GET_EMAIL`.
-- * p_start_date:                   Unused.
-- * p_end_date:                     Unused.
-- * p_employee_id:                  Unused.
-- * p_allow_access_to_schemas:      A list of schemas assigned to the user's workspace to which the user is restricted.
-- * p_person_type:                  Unused.
-- * p_default_schema:               A database schema assigned to the user's workspace, used by default for browsing.
-- * p_groups:                       A list of groups of which the user is a member. See also `GET_GROUPS_USER_BELONGS_TO`
--                                   and `CURRENT_USER_IN_GROUP`.
-- * p_developer_role:               Colon separated list of developer privileges. The following are acceptable
--                                   values for this parameter:
--                                      + `NULL`: Indicates an end user (a user who can only authenticate to developed
--                                      applications).
--                                      + `CREATE:DATA_LOADER:EDIT:HELP:MONITOR:SQL`: Indicates a user with developer
--                                      privileges.
--                                      + `ADMIN:CREATE:DATA_LOADER:EDIT:HELP:MONITOR:SQL`: Indicates a user with full
--                                      workspace administrator and developer privileges.
--                                      See also `GET_USER_ROLES`.
-- * p_description:                  Informational.
-- * p_account_expiry:               Date account password was last reset. See also `END_USER_ACCOUNT_DAYS_LEFT`
--                                   and `WORKSPACE_ACCOUNT_DAYS_LEFT`.
-- * p_account_locked:               Locked/Unlocked indicator `Y` or `N`. See also `GET_ACCOUNT_LOCKED_STATUS`.
-- * p_failed_access_attempts:       Counter for consecutive login failures.
-- * p_change_password_on_first_use: Setting to force password change on first use  `Y` or `N`.
-- * p_first_password_use_occurred:  Indicates whether login with password occurred `Y` or `N`.
--
-- Example:
-- The following example shows how to use the `FETCH_USER` procedure.
-- This procedure is passed the ID of the currently authenticated user for the only IN parameter `p_user_id`.
-- The code then stores all the other OUT parameter values in local variables.
--
--     declare
--         l_workspace                     varchar2(255);
--         l_user_name                     varchar2(100);
--         l_first_name                    varchar2(255);
--         l_last_name                     varchar2(255);
--         l_web_password                  varchar2(255);
--         l_email_address                 varchar2(240);
--         l_start_date                    date;
--         l_end_date                      date;
--         l_employee_id                   number(15,0);
--         l_allow_access_to_schemas       varchar2(4000);
--         l_person_type                   varchar2(1);
--         l_default_schema                varchar2(30);
--         l_groups                        varchar2(1000);
--         l_developer_role                varchar2(60);
--         l_description                   varchar2(240);
--         l_account_expiry                date;
--         l_account_locked                varchar2(1);
--         l_failed_access_attempts        number;
--         l_change_password_on_first_use  varchar2(1);
--         l_first_password_use_occurred   varchar2(1);
--     begin
--         apex_util.fetch_users(
--             p_user_id                       => apex_util.get_current_user_id,
--             p_workspace                     => l_workspace,
--             p_user_name                     => l_user_name,
--             p_first_name                    => l_first_name,
--             p_last_name                     => l_last_name,
--             p_web_password                  => l_web_password,
--             p_email_address                 => l_email_address,
--             p_start_date                    => l_start_date,
--             p_end_date                      => l_end_date,
--             p_employee_id                   => l_employee_id,
--             p_allow_access_to_schemas       => l_allow_access_to_schemas,
--             p_person_type                   => l_person_type,
--             p_default_schema                => l_default_schema,
--             p_groups                        => l_groups,
--             p_developer_role                => l_developer_role,
--             p_description                   => l_description,
--             p_account_expiry                => l_account_expiry,
--             p_account_locked                => l_account_locked,
--             p_failed_access_attempts        => l_failed_access_attempts,
--             p_change_password_on_first_use  => l_change_password_on_first_use,
--             p_first_password_use_occurred   => l_first_password_use_occurred );
--     end;
--
-- See Also:
-- CREATE_USER
-- EDIT_USER
-- FETCH_USER@emp_info
-- FETCH_USER@short
--==============================================================================
procedure fetch_user (
     p_user_id                      in number,
     p_workspace                    out varchar2,
     p_user_name                    out varchar2,
     p_first_name                   out varchar2,
     p_last_name                    out varchar2,
     p_web_password                 out varchar2,
     p_email_address                out varchar2,
     p_start_date                   out varchar2,
     p_end_date                     out varchar2,
     p_employee_id                  out varchar2,
     p_allow_access_to_schemas      out varchar2,
     p_person_type                  out varchar2,
     p_default_schema               out varchar2,
     p_groups                       out varchar2,
     p_developer_role               out varchar2,
     p_description                  out varchar2,
     p_account_expiry               out date,
     p_account_locked               out varchar2,
     p_failed_access_attempts       out number,
     p_change_password_on_first_use out varchar2,
     p_first_password_use_occurred  out varchar2 );

--==============================================================================
-- This procedure enables a user account record to be altered. To execute this procedure,
-- the current user must have administrative privileges in the workspace.
--! Edit user information to
--! wwv_flow_fnd_user, wwv_flow_fnd_group_users and wwv_flow_developers table.
--
-- Note: Currently, the following parameters are named inconsistently between the
-- `CREATE_USER`, `EDIT_USER`, and `FETCH_USER` APIs, although they all relate to
-- the `DEVELOPER_ROLE` field stored in the named user account record:
-- + `CREATE_USER` uses `p_developer_privs`.
-- + `EDIT_USER` uses `p_developer_roles`.
-- + `FETCH_USER` uses `p_developer_role`.
--
-- Parameters:
-- * p_user_id:                      Numeric primary key of the user account.
-- * p_user_name:                    Alphanumeric name used for login. See also `SET_USERNAME`.
-- * p_first_name:                   Informational. See also `SET_FIRST_NAME`.
-- * p_last_name:                    Informational. See also `SET_LAST_NAME`.
-- * p_web_password:                 Clear text password. If using this procedure to update the password
--                                   for the user, values for both `p_web_password` and `p_new_password`
--                                   must not be null and must be identical.
-- * p_new_password:                 Clear text password. If using this procedure to update the password
--                                   for the user, values for both `p_web_password` and `p_new_password`
--                                   must not be null and must be identical.
-- * p_email_address:                Informational. See also `SET_EMAIL`.
-- * p_start_date:                   Unused.
-- * p_end_date:                     Unused.
-- * p_employee_id:                  Unused.
-- * p_allow_access_to_schemas:      A list of schemas assigned to the user's workspace to which the user is restricted.
-- * p_person_type:                  Unused.
-- * p_default_schema:               A database schema assigned to the user's workspace, used by default for browsing.
-- * p_default_date_format:          Oracle Date format for user, used for now only in SQL Workshop.
-- * p_group_ids:                    Colon-separated list of numeric group IDs.
-- * p_developer_roles:               Colon separated list of developer privileges. The following are acceptable
--                                   values for this parameter:
--                                      + `NULL`: Indicates an end user (a user who can only authenticate to developed
--                                      applications).
--                                      + `CREATE:DATA_LOADER:EDIT:HELP:MONITOR:SQL`: Indicates a user with developer
--                                      privileges.
--                                      + `ADMIN:CREATE:DATA_LOADER:EDIT:HELP:MONITOR:SQL`: Indicates a user with full
--                                      workspace administrator and developer privileges.
--                                      See also `GET_USER_ROLES`.
-- * p_description:                  Informational.
-- * p_account_expiry:               The date on which the account password was last updated. See also
--                                   `EXPIRE_END_USER_ACCOUNT`, `EXPIRE_WORKSPACE_ACCOUNT`, `UNEXPIRE_END_USER_ACCOUNT`,
--                                   `UNEXPIRE_WORKSPACE_ACCOUNT`.
-- * p_account_locked:               `Y` or `N` indicating if account is locked or unlocked. See also
--                                   `LOCK_ACCOUNT`, `UNLOCK_ACCOUNT`.
-- * p_failed_access_attempts:       Number of consecutive login failures that have occurred.
-- * p_change_password_on_first_use: Setting to force password change on first use  `Y` or `N`. See also
--                                   `CHANGE_PASSWORD_ON_FIRST_USE`.
-- * p_first_password_use_occurred:  Indicates whether login with password occurred `Y` or `N`. See also
--                                   `PASSWORD_FIRST_USE_OCCURRED`.
--
-- Example:
-- The following example shows how to use the `EDIT_USER` procedure to update a user account. This
-- example shows how you can change the user 'FRANK' from a user with just developer privilege to a user
--  with workspace administrator and developer privilege. Firstly, the `FETCH_USER` procedure is called
-- to assign account details for the user 'FRANK' to local variables. These variables are then used in the call
-- to `EDIT_USER` to preserve the details of the account, with the exception of the value for the
-- `p_developer_roles` parameter, which is set to 'ADMIN:CREATE:DATA_LOADER:EDIT:HELP:MONITOR:SQL'.
--
--     declare
--         l_user_id                       number;
--         l_workspace                     varchar2(255);
--         l_user_name                     varchar2(100);
--         l_first_name                    varchar2(255);
--         l_last_name                     varchar2(255);
--         l_web_password                  varchar2(255);
--         l_email_address                 varchar2(240);
--         l_start_date                    date;
--         l_end_date                      date;
--         l_employee_id                   number(15,0);
--         l_allow_access_to_schemas       varchar2(4000);
--         l_person_type                   varchar2(1);
--         l_default_schema                varchar2(30);
--         l_groups                        varchar2(1000);
--         l_developer_role                varchar2(60);
--         l_description                   varchar2(240);
--         l_account_expiry                date;
--         l_account_locked                varchar2(1);
--         l_failed_access_attempts        number;
--         l_change_password_on_first_use  varchar2(1);
--         l_first_password_use_occurred   varchar2(1);
--
--     begin
--         l_user_id := apex_util.get_user_id('FRANK');
--
--     apex_util.fetch_user(
--         p_user_id                       => l_user_id,
--         p_workspace                     => l_workspace,
--         p_user_name                     => l_user_name,
--         p_first_name                    => l_first_name,
--         p_last_name                     => l_last_name,
--         p_web_password                  => l_web_password,
--         p_email_address                 => l_email_address,
--         p_start_date                    => l_start_date,
--         p_end_date                      => l_end_date,
--         p_employee_id                   => l_employee_id,
--         p_allow_access_to_schemas       => l_allow_access_to_schemas,
--         p_person_type                   => l_person_type,
--         p_default_schema                => l_default_schema,
--         p_groups                        => l_groups,
--         p_developer_role                => l_developer_role,
--         p_description                   => l_description,
--         p_account_expiry                => l_account_expiry,
--         p_account_locked                => l_account_locked,
--         p_failed_access_attempts        => l_failed_access_attempts,
--         p_change_password_on_first_use  => l_change_password_on_first_use,
--         p_first_password_use_occurred   => l_first_password_use_occurred);
--
--     apex_util.edit_user (
--         p_user_id                       => l_user_id,
--         p_user_name                     => l_user_name,
--         p_first_name                    => l_first_name,
--         p_last_name                     => l_last_name,
--         p_web_password                  => l_web_password,
--         p_new_password                  => l_web_password,
--         p_email_address                 => l_email_address,
--         p_start_date                    => l_start_date,
--         p_end_date                      => l_end_date,
--         p_employee_id                   => l_employee_id,
--         p_allow_access_to_schemas       => l_allow_access_to_schemas,
--         p_person_type                   => l_person_type,
--         p_default_schema                => l_default_schema,
--         p_group_ids                     => l_groups,
--         p_developer_roles               => 'ADMIN:CREATE:DATA_LOADER:EDIT:HELP:MONITOR:SQL',
--         p_description                   => l_description,
--         p_account_expiry                => l_account_expiry,
--         p_account_locked                => l_account_locked,
--         p_failed_access_attempts        => l_failed_access_attempts,
--         p_change_password_on_first_use  => l_change_password_on_first_use,
--         p_first_password_use_occurred   => l_first_password_use_occurred);
--     end;
--
-- See Also:
-- FETCH_USER@account_info
--==============================================================================
procedure edit_user (
     p_user_id                      in number,
     p_user_name                    in varchar2,
     p_first_name                   in varchar2 default null,
     p_last_name                    in varchar2 default null,
     p_web_password                 in varchar2 default null,
     p_new_password                 in varchar2 default null,
     p_email_address                in varchar2 default null,
     p_start_date                   in varchar2 default null,
     p_end_date                     in varchar2 default null,
     p_employee_id                  in varchar2 default null,
     p_allow_access_to_schemas      in varchar2 default null,
     p_person_type                  in varchar2 default null,
     p_default_schema               in varchar2 default null,
     p_default_date_format          in varchar2 default null,
     p_group_ids                    in varchar2 default null,
     p_developer_roles              in varchar2 default null,
     p_description                  in varchar2 default null,
     p_account_expiry               in date     default null,
     p_account_locked               in varchar2 default 'N',
     p_failed_access_attempts       in number   default 0,
     p_change_password_on_first_use in varchar2 default 'Y',
     p_first_password_use_occurred  in varchar2 default 'N' );

--==============================================================================
-- This procedure sets a user account status to locked. It must be run by an authenticated
-- workspace administrator in the context of a page request.
--
-- Parameters:
-- * p_user_name: The user name of the user account.
--
-- Example:
-- The following example locks an Oracle APEX account (workspace administrator, developer, or end user)
-- in the current workspace. This action locks the account for use by administrators, developers, and end users.
--
--     begin
--         for c1 in (select user_name from apex_workspace_apex_users) loop
--             apex_util.lock_account(p_user_name => c1.user_name);
--             htp.p('User Account:'||c1.user_name||' is now locked.');
--         end loop;
--     end;
--
-- See Also:
-- UNLOCK_ACCOUNT
-- GET_ACCOUNT_LOCKED_STATUS
--==============================================================================
procedure lock_account (
     p_user_name     in varchar2 );

--==============================================================================
-- This procedure sets a user account status to unlocked. It must be run by an authenticated
-- workspace administrator in the context of a page request.
--
-- Parameters:
-- * p_user_name: The user name of the user account.
--
-- Example:
-- The following example shows how to use the `UNLOCK_ACCOUNT` procedure. Use this procedure to unlock
-- an Oracle APEX account in the current workspace. This action unlocks the account for use by
-- administrators, developers, and end users. This procedure must be run by a user who has administration
-- privileges in the current workspace.
--
--     begin
--         for c1 in (select user_name from apex_workspace_apex_users) loop
--             apex_util.unlock_account(p_user_name => c1.user_name);
--             htp.p('User Account:'||c1.user_name||' is now unlocked.');
--         end loop;
--     end;
--
-- See Also:
-- LOCK_ACCOUNT
-- GET_ACCOUNT_LOCKED_STATUS
--==============================================================================
procedure unlock_account(
     p_user_name     in varchar2 );

--==============================================================================
-- This function returns `TRUE` if the account is locked and `FALSE` if the account is
-- unlocked. It must be run by an authenticated workspace administrator in a page
-- request context.
--
-- Parameters:
-- * p_user_name: The user name of the user account.
--
-- Example:
-- The following example checks if an Oracle APEX user account (workspace administrator,
-- developer, or end user) in the current workspace is locked.
--
--     begin
--         for c1 in (select user_name from apex_workspace_apex_users) loop
--             if apex_util.get_account_locked_status(p_user_name => c1.user_name) then
--                 htp.p('User Account:'||c1.user_name||' is locked.');
--             end if;
--         end loop;
--     end;
--
-- See Also:
-- LOCK_ACCOUNT
-- UNLOCK_ACCOUNT
--==============================================================================
function get_account_locked_status (
    p_user_name in varchar2 )
    return boolean;

--==============================================================================
-- This procedure expires developer or workspace administrator login accounts. It must
-- be run by an authenticated workspace administrator in a page request context.
--
-- Parameters:
-- * p_user_name: The user name of the user account.
--
-- Example:
-- The following example shows how to use the `EXPIRE_WORKSPACE_ACCOUNT` procedure. Use this procedure
-- to expire an Oracle APEX account (workspace administrator, developer, or end user) in the current
-- workspace. This action specifically expires the account for its use by developers or administrators
-- to log in to a workspace, but it may also expire the account for its use by end users to authenticate
-- to developed applications.
--
--     begin
--         for c1 in (select user_name from apex_workspace_apex_users) loop
--             apex_util.expire_workspace_account(p_user_name => c1.user_name);
--             htp.p('Workspace account:'||c1.user_name||' is now expired.');
--         end loop;
--     end;
--
-- See Also:
-- UNEXPIRE_WORKSPACE_ACCOUNT
--==============================================================================
procedure expire_workspace_account (
    p_user_name in varchar2 );

--==============================================================================
-- This procedure unexpires developer and workspace administrator accounts and the associated passwords,
-- enabling the developer or administrator to log into a workspace. This procedure must be run by a user
-- having administration privileges in the current workspace.
--
-- Parameters:
-- * p_user_name: The user name of the user account.
--
-- Example:
-- The following example shows how to use the `UNEXPIRE_WORKSPACE_ACCOUNT` procedure. Use this procedure
-- to renew (unexpire) an Oracle APEX workspace administrator account in the current
-- workspace. This action specifically renews the account for its use by developers or administrators
-- to log in to a workspace, but it may also renew the account for its use by end users to authenticate
-- to developed applications.
--
--     begin
--         for c1 in (select user_name from apex_workspace_apex_users) loop
--             apex_util.unexpire_workspace_account(p_user_name => c1.user_name);
--             htp.p('Workspace account:'||c1.user_name||' is now valid.');
--         end loop;
--     end;
--
-- See Also:
-- EXPIRE_WORKSPACE_ACCOUNT
--==============================================================================
procedure unexpire_workspace_account(
    p_user_name in varchar2 );

--==============================================================================
-- This procedure expires the login account for use as a workspace end user. Must be run by an authenticated
-- workspace administrator in a page request context.
--
-- Parameters:
-- * p_user_name: The user name of the user account.
--
-- Example:
-- The following example shows how to use the `EXPIRE_END_USER_ACCOUNT` procedure. Use this procedure
-- to expire an Oracle APEX account (workspace administrator, developer, or end user) in the current
-- workspace. This action specifically expires the account for its use by end users to authenticate
-- to developed applications, but it may also expire the account for its use by developers or administrators
-- to log in to a workspace.
--
-- Note that this procedure must be run by a user having administration privileges in the current workspace.
--
--     begin
--         for c1 in (select user_name from apex_workspace_apex_users) loop
--             apex_util.expire_end_user_account(p_user_name => c1.user_name);
--             htp.p('End User account:'||c1.user_name||' is now expired.');
--         end loop;
--     end;
--
-- See Also:
-- UNEXPIRE_END_USER_ACCOUNT
--==============================================================================
procedure expire_end_user_account(
    p_user_name in varchar2 );

--==============================================================================
-- This procedure makes expired end users accounts and the associated passwords usable, enabling an
-- end user to log into developed applications.
--
-- Parameters:
-- * p_user_name: The user name of the user account.
--
-- Example:
-- The following example renews (unexpires) an APEX end user account in the current workspace. This
-- action specifically renews the account for use by end users to authenticate to developed applications
-- and may also renew the account for use by developers or administrators to log into a workspace.
--
-- Note that this procedure must be run by a user having administration privileges in the current workspace.
--
--     begin
--         for c1 in (select user_name from apex_workspace_apex_users) loop
--             apex_util.unexpire_end_user_account(p_user_name => c1.user_name);
--             htp.p('End User account:'||c1.user_name||' is now valid.');
--         end loop;
--     end;
--
-- See Also:
-- EXPIRE_END_USER_ACCOUNT
--==============================================================================
procedure unexpire_end_user_account(
    p_user_name in varchar2 );

--==============================================================================
-- This function returns the number of days remaining before the developer or workspace
-- administrator account password expires. Any authenticated user can run this function
-- in a page request context.
--
-- Parameters:
-- * p_user_name: The user name of the user account.
--
-- Example:
-- The following example finds the number of days remaining before an Oracle APEX
-- administrator or developer account in the current workspace expires.
--
--     declare
--         l_days_left number;
--     begin
--         for c1 in (select user_name from apex_workspace_apex_users) loop
--             l_days_left := apex_util.workspace_account_days_left(p_user_name => c1.user_name);
--             htp.p('Workspace Account:'||c1.user_name||' expires in '||l_days_left||' days.');
--         end loop;
--     end;
--
-- See Also:
-- EXPIRE_WORKSPACE_ACCOUNT
-- UNEXPIRE_WORKSPACE_ACCOUNT
--==============================================================================
function workspace_account_days_left(
    p_user_name in varchar2)
    return number;

--==============================================================================
-- This function returns the number of days remaining before an end user account password
-- expires. This function may be run in a page request context by any authenticated user.
--
-- Parameters:
-- * p_user_name: The user name of the user account.
--
-- Example:
-- The following example determines the number of days remaining before an Oracle APEX
-- end user account in the current workspace expires.
--
--     declare
--         l_days_left number;
--     begin
--         for c1 in (select user_name from apex_workspace_apex_users) loop
--             l_days_left := apex_util.end_user_account_days_left(p_user_name => c1.user_name);
--             htp.p('End User Account:'||c1.user_name||' expires in '||l_days_left||' days.');
--         end loop;
--     end;
--
-- See Also:
-- EXPIRE_END_USER_ACCOUNT
-- UNEXPIRE_END_USER_ACCOUNT
--==============================================================================
function end_user_account_days_left(
    p_user_name in varchar2)
    return number;

--==============================================================================
-- This procedure can be called from an application's custom authentication function
-- (that is, credentials verification function). The status passed to this procedure
-- is logged in the Login Access Log.
--
-- Parameters:
-- * p_status: Any text the developer chooses to denote the result of the authentication
--             attempt (up to 4000 characters).
--
-- Example:
-- One way to use the `SET_CUSTOM_AUTH_STATUS` procedure is to include it in the application
-- authentication scheme. This example demonstrates how text and numeric status values can be
-- registered for logging. Note that no credentials verification is performed. The status set
-- using this procedure is visible in the `apex_user_access_log` view and in the reports on this
-- view available to workspace and site administrators.
--
--     create or replace function my_auth(
--         p_username in varchar2,
--         p_password in varchar2)
--     return boolean
--     is
--     begin
--         apex_util.set_custom_auth_status(p_status=>'User:'||p_username||' is back.');
--         if upper(p_username) = 'GOOD' then
--             apex_util.set_authentication_result(24567);
--             return true;
--         else
--             apex_util.set_authentication_result(-666);
--             return false;
--         end if;
--     end;
--
-- See Also:
-- SET_AUTHENTICATION_RESULT
-- GET_AUTHENTICATION_RESULT
--==============================================================================
procedure set_custom_auth_status(
    p_status in varchar2 );

--==============================================================================
-- This procedure can be called from an application's custom authentication function
-- (that is, credentials verification function). The status passed to this procedure is
-- logged in the Login Access Log.
--
-- Parameters:
-- * p_code: Any numeric value the developer chooses. After this value is set in
--           the session using this procedure, it can be retrieved using the
--           `APEX_UTIL.GET_AUTHENTICATION_RESULT` function.
--
-- Example:
-- One way to use this procedure is to include it in the application authentication scheme.
-- This example demonstrates how text and numeric status values can be registered for logging.
-- In this example, no credentials verification is performed, it just demonstrates how text and
-- numeric status values can be registered for logging. Note that the status set using this procedure
-- is visible in the `apex_user_access_log` view and in the reports on this view available to
-- workspace and site administrators.
--
--     create or replace function my_auth(
--         p_username in varchar2,
--         p_password in varchar2)
--     return boolean
--     is
--     begin
--         apex_util.set_custom_auth_status(p_status=>'User:'||p_username||' is back.');
--         if upper(p_username) = 'GOOD' then
--             apex_util.set_authentication_result(24567);
--             return true;
--         else
--             apex_util.set_authentication_result(-666);
--             return false;
--         end if;
--     end;
--
-- See Also:
-- GET_AUTHENTICATION_RESULT
-- SET_CUSTOM_AUTH_STATUS
--==============================================================================
procedure set_authentication_result(
    p_code in number );

--==============================================================================
-- This function enables a developer to check whether this property is enabled or disabled
-- for an end user account.
--
-- Returns:
-- This function returns `TRUE` if the account password must be
-- changed upon first use (after successful authentication) after the password is initially set
-- and after it is changed on the Administration Service, Edit User page.
-- This function returns `FALSE` if the account does not have this property.
--
-- This function may be run in a page request context by any authenticated user.
--
-- Parameters:
-- * p_user_name: The user name of the user account.
--
-- Example:
-- The following example demonstrates how to use the `CHANGE_PASSWORD_ON_FIRST_USE` function.
-- Use this function to check if the password of an APEX user account (workspace administrator, developer,
-- or end user) in the current workspace must be changed by the user the first time it is used.
--
--     begin
--         for c1 in (select user_name from apex_workspace_apex_users) loop
--             if apex_util.change_password_on_first_use(p_user_name => c1.user_name) then
--                 htp.p('User:'||c1.user_name||' requires password to be changed the first time it is used.');
--             end if;
--         end loop;
--     end;
--
-- See Also:
-- PASSWORD_FIRST_USE_OCCURRED
--==============================================================================
function change_password_on_first_use(
    p_user_name in varchar2 )
    return boolean;

--==============================================================================
-- This function returns `TRUE` if the account's password has changed since the account was
-- created, an Oracle APEX administrator performs a password reset operation that results in
-- a new password being emailed to the account holder, or a user has initiated a password reset operation.
-- This function returns `FALSE` if the account's password has not been changed since either of the events
-- just described.
-- This function may be run in a page request context by any authenticated user.
--
-- Parameters:
-- * p_user_name: The user name of the user account.
--
-- Example:
-- The following example demonstrates how to check if the password for an APEX user account (workspace
-- administrator, developer, or end user) in the current workspace has been changed by the user the
-- first time the user logged in after the password was initially set during account creation, or
-- was changed by one of the password reset operations described above. This is meaningful only with
-- accounts for which the `CHANGE_PASSWORD_ON_FIRST_USE` attribute is set to 'Yes'.
--
--     begin
--         for c1 in (select user_name from apex_workspace_apex_users) loop
--             if apex_util.password_first_use_occurred(p_user_name => c1.user_name) then
--                 htp.p('User:'||c1.user_name||' has logged in and updated the password.');
--             end if;
--         end loop;
--     end;
--
-- See Also:
-- CHANGE_PASSWORD_ON_FIRST_USE
--==============================================================================
function password_first_use_occurred(
    p_user_name in varchar2 )
    return boolean;

--==============================================================================
-- Use this function to retrieve the authentication result of the current session.
-- Any authenticated user can call this function in a page request context.
--
-- Example:
-- The following example demonstrates how to use the post-authentication process of an
-- application's authentication scheme to retrieve the authentication result code set
-- during authentication.
--
--     apex_util.set_session_state ('MY_AUTH_STATUS',
--     'Authentication result:'||apex_util.get_authentication_result);
--
-- See Also:
-- SET_AUTHENTICATION_RESULT
-- SET_CUSTOM_AUTH_STATUS
--==============================================================================
function get_authentication_result
    return number;

--==============================================================================
-- The `PREPARE_URL` function serves two purposes:
-- 1. To return an APEX navigation URL with the Session State Protection checksum argument
--    (&cs=) if one is required. For security, the URL will not contain a checksum if the specified
--    application is located in a different workspace.
-- 2. To return an APEX navigation URL with the session ID component replaced with zero (0)
--    if the zero session ID feature is in use and other criteria are met.
--
--! If URL is f?p format, do escape_url on the argument values only.
--
-- Note: Oracle recommends using `APEX_PAGE.GET_URL` instead of `PREPARE_URL` for improved readability.
--
-- Parameters:
-- * p_url:                An APEX navigation URL with all substitutions, e.g., &ITEM_NAME., resolved.
-- * p_url_charset:        The character set name (for example, `UTF-8`) to use when escaping special
--                         characters contained within argument values.
-- * p_checksum_type:      Null or any of the following values:
--                         + `PUBLIC_BOOKMARK` or `1` - Use this when generating links to be used by
--                           any user.
--                           For example, use this value when generating an email which includes links
--                           to an application.
--                         + `PRIVATE_BOOKMARK` or `2` - Use this when generating a link to be used
--                           outside of the current session. This option can only be used by the same
--                           currently authenticated user.
--                         + `SESSION` or `3` - Use this when generating links to an application. This
--                           option can only be used within the current session.
-- * p_triggering_element: A jQuery selector (for example, `#my_button` , where `my_button` is the static
--                         ID for a button element), to identify which element to use to trigger the dialog.
--                         This is required for Modal Dialog support.
-- * p_dialog:             TBD
-- * p_plain_url:          If the page you are calling `APEX_UTIL.PREPARE_URL` from is a modal dialog,
--                         specify `p_plain_url` to omit the unnecessary JavaScript code in the generated link.
--                         By default, if this function is called from a modal dialog, JavaScript code to close
--                         the modal dialog is included in the generated URL.
--
-- Example:
-- The following example shows how to use the `PREPARE_URL` function to return a URL with a valid 'SESSION'
-- level checksum argument. This URL sets the value of `P1_ITEM` page item to 'xyz'.
--
--     declare
--         l_url varchar2(2000);
--         l_app number := v('APP_ID');
--         l_session number := v('APP_SESSION');
--     begin
--         l_url := apex_util.prepare_url(
--             p_url => 'f?p=' || l_app || ':1:'||l_session||'::NO::P1_ITEM:xyz',
--             p_checksum_type => 'SESSION');
--     end;
--
-- Example:
-- The following example shows how to use the `PREPARE_URL` function to return a URL with a zero session
-- ID. In a PL/SQL Dynamic Content region that generates f?p URLs (anchors), call `PREPARE_URL` to ensure
-- that the session ID is set to zero when the zero session ID feature is in use, when the user is a
-- public user (not authenticated), and when the target page is a public page in the current application:
--
--     htp.p(apex_util.prepare_url(p_url => 'f?p=' || :APP_ID || ':10:'|| :APP_SESSION ||'::NO::P10_ITEM:ABC'));
--
-- When using `PREPARE_URL` for this purpose, the `p_url_charset` and `p_checksum_type` arguments can be omitted.
-- However, it is permissible to use them when both the Session State Protection and Zero Session ID features are applicable.
--
-- Also see:
-- APEX_PAGE.GET_URL
--==============================================================================
function prepare_url(
    p_url                    in varchar2,
    p_url_charset            in varchar2 default null,
    p_checksum_type          in varchar2 default null,
    p_triggering_element     in varchar2 default 'this',
    p_dialog                 in varchar2 default 'null',
    p_plain_url              in boolean  default false )
    return varchar2;

--==============================================================================
-- This function sets a package variable (`apex_utilities.g_val_num`) so that it can be retrieved
-- using the function `APEX_UTIL.KEYVAL_NUM`.
--
-- Parameters:
-- * p_val: The numeric value to be saved.
--
-- Example:
-- The following example shows how to use the `SAVEKEY_NUM` function to set the `apex_utilities.g_val_num`
-- package variable to the value of 10.
--
--     declare
--         val number;
--     begin
--         val := apex_util.savekey_num(p_val => 10);
--     end;
--
-- See Also:
-- KEYVAL_NUM
--==============================================================================
function savekey_num (
    p_val in number )
    return number;

--==============================================================================
-- This function gets the value of the package variable `apex_utilities.g_val_num` set by
-- `APEX_UTIL.SAVEKEY_NUM`.
--
-- Example:
-- The following example shows how to use the `KEYVAL_NUM` function to return the current value of the
-- `apex_utilities.g_val_num` package variable.
--
--     declare
--         val number;
--     begin
--         val := apex_util.keyval_num;
--     end;
--
-- See Also:
-- SAVEKEY_NUM
--==============================================================================
function keyval_num
    return number;

--==============================================================================
-- This function sets a package variable (`apex_utilities.g_val_vc2`) so that it can be retrieved
-- using the function `APEX_UTIL.KEYVAL_VC2`.
--
-- Parameters:
-- * p_val: The VARCHAR2 value to be saved.
--
-- Example:
-- The following example shows how to use the `SAVEKEY_VC2` function to set the `apex_utilities.g_val_vc2`
-- package variable to the value of 'XXX'.
--
--     declare
--         val number;
--     begin
--         val := apex_util.savekey_vc2(p_val => 'XXX');
--     end;
--
-- See Also:
-- KEYVAL_VC2
--==============================================================================
function savekey_vc2 (
    p_val in varchar2 )
    return varchar2;

--==============================================================================
-- This function gets the value of the package variable `apex_utilities.g_val_vc2`
-- set by `APEX_UTIL.SAVEKEY_VC2`.
--
-- Example:
--
-- The following example shows how to use the `KEYVAL_VC2` function to return the current value of
-- the package variable `apex_utilities.g_val_vc2`.
--
--     declare
--         val varchar2(4000);
--     begin
--         val := apex_util.keyval_vc2;
--     end;
--
-- See Also:
-- SAVEKEY_VC2
--==============================================================================
function keyval_vc2
    return varchar2;

--==============================================================================
-- Needs Review: Mark internal or add to public documentation?
--==============================================================================
procedure flash2 (
    p                    in varchar2,
    preloaderLoadingText in varchar2 default null,
    swffile              in varchar2 default null,
    preloaderInitText    in varchar2 default null,
    XMLCallDate          in varchar2 default null );

--##############################################################################
--#
--# CALENDAR
--#
--##############################################################################

--==============================================================================
-- Use this procedure to navigate to the next set of days in the calendar. Depending on what the
-- calendar view is, this procedure navigates to the next month, week or day. If it is a Custom
-- Calendar, the total number of days between the start date and end date are navigated.
--
-- Deprecated: 24.2
--
-- Example:
-- In this example, the following code navigates the calendar if you create a button called 'NEXT'
-- on the Calendar page and a process that fires when the button is clicked.
--
--     apex_util.increment_calendar;
--==============================================================================
procedure increment_calendar;

--==============================================================================
-- Deprecated: 24.2
-- Internal:
--==============================================================================
procedure decrement_calendar;

--==============================================================================
-- Deprecated: 24.2
-- Internal:
--==============================================================================
procedure today_calendar;

--==============================================================================
-- Use this procedure to change the existing calendar view to Custom Calendar.
--
-- Deprecated: 24.2
--
-- Parameters:
-- * p_date_type_field: Identifies the item name used to define the type of calendar to be displayed.
--
-- Example:
-- The following example defines a custom calendar based on the hidden calendar type field. Assuming
-- the Calendar is created on Page 9, the following example hides the column called `P9_CALENDAR_TYPE`.
--
--     apex_util.custom_calendar(
--       'P9_CALENDAR_TYPE');
--
--==============================================================================
procedure custom_calendar(p_date_type_field in varchar2 default null );

--==============================================================================
-- Deprecated: 24.2
-- Internal:
--==============================================================================
procedure weekly_calendar(p_date_type_field varchar2 default null);

--==============================================================================
-- Deprecated: 24.2
-- Internal:
--==============================================================================
procedure daily_calendar(p_date_type_field varchar2 default null);

--==============================================================================
-- Deprecated: 24.2
-- Internal:
--==============================================================================
procedure agenda_calendar(p_date_type_field varchar2 default null);

--==============================================================================
-- Deprecated: 24.2
-- Internal:
--==============================================================================
procedure month_calendar(p_date_type_field varchar2 default null);

--##############################################################################
--#
--# PAGE AND REGION CACHING APIS
--#
--##############################################################################

--==============================================================================
-- This procedure purges all cached pages and regions for a given application.
--
-- Parameters:
-- * p_application: The identification number (ID) of the application.
--
-- Example:
-- The following example demonstrates how to use the `CACHE_PURGE_BY_APPLICATION` procedure to purge
-- all the cached pages and regions for the application currently executing.
--
--     begin
--         apex_util.cache_purge_by_application(p_application => :APP_ID);
--     end;
--
-- See Also:
-- CACHE_PURGE_BY_PAGE
--==============================================================================
procedure cache_purge_by_application (
   p_application    in number);

--==============================================================================
-- This procedure purges the cache for a given application and page. If the page itself is not cached
-- but contains one or more cached regions, then the cache for these is also purged.
--
-- Parameters:
-- * p_application: The identification number (ID) of the application.
-- * p_page:        The page number (ID).
-- * p_user_name:   The user associated with cached pages and regions. If `p_user_name is supplied`,
--                  only that user's cached pages and regions will be purged (only relevant if the cache
--                  is set to be by user).
--
-- Example:
-- The following example demonstrates how to use the `CACHE_PURGE_BY_PAGE` procedure to purge the cache
-- for page 9 of the application currently executing. Additionally, if the `p_user_name parameter` is supplied,
-- this procedure would be further restricted by a specific user's cache (only relevant if the cache is set to be
-- by user).
--
--     begin
--         apex_util.cache_purge_by_page(
--             p_application => :APP_ID,
--             p_page => 9);
--     end;
--==============================================================================
procedure cache_purge_by_page (
   p_application  in number,
   p_page         in number,
   p_user_name    in varchar2 default null);

--==============================================================================
-- This procedure deletes all cached pages and regions for the specified application
-- that have passed the defined active time period. When you cache a page or region, you specify an
-- active time period (or Cache Timeout). Once that period has passed, the cache is no longer used,
-- thus removing those unusable pages or regions from the cache.
--
-- Parameters:
-- * p_application: The identification number (ID) of the application.
--
-- Example:
-- The following example demonstrates how to use the `CACHE_PURGE_STALE` procedure to purge all the
-- stale pages and regions in the application currently executing.
--
--     begin
--         apex_util.cache_purge_stale(p_application => :APP_ID);
--     end;
--==============================================================================
procedure cache_purge_stale (
   p_application    in number);

--==============================================================================
-- This function returns the date and time a specified application page was cached either for the user
-- issuing the call, or for all users if the page was not set to be cached by user.
--
-- Parameters:
-- * p_application: The identification number (ID) of the application.
-- * p_page:        The page number (ID).
--
-- Example:
-- The following example demonstrates how to use the `CACHE_GET_DATE_OF_PAGE_CACHE` function to retrieve
-- the cache date and time for page 9 of the currently executing application. If page 9 has been cached,
-- the cache date and time is output using the HTP package. The page could have been cached either
-- by the user issuing the call, or for all users if the page was not to be cached by the user.
--
--     declare
--         l_cache_date date default null;
--     begin
--         l_cache_date := apex_util.cache_get_date_of_page_cache(
--                             p_application => :APP_ID,
--                             p_page        => 9);
--         if l_cache_date is not null then
--             htp.p('Cached on ' || to_char(l_cache_date, 'DD-MON-YY HH24:MI:SS'));
--         end if;
--     end;
--==============================================================================
function cache_get_date_of_page_cache (
   p_application  in number,
   p_page         in number)
   return date;

--==============================================================================
-- This procedure deletes all cached regions for a given application.
--
-- Parameters:
-- * p_application: The identification number (ID) of the application.
--
-- Example:
-- The following example shows how to use `APEX_UTIL.PURGE_REGIONS_BY_APP` to delete all cached regions
-- for application #123.
--
--     begin
--         apex_util.purge_regions_by_app(p_application=> 123);
--     end;
--==============================================================================
procedure purge_regions_by_app (
    p_application in number);

--==============================================================================
-- Deletes all cached values for a region identified by the application ID, page number and region name.
--
-- Parameters:
-- * p_application: The identification number (ID) of the application.
-- * p_page:        The number of the page containing the region to be deleted.
-- * p_region_name: The region name to be deleted.
--
-- Example:
-- The following example shows how to use the `PURGE_REGIONS_BY_NAME` procedure to delete all the
-- cached values for the region 'my_cached_region' on page 1 of the current application.
--
--     begin
--         apex_util.purge_regions_by_name(
--             p_application => :APP_ID,
--             p_page        => 1,
--             p_region_name => 'my_cached_region');
--     end;
--==============================================================================
procedure purge_regions_by_name (
    p_application  in number,
    p_page         in number,
    p_region_name  in varchar2);

--==============================================================================
-- This procedure deletes all cached regions by application and page.
--
-- Parameters:
-- * p_application: The identification number (ID) of the application.
-- * p_page:        The number of the page containing the regions.
--
-- Example:
-- The following example shows how to use the `PURGE_REGIONS_BY_PAGE` procedure to delete all the
-- cached values for regions on page 1 of the current application.
--
--     begin
--         apex_util.purge_regions_by_page(
--             p_application => :APP_ID,
--             p_page        => 1);
--     end;
--==============================================================================
procedure purge_regions_by_page (
    p_application  in number,
    p_page         in number);

--==============================================================================
-- This function returns the date and time a specified region was cached either for the user
-- issuing the call, or for all users if the page was not set to be cached by user.
--
-- Parameters:
-- * p_application: The identification number (ID) of the application.
-- * p_page:        The page number (ID).
-- * p_region_name: The region name.
--
-- Example:
-- The following example demonstrates how to use the `CACHE_GET_DATE_OF_REGION_CACHE` function to retrieve
-- the cache date and time for the region named 'Cached Region' on page 13 of the currently executing
-- application.
-- If the region has been cached, the cache date and time is output using the HTP package. The region
-- could have been cached either by the user issuing the call, or for all users if the page was not
-- to be cached by user.
--
--     declare
--         l_cache_date date default null;
--     begin
--         l_cache_date := apex_util.cache_get_date_of_region_cache(
--                             p_application => :APP_ID,
--                             p_page        => 13,
--                             p_region_name => 'My Cached Region' );
--         if l_cache_date is not null then
--             htp.p('Cached on ' || to_char(l_cache_date, 'DD-MON-YY HH24:MI:SS'));
--         end if;
--     end;
--==============================================================================
function cache_get_date_of_region_cache (
   p_application  in number,
   p_page         in number,
   p_region_name  in varchar2)
   return date;

--##############################################################################
--#
--# EXPORT
--#
--##############################################################################


--==============================================================================
-- Deprecated:
-- Internal:
--==============================================================================
procedure export_application (
    p_workspace_id            in number,
    p_application_id          in number);

--==============================================================================
-- Deprecated:
-- Internal:
--==============================================================================
procedure export_application_page (
    p_workspace_id            in number,
    p_application_id          in number,
    p_page_id                 in number);

--==============================================================================
-- Deprecated:
-- Internal:
--==============================================================================
procedure export_application_component (
    p_workspace_id            in number,
    p_application_id          in number,
    p_component_id            in number,
    p_component_type          in varchar2);

--==============================================================================
--  Needs Review: Mark internal or add to public documentation?
--  For use from SQL prompt
--
-- Example:
--      begin
--          dbms_output.put_line(apex_util.minimum_free_application_id);
--      end;
--==============================================================================
function minimum_free_application_id
    return number;

--==============================================================================
-- Deprecated:
-- Internal:
--==============================================================================
function get_application_id_status (
    p_workspace_id           in  number,
    p_application_id         in  number)
    return number;

--==============================================================================
-- Deprecated:
-- Internal:
--==============================================================================
function get_application_name (
    p_workspace_id           in  number,
    p_application_id         in  number)
    return varchar2;

--##############################################################################
--#
--# JSON
--#
--##############################################################################

--==============================================================================
-- Needs Review: Mark internal or add to public documentation?
--==============================================================================
procedure json_from_sql (
    sqlq             in varchar2 default null,
    p_sub            in varchar2 default 'N',
    p_owner          in varchar2 default null,
    p_optimizer_hint in varchar2 default null );

--==============================================================================
 -- Needs Review: Mark internal or add to public documentation?
--==============================================================================
procedure json_from_array (
    p_rows   in number   default null,
    p_cols   in number   default null,
    p_name01 in varchar2 default null,
    p_name02 in varchar2 default null,
    p_name03 in varchar2 default null,
    p_name04 in varchar2 default null,
    p_name05 in varchar2 default null,
    p_name06 in varchar2 default null,
    p_name07 in varchar2 default null,
    p_name08 in varchar2 default null,
    p_name09 in varchar2 default null,
    p_name10 in varchar2 default null,
    p_f01    in wwv_flow_global.vc_arr2 default wwv_flow.empty_vc_arr,
    p_f02    in wwv_flow_global.vc_arr2 default wwv_flow.empty_vc_arr,
    p_f03    in wwv_flow_global.vc_arr2 default wwv_flow.empty_vc_arr,
    p_f04    in wwv_flow_global.vc_arr2 default wwv_flow.empty_vc_arr,
    p_f05    in wwv_flow_global.vc_arr2 default wwv_flow.empty_vc_arr,
    p_f06    in wwv_flow_global.vc_arr2 default wwv_flow.empty_vc_arr,
    p_f07    in wwv_flow_global.vc_arr2 default wwv_flow.empty_vc_arr,
    p_f08    in wwv_flow_global.vc_arr2 default wwv_flow.empty_vc_arr,
    p_f09    in wwv_flow_global.vc_arr2 default wwv_flow.empty_vc_arr,
    p_f10    in wwv_flow_global.vc_arr2 default wwv_flow.empty_vc_arr,
    p_sub    in varchar2 default 'N');

--==============================================================================
-- Needs Review: Mark internal or add to public documentation?
--==============================================================================
procedure json_from_items(
    p_items      in varchar2 default null,
    p_separator  in varchar2 default ':',
    p_sub        in varchar2 default 'N');

--==============================================================================
-- Needs Review: Mark internal or add to public documentation?
--==============================================================================
procedure json_from_string(
    p_items      in varchar2 default null,
    p_separator  in varchar2 default ':',
    p_sub        in varchar2 default 'N');


--##############################################################################
--#
--# BLOB SUPPORT
--#
--##############################################################################

--==============================================================================
-- Needs Review: Mark internal or add to public documentation?
-- Automatically called from APEX form pages
-- Not designed to be called procedurally
-- Calls to this procedure can be generated by calling the apex_util.get_blob_file_src function
-- Page must have item of type FILE (FILE Browse)
-- Page item source must use the following format "DB_COLUMN:MIMETYPE_COLUMN:FILENAME_COLUMN:LAST_UPDATE_COLUMN:CHARSET_COLUMN:CONTENT_DISPOSITION:DOWNLOAD_LINK"
--    DB_COLUMN           = Required case sensitive name of a valid column which is of type BLOB
--    MIMETYPE_COLUMN     = Optional case sensitive column name of a table column used to store the mimetype
--    FILENAME_COLUMN     = Optional case sensitive column name of a table column used to store the file name
--    LAST_UPDATE_COLUMN  = Optional case sensitive column name of a table column used to store the last update date of the BLOB
--    CHARSET_COLUMN      = Optional case sensitive column name of a table column used to store the file character set
--    CONTENT_DISPOSITION = inline or attachment
--    DOWNLOAD_LINK       = Optional text to be used for the download text, defaults to Download, translated
-- Page item source must include at least the database column name and a trailing colon
-- Mimetype column is required if the mimetype is to be encoded in the download header
-- Page item must be of source type of DATABASE COLUMN
-- Page must have a DML process of type DML_PROCESS_ROW, used to determine the tablename
-- Must be called from an APEX application context
-- Invalid inputs will result in a 404 error
--==============================================================================
procedure get_blob_file(
    s                     in number,                -- APEX session ID
    a                     in number,                -- APEX application ID
    p                     in number,                -- APEX page ID of the form page
    d                     in number,                -- DML process APEX meta data ID
    i                     in number,                -- ITEM of type FILE APEX meta data ID
    p_pk1                 in varchar2,              -- Primary key value
    p_ck                  in varchar2,              -- Checksum used to prevent URL Tampering
    p_pk2                 in varchar2 default null, -- Optional Second Primary Key Value, used for compound keys
    p_mimetype            in varchar2 default null, -- Optional ...
    p_content_disposition in varchar2 default null, -- Optional use "inline" or "attachment" all other values ignored
    p_show_last_mod       in varchar2 default 'Y'); -- Optional ...

--==============================================================================
-- As an alternative to using the built-in methods of providing a download link, you can use the
-- `APEX_UTIL.GET_BLOB_FILE_SRC` function to download a BLOB column content. One advantage of this
-- approach is more specific formatting of the display of the image (with height and width tags).
-- This function must be called from a valid Oracle APEX session and also requires that the parameters
-- that describe the BLOB are listed as the format of a valid item within the application. That item
-- is then referenced by the function.
-- Note:
-- If the URL returned by this function is passed to `APEX_UTIL.PREPARE_URL`, the `p_plain_url` argument
-- must be set to `TRUE` to ensure that no modal dialog code is added when the referenced page item is
-- on a modal page.
--
-- Parameters:
-- * p_item_name:            Name of valid application page item that with type `FILE` that contains
--                           the source type of the DB column.
-- * p_v1:                   Value of primary key column.
-- * p_v2:                   Optional value of second primary key column.
-- * p_content_disposition:  Optional content disposition. Valid values are `inline` and `attachment`,
--                           all other values are ignored.
--
-- Example:
-- As a PL/SQL Function Body:
--
--     return '<img src="'||apex_util.get_blob_file_src('P2_ATTACHMENT', :P2_EMPNO)||'" />';
--
-- Example:
-- As a Region Source of type SQL. This example displays the `BLOB` within the report if it can
-- be displayed, and provides a download link if it cannot be displayed.
--
--     select id,
--            name,
--            case
--                when nvl(dbms_lob.getlength(document),0) = 0 then null
--                else case when attach_mimetype like 'image%' then '<img src="'||apex_util.get_blob_file_src('P4_DOCUMENT',id)||'" />'
--                          else '<a href="'||apex_util.get_blob_file_src('P4_DOCUMENT',id)||'">Download</a>'
--                     end
--            end new_img
--            from test_with_blob
--
-- See Also:
-- PREPARE_URL
--==============================================================================
function get_blob_file_src (
    p_item_name in varchar2 default null,
    p_v1        in varchar2 default null,
    p_v2        in varchar2 default null,
    p_content_disposition in varchar2 default null)
    return varchar2;

--==============================================================================
-- Needs Review: Mark internal or add to public documentation?
-- This procedure will download a blob column given proper inputs
-- This get_blob function is designed to be called from APEX reports automatically
-- Classic and Interactive Reports given a format mask will generate a link to this procedure
-- Report that uses  "select ... dbms_lob.getlength(myblob_column) ..." syntax
--
-- The report column format mask has the following format
-- DOWNLOAD|IMAGE:<blob_tab>:<blob_col>:<pk1_col>:<pk2_col>:<mimetype_col>:<filename_col>:<last_update_col>:<charset_col>:<content disposition>:<download text>
-- All arguments are delimited by colons
--
-- This procedure is NOT designed to be called directly, it is intended to be called by APEX reporting engines
-- Reference function apex_util.get_blob_file_src
--
-- position 1: "DOWNLOAD" or "IMAGE"
--             Download will result in the generation a "a href=" tag
--             Image will result in the generation of an inline "img src=" tag
--             Use image when your files are images
--             Using image for non image based files will result in broken image links
--             Required
-- position 2: Name of the table containing the blob column in question
--             Required and case sensitive
-- position 3: Name of the BLOB column name
--             Required and case sensitive
-- position 4: Name of the primary key column in the table identified in position 2
--             Required and case sensitive
-- position 5: Name of a secondary key column to uniquely identify the row that contains the BLOB column
--             Optional and case sensitive
-- position 6: Name of a column that is used to store the mime type that corresponds to the BLOB column
--             Managing the mimetype allows the mimetype to be encoded in the file download
--             If a mimetype is not specified download will use "application/octet-stream"
--             A proper mimetype allows the browser to pick an appropriate application to display the file
--             Optional and case sensitive
-- position 7: Name of a column that is used to store the filename of the BLOB column identified in position 3
--             Managing the filename allows downloads to default the file name to a useful value
--             Optional and case sensitive
--             Not used for IMAGE format but left in so format can easily be changed between IMAGE and DOWNLOAD
-- position 8: Name of a column that is used to store the date the BLOB was last updated
--             If used, the HTTP header of the file download will indicate the date of last modification and
--                browsers will be able to cache the BLOB.  If not specified, the browser may not be able to cache files.
--             Optional and case sensitive
-- position 9: Name of the character set that is used to store the character set of the file in the BLOB column
--             Most useful for applications that have files in multiple character sets
--             Optional and case sensitive
--             Not used for IMAGE format but left in so format can easily be changed between IMAGE and DOWNLOAD
-- position 10: For DOWNLOAD format masks, identifies the content disposition
--             Defaults to inline
--             Valid values are "inline" and "attachment"
--             A value of inline will cause the browser to render the file inline if it can
--             A value of attachment will prompt the user to download the file
--             Optional, use lower case
--             Not used for IMAGE format but left in so format can easily be changed between IMAGE and DOWNLOAD
-- position 11: For DOWNLOAD format masks, identifies the text used to indicate a download link text
--             Default to "Download"
--             Translated into 10 languages
--             Specify if you wish to over-ride the default text
--             Standard APEX substitutions are performed
--             Only used for DOWNLOAD format masks
--             Optional
--             For IMAGE, String used for the alt tag associated with the image
--
-- Example Report format masks:
--
-- DOWNLOAD:EMP:RESUME:EMPNO::MIMETYPE:FILENAME:RESUME_LAST_UPDATE
-- DOWNLOAD:EMP:RESUME:EMPNO::MIMETYPE:FILENAME:RESUME_LAST_UPDATE:::Resume
-- IMAGE:EMP:EMP_PHOTO:EMPNO::MIMETYPE::PHOTO_LAST_UPDATE:::Employee Photo
--
--==============================================================================
procedure get_blob (
    s  in number   default null, -- session id
    a  in number   default null, -- application id
    c  in number   default null, -- id of the report column
    p  in number   default null, -- page
    k1 in varchar2 default null, -- primary key 1 value
    ck in varchar2 default null, -- checksum
    rt in varchar2 default 'IR', -- optional report type IR (interactive reports) or CR (classic reports)
    k2 in varchar2 default null, -- optional, primary key 2 value
    lm in varchar2 default 'Y'   -- optional, show last modified
    );

--==============================================================================
-- Needs Review: Mark internal or add to public documentation?
-- APEX Listener procedure to set blob
--
-- returns file name
--==============================================================================
function set_blob (
   p_application_id      in varchar2 default null,
   p_page_id             in varchar2 default null,
   p_session_id          in varchar2 default null,
   p_request             in varchar2 default null,
   --
   p_item_id             in varchar2 default null,
   --
   p_file_name           in varchar2 default null,
   p_mime_type           in varchar2 default null,
   p_blob_content        in blob )
   return varchar2;

--==============================================================================
-- This function converts a CLOB to a temporary BLOB.
--
-- Parameters:
-- * p_clob:             CLOB to convert to a BLOB.
-- * p_charset:          Character set to convert the BLOB to. If omitted, no character set conversion
--                       happens.
-- * p_include_bom:      Prepend the generated BLOB with a BOM.
--
-- * p_in_memory:        Create the temporary LOB in memory, if `Y` is specified.
-- * p_free_immediately: Clean up the temporary LOB after the top-level call, if `Y` is specified.
--
-- Returns:
-- Temporary BLOB containing the CLOB contents.
--
-- Example:
-- The following example converts a CLOB to a BLOB, with and without charset conversion.
--
--     declare
--         l_clob clob;
--         l_blob blob;
--     begin
--         l_clob := to_clob( unistr( 'This is some CLOB content with umlauts: \00fc,\00e4,\00f6.' ) );
--
--         l_blob := apex_util.clob_to_blob(
--                       p_clob        => l_clob,
--                       p_charset     => 'AL32UTF8' );
--
--         sys.dbms_output.put_line( 'The utf-8 BLOB has ' || sys.dbms_lob.getlength( l_blob ) || ' bytes.' );
--
--         l_blob := apex_util.clob_to_blob(
--                       p_clob        => l_clob,
--                       p_charset     => 'WE8ISO8859P1' );
--
--         sys.dbms_output.put_line( 'The iso-8859-1 BLOB has ' || sys.dbms_lob.getlength( l_blob ) || ' bytes.' );
--     end;
--
-- See Also:
-- BLOB_TO_CLOB
--==============================================================================
function clob_to_blob(
    p_clob             in clob,
    p_charset          in varchar2 default null,
    p_include_bom      in varchar2 default 'N',
    --
    p_in_memory        in varchar2 default 'Y',
    p_free_immediately in varchar2 default 'Y' )
    return blob;

--==============================================================================
-- This function converts a BLOB to a temporary CLOB.
--
-- Parameters:
-- * p_blob:             BLOB to be converted to a CLOB.
-- * p_charset:          Character set of the BLOB to be converted. If omitted, the database character
--                       set is assumed and no character set conversion happens.
-- * p_in_memory:        Create the temporary LOB in memory, if `Y` is specified.
-- * p_free_immediately: Clean up the temporary LOB after the top-level call, if `Y` is specified.
--
-- Returns:
-- The temporary CLOB containing the BLOB contents.
--
-- Example:
-- The following example grabs website contents as BLOB and converts it to a CLOB.
--
--     declare
--         l_clob clob;
--         l_blob blob;
--     begin
--         l_blob := apex_web_service.make_rest_request_b(
--                       p_url          => 'https://www.example.com/',
--                       p_http_method  => 'GET' );
--
--         l_clob := apex_util.blob_to_clob(
--                       p_blob         => l_blob );
--
--         sys.dbms_output.put_line( 'The CLOB has ' || sys.dbms_lob.getlength( l_clob ) || ' bytes.' );
--         sys.dbms_output.put_line( '--------------------------------' );
--         sys.dbms_output.put_line( sys.dbms_lob.substr( l_clob, 80, 1 ) );
--     end;
--
-- See Also:
-- CLOB_TO_BLOB
--==============================================================================
function blob_to_clob(
    p_blob             in blob,
    p_charset          in varchar2 default null,
    --
    p_in_memory        in varchar2 default 'Y',
    p_free_immediately in varchar2 default 'Y' )
    return clob;

--##############################################################################
--#
--# INTERACTIVE REPORTS
--#
--##############################################################################

--==============================================================================
-- This procedure creates a filter on an interactive report. Only use this procedure in a page submit process.
--
-- Deprecated: 4.2.  Use `APEX_IR.ADD_FILTER` instead.
--
--
--  Parameters:
-- * p_page_id:       Page of the current Oracle APEX application that contains an interactive report.
-- * p_report_column: Name of the report SQL column (or column alias) to be filtered.
-- * p_operator_abbr: Filter type. Valid values are:
--                    + `EQ` = Equals
--                    + `NEQ` = Not Equals
--                    + `LT` = Less than
--                    + `LTE` = Less then or equal to
--                    + `GT` = Greater Than
--                    + `GTE` = Greater than or equal to
--                    + `LIKE` = SQL Like operator
--                    + `N` = Null
--                    + `NN` = Not Null
--                    + `NC` = Not Contains
--                    + `IN` = SQL In Operator
--                    + `NIN` = SQL Not in Operator
-- * p_filter_value:  Value of filter, not used for `N` and `NN`.
-- * p_report_alias:  Identifies the saved report alias within the current application page. To create a
--                    filter on a Primary report, `p_report_alias` must be `PRIMARY` or leave as `NULL`.
--                    To create a filter on a saved report, `p_report_alias` must be the name of the saved
--                    report. For example, to create a filter on report '1234', `p_report_alias` must be '1234'.
--
--
-- Example:
-- The following example shows how to use the `IR_FILTER` procedure to filter the interactive report with
-- the alias of '8101021' in page 1 of the current application, with `DEPTNO` equals 30.
--
--     begin
--         apex_util.ir_filter (
--             p_page_id        => 1,
--             p_report_column  => 'DEPTNO',
--             p_operator_abbr  => 'EQ',
--             p_filter_value   => '30',
--             p_report_alias   => '8101021'
--             );
--     end;
--
-- See Also:
-- APEX_IR.ADD_FILTER Procedure Signature 1
-- APEX_IR.ADD_FILTER Procedure Signature 2
--==============================================================================
procedure ir_filter (
    p_page_id       in number,
    p_report_column in varchar2,
    p_operator_abbr in varchar2 default null,
    p_filter_value  in varchar2,
    p_report_alias in varchar2 default null );

--==============================================================================
-- This procedure resets report settings back to the default report settings. Resetting a report
-- removes any customizations you have made.
--
-- Deprecated: 4.2. Use `APEX_IR.RESET_REPORT` instead.
--
-- Parameters:
-- * p_page_id:      Page of the current Oracle APEX application that contains an interactive report.
-- * p_report_alias: Identifies the saved report alias within the current application page. To reset a
--                   a Primary report, `p_report_alias` must be `PRIMARY` or left as `NULL`.
--                   To reset a saved report, `p_report_alias` must be the name of the saved
--                   report. For example, to reset report '1234', `p_report_alias` must be '1234'.
--
-- Example:
-- The following example shows how to use the `IR_RESET` procedure to reset the Interactive report settings
-- with the alias of '8101021' in page 1 of the current application.
--
--     begin
--         apex_util.ir_reset (
--             p_page_id        => 1,
--             p_report_alias   => '8101021'
--             );
--     end;
--
-- See Also:
-- APEX_IR.RESET_REPORT Procedure Signature 1
-- APEX_IR.RESET_REPORT Procedure Signature 2
--==============================================================================
procedure ir_reset (
    p_page_id      in number,
    p_report_alias in varchar2 default null );

--==============================================================================
-- This procedure clears any report filters including default filters. Only use
-- this procedure in a page submit process.
--
-- Deprecated: 4.2. Use `APEX_IR.CLEAR_REPORT` instead.
--
-- Parameters:
-- * p_page_id:      Page of the current Oracle APEX application that contains an interactive report.
-- * p_report_alias: Identifies the saved report alias within the current application page. To clear a
--                   Primary report, `p_report_alias` must be `PRIMARY` or left as `NULL`.
--                   To clear a saved report, `p_report_alias` must be the name of the saved
--                   report. For example, to clear report '1234', `p_report_alias` must be '1234'.
--
-- Example:
-- The following example shows how to use the `IR_CLEAR` procedure to clear the Interactive report
-- with the alias of '8101021' in page 1 of the current application.
--
--     begin
--         apex_util.ir_clear (
--             p_page_id        => 1,
--             p_report_alias   => '8101021'
--             );
--     end;
--
-- See Also:
-- APEX_IR.CLEAR_REPORT Procedure Signature 1
-- APEX_IR.CLEAR_REPORT Procedure Signature 2
--==============================================================================
procedure ir_clear (
    p_page_id      in number,
    p_report_alias in varchar2 default null );

--==============================================================================
-- This procedure deletes a saved interactive report. It deletes all saved reports except the Primary
-- Default report.
--
-- Deprecated: 4.2. Use `APEX_IR.DELETE_REPORT` instead.
--
-- Parameters:
-- * p_report_id: Report ID to delete within the current Oracle APEX application.
--
-- Example:
-- The following example shows how to use the `IR_DELETE_REPORT` procedure to delete the saved
-- Interactive Report with ID of 880629800374638220 in the current application.
--
--     begin
--         apex_util.ir_delete_report(
--             p_report_id => 880629800374638220);
--     end;
--
-- See Also:
-- APEX_IR.DELETE_REPORT
--==============================================================================
procedure ir_delete_report(
    p_report_id           in number );

--==============================================================================
-- This procedure deletes Interactive Report subscriptions.
--
-- Deprecated: 4.2. Use `APEX_IR.DELETE_SUBSCRIPTION` instead.
--
-- Parameters:
-- * p_subscription_id: Subscription ID to delete within the current workspace.
--
-- Example:
-- The following example shows how to use the `IR_DELETE_SUBSCRIPTION ` procedure to delete the
-- Interactive Report subscription with the ID of 880629800374638220 in the current workspace.
--
--     begin
--         apex_util.ir_delete_subscription(
--             p_report_id => 880629800374638220);
--     end;
--
-- See Also:
-- APEX_IR.DELETE_SUBSCRIPTION
--==============================================================================
procedure ir_delete_subscription(
    p_subscription_id           in number );

--##############################################################################
--#
--# FORMAT MASKS
--#
--! Used by interactive and classic reports to provide additional format masks
--#
--##############################################################################

--==============================================================================
-- Needs Review: Mark internal or add to public documentation?
-- Takes a file size in bytes and returns a rounded off size with trailing KB, MB, GB, or TB
-- Provides a more readable file size
-- Use FILESIZE format mask on numeric columns from within Classic and Interactive Reports
-- Null values will return null
-- KB, MB, GB, and TB are always english language strings
--
-- Example report format masks for interactive and classic reports
--     "FILESIZE"
--
-- Example SQL query syntax
--     "select apex_util.filesize_mask(8675309) from dual"
--==============================================================================
function filesize_mask (
    p_number       in number default null)
    return varchar2;

--==============================================================================
-- Use this function to scale a graph. This function can also be used by classic and interactive reports
-- with format mask of `GRAPH`. This generates a `<div>` tag with inline styles
--
-- Parameters:
-- * p_number:           A number between 0 and 100. Number outside the range 0 to 100 will result in
--                       a `NULL` return.
-- * p_size:             The width of the graph in pixels. The width of the HTML graph will default to
--                       100 px in width.
-- * p_background:       Six character hexadecimal color of chart background (not bar color).
-- * p_bar_background:   Six character hexadecimal background color of chart bar (bar color).
-- * p_format:           If this parameter is supplied, `p_size`, `p_background` and `p_bar_background`
--                       are ignored.
--                       + This parameter uses the following format:
--                       `PCT_GRAPH:<BACKGROUND>:<FOREGROUND>:<CHART_WIDTH>`
--                       + position 1: PCT_GRAPH format mask indicator.
--                       + position 2: Background color in hexadecimal, 6 characters (optional).
--                       + position 3: Foreground "bar" color in hexadecimal, 6 characters (optional).
--                       + position 4: Chart width in Pixels. Numeric and defaults to 100.
--                       + `p_number`is automatically scaled, so 50 would be half of `chart_width`.
-- * p_aria_labelledby:  Space separated list of one or more ID values referencing the elements that label the percent graph.
-- * p_aria_label:       Value that labels the percent graph.
--
-- Example:
-- The following is a SQL example:
--
--     select apex_util.html_pct_graph_mask(33) from dual;
--
-- The following is a report numeric column format mask example:
--
--     PCT_GRAPH:777777:111111:200
--==============================================================================
function html_pct_graph_mask (
    p_number            in number   default null,
    p_size              in number   default 100,
    p_background        in varchar2 default null,
    p_bar_background    in varchar2 default null,
    p_format            in varchar2 default null,
    p_aria_labelledby   in varchar2 default null,
    p_aria_label        in varchar2 default null)
    return varchar2;


--##############################################################################
--#
--# PASSWORD CHECKING
--#
--##############################################################################

--==============================================================================
-- This procedure returns Boolean `OUT` values based on whether a proposed password meets the password
-- strength requirements as defined by the Oracle APEX site administrator.
--
-- Parameters:
-- * p_username:                    The username that identifies the account in the current workspace.
-- * p_password:                    The password to be checked against the password strength rules.
-- * p_old_password:                The current password for the account. Used only to enforce the 'new
--                                  password must differ from old' rule.
-- * p_workspace_name:              The current workspace name, used only to enforce the 'password must not
--                                  contain workspace name' rule.
-- * p_use_strong_rules:            Pass `FALSE` when calling this API.
-- * p_min_length_err:              Result returns `TRUE` or `FALSE`, depending upon whether the password
--                                  meets the minimum length requirement.
-- * p_new_differs_by_err:          Result returns `TRUE` or `FALSE`, depending upon whether the password
--                                  meets the 'new password must differ from old' requirement.
-- * p_one_alpha_err:	            Result returns `TRUE` or `FALSE`, depending upon whether the password
--                                  meets the requirement to contain at least one alphabetic character.
-- * p_one_numeric_err:	            Result returns `TRUE` or `FALSE`, depending upon whether the password
--                                  meets the requirement to contain at least one numeric character.
-- * p_one_punctuation_err:	        Result returns `TRUE` or `FALSE`, depending upon whether the password
--                                  meets the requirement to contain at least one punctuation character.
-- * p_one_upper_err:   	        Result returns `TRUE` or `FALSE`, depending upon whether the password
--                                  meets the requirement to contain at least one upper-case character.
-- * p_one_lower_err:	            Result returns `TRUE` or `FALSE`, depending upon whether the password
--                                  meets the requirement to contain at least one lower-case character.
-- * p_not_like_username_err:	    Result returns `TRUE` or `FALSE`, depending upon whether the password
--                                  meets the requirement that it must not contain the username.
-- * p_not_like_workspace_name_err:	Result returns `TRUE` or `FALSE`, depending upon whether the password
--                                  meets the requirement that it must not contain the workspace name.
-- * p_not_like_words_err:	        Result returns `TRUE` or `FALSE`, depending upon whether the password
--                                  meets the requirement that it must not contain specified simple words.
-- * p_not_reusable_err:            Result returns `TRUE` or `FALSE`, depending upon whether the password
--                                  can be reused based on password history rules.
--
-- Example:
-- The following example checks if the new password 'foo' for the user 'SOMEBODY' meets all the password
-- strength requirements defined by the APEX site administrator. If any of the checks fail, i.e. if any
-- of the associated `OUT` parameters returns `TRUE`, then the example outputs a relevant message.
-- For example, if the APEX site administrator defined that passwords must have at least one numeric
-- character and the password 'foo' is checked, then the `p_one_numeric_err OUT` parameter returns `TRUE`
-- and the message "Password must contain at least one numeric character" is displayed.
--
--     declare
--         l_username                    varchar2(30);
--         l_password                    varchar2(30);
--         l_old_password                varchar2(30);
--         l_workspace_name              varchar2(30);
--         l_min_length_err              boolean;
--         l_new_differs_by_err          boolean;
--         l_one_alpha_err               boolean;
--         l_one_numeric_err             boolean;
--         l_one_punctuation_err         boolean;
--         l_one_upper_err               boolean;
--         l_one_lower_err               boolean;
--         l_not_like_username_err       boolean;
--         l_not_like_workspace_name_err boolean;
--         l_not_like_words_err          boolean;
--         l_not_reusable_err            boolean;
--         l_password_history_days       pls_integer;
--     begin
--         l_username := 'SOMEBODY';
--         l_password := 'foo';
--         l_old_password := 'foo';
--         l_workspace_name := 'XYX_WS';
--         l_password_history_days :=
--             apex_instance_admin.get_parameter ('PASSWORD_HISTORY_DAYS');
--
--         apex_util.strong_password_check(
--             p_username                    => l_username,
--             p_password                    => l_password,
--             p_old_password                => l_old_password,
--             p_workspace_name              => l_workspace_name,
--             p_use_strong_rules            => false,
--             p_min_length_err              => l_min_length_err,
--             p_new_differs_by_err          => l_new_differs_by_err,
--             p_one_alpha_err               => l_one_alpha_err,
--             p_one_numeric_err             => l_one_numeric_err,
--             p_one_punctuation_err         => l_one_punctuation_err,
--             p_one_upper_err               => l_one_upper_err,
--             p_one_lower_err               => l_one_lower_err,
--             p_not_like_username_err       => l_not_like_username_err,
--             p_not_like_workspace_name_err => l_not_like_workspace_name_err,
--             p_not_like_words_err          => l_not_like_words_err,
--             p_not_reusable_err            => l_not_reusable_err);
--
--         if l_min_length_err then
--             htp.p('Password is too short');
--         end if;
--
--         if l_new_differs_by_err then
--             htp.p('Password is too similar to the old password');
--         end if;
--
--         if l_one_alpha_err then
--             htp.p('Password must contain at least one alphabetic character');
--         end if;
--
--         if l_one_numeric_err then
--             htp.p('Password  must contain at least one numeric character');
--         end if;
--
--         if l_one_punctuation_err then
--             htp.p('Password  must contain at least one punctuation character');
--         end if;
--
--         if l_one_upper_err then
--             htp.p('Password must contain at least one upper-case character');
--         end if;
--
--         if l_one_lower_err then
--             htp.p('Password must contain at least one lower-case character');
--         end if;
--
--         if l_not_like_username_err then
--             htp.p('Password may not contain the username');
--         end if;
--
--         if l_not_like_workspace_name_err then
--             htp.p('Password may not contain the workspace name');
--         end if;
--
--         if l_not_like_words_err then
--             htp.p('Password contains one or more prohibited common words');
--         end if;
--
--         if l_not_reusable_err then
--             htp.p('Password cannot be used because it has been used for the
--                 account within the last '||l_password_history_days||' days.');
--         end if;
--     end;
--
-- See Also:
-- STRONG_PASSWORD_VALIDATION
--==============================================================================
procedure strong_password_check(
    p_username                    in  varchar2,
    p_password                    in  varchar2,
    p_old_password                in  varchar2,
    p_workspace_name              in  varchar2,
    p_use_strong_rules            in  boolean,
    p_min_length_err              out boolean,
    p_new_differs_by_err          out boolean,
    p_one_alpha_err               out boolean,
    p_one_numeric_err             out boolean,
    p_one_punctuation_err         out boolean,
    p_one_upper_err               out boolean,
    p_one_lower_err               out boolean,
    p_not_like_username_err       out boolean,
    p_not_like_workspace_name_err out boolean,
    p_not_like_words_err          out boolean,
    p_not_reusable_err            out boolean);

--==============================================================================
-- This function returns formatted HTML in a VARCHAR2 result based on whether a proposed password
-- meets the password strength requirements as defined by the Oracle APEX site administrator.
--
-- Parameters:
-- * p_username:         Username that identifies the account in the current workspace.
-- * p_password:         Password to be checked against password strength rules.
-- * p_old_password:     Current password for the account. Used only to enforce the "new password must differ
--                       from old" rule.
-- * p_workspace_name:   Current workspace name, used only to enforce the "password must not contain workspace
--                       name" rule.
-- * p_use_strong_rules: Pass `FALSE` when calling this API.
--
-- Example:
-- The following example checks if the new password 'foo' for the user `SOMEBODY` meets all the password
-- strength requirements defined by the APEX site administrator. If any of the checks fail, then the
-- example outputs formatted HTML showing details of where the new password fails to meet the requirements.
--
--     declare
--           l_username                    varchar2(30);
--           l_password                    varchar2(30);
--           l_old_password                varchar2(30);
--           l_workspace_name              varchar2(30);
--     begin
--         l_username := 'SOMEBODY';
--         l_password := 'foo';
--         l_old_password := 'foo';
--         l_workspace_name := 'XYX_WS';
--
--         htp.p(apex_util.strong_password_validation(
--             p_username                    => l_username,
--             p_password                    => l_password,
--             p_old_password                => l_old_password,
--             p_workspace_name              => l_workspace_name));
--     end;
--
-- See Also:
-- STRONG_PASSWORD_CHECK
--==============================================================================
function strong_password_validation(
    p_username         in varchar2,
    p_password         in varchar2,
    p_old_password     in varchar2 default null,
    p_workspace_name   in varchar2,
    p_use_strong_rules in boolean default false)
    return varchar2;

--##############################################################################
--#
--# PRINTING
--#
--##############################################################################

--==============================================================================
--! Function Signature 1 in package.
--! Function Signature 4 in public documentation.
-- Overload: data_clob
-- This function returns a document as BLOB using XML based report data (as CLOB) and an RTF or XSL-FO
-- based report layout.
--
-- Parameters:
-- * p_report_data:        XML based report data, must be encoded in UTF-8.
-- * p_report_layout:      Report layout in XSL-FO or RTF format.
-- * p_report_layout_type: Defines the report layout type, that is `xsl-fo` or `rtf`.
-- * p_document_format:    Defines the document format, that is `pdf`, `rtf`, `xls`, `htm`, or `xml`.
-- * p_print_server:       URL of of the print server. If not specified, the print server is
--                         derived from preferences.
--
-- Example:
-- In this example, `GET_PRINT_DOCUMENT` is used with `APEX_MAIL.SEND` and `APEX_MAIL.ADD_ATTACHMENT`
-- to send an email with an attachment of the file returned by `GET_PRINT_DOCUMENT`. Both the report
-- data and layout are taken from values stored in page items (`P1_XML` and `P1_XSL`).
--
--     declare
--         l_id number;
--         l_document blob;
--     begin
--         l_document := apex_util.get_print_document (
--             p_report_data         => :P1_XML,
--             p_report_layout       => :P1_XSL,
--             p_report_layout_type  => 'xsl-fo',
--             p_document_format     => 'pdf');
--
--        l_id := apex_mail.send(
--            p_to        => :p35_mail_to,
--            p_from      => 'admin@example.com',
--            p_subj      => 'sending PDF by using print API',
--            p_body      => 'Please review the attachment.',
--            p_body_html => 'Please review the attachment');
--
--        apex_mail.add_attachment (
--            p_mail_id    => l_id,
--            p_attachment => l_document,
--            p_filename   => 'mydocument.pdf',
--            p_mime_type  => 'application/pdf');
--     end;
--
-- See Also:
-- GET_PRINT_DOCUMENT@data_blob
-- GET_PRINT_DOCUMENT@layout_varchar2
-- GET_PRINT_DOCUMENT@layout_clob
-- DOWNLOAD_PRINT_DOCUMENT@data_clob
-- APEX_MAIL.SEND
-- APEX_MAIL.ADD_ATTACHMENT
--==============================================================================
function get_print_document (
    p_report_data         in clob,
    p_report_layout       in clob,
    p_report_layout_type  in varchar2 default 'xsl-fo',
    p_document_format     in varchar2 default 'pdf',
    p_print_server        in varchar2 default null
) return blob;

--==============================================================================
--! Function Signature 2 in package.
--! Function Signature 1 in public documentation.
-- Overload: data_blob
-- This function returns a document as BLOB using XML based report data (as BLOB) and an RTF or XSL-FO
-- based report layout.
--
-- Parameters:
-- * p_report_data:        XML based report data, must be encoded in UTF-8.
-- * p_report_layout:      Report layout in XSL-FO or RTF format.
-- * p_report_layout_type: Defines the report layout type, that is `xsl-fo` or `rtf`.
-- * p_document_format:    Defines the document format, that is `pdf`, `rtf`, `xls`, `htm`, or `xml`.
-- * p_print_server:       URL of of the print server. If not specified, the print server is
--                         derived from preferences.
--
-- Example:
-- For an example see `GET_PRINT_DOCUMENT Function Signature 1`.
--
-- See Also:
-- GET_PRINT_DOCUMENT@data_clob
-- GET_PRINT_DOCUMENT@layout_varchar2
-- GET_PRINT_DOCUMENT@layout_clob
-- DOWNLOAD_PRINT_DOCUMENT@data_blob
--==============================================================================
function get_print_document (
    p_report_data         in blob,
    p_report_layout       in clob,
    p_report_layout_type  in varchar2 default 'xsl-fo',
    p_document_format     in varchar2 default 'pdf',
    p_print_server        in varchar2 default null
) return blob;

--==============================================================================
--! Function Signature 3 in package.
--! Function Signature 2 in public documentation.
-- Overload: layout_varchar2
-- This function returns a document as BLOB using a pre-defined report query and a pre-defined report
-- layout.
--
-- Parameters:
-- * p_application_id:      Defines the application ID of the report query.
-- * p_report_query_name:   Name of the report query (stored under application's Shared Components).
-- * p_report_layout_name:  Name of the report layout (stored under application's Shared Components).
--                          If `p_report_layout_name` is not specified, the layout associated with
--                          report query will be used.
-- * p_report_layout_type: Defines the report layout type, that is `xsl-fo` or `rtf`.
-- * p_document_format:    Defines the document format, that is `pdf`, `rtf`, `xls`, `htm`, or `xml`.
-- * p_print_server:       URL of of the print server. If not specified, the print server is
--                         derived from preferences.
--
-- Example:
-- For an example see `GET_PRINT_DOCUMENT Function Signature 1`.
--
-- See Also:
-- GET_PRINT_DOCUMENT@data_clob
-- GET_PRINT_DOCUMENT@data_blob
-- GET_PRINT_DOCUMENT@layout_clob
-- DOWNLOAD_PRINT_DOCUMENT@layout_varchar2
--==============================================================================
function get_print_document (
    p_application_id      in number,
    p_report_query_name   in varchar2,
    p_report_layout_name  in varchar2 default null,
    p_report_layout_type  in varchar2 default 'xsl-fo',
    p_document_format     in varchar2 default 'pdf',
    p_print_server        in varchar2 default null
) return blob;

--==============================================================================
--! Function Signature 4 in package.
--! Function Signature 3 in public documentation.
-- Overload: layout_clob
-- This function returns a document as BLOB using a pre-defined report query and an RTF or XSL-FO based
-- report layout.
--
-- Parameters:
-- * p_application_id:      Defines the application ID of the report query.
-- * p_report_query_name:   Name of the report query (stored under application's Shared Components).
-- * p_report_layout:      Report layout in XSL-FO or RTF format.
-- * p_report_layout_type: Defines the report layout type, that is `xsl-fo` or `rtf`.
-- * p_document_format:    Defines the document format, that is `pdf`, `rtf`, `xls`, `htm`, or `xml`.
-- * p_print_server:       URL of of the print server. If not specified, the print server is
--                         derived from preferences.
-- Example:
-- For an example see `GET_PRINT_DOCUMENT Function Signature 1`.
--
-- See Also:
-- GET_PRINT_DOCUMENT@data_clob
-- GET_PRINT_DOCUMENT@data_clob
-- GET_PRINT_DOCUMENT@layout_varchar2
-- DOWNLOAD_PRINT_DOCUMENT@layout_clob
--==============================================================================
function get_print_document (
    p_application_id      in number,
    p_report_query_name   in varchar2,
    p_report_layout       in clob,
    p_report_layout_type  in varchar2 default 'xsl-fo',
    p_document_format     in varchar2 default 'pdf',
    p_print_server        in varchar2 default null
) return blob;

--==============================================================================
--! Procedure Signature 1 in package.
--! Procedure Signature 4 in public documentation.
-- Overload: data_clob
-- This procedure initiates the download of a print document using XML based report data (as a CLOB)
-- and an RTF or XSL-FO based report layout.
--
-- Parameters:
-- * p_file_name:           Defines the filename of the print document.
-- * p_content_disposition: Specifies whether to download the print document or display inline (`attachment`, `inline`).
-- * p_report_data:         XML based report data, must be encoded in UTF-8.
-- * p_report_layout:       Report layout in XSL-FO or RTF format.
-- * p_report_layout_type:  Defines the report layout type, that is `xsl-fo` or `rtf`.
-- * p_document_format:     Defines the document format, that is `pdf`, `rtf`, `xls`, `htm`, or `xml`.
-- * p_print_server:        URL of of the print server. If not specified, the print server is
--                          derived from preferences.
--
-- Example:
-- The following example shows how to use `DOWNLOAD_PRINT_DOCUMENT` using Signature 1 (XML based
-- report data (as a CLOB) and RTF or XSL-FO based report layout). In this example both the report
-- data (XML) and report layout (XSL-FO) are taken from values stored in page items.
--
--     begin
--         apex_util.download_print_document (
--             p_file_name           => 'mydocument',
--             p_content_disposition => 'attachment',
--             p_report_data         => :P1_XML,
--             p_report_layout       => :P1_XSL,
--             p_report_layout_type  => 'xsl-fo',
--             p_document_format     => 'pdf');
--     end;
--
-- See Also:
-- DOWNLOAD_PRINT_DOCUMENT@data_blob
-- DOWNLOAD_PRINT_DOCUMENT@layout_varchar2
-- DOWNLOAD_PRINT_DOCUMENT@layout_clob
-- GET_PRINT_DOCUMENT@data_clob
--==============================================================================
procedure download_print_document (
    p_file_name           in varchar,
    p_content_disposition in varchar  default 'attachment',
    p_report_data         in clob,
    p_report_layout       in clob,
    p_report_layout_type  in varchar2 default 'xsl-fo',
    p_document_format     in varchar2 default 'pdf',
    p_print_server        in varchar2 default null
);

--==============================================================================
--! Procedure Signature 2 in package.
--! Procedure Signature 1 in public documentation.
-- Overload: data_blob
-- This procedure initiates the download of a print document using XML based report data (as a BLOB)
-- and an RTF or XSL-FO based report layout.
--
-- Parameters:
-- * p_file_name:           Defines the filename of the print document.
-- * p_content_disposition: Specifies whether to download the print document or display inline (`attachment`, `inline`).
-- * p_report_data:         XML based report data, must be encoded in UTF-8.
-- * p_report_layout:       Report layout in XSL-FO or RTF format.
-- * p_report_layout_type:  Defines the report layout type, that is `xsl-fo` or `rtf`.
-- * p_document_format:     Defines the document format, that is `pdf`, `rtf`, `xls`, `htm`, or `xml`.
-- * p_print_server:        URL of of the print server. If not specified, the print server is
--                          derived from preferences.
--
-- Example:
-- For an example see `DOWNLOAD_PRINT_DOCUMENT Procedure Signature 1`.
--
-- See Also:
-- DOWNLOAD_PRINT_DOCUMENT@data_clob
-- DOWNLOAD_PRINT_DOCUMENT@layout_varchar2
-- DOWNLOAD_PRINT_DOCUMENT@layout_clob
-- GET_PRINT_DOCUMENT@data_blob
--==============================================================================
procedure download_print_document (
    p_file_name           in varchar,
    p_content_disposition in varchar  default 'attachment',
    p_report_data         in blob,
    p_report_layout       in clob,
    p_report_layout_type  in varchar2 default 'xsl-fo',
    p_document_format     in varchar2 default 'pdf',
    p_print_server        in varchar2 default null
);

--==============================================================================
--! Procedure Signature 3 in package.
--! Procedure Signature 3 in public documentation.
-- Overload: layout_varchar2
-- This procedure initiates the download of a print document using a pre-defined report query and
-- a pre-defined report layout.
--
-- Parameters:
-- * p_file_name:           Defines the filename of the print document.
-- * p_content_disposition: Specifies whether to download the print document or display inline
--                          (`attachment`, `inline`).
-- * p_application_id:      Defines the application ID of the report query.
-- * p_report_query_name:   Name of the report query (stored under application's Shared Components).
-- * p_report_layout_name:  Name of the report layout (stored under application's Shared Components).
--                          If `p_report_layout_name` is not specified, the layout associated with
--                          report query will be used.
-- * p_report_layout_type: Defines the report layout type, that is `xsl-fo` or `rtf`.
-- * p_document_format:    Defines the document format, that is `pdf`, `rtf`, `xls`, `htm`, or `xml`.
-- * p_print_server:       URL of of the print server. If not specified, the print server is
--                         derived from preferences.
--
-- Example:
-- The following example shows how to use the `DOWNLOAD_PRINT_DOCUMENT` using Signature 3 (Pre-defined
-- report query and pre-defined report layout). In this example, the data for the report is taken from
-- a Report Query called 'ReportQuery' stored in the current application's Shared Components >
-- Report Queries. The report layout is taken from a Report Layout called 'ReportLayout' stored in
-- the current application's Shared Components > Report Layouts. Note that if you want to provision
-- dynamic layouts, instead of specifying 'ReportLayout' for the `p_report_layout_name` parameter,
-- you could reference a page item that allowed the user to select one of multiple saved Report Layouts.
-- This example also provides a way for the user to specify how they want to receive the document
-- (as an attachment or inline), through passing the value of `P1_CONTENT_DISP` to the `p_content_disposition``
-- parameter. `P1_CONTENT_DISP` is a page item of type 'Select List' with the following List of Values
-- Definition:
--
--     STATIC:
--     In Browser;inline,
--     Save / Open in separate Window;attachment
--
--     begin
--         apex_util.download_print_document (
--             p_file_name           => 'myreport123',
--             p_content_disposition => :P1_CONTENT_DISP,
--             p_application_id      => :APP_ID,
--             p_report_query_name   => 'ReportQuery',
--             p_report_layout_name  => 'ReportLayout',
--             p_report_layout_type  => 'rtf',
--             p_document_format     => 'pdf');
--     end;
--
-- See Also:
-- DOWNLOAD_PRINT_DOCUMENT@data_clob
-- DOWNLOAD_PRINT_DOCUMENT@data_blob
-- DOWNLOAD_PRINT_DOCUMENT@layout_clob
-- GET_PRINT_DOCUMENT@layout_varchar2
--==============================================================================
procedure download_print_document (
    p_file_name           in varchar,
    p_content_disposition in varchar  default 'attachment',
    p_application_id      in number,
    p_report_query_name   in varchar2,
    p_report_layout_name  in varchar2 default null,
    p_report_layout_type  in varchar2 default 'xsl-fo',
    p_document_format     in varchar2 default 'pdf',
    p_print_server        in varchar2 default null );

--==============================================================================
--! Procedure Signature 4 in package.
--! Procedure Signature 2 in public documentation.
-- Overload: layout_clob
-- This procedure initiates the download of a print document using a pre-defined report query and
-- an RTF or XSL-FO based report layout.
--
-- Parameters:
-- * p_file_name:           Defines the filename of the print document.
-- * p_content_disposition: Specifies whether to download the print document or display inline
--                          (`attachment`, `inline`).
-- * p_application_id:      Defines the application ID of the report query.
-- * p_report_query_name:   Name of the report query (stored under application's Shared Components).
-- * p_report_layout:       Report layout in XSL-FO or RTF format.
-- * p_report_layout_type:  Defines the report layout type, that is `xsl-fo` or `rtf`.
-- * p_document_format:     Defines the document format, that is `pdf`, `rtf`, `xls`, `htm`, or `xml`.
-- * p_print_server:        URL of of the print server. If not specified, the print server is
--                          derived from preferences.
-- Example:
-- The following example shows how to use the `DOWNLOAD_PRINT_DOCUMENT` using Signature 4 (Pre-defined
-- report query and RTF or XSL-FO based report layout). In this example, the data for the report is
-- taken from a Report Query called `ReportQueryAndXSL` stored in the current application's Shared
-- Components > Report Queries. The report layout is taken from a value stored in a page item (`P1_XSL`).
--
--     begin
--         apex_util.download_print_document (
--             p_file_name           => 'mydocument',
--             p_content_disposition => 'attachment',
--             p_application_id      => :APP_ID,
--             p_report_query_name   => 'ReportQueryAndXSL',
--             p_report_layout       => :P1_XSL,
--             p_report_layout_type  => 'xsl-fo',
--             p_document_format     => 'pdf');
--     end;
--
-- See Also:
-- DOWNLOAD_PRINT_DOCUMENT@data_clob
-- DOWNLOAD_PRINT_DOCUMENT@data_blob
-- DOWNLOAD_PRINT_DOCUMENT@layout_varchar2
-- GET_PRINT_DOCUMENT@layout_clob
--==============================================================================
procedure download_print_document (
    p_file_name           in varchar,
    p_content_disposition in varchar  default 'attachment',
    p_application_id      in number,
    p_report_query_name   in varchar2,
    p_report_layout       in clob,
    p_report_layout_type  in varchar2 default 'xsl-fo',
    p_document_format     in varchar2 default 'pdf',
    p_print_server        in varchar2 default null
);

--==============================================================================
-- This function returns the URL to the Oracle APEX instance, depending on the
-- option passed.
--
-- Parameters:
-- * p_option:          Specifies the parts of the URL to include.
--                      Possible values for `p_option` include:
--                      + `NULL:`     Return URL up to port number. For example:
--                                    `https://example.com:7778`
--                      + `SCRIPT:`   Return URL to include script name. For example:
--                          + With friendly URLs enabled, this typically returns:
--                                    `https://example.com:7778/pls/apex/{workspace}/r/{application}`
--                          + With friendly URLs disabled, this typically returns:
--                                    `https://example.com:7778/pls/apex/`
--                      + `APEX_PATH:` Return URL to include script name. For example:
--                                    `https://example.com:7778/pls/apex/`
--                      + `IMGPRE:`    Return URL to include image prefix. For example:
--                                    `https://example.com:7778/i/`
-- Example:
-- The following example returns the URL to the current APEX instance including the script name.
--
--     declare
--         l_host_url      varchar2(4000);
--         l_url           varchar2(4000);
--         l_application   varchar2(30) := 'f?p=100:1';
--         l_email_body    varchar2(32000);
--     begin
--         l_host_url := apex_util.host_url('SCRIPT');
--         l_url := l_host_url||l_application;
--         l_email_body := 'The URL to the application is: '||l_url;
--     end;
--==============================================================================
function host_url(
    p_option          in varchar2 default null )
    return varchar2;

--==============================================================================
-- This procedure calls `owa_util.redirect_url` to tell the browser to redirect to a new URL.
-- Afterwards, it automatically calls `apex_application.stop_apex_engine` to end further processing
-- of the Oracle APEX application.
--
-- Parameters:
-- * p_url:              The URL the browser requests.
-- * p_reset_htp_buffer: Set to `TRUE` to reset the HTP buffer to make sure the browser understands
--                       the redirect to the new URL and is not confused by data that is already written
--                       to the HTP buffer. Set to `FALSE` if the application has its own cookie to
--                       use in the response.
--
-- Example:
-- The following example tells the browser to redirect to `http://www.example.com` and immediately stops
-- further processing.
--
--     apex_util.redirect_url (
--      p_url => 'http://www.example.com/' );
--
-- See Also:
-- APEX_APPLICATION.STOP_APEX_ENGINE
--==============================================================================
procedure redirect_url (
    p_url              in varchar2,
    p_reset_htp_buffer in boolean default true );

--##############################################################################
--#
--# SESSION HANDLING
--#
--##############################################################################
--
--==============================================================================
-- This procedure sets the current session's 'Maximum Session Length in Seconds' value, overriding the
-- corresponding application attribute. This enables developers to dynamically shorten or lengthen
-- the session life based on criteria determined after the user authenticates.
--
-- Parameters:
-- * p_seconds:	A positive integer indicating the number of seconds that the session used by the
--              application can exist.
-- * p_scope:	This parameter is obsolete. The procedure always sets the lifetime for the whole session.
--
-- Example:
-- The following example sets the current application's Maximum Session Length in Seconds attribute to 3600
-- seconds (one hour).
--
--     begin
--        apex_util.set_session_lifetime_seconds(p_seconds => 3600);
--     end;
--==============================================================================
procedure set_session_lifetime_seconds(
    p_seconds in number,
    p_scope   in varchar2 default 'SESSION'); -- Ignore:
--==============================================================================
-- Sets the current application's 'Maximum Session Idle Time in Seconds' value for the current session,
-- overriding the corresponding application attribute. This allows developers to dynamically shorten
-- or lengthen the maximum idle time allowed between page requests based on criteria determined after
-- the user authenticates.
--
-- Parameters:
-- * p_seconds:	A positive integer indicating the number of seconds allowed between page requests.
-- * p_scope:	This parameter is obsolete. The procedure always sets the lifetime for the whole session.
--
-- Example:
-- The following example shows how to use the `SET_SESSION_MAX_IDLE_SECONDS` procedure to set the
-- current application's 'Maximum Session Idle Time in Seconds' attribute to 1200 seconds (twenty minutes).
-- The following example applies to all applications using the current session.
--
--     begin
--        apex_util.set_session_max_idle_seconds (p_seconds => 1200);
--     end;
--==============================================================================
procedure set_session_max_idle_seconds(
    p_seconds  in number,
    p_scope    in varchar2 default 'SESSION'); -- Ignore:

--==============================================================================
-- This procedure sets the time zone to be used for the current user in the current Oracle APEX session.
-- It will also immediately set the database session time zone to this setting.  If the value
-- specified in `p_time_zone` is invalid, then the session time zone will be set to the database time zone.
--
-- Parameters:
-- * p_time_zone: A time zone value in the form of hours and minutes. Examples include: `+09:00`, `04:00`, `-05:00`.
--
-- Example:
-- The following example shows how to use the `SET_SESSION_TIME_ZONE` procedure. It sets the time zone
-- for the current user for the duration of the APEX session.
--
--     begin
--         apex_util.set_session_time_zone( p_time_zone => '-05:00');
--     end;
--
-- See Also:
-- GET_SESSION_TIME_ZONE
--==============================================================================
procedure set_session_time_zone(
    p_time_zone in varchar2 );

--==============================================================================
-- This function returns the time zone for the current user in the current Oracle APEX session. This
-- value is `NULL` if the time zone is not explicitly set by using `APEX_UTIL.SET_SESSION_TIME_ZONE`
-- or if an application's automatic time zone attribute is enabled.
-- When it has not been explicitly set, the database session time zone is always set to the database
-- time zone.
--
-- Example:
-- The following example returns the session time zone for the current user in the current APEX session
-- into a local variable.
--
--     declare
--        val varchar2(255);
--     begin
--        val := apex_util.get_session_time_zone;
--     end;
--
-- See Also:
-- SET_SESSION_TIME_ZONE
--==============================================================================
function get_session_time_zone return varchar2;

--==============================================================================
-- Needs Review: Mark internal or add to public documentation?
-- reset_session_time_zone will clear the time zone setting for the current APEX session.  As well, it will alter
-- the time zone setting of the database session to the database time zone
--==============================================================================
procedure reset_session_time_zone;

--==============================================================================
-- This procedure sets the language for the current user in the current Oracle APEX session. The
-- language must be a valid IANA language name.
--
-- Note:
-- This is only relevant when the application language is derived from 'Session'.
--
-- Parameters:
-- * p_lang: This is an IANA language code. Examples include en, de, de-at, zh-cn, and pt-br.
--
-- Example:
-- The following example sets the language for the current user for the duration of the APEX session.
--
--     begin
--         apex_util.set_session_lang( p_lang => 'en');
--     end;
--
-- See Also:
-- GET_SESSION_LANG
--==============================================================================
procedure set_session_lang(
    p_lang in varchar2 );

--==============================================================================
-- This function returns the language setting for the current user in the current Oracle APEX session.
--
-- Note:
-- This is only relevant when the application language is derived from 'Session'.
--
-- Example:
-- The following example returns the session language for the current user in the current APEX session
-- into a local variable.
--
--     declare
--         val varchar2(5);
--     begin
--         val := apex_util.get_session_lang;
--     end;
--
-- See Also:
-- SET_SESSION_LANG
--==============================================================================
function get_session_lang
    return varchar2;

--==============================================================================
-- Needs Review: Mark internal or add to public documentation?
-- reset_session_lang will clear the language setting for the current APEX session.
--==============================================================================
procedure reset_session_lang;

--==============================================================================
-- This procedure sets the territory to be used for the current user in the current Oracle APEX session.
-- The territory name must be a valid Oracle territory.
--
-- Parameters:
-- * p_territory: A valid Oracle territory name. Examples include: `AMERICA`, `UNITED KINGDOM`, `ISRAEL`,
--                `AUSTRIA`, and `UNITED ARAB EMIRATES`.
--
-- Example:
-- The following example shows how to use the `SET_SESSION_TERRITORY` procedure. It sets the territory
-- for the current user for the duration of the APEX session.
--
--     begin
--         apex_util.set_session_territory( p_territory => 'UNITED KINGDOM');
--     end;
--
-- See Also:
-- GET_SESSION_TERRITORY
--==============================================================================
procedure set_session_territory(
    p_territory in varchar2 );

--==============================================================================
-- This function returns the territory setting for the current user in the current Oracle APEX session.
--
-- Example:
-- The following example returns the session territory setting for the current user in the current
-- APEX session into a local variable.
--
--     declare
--         val varchar2(30);
--     begin
--         val := apex_util.get_session_territory;
--     end;
--
-- See Also:
-- SET_SESSION_TERRITORY
--==============================================================================
function get_session_territory
    return varchar2;

--==============================================================================
-- Needs Review: Mark internal or add to public documentation?
-- reset_session_territory will clear the territory setting for the current APEX session.
--==============================================================================
procedure reset_session_territory;

--##############################################################################
--#
--# PROCESSING THE CURRENT PAGE VIEW
--#
--##############################################################################

--==============================================================================
-- This procedure sets the name of the database edition to be used in all application SQL parsed in
-- the current page view or page submission.
--
-- Note:
-- Support for Edition-Based Redefinition is only available in database version 11.2.0.1 or higher.
--
-- Parameters:
-- * p_edition: The name of the database edition.
--
-- Example:
-- The following example shows how to use the `SET_EDITION` procedure. It sets the database edition name
-- for the session of the current page view.
--
--     begin
--         apex_util.set_edition( p_edition => 'Edition1' );
--     end;
--
-- See Also:
-- GET_EDITION
--==============================================================================
procedure set_edition(
    p_edition in varchar2 );

--==============================================================================
-- This function returns the database edition for the current page view.
--
-- Example:
-- The following example shows how to use the `GET_EDITION` function. It returns the database edition name
-- for the current page view into a local variable.
--
--     declare
--         val varchar2(30);
--     begin
--         val := apex_util.get_edition;
--     end;
--
-- See Also:
-- SET_EDITION
--==============================================================================
function get_edition
    return varchar2;

--==============================================================================
-- This procedure changes the parsing user for the current page view to another
-- workspace schema. You can only call this procedure from within the application's
-- Initialization PL/SQL Code.
--
-- Parameters:
-- * p_schema: The new parsing schema.
--
-- Raises:
-- * `WWV_FLOW.NO_PRIV_ON_SCHEMA`: If `p_schema` is not a valid workspace schema.
-- * `PROGRAM_ERROR`:              When not called from Initialization PL/SQL Code.
--
-- Example:
-- On pages 1-100, change the parsing schema to `:G_PARSING_SCHEMA`.
--
--     begin
--         if :APP_PAGE_ID between 1 and 100 then
--             apex_util.set_parsing_schema_for_request (
--                 p_schema => :G_PARSING_SCHEMA );
--         end if;
--     end;
--==============================================================================
procedure set_parsing_schema_for_request (
    p_schema in varchar2 );

--==============================================================================
-- Use this procedure with `apex_util.find_security_group_id` to ease the use of the mail package in
-- batch mode. This procedure is especially useful when a schema is associated with more than one
-- workspace. For example, you might want to create a procedure that is run by a nightly job to email
-- all outstanding tasks.
--
-- Parameters:
-- * p_security_group_id: This is the security group id of the workspace you are working in.
--
-- Example:
-- The following example sends an alert to each user that has had a task assigned within the last day.
--
--     create or replace procedure new_tasks
--         is
--             l_workspace_id      number;
--             l_subject           varchar2(2000);
--             l_body              clob;
--             l_body_html         clob;
--         begin
--             l_workspace_id := apex_util.find_security_group_id (p_workspace => 'PROJECTS');
--             apex_util.set_security_group_id (p_security_group_id => l_workspace_id);
--
--             l_body := ' ';
--             l_subject := 'You have new tasks';
--             for c1 in (select distinct(p.email_address) email_address, p.user_id
--                          from teamsp_user_profile p, teamsp_tasks t
--                         where p.user_id = t.assigned_to_user_id
--                           and t.created_on > sysdate - 1
--                           and p.email_address is not null ) loop
--                 l_body_html := '<p />The following tasks have been added.';
--                 for c2 in (select task_name, due_date
--                              from teamsp_tasks
--                             where assigned_to_user_id = c1.user_id
--                               and created_on > sysdate - 1 ) loop
--                     l_body_html := l_body_html || '<p />Task: '||c2.task_name||', due '||c2.due_date;
--                 end loop;
--         apex_mail.send (
--                     p_to        => c1.email_address,
--                     p_from      => c1.email_address,
--                     p_body      => l_body,
--                     p_body_html => l_body_html,
--                     p_subj      => l_subject );
--             end loop;
--         end;
--
-- See Also:
-- FIND_SECURITY_GROUP_ID
--==============================================================================
procedure set_security_group_id (
    p_security_group_id  in number );

--==============================================================================
-- This procedure sets the current workspace.
--
-- Parameters:
-- * p_workspace: The workspace's short name.
--
-- Example:
-- This example sets the workspace `MY_WORKSPACE`.
--
--     begin
--         apex_util.set_workspace (
--             p_workspace => 'MY_WORKSPACE' );
--     end;
--
-- See Also:
-- FIND_SECURITY_GROUP_ID
-- SET_SECURITY_GROUP_ID
--==============================================================================
procedure set_workspace (
    p_workspace in varchar2 );

--==============================================================================
-- Needs Review: Mark internal or add to public documentation?
-- set_compatibility_mode will update the compatibility mode setting of the specified
-- application. Valid values for p_compatibility_mode are '4.0' and '4.1'.
--==============================================================================
procedure set_compatibility_mode (
    p_application_id     in number,
    p_compatibility_mode in varchar2 );

--##############################################################################
--#
--# NOTE: The following APIs are for internal use only and are likely to be
--#       changed or removed in a future version of Oracle APEX.
--#
--##############################################################################

--==============================================================================
-- Internal:
-- Deprecated: 24.2
--
-- Returns TRUE if p_value contains dangerous characters which could be used
-- to attack the application/system/... and FALSE if p_value is safe.
--
-- p_check can be one of the following values:
--
--   US_ONLY - Allowlist for a-Z, 0-9 and space
--   WEB_SAFE - Blocklist HTML command characters (<>")
--   NO_SPECIAL_CHAR - Blocklist &<>"/;,*|=% and --
--   NO_SPECIAL_CHAR_NL - Blocklist &<>"/;,*|=% or -- and new line
--==============================================================================
function has_restricted_chars (
    p_value in varchar2,
    p_check in varchar2 )
    return boolean;


--==============================================================================
-- Deprecated: 23.1. Use `APEX_APPLICATION_ADMIN.SET_BUILD_OPTION_STATUS` instead.
-- ___
--
-- Use this procedure to change the build option status of a specified application.
--
-- Note:
-- The build option status will be overwritten when the application is upgraded to a new version. To
-- keep the status set via the API, it is necessary to set the build option attribute On Upgrade Keep
-- Status to Yes.
--
-- Parameters:
-- * p_application_id: The ID of the application that owns the build option under Shared Components.
-- * p_id:             The ID of the build option in the application.
-- * p_build_status:   The new status of the build option. Possible values are `INCLUDE`, `EXCLUDE`
--                     (both upper case).
--
-- Example:
--
--     begin
--         apex_util.set_build_option_status(
--             p_application_id => 101,
--             p_id             => 245935500311121039,
--             p_build_status   =>'INCLUDE');
--     end;
--
-- See Also:
-- APEX_APPLICATION_ADMIN.SET_BUILD_OPTION_STATUS
--==============================================================================
procedure set_build_option_status(
   p_application_id   in number,
   p_id               in number,
   p_build_status     in varchar2 );

--==============================================================================
-- Overload: number
-- Deprecated: 23.1. Use `APEX_APPLICATION_ADMIN.GET_BUILD_OPTION_STATUS Function Signature 1` instead.
-- ___
--
-- Use this function to get the build option status of a specified application by providing the ID
-- of the application build option.
--
-- Parameters:
-- * p_application_id: The ID of the application that owns the build option under Shared Components.
-- * p_id:             The ID of the build option in the application.
--
-- Example:
-- The following code retrieves the current status of the specified build option that is identified
-- by ID.
--
--     declare
--         l_status varchar2(255);
--     begin
--        l_status := apex_util.get_build_option_status(
--                        p_application_id => 101,
--                        p_id             => 245935500311121039);
--     end;
--
-- See Also:
-- APEX_APPLICATION_ADMIN.GET_BUILD_OPTION_STATUS@option_id
-- SET_BUILD_OPTION_STATUS
--==============================================================================
function get_build_option_status(
    p_application_id   in number,
    p_id               in number )
    return varchar2;

--==============================================================================
-- Overload: varchar2
-- Deprecated: 23.1. Use `APEX_APPLICATION_ADMIN.GET_BUILD_OPTION_STATUS@option_name` instead.
-- ___
--
-- Use this function to get the build option status of a specified application by providing the name
-- of the application build option.
--
-- Parameters:
-- * p_application_id:    The ID of the application that owns the build option under Shared Components.
-- * p_build_option_name: The ID of the build option in the application.
--
-- Example:
-- The following code retrieves the current status of the specified build option that is identified
-- by name.
--
--     declare
--         l_status varchar2(255);
--     begin
--         l_status := apex_util.get_build_option_status(
--                         p_application_id    => 101,
--                         p_build_option_name => 'EXCLUDE_FROM_PRODUCTION');
--     end;
--
-- See Also:
-- APEX_APPLICATION_ADMIN.GET_BUILD_OPTION_STATUS@option_name
-- SET_BUILD_OPTION_STATUS
--==============================================================================
function get_build_option_status(
    p_application_id    in number,
    p_build_option_name in varchar2 )
    return varchar2;


--==============================================================================
-- Internal:
-- Deprecated: 24.2
--
-- Close all database links open in the current session
--==============================================================================
procedure close_open_db_links;

--==============================================================================
-- Internal:
-- Deprecated: 24.2
--
-- set format mask for report columns in classic reports
--==============================================================================
procedure set_report_column_format (
    p_column_format in wwv_flow_global.vc_arr2 default wwv_flow.empty_vc_arr
);

--==============================================================================
-- Deprecated: 5.1. Use `APEX_THEME.SET_CURRENT_STYLE` instead.
--==============================================================================
procedure set_current_theme_style (
    p_theme_number   in number,
    p_theme_style_id in number
    );

--==============================================================================
-- Deprecated: 23.1. Use `APEX_APPLICATION_ADMIN.SET_APPLICATION_STATUS` instead.
-- ___
-- This procedure changes the status of the application.
--
-- Parameters:
-- * p_application_id:       The Application ID.
-- * p_application_status:   The new status to set the application to.  Values include:
--                             + `AVAILABLE`             - Application is available with no restrictions.
--                             + `AVAILABLE_W_EDIT_LINK` - Application is available with no restrictions.
--                                                         Developer Toolbar shown to developers.
--                             + `DEVELOPERS_ONLY`       - Application only available to developers.
--                             + `RESTRICTED_ACCESS`     - Application only available to users in
--                                                         `p_restricted_user_list`.
--                             + `UNAVAILABLE`           - Application unavailable.  Message shown in
--                                                         `p_unavailable_value`.
--                             + `UNAVAILABLE_PLSQL`     - Application unavailable.  Message shown from
--                                                         PL/SQL block in `p_unavailable_value`.
--                             + `UNAVAILABLE_URL`       - Application unavailable.  Redirected to URL
--                                                         provided in `p_unavailable_value`.
-- * p_unavailable_value:    The value used when the application is unavailable.  This value has
--                           different semantics dependent upon the value for `p_application_status`.
-- * p_restricted_user_list: A comma separated list of users permitted to access the application, when
--                           `p_application_status` equals `RESTRICTED_ACCESS`.
-- Example:
--
--     begin
--     apex_util.set_application_status(
--         p_application_id     => 117,
--         p_application_status => 'AVAILABLE' );
--     end;
--
--     begin
--     apex_util.set_application_status(
--         p_application_id     => 117,
--         p_application_status => 'AVAILABLE_W_EDIT_LINK' );
--     end;
--
--     begin
--     apex_util.set_application_status(
--         p_application_id     => 117,
--         p_application_status => 'DEVELOPERS_ONLY' );
--     end;
--
--     begin
--     apex_util.set_application_status(
--         p_application_id       => 117,
--         p_application_status   => 'RESTRICTED_ACCESS',
--         p_restricted_user_list => 'xxx.xxx@example.com' );
--     end;
--
--     begin
--     apex_util.set_application_status(
--         p_application_id     => 117,
--         p_application_status => 'UNAVAILABLE',
--         p_unavailable_value  => 'Application not available, sorry' );
--     end;
--
--     begin
--     apex_util.set_application_status(
--         p_application_id     => 117,
--         p_application_status => 'UNAVAILABLE_PLSQL',
--         p_unavailable_value  => 'sys.htp.p(''Application unavailable, sorry'');' );
--     end;
--
--     begin
--     apex_util.set_application_status(
--         p_application_id     => 117,
--         p_application_status => 'UNAVAILABLE_URL',
--         p_unavailable_value  => 'http://www.example.com' );
--     end;
--
-- See Also:
-- APEX_APPLICATION_ADMIN.SET_APPLICATION_STATUS
-- GET_APPLICATION_STATUS
--==============================================================================
procedure set_application_status(
   p_application_id       in number,
   p_application_status   in varchar2,
   p_unavailable_value    in varchar2 default null,
   p_restricted_user_list in varchar2 default null );



--==============================================================================
-- Deprecated: 23.1. Use `APEX_APPLICATION_ADMIN.GET_APPLICATION_STATUS` instead.
-- ___
-- This function returns the current status of the application.
-- Status values include:
-- - `AVAILABLE`
-- - `AVAILABLE_W_EDIT_LINK`
-- - `DEVELOPERS_ONLY`
-- - `RESTRICTED_ACCESS`
-- - `UNAVAILABLE`
-- - `UNAVAILABLE_PLSQL`
-- - `UNAVAILABLE_URL`
--
-- Parameters:
-- * p_application_id:       The Application ID.
--
-- Example:
--
--     declare
--         l_status varchar2(100);
--     begin
--         l_status := apex_util.get_application_status(
--                         p_application_id => 117 );
--         dbms_output.put_line( 'The current application status is: ' || l_status );
--     end;
--
-- See Also:
-- APEX_APPLICATION_ADMIN.GET_APPLICATION_STATUS
-- SET_APPLICATION_STATUS
--==============================================================================
function get_application_status(
   p_application_id   in number )
return varchar2;


--==============================================================================
-- Deprecated: 23.1. Use `APEX_APPLICATION_ADMIN.SET_GLOBAL_NOTIFICATION` instead.
-- ___
-- This procedure is used to set the global notification message which is the message displayed in
-- the page #GLOBAL_NOTIFICATION# substitution string.
--
-- Parameters:
-- * p_application_id:              The Application ID.
-- * p_global_notification_message: Text string to be used for the global notification message.
--
-- Example:
--
--     begin
--         apex_util.set_global_notification(
--             p_application_id              => 117,
--             p_global_notification_message => 'This application will be upgraded this weekend at 2100 UTC' );
--     end;
--
-- See Also:
-- APEX_APPLICATION_ADMIN.SET_GLOBAL_NOTIFICATION
-- GET_GLOBAL_NOTIFICATION
--==============================================================================
procedure set_global_notification(
   p_application_id               in number,
   p_global_notification_message  in varchar2 );

--==============================================================================
-- Deprecated: 23.1. Use `APEX_APPLICATION_ADMIN.GET_GLOBAL_NOTIFICATION` instead.
-- ___
-- This function is used to get the global notification message which is the message displayed in
-- the page #GLOBAL_NOTIFICATION# substitution string.
--
-- Parameters:
-- * p_application_id:              The Application ID.
--
-- Example:
--
--     declare
--         l_global_notification varchar2(100);
--
--     begin
--         l_global_notification := apex_util.get_global_notification(
--                                      p_application_id => 117 );
--         dbms_output.put_line( 'The current global notification is: ' || l_global_notification );
--     end;
--
-- See Also:
-- APEX_APPLICATION_ADMIN.GET_GLOBAL_NOTIFICATION
-- SET_GLOBAL_NOTIFICATION
--==============================================================================
function get_global_notification(
   p_application_id   in number )
return varchar2;


--==============================================================================
-- Deprecated: 23.1. Use `APEX_APPLICATION_ADMIN.SET_BUILD_STATUS` instead.
-- ___
-- This procedure sets the build status of the specified application.
--
-- Parameters:
-- * p_application_id: The Application ID.
-- * p_build_status:   New build status to set application to.  Values include:
--                       - `RUN_AND_BUILD` - The application can be run and can also be edited by
--                                           developers.
--                       - `RUN_ONLY` - The application can be run but cannot be edited by developers.
--
-- Example:
--
--     begin
--         apex_util.set_app_build_status(
--             p_application_id => 170,
--             p_build_status   => 'RUN_ONLY' );
--         commit;
--     end;
--
-- See Also:
-- APEX_APPLICATION_ADMIN.SET_BUILD_STATUS
--==============================================================================
procedure set_app_build_status(
   p_application_id       in number,
   p_build_status         in varchar2 );


--==============================================================================
-- Internal:
-- Deprecated: 5.1.  Use `APEX_UTIL.GET_SINCE` instead.
--
-- Used to get a string representation of the time passed since the input
-- timestamp with time zone
--
-- PARAMETERS
--   p_timestamp: timestamp with time zone  that is used to compute time since p_timestamp and systimestamp
--   p_short: optional parameter to get short since format. e.g. "10h" instead of "10 hours ago".
--==============================================================================
function get_since_tswtz(
    p_timestamp in timestamp with time zone,
    p_short     in varchar2 default 'N' )
    return varchar2;

--==============================================================================
-- Internal:
-- Deprecated: 5.1.  Use `APEX_UTIL.GET_SINCE` instead.
--
-- Used to get a string representation of the time passed since the input
-- timestamp with local time zone
--
-- PARAMETERS
--   p_ltimestamp: timestamp with local time zone  that is used to compute time since p_ltimestamp and localtimestamp
--   p_short: optional parameter to get short since format. e.g. "10h" instead of "10 hours ago".
--==============================================================================
function get_since_tswltz(
    p_ltimestamp in timestamp with local time zone,
    p_short      in varchar2 default 'N' )
    return varchar2;

--==============================================================================
-- This function gets supporting object scripts defined in an application.
-- Note: The workspace ID must be set before the call.
--
-- Parameters:
-- * p_application_id: The application ID to get supporting objects from.
-- * p_script_type:    The supporting objects script type.
--                     Valid values are:
--                     - `apex_util.c_install_script`
--                     - `apex_util.c_upgrade_script`
--                     - `apex_util. c_deinstall_script`
--
-- Example:
-- The following example shows how to set the workspace ID for workspace 'FRED' and how to get the
-- supporting objects from application ID 100 in this workspace.
--
--     declare
--         l_install_script   clob;
--         l_upgrade_script   clob;
--         l_deinstall_script clob;
--     begin
--         apex_util.set_workspace( p_workspace => 'FRED');
--
--         l_install_script := apex_util.get_supporting_object_script(
--                                 p_application_id => 100,
--                                 p_script_type => apex_util.c_install_script );
--         l_upgrade_script := apex_util.get_supporting_object_script(
--                                 p_application_id => 100,
--                                 p_script_type => apex_util.c_upgrade_script );
--         l_deinstall_script := apex_util.get_supporting_object_script(
--                                   p_application_id => 100,
--                                   p_script_type => apex_util.c_deinstall_script );
--     end;
--
-- See Also:
-- GET_SUPPORTING_OBJECT_SCRIPT Procedure
--==============================================================================
function get_supporting_object_script(
    p_application_id  in number,
    p_script_type     in varchar2 ) return clob;

--==============================================================================
-- This procedure gets supporting object scripts and outputs to `sys.dbms_output` buffer or
-- download as a file.
-- Note: The workspace ID must be set before the call.
--
-- Parameters:
-- * p_application_id: The application ID to get supporting objects from.
-- * p_script_type:    The supporting objects script type.
--                     Valid values are:
--                     - `apex_util.c_install_script`
--                     - `apex_util.c_upgrade_script`
--                     - `apex_util. c_deinstall_script`
-- * p_output_type:   The script can be output to `sys.dbms_output` buffer or download as a file.
--                     Valid values are `c_output_as_dbms_output` and `c_output_as_file`.
--                     The default is `c_output_as_dbms_output`.
--
-- Example:
-- The following example shows how to set workspace ID for workspace 'FRED', then get the install script
-- from application ID 100 and output to the command-line buffer.
--
--     set serveroutput on;
--     begin
--         apex_util.set_workspace( p_workspace => 'FRED');
--         apex_util.get_supporting_object_script(
--             p_application_id => 100,
--             p_script_type    => apex_util.c_install_script );
--     end;
--
-- Example:
-- The following example shows how to download the upgrade script file from application ID 100 in the
-- browser. This is useful if the script needs to be downloaded using an application process.
--
--     begin
--         apex_util.set_workspace( p_workspace => 'FRED');
--         apex_util.get_supporting_object_script(
--             p_application_id => 100,
--             p_script_type    => apex_util.c_upgrade_script,
--             p_output_type    => apex_util.c_output_as_file );
--     end;
--
-- See Also:
-- GET_SUPPORTING_OBJECT_SCRIPT Function
--==============================================================================
procedure get_supporting_object_script(
    p_application_id  in number,
    p_script_type     in varchar2,
    p_output_type     in varchar2 default c_output_as_dbms_output );

--==============================================================================
-- Internal:
-- Deprecated: 24.2
-- Returns Timeframe values in seconds (30 min, 1 hour, ...) which can be used
-- in a LOV by entering
--
-- select disp,
--        val
--   from table( apex_util.get_timeframe_lov_data )
--  order by insert_order
--
--==============================================================================
function get_timeframe_lov_data (
    p_max_days_ago in pls_integer default 28 )
    return wwv_flow_t_temp_lov_data;

--==============================================================================
-- This procedure prints a given CLOB to the HTP buffer.
--
-- Parameters:
-- * p_clob:   The CLOB.
-- * p_escape: If `TRUE` (default), escape special characters, using
--             apex_escape.html.
--
-- Example:
-- The following example prints l_clob and escape special characters
--
--     declare
--         l_clob clob := '<script>alert(1)</script>';
--     begin
--         apex_util.prn (
--             p_clob   => l_clob,
--             p_escape => true );
--     end;
--==============================================================================
procedure prn (
    p_clob   in clob,
    p_escape in boolean default true );

--==============================================================================
-- Internal:
-- Deprecated: 24.2
-- Commit transaction in a safe way that supports Application Continuity.
--==============================================================================
procedure do_commit;

--==============================================================================
-- This function returns the name of the schema containing the APEX engine.
--
-- Returns:
-- The schema name which contains the APEX engine.
--
-- Example:
--
--     declare
--         l_apex_owner varchar2(255);
--     begin
--         l_apex_owner := apex_util.get_apex_owner;
--     end;
--==============================================================================
function get_apex_owner return varchar2;

end htmldb_util;
/
show errors
