Rem
Rem  Copyright (c) 2006, 2024, Oracle and/or its affiliates.
Rem



Rem    NAME
Rem      apex_install_pe_data.sql
Rem


Rem    DESCRIPTION
Rem      Insert Property Editor metadata into wwv_pe_* tables during APEX installation
Rem


Rem    MODIFIED     (MM/DD/YYYY)
Rem    arayner       05/13/2013 - Created

set define '^'
set sqlprefix off
set serveroutput on

prompt Enter YES (or hit enter) to load system messages and NO to skip it (NO should only be used during installation)
define LOAD_SYSTEM_MESSAGES = '^1'

prompt
prompt Property Editor - Metadata
prompt
declare
/* Useful SQL queries:

Find gaps in the property id's:

select g.id
  from ( select rownum id
           from sys.dual
        connect by level <= 1000 ) g
 where not exists ( select 1
                      from wwv_pe_properties p
                     where p.id = g.id
                  )
 order by 1;

Find existing properties that could be re-used:

select p.name,
       wwv_flow_lang.system_message(p.prompt) as property_prompt,
       p.property_type,
       p.db_column,
       wwv_flow_lang.system_message(dg.title) as group_title,
       p.*
  from wwv_pe_properties p,
       wwv_pe_display_groups dg
 where p.name like '%FEEDB%'
   and dg.id = p.display_group_id
 order by p.name;

Show all properties of a component type (useful to verify the order and groups):

select wwv_flow_lang.system_message(dg.title) as group_title,
       wwv_flow_lang.system_message(p.prompt) as property_prompt,
       p.name,
       coalesce( cp.db_column, p.db_column )  as db_column
  from wwv_pe_component_types ct,
       wwv_pe_component_props cp,
       wwv_pe_properties p,
       wwv_pe_display_groups dg
 where ct.name              = 'CHANGEME'
   and cp.component_type_id = ct.id
   and p.id                 = cp.property_id
   and dg.id                = p.display_group_id
  order by cp.display_sequence;

Show columns that are missing in the current configuration:

select ct.name, ct.table_name, c.column_name
  from wwv_pe_component_types ct,
       user_tab_cols c
 where c.table_name = ct.table_name
   and c.table_name not in ( 'WWV_FLOW_FILTERS_DEV',
                             'WWV_FLOW_FND_USER_GROUPS'
                           )
   and ct.name      not in ( 'DATA_LOAD_TABLE',
                             'WEB_SRC_SYNC_STEP_WS_PARAM',
                             'AUTOMATION_WS_PARAM',
                             'LOV_WS_PARAM',
                             'CALENDAR_TEMPLATE', -- legacy calendar
                             'PAGE',
                             'TREE' ) -- legacy tree
   and ct.is_query_only  = 'N'
   and ct.page_id_column is null -- no page meta data, they are valid
   and not exists ( select 1
                      from wwv_pe_component_props cp,
                           wwv_pe_properties p
                     where cp.component_type_id = ct.id
                       and p.id                 = cp.property_id
                       and coalesce( cp.db_column, p.db_column ) = c.column_name
                  )
   and c.column_name not in ( 'ID',
                              'FLOW_ID',
                              'PAGE_ID',
                              'SECURITY_GROUP_ID',
                              'CREATED_BY',
                              'CREATED_ON',
                              'APPLICATION_TYPE',
                              'EMAIL_TEMPLATE_ID',
                              'TASK_DEF_ID',
                              'WEB_SRC_MODULE_ID',
                              'WEB_SRC_OPERATION_ID',
                              'WORKFLOW_ID',
                              'FILE_CONTENT',
                              'LAST_REFRESH',
                              'LAST_REFRESHED'
                            )
   and not (   ct.table_name = 'WWV_FLOWS'
           and c.column_name like 'DEFAULT\_%' escape '\' -- not used anymore
           )
   and c.column_name not like 'ATTRIBUTE\___' escape '\'
   and c.column_name not like '%\_COMPILED' escape '\'
   and c.column_name not like 'SYS\_NC%' escape '\'
   and c.column_name not like '%\_RUN\_TIMESTAMP' escape '\'
   and c.column_name not like 'ATTRIBUTE_' -- build_option
   and (c.column_name <> ct.parent_fk_column or ct.parent_fk_column is null)
 order by ct.name, c.column_name;

*/
    type t_parent_component_types is table of number index by binary_integer;
    type t_used_system_messages is table of boolean index by wwv_flow_messages$.name%type;

    c_load_system_messages          constant boolean := ( nvl( upper( '^LOAD_SYSTEM_MESSAGES' ), 'YES' ) = 'YES' );
    ------------------------------------------------------------------------
    -- Component Type Constants
    ------------------------------------------------------------------------
    -- Workspace component types
    c_comp_type_component_type      constant wwv_pe_component_types.id%type :=   1;
    c_comp_type_credential          constant wwv_pe_component_types.id%type :=   10;
    c_comp_type_remote_database     constant wwv_pe_component_types.id%type :=   20;
    c_comp_type_remote_oracle_db    constant wwv_pe_component_types.id%type :=   22;
    c_comp_type_auth_server         constant wwv_pe_component_types.id%type :=   30;
    c_comp_type_print_server        constant wwv_pe_component_types.id%type :=   40;
    c_comp_type_rest_server         constant wwv_pe_component_types.id%type :=   50;
    c_comp_type_workspace_file      constant wwv_pe_component_types.id%type :=   60;
    c_comp_type_app_group           constant wwv_pe_component_types.id%type :=   70;
    c_comp_type_file_server         constant wwv_pe_component_types.id%type :=   80;
    c_comp_type_ai_server           constant wwv_pe_component_types.id%type :=   90;
    c_comp_type_vector_provider     constant wwv_pe_component_types.id%type :=   92;
    c_comp_type_page_pattern        constant wwv_pe_component_types.id%type :=  100;
    --
    -- Application component types
    c_comp_type_application         constant wwv_pe_component_types.id%type := 1000;
    c_comp_type_app_substitution    constant wwv_pe_component_types.id%type := 1005;
    --
    -- Theme related component types
    c_comp_type_theme               constant wwv_pe_component_types.id%type := 2000;
    c_comp_type_theme_style         constant wwv_pe_component_types.id%type := 2010;
    c_comp_type_theme_file          constant wwv_pe_component_types.id%type := 2020;
    c_comp_type_tmpl_option_group   constant wwv_pe_component_types.id%type := 2030;
    c_comp_type_global_tmpl_option  constant wwv_pe_component_types.id%type := 2040;
    --
    -- Templates
    c_comp_type_page_template       constant wwv_pe_component_types.id%type := 2510;
    c_comp_type_page_tmpl_option    constant wwv_pe_component_types.id%type := 2512;
    c_comp_type_display_point       constant wwv_pe_component_types.id%type := 2514;
    c_comp_type_field_template      constant wwv_pe_component_types.id%type := 2520;
    c_comp_type_field_tmpl_option   constant wwv_pe_component_types.id%type := 2522;
    c_comp_type_button_template     constant wwv_pe_component_types.id%type := 2530;
    c_comp_type_button_tmpl_option  constant wwv_pe_component_types.id%type := 2532;
    c_comp_type_region_template     constant wwv_pe_component_types.id%type := 2540;
    c_comp_type_region_tmpl_option  constant wwv_pe_component_types.id%type := 2542;
    c_comp_type_list_template       constant wwv_pe_component_types.id%type := 2550;
    c_comp_type_list_tmpl_option    constant wwv_pe_component_types.id%type := 2552;
    c_comp_type_bc_template         constant wwv_pe_component_types.id%type := 2560;
    c_comp_type_bc_tmpl_option      constant wwv_pe_component_types.id%type := 2562;
    c_comp_type_calendar_template   constant wwv_pe_component_types.id%type := 2570;
    c_comp_type_report_template     constant wwv_pe_component_types.id%type := 2580;
    c_comp_type_report_tmpl_option  constant wwv_pe_component_types.id%type := 2582;
    c_comp_type_popup_lov_template  constant wwv_pe_component_types.id%type := 2590;
    --
    -- Shared component related component types
    c_comp_type_app_item            constant wwv_pe_component_types.id%type := 3010;
    c_comp_type_app_computation     constant wwv_pe_component_types.id%type := 3020;
    c_comp_type_app_process         constant wwv_pe_component_types.id%type := 3030;
    c_comp_type_build_option        constant wwv_pe_component_types.id%type := 3040;
    c_comp_type_authentication      constant wwv_pe_component_types.id%type := 3050;
    c_comp_type_authorization       constant wwv_pe_component_types.id%type := 3060;
    c_comp_type_acl_role            constant wwv_pe_component_types.id%type := 3070;
    --
    c_comp_type_web_src_module      constant wwv_pe_component_types.id%type := 3080;
    c_comp_type_web_src_array_col   constant wwv_pe_component_types.id%type := 3081;
    c_comp_type_web_src_mod_param   constant wwv_pe_component_types.id%type := 3082;
    c_comp_type_web_src_mod_dp      constant wwv_pe_component_types.id%type := 3084;
    c_comp_type_web_src_mod_dp_col  constant wwv_pe_component_types.id%type := 3086;
    c_comp_type_web_src_operation   constant wwv_pe_component_types.id%type := 3088;
    c_comp_type_web_src_oper_param  constant wwv_pe_component_types.id%type := 3090;
    c_comp_type_web_src_oper_dp     constant wwv_pe_component_types.id%type := 3092;
    c_comp_type_web_src_oper_dpcol  constant wwv_pe_component_types.id%type := 3094;
    c_comp_type_web_src_sync_step   constant wwv_pe_component_types.id%type := 3096;
    c_comp_type_web_src_sync_param  constant wwv_pe_component_types.id%type := 3098;  -- web source sync step parameter
    c_comp_type_web_src_query       constant wwv_pe_component_types.id%type := 3100;
    c_comp_type_web_src_param       constant wwv_pe_component_types.id%type := 3102;
    --
    c_comp_type_shortcut            constant wwv_pe_component_types.id%type := 3105;
    --
    c_comp_type_plugin              constant wwv_pe_component_types.id%type := 3110;
    c_comp_type_plugin_std_attr     constant wwv_pe_component_types.id%type := 3112;
    c_comp_type_plugin_attr_grp     constant wwv_pe_component_types.id%type := 3115;
    c_comp_type_plugin_cust_attr    constant wwv_pe_component_types.id%type := 3120;
    c_comp_type_plugin_attr_lov     constant wwv_pe_component_types.id%type := 3125;
    c_comp_type_plugin_action_pos   constant wwv_pe_component_types.id%type := 3130;
    c_comp_type_plugin_action_tmpl  constant wwv_pe_component_types.id%type := 3135;
    c_comp_type_plugin_event        constant wwv_pe_component_types.id%type := 3140;
    c_comp_type_plugin_file         constant wwv_pe_component_types.id%type := 3142;
    c_comp_type_plugin_setting      constant wwv_pe_component_types.id%type := 3144;
    c_comp_type_plugin_slot         constant wwv_pe_component_types.id%type := 3146;
    --
    c_comp_type_application_file    constant wwv_pe_component_types.id%type := 3150;
    c_comp_type_text_message        constant wwv_pe_component_types.id%type := 3170;
    c_comp_type_app_setting         constant wwv_pe_component_types.id%type := 3180;
    --
    c_comp_type_pwa_shortcut        constant wwv_pe_component_types.id%type := 3210;
    c_comp_type_pwa_screenshot      constant wwv_pe_component_types.id%type := 3220;
    --
    c_comp_type_dualityview_src     constant wwv_pe_component_types.id%type := 3310;
    c_comp_type_jsondocument_src    constant wwv_pe_component_types.id%type := 3312;
    c_comp_type_dualityvw_arr_col   constant wwv_pe_component_types.id%type := 3314;
    c_comp_type_jsoncoll_arr_col    constant wwv_pe_component_types.id%type := 3316;
    c_comp_type_dualityvw_dp        constant wwv_pe_component_types.id%type := 3317;
    c_comp_type_dualityvw_dp_col    constant wwv_pe_component_types.id%type := 3318;
    c_comp_type_jsoncoll_dp         constant wwv_pe_component_types.id%type := 3319;
    c_comp_type_jsoncoll_dp_col     constant wwv_pe_component_types.id%type := 3320;
    --
    c_comp_type_ai_config           constant wwv_pe_component_types.id%type := 3400;
    c_comp_type_ai_config_rag_src   constant wwv_pe_component_types.id%type := 3405;
    --
    c_comp_type_classic_nav_bar     constant wwv_pe_component_types.id%type := 3505;
    c_comp_type_breadcrumb          constant wwv_pe_component_types.id%type := 3510;
    c_comp_type_breadcrumb_entry    constant wwv_pe_component_types.id%type := 3515;
    c_comp_type_list                constant wwv_pe_component_types.id%type := 3520;
    c_comp_type_list_entry          constant wwv_pe_component_types.id%type := 3525;
    c_comp_type_lov                 constant wwv_pe_component_types.id%type := 3530;
    c_comp_type_lov_ws_param        constant wwv_pe_component_types.id%type := 3532;  -- web source lov parameter
    c_comp_type_lov_entry           constant wwv_pe_component_types.id%type := 3534;
    c_comp_type_lov_column          constant wwv_pe_component_types.id%type := 3536;
    c_comp_type_ws_ref              constant wwv_pe_component_types.id%type := 3540;
    c_comp_type_ws_ref_operations   constant wwv_pe_component_types.id%type := 3541;
    c_comp_type_ws_ref_oper_p_i     constant wwv_pe_component_types.id%type := 3545;  -- web service input parameters
    c_comp_type_ws_ref_oper_p_o     constant wwv_pe_component_types.id%type := 3546;  -- web service output parameters
    c_comp_type_ws_ref_oper_p_a     constant wwv_pe_component_types.id%type := 3547;  -- web service basic auth parameters
    c_comp_type_ws_ref_oper_p_h     constant wwv_pe_component_types.id%type := 3548;  -- web service http header parameters
    c_comp_type_legacy_data_load    constant wwv_pe_component_types.id%type := 3550;
    -- placeholder: child tables of data load
    c_comp_type_tab_set             constant wwv_pe_component_types.id%type := 3560;
    c_comp_type_report_layout       constant wwv_pe_component_types.id%type := 3570;
    c_comp_type_report_query        constant wwv_pe_component_types.id%type := 3580;
    c_comp_type_report_query_stmt   constant wwv_pe_component_types.id%type := 3585;
    -- placeholder: report layout details
    c_comp_type_automation          constant wwv_pe_component_types.id%type := 3610;
    c_comp_type_autom_ws_param      constant wwv_pe_component_types.id%type := 3612;
    c_comp_type_automation_action   constant wwv_pe_component_types.id%type := 3620;
    c_comp_type_email_template      constant wwv_pe_component_types.id%type := 3630;
    c_comp_type_data_load_def       constant wwv_pe_component_types.id%type := 3640;
    c_comp_type_data_load_dp        constant wwv_pe_component_types.id%type := 3645;
    c_comp_type_data_load_dp_col    constant wwv_pe_component_types.id%type := 3647;
    -- Task definition component type
    c_comp_type_task_def            constant wwv_pe_component_types.id%type := 3700;
    c_comp_type_task_def_param      constant wwv_pe_component_types.id%type := 3705;
    c_comp_type_task_def_action     constant wwv_pe_component_types.id%type := 3710;
    c_comp_type_task_def_particip   constant wwv_pe_component_types.id%type := 3715;
    --
    c_comp_type_language_map        constant wwv_pe_component_types.id%type := 3890;
    c_comp_type_dyn_translation     constant wwv_pe_component_types.id%type := 3891;
    --
    c_comp_type_concatenated_file   constant wwv_pe_component_types.id%type := 3905;
    c_comp_type_search_config       constant wwv_pe_component_types.id%type := 3910;
    c_comp_type_search_ws_param     constant wwv_pe_component_types.id%type := 3912;
    --
    -- Misc component types
    c_comp_type_page_group          constant wwv_pe_component_types.id%type := 4010;
    c_comp_type_tree                constant wwv_pe_component_types.id%type := 4015;
    --
    -- Supporting Objects
    c_comp_type_supobj              constant wwv_pe_component_types.id%type := 4950;
    c_comp_type_supobj_substitut    constant wwv_pe_component_types.id%type := 4955;
    c_comp_type_supobj_build_opt    constant wwv_pe_component_types.id%type := 4960;
    c_comp_type_supobj_validation   constant wwv_pe_component_types.id%type := 4965;
    c_comp_type_supobj_install      constant wwv_pe_component_types.id%type := 4970;
    c_comp_type_supobj_upgrade      constant wwv_pe_component_types.id%type := 4975;
    c_comp_type_supobj_deinstall    constant wwv_pe_component_types.id%type := 4980;
    --
    -- Page related component types
    --
    c_comp_type_page                constant wwv_pe_component_types.id%type := 5000;
    -- Page rendering
    c_comp_type_region              constant wwv_pe_component_types.id%type := 5110;
    c_comp_type_page_item           constant wwv_pe_component_types.id%type := 5120;
    c_comp_type_button              constant wwv_pe_component_types.id%type := 5130;
    c_comp_type_da_event            constant wwv_pe_component_types.id%type := 5140;
    c_comp_type_da_action           constant wwv_pe_component_types.id%type := 5150;
    c_comp_type_meta_tag            constant wwv_pe_component_types.id%type := 5160;
    -- Page processing
    c_comp_type_validation          constant wwv_pe_component_types.id%type := 5510;
    c_comp_type_page_computation    constant wwv_pe_component_types.id%type := 5520;
    c_comp_type_page_process        constant wwv_pe_component_types.id%type := 5530;
    c_comp_type_page_proc_ws_param  constant wwv_pe_component_types.id%type := 5531;
    c_comp_type_branch              constant wwv_pe_component_types.id%type := 5540;
    --
    -- Region type specific component types ( use a new 7xxx hundred for each region type )
    c_comp_type_region_plugin_attr  constant wwv_pe_component_types.id%type := 7000;
    --
    c_comp_type_ir_attributes       constant wwv_pe_component_types.id%type := 7010;
    c_comp_type_ir_saved_report     constant wwv_pe_component_types.id%type := 7020;
    c_comp_type_ir_column_group     constant wwv_pe_component_types.id%type := 7030;
    c_comp_type_ir_column           constant wwv_pe_component_types.id%type := 7040;
    c_comp_type_ir_print            constant wwv_pe_component_types.id%type := 7050;
    --
    c_comp_type_page_proc_ws_p_i    constant wwv_pe_component_types.id%type := 7110;  -- web service input parameters
    c_comp_type_page_proc_ws_p_o    constant wwv_pe_component_types.id%type := 7120;  -- web service output parameters
    c_comp_type_page_proc_ws_p_a    constant wwv_pe_component_types.id%type := 7130;  -- web service basic auth parameters
    c_comp_type_page_proc_ws_p      constant wwv_pe_component_types.id%type := 7150;  -- web source process parameter
    c_comp_type_pg_proc_api_param   constant wwv_pe_component_types.id%type := 7152;  -- Invoke API process parameter
    c_comp_type_region_ws_param     constant wwv_pe_component_types.id%type := 7160;  -- web source region parameter
    --
    c_comp_type_classic_report      constant wwv_pe_component_types.id%type := 7310;
    c_comp_type_classic_rpt_column  constant wwv_pe_component_types.id%type := 7320;
    c_comp_type_classic_rpt_print   constant wwv_pe_component_types.id%type := 7330;
    --
    c_comp_type_tab_form            constant wwv_pe_component_types.id%type := 7410;
    c_comp_type_tab_form_column     constant wwv_pe_component_types.id%type := 7420;
    c_comp_type_tab_form_print      constant wwv_pe_component_types.id%type := 7430;
    --
    c_comp_type_classic_calendar    constant wwv_pe_component_types.id%type := 7610;
    --
    c_comp_type_region_column       constant wwv_pe_component_types.id%type := 7710;
    c_comp_type_region_tmpl_column  constant wwv_pe_component_types.id%type := 7720;
    -- JET Charts
    c_comp_type_jet_chart           constant wwv_pe_component_types.id%type := 7810;
    c_comp_type_jet_chart_series    constant wwv_pe_component_types.id%type := 7820;
    c_comp_type_jet_chart_axes      constant wwv_pe_component_types.id%type := 7830;
    c_comp_type_jet_chart_ws_param  constant wwv_pe_component_types.id%type := 7840;  -- web source jet chart series parameter
    -- interactive grid
    c_comp_type_ig_attributes       constant wwv_pe_component_types.id%type := 7910;
    c_comp_type_ig_saved_report     constant wwv_pe_component_types.id%type := 7920;
    c_comp_type_ig_column_group     constant wwv_pe_component_types.id%type := 7930;
    c_comp_type_ig_column           constant wwv_pe_component_types.id%type := 7940;
    c_comp_type_ig_print            constant wwv_pe_component_types.id%type := 7950;
    -- faceted search
    c_comp_type_facet               constant wwv_pe_component_types.id%type := 8010;
    c_comp_type_facet_group         constant wwv_pe_component_types.id%type := 8020;
    c_comp_type_facet_group_item    constant wwv_pe_component_types.id%type := 8030;
    -- smart filters
    c_comp_type_sfilter             constant wwv_pe_component_types.id%type := 8050;
    c_comp_type_sfilter_group       constant wwv_pe_component_types.id%type := 8051;
    c_comp_type_sfilter_group_item  constant wwv_pe_component_types.id%type := 8052;
    -- card
    c_comp_type_card                constant wwv_pe_component_types.id%type := 8110;
    c_comp_type_card_action         constant wwv_pe_component_types.id%type := 8120;
    --
    c_comp_type_map_region          constant wwv_pe_component_types.id%type := 8210;
    c_comp_type_map_layer           constant wwv_pe_component_types.id%type := 8220;
    c_comp_type_map_layer_ws_param  constant wwv_pe_component_types.id%type := 8230;  -- web source map layer parameter
    c_comp_type_map_background      constant wwv_pe_component_types.id%type := 8240;
    --
    c_comp_type_pg_pr_task_param    constant wwv_pe_component_types.id%type := 8400;   -- task page process parameter
    --
    c_comp_type_search_region       constant wwv_pe_component_types.id%type := 8510;
    c_comp_type_search_source       constant wwv_pe_component_types.id%type := 8520;
    --
    c_comp_type_region_action       constant wwv_pe_component_types.id%type := 8610;
    c_comp_type_ir_column_action    constant wwv_pe_component_types.id%type := 8615;
    --
    c_comp_type_reg_act_menu_entry  constant wwv_pe_component_types.id%type := 8710;
    c_comp_type_ir_act_menu_entry   constant wwv_pe_component_types.id%type := 8715;
    --
    c_comp_type_workflow            constant wwv_pe_component_types.id%type := 8800;
    c_comp_type_workflow_param      constant wwv_pe_component_types.id%type := 8810;
    c_comp_type_workflow_version    constant wwv_pe_component_types.id%type := 8820;
    c_comp_type_workflow_activity   constant wwv_pe_component_types.id%type := 8830;
    c_comp_type_workflow_variable   constant wwv_pe_component_types.id%type := 8840;
    c_comp_type_workflow_act_var    constant wwv_pe_component_types.id%type := 8850;
    c_comp_type_workflow_particip   constant wwv_pe_component_types.id%type := 8860;
    c_comp_type_workflow_trans      constant wwv_pe_component_types.id%type := 8870;
    c_comp_type_workflow_branch     constant wwv_pe_component_types.id%type := 8880;
    --
    c_comp_type_page_proc_wf_param  constant wwv_pe_component_types.id%type := 8900;  -- workflow page process parameter
    c_comp_type_wf_act_task_param   constant wwv_pe_component_types.id%type := 8950;  -- task workflow activity parameter
    c_comp_type_wf_act_api_param    constant wwv_pe_component_types.id%type := 9000;  -- Invoke API workflow activity parameter
    c_comp_type_wf_act_ws_param     constant wwv_pe_component_types.id%type := 9050;  -- Web Source Workflow Activity Parameter
    c_comp_type_wf_act_wf_param     constant wwv_pe_component_types.id%type := 9100;  -- workflow component parameter for Invoke Workflow Activity

    c_comp_type_component_group     constant wwv_pe_component_types.id%type := 9990;  -- Component Group
    c_comp_type_comp_grp_component  constant wwv_pe_component_types.id%type := 9995;  -- Component Group Component
    --

    ------------------------------------------------------------------------
    -- Unit constants
    ------------------------------------------------------------------------
    c_unit_pixels         constant varchar2( 20 ) := 'pixels';
    c_unit_chars          constant varchar2( 20 ) := 'characters';
    c_unit_lines          constant varchar2( 20 ) := 'lines';
    c_unit_degrees        constant varchar2( 20 ) := 'degrees';
    c_unit_seconds        constant varchar2( 20 ) := 'seconds';
    c_unit_days           constant varchar2( 20 ) := 'days';
    c_unit_kb             constant varchar2( 20 ) := 'KB';
    c_unit_milliseconds   constant varchar2( 20 ) := 'milliseconds';

    ------------------------------------------------------------------------
    -- Oracle DB version constants
    ------------------------------------------------------------------------
    c_db_23ai             constant number := 23;

    ------------------------------------------------------------------------
    -- JET Chart constants
    ------------------------------------------------------------------------
    c_jet_area                constant varchar2( 1000 ) := 'area';
    c_jet_bar                 constant varchar2( 1000 ) := 'bar';
    c_jet_box_plot            constant varchar2( 1000 ) := 'boxPlot';
    c_jet_bubble              constant varchar2( 1000 ) := 'bubble';
    c_jet_combo               constant varchar2( 1000 ) := 'combo';
    c_jet_dial                constant varchar2( 1000 ) := 'dial';
    c_jet_donut               constant varchar2( 1000 ) := 'donut';
    c_jet_funnel              constant varchar2( 1000 ) := 'funnel';
    c_jet_gantt               constant varchar2( 1000 ) := 'gantt';

    c_jet_line                constant varchar2( 1000 ) := 'line';
    c_jet_line_with_area      constant varchar2( 1000 ) := 'lineWithArea';
    c_jet_pie                 constant varchar2( 1000 ) := 'pie';
    c_jet_polar               constant varchar2( 1000 ) := 'polar';
    c_jet_pyramid             constant varchar2( 1000 ) := 'pyramid';
    c_jet_radar               constant varchar2( 1000 ) := 'radar';
    c_jet_range               constant varchar2( 1000 ) := 'range';
    c_jet_scatter             constant varchar2( 1000 ) := 'scatter';
    c_jet_stock               constant varchar2( 1000 ) := 'stock';
    c_jet_pie_donut           constant varchar2 (1000 ) := c_jet_pie||','|| c_jet_donut;
    c_jet_single              constant varchar2( 1000 ) := c_jet_area ||','|| c_jet_bar||','|| c_jet_line||','|| c_jet_line_with_area||','|| c_jet_bubble||','|| c_jet_funnel||','|| c_jet_pyramid||','|| c_jet_pie_donut||','|| c_jet_scatter||','|| c_jet_stock||','|| c_jet_dial;
    --
    c_jet_stack               constant varchar2( 1000 ) := c_jet_area || ',' || c_jet_bar || ',' || c_jet_combo || ',' || c_jet_line || ',' || c_jet_line_with_area;
    --c_jet_range               constant varchar2( 1000 ) := c_jet_area || ',' || c_jet_bar;

    c_jet_spark               constant varchar2( 1000 ) := c_jet_area || ',' || c_jet_bar || ',' || c_jet_line || ',' || c_jet_line_with_area;
    c_jet_time_axis_type      constant varchar2( 1000 ) := c_jet_area || ',' || c_jet_bar|| ',' || c_jet_combo || ',' || c_jet_line || ',' || c_jet_line_with_area || ',' || c_jet_stock  || ',' ||c_jet_box_plot;
    c_jet_overview            constant varchar2( 1000 ) := c_jet_area || ',' || c_jet_bar|| ',' || c_jet_combo || ',' || c_jet_line || ',' || c_jet_stock || ',' || c_jet_box_plot || ',' || c_jet_line_with_area;
    c_jet_value_column        constant varchar2( 1000 ) := c_jet_area ||','|| c_jet_bar ||','|| c_jet_combo ||','|| c_jet_dial ||','|| c_jet_funnel ||','|| c_jet_pyramid ||','|| c_jet_line ||','|| c_jet_line_with_area ||','|| c_jet_pie_donut || ',' ||c_jet_box_plot;
    c_jet_series_column       constant varchar2( 1000 ) := c_jet_area ||','|| c_jet_bar ||','|| c_jet_bubble ||','|| c_jet_combo ||','|| c_jet_funnel ||','|| c_jet_pyramid ||','|| c_jet_line ||','|| c_jet_line_with_area ||','|| c_jet_scatter;
    --c_jet_group_column        constant varchar2( 1000 ) := c_jet_area ||','|| c_jet_bar ||','|| c_jet_bubble ||','|| c_jet_combo ||','|| c_jet_line ||','|| c_jet_line_with_area ||','|| c_jet_scatter ||','|| c_jet_stock;
    c_jet_group_column        constant varchar2( 1000 ) := c_jet_area ||','|| c_jet_bar ||','|| c_jet_bubble ||','|| c_jet_line ||','|| c_jet_line_with_area ||','|| c_jet_scatter ||','|| c_jet_stock || ',' ||c_jet_box_plot;
    c_jet_common_series       constant varchar2( 1000 ) := c_jet_area || ',' || c_jet_bar || ',' || c_jet_combo || ',' ||  c_jet_line || ',' || c_jet_line_with_area;
    c_jet_line_charts         constant varchar2( 1000 ) := c_jet_line ||','|| c_jet_line_with_area ||','|| c_jet_combo ||','||c_jet_area || ',' || c_jet_polar || ',' || c_jet_radar || ',' || c_jet_scatter || ',' ||c_jet_box_plot;
    c_jet_line_style_charts   constant varchar2( 1000 ) := c_jet_line ||','|| c_jet_line_with_area ||','|| c_jet_combo ||','|| c_jet_bubble ||','|| c_jet_scatter;
    c_jet_marker_charts       constant varchar2( 1000 ) := c_jet_line ||','|| c_jet_line_with_area ||','|| c_jet_scatter ||','|| c_jet_area ||','|| c_jet_bubble || ','|| c_jet_combo || ',' ||c_jet_box_plot;
    c_jet_no_axes             constant varchar2( 1000 ) := c_jet_dial ||','|| c_jet_funnel ||','|| c_jet_pyramid ||','|| c_jet_pie_donut;
    c_jet_no_sorting          constant varchar2( 1000 ) := c_jet_stock ||','|| c_jet_gantt ||','|| c_jet_dial  ||','|| c_jet_funnel ||','|| c_jet_pyramid  ||','|| c_jet_scatter ||','|| c_jet_pie||','|| c_jet_donut;
    --
    c_data_profile_lookup_example constant varchar2( 32767 ) := q'~
<pre>
ENAME  EMPNO  SAL    DNAME
------ ------ ------ ------------
  7839 KING     5000 ACCOUNTING
  7788 SCOTT    3000 RESEARCH
     : :           : :
</pre>
<ul>
<li>Configure the <strong>DNAME_LOADED</strong> column, as a <strong>Data</strong> column with the selector of <strong>DNAME</strong>, or <strong>4</strong> (based on the selector type).</li>
<li>Use <strong>DEPTNO</strong> as the data profile column name and <strong>Lookup</strong> as the column type.</li>
<li>Use <strong>DEPT</strong> as the <strong>Table Name</strong></li>
<li>Use <strong>DEPTNO</strong> as the <strong>Return Column</strong></li>
<li>Use <strong>DNAME</strong> is the <strong>Table Column 1</strong></li>
<li><strong>DNAME_LOADED</strong> is the <strong>Data Column 1</strong>. Note that this references the data profile column which contains the actually uploaded data.</li>
</ul>
<p>
As a result, &PRODUCT_NAME. will look up <strong>DEPTNO</strong> values from the <strong>DEPT</strong> table, using the uploaded <strong>DNAME</strong> values. These will be uploaded as the <strong>DEPTNO</strong> column of the target table.
</p>~';

    ------------------------------------------------------------------------
    -- Global Variables
    ------------------------------------------------------------------------
    g_display_group_id            number  := 1;
    g_component_property_id       number  := 1;
    g_property_lov_value_id       number  := 1;
    g_component_property_sequence number;
    g_property_lov_value_sequence number;
    --
    g_null_parent_display_seq     number  := 0;
    g_parent_component_types      t_parent_component_types;
    --
    l_message_name                varchar2( 255 );
    g_used_system_messages        t_used_system_messages;

    g_cached_component_type_id    number;
    g_cached_is_page_comp_type    boolean;

    ------------------------------------------------------------------------
    -- Variables used in the generated script
    ------------------------------------------------------------------------
    l_property_id       number;
    l_component_type_id number;

    ------------------------------------------------------------------------
    -- Verifies if the entered configuration matches with our database constraints
    ------------------------------------------------------------------------
    procedure check_configuration (
        p_omit_length_check    in boolean default false,
        p_omit_not_null_check  in boolean default false )
    is
        l_data_length number;
        l_is_nullable varchar2( 1 );
        --
        procedure emit_message (
            p_severity      in varchar2,
            p_type_name     in varchar2,
            p_property_name in varchar2,
            p_db_column     in varchar2,
            p_error         in varchar2 )
        is
        begin
            sys.dbms_output.put_line( upper( p_severity ) || ': Component Type "' || p_type_name || '" - Property: "' || p_property_name || '" - DB Column "' || p_db_column || '" - ' || p_error );
        end emit_message;
    begin
        for l_type in ( select id,
                               name,
                               table_name
                          from wwv_pe_component_types
                         order by id )
        loop
            for l_comp_prop in ( select p.name as property_name,
                                        nvl( cp.db_column, p.db_column ) as db_column,
                                        p.max_length,
                                        cp.is_required,
                                        coalesce( to_char( cp.depending_on_comp_prop_id ), to_char( cp.depending_on_comp_prop_id2 ), cp.plugin_std_attributes, p.deprecated_values) as conditional,
                                        depending_on_condition_type,
                                        depending_on_expression
                                   from wwv_pe_component_props cp,
                                        wwv_pe_properties p
                                  where cp.component_type_id = l_type.id
                                    and p. id = cp.property_id
                                  order by 1 )
            loop
                begin
                    select case
                             when data_type = 'CLOB' then 32767
                             else data_length
                           end as data_length,
                           nullable
                      into l_data_length,
                           l_is_nullable
                      from sys.dba_tab_columns
                     where owner       = wwv_flow.g_flow_schema_owner
                       and table_name  = l_type.table_name
                       and column_name = l_comp_prop.db_column;
                    --
                    if not p_omit_length_check then
                        if l_comp_prop.max_length > l_data_length then
                            emit_message (
                                p_severity      => 'error',
                                p_type_name     => l_type.name,
                                p_property_name => l_comp_prop.property_name,
                                p_db_column     => l_comp_prop.db_column,
                                p_error         => 'Constraint defines ' || l_data_length || ' chars, config uses ' || l_comp_prop.max_length );
                        elsif l_comp_prop.max_length <> l_data_length then
                            emit_message (
                                p_severity      => 'warning',
                                p_type_name     => l_type.name,
                                p_property_name => l_comp_prop.property_name,
                                p_db_column     => l_comp_prop.db_column,
                                p_error         => 'Constraint defines ' || l_data_length || ' chars, config uses ' || l_comp_prop.max_length );
                        end if;
                    end if;

                    if not p_omit_not_null_check then
                        if l_comp_prop.is_required = 'N' and l_is_nullable = 'N' then
                            emit_message (
                                p_severity      => 'error',
                                p_type_name     => l_type.name,
                                p_property_name => l_comp_prop.property_name,
                                p_db_column     => l_comp_prop.db_column,
                                p_error         => 'Has not null constraint, config defines it nullable' );
                        end if;

                        if l_is_nullable = 'N' and l_comp_prop.conditional is not null and l_comp_prop.property_name not in ( 'FIRE_ON_PAGE_LOAD' )
                        then
                            emit_message (
                                p_severity      => 'error',
                                p_type_name     => l_type.name,
                                p_property_name => l_comp_prop.property_name,
                                p_db_column     => l_comp_prop.db_column,
                                p_error         => 'Has not null constraint, property is conditional.' );
                        end if;

                        if    l_comp_prop.is_required = 'Y'
                        and l_is_nullable = 'Y'
                        and l_comp_prop.conditional   is null
                        and l_comp_prop.property_name not like 'GRID_NEW%'
                        then
                            emit_message (
                                p_severity      => 'warning',
                                p_type_name     => l_type.name,
                                p_property_name => l_comp_prop.property_name,
                                p_db_column     => l_comp_prop.db_column,
                                p_error         => 'Is nullable, config always requires value - Add DB NN constraint.' );
                        end if;
                    end if;

                    if    l_comp_prop.depending_on_condition_type in ( 'EQUALS', 'NOT_EQUALS' )
                      and l_comp_prop.depending_on_expression like '%,%'
                    then
                        emit_message (
                            p_severity      => 'error',
                            p_type_name     => l_type.name,
                            p_property_name => l_comp_prop.property_name,
                            p_db_column     => l_comp_prop.db_column,
                            p_error         => 'Depending-on expression contains comma, but ' || l_comp_prop.depending_on_condition_type || ' is used!' );
                    end if;

                exception when no_data_found then
                        emit_message (
                            p_severity      => 'error',
                            p_type_name     => l_type.name,
                            p_property_name => l_comp_prop.property_name,
                            p_db_column     => l_comp_prop.db_column,
                            p_error         => 'Column not found in ' || l_type.table_name );
                end;
            end loop;
        end loop;
        --
    end check_configuration;

    procedure check_com_prop_configuration
    is
        l_groups      wwv_flow_t_varchar2;
        l_issues      wwv_flow_t_varchar2;
        l_cnt         integer := 0;
        --
        procedure check_group_sequence (
            p_component_type_id in number )
        is
            l_group             varchar2(255) := 'DiGa';
            l_found             boolean;
            l_properties        wwv_flow_t_varchar2 := wwv_flow_t_varchar2();
        begin
            l_groups     := wwv_flow_t_varchar2();
            l_issues     := wwv_flow_t_varchar2();
            --
            for l_prop in (
                select wwv_flow_lang.system_message(dg.title) as group_title,
                       wwv_flow_lang.system_message(p.prompt) as property_prompt,
                       p.name                                 as p_name,
                       coalesce( cp.db_column, p.db_column )  as db_column,
                       ct.name                                as ct_name,
                       ct.id                                  as ct_id
                from wwv_pe_component_types ct,
                     wwv_pe_component_props cp,
                     wwv_pe_properties p,
                     wwv_pe_display_groups dg
                where ct.id                = p_component_type_id
                  and cp.component_type_id = ct.id
                  and p.id                 = cp.property_id
                  and dg.id                = p.display_group_id
                order by cp.display_sequence ) loop
                --
                -- Check if the Group is in sequence
                if l_group != l_prop.group_title then
                    l_found := false;
                    for i in 1..l_groups.count loop
                        if l_groups(i) = l_prop.group_title then
                            wwv_flow_string.push(l_properties,l_prop.p_name);
                            wwv_flow_string.push(l_issues,l_prop.group_title||': '||l_prop.p_name);
                            l_found := true;
                        end if;
                    end loop;
                    if not l_found then
                        l_group := l_prop.group_title;
                        wwv_flow_string.push(l_groups, l_prop.group_title);
                    end if;
                end if;
                --
            end loop;
        end check_group_sequence;
    begin
        --
        -- Check for component props having the property groups not in sequence
        for l_ct in (
            select id,
                   name
            from wwv_pe_component_types ) loop
            --
            check_group_sequence(l_ct.id);
            --
            if l_issues.count > 0 then
                if l_cnt = 0 then
                    sys.dbms_output.put_line( 'Property-Group sequencing problems occurred!!' );
                end if;
                l_cnt := l_cnt + 1;
                sys.dbms_output.put_line( case when l_cnt > 1 then chr(10) end||l_ct.name );
            end if;
            --
            for i in 1..l_issues.count loop
                sys.dbms_output.put_line('    '||l_issues(i) );
            end loop;
        end loop;
        --
        if l_cnt = 0 then
            sys.dbms_output.put_line( 'All Property-Groups are in sequence.' );
        end if;
    end check_com_prop_configuration;

    -- Outputs any detected errors in import API configuration (API names and parameter matching)
    procedure check_import_api_configuration
    as
        procedure emit_message (
            p_severity      in varchar2,
            p_type_name     in varchar2,
            p_property_name in varchar2 default null,
            p_error         in varchar2 )
        is
        begin
            sys.dbms_output.put_line (
                upper( p_severity ) || ': Component Type "' || p_type_name || '"' ||
                case when p_property_name is not null then ' - Property: "' || p_property_name || '"' end ||
                '" - ' || p_error );
        end emit_message;

    begin
        --
        -- Check 1: Valid import_api_names
        for ct in (
            select ct.name,
                   ct.msg
              from ( select ct.name,
                            case
                                when ct.import_api_name is null then 'p_import_api_name to be defined'
                                when p.object_name is null then 'Invalid p_import_api_name: ' || ct.import_api_name
                                else null
                                end msg
                       from wwv_pe_component_types ct,
                            sys.user_procedures p
                      where ct.import_api_name = p.object_name(+) || '.' || p.procedure_name(+)  collate binary_ci ) ct
             where ct.msg is not null
            order by ct.name
        ) loop
            emit_message(
                p_severity  => 'error',
                p_type_name => ct.name,
                p_error     => ct.msg );
        end loop;

        --
        -- Check 2: Parameters are correct
        for p in (
            with
                apex_src as (
                    select ct.id component_type_id,
                           cp.id component_prop_id,
                           cp.property_id,
                           --
                           ct.name component_type_name,
                           p.name prop_name,
                           --
                           cp.db_column comp_prop_db_column,
                           p.db_column prop_db_column,
                           ct.parent_fk_column,
                           ct.import_api_name,
                           ct.import_api_params,
                           -- CP import takes precedence over the default prop API parameter
                           nvl(cp.import_api_parameter, p.import_api_parameter) import_api_parameter,
                           cp.import_api_parameter comp_prop_import_api_parameter,
                           p.import_api_parameter prop_import_api_parameter,
                           --
                           ct.is_query_only comp_type_is_query_only,
                           p.is_query_only prop_is_query_only,
                           cp.is_read_only comp_prop_is_read_only
                      from wwv_pe_component_types ct,
                           wwv_pe_component_props cp,
                           wwv_pe_properties p
                     where cp.component_type_id = ct.id
                       and p.id = cp.property_id
                       and ct.import_api_name is not null ), -- Only check paramters for APIs that are defined
                ua as (
                    -- Get arguments for all the APIs that are referenced in wwv_pe_component_types
                    select ct.id component_type_id,
                           ct.name component_type_name,
                           ua.object_id,
                           ct.parent_fk_column,
                           lower(ua.package_name || '.' || ua.object_name) api_name,
                           lower(ua.package_name) package_name,
                           lower(ua.object_name) object_name,
                           lower(ua.argument_name) argument_name,
                           ua.defaulted,
                           nvl2(pj.name, 'Y', 'N') import_api_param_override_yn -- If Y then parameter isn't required to be mapped as an override value will be used
                      from sys.user_arguments ua,
                           wwv_pe_component_types ct,
                           -- Expected format of import_api_params
                           -- [{name: "p_param_name", expression: "'value' or expression"}]
                           json_table( ct.import_api_params,
                                       '$[*]'
                                       columns( name,
                                                expression ) ) pj
                     where ua.package_name || '.' || ua.object_name = ct.import_api_name collate binary_ci
                       and ua.argument_name = pj.name(+) collate binary_ci )
            select *
              from ( select nvl(ua.component_type_name, s.component_type_name) component_type_name,
                            nvl(ua.api_name, s.import_api_name) import_api_name,
                            nvl(s.import_api_parameter, ua.argument_name) import_api_parameter,
                            s.prop_name property_name,
                            case -- Ignore parent component relationships
                                 when 'p_' || lower(nvl(s.parent_fk_column, ua.parent_fk_column)) = ua.argument_name
                                 then null
                                 -- Note: don't factor in defaulted values because they should be defined regardless of if the param has a default value
                                 -- if import_api_param_override_yn = Y then it has an override value defined in import_api_param_json and does not need to be mapped
                                 when nvl(ua.import_api_param_override_yn, 'N') = 'N' and s.component_type_id is null
                                 then 'parameter ' || ua.argument_name || ' is not mapped'
                                 -- Argument doesn't exist but is referenced
                                 when ua.object_id is null
                                 then 'unknown parameter ' || s.import_api_parameter
                                 -- no errors
                                 else null
                                 end err_msg
                       from apex_src s
                            full outer join ua on ua.component_type_id = s.component_type_id
                                              and ua.api_name = s.import_api_name
                                              and ua.argument_name = s.import_api_parameter ) params
             where params.err_msg is not null
                   -- ignore standard parameters
               and params.import_api_parameter not in ( 'p_id',
                                                        'p_flow_id',
                                                        'p_security_group_id',
                                                        'p_created_by',
                                                        'p_created_on',
                                                        'p_updated_by',
                                                        'p_updated_on' )
               and params.import_api_parameter not like 'p_attribute___' -- ingore plug-in parameter p_attribute_00
            order by params.component_type_name,
                     params.import_api_name,
                     params.import_api_parameter
        ) loop
            emit_message(
                p_severity      => 'error',
                p_type_name     => p.component_type_name,
                p_property_name => p.property_name,
                p_error         => p.import_api_name || ': ' ||  p.err_msg );
        end loop;

    end check_import_api_configuration;

    ------------------------------------------------------------------------
    -- Return the edit URL for a given component type
    ------------------------------------------------------------------------
    function get_parent_component_type_url (
        p_component_type_id     wwv_pe_component_types.id%type,
        p_url_type              varchar2 )
        return varchar2
    is
        l_rv        varchar2(4000);
    begin
        select case p_url_type
                    when 'CREATE' then create_url
                    when 'EDIT'   then edit_url
                    when 'COPY'   then copy_url
                    else null
                end
          into l_rv
          from wwv_pe_component_types
         where id = p_component_type_id;
         --
        return replace( l_rv, '%pk_value%', '%parent_pk_value%' );
        --
    end get_parent_component_type_url;
    ------------------------------------------------------------------------
    -- Get the Display Group ID, by name
    ------------------------------------------------------------------------
    function get_display_group_id (
        p_name in wwv_pe_display_groups.name%type )
        return wwv_pe_display_groups.id%type
    is
        l_id wwv_pe_display_groups.id%type;
    begin
        select id
          into l_id
          from wwv_pe_display_groups
         where name = p_name;

        return l_id;
    exception
        when no_data_found then
            raise_application_error( -20999, 'Display Group "' || p_name || '" not found!' );
        -- too_many_rows will never be raised, as name is uniquely constrained
    end get_display_group_id;

    ------------------------------------------------------------------------
    -- Get the Property ID, by name
    ------------------------------------------------------------------------
    function get_property_id (
        p_name in wwv_pe_properties.name%type )
        return wwv_pe_properties.id%type
    is
        l_id wwv_pe_properties.id%type;
    begin
        select id
          into l_id
          from wwv_pe_properties
         where name = p_name;

        return l_id;
    exception
        when no_data_found then
            raise_application_error( -20999, 'Property "' || p_name || '" not found!' );
        -- too_many_rows will never be raised, as name is uniquely constrained
    end get_property_id;

    ------------------------------------------------------------------------
    -- Get the Property Name, by ID
    ------------------------------------------------------------------------
    function get_property_name (
        p_id    in wwv_pe_properties.id%type )
        return wwv_pe_properties.name%type
    is
        l_name wwv_pe_properties.name%type;
    begin
        select name
          into l_name
          from wwv_pe_properties
         where id = p_id;

        return l_name;
    exception
        when no_data_found then
            raise_application_error( -20999, 'Property "' || p_id || '" not found!' );
        -- too_many_rows will never be raised, as ID is unique
    end get_property_name;

    ------------------------------------------------------------------------
    -- Get the Component Property ID, by Component Type ID and Property Name
    ------------------------------------------------------------------------
    function get_component_prop_id (
        p_component_type_id in wwv_pe_component_types.id%type,
        p_property_name     in wwv_pe_properties.name%type )
        return wwv_pe_component_props.id%type
    is
        l_id wwv_pe_component_props.id%type;
    begin
        select cp.id
          into l_id
          from wwv_pe_component_props cp,
               wwv_pe_properties p
         where p.name               = p_property_name
           and cp.component_type_id = p_component_type_id
           and cp.property_id       = p.id;

        return l_id;
    exception
        when no_data_found then
            raise_application_error( -20999, 'Property "' || p_property_name || '" for component type "' || p_component_type_id || '" not found!' );
        -- too_many_rows will never be raised, unique constraint on property_id
        -- and component_type_id
    end get_component_prop_id;

    ------------------------------------------------------------------------
    -- Get the Component Property ID, by Component Type ID and Property Name
    ------------------------------------------------------------------------
    function get_component_type_name (
        p_component_type_id in wwv_pe_component_types.id%type )
        return wwv_pe_component_types.name%type
    is
        l_name wwv_pe_component_types.name%type;
    begin
        select name
          into l_name
          from wwv_pe_component_types
         where id = p_component_type_id;

        return l_name;
    exception
        when no_data_found then
            raise_application_error( -20999, 'Component Type "' || p_component_type_id || '" not found!' );
        -- too_many_rows will never be raised as ID is unique
    end get_component_type_name;

    ------------------------------------------------------------------------
    -- System Message
    ------------------------------------------------------------------------
    procedure store_msg (
        p_name         in out varchar2,
        p_message_text in     varchar2 )
    is
    begin
        p_name := upper( 'PE.' || p_name );

        if p_message_text is not null and c_load_system_messages then
            merge into wwv_flow_messages$ t
            using ( select p_name         as name,
                           p_message_text as message_text
                      from sys.dual ) v
               on (     t.name              = v.name
                    and t.flow_id           = 4411
                    and t.security_group_id = 10
                    and t.message_language  = 'en' )
            when matched then
                 update
                    set t.message_text = v.message_text
                  where t.message_text != v.message_text
            when not matched then
                 insert ( t.flow_id,
                          t.security_group_id,
                          t.name,
                          t.message_text,
                          t.message_language )
                 values ( 4411,
                    10,
                          v.name,
                          v.message_text,
                    'en' );
            g_used_system_messages( p_name ) := true;
        end if;

    end store_msg;
    --
    procedure remove_unused_messages
    is
    begin
        if c_load_system_messages then
            for l_msg in ( select rowid,
                                  name
                             from wwv_flow_messages$
                            where flow_id           = 4411
                              and security_group_id = 10
                              and name              like 'PE.%'
                              and message_language  = 'en' )
            loop
                if not g_used_system_messages.exists ( l_msg.name ) then
                    delete wwv_flow_messages$
                     where rowid = l_msg.rowid;
                end if;
            end loop;
        end if;
    end remove_unused_messages;
    ------------------------------------------------------------------------
    -- Change the sequence number for a given component porperty
    -- Use this function if a comp-prop dependency prevents from storing a
    -- component property in the sequence of it's property display group
    ------------------------------------------------------------------------
    procedure resequence_comp_prop (
        p_component_type_id in wwv_pe_component_types.id%type,
        p_name              in wwv_pe_properties.name%type,
        p_previous_name     in wwv_pe_properties.name%type )
    is
        l_display_sequence      integer;
        l_property_id           number;
        l_previous_property_id  number;
    begin
        --
        l_property_id          := get_property_id( p_name => p_name );
        l_previous_property_id := get_property_id( p_name => p_previous_name );
        --
        -- Get the new display sequence
        select display_sequence + 1
          into l_display_sequence
          from wwv_pe_component_props
         where component_type_id = p_component_type_id
           and property_id       = l_previous_property_id;
        --
        -- Update the moving property display sequence
        update wwv_pe_component_props cp
           set display_sequence = l_display_sequence
         where cp.component_type_id = p_component_type_id
           and cp.property_id       = l_property_id;
        --
    end resequence_comp_prop;
    --
    ------------------------------------------------------------------------
    -- Change the sequence number for a given amount of component porperties
    -- A maximum of 9 properties can be moved as block
    -- Use this function if a comp-prop dependency prevents from storing a
    -- component property in the sequence of it's property display group
    ------------------------------------------------------------------------
    procedure resequence_comp_prop_bulk (
        p_component_type_id in wwv_pe_component_types.id%type,
        p_names             in wwv_flow_t_varchar2,
        p_previous_name     in wwv_pe_properties.name%type )
    is
    begin
        if p_names.count > 9 then
            raise_application_error(-20001,'Too many properties! A maximum of 9 properties can be moved!');
        end if;
        --
        for i in 1..p_names.count loop
            resequence_comp_prop (
                p_component_type_id => p_component_type_id,
                p_name              => p_names(i),
                p_previous_name     => case i when 1 then p_previous_name else p_names(i-1) end );
        end loop;
    end resequence_comp_prop_bulk;
    --
    ------------------------------------------------------------------------
    -- Component Types
    ------------------------------------------------------------------------
    procedure store_component_type (
        p_id                            in number,
        p_name                          in varchar2,
        p_singular_title                in varchar2,
        p_plural_title                  in varchar2,
        p_table_name                    in varchar2,
        p_view_name                     in varchar2 default null,
        p_import_api_name               in varchar2 default null,
        -- Array of default parameter values
        -- Ex: [{name: "p_param_name", expression: "'value' or expression"}]
        p_import_api_params             in varchar2 default null,
        p_pk_column                     in varchar2 default 'ID',
        p_parent_component_type_id      in number   default null,
        p_parent_fk_column              in varchar2 default null,
        p_page_id_column                in varchar2 default null,
        p_display_column                in varchar2,
        p_alt_display_column            in varchar2 default null,
        p_where_clause                  in varchar2 default null,
        p_validation_plsql_ret_error    in varchar2 default null,
        p_plugin_type                   in varchar2 default null,
        p_plugin_column                 in varchar2 default null,
        p_supported_tmpl_components     in varchar2 default null,
        p_create_url                    in varchar2 default null,
        p_edit_url                      in varchar2 default null,
        p_copy_url                      in varchar2 default null,
        p_file_path                     in varchar2 default null,
        p_is_page_designer_shared       in boolean  default false,
        p_is_query_only                 in boolean  default false,
        p_is_shared_designer            in boolean  default false )
    is
        l_is_page_designer_shared       varchar2( 1 ) := case when p_is_page_designer_shared    then 'Y' else 'N' end;
        l_is_query_only                 varchar2( 1 ) := case when p_is_query_only              then 'Y' else 'N' end;
        l_is_shared_designer            varchar2( 1 ) := case when p_is_shared_designer         then 'Y' else 'N' end;
        l_msg_plural_title              wwv_flow_messages$.name%type;
        l_msg_singular_title            wwv_flow_messages$.name%type;
        l_display_sequence              number;
    begin
        ------------------------------------------------------------------------
        -- Store system message
        ------------------------------------------------------------------------
        l_msg_plural_title := 'COMPONEN.TYPE.' || p_name || '.PLURAL';
        store_msg (
            p_name         => l_msg_plural_title,
            p_message_text => p_plural_title );

        l_msg_singular_title := 'COMPONEN.TYPE.' || p_name || '.SINGULAR';
        store_msg (
            p_name         => l_msg_singular_title,
            p_message_text => p_singular_title );

        ------------------------------------------------------------------------
        -- Calculate Display Sequence
        ------------------------------------------------------------------------
        if p_parent_component_type_id is not null then
            if not g_parent_component_types.exists( p_parent_component_type_id ) then
                g_parent_component_types( p_parent_component_type_id ) := 0;
            end if;
            --
            g_parent_component_types( p_parent_component_type_id ) := g_parent_component_types( p_parent_component_type_id ) + 1;
            l_display_sequence := g_parent_component_types( p_parent_component_type_id );
        else
            l_display_sequence := g_null_parent_display_seq + 1;
        end if;

        insert into wwv_pe_component_types (
            id,
            name,
            table_name,
            view_name,
            import_api_name,
            import_api_params,
            plural_title,
            singular_title,
            display_sequence,
            pk_column,
            parent_component_type_id,
            parent_fk_column,
            page_id_column,
            display_column,
            alt_display_column,
            where_clause,
            validation_plsql_ret_error,
            plugin_type,
            plugin_column,
            supported_template_components,
            create_url,
            edit_url,
            copy_url,
            file_path,
            is_page_designer_shared,
            is_query_only,
            is_shared_designer )
        values (
            p_id,
            p_name,
            p_table_name,
            p_view_name,
            p_import_api_name,
            p_import_api_params,
            l_msg_plural_title, -- Store msg reference
            l_msg_singular_title, -- Store msg reference
            l_display_sequence,
            p_pk_column,
            p_parent_component_type_id,
            p_parent_fk_column,
            p_page_id_column,
            p_display_column,
            p_alt_display_column,
            p_where_clause,
            p_validation_plsql_ret_error,
            p_plugin_type,
            p_plugin_column,
            p_supported_tmpl_components,
            p_create_url,
            p_edit_url,
            p_copy_url,
            p_file_path,
            l_is_page_designer_shared,
            l_is_query_only,
            l_is_shared_designer )
        ;

    end store_component_type;

    ------------------------------------------------------------------------
    -- Display Groups
    ------------------------------------------------------------------------
    procedure store_display_group (
        p_name  in varchar2,
        p_title in varchar2 )
    is
        l_msg_name_title wwv_flow_messages$.name%type;
    begin
        ------------------------------------------------------------------------
        -- Store system message
        ------------------------------------------------------------------------
        l_msg_name_title := 'DISPLAY_GROUP.' || p_name;
        store_msg (
            p_name         => l_msg_name_title,
            p_message_text => p_title );

        insert into wwv_pe_display_groups (
            id,
            name,
            title )
        values (
            g_display_group_id,
            p_name,
            l_msg_name_title ) -- Store msg reference
        ;

        -- Always increment global display group id
        g_display_group_id := g_display_group_id + 1;
    end store_display_group;

    ------------------------------------------------------------------------
    -- This function is used to determine if the help text of a property
    -- should be stored in our 4411-system text messages. Properties which are
    -- currently just used by our YAML export, don't have to store help text,
    -- it would just increase our translation budget.
    -- The help text of those properties hasn't been modified yet and would
    -- just lead to unnecessary translation costs!
    ------------------------------------------------------------------------
    function is_new_shared_comp_property (
        p_id in number )
        return boolean
    is
    begin
        /*
        -- Use this query to adjust the return clause of this function.
        -- The query should'nt return any records. It will show
        -- Page Designer or Workflow Designer properties that are in those
        -- ranges.
        --
        select ct.name as component_type_name,
               cp.property_id,
               p.name as property_name
          from wwv_pe_component_types ct,
               wwv_pe_component_props cp,
               wwv_pe_properties      p
         where (  ct.page_id_column      is not null
                or ct.name               = 'PAGE'
                or ct.is_shared_designer = 'Y' )
           and cp.component_type_id = ct.id
           and (   cp.property_id between 1250 and 1280
                or cp.property_id between 1283 and 1363
                or cp.property_id between 1370 and 1401
                or cp.property_id between 1403 and 1466
                or cp.property_id between 1468 and 1500
                or cp.property_id between 1606 and 1658
                or cp.property_id between 1790 and 1999
                or cp.property_id between 2201 and 2327
                or cp.property_id between 5700 and 5977
               )
           and p.id = cp.property_id;
        -- */
        -- Number ranges which are just used by Shared Component Properties and
        -- which are not used by Page Designer or Workflow Designer configuration
        return (  p_id between 1250 and 1280
               or p_id between 1283 and 1363
               or p_id between 1370 and 1401
               or p_id between 1403 and 1466
               or p_id between 1468 and 1500
               or p_id between 1606 and 1658
               or p_id between 1790 and 1999
               or p_id between 2201 and 2327
               or p_id between 5700 and 5977
               );
    end is_new_shared_comp_property;
    --
    --
    ------------------------------------------------------------------------
    -- Properties
    ------------------------------------------------------------------------
    procedure store_property (
        p_id                           in number,
        p_name                         in varchar2,
        p_prompt                       in varchar2,
        p_property_type                in varchar2,
        p_prop_type_yesno_override_y   in varchar2     default null,
        p_prop_type_yesno_override_n   in varchar2     default null,
        p_multi_value_delimiter        in varchar2     default null,
        p_db_column                    in varchar2,
        p_import_api_parameter         in varchar2     default null,
        p_is_query_only                in boolean      default false,
        p_display_length               in number       default null,
        p_max_length                   in number       default null,
        p_min_value                    in number       default null,
        p_max_value                    in number       default null,
        p_unit                         in varchar2     default null,
        p_validation_plsql_ret_error   in varchar2     default null,
        p_text_case                    in varchar2     default null,
        p_restricted_characters        in varchar2     default null,
        p_sql_min_column_count         in number       default null,
        p_sql_max_column_count         in number       default null,
        p_column_data_types            in varchar2     default null,
        p_examples                     in varchar2     default null,
        p_display_group_name           in varchar2,
        p_lov_type                     in varchar2     default null,
        p_lov_component_type_id        in number       default null,
        p_lov_component_type_scope     in varchar2     default null,
        p_is_translatable              in boolean      default false,
        p_is_searchable                in boolean      default true,
        p_deprecated_values            in varchar2     default null,
        p_is_internal                  in boolean      default false,
        p_min_db_version               in number       default null,
        p_help_text                    in varchar2     default null,
        p_comment_text                 in varchar2     default null )
    is
        l_display_group_id      wwv_pe_display_groups.id%type;
        l_msg_name_prop_prompt  wwv_flow_messages$.name%type;
        l_msg_name_prop_help    wwv_flow_messages$.name%type;
        l_msg_name_unit         wwv_flow_messages$.name%type;
        l_msg_name_examples     wwv_flow_messages$.name%type;
        l_import_api_parameter  wwv_pe_properties.import_api_parameter%type;
        l_text_case             wwv_pe_properties.text_case%type             := p_text_case;
        l_display_length        wwv_pe_properties.display_length%type        := p_display_length;
        l_max_length            wwv_pe_properties.max_length%type            := p_max_length;
        l_multi_value_delimiter wwv_pe_properties.multi_value_delimiter%type := p_multi_value_delimiter;
        l_help_text             wwv_flow_messages$.message_text%type         := p_help_text;
        l_examples              wwv_flow_messages$.message_text%type         := p_examples;
        l_is_translatable       varchar2( 1 ) := case when p_is_translatable then 'Y' else 'N' end;
        l_is_searchable         varchar2( 1 ) := case when p_is_searchable   then 'Y' else 'N' end;
        l_is_query_only         varchar2( 1 ) := case when p_is_query_only   then 'Y' else 'N' end;
        l_is_internal           varchar2( 1 ) := case when p_is_internal     then 'Y' else 'N' end;
    begin
        ------------------------------------------------------------------------
        -- Get Display Group ID, by Display Group Name
        ------------------------------------------------------------------------
        l_display_group_id := get_display_group_id( p_display_group_name );

        ------------------------------------------------------------------------
        -- Reset Property LOV Values sequence ready for  Property LOV values
        -- for this Property.
        ------------------------------------------------------------------------
        g_property_lov_value_sequence := 10;

        ------------------------------------------------------------------------
        -- Store system messages
        ------------------------------------------------------------------------
        l_msg_name_prop_prompt := 'PROPERTY.' || p_name || '.PROMPT';
        l_msg_name_prop_help   := 'PROPERTY.' || p_name || '.HELP';
        store_msg (
            p_name         => l_msg_name_prop_prompt,
            p_message_text => p_prompt );
        --
        if p_property_type = 'COLOR' then
            l_help_text := l_help_text ||
q'~
<p>The color is specified using hexadecimal (hex) notation, RGB color codes, or HTML color names. Hex notation is defined as #xxyyzz (a # followed by 3 double digit hex numbers), where xx, yy, and zz represent the two character hex values for red, green and blue respectively.
RGB codes are defined as RGB(xxx,xxx,xxx), where xxx is a number between 0 and 255 each representing the single 8-bit byte value for red, green, and blue respectively.</p>
<p>Search the internet for 'HTML Color Codes' to find the equivalent hex code or RGB code for the color you wish to utilize.</p>~';
        end if;
        --
        -- Don't save the help text yet for the Shared Components properties. The help text
        -- has to be reworked and we don't want to translate them multiple times.
        if is_new_shared_comp_property( p_id ) then
            l_msg_name_prop_help := null;
        else
            store_msg (
                p_name         => l_msg_name_prop_help,
                p_message_text => l_help_text );
        end if;
        --
        if p_unit is not null then
            l_msg_name_unit := 'PROPERTY.' || p_name || '.UNIT';
            store_msg (
                p_name         => l_msg_name_unit,
                p_message_text => p_unit );
        end if;
        --
        if p_property_type = 'COLOR' and l_examples is null then
            l_examples :=
'Examples of major HTML color codes include -
<ul>
  <li><strong><span style="background-color: lightgray; color: black;">&nbsp;&nbsp;Black&nbsp;&nbsp;</span></strong> <code>#000000 RGB(0,0,0)</code></li>
  <li><strong><span style="background-color: lightgray; color: red;">&nbsp;&nbsp;Red&nbsp;&nbsp;</span></strong> <code>#FF0000 RGB(255,0,0)</code></li>
  <li><strong><span style="background-color: lightgray; color: yellow;">&nbsp;&nbsp;Yellow&nbsp;&nbsp;</span></strong> <code>#FFFF00 RGB(255,255,0)</code></li>
  <li><strong><span style="background-color: lightgray; color: lime;">&nbsp;&nbsp;Lime&nbsp;&nbsp;</span></strong> <code>#00FF00 RGB(0,255,0)</code></li>
  <li><strong><span style="background-color: lightgray; color: green;">&nbsp;&nbsp;Green&nbsp;&nbsp;</span></strong> <code>#008000 RGB(0,128,0)</code></li>
  <li><strong><span style="background-color: lightgray; color: aqua;">&nbsp;&nbsp;Aqua&nbsp;&nbsp;</span></strong> <code>#00FFFF RGB(0,255,255)</code></li>
  <li><strong><span style="background-color: lightgray; color: blue;">&nbsp;&nbsp;Blue&nbsp;&nbsp;</span></strong> <code>#0000FF RGB(0,255,255)</code></li>
  <li><strong><span style="background-color: lightgray; color: navy;">&nbsp;&nbsp;Navy&nbsp;&nbsp;</span></strong> <code>#000080 RGB(0,0,128)</code></li>
  <li><strong><span style="background-color: lightgray; color: white;">&nbsp;&nbsp;White&nbsp;&nbsp;</span></strong> <code>#FFFFFF RGB(255,255,255)</code></li>
</ul>';
        end if;
        --
        if l_examples is not null then
            -- Don't save the examples yet for the Shared Components properties. The examples
            -- have to be reworked and we don't want to translate them multiple times.
            if not is_new_shared_comp_property( l_property_id ) then
                l_msg_name_examples := 'PROPERTY.' || p_name || '.EXAMPLES';
                store_msg (
                    p_name         => l_msg_name_examples,
                    p_message_text => l_examples );
            end if;
        end if;
        -- Search doesn't make sense for a few property types
        if   ( p_lov_type is not null and p_lov_type not like '%FORMAT_MASK%' and p_lov_type <> 'DISTINCT' )
          or p_property_type in ( 'YES NO', 'SUBSCRIPTION', 'HIDDEN' )
        then
            l_is_searchable := 'N';
        end if;

        -- Numbers can have up to 22 digits
        if p_property_type = 'NUMBER' and l_max_length is null then
            l_max_length := 22;
        elsif p_property_type = 'COLOR' and ( l_display_length is null or l_max_length is null ) then
            l_display_length := nvl( l_display_length, 7 );
            l_max_length     := nvl( l_max_length, 255 );
        end if;

        -- Items are always in uppercase
        if p_property_type in ( 'ITEM', 'FACET', 'SFILTER' ) then
            l_text_case := 'UPPER';
        end if;

        -- Checkboxes should always use a multi-value delimiter
        if p_property_type = 'CHECKBOXES' and l_multi_value_delimiter is null then
            l_multi_value_delimiter := ':';
        end if;

        -- Set the API parameter here (rather than above) so that if the parameter name is too large it'll be caught in exception handler and p_name will be displayed
        -- Note: substr is TEMPORARY as some columns (ex INIT_GEOMETRY_SDOGEOM_COLUMN don't have a defined parameter yet so need to let pass)
        l_import_api_parameter := 'p_' || lower(coalesce(p_import_api_parameter, substr(p_db_column, 1, 28)));


        insert into wwv_pe_properties (
            id,
            name,
            prompt,
            property_type,
            prop_type_yesno_override_y,
            prop_type_yesno_override_n,
            multi_value_delimiter,
            db_column,
            import_api_parameter,
            is_query_only,
            display_length,
            max_length,
            min_value,
            max_value,
            unit,
            validation_plsql_ret_error,
            text_case,
            restricted_characters,
            sql_min_column_count,
            sql_max_column_count,
            column_data_types,
            examples,
            display_group_id,
            lov_type,
            lov_component_type_id,
            lov_component_type_scope,
            is_translatable,
            is_searchable,
            deprecated_values,
            is_internal,
            min_db_version,
            help_text,
            comment_text )
        values (
            p_id,
            p_name,
            l_msg_name_prop_prompt,
            p_property_type,
            p_prop_type_yesno_override_y,
            p_prop_type_yesno_override_n,
            l_multi_value_delimiter,
            p_db_column,
            l_import_api_parameter,
            l_is_query_only,
            l_display_length,
            l_max_length,
            p_min_value,
            p_max_value,
            l_msg_name_unit,
            p_validation_plsql_ret_error,
            l_text_case,
            p_restricted_characters,
            p_sql_min_column_count,
            p_sql_max_column_count,
            p_column_data_types,
            l_msg_name_examples,
            l_display_group_id,
            p_lov_type,
            p_lov_component_type_id,
            p_lov_component_type_scope,
            l_is_translatable,
            l_is_searchable,
            p_deprecated_values,
            l_is_internal,
            p_min_db_version,
            l_msg_name_prop_help,
            p_comment_text )
        ;

    exception when others then
    for l_i in ( select name from wwv_pe_properties where id = p_id ) loop
    raise_application_error(-20111, 'exists: '||l_i.name||' new: '||p_id||' name: '||p_name);
    end loop;
        raise_application_error( -20111, 'Property Name: ' || p_name || ' sqlerrm: ' || sqlerrm );
    end store_property;

    ------------------------------------------------------------------------
    -- Property LOV Values
    ------------------------------------------------------------------------
    procedure store_property_lov_values (
        p_property_id    in number,
        p_display_value  in varchar2,
        p_return_value   in varchar2,
        p_icon           in varchar2 default null,
        p_is_quick_pick  in varchar2 default 'N',
        p_is_internal    in varchar2 default 'N',
        p_is_deprecated  in varchar2 default 'N',
        p_min_db_version in number   default null,
        p_help_text      in varchar2 default null,
        p_comment_text   in varchar2 default null )
    is
        l_msg_name_prop_lov_val wwv_flow_messages$.name%type;
        l_msg_name_prop_help    wwv_flow_messages$.name%type;
    begin
        ------------------------------------------------------------------------
        -- Store system messages
        ------------------------------------------------------------------------
        l_msg_name_prop_lov_val := 'PROPERTY.' || get_property_name( p_property_id ) || '.LOV.' || upper( p_return_value ) || '.D';
        store_msg (
            p_name         => l_msg_name_prop_lov_val,
            p_message_text => p_display_value );
        --
        if p_help_text is not null then
            --
            -- Don't save the help text yet for the Shared Components properties. The help text
            -- has to be reworked and we don't want to translate them multiple times.
            if not is_new_shared_comp_property( p_property_id ) then
                l_msg_name_prop_help := 'PROPERTY.' || get_property_name( p_property_id ) || '.LOV.' || upper( p_return_value ) || '.HELP';
                store_msg (
                    p_name         => l_msg_name_prop_help,
                    p_message_text => p_help_text );
            end if;
        end if;

        insert into wwv_pe_property_lov_values (
            id,
            property_id,
            display_sequence,
            display_value,
            return_value,
            icon,
            is_quick_pick,
            is_internal,
            is_deprecated,
            min_db_version,
            help_text,
            comment_text )
        values (
            g_property_lov_value_id,
            p_property_id,
            g_property_lov_value_sequence,
            l_msg_name_prop_lov_val,
            p_return_value,
            p_icon,
            p_is_quick_pick,
            p_is_internal,
            p_is_deprecated,
            p_min_db_version,
            l_msg_name_prop_help,
            p_comment_text )
        ;
        --
        g_property_lov_value_id       := g_property_lov_value_id + 1;
        g_property_lov_value_sequence := g_property_lov_value_sequence + 10;

    end store_property_lov_values;

    ------------------------------------------------------------------------
    -- Component Properties
    ------------------------------------------------------------------------
    procedure store_component_prop (
        p_component_type_id             in number,
        p_property_name                 in varchar2,
        p_db_column                     in varchar2     default null,
        p_import_api_parameter          in varchar2     default null,
        p_view_column                   in varchar2     default null,
        p_is_required                   in boolean,
        p_check_uniqueness              in varchar2     default null,
        p_is_common                     in boolean,
        p_is_read_only                  in boolean      default false,
        p_reference_on_delete           in varchar2     default null,
        p_reference_scope               in varchar2     default 'COMPONENT',
        p_supports_substitution         in boolean      default true,
        p_supports_template_directive   in varchar2     default null,
        p_custom_data_type              in varchar2     default null,
        p_default_value                 in varchar2     default null,
        p_null_text                     in varchar2     default null,
        p_null_help_text                in varchar2     default null,
        p_plugin_std_attributes         in varchar2     default null,
        p_features_comp_prop_name       in varchar2     default null,
        p_features                      in varchar2     default null,
        p_depending_on_comp_prop_name   in varchar2     default null,
        p_depending_on_has_to_exist     in boolean      default true,
        p_depending_on_condition_type   in varchar2     default null,
        p_depending_on_expression       in varchar2     default null,
        p_depending_on_comp_prop_name2  in varchar2     default null,
        p_depending_on_has_to_exist2    in boolean      default true,
        p_depending_on_condition_type2  in varchar2     default null,
        p_depending_on_expression2      in varchar2     default null,
        p_parent_comp_type_id           in number       default null,
        p_parent_comp_prop_name         in varchar2     default null,
        p_examples                      in varchar2     default null,
        p_help_text                     in varchar2     default null,
        p_important_for_accessibility   in boolean      default false,
        p_accessibility_help_text       in varchar2     default null,
        p_comment_text                  in varchar2     default null )
    is
        l_property_id                 wwv_pe_properties.id%type;
        l_property_type               wwv_pe_properties.property_type%type;
        l_import_api_parameter        wwv_pe_component_props.import_api_parameter%type;
        l_features_comp_prop_id       wwv_pe_component_props.features_comp_prop_id%type;
        l_depending_on_com_prop_id    wwv_pe_component_props.depending_on_comp_prop_id%type;
        l_depending_on_has_to_exist   varchar2( 1 );
        l_depending_on_com_prop_id2   wwv_pe_component_props.depending_on_comp_prop_id2%type;
        l_depending_on_has_to_exist2  varchar2( 1 );
        l_parent_comp_prop_id         wwv_pe_component_props.parent_comp_prop_id%type;
        l_msg_name_comp_prop_help     wwv_flow_messages$.name%type;
        l_msg_name_null_text          wwv_flow_messages$.name%type;
        l_msg_name_null_help_text     wwv_flow_messages$.name%type;
        l_msg_name_prop_acc_help      wwv_flow_messages$.name%type;
        l_msg_name_examples           wwv_flow_messages$.message_text%type;
        l_is_required                 varchar2( 1 ) := case p_is_required
                                                         when true  then 'Y'
                                                         when false then 'N'
                                                         else null
                                                       end;
        l_is_common                   varchar2( 1 ) := case when p_is_common    then 'Y' else 'N' end;
        l_is_read_only                varchar2( 1 ) := case when p_is_read_only then 'Y' else 'N' end;
        l_supports_substitution       varchar2( 1 ) := case when p_supports_substitution       then 'Y' else 'N' end;
        l_important_for_accessibility varchar2( 1 ) := case when p_important_for_accessibility then 'Y' else 'N' end;
        l_supports_template_directive varchar2( 1 ) := p_supports_template_directive;
        l_reference_on_delete         wwv_pe_component_props.reference_on_delete%type := p_reference_on_delete;
        l_accessibility_help_text     wwv_flow_messages$.message_text%type            := p_accessibility_help_text;
        l_is_page_component_type      varchar2( 1 );

        procedure check_depending_on (
            p_condition_type in varchar2,
            p_expression     in varchar2 )
        is
        begin
            if p_expression like '%,%' and p_condition_type not in ( 'IN_LIST', 'NOT_IN_LIST', 'IN_LIST_LIKE', 'NOT_IN_LIST_LIKE' ) then
                raise_application_error(
                    -20999,
                    'Property "' || p_property_name || '" for component type "' || p_component_type_id ||
                    '" contains list values but does not use a list operator! - ' || p_condition_type );
            end if;

            if p_expression is not null and p_condition_type in ( 'NULL', 'NOT_NULL' ) then
                raise_application_error(
                    -20999,
                    'Property "' || p_property_name || '" for component type "' || p_component_type_id ||
                    '" contains list values but does not use a list operator! - ' || p_condition_type );
            end if;
        end check_depending_on;
    begin
        if g_cached_component_type_id is null or g_cached_component_type_id <> p_component_type_id then
            select id,
                   case
                     when page_id_column is not null or is_shared_designer = 'Y' or id = c_comp_type_page then 'Y'
                     else
                   'N' end
              into g_cached_component_type_id,
                   l_is_page_component_type
              from wwv_pe_component_types
             where id = p_component_type_id;
            --
            g_cached_is_page_comp_type := ( l_is_page_component_type = 'Y' );
        end if;
        ------------------------------------------------------------------------
        -- Get Property ID and type
        ------------------------------------------------------------------------
        l_property_id := get_property_id ( p_property_name );
        select property_type
          into l_property_type
          from wwv_pe_properties
         where id = l_property_id;

        -- Get API Parameter
        if p_db_column is not null or p_import_api_parameter is not null then
            -- Note: substr is TEMPORARY as some columns (ex INIT_GEOMETRY_SDOGEOM_COLUMN don't have a defined parameter yet so need to let pass)
            l_import_api_parameter := 'p_' || lower(coalesce( p_import_api_parameter, substr(p_db_column, 1, 28)));
        end if;

        ------------------------------------------------------------------------
        -- Store system messages (Only if help text is passed)
        ------------------------------------------------------------------------
        if p_null_text is not null then
            l_msg_name_null_text := 'PROPERTY.' || p_property_name || '.COMPONENT_PROPERTY.'||get_component_type_name(p_component_type_id)|| '.NULL_TEXT';
            store_msg (
                p_name         => l_msg_name_null_text,
                p_message_text => p_null_text );
        end if;

        if p_null_text is not null and p_null_help_text is not null then
            -- Don't save the help text yet for the Shared Components properties. The help text
            -- has to be reworked and we don't want to translate them multiple times.
            if g_cached_is_page_comp_type then
                l_msg_name_null_help_text := 'PROPERTY.' || p_property_name || '.COMPONENT_PROPERTY.'||get_component_type_name(p_component_type_id)|| '.NULL_TEXT.HELP';
                store_msg (
                    p_name         => l_msg_name_null_help_text,
                    p_message_text => p_null_help_text );
            end if;
        end if;

        if p_examples is not null then
            -- Don't save the examples yet for the Shared Components properties. The examples
            -- have to be reworked and we don't want to translate them multiple times.
            if not is_new_shared_comp_property( l_property_id ) and g_cached_is_page_comp_type then
                l_msg_name_examples := 'PROPERTY.'||p_property_name||'.COMPONENT_PROPERTY.'||get_component_type_name(p_component_type_id)|| '.EXAMPLES';
                store_msg (
                    p_name         => l_msg_name_examples,
                    p_message_text => p_examples );
            end if;
        end if;

        if p_help_text is not null then
            -- Don't save the help text yet for the Shared Components properties. The help text
            -- has to be reworked and we don't want to translate them multiple times.
            if not is_new_shared_comp_property( l_property_id ) and g_cached_is_page_comp_type then
                l_msg_name_comp_prop_help := 'PROPERTY.'||p_property_name||'.COMPONENT_PROPERTY.'||get_component_type_name(p_component_type_id)|| '.HELP';
                store_msg (
                    p_name         => l_msg_name_comp_prop_help,
                    p_message_text => p_help_text );
            end if;
        end if;

        if p_accessibility_help_text is not null then
            l_msg_name_prop_acc_help := 'PROPERTY.' || p_property_name || '.COMPONENT_PROPERTY.'||get_component_type_name(p_component_type_id)|| '.ACCESSIBILITY_HELP_TEXT';
            store_msg (
                p_name         => l_msg_name_prop_acc_help,
                p_message_text => p_accessibility_help_text );
        end if;

        if p_features is not null then
            l_features_comp_prop_id     := get_component_prop_id (
                                               p_component_type_id => p_component_type_id,
                                               p_property_name     => p_features_comp_prop_name );
        end if;

        if p_depending_on_comp_prop_name is not null then
            l_depending_on_com_prop_id  := get_component_prop_id (
                                               p_component_type_id => p_component_type_id,
                                               p_property_name     => p_depending_on_comp_prop_name );
            l_depending_on_has_to_exist := case when p_depending_on_has_to_exist then 'Y' else 'N' end;
        end if;

        if p_depending_on_comp_prop_name2 is not null then
            l_depending_on_com_prop_id2  := get_component_prop_id (
                                                p_component_type_id => p_component_type_id,
                                                p_property_name     => p_depending_on_comp_prop_name2 );
            l_depending_on_has_to_exist2 := case when p_depending_on_has_to_exist2 then 'Y' else 'N' end;
        end if;

        if p_parent_comp_prop_name is not null then
            l_parent_comp_prop_id := get_component_prop_id (
                                         p_component_type_id => nvl( p_parent_comp_type_id, p_component_type_id ),
                                         p_property_name     => p_parent_comp_prop_name );
        elsif l_property_type = 'REGION COLUMN' then
            l_parent_comp_prop_id := get_component_prop_id (
                                         p_component_type_id => c_comp_type_region,
                                         p_property_name     => 'REGION_SQL' );
        elsif l_property_type = 'COLUMN' then
            raise_application_error( -20999, 'Column property requires a parent property' );
        end if;

        if l_property_type not in ( 'COMBOBOX', 'CSS', 'JAVASCRIPT', 'HTML', 'LINK', 'LINK EXTENDED', 'TEXT', 'TEXT EDITOR', 'TEXTAREA' ) then
            l_supports_substitution       := 'N';
            l_supports_template_directive := null;
        end if;

        if l_property_type = 'YES NO' then
            l_is_required := 'Y';
        end if;

        if l_property_type in ( 'ITEM', 'FACET', 'SFILTER', 'ORDER BY ITEM' ) and l_reference_on_delete is null then
            l_reference_on_delete := 'WEAK_NULL';
        end if;

        -- Check if we do have a valid depending-on config
        check_depending_on (
            p_condition_type => p_depending_on_condition_type,
            p_expression     => p_depending_on_expression );

        check_depending_on (
            p_condition_type => p_depending_on_condition_type2,
            p_expression     => p_depending_on_expression2 );

        insert into wwv_pe_component_props (
            id,
            component_type_id,
            property_id,
            db_column,
            import_api_parameter,
            view_column,
            display_sequence,
            is_required,
            check_uniqueness,
            is_common,
            is_read_only,
            reference_on_delete,
            reference_scope,
            supports_substitution,
            supports_template_directive,
            custom_data_type,
            default_value,
            null_text,
            null_help_text,
            plugin_std_attributes,
            features_comp_prop_id,
            features,
            depending_on_comp_prop_id,
            depending_on_has_to_exist,
            depending_on_condition_type,
            depending_on_expression,
            depending_on_comp_prop_id2,
            depending_on_has_to_exist2,
            depending_on_condition_type2,
            depending_on_expression2,
            parent_comp_prop_id,
            examples,
            help_text,
            important_for_accessibility,
            accessibility_help_text,
            comment_text )
        values (
            g_component_property_id,
            p_component_type_id,
            l_property_id,
            p_db_column,
            l_import_api_parameter,
            p_view_column,
            g_component_property_sequence,
            l_is_required,
            p_check_uniqueness,
            l_is_common,
            l_is_read_only,
            l_reference_on_delete,
            p_reference_scope,
            l_supports_substitution,
            l_supports_template_directive,
            p_custom_data_type,
            p_default_value,
            l_msg_name_null_text,
            l_msg_name_null_help_text,
            p_plugin_std_attributes,
            l_features_comp_prop_id,
            p_features,
            l_depending_on_com_prop_id,
            l_depending_on_has_to_exist,
            p_depending_on_condition_type,
            p_depending_on_expression,
            l_depending_on_com_prop_id2,
            l_depending_on_has_to_exist2,
            p_depending_on_condition_type2,
            p_depending_on_expression2,
            l_parent_comp_prop_id,
            l_msg_name_examples,
            l_msg_name_comp_prop_help,
            l_important_for_accessibility,
            l_msg_name_prop_acc_help,
            p_comment_text )
        ;
        --
        g_component_property_id       := g_component_property_id + 1;
        g_component_property_sequence := g_component_property_sequence + 10;
        --
    exception when others then
        raise_application_error( -20999, 'Type: ' || p_component_type_id || ', Name: ' || p_property_name || ', SQLERRM: ' || sqlerrm );
    end store_component_prop;
    --
    --
    procedure store_plugin_attr_seq_offset (
        p_component_type_id in number )
    is
    begin
        update wwv_pe_component_types
           set plugin_attributes_seq_offset = g_component_property_sequence
         where id = p_component_type_id;
        --
        g_component_property_sequence := g_component_property_sequence + 2500; -- 25 custom attributes
    end store_plugin_attr_seq_offset;
    --
    --
    procedure store_comp_type_tmpl_option (
        p_component_type_id     in number,
        p_parent_comp_type_id   in number,
        p_type_name             in varchar2,
        p_edit_page_no          in integer   )
    is
    begin
        store_component_type (
            p_id                        => p_component_type_id,
            p_name                      => upper(p_type_name)||'_TEMPLATE_OPTION',
            p_singular_title            => 'Template Option',
            p_plural_title              => 'Template Options',
            p_table_name                => 'WWV_FLOW_TEMPLATE_OPTIONS',
            p_view_name                 => 'APEX_APPL_TEMPLATE_OPTIONS',
            p_import_api_name           => 'wwv_flow_imp_shared.create_template_option',
            p_parent_component_type_id  => p_parent_comp_type_id,
            p_parent_fk_column          => upper(p_type_name)||'_TEMPLATE_ID',
            p_display_column            => 'DISPLAY_NAME',
            p_where_clause              => lower(p_type_name)||'_template_id is not null',
            p_edit_url                  => get_parent_component_type_url (
                                                p_component_type_id => p_parent_comp_type_id,
                                                p_url_type          => 'EDIT' ) );
    end store_comp_type_tmpl_option;
    --
    --
    procedure store_comp_type_data_profile (
        p_profile_type_id           in number,
        p_column_type_id            in number,
        p_name_prefix               in varchar2,
        p_parent_type_id            in number,
        p_parent_fk_column          in varchar2 )
    is
    begin
        store_component_type (
            p_id                       => p_profile_type_id,
            p_name                     => p_name_prefix || '_DATA_PROFILE',
            p_singular_title           => 'Data Profile',
            p_plural_title             => 'Data Profiles',
            p_table_name               => 'WWV_FLOW_DATA_PROFILES_DEV',
            p_import_api_name          => 'wwv_flow_imp_shared.create_data_profile',
            p_parent_component_type_id => p_parent_type_id,
            p_parent_fk_column         => p_parent_fk_column,
            p_display_column           => 'NAME',
            p_create_url               => null,
            p_edit_url                 => 'f?p=4000:1925:%session%:::1925:P1925_DATA_PROFILE_ID,FB_FLOW_ID:%pk_value%,%application_id%' );
        --
        store_component_type (
            p_id                       => p_column_type_id,
            p_name                     => p_name_prefix || '_DATA_PROFILE_COLUMN',
            p_singular_title           => 'Column',
            p_plural_title             => 'Columns',
            p_table_name               => 'WWV_FLOW_DATA_PROFILE_COLS_DEV',
            p_import_api_name          => 'wwv_flow_imp_shared.create_data_profile_col',
            p_parent_component_type_id => p_profile_type_id,
            p_parent_fk_column         => 'DATA_PROFILE_ID',
            p_display_column           => 'NAME',
            p_where_clause             => p_parent_fk_column || ' is not null',
            p_create_url               => null,
            p_edit_url                 => 'f?p=4000:1926:%session%:::1926:P1926_DATA_PROFILE_COLUMN_ID,FB_FLOW_ID:%pk_value%,%application_id%' );
    end store_comp_type_data_profile;
    --
    --
    procedure store_order_by_clause_property (
        p_name               in varchar2,
        p_display_group_name in varchar2 )
    is
    begin
        store_property (
            p_id                 => l_property_id,
            p_name               => p_name,
            p_prompt             => 'Order By Clause',
            p_property_type      => 'ORDER BY CLAUSE',
            p_db_column          => 'QUERY_ORDER_BY',
            p_display_length     => 40,
            p_max_length         => 255,
            p_display_group_name => p_display_group_name,
            p_examples           => q'~
    <dl><dt>To show the newest hires first and those which have not been hired yet last. If the date is equal, sort them by department number.</dt>
    <dd><pre>
    HIREDATE DESC NULLS LAST, DEPTNO
    </pre></dd>
    </dl>~',
            p_help_text          => q'~<p>Enter one or more comma separated column names, including directives, to define the order in which records are retrieved.</p>
            <p>Directives include:
              <ul>
                <li>ASC - Return column in ascending order</li>
                <li>DESC - Return column in descending order</li>
                <li>NULLS FIRST - Return column with no value before those with values</li>
                <li>NULLS LAST - Return column with values before those with no value</li>
              </ul>
            </p>~' );
    end store_order_by_clause_property;
    --
    --
    procedure store_ext_order_by_property (
        p_name               in varchar2 )
    is
    begin
        store_property (
            p_id                 => l_property_id,
            p_name               => p_name,
            p_prompt             => 'Order By',
            p_property_type      => 'TEXT EDITOR',
            p_db_column          => 'EXTERNAL_ORDER_BY_EXPR',
            p_display_length     => 40,
            p_max_length         => 255,
            p_display_group_name => 'EXTERNAL_FILTER_ORDER_BY',
            p_examples             => q'~
    <dl><dt>For an ORDS REST Service, order by the "job" attribute ascending, then by "ename", descending.</dt>
    <dd><pre>
    {"job": "asc", "ename": "desc"}
    </pre></dd>
    </dl>
    <dl><dt>For an Oracle Cloud Applications (SaaS) REST Service, order by the "job" attribute ascending, then by "ename", descending.</dt>
    <dd><pre>
    job:asc,ename:desc
    </pre></dd>
    </dl>~',
            p_help_text            => q'~Provide an order by expression involving one or more columns to be passed to the external REST service. The expression will be passed to the remote server as-is. The syntax depends on the type of the REST Data Source. For example, Oracle REST Data Services (ORDS) expects JSON syntax while Oracle Cloud Applications (SaaS) expects a comma-separated list of attribute names with optional direction indicator.~' );
    end store_ext_order_by_property;
    --
    --
    procedure store_lov_type_property (
        p_name in varchar2 )
    is
    begin
        store_property (
            p_id                 => l_property_id,
            p_name               => p_name,
            p_prompt             => 'Type',
            p_property_type      => 'SELECT LIST',
            p_db_column          => 'LOV_TYPE',
            p_display_group_name => 'LIST_OF_VALUES',
            p_lov_type           => 'STATIC',
            p_help_text          => q'~Select the list of values type.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Shared Component',
            p_return_value  => 'SHARED',
            p_help_text     => q'~The list of values is based on a predefined list of values, defined in Shared Components.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'SQL Query',
            p_return_value  => 'SQL_QUERY',
            p_help_text     => q'~The Dynamic list of values is based on the SQL Query you enter.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Static Values',
            p_return_value  => 'STATIC',
            p_help_text     => q'~The Static list of values is based on the text value you enter.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Function Body returning SQL Query', -- @@@
            p_return_value  => 'FUNCTION_BODY',
            p_help_text     => q'~The Dynamic list of values is based on the SQL Query returned by the entered Function Body you enter.~' );
        --
        if p_name = 'FACET_LOV_TYPE' then
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Distinct Values',
                p_return_value  => 'DISTINCT',
                p_help_text     => q'~The list of values is based on an automatically generated query that selects the distinct column values.~' );
        end if;
        --
    end store_lov_type_property;
    --
    --
    procedure store_process_point_property (
        p_property_id in number,
        p_name        in varchar2 )
    is
    begin
        store_property (
            p_id                 => p_property_id,
            p_name               => p_name,
            p_prompt             => 'Point',
            p_property_type      => 'SELECT LIST',
            p_db_column          => 'PROCESS_POINT',
            p_display_group_name => 'EXECUTION',
            p_lov_type           => 'STATIC',
            p_help_text          => q'~Select at what point in page rendering or processing this process is executed.~' );
        --
        if p_name = 'APP_PROCESS_POINT' then
            store_property_lov_values (
                p_property_id   => p_property_id,
                p_display_value => 'After Authentication',
                p_return_value  => 'AFTER_LOGIN',
                p_help_text     => q'~Executes after successful authentication.~' );
        end if;
        --
        store_property_lov_values (
            p_property_id   => p_property_id,
            p_display_value => 'New Session',
            p_return_value  => 'ON_NEW_INSTANCE',
            p_help_text     => q'~Executes when a new session is generated.~' );
        --
        store_property_lov_values (
            p_property_id   => p_property_id,
            p_display_value => 'Before Header',
            p_return_value  => 'BEFORE_HEADER',
            p_help_text     => q'~Rendering: Executes before the page renders.~' );
        --
        store_property_lov_values (
            p_property_id   => p_property_id,
            p_display_value => 'After Header',
            p_return_value  => 'AFTER_HEADER',
            p_help_text     => q'~Rendering: Executes after the header renders.~' );
        --
        store_property_lov_values (
            p_property_id   => p_property_id,
            p_display_value => 'Before Regions',
            p_return_value  => 'BEFORE_BOX_BODY',
            p_is_quick_pick => 'Y',
            p_help_text     => q'~Rendering: Executes before the regions render.~' );
        --
        store_property_lov_values (
            p_property_id   => p_property_id,
            p_display_value => 'After Regions',
            p_return_value  => 'AFTER_BOX_BODY',
            p_help_text     => q'~Rendering: Executes after the regions render.~' );
        --
        store_property_lov_values (
            p_property_id   => p_property_id,
            p_display_value => 'Before Footer',
            p_return_value  => 'BEFORE_FOOTER',
            p_help_text     => q'~Rendering: Executes before the footer renders.~' );
        --
        store_property_lov_values (
            p_property_id   => p_property_id,
            p_display_value => 'After Footer',
            p_return_value  => 'AFTER_FOOTER',
            p_help_text     => q'~Rendering: Executes after the page renders.~' );
        --
        store_property_lov_values (
            p_property_id   => p_property_id,
            p_display_value => 'After Submit',
            p_return_value  => 'ON_SUBMIT_BEFORE_COMPUTATION',
            p_help_text     => q'~Processing: Executes after the page is submitted, but before computations, validations, and page processes performed.~' );
        --
        store_property_lov_values (
            p_property_id   => p_property_id,
            p_display_value => 'Processing',
            p_return_value  => 'AFTER_SUBMIT',
            p_is_quick_pick => 'Y',
            p_help_text     => q'~Processing: Executes after validations and computations performed, when page processing performed.~' );
        --
        store_property_lov_values (
            p_property_id   => p_property_id,
            p_display_value => 'Ajax Callback',
            p_return_value  => 'ON_DEMAND',
            p_help_text     => q'~Executes when called by apex.server JavaScipt APIs.~' );
    end store_process_point_property;
    --
    --
    procedure store_default_svg_shapes_lov(
        p_property_id  in number )
    is
        c_shape_names constant wwv_flow_t_varchar2 :=
            wwv_flow_t_varchar2(
                'Default',
                'Camera',
                'Circle',
                'Circle Alternative',
                'Face Frown',
                'Face Meh',
                'Face Smile',
                'Square',
                'Square Alternative',
                'Pin Circle',
                'Pin Square',
                'Heart',
                'Pentagon',
                'Triangle',
                'Check',
                'Home',
                'Slash',
                'Shine',
                'Street View',
                'Flag Pennant',
                'Flag Swallowtail');

        c_shape_svgs constant wwv_flow_t_varchar2 :=
            wwv_flow_t_varchar2(
                -- default
                'M10 2C6.7 2 4 4.7 4 8c0 4.6 5.4 9.7 5.7 9.9.2.2.5.2.7 0 .2-.2 5.6-5.3 5.6-9.9 0-3.3-2.7-6-6-6zm0 7.9c-1.1 0-2-.9-2-2s.9-2 2-2 2 .9 2 2-.9 2-2 2z',
                -- Camera
                'M15 4h-1.2l-.9-1.2c-.4-.5-1-.8-1.6-.8H8.8c-.7 0-1.3.3-1.6.8L6.2 4H5c-1.1 0-2 .9-2 2v5c0 1.1.9 2 2 2h2.2l2.4 4.7c.1.2.4.3.7.2.1 0 .2-.1.2-.2l2.4-4.7H15c1.1 0 2-.9 2-2V6c0-1.1-.9-2-2-2zm-5 7c-1.4 0-2.5-1.1-2.5-2.5S8.6 6 10 6s2.5 1.1 2.5 2.5S11.4 11 10 11z',
                -- Circle
                'M10 2C6.7 2 4 4.7 4 8c0 2.5 1.6 4.7 3.9 5.6l1.6 4.1c.1.3.4.4.7.3l.3-.3 1.6-4.1c3.1-1.2 4.7-4.6 3.5-7.7C14.7 3.6 12.5 2 10 2zm0 8c-1.1 0-2-.9-2-2s.9-2 2-2 2 .9 2 2-.9 2-2 2z',
                -- Circle Alternative
                'M10 2C6.7 2 4 4.7 4 8v9.5c0 .2.1.4.3.5h.2c.1 0 .3-.1.4-.2l3.7-4c3.2.8 6.5-1.2 7.3-4.4.8-3.2-1.2-6.5-4.4-7.3-.5 0-1-.1-1.5-.1zm0 8c-1.1 0-2-.9-2-2s.9-2 2-2 2 .9 2 2-.9 2-2 2z',
                -- Face Frown
                'M10 2C6.7 2 4 4.7 4 8c0 3.7 5.4 9.6 5.6 9.8.2.2.5.2.7 0 .3-.2 5.7-6.1 5.7-9.8 0-3.3-2.7-6-6-6zM7.2 7c0-.4.3-.8.8-.8.4 0 .8.3.8.8s-.4.8-.8.8-.8-.4-.8-.8zm5 3.8c-.2.1-.4.1-.5-.1-.7-.9-2-1.1-3-.4-.2.1-.3.3-.4.4-.1.2-.4.2-.5.1-.2-.1-.2-.4-.1-.5 1-1.3 2.8-1.5 4-.6l.6.6c.1.1.1.4-.1.5zm-.2-3c-.4 0-.8-.3-.8-.8s.3-.8.8-.8.8.3.8.8-.4.8-.8.8z',
                -- Face Meh
                'M10 2C6.7 2 4 4.7 4 8c0 3.7 5.4 9.6 5.6 9.8.2.2.5.2.7 0 .3-.2 5.7-6.1 5.7-9.8 0-3.3-2.7-6-6-6zM7.2 7c0-.4.3-.8.8-.8.4 0 .8.3.8.8s-.4.8-.8.8-.8-.4-.8-.8zm4.3 3.9h-3c-.2 0-.4-.2-.4-.4s.2-.4.4-.4h3c.2 0 .4.2.4.4s-.2.4-.4.4zm.5-3.1c-.4 0-.8-.3-.8-.8s.3-.8.8-.8.8.3.8.8-.4.8-.8.8z',
                -- Face Smile
                'M10 2C6.7 2 4 4.7 4 8c0 3.7 5.4 9.6 5.6 9.8.2.2.5.2.7 0 .3-.2 5.7-6.1 5.7-9.8 0-3.3-2.7-6-6-6zM7.2 7c0-.4.3-.8.8-.8.4 0 .8.3.8.8s-.4.8-.8.8-.8-.4-.8-.8zm5.1 2.7c-1 1.3-2.8 1.5-4 .6l-.6-.6c-.1-.1-.1-.4.1-.5.2-.1.4-.1.5.1.7.9 2 1.1 3 .4.2-.1.3-.3.4-.4.1-.2.4-.2.5-.1s.2.4.1.5zM12 7.8c-.4 0-.8-.3-.8-.8s.3-.8.8-.8.8.3.8.8-.4.8-.8.8z',
                -- Square
                'M14 2H6c-1.1 0-2 .9-2 2v8c0 1.1.9 2 2 2h2.1l1.5 3.7c.1.3.4.4.7.3l.3-.3 1.5-3.7H14c1.1 0 2-.9 2-2V4c0-1.1-.9-2-2-2zm-4 8c-1.1 0-2-.9-2-2s.9-2 2-2 2 .9 2 2-.9 2-2 2z',
                -- Square Alternative
                'M14 2H6c-1.1 0-2 .9-2 2v13.5c0 .2.1.4.3.5h.2c.1 0 .3-.1.4-.2L8.4 14H14c1.1 0 2-.9 2-2V4c0-1.1-.9-2-2-2zm-4 8c-1.1 0-2-.9-2-2s.9-2 2-2 2 .9 2 2-.9 2-2 2z',
                -- Pin Circle
                'M10 2C7.2 2 5 4.2 5 7c0 2.6 1.9 4.7 4.5 5v5.6c0 .3.2.5.5.5s.5-.2.5-.5V12c2.7-.3 4.7-2.7 4.5-5.4-.3-2.7-2.4-4.6-5-4.6z',
                -- Pin Square
                'M13 2H7c-1.1 0-2 .9-2 2v6c0 1.1.9 2 2 2h2.5v5.5c0 .3.2.5.5.5s.5-.2.5-.5V12H13c1.1 0 2-.9 2-2V4c0-1.1-.9-2-2-2z',
                -- Heart
                'M16 5c0-.8-.3-1.6-.9-2.1-1.2-1.2-3.1-1.2-4.3 0l-.8.8-.8-.8C8 1.7 6.1 1.7 4.9 2.9c-1.2 1.1-1.2 3 0 4.2l4.6 4.6v5.8c0 .3.2.5.5.5s.5-.2.5-.5v-5.8l4.6-4.6c.6-.5.9-1.3.9-2.1z',
                -- Pentagon
                'M15 5.5l-4.7-3.4c-.2-.1-.4-.1-.6 0L5 5.5c-.2.2-.3.4-.2.6l1.8 5.6c.1.2.3.3.5.3h2.4v5.5c0 .3.2.5.5.5s.5-.2.5-.5V12h2.4c.2 0 .4-.1.5-.3l1.8-5.6c.1-.2 0-.4-.2-.6z',
                -- Triangle
                'M15.9 11.2l-5.5-9c-.1-.2-.5-.3-.7-.1-.1 0-.1.1-.1.1l-5.5 9c-.1.2-.1.5.2.7 0 .1.1.1.2.1h5v5.5c0 .3.2.5.5.5s.5-.2.5-.5V12h5c.3 0 .5-.2.5-.5 0-.1 0-.2-.1-.3z',
                -- Check
                'M10 2C6.7 2 4 4.7 4 8c0 3.7 5.4 9.6 5.6 9.8.2.2.5.2.7 0 .3-.2 5.7-6.1 5.7-9.8 0-3.3-2.7-6-6-6zm2.9 4.9l-3 3c-.2.1-.6.1-.8 0L7.6 8.4c-.2-.2-.2-.5 0-.7s.5-.2.7 0l1.1 1.1L12 6.2c.2-.2.5-.2.7 0s.3.5.2.7z',
                -- Home
                '<path d="M17.8 7.6l-7.5-5.5c-.2-.1-.4-.1-.6 0L2.2 7.5c-.2.2-.3.5-.1.7.2.2.5.3.7.1L4 7.5v6c0 .3.2.5.5.5h3.2l1.9 3.7c.1.2.4.3.7.2.1 0 .2-.1.2-.2l1.9-3.7h3.2c.3 0 .5-.2.5-.5v-6l1.2.9c.2.2.5.1.7-.1.1-.3 0-.6-.2-.7z"/><path d="M8.5 8.5v2c0 .3.2.5.5.5h2c.3 0 .5-.2.5-.5v-2c0-.3-.2-.5-.5-.5H9c-.3 0-.5.2-.5.5z" fill="#fff"/>',
                -- Slash
                'M17.9 2.1c-.2-.2-.5-.2-.7 0l-2.3 2.3C13 1.7 9.2 1.1 6.5 3.1 4.9 4.3 4 6.1 4 8c.1 1.9.8 3.8 1.9 5.4l-3.8 3.8c-.2.2-.2.5 0 .7s.5.2.7 0l15-15c.2-.2.2-.6.1-.8zM9.5 9.8c-1.1-.3-1.7-1.4-1.4-2.4S9.5 5.7 10.5 6c.7.2 1.2.7 1.4 1.4L9.5 9.8zM7.1 15c.8 1 1.6 2 2.6 2.8.2.2.5.2.7 0 .2-.1 5.6-5.2 5.6-9.8 0-.6-.1-1.1-.2-1.6L7.1 15z',
                -- Shine
                'M10 7c-2.2 0-4 1.8-4 4 0 2.3 3.2 6.4 3.6 6.8.2.2.5.2.7.1l.1-.1c.4-.5 3.6-4.5 3.6-6.8 0-2.2-1.8-4-4-4zm0 5c-.6 0-1-.4-1-1s.4-1 1-1 1 .4 1 1-.4 1-1 1zM3.6 4.9c-.2-.1-.5-.1-.7.1-.2.3-.1.6.1.7l1.9 1.4c0 .1.1.1.3.1.3 0 .5-.2.5-.5 0-.2-.1-.3-.2-.4L3.6 4.9zm4 .1c.1.2.3.3.5.3h.2c.3-.1.4-.4.3-.6l-.9-2.4c0-.2-.3-.4-.6-.3-.3.1-.4.4-.3.6L7.6 5zm9.5 0c-.2-.2-.5-.3-.7-.1l-1.9 1.4c-.2.2-.3.5-.1.7.1.1.2.2.4.2.1 0 .2 0 .3-.1L17 5.7c.2-.1.3-.4.1-.7zm-5.3.3h.2c.2 0 .4-.1.5-.3l.8-2.3c.1-.3 0-.5-.3-.6-.3-.1-.5 0-.6.3l-.8 2.3c-.2.2-.1.5.2.6z',
                -- Street View
                '<path d="M14.7 11.6c-.3-.1-.5.1-.6.3s.1.5.3.6c1.6.5 2.6 1.2 2.6 1.9 0 1.2-2.9 2.5-7 2.5s-7-1.3-7-2.5c0-.7 1-1.4 2.6-1.9.3 0 .4-.3.3-.5-.1-.3-.4-.4-.6-.3-2.1.6-3.3 1.6-3.3 2.8 0 2 3.4 3.5 8 3.5s8-1.5 8-3.5c0-1.2-1.2-2.2-3.3-2.9z"/><path d="M7.7 10.9c.2.2.3.5.3.7V14c0 .6.4 1 1 1h2c.6 0 1-.4 1-1v-2.4c0-.3.1-.5.3-.7l.1-.1c.4-.4.6-.9.6-1.4V7.8c0-.7-.3-1.3-.9-1.6-1.2 1.2-3.1 1.2-4.2 0-.6.3-.9.9-.9 1.6v1.6c0 .5.2 1 .6 1.4l.1.1z"/><circle cx="10" cy="4" r="2"/>',
                -- Flag Pennant
                'M16.7 6.5L3.7 2c-.3-.1-.6.1-.7.3v15.2c0 .3.2.5.5.5s.5-.2.5-.5v-5.6l12.7-4.4c.3-.1.4-.4.3-.6-.1-.2-.2-.3-.3-.4z',
                -- Flag Swallowtail
                'M13.2 7l3.7-4.2c.2-.2.2-.5 0-.7-.2-.1-.3-.1-.4-.1h-13c-.3 0-.5.2-.5.5v15c0 .3.2.5.5.5s.5-.2.5-.5V12h12.5c.3 0 .5-.2.5-.5 0-.1 0-.2-.1-.3L13.2 7z' );

    begin
        for i in 1 .. c_shape_names.count loop
            store_property_lov_values (
                p_property_id   => p_property_id,
                p_display_value => c_shape_names(i),
                p_return_value  => c_shape_names(i),
                p_help_text     =>    '<svg xmlns="http://www.w3.org/2000/svg" width="20px" height="20px" paint-order="stroke" viewBox="0 0 20 20">'
                                   || case when substr( c_shape_svgs( i ), 1, 5 ) != '<path' then '<path d="' end
                                   || c_shape_svgs( i )
                                   || case when substr( c_shape_svgs( i ), 1, 5 ) != '<path' then '"/>' end
                                   || '</svg>' );

        end loop;
    end store_default_svg_shapes_lov;
    --
    --
    procedure store_color_spectrum_property (
        p_start_property_id  in number,
        p_name_prefix        in varchar2,
        p_db_column_prefix   in varchar2,
        p_display_group_name in varchar2 )
    is
        l_property_id number := p_start_property_id;

        ------------------------------------------------------------------------
        function generate_color_html(
            p_color_list in wwv_flow_t_varchar2 )
            return varchar2
        is
            l_html  varchar2(32767);
            l_width varchar2(4) := '30px';
        begin
            l_html := '<div aria-hidden="true">';
            for i in 1 .. p_color_list.count loop
                l_html := l_html || '<div style="display:inline-block;background-color:' || p_color_list( i ) || '; width:' || l_width || ';">&nbsp;</div>';
            end loop;
            l_html := l_html || '</div>';
            return l_html;
        end generate_color_html;

    begin
        store_property (
            p_id                          => l_property_id,
            p_name                        => p_name_prefix || 'TYPE',
            p_prompt                      => 'Color Scheme',
            p_property_type               => 'SELECT LIST',
            p_lov_type                    => 'STATIC',
            p_db_column                   => p_db_column_prefix || 'TYPE',
            p_display_group_name          => p_display_group_name,
            p_help_text                   => q'~Choose the color scheme type.~' );
        --
        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Sequential',
            p_return_value  => 'SEQUENTIAL',
            p_help_text     => 'This color scheme is suitable to visualize sequential values, going from a low to a high value.' );
        --
        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Diverging',
            p_return_value  => 'DIVERGING',
            p_help_text     => 'This color scheme is suitable to visualize diverging values, going from one extreme to another.' );
        --
        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Qualitative',
            p_return_value  => 'QUALITATIVE',
            p_help_text     => 'This color scheme is suitable to visualize distinct values.' );
        --
        l_property_id := l_property_id + 1;
        --
        store_property (
            p_id                          => l_property_id,
            p_name                        => p_name_prefix || 'SEQ_NAME',
            p_prompt                      => 'Scheme Name',
            p_property_type               => 'SELECT LIST',
            p_lov_type                    => 'STATIC',
            p_db_column                   => p_db_column_prefix || 'NAME',
            p_display_group_name          => p_display_group_name,
            p_help_text                   => q'~Choose one of the predefined sequential color schemes. These are suitable to visualize sequential values, going from a low to a high value. Color schemes from cartocolors (<a rel="noopener noreferrer" href="https://github.com/CartoDB/cartocolor" target="_blank">CARTOcolor on GitHub)</a>.~' );
        --
        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Burgundy',
            p_return_value  => 'Burg',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#ffc6c4','#f4a3a8','#e38191','#cc607d','#ad466c','#8b3058','#672044' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Burgundy to Yellow',
            p_return_value  => 'BurgYl',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#fbe6c5','#f5ba98','#ee8a82','#dc7176','#c8586c','#9c3f5d','#70284a' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Red to Orange',
            p_return_value  => 'RedOr',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#f6d2a9','#f5b78e','#f19c7c','#ea8171','#dd686c','#ca5268','#b13f64' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Orange to Yellow',
            p_return_value  => 'OrYel',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#ecda9a','#efc47e','#f3ad6a','#f7945d','#f97b57','#f66356','#ee4d5a' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Peach',
            p_return_value  => 'Peach',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#fde0c5','#facba6','#f8b58b','#f59e72','#f2855d','#ef6a4c','#eb4a40' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Pink to Yellow',
            p_return_value  => 'PinkYl',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#fef6b5','#ffdd9a','#ffc285','#ffa679','#fa8a76','#f16d7a','#e15383' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Mint',
            p_return_value  => 'Mint',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#e4f1e1','#b4d9cc','#89c0b6','#63a6a0','#448c8a','#287274','#0d585f' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Blue to Green',
            p_return_value  => 'BluGrn',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#c4e6c3','#96d2a4','#6dbc90','#4da284','#36877a','#266b6e','#1d4f60' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Dark Mint',
            p_return_value  => 'DarkMint',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#d2fbd4','#a5dbc2','#7bbcb0','#559c9e','#3a7c89','#235d72','#123f5a' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Emerald',
            p_return_value  => 'Emrld',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#d3f2a3','#97e196','#6cc08b','#4c9b82','#217a79','#105965','#074050' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Blue to Yellow',
            p_return_value  => 'BluYl',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#f7feae','#b7e6a5','#7ccba2','#46aea0','#089099','#00718b','#045275' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Teal',
            p_return_value  => 'Teal',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#d1eeea','#a8dbd9','#85c4c9','#68abb8','#4f90a6','#3b738f','#2a5674' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Teal to Green',
            p_return_value  => 'TealGrn',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#b0f2bc','#89e8ac','#67dba5','#4cc8a3','#38b2a3','#2c98a0','#257d98' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Purple',
            p_return_value  => 'Purp',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#f3e0f7','#e4c7f1','#d1afe8','#b998dd','#9f82ce','#826dba','#63589f' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Purple to Orange',
            p_return_value  => 'PurpOr',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#f9ddda','#f2b9c4','#e597b9','#ce78b3','#ad5fad','#834ba0','#573b88' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Sunset',
            p_return_value  => 'Sunset',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#f3e79b','#fac484','#f8a07e','#eb7f86','#ce6693','#a059a0','#5c53a5' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Magenta',
            p_return_value  => 'Magenta',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#f3cbd3','#eaa9bd','#dd88ac','#ca699d','#b14d8e','#91357d','#6c2167' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Dark Sunset',
            p_return_value  => 'SunsetDark',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#fcde9c','#faa476','#f0746e','#e34f6f','#dc3977','#b9257a','#7c1d6f' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Brown to Yellow',
            p_return_value  => 'BrwnYl',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#ede5cf','#e0c2a2','#d39c83','#c1766f','#a65461','#813753','#541f3f' ) ) );

        --
        l_property_id := l_property_id + 1;
        --
        store_property (
            p_id                          => l_property_id,
            p_name                        => p_name_prefix || 'DIV_NAME',
            p_prompt                      => 'Scheme Name',
            p_property_type               => 'SELECT LIST',
            p_lov_type                    => 'STATIC',
            p_db_column                   => p_db_column_prefix || 'NAME',
            p_display_group_name          => p_display_group_name,
            p_help_text                   => q'~Choose one of the predefined diverging color schemes. These are suitable to visualize diverging values, going from one extreme to another. Color schemes from cartocolors (<a rel="noopener noreferrer" href="https://github.com/CartoDB/cartocolor" target="_blank">CARTOcolor on GitHub)</a>.~' );
        --
        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => '"Army" to Rose',
            p_return_value  => 'ArmyRose',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#798234','#a3ad62','#d0d3a2','#fdfbe4','#f0c6c3','#df91a3','#d46780' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Fall Colors',
            p_return_value  => 'Fall',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#3d5941','#778868','#b5b991','#f6edbd','#edbb8a','#de8a5a','#ca562c' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Geyser',
            p_return_value  => 'Geyser',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#008080','#70a494','#b4c8a8','#f6edbd','#edbb8a','#de8a5a','#ca562c' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Temps',
            p_return_value  => 'Temps',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#009392','#39b185','#9ccb86','#e9e29c','#eeb479','#e88471','#cf597e' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Teal to Rose',
            p_return_value  => 'TealRose',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#009392','#72aaa1','#b1c7b3','#f1eac8','#e5b9ad','#d98994','#d0587e' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Tropic Colors',
            p_return_value  => 'Tropic',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#009B9E','#42B7B9','#A7D3D4','#F1F1F1','#E4C1D9','#D691C1','#C75DAB' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Earth Colors',
            p_return_value  => 'Earth',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#A16928','#bd925a','#d6bd8d','#edeac2','#b5c8b8','#79a7ac','#2887a1' ) ) );

        --
        l_property_id := l_property_id + 1;
        --
        store_property (
            p_id                          => l_property_id,
            p_name                        => p_name_prefix || 'QUA_NAME',
            p_prompt                      => 'Scheme Name',
            p_property_type               => 'SELECT LIST',
            p_lov_type                    => 'STATIC',
            p_db_column                   => p_db_column_prefix || 'NAME',
            p_display_group_name          => p_display_group_name,
            p_help_text                   => q'~Choose one of the predefined qualitative color schemes. These are suitable to visualize distinct values. Color schemes from cartocolors (<a rel="noopener noreferrer" href="https://github.com/CartoDB/cartocolor" target="_blank">CARTOcolor on GitHub)</a>.~' );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Antique',
            p_return_value  => 'Antique',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#855C75','#D9AF6B','#AF6458','#736F4C','#526A83','#625377','#7C7C7C' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Bold',
            p_return_value  => 'Bold',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#7F3C8D','#11A579','#3969AC','#F2B701','#E73F74','#80BA5A','#A5AA99' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Pastel',
            p_return_value  => 'Pastel',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#66C5CC','#F6CF71','#F89C74','#DCB0F2','#87C55F','#9EB9F3','#B3B3B3' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Prism',
            p_return_value  => 'Prism',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#5F4690','#1D6996','#38A6A5','#0F8554','#73AF48','#EDAD08','#666666' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Safe',
            p_return_value  => 'Safe',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#88CCEE','#CC6677','#DDCC77','#117733','#332288','#AA4499','#888888' ) ) );

        store_property_lov_values(
            p_property_id   => l_property_id,
            p_display_value => 'Vivid',
            p_return_value  => 'Vivid',
            p_help_text     => generate_color_html( wwv_flow_t_varchar2( '#E58606','#5D69B1','#52BCA3','#99C945','#CC61B0','#24796C','#A5AA99' ) ) );
    end store_color_spectrum_property;
    --
    --
    procedure store_code_language_properties (
        p_property_id        in number,
        p_name               in varchar2,
        p_db_column          in varchar2,
        p_display_group_name in varchar2,
        p_include_sql        in boolean,
        p_help_text          in varchar2 )
    is
    begin
        store_property (
            p_id                 => p_property_id,
            p_name               => p_name,
            p_prompt             => 'Language',
            p_property_type      => 'CODE LANGUAGE',
            p_db_column          => p_db_column,
            p_display_group_name => p_display_group_name,
            p_lov_type           => 'STATIC',
            p_help_text          => p_help_text );
        --
        if p_include_sql then
            store_property_lov_values (
                p_property_id   => p_property_id,
                p_display_value => 'SQL',
                p_return_value  => 'SQL' );
        end if;
        --
        store_property_lov_values (
            p_property_id   => p_property_id,
            p_display_value => 'PL/SQL',
            p_return_value  => 'PLSQL',
            p_is_quick_pick => 'Y' );
        --
        store_property_lov_values (
            p_property_id   => p_property_id,
            p_display_value => 'JavaScript (MLE)',
            p_return_value  => 'JAVASCRIPT',
            p_is_quick_pick => 'Y' );
        --
        store_property_lov_values (
            p_property_id   => p_property_id,
            p_display_value => 'Java (MLE)',
            p_return_value  => 'JAVA' );
        --
        store_property_lov_values (
            p_property_id   => p_property_id,
            p_display_value => 'Python (MLE)',
            p_return_value  => 'PYTHON' );
    end store_code_language_properties;
    --
    --
    procedure store_expression_properties (
        p_start_property_id  in number,
        p_name_prefix        in varchar2,
        p_return_type        in varchar2,
        p_custom_data_type   in varchar2 default null,
        p_db_column_lang     in varchar2,
        p_db_column_expr     in varchar2,
        p_display_group_name in varchar2,
        p_help_text          in varchar2,
        p_examples_sql       in varchar2 default null,
        p_examples_plsql     in varchar2 default null,
        p_examples_js        in varchar2 default null,
        p_examples_java      in varchar2 default null,
        p_examples_python    in varchar2 default null )
    is
        c_postfix constant varchar2( 30 ) := '_' || p_return_type;
        --
        l_property_id number := p_start_property_id;
    begin
        store_code_language_properties (
            p_property_id        => l_property_id,
            p_name               => p_name_prefix || 'EXPRESSION_LANG' || c_postfix,
            p_db_column          => p_db_column_lang,
            p_display_group_name => p_display_group_name,
            p_include_sql        => true,
            p_help_text          => q'~Select a language that is used to evaluate the entered expression.~' );
        --
        l_property_id := l_property_id + 1;
        store_property (
            p_id                 => l_property_id,
            p_name               => p_name_prefix || 'SQL_EXPRESSION' || c_postfix,
            p_prompt             => 'SQL Expression',
            p_property_type      => 'SQL EXPRESSION' || case when p_return_type in ( 'BOOLEAN', 'CUSTOM' ) then ' ' || p_return_type end,
            p_db_column          => p_db_column_expr,
            p_display_length     => 80,
            p_max_length         => 4000,
            p_display_group_name => p_display_group_name,
            p_help_text          => replace( p_help_text, '%language%', 'SQL' ),
            p_examples           => coalesce(
                                        p_examples_sql,
                                        case p_return_type
                                        when 'BOOLEAN' then
q'~
<dl><dt>To return successfully when the employee is a manager and their department is located in Chicago:</dt>
<dd><pre>
:P2_JOB = 'MANAGER' and
exists ( select 1
           from dept
          where deptno = :P2_DEPTNO
            and loc    = 'CHICAGO' )
</pre></dd>
</dl>~'
                                        else
q'~
<dl><dt>To return YES when the employee is a manager and their department is located in Chicago:</dt>
<dd><pre>
case
when :P2_JOB = 'MANAGER' and
     exists ( select 1
                from dept
               where deptno = :P2_DEPTNO
                 and loc    = 'CHICAGO' ) then 'YES'
else 'NO'
end
</pre></dd>
</dl>~'
                                        end )
            );
        --
        l_property_id := l_property_id + 1;
        store_property (
            p_id                 => l_property_id,
            p_name               => p_name_prefix || 'PLSQL_EXPRESSION' || c_postfix,
            p_prompt             => 'PL/SQL Expression',
            p_property_type      => 'PLSQL EXPRESSION ' || p_return_type,
            p_db_column          => p_db_column_expr,
            p_display_length     => 80,
            p_max_length         => 4000,
            p_display_group_name => p_display_group_name,
            p_help_text          => replace( p_help_text, '%language%', 'PL/SQL' ),
            p_examples           => coalesce(
                                        p_examples_plsql,
                                        case p_return_type
                                        when 'BOOLEAN' then
q'~
<dl><dt>To return successfully when the employee is in department 30 or is a manager:</dt>
<dd><pre>
( :P2_DEPTNO = 30 or :P2_JOB = 'MANAGER' )
</pre></dd>
</dl>~'
                                        else
q'~
<dl><dt>To return YES when the employee is in department 30 or is a manager:</dt>
<dd><pre>
case when ( :P2_DEPTNO = 30 or :P2_JOB = 'MANAGER' ) then 'YES' else 'NO' end
</pre></dd>
</dl>~'
                                        end )
            );
        --
        l_property_id := l_property_id + 1;
        store_property (
            p_id                 => l_property_id,
            p_name               => p_name_prefix || 'JS_EXPRESSION' || c_postfix,
            p_prompt             => 'JavaScript Expression',
            p_property_type      => 'MLE JAVASCRIPT EXPRESSION ' || p_return_type,
            p_db_column          => p_db_column_expr,
            p_display_length     => 80,
            p_max_length         => 4000,
            p_display_group_name => p_display_group_name,
            p_help_text          => replace( p_help_text, '%language%', 'JavaScript' ),
            p_examples           => coalesce(
                                        p_examples_js,
                                        case p_return_type
                                        when 'BOOLEAN' then
q'~
<dl><dt>To return successfully when the employee is in department 30 or is a manager:</dt>
<dd><pre>
( apex.env.P2_DEPTNO === "30" || apex.env.P2_JOB === "MANAGER" )
</pre></dd>
</dl>~'
                                        else
q'~
<dl><dt>To return YES when the employee is in department 30 or is a manager:</dt>
<dd><pre>
( apex.env.P2_DEPTNO === "30" || apex.env.P2_JOB === "MANAGER" ? "YES" : "NO" )
</pre></dd>
</dl>~'
                                        end )
            );
        --
        l_property_id := l_property_id + 1;
        store_property (
            p_id                 => l_property_id,
            p_name               => p_name_prefix || 'JAVA_EXPRESSION' || c_postfix,
            p_prompt             => 'Java Expression',
            p_property_type      => 'MLE JAVA EXPRESSION ' || p_return_type,
            p_db_column          => p_db_column_expr,
            p_display_length     => 80,
            p_max_length         => 4000,
            p_display_group_name => p_display_group_name,
            p_help_text          => replace( p_help_text, '%language%', 'Java' ),
            p_examples           => p_examples_java );
        --
        l_property_id := l_property_id + 1;
        store_property (
            p_id                 => l_property_id,
            p_name               => p_name_prefix || 'PYTHON_EXPRESSION' || c_postfix,
            p_prompt             => 'Python Expression',
            p_property_type      => 'MLE PYTHON EXPRESSION ' || p_return_type,
            p_db_column          => p_db_column_expr,
            p_display_length     => 80,
            p_max_length         => 4000,
            p_display_group_name => p_display_group_name,
            p_help_text          => replace( p_help_text, '%language%', 'Python' ),
            p_examples           => p_examples_python );
        --
    end store_expression_properties;
    --
    --
    procedure store_expression_comp_props (
        p_component_type_id           in number,
        p_name_prefix                 in varchar2,
        p_return_type                 in varchar2,
        p_custom_data_type            in varchar2 default null,
        p_db_column_lang              in varchar2 default null,
        p_db_column_expr              in varchar2 default null,
        p_reference_scope             in varchar2 default 'COMPONENT',
        p_type_property               in varchar2,
        p_depending_on_comp_prop_name in varchar2 default null,
        p_depending_on_condition_type in varchar2 default null,
        p_depending_on_expression     in varchar2 default null )
    is
        c_postfix         constant varchar2( 30 )  := '_' || p_return_type;
        c_expression_lang constant varchar2( 255 ) := p_name_prefix || 'EXPRESSION_LANG' || c_postfix;
    begin
        store_component_prop (
            p_component_type_id            => p_component_type_id,
            p_property_name                => c_expression_lang,
            p_db_column                    => p_db_column_lang,
            p_is_required                  => true,
            p_is_common                    => false,
            p_default_value                => 'PLSQL',
            p_reference_scope              => p_reference_scope,
            p_depending_on_comp_prop_name  => p_type_property,
            p_depending_on_condition_type  => 'EQUALS',
            p_depending_on_expression      => 'EXPRESSION',
            p_depending_on_comp_prop_name2 => p_depending_on_comp_prop_name,
            p_depending_on_condition_type2 => p_depending_on_condition_type,
            p_depending_on_expression2     => p_depending_on_expression );
        --
        store_component_prop (
            p_component_type_id           => p_component_type_id,
            p_property_name               => p_name_prefix || 'SQL_EXPRESSION' || c_postfix,
            p_db_column                   => p_db_column_expr,
            p_is_required                 => true,
            p_is_common                   => false,
            p_reference_scope             => p_reference_scope,
            p_custom_data_type            => p_custom_data_type,
            p_depending_on_comp_prop_name => c_expression_lang,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'SQL' );
        --
        store_component_prop (
            p_component_type_id           => p_component_type_id,
            p_property_name               => p_name_prefix || 'PLSQL_EXPRESSION' || c_postfix,
            p_db_column                   => p_db_column_expr,
            p_is_required                 => true,
            p_is_common                   => false,
            p_reference_scope             => p_reference_scope,
            p_custom_data_type            => p_custom_data_type,
            p_depending_on_comp_prop_name => c_expression_lang,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'PLSQL' );
        --
        store_component_prop (
            p_component_type_id           => p_component_type_id,
            p_property_name               => p_name_prefix || 'JS_EXPRESSION' || c_postfix,
            p_db_column                   => p_db_column_expr,
            p_is_required                 => true,
            p_is_common                   => false,
            p_reference_scope             => p_reference_scope,
            p_custom_data_type            => p_custom_data_type,
            p_depending_on_comp_prop_name => c_expression_lang,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'JAVASCRIPT' );
        --
        store_component_prop (
            p_component_type_id           => p_component_type_id,
            p_property_name               => p_name_prefix || 'JAVA_EXPRESSION' || c_postfix,
            p_db_column                   => p_db_column_expr,
            p_is_required                 => true,
            p_is_common                   => false,
            p_reference_scope             => p_reference_scope,
            p_custom_data_type            => p_custom_data_type,
            p_depending_on_comp_prop_name => c_expression_lang,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'JAVA' );
        --
        store_component_prop (
            p_component_type_id           => p_component_type_id,
            p_property_name               => p_name_prefix || 'PYTHON_EXPRESSION' || c_postfix,
            p_db_column                   => p_db_column_expr,
            p_is_required                 => true,
            p_is_common                   => false,
            p_reference_scope             => p_reference_scope,
            p_custom_data_type            => p_custom_data_type,
            p_depending_on_comp_prop_name => c_expression_lang,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'PYTHON' );
        --
    end store_expression_comp_props;
    --
    --
    procedure store_func_body_properties (
        p_start_property_id    in number,
        p_name_prefix          in varchar2,
        p_postfix              in varchar2 default null,
        p_return_type          in varchar2,
        p_prompt               in varchar2 default null,
        p_db_column_lang       in varchar2,
        p_db_column_func_body  in varchar2,
        p_display_group_name   in varchar2,
        p_max_length           in varchar2 default null,
        p_sql_min_column_count in number   default null,
        p_sql_max_column_count in number   default null,
        p_help_text            in varchar2,
        p_examples_plsql       in varchar2 default null,
        p_examples_js          in varchar2 default null,
        p_examples_java        in varchar2 default null,
        p_examples_python      in varchar2 default null )
    is
        c_postfix        constant varchar2( 30 )  := '_' || coalesce( p_postfix, p_return_type );
        c_func_body_lang constant varchar2( 255 ) := p_name_prefix || 'FUNC_BODY_LANG' || c_postfix;
        c_prompt         constant varchar2( 255 ) := coalesce (
                                                         p_prompt,
                                                         case p_return_type
                                                           when 'SQL' then '%language% Function Body returning SQL Query'
                                                         end,
                                                         '%language% Function Body' );
        c_max_length     constant number          := coalesce (
                                                         p_max_length,
                                                         case when p_return_type in ('SQL', 'CLOB') then 32767 else 4000 end );
        --
        l_property_id number := p_start_property_id;
    begin
        store_code_language_properties (
            p_property_id        => l_property_id,
            p_name               => c_func_body_lang,
            p_db_column          => p_db_column_lang,
            p_display_group_name => p_display_group_name,
            p_include_sql        => false,
            p_help_text          => q'~Select a language that is used to execute the entered function body.~' );
        --
        l_property_id := l_property_id + 1;
        store_property (
            p_id                   => l_property_id,
            p_name                 => p_name_prefix || 'PLSQL_FUNC_BODY' || c_postfix,
            p_prompt               => replace( c_prompt, '%language%', 'PL/SQL' ),
            p_property_type        => 'PLSQL FUNCTION BODY ' || p_return_type,
            p_db_column            => p_db_column_func_body,
            p_display_length       => 80,
            p_max_length           => c_max_length,
            p_sql_min_column_count => p_sql_min_column_count,
            p_sql_max_column_count => p_sql_max_column_count,
            p_display_group_name   => p_display_group_name,
            p_help_text            => replace( p_help_text, '%language%', 'PL/SQL' ),
            p_examples             => coalesce(
                                          p_examples_plsql,
                                          case p_return_type
                                          when 'BOOLEAN' then
q'~
<dl><dt>To return successfully when the employee is in department 30:</dt>
<dd><pre>
declare
    l_deptno number;
begin
    select deptno
      into l_deptno
      from emp
     where empno = :P2_EMPNO;

    if l_deptno = 30 then
        return true;
    else
        return false;
    end if;
end;
</pre></dd>
</dl>~'
                                          when 'VARCHAR2' then
q'~
<dl><dt>To return YES when the employee is in department 30:</dt>
<dd><pre>
declare
    l_deptno number;
begin
    select deptno
      into l_deptno
      from emp
     where empno = :P2_EMPNO;

    if l_deptno = 30 then
        return 'YES';
    else
        return 'NO';
    end if;
end;
</pre></dd>
</dl>~'
                                          when 'CLOB' then
q'~
<dl><dt>To return some data from the EMP table as a stringified JSON:</dt>
<dd><pre>
declare
    l_result clob;
begin
    select json_arrayagg (
               json_object (
                   'empno' value empno,
                   'ename' value ename
               )
           )
      into l_result
      from emp;

    return l_result;
end;
</pre></dd>
</dl>~'
                                          end )
            );
        --
        l_property_id := l_property_id + 1;
        store_property (
            p_id                   => l_property_id,
            p_name                 => p_name_prefix || 'JS_FUNC_BODY' || c_postfix,
            p_prompt               => replace(
                                        replace(
                                            replace(
                                                c_prompt,
                                                '%language%',
                                                'JavaScript' ),
                                            'CLOB',
                                            'String' ),
                                        'VARCHAR2',
                                        'String' ),
            p_property_type        => 'MLE JAVASCRIPT FUNCTION BODY ' || p_return_type,
            p_db_column            => p_db_column_func_body,
            p_display_length       => 80,
            p_max_length           => c_max_length,
            p_sql_min_column_count => p_sql_min_column_count,
            p_sql_max_column_count => p_sql_max_column_count,
            p_display_group_name   => p_display_group_name,
            p_help_text            => replace( p_help_text, '%language%', 'JavaScript' ),
            p_examples             => coalesce(
                                          p_examples_js,
                                          case p_return_type
                                          when 'BOOLEAN' then
q'~
<dl><dt>To return successfully when the employee is in department 30:</dt>
<dd><pre>
let deptno,
    rows;

rows = apex.conn.execute( "select deptno from emp empno = :1", [ apex.env.P2_EMPNO ] ).rows;
for (let row of rows) {
  deptno = row.DEPTNO;
}

return ( deptno === 30 );
</pre></dd>
</dl>~'
                                          when 'VARCHAR2' then
q'~
<dl><dt>To return YES when the employee is in department 30:</dt>
<dd><pre>
let deptno,
    rows;

rows = apex.conn.execute( "select deptno from emp empno = :1", [ apex.env.P2_EMPNO ] ).rows;
for (let row of rows) {
  deptno = row.DEPTNO;
}

return ( deptno === 30 ? "YES" : "NO" );
</pre></dd>
</dl>~'
                                          when 'CLOB' then
q'~
<dl><dt>To return some data from the EMP table as a stringified JSON:</dt>
<dd><pre>
let rows = apex.conn.execute( "select empno, ename from emp" ).rows;

return JSON.stringify( { data: rows } );
</pre></dd>
</dl>~'
                                          end ) );
        --
        l_property_id := l_property_id + 1;
        store_property (
            p_id                   => l_property_id,
            p_name                 => p_name_prefix || 'JAVA_FUNC_BODY' || c_postfix,
            p_prompt               => replace( c_prompt, '%language%', 'Java' ),
            p_property_type        => 'MLE JAVA FUNCTION BODY ' || p_return_type,
            p_db_column            => p_db_column_func_body,
            p_display_length       => 80,
            p_max_length           => c_max_length,
            p_sql_min_column_count => p_sql_min_column_count,
            p_sql_max_column_count => p_sql_max_column_count,
            p_display_group_name   => p_display_group_name,
            p_help_text            => replace( p_help_text, '%language%', 'Java' ),
            p_examples             => p_examples_java );
        --
        l_property_id := l_property_id + 1;
        store_property (
            p_id                   => l_property_id,
            p_name                 => p_name_prefix || 'PYTHON_FUNC_BODY' || c_postfix,
            p_prompt               => replace( c_prompt, '%language%', 'Python' ),
            p_property_type        => 'MLE PYTHON FUNCTION BODY ' || p_return_type,
            p_db_column            => p_db_column_func_body,
            p_display_length       => 80,
            p_max_length           => c_max_length,
            p_sql_min_column_count => p_sql_min_column_count,
            p_sql_max_column_count => p_sql_max_column_count,
            p_display_group_name   => p_display_group_name,
            p_help_text            => replace( p_help_text, '%language%', 'Python' ),
            p_examples             => p_examples_python );
        --
    end store_func_body_properties;
    --
    --
    procedure store_func_body_comp_props (
        p_component_type_id           in number,
        p_name_prefix                 in varchar2,
        p_postfix                     in varchar2 default null,
        p_return_type                 in varchar2,
        p_custom_data_type            in varchar2 default null,
        p_db_column_lang              in varchar2 default null,
        p_db_column_func_body         in varchar2 default null,
        p_reference_scope             in varchar2 default 'COMPONENT',
        p_type_property               in varchar2,
        p_type_operator               in varchar2 default 'EQUALS',
        p_type_value                  in varchar2 default 'FUNCTION_BODY',
        p_parent_comp_prop_name       in varchar2 default null,
        p_depending_on_comp_prop_name in varchar2 default null,
        p_depending_on_condition_type in varchar2 default null,
        p_depending_on_expression     in varchar2 default null,
        p_help_text                   in varchar2 default null,
        p_examples_plsql              in varchar2 default null,
        p_examples_js                 in varchar2 default null,
        p_examples_java               in varchar2 default null,
        p_examples_python             in varchar2 default null )
    is
        c_postfix        constant varchar2( 30 )  := '_' || coalesce( p_postfix, p_return_type );
        c_func_body_lang constant varchar2( 255 ) := p_name_prefix || 'FUNC_BODY_LANG' || c_postfix;
    begin
        store_component_prop (
            p_component_type_id            => p_component_type_id,
            p_property_name                => c_func_body_lang,
            p_db_column                    => p_db_column_lang,
            p_is_required                  => true,
            p_is_common                    => false,
            p_default_value                => 'PLSQL',
            p_reference_scope              => p_reference_scope,
            p_depending_on_comp_prop_name  => p_type_property,
            p_depending_on_condition_type  => p_type_operator,
            p_depending_on_expression      => p_type_value,
            p_depending_on_comp_prop_name2 => p_depending_on_comp_prop_name,
            p_depending_on_condition_type2 => p_depending_on_condition_type,
            p_depending_on_expression2     => p_depending_on_expression );
        --
        store_component_prop (
            p_component_type_id           => p_component_type_id,
            p_property_name               => p_name_prefix || 'PLSQL_FUNC_BODY' || c_postfix,
            p_db_column                   => p_db_column_func_body,
            p_is_required                 => true,
            p_is_common                   => false,
            p_reference_scope             => p_reference_scope,
            p_custom_data_type            => p_custom_data_type,
            p_depending_on_comp_prop_name => c_func_body_lang,
            p_depending_on_condition_type => p_type_operator,
            p_depending_on_expression     => 'PLSQL',
            p_parent_comp_prop_name       => p_parent_comp_prop_name,
            p_help_text                   => replace( p_help_text, '%language%', 'PL/SQL' ),
            p_examples                    => p_examples_python );
        --
        store_component_prop (
            p_component_type_id           => p_component_type_id,
            p_property_name               => p_name_prefix || 'JS_FUNC_BODY' || c_postfix,
            p_db_column                   => p_db_column_func_body,
            p_is_required                 => true,
            p_is_common                   => false,
            p_reference_scope             => p_reference_scope,
            p_custom_data_type            => p_custom_data_type,
            p_depending_on_comp_prop_name => c_func_body_lang,
            p_depending_on_condition_type => p_type_operator,
            p_depending_on_expression     => 'JAVASCRIPT',
            p_parent_comp_prop_name       => p_parent_comp_prop_name,
            p_help_text                   => replace( p_help_text, '%language%', 'JavaScript' ),
            p_examples                    => p_examples_python );
        --
        store_component_prop (
            p_component_type_id           => p_component_type_id,
            p_property_name               => p_name_prefix || 'JAVA_FUNC_BODY' || c_postfix,
            p_db_column                   => p_db_column_func_body,
            p_is_required                 => true,
            p_is_common                   => false,
            p_reference_scope             => p_reference_scope,
            p_custom_data_type            => p_custom_data_type,
            p_depending_on_comp_prop_name => c_func_body_lang,
            p_depending_on_condition_type => p_type_operator,
            p_depending_on_expression     => 'JAVA',
            p_parent_comp_prop_name       => p_parent_comp_prop_name,
            p_help_text                   => replace( p_help_text, '%language%', 'Java' ),
            p_examples                    => p_examples_python );
        --
        store_component_prop (
            p_component_type_id           => p_component_type_id,
            p_property_name               => p_name_prefix || 'PYTHON_FUNC_BODY' || c_postfix,
            p_db_column                   => p_db_column_func_body,
            p_is_required                 => true,
            p_is_common                   => false,
            p_reference_scope             => p_reference_scope,
            p_custom_data_type            => p_custom_data_type,
            p_depending_on_comp_prop_name => c_func_body_lang,
            p_depending_on_condition_type => p_type_operator,
            p_depending_on_expression     => 'PYTHON',
            p_parent_comp_prop_name       => p_parent_comp_prop_name,
            p_help_text                   => replace( p_help_text, '%language%', 'Python' ),
            p_examples                    => p_examples_python );
        --
    end store_func_body_comp_props;
    --
    --
    procedure store_snippet_properties (
        p_start_property_id    in number,
        p_name_prefix          in varchar2,
        p_prompt               in varchar2 default null,
        p_db_column_lang       in varchar2,
        p_db_column_snippet    in varchar2,
        p_display_group_name   in varchar2,
        p_max_length           in varchar2 default 32767,
        p_help_text            in varchar2,
        p_examples_plsql       in varchar2 default null,
        p_examples_js          in varchar2 default null,
        p_examples_java        in varchar2 default null,
        p_examples_python      in varchar2 default null )
    is
        c_snippet_lang constant varchar2( 255 ) := p_name_prefix || 'SNIPPET_LANG';
        c_prompt       constant varchar2( 255 ) := coalesce( p_prompt, '%language% Code' );
        --
        l_property_id number := p_start_property_id;
    begin
        store_code_language_properties (
            p_property_id        => l_property_id,
            p_name               => c_snippet_lang,
            p_db_column          => p_db_column_lang,
            p_display_group_name => p_display_group_name,
            p_include_sql        => false,
            p_help_text          => q'~Select a language that is used to execute the entered code.~' );
        --
        l_property_id := l_property_id + 1;
        store_property (
            p_id                   => l_property_id,
            p_name                 => p_name_prefix || 'PLSQL_CODE',
            p_prompt               => replace( c_prompt, '%language%', 'PL/SQL' ),
            p_property_type        => 'PLSQL',
            p_db_column            => p_db_column_snippet,
            p_display_length       => 80,
            p_max_length           => p_max_length,
            p_display_group_name   => p_display_group_name,
            p_help_text            => replace( p_help_text, '%language%', 'PL/SQL' ),
            p_examples             => coalesce(
                                          p_examples_plsql,
q'~
<dl><dt>Increases the salary by 3% for those employees which are qualified for a salary raise:</dt>
<dd><pre>
begin
    for l_emp ( select empno
                  from emp
                 where deptno = :P2_DEPTNO )
    loop
        if needs_salary_raise( l_empno.empno ) then
            update emp
               set sal = sal * 1.03
             where empno = l_emp.empno;
        end if;
    end loop;
end;
</pre></dd>
</dl>~'
            ));
        --
        l_property_id := l_property_id + 1;
        store_property (
            p_id                   => l_property_id,
            p_name                 => p_name_prefix || 'JS_CODE',
            p_prompt               => replace( c_prompt, '%language%', 'JavaScript' ),
            p_property_type        => 'MLE JAVASCRIPT',
            p_db_column            => p_db_column_snippet,
            p_display_length       => 80,
            p_max_length           => p_max_length,
            p_display_group_name   => p_display_group_name,
            p_help_text            => replace( p_help_text, '%language%', 'JavaScript' ),
            p_examples             => coalesce(
                                          p_examples_js,
q'~
<dl><dt>Increases the salary by 3% for those employees which are qualified for a salary raise:</dt>
<dd><pre>
let rows;

function needsSalaryRaise( empno ) {
    // placeholder for some complicated logic to calculate if employee needs a raise
    return true;
}

rows = apex.conn.execute( "select empno from emp where deptno = :deptno", { deptno: apex.env.P2_DEPTNO } ).rows;
for ( let row of rows ) {
    if ( needsSalaryRaise( row.EMPNO )) {
      apex.conn.execute( "update emp set sal = sal * 1.03 where empno = :empno", { empno: row.EMPNO } );
    }
}
</pre></dd>
</dl>~'
            ));
        --
        l_property_id := l_property_id + 1;
        store_property (
            p_id                   => l_property_id,
            p_name                 => p_name_prefix || 'JAVA_CODE',
            p_prompt               => replace( c_prompt, '%language%', 'Java' ),
            p_property_type        => 'MLE JAVA',
            p_db_column            => p_db_column_snippet,
            p_display_length       => 80,
            p_max_length           => p_max_length,
            p_display_group_name   => p_display_group_name,
            p_help_text            => replace( p_help_text, '%language%', 'Java' ),
            p_examples             => p_examples_java );
        --
        l_property_id := l_property_id + 1;
        store_property (
            p_id                   => l_property_id,
            p_name                 => p_name_prefix || 'PYTHON_CODE',
            p_prompt               => replace( c_prompt, '%language%', 'Python' ),
            p_property_type        => 'MLE PYTHON',
            p_db_column            => p_db_column_snippet,
            p_display_length       => 80,
            p_max_length           => p_max_length,
            p_display_group_name   => p_display_group_name,
            p_help_text            => replace( p_help_text, '%language%', 'Python' ),
            p_examples             => p_examples_python );
        --
    end store_snippet_properties;
    --
    --
    procedure store_snippet_comp_props (
        p_component_type_id     in number,
        p_name_prefix           in varchar2,
        p_db_column_lang        in varchar2 default null,
        p_db_column_snippet     in varchar2 default null,
        p_reference_scope       in varchar2 default 'COMPONENT',
        p_type_property         in varchar2,
        p_type_operator         in varchar2 default 'EQUALS',
        p_type_value            in varchar2,
        p_parent_comp_prop_name in varchar2 default null,
        p_help_text             in varchar2 default null,
        p_examples_plsql        in varchar2 default null,
        p_examples_js           in varchar2 default null,
        p_examples_java         in varchar2 default null,
        p_examples_python       in varchar2 default null )
    is
        c_snippet_lang constant varchar2( 255 ) := p_name_prefix || 'SNIPPET_LANG';
    begin
        store_component_prop (
            p_component_type_id           => p_component_type_id,
            p_property_name               => c_snippet_lang,
            p_db_column                   => p_db_column_lang,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'PLSQL',
            p_reference_scope             => p_reference_scope,
            p_depending_on_comp_prop_name => p_type_property,
            p_depending_on_condition_type => p_type_operator,
            p_depending_on_expression     => p_type_value );
        --
        store_component_prop (
            p_component_type_id           => p_component_type_id,
            p_property_name               => p_name_prefix || 'PLSQL_CODE',
            p_db_column                   => p_db_column_snippet,
            p_is_required                 => true,
            p_is_common                   => false,
            p_reference_scope             => p_reference_scope,
            p_depending_on_comp_prop_name => c_snippet_lang,
            p_depending_on_condition_type => p_type_operator,
            p_depending_on_expression     => 'PLSQL',
            p_parent_comp_prop_name       => p_parent_comp_prop_name,
            p_help_text                   => replace( p_help_text, '%language%', 'PL/SQL' ),
            p_examples                    => p_examples_plsql );
        --
        store_component_prop (
            p_component_type_id           => p_component_type_id,
            p_property_name               => p_name_prefix || 'JS_CODE',
            p_db_column                   => p_db_column_snippet,
            p_is_required                 => true,
            p_is_common                   => false,
            p_reference_scope             => p_reference_scope,
            p_depending_on_comp_prop_name => c_snippet_lang,
            p_depending_on_condition_type => p_type_operator,
            p_depending_on_expression     => 'JAVASCRIPT',
            p_parent_comp_prop_name       => p_parent_comp_prop_name,
            p_help_text                   => replace( p_help_text, '%language%', 'JavaScript' ),
            p_examples                    => p_examples_js );
        --
        store_component_prop (
            p_component_type_id           => p_component_type_id,
            p_property_name               => p_name_prefix || 'JAVA_CODE',
            p_db_column                   => p_db_column_snippet,
            p_is_required                 => true,
            p_is_common                   => false,
            p_reference_scope             => p_reference_scope,
            p_depending_on_comp_prop_name => c_snippet_lang,
            p_depending_on_condition_type => p_type_operator,
            p_depending_on_expression     => 'PYTHON',
            p_parent_comp_prop_name       => p_parent_comp_prop_name,
            p_help_text                   => replace( p_help_text, '%language%', 'Java' ),
            p_examples                    => p_examples_java );
        --
        store_component_prop (
            p_component_type_id           => p_component_type_id,
            p_property_name               => p_name_prefix || 'PYTHON_CODE',
            p_db_column                   => p_db_column_snippet,
            p_is_required                 => true,
            p_is_common                   => false,
            p_reference_scope             => p_reference_scope,
            p_depending_on_comp_prop_name => c_snippet_lang,
            p_depending_on_condition_type => p_type_operator,
            p_depending_on_expression     => 'PYTHON',
            p_parent_comp_prop_name       => p_parent_comp_prop_name,
            p_help_text                   => replace( p_help_text, '%language%', 'Python' ),
            p_examples                    => p_examples_python );
        --
    end store_snippet_comp_props;
    --
    --
    procedure store_cond_type_properties (
        p_start_property_id  in number,
        p_name_prefix        in varchar2,
        p_prompt_type        in varchar2 default 'Type',
        p_help_text_type     in varchar2,
        p_db_column_type     in varchar2,
        p_db_column_expr1    in varchar2,
        p_db_column_expr2    in varchar2,
        p_display_group_name in varchar2 )
    is
        c_name_prefix constant varchar2( 30) := case when p_name_prefix is not null then p_name_prefix || '_' end;
        l_property_id number := p_start_property_id;
    begin
        store_property (
            p_id                 => l_property_id,
            p_name               => c_name_prefix || 'CONDITION_TYPE',
            p_prompt             => p_prompt_type,
            p_property_type      => 'SELECT LIST',
            p_db_column          => p_db_column_type,
            p_display_group_name => p_display_group_name,
            p_lov_type           => 'STATIC',
            p_help_text          => p_help_text_type );
        --
        if p_name_prefix = 'READ_ONLY' then
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Always',
                p_return_value  => 'ALWAYS',
                p_is_deprecated => 'N',
                p_is_quick_pick => 'Y',
                p_help_text     => q'~The component will always be rendered as read only.~' );
            --
        elsif nvl( p_name_prefix, 'CONDITION' ) not in ( 'AUTOM', 'ACTION', 'WF', 'RAG' ) then
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Always',
                p_return_value  => 'ALWAYS',
                p_is_deprecated => 'Y',
                p_help_text     => q'~The component will always be rendered or executed.~' );
        end if;
        --
        if p_name_prefix = 'RAG' then
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Any User Prompt contains',
                p_return_value  => 'ANY_AI_USER_PROMPT_CONTAINS',
                p_is_quick_pick => 'Y',
                p_help_text     => q'~At least one of the comma-separated strings in the <em>List</em> attribute is found in any of the user prompts.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Last User Prompt contains',
                p_return_value  => 'LAST_AI_USER_PROMPT_CONTAINS',
                p_is_quick_pick => 'Y',
                p_help_text     => q'~At least one of the comma-separated strings in the <em>List</em> attribute appears in the most recent user prompt.~' );
        end if;
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Rows returned',
            p_return_value  => 'EXISTS',
            p_is_quick_pick => 'Y',
            p_help_text     => q'~The SQL query returns one or more rows.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'No Rows returned',
            p_return_value  => 'NOT_EXISTS',
            p_help_text     => q'~The SQL Query returns no rows~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Expression',
            p_return_value  => 'EXPRESSION',
            p_help_text     => q'~The expression evaluates to TRUE.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Function Body',
            p_return_value  => 'FUNCTION_BODY',
            p_help_text     => q'~The function body returns TRUE.~' );
        --
        if nvl( p_name_prefix, 'CONDITION' ) not in ( 'AUTOM', 'ACTION', 'SO_VAL', 'SO_COND', 'WF', 'RAG' ) then
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Request = Value',
                p_return_value  => 'REQUEST_EQUALS_CONDITION',
                p_help_text     => q'~The page request is equal to the text you enter into the <em>Value</em> attribute.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Request != Value',
                p_return_value  => 'REQUEST_NOT_EQUAL_CONDITION',
                p_help_text     => q'~The page request is not equal to the text you enter into the <em>Value</em> attribute.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Request is contained in Value',
                p_return_value  => 'REQUEST_IN_CONDITION',
                p_help_text     => q'~The page request is contained in the text you enter into the <em>Value</em> attribute.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Request is NOT contained in Value',
                p_return_value  => 'REQUEST_NOT_IN_CONDITION',
                p_help_text     => q'~The page request is not contained in the text you enter into the <em>Value</em> attribute.~' );
        end if;
        --
        if nvl( p_name_prefix, 'CONDITION' ) not in ( 'AUTOM', 'SO_VAL', 'SO_COND', 'WF', 'RAG' ) then
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Item = Value',
                p_return_value  => 'VAL_OF_ITEM_IN_COND_EQ_COND2',
                p_is_quick_pick => 'Y',
                p_help_text     => q'~The value of the selected <em>Item</em> is equal to the text you enter into the <em>Value</em> attribute.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Item != Value',
                p_return_value  => 'VAL_OF_ITEM_IN_COND_NOT_EQ_COND2',
                p_help_text     => q'~The value of the selected <em>Item</em> is not equal to the text you enter into the <em>Value</em> attribute.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Item is NULL',
                p_return_value  => 'ITEM_IS_NULL',
                p_is_quick_pick => 'Y',
                p_help_text     => q'~The value of the selected <em>Item</em> is empty.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Item is NOT NULL',
                p_return_value  => 'ITEM_IS_NOT_NULL',
                p_is_quick_pick => 'Y',
                p_help_text     => q'~The value of the selected <em>Item</em> is not empty.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Item is zero',
                p_return_value  => 'ITEM_IS_ZERO',
                p_help_text     => q'~The value of the selected <em>Item</em> is the number zero.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Item is NOT zero',
                p_return_value  => 'ITEM_IS_NOT_ZERO',
                p_help_text     => q'~The value of the selected <em>Item</em> is not the number zero.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Item is NULL or zero',
                p_return_value  => 'ITEM_IS_NULL_OR_ZERO',
                p_help_text     => q'~The value of the selected <em>Item</em> is empty or the number zero.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Item is NOT NULL and NOT zero',
                p_return_value  => 'ITEM_NOT_NULL_OR_ZERO',
                p_help_text     => q'~The value of the selected <em>Item</em> is not empty and not the number zero.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Item contains no spaces',
                p_return_value  => 'ITEM_CONTAINS_NO_SPACES',
                p_help_text     => q'~The value of the selected <em>Item</em> has no spaces.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Item is numeric',
                p_return_value  => 'ITEM_IS_NUMERIC',
                p_help_text     => q'~The value of the selected <em>Item</em> is numeric.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Item is NOT numeric',
                p_return_value  => 'ITEM_IS_NOT_NUMERIC',
                p_help_text     => q'~The value of the selected <em>Item</em> is not numeric.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Item is alphanumeric',
                p_return_value  => 'ITEM_IS_ALPHANUMERIC',
                p_help_text     => q'~The value of the selected <em>Item</em> is alphanumeric, containing only letters or numbers and no special characters.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Item is in colon delimited list',
                p_return_value  => 'VALUE_OF_ITEM_IN_CONDITION_IN_COLON_DELIMITED_LIST',
                p_help_text     => q'~The value of the selected <em>Item</em> is completely contained in the text you enter into the <em>Value</em> attribute.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Item is NOT in colon delimited list',
                p_return_value  => 'VALUE_OF_ITEM_IN_CONDITION_NOT_IN_COLON_DELIMITED_LIST',
                p_help_text     => q'~The value of the selected <em>Item</em> is not completely contained in the text you entered into the <em>Value</em> attribute.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Text is contained in Item',
                p_return_value  => 'CONDITION1_IN_VALUE_OF_ITEM_IN_CONDITION2',
                p_is_deprecated => 'Y' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Text = Value',
                p_return_value  => 'DISPLAY_COND_EQUAL_COND_TEXT',
                p_is_deprecated => 'Y' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Text != Value',
                p_return_value  => 'DISP_COND_NOT_EQUAL_COND_TEXT',
                p_is_deprecated => 'Y' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Text is contained in Value',
                p_return_value  => 'DISPLAY_COND_IN_COND_TEXT',
                p_is_deprecated => 'Y' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Text is NOT contained in Value',
                p_return_value  => 'DISPLAY_COND_NOT_IN_COND_TEXT',
                p_is_deprecated => 'Y' );
        end if;
        --
        if nvl( p_name_prefix, 'CONDITION' ) = 'WF' then
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Variable = Value',
                p_return_value  => 'VAL_OF_ITEM_IN_COND_EQ_COND2',
                p_is_quick_pick => 'Y',
                p_help_text     => q'~The value of the selected <em>Variable</em> is equal to the text you enter into the <em>Value</em> attribute.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Variable != Value',
                p_return_value  => 'VAL_OF_ITEM_IN_COND_NOT_EQ_COND2',
                p_help_text     => q'~The value of the selected <em>Variable</em> is not equal to the text you enter into the <em>Value</em> attribute.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Variable is NULL',
                p_return_value  => 'ITEM_IS_NULL',
                p_is_quick_pick => 'Y',
                p_help_text     => q'~The value of the selected <em>Variable</em> is empty.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Variable is NOT NULL',
                p_return_value  => 'ITEM_IS_NOT_NULL',
                p_is_quick_pick => 'Y',
                p_help_text     => q'~The value of the selected <em>Variable</em> is not empty.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Variable is zero',
                p_return_value  => 'ITEM_IS_ZERO',
                p_help_text     => q'~The value of the selected <em>Variable</em> is the number zero.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Variable is NOT zero',
                p_return_value  => 'ITEM_IS_NOT_ZERO',
                p_help_text     => q'~The value of the selected <em>Variable</em> is not the number zero.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Variable is NULL or zero',
                p_return_value  => 'ITEM_IS_NULL_OR_ZERO',
                p_help_text     => q'~The value of the selected <em>Variable</em> is empty or the number zero.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Variable is NOT NULL and NOT zero',
                p_return_value  => 'ITEM_NOT_NULL_OR_ZERO',
                p_help_text     => q'~The value of the selected <em>Variable</em> is not empty and not the number zero.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Variable contains no spaces',
                p_return_value  => 'ITEM_CONTAINS_NO_SPACES',
                p_help_text     => q'~The value of the selected <em>Variable</em> has no spaces.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Variable is numeric',
                p_return_value  => 'ITEM_IS_NUMERIC',
                p_help_text     => q'~The value of the selected <em>Variable</em> is numeric.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Variable is NOT numeric',
                p_return_value  => 'ITEM_IS_NOT_NUMERIC',
                p_help_text     => q'~The value of the selected <em>Variable</em> is not numeric.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Variable is alphanumeric',
                p_return_value  => 'ITEM_IS_ALPHANUMERIC',
                p_help_text     => q'~The value of the selected <em>Variable</em> is alphanumeric, containing only letters or numbers and no special characters.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Variable is in colon delimited list',
                p_return_value  => 'VALUE_OF_ITEM_IN_CONDITION_IN_COLON_DELIMITED_LIST',
                p_help_text     => q'~The value of the selected <em>Variable</em> is completely contained in the text you enter into the <em>Value</em> attribute.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Variable is NOT in colon delimited list',
                p_return_value  => 'VALUE_OF_ITEM_IN_CONDITION_NOT_IN_COLON_DELIMITED_LIST',
                p_help_text     => q'~The value of the selected <em>Variable</em> is not completely contained in the text you entered into the <em>Value</em> attribute.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Text is contained in Variable',
                p_return_value  => 'CONDITION1_IN_VALUE_OF_ITEM_IN_CONDITION2',
                p_is_deprecated => 'Y' );
            --
            -- The following LOV properties are marked as deprecated
            -- This is to stay consistent with the corresponding properties
            -- in the Non-WF Condition LOV defined above.
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Text = Value',
                p_return_value  => 'DISPLAY_COND_EQUAL_COND_TEXT',
                p_is_deprecated => 'Y' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Text != Value',
                p_return_value  => 'DISP_COND_NOT_EQUAL_COND_TEXT',
                p_is_deprecated => 'Y' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Text is contained in Value',
                p_return_value  => 'DISPLAY_COND_IN_COND_TEXT',
                p_is_deprecated => 'Y' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Text is NOT contained in Value',
                p_return_value  => 'DISPLAY_COND_NOT_IN_COND_TEXT',
                p_is_deprecated => 'Y' );
        end if;
        --
        if nvl( p_name_prefix, 'CONDITION' ) not in ( 'AUTOM', 'ACTION', 'SO_VAL', 'SO_COND', 'WF', 'RAG' ) then
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'User Preference = Value',
                p_return_value  => 'USER_PREF_IN_COND_EQ_COND2',
                p_help_text     => q'~The value of the <em>Preference</em> entered is equal to the text you enter into the <em>Value</em> attribute.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'User Preference != Value',
                p_return_value  => 'USER_PREF_IN_COND_NOT_EQ_COND2',
                p_help_text     => q'~The value of the <em>Preference</em> entered is not equal to the text you enter into the <em>Value</em> attribute.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Current Page = Page',
                p_return_value  => 'CURRENT_PAGE_EQUALS_CONDITION',
                p_help_text     => q'~The current page is equal to the value you enter into <em>Page</em>.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Current Page != Page',
                p_return_value  => 'CURRENT_PAGE_NOT_EQUAL_CONDITION',
                p_help_text     => q'~The current page is not equal to the value you enter into <em>Page</em>.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Current Page is in comma delimited list',
                p_return_value  => 'CURRENT_PAGE_IN_CONDITION',
                p_help_text     => q'~The current page is in the comma separated list you enter into <em>Pages</em>.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Current Page is NOT in comma delimited list',
                p_return_value  => 'CURRENT_PAGE_NOT_IN_CONDITION',
                p_help_text     => q'~The current page is not in the comma separated list you enter into <em>Pages</em>.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Current Page = Page Submitted (posted)',
                p_return_value  => 'WHEN_THIS_PAGE_SUBMITTED',
                p_is_deprecated => 'Y' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Current Page != Page Submitted (posted)',
                p_return_value  => 'WHEN_THIS_PAGE_NOT_SUBMITTED',
                p_is_deprecated => 'Y' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Current Page is in Printer Friendly Mode',
                p_return_value  => 'PAGE_IS_IN_PRINTER_FRIENDLY_MODE',
                p_help_text     => q'~The current page has been toggled to 'Printer Friendly Mode' by the end user.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Current page is NOT in Printer Friendly Mode',
                p_return_value  => 'PAGE_IS_NOT_IN_PRINTER_FRIENDLY_MODE',
                p_help_text     => q'~The current page has not been toggled to 'Printer Friendly Mode' by the end user.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Page/Region is Read Only',
                p_return_value  => 'IS_READ_ONLY',
                p_help_text     => q'~Either the page or parent region, if appropriate, 'Read Only' condition evaluates to TRUE.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Page/Region is NOT Read Only',
                p_return_value  => 'IS_NOT_READ_ONLY',
                p_help_text     => q'~The page and parent region, if appropriate, are displayed normally.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'User is Authenticated (not public)',
                p_return_value  => 'USER_IS_NOT_PUBLIC_USER',
                p_help_text     => q'~The end user is logged into the application.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'User is the Public User (user has not authenticated)',
                p_return_value  => 'USER_IS_PUBLIC_USER',
                p_help_text     => q'~The end user is not logged into the application.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Inline Validation Errors displayed',
                p_return_value  => 'DISPLAYING_INLINE_VALIDATION_ERRORS',
                p_help_text     => q'~An inline validation error is displayed on the page, following a page submission.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Inline Validation Errors NOT displayed',
                p_return_value  => 'NOT_DISPLAYING_INLINE_VALIDATION_ERRORS',
                p_help_text     => q'~An inline validation error is not displayed on the page.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'SQL Reports (OK to show the forward button)',
                p_return_value  => 'MAX_ROWS_LT_ROWS_FETCHED',
                p_is_deprecated => 'Y' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'SQL Reports (OK to show the back button)',
                p_return_value  => 'MIN_ROW_GT_THAN_ONE',
                p_is_deprecated => 'Y' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Current Language = Value',
                p_return_value  => 'CURRENT_LANG_EQ_COND1',
                p_help_text     => q'~The end user is currently running the application in the language you enter into the <em>Value</em> attribute.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Current Language != Value',
                p_return_value  => 'CURRENT_LANG_NOT_EQ_COND1',
                p_help_text     => q'~The end user is not currently running the application in the language you enter into the <em>Value</em> attribute.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Current Language is contained in Value',
                p_return_value  => 'CURRENT_LANG_IN_COND1',
                p_help_text     => q'~The end user is currently running the application in one of the languages you enter into the <em>Value</em> attribute.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Current Language is NOT contained in Value',
                p_return_value  => 'CURRENT_LANG_NOT_IN_COND1',
                p_help_text     => q'~The end user is not currently running the application in one of the languages you enter into the <em>Value</em> attribute.~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Client Browser is Mozilla, Netscape 6.x/7x or higher',
                p_return_value  => 'BROWSER_IS_NSCP',
                p_is_deprecated => 'Y' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Client Browser is Microsoft Internet Explorer 7.0 or higher',
                p_return_value  => 'BROWSER_IS_MSIE',
                p_is_deprecated => 'Y' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Client Browser is XHTML / CSS capable browser',
                p_return_value  => 'BROWSER_IS_MSIE_OR_NSCP',
                p_is_deprecated => 'Y' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Client Browser is Other browsers (or older version)',
                p_return_value  => 'BROWSER_IS_OTHER',
                p_is_deprecated => 'Y' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'CGI_ENV: DAD_NAME = Value',
                p_return_value  => 'DAD_NAME_EQ_CONDITION',
                p_is_deprecated => 'Y' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'CGI_ENV: DAD_NAME != Value',
                p_return_value  => 'DAD_NAME_NOT_EQ_CONDITION',
                p_is_deprecated => 'Y' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'CGI_ENV: SERVER_NAME = Value',
                p_return_value  => 'SERVER_NAME_EQ_CONDITION',
                p_is_deprecated => 'Y' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'CGI_ENV: SERVER_NAME != Value',
                p_return_value  => 'SERVER_NAME_NOT_EQ_CONDITION',
                p_is_deprecated => 'Y' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'CGI_ENV: HTTP_HOST = Value',
                p_return_value  => 'HTTP_HOST_EQ_CONDITION',
                p_is_deprecated => 'Y' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'CGI_ENV: HTTP_HOST != Value',
                p_return_value  => 'HTTP_HOST_NOT_EQ_CONDITION',
                p_is_deprecated => 'Y' );
        end if;
        --
        if nvl( p_name_prefix, 'CONDITION' ) in ( 'CONDITION', 'LINK', 'READ_ONLY', 'AUTOM', 'ACTION', 'SO_VAL', 'SO_COND', 'WF', 'RAG' ) then
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Never',
                p_return_value  => 'NEVER',
                p_is_quick_pick => case when nvl( p_name_prefix, 'CONDITION' ) in ( 'CONDITION', 'READ_ONLY', 'AUTOM', 'ACTION', 'SO_VAL', 'SO_COND', 'RAG' ) then 'Y' else 'N' end,
                p_help_text     => case
                                     when p_name_prefix = 'READ_ONLY' then
                                         q'~The component is rendered normally, irrespective of the read only condition for parent component(s).~'
                                     else
                                         q'~<p>The component is never rendered or executed.</p>
<p>Note: Changing a condition to <strong>Never</strong> will remove any existing condition. For temporarily disabling a component, Oracle recommends using the <em>Build Option</em> attribute instead.</p>~'
                                   end );
        else
            -- Remove Never for caching, ...
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Never',
                p_return_value  => 'NEVER',
                p_is_quick_pick => 'N',
                p_is_deprecated => 'Y',
                p_help_text     => q'~~' );
        end if;
        --
        l_property_id := l_property_id + 1;
        store_property (
            p_id                   => l_property_id,
            p_name                 => c_name_prefix || 'CONDITION_SQL_STATEMENT',
            p_prompt               => 'SQL Query',
            p_property_type        => 'SQL',
            p_db_column            => p_db_column_expr1,
            p_display_length       => 80,
            p_max_length           => 4000,
            p_display_group_name   => p_display_group_name,
            p_sql_min_column_count => 1,
            p_help_text            => q'~Enter a SQL query for this condition.~',
            p_examples             => q'~
<dl><dt>To return successfully as one or more rows exist when checking if the employee is in department 30:</dt>
<dd><pre>
select 1
  from emp
 where empno  = :P2_EMPNO
   and deptno = 30
</pre></dd>
<dt>To return successfully as no rows exist when checking if the employee is in department 40:</dt>
<dd><pre>
select 1
  from emp
 where empno  = :P2_EMPNO
   and deptno = 40
</pre></dd>
</dl>~' );
        --
        if nvl( p_name_prefix, 'CONDITION' ) not in ( 'AUTOM', 'SO_VAL', 'SO_COND', 'RAG' ) then
            l_property_id := l_property_id + 1;
            store_property (
                p_id                 => l_property_id,
                p_name               => c_name_prefix || 'CONDITION_VALUE1',
                p_prompt             => 'Value',
                p_property_type      => 'TEXT',
                p_db_column          => p_db_column_expr1,
                p_display_length     => 80,
                p_max_length         => 4000,
                p_display_group_name => p_display_group_name,
                p_help_text          => q'~Enter the value to compare against the condition.~' );
            --
            l_property_id := l_property_id + 1;
            store_property (
                p_id                 => l_property_id,
                p_name               => c_name_prefix || 'CONDITION_ITEM1',
                p_prompt             => 'Item',
                p_property_type      => 'ITEM',
                p_db_column          => p_db_column_expr1,
                p_display_length     => 80,
                p_max_length         => 4000,
                p_display_group_name => p_display_group_name,
                p_help_text          => q'~Enter the page or application item used in this condition. You can type in the name or pick from the list of available items.~' );
        end if;
        --
        l_property_id := l_property_id + 1;
        store_property (
            p_id                 => l_property_id,
            p_name               => c_name_prefix || 'CONDITION_LIST',
            p_prompt             => 'List',
            p_property_type      => 'TEXT',
            p_db_column          => p_db_column_expr2,
            p_display_length     => 80,
            p_max_length         => 4000,
            p_display_group_name => p_display_group_name,
            p_help_text          => q'~Enter a colon(:) separated list of values for this condition.~' );
        --
        if nvl( p_name_prefix, 'CONDITION' ) not in ( 'AUTOM', 'ACTION', 'SO_VAL', 'SO_COND', 'WF', 'RAG' ) then
            l_property_id := l_property_id + 1;
            store_property (
                p_id                 => l_property_id,
                p_name               => c_name_prefix || 'CONDITION_PREFERENCE',
                p_prompt             => 'Preference',
                p_property_type      => 'TEXT',
                p_db_column          => p_db_column_expr1,
                p_display_length     => 80,
                p_max_length         => 4000,
                p_text_case          => 'UPPER',
                p_display_group_name => p_display_group_name,
                p_help_text          => q'~Enter a User Preference that is compared against the specified value for this condition.~' );
            --
            l_property_id := l_property_id + 1;
            store_property (
                p_id                 => l_property_id,
                p_name               => c_name_prefix || 'CONDITION_PAGE',
                p_prompt             => 'Page',
                p_property_type      => 'PAGE',
                p_db_column          => p_db_column_expr1,
                p_display_length     => 80,
                p_max_length         => 4000,
                p_display_group_name => p_display_group_name,
                p_help_text          => q'~Enter the Page Number for this condition.~' );
            --
            l_property_id := l_property_id + 1;
            store_property (
                p_id                    => l_property_id,
                p_name                  => c_name_prefix || 'CONDITION_PAGES',
                p_prompt                => 'Pages',
                p_property_type         => 'PAGE',
                p_multi_value_delimiter => ',',
                p_db_column             => p_db_column_expr1,
                p_display_length        => 80,
                p_max_length            => 4000,
                p_display_group_name    => p_display_group_name,
                p_help_text             => q'~Enter a comma separated list of Page Numbers for this condition.~' );
        end if;
        --
        if nvl( p_name_prefix, 'CONDITION' ) not in ( 'AUTOM', 'SO_VAL', 'SO_COND', 'RAG' ) then
            l_property_id := l_property_id + 1;
            store_property (
                p_id                 => l_property_id,
                p_name               => c_name_prefix || 'CONDITION_TEXT',
                p_prompt             => 'Text',
                p_property_type      => 'TEXT',
                p_db_column          => p_db_column_expr1,
                p_display_length     => 80,
                p_max_length         => 4000,
                p_display_group_name => p_display_group_name,
                p_help_text          => q'~Enter the text value for this condition.~' );
            --
            l_property_id := l_property_id + 1;
            store_property (
                p_id                 => l_property_id,
                p_name               => c_name_prefix || 'CONDITION_VALUE2',
                p_prompt             => 'Value',
                p_property_type      => 'TEXT',
                p_db_column          => p_db_column_expr2,
                p_display_length     => 80,
                p_max_length         => 4000,
                p_display_group_name => p_display_group_name,
                p_help_text          => q'~Enter the value, including <code>&ITEM.</code> substitution strings, for this condition.~' );
            --
            l_property_id := l_property_id + 1;
            store_property (
                p_id                 => l_property_id,
                p_name               => c_name_prefix || 'CONDITION_ITEM2',
                p_prompt             => 'Item',
                p_property_type      => 'ITEM',
                p_db_column          => p_db_column_expr2,
                p_display_length     => 80,
                p_max_length         => 4000,
                p_display_group_name => p_display_group_name,
                p_help_text          => q'~Enter the page or application item used in this condition. You can type in the name or pick from the list of available items.~' );
            --
            l_property_id := l_property_id + 1;
            store_property (
                p_id                 => l_property_id,
                p_name               => c_name_prefix || 'CONDITION_TEXT2',
                p_prompt             => 'Value',
                p_property_type      => 'TEXT',
                p_db_column          => p_db_column_expr2,
                p_display_length     => 80,
                p_max_length         => 4000,
                p_display_group_name => p_display_group_name,
                p_help_text          => q'~Enter the value, including <code>&ITEM.</code> substitution strings, for this condition.~' );
        end if;
        --
        if p_name_prefix = 'RAG' then
            l_property_id := l_property_id + 1;
            store_property (
                p_id                    => l_property_id,
                p_name                  => c_name_prefix || 'AI_WORD_LIST',
                p_prompt                => 'List',
                p_property_type         => 'TEXT',
                p_db_column             => p_db_column_expr1,
                p_multi_value_delimiter => ',',
                p_display_length        => 80,
                p_max_length            => 4000,
                p_display_group_name    => p_display_group_name,
                p_help_text             => q'~Enter a comma-separated list of strings to check for their presence in the user prompt.~' );
        end if;
        --
        -- Note: the calls to store_expression_properties and store_func_body_properties have to be last, because if we have to add
        --       new properties in those procedures, it would change the ID of all properties which are created after
        --
        l_property_id := l_property_id + 1;
        store_expression_properties (
            p_start_property_id  => l_property_id,
            p_name_prefix        => c_name_prefix || 'CONDITION_',
            p_return_type        => 'BOOLEAN',
            p_db_column_lang     => p_db_column_expr2,
            p_db_column_expr     => p_db_column_expr1,
            p_display_group_name => p_display_group_name,
            p_help_text          => q'~Enter a %language% Expression for this condition.~' );
        --
        l_property_id := l_property_id + 20;
        store_func_body_properties (
            p_start_property_id   => l_property_id,
            p_name_prefix         => c_name_prefix || 'CONDITION_',
            p_return_type         => 'BOOLEAN',
            p_db_column_lang      => p_db_column_expr2,
            p_db_column_func_body => p_db_column_expr1,
            p_display_group_name  => p_display_group_name,
            p_help_text           => q'~Enter a %language% Function Body that returns a boolean value for this condition.~' );
    end store_cond_type_properties;
    --
    --
    procedure store_condition_type_comp_prop (
        p_component_type_id           in number,
        p_name_prefix                 in varchar2,
        p_help_text_type              in varchar2,
        p_db_column_type              in varchar2,
        p_view_column_type            in varchar2 default null,
        p_db_column_expr1             in varchar2,
        p_db_column_expr2             in varchar2,
        p_reference_scope             in varchar2 default 'COMPONENT',
        p_is_common                   in boolean  default true,
        p_plugin_std_attributes       in varchar2 default null,
        p_parent_comp_prop_name       in varchar2 default null,
        p_depending_on_comp_prop_name in varchar2 default null,
        p_depending_on_condition_type in varchar2 default null,
        p_depending_on_expression     in varchar2 default null,
        p_has_execute_condition       in boolean  default false,
        p_execute_cond_dep_prop_name  in varchar2 default null,
        p_execute_condition_help_text in varchar2 default null )
    is
        c_name_prefix constant varchar2( 30 ) := case when p_name_prefix is not null then p_name_prefix || '_' end;
    begin
        store_component_prop (
            p_component_type_id           => p_component_type_id,
            p_property_name               => c_name_prefix || 'CONDITION_TYPE',
            p_db_column                   => p_db_column_type,
            p_view_column                 => p_view_column_type,
            p_is_required                 => false,
            p_is_common                   => p_is_common,
            p_reference_scope             => p_reference_scope,
            p_plugin_std_attributes       => p_plugin_std_attributes,
            p_depending_on_comp_prop_name => p_depending_on_comp_prop_name,
            p_depending_on_condition_type => p_depending_on_condition_type,
            p_depending_on_expression     => p_depending_on_expression,
            p_help_text                   => p_help_text_type );
        --
        store_component_prop (
            p_component_type_id           => p_component_type_id,
            p_property_name               => c_name_prefix || 'CONDITION_SQL_STATEMENT',
            p_db_column                   => p_db_column_expr1,
            p_is_required                 => true,
            p_is_common                   => false,
            p_reference_scope             => p_reference_scope,
            p_depending_on_comp_prop_name => c_name_prefix || 'CONDITION_TYPE',
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'EXISTS,NOT_EXISTS' );
        --
        store_expression_comp_props (
            p_component_type_id => p_component_type_id,
            p_name_prefix       => c_name_prefix || 'CONDITION_',
            p_return_type       => 'BOOLEAN',
            p_db_column_lang    => p_db_column_expr2,
            p_db_column_expr    => p_db_column_expr1,
            p_reference_scope   => p_reference_scope,
            p_type_property     => c_name_prefix || 'CONDITION_TYPE' );
        --
        store_func_body_comp_props (
            p_component_type_id   => p_component_type_id,
            p_name_prefix         => c_name_prefix || 'CONDITION_',
            p_return_type         => 'BOOLEAN',
            p_db_column_lang      => p_db_column_expr2,
            p_db_column_func_body => p_db_column_expr1,
            p_reference_scope     => p_reference_scope,
            p_type_property       => c_name_prefix || 'CONDITION_TYPE' );
        --
        if nvl( p_name_prefix, 'CONDITION' ) not in ( 'AUTOM', 'SO_VAL', 'SO_COND', 'WF', 'RAG' ) then
            store_component_prop (
                p_component_type_id           => p_component_type_id,
                p_property_name               => c_name_prefix || 'CONDITION_VALUE1',
                p_db_column                   => p_db_column_expr1,
                p_is_required                 => true,
                p_is_common                   => false,
                p_reference_scope             => p_reference_scope,
                p_depending_on_comp_prop_name => c_name_prefix || 'CONDITION_TYPE',
                p_depending_on_condition_type => 'IN_LIST',
                p_depending_on_expression     => 'REQUEST_EQUALS_CONDITION,REQUEST_NOT_EQUAL_CONDITION,REQUEST_IN_CONDITION,REQUEST_NOT_IN_CONDITION,CURRENT_LANG_IN_COND1,CURRENT_LANG_NOT_IN_COND1,CURRENT_LANG_NOT_EQ_COND1,CURRENT_LANG_EQ_COND1,DAD_NAME_EQ_CONDITION,DAD_NAME_NOT_EQ_CONDITION,SERVER_NAME_EQ_CONDITION,SERVER_NAME_NOT_EQ_CONDITION,HTTP_HOST_EQ_CONDITION,HTTP_HOST_NOT_EQ_CONDITION' );
        end if;
        --
         if nvl( p_name_prefix, 'CONDITION' ) not in ( 'AUTOM', 'SO_VAL', 'SO_COND', 'RAG' ) then
            store_component_prop (
                p_component_type_id           => p_component_type_id,
                p_property_name               => c_name_prefix || 'CONDITION_ITEM1',
                p_db_column                   => p_db_column_expr1,
                p_is_required                 => true,
                p_is_common                   => false,
                p_reference_scope             => p_reference_scope,
                p_parent_comp_prop_name       => p_parent_comp_prop_name,
                p_depending_on_comp_prop_name => c_name_prefix || 'CONDITION_TYPE',
                p_depending_on_condition_type => 'IN_LIST',
                p_depending_on_expression     => 'VAL_OF_ITEM_IN_COND_EQ_COND2,VAL_OF_ITEM_IN_COND_NOT_EQ_COND2,VALUE_OF_ITEM_IN_COND_1_EQ_COND_2,VALUE_OF_ITEM_IN_COND_1_NOT_EQ_COND_2,ITEM_IS_NULL,ITEM_IS_NOT_NULL,ITEM_IS_ZERO,ITEM_IS_NOT_ZERO,ITEM_IS_NULL_OR_ZERO,ITEM_NOT_NULL_OR_ZERO,ITEM_CONTAINS_NO_SPACES,ITEM_IS_NUMERIC,ITEM_IS_NOT_NUMERIC,ITEM_IS_ALPHANUMERIC,VALUE_OF_ITEM_IN_CONDITION_IN_COLON_DELIMITED_LIST,VALUE_OF_ITEM_IN_CONDITION_NOT_IN_COLON_DELIMITED_LIST' );
        end if;
        --
        if nvl( p_name_prefix, 'CONDITION' ) not in ( 'AUTOM', 'ACTION', 'SO_VAL', 'SO_COND', 'WF', 'RAG' ) then
            store_component_prop (
                p_component_type_id           => p_component_type_id,
                p_property_name               => c_name_prefix || 'CONDITION_PREFERENCE',
                p_db_column                   => p_db_column_expr1,
                p_is_required                 => true,
                p_is_common                   => false,
                p_reference_scope             => p_reference_scope,
                p_depending_on_comp_prop_name => c_name_prefix || 'CONDITION_TYPE',
                p_depending_on_condition_type => 'IN_LIST',
                p_depending_on_expression     => 'USER_PREF_IN_COND_EQ_COND2,USER_PREF_IN_COND_NOT_EQ_COND2' );
            --
            store_component_prop (
                p_component_type_id           => p_component_type_id,
                p_property_name               => c_name_prefix || 'CONDITION_PAGE',
                p_db_column                   => p_db_column_expr1,
                p_is_required                 => true,
                p_is_common                   => false,
                p_reference_scope             => p_reference_scope,
                p_depending_on_comp_prop_name => c_name_prefix || 'CONDITION_TYPE',
                p_depending_on_condition_type => 'IN_LIST',
                p_depending_on_expression     => 'CURRENT_PAGE_EQUALS_CONDITION,CURRENT_PAGE_NOT_EQUAL_CONDITION' );
            --
            store_component_prop (
                p_component_type_id           => p_component_type_id,
                p_property_name               => c_name_prefix || 'CONDITION_PAGES',
                p_db_column                   => p_db_column_expr1,
                p_is_required                 => true,
                p_is_common                   => false,
                p_reference_scope             => p_reference_scope,
                p_depending_on_comp_prop_name => c_name_prefix || 'CONDITION_TYPE',
                p_depending_on_condition_type => 'IN_LIST',
                p_depending_on_expression     => 'CURRENT_PAGE_IN_CONDITION,CURRENT_PAGE_NOT_IN_CONDITION' );
        end if;
        --
        if nvl( p_name_prefix, 'CONDITION' ) not in ( 'AUTOM', 'SO_VAL', 'SO_COND', 'RAG' ) then
            store_component_prop (
                p_component_type_id           => p_component_type_id,
                p_property_name               => c_name_prefix || 'CONDITION_TEXT',
                p_db_column                   => p_db_column_expr1,
                p_is_required                 => true,
                p_is_common                   => false,
                p_reference_scope             => p_reference_scope,
                p_depending_on_comp_prop_name => c_name_prefix || 'CONDITION_TYPE',
                p_depending_on_condition_type => 'IN_LIST',
                p_depending_on_expression     => 'CONDITION1_IN_VALUE_OF_ITEM_IN_CONDITION2,DISPLAY_COND_IN_COND_TEXT,DISPLAY_COND_NOT_IN_COND_TEXT,DISPLAY_COND_EQUAL_COND_TEXT,DISP_COND_NOT_EQUAL_COND_TEXT' );
            --
            store_component_prop (
                p_component_type_id           => p_component_type_id,
                p_property_name               => c_name_prefix || 'CONDITION_VALUE2',
                p_db_column                   => p_db_column_expr2,
                p_is_required                 => true,
                p_is_common                   => false,
                p_reference_scope             => p_reference_scope,
                p_depending_on_comp_prop_name => c_name_prefix || 'CONDITION_TYPE',
                p_depending_on_condition_type => 'IN_LIST',
                p_depending_on_expression     => 'VAL_OF_ITEM_IN_COND_EQ_COND2,VAL_OF_ITEM_IN_COND_NOT_EQ_COND2,VALUE_OF_ITEM_IN_COND_1_EQ_COND_2,VALUE_OF_ITEM_IN_COND_1_NOT_EQ_COND_2,USER_PREF_IN_COND_EQ_COND2,USER_PREF_IN_COND_NOT_EQ_COND2' );
            --
            store_component_prop (
                p_component_type_id           => p_component_type_id,
                p_property_name               => c_name_prefix || 'CONDITION_LIST',
                p_db_column                   => p_db_column_expr2,
                p_is_required                 => true,
                p_is_common                   => false,
                p_reference_scope             => p_reference_scope,
                p_depending_on_comp_prop_name => c_name_prefix || 'CONDITION_TYPE',
                p_depending_on_condition_type => 'IN_LIST',
                p_depending_on_expression     => 'VALUE_OF_ITEM_IN_CONDITION_IN_COLON_DELIMITED_LIST,VALUE_OF_ITEM_IN_CONDITION_NOT_IN_COLON_DELIMITED_LIST' );
            --
            store_component_prop (
                p_component_type_id           => p_component_type_id,
                p_property_name               => c_name_prefix || 'CONDITION_ITEM2',
                p_db_column                   => p_db_column_expr2,
                p_is_required                 => true,
                p_is_common                   => false,
                p_reference_scope             => p_reference_scope,
                p_parent_comp_prop_name       => p_parent_comp_prop_name,
                p_depending_on_comp_prop_name => c_name_prefix || 'CONDITION_TYPE',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'CONDITION1_IN_VALUE_OF_ITEM_IN_CONDITION2' );
            --
            store_component_prop (
                p_component_type_id           => p_component_type_id,
                p_property_name               => c_name_prefix || 'CONDITION_TEXT2',
                p_db_column                   => p_db_column_expr2,
                p_is_required                 => true,
                p_is_common                   => false,
                p_reference_scope             => p_reference_scope,
                p_depending_on_comp_prop_name => c_name_prefix || 'CONDITION_TYPE',
                p_depending_on_condition_type => 'IN_LIST',
                p_depending_on_expression     => 'DISPLAY_COND_IN_COND_TEXT,DISPLAY_COND_NOT_IN_COND_TEXT,DISPLAY_COND_EQUAL_COND_TEXT,DISP_COND_NOT_EQUAL_COND_TEXT' );
        end if;
        --
        if p_name_prefix = 'RAG' then
            --
            store_component_prop (
                p_component_type_id           => p_component_type_id,
                p_property_name               => c_name_prefix || 'AI_WORD_LIST',
                p_db_column                   => p_db_column_expr1,
                p_is_required                 => true,
                p_is_common                   => true,
                p_reference_scope             => p_reference_scope,
                p_depending_on_comp_prop_name => c_name_prefix || 'CONDITION_TYPE',
                p_depending_on_condition_type => 'IN_LIST',
                p_depending_on_expression     => 'ANY_AI_USER_PROMPT_CONTAINS,LAST_AI_USER_PROMPT_CONTAINS' );
        end if;
        --
        if p_has_execute_condition then
            store_component_prop (
                p_component_type_id            => l_component_type_id,
                p_property_name                => 'EXECUTE_CONDITION',
                p_db_column                    => null,
                p_is_required                  => true,
                p_is_common                    => false,
                p_reference_scope              => p_reference_scope,
                p_default_value                => 'Y',
                p_depending_on_comp_prop_name  => c_name_prefix || 'CONDITION_TYPE',
                p_depending_on_condition_type  => 'NOT_IN_LIST',
                p_depending_on_expression      => 'NEVER,$NULL$',
                p_depending_on_comp_prop_name2 => p_execute_cond_dep_prop_name,
                p_depending_on_condition_type2 => case when p_execute_cond_dep_prop_name is not null then 'NOT_NULL' end,
                p_help_text                    => p_execute_condition_help_text );
        end if;
    end store_condition_type_comp_prop;
    --
    --
    procedure store_val_type_properties (
        p_start_property_id in number,
        p_name_prefix       in varchar2 )
    is
        c_name_prefix constant   varchar2( 30 )          := case when p_name_prefix is not null then p_name_prefix || '_' end;
        c_item_or_column_lower   constant varchar2( 10 ) := case when p_name_prefix is null then 'item' else 'column' end;
        c_item_or_column_initcap constant varchar2( 10 ) := initcap( c_item_or_column_lower );
        l_property_id number := p_start_property_id;
    begin
        --
        store_property (
            p_id                 => l_property_id,
            p_name               => c_name_prefix || 'VALIDATION_TYPE',
            p_prompt             => 'Type',
            p_property_type      => 'SELECT LIST',
            p_db_column          => 'VALIDATION_TYPE',
            p_display_group_name => 'VALIDATION',
            p_lov_type           => 'STATIC',
            p_help_text          => q'~<p>Select the type of equality to be tested for this validation. Depending on your selection, one or more additional attributes are required to fully define this validation.</p>
<p>If the validation passes the equality test, or evaluates to TRUE, then the validation error message does not display.
Validation error messages display when the validation fails the equality test, or evaluates to FALSE, or a non-empty text string is returned.</p>
<p>A validation failure does not stop subsequent validations from being evaluated, therefore, each subsequent validation may raise an additional validation error message. If you want to stop subsequent validations from executing then you need to add additional logic into the first failed validation, such as setting a page item to a specific value, and then enhancing the conditional logic on subsequent validations as needed. Subsequent processes and branches are not executed if one or more validations fail evaluation.</p>~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Rows returned',
            p_return_value  => 'EXISTS',
            p_help_text     => q'~Passes if the SQL query returns one or more rows.<br>
Fails if the SQL query returns no rows.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'No Rows returned',
            p_return_value  => 'NOT_EXISTS',
            p_is_quick_pick => 'Y',
            p_help_text     => q'~Passes if the SQL query returns no rows.<br>
Fails if the SQL query returns one or more rows.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Expression',
            p_return_value  => 'EXPRESSION',
            p_help_text     => q'~Passes if the expression evaluates to TRUE.<br>
Fails if the expression evaluates to FALSE.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'PL/SQL Error',
            p_return_value  => 'PLSQL_ERROR',
            p_help_text     => q'~Passes if the PL/SQL code executes without generating an error.<br>
Fails if the PL/SQL expression generates an error when it executes.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Function Body (returning Boolean)',
            p_return_value  => 'FUNC_BODY_RETURNING_BOOLEAN',
            p_is_quick_pick => 'Y',
            p_help_text     => q'~Passes if the Function body returns TRUE.<br>
Fails if the Function body returns FALSE.<br>
Note: The result returned from the function must be a boolean.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Function Body (returning Error Text)',
            p_return_value  => 'FUNC_BODY_RETURNING_ERR_TEXT',
            p_help_text     => q'~Passes if the Function body returns an empty string.<br>
Fails if the Function body returns a non-empty string. The returned string displays as the error message for the validation.<br>
Note: The result returned from the function must be a text string.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => c_item_or_column_initcap || ' is NOT NULL',
            p_return_value  => 'ITEM_NOT_NULL',
            p_is_quick_pick => 'Y',
            p_help_text     => q'~Passes if the ~' || c_item_or_column_initcap || q'~ value is not empty.<br>
Fails if the ~' || c_item_or_column_initcap || q'~ value is empty.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => c_item_or_column_initcap || ' is NOT NULL or zero',
            p_return_value  => 'ITEM_NOT_NULL_OR_ZERO',
            p_help_text     => q'~Passes if the ~' || c_item_or_column_initcap || q'~ value is not empty and is not the number zero.<br>
Fails if the ~' || c_item_or_column_initcap || q'~ value is empty or is the number zero.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => c_item_or_column_initcap || ' is NOT zero',
            p_return_value  => 'ITEM_NOT_ZERO',
            p_help_text     => q'~Passes if the ~' || c_item_or_column_initcap || q'~ value is not the number zero.<br>
Fails if the ~' || c_item_or_column_initcap || q'~ value is the number zero.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => c_item_or_column_initcap || ' contains no spaces',
            p_return_value  => 'ITEM_CONTAINS_NO_SPACES',
            p_help_text     => q'~Passes if the ~' || c_item_or_column_initcap || q'~ value has no spaces.<br>
Fails if the ~' || c_item_or_column_initcap || q'~ value contains spaces.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => c_item_or_column_initcap || ' is alphanumeric',
            p_return_value  => 'ITEM_IS_ALPHANUMERIC',
            p_help_text     => q'~Passes if the ~' || c_item_or_column_initcap || q'~ value is alphanumeric, containing only letters and numbers and no special characters.<br>
Fails if the ~' || c_item_or_column_initcap || q'~ value includes special characters.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => c_item_or_column_initcap || ' is numeric',
            p_return_value  => 'ITEM_IS_NUMERIC',
            p_help_text     => q'~Passes if the ~' || c_item_or_column_initcap || q'~ value is numeric.<br>
Fails if the ~' || c_item_or_column_initcap || q'~ value contains non-numeric characters, including spaces, other than leading spaces.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => c_item_or_column_initcap || ' is a valid date',
            p_return_value  => 'ITEM_IS_DATE',
            p_help_text     => q'~Passes if the ~' || c_item_or_column_initcap || q'~ value is a valid date.<br>
Fails if the ~' || c_item_or_column_initcap || q'~ value is not a valid date.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => c_item_or_column_initcap || ' is a valid timestamp',
            p_return_value  => 'ITEM_IS_TIMESTAMP',
            p_help_text     => q'~Passes if the ~' || c_item_or_column_initcap || q'~ value is a valid timestamp.<br>
Fails if the ~' || c_item_or_column_initcap || q'~ value is not a valid timestamp.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => c_item_or_column_initcap || ' = Value',
            p_return_value  => 'ITEM_IN_VALIDATION_EQ_STRING2',
            p_help_text     => q'~Passes if the ~' || c_item_or_column_initcap || q'~ value is equal to the text you enter into the <em>Value</em> attribute.<br>
Fails if the ~' || c_item_or_column_initcap || q'~ value is not equal to the text you enter into the <em>Value</em> attribute.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => c_item_or_column_initcap || ' != Value',
            p_return_value  => 'ITEM_IN_VALIDATION_NOT_EQ_STRING2',
            p_help_text     => q'~Passes if the ~' || c_item_or_column_initcap || q'~ value is not equal to the text you enter into the <em>Value</em> attribute.<br>
Fails if the ~' || c_item_or_column_initcap || q'~ value is equal to the text you enter into the <em>Value</em> attribute.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => c_item_or_column_initcap || ' is contained in Value',
            p_return_value  => 'ITEM_IN_VALIDATION_IN_STRING2',
            p_help_text     => q'~Passes if the ~' || c_item_or_column_initcap || q'~ value is completely contained in the text you enter into the <em>Value</em> attribute.<br>
Fails if the ~' || c_item_or_column_initcap || q'~ value is not completely contained in the text you enter into the <em>Value</em> attribute.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => c_item_or_column_initcap || ' is NOT contained in Value',
            p_return_value  => 'ITEM_IN_VALIDATION_NOT_IN_STRING2',
            p_help_text     => q'~Passes if the ~' || c_item_or_column_initcap || q'~ value is not completely contained in the text you enter into the <em>Value</em> attribute.<br>
Fails if the ~' || c_item_or_column_initcap || q'~ value is completely contained in the text you enter into the <em>Value</em> attribute.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => c_item_or_column_initcap || ' contains only characters specified in Value',
            p_return_value  => 'ITEM_IN_VALIDATION_CONTAINS_ONLY_CHAR_IN_STRING2',
            p_help_text     => q'~Passes if the ~' || c_item_or_column_initcap || q'~ value contains only characters found in the text you enter into the <em>Value</em> attribute.<br>
Fails if the ~' || c_item_or_column_initcap || q'~ value contains any characters that are not in the text you enter into the <em>Value</em> attribute.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => c_item_or_column_initcap || ' contains at least one of the characters in Value',
            p_return_value  => 'ITEM_IN_VALIDATION_CONTAINS_AT_LEAST_ONE_CHAR_IN_STRING2',
            p_help_text     => q'~Passes if the ~' || c_item_or_column_initcap || q'~ value includes at least one character from the text you enter into the <em>Value</em> attribute.<br>
Fails if the ~' || c_item_or_column_initcap || q'~ value does not include any characters from the text you enter into the <em>Value</em> attribute.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => c_item_or_column_initcap || ' does NOT contain any of the characters in Value',
            p_return_value  => 'ITEM_IN_VALIDATION_CONTAINS_NO_CHAR_IN_STRING2',
            p_help_text     => q'~Passes if the ~' || c_item_or_column_initcap || q'~ value does not include any characters from the text you enter into the <em>Value</em> attribute.<br>
Fails if the ~' || c_item_or_column_initcap || q'~ value includes at least one character from the text you enter into the <em>Value</em> attribute.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => c_item_or_column_initcap || ' matches Regular Expression',
            p_return_value  => 'REGULAR_EXPRESSION',
            p_help_text     => q'~Passes if the ~' || c_item_or_column_initcap || q'~ value matches the regular expression you enter into the <em>Value</em> attribute.
Fails if the ~' || c_item_or_column_initcap || q'~ value does not match the regular expression you enter into the <em>Value</em> attribute.<br>
Note: Regular expressions enable you to search for patterns in string data by using standardized syntax conventions, rather than just a straight character comparisons.~' );
        --
        l_property_id := l_property_id + 1;
        store_property (
            p_id                   => l_property_id,
            p_name                 => c_name_prefix || 'VAL_SQL_STATEMENT',
            p_prompt               => 'SQL Query',
            p_property_type        => 'SQL',
            p_db_column            => 'VALIDATION',
            p_display_length       => 80,
            p_max_length           => 4000,
            p_display_group_name   => 'VALIDATION',
            p_sql_min_column_count => 1,
            p_help_text            => q'~Enter a SQL query to define this validation.~',
            p_examples             => q'~
<dl><dt>For <strong>Rows Returned</strong>, to pass the validation, and not raise an error message, when the employee is in department 30:</dt>
<dd><pre>
select 1
  from emp
 where empno  = :P2_EMPNO
   and deptno = 30
</pre></dd>
<dt>For <strong>No Rows Returned</strong>, to pass the validation, and not raise an error message, when the employee is in department 40:</dt>
<dd><pre>
select 1
  from emp
 where empno  = :P2_EMPNO
   and deptno <> 40
</pre></dd>
</dl>~' );
        --
        if p_name_prefix is null then
            l_property_id := l_property_id + 1;
            store_property (
                p_id                       => l_property_id,
                p_name                     => 'VAL_ITEM',
                p_prompt                   => 'Item',
                p_property_type            => 'ITEM',
                p_db_column                => 'VALIDATION',
                p_display_length           => 80,
                p_max_length               => 255,
                p_display_group_name       => 'VALIDATION',
                p_lov_component_type_scope => 'APPLICATION',
                p_help_text                => q'~Enter a page or application item used in this validation. You can type in the name or pick from the list of available items.~' );
        end if;
        --
        if p_name_prefix = 'REGION' then
            l_property_id := l_property_id + 1;
            store_property (
                p_id                 => l_property_id,
                p_name               => c_name_prefix || 'VAL_COLUMN',
                p_prompt             => 'Column',
                p_property_type      => 'REGION COLUMN',
                p_db_column          => 'VALIDATION',
                p_display_length     => 80,
                p_max_length         => 128,
                p_display_group_name => 'VALIDATION',
                p_help_text          => q'~Enter a case-sensitive column name for this validation. You can type in the name or pick from the list of available columns.~' );
        end if;
        --
        l_property_id := l_property_id + 1;
        store_property (
            p_id                 => l_property_id,
            p_name               => c_name_prefix || 'VAL_VALUE',
            p_prompt             => 'Value',
            p_property_type      => 'TEXT EDITOR',
            p_db_column          => 'VALIDATION2',
            p_display_length     => 80,
            p_max_length         => 4000,
            p_display_group_name => 'VALIDATION',
            p_help_text          => q'~Enter the value used to compare against the value of the ~' || c_item_or_column_lower || q'~ column specified for this validation.~' );
        --
        l_property_id := l_property_id + 1;
        store_property (
            p_id                 => l_property_id,
            p_name               => c_name_prefix || 'VAL_REGULAR_EXPRESSION',
            p_prompt             => 'Regular Expression',
            p_property_type      => 'TEXT',
            p_db_column          => 'VALIDATION2',
            p_display_length     => 80,
            p_max_length         => 4000,
            p_display_group_name => 'VALIDATION',
            p_help_text          => q'~Enter the regular expression used to match the ~' || c_item_or_column_lower || q'~ specified for this validation.~' );
        --
        -- Note: the calls to store_expression_properties and store_func_body_properties have to be last, because if we have to add
        --       new properties in those procedures, it would change the ID of all properties which are created after
        --
        --
        l_property_id := l_property_id + 1;
        store_func_body_properties (
            p_start_property_id   => l_property_id,
            p_name_prefix         => c_name_prefix || 'VAL_',
            p_postfix             => 'ERROR',
            p_return_type         => 'VARCHAR2',
            p_prompt              => '%language% Function Body Returning Error Text',
            p_db_column_lang      => 'VALIDATION2',
            p_db_column_func_body => 'VALIDATION',
            p_display_group_name  => 'VALIDATION',
            p_help_text           => q'~Enter a %language% function body that returns the error message text as the basis for this validation.~',
            p_examples_plsql      => q'~
<dl><dt>To pass the validation, and not raise the specified error message, when the employee is in department 30:</dt>
<dd><pre>
begin
    if :P2_DEPTNO <> 30 then
        return 'Employee is in Department ' || :P2_DEPTNO || ' and not Department 30';
    end if;
    return null;
end;
</pre></dd>
</dl>~',
            p_examples_js         => q'~
<dl><dt>To pass the validation, and not raise the specified error message, when the employee is in department 30:</dt>
<dd><pre>
if ( apex.env.P2_DEPTNO !== "30" ) {
    return "Employee is in Department " + apex.env.P2_DEPTNO + " and not Department 30";
}
return "";
</pre></dd>
</dl>~' );
        --
        l_property_id := l_property_id + 20;
        store_property (
            p_id                 => l_property_id,
            p_name               => c_name_prefix || 'VAL_PLSQL',
            p_prompt             => 'PL/SQL Code raising Error',
            p_property_type      => 'PLSQL',
            p_db_column          => 'VALIDATION',
            p_display_length     => 80,
            p_max_length         => 4000,
            p_display_group_name => 'VALIDATION',
            p_help_text          => q'~Enter a PL/SQL expression that raises an error as the basis for this validation.~',
            p_examples           => q'~
<dl><dt>To pass the validation, and not raise the specified error, when the employee is in department 30:</dt>
<dd><pre>
begin
    if :P2_DEPTNO <> 30 then
        raise_application_error( -20000, 'Employee is in Department ' || :P2_DEPTNO || ' and not Department 30' );
    end if;
end;
</pre></dd>
</dl>~' );
        l_property_id := l_property_id + 1;
        store_expression_properties (
            p_start_property_id  => l_property_id,
            p_name_prefix        => c_name_prefix || 'VAL_',
            p_return_type        => 'BOOLEAN',
            p_db_column_lang     => 'VALIDATION2',
            p_db_column_expr     => 'VALIDATION',
            p_display_group_name => 'VALIDATION',
            p_help_text          => q'~Enter a %language% expression to define the validation.~',
            p_examples_sql       => q'~
<dl><dt>To pass the validation, and not raise an error message, when the employee is a manager or the department is located in Chicago:</dt>
<dd><pre>
:P2_JOB = 'MANAGER' or
exists ( select 1
           from dept
          where deptno = :P2_DEPTNO
            and loc    = 'CHICAGO' )
</pre></dd>
</dl>~',
            p_examples_plsql     => q'~
<dl><dt>To pass the validation, and not raise an error message, when the employee is in department 30 or is a manager:</dt>
<dd><pre>
( :P2_DEPTNO = 30 or :P2_JOB = 'MANAGER' )
</pre></dd>
</dl>~',
            p_examples_js        => q'~
<dl><dt>To pass the validation, and not raise an error message, when the employee is in department 30 or is a manager:</dt>
<dd><pre>
( apex.env.P2_DEPTNO === "30" || apex.env.P2_JOB == "MANAGER" )
</pre></dd>
</dl>~' );
        --
        l_property_id := l_property_id + 20;
        store_func_body_properties (
            p_start_property_id   => l_property_id,
            p_name_prefix         => c_name_prefix || 'VAL_',
            p_return_type         => 'BOOLEAN',
            p_db_column_lang      => 'VALIDATION2',
            p_db_column_func_body => 'VALIDATION',
            p_display_group_name  => 'VALIDATION',
            p_help_text           => q'~Enter a %language% function body that returns a BOOLEAN value as the basis for this validation.~',
            p_examples_plsql      => q'~
<dl><dt>To pass the validation, and not raise an error message, when the employee is in department 30:</dt>
<dd><pre>
begin
    if :P2_DEPTNO = 30 then
        return true;
    else
        return false;
    end if;
end;
</pre></dd>
</dl>~',
            p_examples_js         => q'~
<dl><dt>To pass the validation, and not raise an error message, when the employee is in department 30:</dt>
<dd><pre>
if ( apex.env.P2_DEPTNO === "30" ) {
    return true;
} else {
    return false;
}
</pre></dd>
</dl>~' );
        --
    end store_val_type_properties;
    --
    --
    procedure store_report_template_props (
        p_start_property_id     in integer )
    is
        l_type      constant wwv_flow_t_varchar2 := wwv_flow_t_varchar2('COLUMN','ROW');
    begin
        l_property_id :=  p_start_property_id - 1; -- Decrease the property id by 1 -> the increase is done again during the loop
        for t in 1..l_type.count loop
            for i in 1..4 loop
                l_property_id := l_property_id + 1;
                store_property (
                    p_id                 => l_property_id,
                    p_name               => l_type(t)||'_TEMPLATE_'||i,
                    p_prompt             => 'Template',
                    p_property_type      => 'HTML',
                    p_db_column          => 'ROW_TEMPLATE'||i,
                    p_max_length         => 4000,
                    p_display_group_name => l_type(t)||'_TEMPLATE_'||i,
                    p_is_translatable    => true,
                    p_examples           => q'~<pre>&lt;td #ALIGNMENT#&gt;#COLUMN_VALUE#&lt;/td&gt;</pre>~',
                    p_help_text          => q'~<p>Defines the ~'||lower(l_type(t))||q'~ template. You can use these substitution strings here:</p>
<ul class="noIndent">
<li>#COLUMN_VALUE# - Will be replaced with the value of a column</li>
<li>#ALIGNMENT# - Specified by the user, determines the alignment of a column</li>
<li>#COLNUM# - Current Column Number</li>
<li>#COLUMN_HEADER# - Column Header</li>
<li>#COLCOUNT# - Number of Columns</li>
<li>#ROWNUM# - Current Row Number</li>~'||
case l_type(t)
    when 'COLUMN' then
        q'~<li>#ACCESSIBLE_HEADERS# - Adds the row and column headers for the cell</li>
</ul><p>You can define up to four column templates, each of which can be conditional.  For example, you can have different background colors for even and odd rows, or highlight rows which meet a PL/SQL defined condition.</p>~'
    else
        q'~</ul><p>You can also use column number syntax to reference columns by select list position in the query, such as #1#, #2#.</p>~'
end );
                --
                l_property_id := l_property_id + 1;
                store_property (
                    p_id                 => l_property_id,
                    p_name               => l_type(t)||'_TEMPLATE_'||i||'_CONDITION',
                    p_prompt             => 'Condition',
                    p_property_type      => 'SELECT LIST',
                    p_db_column          => 'ROW_TEMPLATE_DISPLAY_COND'||i,
                    p_display_group_name => l_type(t)||'_TEMPLATE_'||i,
                    p_lov_type           => 'STATIC',
                    p_help_text          => q'~<p>Optionally select a condition that must be met in order to apply this ~'||lower(l_type(t))||q'~ template. You can select either even or odd row numbers, or supply a PL/SQL expression.</p>~' );
                --
                store_property_lov_values (
                    p_property_id   => l_property_id,
                    p_display_value => 'Use Based on PL/SQL Expression',
                    p_return_value  => 'NOT_CONDITIONAL' );
                --
                store_property_lov_values (
                    p_property_id   => l_property_id,
                    p_display_value => 'Use for Even Numbered Rows',
                    p_return_value  => 'EVEN_ROW_NUMBERS' );
                --
                store_property_lov_values (
                    p_property_id   => l_property_id,
                    p_display_value => 'Use for Odd Numbered Rows',
                    p_return_value  => 'ODD_ROW_NUMBERS' );
                --
                l_property_id := l_property_id + 1;
                store_property (
                    p_id                 => l_property_id,
                    p_name               => l_type(t)||'_TEMPLATE_'||i||'_EXPRESSION',
                    p_prompt             => 'PL/SQL Expression',
                    p_property_type      => 'PLSQL EXPRESSION BOOLEAN',
                    p_db_column          => 'ROW_TEMPLATE_CONDITION'||i,
                    p_max_length         => 4000,
                    p_display_group_name => l_type(t)||'_TEMPLATE_'||i,
                    p_examples           => q'~<p>If you wanted to display a value in bold if the salary was greater than 2000, you could use the following template condition:</p>
<pre>:SAL &gt; 2000</pre>~',
                    p_help_text          => q'~<p>Used only when the ~'||lower(l_type(t))||q'~ condition is set to Use Based on PL/SQL Expression. Enables you to format ~'||lower(l_type(t))||q'~s based on the data in that row.</p>
<p>Note that column names should be referenced using bind variable syntax, that is, use :SAL rather than #SAL#. You can also use the substitution string #ROWNUM# here.</p>~' );
            end loop;
        end loop;
    end store_report_template_props;
    --
    procedure store_list_tmpl_attr_props (
        p_start_property_id in number,
        p_attribute_count   in integer )
    is
        l_idx         varchar2(2);
    begin
        l_property_id := p_start_property_id - 1;
        --
        for i in 1..p_attribute_count loop
            l_idx         := lpad(i,2,'0');
            l_property_id := l_property_id + 1;
            store_property (
                p_id                 => l_property_id,
                p_name               => 'A'||l_idx||'_DESCRIPTION',
                p_prompt             => 'A'||l_idx,
                p_property_type      => 'TEXT',
                p_db_column          => 'A'||l_idx||'_LABEL',
                p_display_length     => 80,
                p_max_length         => 4000,
                p_display_group_name => 'ATTRIBUTE_DESCRIPTIONS',
                p_help_text          => 'Specify a brief description for the substitution string #A'||l_idx||'#, to assist in understanding its use in the current list template.' );
        end loop;
    end store_list_tmpl_attr_props;
    --
    procedure store_list_tmpl_att_comp_props (
        p_componant_type_id in number,
        p_attribute_count   in integer )
    is
        l_idx         varchar2(2);
    begin
        for i in 1..p_attribute_count loop
            l_idx := lpad(i,2,'0');
            store_component_prop (
                p_component_type_id     => l_component_type_id,
                p_property_name         => 'A'||l_idx||'_DESCRIPTION',
                p_db_column             => null,
                p_is_required           => false,
                p_is_common             => false,
                p_supports_substitution => false );
        end loop;
    end store_list_tmpl_att_comp_props;
    --
    --
    procedure store_comp_prop_report_tmpl
    is
        l_type          constant wwv_flow_t_varchar2 := wwv_flow_t_varchar2('COLUMN'         ,'ROW');
        l_expression    constant wwv_flow_t_varchar2 := wwv_flow_t_varchar2('GENERIC_COLUMNS','NAMED_COLUMNS');
        --
    begin
        for t in 1..l_type.count loop
            for i in 1..4 loop
                store_component_prop (
                    p_component_type_id           => l_component_type_id,
                    p_property_name               => l_type(t)||'_TEMPLATE_'||i,
                    p_db_column                   => null,
                    p_is_required                 => case i when 1 then true else false end,
                    p_is_common                   => false,
                    p_default_value               => case when l_type(t) = 'ROW' and i = 1 then '<tr><td>#1#</td><td>#2#</td><td>#3#</td><td>#4#</td><td>#5#</td></tr>' end,
                    p_supports_substitution       => false,
                    p_depending_on_comp_prop_name => 'CLASSIC_REPORT_TEMPLATE_TYPE',
                    p_depending_on_condition_type => 'EQUALS',
                    p_depending_on_expression     => l_expression(t) );
                --
                store_component_prop (
                    p_component_type_id           => l_component_type_id,
                    p_property_name               => l_type(t)||'_TEMPLATE_'||i||'_CONDITION',
                    p_db_column                   => null,
                    p_is_required                 => false,
                    p_is_common                   => false,
                    p_depending_on_comp_prop_name => l_type(t)||'_TEMPLATE_'||i,
                    p_depending_on_condition_type => 'NOT_NULL' );
                --
                store_component_prop (
                    p_component_type_id           => l_component_type_id,
                    p_property_name               => l_type(t)||'_TEMPLATE_'||i||'_EXPRESSION',
                    p_db_column                   => null,
                    p_is_required                 => true,
                    p_is_common                   => false,
                    p_supports_substitution       => false,
                    p_depending_on_comp_prop_name => l_type(t)||'_TEMPLATE_'||i||'_CONDITION',
                    p_depending_on_condition_type => 'EQUALS',
                    p_depending_on_expression     => 'NOT_CONDITIONAL' );
            end loop;
        end loop;
    end store_comp_prop_report_tmpl;
    --
    --
    procedure store_order_by_item_property (
        p_name               in varchar2,
        p_prompt             in varchar2,
        p_display_group_name in varchar2 )
    is
    begin
        store_property (
            p_id                 => l_property_id,
            p_name               => p_name,
            p_prompt             => p_prompt,
            p_property_type      => 'ORDER BY ITEM',
            p_db_column          => 'QUERY_ORDER_BY',
            p_display_group_name => p_display_group_name,
            p_examples           => q'~
<dl><dt>To show the newest hires first and those which have not been hired yet last. If the date is equal, sort them by department number.</dt>
<dd><pre>
HIREDATE DESC NULLS LAST, DEPTNO
</pre></dd>
</dl>~',
            p_help_text          => q'~<p>Enter multiple Order By Clauses that can be picked by end-users. Each can contain one or more comma-separated column names, including directives, to define the order in which records are retrieved.</p>
            <p>Directives include:
              <ul>
                <li>ASC - Return column in ascending order</li>
                <li>DESC - Return column in descending order</li>
                <li>NULLS FIRST - Return column with no value before those with values</li>
                <li>NULLS LAST - Return column with values before those with no value</li>
              </ul>
            </p>~' );
    end store_order_by_item_property;
    --
    --
    procedure store_post_processing_props(
        p_prefix        in varchar2,
        p_ids           in wwv_flow_t_number )
    is
        l_source_type varchar2(255) := case p_prefix
                                            when 'DUALITYVIEW_'    then 'Duality View'
                                            when 'JSONCOLLECTION_' then 'JSON'
                                            else                       'REST Source'
                                       end;
    begin
        l_property_id := p_ids( 1 );
        store_property (
            p_id                 => l_property_id,
            p_name               => p_prefix || 'SOURCE_POST_PROCESSING',
            p_prompt             => 'Type',
            p_property_type      => 'SELECT LIST',
            p_db_column          => 'SOURCE_POST_PROCESSING',
            p_display_group_name => 'LOCAL_POST_PROCESSING',
            p_lov_type           => 'STATIC',
            p_help_text          => 'Select how to post process the retrieved ' || l_source_type || ' data in the local database.' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Where/Order By Clause',
            p_return_value  => 'WHERE_ORDER_BY_CLAUSE',
            p_help_text     => q'~Enter the where and/or order by clause to restrict and order the data retrieved.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'SQL Query',
            p_return_value  => 'SQL',
            p_help_text     => q'~Enter a SQL Query to transform the data retrieved.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'PL/SQL Function Body returning SQL Query',
            p_return_value  => 'FUNC_BODY_RETURNING_SQL',
            p_help_text     => q'~Enter a PL/SQL function body, where the SQL query returned transforms the data retrieved.~' );
        --
        l_property_id := p_ids( 2 );
        store_property (
            p_id                 => l_property_id,
            p_name               => p_prefix || 'POST_PROC_WHERE',
            p_prompt             => 'Where Clause',
            p_property_type      => 'WHERE CLAUSE',
            p_db_column          => 'QUERY_WHERE',
            p_display_length     => 40,
            p_max_length         => 4000,
            p_display_group_name => 'LOCAL_POST_PROCESSING',
            p_examples           => q'~
    <dl><dt>Only include those employees which have a manager or have a job of salesman.</dt>
    <dd><pre>
    ( MGR is not null or JOB = 'SALESMAN' )
    </pre></dd>
    </dl>~',
            p_help_text          => q'~Enter a valid SQL where condition, used to limit the records retrieved.~' );
        --
        l_property_id := p_ids( 3 );
        store_property (
            p_id                 => l_property_id,
            p_name               => p_prefix || 'POST_PROC_ORDER_BY',
            p_prompt             => 'Order By Clause',
            p_property_type      => 'ORDER BY CLAUSE',
            p_db_column          => 'QUERY_ORDER_BY',
            p_display_length     => 40,
            p_max_length         => 255,
            p_display_group_name => 'LOCAL_POST_PROCESSING',
            p_examples           => q'~
    <dl><dt>To show the newest hires first and those which have not been hired yet last. If the date is equal, sort them by department number.</dt>
    <dd><pre>
    HIREDATE DESC NULLS LAST, DEPTNO
    </pre></dd>
    </dl>~',
            p_help_text          => q'~<p>Enter one or more comma separated column names, including directives, to define the order in which records are retrieved.</p>
            <p>Directives include:
              <ul>
                <li>ASC - Return column in ascending order</li>
                <li>DESC - Return column in descending order</li>
                <li>NULLS FIRST - Return column with no value before those with values</li>
                <li>NULLS LAST - Return column with values before those with no value</li>
              </ul>
            </p>~' );
        --
        l_property_id := p_ids( 4 );
        store_property (
            p_id                   => l_property_id,
            p_name                 => p_prefix || 'POST_PROC_SQL_QUERY',
            p_prompt               => 'SQL Query',
            p_property_type        => 'SQL',
            p_db_column            => 'PLUG_SOURCE',
            p_display_length       => 80,
            p_max_length           => 32767,
            p_sql_min_column_count => 1,
            p_display_group_name   => 'LOCAL_POST_PROCESSING',
            p_examples             => q'~
    <dl><dt>Select the EMPNO, ENAME and SAL columns and filter out all rows with a SAL less then 3000.</dt>
    <dd><pre>
    select EMPNO, ENAME, SAL
      from #APEX$SOURCE_DATA#
     where SAL < 3000
    </pre></dd>
    </dl>~',
            p_help_text            => q'~Enter a SQL Query to transform the data retrieved from the REST Data Source. Select from the pseudo-table #APEX$SOURCE_DATA# and specify columns available in the data profile. Subquery factoring using the SQL WITH clause is not supported here.~' );
        --
        store_func_body_properties (
            p_start_property_id   => p_ids( 8 ),
            p_name_prefix         => p_prefix || 'POST_PROC_',
            p_return_type         => 'SQL',
            p_db_column_lang      => 'FUNCTION_BODY_LANGUAGE',
            p_db_column_func_body => 'DATA_SOURCE',
            p_display_group_name  => 'LOCAL_POST_PROCESSING',
            p_help_text           => q'~Enter a %language% function body, where the SQL query returned transforms the data retrieved from the Web Source. The SQL query returned by the %language% function body must select from the #APEX$SOURCE_DATA# pseudo-table and read columns available in the data profile. Subquery factoring using the SQL WITH clause is not supported here. ~',
            p_examples_plsql      => q'~
    <dl><dt>Select the EMPNO, ENAME and SAL columns and filter out all rows with a SAL less then 3000.</dt>
    <dd><pre>
    begin
        return 'select EMPNO, ENAME, SAL from #APEX$SOURCE_DATA where SAL < 3000';
    end;
    </pre></dd>
    </dl>~',
            p_examples_js         => q'~
    <dl><dt>Select the EMPNO, ENAME and SAL columns and filter out all rows with a SAL less then 3000.</dt>
    <dd><pre>
    return "select EMPNO, ENAME, SAL from #APEX$SOURCE_DATA where SAL < 3000";
    </pre></dd>
    </dl>~' );

        l_property_id := p_ids( 5 );
        store_property (
            p_id                 => l_property_id,
            p_name               => p_prefix || 'POST_PROC_ORDER_BY_TYPE',
            p_prompt             => 'Order By Type',
            p_property_type      => 'SELECT LIST',
            p_db_column          => 'QUERY_ORDER_BY_TYPE',
            p_display_group_name => 'LOCAL_POST_PROCESSING',
            p_lov_type           => 'STATIC',
            p_help_text          => q'~Select how this component is sorted.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Static Value',
            p_return_value  => 'STATIC',
            p_is_quick_pick => 'Y',
            p_help_text     => q'~Based on the ORDER BY clause entered into <em>Order By Clause</em>.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Item',
            p_return_value  => 'ITEM',
            p_is_quick_pick => 'Y',
            p_help_text     => q'~Based on the value held in session state for the <em>Item</em> selected.~' );
        --
        l_property_id := p_ids( 6 );
        store_order_by_item_property (
            p_name               => p_prefix || 'POST_PROC_ORDER_BY_ITEM',
            p_prompt             => 'Item',
            p_display_group_name => 'LOCAL_POST_PROCESSING' );
        --
        l_property_id := p_ids( 7 );
        store_order_by_item_property (
            p_name               => p_prefix || 'POST_PROC_ORDER_BY_ITEM2',
            p_prompt             => 'Order By Item',
            p_display_group_name => 'LOCAL_POST_PROCESSING' );
    end store_post_processing_props;
    --
    --
    procedure store_val_type_comp_props (
        p_name_prefix in varchar2 )
    is
        c_name_prefix     constant varchar2( 30 ) := case when p_name_prefix is not null then p_name_prefix || '_' end;
        c_type_property   constant varchar2( 30 ) := c_name_prefix || 'VALIDATION_TYPE';
        c_reference_scope constant varchar2( 30 ) := case when p_name_prefix is not null then 'ROW' else 'COMPONENT' end;
    begin
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => c_type_property,
            p_db_column                   => null,
            p_view_column                 => 'VALIDATION_TYPE',
            p_is_required                 => true,
            p_is_common                   => true,
            p_default_value               => 'EXPRESSION',
            p_depending_on_comp_prop_name => 'VALIDATION_REGION',
            p_depending_on_condition_type => case when p_name_prefix is not null then 'NOT_NULL' else 'NULL' end );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => c_name_prefix || 'VAL_SQL_STATEMENT',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_reference_scope             => c_reference_scope,
            p_depending_on_comp_prop_name => c_type_property,
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'EXISTS,NOT_EXISTS' );
        --
        store_expression_comp_props (
            p_component_type_id => l_component_type_id,
            p_name_prefix       => c_name_prefix || 'VAL_',
            p_return_type       => 'BOOLEAN',
            p_reference_scope   => c_reference_scope,
            p_type_property     => c_type_property );
        --
        store_func_body_comp_props (
            p_component_type_id => l_component_type_id,
            p_name_prefix       => c_name_prefix || 'VAL_',
            p_return_type       => 'BOOLEAN',
            p_reference_scope   => c_reference_scope,
            p_type_property     => c_type_property,
            p_type_value        => 'FUNC_BODY_RETURNING_BOOLEAN' );
        --
        store_func_body_comp_props (
            p_component_type_id   => l_component_type_id,
            p_name_prefix         => c_name_prefix || 'VAL_',
            p_postfix             => 'ERROR',
            p_return_type         => 'VARCHAR2',
            p_db_column_lang      => null,
            p_db_column_func_body => null,
            p_reference_scope     => c_reference_scope,
            p_type_property       => c_type_property,
            p_type_value          => 'FUNC_BODY_RETURNING_ERR_TEXT' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => c_name_prefix || 'VAL_PLSQL',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_reference_scope             => c_reference_scope,
            p_depending_on_comp_prop_name => c_type_property,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'PLSQL_ERROR' );
        --
        if p_name_prefix is null then
            store_component_prop (
                p_component_type_id            => l_component_type_id,
                p_property_name                => c_name_prefix || 'VAL_ITEM',
                p_db_column                    => null,
                p_is_required                  => true,
                p_is_common                    => true,
                p_reference_on_delete          => 'WEAK_CASCADE',
                p_depending_on_comp_prop_name  => c_type_property,
                p_depending_on_condition_type  => 'IN_LIST_LIKE',
                p_depending_on_expression      => 'ITEM,REGULAR_EXPRESSION' );
        end if;
        --
        if p_name_prefix = 'REGION' then
            store_component_prop (
                p_component_type_id            => l_component_type_id,
                p_property_name                => c_name_prefix || 'VAL_COLUMN',
                p_db_column                    => null,
                p_is_required                  => true,
                p_is_common                    => true,
                p_parent_comp_prop_name        => 'VALIDATION_REGION',
                p_depending_on_comp_prop_name  => c_type_property,
                p_depending_on_condition_type  => 'IN_LIST_LIKE',
                p_depending_on_expression      => 'ITEM,REGULAR_EXPRESSION' );
        end if;
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => c_name_prefix || 'VAL_REGULAR_EXPRESSION',
            p_db_column                    => null,
            p_is_required                  => true,
            p_is_common                    => true,
            p_reference_scope              => 'DEPENDS',
            p_depending_on_comp_prop_name  => c_type_property,
            p_depending_on_condition_type  => 'EQUALS',
            p_depending_on_expression      => 'REGULAR_EXPRESSION' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => c_name_prefix || 'VAL_VALUE',
            p_db_column                    => null,
            p_is_required                  => true,
            p_is_common                    => true,
            p_reference_scope              => c_reference_scope,
            p_depending_on_comp_prop_name  => c_type_property,
            p_depending_on_condition_type  => 'IN_LIST',
            p_depending_on_expression      => 'ITEM_IN_VALIDATION_EQ_STRING2,ITEM_IN_VALIDATION_NOT_EQ_STRING2,ITEM_IN_VALIDATION_IN_STRING2,ITEM_IN_VALIDATION_NOT_IN_STRING2,ITEM_IN_VALIDATION_CONTAINS_AT_LEAST_ONE_CHAR_IN_STRING2,ITEM_IN_VALIDATION_CONTAINS_ONLY_CHAR_IN_STRING2,ITEM_IN_VALIDATION_CONTAINS_NO_CHAR_IN_STRING2' );
        --
    end store_val_type_comp_props;
    --
    --
    procedure store_audit_info_comp_prop (
        p_component_type_id in number,
        p_db_column_prefix  in varchar2 default 'LAST_' )
    is
    begin

        store_component_prop (
            p_component_type_id     => p_component_type_id,
            p_property_name         => 'CHANGED_BY',
            p_db_column             => p_db_column_prefix || 'UPDATED_BY',
            p_import_api_parameter  => 'updated_by',
            p_is_required           => false,
            p_is_common             => false,
            p_is_read_only          => true,
            p_supports_substitution => false );
        --
        store_component_prop (
            p_component_type_id     => p_component_type_id,
            p_property_name         => 'CHANGED_ON',
            p_db_column             => p_db_column_prefix || 'UPDATED_ON',
            p_import_api_parameter  => 'updated_on',
            p_is_required           => false,
            p_is_common             => false,
            p_is_read_only          => true,
            p_supports_substitution => false );
    end store_audit_info_comp_prop;
    --
    --
    procedure store_standard_attr_comp_prop (
        p_component_type_id           in number   default l_component_type_id,
        p_has_condition               in boolean  default false,
        p_cond_type_name_prefix       in varchar2 default null,
        p_cond_type_help_text         in varchar2 default null,
        p_cond_type_db_column         in varchar2 default null,
        p_cond_type_view_db_column    in varchar2 default null,
        p_cond_exp1_db_column         in varchar2 default null,
        p_cond_exp2_db_column         in varchar2 default null,
        p_has_authorization_scheme    in boolean  default false,
        p_authorization_db_column     in varchar2 default null,
        p_authz_dep_on_comp_prop_name in varchar2 default null,
        p_authz_dep_on_condition_type in varchar2 default null,
        p_authz_dep_on_expression     in varchar2 default null,
        p_has_build_option            in boolean  default false,
        p_build_option_db_column      in varchar2 default null,
        p_has_comments                in boolean  default false,
        p_comment_db_column           in varchar2 default null,
        p_comment_import_api_param    in varchar2 default null,
        p_audit_db_column_prefix      in varchar2 default 'LAST_' )
    is
    begin
        if p_has_condition then
            store_condition_type_comp_prop (
                p_component_type_id=> p_component_type_id,
                p_name_prefix      => p_cond_type_name_prefix,
                p_help_text_type   => p_cond_type_help_text,
                p_db_column_type   => p_cond_type_db_column,
                p_view_column_type => p_cond_type_view_db_column,
                p_db_column_expr1  => p_cond_exp1_db_column,
                p_db_column_expr2  => p_cond_exp2_db_column );
        end if;
        --
        if p_has_authorization_scheme or p_authorization_db_column is not null then
            store_component_prop (
                p_component_type_id           => p_component_type_id,
                p_property_name               => 'AUTHORIZATION_SCHEME',
                p_db_column                   => p_authorization_db_column,
                p_is_required                 => false,
                p_is_common                   => false,
                p_reference_on_delete         => 'WEAK_NULL',
                p_depending_on_comp_prop_name => p_authz_dep_on_comp_prop_name,
                p_depending_on_condition_type => p_authz_dep_on_condition_type,
                p_depending_on_expression     => p_authz_dep_on_expression );
        end if;
        --
        if p_has_build_option or p_build_option_db_column is not null then
            store_component_prop (
                p_component_type_id   => p_component_type_id,
                p_property_name       => 'BUILD_OPTION',
                p_db_column           => p_build_option_db_column,
                p_is_required         => false,
                p_is_common           => false,
                p_reference_on_delete => 'WEAK_NULL' );
        end if;
        --
        if p_has_comments or p_comment_db_column is not null then
            store_component_prop (
                p_component_type_id     => p_component_type_id,
                p_property_name         => 'COMMENTS',
                p_db_column             => p_comment_db_column,
                p_import_api_parameter  => p_comment_import_api_param,
                p_is_required           => false,
                p_is_common             => false,
                p_supports_substitution => false );
        end if;
        --
        store_audit_info_comp_prop (
            p_component_type_id => p_component_type_id,
            p_db_column_prefix  => p_audit_db_column_prefix );
        --
    end store_standard_attr_comp_prop;
    --
    --
    procedure store_process_type_comp_prop (
        p_name_db_column           in varchar2,
        p_name_help_text           in varchar2,
        p_db_column_prefix         in varchar2,
        p_execution_seq_db_column  in varchar2,
        p_error_message_db_column  in varchar2,
        p_error_message_help_text  in varchar2 default null,
        p_stop_execution_help_text in varchar2 default null,
        p_has_authorization_scheme in boolean,
        p_build_option_db_column   in varchar2,
        p_comment_db_column        in varchar2,
        p_comment_import_api_param in varchar2 default null )
    is
    begin
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'NAME',
            p_db_column             => p_name_db_column,
            p_is_required           => true,
            p_is_common             => true,
            p_check_uniqueness      => case l_component_type_id
                                         when c_comp_type_app_process  then 'APPLICATION'
                                         when c_comp_type_page_process then 'PAGE'
                                         else 'PARENT'
                                       end,
            p_supports_substitution => false,
            p_default_value         => '#PLEASE_CHANGE#',
            p_help_text             => p_name_help_text );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'PROCESS_TYPE',
            p_db_column         => case when p_db_column_prefix is not null then p_db_column_prefix || 'TYPE' end,
            p_is_required       => true,
            p_is_common         => true,
            p_default_value     => 'NATIVE_PLSQL' );
        --
        if l_component_type_id = c_comp_type_page_process then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'PARENT_PROCESS',
                p_is_required                 => false,
                p_is_common                   => true,
                p_null_text                   => 'None',
                p_reference_on_delete         => 'WEAK_CASCADE' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'INTERNAL_UID',
                p_is_required                 => false,
                p_is_read_only                => true,
                p_is_common                   => true );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'PROCESS_DATA_SOURCE_REGION',
                p_db_column                   => null,
                p_is_required                 => null, -- defined by plug-in standard attributes
                p_is_common                   => true,
                p_reference_on_delete         => 'WEAK_CASCADE',
                p_plugin_std_attributes       => 'FORM' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'PROCESS_REGION',
                p_db_column                   => null,
                p_is_required                 => null, -- defined by plug-in standard attributes
                p_is_common                   => true,
                p_reference_on_delete         => 'WEAK_CASCADE',
                p_plugin_std_attributes       => 'REGION' );
        end if;
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'PLSQL_PROCESS_LOCATION',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'LOCAL',
            p_depending_on_comp_prop_name => 'PROCESS_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'NATIVE_PLSQL' );
        --
        if l_component_type_id = c_comp_type_workflow_activity then
            store_snippet_comp_props (
                p_component_type_id           => l_component_type_id,
                p_name_prefix                 => 'SOURCE_',
                p_db_column_lang              => 'ACTIVITY_CODE_LANGUAGE',
                p_db_column_snippet           => 'ACTIVITY_CODE',
                p_reference_scope             => 'COMPONENT',
                p_type_property               => 'PLSQL_PROCESS_LOCATION',
                p_type_value                  => 'LOCAL' );
        --
        else
            store_snippet_comp_props (
                p_component_type_id           => l_component_type_id,
                p_name_prefix                 => 'SOURCE_',
                p_db_column_lang              => case when p_db_column_prefix is not null then p_db_column_prefix || 'CLOB_LANGUAGE' end,
                p_db_column_snippet           => case when p_db_column_prefix is not null then p_db_column_prefix || 'CLOB' end,
                p_reference_scope             => case l_component_type_id
                                                   when c_comp_type_page_process then 'DEPENDS'
                                                   else 'COMPONENT'
                                                  end,
                p_type_property               => 'PLSQL_PROCESS_LOCATION',
                p_type_value                  => 'LOCAL' );
        end if;
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'SOURCE_REMOTE_ORACLE_DATABASE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => 'PLSQL_PROCESS_LOCATION',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'REMOTE' );
        --
        if l_component_type_id = c_comp_type_workflow_activity then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'PLSQL_CODE',
                p_db_column                   => 'ACTIVITY_CODE',
                p_is_required                 => true,
                p_is_common                   => true,
                p_reference_scope             => 'COMPONENT',
                p_depending_on_comp_prop_name => 'PLSQL_PROCESS_LOCATION',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'REMOTE',
                p_help_text                   => q'~Enter the PL/SQL code to be executed on the remote database.~' );
        else
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'PLSQL_CODE',
                p_db_column                   => case
                                               when p_db_column_prefix is not null then p_db_column_prefix || 'CLOB'
                                               else 'PROCESS_SQL_CLOB'
                                             end,
                p_is_required                 => true,
                p_is_common                   => true,
                p_reference_scope             => case l_component_type_id
                                                   when c_comp_type_page_process then 'DEPENDS'
                                                   else 'COMPONENT'
                                                 end,
                p_depending_on_comp_prop_name => 'PLSQL_PROCESS_LOCATION',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'REMOTE',
                p_help_text                   => q'~Enter the PL/SQL code to be executed on the remote database.~' );
        end if;
        --
        store_plugin_attr_seq_offset (
            p_component_type_id => l_component_type_id );
        --
        if l_component_type_id = c_comp_type_task_def_action then
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'LOG_MESSAGE_TYPE',
                p_db_column         => null,
                p_is_required       => true,
                p_is_common         => true,
                p_default_value     => 'NONE' );
            --
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'ON_EVENT',
                p_db_column         => null,
                p_is_required       => true,
                p_is_common         => true,
                p_default_value     => 'COMPLETE' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'OUTCOME',
                p_db_column                   => null,
                p_is_required                 => false,
                p_is_common                   => true,
                p_depending_on_comp_prop_name => 'ON_EVENT',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'COMPLETE' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'BEFORE_EXPIRE_INTERVAL',
                p_db_column                   => null,
                p_is_required                 => false,
                p_is_common                   => true,
                p_depending_on_comp_prop_name => 'ON_EVENT',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'BEFORE_EXPIRE' );
        end if;
        --
        if l_component_type_id = c_comp_type_workflow_activity then

            store_component_prop (
                p_component_type_id        => l_component_type_id,
                p_property_name            => 'WF_SOURCE_SQL_QUERY',
                p_db_column                => 'QUERY_SOURCE',
                p_is_required              => false,
                p_is_common                => true,
                p_help_text                => q'~<p>Enter a SQL query to read additional data.</p>
                <p>The column names of the query can be used as bind variables and substitution
                strings to evaluate conditions or activity variables during the execution of the activity.</p>
                <p>The result of the query should contain only one row.</p>
                <p> If the query returns no result or more than one row then the activity fails during execution.</p>~' );

            store_component_prop (
                p_component_type_id     => l_component_type_id,
                p_property_name         => 'DUE_ON_TYPE',
                p_db_column             => null,
                p_is_required           => false,
                p_null_text             => 'None',
                p_is_common             => true,
                p_depending_on_comp_prop_name => 'PROCESS_TYPE',
                p_depending_on_condition_type => 'NOT_EQUALS',
                p_depending_on_expression     => 'NATIVE_WORKFLOW_WAIT');
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'DUE_ON_INTERVAL_VALUE',
                p_db_column                   => 'DUE_ON_VALUE',
                p_is_required                 => true,
                p_is_common                   => true,
                p_depending_on_comp_prop_name => 'DUE_ON_TYPE',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'INTERVAL' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'DUE_ON_QUERY_VALUE',
                p_db_column                   => 'DUE_ON_VALUE',
                p_is_required                 => true,
                p_is_common                   => true,
                p_depending_on_comp_prop_name => 'DUE_ON_TYPE',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'SQL_QUERY' );
            --
            store_expression_comp_props (
                p_component_type_id => l_component_type_id,
                p_name_prefix       => 'DUE_ON_',
                p_return_type       => 'CUSTOM',
                p_custom_data_type  => 'TIMESTAMP WITH TIME ZONE',
                p_type_property     => 'DUE_ON_TYPE' );
            --
            store_func_body_comp_props (
                p_component_type_id => l_component_type_id,
                p_name_prefix       => 'DUE_ON_',
                p_return_type       => 'CUSTOM',
                p_custom_data_type  => 'TIMESTAMP WITH TIME ZONE',
                p_type_property     => 'DUE_ON_TYPE' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'DUE_ON_SCHEDULER_VALUE',
                p_db_column                   => 'DUE_ON_VALUE',
                p_is_required                 => true,
                p_is_common                   => true,
                p_depending_on_comp_prop_name => 'DUE_ON_TYPE',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'SCHEDULER_EXPRESSION' );
            --

        end if;

        --
        if l_component_type_id = c_comp_type_workflow_activity then
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'DISPLAY_SEQUENCE',
                p_db_column         => 'DISPLAY_SEQUENCE',
                p_is_required       => true,
                p_is_common         => true );
        else
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'EXECUTION_SEQUENCE',
                p_db_column         => p_execution_seq_db_column,
                p_is_required       => true,
                p_is_common         => true );
        end if;
        --
        if l_component_type_id = c_comp_type_app_process then
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'APP_PROCESS_POINT',
                p_db_column         => null,
                p_is_required       => true,
                p_is_common         => true,
                p_default_value     => 'BEFORE_HEADER' );
            --
        elsif l_component_type_id = c_comp_type_page_process then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'PROCESS_POINT',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => true,
                p_default_value               => 'AFTER_SUBMIT',
                p_depending_on_comp_prop_name => 'PARENT_PROCESS',
                p_depending_on_condition_type => 'NULL' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'RUN_PROCESS',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => false,
                p_default_value               => 'N',
                p_depending_on_comp_prop_name => 'PARENT_PROCESS',
                p_depending_on_condition_type => 'NULL' );
        end if;

        if l_component_type_id in ( c_comp_type_page_process, c_comp_type_task_def_action ) then
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'SUCCESS_MESSAGE',
                p_db_column         => case when l_component_type_id = c_comp_type_task_def_action then 'SUCCESS_MESSAGE' end,
                p_is_required       => false,
                p_is_common         => true,
                p_help_text         => q'~<p>Enter the success message for this process.
If the process runs and does not generate an error, then this process success message displays in the notification section of the resulting page displayed.
If you are branching to another page via a URL redirect, you may need to check the <em>preserve success message</em> attribute.</p>
<p>For multi row processes, the following substitution string can be used to get the number of records processed:</p>
<ul>
  <li>Insert: <code>#MRI_COUNT#</code></li>
  <li>Update: <code>#MRU_COUNT#</code></li>
  <li>Delete: <code>#MRD_COUNT#</code></li>
</ul>
<p>For the Send E-Mail process type the substitution string <code>#TO#</code> can be used to get the addressees of the e-mail.</p>
<p>Plug-ins can have other substitution strings as well. See Plug-in documentation for details.</p>~' );
            --
        end if;
        --
        if l_component_type_id != c_comp_type_workflow_activity then
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'ERROR_MESSAGE',
                p_db_column         => p_error_message_db_column,
                p_is_required       => false,
                p_is_common         => false,
                p_help_text         => coalesce( p_error_message_help_text, q'~<p>Enter the error message for this process.
This message displays if an unhandled exception is raised. After any error processing stops, a rollback is issued and an error message displays.</p>
<p>Note: the SQL error message sqlerrm displays by default if <strong>On Error Page</strong> is defined as the error display location, there is no need to add <code>#SQLERRM#</code> to your error text.</p>
<p>Error messages can include the following substitution strings:</p>
<dl>
  <dt><code>#SQLERRM_TEXT#</code></dt>
  <dd>Text of error message without the error number.</dd>
  <dt><code>#SQLERRM#</code></dt>
  <dd>Complete error message.</dd>
</dl>~' ) );
        end if;
        --
        if l_component_type_id in ( c_comp_type_app_process, c_comp_type_page_process ) then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'PROCESS_ERROR_DISPLAY_LOCATION',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => false,
                p_default_value               => 'INLINE_IN_NOTIFICATION',
                p_depending_on_comp_prop_name => case l_component_type_id
                                                   when c_comp_type_app_process  then 'APP_PROCESS_POINT'
                                                   when c_comp_type_page_process then 'PROCESS_POINT'
                                                 end,
                p_depending_on_condition_type => 'IN_LIST',
                p_depending_on_expression     => 'ON_SUBMIT_BEFORE_COMPUTATION,AFTER_SUBMIT' );
        else
            if l_component_type_id != c_comp_type_workflow_activity then
                store_component_prop (
                    p_component_type_id => l_component_type_id,
                    p_property_name     => 'ERROR_STOP_EXECUTION_ON_ERROR',
                    p_db_column         => null,
                    p_is_required       => true,
                    p_is_common         => true,
                    p_default_value     => 'Y',
                    p_help_text         => p_stop_execution_help_text );
            end if;
        end if;
        --
        if l_component_type_id = c_comp_type_app_process then
            --
            store_condition_type_comp_prop (
                p_component_type_id => l_component_type_id,
                p_name_prefix       => null,
                p_help_text_type    => null,
                p_db_column_type    => 'PROCESS_WHEN_TYPE',
                p_view_column_type  => 'CONDITION_TYPE',
                p_db_column_expr1   => 'PROCESS_WHEN',
                p_db_column_expr2   => 'PROCESS_WHEN2' );
            --
        elsif l_component_type_id = c_comp_type_page_process then
            --
            store_component_prop (
                p_component_type_id   => l_component_type_id,
                p_property_name       => 'WHEN_BUTTON_PRESSED',
                p_db_column           => 'PROCESS_WHEN_BUTTON_ID',
                p_is_required         => false,
                p_is_common           => true,
                p_reference_on_delete => 'WEAK_NULL' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'EXECUTION_SCOPE',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => false,
                p_default_value               => 'Y',
                p_depending_on_comp_prop_name => 'PROCESS_REGION',
                p_depending_on_condition_type => 'NOT_NULL' );
            --
            store_condition_type_comp_prop (
                p_component_type_id           => l_component_type_id,
                p_name_prefix                 => null,
                p_help_text_type              => null,
                p_reference_scope             => 'DEPENDS',
                p_db_column_type              => 'PROCESS_WHEN_TYPE',
                p_view_column_type            => 'CONDITION_TYPE',
                p_db_column_expr1             => 'PROCESS_WHEN',
                p_db_column_expr2             => 'PROCESS_WHEN2',
                p_has_execute_condition       => true,
                p_execute_cond_dep_prop_name  => 'PROCESS_REGION' );
        elsif l_component_type_id != c_comp_type_workflow_activity then
            -- Automation and Task Def Actions

                store_condition_type_comp_prop (
                    p_component_type_id           => l_component_type_id,
                    p_name_prefix                 => 'ACTION',
                    p_help_text_type              => null,
                    p_db_column_type              => 'CONDITION_TYPE',
                    p_view_column_type            => 'CONDITION_TYPE',
                    p_db_column_expr1             => 'CONDITION_EXPR1',
                    p_db_column_expr2             => 'CONDITION_EXPR2',
                    p_has_execute_condition       => ( l_component_type_id = c_comp_type_automation_action ),
                    p_execute_condition_help_text => q'~Specify whether the condition is to be executed for each row or only once.~' );
        end if;
        --
        if l_component_type_id != c_comp_type_workflow_activity then
           store_standard_attr_comp_prop (
                p_component_type_id        => l_component_type_id,
                p_has_authorization_scheme => p_has_authorization_scheme,
                p_authorization_db_column  => case when p_has_authorization_scheme then 'SECURITY_SCHEME' end,
                p_has_build_option         => true,
                p_build_option_db_column   => p_build_option_db_column,
                p_has_comments             => true,
                p_comment_db_column        => p_comment_db_column,
                p_comment_import_api_param => p_comment_import_api_param );
        end if;
        --
    end store_process_type_comp_prop;
    --
    --
    procedure store_computation_comp_prop (
        p_component_type_id in number )
    is
    begin
        --
        store_component_prop (
            p_component_type_id   => p_component_type_id,
            p_property_name       => 'COMPUTATION_ITEM_NAME',
            p_db_column           => null,
            p_is_required         => true,
            p_is_common           => true,
            p_reference_on_delete => 'WEAK_CASCADE' );
        --
        store_component_prop (
            p_component_type_id => p_component_type_id,
            p_property_name     => 'EXECUTION_SEQUENCE',
            p_db_column         => 'COMPUTATION_SEQUENCE',
            p_is_required       => true,
            p_is_common         => false );
        --
        -- page and application computations have different default values
        -- for computation point and computation type
        if p_component_type_id = c_comp_type_page_computation then
            --
            store_component_prop (
                p_component_type_id => p_component_type_id,
                p_property_name     => 'COMPUTATION_POINT',
                p_db_column         => null,
                p_is_required       => true,
                p_is_common         => true,
                p_default_value     => 'BEFORE_BOX_BODY' );
            --
            store_component_prop (
                p_component_type_id => p_component_type_id,
                p_property_name     => 'COMPUTATION_TYPE',
                p_db_column         => null,
                p_view_column       => 'COMPUTATION_TYPE',
                p_is_required       => true,
                p_is_common         => true,
                p_default_value     => 'QUERY' );
        else
            --
            store_component_prop (
                p_component_type_id => p_component_type_id,
                p_property_name     => 'COMPUTATION_POINT',
                p_db_column         => null,
                p_is_required       => true,
                p_is_common         => true,
                p_default_value     => 'ON_NEW_INSTANCE' );
            --
            store_component_prop (
                p_component_type_id => p_component_type_id,
                p_property_name     => 'COMPUTATION_TYPE',
                p_db_column         => null,
                p_view_column       => 'COMPUTATION_TYPE',
                p_is_required       => true,
                p_is_common         => true,
                p_default_value     => 'STATIC' );
        end if;
        --
        store_component_prop (
            p_component_type_id           => p_component_type_id,
            p_property_name               => 'COMPUTATION_STATIC_VALUE',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => 'COMPUTATION_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'STATIC_ASSIGNMENT' );
        --
        store_component_prop (
            p_component_type_id           => p_component_type_id,
            p_property_name               => 'COMPUTATION_SQL_STATEMENT',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => 'COMPUTATION_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'QUERY' );
        --
        store_component_prop (
            p_component_type_id           => p_component_type_id,
            p_property_name               => 'COMPUTATION_SQL_COLON',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => 'COMPUTATION_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'QUERY_COLON' );
        --
        store_expression_comp_props (
            p_component_type_id => p_component_type_id,
            p_name_prefix       => 'COMPUTATION_',
            p_return_type       => 'VARCHAR2',
            p_type_property     => 'COMPUTATION_TYPE' );
        --
        store_func_body_comp_props (
            p_component_type_id => p_component_type_id,
            p_name_prefix       => 'COMPUTATION_',
            p_return_type       => 'VARCHAR2',
            p_type_property     => 'COMPUTATION_TYPE' );
        --
        store_component_prop (
            p_component_type_id           => p_component_type_id,
            p_property_name               => 'COMPUTATION_ITEM_VALUE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => 'COMPUTATION_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'ITEM_VALUE' );
        --
        store_component_prop (
            p_component_type_id           => p_component_type_id,
            p_property_name               => 'COMPUTATION_PREFERENCE_VALUE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => 'COMPUTATION_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'SET_ITEM_EQUAL_THIS_PREFERENCE' );
        --
        store_component_prop (
            p_component_type_id => p_component_type_id,
            p_property_name     => 'ERROR_MESSAGE',
            p_db_column         => 'COMPUTATION_ERROR_MESSAGE',
            p_is_required       => false,
            p_is_common         => false,
        p_help_text         => q'~<p>Enter an error message that displays if this computation fails.
    Use <code>#SQLERRM#</code> as a substitution string for the SQL error message resulting from a failed computation.</p>
    <p>Computations are designed to always succeed, and only fail due to unanticipated errors.
    It is not advisable to implement a computation that regularly fails and acts as a pseudo-validation utilizing this error message.~' );
        --
        store_standard_attr_comp_prop (
            p_component_type_id        => p_component_type_id,
            p_has_condition            => true,
            p_cond_type_db_column      => 'COMPUTE_WHEN_TYPE',
            p_cond_type_view_db_column => 'CONDITION_TYPE',
            p_cond_exp1_db_column      => 'COMPUTE_WHEN',
            p_cond_exp2_db_column      => 'COMPUTE_WHEN_TEXT',
            p_has_authorization_scheme => true,
            p_authorization_db_column  => 'SECURITY_SCHEME',
            p_has_build_option         => true,
            p_has_comments             => true,
            p_comment_db_column        => 'COMPUTATION_COMMENT' );
        --
    end store_computation_comp_prop;
    --
    procedure store_grid_comp_prop (
        p_component_type_id     in number,
        p_new_grid_db_column    in varchar2 default 'GRID_NEW_GRID',           -- Pass if component's DB column is not GRID_NEW_GRID
        p_new_row_db_column     in varchar2 default 'GRID_NEW_ROW',            -- Pass if component's DB column is not GRID_NEW_ROW
        p_row_css_db_column     in varchar2 default 'GRID_ROW_CSS_CLASSES',    -- Pass if component's DB column is not GRID_ROW_CSS_CLASSES
        p_column_db_column      in varchar2 default 'GRID_COLUMN',             -- Pass if component's DB column is not GRID_COLUMN
        p_new_column_db_column  in varchar2 default 'GRID_NEW_COLUMN',         -- Pass if component's DB column is not GRID_NEW_COLUMN
        p_column_span_db_column in varchar2 default 'GRID_COLUMN_SPAN',        -- Pass if component's DB column is not GRID_COLUMN_SPAN
        p_row_span_db_column    in varchar2 default 'GRID_ROW_SPAN',           -- Pass if component's DB column is not GRID_ROW_SPAN
        p_column_css_db_column  in varchar2 default 'GRID_COLUMN_CSS_CLASSES', -- Pass if component's DB column is not GRID_COLUMN_CSS_CLASSES
        p_column_attr_db_column in varchar2 default 'GRID_COLUMN_ATTRIBUTES',  -- Pass if component's DB column is not GRID_COLUMN_ATTRIBUTES
        p_has_label_column_span in boolean  default false,
        p_plugin_std_attributes in varchar2 default null )
    is
    begin
        --
        store_component_prop (
            p_component_type_id     => p_component_type_id,
            p_property_name         => 'GRID_NEW_GRID',
            p_db_column             => p_new_grid_db_column,
            p_is_required           => true,
            p_is_common             => true,
            p_default_value         => 'N',
            p_plugin_std_attributes => p_plugin_std_attributes );
        --
        store_component_prop (
            p_component_type_id     => p_component_type_id,
            p_property_name         => 'GRID_NEW_ROW',
            p_db_column             => p_new_row_db_column,
            p_is_required           => true,
            p_is_common             => true,
            p_default_value         => 'Y',
            p_plugin_std_attributes => p_plugin_std_attributes );
        --
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'GRID_ROW_CSS_CLASSES',
            p_db_column             => p_row_css_db_column,
            p_is_required           => false,
            p_is_common             => false,
            p_plugin_std_attributes => p_plugin_std_attributes );
        --
        store_component_prop (
            p_component_type_id     => p_component_type_id,
            p_property_name         => 'GRID_COLUMN',
            p_db_column             => p_column_db_column,
            p_is_required           => false,
            p_is_common             => true,
            p_null_text             => 'Automatic',
            p_plugin_std_attributes => p_plugin_std_attributes );
        --
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'GRID_NEW_COLUMN',
            p_db_column             => p_new_column_db_column,
            p_is_required           => true,
            p_is_common             => true,
            p_default_value         => 'Y',
            p_plugin_std_attributes => p_plugin_std_attributes );
        --
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'GRID_COLUMN_SPAN',
            p_db_column             => p_column_span_db_column,
            p_is_required           => false,
            p_is_common             => true,
            p_null_text             => 'Automatic',
            p_plugin_std_attributes => p_plugin_std_attributes );
        --
        if p_row_span_db_column is not null then
            store_component_prop (
                p_component_type_id     => l_component_type_id,
                p_property_name         => 'GRID_ROW_SPAN',
                p_db_column             => p_row_span_db_column,
                p_is_required           => false,
                p_is_common             => false,
                p_plugin_std_attributes => p_plugin_std_attributes );
        end if;
        --
        if p_has_label_column_span then
            store_component_prop (
                p_component_type_id     => l_component_type_id,
                p_property_name         => 'GRID_LABEL_COLUMN_SPAN',
                p_db_column             => null,
                p_is_required           => false,
                p_is_common             => false,
                p_null_text             => 'Page Template Default',
                p_plugin_std_attributes => p_plugin_std_attributes );
        end if;
        --
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'GRID_COLUMN_CSS_CLASSES',
            p_db_column             => p_column_css_db_column,
            p_is_required           => false,
            p_is_common             => false,
            p_plugin_std_attributes => p_plugin_std_attributes );
        --
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'GRID_COLUMN_ATTRIBUTES',
            p_db_column             => p_column_attr_db_column,
            p_is_required           => false,
            p_is_common             => false,
            p_plugin_std_attributes => p_plugin_std_attributes );

    end store_grid_comp_prop;
    --
    procedure store_report_attributes
    is
    begin
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'NUMBER_OF_ROWS_TYPE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'STATIC' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'NUMBER_OF_ROWS',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_default_value               => '15',
            p_depending_on_comp_prop_name => 'NUMBER_OF_ROWS_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'STATIC' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'NUMBER_OF_ROWS_ITEM',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'NUMBER_OF_ROWS_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'ITEM' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'TEMPLATE_TYPE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'THEME' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'REPORT_TEMPLATE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_default_value               => '#DEFAULT_REPORT_TEMPLATE#',
            p_depending_on_comp_prop_name => 'TEMPLATE_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'THEME' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'REPORT_PREDEFINED_TEMPLATE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_default_value               => '1',
            p_depending_on_comp_prop_name => 'TEMPLATE_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'PREDEFINED' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'TEMPLATE_OPTIONS',
            p_db_column                   => 'COMPONENT_TEMPLATE_OPTIONS',
            p_is_required                 => false,
            p_is_common                   => false,
            p_default_value               => '#DEFAULT##PRESET_TEMPLATE_OPTIONS#',
            p_parent_comp_prop_name       => 'REPORT_TEMPLATE',
            p_depending_on_comp_prop_name => 'REPORT_TEMPLATE',
            p_depending_on_condition_type => 'NOT_NULL' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'APPEARANCE_CSS_CLASSES',
            p_db_column                   => 'REGION_SUB_CSS_CLASSES',
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'TEMPLATE_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'THEME',
            p_help_text                   => q'~Enter additional space delimited CSS classes you want to add to your report. Your report template must include the <code>#COMPONENT_CSS_CLASSES#</code> substitution string.~',
            p_comment_text                => '$$$ we should only show this property if the template supports the #COMPONENT_CSS_CLASSES# placeholder' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'SHOW_NULL_VALUES_AS',
            p_db_column         => 'PLUG_QUERY_SHOW_NULLS_AS',
            p_is_required       => false,
            p_is_common         => false,
            p_default_value     => null );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'CLASSIC_REPORT_PAGINATION_TYPE',
            p_db_column         => null,
            p_is_required       => false,
            p_is_common         => true,
            p_default_value     => 'NEXT_PREVIOUS_LINKS',
            p_null_text         => 'No Pagination (Show All Rows)' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'PAGINATION_DISPLAY_POSITION',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'BOTTOM_RIGHT',
            p_depending_on_comp_prop_name => 'CLASSIC_REPORT_PAGINATION_TYPE',
            p_depending_on_condition_type => 'NOT_NULL' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'PARTIAL_PAGE_REFRESH',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => false,
            p_default_value     => 'Y' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'LAZY_LOADING',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'N',
            p_depending_on_comp_prop_name => 'PARTIAL_PAGE_REFRESH',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'Y' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'MAXIMUM_ROW_COUNT',
            p_db_column         => 'PLUG_QUERY_ROW_COUNT_MAX',
            p_is_required       => false,
            p_is_common         => false,
            p_comment_text      => q'~$$$ should be moved to region attributes, I think that is needed by all plug-ins~' );
        --
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'WHEN_NO_DATA_FOUND_MESSAGE',
            p_db_column             => 'PLUG_QUERY_NO_DATA_FOUND',
            p_import_api_parameter  => 'query_no_data_found',
            p_is_required           => false,
            p_is_common             => false );
        --
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'WHEN_MORE_DATA_FOUND_MESSAGE',
            p_db_column             => null,
            p_import_api_parameter  => 'query_more_data',
            p_is_required           => false,
            p_is_common             => false );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'REPORT_SUM_LABEL',
            p_db_column         => null,
            p_is_required       => false,
            p_is_common         => false );
        --
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'BREAK_COLUMNS',
            p_db_column             => null,
            p_import_api_parameter  => 'query_break_cols',
            p_is_required           => false,
            p_is_common             => false,
            p_null_text             => 'No Break' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'BREAK_COLUMN_STYLE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'DEFAULT_BREAK_FORMATTING',
            p_depending_on_comp_prop_name => 'BREAK_COLUMNS',
            p_depending_on_condition_type => 'NOT_NULL' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'BREAK_REPEAT_HEADING_FORMAT',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'BREAK_COLUMN_STYLE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'REPEAT_HEADINGS_ON_BREAK_1' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'BREAK_COLUMN_TEXT_FORMAT',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'BREAK_COLUMNS',
            p_depending_on_condition_type => 'NOT_NULL' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'BREAK_BEFORE_ROW',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'BREAK_COLUMNS',
            p_depending_on_condition_type => 'NOT_NULL' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'BREAK_GENERIC_COLUMN',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'BREAK_COLUMNS',
            p_depending_on_condition_type => 'NOT_NULL' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'BREAK_AFTER_ROW',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'BREAK_COLUMNS',
            p_depending_on_condition_type => 'NOT_NULL' );
        --
        store_component_prop (
            p_component_type_id    => l_component_type_id,
            p_property_name        => 'CUSTOM_ATTRIBUTES',
            p_db_column            => 'REPORT_ATTRIBUTES_SUBSTITUTION',
            p_is_required          => false,
            p_is_common            => false,
            p_help_text            => q'~The report template substitution string <code>#REPORT_ATTRIBUTES#</code> is replaced with this value.  The <code>#REPORT_ATTRIBUTES#</code> substitution string is only replaced in the before rows attribute of the report template.~' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'STRIP_HTML',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => false,
            p_default_value     => 'N' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'SORT_NULLS',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => false,
            p_default_value     => 'L' );
        --
        store_component_prop (
            p_component_type_id    => l_component_type_id,
            p_property_name        => 'HEADINGS_TYPE',
            p_db_column            => null,
            p_import_api_parameter => 'query_headings_type',
            p_is_required          => true,
            p_is_common            => false,
            p_default_value        => 'COLON_DELMITED_LIST' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'HEADING_TYPE_PLSQL_FUNCTION_BODY',
            p_db_column                   => null,
            p_import_api_parameter        => 'query_headings',
            p_is_required                 => true,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'HEADINGS_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'FUNCTION_BODY_RETURNING_COLON_DELIMITED_LIST' );
        --
        /* Add back when we have fixed header support for classic reports / tabular forms
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'FIXED_HEADER',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'PAGE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'FIXED_HEADER_MAX_HEIGHT',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'FIXED_HEADER',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'REGION' );
        */
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'CSV_EXPORT_ENABLED',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => true,
            p_default_value     => 'N' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'CSV_SEPARATOR',
            p_db_column                   => 'PLUG_QUERY_EXP_SEPARATOR',
            p_is_required                 => false,
            p_is_common                   => true,
            p_supports_substitution       => false,
            p_depending_on_comp_prop_name => 'CSV_EXPORT_ENABLED',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'Y' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'CSV_ENCLOSED_BY',
            p_db_column                   => 'PLUG_QUERY_EXP_ENCLOSED_BY',
            p_is_required                 => false,
            p_is_common                   => true,
            p_supports_substitution       => false,
            p_depending_on_comp_prop_name => 'CSV_EXPORT_ENABLED',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'Y' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'DOWNLOAD_LINK_TEXT',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'Download',
            p_depending_on_comp_prop_name => 'CSV_EXPORT_ENABLED',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'Y' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'DOWNLOAD_FILENAME',
            p_db_column                   => 'PLUG_QUERY_EXP_FILENAME',
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'CSV_EXPORT_ENABLED',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'Y',
            p_help_text                   => q'~Specify a name for the CSV download file. If no name is specified, the region name is used followed by the extension .csv.~' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'SUPPLEMENTAL_TEXT',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_default_value               => null );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'ENABLE_PRINTING',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => true,
            p_default_value     => 'N' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'EXTERNAL_PROCESSING_ENABLED',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => false,
            p_default_value     => 'N' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'EXTERNAL_PROCESSING_URL',
            p_db_column                   => null,
            p_view_column                 => 'URL',
            p_is_required                 => true,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'EXTERNAL_PROCESSING_ENABLED',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'Y' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'EXTERNAL_PROCESSING_LINK_TEXT',
            p_db_column                   => null,
            p_view_column                 => 'LINK_LABEL',
            p_is_required                 => true,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'EXTERNAL_PROCESSING_ENABLED',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'Y' );
        --
    end store_report_attributes;
    --
    procedure store_report_columns (
        p_type_property_name in varchar2 )
    is
    begin
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'COLUMN_NAME',
            p_db_column         => 'COLUMN_ALIAS',
            p_is_required       => true,
            p_is_common         => true,
            p_is_read_only      => true,
            p_check_uniqueness  => 'PARENT' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'QUERY_COLUMN_ID',
            p_db_column         => null,
            p_is_required       => false,
            p_is_common         => true );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'DERIVED_COLUMN',
            p_db_column         => null,
            p_is_required       => false,
            p_is_common         => true,
            p_default_value     => 'N' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => p_type_property_name,
            p_db_column         => null,
            p_view_column       => 'DISPLAY_AS',
            p_is_required       => true,
            p_is_common         => true,
            p_default_value     => 'PLAIN' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'COLUMN_HEADING',
            p_db_column                   => 'COLUMN_HEADING',
            p_is_required                 => false,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'NOT_IN_LIST',
            p_depending_on_expression     => 'HIDDEN_COLUMN,HIDDEN_FIELD',
            p_important_for_accessibility => true,
            p_accessibility_help_text     => q'~<p>Column headings play a crucial role in organizing and presenting tabular data in a clear and understandable format.</p>
<strong>Best Practices:</strong>
<ol>
<li><strong>Descriptive and Meaningful:</strong> Use descriptive and meaningful column headings that accurately represent the content of each column. Column headings should succinctly convey the information contained within the respective columns.</li>
<li><strong>Consistency:</strong> Maintain consistency in column headings throughout the table. Consistent column headings facilitate navigation and comprehension for users, especially when interacting with complex tables containing multiple columns.</li>
</ol>~' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'HEADING_ALIGNMENT',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'LEFT',
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'NOT_IN_LIST',
            p_depending_on_expression     => 'HIDDEN_COLUMN,HIDDEN_FIELD' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'DISPLAY_SEQUENCE',
            p_db_column         => 'COLUMN_DISPLAY_SEQUENCE',
            p_is_required       => true,
            p_is_common         => true,
            p_comment_text      => q'~$$$ make this read only for now~' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'COLUMN_ALIGNMENT',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'LEFT',
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'NOT_IN_LIST',
            p_depending_on_expression     => 'HIDDEN_COLUMN,HIDDEN_FIELD' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'USE_AS_ROW_HEADER',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'N',
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'NOT_IN_LIST',
            p_depending_on_expression     => 'HIDDEN_COLUMN,HIDDEN_FIELD,PCT_GRAPH,ROW_SELECTOR,RICH_TEXT',
            p_important_for_accessibility => true,
            p_accessibility_help_text     => q'~<p>It's crucial to define row headers for tabular report data, such as in Classic or Interactive Reports, to enhance usability for users of assistive technologies like screen readers.</p>
<strong>Best Practices:</strong>
<ol>
<li><strong>Impact on Usability for Screen Reader Users:</strong>
<ul>
<li>When a screen reader user navigates through different rows of the report, the column or columns designated as identifying the row will be announced as the row header upon moving to the new row.</li>
<li>Failure to define a row header column makes it significantly more challenging for screen reader users to orient themselves while navigating the report.</li>
</ul>
</li>
<li><strong>Enhancing Orientation and Navigation:</strong>
<ul>
<li>Defining this attribute improves the navigation experience for screen reader users by providing clear orientation points within the tabular data.</li>
<li>This ensures smoother navigation and comprehension of the report's content for users relying on assistive technologies.</li>
</ul>
</li>
<li><strong>Recommendations:</strong>
<ul>
<li>Define, one or more columns that identify the row, that will function as the row header for users of Assistive Technology.</li>
<li>This will be announced every time a user moves between rows, so the entire row header should not be too long to become overly verbose.</li>
</ul>
</li>
</ol>~' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'IMAGE_DESCRIPTION',
            p_db_column                   => 'IMAGE_DESCRIPTION',
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'IMAGE',
            p_important_for_accessibility => true,
            p_accessibility_help_text     => q'~<p>The Image Description property serves as a vital component in improving accessibility for users with visual impairments. It enables the provision of text or a reference to a column containing text to describe the image.  It is important to define descriptions for images that can be displayed as part of reports, such as in Classic or Interactive Reports, to improve usability for users of assistive technologies, such as screen readers.</p>
<strong>Best practices:</strong>
<ol>
<li>Informative Images:
Identify images that contribute to the comprehension or context of the webpage. If removing the image would diminish the page's meaning or significance, it requires the image description. If removing the image would not diminish the page's meaning (for example if it was just decorative), then leave the image description blank and it will be ignored by screen readers.</li>
<li>Purposeful Description:
<ul>
<li>Alt Text should succinctly convey the significance of the image in relation to the surrounding content.</li>
<li>Focus on describing the "why" of the image rather than stating what the image is.</li>
<li>Keep Alt Text clear, concise, and descriptive.</li>
</ul>
</li>
<strong>Example:</strong>
<p>"Stylish pink tote bag with faux leather straps, perfect for adding a pop of color to your outfit".</p>
<li>Avoid Redundancy:
Omit phrases like "image of" as screen readers automatically announce an image as such.</li>
</ol>~' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'FORMAT_MASK',
            p_db_column                   => 'COLUMN_FORMAT',
            p_is_required                 => false,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'PLAIN,LINK,DISPLAY_AND_SAVE,TEXT,DATE_PICKER',
            p_help_text                   => q'~<p>Enter the format mask to apply to this column.
    You can type in the format mask or pick from the predefined list, based on a sample representation of how that format mask is displayed.</p>
    <p>It is important that number format masks are only applied to columns that contain numbers and date format masks are only applied to columns that contain dates.
    Otherwise, a runtime error is raised when any record contains a value that can not be converted using the specified format mask.</p>~' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'DATE_PICKER_FORMAT_MASK',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'DATE_POPUP' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'PCT_GRAPH_BACKGROUD_COLOR',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'PCT_GRAPH' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'PCT_GRAPH_FOREGROUND_COLOR',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'PCT_GRAPH' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'PCT_GRAPH_BAR_WIDTH',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'PCT_GRAPH' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'ELEMENT_WIDTH',
            p_db_column                   => 'COLUMN_WIDTH',
            p_is_required                 => false,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'TEXT,TEXTAREA,DATE_POPUP,DATE_PICKER,POPUP,POPUPKEY' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'ELEMENT_HEIGHT',
            p_db_column                   => 'COLUMN_HEIGHT',
            p_is_required                 => false,
            p_is_common                   => true,
            p_default_value               => '3',
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'TEXTAREA' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'CONTENT_DISPOSITION',
            p_db_column                   => 'BLOB_CONTENT_DISPOSITION',
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'attachment',
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'DOWNLOAD' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'FIELD_TEMPLATE',
            p_db_column                   => 'COLUMN_FIELD_TEMPLATE',
            p_is_required                 => false,
            p_is_common                   => false,
            p_reference_on_delete         => 'WEAK_NULL',
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'DISPLAY_AND_SAVE,TEXT,TEXTAREA,DATE_POPUP,DATE_PICKER,HIDDEN_FIELD,HIDDEN_PROTECTED,SELECT_LIST,RADIOGROUP,POPUP,POPUPKEY,SIMPLE_CHECKBOX',
            p_help_text                   => q'~Choose the field template to be used when rendering the tabular form field. When no template is selected, the field is rendered using the default accessibility compliant label.~' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'RICH_TEXT_FORMAT',
            p_db_column                   => 'ATTRIBUTE_01',
            p_is_required                 => true,
            p_is_common                   => true,
            p_default_value               => 'MARKDOWN',
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'RICH_TEXT' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'BLOB_TABLE_OWNER',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'IMAGE,DOWNLOAD',
            p_comment_text                => q'~$$$ Note: A owner column is not yet supported by the format mask~' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'BLOB_TABLE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'IMAGE,DOWNLOAD',
            p_parent_comp_prop_name       => 'BLOB_TABLE_OWNER' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'DOWNLOAD_TEXT',
            p_db_column                   => 'BLOB_DOWNLOAD_TEXT',
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'DOWNLOAD',
            p_parent_comp_prop_name       => 'BLOB_TABLE' );
        --
        -- Download Text is part of Display Group "Appearance" but also has a dependency
        -- to Component-Prop 'BLOB_TABLE', so we have to move it up after creation
        resequence_comp_prop (
            p_component_type_id => l_component_type_id,
            p_name              => 'DOWNLOAD_TEXT',
            p_previous_name     => 'FIELD_TEMPLATE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'BLOB_CONTENT_COLUMN',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'IMAGE,DOWNLOAD',
            p_parent_comp_prop_name       => 'BLOB_TABLE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'BLOB_PK_COLUMN1',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'IMAGE,DOWNLOAD',
            p_parent_comp_prop_name       => 'BLOB_TABLE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'BLOB_PK_COLUMN2',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'IMAGE,DOWNLOAD',
            p_parent_comp_prop_name       => 'BLOB_TABLE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'BLOB_MIME_TYPE_COLUMN',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'IMAGE,DOWNLOAD',
            p_parent_comp_prop_name       => 'BLOB_TABLE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'BLOB_FILENAME_COLUMN',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'IMAGE,DOWNLOAD',
            p_parent_comp_prop_name       => 'BLOB_TABLE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'BLOB_LAST_UPDATED_COLUMN',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'IMAGE,DOWNLOAD',
            p_parent_comp_prop_name       => 'BLOB_TABLE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'BLOB_CHARSET_COLUMN',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'DOWNLOAD',
            p_parent_comp_prop_name       => 'BLOB_TABLE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'LOV_TYPE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'PLAIN_LOV,SELECT_LIST,RADIOGROUP,POPUP,POPUPKEY',
            p_comment_text                => q'~$$$ for POPUP and POPUPKEY we should NOT show STATIC~' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'NAMED_LOV',
            p_db_column                   => 'NAMED_LOV',
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => 'LOV_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'SHARED' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'LOV_SQL',
            p_db_column                   => 'INLINE_LOV',
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => 'LOV_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'SQL_QUERY' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'LOV_STATIC_VALUES',
            p_db_column                   => 'INLINE_LOV',
            p_is_required                 => true,
            p_is_common                   => true,
            p_default_value               => 'STATIC:Display1;Return1,Display2;Return2',
            p_depending_on_comp_prop_name => 'LOV_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'STATIC' );
        --
        store_func_body_comp_props (
            p_component_type_id   => l_component_type_id,
            p_name_prefix         => 'LOV_',
            p_return_type         => 'SQL',
            p_db_column_func_body => 'INLINE_LOV',
            p_type_property       => 'LOV_TYPE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'SIMPLE_CHECKBOX_VALUES',
            p_db_column                   => 'INLINE_LOV',
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'SIMPLE_CHECKBOX' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'DISPLAY_EXTRA_VALUES',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'Y',
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'SELECT_LIST,RADIOGROUP' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'DISPLAY_NULL_VALUE',
            p_db_column                   => 'LOV_SHOW_NULLS',
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'Y',
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'SELECT_LIST,RADIOGROUP,POPUP,POPUPKEY' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'NULL_DISPLAY_VALUE',
            p_db_column                   => 'LOV_NULL_TEXT',
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'DISPLAY_NULL_VALUE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'Y' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'NULL_RETURN_VALUE',
            p_db_column                   => 'LOV_NULL_VALUE',
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'DISPLAY_NULL_VALUE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'Y' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'LINK_TARGET',
            p_db_column                   => 'COLUMN_LINK',
            p_is_required                 => false,
            p_is_common                   => true,
            p_reference_scope             => 'ROW',
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'LINK,PCT_GRAPH',
            p_comment_text                => q'~$$$ IMAGE raises the runtime error "Numeric value error" if a value is set~' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'LINK_TEXT',
            p_db_column                   => 'COLUMN_LINKTEXT',
            p_is_required                 => true,
            p_is_common                   => true,
            p_default_value               => '##CURRENT_COLUMN_NAME##',
            p_reference_scope             => 'ROW',
            p_depending_on_comp_prop_name => 'LINK_TARGET',
            p_depending_on_condition_type => 'NOT_NULL',
            p_important_for_accessibility => true,
            p_accessibility_help_text     => q'~<p>Accessible link text is essential for users of assistive technologies, such as screen readers, to navigate and understand content effectively.</p>
<strong>Best Practices:</strong>
<ol>
<li><strong>Conciseness:</strong> Keep link text concise while ensuring it conveys the necessary information. Aim for a balance between brevity and descriptiveness.</li>
<li><strong>Accessibility Attributes:</strong> Ensure report column row headers are defined. Defining this attribute improves the navigation experience for screen reader users by providing clear orientation points within the tabular data.</li>
<li><strong>Repeated links:</strong> In a typical report, each row often features a repeating link, such as an edit option. In such cases, the most effective link text would be 'Edit', and paired with a clear row header.</li>
</ol>
<p>By following these best practices for link text, you can create a more inclusive and accessible browsing experience for all users, including those with disabilities.</p>~' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'LINK_ATTRIBUTES',
            p_db_column                   => 'COLUMN_LINK_ATTR',
            p_is_required                 => false,
            p_is_common                   => false,
            p_reference_scope             => 'ROW',
            p_depending_on_comp_prop_name => 'LINK_TARGET',
            p_depending_on_condition_type => 'NOT_NULL' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'RADIO_COLUMNS',
            p_db_column                   => 'ATTRIBUTE_01',
            p_is_required                 => false,
            p_is_common                   => false,
            p_default_value               => '1',
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'RADIOGROUP' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'HTML_EXPRESSION',
            p_db_column                   => 'COLUMN_HTML_EXPRESSION',
            p_is_required                 => false,
            p_is_common                   => false,
            p_supports_template_directive => 'S',
            p_reference_scope             => 'ROW',
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'PLAIN' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'FORMATTING_CSS_CLASSES',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'PLAIN,LINK,PCT_GRAPH',
            p_comment_text                => q'~$$$ IMAGE or DOWNLOAD do raise the runtime error "Numeric value error" if a value is set~' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'CSS_STYLE',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'PLAIN,LINK,PCT_GRAPH',
            p_comment_text                => q'~$$$ IMAGE or DOWNLOAD do raise the runtime error "Numeric value error" if a value is set~' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'HIGHLIGHT_WORDS',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'PLAIN' );
        --
        if p_type_property_name = 'TAB_FORM_COLUMN_TYPE' then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'COLUMN_DEFAULT_TYPE',
                p_db_column                   => null,
                p_is_required                 => false,
                p_is_common                   => false,
                p_depending_on_comp_prop_name => p_type_property_name,
                p_depending_on_condition_type => 'IN_LIST',
                p_depending_on_expression     => 'DISPLAY_AND_SAVE,DATE_POPUP,DATE_PICKER,TEXT,TEXTAREA,SELECT_LIST,RADIOGROUP,HIDDEN_FIELD,POPUP,POPUPKEY,SIMPLE_CHECKBOX,ROW_SELECTOR',
                p_comment_text                => '$$$ Property should only be visible if the current column name NOT matches with the primary key column(s) in the tabular form DML process' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'DEFAULT_VALUE_ITEM',
                p_db_column                   => 'COLUMN_DEFAULT',
                p_is_required                 => true,
                p_is_common                   => false,
                p_depending_on_comp_prop_name => 'COLUMN_DEFAULT_TYPE',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'ITEM' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'DEFAULT_VALUE_PLSQL_EXPRESSION_VARCHAR2',
                p_db_column                   => 'COLUMN_DEFAULT',
                p_is_required                 => true,
                p_is_common                   => false,
                p_depending_on_comp_prop_name => 'COLUMN_DEFAULT_TYPE',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'FUNCTION' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'PK_SOURCE_TYPE',
                p_db_column                   => null,
                p_is_required                 => false,
                p_is_common                   => false,
                p_depending_on_comp_prop_name => p_type_property_name,
                p_depending_on_condition_type => 'NOT_EQUALS',
                p_depending_on_expression     => 'ROW_SELECTOR',
                p_comment_text                => '$$$ Property should only be visible if the current column name matches with the primary key column(s) in the tabular form DML process' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'PK_SOURCE_SEQUENCE_NAME',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => false,
                p_depending_on_comp_prop_name => 'PK_SOURCE_TYPE',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'S' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'PK_SOURCE_PLSQL_EXPRESSION',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => false,
                p_depending_on_comp_prop_name => 'PK_SOURCE_TYPE',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'F' );
            --
        end if;
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'COLUMN_SORT_SEQUENCE',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'NOT_IN_LIST',
            p_depending_on_expression     => 'ROW_SELECTOR' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'COLUMN_SORT_DIRECTION',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_null_text                   => 'Ascending',
            p_depending_on_comp_prop_name => 'COLUMN_SORT_SEQUENCE',
            p_depending_on_condition_type => 'NOT_NULL' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'DISABLE_SORT_COLUMN',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'N',
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'NOT_IN_LIST',
            p_depending_on_expression     => 'HIDDEN_COLUMN,ROW_SELECTOR' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'INCLUDE_IN_EXPORT_PRINT',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'Y',
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'NOT_IN_LIST',
            p_depending_on_expression     => 'HIDDEN_COLUMN,ROW_SELECTOR',
            p_comment_text                => q'~$$$ should we hide it for LINK as well? Is Hidden supported?~' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'PRINT_COLUMN_WIDTH',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'INCLUDE_IN_EXPORT_PRINT',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'Y' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'ADVANCED_CSS_CLASSES',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'TEXT,TEXTAREA,DATE_POPUP,DATE_PICKER,HIDDEN_FIELD,HIDDEN_PROTECTED,SELECT_LIST,RADIOGROUP,POPUP,POPUPKEY,SIMPLE_CHECKBOX' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'CUSTOM_ATTRIBUTES',
            p_db_column                   => 'CATTRIBUTES',
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'TEXT,TEXTAREA,DATE_POPUP,DATE_PICKER,HIDDEN_FIELD,HIDDEN_PROTECTED,SELECT_LIST,RADIOGROUP,POPUP,POPUPKEY,SIMPLE_CHECKBOX' );
        --
    /*
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'ELEMENT_OPTION_HTML_ATTRIBUTES', -- $$$ is this property really used by reports3.plb, can't find it in the rendering code
            p_db_column                   => 'CATTRIBUTES_ELEMENT',
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => p_type_property_name, -- $$$ would it be better to restricted to specific display types?
            p_depending_on_condition_type => 'NOT_EQUALS',
            p_depending_on_expression     => 'HIDDEN_COLUMN' );
    */
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'COLUMN_WIDTH',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'NOT_IN_LIST',
            p_depending_on_expression     => 'HIDDEN_COLUMN,HIDDEN_FIELD' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'COMPUTE_SUM',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'N',
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'NOT_IN_LIST',
            p_depending_on_expression     => 'HIDDEN_COLUMN,HIDDEN_FIELD,ROW_SELECTOR,DATE_POPUP,DATE_PICKER,RICH_TEXT' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'UI_DEFAULT_REF_TABLE_OWNER',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'DERIVED_COLUMN',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'N' );
        --
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'UI_DEFAULT_REF_TABLE',
            p_db_column             => null,
            p_is_required           => false,
            p_is_common             => false,
            p_parent_comp_prop_name => 'UI_DEFAULT_REF_TABLE_OWNER' );
        --
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'UI_DEFAULT_REF_COLUMN',
            p_db_column             => null,
            p_is_required           => false,
            p_is_common             => false,
            p_parent_comp_prop_name => 'UI_DEFAULT_REF_TABLE' );
        --
        store_condition_type_comp_prop (
            p_component_type_id => l_component_type_id,
            p_name_prefix       => null,
            p_help_text_type    => null,
            p_db_column_type    => 'DISPLAY_WHEN_COND_TYPE',
            p_view_column_type  => 'CONDITION_TYPE',
            p_db_column_expr1   => 'DISPLAY_WHEN_CONDITION',
            p_db_column_expr2   => 'DISPLAY_WHEN_CONDITION2' );
        --
        store_component_prop (
            p_component_type_id   => l_component_type_id,
            p_property_name       => 'AUTHORIZATION_SCHEME',
            p_db_column           => 'REPORT_COLUMN_REQUIRED_ROLE',
            p_is_required         => false,
            p_is_common           => false,
            p_reference_on_delete => 'WEAK_NULL' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'ESCAPE_SPECIAL_CHARACTERS',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'Y',
            p_depending_on_comp_prop_name => p_type_property_name,
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'PLAIN,LINK,HIDDEN_COLUMN,PLAIN_LOV' );
        --
        store_standard_attr_comp_prop (
            p_component_type_id => l_component_type_id,
            p_has_build_option  => true,
            p_has_comments      => true,
            p_comment_db_column => 'COLUMN_COMMENT' );
    end store_report_columns;
    --
    procedure store_comp_props_doc_src(
        p_component_type_id    in number )
    is
    begin
        store_component_prop (
            p_component_type_id     => p_component_type_id,
            p_property_name         => 'NAME',
            p_db_column             => null,
            p_is_required           => true,
            p_is_common             => true,
            p_supports_substitution => false,
            p_check_uniqueness      => 'APPLICATION',
            p_default_value         => '#PLEASE_CHANGE#',
            p_help_text             => case p_component_type_id
                                           when c_comp_type_dualityview_src
                                           then q'~Enter a name of the Duality View Source.~'
                                           when c_comp_type_jsondocument_src
                                           then q'~Enter a name of the JSON Source.~'
                                       end );
        --
        store_component_prop (
            p_component_type_id     => p_component_type_id,
            p_property_name         => 'STATIC_ID',
            p_db_column             => null,
            p_is_required           => true,
            p_is_common             => true,
            p_supports_substitution => false,
            p_help_text             => case p_component_type_id
                                           when c_comp_type_dualityview_src
                                           then q'~Use the Static ID to reference the Duality View Source in API Calls.~'
                                           when c_comp_type_jsondocument_src
                                           then q'~Use the Static ID to reference the JSON Source in API Calls.~'
                                       end );
        --
        if p_component_type_id = c_comp_type_jsondocument_src then
            store_component_prop (
                p_component_type_id => p_component_type_id,
                p_property_name     => 'JSON_COLLECTION_TYPE',
                p_db_column         => null,
                p_is_required       => true,
                p_is_common         => false );
        end if;
        --
        store_component_prop (
            p_component_type_id => p_component_type_id,
            p_property_name     => 'DOCUMENT_SOURCE_LOCATION',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => false );
        --
        store_component_prop (
            p_component_type_id => p_component_type_id,
            p_property_name     => 'SOURCE_QUERY_OWNER',
            p_db_column         => 'OBJECT_OWNER',
            p_is_required       => false,
            p_is_common         => false,
            p_help_text         => 'Select the schema that owns the data source.' );
        --
        store_component_prop (
            p_component_type_id => p_component_type_id,
            p_property_name     => 'SOURCE_QUERY_TABLE',
            p_db_column         => 'OBJECT_NAME',
            p_is_required       => true,
            p_is_common         => false,
            p_help_text         => case p_component_type_id
                                        when c_comp_type_dualityview_src
                                        then q'~Enter the case-sensitive duality view name. You can type in the name or pick from the list.~'
                                        when c_comp_type_jsondocument_src
                                        then q'~Enter the case-sensitive table name. You can type in the name or pick from the list.~'
                                   end );
        --
        store_component_prop (
            p_component_type_id => p_component_type_id,
            p_property_name     => 'SOURCE_QUERY_WHERE',
            p_db_column         => 'WHERE_CLAUSE',
            p_is_required       => false,
            p_is_common         => false );
        --
        store_component_prop (
            p_component_type_id           => p_component_type_id,
            p_property_name               => 'REMOTE_CACHING',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_null_text                   => 'Disabled',
            p_null_help_text              => q'~Result is not cached. Data is fetched from the remote database for each request.~',
            p_depending_on_comp_prop_name => 'DOCUMENT_SOURCE_LOCATION',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'REMOTE' );
        --
        store_component_prop (
            p_component_type_id           => p_component_type_id,
            p_property_name               => 'REMOTE_INVALIDATE_WHEN',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'REMOTE_CACHING',
            p_depending_on_condition_type => 'NOT_NULL' );
        --
        store_component_prop (
            p_component_type_id           => p_component_type_id,
            p_property_name               => case p_component_type_id
                                                 when c_comp_type_dualityview_src  then 'DUALITY_VIEW_SRC'
                                                 when c_comp_type_jsondocument_src then 'JSON_COLLECTION_SRC'
                                             end || '_SUBSCRIBED_FROM',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false );
        --
        store_component_prop (
            p_component_type_id           => p_component_type_id,
            p_property_name               => 'VERSION_SCN',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false );
        --
        store_standard_attr_comp_prop (
            p_component_type_id           => p_component_type_id,
            p_has_build_option            => false,
            p_has_comments                => true,
            p_comment_db_column           => 'SOURCE_COMMENT' );
    end store_comp_props_doc_src;
    --
    --
    procedure store_src_comp_props_objsrc(
        p_prefix                      in varchar2,
        p_source_prop_name            in varchar2,
        p_location_property_name      in varchar2,
        p_has_order_by                in boolean,
        p_has_order_by_item           in boolean,
        p_db_column_source            in varchar2,
        p_db_column_source_lang       in varchar2 )
    is
    begin
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => p_prefix || 'SOURCE_POST_PROCESSING',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_default_value               => null,
            p_depending_on_comp_prop_name => p_source_prop_name,
            p_depending_on_condition_type => 'NOT_NULL',
            p_parent_comp_prop_name       => p_source_prop_name );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => p_prefix || 'POST_PROC_WHERE',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => p_prefix || 'SOURCE_POST_PROCESSING',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'WHERE_ORDER_BY_CLAUSE',
            p_parent_comp_prop_name       => p_source_prop_name );
        --
        if p_has_order_by and not p_has_order_by_item then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => p_prefix || 'POST_PROC_ORDER_BY',
                p_db_column                   => null,
                p_is_required                 => false,
                p_is_common                   => false,
                p_depending_on_comp_prop_name => p_prefix || 'SOURCE_POST_PROCESSING',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'WHERE_ORDER_BY_CLAUSE',
                p_parent_comp_prop_name       => p_source_prop_name );
        end if;
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => p_prefix || 'POST_PROC_SQL_QUERY',
            p_db_column                   => p_db_column_source,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => p_prefix || 'SOURCE_POST_PROCESSING',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'SQL',
            p_parent_comp_prop_name       => p_source_prop_name );
        --
        store_func_body_comp_props (
            p_component_type_id     => l_component_type_id,
            p_name_prefix           => p_prefix || 'POST_PROC_',
            p_db_column_lang        => p_db_column_source_lang,
            p_db_column_func_body   => p_db_column_source,
            p_return_type           => 'SQL',
            p_type_property         => p_prefix || 'SOURCE_POST_PROCESSING',
            p_type_value            => 'FUNC_BODY_RETURNING_SQL',
            p_parent_comp_prop_name => p_source_prop_name );
        --
        if p_has_order_by_item then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => p_prefix || 'POST_PROC_ORDER_BY_TYPE',
                p_db_column                   => null,
                p_is_required                 => false,
                p_is_common                   => true,
                p_null_text                   => 'None',
                p_depending_on_comp_prop_name => p_prefix || 'SOURCE_POST_PROCESSING',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'WHERE_ORDER_BY_CLAUSE' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => p_prefix || 'POST_PROC_ORDER_BY',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => false,
                p_depending_on_comp_prop_name => p_prefix || 'POST_PROC_ORDER_BY_TYPE',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'STATIC',
                p_parent_comp_prop_name       => 'SOURCE_WEB_SRC_QUERY' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => p_prefix || 'POST_PROC_ORDER_BY_ITEM',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => false,
                p_depending_on_comp_prop_name => p_prefix || 'POST_PROC_ORDER_BY_TYPE',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'ITEM',
                p_parent_comp_prop_name       => p_source_prop_name );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => p_prefix || 'POST_PROC_ORDER_BY_ITEM2',
                p_db_column                   => null,
                p_is_required                 => false,
                p_is_common                   => false,
                p_depending_on_comp_prop_name => p_prefix || 'SOURCE_POST_PROCESSING',
                p_depending_on_condition_type => 'IN_LIST',
                p_depending_on_expression     => 'SQL,FUNC_BODY_RETURNING_SQL',
                p_parent_comp_prop_name       => p_prefix || 'POST_PROC_SQL_QUERY' );
        end if;
    end store_src_comp_props_objsrc;
    --
    --
    procedure store_comp_props_array_col(
        p_component_type_id    in number )
    is
    begin
        store_component_prop (
            p_component_type_id => p_component_type_id,
            p_property_name     => 'NAME',
            p_db_column         => null,
            p_check_uniqueness  => 'PARENT',
            p_is_required       => true,
            p_is_common         => true,
            p_default_value     => '#PLEASE_CHANGE#' );
        --
        store_component_prop (
            p_component_type_id => p_component_type_id,
            p_property_name     => 'LABEL',
            p_db_column         => 'DISPLAY_NAME',
            p_is_required       => true,
            p_is_common         => true );
    end store_comp_props_array_col;
    --
    --
    procedure store_comp_props_web_src_param(
        p_for_operation_params in boolean )
    is
    begin
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'NAME',
            p_db_column                    => null,
            p_check_uniqueness             => 'PARENT',
            p_is_required                  => true,
            p_is_common                    => true,
            p_default_value                => '#PLEASE_CHANGE#' );
        --
        if p_for_operation_params then
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'DATABASE_OPERATION',
                p_db_column         => null,
                p_is_required       => true,
                p_is_read_only      => true,
                p_is_common         => true );
        end if;
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'WEB_SRC_PARAM_TYPE',
            p_db_column                    => null,
            p_is_required                  => true,
            p_is_common                    => true,
            p_default_value                => 'HEADER' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'WEB_SRC_PARAM_DATA_TYPE',
            p_db_column                    => null,
            p_is_required                  => false,
            p_is_common                    => true,
            p_default_value                => 'VARCHAR2',
            p_depending_on_comp_prop_name  => 'WEB_SRC_PARAM_TYPE',
            p_depending_on_condition_type  => 'IN_LIST',
            p_depending_on_expression      => 'HEADER,COOKIE,URL_PATTERN,QUERY_STRING,BODY' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'FORMAT_MASK',
            p_db_column                   => 'OUTPUT_FORMAT_MASK',
            p_is_required                 => false,
            p_is_common                   => false,
            p_default_value               => null,
            p_depending_on_comp_prop_name => 'WEB_SRC_PARAM_DATA_TYPE',
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'BOOLEAN,NUMBER,DATE,TIMESTAMP,TIMESTAMP WITH TIME ZONE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'VALIDATION_REGEXP',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_default_value               => null,
            p_depending_on_comp_prop_name => 'WEB_SRC_PARAM_DATA_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'VARCHAR2' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'PARAM_DIRECTION',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'IS_STATIC',
            p_db_column                    => null,
            p_is_required                  => true,
            p_is_common                    => true,
            p_default_value                => 'N',
            p_depending_on_comp_prop_name  => 'PARAM_DIRECTION',
            p_depending_on_condition_type  => 'NOT_EQUALS',
            p_depending_on_expression      => 'OUT' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'PARAMETER_VALUE',
            p_db_column                    => null,
            p_is_required                  => false, -- $$$ should be required depending on IS_STATIC = Y
            p_is_common                    => true );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'PARAMETER_REQUIRED',
            p_db_column                    => null,
            p_is_required                  => true,
            p_is_common                    => true,
            p_default_value                => 'N',
            p_depending_on_comp_prop_name  => 'IS_STATIC',
            p_depending_on_condition_type  => 'EQUALS',
            p_depending_on_expression      => 'N' );
        --
        if l_component_type_id = c_comp_type_web_src_mod_param then
            -- $$$ todo only one parameter can have USE_FOR_ROW_SEARCH = Y
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'USE_FOR_ROW_SEARCH',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => true,
                p_default_value               => 'N',
                p_depending_on_comp_prop_name => 'PARAM_DIRECTION',
                p_depending_on_condition_type => 'NOT_EQUALS',
                p_depending_on_expression     => 'OUT' );
        end if;
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'OMIT_WHEN_NULL',
            p_db_column                    => null,
            p_is_required                  => true,
            p_is_common                    => true,
            p_default_value                => 'N',
            p_depending_on_comp_prop_name  => 'WEB_SRC_PARAM_TYPE',
            p_depending_on_condition_type  => 'IN_LIST',
            p_depending_on_expression      => 'HEADER,COOKIE,QUERY_STRING' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'MULTIPLE_VALUES',
            p_db_column                    => null,
            p_is_required                  => false,
            p_is_common                    => false,
            p_default_value                => 'N',
            p_depending_on_comp_prop_name  => 'WEB_SRC_PARAM_TYPE',
            p_depending_on_condition_type  => 'EQUALS',
            p_depending_on_expression      => 'QUERY_STRING',
            p_depending_on_comp_prop_name2 => 'OMIT_WHEN_NULL',
            p_depending_on_condition_type2 => 'EQUALS',
            p_depending_on_expression2     => 'Y' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'MULTI_VALUE_SEPARATOR',
            p_db_column                    => 'ARRAY_SEPARATOR_CHAR',
            p_is_required                  => false,
            p_is_common                    => false,
            p_supports_substitution        => false,
            p_depending_on_comp_prop_name  => 'MULTIPLE_VALUES',
            p_depending_on_condition_type  => 'EQUALS',
            p_depending_on_expression      => 'Y',
            p_help_text                    => q'~
    Enter the separator character to use for splitting up the value. If empty, values will be separated by
    the colon character (":").~' );
        --
        store_standard_attr_comp_prop (
            p_component_type_id => l_component_type_id,
            p_comment_db_column => 'PARAM_COMMENT' );
    end store_comp_props_web_src_param;
    --
    procedure store_comp_props_ws_param_comp
    is
    begin
        store_component_prop (
            p_component_type_id   => l_component_type_id,
            p_property_name       => 'WEB_SRC_PARAM',
            p_db_column           => null,
            p_is_required         => true,
            p_is_common           => true,
            p_is_read_only        => true,
            p_reference_on_delete => 'DB_CASCADE' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'HTTP_METHOD',
            p_db_column         => 'OPERATION',
            p_is_required       => false,
            p_is_read_only      => true,
            p_null_text         => 'All',
            p_is_common         => true );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'DATABASE_OPERATION',
            p_db_column         => 'DATABASE_OPERATION',
            p_is_required       => false,
            p_null_text         => 'All',
            p_is_read_only      => true,
            p_is_common         => true );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'API_DIRECTION',
            p_is_required       => true,
            p_is_common         => true,
            p_db_column         => null,
            p_is_read_only      => true );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'WEB_SRC_PARAM_DATA_TYPE',
            p_is_required                  => false,
            p_is_common                    => true,
            p_db_column                    => null,
            p_is_read_only                 => true,
            p_depending_on_comp_prop_name  => 'API_DIRECTION',
            p_depending_on_condition_type  => 'IN_LIST',
            p_depending_on_expression      => 'IN,IN_OUT' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'IGNORE_OUTPUT',
            p_db_column                    => null,
            p_is_required                  => true,
            p_default_value                => 'Y',
            p_is_common                    => true,
            p_depending_on_comp_prop_name  => 'API_DIRECTION',
            p_depending_on_condition_type  => 'IN_LIST',
            p_depending_on_expression      => 'IN_OUT,OUT' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'HIDDEN_REQUIRED_VALUE_TYPE',
            p_is_required                  => false,
            p_is_common                    => true,
            p_default_value                => 'BASIC' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'HIDDEN_VALUE_TYPE',
            p_db_column                    => 'VALUE_TYPE',
            p_is_required                  => false,
            p_is_common                    => true,
            p_default_value                => 'ITEM' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'OUT_VALUE_TYPE',
            p_db_column                    => null,
            p_is_required                  => true,
            p_is_common                    => true,
            p_default_value                => 'ITEM',
            p_depending_on_comp_prop_name  => 'HIDDEN_REQUIRED_VALUE_TYPE',
            p_depending_on_condition_type  => 'EQUALS',
            p_depending_on_expression      => 'ITEM' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'VALUE_TYPE',
            p_db_column                    => null,
            p_is_required                  => true,
            p_is_common                    => true,
            p_default_value                => 'ITEM',
            p_depending_on_comp_prop_name  => 'HIDDEN_REQUIRED_VALUE_TYPE',
            p_depending_on_condition_type  => 'EQUALS',
            p_depending_on_expression      => 'BASIC' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'VALUE_STATIC_VALUE',
            p_db_column                    => null,
            p_is_required                  => true,
            p_is_common                    => true,
            p_depending_on_comp_prop_name  => 'HIDDEN_VALUE_TYPE',
            p_depending_on_condition_type  => 'EQUALS',
            p_depending_on_expression      => 'STATIC',
            p_depending_on_comp_prop_name2 => 'WEB_SRC_PARAM_DATA_TYPE',
            p_depending_on_condition_type2 => 'NOT_EQUALS',
            p_depending_on_expression2     => 'BOOLEAN' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'VALUE_STATIC_VALUE_BOOLEAN',
            p_db_column                    => null,
            p_is_required                  => true,
            p_is_common                    => true,
            p_depending_on_comp_prop_name  => 'HIDDEN_VALUE_TYPE',
            p_depending_on_condition_type  => 'EQUALS',
            p_depending_on_expression      => 'STATIC',
            p_depending_on_comp_prop_name2 => 'WEB_SRC_PARAM_DATA_TYPE',
            p_depending_on_condition_type2 => 'EQUALS',
            p_depending_on_expression2     => 'BOOLEAN' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'VALUE_QUERY',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => 'HIDDEN_VALUE_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'SQL_QUERY' );
        --
    /*
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'VALUE_QUERY_COLON',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => 'VALUE_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'SQL_QUERY_COLON' );

        --
    */
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'VALUE_ITEM',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => 'HIDDEN_VALUE_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'ITEM' );
        --
        store_expression_comp_props (
            p_component_type_id => l_component_type_id,
            p_name_prefix       => 'VALUE_',
            p_return_type       => 'CUSTOM',
            p_reference_scope   => 'DEPENDS',
            p_type_property     => 'HIDDEN_VALUE_TYPE' );
        --
        store_func_body_comp_props (
            p_component_type_id => l_component_type_id,
            p_name_prefix       => 'VALUE_',
            p_return_type       => 'CUSTOM',
            p_reference_scope   => 'DEPENDS',
            p_type_property     => 'HIDDEN_VALUE_TYPE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'VALUE_PREFERENCE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_supports_substitution       => false,
            p_depending_on_comp_prop_name => 'HIDDEN_VALUE_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'PREFERENCE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'VALUE_COLLECTION',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_supports_substitution       => false,
            p_depending_on_comp_prop_name => 'HIDDEN_VALUE_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'COLLECTION' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'VALUE_FORMAT_MASK',
            p_is_required                  => false,
            p_is_common                    => false,
            p_db_column                    => null,
            p_is_read_only                 => false,
            p_depending_on_comp_prop_name  => 'WEB_SRC_PARAM_DATA_TYPE',
            p_depending_on_condition_type  => 'IN_LIST',
            p_depending_on_expression      => 'NUMBER,DATE,TIMESTAMP,TIMESTAMP WITH TIME ZONE',
            p_depending_on_comp_prop_name2 => 'HIDDEN_VALUE_TYPE',
            p_depending_on_condition_type2 => 'IN_LIST',
            p_depending_on_expression2     => 'STATIC,ITEM,PREFERENCE',
            p_help_text                    => 'Format mask to convert the <strong>Static</strong>, <strong>Item</strong> or <strong>Preference</strong> value to the data type of the REST Source Parameter.' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'VALUE_BOOLEAN_TRUE',
            p_db_column                    => null,
            p_is_required                  => false,
            p_is_common                    => true,
            p_supports_substitution        => false,
            p_depending_on_comp_prop_name  => 'HIDDEN_VALUE_TYPE',
            p_depending_on_condition_type  => 'IN_LIST',
            p_depending_on_expression      => 'ITEM,PREFERENCE',
            p_depending_on_comp_prop_name2 => 'WEB_SRC_PARAM_DATA_TYPE',
            p_depending_on_condition_type2 => 'EQUALS',
            p_depending_on_expression2     => 'BOOLEAN' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'VALUE_PARENT_COLUMN',
            p_db_column                    => null,
            p_is_required                  => true,
            p_is_common                    => false,
            p_supports_substitution        => false,
            p_reference_on_delete          => 'WEAK_NULL',
            p_depending_on_comp_prop_name  => 'HIDDEN_VALUE_TYPE',
            p_depending_on_condition_type  => 'EQUALS',
            p_depending_on_expression      => 'PARENT_COLUMN' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'VALUE_BOOLEAN_FALSE',
            p_db_column                    => null,
            p_is_required                  => false,
            p_is_common                    => true,
            p_supports_substitution        => false,
            p_depending_on_comp_prop_name  => 'HIDDEN_VALUE_TYPE',
            p_depending_on_condition_type  => 'IN_LIST',
            p_depending_on_expression      => 'ITEM,PREFERENCE',
            p_depending_on_comp_prop_name2 => 'WEB_SRC_PARAM_DATA_TYPE',
            p_depending_on_condition_type2 => 'EQUALS',
            p_depending_on_expression2     => 'BOOLEAN' );
        --
        store_audit_info_comp_prop (
            p_component_type_id => l_component_type_id );
        --
    end store_comp_props_ws_param_comp;
    --
    procedure store_comp_props_api_param
    is
    begin
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'NAME',
            p_db_column             => null,
            p_check_uniqueness      => 'PARENT',
            p_is_required           => true,
            p_is_common             => true,
            p_supports_substitution => false,
            p_default_value         => 'p_',
            p_help_text             => q'~Enter the name of a parameter declared in the selected procedure or function.~' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'API_DIRECTION',
            p_db_column                    => null,
            p_is_required                  => true,
            p_is_common                    => true,
            p_default_value                => 'IN' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'API_DATA_TYPE',
            p_db_column                    => null,
            p_is_required                  => true,
            p_is_common                    => true,
            p_default_value                => 'VARCHAR2' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'API_CUSTOM_DATA_TYPE',
            p_db_column                    => null,
            p_is_required                  => true,
            p_is_common                    => true,
            p_supports_substitution        => false,
            p_depending_on_comp_prop_name  => 'API_DATA_TYPE',
            p_depending_on_condition_type  => 'EQUALS',
            p_depending_on_expression      => 'CUSTOM' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'HAS_DEFAULT',
            p_db_column                    => null,
            p_is_required                  => true,
            p_is_common                    => true,
            p_default_value                => 'N',
            p_depending_on_comp_prop_name  => 'API_DIRECTION',
            p_depending_on_condition_type  => 'EQUALS',
            p_depending_on_expression      => 'IN' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'IGNORE_OUTPUT',
            p_db_column                    => null,
            p_is_required                  => true,
            p_is_common                    => true,
            p_default_value                => 'N',
            p_depending_on_comp_prop_name  => 'API_DIRECTION',
            p_depending_on_condition_type  => 'IN_LIST',
            p_depending_on_expression      => 'IN_OUT,OUT',
            p_depending_on_comp_prop_name2 => 'API_DATA_TYPE',
            p_depending_on_condition_type2 => 'NOT_IN_LIST',
            p_depending_on_expression2     => 'CUSTOM,BLOB' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'HIDDEN_REQUIRED_VALUE_TYPE',
            p_db_column                    => null,
            p_default_value                => 'BASIC',
            p_is_required                  => false,
            p_is_common                    => false );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'HIDDEN_VALUE_TYPE',
            p_db_column                    => null,
            p_default_value                => 'ITEM', /* keep in sync with default of API_VALUE_TYPE_BASIC */
            p_is_required                  => false,
            p_is_common                    => false );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'API_VALUE_TYPE_BASIC',
            p_db_column                    => null,
            p_is_required                  => true,
            p_is_common                    => true,
            p_default_value                => 'ITEM',
            p_depending_on_comp_prop_name  => 'HIDDEN_REQUIRED_VALUE_TYPE',
            p_depending_on_condition_type  => 'EQUALS',
            p_depending_on_expression      => 'BASIC' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'API_VALUE_TYPE_BLOB',
            p_db_column                    => null,
            p_is_required                  => true,
            p_is_common                    => true,
            p_default_value                => 'SQL_QUERY',
            p_depending_on_comp_prop_name  => 'HIDDEN_REQUIRED_VALUE_TYPE',
            p_depending_on_condition_type  => 'EQUALS',
            p_depending_on_expression      => 'BLOB' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'API_VALUE_TYPE_CUSTOM',
            p_db_column                    => null,
            p_is_required                  => true,
            p_is_common                    => true,
            p_default_value                => 'FUNCTION_BODY',
            p_depending_on_comp_prop_name  => 'HIDDEN_REQUIRED_VALUE_TYPE',
            p_depending_on_condition_type  => 'EQUALS',
            p_depending_on_expression      => 'CUSTOM' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'VALUE_STATIC_VALUE',
            p_db_column                    => null,
            p_is_required                  => true,
            p_is_common                    => true,
            p_reference_scope              => 'DEPENDS',
            p_depending_on_comp_prop_name  => 'HIDDEN_VALUE_TYPE',
            p_depending_on_condition_type  => 'EQUALS',
            p_depending_on_expression      => 'STATIC',
            p_depending_on_comp_prop_name2 => 'API_DATA_TYPE',
            p_depending_on_condition_type2 => 'NOT_EQUALS',
            p_depending_on_expression2     => 'BOOLEAN' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'VALUE_STATIC_VALUE_BOOLEAN',
            p_db_column                    => null,
            p_is_required                  => true,
            p_is_common                    => true,
            p_depending_on_comp_prop_name  => 'HIDDEN_VALUE_TYPE',
            p_depending_on_condition_type  => 'EQUALS',
            p_depending_on_expression      => 'STATIC',
            p_depending_on_comp_prop_name2 => 'API_DATA_TYPE',
            p_depending_on_condition_type2 => 'EQUALS',
            p_depending_on_expression2     => 'BOOLEAN' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'VALUE_QUERY',
            p_db_column                    => null,
            p_is_required                  => true,
            p_is_common                    => true,
            p_reference_scope              => 'DEPENDS',
            p_depending_on_comp_prop_name  => 'HIDDEN_VALUE_TYPE',
            p_depending_on_condition_type  => 'EQUALS',
            p_depending_on_expression      => 'SQL_QUERY' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'VALUE_ITEM',
            p_db_column                    => null,
            p_is_required                  => true,
            p_is_common                    => true,
            p_reference_scope              => 'DEPENDS',
            p_depending_on_comp_prop_name  => 'HIDDEN_VALUE_TYPE',
            p_depending_on_condition_type  => 'EQUALS',
            p_depending_on_expression      => 'ITEM' );
        --
        store_expression_comp_props (
            p_component_type_id           => l_component_type_id,
            p_name_prefix                 => 'VALUE_',
            p_return_type                 => 'CUSTOM',
            p_reference_scope             => 'DEPENDS',
            p_type_property               => 'HIDDEN_VALUE_TYPE',
            p_depending_on_comp_prop_name => 'API_DATA_TYPE',
            p_depending_on_condition_type => 'NOT_EQUALS',
            p_depending_on_expression     => 'CUSTOM' );
        --
        store_func_body_comp_props (
            p_component_type_id           => l_component_type_id,
            p_name_prefix                 => 'VALUE_',
            p_return_type                 => 'CUSTOM',
            p_reference_scope             => 'DEPENDS',
            p_type_property               => 'HIDDEN_VALUE_TYPE',
            p_depending_on_comp_prop_name => 'API_DATA_TYPE',
            p_depending_on_condition_type => 'NOT_EQUALS',
            p_depending_on_expression     => 'CUSTOM' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'VALUE_PLSQL_EXPRESSION_NO_LANG',
            p_db_column                    => null,
            p_is_required                  => true,
            p_is_common                    => true,
            p_reference_scope              => 'DEPENDS',
            p_depending_on_comp_prop_name  => 'HIDDEN_VALUE_TYPE',
            p_depending_on_condition_type  => 'EQUALS',
            p_depending_on_expression      => 'EXPRESSION',
            p_depending_on_comp_prop_name2 => 'API_DATA_TYPE',
            p_depending_on_condition_type2 => 'EQUALS',
            p_depending_on_expression2     => 'CUSTOM' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'VALUE_PLSQL_FUNC_BODY_NO_LANG',
            p_db_column                    => null,
            p_is_required                  => true,
            p_is_common                    => true,
            p_reference_scope              => 'DEPENDS',
            p_depending_on_comp_prop_name  => 'HIDDEN_VALUE_TYPE',
            p_depending_on_condition_type  => 'EQUALS',
            p_depending_on_expression      => 'FUNCTION_BODY',
            p_depending_on_comp_prop_name2 => 'API_DATA_TYPE',
            p_depending_on_condition_type2 => 'EQUALS',
            p_depending_on_expression2     => 'CUSTOM' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'VALUE_PREFERENCE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_supports_substitution       => false,
            p_depending_on_comp_prop_name => 'HIDDEN_VALUE_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'PREFERENCE' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'VALUE_FORMAT_MASK',
            p_db_column                    => null,
            p_is_required                  => false,
            p_is_common                    => true,
            p_supports_substitution        => false,
            p_reference_scope              => 'DEPENDS',
            p_depending_on_comp_prop_name  => 'HIDDEN_VALUE_TYPE',
            p_depending_on_condition_type  => 'IN_LIST',
            p_depending_on_expression      => 'STATIC,ITEM,PREFERENCE',
            p_depending_on_comp_prop_name2 => 'API_DATA_TYPE',
            p_depending_on_condition_type2 => 'IN_LIST',
            p_depending_on_expression2     => 'NUMBER,DATE,TIMESTAMP,TIMESTAMP WITH TIME ZONE,TIMESTAMP WITH LOCAL TIME ZONE' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'VALUE_BOOLEAN_TRUE',
            p_db_column                    => null,
            p_is_required                  => false,
            p_is_common                    => true,
            p_supports_substitution        => false,
            p_depending_on_comp_prop_name  => 'HIDDEN_VALUE_TYPE',
            p_depending_on_condition_type  => 'IN_LIST',
            p_depending_on_expression      => 'ITEM,PREFERENCE',
            p_depending_on_comp_prop_name2 => 'API_DATA_TYPE',
            p_depending_on_condition_type2 => 'EQUALS',
            p_depending_on_expression2     => 'BOOLEAN' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'VALUE_BOOLEAN_FALSE',
            p_db_column                    => null,
            p_is_required                  => false,
            p_is_common                    => true,
            p_supports_substitution        => false,
            p_depending_on_comp_prop_name  => 'HIDDEN_VALUE_TYPE',
            p_depending_on_condition_type  => 'IN_LIST',
            p_depending_on_expression      => 'ITEM,PREFERENCE',
            p_depending_on_comp_prop_name2 => 'API_DATA_TYPE',
            p_depending_on_condition_type2 => 'EQUALS',
            p_depending_on_expression2     => 'BOOLEAN' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'PARAM_DISPLAY_SEQUENCE',
            p_db_column                    => null,
            p_is_required                  => true,
            p_is_common                    => false );
        --
        store_standard_attr_comp_prop (
            p_component_type_id => l_component_type_id,
            p_comment_db_column => 'PARAM_COMMENT' );
        --
    end store_comp_props_api_param;
    --
    procedure store_comp_props_wf_param
    is
    begin
        store_component_prop (
            p_component_type_id   => l_component_type_id,
            p_property_name       => 'WORKFLOW_PARAM',
            p_db_column           => null,
            p_is_required         => true,
            p_is_common           => true,
            p_is_read_only        => true,
            p_reference_on_delete => 'DB_CASCADE' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'API_DIRECTION',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => true,
            p_default_value     => 'IN',
            p_is_read_only      => true,
            p_help_text         => q'~The direction of the parameter as defined within the workflow.~' );
        --
        -- Only workflow activities can wait on the result of an invoked workflow and use the output
        --
        if l_component_type_id = c_comp_type_wf_act_wf_param then
            store_component_prop (
                p_component_type_id            => l_component_type_id,
                p_property_name                => 'IGNORE_OUTPUT',
                p_db_column                    => null,
                p_is_required                  => true,
                p_default_value                => 'Y',
                p_is_common                    => true,
                p_depending_on_comp_prop_name  => 'API_DIRECTION',
                p_depending_on_condition_type  => 'IN_LIST',
                p_depending_on_expression      => 'OUT,IN_OUT' );
        end if;
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'WORKFLOW_DATA_TYPE',
            p_db_column                    => null,
            p_is_required                  => true,
            p_is_common                    => true,
            p_is_read_only                 => true );
        --
        -- This hidden property is populated by dev_views.sql and pe.callbacks.js and uses the direction and ignore_output to determine if it can
        -- accept any value or item (=BASIC), or because it's an OUT direction where ignore_output = N only items (=ITEM).
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'HIDDEN_REQUIRED_VALUE_TYPE',
            p_is_required                  => false,
            p_is_common                    => true,
            p_default_value                => 'BASIC' );
        --
        -- All VALUE_* properties are dependent on this hidden property. It has to be hidden because if HIDDEN_REQUIRED_VALUE_TYPE is set to ITEM
        -- a developer is NOT able to choose how they want to set the parameter, they can only assign an item.
        -- In pe.callbacks.js, HIDDEN_VALUE_TYPE is set to the value of PARAMETER_VALUE_TYPE if it's visible so that all VALUE_* properties work.
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'HIDDEN_VALUE_TYPE',
            p_db_column                    => 'VALUE_TYPE',
            p_is_required                  => false,
            p_is_common                    => true,
            p_default_value                => 'ITEM' ); /* keep in sync with default of PARAMETER_VALUE_TYPE */
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'PARAMETER_VALUE_TYPE',
            p_db_column                    => null,
            p_is_required                  => true,
            p_is_common                    => true,
            p_default_value                => 'ITEM',
            p_depending_on_comp_prop_name  => 'HIDDEN_REQUIRED_VALUE_TYPE',
            p_depending_on_condition_type  => 'EQUALS',
            p_depending_on_expression      => 'BASIC' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'VALUE_STATIC_VALUE',
            p_db_column                    => null,
            p_is_required                  => true,
            p_is_common                    => true,
            p_depending_on_comp_prop_name  => 'HIDDEN_VALUE_TYPE',
            p_depending_on_condition_type  => 'EQUALS',
            p_depending_on_expression      => 'STATIC',
            p_depending_on_comp_prop_name2 => 'WORKFLOW_DATA_TYPE',
            p_depending_on_condition_type2 => 'NOT_EQUALS',
            p_depending_on_expression2     => 'BOOLEAN' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'VALUE_STATIC_VALUE_BOOLEAN',
            p_db_column                    => null,
            p_is_required                  => true,
            p_is_common                    => true,
            p_depending_on_comp_prop_name  => 'HIDDEN_VALUE_TYPE',
            p_depending_on_condition_type  => 'EQUALS',
            p_depending_on_expression      => 'STATIC',
            p_depending_on_comp_prop_name2 => 'WORKFLOW_DATA_TYPE',
            p_depending_on_condition_type2 => 'EQUALS',
            p_depending_on_expression2     => 'BOOLEAN' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'VALUE_QUERY',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => 'HIDDEN_VALUE_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'SQL_QUERY' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'VALUE_ITEM',
            p_db_column                    => null,
            p_is_required                  => true,
            p_is_common                    => true,
            p_depending_on_comp_prop_name  => 'HIDDEN_VALUE_TYPE',
            p_depending_on_condition_type  => 'EQUALS',
            p_depending_on_expression      => 'ITEM' );
        --
        store_expression_comp_props (
            p_component_type_id => l_component_type_id,
            p_name_prefix       => 'VALUE_',
            p_return_type       => 'CUSTOM',
            p_reference_scope   => 'DEPENDS',
            p_type_property     => 'HIDDEN_VALUE_TYPE' );
        --
        store_func_body_comp_props (
            p_component_type_id => l_component_type_id,
            p_name_prefix       => 'VALUE_',
            p_return_type       => 'CUSTOM',
            p_reference_scope   => 'DEPENDS',
            p_type_property     => 'HIDDEN_VALUE_TYPE' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'VALUE_FORMAT_MASK',
            p_is_required                  => false,
            p_is_common                    => false,
            p_db_column                    => null,
            p_is_read_only                 => false,
            p_depending_on_comp_prop_name  => 'WORKFLOW_DATA_TYPE',
            p_depending_on_condition_type  => 'IN_LIST',
            p_depending_on_expression      => 'NUMBER,DATE,TIMESTAMP,TIMESTAMP WITH LOCAL TIME ZONE',
            p_depending_on_comp_prop_name2 => 'HIDDEN_VALUE_TYPE',
            p_depending_on_condition_type2 => 'IN_LIST',
            p_depending_on_expression2     => 'STATIC,ITEM',
            p_help_text                    => 'Format mask to convert the <strong>Static</strong> or <strong>Item</strong> value to the data type of the workflow parameter.' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'VALUE_BOOLEAN_TRUE',
            p_db_column                    => null,
            p_is_required                  => false,
            p_is_common                    => true,
            p_supports_substitution        => false,
            p_depending_on_comp_prop_name  => 'HIDDEN_VALUE_TYPE',
            p_depending_on_condition_type  => 'IN_LIST',
            p_depending_on_expression      => 'ITEM,PREFERENCE',
            p_depending_on_comp_prop_name2 => 'WORKFLOW_DATA_TYPE',
            p_depending_on_condition_type2 => 'EQUALS',
            p_depending_on_expression2     => 'BOOLEAN' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'VALUE_BOOLEAN_FALSE',
            p_db_column                    => null,
            p_is_required                  => false,
            p_is_common                    => true,
            p_supports_substitution        => false,
            p_depending_on_comp_prop_name  => 'HIDDEN_VALUE_TYPE',
            p_depending_on_condition_type  => 'IN_LIST',
            p_depending_on_expression      => 'ITEM,PREFERENCE',
            p_depending_on_comp_prop_name2 => 'WORKFLOW_DATA_TYPE',
            p_depending_on_condition_type2 => 'EQUALS',
            p_depending_on_expression2     => 'BOOLEAN' );
        --
        store_audit_info_comp_prop (
            p_component_type_id => l_component_type_id );
        --
    end store_comp_props_wf_param;
    --
    procedure store_comp_props_task_param (
        p_parent_comp_type_id in number
    )
    is
    begin
        store_component_prop (
            p_component_type_id   => l_component_type_id,
            p_property_name       => 'TASK_DEF_PARAM',
            p_db_column           => null,
            p_is_required         => true,
            p_is_common           => true,
            p_is_read_only        => true,
            p_reference_on_delete => 'DB_CASCADE' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'TASK_VALUE_TYPE',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => true,
            p_default_value     => 'STATIC' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'VALUE_STATIC_VALUE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => 'TASK_VALUE_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'STATIC' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'VALUE_QUERY',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => 'TASK_VALUE_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'SQL_QUERY' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'VALUE_ITEM',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => 'TASK_VALUE_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'ITEM' );
        --
        store_expression_comp_props (
            p_component_type_id => l_component_type_id,
            p_name_prefix       => 'VALUE_',
            p_return_type       => 'VARCHAR2',
            p_type_property     => 'TASK_VALUE_TYPE' );
        --
        store_func_body_comp_props (
            p_component_type_id => l_component_type_id,
            p_name_prefix       => 'VALUE_',
            p_return_type       => 'VARCHAR2',
            p_type_property     => 'TASK_VALUE_TYPE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'VALUE_PREFERENCE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_supports_substitution       => false,
            p_depending_on_comp_prop_name => 'TASK_VALUE_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'PREFERENCE' );
        --
        store_audit_info_comp_prop (
            p_component_type_id => l_component_type_id );
        --
    end store_comp_props_task_param;
    --
    procedure store_property_series_source (
        p_prefix               in varchar2,
        p_property_id_type     in number,
        p_property_id_sql      in number,
        p_property_id_plsql    in number,
        p_sql_min_column_count in number,
        p_sql_max_column_count in number,
        p_examples_sql         in varchar2,
        p_examples_plsql       in varchar2 )
    is
    begin
        l_property_id := p_property_id_type;
        store_property (
            p_id                 => l_property_id,
            p_name               => p_prefix || '_SERIES_SOURCE_TYPE',
            p_prompt             => 'Type',
            p_property_type      => 'SELECT LIST',
            p_db_column          => 'SERIES_QUERY_TYPE',
            p_display_group_name => 'SOURCE',
            p_lov_type           => 'STATIC',
            p_help_text          => q'~Select the query source type for the chart series.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'SQL Query',
            p_return_value  => 'SQL_QUERY' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Function Body returning SQL Query',
            p_return_value  => 'FUNCTION_RETURNING_SQL_QUERY' );
        --
        l_property_id := p_property_id_sql;
        store_property (
            p_id                   => l_property_id,
            p_name                 => p_prefix || '_SOURCE_QUERY',
            p_prompt               => 'SQL Query',
            p_property_type        => 'SQL',
            p_db_column            => 'SERIES_QUERY',
            p_display_length       => 80,
            p_max_length           => 32767,
            p_sql_min_column_count => p_sql_min_column_count,
            p_sql_max_column_count => p_sql_max_column_count,
            p_examples             => p_examples_sql,
            p_display_group_name   => 'SOURCE',
            p_help_text            => q'~Enter the SQL Query for the chart series.~',
            p_comment_text         => '$$$ how can we check the data types of the SQL query?' );
        --
        l_property_id := p_property_id_plsql;
        store_property (
            p_id                   => l_property_id,
            p_name                 => p_prefix || '_SOURCE_FUNC_RETURNING_SQL',
            p_prompt               => 'PL/SQL Function Body returning SQL Query',
            p_property_type        => 'PLSQL FUNCTION BODY SQL',
            p_db_column            => 'SERIES_QUERY',
            p_display_length       => 80,
            p_max_length           => 32767,
            p_display_group_name   => 'SOURCE',
            p_sql_min_column_count => p_sql_min_column_count,
            p_sql_max_column_count => p_sql_max_column_count,
            p_examples             => p_examples_plsql,
            p_help_text            => q'~Enter a PL/SQL function body that returns an SQL Query for the chart series.~',
            p_comment_text         => '$$$ how can we check the data types of the SQL query?' );
    end store_property_series_source;
    --
    procedure store_comp_prop_wf_variable
    is
    begin
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'WF_IDENTIFICATION_STATIC_ID',
            p_db_column             => 'STATIC_ID',
            p_check_uniqueness      => 'WORKFLOW', -- limited to the workflow version
            p_is_required           => true,
            p_is_common             => true,
            p_supports_substitution => false,
            p_default_value         => '#PLEASE_CHANGE#' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'LABEL',
            p_db_column         => 'LABEL',
            p_is_required       => true, -- $$$ why is it optional for parameters but required for variables?
            p_is_common         => true,
            p_default_value     => '#PLEASE_CHANGE#',
            p_check_uniqueness  => 'PARENT',
            p_help_text         => q'~An end-user friendly name of the workflow variable.~' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'WORKFLOW_DATA_TYPE',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => true,
            p_default_value     => 'VARCHAR2' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'WORKFLOW_VALUE_TYPE',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => true,
            p_default_value     => 'NULL' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'VALUE_STATIC_VALUE',
            p_db_column                    => null,
            p_is_required                  => true,
            p_is_common                    => true,
            p_depending_on_comp_prop_name  => 'WORKFLOW_VALUE_TYPE',
            p_depending_on_condition_type  => 'EQUALS',
            p_depending_on_expression      => 'STATIC',
            p_depending_on_comp_prop_name2 => 'WORKFLOW_DATA_TYPE',
            p_depending_on_condition_type2 => 'NOT_EQUALS',
            p_depending_on_expression2     => 'BOOLEAN' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'VALUE_STATIC_VALUE_BOOLEAN',
            p_db_column                    => null,
            p_is_required                  => true,
            p_is_common                    => true,
            p_depending_on_comp_prop_name  => 'WORKFLOW_VALUE_TYPE',
            p_depending_on_condition_type  => 'EQUALS',
            p_depending_on_expression      => 'STATIC',
            p_depending_on_comp_prop_name2 => 'WORKFLOW_DATA_TYPE',
            p_depending_on_condition_type2 => 'EQUALS',
            p_depending_on_expression2     => 'BOOLEAN' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'VALUE_ITEM',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => 'WORKFLOW_VALUE_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'ITEM' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'VALUE_QUERY',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => 'WORKFLOW_VALUE_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'SQL_QUERY' );
        --
        store_expression_comp_props (
            p_component_type_id => l_component_type_id,
            p_name_prefix       => 'VALUE_',
            p_return_type       => 'CUSTOM',
            p_type_property     => 'WORKFLOW_VALUE_TYPE' );
        --
        store_func_body_comp_props (
            p_component_type_id => l_component_type_id,
            p_name_prefix       => 'VALUE_',
            p_return_type       => 'CUSTOM',
            p_type_property     => 'WORKFLOW_VALUE_TYPE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'SESSION_FORMAT_MASK',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_supports_substitution       => false,
            p_depending_on_comp_prop_name => 'WORKFLOW_DATA_TYPE',
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'NUMBER,TIMESTAMP,TIMESTAMP_TZ,TIMESTAMP_LTZ',
            p_help_text                   => q'~Format mask to be used to convert the workflow variable into a NUMBER or TIMESTAMP and back into a VARCHAR2 when the parameter is used in a bind variable or substitution string. ~' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'VARIABLE_BOOLEAN_TRUE',
            p_db_column                    => null,
            p_is_required                  => false,
            p_is_common                    => true,
            p_supports_substitution        => false,
            p_depending_on_comp_prop_name  => 'WORKFLOW_DATA_TYPE',
            p_depending_on_condition_type  => 'EQUALS',
            p_depending_on_expression      => 'BOOLEAN',
            p_depending_on_comp_prop_name2 => 'WORKFLOW_VALUE_TYPE',
            p_depending_on_condition_type2 => 'NOT_IN_LIST',
            p_depending_on_expression2     => 'STATIC' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'VARIABLE_BOOLEAN_FALSE',
            p_db_column                    => null,
            p_is_required                  => false,
            p_is_common                    => true,
            p_supports_substitution        => false,
            p_depending_on_comp_prop_name  => 'WORKFLOW_DATA_TYPE',
            p_depending_on_condition_type  => 'EQUALS',
            p_depending_on_expression      => 'BOOLEAN',
            p_depending_on_comp_prop_name2 => 'WORKFLOW_VALUE_TYPE',
            p_depending_on_condition_type2 => 'NOT_IN_LIST',
            p_depending_on_expression2     => 'STATIC' );
        --
        store_standard_attr_comp_prop (
            p_component_type_id => l_component_type_id,
            p_has_comments      => true,
            p_comment_db_column => 'VARIABLE_COMMENT' );
        --
end store_comp_prop_wf_variable;

    --
    procedure store_chart_font_properties (
        p_font_group             in varchar2,
        p_prompt_prefix          in varchar2 default null,
        p_display_group_name     in varchar2,
        p_property_id_font       in number,
        p_property_id_font_size  in number,
        p_property_id_font_color in number )
    is
        c_prompt_prefix varchar2( 30 ) := p_prompt_prefix || case when p_prompt_prefix is not null then ' ' end;
    begin
        --
        l_property_id := p_property_id_font;
        store_property (
            p_id                 => l_property_id,
            p_name               => p_font_group || '_FACE',
            p_prompt             => c_prompt_prefix || 'Font Face',
            p_property_type      => 'SELECT LIST',
            p_db_column          => p_font_group || '_FACE',
            p_display_group_name => p_display_group_name,
            p_lov_type           => 'STATIC',
            p_help_text          => q'~Select a font face.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Arial',
            p_return_value  => 'Arial' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Helvetica',
            p_return_value  => 'Helvetica' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Geneva',
            p_return_value  => 'Geneva' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Sans-serif',
            p_return_value  => 'sans-serif' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Verdana',
            p_return_value  => 'Verdana' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Tahoma',
            p_return_value  => 'Tahoma' );
        --
        l_property_id := p_property_id_font_size;
        store_property (
            p_id                 => l_property_id,
            p_name               => p_font_group || '_SIZE',
            p_prompt             => c_prompt_prefix || 'Font Size',
            p_property_type      => 'SELECT LIST',
            p_db_column          => p_font_group || '_SIZE',
            p_display_group_name => p_display_group_name,
            p_lov_type           => 'STATIC',
            p_help_text          => q'~Select the font size.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => '2',
            p_return_value  => '2' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => '4',
            p_return_value  => '4' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => '6',
            p_return_value  => '6' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => '8',
            p_return_value  => '8' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => '10',
            p_return_value  => '10' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => '12',
            p_return_value  => '12' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => '14',
            p_return_value  => '14' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => '16',
            p_return_value  => '16' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => '18',
            p_return_value  => '18' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => '20',
            p_return_value  => '20' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => '22',
            p_return_value  => '22' );
        --
        l_property_id := p_property_id_font_color;
        store_property (
            p_id                 => l_property_id,
            p_name               => p_font_group || '_COLOR',
            p_prompt             => c_prompt_prefix || 'Font Color',
            p_property_type      => 'COLOR',
            p_db_column          => p_font_group || '_COLOR',
            p_display_group_name => p_display_group_name,
            p_help_text          => q'~<p>Enter the color used to display the font for the specified attribute.</p>~' );

    end store_chart_font_properties;
    --
    procedure store_chart_font_comp_props (
        p_font_group                  in varchar2,
        p_default_font_size           in number   default 10,
        p_depending_on_comp_prop_name in varchar2 default null,
        p_depending_on_condition_type in varchar2 default null,
        p_depending_on_expression     in varchar2 default null )
    is
    begin
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => p_font_group || '_FACE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'Tahoma',
            p_depending_on_comp_prop_name => p_depending_on_comp_prop_name,
            p_depending_on_condition_type => p_depending_on_condition_type,
            p_depending_on_expression     => p_depending_on_expression );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => p_font_group || '_SIZE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => p_default_font_size,
            p_depending_on_comp_prop_name => p_depending_on_comp_prop_name,
            p_depending_on_condition_type => p_depending_on_condition_type,
            p_depending_on_expression     => p_depending_on_expression );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => p_font_group || '_COLOR',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => '#000000',
            p_depending_on_comp_prop_name => p_depending_on_comp_prop_name,
            p_depending_on_condition_type => p_depending_on_condition_type,
            p_depending_on_expression     => p_depending_on_expression );
        --
    end store_chart_font_comp_props;
    --
    procedure store_jet_font_properties (
        p_font_group              in varchar2,
        p_display_group_name      in varchar2,
        p_property_id_font_family in number,
        p_property_id_font_style  in number,
        p_property_id_font_size   in number,
        p_property_id_font_color  in number )
    is
    begin
        --
        l_property_id := p_property_id_font_family;
        store_property (
            p_id                 => l_property_id,
            p_name               => 'JET_' ||p_font_group || '_FAMILY',
            p_prompt             => 'Font Family',
            p_property_type      => 'SELECT LIST',
            p_db_column          => p_font_group || '_FAMILY',
            p_display_group_name => p_display_group_name,
            p_lov_type           => 'STATIC',
            p_help_text          => q'~Select the font family.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Arial',
            p_return_value  => 'Arial');
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Arial Black',
            p_return_value  => 'Arial Black');
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Bookman',
            p_return_value  => 'Bookman');
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Comic Sans MS',
            p_return_value  => 'Comic Sans MS');
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Courier',
            p_return_value  => 'Courier');
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Courier New',
            p_return_value  => 'Courier New');
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Garamond',
            p_return_value  => 'Garamond');
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Georgia',
            p_return_value  => 'Georgia');
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Helvetica',
            p_return_value  => 'Helvetica');
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Impact',
            p_return_value  => 'Impact');
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Palatino',
            p_return_value  => 'Palatino');
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Times',
            p_return_value  => 'Times');
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Times New Roman',
            p_return_value  => 'Times New Roman');
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Trebuchet MS',
            p_return_value  => 'Trebuchet MS');
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Verdana',
            p_return_value  => 'Verdana');
        --
        l_property_id := p_property_id_font_style;
        store_property (
            p_id                 => l_property_id,
            p_name               => 'JET_' ||p_font_group || '_STYLE',
            p_prompt             => 'Font Style',
            p_property_type      => 'SELECT LIST',
            p_db_column          => p_font_group || '_STYLE',
            p_display_group_name => p_display_group_name,
            p_lov_type           => 'STATIC',
            p_help_text          => q'~Select the font style.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Normal',
            p_return_value  => 'normal');
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Italic',
            p_return_value  => 'italic');
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Oblique',
            p_return_value  => 'oblique');
        --
        l_property_id := p_property_id_font_size;
        store_property (
            p_id                 => l_property_id,
            p_name               => 'JET_' ||p_font_group || '_SIZE',
            p_prompt             => 'Font Size',
            p_property_type      => 'SELECT LIST',
            p_db_column          => p_font_group || '_SIZE',
            p_display_group_name => p_display_group_name,
            p_lov_type           => 'STATIC',
            p_help_text          => q'~Select the font size.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => '8',
            p_return_value  => '8' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => '10',
            p_return_value  => '10' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => '12',
            p_return_value  => '12' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => '14',
            p_return_value  => '14' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => '16',
            p_return_value  => '16' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => '18',
            p_return_value  => '18' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => '20',
            p_return_value  => '20' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => '22',
            p_return_value  => '22' );
        --
        l_property_id := p_property_id_font_color;
        store_property (
            p_id                 => l_property_id,
            p_name               => 'JET_' ||p_font_group || '_COLOR',
            p_prompt             => 'Font Color',
            p_property_type      => 'COLOR',
            p_db_column          => p_font_group || '_COLOR',
            p_display_group_name => p_display_group_name,
            p_help_text          => q'~<p>Enter the color used to display the font for the specified attribute.</p>~' );

    end store_jet_font_properties;
    --
    procedure store_jet_font_comp_props (
        p_font_group                   in varchar2,
        p_depending_on_comp_prop_name  in varchar2 default null,
        p_depending_on_condition_type  in varchar2 default null,
        p_depending_on_expression      in varchar2 default null,
        p_depending_on_comp_prop_name2 in varchar2 default null,
        p_depending_on_condition_type2 in varchar2 default null,
        p_depending_on_expression2     in varchar2 default null )
    is
    begin
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'JET_'|| p_font_group || '_FAMILY',
            p_db_column                    => null,
            p_is_required                  => false,
            p_is_common                    => false,
            p_depending_on_comp_prop_name  => p_depending_on_comp_prop_name,
            p_depending_on_condition_type  => p_depending_on_condition_type,
            p_depending_on_expression      => p_depending_on_expression,
            p_depending_on_comp_prop_name2 => p_depending_on_comp_prop_name2,
            p_depending_on_condition_type2 => p_depending_on_condition_type2,
            p_depending_on_expression2     => p_depending_on_expression2);
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'JET_'|| p_font_group || '_STYLE',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => p_depending_on_comp_prop_name,
            p_depending_on_condition_type => p_depending_on_condition_type,
            p_depending_on_expression     => p_depending_on_expression,
            p_depending_on_comp_prop_name2 => p_depending_on_comp_prop_name2,
            p_depending_on_condition_type2 => p_depending_on_condition_type2,
            p_depending_on_expression2     => p_depending_on_expression2 );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'JET_'|| p_font_group || '_SIZE',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => p_depending_on_comp_prop_name,
            p_depending_on_condition_type => p_depending_on_condition_type,
            p_depending_on_expression     => p_depending_on_expression,
            p_depending_on_comp_prop_name2 => p_depending_on_comp_prop_name2,
            p_depending_on_condition_type2 => p_depending_on_condition_type2,
            p_depending_on_expression2     => p_depending_on_expression2 );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'JET_'|| p_font_group || '_COLOR',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => p_depending_on_comp_prop_name,
            p_depending_on_condition_type => p_depending_on_condition_type,
            p_depending_on_expression     => p_depending_on_expression,
            p_depending_on_comp_prop_name2 => p_depending_on_comp_prop_name2,
            p_depending_on_condition_type2 => p_depending_on_condition_type2,
            p_depending_on_expression2     => p_depending_on_expression2 );
        --
    end store_jet_font_comp_props;
    --
    --
    procedure store_cal_weekday_property (
        p_property_id        in number,
        p_name               in varchar2,
        p_prompt             in varchar2,
        p_db_column          in varchar2,
        p_display_group_name in varchar2,
        p_help_text          in varchar2 )
    is
    begin
        l_property_id := p_property_id;
        store_property (
            p_id                 => l_property_id,
            p_name               => p_name,
            p_prompt             => p_prompt,
            p_property_type      => 'SELECT LIST',
            p_db_column          => p_db_column,
            p_display_group_name => p_display_group_name,
            p_lov_type           => 'STATIC',
            p_help_text          => p_help_text );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Sunday',
            p_return_value  => '19721105' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Monday',
            p_return_value  => '19721106' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Tuesday',
            p_return_value  => '19721107' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Wednesday',
            p_return_value  => '19721108' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Thursday',
            p_return_value  => '19721109' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Friday',
            p_return_value  => '19721110' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Saturday',
            p_return_value  => '19721111' );
        --
    end store_cal_weekday_property;
    --
    --
    procedure store_print_font_properties (
        p_font_group                in varchar2,
        p_display_group_name        in varchar2,
        p_db_column_prefix          in varchar2,
        p_property_id_font          in number,
        p_property_id_font_weight   in number,
        p_property_id_font_size     in number,
        p_property_id_font_color    in number,
        p_property_id_font_bg_color in number default null )
    is
    begin
        --
        l_property_id := p_property_id_font;
        store_property (
            p_id                 => l_property_id,
            p_name               => p_font_group || '_FONT_FAMILY',
            p_prompt             => 'Font',
            p_property_type      => 'SELECT LIST',
            p_db_column          => p_db_column_prefix || '_FONT_FAMILY',
            p_display_group_name => p_display_group_name,
            p_lov_type           => 'STATIC',
            p_help_text          => q'~Select a font face for this component.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Helvetica',
            p_return_value  => 'Helvetica' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Times',
            p_return_value  => 'Times' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Courier',
            p_return_value  => 'Courier' );
        --
        l_property_id := p_property_id_font_weight;
        store_property (
            p_id                 => l_property_id,
            p_name               => p_font_group || '_FONT_WEIGHT',
            p_prompt             => 'Font Weight',
            p_property_type      => 'SELECT LIST',
            p_db_column          => p_db_column_prefix || '_FONT_WEIGHT',
            p_display_group_name => p_display_group_name,
            p_lov_type           => 'STATIC',
            p_help_text          => q'~Select a font weight for this component.~' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Normal',
            p_return_value  => 'normal' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Bold',
            p_return_value  => 'bold' );
        --
        l_property_id := p_property_id_font_size;
        store_property (
            p_id                 => l_property_id,
            p_name               => p_font_group || '_FONT_SIZE',
            p_prompt             => 'Font Size',
            p_property_type      => 'INTEGER',
            p_db_column          => p_db_column_prefix || '_FONT_SIZE',
            p_display_length     => 4,
            p_display_group_name => p_display_group_name,
            p_help_text          => q'~Enter a font size for this component.~' );
        --
        l_property_id := p_property_id_font_color;
        store_property (
            p_id                 => l_property_id,
            p_name               => p_font_group || '_FONT_COLOR',
            p_prompt             => 'Font Color',
            p_property_type      => 'COLOR',
            p_db_column          => p_db_column_prefix || '_FONT_COLOR',
            p_display_group_name => p_display_group_name,
            p_help_text          => q'~<p>Enter the color used to display this component.</p>~' );
        --
        if p_property_id_font_bg_color is not null then
            l_property_id := p_property_id_font_bg_color;
            store_property (
                p_id                 => l_property_id,
                p_name               => p_font_group || '_BG_COLOR',
                p_prompt             => 'Background Color',
                p_property_type      => 'COLOR',
                p_db_column          => p_db_column_prefix || '_BG_COLOR',
                p_display_group_name => p_display_group_name,
                p_help_text          => q'~<p>Enter the color used to display the background for this component.</p>~' );
        end if;
        --
    end store_print_font_properties;
    --
    procedure store_print_font_comp_props (
        p_font_group                  in varchar2,
        p_default_size                in varchar2,
        p_default_weight              in varchar2 default 'normal',
        p_has_bg_color                in boolean  default false,
        p_default_bg_color            in varchar2 default null,
        p_depending_on_comp_prop_name in varchar2 default null,
        p_depending_on_condition_type in varchar2 default null,
        p_depending_on_expression     in varchar2 default null )
    is
    begin
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => p_font_group || '_FONT_FAMILY',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'Helvetica',
            p_depending_on_comp_prop_name => p_depending_on_comp_prop_name,
            p_depending_on_condition_type => p_depending_on_condition_type,
            p_depending_on_expression     => p_depending_on_expression );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => p_font_group || '_FONT_WEIGHT',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => p_default_weight,
            p_depending_on_comp_prop_name => p_depending_on_comp_prop_name,
            p_depending_on_condition_type => p_depending_on_condition_type,
            p_depending_on_expression     => p_depending_on_expression );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => p_font_group || '_FONT_SIZE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => p_default_size,
            p_depending_on_comp_prop_name => p_depending_on_comp_prop_name,
            p_depending_on_condition_type => p_depending_on_condition_type,
            p_depending_on_expression     => p_depending_on_expression );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => p_font_group || '_FONT_COLOR',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => '#000000',
            p_depending_on_comp_prop_name => p_depending_on_comp_prop_name,
            p_depending_on_condition_type => p_depending_on_condition_type,
            p_depending_on_expression     => p_depending_on_expression );
        --
        if p_has_bg_color then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => p_font_group || '_BG_COLOR',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => false,
                p_default_value               => p_default_bg_color,
                p_depending_on_comp_prop_name => p_depending_on_comp_prop_name,
                p_depending_on_condition_type => p_depending_on_condition_type,
                p_depending_on_expression     => p_depending_on_expression );
        end if;
        --
    end store_print_font_comp_props;
    --
    --
    procedure store_print_comp_props (
        p_component_type_id in number )
    is
    begin
        --
        if p_component_type_id in ( c_comp_type_classic_rpt_print, c_comp_type_tab_form_print ) then
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'PRINT_LINK_TEXT',
                p_db_column         => null,
                p_is_required       => false,
                p_is_common         => false,
                p_default_value     => '#PRINT_LINK_TEXT#' );
            --
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'PRINT_OUTPUT_FORMAT',
                p_db_column         => null,
                p_is_required       => false,
                p_is_common         => true,
                p_default_value     => 'PDF',
                p_null_text         => 'Derive from Item');
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'PRINT_OUTPUT_FORMAT_ITEM',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => false,
                p_depending_on_comp_prop_name => 'PRINT_OUTPUT_FORMAT',
                p_depending_on_condition_type => 'NULL' );
        end if;
        --
        if p_component_type_id != c_comp_type_ig_print then
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'REPORT_LAYOUT',
                p_db_column         => null,
                p_is_required       => false,
                p_is_common         => false,
                p_null_text         => 'Default Report Layout' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'PRINT_VIEW_FILE_AS',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => false,
                p_default_value               => 'ATTACHMENT' );
        end if;
        --
        if p_component_type_id in ( c_comp_type_classic_rpt_print, c_comp_type_tab_form_print ) then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'PRINT_FILE_NAME',
                p_db_column                   => null,
                p_is_required                 => false,
                p_is_common                   => false );
            --
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'LINK_EXAMPLE',
                p_db_column         => 'PRINT_EXAMPLE_URL',
                p_is_required       => false,
                p_is_common         => false,
                p_help_text         => q'~<p>Report regions can be printed using the standard print link, shown below the report region, or using a button. If you want to use a button, remove the link label and create a button on your page, using the URL shown in Print URL as the button target.</p>
<p>Using buttons to print report regions enables you to set session state. This is useful if you are planning to support printing to several different output formats (Word, Excel, PDF, and so on) and want to display a button, link, or image for each option. For this implementation, use the output format derived from item option and include the item name and value in your Print URL.</p>~' );
        end if;
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'PRINT_PAGE_SIZE',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => true,
            p_default_value     => 'LETTER',
            p_comment_text      => '$$$ default based on current language' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'PRINT_ORIENTATION',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => true,
            p_default_value     => 'HORIZONTAL' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'PRINT_UNITS',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => false,
            p_default_value     => 'INCHES',
            p_comment_text      => '$$$ default based on current language' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'PRINT_WIDTH',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => false,
            p_default_value     => '11',
            p_comment_text      => '$$$ default based on current language' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'PRINT_HEIGHT',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => false,
            p_default_value     => '8.5',
            p_comment_text      => '$$$ default based on current language' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'PRINT_BORDER_WIDTH',
            p_db_column         => null,
            p_is_required       => false,
            p_is_common         => false,
            p_default_value     => '.5' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'PRINT_BORDER_COLOR',
            p_db_column         => null,
            p_is_required       => false,
            p_is_common         => false,
            p_default_value     => '#666666' );
        --
        store_print_font_comp_props (
            p_font_group   => 'PRINT_PAGE_HEADER',
            p_default_size => 12 );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'PRINT_PAGE_HEADER_ALIGNMENT',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => false,
            p_default_value     => 'CENTER' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'PRINT_PAGE_HEADER_TEXT',
            p_db_column         => null,
            p_is_required       => false,
            p_is_common         => true );
        --
        store_print_font_comp_props (
            p_font_group       => 'PRINT_COLUMN_HEADING',
            p_default_size     => 10,
            p_default_weight   => 'bold',
            p_has_bg_color     => true,
            p_default_bg_color => '#EEEEEE' );
        --
        store_print_font_comp_props (
            p_font_group       => 'PRINT_COLUMN',
            p_default_size     => 10,
            p_has_bg_color     => true,
            p_default_bg_color => '#FFFFFF' );
        --
        if p_component_type_id in ( c_comp_type_classic_rpt_print, c_comp_type_tab_form_print ) then
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'PRINT_COLUMN_WIDTH_UNITS',
                p_db_column         => null,
                p_is_required       => true,
                p_is_common         => false,
                p_default_value     => 'PERCENTAGE' );
        end if;
        --
        store_print_font_comp_props (
            p_font_group   => 'PRINT_PAGE_FOOTER',
            p_default_size => 12 );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'PRINT_PAGE_FOOTER_ALIGNMENT',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => false,
            p_default_value     => 'CENTER' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'PRINT_PAGE_FOOTER_TEXT',
            p_db_column         => null,
            p_is_required       => false,
            p_is_common         => false );
        --
    end store_print_comp_props;
    --
    --
    procedure store_lov_type_component_props (
        p_name                        in varchar2,
        p_db_column_named_lov         in varchar2 default null,
        p_db_column_source            in varchar2 default null,
        p_is_required                 in boolean,
        p_plugin_std_attributes       in varchar2 default null,
        p_depending_on_comp_prop_name in varchar2 default null,
        p_depending_on_condition_type in varchar2 default null,
        p_depending_on_expression     in varchar2 default null )
    is
    begin
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => p_name,
            p_db_column                   => null,
            p_is_required                 => p_is_required,
            p_is_common                   => true,
            p_plugin_std_attributes       => p_plugin_std_attributes,
            p_depending_on_comp_prop_name => p_depending_on_comp_prop_name,
            p_depending_on_condition_type => p_depending_on_condition_type,
            p_depending_on_expression     => p_depending_on_expression );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'NAMED_LOV',
            p_db_column                   => p_db_column_named_lov,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => p_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'SHARED' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'LOV_SQL',
            p_db_column                   => p_db_column_source,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => p_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'SQL_QUERY',
            p_help_text                   => q'~<p>Enter the SQL query definition to populate this list of values. Generally list of value queries are of the form:</p>
<pre>
select [displayValue],
       [returnValue]
  from ...
 where ...
 order by ...
</pre>
<p>Each column selected must have a unique name or alias. Oracle recommends using an alias on any column that includes an SQL expression.</p>
<p>Note: When defining a Popup LOV item type, if you would like to display multiple columns in the popup, you must instead define your List of Values in Shared Components,
with the required additional metadata. Inline list of values can only be used to display single columns for Popup LOVs.</p>~' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'LOV_STATIC_VALUES',
            p_db_column                   => p_db_column_source,
            p_is_required                 => true,
            p_is_common                   => true,
            p_default_value               => 'STATIC:Display1;Return1,Display2;Return2',
            p_depending_on_comp_prop_name => p_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'STATIC' );
        --
        store_func_body_comp_props (
            p_component_type_id   => l_component_type_id,
            p_name_prefix         => 'LOV_',
            p_return_type         => 'SQL',
            p_db_column_func_body => p_db_column_source,
            p_type_property       => p_name );
        --
    end store_lov_type_component_props;
    --
    --
    procedure store_comp_prop_fc_attributes
    is
        c_is_facet          constant boolean := ( l_component_type_id in ( c_comp_type_facet,   c_comp_type_facet_group_item ) );
        c_is_facet_or_group constant boolean := ( l_component_type_id in ( c_comp_type_facet,   c_comp_type_facet_group ) );
        c_is_sfilter        constant boolean := ( l_component_type_id in ( c_comp_type_sfilter, c_comp_type_sfilter_group_item ) );
    begin
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'FC_COMPUTE_COUNTS',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'Y',
            p_plugin_std_attributes       => case when c_is_facet or c_is_sfilter then 'FC_HAS_FEEDBACK' end );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'FC_SHOW_COUNTS',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'Y',
            p_depending_on_comp_prop_name => 'FC_COMPUTE_COUNTS',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'Y' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'FC_ZERO_COUNT_ENTRIES',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'H',
            p_depending_on_comp_prop_name => 'FC_COMPUTE_COUNTS',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'Y' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'FC_SORT_BY_TOP_COUNTS',
            p_db_column                    => null,
            p_is_required                  => true,
            p_is_common                    => false,
            p_default_value                => 'Y',
            p_depending_on_comp_prop_name  => 'FC_SHOW_COUNTS',
            p_depending_on_condition_type  => 'EQUALS',
            p_depending_on_expression      => 'Y',
            p_depending_on_comp_prop_name2 => case when c_is_facet or c_is_sfilter then 'ITEM_TYPE'                         end,
            p_depending_on_condition_type2 => case when c_is_facet or c_is_sfilter then 'IN_LIST'                           end,
            p_depending_on_expression2     => case when c_is_facet or c_is_sfilter then 'NATIVE_CHECKBOX,NATIVE_RADIOGROUP' end );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'FC_SHOW_SELECTED_FIRST',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'N',
            p_plugin_std_attributes       => case when c_is_facet or c_is_sfilter then 'FC_SHOW_SELECTED_FIRST' end,
            p_depending_on_comp_prop_name => case when c_is_facet or c_is_sfilter then 'FACET_LOV_TYPE'         end,
            p_depending_on_condition_type => case when c_is_facet or c_is_sfilter then 'NOT_NULL'               end );
        --
        if c_is_facet or l_component_type_id = c_comp_type_facet_group then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'FC_SHOW_MORE_COUNT',
                p_db_column                   => null,
                p_is_required                 => false,
                p_is_common                   => false,
                p_default_value               => '7',
                p_plugin_std_attributes       => case when c_is_facet then 'FC_SHOW_MORE_COUNT' end,
                p_depending_on_comp_prop_name => case when c_is_facet then 'FACET_LOV_TYPE'     end,
                p_depending_on_condition_type => case when c_is_facet then 'NOT_NULL'           end );
        end if;
        --
        if c_is_facet_or_group then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'FC_DISPLAY_AS',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => false,
                p_default_value               => 'INLINE',
                p_depending_on_comp_prop_name => case when c_is_facet then 'ITEM_TYPE'     end,
                p_depending_on_condition_type => case when c_is_facet then 'NOT_EQUALS'    end,
                p_depending_on_expression     => case when c_is_facet then 'NATIVE_SEARCH' end );

            resequence_comp_prop (
                p_component_type_id => l_component_type_id,
                p_name              => 'FC_DISPLAY_AS',
                p_previous_name     => 'ICON_CSS_CLASSES' );
        end if;
        --
        if l_component_type_id = c_comp_type_sfilter then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'FC_FILTER_VALUES',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => false,
                p_default_value               => 'N',
                p_plugin_std_attributes       => 'FC_FILTER_VALUES',
                p_depending_on_comp_prop_name => 'FACET_LOV_TYPE',
                p_depending_on_condition_type => 'NOT_NULL' );
            --
        elsif l_component_type_id = c_comp_type_facet then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'FC_FILTER_VALUES_INITIALLY',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => false,
                p_default_value               => 'N',
                p_plugin_std_attributes       => 'FC_FILTER_VALUES',
                p_depending_on_comp_prop_name => 'FACET_LOV_TYPE',
                p_depending_on_condition_type => 'NOT_NULL' );
            --
            store_component_prop (
                p_component_type_id            => l_component_type_id,
                p_property_name                => 'FC_ACTIONS_FILTER',
                p_db_column                    => null,
                p_is_required                  => true,
                p_is_common                    => false,
                p_default_value                => 'Y',
                p_plugin_std_attributes        => 'FC_FILTER_VALUES',
                p_depending_on_comp_prop_name  => 'FACET_LOV_TYPE',
                p_depending_on_condition_type  => 'NOT_NULL',
                p_depending_on_comp_prop_name2 => 'FC_DISPLAY_AS',
                p_depending_on_condition_type2 => 'EQUALS',
                p_depending_on_expression2     => 'INLINE' );
            --
            store_component_prop (
                p_component_type_id            => l_component_type_id,
                p_property_name                => 'FC_ACTIONS_CHART',
                p_db_column                    => null,
                p_is_required                  => true,
                p_is_common                    => false,
                p_default_value                => 'Y',
                p_depending_on_comp_prop_name  => 'FC_COMPUTE_COUNTS',
                p_depending_on_condition_type  => 'EQUALS',
                p_depending_on_expression      => 'Y',
                p_depending_on_comp_prop_name2 => 'FACET_LOV_TYPE',
                p_depending_on_condition_type2 => 'NOT_NULL' );
        --
        elsif l_component_type_id = c_comp_type_facet_group then
            store_component_prop (
                p_component_type_id            => l_component_type_id,
                p_property_name                => 'FC_ACTIONS_CHART',
                p_db_column                    => null,
                p_is_required                  => true,
                p_is_common                    => false,
                p_default_value                => 'Y',
                p_depending_on_comp_prop_name  => 'FC_COMPUTE_COUNTS',
                p_depending_on_condition_type  => 'EQUALS',
                p_depending_on_expression      => 'Y',
                p_depending_on_comp_prop_name2 => 'FC_DISPLAY_AS',
                p_depending_on_condition_type2 => 'EQUALS',
                p_depending_on_expression2     => 'INLINE' );
        end if;
        --
        if c_is_facet_or_group then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'FC_INITIAL_CHART',
                p_db_column                   => null,
                p_is_required                 => false,
                p_is_common                   => false,
                p_default_value               => 'N',
                p_depending_on_comp_prop_name => 'FC_ACTIONS_CHART',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'Y',
                p_depending_on_comp_prop_name2 => 'FC_DISPLAY_AS',
                p_depending_on_condition_type2 => 'EQUALS',
                p_depending_on_expression2     => 'INLINE' );
        end if;
        --
        if l_component_type_id in ( c_comp_type_facet, c_comp_type_facet_group, c_comp_type_facet_group_item ) then
            --
            store_component_prop (
                p_component_type_id            => l_component_type_id,
                p_property_name                => 'FC_COLLAPSIBLE',
                p_db_column                    => null,
                p_is_required                  => true,
                p_is_common                    => false,
                p_default_value                => 'N',
                p_depending_on_comp_prop_name  => case when c_is_facet then 'ITEM_TYPE'     end,
                p_depending_on_condition_type  => case when c_is_facet then 'NOT_EQUALS'    end,
                p_depending_on_expression      => case when c_is_facet then 'NATIVE_SEARCH' end,
                p_depending_on_comp_prop_name2 => case when c_is_facet_or_group then 'FC_DISPLAY_AS' end,
                p_depending_on_condition_type2 => case when c_is_facet_or_group then 'EQUALS' end,
                p_depending_on_expression2     => case when c_is_facet_or_group then 'INLINE' end );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'FC_INITIAL_COLLAPSED',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => false,
                p_default_value               => 'N',
                p_depending_on_comp_prop_name => 'FC_COLLAPSIBLE',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'Y' );
        end if;
        --
        if l_component_type_id in ( c_comp_type_sfilter, c_comp_type_sfilter_group ) then
            -- Suggestion Chips
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'SUGGESTIONS_TYPE',
                p_db_column                   => null,
                p_is_required                 => false,
                p_is_common                   => true,
                p_default_value               => 'DYNAMIC',
                p_null_text                   => 'None',
                p_depending_on_comp_prop_name => case when c_is_sfilter then 'ITEM_TYPE'     end,
                p_depending_on_condition_type => case when c_is_sfilter then 'NOT_EQUALS'    end,
                p_depending_on_expression     => case when c_is_sfilter then 'NATIVE_SEARCH' end );
            --
            /* $$$ Phase 2
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'SEARCH_SUGGESTIONS_TYPE',
                p_db_column                   => null,
                p_is_required                 => false,
                p_is_common                   => true );
                p_depending_on_comp_prop_name => case when c_is_sfilter then 'ITEM_TYPE'     end,
                p_depending_on_condition_type => case when c_is_sfilter then 'EQUALS'        end,
                p_depending_on_expression     => case when c_is_sfilter then 'NATIVE_SEARCH' end );
            --
            -- For the depending properties we would use
            --
            -- p_depending_on_comp_prop_name  => 'SUGGESTIONS_TYPE',
            -- p_depending_on_has_to_exist    => 'N',
            -- ...
            -- p_depending_on_comp_prop_name2 => 'SEARCH_SUGGESTIONS_TYPE',
            -- p_depending_on_has_to_exist2   => 'N',
            -- ...
            */
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'SUGGESTIONS_STATIC_VALUES',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => true,
                p_depending_on_comp_prop_name => 'SUGGESTIONS_TYPE',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'STATIC' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'SUGGESTIONS_SQL_QUERY',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => true,
                p_depending_on_comp_prop_name => 'SUGGESTIONS_TYPE',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'SQL_QUERY' );
            --
            /* $$$ phase 2
            store_func_body_comp_props (
                p_component_type_id => l_component_type_id,
                p_name_prefix       => 'SUGGESTIONS_',
                p_return_type       => 'ARRAY',
                p_type_property     => 'SUGGESTIONS_TYPE' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'MAXIMUM_SUGGESTIONS',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => false,
                p_default_value               => '1',
                p_depending_on_comp_prop_name => case when c_is_sfilter then 'ITEM_TYPE'     end,
                p_depending_on_condition_type => case when c_is_sfilter then 'NOT_EQUALS'    end,
                p_depending_on_expression     => case when c_is_sfilter then 'NATIVE_SEARCH' end );
            --
            if l_component_type_id = c_comp_type_sfilter then
                store_component_prop (
                    p_component_type_id           => l_component_type_id,
                    p_property_name               => 'MAXIMUM_SEARCH_SUGGESTIONS',
                    p_db_column                   => null,
                    p_is_required                 => true,
                    p_is_common                   => false,
                    p_default_value               => '5',
                    p_depending_on_comp_prop_name => 'ITEM_TYPE',
                    p_depending_on_condition_type => 'EQUALS',
                    p_depending_on_expression     => 'NATIVE_SEARCH' );
            end if;
            */
            --
            store_component_prop (
                p_component_type_id            => l_component_type_id,
                p_property_name                => 'SHOW_SUGGESTIONS_LABEL',
                p_db_column                    => null,
                p_is_required                  => true,
                p_is_common                    => false,
                p_default_value                => case l_component_type_id
                                                    when c_comp_type_sfilter       then 'Y'
                                                    when c_comp_type_sfilter_group then 'N'
                                                  end,
                p_depending_on_comp_prop_name  => 'SUGGESTIONS_TYPE',
                p_depending_on_condition_type  => 'NOT_NULL',
                p_depending_on_comp_prop_name2 => case when c_is_sfilter then 'ITEM_TYPE'     end,
                p_depending_on_condition_type2 => case when c_is_sfilter then 'NOT_EQUALS'    end,
                p_depending_on_expression2     => case when c_is_sfilter then 'NATIVE_SEARCH' end );
        end if;
        --
    end store_comp_prop_fc_attributes;
    --
    --
    function replace_facet_sfil_help_text (
        p_help_text varchar2 )
        return varchar2
    is
    begin
        return replace (
                   p_help_text,
                   '%component%',
                   case
                     when l_component_type_id in ( c_comp_type_facet,   c_comp_type_facet_group,   c_comp_type_facet_group_item )   then 'facet'
                     when l_component_type_id in ( c_comp_type_sfilter, c_comp_type_sfilter_group, c_comp_type_sfilter_group_item ) then 'filter'
                   end );
    end replace_facet_sfil_help_text;
    --
    --
    procedure store_comp_prop_facet_sfilter (
        p_default_item_type          in varchar2,
        p_depending_on_property_name in varchar2 )
    is
        c_cascading_lov_parent_prop constant varchar2(30) := case
                                                               when l_component_type_id in ( c_comp_type_facet,   c_comp_type_facet_group_item )   then 'CASCADING_LOV_PARENT_FACET'
                                                               when l_component_type_id in ( c_comp_type_sfilter, c_comp_type_sfilter_group_item ) then 'CASCADING_LOV_PARENT_FILTER'
                                                             end;
    begin
        if l_component_type_id in ( c_comp_type_facet_group_item, c_comp_type_sfilter_group_item ) then
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'HIDDEN_PARENT_REGION',
                p_is_required       => false,
                p_is_common         => false );
        end if;
        --
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'ITEM_NAME',
            p_db_column             => null,
            p_is_required           => true,
            p_is_common             => true,
            p_check_uniqueness      => 'APPLICATION',
            p_supports_substitution => false,
            p_default_value         => 'P#PAGE_ID#_NEW' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'ITEM_TYPE',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => true,
            p_default_value     => p_default_item_type );
        --
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'LABEL',
            p_db_column             => null,
            p_is_required           => false,
            p_is_common             => true,
            p_default_value         => '#PLEASE_CHANGE#',
            p_plugin_std_attributes => 'VISIBLE',
            p_help_text             => replace_facet_sfil_help_text( q'~<p>Enter the label for the %component%.
    The label displays on the page only if the %component% displays. The label for type <strong>Search</strong> is visually hidden, but available to assistive technology.</p>~' ));
        --
        if l_component_type_id in ( c_comp_type_facet, c_comp_type_facet_group_item ) then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'SHOW_CURRENT_FACET_LABEL',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => true,
                p_default_value               => 'Y',
                p_plugin_std_attributes       => 'VISIBLE',
                p_depending_on_comp_prop_name => 'ITEM_TYPE',
                p_depending_on_condition_type => 'NOT_EQUALS',
                p_depending_on_expression     => 'NATIVE_SEARCH' );
        end if;
        --
        store_plugin_attr_seq_offset (
            p_component_type_id => l_component_type_id );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'DISPLAY_SEQUENCE',
            p_db_column         => 'ITEM_SEQUENCE',
            p_is_required       => true,
            p_is_common         => true );
        --
        if l_component_type_id = c_comp_type_facet_group_item then
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'FACET_GROUP',
                p_db_column         => null,
                p_is_required       => true,
                p_is_common         => true );
        elsif l_component_type_id = c_comp_type_sfilter_group_item then
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'SFILTER_GROUP',
                p_db_column         => null,
                p_is_required       => true,
                p_is_common         => true );
        end if;
        --
        if l_component_type_id = c_comp_type_facet then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'FC_MAX_HEIGHT',
                p_db_column                   => null,
                p_is_required                 => false,
                p_is_common                   => false,
                p_depending_on_comp_prop_name => 'ITEM_TYPE',
                p_depending_on_condition_type => 'NOT_IN_LIST',
                p_depending_on_expression     => 'NATIVE_SEARCH,NATIVE_INPUT' );
        end if;
        --
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'ELEMENT_VALUE_PLACEHOLDER',
            p_db_column             => null,
            p_is_required           => false,
            p_is_common             => true,
            p_plugin_std_attributes => 'PLACEHOLDER' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'ICON_CSS_CLASSES',
            p_db_column                   => 'ITEM_ICON_CSS_CLASSES',
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'ITEM_TYPE',
            p_depending_on_condition_type => 'NOT_EQUALS',
            p_depending_on_expression     => 'NATIVE_SEARCH',
            p_help_text                   => replace_facet_sfil_help_text (
q'~<p>Enter the classes for the icon you want to use to identify this %component%. You may add multiple classes by separating them with spaces.</p>
            <p>If your theme uses Font APEX then review the <a rel="noopener noreferrer" target="_blank" href="https://apex.oracle.com/ut">Universal Theme Sample Application</a> to view available icons and modifiers.</p>~' ));
        --
        store_lov_type_component_props (
            p_name                  => 'FACET_LOV_TYPE',
            p_is_required           => null, -- defined by plug-in standard attributes
            p_plugin_std_attributes => 'LOV' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'LOV_SORT_DIRECTION',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'ASC',
            p_depending_on_comp_prop_name => 'FACET_LOV_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'DISTINCT' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'INCLUDE_NULL_OPTION',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'N',
            p_plugin_std_attributes       => 'FC_LOV_DISPLAY_NULL',
            p_depending_on_comp_prop_name => 'FACET_LOV_TYPE',
            p_depending_on_condition_type => 'NOT_NULL' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'NULL_DISPLAY_VALUE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'INCLUDE_NULL_OPTION',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'Y',
            p_help_text                   => q'~Enter the text to be displayed within the list NULL option at the top of this list.~' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => c_cascading_lov_parent_prop,
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_reference_on_delete         => 'WEAK_NULL',
            p_depending_on_comp_prop_name => 'FACET_LOV_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'SHARED' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'CASCADING_LOV_PARENT_COLUMN',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_supports_substitution       => false,
       --     p_plugin_std_attributes       => 'CASCADING_LOV',
            p_depending_on_comp_prop_name => c_cascading_lov_parent_prop,
            p_depending_on_condition_type => 'NOT_NULL' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'CASCADING_LOV_PARENT_REQUIRED',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_default_value               => 'Y',
            p_depending_on_comp_prop_name => c_cascading_lov_parent_prop,
            p_depending_on_condition_type => 'NOT_NULL' );
        --
        if l_component_type_id in ( c_comp_type_facet, c_comp_type_sfilter ) then
            store_comp_prop_fc_attributes;
            --
            declare
                c_is_facet  constant boolean := l_component_type_id = c_comp_type_facet;
            begin
                store_component_prop (
                    p_component_type_id            => l_component_type_id,
                    p_property_name                => 'ADVANCED_CSS_CLASSES',
                    p_db_column                    => 'ITEM_CSS_CLASSES',
                    p_is_required                  => false,
                    p_is_common                    => false,
                    p_depending_on_comp_prop_name  => 'ITEM_TYPE',
                    p_depending_on_condition_type  => 'NOT_EQUALS',
                    p_depending_on_expression      => 'NATIVE_SEARCH',
                    p_depending_on_comp_prop_name2 => case when c_is_facet then 'FC_DISPLAY_AS' end,
                    p_depending_on_condition_type2 => case when c_is_facet then 'EQUALS' end,
                    p_depending_on_expression2     => case when c_is_facet then 'INLINE' end,
                    p_help_text                    => replace_facet_sfil_help_text( q'~Enter classes to add to this %component%. You may add multiple classes by separating them with spaces.~' ));
            end;
        end if;
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'SOURCE_DB_COLUMN',
            p_db_column                   => 'SOURCE',
            p_is_required                 => true,
            p_is_common                   => true,
            p_supports_substitution       => false,
            p_depending_on_comp_prop_name => 'ITEM_TYPE',
            p_depending_on_condition_type => 'NOT_EQUALS',
            p_depending_on_expression     => 'NATIVE_SEARCH',
            p_help_text                   => replace_facet_sfil_help_text( q'~<p>Enter the case sensitive database column name used as the source for this %component%.</p>~' ));
        --
        if l_component_type_id in ( c_comp_type_facet, c_comp_type_sfilter ) then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'TEXT_SEARCH_DB_COLUMNS',
                p_db_column                   => 'SOURCE',
                p_is_required                 => false,
                p_is_common                   => true,
                p_supports_substitution       => false,
                p_depending_on_comp_prop_name => 'ITEM_TYPE',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'NATIVE_SEARCH' );
        end if;
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'DATA_TYPE',
            p_db_column                   => 'SOURCE_DATA_TYPE',
            p_is_required                 => true,
            p_is_common                   => false,
            p_supports_substitution       => false,
            p_default_value               => 'VARCHAR2',
            p_depending_on_comp_prop_name => 'ITEM_TYPE',
            p_depending_on_condition_type => 'NOT_EQUALS',
            p_depending_on_expression     => 'NATIVE_SEARCH' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'FORMAT_MASK_DATE',
            p_db_column                   => 'FORMAT_MASK',
            p_is_required                 => false,
            p_is_common                   => false,
            p_plugin_std_attributes       => 'FORMAT_MASK_DATE',
            p_depending_on_comp_prop_name => 'DATA_TYPE',
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'DATE,TIMESTAMP,TIMESTAMP_TZ,TIMESTAMP_LTZ' );
        --
        -- FORMAT_MASK_DATE is part of Display Group "Appearance" but also has a dependency
        -- to Component-Prop 'DATA_TYPE', so we have to move it up after creation
        resequence_comp_prop (
            p_component_type_id => l_component_type_id,
            p_name              => 'FORMAT_MASK_DATE',
            p_previous_name     => 'ICON_CSS_CLASSES' );
        --
        if l_component_type_id in ( c_comp_type_facet, c_comp_type_sfilter ) then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'MULTI_VALUE_TYPE',
                p_db_column                   => null,
                p_is_required                 => false,
                p_is_common                   => false,
                p_null_text                   => 'No',
                p_null_help_text              => 'Column does not contain multiple values.',
                p_depending_on_comp_prop_name => 'ITEM_TYPE',
                p_depending_on_condition_type => 'IN_LIST',
                p_depending_on_expression     => 'NATIVE_CHECKBOX,NATIVE_RADIOGROUP,NATIVE_SELECT_LIST' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'MULTI_VALUE_SEPARATOR',
                p_db_column                   => null,
                p_default_value               => ':',
                p_is_required                 => true,
                p_is_common                   => false,
                p_supports_substitution       => false,
                p_depending_on_comp_prop_name => 'MULTI_VALUE_TYPE',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'SEPARATED' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'MULTI_VALUE_TRIM_SPACE',
                p_db_column                   => null,
                p_default_value               => 'N',
                p_is_required                 => true,
                p_is_common                   => false,
                p_depending_on_comp_prop_name => 'MULTI_VALUE_TYPE',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'SEPARATED' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'FC_MULTI_VALUE_COMBINATION',
                p_is_required                 => true,
                p_default_value               => 'OR',
                p_is_common                   => false,
                p_depending_on_comp_prop_name => 'MULTI_VALUE_TYPE',
                p_depending_on_condition_type => 'NOT_NULL' );
        end if;
        --
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'DEFAULT_VALUE_TYPE',
            p_db_column             => null,
            p_is_required           => false,
            p_is_common             => true,
            p_plugin_std_attributes => 'SOURCE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'DEFAULT_VALUE_STATIC_TEXT',
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => 'DEFAULT_VALUE_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'STATIC_TEXT_WITH_SUBSTITUTIONS' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'DEFAULT_VALUE_ITEM',
            p_is_required                 => true,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'DEFAULT_VALUE_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'ITEM' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'DEFAULT_VALUE_SQL_QUERY',
            p_db_column                   => 'ITEM_DEFAULT',
            p_is_required                 => true,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'DEFAULT_VALUE_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'SQL_QUERY' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'DEFAULT_VALUE_SQL_QUERY_COLON',
            p_db_column                   => 'ITEM_DEFAULT',
            p_is_required                 => true,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'DEFAULT_VALUE_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'SQL_QUERY_COLON' );
        --
        store_expression_comp_props (
            p_component_type_id => l_component_type_id,
            p_name_prefix       => 'DEFAULT_VALUE_',
            p_return_type       => 'VARCHAR2',
            p_type_property     => 'DEFAULT_VALUE_TYPE' );
        --
        store_func_body_comp_props (
            p_component_type_id => l_component_type_id,
            p_name_prefix       => 'DEFAULT_VALUE_',
            p_return_type       => 'VARCHAR2',
            p_type_property     => 'DEFAULT_VALUE_TYPE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'DEFAULT_VALUE_SEQUENCE',
            p_db_column                   => 'ITEM_DEFAULT',
            p_is_required                 => true,
            p_is_common                   => false,
            p_supports_substitution       => false,
            p_depending_on_comp_prop_name => 'DEFAULT_VALUE_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'SEQUENCE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => p_depending_on_property_name,
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_reference_on_delete         => 'DB_NULL',
            p_depending_on_comp_prop_name => 'ITEM_TYPE',
            p_depending_on_condition_type => 'NOT_EQUALS',
            p_depending_on_expression     => 'NATIVE_SEARCH' );
        --
        store_component_prop (
            p_component_type_id             => l_component_type_id,
            p_property_name                 => 'DEPENDING_ON_CONDITION',
            p_db_column                     => null,
            p_is_required                   => true,
            p_is_common                     => false,
            p_default_value                 => 'NOT_NULL',
            p_depending_on_comp_prop_name   => p_depending_on_property_name,
            p_depending_on_condition_type   => 'NOT_NULL' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'DEPENDING_ON_COND_VALUE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'DEPENDING_ON_CONDITION',
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'EQUALS,NOT_EQUALS' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'DEPENDING_ON_COND_LIST',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'DEPENDING_ON_CONDITION',
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'IN_LIST,NOT_IN_LIST' );
        --
        if l_component_type_id = c_comp_type_sfilter then
            store_condition_type_comp_prop (
                p_component_type_id           => l_component_type_id,
                p_name_prefix                 => null,
                p_help_text_type              => null,
                p_db_column_type              => 'DISPLAY_WHEN_TYPE',
                p_view_column_type            => 'CONDITION_TYPE',
                p_db_column_expr1             => 'DISPLAY_WHEN',
                p_db_column_expr2             => 'DISPLAY_WHEN2',
                p_depending_on_comp_prop_name => 'ITEM_TYPE',
                p_depending_on_condition_type => 'NOT_EQUALS',
                p_depending_on_expression     => 'NATIVE_SEARCH' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'AUTHORIZATION_SCHEME',
                p_db_column                   => 'SECURITY_SCHEME',
                p_is_required                 => false,
                p_is_common                   => false,
                p_reference_on_delete         => 'WEAK_NULL',
                p_depending_on_comp_prop_name => 'ITEM_TYPE',
                p_depending_on_condition_type => 'NOT_EQUALS',
                p_depending_on_expression     => 'NATIVE_SEARCH' );
        else
            store_condition_type_comp_prop (
                p_component_type_id => l_component_type_id,
                p_name_prefix       => null,
                p_help_text_type    => null,
                p_db_column_type    => 'DISPLAY_WHEN_TYPE',
                p_view_column_type  => 'CONDITION_TYPE',
                p_db_column_expr1   => 'DISPLAY_WHEN',
                p_db_column_expr2   => 'DISPLAY_WHEN2' );
            --
            store_component_prop (
                p_component_type_id   => l_component_type_id,
                p_property_name       => 'AUTHORIZATION_SCHEME',
                p_db_column           => 'SECURITY_SCHEME',
                p_is_required         => false,
                p_is_common           => false,
                p_reference_on_delete => 'WEAK_NULL' );
        end if;
        --
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'ENCRYPT_SESSION_STATE',
            p_db_column             => null,
            p_is_required           => true,
            p_is_common             => false,
            p_default_value         => 'Y',
            p_plugin_std_attributes => 'ENCRYPT' );
        --
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'ESCAPE_SPECIAL_CHARACTERS',
            p_db_column             => null,
            p_is_required           => true,
            p_is_common             => false,
            p_default_value         => 'Y',
            p_plugin_std_attributes => 'ESCAPE_OUTPUT' );
        --
        store_standard_attr_comp_prop (
            p_component_type_id     => l_component_type_id,
            p_has_build_option      => true,
            p_has_comments          => true,
            p_comment_db_column     => 'ITEM_COMMENT' );
        --
    end store_comp_prop_facet_sfilter;
    --
    --
    procedure store_comp_prop_facet_sfil_grp
    is
    begin
        store_component_prop (
            p_component_type_id      => l_component_type_id,
            p_property_name          => 'LABEL',
            p_db_column              => 'LABEL',
            p_is_required            => true,
            p_is_common              => true,
            p_check_uniqueness       => 'PARENT',
            p_supports_substitution  => true,
            p_default_value          => '#PLEASE_CHANGE#',
            p_help_text              => replace_facet_sfil_help_text( '~Enter the label of the %component% group.~' ));
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'DISPLAY_SEQUENCE',
            p_db_column         => 'DISPLAY_SEQUENCE',
            p_is_required       => true,
            p_is_common         => true );
        --
        if l_component_type_id = c_comp_type_facet_group then
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'FC_MAX_HEIGHT',
                p_db_column         => null,
                p_is_required       => false,
                p_is_common         => false );
        end if;
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'ICON_CSS_CLASSES',
            p_db_column         => 'ICON_CSS_CLASSES',
            p_is_required       => false,
            p_is_common         => false,
            p_help_text         => replace_facet_sfil_help_text(
q'~<p>Enter the classes for the icon you want to use to identify this %component%. You may add multiple classes by separating them with spaces.</p>
            <p>If your theme uses Font APEX then review the <a rel="noopener noreferrer" target="_blank" href="https://apex.oracle.com/ut">Universal Theme Sample Application</a> to view available icons and modifiers.</p>~' ));
        --
        store_comp_prop_fc_attributes;
        --
        declare
            c_is_facet  constant boolean := l_component_type_id = c_comp_type_facet_group;
        begin
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'ADVANCED_CSS_CLASSES',
                p_db_column                   => 'CSS_CLASSES',
                p_is_required                 => false,
                p_is_common                   => false,
                p_depending_on_comp_prop_name => case when c_is_facet then 'FC_DISPLAY_AS' end,
                p_depending_on_condition_type => case when c_is_facet then 'EQUALS' end,
                p_depending_on_expression     => case when c_is_facet then 'INLINE' end,
                p_help_text                   => replace_facet_sfil_help_text( q'~Enter classes to add to this %component% group. You may add multiple classes by separating them with spaces.~' ) );
        end;
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'FC_FILTER_COMBINATION',
            p_is_required       => true,
            p_is_common         => false,
            p_default_value     => 'OR' );
        --
        store_standard_attr_comp_prop (
            p_component_type_id      => l_component_type_id,
            p_has_condition            => true,
            p_cond_type_db_column      => 'CONDITION_TYPE',
            p_cond_exp1_db_column      => 'CONDITION_EXPR1',
            p_cond_exp2_db_column      => 'CONDITION_EXPR2',
            p_has_authorization_scheme => true,
            p_authorization_db_column  => 'AUTHORIZATION_SCHEME',
            p_has_build_option         => true,
            p_build_option_db_column   => 'BUILD_OPTION_ID',
            p_has_comments             => true,
            p_comment_db_column        => 'GROUP_COMMENT' );
        --
    end store_comp_prop_facet_sfil_grp;
    --
    --
    procedure store_comp_prop_js_condition (
        p_db_column_type         in varchar2,
        p_db_column_element_type in varchar2,
        p_db_column_element      in varchar2,
        p_db_column_expression   in varchar2,
        p_help_text_type         in varchar2 default null )
    is
        procedure store_comp_props (
            p_is_ig in boolean )
        is
            c_postfix      constant varchar2(30) := case when p_is_ig then '_WHEN_IG' end;
            c_js_condition constant varchar2(30) := 'JS_CONDITION' || c_postfix;
        begin
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => c_js_condition,
                p_db_column                   => p_db_column_type,
                p_is_required                 => false,
                p_is_common                   => true,
                p_depending_on_comp_prop_name => 'IG_REGION',
                p_depending_on_condition_type => case when p_is_ig then 'NOT_NULL' else 'NULL' end,
                p_help_text                   => p_help_text_type );
            --
            if p_is_ig then
                store_component_prop (
                    p_component_type_id           => l_component_type_id,
                    p_property_name               => 'JS_CONDITION_ELEMENT_TYPE',
                    p_db_column                   => p_db_column_element_type,
                    p_is_required                 => true,
                    p_is_common                   => true,
                    p_default_value               => 'COLUMN',
                    p_depending_on_comp_prop_name => 'JS_CONDITION_WHEN_IG',
                    p_depending_on_condition_type => 'IN_LIST',
                    p_depending_on_expression     => 'EQUALS,NOT_EQUALS,GREATER_THAN,GREATER_THAN_OR_EQUAL,LESS_THAN,LESS_THAN_OR_EQUAL,NULL,NOT_NULL,IN_LIST,NOT_IN_LIST' );
                --
                store_component_prop (
                    p_component_type_id           => l_component_type_id,
                    p_property_name               => 'JS_CONDITION_ITEM_WHEN_IG',
                    p_db_column                   => p_db_column_element,
                    p_is_required                 => true,
                    p_is_common                   => true,
                    p_depending_on_comp_prop_name => 'JS_CONDITION_ELEMENT_TYPE',
                    p_depending_on_condition_type => 'EQUALS',
                    p_depending_on_expression     => 'ITEM' );
                --
                store_component_prop (
                    p_component_type_id           => l_component_type_id,
                    p_property_name               => 'JS_CONDITION_COLUMN_WHEN_IG',
                    p_db_column                   => p_db_column_element,
                    p_is_required                 => true,
                    p_is_common                   => true,
                    p_parent_comp_prop_name       => 'IG_REGION',
                    p_reference_on_delete         => 'WEAK_CASCADE',
                    p_depending_on_comp_prop_name => 'JS_CONDITION_ELEMENT_TYPE',
                    p_depending_on_condition_type => 'EQUALS',
                    p_depending_on_expression     => 'COLUMN' );
            else
                store_component_prop (
                    p_component_type_id           => l_component_type_id,
                    p_property_name               => 'JS_CONDITION_ITEM' || c_postfix,
                    p_db_column                   => p_db_column_element,
                    p_is_required                 => true,
                    p_is_common                   => true,
                    p_depending_on_comp_prop_name => c_js_condition,
                    p_depending_on_condition_type => 'IN_LIST',
                    p_depending_on_expression     => 'EQUALS,NOT_EQUALS,GREATER_THAN,GREATER_THAN_OR_EQUAL,LESS_THAN,LESS_THAN_OR_EQUAL,NULL,NOT_NULL,IN_LIST,NOT_IN_LIST',
                    p_help_text                   => q'~<p>Enter the page item used in this condition. You can type in the name or pick from the list of available items.</p>~' );
            end if;
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'JS_COND_VALUE' || c_postfix,
                p_db_column                   => p_db_column_expression,
                p_is_required                 => true,
                p_is_common                   => true,
                p_depending_on_comp_prop_name => c_js_condition,
                p_depending_on_condition_type => 'IN_LIST',
                p_depending_on_expression     => 'EQUALS,NOT_EQUALS,GREATER_THAN,GREATER_THAN_OR_EQUAL,LESS_THAN,LESS_THAN_OR_EQUAL' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'JS_COND_LIST' || c_postfix,
                p_db_column                   => p_db_column_expression,
                p_is_required                 => true,
                p_is_common                   => true,
                p_depending_on_comp_prop_name => c_js_condition,
                p_depending_on_condition_type => 'IN_LIST',
                p_depending_on_expression     => 'IN_LIST,NOT_IN_LIST' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'JS_COND_JS_EXPRESSION' || c_postfix,
                p_db_column                   => p_db_column_expression,
                p_is_required                 => true,
                p_is_common                   => true,
                p_depending_on_comp_prop_name => c_js_condition,
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'JAVASCRIPT_EXPRESSION' );
        end store_comp_props;
    begin
        -- for page items
        store_comp_props (
            p_is_ig => false );
        -- for IG columns
        store_comp_props (
            p_is_ig => true );
    end store_comp_prop_js_condition;
    --
    --
    procedure store_comp_prop_remote_server (
        p_server_type in varchar2 )
    is
        c_server_type_desc constant varchar2( 255 ) := case p_server_type
                                                       when 'REMOTE_SQL'     then 'REST Enabled SQL Service'
                                                       when 'AUTHENTICATION' then 'Authentication Server'
                                                       when 'PRINT_SERVER'   then 'Print Server'
                                                       when 'WEB_SOURCE'     then 'REST Data Source Server'
                                                       when 'GENERATIVE_AI'  then 'Generative AI Service'
                                                       end;
        function get_help (
            p_help_text in varchar2 )
            return varchar2
        is
        begin
            return replace( p_help_text, '%type%', c_server_type_desc );
        end get_help;
    begin
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'NAME',
            p_db_column             => null,
            p_is_required           => true,
            p_is_common             => true,
            p_check_uniqueness      => 'WORKSPACE',
            p_supports_substitution => false,
            p_default_value         => '#PLEASE_CHANGE#',
            p_help_text             => get_help( q'~Enter a descriptive name for this %type%.~' ));
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'REMOTE_SERVER_TYPE',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => true,
            p_default_value     => p_server_type );
        --
        if p_server_type = 'PRINT_SERVER' then
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'PRINT_SERVER_TYPE',
                p_db_column         => null,
                p_is_required       => true,
                p_is_common         => true,
                p_default_value     => 'BIP' );
            end if;
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'ENDPOINT_URL',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => true,
            p_help_text         => case p_server_type
                                   when 'REMOTE_SQL' then q'~Enter the endpoint URL for this REST Enabled SQL Service. Include the ORDS context root and schema URL prefix, e.g. <em>https://{host}:{port}/ords/{schema}</em>. Do not append /_/sql; Application Express will take care of that.~'
                                   else get_help( q'~Enter the endpoint URL for this %type%.~' )
                                   end
            );
        --
        if p_server_type = 'REMOTE_SQL' then
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'CREDENTIALS',
                p_db_column         => null,
                p_is_required       => true,
                p_is_common         => true );
            --
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'DB_SESSION_INIT',
                p_db_column         => 'ORDS_INIT_CODE',
                p_is_required       => false,
                p_is_common         => false,
                p_help_text         => q'~Enter code to be executed immediately after connecting to the REST Enabled SQL Service and before the component SQL is being executed.~' );
            --
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'DB_SESSION_CLEANUP',
                p_db_column         => 'ORDS_CLEANUP_CODE',
                p_is_required       => false,
                p_is_common         => false,
                p_help_text         => q'~Enter code to be executed immediately after the component SQL is being executed.~' );
        end if;
        --
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'STATIC_ID',
            p_db_column             => null,
            p_is_required           => true,
            p_is_common             => true,
            p_check_uniqueness      => 'WORKSPACE',
            p_supports_substitution => false,
            p_help_text             => get_help( q'~Use the Static ID to reference the %type% in API Calls. Static IDs are also used to identify an existing %type% when the application is being exported and imported to another workspace.~' ));
        --
        if p_server_type in ( 'REMOTE_SQL', 'AUTHENTICATION' ) then
            store_component_prop (
                p_component_type_id  => l_component_type_id,
                p_property_name      => 'RS_PLSQL_CODE',
                p_db_column          => null,
                p_is_required        => false,
                p_is_common          => false,
                p_help_text          => q'~A procedure which configures the Endpoint URL at runtime can be defined here.</p>~',
                p_examples           => q'~<p>Some examples:</p>
<pre><code>
procedure my_server_config(
    p_info   in  apex_plugin.t_remote_server_info,
    p_config out apex_plugin.t_remote_server_config )
is
begin
    if p_info.application_id = 100
    then
      p_config.base_url := 'https://#cust#.example.com';
      p_config.substitutions := apex_t_varchar2();
      apex_string.plist_put( p_config.substitutions, 'cust', v('P3_CUSTOMER') );
    else
      p_config.base_url := 'https://test.example.com';
    end if;
end;

procedure my_server_config(
    p_info   in  apex_plugin.t_remote_server_info,
    p_config out apex_plugin.t_remote_server_config )
is
begin
  p_config.base_url := sys.owa_util.get_cgi_env('HTTP_HOST');
end;

procedure my_server_config(
    p_info   in  apex_plugin.t_remote_server_info,
    p_config out apex_plugin.t_remote_server_config )
is
begin
    -- expect endpoint url to be #host#/my_service
    p_config.substitutions := apex_t_varchar2();
    apex_string.plist_put( p_config.substitutions, 'host', sys.owa_util.get_cgi_env('HTTP_HOST') );
end;
</code></pre>~' );
            --
            store_component_prop (
                p_component_type_id  => l_component_type_id,
                p_property_name      => 'RS_CONFIGURATION_PROCEDURE',
                p_db_column          => null,
                p_is_required        => false,
                p_is_common          => false,
                p_help_text          => q'~<p>Enter the name of a procedure which configures the Endpoint URL at runtime.
This can be a stored procedure or a procedure defined in item <em>PL/SQL Code</em>.
The code below is for an application that implements multi-tenancy and supports
different variations of <em>Endpoint URL</em> for the tenants.
It changes the domain name in the URL, depending on the customer (Example URLs:
https://cust-01.example.com, https://cust-02.example.com):
</p>~',
                p_examples           => q'~<pre><code>
procedure my_server_config(
    p_info   in  apex_plugin.t_remote_server_info,
    p_config out apex_plugin.t_remote_server_config )
is
begin
    if p_info.application_id = 100
    then
      p_config.base_url := 'https://#cust#.example.com';
      p_config.substitutions := apex_t_varchar2();
      apex_string.plist_put( p_config.substitutions, 'cust', v('P3_CUSTOMER') );
    else
      p_config.base_url := 'https://test.example.com';
    end if;
end;
</code></pre><p>
The procedure can change 2 attributes of the in/out parameter <code>p_config</code>,
both of them are optional:
<ul>
    <li><code>base_url</code>:
    Changes the <em>Endpoint URL</em>.
    </li>
    <li><code>substitutions</code>:
    Assign name/value pairs using <code>apex_t_varchar2</code>.
    &PRODUCT_NAME. substitutes each <code>#NAME#</code> in the <em>base url</em> or <em>Endpoint URL</em>.
    </li>
</ul>
</p>~' );
        end if;
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'PROMPT_ON_INSTALL',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => false,
            p_default_value     => 'N',
            p_help_text         => get_help( q'~Choose whether prompts for this %type% should be displayed when the application is being imported on another Application Express instance.~' ));
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'HTTPS_HOST_NAME',
            p_db_column         => null,
            p_is_required       => false,
            p_is_common         => false,
            p_help_text         => q'~The host name to be matched against the common name (CN) of the remote server's certificate for an HTTPS request. It can also be a domain name like *.example.com. If NULL, the host name in the given URL will be used.~' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'SERVER_TIMEOUT',
            p_db_column         => null,
            p_is_required       => false,
            p_is_common         => false,
            p_default_value     => '300' );
        --
        if p_server_type = 'REMOTE_SQL' then
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'SERVER_TIME_ZONE',
                p_db_column         => null,
                p_is_required       => false,
                p_is_common         => false );
            --
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'REMOTE_SQL_DATABASE_TYPE',
                p_db_column         => null,
                p_is_required       => false,
                p_is_read_only      => true,
                p_is_common         => false );
        end if;
        --
        --
        if p_server_type = 'GENERATIVE_AI' then
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'AI_PROVIDER_TYPE',
                p_db_column         => null,
                p_is_required       => true,
                p_is_common         => false );
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'AI_IS_BUILDER_SERVICE',
                p_db_column         => null,
                p_is_required       => true,
                p_default_value     => 'N',
            p_is_common         => false );
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'AI_MODEL_NAME',
                p_db_column         => null,
                p_is_required       => false,
                p_is_common         => false );
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'AI_HTTP_HEADERS',
                p_db_column         => null,
                p_is_required       => false,
                p_is_common         => false );
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'AI_ATTRIBUTES',
                p_db_column         => null,
                p_is_required       => false,
                p_is_common         => false );
        end if;
        --
        if p_server_type = 'VECTOR' then
            store_component_prop (
                p_component_type_id            => l_component_type_id,
                p_property_name                => 'EMBEDDING_TYPE',
                p_db_column                    => 'EMBEDDING_TYPE',
                p_is_required                  => true,
                p_is_common                    => true );
            store_component_prop (
                p_component_type_id            => l_component_type_id,
                p_property_name                => 'EMB_LOCAL_MODEL_OWNER',
                p_db_column                    => 'EMB_LOCAL_MODEL_OWNER',
                p_is_required                  => true,
                p_is_common                    => true,
                p_depending_on_comp_prop_name  => 'EMBEDDING_TYPE',
                p_depending_on_condition_type  => 'EQUALS',
                p_depending_on_expression      => 'ONNX' );
            store_component_prop (
                p_component_type_id            => l_component_type_id,
                p_property_name                => 'EMB_LOCAL_MODEL_NAME',
                p_db_column                    => 'EMB_LOCAL_MODEL_NAME',
                p_is_required                  => true,
                p_is_common                    => true,
                p_depending_on_comp_prop_name  => 'EMBEDDING_TYPE',
                p_depending_on_condition_type  => 'EQUALS',
                p_depending_on_expression      => 'ONNX' );
            store_component_prop (
                p_component_type_id            => l_component_type_id,
                p_property_name                => 'EMB_FUNCTION',
                p_db_column                    => 'EMB_FUNCTION',
                p_is_required                  => true,
                p_is_common                    => true,
                p_depending_on_comp_prop_name  => 'EMBEDDING_TYPE',
                p_depending_on_condition_type  => 'EQUALS',
                p_depending_on_expression      => 'PLSQL' );

            store_component_prop (
                p_component_type_id            => l_component_type_id,
                p_property_name                => 'AI_PROVIDER_TYPE',
                p_db_column                    => null,
                p_is_required                  => true,
                p_is_common                    => false,
                p_depending_on_comp_prop_name  => 'EMBEDDING_TYPE',
                p_depending_on_condition_type  => 'EQUALS',
                p_depending_on_expression      => 'GENAI_PROVIDER' );
            store_component_prop (
                p_component_type_id            => l_component_type_id,
                p_property_name                => 'AI_IS_BUILDER_SERVICE',
                p_db_column                    => null,
                p_is_required                  => true,
                p_default_value                => 'N',
            p_is_common                        => false,
                p_depending_on_comp_prop_name  => 'EMBEDDING_TYPE',
                p_depending_on_condition_type  => 'EQUALS',
                p_depending_on_expression      => 'GENAI_PROVIDER' );
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'AI_MODEL_NAME',
                p_db_column         => null,
                p_is_required       => false,
                p_is_common         => false,
                p_depending_on_comp_prop_name  => 'EMBEDDING_TYPE',
                p_depending_on_condition_type  => 'EQUALS',
                p_depending_on_expression      => 'GENAI_PROVIDER' );
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'AI_HTTP_HEADERS',
                p_db_column         => null,
                p_is_required       => false,
                p_is_common         => false,
                p_depending_on_comp_prop_name  => 'EMBEDDING_TYPE',
                p_depending_on_condition_type  => 'EQUALS',
                p_depending_on_expression      => 'GENAI_PROVIDER' );
            store_component_prop (
                p_component_type_id => l_component_type_id,
                p_property_name     => 'AI_ATTRIBUTES',
                p_db_column         => null,
                p_is_required       => false,
                p_is_common         => false,
                p_depending_on_comp_prop_name  => 'EMBEDDING_TYPE',
                p_depending_on_condition_type  => 'EQUALS',
                p_depending_on_expression      => 'GENAI_PROVIDER' );
        end if;
        --
        store_standard_attr_comp_prop (
            p_component_type_id => l_component_type_id,
            p_has_comments      => true,
            p_comment_db_column => 'SERVER_COMMENT' );
        --
    end store_comp_prop_remote_server;
    --
    --
    procedure store_comp_prop_static_file
    is
    begin
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'FILE_NAME',
            p_db_column             => null,
            p_is_required           => true,
            p_is_common             => true,
            p_check_uniqueness      => case l_component_type_id
                                         when c_comp_type_workspace_file   then 'WORKSPACE'
                                         when c_comp_type_application_file then 'APPLICATION'
                                         when c_comp_type_plugin_file      then 'PLUGIN'
                                         when c_comp_type_theme_file       then 'THEME'
                                       end,
            p_supports_substitution => false,
            p_help_text             => case l_component_type_id
                                         when c_comp_type_workspace_file   then null -- use property level help
                                         when c_comp_type_application_file then
                                             q'~This is the name of the file in the application file repository. Use <strong><code>&#35;APP_IMAGES#filename</code></strong> in your application to reference a file.~'
                                         else
                                             q'~The name of the file.~'
                                       end
            );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'MIME_TYPE',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => true );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'FILE_CHARACTER_SET',
            p_db_column         => null,
            p_is_required       => false,
            p_is_common         => true ); -- $$$ depending on upper(:P315_MIME_TYPE) like 'TEXT/%' or upper(:P315_MIME_TYPE) like '%JAVASCRIPT%'
        --
        store_audit_info_comp_prop (
            p_component_type_id => l_component_type_id );
    end store_comp_prop_static_file;
    --
    --
    procedure store_comp_prop_location (
        p_location_property_name      in varchar2,
        p_query_type_property_name    in varchar2,
        p_db_column_query_type        in varchar2,
        p_db_column_source_lang       in varchar2,
        p_db_column_source            in varchar2,
        p_has_items_to_submit         in boolean,
        p_db_column_items_to_submit   in varchar2,
        p_has_include_rowid           in boolean,
        p_has_order_by                in boolean,
        p_has_external_order_by       in boolean default null,
        p_has_order_by_item           in boolean,
        p_has_web_src_has_local_table in boolean,
        p_has_remote_caching          in boolean,
        p_examples_sql_query          in varchar2,
        p_help_text_items_to_submit   in varchar2,
        p_is_region                   in boolean  default false,
        p_depending_on_comp_prop_name in varchar2 default null,
        p_depending_on_condition_type in varchar2 default null,
        p_depending_on_expression     in varchar2 default null )
    is
        c_is_automation constant boolean := ( l_component_type_id = c_comp_type_automation );
    begin
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => p_location_property_name,
            p_db_column                   => null,
            p_is_required                 => case when p_is_region then null else true end, -- defined by plug-in standard attributes
            p_is_common                   => false,
            p_default_value               => 'LOCAL',
            p_plugin_std_attributes       => case when p_is_region then 'SOURCE_LOCATION' end,
            p_null_text                   => case when p_is_region then 'None' end,
            p_null_help_text              => case when p_is_region then 'No data source.' end,
            p_depending_on_comp_prop_name => p_depending_on_comp_prop_name,
            p_depending_on_condition_type => p_depending_on_condition_type,
            p_depending_on_expression     => p_depending_on_expression );
        --
        if c_is_automation then
            -- Initiated = Function Body Returning Boolean
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'PLSQL_PROCESS_LOCATION',
                p_db_column                   => 'LOCATION',
                p_is_required                 => true,
                p_is_common                   => false,
                p_default_value               => 'LOCAL',
                p_depending_on_comp_prop_name => 'ACTIONS_INITIATED_ON',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'FUNC_BODY_RETURNING_BOOLEAN' );
            --
            store_component_prop (
                p_component_type_id            => l_component_type_id,
                p_property_name                => 'SOURCE_REMOTE_DATABASE', -- todo $$$ should be SOURCE_REMOTE_ORACLE_DATABASE, but only for pl/sql. Has to be checked again.
                p_db_column                    => null,
                p_is_required                  => true,
                p_is_common                    => true,
                p_depending_on_comp_prop_name  => p_location_property_name,
                p_depending_on_condition_type  => 'EQUALS',
                p_depending_on_has_to_exist    => false,
                p_depending_on_expression      => 'REMOTE',
                p_depending_on_comp_prop_name2 => 'PLSQL_PROCESS_LOCATION',
                p_depending_on_condition_type2 => 'EQUALS',
                p_depending_on_has_to_exist2   => false,
                p_depending_on_expression2     => 'REMOTE' );
            --
            store_func_body_comp_props (
                p_component_type_id           => l_component_type_id,
                p_name_prefix                 => 'AUTOM_',
                p_return_type                 => 'BOOLEAN',
                p_db_column_lang              => 'FUNCTION_BODY_LANGUAGE',
                p_db_column_func_body         => 'QUERY_SOURCE',
                p_type_property               => 'ACTIONS_INITIATED_ON',
                p_type_value                  => 'FUNC_BODY_RETURNING_BOOLEAN' );
            --
        else
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'SOURCE_REMOTE_DATABASE',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => true,
                p_depending_on_comp_prop_name => p_location_property_name,
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'REMOTE' );
        end if;
        --
        if p_has_include_rowid then
            store_component_prop (
                p_component_type_id            => l_component_type_id,
                p_property_name                => 'SOURCE_REMOTE_DATABASE_TYPE',
                p_db_column                    => null,
                p_is_required                  => false,
                p_is_common                    => false,
                p_depending_on_comp_prop_name  => p_location_property_name,
                p_depending_on_condition_type  => 'EQUALS',
                p_depending_on_expression      => 'REMOTE',
                p_depending_on_comp_prop_name2 => 'SOURCE_REMOTE_DATABASE',
                p_depending_on_condition_type2 => 'NOT_NULL' );
        end if;
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => p_query_type_property_name,
            p_db_column                    => p_db_column_query_type,
            p_is_required                  => true,
            p_is_common                    => true,
            p_default_value                => 'TABLE',
            p_depending_on_comp_prop_name  => p_location_property_name,
            p_depending_on_condition_type  => 'IN_LIST',
            p_depending_on_expression      => 'LOCAL,REMOTE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'SOURCE_QUERY_OWNER',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => p_query_type_property_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'TABLE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'SOURCE_QUERY_TABLE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => p_query_type_property_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'TABLE',
            p_parent_comp_prop_name       => 'SOURCE_QUERY_OWNER' );
        --
        if p_has_include_rowid then
            store_component_prop (
                p_component_type_id            => l_component_type_id,
                p_property_name                => 'INCLUDE_ROWID_COLUMN',
                p_db_column                    => null,
                p_is_required                  => true,
                p_is_common                    => false,
                p_default_value                => 'N',
                p_depending_on_comp_prop_name  => p_query_type_property_name,
                p_depending_on_condition_type  => 'EQUALS',
                p_depending_on_expression      => 'TABLE',
                p_depending_on_comp_prop_name2 => 'SOURCE_REMOTE_DATABASE_TYPE',
                p_depending_on_condition_type2 => 'IN_LIST',
                p_depending_on_expression2     => 'ORACLE,$NULL$',
                p_depending_on_has_to_exist2   => false );
        end if;
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'SOURCE_QUERY_WHERE',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => p_query_type_property_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'TABLE',
            p_parent_comp_prop_name       => 'SOURCE_QUERY_TABLE' );
        --
        -- Property Graph properties
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'SOURCE_GRAPH_OWNER',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => p_query_type_property_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'GRAPH' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'SOURCE_GRAPH_NAME',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => p_query_type_property_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'GRAPH',
            p_parent_comp_prop_name       => 'SOURCE_GRAPH_OWNER' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'SOURCE_GRAPH_MATCH_CLAUSE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_supports_substitution       => false,
            p_depending_on_comp_prop_name => p_query_type_property_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'GRAPH',
            p_parent_comp_prop_name       => 'SOURCE_GRAPH_NAME' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'SOURCE_GRAPH_COLUMNS_CLAUSE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_supports_substitution       => false,
            p_depending_on_comp_prop_name => p_query_type_property_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'GRAPH',
            p_parent_comp_prop_name       => 'SOURCE_GRAPH_MATCH_CLAUSE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'SOURCE_GRAPH_WHERE_CLAUSE',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_supports_substitution       => false,
            p_depending_on_comp_prop_name => p_query_type_property_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'GRAPH',
            p_parent_comp_prop_name       => 'SOURCE_GRAPH_MATCH_CLAUSE' );
        --
        if p_has_order_by then
            if p_is_region then
                store_component_prop (
                    p_component_type_id            => l_component_type_id,
                    p_property_name                => 'SOURCE_QUERY_STATIC_ORDER_BY',
                    p_db_column                    => null,
                    p_is_required                  => false,
                    p_is_common                    => false,
                    p_depending_on_comp_prop_name  => p_query_type_property_name,
                    p_depending_on_condition_type  => 'EQUALS',
                    p_depending_on_expression      => 'TABLE',
                    p_depending_on_comp_prop_name2 => 'REGION_TYPE',
                    p_depending_on_condition_type2 => 'IN_LIST',
                    p_depending_on_expression2     => 'NATIVE_IG,NATIVE_IR,NATIVE_MAP_REGION,NATIVE_FORM',
                    p_parent_comp_prop_name        => 'SOURCE_QUERY_TABLE' );
                --
                store_component_prop (
                    p_component_type_id            => l_component_type_id,
                    p_property_name                => 'SOURCE_GRAPH_STATIC_ORDER_BY',
                    p_db_column                    => null,
                    p_is_required                  => false,
                    p_is_common                    => false,
                    p_depending_on_comp_prop_name  => p_query_type_property_name,
                    p_depending_on_condition_type  => 'EQUALS',
                    p_depending_on_expression      => 'GRAPH',
                    p_depending_on_comp_prop_name2 => 'REGION_TYPE',
                    p_depending_on_condition_type2 => 'IN_LIST',
                    p_depending_on_expression2     => 'NATIVE_IG,NATIVE_IR,NATIVE_MAP_REGION,NATIVE_FORM',
                    p_parent_comp_prop_name        => 'SOURCE_GRAPH_COLUMNS_CLAUSE' );
                --
            elsif not p_has_order_by_item then
                store_component_prop (
                    p_component_type_id            => l_component_type_id,
                    p_property_name                => 'SOURCE_QUERY_STATIC_ORDER_BY',
                    p_db_column                    => null,
                    p_is_required                  => false,
                    p_is_common                    => false,
                    p_depending_on_comp_prop_name  => p_query_type_property_name,
                    p_depending_on_condition_type  => 'EQUALS',
                    p_depending_on_expression      => 'TABLE',
                    p_parent_comp_prop_name        => 'SOURCE_QUERY_TABLE' );
                --
                store_component_prop (
                    p_component_type_id            => l_component_type_id,
                    p_property_name                => 'SOURCE_GRAPH_STATIC_ORDER_BY',
                    p_db_column                    => null,
                    p_is_required                  => false,
                    p_is_common                    => false,
                    p_depending_on_comp_prop_name  => p_query_type_property_name,
                    p_depending_on_condition_type  => 'EQUALS',
                    p_depending_on_expression      => 'GRAPH',
                    p_parent_comp_prop_name        => 'SOURCE_GRAPH_COLUMNS_CLAUSE' );
            end if;
        end if;
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'SOURCE_SQL_QUERY',
            p_db_column                   => p_db_column_source,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => p_query_type_property_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'SQL',
            p_examples                    => p_examples_sql_query );
        --
        if p_query_type_property_name = 'LOV_QUERY_TYPE' then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'LOV_LEGACY_SQL_QUERY',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => true,
                p_depending_on_comp_prop_name => p_query_type_property_name,
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'LEGACY_SQL' );
        end if;
        --
        store_func_body_comp_props (
            p_component_type_id   => l_component_type_id,
            p_name_prefix         => 'SOURCE_QUERY_',
            p_return_type         => 'SQL',
            p_db_column_lang      => p_db_column_source_lang,
            p_db_column_func_body => p_db_column_source,
            p_type_property       => p_query_type_property_name,
            p_type_value          => 'FUNC_BODY_RETURNING_SQL' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'SOURCE_WEB_SRC_QUERY',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => p_location_property_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'WEB_SOURCE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'DUALITY_VIEW_SRC_ID',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => p_location_property_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'DUALITY_VIEW' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'JSON_COLLECTION_SRC_ID',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => p_location_property_name,
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'JSON_COLLECTION' );
        --
        if p_is_region then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'SOURCE_LIST',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => true,
                p_depending_on_comp_prop_name => 'REGION_TYPE',
                p_depending_on_condition_type => 'IN_LIST',
                p_depending_on_expression     => 'NATIVE_LIST' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'SOURCE_BREADCRUMB',
                p_db_column                   => 'MENU_ID',
                p_is_required                 => true,
                p_is_common                   => true,
                p_depending_on_comp_prop_name => 'REGION_TYPE',
                p_depending_on_condition_type => 'IN_LIST',
                p_depending_on_expression     => 'NATIVE_BREADCRUMB' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'SOURCE_FILTERED_REGION',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => true,
                p_reference_on_delete         => 'WEAK_CASCADE',
                p_depending_on_comp_prop_name => 'REGION_TYPE',
                p_depending_on_condition_type => 'IN_LIST',
                p_depending_on_expression     => 'NATIVE_FACETED_SEARCH,NATIVE_SMART_FILTERS' );
        end if;
        --
        if p_is_region then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'SOURCE_PAGE_ITEMS_TO_SUBMIT',
                p_db_column                   => p_db_column_items_to_submit,
                p_is_required                 => false,
                p_is_common                   => true,
        /* $$$ has to be investigated, doesn't work in wwv_flow_property_dev.clear_property_values the reason might be that SOURCE_LOCATION has a higher property id and
               that's why the order in which the properties are processed causes the issue

                p_depending_on_comp_prop_name => p_location_property_name,
                p_depending_on_has_to_exist   => false,
                p_depending_on_condition_type => 'NOT_NULL',
        */
                p_plugin_std_attributes       => case when p_is_region then 'AJAX_ITEMS_TO_SUBMIT' end,
                p_help_text                   => p_help_text_items_to_submit );
        elsif p_has_items_to_submit then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'SOURCE_PAGE_ITEMS_TO_SUBMIT',
                p_db_column                   => p_db_column_items_to_submit,
                p_is_required                 => false,
                p_is_common                   => true,
                p_depending_on_comp_prop_name => p_location_property_name,
                p_depending_on_condition_type => 'NOT_EQUALS',
                p_depending_on_expression     => 'REGION_SOURCE',
                p_help_text                   => p_help_text_items_to_submit );
        end if;
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'OPTIMIZER_HINT',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_supports_substitution       => false,
            p_depending_on_comp_prop_name => p_location_property_name,
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'LOCAL,REMOTE,WEB_SOURCE,DUALITY_VIEW,JSON_COLLECTION' );
        --
        if p_is_region then
            store_component_prop (
                p_component_type_id            => l_component_type_id,
                p_property_name                => 'HAS_GENERIC_COLUMNS',
                p_db_column                    => null,
                p_is_required                  => true,
                p_is_common                    => false,
                p_default_value                => 'DERIVED_REPORT_COLUMNS',
                p_depending_on_comp_prop_name  => 'REGION_TYPE',
                p_depending_on_condition_type  => 'EQUALS',
                p_depending_on_expression      => 'NATIVE_SQL_REPORT',
                p_depending_on_comp_prop_name2 => p_query_type_property_name,
                p_depending_on_condition_type2 => 'EQUALS',
                p_depending_on_expression2     => 'FUNC_BODY_RETURNING_SQL' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'GENERIC_COLUMN_COUNT',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => false,
                p_depending_on_comp_prop_name => 'HAS_GENERIC_COLUMNS',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'GENERIC_REPORT_COLUMNS' );
        end if;
        --
        if p_has_order_by_item then
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'SOURCE_QUERY_ORDER_BY_TYPE',
                p_db_column                   => null,
                p_is_required                 => false,
                p_is_common                   => true,
                p_null_text                   => 'None',
                p_plugin_std_attributes       => 'ORDER_BY',
                p_depending_on_comp_prop_name => p_query_type_property_name,
                p_depending_on_condition_type => 'IN_LIST',
                p_depending_on_expression     => 'TABLE,GRAPH,SQL,FUNC_BODY_RETURNING_SQL' );
            --
            store_component_prop (
                p_component_type_id            => l_component_type_id,
                p_property_name                => 'SOURCE_QUERY_ORDER_BY',
                p_db_column                    => null,
                p_is_required                  => true,
                p_is_common                    => false,
                p_depending_on_comp_prop_name  => 'SOURCE_QUERY_ORDER_BY_TYPE',
                p_depending_on_condition_type  => 'EQUALS',
                p_depending_on_expression      => 'STATIC',
                --
                p_depending_on_comp_prop_name2 => p_query_type_property_name,
                p_depending_on_condition_type2 => 'IN_LIST',
                p_depending_on_expression2     => 'TABLE,SQL,FUNC_BODY_RETURNING_SQL' );
            --
            store_component_prop (
                p_component_type_id            => l_component_type_id,
                p_property_name                => 'SOURCE_QUERY_ORDER_BY_ITEM',
                p_db_column                    => null,
                p_is_required                  => true,
                p_is_common                    => false,
                p_depending_on_comp_prop_name  => 'SOURCE_QUERY_ORDER_BY_TYPE',
                p_depending_on_condition_type  => 'EQUALS',
                p_depending_on_expression      => 'ITEM',
                --
                p_depending_on_comp_prop_name2 => p_query_type_property_name,
                p_depending_on_condition_type2 => 'IN_LIST',
                p_depending_on_expression2     => 'TABLE,SQL,FUNC_BODY_RETURNING_SQL' );
            --
            store_component_prop (
                p_component_type_id            => l_component_type_id,
                p_property_name                => 'SOURCE_GRAPH_ORDER_BY',
                p_db_column                    => null,
                p_is_required                  => true,
                p_is_common                    => false,
                p_depending_on_comp_prop_name  => 'SOURCE_QUERY_ORDER_BY_TYPE',
                p_depending_on_condition_type  => 'EQUALS',
                p_depending_on_expression      => 'STATIC',
                --
                p_depending_on_comp_prop_name2 => p_query_type_property_name,
                p_depending_on_condition_type2 => 'EQUALS',
                p_depending_on_expression2     => 'GRAPH',
                --
                p_parent_comp_prop_name        => 'SOURCE_GRAPH_COLUMNS_CLAUSE' );
            --
            store_component_prop (
                p_component_type_id            => l_component_type_id,
                p_property_name                => 'SOURCE_GRAPH_ORDER_BY_ITEM',
                p_db_column                    => null,
                p_is_required                  => true,
                p_is_common                    => false,
                p_depending_on_comp_prop_name  => 'SOURCE_QUERY_ORDER_BY_TYPE',
                p_depending_on_condition_type  => 'EQUALS',
                p_depending_on_expression      => 'ITEM',
                --
                p_depending_on_comp_prop_name2 => p_query_type_property_name,
                p_depending_on_condition_type2 => 'EQUALS',
                p_depending_on_expression2     => 'GRAPH',
                --
                p_parent_comp_prop_name        => 'SOURCE_GRAPH_COLUMNS_CLAUSE' );
        end if;
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'WEB_SRC_HAS_ARRAY_COLUMNS',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'Y',
            p_depending_on_comp_prop_name => 'SOURCE_WEB_SRC_QUERY',
            p_depending_on_condition_type => 'NOT_NULL' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'SOURCE_WEB_SRC_ARRAY_COL',
            p_db_column                    => null,
            p_is_required                  => false,
            p_null_text                    => 'None',
            p_is_common                    => false,
            p_parent_comp_prop_name        => 'SOURCE_WEB_SRC_QUERY',
            p_depending_on_comp_prop_name  => 'SOURCE_WEB_SRC_QUERY',
            p_depending_on_condition_type  => 'NOT_NULL',
            p_depending_on_comp_prop_name2 => 'WEB_SRC_HAS_ARRAY_COLUMNS',
            p_depending_on_condition_type2 => 'EQUALS',
            p_depending_on_expression2     => 'Y' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'DOC_SRC_HAS_ARRAY_COLUMNS',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'Y',
            p_depending_on_comp_prop_name => p_location_property_name,
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'DUALITY_VIEW,JSON_COLLECTION' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'DUALITY_VIEW_ARRAY_COL_ID',
            p_db_column                    => null,
            p_is_required                  => false,
            p_null_text                    => 'None',
            p_is_common                    => false,
            p_parent_comp_prop_name        => 'DUALITY_VIEW_SRC_ID',
            p_depending_on_comp_prop_name  => 'DUALITY_VIEW_SRC_ID',
            p_depending_on_condition_type  => 'NOT_NULL',
            p_depending_on_comp_prop_name2 => 'DOC_SRC_HAS_ARRAY_COLUMNS',
            p_depending_on_condition_type2 => 'EQUALS',
            p_depending_on_expression2     => 'Y' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'JSON_COLLECTION_ARRAY_COL_ID',
            p_db_column                    => null,
            p_is_required                  => false,
            p_null_text                    => 'None',
            p_is_common                    => false,
            p_parent_comp_prop_name        => 'JSON_COLLECTION_SRC_ID',
            p_depending_on_comp_prop_name  => 'JSON_COLLECTION_SRC_ID',
            p_depending_on_condition_type  => 'NOT_NULL',
            p_depending_on_comp_prop_name2 => 'DOC_SRC_HAS_ARRAY_COLUMNS',
            p_depending_on_condition_type2 => 'EQUALS',
            p_depending_on_expression2     => 'Y' );
        --
        if p_has_web_src_has_local_table then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'WEB_SRC_HAS_LOCAL_TABLE',
                p_db_column                   => null,
                p_is_required                 => false,
                p_is_common                   => false,
                p_default_value               => 'N',
                p_depending_on_comp_prop_name => 'SOURCE_WEB_SRC_QUERY',
                p_depending_on_condition_type => 'NOT_NULL' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'WEB_SRC_USE_LOCAL_TABLE',
                p_db_column                   => null,
                p_is_required                 => false,
                p_is_common                   => false,
                p_default_value               => 'N',
                p_depending_on_comp_prop_name => 'WEB_SRC_HAS_LOCAL_TABLE',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'Y' );
        end if;
        --
        if p_has_web_src_has_local_table then
            store_component_prop (
                p_component_type_id            => l_component_type_id,
                p_property_name                => 'USE_EXTERNAL_EXPR',
                p_db_column                    => null,
                p_is_required                  => false,
                p_is_common                    => false,
                p_default_value                => 'N',
                p_depending_on_comp_prop_name  => 'SOURCE_WEB_SRC_QUERY',
                p_depending_on_condition_type  => 'NOT_NULL',
                --
                p_depending_on_comp_prop_name2 => 'WEB_SRC_USE_LOCAL_TABLE',
                p_depending_on_has_to_exist2   => false,
                p_depending_on_condition_type2 => 'IN_LIST',
                p_depending_on_expression2     => 'N,$NULL$' );
        else
            store_component_prop (
                p_component_type_id            => l_component_type_id,
                p_property_name                => 'USE_EXTERNAL_EXPR',
                p_db_column                    => null,
                p_is_required                  => false,
                p_is_common                    => false,
                p_default_value                => 'N',
                p_depending_on_comp_prop_name  => 'SOURCE_WEB_SRC_QUERY',
                p_depending_on_condition_type  => 'NOT_NULL' );
        end if;
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'EXTERNAL_FILTER_EXPR',
            p_db_column                    => null,
            p_is_required                  => false,
            p_is_common                    => false,
            p_default_value                => null,
            p_depending_on_comp_prop_name  => 'USE_EXTERNAL_EXPR',
            p_depending_on_condition_type  => 'EQUALS',
            p_depending_on_expression      => 'Y' );
        --
        if p_has_order_by_item then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'EXTERNAL_ORDER_BY_TYPE',
                p_db_column                   => null,
                p_is_required                 => false,
                p_is_common                   => true,
                p_null_text                   => 'None',
                p_plugin_std_attributes       => 'ORDER_BY',
                p_depending_on_comp_prop_name => 'USE_EXTERNAL_EXPR',
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'Y' );
            --
            store_component_prop (
                p_component_type_id            => l_component_type_id,
                p_property_name                => 'EXTERNAL_ORDER_BY_EXPR',
                p_db_column                    => null,
                p_is_required                  => true,
                p_is_common                    => false,
                p_default_value                => null,
                p_depending_on_comp_prop_name  => 'EXTERNAL_ORDER_BY_TYPE',
                p_depending_on_condition_type  => 'EQUALS',
                p_depending_on_expression      => 'STATIC' );
            --
            store_component_prop (
                p_component_type_id            => l_component_type_id,
                p_property_name                => 'EXTERNAL_ORDER_BY_ITEM',
                p_db_column                    => null,
                p_is_required                  => true,
                p_is_common                    => false,
                p_default_value                => null,
                p_depending_on_comp_prop_name  => 'EXTERNAL_ORDER_BY_TYPE',
                p_depending_on_condition_type  => 'EQUALS',
                p_depending_on_expression      => 'ITEM' );
            --
            if p_is_region then
                store_component_prop (
                    p_component_type_id            => l_component_type_id,
                    p_property_name                => 'EXTERNAL_STATIC_ORDER_BY_EXPR',
                    p_db_column                    => null,
                    p_is_required                  => false,
                    p_is_common                    => false,
                    p_depending_on_comp_prop_name  => 'USE_EXTERNAL_EXPR',
                    p_depending_on_condition_type  => 'EQUALS',
                    p_depending_on_expression      => 'Y',
                    p_depending_on_comp_prop_name2 => 'REGION_TYPE',
                    p_depending_on_condition_type2 => 'IN_LIST',
                    p_depending_on_expression2     => 'NATIVE_IG,NATIVE_IR,NATIVE_MAP_REGION,NATIVE_FORM' );
            end if;
        elsif p_has_order_by or p_has_external_order_by then
            store_component_prop (
                p_component_type_id            => l_component_type_id,
                p_property_name                => 'EXTERNAL_STATIC_ORDER_BY_EXPR',
                p_db_column                    => null,
                p_is_required                  => false,
                p_is_common                    => false,
                p_depending_on_comp_prop_name  => 'USE_EXTERNAL_EXPR',
                p_depending_on_condition_type  => 'EQUALS',
                p_depending_on_expression      => 'Y' );
        end if;
        --
        --
        store_src_comp_props_objsrc(
            p_prefix                    => null,
            p_source_prop_name          => 'SOURCE_WEB_SRC_QUERY',
            p_location_property_name    => p_location_property_name,
            p_has_order_by              => p_has_order_by,
            p_has_order_by_item         => p_has_order_by_item,
            p_db_column_source          => p_db_column_source,
            p_db_column_source_lang     => p_db_column_source_lang );

        store_src_comp_props_objsrc(
            p_prefix                    => 'DUALITYVIEW_',
            p_source_prop_name          => 'DUALITY_VIEW_SRC_ID',
            p_location_property_name    => p_location_property_name,
            p_has_order_by              => p_has_order_by,
            p_has_order_by_item         => p_has_order_by_item,
            p_db_column_source          => p_db_column_source,
            p_db_column_source_lang     => p_db_column_source_lang );

        store_src_comp_props_objsrc(
            p_prefix                    => 'JSONCOLLECTION_',
            p_source_prop_name          => 'JSON_COLLECTION_SRC_ID',
            p_location_property_name    => p_location_property_name,
            p_has_order_by              => p_has_order_by,
            p_has_order_by_item         => p_has_order_by_item,
            p_db_column_source          => p_db_column_source,
            p_db_column_source_lang     => p_db_column_source_lang );

        --
        if p_has_remote_caching then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'REMOTE_CACHING',
                p_db_column                   => null,
                p_is_required                 => false,
                p_is_common                   => false,
                p_null_text                   => 'Disabled',
                p_null_help_text              => q'~Result is not cached. Data is fetched from the remote database for each request.~',
                p_depending_on_comp_prop_name => p_location_property_name,
                p_depending_on_condition_type => 'EQUALS',
                p_depending_on_expression     => 'REMOTE' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'REMOTE_INVALIDATE_WHEN',
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => false,
                p_depending_on_comp_prop_name => 'REMOTE_CACHING',
                p_depending_on_condition_type => 'NOT_NULL' );
        end if;
        --
    end store_comp_prop_location;
    --
    --
    procedure store_i18n_lang_prop_lov_value
    is
    begin
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Afrikaans (af)',
            p_return_value  => 'af' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Albanian',
            p_return_value  => 'sq' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Arabic (Algeria) (ar-dz)',
            p_return_value  => 'ar-dz' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Arabic (Bahrain) (ar-bh)',
            p_return_value  => 'ar-bh' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Arabic (Egypt) (ar-eg)',
            p_return_value  => 'ar-eg' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Arabic (Iraq) (ar-iq)',
            p_return_value  => 'ar-iq' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Arabic (Jordan) (ar-jo)',
            p_return_value  => 'ar-jo' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Arabic (Kuwait) (ar-kw)',
            p_return_value  => 'ar-kw' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Arabic (Lebanon) (ar-lb)',
            p_return_value  => 'ar-lb' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Arabic (Libya) (ar-ly)',
            p_return_value  => 'ar-ly' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Arabic (Morocco) (ar-ma)',
            p_return_value  => 'ar-ma' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Arabic (Oman) (ar-om)',
            p_return_value  => 'ar-om' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Arabic (Qatar) (ar-qa)',
            p_return_value  => 'ar-qa' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Arabic (Saudi Arabia) (ar-sa)',
            p_return_value  => 'ar-sa' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Arabic (Syria) (ar-sy)',
            p_return_value  => 'ar-sy' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Arabic (Tunisia) (ar-tn)',
            p_return_value  => 'ar-tn' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Arabic (U.A.E.) (ar-ae)',
            p_return_value  => 'ar-ae' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Arabic (Yemen) (ar-ye)',
            p_return_value  => 'ar-ye' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Arabic (ar)',
            p_return_value  => 'ar' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Armenian',
            p_return_value  => 'hy' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Assamese (as)',
            p_return_value  => 'as' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Azerbaijani (az)',
            p_return_value  => 'az' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Basque (eu)',
            p_return_value  => 'eu' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Belarusian (be)',
            p_return_value  => 'be' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Bengali (bn)',
            p_return_value  => 'bn' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Bulgarian (bg)',
            p_return_value  => 'bg' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Khmer (Cambodia) (km)',
            p_return_value  => 'km' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Catalan (ca)',
            p_return_value  => 'ca' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Chinese (China) (zh-cn)',
            p_return_value  => 'zh-cn' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Chinese (Hong Kong SAR) (zh-hk)',
            p_return_value  => 'zh-hk' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Chinese (Macau SAR) (zh-mo)',
            p_return_value  => 'zh-mo' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Chinese (Singapore) (zh-sg)',
            p_return_value  => 'zh-sg' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Chinese (Taiwan) (zh-tw)',
            p_return_value  => 'zh-tw' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Chinese (zh)',
            p_return_value  => 'zh' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Croatian (hr)',
            p_return_value  => 'hr' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Czech (cs)',
            p_return_value  => 'cs' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Danish (da)',
            p_return_value  => 'da' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Dutch (Belgium) (nl-be)',
            p_return_value  => 'nl-be' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Dutch (Netherlands) (nl)',
            p_return_value  => 'nl' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'English (Australia) (en-au)',
            p_return_value  => 'en-au' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'English (Belize) (en-bz)',
            p_return_value  => 'en-bz' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'English (Canada) (en-ca)',
            p_return_value  => 'en-ca' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'English (Ireland) (en-ie)',
            p_return_value  => 'en-ie' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'English (Jamaica) (en-jm)',
            p_return_value  => 'en-jm' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'English (New Zealand) (en-nz)',
            p_return_value  => 'en-nz' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'English (Philippines) (en-ph)',
            p_return_value  => 'en-ph' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'English (South Africa) (en-za)',
            p_return_value  => 'en-za' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'English (Trinidad) (en-tt)',
            p_return_value  => 'en-tt' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'English (United Kingdom) (en-gb)',
            p_return_value  => 'en-gb' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'English (United States) (en-us)',
            p_return_value  => 'en-us' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'English (Zimbabwe) (en-zw)',
            p_return_value  => 'en-zw' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'English (en)',
            p_return_value  => 'en' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Estonian (et)',
            p_return_value  => 'et' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'FYRO Macedonian (mk)',
            p_return_value  => 'mk' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Faeroese (fo)',
            p_return_value  => 'fo' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Farsi (fa)',
            p_return_value  => 'fa' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Finnish (fi)',
            p_return_value  => 'fi' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'French (Belgium) (fr-be)',
            p_return_value  => 'fr-be' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'French (Canada) (fr-ca)',
            p_return_value  => 'fr-ca' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'French (France) (fr)',
            p_return_value  => 'fr' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'French (Luxembourg) (fr-lu)',
            p_return_value  => 'fr-lu' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'French (Monaco) (fr-mc)',
            p_return_value  => 'fr-mc' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'French (Switzerland) (fr-ch)',
            p_return_value  => 'fr-ch' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Gaelic (gd)',
            p_return_value  => 'gd' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Galician (gl)',
            p_return_value  => 'gl' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'German (Austria) (de-at)',
            p_return_value  => 'de-at' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'German (Germany) (de)',
            p_return_value  => 'de' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'German (Liechtenstein) (de-li)',
            p_return_value  => 'de-li' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'German (Luxemgourg) (de-lu)',
            p_return_value  => 'de-lu' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'German (Switzerland) (de-ch)',
            p_return_value  => 'de-ch' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Greek (el)',
            p_return_value  => 'el' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Gujarati (gu)',
            p_return_value  => 'gu' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Hebrew (he)',
            p_return_value  => 'he' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Hindi (hi)',
            p_return_value  => 'hi' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Hungarian (hu)',
            p_return_value  => 'hu' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Icelandic (is)',
            p_return_value  => 'is' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Indonesian (id)',
            p_return_value  => 'id' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Irish (ga)',
            p_return_value  => 'ga' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Italian (Italy) (it)',
            p_return_value  => 'it' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Italian (Switzerland) (it-ch)',
            p_return_value  => 'it-ch' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Japanese (ja)',
            p_return_value  => 'ja' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Kannada (kn)',
            p_return_value  => 'kn' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Kazakh (kk)',
            p_return_value  => 'kk' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Konkani (kok)',
            p_return_value  => 'kok' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Korean (ko)',
            p_return_value  => 'ko' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Kyrgyz (kz)',
            p_return_value  => 'kz' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Latvian (lv)',
            p_return_value  => 'lv' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Lithuanian (lt)',
            p_return_value  => 'lt' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Malay (Malaysia) (ms)',
            p_return_value  => 'ms' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Malayalam (ml)',
            p_return_value  => 'ml' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Maltese (mt)',
            p_return_value  => 'mt' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Marathi (mr)',
            p_return_value  => 'mr' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Nepali (India) (ne)',
            p_return_value  => 'ne' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Norwegian (Bokmal) (nb-no)',
            p_return_value  => 'nb-no' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Norwegian (Bokmal) (no)',
            p_return_value  => 'no' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Norwegian (Nynorsk) (nn-no)',
            p_return_value  => 'nn-no' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Oriya (or)',
            p_return_value  => 'or' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Polish (pl)',
            p_return_value  => 'pl' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Portuguese (Brazil) (pt-br)',
            p_return_value  => 'pt-br' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Portuguese (Portugal) (pt)',
            p_return_value  => 'pt' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Punjabi (pa)',
            p_return_value  => 'pa' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Romanian (ro)',
            p_return_value  => 'ro' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Russian (Moldova) (ru-md)',
            p_return_value  => 'ru-md' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Russian (ru)',
            p_return_value  => 'ru' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Serbian (sr)',
            p_return_value  => 'sr' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Slovak (sk)',
            p_return_value  => 'sk' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Slovenian (sl)',
            p_return_value  => 'sl' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Spanish (Argentina) (es-ar)',
            p_return_value  => 'es-ar' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Spanish (Bolivia) (es-bo)',
            p_return_value  => 'es-bo' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Spanish (Chile) (es-cl)',
            p_return_value  => 'es-cl' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Spanish (Colombia) (es-co)',
            p_return_value  => 'es-co' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Spanish (Costa Rica) (es-cr)',
            p_return_value  => 'es-cr' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Spanish (Dominican Republic) (es-do)',
            p_return_value  => 'es-do' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Spanish (Ecuador) (es-ec)',
            p_return_value  => 'es-ec' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Spanish (El Salvador) (es-sv)',
            p_return_value  => 'es-sv' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Spanish (Guatemala) (es-gt)',
            p_return_value  => 'es-gt' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Spanish (Honduras) (es-hn)',
            p_return_value  => 'es-hn' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Spanish (Mexico) (es-mx)',
            p_return_value  => 'es-mx' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Spanish (Nicaragua) (es-ni)',
            p_return_value  => 'es-ni' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Spanish (Panama) (es-pa)',
            p_return_value  => 'es-pa' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Spanish (Paraguay) (es-py)',
            p_return_value  => 'es-py' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Spanish (Peru) (es-pe)',
            p_return_value  => 'es-pe' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Spanish (Puerto Rico) (es-pr)',
            p_return_value  => 'es-pr' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Spanish (Traditional Sort) (es)',
            p_return_value  => 'es' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Spanish (United States) (es-us)',
            p_return_value  => 'es-us' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Spanish (Uruguay) (es-uy)',
            p_return_value  => 'es-uy' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Spanish (Venezuela) (es-ve)',
            p_return_value  => 'es-ve' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Swahili (sw)',
            p_return_value  => 'sw' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Swedish (Finland) (sv-fi)',
            p_return_value  => 'sv-fi' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Swedish (sv)',
            p_return_value  => 'sv' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Tamil (ta)',
            p_return_value  => 'ta' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Telugu (te)',
            p_return_value  => 'te' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Thai (th)',
            p_return_value  => 'th' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Turkish (tr)',
            p_return_value  => 'tr' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Ukrainian (uk)',
            p_return_value  => 'uk' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Urdu (ur)',
            p_return_value  => 'ur' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Uzbek (uz)',
            p_return_value  => 'uz' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Vietnamese (vi)',
            p_return_value  => 'vi' );
        --
        store_property_lov_values (
            p_property_id   => l_property_id,
            p_display_value => 'Welsh (cy)',
            p_return_value  => 'cy' );
        --
    end store_i18n_lang_prop_lov_value;
    --
    --
    procedure store_comp_prop_data_profile (
        p_profile_type_id in number,
        p_column_type_id  in number )
    is
    begin
        --
        -- Component Properties for XZY_DATA_PROFILE
        --
        l_component_type_id           := p_profile_type_id;
        -- Reset Component Property sequence for this component type
        g_component_property_sequence := 10;
        --
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'NAME',
            p_db_column             => null,
            p_is_required           => true,
            p_is_common             => true,
            p_supports_substitution => false,
            p_check_uniqueness      => 'APPLICATION',
            p_default_value         => '#PLEASE_CHANGE#',
            p_help_text             => q'~Enter a name for the data profile.~' );
        --
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'DATA_PROFILE_FORMAT',
            p_db_column             => null,
            p_is_required           => true,
            p_is_common             => true,
            p_default_value         => 'JSON' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'DATA_PROFILE_ROW_SELECTOR',
            p_db_column                    => null,
            p_is_required                  => false,
            p_is_common                    => false,
            p_supports_substitution        => false,
            p_depending_on_comp_prop_name  => 'DATA_PROFILE_FORMAT',
            p_depending_on_condition_type  => 'IN_LIST',
            p_depending_on_expression      => 'JSON,XML' );
        --
        if l_component_type_id in ( c_comp_type_web_src_mod_dp, c_comp_type_web_src_oper_dp ) then
            store_component_prop (
                p_component_type_id            => l_component_type_id,
                p_property_name                => 'DATA_PROFILE_IS_SINGLE_ROW',
                p_db_column                    => null,
                p_is_required                  => true,
                p_is_common                    => true,
                p_default_value                => 'N',
                p_depending_on_comp_prop_name  => 'DATA_PROFILE_FORMAT',
                p_depending_on_condition_type  => 'IN_LIST',
                p_depending_on_expression      => 'JSON,XML' );
            --
            store_component_prop (
                p_component_type_id            => l_component_type_id,
                p_property_name                => 'DATA_PROFILE_SINGLE_ROW_SELECTOR',
                p_db_column                    => null,
                p_is_required                  => false,
                p_is_common                    => true,
                p_supports_substitution        => false,
                p_depending_on_comp_prop_name  => 'DATA_PROFILE_IS_SINGLE_ROW',
                p_depending_on_condition_type  => 'EQUALS',
                p_depending_on_expression      => 'N',
                p_depending_on_comp_prop_name2 => 'DATA_PROFILE_FORMAT',
                p_depending_on_condition_type2 => 'NOT_EQUALS',
                p_depending_on_expression2     => 'CSV' );
        end if;
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'DATA_PROFILE_USE_RAW_JSON_SELECTORS',
            p_db_column                    => null,
            p_is_required                  => true,
            p_is_common                    => false,
            p_default_value                => 'N',
            p_depending_on_comp_prop_name  => 'DATA_PROFILE_FORMAT',
            p_depending_on_condition_type  => 'EQUALS',
            p_depending_on_expression      => 'JSON' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'I18N_IANA_CHARSET_ENCODING',
            p_db_column                    => null,
            p_is_required                  => false,
            p_is_common                    => false,
            p_null_text                    => 'Database Character Set',
            p_null_help_text               => q'~Treat the data as encoded in the <em>database character set</em>.~' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'DATA_PROFILE_NUMERIC_CHARACTERS',
            p_db_column                    => null,
            p_is_required                  => false,
            p_is_common                    => false,
            p_supports_substitution        => false );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'XML_NAMESPACES',
            p_db_column                    => null,
            p_is_required                  => false,
            p_is_common                    => false,
            p_supports_substitution        => false,
            p_depending_on_comp_prop_name  => 'DATA_PROFILE_FORMAT',
            p_depending_on_condition_type  => 'EQUALS',
            p_depending_on_expression      => 'XML' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'DATA_PROFILE_CSV_SEPARATOR',
            p_db_column                    => null,
            p_is_required                  => false,
            p_is_common                    => false,
            p_depending_on_comp_prop_name  => 'DATA_PROFILE_FORMAT',
            p_depending_on_condition_type  => 'EQUALS',
            p_depending_on_expression      => 'CSV' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'DATA_PROFILE_CSV_ENCLOSED_BY',
            p_db_column                    => null,
            p_is_required                  => false,
            p_is_common                    => false,
            p_depending_on_comp_prop_name  => 'DATA_PROFILE_FORMAT',
            p_depending_on_condition_type  => 'EQUALS',
            p_depending_on_expression      => 'CSV' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'DEFAULT_XLSX_SHEET_NAME',
            p_db_column                    => null,
            p_is_required                  => false,
            p_is_common                    => false,
            p_supports_substitution        => false,
            p_depending_on_comp_prop_name  => 'DATA_PROFILE_FORMAT',
            p_depending_on_condition_type  => 'EQUALS',
            p_depending_on_expression      => 'XLSX' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'DATA_PROFILE_SKIP_ROWS',
            p_db_column                    => null,
            p_is_required                  => false,
            p_is_common                    => false,
            p_depending_on_comp_prop_name  => 'DATA_PROFILE_FORMAT',
            p_depending_on_condition_type  => 'IN_LIST',
            p_depending_on_expression      => 'CSV,XLSX' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'DATA_PROFILE_HAS_HEADER_ROW',
            p_db_column                    => null,
            p_is_required                  => true,
            p_is_common                    => false,
            p_default_value                => 'Y',
            p_depending_on_comp_prop_name  => 'DATA_PROFILE_FORMAT',
            p_depending_on_condition_type  => 'IN_LIST',
            p_depending_on_expression      => 'CSV,XLSX' );
        --
        store_standard_attr_comp_prop (
            p_component_type_id => l_component_type_id,
            p_has_comments      => true,
            p_comment_db_column => 'PROFILE_COMMENT' );
        --
        --
        -- Component Properties for XYZ_DATA_PROFILE_COLUMN
        --
        l_component_type_id           := p_column_type_id;
        -- Reset Component Property sequence for this component type
        g_component_property_sequence := 10;
        --
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'COLUMN_NAME',
            p_db_column             => 'NAME',
            p_is_required           => true,
            p_is_common             => true,
            p_supports_substitution => false,
            p_check_uniqueness      => 'PARENT',
            p_help_text             => q'~<p>Name under which the column is exposed to &PRODUCT_NAME. components.</p>
    <p>If the data profile is part of a <em>Data Load Definition</em>, this name is used to map data profile columns to the target table. For instance, the <strong>ENAME</strong> data profile column will be loaded to the <strong>ENAME</strong> column of the target table. If the profile column does <em>not exist</em> in the target table, it will be ignored during data load.</p>~' );
        --
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'DATA_PROFILE_FORMAT_HIDDEN',
            p_db_column             => null,
            p_is_required           => false,
            p_is_common             => false );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'DATA_PROFILE_COLUMN_TYPE',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => true,
            p_default_value     => 'DATA' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'DATA_PROFILE_COLUMN_DATA_TYPE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_default_value               => 'VARCHAR2' ); -- $$$ todo: Page 1926 has some advanced defaulting logic which has to go into pe.callbacks.js
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'VALIDATION_REGEXP',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_default_value               => null,
            p_depending_on_comp_prop_name => 'DATA_PROFILE_COLUMN_DATA_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'VARCHAR2' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'MAX_LENGTH',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_default_value               => '4000',
            p_depending_on_comp_prop_name => 'DATA_PROFILE_COLUMN_DATA_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'VARCHAR2' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'DATA_PROFILE_DECIMAL_CHARACTER',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_supports_substitution       => false,
            p_depending_on_comp_prop_name => 'DATA_PROFILE_COLUMN_DATA_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'NUMBER' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'DATA_PROFILE_GROUP_CHARACTER',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_supports_substitution       => false,
            p_depending_on_comp_prop_name => 'DATA_PROFILE_COLUMN_DATA_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'NUMBER' );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'IS_PRIMARY_KEY',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => true,
            p_default_value     => 'N',
            p_help_text         => q'~Determines whether this is a primary key column. A primary key can consist of multiple columns.~' );
        --
        -- Parsing
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'DATA_PROFILE_SELECTOR_TYPE',
            p_db_column                    => null,
            p_is_required                  => true,
            p_is_common                    => true,
            p_default_value                => 'NAME',
            p_depending_on_comp_prop_name  => 'DATA_PROFILE_COLUMN_TYPE',
            p_depending_on_condition_type  => 'EQUALS',
            p_depending_on_expression      => 'DATA',
            p_depending_on_comp_prop_name2 => 'DATA_PROFILE_FORMAT_HIDDEN',
            p_depending_on_condition_type2 => 'IN_LIST',
            p_depending_on_expression2     => 'CSV,XLSX' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'DATA_PROFILE_COLUMN_SELECTOR',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_supports_substitution       => false,
            p_depending_on_comp_prop_name => 'DATA_PROFILE_SELECTOR_TYPE',
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'NAME,REGEXP' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'DATA_PROFILE_COLUMN_SELECTOR_NO',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_supports_substitution       => false,
            p_depending_on_comp_prop_name => 'DATA_PROFILE_SELECTOR_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'SEQ' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'DATA_PROFILE_COLUMN_PATH',
            p_db_column                    => null,
            p_is_required                  => true,
            p_is_common                    => true,
            p_supports_substitution        => false,
            p_depending_on_comp_prop_name  => 'DATA_PROFILE_FORMAT_HIDDEN',
            p_depending_on_condition_type  => 'IN_LIST',
            p_depending_on_expression      => 'JSON,XML',
            p_depending_on_comp_prop_name2 => 'DATA_PROFILE_COLUMN_TYPE',
            p_depending_on_condition_type2 => 'EQUALS',
            p_depending_on_expression2     => 'DATA' );
        --
        store_component_prop (
            p_component_type_id            => l_component_type_id,
            p_property_name                => 'DATA_PROFILE_TABLE_COLUMN',
            p_db_column                    => null,
            p_is_required                  => true,
            p_is_common                    => true,
            p_supports_substitution        => false,
            p_depending_on_comp_prop_name  => 'DATA_PROFILE_FORMAT_HIDDEN',
            p_depending_on_condition_type  => 'IN_LIST',
            p_depending_on_expression      => 'JSON',
            p_depending_on_comp_prop_name2 => 'DATA_PROFILE_COLUMN_TYPE',
            p_depending_on_condition_type2 => 'EQUALS',
            p_depending_on_expression2     => 'TABLE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'PARSING_FORMAT_MASK',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'DATA_PROFILE_COLUMN_DATA_TYPE',
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'NUMBER,DATE,TIMESTAMP,TIMESTAMP WITH TIME ZONE,TIMESTAMP WITH LOCAL TIME ZONE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'HAS_TIME_ZONE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_default_value               => 'Y',
            p_depending_on_comp_prop_name => 'DATA_PROFILE_COLUMN_DATA_TYPE',
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'DATE,TIMESTAMP,TIMESTAMP WITH TIME ZONE,TIMESTAMP WITH LOCAL TIME ZONE' );
        --
        -- Computation
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'COMPUTATION_SQL_EXPRESSION_VARCHAR2',
            p_db_column                   => 'EXPRESSION1',
            p_is_required                 => true,
            p_is_common                   => true,
            p_supports_substitution       => false,
            p_depending_on_comp_prop_name => 'DATA_PROFILE_COLUMN_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'SQL_EXPRESSION',
            p_help_text                   => q'~Enter the SQL Expression to compute the value of this column. The SQL expression can reference data profile columns of the <strong>Data</strong> column type.~' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'COMPUTATION_SQL_STATEMENT',
            p_db_column                   => 'EXPRESSION1',
            p_is_required                 => true,
            p_is_common                   => true,
            p_supports_substitution       => false,
            p_depending_on_comp_prop_name => 'DATA_PROFILE_COLUMN_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'SQL_QUERY',
            p_help_text                   => q'~Enter the SQL Query to compute the value of this column. The SQL query must return a single value and can reference data profile columns of the <strong>Data</strong> column type.~' );
        --
        -- Lookup
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'LOOKUP_OWNER',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'DATA_PROFILE_COLUMN_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'LOOKUP' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'LOOKUP_TABLE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_parent_comp_prop_name       => 'LOOKUP_OWNER',
            p_depending_on_comp_prop_name => 'DATA_PROFILE_COLUMN_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'LOOKUP' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'LOOKUP_RETURN_COLUMN',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_parent_comp_prop_name       => 'LOOKUP_TABLE',
            p_depending_on_comp_prop_name => 'DATA_PROFILE_COLUMN_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'LOOKUP' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'LOOKUP_WHERE_CLAUSE',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => true,
            p_parent_comp_prop_name       => 'LOOKUP_TABLE',
            p_depending_on_comp_prop_name => 'DATA_PROFILE_COLUMN_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'LOOKUP' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'LOOKUP_TABLE_COLUMN1',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_parent_comp_prop_name       => 'LOOKUP_TABLE',
            p_depending_on_comp_prop_name => 'DATA_PROFILE_COLUMN_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'LOOKUP' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'LOOKUP_DATA_COLUMN1',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_depending_on_comp_prop_name => 'DATA_PROFILE_COLUMN_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'LOOKUP' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'LOOKUP_TABLE_COLUMN2',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_parent_comp_prop_name       => 'LOOKUP_TABLE',
            p_depending_on_comp_prop_name => 'LOOKUP_TABLE_COLUMN1',
            p_depending_on_condition_type => 'NOT_NULL' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'LOOKUP_DATA_COLUMN2',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'LOOKUP_TABLE_COLUMN2',
            p_depending_on_condition_type => 'NOT_NULL' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'LOOKUP_TABLE_COLUMN3',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_parent_comp_prop_name       => 'LOOKUP_TABLE',
            p_depending_on_comp_prop_name => 'LOOKUP_TABLE_COLUMN2',
            p_depending_on_condition_type => 'NOT_NULL' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'LOOKUP_DATA_COLUMN3',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'LOOKUP_TABLE_COLUMN3',
            p_depending_on_condition_type => 'NOT_NULL' );
        --
        -- Transformation
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'TRANSFORMATION_TYPE',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_depending_on_comp_prop_name => 'DATA_PROFILE_COLUMN_TYPE',
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'DATA,LOOKUP' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'TRANSFORMATION_TRIM_CHARS',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_supports_substitution       => true,
            p_depending_on_comp_prop_name => 'TRANSFORMATION_TYPE',
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'LTRIM,RTRIM,TRIM' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'TRANSFORMATION_FIND',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_supports_substitution       => true,
            p_depending_on_comp_prop_name => 'TRANSFORMATION_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'REPLACE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'TRANSFORMATION_REGEXP',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_supports_substitution       => true,
            p_depending_on_comp_prop_name => 'TRANSFORMATION_TYPE',
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'REPLACE,REGEXP_NULLIF' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'TRANSFORMATION_REPLACE_WITH',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_supports_substitution       => true,
            p_depending_on_comp_prop_name => 'TRANSFORMATION_TYPE',
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'REPLACE,REGEXP_REPLACE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'TRANSFORMATION_NULLIF',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_supports_substitution       => true,
            p_depending_on_comp_prop_name => 'TRANSFORMATION_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'NULLIF' );
        --
        -- Remote REST Data Source
        if l_component_type_id in ( c_comp_type_web_src_mod_dp_col, c_comp_type_web_src_oper_dpcol ) then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'REMOTE_IS_FILTERABLE',
                p_db_column                   => null,
                p_is_required                 => false,
                p_is_common                   => false,
                p_supports_substitution       => false,
                p_default_value               => 'Y' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'REMOTE_IS_COMMON',
                p_db_column                   => null,
                p_is_required                 => false,
                p_is_common                   => false,
                p_supports_substitution       => false,
                p_default_value               => 'Y' );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'REMOTE_ATTRIBUTE_NAME',
                p_db_column                   => null,
                p_is_required                 => false,
                p_is_common                   => false,
                p_supports_substitution       => false );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'REMOTE_DATA_TYPE',
                p_db_column                   => null,
                p_is_required                 => false,
                p_is_common                   => false,
                p_supports_substitution       => false );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'ADDITIONAL_INFO',
                p_db_column                   => null,
                p_is_required                 => false,
                p_is_common                   => false,
                p_supports_substitution       => false );
        end if;
        --
        -- Advanced
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'ADVANCED_SEQUENCE',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => true );
        --
        store_component_prop (
            p_component_type_id => l_component_type_id,
            p_property_name     => 'IS_HIDDEN',
            p_db_column         => null,
            p_is_required       => true,
            p_is_common         => false,
            p_default_value     => 'N' );
        --
        store_standard_attr_comp_prop (
            p_component_type_id => l_component_type_id,
            p_has_comments      => true,
            p_comment_db_column => 'COLUMN_COMMENT' );
        --
    end store_comp_prop_data_profile;
    --
    --
    procedure store_comp_prop_sup_obj_script
    is
    begin
        --
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'NAME',
            p_db_column             => null,
            p_is_required           => true,
            p_is_common             => true,
            p_check_uniqueness      => 'PARENT',
            p_supports_substitution => false,
            p_default_value         => '#PLEASE_CHANGE#',
            p_help_text             => q'~The name of this installation script.~' );
        --
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'EXECUTION_SEQUENCE',
            p_db_column             => 'SEQUENCE',
            p_is_required           => true,
            p_is_common             => true,
            p_help_text             => q'~The sequence number of this installation script. Scripts run in order of ascending sequence number.~' );
        --
        store_standard_attr_comp_prop (
            p_component_type_id     => l_component_type_id,
            p_has_condition         => true,
            p_cond_type_name_prefix => 'SO_COND' );
        --
    end store_comp_prop_sup_obj_script;
    --
    --
    procedure store_comp_prop_tmpl_options (
        p_component_type_id     in number,
        p_type_name             in varchar2,
        p_default_value         in varchar2 default null )
    is
    begin
        --
        if p_component_type_id = c_comp_type_global_tmpl_option then
            store_component_prop (
                p_component_type_id     => p_component_type_id,
                p_property_name         => 'TEMPLATE_OPTION_TYPE',
                p_db_column             => null,
                p_is_required           => true,
                p_is_common             => true,
                p_supports_substitution => false );
        else
            store_component_prop (
                p_component_type_id     => p_component_type_id,
                p_property_name         => 'TEMPLATE_OPTION_TYPE_HIDDEN',
                p_db_column             => null,
                p_is_required           => true,
                p_is_common             => true,
                p_supports_substitution => false,
                p_default_value         => p_default_value );
        end if;
        --
        store_component_prop (
            p_component_type_id     => p_component_type_id,
            p_property_name         => 'NAME',
            p_db_column             => 'DISPLAY_NAME',
            p_is_required           => true,
            p_is_common             => true,
            p_check_uniqueness      => 'PARENT',
            p_supports_substitution => false,
            p_help_text             => q'~The display name of this template option.~' );
        --
        store_component_prop (
            p_component_type_id     => p_component_type_id,
            p_property_name         => 'IDENTIFIER',
            p_db_column             => 'NAME',
            p_is_required           => true,
            p_is_common             => true,
            p_check_uniqueness      => 'PARENT',
            p_supports_substitution => false,
            p_help_text             => q'~The name of this template option.~' );
        --
        store_component_prop (
            p_component_type_id     => p_component_type_id,
            p_property_name         => 'TEMPLATE_OPTION_GROUP',
            p_db_column             => null,
            p_is_required           => true,
            p_is_common             => true,
            p_supports_substitution => false );
        --
        store_component_prop (
            p_component_type_id     => p_component_type_id,
            p_property_name         => 'IDENTIFICATION_DISPLAY_SEQUENCE',
            p_db_column             => 'DISPLAY_SEQUENCE',
            p_is_required           => true,
            p_is_common             => true,
            p_supports_substitution => false,
            p_help_text             => q'~<p>Enter the display sequence for this template option.</p>~' );
        --
        store_component_prop (
            p_component_type_id     => l_component_type_id,
            p_property_name         => 'REFERENCE_ID',
            p_db_column             => null,
            p_is_required           => false,
            p_is_common             => false );
        --
        store_component_prop (
            p_component_type_id     => p_component_type_id,
            p_property_name         => 'IS_ADVANCED',
            p_db_column             => 'IS_ADVANCED',
            p_is_required           => true,
            p_is_common             => false,
            p_supports_substitution => false,
            p_default_value         => 'N' );
        --
        store_component_prop (
            p_component_type_id     => p_component_type_id,
            p_property_name         => 'COMPONENT_CSS_CLASSES',
            p_db_column             => 'CSS_CLASSES',
            p_is_required           => true,
            p_is_common             => true,
            p_check_uniqueness      => 'PARENT',
            p_supports_substitution => false );
        --
        store_component_prop (
            p_component_type_id     => p_component_type_id,
            p_property_name         => 'HELP_TEXT',
            p_db_column             => null,
            p_is_required           => false,
            p_is_common             => true,
            p_supports_substitution => false,
            p_help_text             => q'~<p>Use this attribute to enter help text for this template option.</p>~' );
        --
        store_standard_attr_comp_prop (
            p_component_type_id     => p_component_type_id );
        --
    end store_comp_prop_tmpl_options;
    --
    --
    procedure store_action_comp_props (
        p_is_region in boolean )
    is
        l_dynamic_reference_scope varchar2(7) := case when p_is_region then 'DEPENDS' else 'ROW' end;
    begin
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'HIDDEN_PLUGIN_NAME',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false ); /* property is hidden */
        --
        -- this property gets populated by view wwv_flow_component_actions_dev and
        -- pe.callbacks.js. We need it because ACTION_TEMPLATE isn't always visible,
        -- for example in the case of a LINK position or if the position has a
        -- predefined template, but we still need a property for all of our
        -- dependend-on checks. That's why we have this property which gets synchronized
        -- with the value of ACTION_TEMPLATE or wwv_flow_plugin_act_positions.template_id.
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'HIDDEN_TEMPLATE_ID',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false ); /* property is hidden */
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'ACTION_POSITION',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_parent_comp_prop_name       => 'HIDDEN_PLUGIN_NAME' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'ACTION_TEMPLATE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_parent_comp_prop_name       => 'HIDDEN_PLUGIN_NAME',
            p_features_comp_prop_name     => 'ACTION_POSITION',
            p_features                    => 'TEMPLATE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'BUTTON_LABEL',
            p_db_column                   => 'LABEL',
            p_is_required                 => true,
            p_is_common                   => true,
            p_reference_scope             => 'ROW',
            p_default_value               => '#PLEASE_CHANGE#',
            p_features_comp_prop_name     => 'HIDDEN_TEMPLATE_ID',
            p_features                    => 'LABEL' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'DISPLAY_SEQUENCE',
            p_db_column                   => 'DISPLAY_SEQUENCE',
            p_is_required                 => true,
            p_is_common                   => true,
            p_help_text                   => q'~Enter the display sequence for this action. This is used to evaluate the display condition and render in the order defined.~' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'ACTION_LINK_TARGET_TYPE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_default_value               => 'REDIRECT_PAGE',
            p_features_comp_prop_name     => 'HIDDEN_TEMPLATE_ID',
            p_features                    => 'LINK_URL' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'LINK_TARGET',
            p_db_column                   => 'LINK_TARGET',
            p_is_required                 => true,
            p_is_common                   => true,
            p_reference_scope             => l_dynamic_reference_scope,
            p_depending_on_comp_prop_name => 'ACTION_LINK_TARGET_TYPE',
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'REDIRECT_PAGE,REDIRECT_APP,REDIRECT_URL' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'LINK_ATTRIBUTES',
            p_db_column                   => 'LINK_ATTRIBUTES',
            p_is_required                 => false,
            p_is_common                   => false,
            p_reference_scope             => l_dynamic_reference_scope,
            p_depending_on_comp_prop_name => 'LINK_TARGET',
            p_depending_on_condition_type => 'NOT_NULL',
            p_features_comp_prop_name     => 'HIDDEN_TEMPLATE_ID',
            p_features                    => 'LINK_ATTR' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'BUTTON_DISPLAY_TYPE',
            p_db_column                   => 'BUTTON_DISPLAY_TYPE',
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'TEXT',
            p_features_comp_prop_name     => 'HIDDEN_TEMPLATE_ID',
            p_features                    => 'ICON_CLASSES' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'ICON_CSS_CLASSES',
            p_db_column                   => 'ICON_CSS_CLASSES',
            p_is_required                 => true,
            p_is_common                   => false,
            p_reference_scope             => l_dynamic_reference_scope,
            p_depending_on_comp_prop_name => 'BUTTON_DISPLAY_TYPE',
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'ICON,TEXT_WITH_ICON' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'BUTTON_IS_HOT',
            p_db_column                   => 'IS_HOT',
            p_is_required                 => true,
            p_is_common                   => true,
            p_help_text                   => q'~Specify whether to use the <strong>Normal</strong> or <strong>Hot</strong> button when rendering this button.~',
            p_default_value               => 'N',
            p_features_comp_prop_name     => 'HIDDEN_TEMPLATE_ID',
            p_features                    => 'IS_HOT' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'SHOW_AS_DISABLED',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_default_value               => 'N',
            p_features_comp_prop_name     => 'HIDDEN_TEMPLATE_ID',
            p_features                    => 'IS_DISABLED' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'APPEARANCE_CSS_CLASSES',
            p_db_column                   => 'ACTION_CSS_CLASSES',
            p_is_required                 => false,
            p_is_common                   => false,
            p_reference_scope             => l_dynamic_reference_scope,
            p_help_text                   => q'~Enter classes to add to this action. You may add multiple classes by separating them with spaces.~',
            p_features_comp_prop_name     => 'HIDDEN_TEMPLATE_ID',
            p_features                    => 'CSS_CLASSES' );
        --
        store_condition_type_comp_prop (
            p_component_type_id           => l_component_type_id,
            p_name_prefix                 => null,
            p_help_text_type              => null,
            p_reference_scope             => 'DEPENDS',
            p_db_column_type              => 'CONDITION_TYPE',
            p_view_column_type            => 'CONDITION_TYPE',
            p_db_column_expr1             => 'CONDITION_EXPR1',
            p_db_column_expr2             => 'CONDITION_EXPR2',
            p_has_execute_condition       => true );
        --
        store_standard_attr_comp_prop (
            p_component_type_id        => l_component_type_id,
            p_has_authorization_scheme => true,
            p_authorization_db_column  => 'AUTHORIZATION_SCHEME',
            p_has_build_option         => true,
            p_build_option_db_column   => 'BUILD_OPTION_ID',
            p_has_comments             => true,
            p_comment_db_column        => 'ACTION_COMMENTS' );
    end store_action_comp_props;
    --
    --
    procedure store_action_menu_comp_props (
        p_parent_menu_property_name in varchar2,
        p_is_region                 in boolean )
    is
        l_dynamic_reference_scope varchar2(7) := case when p_is_region then 'DEPENDS' else 'ROW' end;
    begin
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'MENU_ENTRY_TYPE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_default_value               => 'ENTRY' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'BUTTON_LABEL',
            p_db_column                   => 'LABEL',
            p_is_required                 => true,
            p_is_common                   => true,
            p_reference_scope             => l_dynamic_reference_scope,
            p_help_text                   => q'~Enter the text to be displayed in the menu.~',
            p_default_value               => '#PLEASE_CHANGE#',
            p_depending_on_comp_prop_name => 'MENU_ENTRY_TYPE',
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'ENTRY,SUB_MENU' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => p_parent_menu_property_name,
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => true,
            p_null_text                   => 'No Parent',
            p_reference_on_delete         => 'DB_CASCADE' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'DISPLAY_SEQUENCE',
            p_db_column                   => 'DISPLAY_SEQUENCE',
            p_is_required                 => true,
            p_is_common                   => true,
            p_help_text                   => q'~Enter the display sequence for this menu entry. This is used to evaluate the display condition and render in the order defined.~' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'ACTION_LINK_TARGET_TYPE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => true,
            p_default_value               => 'REDIRECT_PAGE',
            p_depending_on_comp_prop_name => 'MENU_ENTRY_TYPE',
            p_depending_on_condition_type => 'EQUALS',
            p_depending_on_expression     => 'ENTRY' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'LINK_TARGET',
            p_db_column                   => 'LINK_TARGET',
            p_is_required                 => true,
            p_is_common                   => true,
            p_reference_scope             => l_dynamic_reference_scope,
            p_depending_on_comp_prop_name => 'ACTION_LINK_TARGET_TYPE',
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'REDIRECT_PAGE,REDIRECT_APP,REDIRECT_URL' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'LINK_ATTRIBUTES',
            p_db_column                   => 'LINK_ATTRIBUTES',
            p_is_required                 => false,
            p_is_common                   => false,
            p_reference_scope             => l_dynamic_reference_scope,
            p_depending_on_comp_prop_name => 'LINK_TARGET',
            p_depending_on_condition_type => 'NOT_NULL' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'ICON_CSS_CLASSES',
            p_db_column                   => 'ICON_CSS_CLASSES',
            p_is_required                 => false,
            p_is_common                   => false,
            p_reference_scope             => l_dynamic_reference_scope,
            p_depending_on_comp_prop_name => 'MENU_ENTRY_TYPE',
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'ENTRY,SUB_MENU' );
        --
        store_condition_type_comp_prop (
            p_component_type_id           => l_component_type_id,
            p_name_prefix                 => null,
            p_help_text_type              => null,
            p_reference_scope             => 'DEPENDS',
            p_db_column_type              => 'CONDITION_TYPE',
            p_view_column_type            => 'CONDITION_TYPE',
            p_db_column_expr1             => 'CONDITION_EXPR1',
            p_db_column_expr2             => 'CONDITION_EXPR2',
            p_depending_on_comp_prop_name => 'MENU_ENTRY_TYPE',
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'ENTRY,SUB_MENU',
            p_has_execute_condition       => true );
        --
        store_standard_attr_comp_prop (
            p_component_type_id           => l_component_type_id,
            p_has_authorization_scheme    => true,
            p_authorization_db_column     => 'AUTHORIZATION_SCHEME',
            p_authz_dep_on_comp_prop_name => 'MENU_ENTRY_TYPE',
            p_authz_dep_on_condition_type => 'IN_LIST',
            p_authz_dep_on_expression     => 'ENTRY,SUB_MENU',
            p_has_build_option            => true,
            p_build_option_db_column      => 'BUILD_OPTION_ID',
            p_has_comments                => true,
            p_comment_db_column           => 'MENU_ENTRY_COMMENTS' );
    end store_action_menu_comp_props;
    --
    procedure store_attr_cond_comp_props
    is
    begin
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'DEPENDING_ON_ATTRIBUTE',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false,
            p_supports_substitution       => false );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'DEPENDING_ON_CONDITION_TYPE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_supports_substitution       => false,
            p_depending_on_comp_prop_name => 'DEPENDING_ON_ATTRIBUTE',
            p_depending_on_condition_type => 'NOT_NULL' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'DEPENDING_ON_EXPRESSION',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_supports_substitution       => false,
            p_depending_on_comp_prop_name => 'DEPENDING_ON_CONDITION_TYPE',
            p_depending_on_condition_type => 'IN_LIST',
            p_depending_on_expression     => 'EQUALS,NOT_EQUALS,IN_LIST,NOT_IN_LIST' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'DEPENDING_ON_HAS_TO_EXIST',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_supports_substitution       => false,
            p_default_value               => 'Y',
            p_depending_on_comp_prop_name => 'DEPENDING_ON_ATTRIBUTE',
            p_depending_on_condition_type => 'NOT_NULL' );
        --
    end store_attr_cond_comp_props;
    --
    procedure store_attribute_props (
        p_start_property_id     in integer,
        p_attribute_count       in integer,
        p_column_name_prefix    in varchar2 )
    is
    begin
        -- Init the start property id to the id before the starting id
        -- to do a correct increase inside the loop
        l_property_id := p_start_property_id - 1;
        --
        for i in 1..p_attribute_count loop
            --
            l_property_id := l_property_id + 1;
            store_property (
                p_id                        => l_property_id,
                p_name                      => p_column_name_prefix||'_'||lpad(i,2,'0'),
                p_prompt                    => initcap(p_column_name_prefix)||' '||i,
                p_property_type             => 'TEXT',
                p_db_column                 => p_column_name_prefix||'_'||lpad(i,2,'0'),
                p_display_group_name        => 'SETTINGS' );
            --
        end loop;
    end store_attribute_props;
    --
    procedure store_attribute_comp_props (
        p_attribute_count       in integer,
        p_column_name_prefix    in varchar2 )
    is
    begin
        for i in 1..p_attribute_count loop
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => p_column_name_prefix||'_'||lpad(i,2,'0'),
                p_db_column                   => null,
                p_is_required                 => true,
                p_is_common                   => false,
                p_supports_substitution       => false );
        end loop;
    end store_attribute_comp_props;
    --
    procedure store_subscription_prop (
        p_property_id                in number,
        p_name                       in varchar2,
        p_lov_component_type_id      in number,
        p_help_text                  in varchar2 default null,
        p_validation_plsql_ret_error in varchar2 default null )
    is
    begin
        --
        store_property (
            p_id                         => p_property_id,
            p_name                       => p_name || '_SUBSCRIBED_FROM',
            p_prompt                     => 'Master',
            p_property_type              => 'SUBSCRIPTION',
            p_db_column                  => 'REFERENCE_ID',
            p_display_group_name         => 'SUBSCRIPTION',
            p_lov_type                   => 'COMPONENT',
            p_lov_component_type_id      => p_lov_component_type_id,
            p_lov_component_type_scope   => 'WORKSPACE',
            p_help_text                  => p_help_text,
            p_validation_plsql_ret_error => p_validation_plsql_ret_error );
        --
    end store_subscription_prop;
    --
    procedure store_subscription_comp_prop (
        p_component_type_id in number,
        p_name_prefix       in varchar2 )
    is
    begin
        store_component_prop (
            p_component_type_id => p_component_type_id,
            p_property_name     => p_name_prefix || '_SUBSCRIBED_FROM',
            p_db_column         => null,
            p_is_required       => false,
            p_is_common         => false );
        --
        store_component_prop (
            p_component_type_id => p_component_type_id,
            p_property_name     => 'VERSION_SCN',
            p_db_column         => null,
            p_is_required       => false,
            p_is_common         => false );
        --
    end store_subscription_comp_prop;
    --
    procedure store_tmpl_subscription_prop (
        p_property_id           in number,
        p_name                  in varchar2,
        p_lov_component_type_id in number )
    is
    begin
        store_subscription_prop (
            p_property_id           => p_property_id,
            p_name                  => p_name,
            p_lov_component_type_id => p_lov_component_type_id,
            p_help_text             => q'~<p>You may select an existing template in this workspace or another template in your workspace's schema. By selecting an existing template, you become a subscriber to that template.</p>
<p>To load a new copy of any master template, click <span class="fielddatabold">Refresh</span>.</p>~');
    end store_tmpl_subscription_prop;
    --
    procedure store_template_std_comp_props (
        p_template_type             in varchar2,
        p_name_db_column            in varchar2 default null,
        p_name_import_api_param     in varchar2 default null,
        p_identifier_db_column      in varchar2 default null,
        p_template_type_db_column   in varchar2 default null )
    is
    begin
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'NAME',
            p_db_column                   => p_name_db_column,
            p_import_api_parameter        => p_name_import_api_param,
            p_is_required                 => true,
            p_is_common                   => true,
            p_supports_substitution       => false,
            p_check_uniqueness            => 'PARENT',
            p_help_text                   => q'~Template identifier, this is used when switching and upgrading themes and cannot be modified.~' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'TEMPLATE_IDENTIFIER',
            p_db_column                   => p_identifier_db_column,
            p_is_required                 => true,
            p_is_common                   => true,
            p_supports_substitution       => false,
            p_check_uniqueness            => 'PARENT' );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => p_template_type||'_TEMPLATE_CLASS',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => p_template_type||'_TEMPLATE_SUBSCRIBED_FROM',
            p_db_column                   => null,
            p_is_required                 => false,
            p_is_common                   => false );
        --
        store_component_prop (
            p_component_type_id           => l_component_type_id,
            p_property_name               => 'TEMPLATE_TRANSLATABLE',
            p_db_column                   => null,
            p_is_required                 => true,
            p_is_common                   => false,
            p_default_value               => 'N' );
        --
        if p_template_type not in  ( 'CALENDAR', 'POPUP_LOV' ) then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'PRESET_TEMPLATE_OPTIONS',
                p_db_column                   => null,
                p_is_required                 => false,
                p_is_common                   => false );
            --
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => 'DEFAULT_TEMPLATE_OPTIONS',
                p_db_column                   => null,
                p_is_required                 => false,
                p_is_common                   => false );
        end if;
        --
        if p_template_type in ( 'PAGE', 'CLASSIC_REPORT' ) then
            store_component_prop (
                p_component_type_id           => l_component_type_id,
                p_property_name               => p_template_type||'_TEMPLATE_TYPE',
                p_db_column                   => p_template_type_db_column,
                p_is_required                 => case p_template_type when 'PAGE' then true else false end,
                p_is_common                   => false );
        end if;
        --
    end store_template_std_comp_props;
    --
    procedure store_multiple_value_type_prop (
        p_property_id           in number,
        p_name                  in varchar2,
        p_help_text             in varchar2 default null )
    is
    begin
        --
        store_property (
            p_id                 => p_property_id,
            p_name               => p_name,
            p_prompt             => 'Type',
            p_property_type      => 'SELECT LIST',
            p_lov_type           => 'STATIC',
            p_db_column          => 'MULTI_VALUE_TYPE',
            p_display_group_name => 'MULTIPLE_VALUES',
            p_help_text          => p_help_text );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'Delimited List',
                p_return_value  => 'SEPARATED',
                p_help_text     => q'~<p>Column contains values as a delimited list.</p><pre>APEX,SQL,PL/SQL</pre><p>Use the APEX_STRING.SPLIT function for processing, as in the following example:</p><pre>select column_value from apex_string.split ( <em>:ITEM_NAME</em>, ',' );</pre>~' );
            --
            store_property_lov_values (
                p_property_id   => l_property_id,
                p_display_value => 'JSON Array',
                p_return_value  => 'JSON_ARRAY',
                p_help_text     => q'~<p>The Source column contains a JSON array of scalar values. &PRODUCT_NAME. will raise error messages if the column contains invalid JSON syntax.</p><pre>["APEX", "SQL", "PL/SQL"]</pre><p>Use the JSON_TABLE SQL function for processing, as in the following example:</p><pre>select
    element
from
    json_table ( <em>:ITEM_NAME</em>, '$[*]'
        columns
            element varchar2 path '$'
    );
</pre>~' );

    end store_multiple_value_type_prop;
    --
    procedure store_multiple_value_sep_prop (
        p_property_id           in number,
        p_name                  in varchar2 )
    is
    begin
        --
        store_property (
            p_id                 => p_property_id,
            p_name               => p_name,
            p_prompt             => 'Separator',
            p_property_type      => 'TEXT',
            p_display_length     => 5,
            p_max_length         => 5,
            p_db_column          => 'MULTI_VALUE_SEPARATOR',
            p_display_group_name => 'MULTIPLE_VALUES',
            p_help_text          => q'~<p>Identifies the character which separates values from each other. Note that the <strong>"</strong> (double-quote) and <strong>\</strong> (backslash) characters are not supported as a Value Separator, also not in the <strong>\xxxx</strong> format.</p>~',
            p_examples           => q'~<dl>
<dt>Normal characters:</dt>
<dd><pre>,</pre></dd>
<dd><pre>:</pre></dd>
<dt>Control Characters:</dt>
<dd>Tab (<em>ASCII: 9</em>) <pre>\t</pre></dd>
<dd>Linefeed (<em>ASCII: 10</em>) <pre>\n</pre></dd>
<dt>Other Characters:</dt>
<dd>Use a Unicode encoding value of the form '\xxxx' where 'xxxx' is the hexadecimal value of a character in UCS-2 encoding format.</dd>
<dd>Start of Heading (<em>ASCII: 1</em>) <pre>\0001</pre></dd>
<dd>"@"-Symbol (<em>ASCII: 64</em>) <pre>\0040</pre></dd>
</dl>~' );

    end store_multiple_value_sep_prop;
    --
begin
    sys.dbms_output.enable( 1000000 );

    ------------------------------------------------------------------------
    -- Delete all configuration first
    ------------------------------------------------------------------------
    delete from wwv_pe_properties;
    delete from wwv_pe_component_types;
    delete from wwv_pe_display_groups;

    ------------------------------------------------------------------------
    -- Component Types
    ------------------------------------------------------------------------

    -- Workspace component types
    store_component_type (
        p_id                        => c_comp_type_component_type,
        p_name                      => 'COMPONENT_TYPE',
        p_singular_title            => 'Component Type',
        p_plural_title              => 'Component Types',
        p_table_name                => 'WWV_PE_COMPONENT_TYPES',
        p_display_column            => 'NAME',
        p_create_url                => null,
        p_edit_url                  => null,
        p_copy_url                  => null,
        p_file_path                 => null,
        p_is_page_designer_shared   => false );
    --
    store_component_type (
        p_id                        => c_comp_type_credential,
        p_name                      => 'CREDENTIAL',
        p_singular_title            => 'Web Credential',
        p_plural_title              => 'Web Credentials',
        p_table_name                => 'WWV_CREDENTIALS',
        p_import_api_name           => 'wwv_imp_workspace.create_credential',
        p_display_column            => 'NAME',
        p_create_url                => null,
        p_edit_url                  => 'f?p=4000:1621:%session%:::RP,1621:P1621_ID,P1621_BACK_TO_PAGE:%pk_value%,1621',
        p_copy_url                  => null,
        p_file_path                 => 'workspace/credentials#ext#',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_remote_database,
        p_name                      => 'REMOTE_DATABASE',
        p_singular_title            => 'REST Enabled SQL Database',
        p_plural_title              => 'REST Enabled SQL Databases',
        p_table_name                => 'WWV_REMOTE_SERVERS',
        p_import_api_name           => 'wwv_imp_workspace.create_remote_server',
        p_import_api_params         => q'~[ { name: "p_server_type", expression: "'WEB_SERVICE'" } ]~',
        p_display_column            => 'NAME',
        p_where_clause              => q'~server_type = 'REMOTE_SQL' and remote_sql_database_type is not null ~',
        p_create_url                => null,
        p_edit_url                  => 'f?p=4000:1601:%session%:::RP,1601:P1601_ID:%pk_value%',
        p_copy_url                  => null,
        p_file_path                 => 'workspace/remote_databases#ext#',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_remote_oracle_db,
        p_name                      => 'REMOTE_ORACLE_DATABASE',
        p_singular_title            => 'Remote Oracle Database',
        p_plural_title              => 'Remote Oracle Databases',
        p_table_name                => 'WWV_REMOTE_SERVERS',
        p_import_api_name           => 'wwv_imp_workspace.create_remote_server',
        p_display_column            => 'NAME',
        p_where_clause              => q'~server_type = 'REMOTE_SQL' and remote_sql_database_type = 'ORACLE' ~',
        p_create_url                => null,
        p_edit_url                  => 'f?p=4000:27:%session%::NO::F4000_P27_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_copy_url                  => null,
        p_is_page_designer_shared   => true,
        p_is_query_only             => true );
    --
    store_component_type (
        p_id                        => c_comp_type_auth_server,
        p_name                      => 'AUTHENTICATION_SERVER',
        p_singular_title            => 'Authentication Server',
        p_plural_title              => 'Authentication Servers',
        p_table_name                => 'WWV_REMOTE_SERVERS',
        p_import_api_name           => 'wwv_imp_workspace.create_remote_server',
        p_display_column            => 'NAME',
        p_where_clause              => q'~server_type = 'AUTHENTICATION' ~',
        p_create_url                => null,
        p_edit_url                  => 'f?p=4000:1932:%session%:::RP,1932:P1932_ID,P1932_BACK_TO_PAGE:%pk_value%,1932',
        p_copy_url                  => null,
        p_file_path                 => 'workspace/authentication_servers#ext#' );
    --
    store_component_type (
        p_id                        => c_comp_type_print_server,
        p_name                      => 'PRINT_SERVER',
        p_singular_title            => 'Print Server',
        p_plural_title              => 'Print Servers',
        p_table_name                => 'WWV_REMOTE_SERVERS',
        p_import_api_name           => 'wwv_imp_workspace.create_remote_server',
        p_display_column            => 'NAME',
        p_where_clause              => q'~server_type = 'PRINT_SERVER' ~',
        p_create_url                => null,
        p_edit_url                  => 'f?p=4000:1932:%session%:::RP,1932:P1932_ID,P1932_BACK_TO_PAGE:%pk_value%,1932',
        p_copy_url                  => null,
        p_file_path                 => 'workspace/print_servers#ext#' );

    store_component_type (
        p_id                        => c_comp_type_file_server,
        p_name                      => 'FILE',
        p_singular_title            => 'File Server',
        p_plural_title              => 'File Servers',
        p_table_name                => 'WWV_REMOTE_SERVERS',
        p_import_api_name           => 'wwv_imp_workspace.create_remote_server',
        p_display_column            => 'NAME',
        p_where_clause              => q'~server_type = 'CLOUD_OBJECT_STORE' ~',
        p_create_url                => null,
        p_edit_url                  => 'f?p=4000:1932:%session%:::RP,1932:P1932_ID,P1932_BACK_TO_PAGE:%pk_value%,1932',
        p_copy_url                  => null,
        p_file_path                 => 'workspace/file_servers#ext#' );

    store_component_type (
        p_id                        => c_comp_type_ai_server,
        p_name                      => 'GENERATIVE_AI',
        p_singular_title            => 'Generative AI Service',
        p_plural_title              => 'Generative AI Services',
        p_table_name                => 'WWV_REMOTE_SERVERS',
        p_import_api_name           => 'wwv_imp_workspace.create_remote_server',
        p_display_column            => 'NAME',
        p_where_clause              => q'~server_type = 'GENERATIVE_AI' ~',
        p_create_url                => null,
        p_edit_url                  => 'f?p=4000:9801:%session%:::RP,9801:P9801_ID:%pk_value%',
        p_copy_url                  => null,
        p_file_path                 => 'workspace/generative_ai_servers#ext#',
        p_is_page_designer_shared   => true );

    store_component_type (
        p_id                        => c_comp_type_vector_provider,
        p_name                      => 'VECTOR_PROVIDER',
        p_singular_title            => 'Vector Provider',
        p_plural_title              => 'Vector Providers',
        p_table_name                => 'WWV_REMOTE_SERVERS',
        p_import_api_name           => 'wwv_imp_workspace.create_remote_server',
        p_display_column            => 'NAME',
        p_where_clause              => q'~server_type = 'VECTOR' ~',
        p_create_url                => null,
        p_edit_url                  => 'f?p=4000:9811:%session%:::RP,9811:P9811_ID:%pk_value%',
        p_copy_url                  => null,
        p_file_path                 => 'workspace/vector_providers#ext#',
        p_is_page_designer_shared   => true );

    --
    store_component_type (
        p_id                        => c_comp_type_rest_server,
        p_name                      => 'REST_SERVER',
        p_singular_title            => 'REST Data Source Server',
        p_plural_title              => 'REST Data Source Servers',
        p_table_name                => 'WWV_REMOTE_SERVERS',
        p_import_api_name           => 'wwv_imp_workspace.create_remote_server',
        p_display_column            => 'NAME',
        p_where_clause              => q'~server_type = 'WEB_SERVICE' ~',
        p_create_url                => null,
        p_edit_url                  => 'f?p=4000:1932:%session%:::RP,1932:P1932_ID,P1932_BACK_TO_PAGE:%pk_value%,1932',
        p_copy_url                  => null,
        p_file_path                 => 'workspace/rest_servers#ext#' );
    --
    store_component_type (
        p_id                        => c_comp_type_workspace_file,
        p_name                      => 'STATIC_WORKSPACE_FILE',
        p_singular_title            => 'Static Workspace File',
        p_plural_title              => 'Static Workspace Files',
        p_table_name                => 'WWV_FLOW_COMPANY_STATIC_FILES',
        p_import_api_name           => 'wwv_flow_imp_shared.create_static_file',
        p_display_column            => 'FILE_NAME',
        p_create_url                => 'f?p=4000:314:%session%:::RP,314',
        p_edit_url                  => 'f?p=4000:315:%session%:::RP,315:P315_ID:%pk_value%',
        p_copy_url                  => null,
        p_file_path                 => 'workspace/static_files#ext#' );
    --
    store_component_type (
        p_id                        => c_comp_type_page_pattern,
        p_name                      => 'PAGE_PATTERN',
        p_singular_title            => 'Page Pattern',
        p_plural_title              => 'Page Patterns',
        p_table_name                => 'WWV_FLOW_STEPS_DEV',
        p_view_name                 => 'APEX_APPLICATION_PAGES',
        p_import_api_name           => 'wwv_flow_imp_page.create_page',
        p_pk_column                 => 'INTERNAL_ID',
        p_display_column            => 'NAME',
        p_where_clause              => q'~is_page_pattern = 'Y' ~',
        p_file_path                 => 'workspace/page_patterns#ext#' );
    --
    store_component_type (
        p_id                        => c_comp_type_app_group,
        p_name                      => 'APP_GROUP',
        p_singular_title            => 'Application Group',
        p_plural_title              => 'Application Groups',
        p_table_name                => 'WWV_FLOW_APPLICATION_GROUPS',
        p_display_column            => 'GROUP_NAME',
        p_create_url                => null,
        p_edit_url                  => null,
        p_copy_url                  => null,
        p_file_path                 => 'workspace/app_groups#ext#' );

    -- Application related component types
    store_component_type (
        p_id                        => c_comp_type_application,
        p_name                      => 'APPLICATION',
        p_singular_title            => 'Application',
        p_plural_title              => 'Applications',
        p_table_name                => 'WWV_FLOWS',
        p_import_api_name           => 'wwv_imp_workspace.create_flow',
        p_display_column            => 'NAME',
        p_edit_url                  => 'f?p=4000:4001:%session%::::F4000_P1_FLOW,FB_FLOW_ID:%pk_value%,%pk_value%',
        p_file_path                 => 'application/',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_app_substitution,
        p_name                      => 'APP_SUBSTITUTION',
        p_singular_title            => 'Substitution',
        p_plural_title              => 'Substitutions',
        p_table_name                => 'WWV_FLOW_SUBSTITUTIONS_DEV',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'SUBSTITUTION_STRING',
        p_edit_url                  => 'f?p=4000:4001:%session%::NO:2:FB_FLOW_ID:%application_id%' );

    -- Theme related component types
    store_component_type (
        p_id                        => c_comp_type_theme,
        p_name                      => 'THEME',
        p_singular_title            => 'Theme',
        p_plural_title              => 'Themes',
        p_table_name                => 'WWV_FLOW_THEMES',
        p_import_api_name           => 'wwv_flow_imp_shared.create_theme',
        p_pk_column                 => 'THEME_ID',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'THEME_NAME',
        p_file_path                 => 'shared_components/theme_#id#/',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_theme_file,
        p_name                      => 'THEME_FILE',
        p_singular_title            => 'File',
        p_plural_title              => 'Files',
        p_table_name                => 'WWV_FLOW_THEME_FILES',
        p_import_api_name           => 'wwv_flow_imp_shared.create_static_file',
        p_view_name                 => 'APEX_APPLICATION_THEME_FILES',
        p_parent_component_type_id  => c_comp_type_theme,
        p_parent_fk_column          => 'THEME_ID',
        p_display_column            => 'FILE_NAME',
        p_edit_url                  => 'f?p=4000:118:%session%:::118:P118_SCOPE,P118_ID,FB_FLOW_ID:THEME,%pk_value%,%application_id%' );
    --
    store_component_type (
        p_id                        => c_comp_type_theme_style,
        p_name                      => 'THEME_STYLE',
        p_singular_title            => 'Style',
        p_plural_title              => 'Styles',
        p_table_name                => 'WWV_FLOW_THEME_STYLES',
        p_import_api_name           => 'wwv_flow_imp_shared.create_theme_style',
        p_view_name                 => 'APEX_APPLICATION_THEME_STYLES',
        p_parent_component_type_id  => c_comp_type_theme,
        p_parent_fk_column          => 'THEME_ID',
        p_display_column            => 'NAME',
        p_edit_url                  => 'f?p=4000:177:%session%:::177:P177_ID,FB_FLOW_ID:%pk_value%,%application_id%' );
    --
    store_component_type (
        p_id                        => c_comp_type_global_tmpl_option,
        p_name                      => 'GLOBAL_TEMPLATE_OPTION',
        p_singular_title            => 'Global Template Option',
        p_plural_title              => 'Global Template Options',
        p_table_name                => 'WWV_FLOW_TEMPLATE_OPTIONS',
        p_import_api_name           => 'wwv_flow_imp_shared.create_template_option',
        p_view_name                 => 'APEX_APPL_TEMPLATE_OPTIONS',
        p_parent_component_type_id  => c_comp_type_theme,
        p_parent_fk_column          => 'THEME_ID',
        p_display_column            => 'DISPLAY_NAME',
        p_where_clause              => q'~page_template_id       is null
and region_template_id     is null
and report_template_id     is null
and breadcrumb_template_id is null
and list_template_id       is null
and field_template_id      is null
and button_template_id     is null~',
        p_edit_url                  => get_parent_component_type_url (
                                                p_component_type_id => c_comp_type_theme,
                                                p_url_type          => 'EDIT' ),
        p_file_path                 => 'global_template_options#ext#' );
    --
    store_component_type (
        p_id                        => c_comp_type_tmpl_option_group,
        p_name                      => 'TEMPLATE_OPTION_GROUP',
        p_singular_title            => 'Template Option Group',
        p_plural_title              => 'Template Option Groups',
        p_table_name                => 'WWV_FLOW_TEMPLATE_OPT_GROUPS',
        p_import_api_name           => 'wwv_flow_imp_shared.create_template_opt_group',
        p_view_name                 => 'APEX_APPL_TEMPLATE_OPT_GROUPS',
        p_parent_component_type_id  => c_comp_type_theme,
        p_parent_fk_column          => 'THEME_ID',
        p_display_column            => 'DISPLAY_NAME',
        p_edit_url                  => 'f?p=4000:81:%session%:::RP,81::',
        p_file_path                 => 'template_option_groups#ext#' );

    -- Templates
    store_component_type (
        p_id                        => c_comp_type_page_template,
        p_name                      => 'PAGE_TEMPLATE',
        p_singular_title            => 'Page Template',
        p_plural_title              => 'Page Templates',
        p_table_name                => 'WWV_FLOW_TEMPLATES',
        p_import_api_name           => 'wwv_flow_imp_shared.create_template',
        p_parent_component_type_id  => c_comp_type_theme,
        p_parent_fk_column          => 'THEME_ID',
        p_display_column            => 'NAME',
        p_create_url                => 'f?p=4000:162:%session%:::162:FB_FLOW_ID:%application_id%',
        p_edit_url                  => 'f?p=4000:4307:%session%:::4307:F4000_P4307_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_copy_url                  => 'f?p=4000:568:%session%:::568:FB_FLOW_ID,P568_FROM_TEMPLATE_ID:%application_id%,%pk_value%',
        p_file_path                 => 'page_templates#ext#',
        p_is_page_designer_shared   => true );
    --
    store_comp_type_tmpl_option (
        p_component_type_id         => c_comp_type_page_tmpl_option,
        p_parent_comp_type_id       => c_comp_type_page_template,
        p_type_name                 => 'PAGE',
        p_edit_page_no              => 4307 );
    --
    store_component_type (
        p_id                        => c_comp_type_region_template,
        p_name                      => 'REGION_TEMPLATE',
        p_singular_title            => 'Region Template',
        p_plural_title              => 'Region Templates',
        p_table_name                => 'WWV_FLOW_PAGE_PLUG_TEMPLATES',
        p_import_api_name           => 'wwv_flow_imp_shared.create_plug_template',
        p_parent_component_type_id  => c_comp_type_theme,
        p_parent_fk_column          => 'THEME_ID',
        p_display_column            => 'PAGE_PLUG_TEMPLATE_NAME',
        p_create_url                => 'f?p=4000:162:%session%:::162:FB_FLOW_ID:%application_id%',
        p_edit_url                  => 'f?p=4000:4653:%session%:::4653:F4000_P4653_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_copy_url                  => 'f?p=4000:538:%session%:::538:FB_FLOW_ID,P538_FROM_TEMPLATE_ID:%application_id%,%pk_value%',
        p_file_path                 => 'region_templates#ext#',
        p_is_page_designer_shared   => true );
    --
    store_comp_type_tmpl_option (
        p_component_type_id         => c_comp_type_region_tmpl_option,
        p_parent_comp_type_id       => c_comp_type_region_template,
        p_type_name                 => 'REGION',
        p_edit_page_no              => 4653 );
    --
    store_component_type (
        p_id                        => c_comp_type_field_template,
        p_name                      => 'FIELD_TEMPLATE',
        p_singular_title            => 'Field Template',
        p_plural_title              => 'Field Templates',
        p_table_name                => 'WWV_FLOW_FIELD_TEMPLATES',
        p_import_api_name           => 'wwv_flow_imp_shared.create_field_template',
        p_parent_component_type_id  => c_comp_type_theme,
        p_parent_fk_column          => 'THEME_ID',
        p_display_column            => 'TEMPLATE_NAME',
        p_create_url                => 'f?p=4000:162:%session%:::162:FB_FLOW_ID:%application_id%',
        p_edit_url                  => 'f?p=4000:85:%session%:::85:F4000_P85_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_copy_url                  => 'f?p=4000:574:%session%:::574:FB_FLOW_ID,P574_FROM_TEMPLATE_ID:%application_id%,%pk_value%',
        p_file_path                 => 'field_templates#ext#',
        p_is_page_designer_shared   => true );
    --
    store_comp_type_tmpl_option (
        p_component_type_id         => c_comp_type_field_tmpl_option,
        p_parent_comp_type_id       => c_comp_type_field_template,
        p_type_name                 => 'FIELD',
        p_edit_page_no              => 85 );
    --
    store_component_type (
        p_id                        => c_comp_type_button_template,
        p_name                      => 'BUTTON_TEMPLATE',
        p_singular_title            => 'Button Template',
        p_plural_title              => 'Button Templates',
        p_table_name                => 'WWV_FLOW_BUTTON_TEMPLATES',
        p_import_api_name           => 'wwv_flow_imp_shared.create_button_templates',
        p_parent_component_type_id  => c_comp_type_theme,
        p_parent_fk_column          => 'THEME_ID',
        p_display_column            => 'TEMPLATE_NAME',
        p_create_url                => 'f?p=4000:162:%session%:::162:FB_FLOW_ID:%application_id%',
        p_edit_url                  => 'f?p=4000:204:%session%:::204:F4000_P204_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_copy_url                  => 'f?p=4000:571:%session%:::571:FB_FLOW_ID,P571_FROM_TEMPLATE_ID:%application_id%,%pk_value%',
        p_file_path                 => 'button_templates#ext#',
        p_is_page_designer_shared   => true );
    --
    store_comp_type_tmpl_option (
        p_component_type_id         => c_comp_type_button_tmpl_option,
        p_parent_comp_type_id       => c_comp_type_button_template,
        p_type_name                 => 'BUTTON',
        p_edit_page_no              => 204 );
    --
    store_component_type (
        p_id                        => c_comp_type_list_template,
        p_name                      => 'LIST_TEMPLATE',
        p_singular_title            => 'List Template',
        p_plural_title              => 'List Templates',
        p_table_name                => 'WWV_FLOW_LIST_TEMPLATES',
        p_import_api_name           => 'wwv_flow_imp_shared.create_list_template',
        p_parent_component_type_id  => c_comp_type_theme,
        p_parent_fk_column          => 'THEME_ID',
        p_display_column            => 'LIST_TEMPLATE_NAME',
        p_create_url                => 'f?p=4000:162:%session%:::162:FB_FLOW_ID:%application_id%',
        p_edit_url                  => 'f?p=4000:4655:%session%:::4655:F4000_P4655_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_copy_url                  => 'f?p=4000:573:%session%:::573:FB_FLOW_ID,P573_FROM_TEMPLATE_ID:%application_id%,%pk_value%',
        p_file_path                 => 'list_templates#ext#',
        p_is_page_designer_shared   => true );
    --
    store_comp_type_tmpl_option (
        p_component_type_id         => c_comp_type_list_tmpl_option,
        p_parent_comp_type_id       => c_comp_type_list_template,
        p_type_name                 => 'LIST',
        p_edit_page_no              => 4655 );
    --
    store_component_type (
        p_id                        => c_comp_type_bc_template,
        p_name                      => 'BREADCRUMB_TEMPLATE',
        p_singular_title            => 'Breadcrumb Template',
        p_plural_title              => 'Breadcrumb Templates',
        p_table_name                => 'WWV_FLOW_MENU_TEMPLATES',
        p_import_api_name           => 'wwv_flow_imp_shared.create_menu_template',
        p_parent_component_type_id  => c_comp_type_theme,
        p_parent_fk_column          => 'THEME_ID',
        p_display_column            => 'NAME',
        p_create_url                => 'f?p=4000:162:%session%:::162:FB_FLOW_ID:%application_id%',
        p_edit_url                  => 'f?p=4000:289:%session%:::289:F4000_P289_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_copy_url                  => 'f?p=4000:636:%session%:::636,596:FB_FLOW_ID,P636_FROM_TEMPLATE_ID:%application_id%,%pk_value%',
        p_file_path                 => 'breadcrumb_templates#ext#',
        p_is_page_designer_shared   => true );
    --
    store_comp_type_tmpl_option (
        p_component_type_id         => c_comp_type_bc_tmpl_option,
        p_parent_comp_type_id       => c_comp_type_bc_template,
        p_type_name                 => 'BREADCRUMB',
        p_edit_page_no              => 289 );
    --
    store_component_type (
        p_id                        => c_comp_type_calendar_template,
        p_name                      => 'CALENDAR_TEMPLATE',
        p_singular_title            => 'Legacy Calendar Template',
        p_plural_title              => 'Legacy Calendar Templates',
        p_table_name                => 'WWV_FLOW_CAL_TEMPLATES',
        p_import_api_name           => 'wwv_flow_imp_shared.create_calendar_template',
        p_parent_component_type_id  => c_comp_type_theme,
        p_parent_fk_column          => 'THEME_ID',
        p_display_column            => 'NAME',
        p_create_url                => 'f?p=4000:162:%session%:::162:FB_FLOW_ID:%application_id%',
        p_edit_url                  => 'f?p=4000:697:%session%:::697:F4000_P697_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_file_path                 => 'legacy_calendar_templates#ext#',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_report_template,
        p_name                      => 'REPORT_TEMPLATE',
        p_singular_title            => 'Classic Report Template',
        p_plural_title              => 'Classic Report Templates',
        p_table_name                => 'WWV_FLOW_ROW_TEMPLATES',
        p_import_api_name           => 'wwv_flow_imp_shared.create_row_template',
        p_parent_component_type_id  => c_comp_type_theme,
        p_parent_fk_column          => 'THEME_ID',
        p_display_column            => 'ROW_TEMPLATE_NAME',
        p_create_url                => 'f?p=4000:162:%session%:::162:FB_FLOW_ID:%application_id%',
        p_edit_url                  => 'f?p=4000:258:%session%:::258:F4000_P258_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_copy_url                  => 'f?p=4000:569:%session%:::569:FB_FLOW_ID,P569_FROM_TEMPLATE_ID:%application_id%,%pk_value%',
        p_file_path                 => 'classic_report_templates#ext#',
        p_is_page_designer_shared   => true );
    --
    store_comp_type_tmpl_option (
        p_component_type_id     => c_comp_type_report_tmpl_option,
        p_parent_comp_type_id   => c_comp_type_report_template,
        p_type_name             => 'REPORT',
        p_edit_page_no          => 258 );       --column template
    --
/* $$$ the following block is commented out due to the presence of 2 edit pages for report templates $$$ */
/*    store_comp_type_tmpl_option (
        p_component_type_id     => c_comp_type_report_tmpl_option,
        p_parent_comp_type_id   => c_comp_type_report_template,
        p_type_name             => 'REPORT_ROW',
        p_edit_page_no          => 4654 );      -- row template */
    --
    store_component_type (
        p_id                        => c_comp_type_popup_lov_template,
        p_name                      => 'POPUP_LOV_TEMPLATE',
        p_singular_title            => 'Popup LOV Template',
        p_plural_title              => 'Popup LOV Templates',
        p_table_name                => 'WWV_FLOW_POPUP_LOV_TEMPLATE',
        p_import_api_name           => 'wwv_flow_imp_shared.create_popup_lov_template',
        p_parent_component_type_id  => c_comp_type_theme,
        p_parent_fk_column          => 'THEME_ID',
        p_display_column            => 'PAGE_TITLE',
        p_edit_url                  => 'f?p=4000:251:%session%:::251:F4000_P251_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_file_path                 => 'popup_lov_templates#ext#',
        p_is_page_designer_shared   => true );
    --
    -- Shared Component related component types
    store_component_type (
        p_id                        => c_comp_type_app_item,
        p_name                      => 'APP_ITEM',
        p_singular_title            => 'Application Item',
        p_plural_title              => 'Application Items',
        p_table_name                => 'WWV_FLOW_ITEMS',
        p_import_api_name           => 'wwv_flow_imp_shared.create_flow_item',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'NAME',
        p_create_url                => 'f?p=4000:4303:%session%:::4303:FB_FLOW_ID:%application_id%',
        p_edit_url                  => 'f?p=4000:4303:%session%:::4303:F4000_P4303_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_file_path                 => 'shared_components/app_items#ext#',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_app_computation,
        p_name                      => 'APP_COMPUTATION',
        p_singular_title            => 'Application Computation',
        p_plural_title              => 'Application Computations',
        p_table_name                => 'WWV_FLOW_COMPUTATIONS',
        p_import_api_name           => 'wwv_flow_imp_shared.create_flow_computation',
        p_view_name                 => 'APEX_APPLICATION_COMPUTATIONS',
        p_parent_component_type_id  => C_COMP_TYPE_APPLICATION,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'COMPUTATION_ITEM',
        p_edit_url                  => 'f?p=4000:4304:%session%::::F4000_P4304_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_file_path                 => 'shared_components/app_computations#ext#' );
    --
    store_component_type (
        p_id                        => c_comp_type_app_process,
        p_name                      => 'APP_PROCESS',
        p_singular_title            => 'Application Process',
        p_plural_title              => 'Application Processes',
        p_table_name                => 'WWV_FLOW_PROCESSING',
        p_import_api_name           => 'wwv_flow_imp_shared.create_flow_process',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'PROCESS_NAME',
        p_create_url                => 'f?p=4000:147:%session%:::147,148,150:FB_FLOW_ID:%application_id%',
        p_edit_url                  => 'f?p=4000:4309:%session%::::F4000_P4309_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_file_path                 => 'shared_components/app_processes#ext#' );
    --
    store_component_type (
        p_id                        => c_comp_type_build_option,
        p_name                      => 'BUILD_OPTION',
        p_singular_title            => 'Build Option',
        p_plural_title              => 'Build Options',
        p_table_name                => 'WWV_FLOW_PATCHES',
        p_import_api_name           => 'wwv_flow_imp_shared.create_build_option',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'PATCH_NAME',
        p_create_url                => 'f?p=4000:4911:%session%:::4911:FB_FLOW_ID:%application_id%',
        p_edit_url                  => 'f?p=4000:4911:%session%:::4911:F4000_P4911_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_copy_url                  => 'f?p=4000:4912:%session%:::4912:P4912_ID,FB_FLOW_ID,P4912_COPY_SOURCE,P4912_COPY_FROM_APPLICATION_ID:%pk_value%,%application_id%,THIS,%application_id%',
        p_file_path                 => 'shared_components/build_options#ext#',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_authentication,
        p_name                      => 'AUTHENTICATION',
        p_singular_title            => 'Authentication',
        p_plural_title              => 'Authentications',
        p_table_name                => 'WWV_FLOW_AUTHENTICATIONS',
        p_import_api_name           => 'wwv_flow_imp_shared.create_authentication',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'NAME',
        p_plugin_type               => 'AUTHENTICATION TYPE',
        p_plugin_column             => 'SCHEME_TYPE',
        p_create_url                => 'f?p=4000:815:%session%:::815,4495:FB_FLOW_ID:%application_id%',
        p_edit_url                  => 'f?p=4000:4495:%session%::::P4495_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_file_path                 => 'shared_components/authentications#ext#',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_authorization,
        p_name                      => 'AUTHORIZATION',
        p_singular_title            => 'Authorization',
        p_plural_title              => 'Authorizations',
        p_table_name                => 'WWV_FLOW_SECURITY_SCHEMES',
        p_import_api_name           => 'wwv_flow_imp_shared.create_security_scheme',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'NAME',
        p_plugin_type               => 'AUTHORIZATION TYPE',
        p_plugin_column             => 'SCHEME_TYPE',
        p_create_url                => 'f?p=4000:184:%session%::::FB_FLOW_ID:%application_id%',
        p_edit_url                  => 'f?p=4000:4008:%session%::::F4000_P4008_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_copy_url                  => 'f?p=4000:493:%session%:::493:FB_FLOW_ID,P493_ID:%application_id%,%pk_value%',
        p_file_path                 => 'shared_components/authorizations#ext#',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_acl_role,
        p_name                      => 'ACL_ROLE',
        p_singular_title            => 'Role',
        p_plural_title              => 'Roles',
        p_table_name                => 'WWV_FLOW_FND_USER_GROUPS',
        p_import_api_name           => 'wwv_flow_imp_shared.create_acl_role',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'GROUP_NAME',
        p_create_url                => 'f?p=4000:2320:%session%:::RP,2320:FB_FLOW_ID:%application_id%',
        p_edit_url                  => 'f?p=4000:2320:%session%:::RP,2320:F4000_P2320_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_file_path                 => 'shared_components/acl_roles#ext#' );
    --
    store_component_type (
        p_id                        => c_comp_type_dualityview_src,
        p_name                      => 'DUALITY_VIEW_SOURCE',
        p_singular_title            => 'Duality View',
        p_plural_title              => 'Duality Views',
        p_table_name                => 'WWV_FLOW_DUALITY_VIEW_SOURCES',
        p_import_api_name           => 'wwv_flow_imp_shared.create_document_source',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'NAME',
        p_create_url                => null,
        p_edit_url                  => 'f?p=4000:7110:%session%:::RP,7110:FB_FLOW_ID,P7100_SOURCE_TYPE,P7110_ID:%application_id%,DUALITY_VIEW,%pk_value%',
        p_copy_url                  => null,
        p_file_path                 => 'shared_components/duality_view_sources#ext#',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_jsondocument_src,
        p_name                      => 'JSON_DOCUMENT_SOURCE',
        p_singular_title            => 'JSON Source',
        p_plural_title              => 'JSON Sources',
        p_table_name                => 'WWV_FLOW_JSON_SOURCES',
        p_import_api_name           => 'wwv_flow_imp_shared.create_document_source',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'NAME',
        p_create_url                => null,
        p_edit_url                  => 'f?p=4000:7110:%session%:::RP,7110:FB_FLOW_ID,P7100_SOURCE_TYPE,P7110_ID:%application_id%,JSON_TABLE,%pk_value%',
        p_copy_url                  => null,
        p_file_path                 => 'shared_components/json_collection_sources#ext#',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_ai_config,
        p_name                      => 'AI_CONFIG',
        p_singular_title            => 'AI Configuration',
        p_plural_title              => 'AI Configurations',
        p_table_name                => 'WWV_FLOW_AI_CONFIGS',
        p_import_api_name           => 'wwv_flow_imp_shared.create_ai_config',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'NAME',
        p_create_url                => 'f?p=4000:3504:%session%:::RP,3504:FB_FLOW_ID:%application_id%',
        p_edit_url                  => 'f?p=4000:3504:%session%:::RP,3504:FB_FLOW_ID,P3504_ID:%application_id%,%pk_value%',
        p_copy_url                  => null,
        p_file_path                 => 'shared_components/ai_configs#ext#',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_ai_config_rag_src,
        p_name                      => 'AI_CONFIG_RAG_SOURCE',
        p_singular_title            => 'RAG Source',
        p_plural_title              => 'RAG Sources',
        p_table_name                => 'WWV_FLOW_AI_CONFIG_RAG_SOURCES',
        p_import_api_name           => 'wwv_flow_imp_shared.create_ai_config_rag_source',
        p_parent_component_type_id  => c_comp_type_ai_config,
        p_parent_fk_column          => 'AI_CONFIG_ID',
        p_display_column            => 'NAME',
        p_create_url                => null,
        p_edit_url                  => 'f?p=4000:3505:%session%:::RP,3505:FB_FLOW_ID,P3505_ID:%application_id%,%pk_value%',
        p_copy_url                  => null );
    --
    store_component_type (
        p_id                        => c_comp_type_web_src_module,
        p_name                      => 'WEB_SRC_MODULE',
        p_singular_title            => 'REST Data Source',
        p_plural_title              => 'REST Data Sources',
        p_table_name                => 'WWV_FLOW_WEB_SRC_MODULES',
        p_import_api_name           => 'wwv_flow_imp_shared.create_web_source_module',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'NAME',
        p_plugin_type               => 'WEB SOURCE TYPE',
        p_plugin_column             => 'WEB_SOURCE_TYPE',
        p_where_clause              => null,
        p_create_url                => 'f?p=4000:1905:%session%:::1941,1942,1943,1944,1946,1905:FB_FLOW_ID:%application_id%',
        p_edit_url                  => 'f?p=4000:1921:%session%:::RP,1921:P1921_WEB_SRC_MODULE_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_copy_url                  => null,
        p_file_path                 => 'shared_components/rest_data_sources#ext#',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_web_src_array_col,
        p_name                      => 'WEB_SRC_ARRAY_COLUMN',
        p_singular_title            => 'Nested Rows',
        p_plural_title              => 'Nested Row Columns',
        p_table_name                => 'WWV_FLOW_SRC_ARR_COLS_DEV',
        p_import_api_name           => 'wwv_flow_imp_shared.create_data_profile_col',
        p_parent_component_type_id  => c_comp_type_web_src_module,
        p_parent_fk_column          => 'WEB_SRC_MODULE_ID',
        p_display_column            => 'DISPLAY_NAME',
        p_where_clause              => null,
        p_create_url                => null,
        p_edit_url                  => null,
        p_copy_url                  => null,
        p_file_path                 => 'shared_components/rest_data_sources/nested_rows#ext#',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_dualityvw_arr_col,
        p_name                      => 'DUALITY_VIEW_SRC_ARRAY_COLUMN',
        p_singular_title            => 'Nested Rows',
        p_plural_title              => 'Nested Row Columns',
        p_table_name                => 'WWV_FLOW_SRC_ARR_COLS_DEV',
        p_import_api_name           => 'wwv_flow_imp_shared.create_data_profile_col',
        p_parent_component_type_id  => c_comp_type_dualityview_src,
        p_parent_fk_column          => 'DOCUMENT_SOURCE_ID',
        p_display_column            => 'DISPLAY_NAME',
        p_where_clause              => null,
        p_create_url                => null,
        p_edit_url                  => null,
        p_copy_url                  => null,
        p_file_path                 => 'shared_components/duality_view_sources/nested_rows#ext#',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_jsoncoll_arr_col,
        p_name                      => 'JSON_COLLECTION_SRC_ARRAY_COLUMN',
        p_singular_title            => 'Nested Rows',
        p_plural_title              => 'Nested Row Columns',
        p_table_name                => 'WWV_FLOW_SRC_ARR_COLS_DEV',
        p_import_api_name           => 'wwv_flow_imp_shared.create_data_profile_col',
        p_parent_component_type_id  => c_comp_type_jsondocument_src,
        p_parent_fk_column          => 'DOCUMENT_SOURCE_ID',
        p_display_column            => 'DISPLAY_NAME',
        p_where_clause              => null,
        p_create_url                => null,
        p_edit_url                  => null,
        p_copy_url                  => null,
        p_file_path                 => 'shared_components/json_collection_sources/nested_rows#ext#',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_web_src_mod_param,
        p_name                      => 'WEB_SRC_MODULE_PARAM',
        p_singular_title            => 'Parameter',
        p_plural_title              => 'Parameters',
        p_table_name                => 'WWV_FLOW_WEB_SRC_PARAMS_DEV',
        p_import_api_name           => 'wwv_flow_imp_shared.create_web_source_param',
        p_parent_component_type_id  => c_comp_type_web_src_module,
        p_parent_fk_column          => 'WEB_SRC_MODULE_ID',
        p_display_column            => 'NAME',
        p_where_clause              => 'web_src_operation_id is null',
        p_create_url                => null,
        p_edit_url                  => null,
        p_copy_url                  => null,
        p_is_page_designer_shared   => true );
    --
    store_comp_type_data_profile (
        p_profile_type_id           => c_comp_type_web_src_mod_dp,
        p_column_type_id            => c_comp_type_web_src_mod_dp_col,
        p_name_prefix               => 'WEB_SRC_MODULE',
        p_parent_type_id            => c_comp_type_web_src_module,
        p_parent_fk_column          => 'WEB_SRC_MODULE_ID' );
    --
    store_comp_type_data_profile (
        p_profile_type_id           => c_comp_type_dualityvw_dp,
        p_column_type_id            => c_comp_type_dualityvw_dp_col,
        p_name_prefix               => 'DUALITY_VIEW',
        p_parent_type_id            => c_comp_type_dualityview_src,
        p_parent_fk_column          => 'DOCUMENT_SOURCE_ID' );
    --
    store_comp_type_data_profile (
        p_profile_type_id           => c_comp_type_jsoncoll_dp,
        p_column_type_id            => c_comp_type_jsoncoll_dp_col,
        p_name_prefix               => 'JSON_COLLECTION',
        p_parent_type_id            => c_comp_type_jsondocument_src,
        p_parent_fk_column          => 'DOCUMENT_SOURCE_ID' );
    --
    store_component_type (
        p_id                        => c_comp_type_web_src_operation,
        p_name                      => 'WEB_SRC_OPERATION',
        p_singular_title            => 'Operation',
        p_plural_title              => 'Operations',
        p_table_name                => 'WWV_FLOW_WEB_SRC_OPERS_DEV',
        p_import_api_name           => 'wwv_flow_imp_shared.create_web_source_operation',
        p_parent_component_type_id  => c_comp_type_web_src_module,
        p_parent_fk_column          => 'WEB_SRC_MODULE_ID',
        p_display_column            => 'DISPLAY_NAME',
        p_where_clause              => null,
        p_create_url                => null,
        p_edit_url                  => 'f?p=4000:1922:%session%::NO:1922,RP:P1922_ID:%pk_value%',
        p_copy_url                  => null,
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_web_src_oper_param,
        p_name                      => 'WEB_SRC_OPER_PARAM',
        p_singular_title            => 'Parameter',
        p_plural_title              => 'Parameters',
        p_table_name                => 'WWV_FLOW_WEB_SRC_PARAMS_DEV',
        p_import_api_name           => 'wwv_flow_imp_shared.create_web_source_param',
        p_parent_component_type_id  => c_comp_type_web_src_operation,
        p_parent_fk_column          => 'WEB_SRC_OPERATION_ID',
        p_display_column            => 'NAME',
        p_where_clause              => 'web_src_operation_id is not null',
        p_create_url                => null,
        p_edit_url                  => 'f?p=4000:1922:%session%::NO:1922,RP:P1922_ID:%parent_pk_value%',
        p_copy_url                  => null,
        p_is_page_designer_shared   => true );
    --
    store_comp_type_data_profile (
        p_profile_type_id           => c_comp_type_web_src_oper_dp,
        p_column_type_id            => c_comp_type_web_src_oper_dpcol,
        p_name_prefix               => 'WEB_SRC_OPERATION',
        p_parent_type_id            => c_comp_type_web_src_operation,
        p_parent_fk_column          => 'WEB_SRC_OPERATION_ID' );
    --
    store_component_type (
        p_id                        => c_comp_type_web_src_sync_step,
        p_name                      => 'WEB_SRC_SYNC_STEP',
        p_singular_title            => 'Synchronization Step',
        p_plural_title              => 'Synchronization Steps',
        p_table_name                => 'WWV_FLOW_WEB_SRC_SYNC_STEPS',
        p_import_api_name           => 'wwv_flow_imp_shared.create_web_source_sync_step',
        p_parent_component_type_id  => c_comp_type_web_src_module,
        p_parent_fk_column          => 'WEB_SRC_MODULE_ID',
        p_display_column            => 'STATIC_ID',
        p_where_clause              => null,
        p_create_url                => null,
        p_edit_url                  => 'f?p=4000:1951:%session%::NO:1951,RP:P1951_WEB_SRC_MODULE_ID:%parent_pk_value%',
        p_copy_url                  => null );
    --
    store_component_type (
        p_id                        => c_comp_type_web_src_sync_param,
        p_name                      => 'WEB_SRC_SYNC_STEP_WS_PARAM',
        p_singular_title            => 'Parameter',
        p_plural_title              => 'Parameters',
        p_table_name                => 'WWV_FLOW_WEB_SRC_CPARAMS_DEV',
        p_import_api_name           => 'wwv_flow_imp_shared.create_web_source_comp_param',
        p_view_name                 => 'APEX_APPL_WEB_SRC_COMP_PARAMS',
        p_parent_component_type_id  => c_comp_type_web_src_sync_step,
        p_parent_fk_column          => 'MODULE_SYNC_STEP_ID',
        p_display_column            => 'WEB_SRC_PARAM_ID' );
    --
    store_component_type (
        p_id                        => c_comp_type_web_src_query,
        p_name                      => 'WEB_SRC_QUERY',
        p_singular_title            => 'REST Source Query',
        p_plural_title              => 'REST Source Query',
        p_table_name                => 'WWV_FLOW_WEB_SRC_MODULES',
        p_import_api_name           => 'wwv_flow_imp_shared.create_web_source_module',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'NAME',
        p_where_clause              => q'~exists( select 1 from wwv_flow_web_src_operations where web_src_module_id=wwv_flow_web_src_modules.id and database_operation = 'FETCH_COLLECTION')~',
        p_create_url                => null,
        p_edit_url                  => 'f?p=4000:1921:%session%::NO:1921,RP:P1921_WEB_SRC_MODULE_ID:%pk_value%',
        p_copy_url                  => null,
        p_is_page_designer_shared   => true,
        p_is_query_only             => true );
    --
    -- Helper component type to get all parameters of a web src module (incl. operation parameters)
    -- This is needed for wwv_flow_web_src_comp_params
    store_component_type (
        p_id                        => c_comp_type_web_src_param,
        p_name                      => 'WEB_SRC_PARAM',
        p_singular_title            => 'Parameter',
        p_plural_title              => 'Parameters',
        p_table_name                => 'WWV_FLOW_WEB_SRC_PARAMS_DEV',
        p_import_api_name           => 'wwv_flow_imp_shared.create_web_source_param',
        p_parent_component_type_id  => c_comp_type_web_src_module,
        p_parent_fk_column          => 'WEB_SRC_MODULE_ID',
        p_display_column            => 'NAME',
        p_where_clause              => null,
        p_create_url                => null,
        p_edit_url                  => 'f?p=4000:1922:%session%::NO:1922,RP:P1922_ID:%parent_pk_value%',
        p_copy_url                  => null,
        p_is_page_designer_shared   => true,
        p_is_query_only             => true );
    --
    store_component_type (
        p_id                        => c_comp_type_application_file,
        p_name                      => 'STATIC_APPLICATION_FILE',
        p_singular_title            => 'Static Application File',
        p_plural_title              => 'Static Application Files',
        p_table_name                => 'WWV_FLOW_STATIC_FILES',
        p_import_api_name           => 'wwv_flow_imp_shared.create_static_file',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'FILE_NAME',
        p_create_url                => 'f?p=4000:271:%session%:::RP,271:FB_FLOW_ID:%application_id%',
        p_edit_url                  => 'f?p=4000:275:%session%:::RP,275:P275_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_copy_url                  => null,
        p_file_path                 => 'shared_components/app_static_files#ext#' );
    --
    store_component_type (
        p_id                        => c_comp_type_text_message,
        p_name                      => 'TEXT_MESSAGE',
        p_singular_title            => 'Text Message',
        p_plural_title              => 'Text Messages',
        p_table_name                => 'WWV_FLOW_MESSAGES$',
        p_import_api_name           => 'wwv_flow_imp_shared.create_message',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'NAME',
        p_edit_url                  => 'f?p=4000:4009:%session%:::4009:P4009_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_file_path                 => 'shared_components/messages#ext#' );
    --
    store_component_type (
        p_id                        => c_comp_type_app_setting,
        p_name                      => 'APP_SETTING',
        p_singular_title            => 'Application Setting',
        p_plural_title              => 'Application Settings',
        p_table_name                => 'WWV_FLOW_APP_SETTINGS',
        p_import_api_name           => 'wwv_flow_imp_shared.create_app_setting',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'NAME',
        p_edit_url                  => 'f?p=4000:4851:%session%:::4851:F4000_P4851_ID,FB_FLOW_ID:%pk_value%,%application_id%' );
    --
    store_component_type (
        p_id                        => c_comp_type_pwa_shortcut,
        p_name                      => 'PWA_SHORTCUT',
        p_singular_title            => 'PWA Shortcut',
        p_plural_title              => 'PWA Shortcuts',
        p_table_name                => 'WWV_FLOW_PWA_SHORTCUTS',
        p_import_api_name           => 'wwv_flow_imp_shared.create_pwa_shortcut',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'NAME',
        p_edit_url                  => 'f?p=4000:508:%session%:::508:F4000_P508_ID,FB_FLOW_ID:%pk_value%,%application_id%' );
    --
    store_component_type (
        p_id                        => c_comp_type_pwa_screenshot,
        p_name                      => 'PWA_SCREENSHOT',
        p_singular_title            => 'PWA Screenshot',
        p_plural_title              => 'PWA Screenshots',
        p_table_name                => 'WWV_FLOW_PWA_SCREENSHOTS',
        p_import_api_name           => 'wwv_flow_imp_shared.create_pwa_screenshot',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'LABEL',
        p_edit_url                  => 'f?p=4000:519:%session%:::519:F4000_P519_ID,FB_FLOW_ID:%pk_value%,%application_id%' );
    --
    store_component_type (
        p_id                        => c_comp_type_classic_nav_bar,
        p_name                      => 'CLASSIC_NAV_BAR',
        p_singular_title            => 'Classic Navigation Bar Entry',
        p_plural_title              => 'Classic Navigation Bar Entries',
        p_table_name                => 'WWV_FLOW_ICON_BAR',
        p_import_api_name           => 'wwv_flow_imp_shared.create_icon_bar_item',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'ICON_SUBTEXT',
        p_edit_url                  => 'f?p=4000:4308:%session%::::F4000_P4308_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_file_path                 => 'shared_components/classic_navigation_bar_entries#ext#' );
    --
    store_component_type (
        p_id                        => c_comp_type_breadcrumb,
        p_name                      => 'BREADCRUMB',
        p_singular_title            => 'Breadcrumb',
        p_plural_title              => 'Breadcrumbs',
        p_table_name                => 'WWV_FLOW_MENUS',
        p_import_api_name           => 'wwv_flow_imp_shared.create_menu',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'NAME',
        p_create_url                => 'f?p=4000:288:%session%:::RP,288:FB_FLOW_ID:%application_id%',
        p_edit_url                  => 'f?p=4000:104:%session%:::RP,104:FB_FLOW_ID,FB_FLOW_PAGE_ID,P104_PAGE_ID,P104_MENU_ID:%application_id%,%page_id%,%page_id%,%pk_value%',
        p_copy_url                  => null,
        p_file_path                 => 'shared_components/breadcrumbs#ext#',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_breadcrumb_entry,
        p_name                      => 'BREADCRUMB_ENTRY',
        p_singular_title            => 'Entry',
        p_plural_title              => 'Entries',
        p_table_name                => 'WWV_FLOW_MENU_OPTIONS',
        p_import_api_name           => 'wwv_flow_imp_shared.create_menu_option',
        p_parent_component_type_id  => c_comp_type_breadcrumb,
        p_parent_fk_column          => 'MENU_ID',
        p_display_column            => 'SHORT_NAME',
        p_create_url                => 'f?p=4000:290:%session%:::RP,290:FB_FLOW_ID:%application_id%',
        p_edit_url                  => 'f?p=4000:290:%session%:::RP,290:F4000_P290_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_copy_url                  => null );
    --
    store_component_type (
        p_id                        => c_comp_type_list,
        p_name                      => 'LIST',
        p_singular_title            => 'List',
        p_plural_title              => 'Lists',
        p_table_name                => 'WWV_FLOW_LISTS',
        p_import_api_name           => 'wwv_flow_imp_shared.create_list',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'NAME',
        p_create_url                => 'f?p=4000:427:%session%:::427,440,464,529,734:FB_FLOW_ID:%application_id%',
        p_edit_url                  => 'f?p=4000:4050:%session%:::RP,4050:FB_FLOW_ID,FB_FLOW_PAGE_ID,F4000_P4050_LIST_ID:%application_id%,%page_id%,%pk_value%',
        p_copy_url                  => 'f?p=4000:663:%session%:::663:FB_FLOW_ID,P663_ID:%application_id%,%pk_value%',
        p_file_path                 => 'shared_components/lists#ext#',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_list_entry,
        p_name                      => 'LIST_ENTRY',
        p_singular_title            => 'Entry',
        p_plural_title              => 'Entries',
        p_table_name                => 'WWV_FLOW_LIST_ITEMS',
        p_import_api_name           => 'wwv_flow_imp_shared.create_list_item',
        p_parent_component_type_id  => c_comp_type_list,
        p_parent_fk_column          => 'LIST_ID',
        p_display_column            => 'LIST_ITEM_LINK_TEXT',
        p_create_url                => 'f?p=4000:4052:%session%:::RP,4052,4050:F4000_P4050_LIST_ID,FB_FLOW_ID:%parent_pk_value%,%application_id%',
        p_edit_url                  => 'f?p=4000:4052:%session%:::RP,4052,4050:F4000_P4052_ID,F4000_P4050_LIST_ID,FB_FLOW_ID:%pk_value%,%parent_pk_value%,%application_id%',
        p_copy_url                  => null );
    --
    store_component_type (
        p_id                        => c_comp_type_lov,
        p_name                      => 'LOV',
        p_singular_title            => 'List of Values',
        p_plural_title              => 'Lists of Values',
        p_table_name                => 'WWV_FLOW_LOVS_DEV',
        p_import_api_name           => 'wwv_flow_imp_shared.create_list_of_values',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'LOV_NAME',
        p_create_url                => 'f?p=4000:137:%session%:::4004,4007,4014,137,138,139:FB_FLOW_ID:%application_id%',
        p_edit_url                  => 'f?p=4000:4111:%session%:::4111:F4000_P4111_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_copy_url                  => 'f?p=4000:603:%session%:::603:FB_FLOW_ID,P603_ID:%application_id%,%pk_value%',
        p_file_path                 => 'shared_components/lovs#ext#',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_lov_entry,
        p_name                      => 'LOV_ENTRY',
        p_singular_title            => 'Entry',
        p_plural_title              => 'Entries',
        p_table_name                => 'WWV_FLOW_LIST_OF_VALUES_DATA',
        p_import_api_name           => 'wwv_flow_imp_shared.create_static_lov_data',
        p_parent_component_type_id  => c_comp_type_lov,
        p_parent_fk_column          => 'LOV_ID',
        p_display_column            => 'LOV_DISP_VALUE',
        p_create_url                => 'f?p=4000:4113:%session%:::4113,4111:F4000_P4111_ID,FB_FLOW_ID:%parent_pk_value%,%application_id%',
        p_edit_url                  => 'f?p=4000:4113:%session%:::4113,4111:F4000_P4113_ID,F4000_P4111_ID,FB_FLOW_ID:%pk_value%,%parent_pk_value%,%application_id%' );
    --
    store_component_type (
        p_id                        => c_comp_type_lov_column,
        p_name                      => 'LOV_COLUMN',
        p_singular_title            => 'Column',
        p_plural_title              => 'Columns',
        p_table_name                => 'WWV_FLOW_LIST_OF_VALUES_COLS',
        p_import_api_name           => 'wwv_flow_imp_shared.create_list_of_values_cols',
        p_parent_component_type_id  => c_comp_type_lov,
        p_parent_fk_column          => 'LOV_ID',
        p_display_column            => 'QUERY_COLUMN_NAME',
        p_create_url                => 'f?p=4000:4111:%session%:::4111:F4000_P4111_ID,FB_FLOW_ID:%parent_pk_value%,%application_id%',
        p_edit_url                  => 'f?p=4000:4111:%session%:::4111:F4000_P4111_ID,FB_FLOW_ID:%parent_pk_value%,%application_id%' );
    --
    store_component_type (
        p_id                        => c_comp_type_lov_ws_param,
        p_name                      => 'LOV_WS_PARAM',
        p_singular_title            => 'Parameter',
        p_plural_title              => 'Parameters',
        p_table_name                => 'WWV_FLOW_WEB_SRC_CPARAMS_DEV',
        p_import_api_name           => 'wwv_flow_imp_shared.create_web_source_comp_param',
        p_view_name                 => 'APEX_APPL_WEB_SRC_COMP_PARAMS',
        p_parent_component_type_id  => c_comp_type_lov,
        p_parent_fk_column          => 'SHARED_LOV_ID',
        p_display_column            => 'WEB_SRC_PARAM_ID' );
    --
    store_component_type (
        p_id                        => c_comp_type_ws_ref,
        p_name                      => 'WEB_SERVICE_REFERENCE',
        p_singular_title            => 'Web Service Reference',
        p_plural_title              => 'Web Service References',
        p_table_name                => 'WWV_FLOW_SHARED_WEB_SERVICES',
        p_import_api_name           => 'wwv_flow_imp_shared.create_web_service',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'NAME',
        p_create_url                => 'f?p=4000:870:%session%:::550,555,560,565,570,575,580,585,595,870,871,872,873,874,875,REST_INPUT_PARMS,REST_OUTPUT_PARMS,REST_HTTP_HEADERS:FB_FLOW_ID:%application_id%',
        p_copy_url                  => '',
        p_edit_url                  => '',  /* web services do have different edit pages */
        p_file_path                 => 'shared_components/legacy_web_service_references#ext#',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_ws_ref_operations,
        p_name                      => 'WEB_SERVICE_REFERENCE_OPERATION',
        p_singular_title            => 'Operation',
        p_plural_title              => 'Operations',
        p_table_name                => 'WWV_FLOW_WS_OPERATIONS_DEV',
        p_import_api_name           => 'wwv_flow_imp_shared.create_ws_operations',
        p_parent_component_type_id  => c_comp_type_ws_ref,
        p_parent_fk_column          => 'WS_ID',
        p_display_column            => 'FULL_NAME',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_ws_ref_oper_p_i,
        p_name                      => 'WEB_SERVICE_REF_OPER_PARAM_I',
        p_singular_title            => 'In Parameter',
        p_plural_title              => 'In Parameters',
        p_table_name                => 'WWV_FLOW_WS_OPER_PARAM_DEV',
        p_import_api_name           => 'wwv_flow_imp_shared.create_ws_parameters',
        p_parent_component_type_id  => c_comp_type_ws_ref_operations,
        p_parent_fk_column          => 'WS_OPERS_ID',
        p_display_column            => 'NAME',
        p_where_clause              => q'~input_or_output = 'I' ~',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_ws_ref_oper_p_o,
        p_name                      => 'WEB_SERVICE_REF_OPER_PARAM_O',
        p_singular_title            => 'Out Parameter',
        p_plural_title              => 'Out Parameters',
        p_table_name                => 'WWV_FLOW_WS_OPER_PARAM_DEV',
        p_import_api_name           => 'wwv_flow_imp_shared.create_ws_parameters',
        p_parent_component_type_id  => c_comp_type_ws_ref_operations,
        p_parent_fk_column          => 'WS_OPERS_ID',
        p_display_column            => 'NAME',
        p_where_clause              => q'~input_or_output = 'O' ~',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_ws_ref_oper_p_a,
        p_name                      => 'WEB_SERVICE_REF_OPER_PARAM_A',
        p_singular_title            => 'Basic Authentication Parameter',
        p_plural_title              => 'Basic Authentication Parameters',
        p_table_name                => 'WWV_FLOW_WS_OPER_PARAM_DEV',
        p_import_api_name           => 'wwv_flow_imp_shared.create_ws_parameters',
        p_parent_component_type_id  => c_comp_type_ws_ref_operations,
        p_parent_fk_column          => 'WS_OPERS_ID',
        p_display_column            => 'NAME',
        p_where_clause              => q'~input_or_output = 'A' ~',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_ws_ref_oper_p_h,
        p_name                      => 'WEB_SERVICE_REF_OPER_PARAM_H',
        p_singular_title            => 'HTTP Header Parameter',
        p_plural_title              => 'HTTP Header Parameters',
        p_table_name                => 'WWV_FLOW_WS_OPER_PARAM_DEV',
        p_import_api_name           => 'wwv_flow_imp_shared.create_ws_parameters',
        p_parent_component_type_id  => c_comp_type_ws_ref_operations,
        p_parent_fk_column          => 'WS_OPERS_ID',
        p_display_column            => 'NAME',
        p_where_clause              => q'~input_or_output = 'H' ~',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_data_load_def,
        p_name                      => 'DATA_LOAD_DEFINITION',
        p_singular_title            => 'Data Load Definition',
        p_plural_title              => 'Data Load Definitions',
        p_table_name                => 'WWV_FLOW_LOAD_TABLES',
        p_import_api_name           => 'wwv_flow_imp_shared.create_load_table',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'NAME',
        p_where_clause              => q'~target_type is not null ~',
        p_edit_url                  => 'f?p=4000:1820:%session%:::1820:P1820_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_file_path                 => 'shared_components/data_load_definitions#ext#',
        p_is_page_designer_shared   => true );
    --
    store_comp_type_data_profile (
        p_profile_type_id           => c_comp_type_data_load_dp,
        p_column_type_id            => c_comp_type_data_load_dp_col,
        p_name_prefix               => 'DATA_LOAD_DEF',
        p_parent_type_id            => c_comp_type_data_load_def,
        p_parent_fk_column          => 'LOAD_TABLE_ID' );
    --
    store_component_type (
        p_id                        => c_comp_type_legacy_data_load,
        p_name                      => 'DATA_LOAD_TABLE',
        p_singular_title            => 'Legacy Data Load Definition',
        p_plural_title              => 'Legacy Data Load Definitions',
        p_table_name                => 'WWV_FLOW_LOAD_TABLES',
        p_import_api_name           => 'wwv_flow_imp_shared.create_load_table',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'NAME',
        p_where_clause              => q'~target_type is null ~',
        p_edit_url                  => 'f?p=4000:1801:%session%:::1801:P1801_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_file_path                 => 'shared_components/legacy_data_load_definitions#ext#',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_tab_set,
        p_name                      => 'TAB SET',
        p_singular_title            => 'Standard Tab Set',
        p_plural_title              => 'Standard Tab Sets',
        p_table_name                => 'WWV_FLOW_TABSET_VIEW',
        p_import_api_name           => 'wwv_flow_imp_shared.create_tab',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'D',
        p_edit_url                  => 'f?p=4000:643:%session%:::643:FB_FLOW_ID:%application_id%',
        p_file_path                 => 'shared_components/legacy_tabs#ext#',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_report_layout,
        p_name                      => 'REPORT LAYOUT',
        p_singular_title            => 'Report Layout',
        p_plural_title              => 'Report Layouts',
        p_table_name                => 'WWV_FLOW_REPORT_LAYOUTS',
        p_import_api_name           => 'wwv_flow_imp_shared.create_report_layout',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'REPORT_LAYOUT_NAME',
        p_edit_url                  => 'f?p=4000:4111:%session%:::4111:F4000_P4111_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_file_path                 => 'shared_components/report_layouts#ext#',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_report_query,
        p_name                      => 'REPORT_QUERY',
        p_singular_title            => 'Report Query',
        p_plural_title              => 'Report Queries',
        p_table_name                => 'WWV_FLOW_SHARED_QUERIES',
        p_import_api_name           => 'wwv_flow_imp_shared.create_shared_query',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'NAME',
        p_edit_url                  => 'f?p=4000:1116:%session%:::1116,1117,1112:P1116_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_file_path                 => 'shared_components/report_queries#ext#',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_report_query_stmt,
        p_name                      => 'REPORT_QUERY_STATEMENT',
        p_singular_title            => 'Source Query',
        p_plural_title              => 'Source Queries',
        p_table_name                => 'WWV_FLOW_SHARED_QRY_STMTS_DEV',
        p_import_api_name           => 'wwv_flow_imp_shared.create_shared_query_stmnt',
        p_parent_component_type_id  => c_comp_type_report_query,
        p_parent_fk_column          => 'SHARED_QUERY_ID',
        p_display_column            => 'NAME',
        p_edit_url                  => 'f?p=4000:1116:%session%:::1116,1117,1112:P1116_ID,FB_FLOW_ID:%parent_pk_value%,%application_id%' ); -- intentionally pointing to parent page because collections have to be populated
    --
    store_component_type (
        p_id                        => c_comp_type_automation,
        p_name                      => 'AUTOMATION',
        p_singular_title            => 'Automation',
        p_plural_title              => 'Automations',
        p_table_name                => 'WWV_FLOW_AUTOMATIONS_DEV',
        p_import_api_name           => 'wwv_flow_imp_shared.create_automation',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'NAME',
        p_edit_url                  => 'f?p=4000:7021:%session%:::RP,7021:P7021_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_file_path                 => 'shared_components/automations#ext#' );
    --
    store_component_type (
        p_id                        => c_comp_type_autom_ws_param,
        p_name                      => 'AUTOMATION_WS_PARAM',
        p_singular_title            => 'Parameter',
        p_plural_title              => 'Parameters',
        p_table_name                => 'WWV_FLOW_WEB_SRC_CPARAMS_DEV',
        p_import_api_name           => 'wwv_flow_imp_shared.create_web_source_comp_param',
        p_view_name                 => 'APEX_APPL_WEB_SRC_COMP_PARAMS',
        p_parent_component_type_id  => c_comp_type_automation,
        p_parent_fk_column          => 'AUTOMATION_ID',
        p_display_column            => 'WEB_SRC_PARAM_ID' );
    --
    store_component_type (
        p_id                        => c_comp_type_automation_action,
        p_name                      => 'AUTOMATION_ACTION',
        p_singular_title            => 'Action',
        p_plural_title              => 'Actions',
        p_table_name                => 'WWV_FLOW_AUTOMATION_ACTIONS',
        p_import_api_name           => 'wwv_flow_imp_shared.create_automation_action',
        p_parent_component_type_id  => c_comp_type_automation,
        p_parent_fk_column          => 'AUTOMATION_ID',
        p_display_column            => 'NAME',
        p_plugin_type               => 'PROCESS TYPE',
        p_plugin_column             => 'ACTION_TYPE',
        p_edit_url                  => 'f?p=4000:7022:%session%:::RP,7022:P7022_ID,FB_FLOW_ID:%pk_value%,%application_id%' );
    --
    store_component_type (
        p_id                        => c_comp_type_search_config,
        p_name                      => 'SEARCH_CONFIG',
        p_singular_title            => 'Search Configuration',
        p_plural_title              => 'Search Configurations',
        p_table_name                => 'WWV_FLOW_SEARCH_CONFIGS_DEV',
        p_import_api_name           => 'wwv_flow_imp_shared.create_search_config',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'LABEL',
        p_edit_url                  => 'f?p=4000:3120:%session%:::3120:P3120_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_file_path                 => 'shared_components/search_configs#ext#',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_search_ws_param,
        p_name                      => 'SEARCH_CONFIG_WS_PARAM',
        p_singular_title            => 'Parameter',
        p_plural_title              => 'Parameters',
        p_table_name                => 'WWV_FLOW_WEB_SRC_CPARAMS_DEV',
        p_view_name                 => 'APEX_APPL_WEB_SRC_COMP_PARAMS',
        p_import_api_name           => 'wwv_flow_imp_shared.create_web_source_comp_param',
        p_parent_component_type_id  => c_comp_type_search_config,
        p_parent_fk_column          => 'SEARCH_CONFIG_ID',
        p_display_column            => 'WEB_SRC_PARAM_ID' );
    --
    store_component_type (
        p_id                        => c_comp_type_email_template,
        p_name                      => 'EMAIL_TEMPLATE',
        p_singular_title            => 'Email Template',
        p_plural_title              => 'Email Templates',
        p_table_name                => 'WWV_FLOW_EMAIL_TEMPLATES',
        p_import_api_name           => 'wwv_flow_imp_shared.create_email_template',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'NAME',
        p_create_url                => 'f?p=4000:4861:%session%:::RP,4861:FB_FLOW_ID:%application_id%',
        p_edit_url                  => 'f?p=4000:4861:%session%:::RP,4861:F4000_P4861_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_file_path                 => 'shared_components/email_templates#ext#',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_language_map,
        p_name                      => 'LANGUAGE_MAP',
        p_singular_title            => 'Language Mapping',
        p_plural_title              => 'Language Mappings',
        p_table_name                => 'WWV_FLOW_LANGUAGE_MAP',
        p_import_api_name           => 'wwv_flow_imp_shared.create_language_map',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'PRIMARY_LANGUAGE_FLOW_ID',
        p_display_column            => 'TRANSLATION_FLOW_LANGUAGE_CODE',
        p_edit_url                  => 'f?p=4000:4754:%session%:::4754:P4754_ID,FB_FLOW_ID:%pk_value%,%application_id%' );
    --
    store_component_type (
        p_id                        => c_comp_type_dyn_translation,
        p_name                      => 'DYNAMIC_TRANSLATION',
        p_singular_title            => 'Dynamic Translation',
        p_plural_title              => 'Dynamic Translations',
        p_table_name                => 'WWV_FLOW_DYNAMIC_TRANSLATIONS$',
        p_import_api_name           => 'wwv_flow_imp_shared.create_dynamic_translation',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'TRANSLATE_FROM_TEXT',
        p_edit_url                  => 'f?p=4000:4757:%session%:::4757:F4000_P4757_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_file_path                 => 'shared_components/dynamic_translations#ext#' );
    --
    store_component_type (
        p_id                        => c_comp_type_concatenated_file,
        p_name                      => 'CONCATENATED_FILE',
        p_singular_title            => 'Concatenated File',
        p_plural_title              => 'Concatenated Files',
        p_table_name                => 'WWV_FLOW_COMBINED_FILES',
        p_import_api_name           => 'wwv_flow_imp_shared.create_combined_file',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'COMBINED_FILE_URL',
        p_edit_url                  => 'f?p=4000:209:%session%::NO:209:P209_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_file_path                 => 'shared_components/concatenated_files#ext#' );
    --
    store_component_type (
        p_id                        => c_comp_type_task_def,
        p_name                      => 'TASK_DEF',
        p_singular_title            => 'Task Definition',
        p_plural_title              => 'Task Definitions',
        p_table_name                => 'WWV_FLOW_TASK_DEFS',
        p_import_api_name           => 'wwv_flow_imp_shared.create_task_def',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'NAME',
        p_edit_url                  => 'f?p=4000:9502:%session%:::9502:P9502_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_file_path                 => 'shared_components/task_definitions#ext#',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_task_def_param,
        p_name                      => 'TASK_DEF_PARAM',
        p_singular_title            => 'Parameter',
        p_plural_title              => 'Parameters',
        p_table_name                => 'WWV_FLOW_TASK_DEF_PARAMS',
        p_import_api_name           => 'wwv_flow_imp_shared.create_task_def_param',
        p_parent_component_type_id  => c_comp_type_task_def,
        p_parent_fk_column          => 'TASK_DEF_ID',
        p_display_column            => 'LABEL',
        p_where_clause              => null,
        p_create_url                => null,
        p_edit_url                  => 'f?p=4000:9502:%session%::NO:RP,9502:P9502_ID,FB_FLOW_ID:%parent_pk_value%,%application_id%',
        p_copy_url                  => null,
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_task_def_action,
        p_name                      => 'TASK_DEF_ACTION',
        p_singular_title            => 'Action',
        p_plural_title              => 'Actions',
        p_table_name                => 'WWV_FLOW_TASK_DEF_ACTIONS',
        p_import_api_name           => 'wwv_flow_imp_shared.create_task_def_action',
        p_parent_component_type_id  => c_comp_type_task_def,
        p_parent_fk_column          => 'TASK_DEF_ID',
        p_display_column            => 'NAME',
        p_edit_url                  => 'f?p=4000:9506:%session%::NO:RP,9506:P9506_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_plugin_type               => 'PROCESS TYPE',
        p_plugin_column             => 'ACTION_TYPE' );
    --
    store_component_type (
        p_id                        => c_comp_type_task_def_particip,
        p_name                      => 'TASK_DEF_PARTICIPANT',
        p_singular_title            => 'Participant',
        p_plural_title              => 'Participants',
        p_table_name                => 'WWV_FLOW_TASK_DEF_PARTICIPANTS',
        p_import_api_name           => 'wwv_flow_imp_shared.create_task_def_participant',
        p_parent_component_type_id  => c_comp_type_task_def,
        p_parent_fk_column          => 'TASK_DEF_ID',
        p_display_column            => 'PARTICIPANT_TYPE',
        p_edit_url                  => 'f?p=4000:9502:%session%::NO:RP,9502:P9502_ID,FB_FLOW_ID:%parent_pk_value%,%application_id%');
    --
    store_component_type (
        p_id                        => c_comp_type_workflow,
        p_name                      => 'WORKFLOW',
        p_singular_title            => 'Workflow',
        p_plural_title              => 'Workflows',
        p_table_name                => 'WWV_FLOW_WORKFLOWS',
        p_import_api_name           => 'wwv_flow_imp_shared.create_workflow',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'NAME',
        p_edit_url                  => null,
        p_file_path                 => 'shared_components/workflows#ext#',
        p_is_page_designer_shared   => true,
        p_is_shared_designer        => true );
    --
    store_component_type (
        p_id                        => c_comp_type_workflow_param,
        p_name                      => 'WORKFLOW_PARAM',
        p_singular_title            => 'Parameter',
        p_plural_title              => 'Parameters',
        p_table_name                => 'WWV_FLOW_WORKFLOW_VARS_DEV',
        p_import_api_name           => 'wwv_flow_imp_shared.create_workflow_variable',
        p_parent_component_type_id  => c_comp_type_workflow,
        p_parent_fk_column          => 'WORKFLOW_ID',
        p_display_column            => 'LABEL',
        p_where_clause              => null,
        p_create_url                => null,
        p_edit_url                  => null,
        p_copy_url                  => null,
        p_is_page_designer_shared   => true,
        p_is_shared_designer        => true );
    --
    store_component_type (
        p_id                        => c_comp_type_workflow_version,
        p_name                      => 'WORKFLOW_VERSION',
        p_singular_title            => 'Version',
        p_plural_title              => 'Versions',
        p_table_name                => 'WWV_FLOW_WORKFLOW_VERSIONS',
        p_import_api_name           => 'wwv_flow_imp_shared.create_workflow_version',
        p_parent_component_type_id  => c_comp_type_workflow,
        p_parent_fk_column          => 'WORKFLOW_ID',
        p_display_column            => 'VERSION',
        p_is_page_designer_shared   => false,
        p_is_shared_designer        => true );
    --
    store_component_type (
        p_id                        => c_comp_type_workflow_variable,
        p_name                      => 'WORKFLOW_VARIABLE',
        p_singular_title            => 'Variable',
        p_plural_title              => 'Variables',
        p_table_name                => 'WWV_FLOW_WORKFLOW_VARS_DEV',
        p_import_api_name           => 'wwv_flow_imp_shared.create_workflow_variable',
        p_parent_component_type_id  => c_comp_type_workflow_version,
        p_parent_fk_column          => 'WORKFLOW_VERSION_ID',
        p_display_column            => 'LABEL',
        p_is_page_designer_shared   => false,
        p_is_shared_designer        => true );
    --
    store_component_type (
        p_id                        => c_comp_type_workflow_activity,
        p_name                      => 'WORKFLOW_ACTIVITY',
        p_singular_title            => 'Activity',
        p_plural_title              => 'Activities',
        p_table_name                => 'WWV_FLOW_WF_ACTIVITIES_DEV',
        p_import_api_name           => 'wwv_flow_imp_shared.create_workflow_activity',
        p_parent_component_type_id  => c_comp_type_workflow_version,
        p_parent_fk_column          => 'WORKFLOW_VERSION_ID',
        p_display_column            => 'NAME',
        p_plugin_type               => 'PROCESS TYPE',
        p_plugin_column             => 'ACTIVITY_TYPE',
        p_is_page_designer_shared   => false,
        p_is_shared_designer        => true );
    --
    store_component_type (
        p_id                        => c_comp_type_workflow_act_var,
        p_name                      => 'WORKFLOW_ACTIVITY_VARIABLE',
        p_singular_title            => 'Activity Variable',
        p_plural_title              => 'Activity Variables',
        p_table_name                => 'WWV_FLOW_WORKFLOW_VARS_DEV',
        p_import_api_name           => 'wwv_flow_imp_shared.create_workflow_variable',
        p_parent_component_type_id  => c_comp_type_workflow_activity,
        p_parent_fk_column          => 'ACTIVITY_ID',
        p_display_column            => 'LABEL',
        p_is_page_designer_shared   => false,
        p_is_shared_designer        => true );
    --
    store_component_type (
        p_id                        => c_comp_type_workflow_trans,
        p_name                      => 'WORKFLOW_TRANSITION',
        p_singular_title            => 'Connection',
        p_plural_title              => 'Connections',
        p_table_name                => 'WWV_FLOW_WORKFLOW_TRANSITIONS',
        p_import_api_name           => 'wwv_flow_imp_shared.create_workflow_transition',
        p_parent_component_type_id  => c_comp_type_workflow_activity,
        p_parent_fk_column          => 'FROM_ACTIVITY_ID',
        p_display_column            => 'NAME',
        p_where_clause              => q'~transition_type <> 'BRANCH' ~',
        p_is_page_designer_shared   => false,
        p_is_shared_designer        => true );
    --
    store_component_type (
        p_id                        => c_comp_type_workflow_branch,
        p_name                      => 'WORKFLOW_BRANCH',
        p_singular_title            => 'Connection',
        p_plural_title              => 'Connections',
        p_table_name                => 'WWV_FLOW_WORKFLOW_BRANCHES_DEV',
        p_parent_component_type_id  => c_comp_type_workflow_activity,
        p_parent_fk_column          => 'FROM_ACTIVITY_ID',
        p_display_column            => 'NAME',
        p_is_page_designer_shared   => false,
        p_is_shared_designer        => true );
    --
    store_component_type (
        p_id                        => c_comp_type_workflow_particip,
        p_name                      => 'WORKFLOW_PARTICIPANT',
        p_singular_title            => 'Participant',
        p_plural_title              => 'Participants',
        p_table_name                => 'WWV_FLOW_WORKFLOW_PARTICIPANTS',
        p_import_api_name           => 'wwv_flow_imp_shared.create_workflow_participant',
        p_parent_component_type_id  => c_comp_type_workflow_version,
        p_parent_fk_column          => 'WORKFLOW_VERSION_ID',
        p_display_column            => 'PARTICIPANT_TYPE',
        p_is_page_designer_shared   => false,
        p_is_shared_designer        => true );
    --
    -- Misc component types
    store_component_type (
        p_id                        => c_comp_type_page_group,
        p_name                      => 'PAGE_GROUP',
        p_singular_title            => 'Page Group',
        p_plural_title              => 'Page Groups',
        p_table_name                => 'WWV_FLOW_PAGE_GROUPS',
        p_import_api_name           => 'wwv_flow_imp_page.create_page_group',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'GROUP_NAME',
        p_edit_url                  => 'f?p=4000:521:%session%::NO:521,RP:P521_ID,FB_FLOW_ID:%parent_pk_value%,%application_id%',
        p_file_path                 => 'page_groups#ext#',
        p_is_page_designer_shared   => true );
    --
    -- Page related component types
    --
    store_component_type (
        p_id                        => c_comp_type_page,
        p_name                      => 'PAGE',
        p_singular_title            => 'Page',
        p_plural_title              => 'Pages',
        p_table_name                => 'WWV_FLOW_STEPS_DEV',
        p_view_name                 => 'APEX_APPLICATION_PAGES',
        p_import_api_name           => 'wwv_flow_imp_page.create_page',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'NAME',
        p_create_url                => 'f?p=4000:2500:%session%:::2500,2501:FB_FLOW_ID:%application_id%',
        p_copy_url                  => 'f?p=4000:618:%session%:::618,301,302,305,308,309:FB_FLOW_ID:%application_id%',
        p_file_path                 => 'pages/' );

    -- Page Rendering
    store_component_type (
        p_id                        => c_comp_type_region,
        p_name                      => 'REGION',
        p_singular_title            => 'Region',
        p_plural_title              => 'Regions',
        p_table_name                => 'WWV_FLOW_PAGE_PLUGS_DEV',
        p_view_name                 => 'APEX_APPLICATION_PAGE_REGIONS',
        p_import_api_name           => 'wwv_flow_imp_page.create_page_plug',
        p_parent_component_type_id  => c_comp_type_page,
        p_parent_fk_column          => 'PAGE_ID',
        p_page_id_column            => 'PAGE_ID',
        p_plugin_type               => 'REGION TYPE',
        p_plugin_column             => 'PLUG_SOURCE_TYPE',
        p_supported_tmpl_components => 'REGION_ONLY:PARTIAL:REPORT', -- The order of the values is important, see wwv_meta_meta_data.populate_custom_plugins
        p_display_column            => 'PLUG_NAME',
        p_copy_url                  => 'f?p=4000:194:%session%:::194:FB_FLOW_ID,F4000_P194_PLUG_ID_FROM,F4000_P194_COPY_FROM_PAGE_ID:%application_id%,%pk_value%,%page_id%' );
    --
    store_component_type (
        p_id                        => c_comp_type_page_item,
        p_name                      => 'PAGE_ITEM',
        p_singular_title            => 'Page Item',
        p_plural_title              => 'Page Items',
        p_table_name                => 'WWV_FLOW_STEP_ITEMS_DEV',
        p_view_name                 => 'APEX_APPLICATION_PAGE_ITEMS',
        p_import_api_name           => 'wwv_flow_imp_page.create_page_item',
        p_parent_component_type_id  => c_comp_type_page,
        p_parent_fk_column          => 'FLOW_STEP_ID',
        p_page_id_column            => 'FLOW_STEP_ID',
        p_display_column            => 'NAME',
        p_plugin_type               => 'ITEM TYPE',
        p_plugin_column             => 'DISPLAY_AS',
        p_copy_url                  => 'f?p=4000:171:%session%:::171,172:FB_FLOW_ID,F4000_P171_COPY_FROM_ID,F4000_P171_COPY_FROM_PAGE_ID:%application_id%,%pk_value%,%page_id%' );
    --
    store_component_type (
        p_id                        => c_comp_type_button,
        p_name                      => 'BUTTON',
        p_singular_title            => 'Button',
        p_plural_title              => 'Buttons',
        p_table_name                => 'WWV_FLOW_STEP_BUTTONS',
        p_view_name                 => 'APEX_APPLICATION_PAGE_BUTTONS',
        p_import_api_name           => 'wwv_flow_imp_page.create_page_button',
        p_parent_component_type_id  => c_comp_type_page,
        p_parent_fk_column          => 'FLOW_STEP_ID',
        p_page_id_column            => 'FLOW_STEP_ID',
        p_display_column            => 'BUTTON_NAME',
        p_copy_url                  => 'f?p=4000:231:%session%:::231,191:FB_FLOW_ID,F4000_P231_COPY_FROM_ID,F4000_P231_COPY_FROM_PAGE_ID:%application_id%,%pk_value%,%page_id%' );
    --
    store_component_type (
        p_id                        => c_comp_type_da_event,
        p_name                      => 'DYNAMIC_ACTION_EVENT',
        p_singular_title            => 'Dynamic Action',
        p_plural_title              => 'Dynamic Actions',
        p_table_name                => 'WWV_FLOW_PAGE_DA_EVENTS_DEV',
        p_view_name                 => 'APEX_APPLICATION_PAGE_DA',
        p_import_api_name           => 'wwv_flow_imp_page.create_page_da_event',
        p_parent_component_type_id  => c_comp_type_page,
        p_parent_fk_column          => 'PAGE_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => 'NAME',
        p_copy_url                  => 'f?p=4000:796:%session%:::128,796,797:FB_FLOW_ID,F4000_P796_COPY_FROM_ID,F4000_P796_COPY_FROM_PAGE_ID:%application_id%,%pk_value%,%page_id%' );
    --
    store_component_type (
        p_id                        => c_comp_type_da_action,
        p_name                      => 'DYNAMIC_ACTION_ACTION',
        p_singular_title            => 'Action',
        p_plural_title              => 'Actions',
        p_table_name                => 'WWV_FLOW_PAGE_DA_ACTIONS_DEV',
        p_view_name                 => 'APEX_APPLICATION_PAGE_DA_ACTS',
        p_import_api_name           => 'wwv_flow_imp_page.create_page_da_action',
        p_parent_component_type_id  => c_comp_type_da_event,
        p_parent_fk_column          => 'EVENT_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => 'NAME_HIDDEN',
        p_plugin_type               => 'DYNAMIC ACTION',
        p_plugin_column             => 'ACTION' );
    --
    store_component_type (
        p_id                        => c_comp_type_meta_tag,
        p_name                      => 'PAGE_META_TAG',
        p_singular_title            => 'Meta Tag',
        p_plural_title              => 'Meta Tags',
        p_table_name                => 'WWV_FLOW_PAGE_META_TAGS',
        p_view_name                 => 'APEX_APPL_PAGE_META_TAGS',
        p_import_api_name           => 'wwv_flow_imp_page.create_page_meta_tag',
        p_parent_component_type_id  => c_comp_type_page,
        p_parent_fk_column          => 'PAGE_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => 'META_TAG_NAME' );

    -- Page Processing
    store_component_type (
        p_id                        => c_comp_type_validation,
        p_name                      => 'VALIDATION',
        p_singular_title            => 'Validation',
        p_plural_title              => 'Validations',
        p_table_name                => 'WWV_FLOW_STEP_VALIDATIONS',
        p_view_name                 => 'APEX_APPLICATION_PAGE_VAL',
        p_import_api_name           => 'wwv_flow_imp_page.create_page_validation',
        p_parent_component_type_id  => c_comp_type_page,
        p_parent_fk_column          => 'FLOW_STEP_ID',
        p_page_id_column            => 'FLOW_STEP_ID',
        p_display_column            => 'VALIDATION_NAME',
        p_copy_url                  => 'f?p=4000:583:%session%:::583,596:FB_FLOW_ID,P583_ID:%application_id%,%pk_value%' );
    --
    store_component_type (
        p_id                        => c_comp_type_page_computation,
        p_name                      => 'PAGE_COMPUTATION',
        p_singular_title            => 'Computation',
        p_plural_title              => 'Computations',
        p_table_name                => 'WWV_FLOW_STEP_COMPUTATIONS',
        p_view_name                 => 'APEX_APPLICATION_PAGE_COMP',
        p_import_api_name           => 'wwv_flow_imp_page.create_page_computation',
        p_parent_component_type_id  => c_comp_type_page,
        p_parent_fk_column          => 'FLOW_STEP_ID',
        p_page_id_column            => 'FLOW_STEP_ID',
        p_display_column            => 'COMPUTATION_ITEM',
        p_copy_url                  => 'f?p=4000:592:%session%:::592,597:FB_FLOW_ID,P592_ID:%application_id%,%pk_value%' );
    --
    store_component_type (
        p_id                        => c_comp_type_page_process,
        p_name                      => 'PAGE_PROCESS',
        p_singular_title            => 'Process',
        p_plural_title              => 'Processes',
        p_table_name                => 'WWV_FLOW_STEP_PROCESSING_DEV',
        p_view_name                 => 'APEX_APPLICATION_PAGE_PROC',
        p_import_api_name           => 'wwv_flow_imp_page.create_page_process',
        p_parent_component_type_id  => c_comp_type_page,
        p_parent_fk_column          => 'FLOW_STEP_ID',
        p_page_id_column            => 'FLOW_STEP_ID',
        p_display_column            => 'PROCESS_NAME',
        p_plugin_type               => 'PROCESS TYPE',
        p_plugin_column             => 'PROCESS_TYPE',
        p_copy_url                  => 'f?p=4000:590:%session%:::590,645:FB_FLOW_ID,P590_ID:%application_id%,%pk_value%' );
    --
    store_component_type (
        p_id                        => c_comp_type_branch,
        p_name                      => 'BRANCH',
        p_singular_title            => 'Branch',
        p_plural_title              => 'Branches',
        p_table_name                => 'WWV_FLOW_STEP_BRANCHES_DEV',
        p_view_name                 => 'APEX_APPLICATION_PAGE_BRANCHES',
        p_import_api_name           => 'wwv_flow_imp_page.create_page_branch',
        p_parent_component_type_id  => c_comp_type_page,
        p_parent_fk_column          => 'FLOW_STEP_ID',
        p_page_id_column            => 'FLOW_STEP_ID',
        p_display_column            => 'BRANCH_NAME',
        p_copy_url                  => 'f?p=4000:587:%session%:::587,685:FB_FLOW_ID,P587_ID:%application_id%,%pk_value%' );
    --
    -- Region type specific component types
    --
    -- Region Plug-in Attributes
    store_component_type (
        p_id                       => c_comp_type_region_plugin_attr,
        p_name                     => 'REGION_PLUGIN_ATTRIBUTES',
        p_singular_title           => 'Attributes',
        p_plural_title             => 'Attributes',
        p_table_name               => 'WWV_FLOW_REGION_PLUGIN_DEV',
        -- Requires new API don't use wwv_flow_imp_page.create_page_plug
        -- p_import_api_name           => 'wwv_flow_imp_page.create_page_plug',
        p_pk_column                => 'REGION_ID',
        p_parent_component_type_id => c_comp_type_region,
        p_parent_fk_column         => 'REGION_ID',
        p_page_id_column           => 'PAGE_ID',
        p_plugin_type              => 'REGION TYPE',
        p_plugin_column            => 'PLUG_SOURCE_TYPE',
        p_display_column           => null );

    -- Interactive Report
    store_component_type (
        p_id                        => c_comp_type_ir_attributes,
        p_name                      => 'IR_ATTRIBUTES',
        p_singular_title            => 'Attributes',
        p_plural_title              => 'Attributes',
        p_table_name                => 'WWV_FLOW_WORKSHEETS_DEV',
        p_import_api_name           => 'wwv_flow_imp_page.create_worksheet',
        p_view_name                 => 'APEX_APPLICATION_PAGE_IR',
        p_parent_component_type_id  => c_comp_type_region,
        p_parent_fk_column          => 'REGION_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => null );
    --
    store_component_type (
        p_id                        => c_comp_type_ir_saved_report,
        p_name                      => 'IR_SAVED_REPORT',
        p_singular_title            => 'Saved Report',
        p_plural_title              => 'Saved Reports',
        p_table_name                => 'WWV_FLOW_WORKSHEET_RPTS_DEV',
        p_import_api_name           => 'wwv_flow_imp_page.create_worksheet_rpt',
        p_parent_component_type_id  => c_comp_type_ir_attributes,
        p_parent_fk_column          => 'WORKSHEET_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => 'NAME',
        p_where_clause              => 'visibility <> ''PRIVATE''' );
    --
    store_component_type (
        p_id                        => c_comp_type_ir_column_group,
        p_name                      => 'IR_COLUMN_GROUP',
        p_singular_title            => 'Column Group',
        p_plural_title              => 'Column Groups',
        p_table_name                => 'WWV_FLOW_WORKSHEET_COL_GRP_DEV',
        p_import_api_name           => 'wwv_flow_imp_page.create_worksheet_col_group',
        p_parent_component_type_id  => c_comp_type_ir_attributes,
        p_parent_fk_column          => 'WORKSHEET_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => 'NAME' );
    --
    store_component_type (
        p_id                        => c_comp_type_ir_column,
        p_name                      => 'IR_COLUMN',
        p_singular_title            => 'Column',
        p_plural_title              => 'Columns',
        p_table_name                => 'WWV_FLOW_WORKSHEET_COL_DEV',
        p_view_name                 => 'APEX_APPLICATION_PAGE_IR_COL',
        p_import_api_name           => 'wwv_flow_imp_page.create_worksheet_column',
        p_parent_component_type_id  => c_comp_type_ir_attributes,
        p_parent_fk_column          => 'WORKSHEET_ID',
        p_page_id_column            => 'PAGE_ID',
        p_plugin_type               => 'IR COLUMN',
        p_plugin_column             => 'DISPLAY_TEXT_AS',
        p_supported_tmpl_components => 'PARTIAL',
        p_display_column            => 'DB_COLUMN_NAME' );
    --
    store_component_type (
        p_id                        => c_comp_type_ir_print,
        p_name                      => 'IR_PRINT_ATTR',
        p_singular_title            => 'Printing',
        p_plural_title              => 'Printing',
        p_table_name                => 'WWV_FLOW_REGION_PRINT_DEV',
        p_pk_column                 => 'WORKSHEET_ID',
        p_parent_component_type_id  => c_comp_type_ir_attributes,
        p_parent_fk_column          => 'WORKSHEET_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => null,
        p_where_clause              => q'~plug_source_type = 'NATIVE_IR' and ( download_formats like '%XLSX%' or download_formats like '%PDF%' ) ~' );

    -- Interactive grid
    store_component_type (
        p_id                        => c_comp_type_ig_attributes,
        p_name                      => 'IG_ATTRIBUTES',
        p_singular_title            => 'Attributes',
        p_plural_title              => 'Attributes',
        p_table_name                => 'WWV_FLOW_INTERACTIVE_GRIDS_DEV',
        p_view_name                 => 'APEX_APPLICATION_PAGE_IG',
        p_import_api_name           => 'wwv_flow_imp_page.create_interactive_grid',
        p_parent_component_type_id  => c_comp_type_region,
        p_parent_fk_column          => 'REGION_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => null );
    --
    store_component_type (
        p_id                        => c_comp_type_ig_column_group,
        p_name                      => 'IG_COLUMN_GROUP',
        p_singular_title            => 'Column Group',
        p_plural_title              => 'Column Groups',
        p_table_name                => 'WWV_FLOW_REGION_COLUMN_GROUPS',
        p_import_api_name           => 'wwv_flow_imp_page.create_region_column_group',
        p_parent_component_type_id  => c_comp_type_region,
        p_parent_fk_column          => 'REGION_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => 'HEADING' );
    --
    store_component_type (
        p_id                        => c_comp_type_ig_column,
        p_name                      => 'IG_COLUMN',
        p_singular_title            => 'Column',
        p_plural_title              => 'Columns',
        p_table_name                => 'WWV_FLOW_IG_COLUMNS_DEV',
        p_view_name                 => 'APEX_APPLICATION_PAGE_IG_COL',
        p_import_api_name           => 'wwv_flow_imp_page.create_region_column',
        p_parent_component_type_id  => c_comp_type_region,
        p_parent_fk_column          => 'REGION_ID',
        p_page_id_column            => 'PAGE_ID',
        p_plugin_type               => 'ITEM TYPE',
        p_plugin_column             => 'ITEM_TYPE',
        p_display_column            => 'NAME' );
    --
    store_component_type (
        p_id                        => c_comp_type_ig_saved_report,
        p_name                      => 'IG_SAVED_REPORT',
        p_singular_title            => 'Saved Report',
        p_plural_title              => 'Saved Reports',
        p_table_name                => 'WWV_FLOW_IG_REPORTS_DEV',
        p_import_api_name           => 'wwv_flow_imp_page.create_ig_report',
        p_parent_component_type_id  => c_comp_type_ig_attributes,
        p_parent_fk_column          => 'INTERACTIVE_GRID_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => 'NAME',
        p_where_clause              => 'visibility in (''PRIMARY'',''ALTERNATIVE'')' );
    --
    store_component_type (
        p_id                        => c_comp_type_ig_print,
        p_name                      => 'IG_PRINT_ATTR',
        p_singular_title            => 'Printing',
        p_plural_title              => 'Printing',
        p_table_name                => 'WWV_FLOW_REGION_PRINT_DEV',
        p_pk_column                 => 'REGION_ID',
        p_parent_component_type_id  => c_comp_type_region,
        p_parent_fk_column          => 'REGION_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => null,
        p_where_clause              => q'~plug_source_type = 'NATIVE_IG' and ( download_formats like '%XLSX%' or download_formats like '%PDF%' ) ~' );

    -- Web Service
    store_component_type (
        p_id                        => c_comp_type_page_proc_ws_p_i,
        p_name                      => 'PAGE_PROCESS_WS_PARAM_IN',
        p_singular_title            => 'In Parameter',
        p_plural_title              => 'In Parameters',
        p_table_name                => 'WWV_FLOW_PAGE_PROC_WS_PARM_DEV',
        p_import_api_name           => 'wwv_flow_imp_shared.create_ws_process_parms_map',
        p_parent_component_type_id  => c_comp_type_page_process,
        p_parent_fk_column          => 'PROCESS_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => 'PARAMETER_NAME',
        p_where_clause              => q'~parameter_type in ( 'I', 'H' ) ~' );
    --
    store_component_type (
        p_id                        => c_comp_type_page_proc_ws_p_o,
        p_name                      => 'PAGE_PROCESS_WS_PARAM_OUT',
        p_singular_title            => 'Out Parameter',
        p_plural_title              => 'Out Parameters',
        p_table_name                => 'WWV_FLOW_PAGE_PROC_WS_PARM_DEV',
        p_import_api_name           => 'wwv_flow_imp_shared.create_ws_process_parms_map',
        p_parent_component_type_id  => c_comp_type_page_process,
        p_parent_fk_column          => 'PROCESS_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => 'PARAMETER_NAME',
        p_where_clause              => q'~parameter_type = 'O' ~' );
    --
    store_component_type (
        p_id                        => c_comp_type_page_proc_ws_p_a,
        p_name                      => 'PAGE_PROCESS_WS_PARAM_AUTH',
        p_singular_title            => 'Basic Authentication Parameter',
        p_plural_title              => 'Basic Authentication Parameters',
        p_table_name                => 'WWV_FLOW_PAGE_PROC_WS_PARM_DEV',
        p_import_api_name           => 'wwv_flow_imp_shared.create_ws_process_parms_map',
        p_parent_component_type_id  => c_comp_type_page_process,
        p_parent_fk_column          => 'PROCESS_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => 'PARAMETER_NAME',
        p_where_clause              => q'~parameter_type = 'A' ~' );
    --
    store_component_type (
        p_id                        => c_comp_type_pg_proc_api_param,
        p_name                      => 'PAGE_PROCESS_INVOKE_API_PARAM',
        p_singular_title            => 'Parameter',
        p_plural_title              => 'Parameters',
        p_table_name                => 'WWV_FLOW_INVOKEAPI_CPARAMS_DEV',
        p_import_api_name           => 'wwv_flow_imp_shared.create_invokeapi_comp_param',
        p_parent_component_type_id  => c_comp_type_page_process,
        p_parent_fk_column          => 'PAGE_PROCESS_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => 'NAME' );
    --
    store_component_type (
        p_id                        => c_comp_type_page_proc_ws_param,
        p_name                      => 'PAGE_PROCESS_WS_PARAM',
        p_singular_title            => 'Parameter',
        p_plural_title              => 'Parameters',
        p_table_name                => 'WWV_FLOW_WEB_SRC_CPARAMS_DEV',
        p_view_name                 => 'APEX_APPL_WEB_SRC_COMP_PARAMS',
        p_import_api_name           => 'wwv_flow_imp_shared.create_web_source_comp_param',
        p_parent_component_type_id  => c_comp_type_page_process,
        p_parent_fk_column          => 'PAGE_PROCESS_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => 'WEB_SRC_PARAM_ID' );
    --
    store_component_type (
        p_id                        => c_comp_type_region_ws_param,
        p_name                      => 'REGION_WS_PARAM',
        p_singular_title            => 'Parameter',
        p_plural_title              => 'Parameters',
        p_table_name                => 'WWV_FLOW_WEB_SRC_CPARAMS_DEV',
        p_view_name                 => 'APEX_APPL_WEB_SRC_COMP_PARAMS',
        p_import_api_name           => 'wwv_flow_imp_shared.create_web_source_comp_param',
        p_parent_component_type_id  => c_comp_type_region,
        p_parent_fk_column          => 'PAGE_PLUG_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => 'WEB_SRC_PARAM_ID' );
    -- Classic Report
    store_component_type (
        p_id                       => c_comp_type_classic_report,
        p_name                     => 'CLASSIC_REPORT',
        p_singular_title           => 'Attributes',
        p_plural_title             => 'Attributes',
        p_table_name               => 'WWV_FLOW_REGION_RPT_DEV',
        p_view_name                => 'APEX_APPLICATION_PAGE_REGIONS',
        -- Requires new API don't use wwv_flow_imp_page.create_report_region
        -- p_import_api_name           => 'wwv_flow_imp_page.create_report_region',
        p_pk_column                => 'REGION_ID',
        p_parent_component_type_id => c_comp_type_region,
        p_parent_fk_column         => 'REGION_ID',
        p_page_id_column           => 'PAGE_ID',
        p_display_column           => null,
        p_where_clause             => q'~plug_source_type = 'NATIVE_SQL_REPORT' ~' );
    --
    store_component_type (
        p_id                       => c_comp_type_classic_rpt_column,
        p_name                     => 'CLASSIC_REPORT_COLUMN',
        p_singular_title           => 'Column',
        p_plural_title             => 'Columns',
        p_table_name               => 'WWV_FLOW_REGION_RPT_COL_DEV',
        p_view_name                => 'APEX_APPLICATION_PAGE_RPT_COLS',
        p_import_api_name          => 'wwv_flow_imp_page.create_report_columns',
        p_parent_component_type_id => c_comp_type_region,
        p_parent_fk_column         => 'REGION_ID',
        p_page_id_column           => 'FLOW_STEP_ID',
        p_display_column           => 'COLUMN_ALIAS',
        p_where_clause             => q'~exists ( select 1 from wwv_flow_page_plugs a where a.id = region_id and a.plug_source_type = 'NATIVE_SQL_REPORT' ) ~' );
    --
    store_component_type (
        p_id                       => c_comp_type_classic_rpt_print,
        p_name                     => 'CLASSIC_REPORT_PRINT_ATTR',
        p_singular_title           => 'Printing',
        p_plural_title             => 'Printing',
        p_table_name               => 'WWV_FLOW_REGION_PRINT_DEV',
        p_pk_column                => 'REGION_ID',
        p_parent_component_type_id => c_comp_type_classic_report,
        p_parent_fk_column         => 'REGION_ID',
        p_page_id_column           => 'PAGE_ID',
        p_display_column           => null,
        p_where_clause             => q'~plug_source_type = 'NATIVE_SQL_REPORT' and prn_output = 'Y' ~' );

    -- Tabular Form
    store_component_type (
        p_id                       => c_comp_type_tab_form,
        p_name                     => 'TABULAR_FORM',
        p_singular_title           => 'Attributes',
        p_plural_title             => 'Attributes',
        p_table_name               => 'WWV_FLOW_REGION_RPT_DEV',
        p_view_name                => 'APEX_APPLICATION_PAGE_REGIONS',
        -- Requires new API don't use wwv_flow_imp_page.create_report_region
        -- p_import_api_name           => 'wwv_flow_imp_page.create_report_region',
        p_pk_column                => 'REGION_ID',
        p_parent_component_type_id => c_comp_type_region,
        p_parent_fk_column         => 'REGION_ID',
        p_page_id_column           => 'PAGE_ID',
        p_display_column           => null,
        p_where_clause             => q'~plug_source_type = 'NATIVE_TABFORM' ~' );
    --
    store_component_type (
        p_id                       => c_comp_type_tab_form_column,
        p_name                     => 'TABULAR_FORM_COLUMN',
        p_singular_title           => 'Column',
        p_plural_title             => 'Columns',
        p_table_name               => 'WWV_FLOW_REGION_RPT_COL_DEV',
        p_view_name                => 'APEX_APPLICATION_PAGE_RPT_COLS',
        p_import_api_name          => 'wwv_flow_imp_page.create_report_columns',
        p_parent_component_type_id => c_comp_type_region,
        p_parent_fk_column         => 'REGION_ID',
        p_page_id_column           => 'FLOW_STEP_ID',
        p_display_column           => 'COLUMN_ALIAS',
        p_where_clause             => q'~exists ( select 1 from wwv_flow_page_plugs a where a.id = region_id and a.plug_source_type = 'NATIVE_TABFORM' ) ~' );
    --
    store_component_type (
        p_id                       => c_comp_type_tab_form_print,
        p_name                     => 'TABULAR_FORM_PRINT_ATTR',
        p_singular_title           => 'Printing',
        p_plural_title             => 'Printing',
        p_table_name               => 'WWV_FLOW_REGION_PRINT_DEV',
        p_pk_column                => 'REGION_ID',
        p_parent_component_type_id => c_comp_type_tab_form,
        p_parent_fk_column         => 'REGION_ID',
        p_page_id_column           => 'PAGE_ID',
        p_display_column           => null,
        p_where_clause             => q'~plug_source_type = 'NATIVE_TABFORM' and prn_output = 'Y' ~' );
    -- Classic Calendar
    store_component_type (
        p_id                        => c_comp_type_classic_calendar,
        p_name                      => 'CLASSIC_CALENDAR',
        p_singular_title            => 'Attributes',
        p_plural_title              => 'Attributes',
        p_table_name                => 'WWV_FLOW_CALS_DEV',
        p_import_api_name           => 'wwv_flow_imp_page.create_calendar',
        p_parent_component_type_id  => c_comp_type_region,
        p_parent_fk_column          => 'PLUG_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => null );
    --
    store_component_type (
        p_id                        => c_comp_type_region_column,
        p_name                      => 'REGION_COLUMN',
        p_singular_title            => 'Column',
        p_plural_title              => 'Columns',
        p_table_name                => 'WWV_FLOW_REGION_COLUMNS_DEV',
        p_view_name                 => 'APEX_APPLICATION_PAGE_REG_COLS',
        p_import_api_name           => 'wwv_flow_imp_page.create_region_column',
        p_parent_component_type_id  => c_comp_type_region,
        p_parent_fk_column          => 'REGION_ID',
        p_page_id_column            => 'PAGE_ID',
        p_plugin_type               => 'REGION TYPE',
        p_plugin_column             => 'REGION_TYPE',
        p_display_column            => 'NAME',
        p_where_clause              => q'~region_type not like 'TMPL\_%' escape '\' ~' );
    --
    store_component_type (
        p_id                        => c_comp_type_region_tmpl_column,
        p_name                      => 'REGION_TMPL_COLUMN',
        p_singular_title            => 'Column',
        p_plural_title              => 'Columns',
        p_table_name                => 'WWV_FLOW_REGION_COLUMNS_DEV',
        p_view_name                 => 'APEX_APPLICATION_PAGE_REG_COLS',
        p_import_api_name           => 'wwv_flow_imp_page.create_region_column',
        p_parent_component_type_id  => c_comp_type_region,
        p_parent_fk_column          => 'REGION_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => 'NAME',
        p_where_clause              => q'~region_type like 'TMPL\_%' escape '\' ~' );

    -- JET Chart
    store_component_type (
        p_id                        => c_comp_type_jet_chart,
        p_name                      => 'JET_CHART',
        p_singular_title            => 'Attributes',
        p_plural_title              => 'Attributes',
        p_table_name                => 'WWV_FLOW_JET_CHARTS',
        p_import_api_name           => 'wwv_flow_imp_page.create_jet_chart',
        p_parent_component_type_id  => c_comp_type_region,
        p_parent_fk_column          => 'REGION_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => null );
    --
    store_component_type (
        p_id                        => c_comp_type_jet_chart_series,
        p_name                      => 'JET_CHART_SERIES',
        p_singular_title            => 'Series',
        p_plural_title              => 'Series',
        p_table_name                => 'WWV_FLOW_JET_SERIES_DEV',
        p_view_name                 => 'APEX_APPLICATION_PAGE_CHART_S',
        p_import_api_name           => 'wwv_flow_imp_page.create_jet_chart_series',
        p_parent_component_type_id  => c_comp_type_jet_chart,
        p_parent_fk_column          => 'CHART_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => 'NAME' );
    --
    store_component_type (
        p_id                        => c_comp_type_jet_chart_axes,
        p_name                      => 'JET_CHART_AXES',
        p_singular_title            => 'Axis',
        p_plural_title              => 'Axes',
        p_table_name                => 'WWV_FLOW_JET_AXES_DEV',
        p_view_name                 => 'APEX_APPLICATION_PAGE_CHART_A',
        p_import_api_name           => 'wwv_flow_imp_page.create_jet_chart_axis',
        p_parent_component_type_id  => c_comp_type_jet_chart,
        p_parent_fk_column          => 'CHART_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => 'AXIS' );
    --
    store_component_type (
        p_id                        => c_comp_type_jet_chart_ws_param,
        p_name                      => 'JET_CHART_WS_PARAM',
        p_singular_title            => 'Parameter',
        p_plural_title              => 'Parameters',
        p_table_name                => 'WWV_FLOW_WEB_SRC_CPARAMS_DEV',
        p_view_name                 => 'APEX_APPL_WEB_SRC_COMP_PARAMS',
        p_import_api_name           => 'wwv_flow_imp_shared.create_web_source_comp_param',
        p_parent_component_type_id  => c_comp_type_jet_chart_series,
        p_parent_fk_column          => 'JET_CHART_SERIES_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => 'WEB_SRC_PARAM_ID' );
    --
    store_component_type (
        p_id                        => c_comp_type_map_region,
        p_name                      => 'MAP_REGION',
        p_singular_title            => 'Attributes',
        p_plural_title              => 'Attributes',
        p_table_name                => 'WWV_FLOW_MAP_REGIONS_DEV',
        p_view_name                 => 'APEX_APPL_PAGE_MAP_REGIONS',
        p_import_api_name           => 'wwv_flow_imp_page.create_map_region',
        p_parent_component_type_id  => c_comp_type_region,
        p_parent_fk_column          => 'REGION_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => null );
    --
    store_component_type (
        p_id                        => c_comp_type_map_layer,
        p_name                      => 'MAP_LAYER',
        p_singular_title            => 'Layer',
        p_plural_title              => 'Layers',
        p_table_name                => 'WWV_FLOW_MAP_LAYERS_DEV',
        p_view_name                 => 'APEX_APPL_PAGE_MAP_LAYERS',
        p_import_api_name           => 'wwv_flow_imp_page.create_map_region_layer',
        p_parent_component_type_id  => c_comp_type_map_region,
        p_parent_fk_column          => 'MAP_REGION_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => 'NAME' );
    --
    store_component_type (
        p_id                        => c_comp_type_map_layer_ws_param,
        p_name                      => 'MAP_LAYER_WS_PARAM',
        p_singular_title            => 'Parameter',
        p_plural_title              => 'Parameters',
        p_table_name                => 'WWV_FLOW_WEB_SRC_CPARAMS_DEV',
        p_view_name                 => 'APEX_APPL_WEB_SRC_COMP_PARAMS',
        p_import_api_name           => 'wwv_flow_imp_shared.create_web_source_comp_param',
        p_parent_component_type_id  => c_comp_type_map_layer,
        p_parent_fk_column          => 'MAP_REGION_LAYER_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => 'WEB_SRC_PARAM_ID' );
    --
    store_component_type (
        p_id                        => c_comp_type_map_background,
        p_name                      => 'MAP_BACKGROUND',
        p_singular_title            => 'Map Background',
        p_plural_title              => 'Map Backgrounds',
        p_table_name                => 'WWV_FLOW_MAP_BACKGROUNDS',
        p_import_api_name           => 'wwv_flow_imp_shared.create_map_background',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'NAME',
        p_edit_url                  => 'f?p=4000:6120:%session%:::6120:P6120_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_file_path                 => 'shared_components/map_backgrounds#ext#',
        p_is_page_designer_shared   => true );
    --
    store_component_type (
        p_id                        => c_comp_type_component_group,
        p_name                      => 'COMPONENT_GROUP',
        p_singular_title            => 'Component Group',
        p_plural_title              => 'Component Groups',
        p_table_name                => 'WWV_FLOW_COMPONENT_GROUPS',
        p_import_api_name           => 'wwv_flow_imp_shared.create_component_group',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'NAME',
        p_edit_url                  => 'f?p=4000:6201:%session%:::6201:P6201_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_file_path                 => 'shared_components/component_groups#ext#' );
    --
    store_component_type (
        p_id                        => c_comp_type_comp_grp_component,
        p_name                      => 'COMP_GRP_COMPONENT',
        p_singular_title            => 'Component',
        p_plural_title              => 'Components',
        p_table_name                => 'WWV_FLOW_COMP_GRP_COMPONENTS',
        p_import_api_name           => 'wwv_flow_imp_shared.create_comp_grp_component',
        p_parent_component_type_id  => c_comp_type_component_group,
        p_parent_fk_column          => 'COMPONENT_GROUP_ID',
        p_display_column            => 'COMPONENT_TYPE',
        p_edit_url                  => 'f?p=4000:6201:%session%:::6201:P6201_ID,FB_FLOW_ID:%parent_pk_value%,%application_id%' );
    --
    store_component_type (
        p_id                        => c_comp_type_facet,
        p_name                      => 'FACET',
        p_singular_title            => 'Facet',
        p_plural_title              => 'Facets',
        p_table_name                => 'WWV_FLOW_FILTERS_DEV',
        p_view_name                 => 'APEX_APPL_PAGE_FILTERS',
        p_parent_component_type_id  => c_comp_type_region,
        p_parent_fk_column          => 'ITEM_PLUG_ID',
        p_page_id_column            => 'FLOW_STEP_ID',
        p_display_column            => 'NAME',
        p_plugin_type               => 'ITEM TYPE',
        p_plugin_column             => 'DISPLAY_AS',
        p_copy_url                  => null,
        p_where_clause              => 'item_group_id is null' );
    --
    store_component_type (
        p_id                        => c_comp_type_facet_group,
        p_name                      => 'FACET_GROUP',
        p_singular_title            => 'Facet Group',
        p_plural_title              => 'Facet Groups',
        p_table_name                => 'WWV_FLOW_STEP_ITEM_GROUPS',
        p_view_name                 => 'APEX_APPL_PAGE_FILTER_GROUPS',
        p_import_api_name           => 'wwv_flow_imp_page.create_page_item_group',
        p_parent_component_type_id  => c_comp_type_region,
        p_parent_fk_column          => 'PAGE_PLUG_ID',
        p_page_id_column            => 'FLOW_STEP_ID',
        p_display_column            => 'LABEL',
        p_copy_url                  => null );
    --
    store_component_type (
        p_id                        => c_comp_type_facet_group_item,
        p_name                      => 'FACET_GROUP_ITEM',
        p_singular_title            => 'Checkbox',
        p_plural_title              => 'Checkboxes',
        p_table_name                => 'WWV_FLOW_FILTERS_DEV',
        p_view_name                 => 'APEX_APPL_PAGE_FILTERS',
        p_import_api_name           => 'wwv_flow_imp_page.create_page_item',
        p_parent_component_type_id  => c_comp_type_facet_group,
        p_parent_fk_column          => 'ITEM_GROUP_ID',
        p_page_id_column            => 'FLOW_STEP_ID',
        p_display_column            => 'NAME',
        p_plugin_type               => 'ITEM TYPE',
        p_plugin_column             => 'DISPLAY_AS',
        p_copy_url                  => null,
        p_where_clause              => 'item_group_id is not null' );
    --
    store_component_type (
        p_id                        => c_comp_type_sfilter,
        p_name                      => 'SFILTER',
        p_singular_title            => 'Filter',
        p_plural_title              => 'Filters',
        p_table_name                => 'WWV_FLOW_FILTERS_DEV',
        p_view_name                 => 'APEX_APPL_PAGE_FILTERS',
        p_import_api_name           => 'wwv_flow_imp_page.create_page_item',
        p_parent_component_type_id  => c_comp_type_region,
        p_parent_fk_column          => 'ITEM_PLUG_ID',
        p_page_id_column            => 'FLOW_STEP_ID',
        p_display_column            => 'NAME',
        p_plugin_type               => 'ITEM TYPE',
        p_plugin_column             => 'DISPLAY_AS',
        p_copy_url                  => null,
        p_where_clause              => 'item_group_id is null' );
    --
    store_component_type (
        p_id                        => c_comp_type_sfilter_group,
        p_name                      => 'SFILTER_GROUP',
        p_singular_title            => 'Filter Group',
        p_plural_title              => 'Filter Groups',
        p_table_name                => 'WWV_FLOW_STEP_ITEM_GROUPS',
        p_view_name                 => 'APEX_APPL_PAGE_FILTER_GROUPS',
        p_import_api_name           => 'wwv_flow_imp_page.create_page_item_group',
        p_parent_component_type_id  => c_comp_type_region,
        p_parent_fk_column          => 'PAGE_PLUG_ID',
        p_page_id_column            => 'FLOW_STEP_ID',
        p_display_column            => 'LABEL',
        p_copy_url                  => null );
    --
    store_component_type (
        p_id                        => c_comp_type_sfilter_group_item,
        p_name                      => 'SFILTER_GROUP_ITEM',
        p_singular_title            => 'Checkbox',
        p_plural_title              => 'Checkboxes',
        p_table_name                => 'WWV_FLOW_FILTERS_DEV',
        p_view_name                 => 'APEX_APPL_PAGE_FILTERS',
        p_import_api_name           => 'wwv_flow_imp_page.create_page_item',
        p_parent_component_type_id  => c_comp_type_sfilter_group,
        p_parent_fk_column          => 'ITEM_GROUP_ID',
        p_page_id_column            => 'FLOW_STEP_ID',
        p_display_column            => 'NAME',
        p_plugin_type               => 'ITEM TYPE',
        p_plugin_column             => 'DISPLAY_AS',
        p_copy_url                  => null,
        p_where_clause              => 'item_group_id is not null' );
    --
    store_component_type (
        p_id                        => c_comp_type_card,
        p_name                      => 'CARD',
        p_singular_title            => 'Attributes',
        p_plural_title              => 'Attributes',
        p_table_name                => 'WWV_FLOW_CARDS_DEV',
        p_view_name                 => 'APEX_APPL_PAGE_CARDS',
        p_import_api_name           => 'wwv_flow_imp_page.create_card',
        p_parent_component_type_id  => c_comp_type_region,
        p_parent_fk_column          => 'REGION_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => null );
    --
    store_component_type (
        p_id                        => c_comp_type_card_action,
        p_name                      => 'CARD_ACTION',
        p_singular_title            => 'Action',
        p_plural_title              => 'Actions',
        p_table_name                => 'WWV_FLOW_CARD_ACTIONS',
        p_view_name                 => 'APEX_APPL_PAGE_CARD_ACTIONS',
        p_import_api_name           => 'wwv_flow_imp_page.create_card_action',
        p_parent_component_type_id  => c_comp_type_card,
        p_parent_fk_column          => 'CARD_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => 'LABEL',
        p_alt_display_column        => 'ACTION_TYPE' );
    --
    store_component_type (
        p_id                        => c_comp_type_pg_pr_task_param,
        p_name                      => 'PAGE_PROCESS_TASK_PARAM',
        p_singular_title            => 'Parameter',
        p_plural_title              => 'Parameters',
        p_table_name                => 'WWV_FLOW_TASK_DEF_COMP_PARAMS',
        p_import_api_name           => 'wwv_flow_imp_shared.create_task_def_comp_param',
        p_parent_component_type_id  => c_comp_type_page_process,
        p_parent_fk_column          => 'PAGE_PROCESS_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => 'TASK_DEF_PARAM_ID' );
    --
    store_component_type (
        p_id                        => c_comp_type_wf_act_task_param,
        p_name                      => 'WF_ACTIVITY_TASK_PARAM',
        p_singular_title            => 'Parameter',
        p_plural_title              => 'Parameters',
        p_table_name                => 'WWV_FLOW_TASK_DEF_COMP_PARAMS',
        p_import_api_name           => 'wwv_flow_imp_shared.create_task_def_comp_param',
        p_parent_component_type_id  => c_comp_type_workflow_activity,
        p_parent_fk_column          => 'WORKFLOW_ACTIVITY_ID',
        p_display_column            => 'TASK_DEF_PARAM_ID',
        p_is_page_designer_shared   => true,
        p_is_shared_designer        => true );
    --
    store_component_type (
        p_id                        => c_comp_type_page_proc_wf_param,
        p_name                      => 'PAGE_PROCESS_WF_PARAM',
        p_singular_title            => 'Parameter',
        p_plural_title              => 'Parameters',
        p_table_name                => 'WWV_FLOW_WORKFLOW_CPARAMS_DEV',
        p_import_api_name           => 'wwv_flow_imp_shared.create_workflow_comp_param',
        p_parent_component_type_id  => c_comp_type_page_process,
        p_parent_fk_column          => 'PAGE_PROCESS_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => 'WORKFLOW_VARIABLE_ID',
        p_is_page_designer_shared   => true,
        p_where_clause              => q'~page_id is not null ~' );
    --
    store_component_type (
        p_id                        => c_comp_type_wf_act_wf_param,
        p_name                      => 'WF_ACTIVITY_WF_PARAM',
        p_singular_title            => 'Parameter',
        p_plural_title              => 'Parameters',
        p_table_name                => 'WWV_FLOW_WORKFLOW_CPARAMS_DEV',
        p_import_api_name           => 'wwv_flow_imp_shared.create_workflow_comp_param',
        p_parent_component_type_id  => c_comp_type_workflow_activity,
        p_parent_fk_column          => 'WORKFLOW_ACTIVITY_ID',
        p_display_column            => 'WORKFLOW_VARIABLE_ID',
        p_is_shared_designer        => true,
        p_where_clause              => q'~workflow_activity_id is not null ~' );
    --
    store_component_type (
        p_id                        => c_comp_type_wf_act_api_param,
        p_name                      => 'WF_ACTIVITY_INVOKE_API_PARAM',
        p_singular_title            => 'Parameter',
        p_plural_title              => 'Parameters',
        p_table_name                => 'WWV_FLOW_INVOKEAPI_CPARAMS_DEV',
        p_import_api_name           => 'wwv_flow_imp_shared.create_invokeapi_comp_param',
        p_parent_component_type_id  => c_comp_type_workflow_activity,
        p_parent_fk_column          => 'WORKFLOW_ACTIVITY_ID',
        p_display_column            => 'NAME',
        p_is_page_designer_shared   => true,
        p_is_shared_designer        => true );
    --
    store_component_type (
        p_id                        => c_comp_type_wf_act_ws_param,
        p_name                      => 'WF_ACTIVITY_WS_PARAM',
        p_singular_title            => 'Parameter',
        p_plural_title              => 'Parameters',
        p_table_name                => 'WWV_FLOW_WEB_SRC_CPARAMS_DEV',
        p_view_name                 => 'APEX_APPL_WEB_SRC_COMP_PARAMS',
        p_import_api_name           => 'wwv_flow_imp_shared.create_web_source_comp_param',
        p_parent_component_type_id  => c_comp_type_workflow_activity,
        p_parent_fk_column          => 'WORKFLOW_ACTIVITY_ID',
        p_display_column            => 'WEB_SRC_PARAM_ID',
        p_is_page_designer_shared   => true,
        p_is_shared_designer        => true );
    --
    store_component_type (
        p_id                        => c_comp_type_shortcut,
        p_name                      => 'SHORTCUT',
        p_singular_title            => 'Shortcut',
        p_plural_title              => 'Shortcuts',
        p_table_name                => 'WWV_FLOW_SHORTCUTS',
        p_view_name                 => 'APEX_APPLICATION_SHORTCUTS',
        p_import_api_name           => 'wwv_flow_imp_shared.create_shortcut',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'SHORTCUT_NAME',
        p_create_url                => 'f?p=4000:131:%session%:::RP,131:FB_FLOW_ID:%application_id%',
        p_edit_url                  => 'f?p=4000:4048:%session%:::RP,4048:F4000_P4048_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_file_path                 => 'shared_components/shortcuts#ext#' );
    --
    store_component_type (
        p_id                        => c_comp_type_tree,
        p_name                      => 'TREE',
        p_singular_title            => 'Tree',
        p_plural_title              => 'Trees',
        p_table_name                => 'WWV_FLOW_TREES',
        p_view_name                 => 'APEX_APPLICATION_TREES',
        p_import_api_name           => 'wwv_flow_imp_shared.create_tree',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'TREE_NAME',
        p_edit_url                  => 'f?p=4000:27:%session%::NO::F4000_P27_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_file_path                 => 'shared_components/legacy_trees#ext#' );
    --
    store_component_type (
        p_id                        => c_comp_type_plugin,
        p_name                      => 'PLUGIN',
        p_singular_title            => 'Plugin',
        p_plural_title              => 'Plugins',
        p_table_name                => 'WWV_FLOW_PLUGINS',
        p_view_name                 => 'APEX_APPL_PLUGINS',
        p_import_api_name           => 'wwv_flow_imp_shared.create_plugin',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'DISPLAY_NAME',
        p_file_path                 => 'shared_components/plugins#ext#',
        p_edit_url                  => 'f?p=4000:4410:%session%:::RP,4410:P4410_ID,FB_FLOW_ID:%pk_value%,%application_id%' );
    --
    store_component_type (
        p_id                        => c_comp_type_plugin_std_attr,
        p_name                      => 'PLUGIN_STD_ATTRIBUTE',
        p_singular_title            => 'Standard Attribute',
        p_plural_title              => 'Standard Attributes',
        p_table_name                => 'WWV_FLOW_PLUGIN_STD_ATTRIBUTES',
        p_view_name                 => 'APEX_APPL_PLUGIN_STD_ATTRS',
        p_import_api_name           => 'wwv_flow_imp_shared.create_plugin_std_attribute',
        p_parent_component_type_id  => c_comp_type_plugin,
        p_parent_fk_column          => 'PLUGIN_ID',
        p_display_column            => 'NAME',
        p_edit_url                  => 'f?p=4000:4411:%session%:::RP,4411:P4411_ID,FB_FLOW_ID:%pk_value%,%application_id%' );
    --
    store_component_type (
        p_id                        => c_comp_type_plugin_attr_grp,
        p_name                      => 'PLUGIN_ATTRIBUTE_GROUP',
        p_singular_title            => 'Attribute Group',
        p_plural_title              => 'Attribute Groups',
        p_table_name                => 'WWV_FLOW_PLUGIN_ATTR_GROUPS',
        p_view_name                 => 'APEX_APPL_PLUGIN_ATTR_GROUPS',
        p_import_api_name           => 'wwv_flow_imp_shared.create_plugin_attr_group',
        p_parent_component_type_id  => c_comp_type_plugin,
        p_parent_fk_column          => 'PLUGIN_ID',
        p_display_column            => 'TITLE',
        p_edit_url                  => get_parent_component_type_url (
                                                p_component_type_id => c_comp_type_plugin,
                                                p_url_type          => 'EDIT' ) );
    --
    store_component_type (
        p_id                        => c_comp_type_plugin_cust_attr,
        p_name                      => 'PLUGIN_CUSTOM_ATTRIBUTE',
        p_singular_title            => 'Custom Attribute',
        p_plural_title              => 'Custom Attributes',
        p_table_name                => 'WWV_FLOW_PLUGIN_ATTRIBUTES',
        p_view_name                 => 'APEX_APPL_PLUGIN_ATTRIBUTES',
        p_import_api_name           => 'wwv_flow_imp_shared.create_plugin_attribute',
        p_parent_component_type_id  => c_comp_type_plugin,
        p_parent_fk_column          => 'PLUGIN_ID',
        p_display_column            => 'PROMPT',
        p_edit_url                  => 'f?p=4000:4415:%session%:::RP,4415:P4415_ID,FB_FLOW_ID:%pk_value%,%application_id%' );
    --
    store_component_type (
        p_id                        => c_comp_type_plugin_attr_lov,
        p_name                      => 'PLUGIN_ATTRIBUTE_LOV_ENTRY',
        p_singular_title            => 'Entry',
        p_plural_title              => 'Entries',
        p_table_name                => 'WWV_FLOW_PLUGIN_ATTR_VALUES',
        p_view_name                 => 'APEX_APPL_PLUGIN_ATTR_VALUES',
        p_import_api_name           => 'wwv_flow_imp_shared.create_plugin_attr_value',
        p_parent_component_type_id  => c_comp_type_plugin_cust_attr,
        p_parent_fk_column          => 'PLUGIN_ATTRIBUTE_ID',
        p_display_column            => 'DISPLAY_VALUE',
        p_edit_url                  => 'f?p=4000:4416:%session%:::RP,4416:P4416_ID,FB_FLOW_ID:%pk_value%,%application_id%' );
    --
    store_component_type (
        p_id                        => c_comp_type_plugin_action_pos,
        p_name                      => 'PLUGIN_ACTION_POSITION',
        p_singular_title            => 'Action Position',
        p_plural_title              => 'Action Positions',
        p_table_name                => 'WWV_FLOW_PLUGIN_ACT_POSITIONS',
        p_view_name                 => 'APEX_APPL_PLUGIN_ACT_POSITIONS',
        p_import_api_name           => 'wwv_flow_imp_shared.create_plugin_act_position',
        p_parent_component_type_id  => c_comp_type_plugin,
        p_parent_fk_column          => 'PLUGIN_ID',
        p_display_column            => 'NAME',
        p_edit_url                  => 'f?p=4000:4417:%session%:::RP,4417:P4417_ID,FB_FLOW_ID:%pk_value%,%application_id%' );
    --
    store_component_type (
        p_id                        => c_comp_type_plugin_slot,
        p_name                      => 'PLUGIN_SLOT',
        p_singular_title            => 'Slot',
        p_plural_title              => 'Slots',
        p_table_name                => 'WWV_FLOW_PLUGIN_SLOTS',
        p_view_name                 => 'APEX_APPL_PLUGIN_SLOTS',
        p_import_api_name           => 'wwv_flow_imp_shared.create_plugin_slot',
        p_parent_component_type_id  => c_comp_type_plugin,
        p_parent_fk_column          => 'PLUGIN_ID',
        p_display_column            => 'NAME',
        p_edit_url                  => 'f?p=4000:4410:%session%::NO::P4410_ID,FB_FLOW_ID:%parent_pk_value%,%application_id%' );
    --
    store_component_type (
        p_id                        => c_comp_type_plugin_action_tmpl,
        p_name                      => 'PLUGIN_ACTION_TEMPLATE',
        p_singular_title            => 'Action Template',
        p_plural_title              => 'Action Templates',
        p_table_name                => 'WWV_FLOW_PLUGIN_ACT_TEMPLATES',
        p_view_name                 => 'APEX_APPL_PLUGIN_ACT_TEMPLATES',
        p_import_api_name           => 'wwv_flow_imp_shared.create_plugin_act_template',
        p_parent_component_type_id  => c_comp_type_plugin,
        p_parent_fk_column          => 'PLUGIN_ID',
        p_display_column            => 'NAME',
        p_edit_url                  => 'f?p=4000:4418:%session%:::RP,4418:P4418_ID,FB_FLOW_ID:%pk_value%,%application_id%' );
    --
    store_component_type (
        p_id                        => c_comp_type_plugin_event,
        p_name                      => 'PLUGIN_EVENT',
        p_singular_title            => 'Event',
        p_plural_title              => 'Events',
        p_table_name                => 'WWV_FLOW_PLUGIN_EVENTS',
        p_view_name                 => 'APEX_APPL_PLUGIN_EVENTS',
        p_import_api_name           => 'wwv_flow_imp_shared.create_plugin_event',
        p_parent_component_type_id  => c_comp_type_plugin,
        p_parent_fk_column          => 'PLUGIN_ID',
        p_display_column            => 'NAME',
        p_edit_url                  => 'f?p=4000:4410:%session%::NO::P4410_ID,FB_FLOW_ID:%parent_pk_value%,%application_id%' );
    --
    store_component_type (
        p_id                        => c_comp_type_plugin_file,
        p_name                      => 'PLUGIN_FILE',
        p_singular_title            => 'File',
        p_plural_title              => 'Files',
        p_table_name                => 'WWV_FLOW_PLUGIN_FILES',
        p_view_name                 => 'APEX_APPL_PLUGIN_FILES',
        p_import_api_name           => 'wwv_flow_imp_shared.create_plugin_file',
        p_parent_component_type_id  => c_comp_type_plugin,
        p_parent_fk_column          => 'PLUGIN_ID',
        p_display_column            => 'FILE_NAME',
        p_edit_url                  => 'f?p=4000:118:%session%:::RP,118:P118_ID,FB_FLOW_ID:%pk_value%,%application_id%' );
    --
    store_component_type (
        p_id                        => c_comp_type_supobj,
        p_name                      => 'SUPPORT_OBJECTS',
        p_singular_title            => 'Supporting Object',
        p_plural_title              => 'Supporting Object',
        p_table_name                => 'WWV_FLOW_INSTALL',
        p_import_api_name           => 'wwv_flow_imp_shared.create_install',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'FLOW_ID',
        p_edit_url                  => 'f?p=4000:567:%session%::NO:567,RP:FB_FLOW_ID:%application_id%',
        p_file_path                 => 'supporting_objects/supporting_objects#ext#' );
    --
    store_component_type (
        p_id                        => c_comp_type_plugin_setting,
        p_name                      => 'PLUGIN_SETTING',
        p_singular_title            => 'Component Setting',
        p_plural_title              => 'Component Settings',
        p_table_name                => 'WWV_FLOW_PLUGIN_SETTINGS',
        p_view_name                 => 'APEX_APPL_PLUGIN_SETTINGS',
        p_import_api_name           => 'wwv_flow_imp_shared.create_plugin_setting',
        p_parent_component_type_id  => c_comp_type_application,
        p_parent_fk_column          => 'FLOW_ID',
        p_display_column            => 'PLUGIN',
        p_edit_url                  => 'f?p=4000:4446:%session%:::RP,4446:P4446_ID,FB_FLOW_ID:%pk_value%,%application_id%' );
    --
    store_component_type (
        p_id                        => c_comp_type_supobj_substitut,
        p_name                      => 'SUPPORT_OBJECTS_SUBSTITUTION',
        p_singular_title            => 'Substitution',
        p_plural_title              => 'Substitutions',
        p_table_name                => 'WWV_FLOW_INSTALL_SUBST_DEV',
        -- Requires new api that updates just those substitution_string_* columns in wwv_flow_install
        -- p_import_api_name           => '',
        p_parent_component_type_id  => c_comp_type_supobj,
        p_parent_fk_column          => 'INSTALL_ID',
        p_display_column            => 'SUBSTITUTION_STRING',
        p_edit_url                  => 'f?p=4000:17:%session%::NO:17:FB_FLOW_ID:%application_id%',
        p_file_path                 => 'supporting_objects/substitutions#ext#' );
    --
    store_component_type (
        p_id                        => c_comp_type_supobj_build_opt,
        p_name                      => 'SUPPORT_OBJECTS_BUILD_OPTION',
        p_singular_title            => 'Build Option',
        p_plural_title              => 'Build Options',
        p_table_name                => 'WWV_FLOW_INSTALL_BUILD_OPT',
        p_import_api_name           => 'wwv_flow_imp_shared.create_install_build_option',
        p_parent_component_type_id  => c_comp_type_supobj,
        p_parent_fk_column          => 'INSTALL_ID',
        p_display_column            => 'BUILD_OPT_ID',
        p_edit_url                  => 'f?p=4000:511:%session%::NO:511,RP:FB_FLOW_ID:%application_id%',
        p_file_path                 => 'supporting_objects/build_options#ext#' );
    --
    store_component_type (
        p_id                        => c_comp_type_supobj_validation,
        p_name                      => 'SUPPORT_OBJECTS_VALIDATION',
        p_singular_title            => 'Validation',
        p_plural_title              => 'Validations',
        p_table_name                => 'WWV_FLOW_INSTALL_CHECKS',
        p_import_api_name           => 'wwv_flow_imp_shared.create_install_check',
        p_parent_component_type_id  => c_comp_type_supobj,
        p_parent_fk_column          => 'INSTALL_ID',
        p_display_column            => 'NAME',
        p_edit_url                  => 'f?p=4000:376:%session%::NO:RP,376:P376_ID,FB_FLOW_ID:%pk_value%,%application_id%',
        p_file_path                 => 'supporting_objects/validations#ext#' );
    --
    store_component_type (
        p_id                        => c_comp_type_supobj_install,
        p_name                      => 'SUPPORT_OBJECTS_INSTALL',
        p_singular_title            => 'Install Script',
        p_plural_title              => 'Install Scripts',
        p_table_name                => 'WWV_FLOW_INSTALL_SCRIPTS',
        p_import_api_name           => 'wwv_flow_imp_shared.create_install_script',
        p_parent_component_type_id  => c_comp_type_supobj,
        p_parent_fk_column          => 'INSTALL_ID',
        p_display_column            => 'NAME',
        p_edit_url                  => 'f?p=4000:328:%session%::NO:328,RP:P328_ID,F4000_LAST_VIEW,FB_FLOW_ID:%pk_value%,473,%application_id%',
        p_where_clause              => q'~coalesce( script_type, 'INSTALL' ) = 'INSTALL'~', -- $$$ todo we should do a data migration
        p_file_path                 => 'supporting_objects/install_scripts#ext#' );
    --
    store_component_type (
        p_id                        => c_comp_type_supobj_upgrade,
        p_name                      => 'SUPPORT_OBJECTS_UPGRADE',
        p_singular_title            => 'Upgrade Script',
        p_plural_title              => 'Upgrade Scripts',
        p_table_name                => 'WWV_FLOW_INSTALL_SCRIPTS',
        p_import_api_name           => 'wwv_flow_imp_shared.create_install_script',
        p_parent_component_type_id  => c_comp_type_supobj,
        p_parent_fk_column          => 'INSTALL_ID',
        p_display_column            => 'NAME',
        p_edit_url                  => 'f?p=4000:328:%session%::NO:328,RP:P328_ID,F4000_LAST_VIEW,FB_FLOW_ID:%pk_value%,667,%application_id%',
        p_where_clause              => q'~script_type = 'UPGRADE'~',
        p_file_path                 => 'supporting_objects/upgrade_scripts#ext#' );
    --
    store_component_type (
        p_id                        => c_comp_type_search_source,
        p_name                      => 'SEARCH_REGION_SOURCES',
        p_singular_title            => 'Search Source',
        p_plural_title              => 'Search Sources',
        p_table_name                => 'WWV_FLOW_SEARCH_REGION_SOURCES',
        p_view_name                 => 'APEX_APPL_PAGE_SEARCH_SOURCES',
        p_import_api_name           => 'wwv_flow_imp_page.create_search_region_source',
        p_parent_component_type_id  => c_comp_type_region,
        p_parent_fk_column          => 'SEARCH_REGION_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => 'NAME',
        p_copy_url                  => null );
    --
    store_component_type (
        p_id                        => c_comp_type_region_action,
        p_name                      => 'REGION_ACTION',
        p_singular_title            => 'Action',
        p_plural_title              => 'Actions',
        p_table_name                => 'WWV_FLOW_COMPONENT_ACTIONS_DEV',
        p_view_name                 => 'APEX_APPL_PAGE_COMP_ACTIONS',
        p_import_api_name           => 'wwv_flow_imp_page.create_component_action',
        p_parent_component_type_id  => c_comp_type_region,
        p_parent_fk_column          => 'REGION_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => 'LABEL',
        p_alt_display_column        => 'POSITION_ID',
        p_where_clause              => q'~region_id is not null~',
        p_copy_url                  => null );
    --
    store_component_type (
        p_id                        => c_comp_type_reg_act_menu_entry,
        p_name                      => 'REGION_ACTION_MENU_ENTRY',
        p_singular_title            => 'Menu',
        p_plural_title              => 'Menus',
        p_table_name                => 'WWV_FLOW_COMP_MENU_ENTRIES',
        p_view_name                 => 'APEX_APPL_PAGE_MENU_ENTRIES',
        p_import_api_name           => 'wwv_flow_imp_page.create_comp_menu_entry',
        p_parent_component_type_id  => c_comp_type_region_action,
        p_parent_fk_column          => 'COMPONENT_ACTION_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => 'LABEL',
        p_alt_display_column        => 'MENU_ENTRY_TYPE',
        p_where_clause              => q'~component_action_id is not null~',
        p_copy_url                  => null );
    --
    store_component_type (
        p_id                        => c_comp_type_ir_column_action,
        p_name                      => 'IR_COLUMN_ACTION',
        p_singular_title            => 'Action',
        p_plural_title              => 'Actions',
        p_table_name                => 'WWV_FLOW_COMPONENT_ACTIONS_DEV',
        p_view_name                 => 'APEX_APPL_PAGE_COMP_ACTIONS',
        p_import_api_name           => 'wwv_flow_imp_page.create_component_action',
        p_parent_component_type_id  => c_comp_type_ir_column,
        p_parent_fk_column          => 'IR_COLUMN_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => 'LABEL',
        p_alt_display_column        => 'POSITION_ID',
        p_where_clause              => q'~ir_column_id is not null~',
        p_copy_url                  => null );
    --
    store_component_type (
        p_id                        => c_comp_type_ir_act_menu_entry,
        p_name                      => 'IR_COLUMN_ACTION_MENU_ENTRY',
        p_singular_title            => 'Menu',
        p_plural_title              => 'Menus',
        p_table_name                => 'WWV_FLOW_COMP_MENU_ENTRIES',
        p_view_name                 => 'APEX_APPL_PAGE_MENU_ENTRIES',
        p_import_api_name           => 'wwv_flow_imp_page.create_comp_menu_entry',
        p_parent_component_type_id  => c_comp_type_ir_column_action,
        p_parent_fk_column          => 'COMPONENT_ACTION_ID',
        p_page_id_column            => 'PAGE_ID',
        p_display_column            => 'LABEL',
        p_alt_display_column        => 'MENU_ENTRY_TYPE',
        p_where_clause              => q'~component_action_id is not null~',
        p_copy_url                  => null );
    --
    --
    store_component_type (
        p_id                        => c_comp_type_display_point,
        p_name                      => 'PAGE_TEMPLATE_DISPLAY_POINT',
        p_singular_title            => 'Slot',
        p_plural_title              => 'Slots',
        p_table_name                => 'WWV_FLOW_PAGE_TMPL_DISP_POINTS',
        p_import_api_name           => 'wwv_flow_imp_shared.create_page_tmpl_display_point',
        p_parent_component_type_id  => c_comp_type_page_template,
        p_parent_fk_column          => 'PAGE_TEMPLATE_ID',
        p_display_column            => 'NAME',
        p_edit_url                  => 'f?p=4000:4307:%session%::NO:4307:F4000_P4307_ID,FB_FLOW_ID:%parent_pk_value%,%application_id%' );
    --
    --*******************
    -- END COMPONENT TYPE
    --*******************
    --
    ------------------------------------------------------------------------
    -- Display Groups
    ------------------------------------------------------------------------
    store_display_group (
        p_name  => 'IDENTIFICATION',
        p_title => 'Identification' );
    --
    store_display_group (
        p_name  => 'SETTINGS',
        p_title => 'Settings' );
    --
    store_display_group (
        p_name  => 'ATTRIBUTES',
        p_title => 'Attributes' );
    --
    store_display_group (
        p_name  => 'COMMENTS',
        p_title => 'Comments' );
    --
    store_display_group (
        p_name  => 'ERROR',
        p_title => 'Error' );
    --
    store_display_group (
        p_name  => 'CONFIGURATION',
        p_title => 'Configuration' );
    --
    store_display_group (
        p_name  => 'CSS',
        p_title => 'CSS' );
    --
    store_display_group (
        p_name  => 'DIALOG',
        p_title => 'Dialog' );
    --
    store_display_group (
        p_name  => 'ERROR_HANDLING',
        p_title => 'Error Handling' );
    --
    store_display_group (
        p_name  => 'HEADER_AND_FOOTER',
        p_title => 'Header and Footer' );
    --
    store_display_group (
        p_name  => 'HELP',
        p_title => 'Help' );
    --
    store_display_group (
        p_name  => 'HTML_HEADER',
        p_title => 'HTML Header' );
    --
    store_display_group (
        p_name  => 'JAVASCRIPT',
        p_title => 'JavaScript' );
    --
    store_display_group (
        p_name  => 'READ_ONLY',
        p_title => 'Read Only' );
    --
    store_display_group (
        p_name  => 'SECURITY',
        p_title => 'Security' );
    --
    store_display_group (
        p_name  => 'SERVER_CACHE',
        p_title => 'Server Cache' );
    --
    store_display_group (
        p_name  => 'SERVER_SIDE_CONDITION',
        p_title => 'Server-side Condition' );
    store_display_group (
        p_name  => 'CLIENT_SIDE_CONDITION',
        p_title => 'Client-side Condition' );
    --
    store_display_group (
        p_name  => 'EXECUTION',
        p_title => 'Execution' );
    --
    store_display_group (
        p_name  => 'COMPUTATION',
        p_title => 'Computation' );
    --
    store_display_group (
        p_name  => 'ADVANCED',
        p_title => 'Advanced' );
    --
    store_display_group (
        p_name  => 'AUTHORIZATION',
        p_title => 'Authorization' );
    --
    store_display_group (
        p_name  => 'WHEN',
        p_title => 'When' );
    --
    store_display_group (
        p_name  => 'AFFECTED_ELEMENTS',
        p_title => 'Affected Elements' );
    --
    store_display_group (
        p_name  => 'DEFAULT',
        p_title => 'Default' );
    --
    store_display_group (
        p_name  => 'ELEMENT',
        p_title => 'Element' );
    --
    store_display_group (
        p_name  => 'LABEL',
        p_title => 'Label' );
    --
    store_display_group (
        p_name  => 'LIST_OF_VALUES',
        p_title => 'List of Values' );
    --
    store_display_group (
        p_name  => 'QUICK_PICKS',
        p_title => 'Quick Picks' );
    --
    store_display_group (
        p_name  => 'SOURCE',
        p_title => 'Source' );
    --
    store_display_group (
        p_name  => 'VALIDATION',
        p_title => 'Validation' );
    --
    store_display_group (
        p_name  => 'BEHAVIOR',
        p_title => 'Behavior' );
    --
    store_display_group (
        p_name  => 'APPEARANCE',
        p_title => 'Appearance' );
    --
    store_display_group (
        p_name  => 'REGION_IMAGE',
        p_title => 'Image' );
    --
    store_display_group (
        p_name  => 'THRESHOLDS',
        p_title => 'Thresholds' );
    --
    store_display_group (
        p_name  => 'LAYOUT',
        p_title => 'Layout' );
    --
    store_display_group (
        p_name  => 'LOGIN_PROCESSING',
        p_title => 'Login Processing' );
    --
    store_display_group (
        p_name  => 'POST_LOGOUT_URL',
        p_title => 'Post-Logout URL' );
    --
    store_display_group (
        p_name  => 'REAL_APPLICATION_SECURITY',
        p_title => 'Real Application Security' );
    --
    store_display_group (
        p_name  => 'REFERENCE_LINES',
        p_title => 'Reference Lines' );
    --
    store_display_group (
        p_name  => 'SESSION_SHARING',
        p_title => 'Session Sharing' );
    --
    store_display_group (
        p_name  => 'SESSION_NOT_VALID',
        p_title => 'Session Not Valid' );
    --
    store_display_group (
        p_name  => 'SUBSCRIPTION',
        p_title => 'Subscription' );
    --
    store_display_group (
        p_name  => 'CUSTOMIZATION',
        p_title => 'Customization' );
    --
    store_display_group (
        p_name  => 'NAVIGATION',
        p_title => 'Navigation' );
    --
    store_display_group (
        p_name  => 'NAVIGATION_MENU',
        p_title => 'Navigation Menu' );
    --
    store_display_group (
        p_name  => 'TABS',
        p_title => 'Tabs' );
    --
    store_display_group (
        p_name  => 'SEARCH_BAR',
        p_title => 'Search Bar' );
    --
    store_display_group (
        p_name  => 'ACTIONS_MENU',
        p_title => 'Actions Menu' );
    --
    store_display_group (
        p_name  => 'COLUMN_FORMATTING',
        p_title => 'Column Formatting' );
    --
    store_display_group (
        p_name  => 'ENABLE_USER_TO',
        p_title => 'Enable Users To' );
    --
    store_display_group (
        p_name  => 'DESCRIPTION',
        p_title => 'Description' );
    --
    store_display_group (
        p_name  => 'DETAIL_VIEW',
        p_title => 'Detail View' );
    --
    store_display_group (
        p_name  => 'DOWNLOAD',
        p_title => 'Download' );
    --
    store_display_group (
        p_name  => 'ICON_VIEW',
        p_title => 'Icon View' );
    --
    store_display_group (
        p_name  => 'PAGINATION',
        p_title => 'Pagination' );
    --
    store_display_group (
        p_name  => 'PERFORMANCE',
        p_title => 'Performance' );
    --
    store_display_group (
        p_name  => 'SINGLE_ROW_VIEW',
        p_title => 'Single Row View' );
    --
    store_display_group (
        p_name  => 'LINK',
        p_title => 'Link' );
    --
    store_display_group (
         p_name  => 'FILE_STORAGE',
         p_title => 'File Storage' );
    --
    store_display_group (
        p_name  => 'SUCCESS_MESSAGE',
        p_title => 'Success Message' );
    --
    store_display_group (
        p_name  => 'PARAMETER',
        p_title => 'Parameter' );
    --
    store_display_group (
        p_name  => 'AUDIT_INFORMATION',
        p_title => 'Audit Information' );
    --
    store_display_group (
        p_name  => 'SORTING',
        p_title => 'Sorting' );
    --
    store_display_group (
        p_name  => 'EXPORT_PRINTING',
        p_title => 'Export / Printing' );
    --
    store_display_group (
        p_name  => 'BLOB',
        p_title => 'BLOB Attributes' );
    --
    store_display_group (
        p_name  => 'BREAK_FORMATTING',
        p_title => 'Break Formatting' );
    --
    store_display_group (
        p_name  => 'EXTERNAL_PROCESSING',
        p_title => 'External Processing' );
    --
    store_display_group (
        p_name  => 'MESSAGES',
        p_title => 'Messages' );
    --
    store_display_group (
        p_name  => 'HEADING',
        p_title => 'Heading' );
    --
    store_display_group (
        p_name  => 'UI_DEFAULTS_REFERENCE',
        p_title => 'UI Defaults Reference' );
    --
    store_display_group (
        p_name  => 'PRIMARY_KEY_SOURCE',
        p_title => 'Primary Key Source' );
    --
    store_display_group (
        p_name  => 'AUTOMATIC_REFRESH',
        p_title => 'Automatic Refresh' );
    --
    store_display_group (
        p_name  => 'LEGEND',
        p_title => 'Legend' );
    --
    store_display_group (
        p_name  => 'GAUGE',
        p_title => 'Gauge' );
    --
    store_display_group (
        p_name  => 'VALUE',
        p_title => 'Value' );
    --
    store_display_group (
        p_name  => 'TOOLTIP',
        p_title => 'Tooltip' );
    --
    store_display_group (
        p_name  => 'TITLE',
        p_title => 'Title' );
    --
    store_display_group (
        p_name  => 'GANTT_TIMELINE',
        p_title => 'Timeline' );
    --
    store_display_group (
        p_name  => 'CHART',
        p_title => 'Chart' );
    --
    store_display_group (
        p_name  => 'MONTHLY_CALENDAR',
        p_title => 'Monthly Calendar' );
    --
    store_display_group (
        p_name  => 'WEEKLY_CALENDAR',
        p_title => 'Weekly Calendar' );
    --
    store_display_group (
        p_name  => 'DAILY_WEEKLY_CALENDAR',
        p_title => 'Daily & Weekly Calendar' );
    --
    store_display_group (
        p_name  => 'LIST_VIEW_CALENDAR',
        p_title => 'Agenda View Calendar' );
    --
    store_display_group (
        p_name  => 'COLORS',
        p_title => 'Colors' );
    --
    store_display_group (
        p_name  => 'DRAG_AND_DROP',
        p_title => 'Drag and Drop' );
    --
    store_display_group (
        p_name  => 'CALENDAR_STATE_ITEMS',
        p_title => 'Calendar State Items' );
    --
    store_display_group (
        p_name  => 'ENTRY',
        p_title => 'Entry' );
    --
    store_display_group (
        p_name  => 'CONTROLS',
        p_title => 'Controls' );
    --
    store_display_group (
        p_name  => 'COLUMN_FILTER',
        p_title => 'Column Filter' );
    --
    store_display_group (
        p_name  => 'PRINTING',
        p_title => 'Printing' );
    --
    store_display_group (
        p_name  => 'PAGE',
        p_title => 'Page' );
    --
    store_display_group (
        p_name  => 'COLUMN_HEADINGS',
        p_title => 'Column Headings' );
    --
    store_display_group (
        p_name  => 'COLUMNS',
        p_title => 'Columns' );
    --
    store_display_group (
        p_name  => 'PAGE_HEADER',
        p_title => 'Page Header' );
    --
    store_display_group (
        p_name  => 'PAGE_FOOTER',
        p_title => 'Page Footer' );
    --
    store_display_group (
        p_name  => 'OUTPUT',
        p_title => 'Output' );
    --
    store_display_group (
        p_name  => 'EDIT',
        p_title => 'Edit' );
    --
    store_display_group (
        p_name  => 'EDIT_AUTH',
        p_title => 'Edit Authorization' );
    --
    store_display_group (
        p_name  => 'TOOLBAR',
        p_title => 'Toolbar' );
    --
    store_display_group (
        p_name  => 'OVERVIEW',
        p_title => 'Overview' );
    --
    store_display_group (
        p_name  => 'DATA_CURSOR',
        p_title => 'Data Cursor' );
    --
    store_display_group (
        p_name  => 'COLUMN_MAPPING',
        p_title => 'Column Mapping' );
    --
    store_display_group (
        p_name  => 'LINE',
        p_title => 'Line' );
    --
    store_display_group (
        p_name  => 'MARKER',
        p_title => 'Marker' );
    --
    store_display_group (
        p_name  => 'MAJOR_TICKS',
        p_title => 'Major Ticks' );
    --
    store_display_group (
        p_name  => 'MINOR_TICKS',
        p_title => 'Minor Ticks' );
    --
    store_display_group (
        p_name  => 'TICK_LABEL',
        p_title => 'Tick Label' );
    --
    store_display_group (
        p_name  => 'DUAL_Y_AXES',
        p_title => 'Dual Y Axes' );
    --
    store_display_group (
        p_name  => 'ANIMATION',
        p_title => 'Animation' );
    --
    store_display_group (
        p_name  => 'MASTER_DETAIL',
        p_title => 'Master Detail' );
    --
    store_display_group (
        p_name  => 'REMOTE_CACHE',
        p_title => 'Remote Cache' );
    --
    store_display_group (
        p_name  => 'ZOOM_SCALE',
        p_title => 'Zoom Scale' );
    --
    store_display_group (
        p_name  => 'TIME_SCALE',
        p_title => 'Time Scale' );
    --
    store_display_group (
        p_name  => 'GANTT_VIEWPORT',
        p_title => 'Viewport' );
    --
    store_display_group (
        p_name  => 'LOCAL_POST_PROCESSING',
        p_title => 'Local Post Processing' );
    --
    store_display_group (
        p_name  => 'MULTI_SERIES',
        p_title => 'Multi-Series Chart Data' );
    --
    store_display_group (
        p_name  => 'LIST_ENTRIES',
        p_title => 'List Entries' );
    --
    store_display_group (
        p_name  => 'DEPENDING_ON',
        p_title => 'Depending On' );
    --
    store_display_group (
        p_name  => 'CASCADING_LOV',
        p_title => 'Cascading List of Values' );
    --
    store_display_group (
        p_name  => 'QUERY',
        p_title => 'Query' );
    --
    store_display_group (
        p_name  => 'CARD',
        p_title => 'Card' );
    --
    store_display_group (
        p_name  => 'MAP_REGION',
        p_title => 'Map' );
    --
    store_display_group (
        p_name  => 'MAP_LAYER_ZOOM',
        p_title => 'Zoom Levels' );
    --
    store_display_group (
        p_name  => 'MAP_REGION_INIT',
        p_title => 'Initial Position and Zoom' );
    --
    store_display_group (
        p_name  => 'MAP_REGION_POINT_FEATURE_STYLE',
        p_title => 'Point Objects' );
    --
    store_display_group (
        p_name  => 'MAP_REGION_POINT_CLUSTER',
        p_title => 'Point Clustering' );
    --
    store_display_group (
        p_name  => 'MAP_REGION_ROW_SELECTION',
        p_title => 'Row Assignment' );
    --
    store_display_group (
        p_name  => 'MAP_REGION_FEATURE_INFO_WINDOW',
        p_title => 'Info Window' );
    --
    store_display_group (
        p_name  => 'SUB_TITLE',
        p_title => 'Subtitle' );
    --
    store_display_group (
        p_name  => 'BODY',
        p_title => 'Body' );
    --
    store_display_group (
        p_name  => 'SECONDARY_BODY',
        p_title => 'Secondary Body' );
    --
    store_display_group (
        p_name  => 'ICON_AND_BADGE',
        p_title => 'Icon and Badge' );
    --
    store_display_group (
        p_name  => 'MEDIA',
        p_title => 'Media' );
    --
    store_display_group (
        p_name  => 'AUTHORIZATION_SCHEME',
        p_title => 'Authorization Scheme' );
    --
    store_display_group (
        p_name  => 'DATABASE_SESSION',
        p_title => 'Database Session' );
    --
    store_display_group (
        p_name  => 'NAME',
        p_title => 'Name' );
    --
    store_display_group (
        p_name  => 'SESSION',
        p_title => 'Session' );
    --
    store_display_group (
        p_name  => 'ADDITIONAL_CODE',
        p_title => 'Additional Code Execution' );
    --
    store_display_group (
        p_name  => 'AUTHENTICATION',
        p_title => 'Authentication' );
    --
    store_display_group (
        p_name  => 'DATABASE_CREDENTIAL',
        p_title => 'Database Credential' );
    --
    store_display_group (
        p_name  => 'ENDPOINT_URL',
        p_title => 'Endpoint URL' );
    --
    store_display_group (
        p_name  => 'HTML_FORMAT',
        p_title => 'HTML Format' );
    --
    store_display_group (
        p_name  => 'PLAIN_TEXT_FORMAT',
        p_title => 'Plain Text Format' );
    --
    store_display_group (
        p_name  => 'MESSAGE',
        p_title => 'Message' );
    --
    store_display_group (
        p_name  => 'SESSION_STATE',
        p_title => 'Session State' );
    --
    store_display_group (
        p_name  => 'STATUS',
        p_title => 'Status' );
    --
    store_display_group (
        p_name  => 'ICON',
        p_title => 'Icon' );
    --
    store_display_group (
        p_name  => 'IS_CURRENT',
        p_title => 'Is Current' );
    --
    store_display_group (
        p_name  => 'CLICK_COUNTING',
        p_title => 'Click Counting' );
    --
    store_display_group (
        p_name  => 'USER_DEFINED_ATTRIBUTES',
        p_title => 'User Defined Attributes' );
    --
    store_display_group (
        p_name  => 'GLOBALIZATION',
        p_title => 'Globalization' );
    --
    store_display_group (
        p_name  => 'ACTION_EXECUTION',
        p_title => 'Action Execution' );
    --
    store_display_group (
        p_name  => 'MULTIPLE_VALUES',
        p_title => 'Multiple Values' );
    --
    store_display_group (
        p_name  => 'CONFIRMATION',
        p_title => 'Confirmation' );
    --
    store_display_group (
        p_name  => 'SUGGESTIONS',
        p_title => 'Suggestions' );
  --
    store_display_group (
        p_name  => 'ACCESSIBILITY',
        p_title => 'Accessibility' );
    --
    store_display_group (
        p_name  => 'EXTERNAL_FILTER_ORDER_BY',
        p_title => 'External Filter and Order By' );
    --
    store_display_group (
        p_name  => 'SOURCE_ORDER_BY',
        p_title => 'Order By' );
    --
    store_display_group (
        p_name  => 'DATA_PROFILE',
        p_title => 'Data Profile' );
    --
    store_display_group (
        p_name  => 'PARSING',
        p_title => 'Parsing' );
    --
    store_display_group (
        p_name  => 'LOOKUP',
        p_title => 'Lookup' );
    --
    store_display_group (
        p_name  => 'TRANSFORMATION',
        p_title => 'Transformation' );
    --
    store_display_group (
        p_name  => 'REMOTE_REST',
        p_title => 'Remote REST Data Source Server' );
    --
    store_display_group (
        p_name  => 'TARGET',
        p_title => 'Target' );
    --
    store_display_group (
        p_name  => 'SYNCHRONIZATION',
        p_title => 'REST Synchronization' );
    --
    store_display_group (
        p_name  => 'SYNCHRONIZATION_RATE_LIMIT',
        p_title => 'Synchronization Rate Limit' );
    --
    store_display_group (
        p_name  => 'OPERATION',
        p_title => 'Operation' );
    --
    store_display_group (
        p_name  => 'ROW_SEARCH',
        p_title => 'Row Search' );
    --
    store_display_group (
        p_name  => 'AVAILABILITY',
        p_title => 'Availability' );
    --
    store_display_group (
        p_name  => 'REPORT_PRINTING',
        p_title => 'Report Printing' );
    --
    store_display_group (
        p_name  => 'SESSION_MANAGEMENT',
        p_title => 'Session Management' );
    --
    store_display_group (
        p_name  => 'SESSION_STATE_PROTECTION',
        p_title => 'Session State Protection' );
    --
    store_display_group (
        p_name  => 'APP_FORMAT_MASKS',
        p_title => 'Application Format Masks' );
    --
    store_display_group (
        p_name  => 'CHARACTER_VALUE_COMPARISON',
        p_title => 'Character Value Comparison' );
    --
    store_display_group (
        p_name  => 'LOGO',
        p_title => 'Logo' );
    --
    store_display_group (
        p_name  => 'NAVIGATION_BAR',
        p_title => 'Navigation Bar' );
    --
    store_display_group (
        p_name  => 'PWA',
        p_title => 'Progressive Web App' );
    --
    store_display_group (
        p_name  => 'INSTALLATION',
        p_title => 'Installation' );
    --
    store_display_group (
        p_name  => 'PREREQUISITES',
        p_title => 'Prerequisites' );
    --
    store_display_group (
        p_name  => 'INSTALL_MESSAGES',
        p_title => 'Installation Messages' );
    --
    store_display_group (
        p_name  => 'UPGRADE_MESSAGES',
        p_title => 'Upgrade Messages' );
    --
    store_display_group (
        p_name  => 'DEINSTALL_MESSAGES',
        p_title => 'Deinstallation Messages' );
    --
    store_display_group (
        p_name  => 'REST_SOURCE_CATALOG',
        p_title => 'REST Source Catalog' );
    --
    store_display_group (
        p_name  => 'TEMPLATES',
        p_title => 'Templates' );
    --
    store_display_group (
        p_name  => 'DEADLINE',
        p_title => 'Deadline' );
    --
    store_display_group (
        p_name  => 'ENTITY_TITLE',
        p_title => 'Entity Title' );
    --
    store_display_group (
        p_name  => 'TEMPLATE_OPTIONS',
        p_title => 'Template Options' );
    --
    store_display_group (
        p_name  => 'DEFINITION',
        p_title => 'Definition' );
    --
    store_display_group (
        p_name  => 'FIELD_CONTAINER',
        p_title => 'Field Container' );
    --
    store_display_group (
        p_name  => 'FIRST_ENTRY',
        p_title => 'First Entry' );
    --
    store_display_group (
        p_name  => 'SUBLIST_ENTRY',
        p_title => 'Sublist Entry' );
    --
    store_display_group (
        p_name  => 'AFTER_LIST_ENTRY',
        p_title => 'After List Entry' );
    --
    store_display_group (
        p_name  => 'CASCADING_STYLE_SHEET',
        p_title => 'Cascading Style Sheet' );
    --
    store_display_group (
        p_name  => 'ATTRIBUTE_DESCRIPTIONS',
        p_title => 'Attribute Descriptions' );
    --
    store_display_group (
        p_name  => 'SUB_REGIONS',
        p_title => 'Sub Regions' );
    --
    store_display_group (
        p_name  => 'IMAGE',
        p_title => 'Image' );
    --
    store_display_group (
        p_name  => 'REPORT_TEMPLATE',
        p_title => 'Report Template' );
    --
    store_display_group (
        p_name  => 'COLUMN_TEMPLATES',
        p_title => 'Column Templates' );
    --
    store_display_group (
        p_name  => 'COLUMN_TEMPLATE_1',
        p_title => 'Column Template 1' );
    --
    store_display_group (
        p_name  => 'COLUMN_TEMPLATE_2',
        p_title => 'Column Template 2' );
    --
    store_display_group (
        p_name  => 'COLUMN_TEMPLATE_3',
        p_title => 'Column Template 3' );
    --
    store_display_group (
        p_name  => 'COLUMN_TEMPLATE_4',
        p_title => 'Column Template 4' );
    --
    store_display_group (
        p_name  => 'ROW_TEMPLATES',
        p_title => 'Row Templates' );
    --
    store_display_group (
        p_name  => 'ROW_TEMPLATE_1',
        p_title => 'Row Template 1' );
    --
    store_display_group (
        p_name  => 'ROW_TEMPLATE_2',
        p_title => 'Row Template 2' );
    --
    store_display_group (
        p_name  => 'ROW_TEMPLATE_3',
        p_title => 'Row Template 3' );
    --
    store_display_group (
        p_name  => 'ROW_TEMPLATE_4',
        p_title => 'Row Template 4' );
    --
    store_display_group (
        p_name  => 'PAGINATION_SUBTEMPLATE',
        p_title => 'Pagination Subtemplates' );
    --
    store_display_group (
        p_name  => 'ROW_HIGHLIGHTING',
        p_title => 'Row Highlighting' );
    --
    store_display_group (
        p_name  => 'PAGE_SUBTEMPLATE',
        p_title => 'Subtemplate' );
    --
    store_display_group (
        p_name  => 'IMAGE_BASED_TAB',
        p_title => 'Image Based Tab' );
    --
    store_display_group (
        p_name  => 'STANDARD_TAB',
        p_title => 'Standard Tab' );
    --
    store_display_group (
        p_name  => 'PARENT_TAB',
        p_title => 'Parent Tab' );
    --
    store_display_group (
        p_name  => 'POSITIONS',
        p_title => 'Positions' );
    --
    store_display_group (
        p_name  => 'DEFAULT_SLOTS',
        p_title => 'Default Slots' );
    --
    store_display_group (
        p_name  => 'ERROR_PAGE',
        p_title => 'Error Page' );
    --
    store_display_group (
        p_name  => 'CALLBACKS',
        p_title => 'Callbacks' );
    --
    store_display_group (
        p_name  => 'SUPPORTED_FOR',
        p_title => 'Supported for' );
    --
    store_display_group (
        p_name  => 'STANDARD_ATTRIBUTES',
        p_title => 'Standard Attributes' );
    --
    store_display_group (
        p_name  => 'CUSTOM_ATTRIBUTES',
        p_title => 'Custom Attributes' );
    --
    store_display_group (
        p_name  => 'FILES',
        p_title => 'Files' );
    --
    store_display_group (
        p_name  => 'FILE_URLS_TO_LOAD',
        p_title => 'File URLs to Load' );
    --
    store_display_group (
        p_name  => 'INFORMATION',
        p_title => 'Information' );
    --
    store_display_group (
        p_name  => 'CONDITION',
        p_title => 'Condition' );
    --
    store_display_group (
        p_name  => 'EXAMPLES',
        p_title => 'Examples' );
    --
    store_display_group (
        p_name  => 'COMPONENT_LIST_OF_VALUES',
        p_title => 'Component List of Values' );
    --
    store_display_group (
        p_name  => 'COMPONENT_DEFAULTS',
        p_title => 'Component Defaults' );
    --
    store_display_group (
        p_name  => 'REGION_DEFAULTS',
        p_title => 'Region Defaults' );
    --
    store_display_group (
        p_name  => 'DIALOG_DEFAULTS',
        p_title => 'Dialog Defaults' );
    --
    store_display_group (
        p_name  => 'ICONS',
        p_title => 'Icons' );
    --
    store_display_group (
        p_name  => 'THEME_ROLLER_ATTRIBUTES',
        p_title => 'Theme Roller Attributes' );
    --
    store_display_group (
        p_name  => 'ACTIVITY',
        p_title => 'Activity' );
    --
    store_display_group (
        p_name  => 'VARIABLE',
        p_title => 'Variable' );
    --
    --
    store_display_group (
        p_name  => 'SUBENTRIES',
        p_title => 'Subentries' );
    --
    store_display_group (
        p_name  => 'WORKING_COPY',
        p_title => 'Working Copy' );
    --
    store_display_group (
        p_name  => 'LIST',
        p_title => 'List' );
    --
    store_display_group (
        p_name  => 'SUBLIST',
        p_title => 'SubList' );
    --
    store_display_group (
        p_name  => 'ITEM',
        p_title => 'Item' );
    --
    store_display_group (
        p_name  => 'ERROR_TEMPLATES',
        p_title => 'Error Templates' );
    --
    store_display_group (
        p_name  => 'COMPONENT_TYPE_SUPPORT',
        p_title => 'Component Type Support' );
    --
    store_display_group (
        p_name  => 'GRID_LAYOUT_RENDERING',
        p_title => 'Grid Layout Rendering' );
    --
    store_display_group (
        p_name  => 'FEEDBACK',
        p_title => 'Feedback' );
    --
    store_display_group (
        p_name  => 'TEMPLATE',
        p_title => 'Template' );
    --
    store_display_group (
        p_name  => 'POPUP_ICON',
        p_title => 'Popup Icon' );
    --
    store_display_group (
        p_name  => 'SEARCH_FIELD',
        p_title => 'Search Field' );
    --
    store_display_group (
        p_name  => 'FIND_BUTTON',
        p_title => 'Find Button' );
    --
    store_display_group (
        p_name  => 'CLOSE_BUTTON',
        p_title => 'Close Button' );
    --
    store_display_group (
        p_name  => 'NEXT_BUTTON',
        p_title => 'Next Button' );
    --
    store_display_group (
        p_name  => 'PREVIOUS_BUTTON',
        p_title => 'Previous Button' );
    --
    store_display_group (
        p_name  => 'WINDOW',
        p_title => 'Window' );
    --
    store_display_group (
        p_name  => 'RESULT_SET',
        p_title => 'Result Set' );
    --
    store_display_group (
        p_name  => 'PAGE_PATTERN',
        p_title => 'Page Pattern' );
    --
    store_display_group (
        p_name  => 'ADDITIONAL_DATA',
        p_title => 'Additional Data' );
    --
    store_display_group (
        p_name  => 'GENERATIVE_AI_SERVICE',
        p_title => 'Generative AI Service');
    --
    store_display_group (
        p_name  => 'AI',
        p_title => 'Generative AI' );
    --
    store_display_group (
        p_name  => 'VECTOR_PROVIDER',
        p_title => 'Vector Provider');
    --
    store_display_group (
        p_name  => 'VECTOR_ATTRIBUTES',
        p_title => 'Vector Attributes');
    --
    store_display_group (
        p_name  => 'ROW_SELECTION',
        p_title => 'Row Selection' );
    --
    store_display_group (
        p_name  => 'COMPONENT',
        p_title => 'Component' );
    --
    store_display_group (
        p_name  => 'DISPLAY_FORMAT_MASKS',
        p_title => 'Display Format Masks' );
    --
    store_display_group (
        p_name  => 'OPENTELEMETRY',
        p_title => 'OpenTelemetry' );
    --
    --******************
    -- END DISPLAY GROUP
    --******************
    --
    ------------------------------------------------------------------------
    -- Error Messages
    ------------------------------------------------------------------------
    l_message_name := 'ERROR.APP_ID_EXISTS';
    store_msg (
        p_name         => l_message_name,
        p_message_text => 'Application with this ID already exists.' );
    --
    l_message_name := 'ERROR.TRANS_APP_ID_NO_ZERO';
    store_msg (
        p_name         => l_message_name,
        p_message_text => 'Translation Application must not end in zero.' );
    --
    --*******************
    -- END ERROR MESSAGES
    --*******************
    --
    ------------------------------------------------------------------------
    -- Properties
    ------------------------------------------------------------------------
    --
    l_property_id := 1;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'NAME',
        p_prompt             => 'Name',
        p_property_type      => 'TEXT',
        p_db_column          => 'NAME',
        p_display_length     => 32,
        p_max_length         => 255,
        p_display_group_name => 'IDENTIFICATION',
        p_is_translatable    => true,
        p_help_text          => q'~Enter a descriptive name for this component.~',
        p_comment_text       => q'~$$$ Note: Only true because name used by PAGE is defined as translatable in old metadata! Check.~' );
    --
    l_property_id := 2;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'BUILD_OPTION_STATUS',
        p_prompt             => 'Status',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'PATCH_STATUS',
        p_display_group_name => 'STATUS',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select a build option status for this component.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Include',
        p_return_value  => 'INCLUDE',
        p_help_text     => q'~Use all components flagged with this build option.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Exclude',
        p_return_value  => 'EXCLUDE',
        p_help_text     => q'~Do not use any component flagged with this build option.~' );
    --
    l_property_id := 3;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'DEFAULT_ON_EXPORT',
        p_prompt             => 'Default on Export',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'DEFAULT_ON_EXPORT',
        p_display_group_name => 'STATUS',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select the default build option applied to this component when exporting. This option can be over ridden by export options.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Exclude',
        p_return_value  => 'EXCLUDE',
        p_help_text     => q'~Disabled and not a part of the application.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Include',
        p_return_value  => 'INCLUDE',
        p_help_text     => q'~Enabled and part of the application.~' );
    --
    l_property_id := 4;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'COMMENTS',
        p_prompt             => 'Comments',
        p_property_type      => 'TEXT EDITOR',
        p_db_column          => 'COMMENTS',
        p_display_length     => 50,
        p_max_length         => 4000,
        p_display_group_name => 'COMMENTS',
        p_help_text          => q'~Enter any developer comments or notes here. These comments only display within App Builder and never display when running the application.~' );
    --
    l_property_id := 5;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'AUTHORIZATION_TYPE',
        p_prompt             => 'Type',
        p_property_type      => 'SUPPORTED UI',
        p_db_column          => 'SCHEME_TYPE',
        p_display_group_name => 'AUTHORIZATION_SCHEME',
        p_lov_type           => 'PLUGINS',
        p_help_text          => q'~<p>Specify how this authorization scheme will be applied. The following types are supported:</p>
#NATIVE_PLUGINS#
<p>Depending on your application configuration, you may also have additional authorization type plug-ins available here.
These display as <em>&lt;My Plug-in&gt; [Plug-in]</em>.
The following plug-ins install into this application's shared components:</p>
#CUSTOM_PLUGINS#~' );
    --
    l_property_id := 6;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'ERROR_MESSAGE',
        p_prompt             => 'Error Message',
        p_property_type      => 'TEXT EDITOR',
        p_db_column          => 'ERROR_MESSAGE',
        p_display_length     => 70,
        p_max_length         => 4000,
        p_examples           => '#LABEL# must have some value.',
        p_display_group_name => 'ERROR',
        p_is_translatable    => true,
        p_help_text          => q'~<p>Enter the text to be displayed in the event that the validation does not pass.</p>
<p>You can use <code>#LABEL#</code> as placeholder for the label of the associated page item or <code>#COLUMN_HEADER#</code> for the heading of the associated editable region column. The placeholders are not supported if error display location is set to On Error Page.</p>~' );
    --
    l_property_id := 7;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'EVALUATION_POINT',
        p_prompt             => 'Evaluation Point',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'CACHING',
        p_display_group_name => 'SERVER_CACHE',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~<p>Authorization schemes are evaluated on first use in a session.
Select the appropriate evaluation point to determine whether subsequent use of the given authorization obtains the result from memory or re-evaluates the authorization scheme.</p>
#LOV_VALUES#
<p>The default value <em>Once per session</em> is the most efficient.
You should choose another value if the authorization check depends on changing session state or other factors that are not consistent over an entire session.
The options <em>Once per component</em> and <em>Always (No Caching)</em> provide additional parameters that can be defined for use by the authorization scheme.
In authorization plugins, you can use the <em>component.type</em>, <em>component.id</em> and <em>component.name</em> attributes of the <em>t_authorization</em> parameter.
For the built-in <em>Exists SQL Query</em> and <em>PL/SQL Function Body Returning Boolean</em> types, equivalent bind variables <em>:APP_COMPONENT_TYPE</em>, <em>:APP_COMPONENT_ID</em> and <em>:APP_COMPONENT_NAME</em> are available.
</p>~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Once per Session',
        p_return_value  => 'BY_USER_BY_SESSION',
        p_help_text     => q'~Evaluates only once and then uses the result in memory.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Once per Page View',
        p_return_value  => 'BY_USER_BY_PAGE_VIEW',
        p_help_text     => q'~Evaluates once for each request that is processed. Use the result in memory if the authorization scheme is referenced in more than one component on the page.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Once per Component',
        p_return_value  => 'BY_COMPONENT',
        p_help_text     => q'~Evaluates once for each component that references the authorization scheme, but saves the result in the session. Further requests to render or process the scheme use the result in memory.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Always (No Caching)',
        p_return_value  => 'NOCACHE',
        p_help_text     => q'~The authorization scheme is always evaluated.~' );
    --
    l_property_id := 8;
    store_property (
        p_id                    => l_property_id,
        p_name                  => 'ALIAS',
        p_prompt                => 'Alias',
        p_property_type         => 'TEXT',
        p_db_column             => 'ALIAS',
        p_display_length        => 30,
        p_max_length            => 80,
        p_text_case             => 'LOWER',
        p_restricted_characters => 'SAFE_COMPONENT_NAME_NO_NUMBER',
        p_display_group_name    => 'IDENTIFICATION',
        p_validation_plsql_ret_error => q'~
begin
    if wwv_flow_page_dev.is_alias_reserved_word( p_alias => p_value ) then
        return wwv_flow_lang.message( 'PD.ERROR.INVALID_ALIAS_RESERVED_WORD', p_value );
    end if;
    return null;
end;~',
        p_help_text             => q'~Enter an alphanumeric alias for this page. You may use this alias to reference the page instead of the page number.
For example, if you were working on page 1 of application 100, you could create an alias called <code>home</code>.
You could then access this page from other pages using the following f?p syntax:
<pre>f?p=100:home</pre>~' );
    --
    l_property_id := 9;
    store_property (
        p_id                    => l_property_id,
        p_name                  => 'PAGE_GROUP',
        p_prompt                => 'Page Group',
        p_property_type         => 'COMPONENT',
        p_db_column             => 'GROUP_ID',
        p_display_group_name    => 'IDENTIFICATION',
        p_lov_type              => 'COMPONENT',
        p_lov_component_type_id => c_comp_type_page_group,
        p_help_text             => q'~Specify the page group to associate with this page.
Page groups do not affect functionality, but help developers manage the pages within an application.~' );
    --
    l_property_id := 10;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'PAGE_MODE',
        p_prompt             => 'Page Mode',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'PAGE_MODE',
        p_display_group_name => 'APPEARANCE',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~<p>Select the page mode for this page.</p>
#LOV_VALUES#
<p>Note that options <em>Modal</em> and <em>Non-Modal Dialog</em> will only be available for selection where the current application theme contains a default <em>Dialog Page Template</em>.</p>~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Normal',
        p_return_value  => 'NORMAL',
        p_help_text     => q'~The page is presented as a normal &PRODUCT_NAME. application page.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Modal Dialog',
        p_return_value  => 'MODAL',
        p_help_text     => q'~The page is presented as a modal dialog.<br>
A modal dialog is an overlay window positioned within the viewport, which remains active and focused until the end user dismisses (closes) it.
The underlying page is grayed out and the end user is prevented from interacting with the rest of the page until the dialog is closed.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Non-Modal Dialog',
        p_return_value  => 'NON_MODAL',
        p_help_text     => q'~The page is presented as a non-modal dialog.<br>
A non-modal dialog is an overlay window positioned within the viewport, where the end user can interact with the dialog, and content on the page that launched the dialog.
A non-modal dialog can be used when the requested information is not essential to continue, work can continue elsewhere while the window is still open.~' );
    --
    l_property_id := 11;
    store_property (
        p_id                       => l_property_id,
        p_name                     => 'PAGE_TEMPLATE',
        p_prompt                   => 'Page Template',
        p_property_type            => 'COMPONENT',
        p_db_column                => 'STEP_TEMPLATE',
        p_display_group_name       => 'APPEARANCE',
        p_lov_type                 => 'COMPONENT',
        p_lov_component_type_id    => c_comp_type_page_template,
        p_lov_component_type_scope => 'THEME',
        p_help_text                => q'~<p>Select a page template to define the appearance and layout of this page.</p>
        <p>Page templates are defined in the application theme.
        When a page is created, the template is automatically set to <em>Theme Default</em> which will render the page using the default page template defined in the current theme.
        Making a selection here overrides the default page template.</p>
        <p>Many themes contain page templates with specific purposes, such as for displaying content with side columns, login page, dialog pages, and so forth.<br>
        Universal Theme provides the following page templates:
          <dl>
            <dt>Standard</dt>
              <dd>The default page template. Ideal for home pages, dashboards, large reports, and other layouts where maximum screen real-estate is desired.</dd>
            <dt>Left Side Column</dt>
              <dd>Provides a slot on the left side where search and filter regions can be placed.</dd>
            <dt>Right Side Column</dt>
              <dd>Provides a collapsible slot on the right side where action-oriented controls such as buttons or lists can be displayed.</dd>
            <dt>Left and Right Side Columns</dt>
              <dd>Provides both the left-side column and collapsible right-side column that is well suited for very complex pages.</dd>
            <dt>Login</dt>
              <dd>Used exclusively for the application login page.</dd>
            <dt>Marquee</dt>
              <dd>Provides a collapsible right side column and a title-bar area which contains primary information.
              This page template is well suited for displaying a single record which has several related reports or data.
              Marquee is commonly used for displaying Master-Detail relationships where a single master record is first selected from a report. </dd>
            <dt>Minimal (No Navigation)</dt>
              <dd>Useful for very simple pages, disclaimers, single page applications, or where navigation is not necessary.</dd>
          </dl>
          See Also: <a rel="noopener noreferrer" target="_blank" href="https://apex.oracle.com/ut">Universal Theme Sample Application</a>
        </p>~' );
    --
    l_property_id := 12;
    store_property (
        p_id                    => l_property_id,
        p_name                  => 'STANDARD_TAB_SET',
        p_prompt                => 'Standard Tab Set',
        p_property_type         => 'COMPONENT',
        p_db_column             => 'TAB_SET',
        p_display_group_name    => 'TABS',
        p_lov_type              => 'COMPONENT',
        p_lov_component_type_id => c_comp_type_tab_set,
        p_help_text             => q'~Select a standard tab set used for this page. A standard tab set is associated with a specific page number. You can use standard tabs to link users to a specific page.~' );
    --
    l_property_id := 13;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'TITLE',
        p_prompt             => 'Title',
        p_property_type      => 'TEXT',
        p_db_column          => 'TITLE',
        p_display_length     => 70,
        p_max_length         => 255,
        p_display_group_name => 'IDENTIFICATION',
        p_is_translatable    => true,
        p_help_text          => q'~Enter the page title. The page title is displayed in the title bar of the browser window.~' );
    --
    l_property_id := 14;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'CURSOR_FOCUS',
        p_prompt             => 'Cursor Focus',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'FIRST_ITEM',
        p_display_group_name => 'NAVIGATION',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Specify whether the cursor is focused when first entering the page.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'First item on page',
        p_return_value  => 'AUTO_FIRST_ITEM',
        p_help_text     => q'~The cursor focus is placed on the first field on the page using JavaScript, generated by the &PRODUCT_NAME. engine.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Do not focus cursor',
        p_return_value  => 'NO_FIRST_ITEM',
        p_help_text     => q'~The cursor is not focused and no JavaScript is generated.<br>
Select this option if the generated JavaScript conflicts with custom JavaScript.~' );
    --
    l_property_id := 15;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'JS_FILE_URLS',
        p_prompt             => 'File URLs',
        p_property_type      => 'FILE URLS JAVASCRIPT',
        p_db_column          => 'JAVASCRIPT_FILE_URLS',
        p_display_length     => 80,
        p_max_length         => 4000,
        p_examples           => q'~<dl>
<dt>Standard file reference</dt>
<dd><pre>/myjs/main.js</pre></dd>
<dt>Standard file reference which loads the minified file <code>main.min.js</code> from <code>/myjs/minified/</code> for regular page views and <code>main.js</code> from <code>/myjs/</code> in debug mode or when running the application from App Builder.</dt>
<dd><pre>/myjs/#MIN_DIRECTORY#main#MIN#.js</pre></dd>
<dt>Use this syntax if the library is also used from a RequireJS require call. The define name must match what is used in the requirejs.config paths and the value must be the global symbol exposed by the library. Only applies when using RequireJS. This avoids RequireJS  loading that library a second time.</dt>
<dd><pre>[define hammerjs=Hammer]/myjs/hammer.js</pre></dd>
<dt>Use this syntax if the file has a dependency on RequireJS. This will automatically load RequireJS before this file.</dt>
<dd><pre>[require requirejs]/myjs/main.js</pre></dd>
<dt>Use this syntax if the file has a dependency on Oracle JET. This will automatically load RequireJS and the Oracle JET RequireJS configuration before this file.</dt>
<dd><pre>[require jet]/myjs/main.js</pre></dd>
<dt>Use this syntax if the file is a JavaScript ES Module. It will add <code>type="module"</code> to the script tag.</dt>
<dd><pre>[module]/myjs/main.js</pre></dd>
<dt>Use this syntax if the file should be loaded asynchronously. It will add attribute <code>async</code> to the script tag.</dt>
<dd><pre>[async]/myjs/main.js</pre></dd>
<dt>Use this syntax if the file should only be executed after the document was been parsed. It will add attribute <code>defer</code> to the script tag. Note that modules are deferred by default.</dt>
<dd><pre>[defer]/myjs/main.js</pre></dd>
<dt>Multiple File URL directives can be provided but must be comma separated.</dt>
<dd><pre>[module, async]/myjs/main.js</pre></dd>
<dt>File reference using a query string in the URL referencing the application version, such that when the application is upgraded (and the version is incremented), the browser is instructed to always load the new file and ignore any previously cached files.</dt>
<dd><pre>/myjs/main.js?version=#APP_VERSION#</pre></dd>
</dl>~',
        p_display_group_name => 'JAVASCRIPT',
        p_help_text          => q'~<p>Enter JavaScript file URLs for code loaded on this page. Each URL must be written on a new line.
If you provide a minified version of your file you can use the substitution string <code>#MIN#</code> to include <code>.min</code> or <code>#MIN_DIRECTORY#</code> to include <code>minified/</code> in your file URL for a regular page view and an empty string if the page is viewed in debug mode or when running the application from App Builder.
You also have access to the substitution string <code>#APP_VERSION#</code> if you want to include the application's version in the file URL.</p>
<p>JavaScript file URLs you enter here replaces the <code>#PAGE_JAVASCRIPT#</code> substitution string in the page template.</p>
<p>Each file URL can be prefixed with a pair of brackets (<code>[...]</code>) containing one or more of the following directives. See the examples for details.</p>
<p><ul>
    <li>module</li>
    <li>defer</li>
    <li>async</li>
    <li>define</li>
    <li>require requirejs</li>
    <li>require jet</li>
</ul></p>
<p>Note: You do not need to include opening or closing <code>script</code> tags. Just specify the URL.</p>
<p>To comment out a File URL, prefix the line with two dashes: <code>--</code>. To comment out multiple lines, use the typical code block syntax: <code>/* */</code>.</p>~' );
    --
    l_property_id := 16;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'JS_FUNCTION_AND_GLOBAL_VARIABLE_DECLARATION',
        p_prompt             => 'Function and Global Variable Declaration',
        p_property_type      => 'JAVASCRIPT',
        p_db_column          => 'JAVASCRIPT_CODE',
        p_display_length     => 80,
        p_max_length         => 32767,
        p_display_group_name => 'JAVASCRIPT',
        p_help_text          => q'~<p>Enter JavaScript code such as functions or global variable declarations referenced on this page.
To reference a shortcut, use <code>"SHORTCUTNAME"</code>.
If the same JavaScript is required on multiple pages, consider putting the code into an external JavaScript file to avoid duplication and to leverage browser caching of static files.</p>
<p>Code you enter here replaces the <code>#PAGE_JAVASCRIPT#</code> substitution string in the page template.</p>
<p>Note: You do not need to include opening or closing <code>script</code> tags. Just include the JavaScript code.</p>~' );
    --
    l_property_id := 17;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'JS_EXECUTE_WHEN_PAGE_LOADS',
        p_prompt             => 'Execute when Page Loads',
        p_property_type      => 'JAVASCRIPT',
        p_db_column          => 'JAVASCRIPT_CODE_ONLOAD',
        p_display_length     => 80,
        p_max_length         => 32767,
        p_display_group_name => 'JAVASCRIPT',
        p_help_text          => q'~<p>Enter JavaScript code to execute when the page loads.
The code is executed after the JavaScript code generated by &PRODUCT_NAME..</p>
<p>Code you enter here replaces the <code>#GENERATED_JAVASCRIPT#</code> substitution string in the page template.</p>
<p>Note: You do not need to add any opening or closing <code>script</code> tags, just the JavaScript code.</p>~' );
    --
    l_property_id := 18;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'INCLUDE_STANDARD_JAVASCRIPT_AND_CSS',
        p_prompt             => 'Include Standard JavaScript and CSS',
        p_property_type      => 'YES NO',
        p_db_column          => 'INCLUDE_APEX_CSS_JS_YN',
        p_display_group_name => 'JAVASCRIPT',
        p_deprecated_values  => 'N',
        p_help_text          => q'~<p>&PRODUCT_NAME. typically requires specific JavaScript and CSS files.
For some applications, such as mobile applications, you may want to suppress the inclusion of these files on the page.</p>
<p>To Include the standard &PRODUCT_NAME. and JavaScript files select <strong>On</strong>. Selecting <strong>Off</strong> suppresses the inclusion of these files in the HTML Header.</p>
<p>Suppressing inclusion of these files can have adverse effects on an application and may cause it to break.
Therefore, disabling this attribute is only recommended for advanced developers.</p>~' );
    --
    l_property_id := 19;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'HTML_HEADER',
        p_prompt             => 'HTML Header',
        p_property_type      => 'HTML',
        p_db_column          => 'HTML_PAGE_HEADER',
        p_display_length     => 80,
        p_max_length         => 32767,
        p_display_group_name => 'HTML_HEADER',
        p_help_text          => q'~<p>Enter text to replace the <code>#HEAD#</code> substitution string in the page template header.
The values entered here are inserted after the HTML <code>&lt;head&gt;</code> tag. Use this page attribute to add page specific meta tags.
</p>
<p>Note: JavaScript code or Cascading Style Sheets should be added to the dedicated JavaScript and CSS attributes.</p>~' );
    --
    l_property_id := 20;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'PAGE_HTML_BODY_ATTRIBUTE',
        p_prompt             => 'Page HTML Body Attribute',
        p_property_type      => 'TEXT EDITOR',
        p_db_column          => 'HTML_PAGE_ONLOAD',
        p_display_length     => 80,
        p_max_length         => 4000,
        p_display_group_name => 'HTML_HEADER',
        p_help_text          => q'~Enter text substituted in for your page template's <code>#ONLOAD#</code> substitution string.
To use this feature, your page template must include the <code>#ONLOAD#</code> substitution string.~' );
    --
    l_property_id := 21;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'HEADER_TEXT',
        p_prompt             => 'Header Text',
        p_property_type      => 'HTML',
        p_db_column          => 'WELCOME_TEXT',
        p_display_length     => 80,
        p_max_length         => 4000,
        p_display_group_name => 'HEADER_AND_FOOTER',
        p_is_translatable    => true,
        p_help_text          => q'~Enter text or HTML you want to appear immediately following the page header and before the body content.~' );
    --
    l_property_id := 22;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'BODY_HEADER',
        p_prompt             => 'Body Header',
        p_property_type      => 'HTML',
        p_db_column          => 'BOX_WELCOME_TEXT',
        p_display_length     => 80,
        p_max_length         => 4000,
        p_display_group_name => 'HEADER_AND_FOOTER',
        p_is_translatable    => true,
        p_deprecated_values  => '$NOT_NULL$',
        p_help_text          => q'~Enter text or HTML you want to appear before regions that display on the running page.
Text you enter here appears before the page template <code>#BODY#</code> substitution string.~' );
    --
    l_property_id := 23;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'FOOTER_TEXT',
        p_prompt             => 'Footer Text',
        p_property_type      => 'HTML',
        p_db_column          => 'FOOTER_TEXT',
        p_display_length     => 80,
        p_max_length         => 4000,
        p_display_group_name => 'HEADER_AND_FOOTER',
        p_is_translatable    => true,
        p_help_text          => q'~Enter text or HTML that appears before the page template footer.~' );
    --
    l_property_id := 24;
    store_property (
        p_id                    => l_property_id,
        p_name                  => 'AUTHORIZATION_SCHEME',
        p_prompt                => 'Authorization Scheme',
        p_property_type         => 'COMPONENT',
        p_db_column             => 'REQUIRED_ROLE',
        p_display_group_name    => 'SECURITY',
        p_lov_type              => 'COMPONENT',
        p_lov_component_type_id => c_comp_type_authorization,
        p_help_text             => q'~Select an authorization scheme to control this component. The authorization must evaluate to TRUE in order for this component to be rendered or executed.~' );
    --
    l_property_id := 25;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'PAGE_AUTHENTICATION',
        p_prompt             => 'Authentication',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'PAGE_IS_PUBLIC_Y_N',
        p_display_group_name => 'SECURITY',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Specify whether this page is public or requires authentication.
If a page is identified as public, the page can be viewed by any user who accesses the page.
The application's page sentry function may access this page attribute to identify pages that do not require prior authentication to view.
The implementation of the authentication scheme's page sentry function determines whether this attribute has any effect.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Page Requires Authentication',
        p_return_value  => 'N' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Page Is Public',
        p_return_value  => 'Y' );
    --
    l_property_id := 26;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'DEEP_LINKING',
        p_prompt             => 'Deep Linking',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'DEEP_LINKING',
        p_display_group_name => 'SECURITY',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select how external URL requests to a page are handled.
Page requests from within this application to this page, are not considered deep linking, and are not governed by this attribute's setting.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Enabled',
        p_return_value  => 'Y',
        p_help_text     => q'~If this page is specified as the target page in a URL, it ultimately redirects to this page.<br>
If the user is not already authenticated and the page is not public, then the user is prompted to login.
After successfully logging into the application, the user is redirected to this page.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Disabled',
        p_return_value  => 'N',
        p_help_text     => q'~If the URL does not contain a valid session ID, &PRODUCT_NAME. starts a new session and redirects the user to the application home page, rather than to this page.~' );
    --
    l_property_id := 27;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'PAGE_ACCESS_PROTECTION',
        p_prompt             => 'Page Access Protection',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'PROTECTION_LEVEL',
        p_display_group_name => 'SECURITY',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select what type of session state protection is implemented for this page.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Unrestricted',
        p_return_value  => 'N',
        p_help_text     => q'~The page may be requested using a URL, with or without session state arguments, and without having to have a checksum.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Arguments Must Have Checksum',
        p_return_value  => 'C',
        p_help_text     => q'~If Request, Clear Cache, or Name/Value Pair arguments appear in the URL, a checksum must also be provided.<br>
The checksum type must be compatible with the most stringent <em>Session State Protection</em> attribute of all the items passed as arguments.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'No Arguments Supported',
        p_return_value  => 'U',
        p_help_text     => q'~A URL may be used to request the page, but the URL can not contain Request, Clear Cache, or Name/Value Pair arguments.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'No URL Access',
        p_return_value  => 'D',
        p_help_text     => q'~The page may not be accessed using a URL.<br>
However, the page may be the target of a <em>Branch to Page</em> branch type, as this does not perform a URL redirect.~' );
    --
    l_property_id := 28;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'ALLOW_DUPLICATE_PAGE_SUBMISSIONS',
        p_prompt             => 'Enable duplicate page submissions',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'ALLOW_DUPLICATE_SUBMISSIONS',
        p_display_group_name => 'ADVANCED',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select whether the end user may submit the same page more than once.
Use this feature to avoid duplicate inserts into the same table.
This can happen when an end user clicks the browser's back button and then submits the page again, or if the end user clicks the browser's reload button.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'No - Prevent page from being re-posted',
        p_return_value  => 'N' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Yes - Enable page to be re-posted',
        p_return_value  => 'Y' );
    --
    l_property_id := 29;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'ON_DUPLICATE_PAGE_SUBMISSIONS_GO_TO_THIS_URL',
        p_prompt             => 'On duplicate page submissions go to this URL',
        p_property_type      => 'TEXT',
        p_db_column          => 'ON_DUP_SUBMISSION_GOTO_URL',
        p_display_length     => 70,
        p_max_length         => 4000,
        p_display_group_name => 'ADVANCED',
        p_help_text          => q'~Enter a URL target for when an end user attempts to perform a second page submission. &PRODUCT_NAME. displays an error message and a link to this URL to enable the user to continue.~' );
    --
    l_property_id := 30;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'FORM_AUTO_COMPLETE',
        p_prompt             => 'Form Auto Complete',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'AUTOCOMPLETE_ON_OFF',
        p_display_group_name => 'SECURITY',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Specify whether auto-complete is disabled for this page. A value of <strong>On</strong> has no effect on this page. A value of <strong>Off</strong> generates the following HTML in the FORM tag:
<pre>
autocomplete="off"
</pre>
<p>Note: Different browsers may have different rules around when <code>autocomplete="off"</code> is respected or ignored. Therefore please bear this in mind specifically when setting this to <strong>Off</strong>, and test in your target browsers.</p>~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'On',
        p_return_value  => 'ON' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Off',
        p_return_value  => 'OFF' );
    --
    l_property_id := 31;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'HIDDEN_REQUIRED_VALUE_TYPE',
        p_prompt             => 'Required Type',
        p_property_type      => 'HIDDEN',
        p_db_column          => 'REQUIRED_VALUE_TYPE',
        p_display_group_name => 'PARAMETER' );
    --
    l_property_id := 32;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'CACHE_TIMEOUT',
        p_prompt             => 'Cache Timeout',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'CACHE_TIMEOUT_SECONDS',
        p_display_group_name => 'SERVER_CACHE',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select how long a cached page is valid. Subsequent page requests do not utilize the cached page if the timeout expires.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '10 seconds',
        p_return_value  => '10' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '1 minute',
        p_return_value  => '60',
        p_is_quick_pick => 'Y' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '10 minutes',
        p_return_value  => '600' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '30 minutes',
        p_return_value  => '1800' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '1 hour',
        p_return_value  => '3600',
        p_is_quick_pick => 'Y' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '2 hours',
        p_return_value  => '7200' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '3 hours',
        p_return_value  => '10800' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '4 hours',
        p_return_value  => '14400' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '6 hours',
        p_return_value  => '21600' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '12 hours',
        p_return_value  => '43200' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '1 day',
        p_return_value  => '86400',
        p_is_quick_pick => 'Y' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '2 days',
        p_return_value  => '172800' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '3 days',
        p_return_value  => '259200' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '4 days',
        p_return_value  => '345600' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '1 week',
        p_return_value  => '604800' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '2 weeks',
        p_return_value  => '1209600' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '4 Weeks',
        p_return_value  => '2419200' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '10 Weeks',
        p_return_value  => '6048000' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => '1 Year',
        p_return_value  => '31536000' );
    --
    l_property_id := 33;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'BROWSER_CACHE',
        p_prompt             => 'Browser Cache',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'BROWSER_CACHE',
        p_display_group_name => 'SECURITY',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~<p>If the cache is enabled, the browser can save the contents of your application's pages in its cache, both in memory and on disk.
If a user clicks the browser back button, the page typically loads from the cache, not from the server.
If the cache is disabled, the browser is instructed to not save page content and requests it anew from the server.</p>
<p>Oracle recommends, from a security perspective, that the cache is disabled, so the browser does not store sensitive data and always requests pages if the URL changes.
Otherwise, it may even be possible to go back in the browser history after a log out and see cached content of the former session.
Disabling the browser cache prevents subtle back button issues with pages that use partial page refresh, such as interactive reports.</p>
<p>If this attribute is set to <strong>Disabled</strong>, &PRODUCT_NAME. sends the HTTP header <em>cache-control: no-store</em> which instructs the Browser to not cache the page contents on disk or in memory.
If it is set to <strong>Application Default</strong>, the value at runtime is taken from the application's <em>Browser Cache</em> attribute.</p>
<p>Note: This feature requires modern browsers that support the HTTP header response variable <code>cache-control</code>.</p>~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Enabled',
        p_return_value  => 'Y' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Disabled',
        p_return_value  => 'N' );
    --
    l_property_id := 34;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'API_DIRECTION',
        p_prompt             => 'Direction',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'DIRECTION',
        p_display_group_name => 'PARAMETER',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select the direction of the parameter as it's defined in the selected procedure or function.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'In',
        p_return_value  => 'IN' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Out',
        p_return_value  => 'OUT' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'In/Out',
        p_return_value  => 'IN_OUT' );
    --
    l_property_id := 35;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'IGNORE_OUTPUT',
        p_prompt             => 'Ignore Output',
        p_property_type      => 'YES NO',
        p_db_column          => 'IGNORE_OUTPUT',
        p_display_group_name => 'PARAMETER',
        p_help_text          => q'~Specify whether the parameters IN OUT, OUT or function result should be ignored.~' );
    --
    l_property_id := 36;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'API_DATA_TYPE',
        p_prompt             => 'Data Type',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'DATA_TYPE',
        p_display_group_name => 'PARAMETER',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Data type of the parameter as it's defined in the selected procedure or function.~' );
    --
    -- Supported standard parameter data types
    -- Note: Keep in sync with wwv_flow_property_dev.plb / emit_plsql_method_parameters,
    --                         wwv_flow_process_native.plb / invoke_api
    --                         dev_views.sql / wwv_flow_invokeapi_cparams_dev
    --                         PD config API_DATA_TYPE
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'VARCHAR2',
        p_return_value  => 'VARCHAR2' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'CLOB',
        p_return_value  => 'CLOB' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'NUMBER',
        p_return_value  => 'NUMBER' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'PL/SQL INTEGER',
        p_return_value  => 'PLS_INTEGER' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'BOOLEAN',
        p_return_value  => 'BOOLEAN' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'DATE',
        p_return_value  => 'DATE' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'TIMESTAMP',
        p_return_value  => 'TIMESTAMP' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'TIMESTAMP WITH LOCAL TIME ZONE',
        p_return_value  => 'TIMESTAMP WITH LOCAL TIME ZONE' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'TIMESTAMP WITH TIME ZONE',
        p_return_value  => 'TIMESTAMP WITH TIME ZONE' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'INTERVAL YEAR TO MONTH',
        p_return_value  => 'INTERVAL YEAR TO MONTH' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'INTERVAL DAY TO SECOND',
        p_return_value  => 'INTERVAL DAY TO SECOND' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'BLOB',
        p_return_value  => 'BLOB' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'CHAR',
        p_return_value  => 'CHAR' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'ROWID',
        p_return_value  => 'ROWID' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Custom',
        p_return_value  => 'CUSTOM',
        p_help_text     => q'~Use Custom if it's not a standard PL/SQL data type. Custom allows to specify the name of a PL/SQL record type or Database object type.~' );
    --
    l_property_id := 37;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'API_CUSTOM_DATA_TYPE',
        p_prompt             => 'Data Type Name',
        p_property_type      => 'TEXT',
        p_db_column          => 'CUSTOM_DATA_TYPE',
        p_display_group_name => 'PARAMETER',
        p_help_text          => q'~Enter the name of the custom data type of the parameter.~' );
    --
    l_property_id := 38;
    store_property (
        p_id                    => l_property_id,
        p_name                  => 'BUILD_OPTION',
        p_prompt                => 'Build Option',
        p_property_type         => 'COMPONENT',
        p_db_column             => 'REQUIRED_PATCH',
        p_display_group_name    => 'CONFIGURATION',
        p_lov_type              => 'COMPONENT',
        p_lov_component_type_id => c_comp_type_build_option,
        p_help_text             => q'~<p>Select a build option for this component.</p>
                                     <p>Build Options allow application components and functionality to be conditionally enabled or disabled when the application is installed, or at runtime via API (APEX_UTIL.SET_BUILD_OPTION_STATUS).<br>
                                       Build Options can set this component to be Included or Excluded in the runtime application:</p>
                                     <p><strong>Included</strong> - Enabled and are part of the application.</p>
                                     <p><strong>Excluded</strong> - Disabled and are not a part of the application.</p>~',
        p_examples              => q'~<p>1. You are developing an application that will have many installations.
                                        Additionally, each installation may choose to enable or disable certain application functionality.
                                        With Build Options, you can develop one application and when installed, build options can be configured to include or exclude associated application functionality.
                                        You may also use the API to expose this feature within the application.
                                        This allows privileged end-users to selectively enable or disable application functionality.</p>
                                      <p>2. You are developing functionality which is not yet complete, so you do not wish to expose to end-users.
                                        Using Build Options enables you to disable this functionality from the application when it is built into the test and production environments.
                                        The benefit of this approach is that the application can still be deployed without including incomplete functionality.</p>
                                      <p>3. You are removing functionality you think is not needed by the application, but are unsure this is a safe operation.
                                        In this case, you can use Build Options to essentially "comment out" specific functionality.
                                        If on deployment, the removed functionality was deemed to be necessary, it can be re-instated.
                                        If this functionality was deemed to be unnecessary, then it can be safely removed from the application.</p>~' );
    --
    l_property_id := 39;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'INLINE_ERROR_NOTIFICATION_TEXT',
        p_prompt             => 'In-line Error Notification Text',
        p_property_type      => 'TEXT EDITOR',
        p_db_column          => 'ERROR_NOTIFICATION_TEXT',
        p_display_length     => 80,
        p_max_length         => 4000,
        p_display_group_name => 'ERROR_HANDLING',
        p_help_text          => q'~Enter the error message displayed when an error occurs on this page. The error message is substituted for <code>#NOTIFICATION_MESSAGE#</code> in the page template.~' );
    --
    l_property_id := 40;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'HELP_TEXT',
        p_prompt             => 'Help Text',
        p_property_type      => 'HTML',
        p_db_column          => 'HELP_TEXT',
        p_display_length     => 80,
        p_max_length         => 4000,
        p_display_group_name => 'HELP',
        p_is_translatable    => true,
        p_help_text          => q'~<p>Use this attribute to enter help text for the current page. Page level help supports shortcuts using the following syntax:</p>
<pre>"SHORTCUT_NAME"</pre>
<p>Page help text is not displayed by default. However, you can readily develop a help system that exposes this text.</p>~' );
    --
    l_property_id := 41;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'ERROR_HANDLING_FUNCTION',
        p_prompt             => 'Error Handling Function',
        p_property_type      => 'TEXT', -- $$$ for Advisor we should have a new property_type FUNCTION_NAME/PROCEDURE_NAME
        p_db_column          => 'ERROR_HANDLING_FUNCTION',
        p_display_length     => 78,
        p_max_length         => 255,
        p_display_group_name => 'ERROR_HANDLING',
        p_help_text          => q'~<p>Enter the name of the PL/SQL function called when an error occurs within the application.
You can use this function to act on various error codes including those emanating from the Oracle database, and log such errors in a custom table you define if necessary.
The function can include logic to modify the error message and display location, transforming Oracle database errors into user friendly messages.
For example, you can capture a unique key violation on <code>DEMO_CUSTOMERS</code> and replace the <code>ORA-00001</code> message with this text: Customer Name must be unique.
Installing the <strong>Poll</strong> Starter App, from the Gallery, creates the <code>EBA_QPOLL_FW</code> package, which includes the <code>apex_error_handling</code> function.
This error handling function includes extensive comments to make it easy for you to utilize as the basis for defining your own function.</p>
<p>The function name entered can reference a package function or stand alone function in the database. For example:
<pre>
log_apex_error
</pre>
</p>
<p>When referencing a database PL/SQL package or stand alone function, you can use the <code>#OWNER#</code> substitution string to reference the parsing schema of the current application. For example:
<pre>#OWNER#.log_apex_error</pre>
</p>
<p>If an error handling function is specified on a page then it is utilized, instead of this application level setting, for that page.</p>
<p>Error Handling Functions must implement the following interface:
<pre>
function &lt;name of function> (
    p_error in apex_error.t_error )
    return apex_error.t_error_result
</pre>
</p>
<p>See the API Reference manual for a description of the package <code>apex_error</code>.</p>~' );
    --
    l_property_id := 42;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'HIDDEN_VALUE_TYPE',
        p_prompt             => 'Hidden Value Type',
        p_property_type      => 'HIDDEN',
        p_db_column          => 'HIDDEN_VALUE_TYPE',
        p_display_group_name => 'PARAMETER' );
    --
    l_property_id := 43;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'ENABLE_META_TAGS',
        p_prompt             => 'Enable Meta Tags',
        p_property_type      => 'YES NO',
        p_db_column          => 'ENABLE_META_TAGS',
        p_display_group_name => 'ADVANCED',
        p_help_text          => q'~Select if meta tags can be added on this public page.~' );
    --
    l_property_id := 44;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'MEDIA_TYPE',
        p_prompt             => 'Media Type',
        p_property_type      => 'TEXT',
        p_db_column          => 'MEDIA_TYPE',
        p_display_length     => 30,
        p_max_length         => 255,
        p_display_group_name => 'APPEARANCE',
        p_help_text          => q'~<p>Enter the Internet media type. An Internet media type is a two part identifier for file formats on the Internet.
A media type is composed of at least two parts: a type, a subtype, and one or more optional parameters.
This media type is used in the Content-Type HTTP header when rendering the page.</p>
<p>The page-level Media type overrides the application-level media type.
If both the page-level and application-level values for media type are <code>NULL</code>, the media type <code>text/html</code> is used.</p>~',
        p_comment_text       => q'~$$$ Should this property be deprecated???~' );
    --
    l_property_id := 45;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'CSS_FILE_URLS',
        p_prompt             => 'File URLs',
        p_property_type      => 'FILE URLS CSS',
        p_db_column          => 'CSS_FILE_URLS',
        p_display_length     => 80,
        p_max_length         => 4000,
        p_examples           => q'~<dl>
<dt>Standard file reference</dt>
<dd><pre>/mycss/main.css</pre></dd>
<dt>Standard file reference which loads the minified file main.min.css from /mycss/minified/ for regular page views and main.css from /mycss/ in debug mode or when running the application from App Builder.</dt>
<dd><pre>/mycss/#MIN_DIRECTORY#main#MIN#.css</pre></dd>
<dt>Conditional media query</dt>
<dd><pre>[media="only screen and (max-device-width: 480px)"]/mycss/smartphone.css</pre></dd>
<dt>File reference using a query string in the URL referencing the application version, such that when the application is upgraded (and the version is incremented), the browser is instructed to always load the new file and ignore any previously cached files.</dt>
<dd><pre>/mycss/main.css?version=#APP_VERSION#</pre></dd>
</dl>~',
        p_display_group_name => 'CSS',
        p_help_text          => q'~<p>Enter Cascading Style Sheet (CSS) file URLs loaded on this page. Each URL must be written on a new line.
If you provide a minified version of your file you can use the substitution string <code>#MIN#</code> to include <code>.min</code> or <code>#MIN_DIRECTORY#</code> to include <code>minified/</code> in your file URL for a regular page view and an empty string if the page is viewed in debug mode or when running the application from App Builder.
You also have access to the substitution string <code>#APP_VERSION#</code> if you want to include the application's version in the file URL.</p>
<p>File URLs you enter here replace the <code>#PAGE_CSS#</code> substitution string in the page template.</p>
<p>Each file URL can optionally be prefixed with <code>[media ...]</code> for conditional loading. See the examples for details.</p>
<p>Note: You do not need to include opening or closing <code>link</code> tags. Just include the file URL.</p>
<p>To comment out a File URL, prefix the line with two dashes: <code>--</code>. To comment out multiple lines, use the typical code block syntax: <code>/* */</code>.</p>~' );
    --
    l_property_id := 46;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'INLINE_CSS',
        p_prompt             => 'Inline',
        p_property_type      => 'CSS',
        p_db_column          => 'INLINE_CSS',
        p_display_length     => 80,
        p_max_length         => 32767,
        p_examples           => q'~<dl>
<dt>Sets the CSS class <code>display_only</code> to bold</dt>
<dd><pre>
.display_only {
  font-weight: bold;
}
</pre></dd>
</dl>~',
        p_display_group_name => 'CSS',
        p_help_text          => q'~<p>Enter CSS code for this page. Any CSS entered here will only be applied to this page and will not affect any other pages. The entered CSS code will replace the #PAGE_CSS# substitution string in the page template.</p>
<p>If the same CSS is required on several pages, it's best to put the code into an external file. This will help to avoid duplication and take advantage of browser caching of static files. You can reference the external file at the application level, under Shared Components, User Interface Attributes, CSS, File URLs.</p>
<p>Note that you don't need to include opening or closing <code>style</code> tags. Simply add your CSS code.</p>~' );
    --
    l_property_id := 47;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'DIALOG_WIDTH',
        p_prompt             => 'Width',
        p_property_type      => 'TEXT',
        p_db_column          => 'DIALOG_WIDTH',
        p_display_length     => 10,
        p_max_length         => 20,
        p_display_group_name => 'DIALOG',
        p_help_text          => q'~<p>Enter the width of the dialog, in pixels.
The substitution string <code>#DIALOG_WIDTH#</code> in the dialog initialization code of the page template is replaced with the value entered in this item.</p>~' );
    --
    l_property_id := 48;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'DIALOG_MAX_WIDTH',
        p_prompt             => 'Maximum Width',
        p_property_type      => 'TEXT',
        p_db_column          => 'DIALOG_MAX_WIDTH',
        p_display_length     => 10,
        p_max_length         => 20,
        p_display_group_name => 'DIALOG',
        p_help_text          => q'~<p>Enter the maximum width of the dialog in pixels.
The substitution string <code>#DIALOG_MAX_WIDTH#</code> in the dialog initialization code of the page template is replaced with the value entered in this item.</p>~' );
    --
    l_property_id := 49;
    store_property (
        p_id                    => l_property_id,
        p_name                  => 'DIALOG_ATTRIBUTES',
        p_prompt                => 'Attributes',
        p_property_type         => 'COMBOBOX',
        p_multi_value_delimiter => ',',
        p_db_column             => 'DIALOG_ATTRIBUTES',
        p_display_length        => 100,
        p_max_length            => 255,
        p_display_group_name    => 'DIALOG',
        p_lov_type              => 'DISTINCT',
        p_help_text             => q'~A subset of dialog attributes are declaratively supported: <em>height, width, and maximum width</em>.
To define any additional attributes for the dialog, enter the name/value pairings in a comma separated list.
<p>The substitution string <code>#DIALOG_ATTRIBUTES#</code> in the dialog initialization code defined at page template level, is replaced by any value(s) entered in this item.</p>
<p>The jQuery UI dialog widget is utilized.</p>
<p>For example, a dialog page could have the following attributes defined: <code>draggable:true</code></p>
<p>For further information on the attributes available, refer to the <a rel="noopener noreferrer" target="_blank" href="http://api.jqueryui.com/dialog/">jQuery UI dialog widget documentation</a>.</p>~' );
    --
    l_property_id := 50;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'DESCRIPTION',
        p_prompt             => 'Description',
        p_property_type      => 'TEXT EDITOR',
        p_db_column          => 'DESCRIPTION',
        p_display_length     => 60,
        p_max_length         => 4000,
        p_display_group_name => 'DESCRIPTION',
        p_help_text          => q'~Enter a description.~' );
    --
    l_property_id := 51;
    store_property (
        p_id                       => l_property_id,
        p_name                     => 'COMPUTATION_ITEM_NAME',
        p_prompt                   => 'Item Name',
        p_property_type            => 'ITEM',
        p_db_column                => 'COMPUTATION_ITEM',
        p_display_length           => 50,
        p_max_length               => 255,
        p_display_group_name       => 'IDENTIFICATION',
        p_lov_component_type_scope => 'APPLICATION',
        p_help_text                => q'~Enter the page or application item populated by this computation. You can type in the name or pick from the list of available items.~' );
    --
    l_property_id := 52;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'COMPUTATION_TYPE',
        p_prompt             => 'Type',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'COMPUTATION_TYPE',
        p_display_group_name => 'COMPUTATION',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select a computation type from the following list.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Static Value',
        p_return_value  => 'STATIC_ASSIGNMENT',
        p_help_text     => q'~Set to the text entered into <em>Static Value</em>.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Item',
        p_return_value  => 'ITEM_VALUE',
        p_help_text     => q'~Set to the value held in session state for the <em>Item</em> selected.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'SQL Query (return single value)',
        p_return_value  => 'QUERY',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~Set to the first value returned from the <em>SQL Query</em> entered.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'SQL Query (return multiple values)',
        p_return_value  => 'QUERY_COLON',
        p_help_text     => q'~Set to multiple values returned from the <em>SQL Query</em>. The format of what is returned is determined by the current item's Multiple Values setting. If Multiple Values is set to <em>Delimited Values</em>, the value will be a single concatenated list separated by the defined Separator. If Multiple Values is set to <em>JSON Array</em>, the value will be a JSON Array of the values. If Multiple Values is set to <em>No</em> or the item does not support multiple values, then the value is a single concatenated list separated by a colon.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Expression',
        p_return_value  => 'EXPRESSION',
        p_help_text     => q'~Set to the value that results from executing the <em>Expression</em> entered.~'  );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Function Body',
        p_return_value  => 'FUNCTION_BODY',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~Set to the value returned from the <em>Function Body</em> entered.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Preference',
        p_return_value  => 'SET_ITEM_EQUAL_THIS_PREFERENCE',
        p_help_text     => q'~Set to the value of the <em>Preference</em> entered.~' );
    --
    l_property_id := 53;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'COMPUTATION_STATIC_VALUE',
        p_prompt             => 'Static Value',
        p_property_type      => 'TEXT',
        p_db_column          => 'COMPUTATION',
        p_display_length     => 80,
        p_max_length         => 4000,
        p_display_group_name => 'COMPUTATION',
        p_help_text          => q'~Enter the text that is used to populate the computation item.~' );
    --
    l_property_id := 54;
    store_property (
        p_id                   => l_property_id,
        p_name                 => 'COMPUTATION_SQL_STATEMENT',
        p_prompt               => 'SQL Query',
        p_property_type        => 'SQL',
        p_db_column            => 'COMPUTATION',
        p_display_length       => 80,
        p_max_length           => 4000,
        p_sql_min_column_count => 1,
        p_sql_max_column_count => 1,
        p_display_group_name   => 'COMPUTATION',
        p_help_text            => q'~Enter a SQL query that provides a value used to populate the computation item. If the result contains multiple rows then only the value from the first row is returned.~' );
    --
    l_property_id := 55;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'PRINT_SERVER_TYPE',
        p_prompt             => 'Type',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'PRINT_SERVER_TYPE',
        p_display_group_name => 'IDENTIFICATION',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select the type of print server.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Oracle BI Publisher',
        p_return_value  => 'BIP',
        p_help_text     => q'~Select this option if you are using a licensed version of Oracle BI Publisher.
This option enables you to take report query results and convert them from XML to RTF format using Oracle BI Publisher.
Select this option if you want to upload your own customized RTF or XSL-FO templates for printing reports within &PRODUCT_NAME..~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Apache FOP',
        p_return_value  => 'FOP',
        p_help_text     => q'~Select this option if you are using Apache FOP on an external J2EE server.
This option enables you to use the basic printing functionality, which includes creating report queries and
printing report regions using the default templates provided in &PRODUCT_NAME. and using your own customized XSL-FO templates.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'APEX Office Print',
        p_return_value  => 'AOP',
        p_help_text     => q'~Select this option if you are using APEX Office Print.~' );
    --
    l_property_id := 56;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'CREDENTIAL_TYPE',
        p_prompt             => 'Type',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'AUTHENTICATION_TYPE',
        p_display_group_name => 'AUTHENTICATION',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~<p>Specify the Authentication type here. &PRODUCT_NAME. supports the following authentication types:</p>
<ul>
    <li><strong>Basic Authentication</strong> - Sends username and password in Base64-encoded form as the <strong>Authorization</strong> request header.</li>
    <li><strong>OAuth2 Client Credentials</strong> - &PRODUCT_NAME. exchanges the client ID and client secret for an <strong>Access Token</strong> using a token server URL. The access token is t
hen used to perform the actual request. If the access token is expired, &PRODUCT_NAME. will transparently request a new one.</li>
    <li><strong>OCI Native Authentication</strong> - &PRODUCT_NAME. signs requests to the <em>Oracle Cloud Infrastructur
e (OCI)</em> REST API, as described in the <a rel="noopener noreferrer" href="https://docs.cloud.oracle.com/en-us/iaas/Content/API/Concepts/signingrequests.htm">OCI Documentation</a>.</li>
    <li><strong>HTTP Hea
der</strong> - The credential is added to the REST request as an HTTP Header. The name of the credential is used as the HTTP Header name, and the Secret of the credential is used as the HTTP Header value. &PRODUCT_NAME. does not add these secrets to
the Debug log or any other logs.</li>
    <li><strong>URL Query String</strong> -  The credential is added to the URL of the REST request as a <em>Query String</em> Parameter (for example: <strong>?name=value</strong>). This option can be useful with
 API keys. &PRODUCT_NAME. does not add these secrets to the Debug log or any other logs. However, web proxy or other intermediate servers may record these credentials in their log files.</li>
    <li><strong>Key Pair</strong> - A key pair credential consists of a public key, openly shared for encrypting data, and a private key, securely kept secret for decrypting data, together ensuring a secure data exchange.</li>
</ul>~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Basic Authentication',
        p_return_value  => 'BASIC' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'OAuth2 Client Credentials Flow',
        p_return_value  => 'OAUTH2_CLIENT_CREDENTIALS' );
    --
    /* $$$ currently commented out
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'JSON Web Token',
        p_return_value  => 'JWT' );
    */
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Oracle Cloud Infrastructure (OCI)',
        p_return_value  => 'OCI' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Certificate/Private Key Pair',
        p_return_value  => 'CERTIFICATE_PAIR',
        p_comment_text  => q'~$$$ Condition of type wwv_flow_platform.get_preference('SAML_ENABLED')='Y'~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'HTTP Header',
        p_return_value  => 'HTTP_HEADER' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'URL Query String',
        p_return_value  => 'HTTP_QUERY_STRING' );
    --
    l_property_id := 57;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'OAUTH_SCOPE',
        p_prompt             => 'OAuth Scope',
        p_property_type      => 'TEXT',
        p_db_column          => 'SCOPE',
        p_display_length     => 50,
        p_max_length         => 255,
        p_display_group_name => 'AUTHENTICATION',
        p_help_text          => q'~
<p>
Permissions represented by the Access Token in OAuth 2.0 terms are known as scopes. The scope parameter allows the application to express the desired scope of the access request.
</p>
<p>
If your authentication server requires a scope to be specified for the access token request, provide it here. The OAuth2 access token will then be requested with the following request body:
<pre>
grant_type=client_credentials&amp;scope=<strong>{scope}</strong>
</pre>
</p>~' );
    --
    l_property_id := 58;
    store_property (
        p_id                       => l_property_id,
        p_name                     => 'COMPUTATION_ITEM_VALUE',
        p_prompt                   => 'Item Name',
        p_property_type            => 'ITEM',
        p_db_column                => 'COMPUTATION',
        p_display_length           => 80,
        p_max_length               => 4000,
        p_display_group_name       => 'COMPUTATION',
        p_lov_component_type_scope => 'APPLICATION',
        p_help_text                => q'~Enter the item whose value is used to populate the computation item. You can type in the name or pick from the list of available items.~' );
    --
    l_property_id := 59;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'COMPUTATION_PREFERENCE_VALUE',
        p_prompt             => 'Preference',
        p_property_type      => 'TEXT',
        p_db_column          => 'COMPUTATION',
        p_display_length     => 80,
        p_max_length         => 4000,
        p_display_group_name => 'COMPUTATION',
        p_help_text          => q'~Enter a User Preference that has a value used to populate the computation item.~' );
    --
    l_property_id := 60;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'EXECUTION_SEQUENCE',
        p_prompt             => 'Sequence',
        p_property_type      => 'NUMBER',
        p_db_column          => 'SEQUENCE',
        p_display_length     => 10,
        p_display_group_name => 'EXECUTION',
        p_help_text          => q'~<p>Specify the sequence for this component. The sequence determines the order of execution.</p>
<p>Note: If two components have the same sequence value, then they may be executed in a different order when the application is exported and imported into another environment.</p>~' );
    --
    l_property_id := 61;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'COMPUTATION_POINT',
        p_prompt             => 'Point',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'COMPUTATION_POINT',
        p_display_group_name => 'EXECUTION',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select at what point in page rendering or processing this computation is executed.
The most typical computation point is <strong>After Submit</strong>.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'New Session',
        p_return_value  => 'ON_NEW_INSTANCE',
        p_help_text     => q'~Executes when a new session is generated.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Before Header',
        p_return_value  => 'BEFORE_HEADER',
        p_help_text     => q'~Rendering: Executes before the page renders.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'After Header',
        p_return_value  => 'AFTER_HEADER',
        p_help_text     => q'~Rendering: Executes after the header renders.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Before Regions',
        p_return_value  => 'BEFORE_BOX_BODY',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~Rendering: Executes before regions render.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'After Regions',
        p_return_value  => 'AFTER_BOX_BODY',
        p_help_text     => q'~Rendering: Executes after regions render.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Before Footer',
        p_return_value  => 'BEFORE_FOOTER',
        p_help_text     => q'~Rendering: Executes before footer renders.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'After Footer',
        p_return_value  => 'AFTER_FOOTER',
        p_help_text     => q'~Rendering: Executes after footer renders.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'After Submit',
        p_return_value  => 'AFTER_SUBMIT',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~Processing: Executes after the page is submitted, but before page processing.~' );
    --
    l_property_id := 62;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'EVENT',
        p_prompt             => 'Event',
        p_property_type      => 'SUPPORTED UI',
        p_db_column          => 'BIND_EVENT_TYPE',
        p_display_group_name => 'WHEN',
        p_lov_type           => 'EVENTS',
        p_help_text          => q'~<p>Specify the event that causes the dynamic action to fire.
For information about specific events, please see the <a rel="noopener noreferrer" href="#SYSTEM_HELP_URL#CSH:pd-event" target="_blank">relevant section</a> in the <em>App Builder User's Guide</em>.</p>
Events are categorized into the following types:
<dl>
  <dt>Browser Events</dt>
    <dd>Standard browser events.</dd>
  <dt>Framework Events</dt>
    <dd>Framework events are triggered from within the Oracle APEX framework.</dd>
  <dt>Component Events</dt>
    <dd>Only available when there is a component available to your application that triggers a custom event. These events are triggered from either native components or plug-in components you have installed in your application.</dd>
  <dt>Custom Event</dt>
    <dd>Any custom event triggered from your application.</dd>
</dl>~' );
    --
    l_property_id := 63;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'CUSTOM_EVENT',
        p_prompt             => 'Custom Event',
        p_property_type      => 'TEXT',
        p_db_column          => 'BIND_EVENT_TYPE_CUSTOM',
        p_display_length     => 50,
        p_max_length         => 130,
        p_display_group_name => 'WHEN',
        p_help_text          => q'~Specify a custom JavaScript event name if the declarative events provided are not sufficient.~' );
    --
    l_property_id := 64;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'WHEN_TYPE',
        p_prompt             => 'Selection Type',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'TRIGGERING_ELEMENT_TYPE',
        p_display_group_name => 'WHEN',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select the type of page element or construct to be used to trigger the event. For examples of using a specific selection type, see Help on the associated attribute.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Item(s)',
        p_return_value  => 'ITEM',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~Triggered by one or more page items.~');
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Button',
        p_return_value  => 'BUTTON',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~Triggered by a button.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Region',
        p_return_value  => 'REGION',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~Triggered by a region.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Column(s)',
        p_return_value  => 'COLUMN',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~Triggered by one or more columns.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'DOM Object',
        p_return_value  => 'DOM_OBJECT',
        p_is_deprecated => 'Y',
        p_help_text     => q'~<p>Triggered by the DOM object (<code>document</code>, <code>window</code>) or the <em>id</em> of a DOM object (<code>myElementID</code>).</p>
        <p>Note: <em>DOM Object</em> is deprecated and should be replaced by <em>JavaScript Expression</em> or <em>jQuery Selector</em>.</p>~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'jQuery Selector',
        p_return_value  => 'JQUERY_SELECTOR',
        p_help_text     => q'~Triggered by one or more page elements returned by the jQuery Selector.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'JavaScript Expression',
        p_return_value  => 'JAVASCRIPT_EXPRESSION',
        p_help_text     => q'~Triggered by the page elements returned by the JavaScript expression.~' );
    --
    l_property_id := 65;
    store_property (
        p_id                       => l_property_id,
        p_name                     => 'WHEN_REGION',
        p_prompt                   => 'Region',
        p_property_type            => 'COMPONENT',
        p_db_column                => 'TRIGGERING_REGION_ID',
        p_display_group_name       => 'WHEN',
        p_lov_type                 => 'COMPONENT',
        p_lov_component_type_id    => c_comp_type_region,
        p_lov_component_type_scope => 'PAGE_AND_GLOBAL',
        p_help_text                => q'~<p>Select the name of the region that triggers the dynamic action.
For example this could be an 'Interactive Report' region, used in conjunction with an 'After Refresh' event, to define a dynamic action that fire whenever the report refreshes.</p>
<p>Having region trigger a dynamic action can be useful when you want to define similar behavior for multiple page items in a region. However, this is only possible for certain events that are raised by items within that region.
For a list of events that support this type of definition and an example of how they could be utilized, see item Help for the <em>Affected Elements</em> &gt; <em>Selection Type</em> of Event Scope on the Action page.</p>~' );
    --
    l_property_id := 66;
    store_property (
        p_id                       => l_property_id,
        p_name                     => 'WHEN_BUTTON',
        p_prompt                   => 'Button',
        p_property_type            => 'COMPONENT',
        p_db_column                => 'TRIGGERING_BUTTON_ID',
        p_display_group_name       => 'WHEN',
        p_lov_type                 => 'COMPONENT',
        p_lov_component_type_id    => c_comp_type_button,
        p_lov_component_type_scope => 'PAGE_AND_GLOBAL',
        p_help_text                => q'~Select the button to trigger the dynamic action.~' );
    --
    l_property_id := 67;
    store_property (
        p_id                       => l_property_id,
        p_name                     => 'WHEN_ITEMS',
        p_prompt                   => 'Item(s)',
        p_property_type            => 'ITEM',
        p_multi_value_delimiter    => ',',
        p_db_column                => 'TRIGGERING_ELEMENT',
        p_display_length           => 60,
        p_max_length               => 4000,
        p_display_group_name       => 'WHEN',
        p_lov_component_type_scope => 'PAGE_AND_GLOBAL',
        p_help_text                => q'~Enter page or application items to trigger the dynamic action.
For multiple items, separate each item name with a comma.
You can type in the name or pick from the list of available items.
If you pick from the list and there is already text entered, a comma is placed at the end of the existing text, followed by the item name returned from the list.~' );
    --
    l_property_id := 68;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'WHEN_DOM_OBJECT',
        p_prompt             => 'DOM Object',
        p_property_type      => 'TEXT',
        p_db_column          => 'TRIGGERING_ELEMENT',
        p_display_length     => 70,
        p_max_length         => 4000,
        p_display_group_name => 'WHEN',
        p_help_text          => q'~<p>Enter either a DOM object (<code>document</code>, <code>window</code>) or the <em>id</em> of a DOM object (<code>myElementID</code>) as the element to trigger the dynamic action.</p>
<p>Note: <em>DOM Object</em> is deprecated and should be replaced by <em>JavaScript Expression</em> or <em>jQuery Selector</em>.</p>~' );
    --
    l_property_id := 69;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'WHEN_JQUERY_SELECTOR',
        p_prompt             => 'jQuery Selector',
        p_property_type      => 'TEXT',
        p_db_column          => 'TRIGGERING_ELEMENT',
        p_display_length     => 70,
        p_max_length         => 4000,
        p_examples           => q'~<dl>
<dt>To select a page element with an DOM id of <code>my_id</code></dt>
<dd><pre>#my_id</pre></dd>
<dt>To select all the page elements with a class of <code>my_class</code></dt>
<dd><pre>.my_class</pre></dd>
<dt>To select all the page elements that are inputs</dt>
<dd><pre>input</pre></dd>
</dl>~',
        p_display_group_name => 'WHEN',
        p_help_text          => q'~<p>Enter jQuery Selector syntax to return one or more page element to trigger the dynamic action. The jQuery library offers a very powerful selector syntax which provides a way to easily access one or more elements from a page.</p>
<p>Please see the official documentation available from the <a rel="noopener noreferrer" target="_blank" href="http://docs.jquery.com">jQuery homepage</a> for further information about the selector syntax.</p>~' );
    --
    l_property_id := 70;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'JS_CONDITION',
        p_prompt             => 'Type',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'TRIGGERING_CONDITION_TYPE',
        p_display_group_name => 'CLIENT_SIDE_CONDITION',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Specify a client-side condition to control whether True or False actions fire. If the condition
                                  evaluates to True, then True actions fire, if the condition evaluates to False, then False actions fire.
                                  If no condition is specified, only True actions fire.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Item = Value',
        p_return_value  => 'EQUALS',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~Checks if the value of the selected <em>Item</em> is equal to the <em>Value</em> specified.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Item != Value',
        p_return_value  => 'NOT_EQUALS',
        p_help_text     => q'~Checks if the value of the selected <em>Item</em> is not equal to the <em>Value</em> specified.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Item > Value',
        p_return_value  => 'GREATER_THAN',
        p_help_text     => q'~Checks if the value of the selected <em>Item</em> is greater than the <em>Value</em> specified.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Item >= Value',
        p_return_value  => 'GREATER_THAN_OR_EQUAL',
        p_help_text     => q'~Checks if the value of the selected <em>Item</em> is greater than or equal to the <em>Value</em> specified.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Item < Value',
        p_return_value  => 'LESS_THAN',
        p_help_text     => q'~Checks if the value of the selected <em>Item</em> is less than the <em>Value</em> specified.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Item <= Value',
        p_return_value  => 'LESS_THAN_OR_EQUAL',
        p_help_text     => q'~Checks if the value of the selected <em>Item</em> is less than or equal to the <em>Value</em> specified.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Item is null',
        p_return_value  => 'NULL',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~Checks if the selected <em>Item</em> is empty.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Item is not null',
        p_return_value  => 'NOT_NULL',
        p_help_text     => q'~Checks if the selected <em>Item</em> is not empty.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Item is in list',
        p_return_value  => 'IN_LIST',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~Checks if the value of the selected <em>Item</em> is in the <em>List</em> specified.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Item is not in list',
        p_return_value  => 'NOT_IN_LIST',
        p_help_text     => q'~Checks if the value of the selected <em>Item</em> is not in the <em>List</em> specified.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'JavaScript expression',
        p_return_value  => 'JAVASCRIPT_EXPRESSION',
        p_help_text     => q'~Evaluates the JavaScript Expression specified.~' );
    --
    l_property_id := 71;
    store_property (
        p_id                       => l_property_id,
        p_name                     => 'WHEN_IG_REGION',
        p_prompt                   => 'Interactive Grid',
        p_property_type            => 'COMPONENT',
        p_db_column                => 'TRIGGERING_REGION_ID',
        p_display_group_name       => 'WHEN',
        p_lov_type                 => 'COMPONENT',
        p_lov_component_type_id    => c_comp_type_region,
        p_lov_component_type_scope => 'PAGE_AND_GLOBAL',
        p_help_text                => q'~<p>Select the name of the editable interactive grid region that triggers the dynamic action.~' );
    --
    l_property_id := 72;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'API_VALUE_TYPE_BASIC',
        p_prompt             => 'Type',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'VALUE_TYPE',
        p_display_group_name => 'VALUE',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select how this parameter is populated.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Static Value',
        p_return_value  => 'STATIC',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~Set to the value entered into <em>Static Value</em>.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Item',
        p_return_value  => 'ITEM',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~Set to the value held in session state for the <em>Item</em> selected.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'SQL Query (return single value)',
        p_return_value  => 'SQL_QUERY',
        p_help_text     => q'~Set to the first value returned from the <em>SQL Query</em> entered.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Expression',
        p_return_value  => 'EXPRESSION',
        p_help_text     => q'~Set to the value that results from executing the <em>Expression</em> entered.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Function Body',
        p_return_value  => 'FUNCTION_BODY',
        p_help_text     => q'~Set to the value returned from the <em>Function Body</em> entered.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Preference',
        p_return_value  => 'PREFERENCE',
        p_help_text     => q'~Set to the value of the <em>Preference</em> entered.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'API Default',
        p_return_value  => 'API_DEFAULT',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~Use the value defined as default in the API parameter declaration.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Null',
        p_return_value  => 'NULL',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~Set to NULL as no value specified.~' );
    --
    l_property_id := 73;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'API_VALUE_TYPE_BLOB',
        p_prompt             => 'Type',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'VALUE_TYPE',
        p_display_group_name => 'VALUE',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select how this parameter is populated.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'SQL Query (return single value)',
        p_return_value  => 'SQL_QUERY',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~Set to the first value returned from the <em>SQL Query</em> entered.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Expression',
        p_return_value  => 'EXPRESSION',
        p_help_text     => q'~Set to the value that results from executing the <em>Expression</em> entered.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Function Body',
        p_return_value  => 'FUNCTION_BODY',
        p_help_text     => q'~Set to the value returned from the <em>Function Body</em> entered.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'API Default',
        p_return_value  => 'API_DEFAULT',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~Use the value defined as default in the API parameter declaration.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Null',
        p_return_value  => 'NULL',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~Set to NULL as no BLOB specified.~' );
    --
    l_property_id := 74;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'API_VALUE_TYPE_CUSTOM',
        p_prompt             => 'Type',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'VALUE_TYPE',
        p_display_group_name => 'VALUE',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select how this parameter is populated.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'PL/SQL Expression',
        p_return_value  => 'EXPRESSION',
        p_help_text     => q'~Set to the value that results from executing the <em>PL/SQL Expression</em> entered.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'PL/SQL Function Body',
        p_return_value  => 'FUNCTION_BODY',
        p_help_text     => q'~Set to the value returned from the <em>PL/SQL Function Body</em> entered.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'API Default',
        p_return_value  => 'API_DEFAULT',
        p_help_text     => q'~Use the value defined as default in the API parameter declaration.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Null',
        p_return_value  => 'NULL',
        p_is_quick_pick => 'Y',
        p_help_text     => q'~Set to an empty data type.~' );
    --
    l_property_id := 75;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'PARAM_DISPLAY_SEQUENCE',
        p_prompt             => 'Display Sequence',
        p_property_type      => 'INTEGER',
        p_db_column          => 'DISPLAY_SEQUENCE',
        p_display_length     => 6,
        p_min_value          => '0',
        p_display_group_name => 'ADVANCED',
        p_help_text          => q'~Enter the display sequence for this parameter in Page Designer.~' );
    --
    l_property_id := 76;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'VALUE_STATIC_VALUE_BOOLEAN',
        p_prompt             => 'Static Value',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'VALUE',
        p_display_group_name => 'VALUE',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select the BOOLEAN value that is used as the source for this parameter value.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'True',
        p_return_value  => 'TRUE' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'False',
        p_return_value  => 'FALSE' );
    --
    l_property_id := 77;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'EVENT_SCOPE',
        p_prompt             => 'Event Scope',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'BIND_TYPE',
        p_display_group_name => 'EXECUTION',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~Select the scope of the event, which determines when the event is evaluated a second or successive time.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Static',
        p_return_value  => 'bind',
        p_help_text     => q'~Binds the event handler to the triggering element(s) for the lifetime of the current page, but is no longer bound if a triggering element is updated via Partial Page Refresh (PPR).~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Dynamic',
        p_return_value  => 'live',
        p_help_text     => q'~Binds the event handler to the triggering element(s) for the lifetime of the current page, irrespective of any triggering elements being recreated via Partial Page Refresh (PPR).~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Once',
        p_return_value  => 'one',
        p_help_text     => q'~Binds the event handler to the triggering element(s) only once. The dynamic action is not triggered again until after the page has been fully refreshed.~' );
    --
    l_property_id := 78;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'STATIC_CONTAINER_JQUERY',
        p_prompt             => 'Static Container (jQuery Selector)',
        p_property_type      => 'TEXT',
        p_db_column          => 'BIND_DELEGATE_TO_SELECTOR',
        p_display_length     => 50,
        p_max_length         => 255,
        p_display_group_name => 'EXECUTION',
        p_help_text          => q'~<p>Enter the jQuery Selector, of a static container, for use by this dynamic action.</p>
<p>You can improve the performance of dynamic actions that have an event scope of <strong>Dynamic</strong>.
This must be an element on the page that itself does not get recreated, but contains the triggering elements that are recreated using Partial Page Refresh (PPR).
This element reference needs to be defined as a jQuery selector.</p>~' );
    --
    l_property_id := 79;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'DA_EXECUTION_TYPE',
        p_prompt             => 'Type',
        p_property_type      => 'SELECT LIST',
        p_db_column          => 'EXECUTION_TYPE',
        p_display_group_name => 'EXECUTION',
        p_lov_type           => 'STATIC',
        p_help_text          => q'~<p>Choose the type of execution method for the Dynamic Action. This defines if the actions are fired immediately, debounced, or throttled.</p>
        <p>If a Client-side Condition is defined, it will be evaluated when the Dynamic Action and its Actions are executed, not when the event fires.</p>~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Immediate',
        p_return_value  => 'IMMEDIATE',
        p_help_text     => q'~Actions are executed immediately.~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Debounce',
        p_return_value  => 'DEBOUNCE',
        p_help_text     => q'~<p>Actions are only executed once after a specified time at the end of a single or series of events.</p>
        <p>Example 1:</p>
        <p>Update a text field if the user has stopped typing for 2 seconds and use the values last entered.</p>
        <p>Example 2:</p>
        <p>Refresh a region once after a click and ignore all other clicks in the next 5 seconds.</p>~' );
    --
    store_property_lov_values (
        p_property_id   => l_property_id,
        p_display_value => 'Throttle',
        p_return_value  => 'THROTTLE',
        p_help_text     => q'~<p>Actions are executed at timed intervals when the action is fired multiple times.</p>
        <p>Example 1:</p>
        <p>During scrolling, update the current screen position at least every 2 seconds. The action will continue to fire if the user keeps scrolling.</p>
        <p>Example 2:</p>
        <p>As a user types, update a text field with the values entered at least every 2 seconds. The action will continue to fire as the user keeps typing.</p>~' );
    --
    l_property_id := 80;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'DA_EXECUTION_TIME',
        p_prompt             => 'Time',
        p_property_type      => 'INTEGER',
        p_db_column          => 'EXECUTION_TIME',
        p_display_group_name => 'EXECUTION',
        p_min_value          => 1,
        p_unit               => c_unit_milliseconds,
        p_help_text          => q'~Specify the amount of time to delay the execution of actions based on the execution type. Measured in milliseconds where 1 second = 1000 milliseconds.~' );
    --
    l_property_id := 81;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'DA_EXECUTION_IMMEDIATE',
        p_prompt             => 'Immediate',
        p_property_type      => 'YES NO',
        p_db_column          => 'EXECUTION_IMMEDIATE',
        p_display_group_name => 'EXECUTION',
        p_help_text          => q'~Toggle when the selected execution type is invoked. Set to Yes for actions to be executed immediately before the timer starts. Set to No for actions to be executed after the value set in Time.~' );
    --
    l_property_id := 82;
    store_property (
        p_id                 => l_property_id,
        p_name               => 'DA_ACTION_TYPE',
        p_prompt             => 'Action',
        p_property_type      => 'SUPPORTED UI',
        p_db_column          => 'ACTION',
        p_display_group_name => 'IDENTIFICATION',
        p_lov_type           => 'PLUGINS',
        p_help_text          => q'~<p>Specify which action you want to perform. The following actions are supported:</p>
#NATIVE_PLUGINS#
<p>Depending on your application configuration, you may also have additional dynamic action plug-ins available here.
These are displayed as <em>&lt;My Plug-in&gt; [Plug-in]</em>.
The following plug-ins are installed within this application's shared components:</p>
#CUSTOM_PLUGINS#~' );
    --
    l_property_id := 83;
    store_property (
        p_id                 => l_property_id,
        p_name               =