/**
 * A TestDriver implements the platform-specific commands needed for testing
 * with adapters.
 */
export interface TestDriver {
    /**
     * Click on the given TestElement
     * @param element The TestElement on which to click
     */
    click(element: TestElement): Promise<void>;
    /**
     * Execute a script in the browser environment.
     * @param script The script to execute
     * @param args Any arguments to pass to the script. Access each argument through
     * the <code>arguments</code> array.
     */
    executeScript<T>(script: string, ...args: any): Promise<T>;
    /**
     * Find an element in the browser environment that matches the given CSS query.
     * This method should throw an exception if no matching elements are found.
     * @param query The CSS query to use in locate the element
     */
    findElement(query: string): Promise<TestElement>;
    /**
     * Find elements in the browser environment matching the gven CSS query. If no
     * matching elements are found, return a blank array.
     * @param query The CSS query to use in locating the elements.
     */
    findElements(query: string): Promise<TestElement[]>;
    /**
     *
     * @param element The element to receive the keystrokes
     * @param text
     */
    sendKeys(element: TestElement, text: string): Promise<void>;
}
/**
 * A TestElement represents the DOM element in the browser environment. This interface
 * is used by test adapters to work with elements without being bound to a specific
 * platform implementation.
 */
export interface TestElement {
    findElement(query: string): Promise<TestElement>;
    findElements(query: string): Promise<TestElement[]>;
}
/**
 * Set the TestDriver instance to be used for testing. This method should be called by
 * tests during setup, before calling any test adapter methods.
 *
 * @param driver The TestDriver instance
 */
export declare function setDriver(driver: TestDriver): void;
/**
 * Get the configured TestDriver instance to be used for testing. This method should
 * only be called by test adapters to interact with the browser environment.
 *
 * @returns {TestDriver} The TestDriver instance
 */
export declare function getDriver(): TestDriver;
